# //                          FILE
# // Associated with ziWireframeViewport_<version>.mll
# //
# //                          AUTHOR
# //  stuzzi(contact@vertexture.org)
# //  www.vertexture.org
# //  Please read on the website terms of use and licensing. Tutorials can be found also
# //
# //                          DATE
# //  25/02/2020
# //
# //                          DESCRIPTION
# //      Change mesh surface properties mesh display within the vertexture viewport
# //
# ////////////////////////////////////////////////////////////////////////////////////*/

from Qt.QtWidgets import *
from Qt.QtCore import *
from Qt.QtGui import *
import Qt.QtCompat

import zi_Widget.zi_Windows
import zi_UI.ziRessources_rc

import maya.cmds as cmds
from maya.mel import eval
import re

from pdb import set_trace as db

help = """
<html><head/><body><p align="center"><img src=":/vertexture/skin/vertexture/logoHs_color.png"/></p><p><br/><br/></p><p><span style=" font-size:10pt; font-weight:600; color:#7e835a;">ziWireframeViewport </span>displays the selected mesh geometry in a wireframe mode. This is a visual companion tool for retopology task.</p><p><br/></p><p><span style=" font-weight:600;">Surface Color - </span>Please pick the desired geometry displayed rgb color (<span style=" font-weight:600;">rgb.</span>a)</p><p><span style=" font-weight:600;">Line color</span> - This stands for wireframe color itself.</p><p><span style=" font-weight:600;">Surface Alpha</span> - so you can specify the alpha component of the Surface Alpha (rgb.<span style=" font-weight:600;">a</span>)</p><p><span style=" font-weight:600;">Depth Priority</span> - Actually the Zdeph test for the wireframe display. Increasing this value would draw the wireframe ontop of the rest of scene.</p><p><span style=" font-weight:600;">Line Width</span> - The thickness of the wireframe</p><p><span style=" font-weight:600;">Backface Culling </span>- The wireframe polygons facing away the camera won't be display if the value is low. </p><p><span style=" font-weight:600;">Source Backface </span>- If ticked on, the DAG mesh geometry will be set as backface culled.</p><p><span style=" font-weight:600;">Override Shading </span>- if on, the DAG mesh geometry will display its polygon surface.</p><p><br/></p><p><br/></p><p><span style=" font-weight:600;">Angle Limit</span>- While extruding a boundary edge loop, this attribute (set as degree  angle) will determine which edges to select. </p><p><span style=" font-weight:600;">Split Threshold </span>- A convenient attribute for <span style=" font-size:10pt; font-weight:600; color:#7e835a;">ziCut</span>. The resulting vertices will be merged with the existing ones depending of this distance value.<br/></p><p>Colors are linear </p><p>------------------------------------------------------------</p><p>for more informations and video tutorials, please visit</p><p><a href="www.vertexture.org"><span style=" text-decoration: underline; color:#0000ff;">www.vertexture.org</span></a></p><p align="justify"><span style=" font-weight:600;">MMB+DRAG</span> on whatever widgets to move the window.</p><p align="justify"><span style=" font-weight:600;">RMB+DRAG</span> on whatever widgets to scale up or down the window.</p><p align="justify"><br/></p></body></html>
"""

kDoublesWire = {"Surface Alpha": 'ziCutSa',
                "Line Width": 'ziCutLw',
                "BackFace Culling": 'ziCutBf',
                "Depth Priority": 'ziCutDp',

                }

kDoublesCut = {"Split Threshold": 'ziSplitThres',
               "Angle Limit": 'ziAngleLimit'
               }

kDoubles = dict(kDoublesWire)
kDoubles.update(kDoublesCut)

kColors = {"Surface Color": ['ziCutSurfCr', 'ziCutSurfCg', 'ziCutSurfCb'],
           "Line Color": ["ziCutLineCr", "ziCutLineCg", "ziCutLineCb"]
           }

kChecks = {"Override Shading": 'ziCutShading',
           "Source BackFace": 'ziCutSrcBf'
           }

__tool__ = "ziWireframe"
__author = "VERTEXTURE"

NAMEPLUGS = ["ziWireframeViewport", "ziCut"]
kIncrement = 100
DEBUG = False


class OptVar(object):

    def __init__(self):
        pass

    def load(self, obj):
        """Description
        """
        for suffix, typ in zip(["_line", "_slider"], [str, float]):

            for name in kDoubles.keys():
                widget = obj.findWidget(name, suffix)

                if not widget:
                    continue

                obj.setVar(widget, typ, name)

        for key, values in kColors.items():

            if key == "Surface Color":
                widget = obj.findWidget(key, "_color")
                obj.setColor(widget, self.surfColor)

            if key == "Line Color":
                widget = obj.findWidget(key, "_color")
                obj.setColor(widget, self.lineColor)

    def getDouble(self, value, default):
        """Description
        """
        result = default

        if cmds.optionVar(exists=value):
            result = cmds.optionVar(q=value)

        return result

    @property
    def lineWidth(self):
        return self.getDouble(kDoubles["Line Width"], 2.3)

    @property
    def surfaceAlpha(self):
        return self.getDouble(kDoubles["Surface Alpha"], 0.3)

    @property
    def backface(self):
        return self.getDouble(kDoubles["BackFace Culling"], 0.01)

    @property
    def depth(self):
        return self.getDouble(kDoubles["Depth Priority"], 2000)

    @property
    def spiltT(self):
        return self.getDouble(kDoubles["Split Threshold"], 0.02)

    @property
    def angleL(self):
        return self.getDouble(kDoubles["Angle Limit"], 150)

    @property
    def surfColor(self):
        return (self.getDouble('ziCutSurfCr', 0.14),
                self.getDouble('ziCutSurfCg', 0.14),
                self.getDouble('ziCutSurfCb', 0.17)
                )

    @property
    def lineColor(self):

        return (self.getDouble('ziCutLineCr', 0.08),
                self.getDouble('ziCutLineCg', 0.06),
                self.getDouble('ziCutLineCb', 0.17)
                )

    @property
    def meshName(self):
        varname = "ziCutMeshName"
        return cmds.optionVar(q=varname) if cmds.optionVar(ex=varname) else ""

    def clear(self):

        for var in kDoubles.items() + kChecks.items():
            cmds.optionVar(remove=var)

        for vars in kColors.items():
            for var in vars:
                cmds.optionVar(remove=var)


class Win(zi_Widget.zi_Windows.Frameless):

    def __init__(self):
        super(Win, self).__init__()

        self.var = OptVar()

        self.loadPlugin()
        self.setWinLayout()
        self.setConnections()
        self.setWin()

        self.var.load(self)
        self.show()

    def setWin(self):
        """Description
        """
        self.addBar(help)
        self.setWindowTitle("%s.properties" % __name__)

        self.setMaximumSize(400, 500)
        self.setMinimumSize(265, 410)
        self.resize(315, 410)

        if not cmds.objExists(self.var.meshName):
            return

        attr = "%s.overrideShading" % self.var.meshName
        self.findWidget("Override Shading", "_check").setChecked(cmds.getAttr(attr))

        attr = "%s.backfaceCulling" % self.var.meshName

        value = True if cmds.getAttr(attr) == 3 else False
        self.findWidget("Source BackFace", "_check").setChecked(value)

    def setConnections(self):
        """Description
        """
        for key in kDoubles.keys():
            self.findWidget(key, "_slider").valueChanged.connect(
                self.slid2Line)

        for key in kDoubles.keys():
            self.findWidget(key, "_line").editingFinished.connect(
                self.line2Slid)

        for key in kColors.keys():
            self.findWidget(key, "_color").clicked.connect(self.defineColor)

        for key in kChecks.keys():
            self.findWidget(key, "_check").clicked.connect(self.setCheck)

        self.ziUnlock.changedValue.connect(self.unlock)

    def loadPlugin(self):
        """Description
        """
        version = cmds.about(v=True)
        for plug in NAMEPLUGS:

            name = "{name}_{ver}".format(name=plug, ver=version)

            if not cmds.pluginInfo(name, q=True, loaded=True):
                try:
                    cmds.loadPlugin(name)
                except:
                    pass


    def unlock(self):
        """Description
        """
        panels = cmds.getPanel(type="modelPanel")

        for panel in panels or []:

            if self.ziUnlock.value == "ON":
                cmd = "setRendererAndOverrideInModelPanel $gViewport2 %s %s"
                eval(cmd % ("VertextureViewport", panel))

            if self.ziUnlock.value == "OFF":
                cmd = "setRendererInModelPanel $gViewport2 %s"
                eval(cmd % (panel))

    def setWidgetLine(self, name, value):

        self.findWidget(name, "_slider").setValue(value * kIncrement)
        self.findWidget(name, "_line").setText(str(value))

    def setWidgetcolor(self, name, value):
        """Description

        :Param name:
        :Type name:

        :Param value:
        :Type value:
        """
        obj = self.findWidget(name, '_color')
        self.setColor(obj, value)

        for i in xrange(3):
            cmds.optionVar(fv=(kColors[name][i], value[i]))

    def reset(self):
        """Description
        """
        self.setWidgetLine("Surface Alpha", 0.5)
        self.setWidgetLine("Line Width", 2.5)
        self.setWidgetLine("BackFace Culling", 0.1)
        self.setWidgetLine("Depth Priority", 1500.0)
        self.setWidgetLine("Split Threshold", 0.05)
        self.setWidgetLine("Angle Limit", 150)

        self.setWidgetcolor("Surface Color", [0.014, 0.014, 0.17])
        self.setWidgetcolor("Line Color", [0.01, 0.01, 0.13])

        if cmds.objExists(self.var.meshName):

            self.findWidget("Source BackFace", "_check").setChecked(True)
            cmds.setAttr("%s.backfaceCulling" % self.var.meshName, 3)

            self.findWidget("Override Shading", "_check").setChecked(False)
            cmds.setAttr("%s.overrideEnabled" % self.var.meshName, True)
            cmds.setAttr("%s.overrideShading" % self.var.meshName, False)


        cmds.refresh(cv=True, f=True)

    def setCheck(self):
        """Description
        """
        if not self.var.meshName:
            return

        checked = self.sender().isChecked()

        if "Shading" in self.sender().objectName():
            attr = "%s.overrideEnabled" % self.var.meshName
            cmds.setAttr(attr, 1)

            attr = "%s.overrideShading" % self.var.meshName
            cmds.setAttr(attr, checked)

        if "BackFace" in self.sender().objectName():
            attr = "%s.backfaceCulling" % self.var.meshName

            bfState = 3 if checked else 0
            cmds.setAttr(attr, bfState)

    def defineColor(self):
        """Description
        """
        sender = self.sender()
        colors = self.getColor(sender)
        self.setColor(sender, colors)

        name = sender.objectName().split("_")

        for i in xrange(3):
            cmds.optionVar(fv=(kColors[name[0]][i], colors[i]))

        cmds.refresh(cv=True, f=True)

    def getColor(self, widget):
        """Description
        """
        butColor = self.butColor(widget)
        color = QColorDialog().getColor(butColor)

        if not color.isValid():
            color = butColor

        r, g, b, a = color.getRgbF()
        return (r, g, b)

    def setColor(self, widget, colors):
        """Description

        :Param  colors:
        :Type  colors:
        """
        style = "background-color: rgb({red},{green},{blue})"

        widget.setStyleSheet(style.format(red=colors[0] * 255,
                                          green=colors[1] * 255,
                                          blue=colors[2] * 255)
                             )

        cmds.refresh(cv=True, f=True)

    def butColor(self, widget):
        """Description

        :Param widget:
        :Type widget:
        """
        txt = widget.styleSheet()
        values = re.search(r'rgb\((.+),(.+),(.+)\)', txt)

        color = QColor()
        color.setRgb(*map(float, values.groups()))

        return color

    def slid2Line(self):
        """Description
        """
        target = self.sender().objectName().split("_")
        value = float(self.sender().value()) / kIncrement
        self.findWidget(target[0], "_line").setText(str(value))

        cmds.optionVar(fv=(kDoubles[target[0]], value))
        cmds.refresh(cv=True, f=True)

    def line2Slid(self):
        """Description
        """
        target = self.sender().objectName().split("_")
        value = float(self.sender().text()) * kIncrement
        self.findWidget(target[0], "_slider").setValue(value)

    def createTitle(self, mainLayout, txt):
        """Description

        :Param txt:
        :Type txt:
        """
        title = QLabel(txt.title())
        title.setMinimumHeight(20)
        title.setAlignment(Qt.QtCore.Qt.AlignCenter)

        hlayout = QHBoxLayout()
        hlayout.addWidget(title)
        mainLayout.addLayout(hlayout)

    def createButton(self, mainlayout, layout, label,  func):
        """Description

        :Param layout:
        :Type layout:

        :Param label:
        :Type label:

        :Param  func:
        :Type  func:
        """
        butt = QPushButton(label.title())

        layout.addWidget(butt)
        mainlayout.addLayout(layout)

        butt.clicked.connect(func)

    def createLabel(self, txt):
        """Description

        :Param txt:
        :Type txt:
        """
        label = QLabel(txt.title())
        label.setFixedWidth(100)
        label.setAlignment(Qt.QtCore.Qt.AlignRight)
        label.setObjectName("%s_label" % txt)

        return label

    def createColor(self,  txt):
        """Description

        :Param  txt:
        :Type  txt:
        """
        color = QPushButton("")
        color.setObjectName("%s_color" % txt)
        color.setMaximumWidth(70)
        color.setFlat(True)

        return color

    def createSlider(self, txt):
        """Description

        :Param txt:
        :Type txt:
        """
        slider = QSlider(Qt.QtCore.Qt.Horizontal)
        slider.setObjectName("%s_slider" % txt)

        if txt == "Surface Alpha":
            slider.setMaximum(1.0 * kIncrement)

        if txt == "Line Width":
            slider.setMaximum(6.0 * kIncrement)

        if txt == "BackFace Culling":
            slider.setMaximum(30 * kIncrement)
            slider.setMinimum(0.01)

        if txt == "Depth Priority":
            slider.setMaximum(4000 * kIncrement)

        if txt == "Split Threshold":
            slider.setMaximum(0.4 * kIncrement)

        if txt == "Angle Limit":
            slider.setMaximum(200 * kIncrement)

        return slider

    def createCheckBox(self,  txt):
        """Description

        :Param  txt:
        :Type  txt:
        """
        box = QCheckBox("")
        box.setObjectName("%s_check" % txt)

        return box

    def createLine(self, txt):
        """Description
        """
        line = QLineEdit("-0")
        line.setMaximumWidth(50)
        line.setObjectName("%s_line" % txt)

        return line

    def setVar(self, widget, typ, name):
        """Description

        :Param  name:
        :Type  name:
        """
        func = None

        if typ == str:
            func = widget.setText

        if typ == float:
            func = widget.setValue

        if "Surface Alpha" in name:
            func(typ(self.var.surfaceAlpha * kIncrement))

        if "Line Width" in name:
            func(typ(self.var.lineWidth * kIncrement))

        if "BackFace Culling" in name:
            func(typ(self.var.backface * kIncrement))

        if "Depth Priority" in name:
            func(typ(self.var.depth * kIncrement))

        if "Split Threshold" in name:
            func(typ(self.var.spiltT * kIncrement))

        if "Angle Limit" in name:
            func(typ(self.var.angleL * kIncrement))

    def findWidget(self, name, suffix):
        """Description
        """
        reg = QRegExp(r'%s%s' % (name, suffix))
        widget = self.findChildren(QWidget, reg)

        return widget[0] or None

    def createSeparator(self, layout):
        """Description
        """
        line = QFrame()
        line.setFrameShape(QFrame.HLine)
        line.setFrameShadow(QFrame.Sunken)
        layout.addWidget(line)

    def createSplitter(self, layout):
        """Description
        """
        splitter = QSplitter()
        layout.addWidget(splitter)

    def setWinLayout(self):
        """Description
        """
        self.mainWin = self
        wid = QWidget(self.mainWin)
        mainLayout = QVBoxLayout(wid)

        self.createSeparator(mainLayout)

        self.ziUnlock = ziUnlockSlider("ON", "OFF", "VIEWPORT")
        hlayout = QHBoxLayout()
        hlayout.addWidget(self.ziUnlock)
        mainLayout.addLayout(hlayout)

        self.createSeparator(mainLayout)

        for label in kColors.keys():

            widgets = []
            hlayout = QHBoxLayout()

            widgets.append(self.createLabel(label))
            widgets.append(self.createColor(label))

            for widget in widgets:
                hlayout.addWidget(widget)
                mainLayout.addLayout(hlayout)

        self.createSeparator(mainLayout)

        title = ["Wireframe Properties", "ziCut Properties"]

        count = 0
        for double in (kDoublesWire, kDoublesCut):

            self.createTitle(mainLayout, title[count])
            count += 1

            for label in double.keys():

                widgets = []
                hlayout = QHBoxLayout()

                widgets.append(self.createLabel(label))
                widgets.append(self.createLine(label))
                widgets.append(self.createSlider(label))

                for widget in widgets:
                    hlayout.addWidget(widget)
                    mainLayout.addLayout(hlayout)

            self.createSeparator(mainLayout)

        for label in kChecks:

            widgets = []
            hlayout = QHBoxLayout()

            widgets.append(self.createLabel(label))
            widgets.append(self.createCheckBox(label))

            for widget in widgets:
                hlayout.addWidget(widget)
                mainLayout.addLayout(hlayout)

        self.createSeparator(mainLayout)

        hlayout = QHBoxLayout()
        self.createButton(mainLayout, hlayout, "reset", self.reset)

        wid.setLayout(mainLayout)
        self.mainWin.setCentralWidget(wid)


class ziUnlockSlider(QWidget, QObject):
    """Display a iphone's slider like
    """

    try:
        changedValue = pyqtSignal(str)
    except:
        changedValue = Signal(str)

    def __init__(self, text1='ON', text2='OFF', text3='', parent=None):
        QWidget.__init__(self, parent)

        self.setMouseTracking(True)
        self.cur = self.cursor()

        self.textValue = text1
        self.textL = text1
        self.textR = text2
        self.textM = text3

        self.position = QPointF()
        self.height = 20

        self.buttGrad = QLinearGradient()
        self.textGrad = QLinearGradient()
        self.backgGrad = QLinearGradient()

        self.colorR = QColor(190, 190, 190, 205)
        self.colorL = QColor(60, 60, 60, 205)
        self.poly = QPolygonF()

        self.setMinimumHeight(self.height)
        self.hover = False

    @property
    def value(self):
        return self.textValue

    def mousePressEvent(self, event):

        if event.buttons() == Qt.QtCore.Qt.LeftButton:

            # -- pixmap.contains added
            if self.buttPath.contains(event.pos()):
                self.setCursor(Qt.QtCore.Qt.PointingHandCursor)
                event.accept()

    def mouseMoveEvent(self, event):

        self.hover = False

        if event.buttons() == Qt.QtCore.Qt.LeftButton:
            self.position = event.pos()

        self.update()

    def mouseReleaseEvent(self, event):

        # -- left
        if self.position.x() > self.size().width() * .5:
            self.position = QPoint(self.size().width(), 0)
            self.colorL = QColor(190, 190, 190, 205)
            self.colorR = QColor(60, 60, 60, 205)

            self.textValue = self.textL
            self.changedValue.emit(self.textL)

        # -- right
        else:
            self.colorR = QColor(190, 190, 190, 205)
            self.colorL = QColor(60, 60, 60, 205)

            self.textValue = self.textR
            self.changedValue.emit(self.textR)

            self.position = QPoint(0, 0)

        self.setCursor(self.cur)
        self.update()

    def paintEvent(self, event):

        painter = QPainter(self)
        painter.setRenderHint(QPainter.HighQualityAntialiasing)

        buttsize = self.size().width() * .5

        self.buttPath = QPainterPath()
        circlePath = QPainterPath()
        textMPath = QPainterPath()
        textPath = QPainterPath()
        backPath = QPainterPath()

        buttpos = self.position.x() - (buttsize * .5)

        # -- clamping values, so the button does not go outside the boundaries
        if buttpos < 0:
            buttpos = 0

        if buttpos > self.size().width() - buttsize:
            buttpos = self.size().width() - buttsize

        # -- background
        backPath.addRoundedRect(
            QRectF(0, 0, self.size().width(), self.height), 3, 3)

        self.backgGrad.setColorAt(0.00, QColor(20, 20, 20, 205))
        self.backgGrad.setColorAt(0.45, QColor(25, 25, 25, 205))
        self.backgGrad.setColorAt(0.5, QColor(10, 10, 10, 205))
        self.backgGrad.setColorAt(1.0, QColor(35, 35, 35, 205))
        self.backgGrad.setStart(QPointF(0, 0))
        self.backgGrad.setFinalStop(QPointF(0, self.height))

        painter.setBrush(self.backgGrad)
        painter.setPen(QPen(QColor(20, 20, 20, 255)))
        painter.drawPath(backPath)

        # -- texts
        font = QFont('Lato', self.height * .5)
        textPath.addText(QPointF(len(self.textL) * self.size().width() * .1,
                                 self.height * .75),
                         font,
                         self.textL)

        textPath.addText(QPointF(len(self.textR) * self.size().width() * .23,
                                 self.height * .75),
                         font,
                         self.textR)

        self.textGrad.setStart(QPointF(0, 0))
        self.textGrad.setFinalStop(QPointF(buttsize * 2, 0))

        self.textGrad.setColorAt(0.48, self.colorL)
        self.textGrad.setColorAt(0.50, QColor(80, 80, 80, 255))  # -- mid
        self.textGrad.setColorAt(0.52, self.colorR)  # -- right

        painter.setBrush(self.textGrad)
        painter.setPen(QPen(self.textGrad, 0))
        painter.drawPath(textPath)

        # -- circle
        painter.setBrush(self.backgGrad)
        painter.setPen(QPen(self.backgGrad, 0))

        # -- butt
        baseColor = QColor(128, 129, 138, 255)
        self.buttGrad.setColorAt(0.00, baseColor.lighter(40))
        self.buttGrad.setColorAt(0.45, baseColor.lighter(45))
        self.buttGrad.setColorAt(0.50, baseColor.lighter(30))
        self.buttGrad.setColorAt(1.00, baseColor.lighter(55))
        self.buttGrad.setStart(QPointF(0, 0))
        self.buttGrad.setFinalStop(QPointF(0, self.height))

        self.buttPath.addRoundedRect(QRectF(0, 0, buttsize, self.height),
                                     3, 3)

        self.buttPath.translate(buttpos, 0)

        painter.setBrush(self.buttGrad)
        painter.setPen(QPen(QColor(20, 20, 20, 255)))
        painter.drawPath(self.buttPath)

        # -- if mouse over the button
        if self.hover:
            hoverGrad = QRadialGradient(QPointF(buttpos + (buttsize * .5),
                                                self.height * .5),
                                        self.height * .7)

            hoverGrad.setColorAt(.81, QColor(170, 170, 170, 255))
            hoverGrad.setColorAt(1, QColor(160, 160, 160, 255))

            painter.setBrush(hoverGrad)
            painter.setPen(QPen(hoverGrad, 1))

        else:
            painter.setBrush(self.backgGrad)
            painter.setPen(QPen(self.backgGrad, 1))

        # -- circle
        if not self.textM:
            circlePath.addEllipse(QPointF(buttpos + (buttsize * .5),
                                          self.height * .5),
                                  self.height * .4,
                                  self.height * .4)

            painter.drawPath(circlePath)

        # -- specified text
        if self.textM:
            textMPath.addText(QPointF(0, 0), font, self.textM)
            bound = textMPath.boundingRect()

            textMPath.translate(
                buttpos + (buttsize * .5) - (bound.right() * .5),
                self.height * .75)

            painter.drawPath(textMPath)


def console(*txt):
    """Description
    """
    if not DEBUG:
        return

    txt = map(str, txt)
    print("ziCutDebug{:_>20}".format(' '.join(txt)))


def main():
    global ziCutOptions

    try:
        ziCutOptions.deleteLater()
    except:
        pass

    ziCutOptions = Win()
    return ziCutOptions
