# //                          FILE
# // Associated with ziRail_<version>.mll
# //
# //                          AUTHOR
# //  stuzzi(contact@vertexture.org)
# //  www.vertexture.org
# //  Please read on the website terms of use and licensing. Tutorials can be found also
# //
# //                          DATE
# //  01/05/2020
# //
# //                          DESCRIPTION
# //      Retopo tool inspired by blender Bsurface
# //
# ////////////////////////////////////////////////////////////////////////////////////*/

from Qt.QtWidgets import *
from Qt.QtCore import *
from Qt.QtGui import *
import Qt.QtCompat

import maya.cmds as cmds
from maya.OpenMaya import *
from maya.mel import eval

import zi_UI.zi_RailUI
import zi_Widget.zi_Windows
import zi_UI.ziRessources_rc

import math

help = """
<html><head/><body><p align="center"><img src=":/vertexture/skin/vertexture/logoHs_color.png"/></p><p><br/><br/></p><p><span style=" font-size:10pt; font-weight:600; color:#7e835a;">ziRail </span>creates patch polygons along your strokes. These strokes are drawn directly on a mesh. By doing so, creating polygons become intuitiv. This can be used as a companion tool for retopology tasks. <a href="https://www.vertexture.org/?page_id=1809"><span style=" text-decoration: underline; color:#0000ff;">ziWireframe</span></a>is shipped with it.</p><p><br/></p><p><span style=" font-weight:600;">Source - </span>Specify the target mesh. The newly created polygons will be snap on it.</p><p><span style=" font-weight:600;">Uspans - </span>Specify the amount of subdivisions along the U direction of the patch. This can be done before and after the patch creation. The last button on the right will set the value as 1.</p><p><span style=" font-weight:600;">Vspans- </span>Specify the amount of subdivisions along the V direction of the patch. This can be done before and after the patch creation. The last button on the right will set the value as 1.</p><p><span style=" font-weight:600;">Bridge Subdivs</span> - When creating a bridge surface, this will set the amount of subdvision.</p><p><span style=" font-weight:600;">Display HUD</span> - So you can have a help guide in the viewport</p><p><span style=" font-weight:600;">Close Strokes</span> - While creating grid you may need to have closed strokes (ie. eyelids). </p><p><span style=" font-weight:600;">Relax Brush </span>- The brush manipulators does a basic relax task while snapping on the source mesh.</p><p><span style=" font-weight:600;">Relax Patch </span>- Relax the last created patch prior validation.</p><p><span style=" font-weight:600;">Intensity </span>- The relax brush strength</p><p><span style=" font-weight:600;">Radius </span>- The relax brush diameter</p><p><span style=" font-weight:600;">Rail Mode </span>- Start drawing the strokes.<br/></p><p>---</p><p align="center"><span style=" font-weight:600;">RAIL MODE</span></p><p><span style=" font-weight:600;">LMB + DRAG </span>- On source mesh will draw a stroke. You can repeat the process as much as need to have several strokes. by pressing <span style=" font-weight:600;">ENTER</span>, you will create patches along the strokes. The direction of the strokes has to be consistent so you have a expected result. The strokes are the guides for the resulting patches the polygons in between are interpolated. You may have a better result by reducing the distance between these strokes. You can append polygons from an existing mesh (select it) or from scratch (with an empty selection)</p><p><span style=" font-weight:600;">LMB+ CTRL + DRAG </span>- Will modify the last created patch so its fits this stroke. After pressing <span style=" font-weight:600;">ENTER, </span>you can still modify the last line of a patch. For doing so, select a boundary line (explain bellow).</p><p><span style=" font-weight:600;">CTRL + SHIFT + LMB </span>- On a boundary vertex to have an anchor point. Reproduce the same step another boundary vertex to have a boundary line. This boundary line is basically a stroke. Drawing another stroke will create a patch. </p><p><span style=" font-weight:600;">LMB + DRAG </span>- With a boundary line selected will create a patch to this stroke ( considered as a profile).</p><p><span style=" font-weight:600;">LMB + SHIFT + DRAG </span>- With a boundary line selected will create a patch along this stroke (considered as a path).</p><p><span style=" font-weight:600;">MMB </span>(with boundary line) - With a boundary line selected, MMB on a boundary vertex to prepare a polygBridge creation. </p><p><span style=" font-weight:600;">MMB + SHIFT </span>(with boundary line) - With a boundary line selected, MMB + SHIFT  on  boundary vertex to merge the loops</p><p><span style=" font-weight:600;">MMB + DRAG </span>- On a boundary vertex and another one to merge them</p><p><br/></p><p>------------------------------------------------------------</p><p>for more informations and video tutorials, please visit</p><p><a href="www.vertexture.org"><span style=" text-decoration: underline; color:#0000ff;">www.vertexture.org</span></a></p><p align="justify"><span style=" font-weight:600;">MMB+DRAG</span> on whatever widgets to move the window.</p><p align="justify"><span style=" font-weight:600;">RMB+DRAG</span> on whatever widgets to scale up or down the window.</p><p align="justify"><br/><span style=" font-size:7pt;">LMB (Left Mouse Button)</span></p><p align="justify"><span style=" font-size:7pt;">MMB (Middle Mouse Button)</span></p><p align="justify"><span style=" font-size:7pt;">RMB (Right Mouse Button)</span></p><p>&quot;&quot;&quot; </p></body></html>"""

__tool__ = "ziRail"
__author = "VERTEXTURE"

NAMEPLUGS = ["ziRail", "ziWireframeViewport"]
VERBOSE = True

from pdb import set_trace as db


class Options(object):

    attrU = 'zi_uspan'
    attrV = 'zi_vspan'
    attrB = 'zi_bdiv'
    attrD = 'zi_distancesnap'
    attrMT = 'zi_mergThreshold'

    def __init__(self):
        self._source = None

    @property
    def source(self):
        return self._source

    @source.setter
    def sourceShape(self, shape):
        self._source = shape

    @property
    def mergeT(self):
        val = 5
        if cmds.optionVar(exists=self.attrMT):
            val = cmds.optionVar(q=self.attrMT)

        return val

    @property
    def distance(self):
        val = 100
        if cmds.optionVar(exists=self.attrD):
            val = cmds.optionVar(q=self.attrD)

        return val

    @property
    def u(self):
        val = 5
        if cmds.optionVar(exists=self.attrU):
            val = cmds.optionVar(q=self.attrU)

        return val

    @property
    def v(self):
        val = 5
        if cmds.optionVar(exists=self.attrV):
            val = cmds.optionVar(q=self.attrV)

        return val

    @property
    def bdiv(self):
        val = 1
        if cmds.optionVar(exists=self.attrB):
            val = cmds.optionVar(q=self.attrB)

        return val

    @v.setter
    def v(self, value):
        cmds.optionVar(iv=[self.attrV, value])

    @u.setter
    def u(self, value):
        cmds.optionVar(iv=[self.attrU, value])

    @bdiv.setter
    def bdiv(self, value):
        cmds.optionVar(iv=[self.attrB, value])

    @mergeT.setter
    def mergeT(self, value):
        cmds.optionVar(fv=[self.attrMT, value])

    @distance.setter
    def distance(self, value):
        print "set {} to {}".format(self.attrD, value)
        cmds.optionVar(fv=[self.attrD, value])

class Mesh(object):

    def __init__(self):
        pass

    @staticmethod
    def shape(transform):
        shapes = cmds.listRelatives(transform, shapes=True, type='mesh')
        return shapes[0] if shapes else None

    @staticmethod
    def node(shape, obj):

        nodes = cmds.ls(typ=__tool__)

        if nodes and shape:
            if not cmds.listConnections('%s.ziRailMesh' % nodes[0]):
                cmds.connectAttr('%s.outMesh' %
                                 shape, '%s.ziRailMesh' % nodes[0], f=True)
            return nodes[0]

        obj.createStream()
        Mesh.node(shape, obj)


class Win(zi_Widget.zi_Windows.Frameless,
          QObject,
          zi_UI.zi_RailUI.Ui_MainWindow):

    def __init__(self):
        super(Win, self).__init__()

        self.ctx = None
        self.setupUi(self)
        self.loadPlugin()

        self.options = Options()

        # self.setHotkeys()
        self.setConnections()
        self.setIcons()
        self.setWin()
        self.restoreSession()

        self.show()

    def setWin(self):
        """Set misc preference for the QMainWindow and QWidgets
        """
        self.addBar(help)
        self.setWindowTitle("%s.properties" % __name__)

        self.logo.setPixmap(
            self.logo.pixmap()
            .scaledToWidth(90, Qt.QtCore.Qt.SmoothTransformation))

        self.setMinimumSize(200, 370)
        self.setMaximumSize(500, 500)
        self.resize(200, 295)

        self.hudChk.setHidden(True)
        self.Uspin.setValue(self.options.u)
        self.Vspin.setValue(self.options.v)
        self.mergeTSpin.setValue(self.options.mergeT)
        self.bridgeSpin.setValue(self.options.bdiv)
        self.distanceSpin.setValue(self.options.distance)


        map(lambda x: x.setVisible(False), [self.col1,
                                            self.col2,
                                            self.col3,
                                            self.col4])

    def restoreSession(self):
        """Check if a ziRail connection already exist to set the source mesh
        """
        node = self.ziRailNode()

        if node:
            self.hudDisplay()

            meshes = cmds.listConnections('%s.ziRailMesh' % node[0])
            if meshes:
                # -- saving the current selection
                prevSelection = cmds.ls(sl=True)

                cmds.select(meshes[0], replace=True)
                self.pickSrcBtn.clicked.emit()

                cmds.select(prevSelection, replace=True)

    def setIcons(self):
        """Set QPushButton::QIcon images for UI
        """
        self.viewportBtn.setIcon(QIcon(":render_layeredShader.png"))
        self.closeStrokeBtn.setIcon(QIcon(":attachCurves.png"))

        self.launch.setIcon(QIcon(":birail1Gen.png"))
        self.launch.setIcon(QIcon(":birail1Gen.png"))
        self.relaxBrBtn.setIcon(QIcon(":putty.png"))
        # self.relaxBtn.setIcon(QIcon(":Relax.png"))

    def setHotkeys(self):
        """Set hotkeys to speed up the workflow, deactivated for now.
        """
        mayWin = zi_Widget.zi_Windows.getMayaWin()

        self.plus = QShortcut(QKeySequence(Qt.QtCore.Qt.Key_Plus), mayWin)
        self.minus = QShortcut(QKeySequence(Qt.QtCore.Qt.Key_Minus), mayWin)
        self.plus.activated.connect(lambda: self.spansArrow(self.Vspin, 1))
        self.minus.activated.connect(lambda: self.spansArrow(self.Vspin, -1))

    # def closeEvent(self, event):
        """Deactivated since the class variables related to hotkeys are not initialized
        """
        # self.plus.activated.disconnect()
        # self.minus.activated.disconnect()

    def setConnections(self):
        """Set QSignals and QSlots for QWidgets
        """
        self.pickSrcBtn.clicked.connect(self.pickSource)

        self.blankVBtn.clicked.connect(lambda: self.blank(self.blankVBtn, "v"))
        self.blankUBtn.clicked.connect(lambda: self.blank(self.blankUBtn, "u"))

        self.upUBtn.clicked.connect(lambda: self.spansArrow(self.Uspin, 1))
        self.dnUBtn.clicked.connect(lambda: self.spansArrow(self.Uspin, -1))
        self.upVBtn.clicked.connect(lambda: self.spansArrow(self.Vspin, 1))
        self.dnVBtn.clicked.connect(lambda: self.spansArrow(self.Vspin, -1))

        self.shaderApplyBtn.clicked.connect(self.applyShader)

        self.distanceSpin.valueChanged.connect(self.changeDistance)
        self.mergeTSpin.valueChanged.connect(self.changeThreshold)
        self.bridgeSpin.valueChanged.connect(self.changeBridge)

        self.viewportBtn.clicked.connect(self.setViewport)

        self.closeStrokeBtn.clicked.connect(self.closeStroke)
        self.radiusSlid.valueChanged.connect(self.relaxBrush)
        self.forceSlid.valueChanged.connect(self.relaxBrush)

         # -- need to change the signal and focus
        self.Uspin.valueChanged.connect(self.spansChanged)
        self.Vspin.valueChanged.connect(self.spansChanged)

        self.relaxBrBtn.clicked.connect(self.relaxBrush)
        # self.relaxBtn.clicked.connect(self.relax)

        self.hudChk.clicked.connect(self.hudDisplay)

        self.launch.clicked.connect(self.launching)

    def spansArrow(self, func, incr):
        """Called function for changing spans u or v direction
        """
        func.setValue(func.value()+incr)


    def changeDistance(self):
        """Description
        """
        self.options.distance = float(self.distanceSpin.value())


    def changeThreshold(self):
        """Description
        """
        self.options.mergeT = float(self.mergeTSpin.value())

    def spansChanged(self, dummy):
        """Change the u or v spans sudbdivisions of the last created patch
        """
        if not self.ctx:
            return

        if self.getNode():
            vspan = int(self.Vspin.value())
            uspan = int(self.Uspin.value())

            cmds.ziRailCmd(v=vspan, u=uspan)
            cmds.ziRailContext(self.ctx, e=True, refresh=True)

    def getSelection(self):
        """Description
        """
        sels = cmds.ls(sl=True, typ="transform")

        if not sels:
            cmds.error("please select a mesh")

        shapes = cmds.listRelatives(sels[0], shapes=True)

        if not shapes:
            cmds.error("please select a valid mesh")

        if cmds.nodeType(shapes[0]) == "mesh":
            return shapes[0]

        else:
            cmds.error('cannot retrieves a valid mesh from selection')

    def applyShader(self):
        """Description
        """
        sel = self.getSelection()

        panels = cmds.getPanel(type="modelPanel")
        for panel in panels or []:

            if self.shaderApplyBtn.isChecked():

                cmd = "setRendererAndOverrideInModelPanel $gViewport2 %s %s"
                eval(cmd % ("VertextureViewport", panel))
                cmds.setAttr("%s.overrideEnabled" % sel, True)
                cmds.setAttr("%s.overrideShading" % sel, False)

        if not self.shaderApplyBtn.isChecked():
            cmd = "setRendererInModelPanel $gViewport2 %s"
            eval(cmd % (panel))

            cmds.setAttr("%s.overrideEnabled" % sel, False)
            cmds.setAttr("%s.overrideShading" % sel, True)

    def setViewport(self):
        """Description
        """
        self.loadPlugin()

        import zi_wireframe
        zi_wireframe.main()

    def closeStroke(self):
        cmds.ziRailCmd(close=True)

    def closeV(self):
        cmds.ziRailCmd(cu=True)

    def relax(self):
        cmds.ziRailCmd(relax=True)

    def relaxBrush(self):

        if self.relaxBrBtn.isChecked():

            if not self.ctx:
                self.ctx = cmds.ziRailContext()

            map(lambda x: x.setEnabled(True), [
                self.forceSlid, self.radiusSlid])

            radius = self.radiusSlid.value() * 5
            intens = self.forceSlid.value() * .0005

            cmds.ziRailContext(self.ctx,
                               e=True,
                               relaxBrush=True,
                               rad=radius,
                               intensity=intens
                               )

            cmds.ziRailContext(self.ctx, e=True, refresh=True)
            return

        self.launching()

    def blank(self, obj, span):
        """Description
        """
        func = self.disableU if span == "u" else self.disableV
        mode = False if obj.isChecked() else True
        func(mode)

        if span == 'u':
            self.options.u = 1 if obj.isChecked() else self.Uspin.value()
        if span == 'v':
            self.options.v = 1 if obj.isChecked() else self.Vspin.value()

    def disableU(self, mode):
        map(lambda x: x.setEnabled(mode), [self.upUBtn,
                                           self.dnUBtn,
                                           self.Uspin])

    def disableV(self, mode):
        map(lambda x: x.setEnabled(mode), [self.upVBtn,
                                           self.dnVBtn,
                                           self.Vspin])

    def hudDisplay(self):
        """Set weither the user needs to have HUD display as support
        """
        for node in self.ziRailNode():
            cmds.setAttr("%s.displayInfo" % node, self.hudChk.isChecked())
            console("hudDisplay set as {}".format(self.hudChk.isChecked()))

    def changeBridge(self):
        """Description
        """
        self.options.bdiv = int(self.bridgeSpin.value())

    def launching(self):
        """Launch the main context tool
        """
        name = "{name}_{ver}".format(name=NAMEPLUGS[0], ver=cmds.about(v=True))

        if not cmds.pluginInfo(name, q=True, loaded=True):
            cmds.error("Please load the plugin %s" % name)

        if not cmds.constructionHistory(q=True, tgl=True):
            cmds.error("Please activate the construction history, ziRail is a\
                custom node based tool, history is required")

        map(lambda x: x.setEnabled(False), [self.forceSlid, self.radiusSlid])
        Mesh.node(self.options.source, self)

        if self.pickSrcBtn.text() in cmds.ls(sl=True):
            cmds.error("please another mesh, it should not be the source mesh")

        self.ctx = cmds.ziRailContext()
        cmds.setToolTo(self.ctx)

        self.relaxBrBtn.setChecked(False)
        self.hudDisplay()

    def pickSource(self):
        """Specify the source mesh and make its connections
        """
        shape = str()
        sels = cmds.ls(sl=True)

        if not sels:
            cmds.error("invalid selection")

        if not cmds.nodeType(sels[0]) == 'mesh':

            shapes = cmds.listRelatives(sels[0], shapes=True, type='mesh')
            if shapes:
                shape = shapes[0]

        if not cmds.nodeType(sels[0]) == 'mesh':
            shape = shapes[0]

        if not shape:
            cmds.error("invalid selection")

        self.options.sourceShape = shape

        Mesh.node(shape, self)
        nodes = self.ziRailNode()

        if shapes:
            self.sender().setText(sels[0])
            self.options.sourceShape = shape
            console("%s set as source" % sels[0])

            if nodes:
                cmds.connectAttr('%s.outMesh' % shape,
                                 '%s.ziRailMesh' % nodes[0], f=True)

    def setupNetwork(self):
        """Create the networks connection
        """
        node = self.getNode()

        if not node:
            self.createStream()

    def createStream(self):
        """Create the connection and node if not exist
        """
        # -- saving the current selection
        sels = cmds.ls(sl=True)

        node = ''
        currentNodes = cmds.ls(typ=__tool__)

        if not currentNodes:
            node = cmds.createNode(__tool__, n='ziRailShape')
        else:
            node = currentNodes[0]

        if not cmds.nodeType(node) == __tool__:
            cmds.delete(node)
            cmds.error("please load the plugin")

        sshape = self.options.sourceShape

        if not sshape:
            cmds.error("Please specify a source mesh first")

        if not cmds.objExists(sshape):
            cmds.error("please specify a valid source")

        cmds.connectAttr('%s.outMesh' % sshape, '%s.ziRailMesh' % node, f=True)
        console("\"%s\" connected to %s" % (sshape, node))

        # -- restoring previous selection
        cmds.select(sels, replace=True)

    def ziRailNode(self):
        """Description
        """
        return cmds.ls(typ=__tool__)

    def getNode(self):
        """Get a ziRail node or create one if none
        """
        return Mesh.node(self.options.source, self)

    def loadPlugin(self):
        version = cmds.about(v=True)

        for plug in NAMEPLUGS:
            name = "{name}_{ver}".format(name=plug, ver=version)

            if not cmds.pluginInfo(name, q=True, loaded=True):

                try:
                    cmds.loadPlugin(name)
                    console("{} loaded".format(name))
                except:
                    cmds.error(
                        "Cannot load plugin %s please make sure the *.mll file is in the correct folder, a video tutorial is on the website" % name)

    @staticmethod
    def loc(pos, name=''):
        grpName = 'locs'
        if not cmds.objExists(grpName):
            cmds.createNode('transform', n=grpName)

        node = cmds.spaceLocator(n=name, p=(pos[0], pos[1], pos[2]))[0]
        cmds.setAttr('%s.localScaleX' % node, 0.1)
        cmds.setAttr('%s.localScaleY' % node, 0.1)
        cmds.setAttr('%s.localScaleZ' % node, 0.1)

        cmds.delete(node, ch=True)
        cmds.xform(node, cpc=True)
        cmds.parent(node, grpName)

    @staticmethod
    def locs(pos, name=""):
        [Win.loc(p, name) for p in pos]


def console(*txt):
    """Description
    """
    if not VERBOSE:
        return

    txt = map(str, txt)
    print("ziRailInfo {:_>20}".format(' '.join(txt)))


def main():
    global ziRailwin

    try:
        ziRailwin.deleteLater()
    except:
        pass

    ziRailwin = Win()
    return ziRailwin
