<?php

/**
 * Xfcloud - Gallery
 *
 * This file is licensed under the Affero General Public License version 3 or
 * later. See the COPYING file.
 *
 * @author Olivier Paroz <galleryapps@oparoz.com>
 *
 * @copyright Olivier Paroz 2017
 */

namespace OCA\Gallery\Controller;

use OCP\ILogger;
use OCP\AppFramework\Http;
use OCA\Gallery\Service\ConfigService;

/**
 * Trait Config
 *
 * @package OCA\Gallery\Controller
 */
trait Config {

    /**
     * @var ConfigService
     */
    private $configService;

    /**
     * @var ILogger
     */
    private $logger;

    /**
     * @NoAdminRequired
     *
     * Returns an app configuration array
     *
     * @param bool $extraMediaTypes
     *
     * @return array <string,null|array>
     */
    private function getConfig($extraMediaTypes = false) {
        $features = $this->configService->getFeaturesList();

        //$this->logger->debug("Features: {features}", ['features' => $features]);

        $nativeSvgSupport = $this->isNativeSvgActivated($features);
        $mediaTypes = $this->configService->getSupportedMediaTypes($extraMediaTypes, $nativeSvgSupport);

        return ['features' => $features,
            'mediatypes' => $mediaTypes,
           /* 'videotypes' =>$this->get_type_mime(['mp4', 'asf', 'aif', 'aac', 'avi', 'flv', 'f4v', 'mp4a', 'mp4v',
                'mpe', 'mpeg', 'mpg', 'mpg4', 'wav', 'mov', 'wm', 'mkv', 'dv', 'webm', 'ogv'
            ]),*/
            'videotypes'=>  $this->videoMime(),
            'txttypes' => $this->get_type_mime(['asc', 'csv', 'doc', 'dot', 'docx',
                'wps', 'kset', 'ksdps', 'ini', 'pdf', 'ppt', 'pptx', 'ps', 'xls', 'xlsx']),
 
            'musictypes' => $this->get_type_mime(['aac', 'aif', 'flac', 'mid', 'midi', 'mp3', 'mp4a', 'oga', 'ogg', 'wav', 'wma']),
            'othertypes' => $this->get_type_mime(['7z', 'atom', 'bz2', 'cer', 'crl', 'crt', 'css', 'csv',
                'cu', 'deb', 'eot', 'eps', 'epub', 'etx', 'gz', 'htm', 'ics', 'iso', 'jar', 'json', 'latex', 'log',
                'ogx', 'rar', 'rss', 'rtf', 'sgm', 'sgml', 'zip', 'js', 'php', 'sh', 'psd', 'apk', 'exe', 'pem',
                'swf', 'tar', 'torrent', 'ttf', 'xml', 'yaml', 'yml']
            )
          
        ];
    }

    private function videoMime() {
        return array_values( array(
            'rmvb' => 'video/vnd.rn-realvideo',            
            'm4v' => 'video/mp4',
            'mov' => 'video/quicktime',
            'mp4' => 'video/mp4',
            'ogv' => 'video/ogg',
            '3gp'=>'video/3gpp',
            'webm' => 'video/webm',  
            'wmv' => 'video/x-ms-wmv',
            'wma'=>'video/x-ms-asf',
            'all'=> 'application/octet-stream',
            'm4u'=>'video/vnd.mpegurl',
            'm4v'=>'video/x-m4v"',
            'mp4v' => 'video/mp4',
            'mpe' => 'video/mpeg',   
            'vivo'=>'video/vnd.vivo',
            'qt' => 'video/quicktime',
            'asf' => 'video/x-ms-asf',
            'avi' => 'video/x-msvideo',
            'flv' => 'video/x-flv',
            'mkv'=> 'video/x-matroska',
            'rmvb1'=>'video/x-pn-realvideo', 
            'rmvb2'=>'application/vnd.rn-realmedia-vbr',
            'rmvb3'=>'application/octet-stream'
        ));
    }

    private function get_type_mime($extname, $types=[]) {
        $mimetypes = [
            'sh' => 'text/x-shellscript',
            'php' => 'application/x-php',
            'js' => 'application/javascript',
            '7z' => 'application/x-7z-compressed',
            'aac' => 'audio/x-aac',
            'ai' => 'application/postscript',
            'aif' => 'audio/x-aiff',
            'asc' => 'text/plain',           
            'atom' => 'application/atom+xml',
            'apk' => 'application/vnd.android.package-archive',           
            'bmp' => 'image/bmp',
            'bz2' => 'application/x-bzip2',
            'cer' => 'application/pkix-cert',
            'crl' => 'application/pkix-crl',
            'crt' => 'application/x-x509-ca-cert',
            'css' => 'text/css',
            'csv' => 'text/csv',
            'cu' => 'application/cu-seeme',
            'deb' => 'application/x-debian-package',
            'doc' => 'application/msword',
            'docx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'dvi' => 'application/x-dvi',
            'eot' => 'application/vnd.ms-fontobject',
            'eps' => 'application/postscript',
            'epub' => 'application/epub+zip',
            'ext' => 'application/octet-stream',
            'etx' => 'text/x-setext',
            'flac' => 'audio/flac',        
            'gif' => 'image/gif',
            'gz' => 'application/gzip',
            'htm' => 'text/html',
            'html' => 'text/html',
            'ico' => 'image/x-icon',
            'ics' => 'text/calendar',
            'ini' => 'text/plain',
            'iso' => 'application/x-iso9660-image',
            'jar' => 'application/java-archive',
            'jpe' => 'image/jpeg',
            'jpeg' => 'image/jpeg',
            'jpg' => 'image/jpeg',
            'js' => 'text/javascript',
            'json' => 'application/json',
            'latex' => 'application/x-latex',
            'log' => 'text/plain',
            'm4a' => 'audio/mp4',
            'mid' => 'audio/midi',
            'midi' => 'audio/midi',
            'mp3' => 'audio/mpeg',
            'mp4a' => 'audio/mp4',  
            'oga' => 'audio/ogg',
            'ogg' => 'audio/ogg',
            'ogx' => 'application/ogg',
            'pbm' => 'image/x-portable-bitmap',
            'pdf' => 'application/pdf',
            'pgm' => 'image/x-portable-graymap',
            'png' => 'image/png',
            'pnm' => 'image/x-portable-anymap',
            'ppm' => 'image/x-portable-pixmap',
            'ppt' => 'application/vnd.ms-powerpoint',
            'pptx' => 'application/vnd.openxmlformats-officedocument.presentationml.presentation',
            'ps' => 'application/postscript',           
            'rar' => 'application/x-rar-compressed',
            'ras' => 'image/x-cmu-raster',
            'rss' => 'application/rss+xml',
            'rtf' => 'application/rtf',
            'sgm' => 'text/sgml',
            'sgml' => 'text/sgml',
            'svg' => 'image/svg+xml',
            'swf' => 'application/x-shockwave-flash',
            'tar' => 'application/x-tar',
            'tif' => 'image/tiff',
            'tiff' => 'image/tiff',
            'torrent' => 'application/x-bittorrent',
            'ttf' => 'application/x-font-ttf',
            'txt' => 'text/plain',
            'wav' => 'audio/x-wav',
            'webm' => 'video/webm',
            'wma' => 'audio/x-ms-wma',           
            'woff' => 'application/x-font-woff',
            'wsdl' => 'application/wsdl+xml',
            'xbm' => 'image/x-xbitmap',
            'xls' => 'application/vnd.ms-excel',
            'xlsx' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'xml' => 'application/xml',
            'xpm' => 'image/x-xpixmap',
            'xwd' => 'image/x-xwindowdump',
            'yaml' => 'text/yaml',
            'yml' => 'text/yaml',
            'zip' => 'application/zip',
            'psd' => 'image/x-photoshop',
            'pem' => 'application/x-pem-file',
            'wps' => 'application/kswps', 'kset' => 'application/kset', 'ksdps' => 'application/ksdps',
        ]; 
        $str = [];
        foreach ($extname as $name) {
            if (isset($mimetypes[$name])) {
                $str[] = $mimetypes[$name];
            }
        }
        return $str;
    }

    /**
     * Determines if the native SVG feature has been activated
     *
     * @param array $features
     *
     * @return bool
     */
    private function isNativeSvgActivated($features) {
        $nativeSvgSupport = false;
        if (!empty($features) && in_array('native_svg', $features)) {
            $nativeSvgSupport = true;
        }

        return $nativeSvgSupport;
    }

}
