import { parse } from './util'

describe('Path', () => {
  describe('parse()', () => {
    it('should parse path string', () => {
      const paths = [
        [
          'M 10 10 H 20',
          [
            ['M', 10, 10],
            ['H', 20],
          ],
        ],
        [
          'M 10 10 V 20',
          [
            ['M', 10, 10],
            ['V', 20],
          ],
        ],
        [
          'M 10 20 C 10 10 25 10 25 20 S 40 30 40 20',
          [
            ['M', 10, 20],
            ['C', 10, 10, 25, 10, 25, 20],
            ['S', 40, 30, 40, 20],
          ],
        ],
        [
          'M 20 20 Q 40 0 60 20',
          [
            ['M', 20, 20],
            ['Q', 40, 0, 60, 20],
          ],
        ],
        [
          'M 20 20 Q 40 0 60 20 T 100 20',
          [
            ['M', 20, 20],
            ['Q', 40, 0, 60, 20],
            ['T', 100, 20],
          ],
        ],
        [
          'M 30 15 A 15 15 0 0 0 15 30',
          [
            ['M', 30, 15],
            ['A', 15, 15, 0, 0, 0, 15, 30],
          ],
        ],
        ['m 10 10', [['M', 10, 10]]],
        [
          'M 10 10 m 10 10',
          [
            ['M', 10, 10],
            ['M', 20, 20],
          ],
        ],
        [
          'M 10 10 l 10 10',
          [
            ['M', 10, 10],
            ['L', 20, 20],
          ],
        ],
        [
          'M 10 10 c 0 10 10 10 10 0',
          [
            ['M', 10, 10],
            ['C', 10, 20, 20, 20, 20, 10],
          ],
        ],
        ['M 10 10 z', [['M', 10, 10], ['Z']]],
        [
          'M 10 10 20 20',
          [
            ['M', 10, 10],
            ['L', 20, 20],
          ],
        ],
        [
          'M 10 10 L 20 20 30 30',
          [
            ['M', 10, 10],
            ['L', 20, 20],
            ['L', 30, 30],
          ],
        ],
        [
          'M 10 10 C 10 20 20 20 20 10 20 0 30 0 30 10',
          [
            ['M', 10, 10],
            ['C', 10, 20, 20, 20, 20, 10],
            ['C', 20, 0, 30, 0, 30, 10],
          ],
        ],

        // edge cases
        ['L 10 10', [['L', 10, 10]]],
        ['C 10 20 20 20 20 10', [['C', 10, 20, 20, 20, 20, 10]]],
        ['Z', [['Z']]],
        ['M 10 10 Z L 20 20', [['M', 10, 10], ['Z'], ['L', 20, 20]]],
        [
          'M 10 10 Z C 10 20 20 20 20 10',
          [['M', 10, 10], ['Z'], ['C', 10, 20, 20, 20, 20, 10]],
        ],
        ['M 10 10 Z Z', [['M', 10, 10], ['Z'], ['Z']]],

        // empty string
        ['', []],

        // invalid command
        ['X 10 10', []],

        // no arguments for a command that needs them
        ['M', []],

        // too few arguments
        ['M 10', []],

        // too many arguments
        ['M 10 10 20', [['M', 10, 10]]],

        // mixing invalid and valid commands
        ['X M 10 10', [['M', 10, 10]]],

        // invalid commands interspersed with valid commands
        [
          'X m 10 10 X 20 20',
          [
            ['M', 10, 10],
            ['L', 30, 30],
          ],
        ],

        [
          'M 0. .0 A 1e1 1 0 1 0 -1.5 -1',
          [
            ['M', 0, 0],
            ['A', 1e1, 1, 0, 1, 0, -1.5, -1],
          ],
        ],
        [
          'M 14.4 29.52 a .72 .72 0 1 0 -.72 -.72 A .72 .72 0 0 0 14.4 29.52Z',
          [
            ['M', 14.4, 29.52],
            ['A', 0.72, 0.72, 0, 1, 0, 13.68, 28.8],
            ['A', 0.72, 0.72, 0, 0, 0, 14.4, 29.52],
            ['Z'],
          ],
        ],
        ['M 1.5.8', [['M', 1.5, 0.8]]],
        ['M 1.5-8', [['M', 1.5, -8]]],
      ]

      paths.forEach((item) => {
        expect(parse(item[0] as string)).toEqual(item[1] as any)
      })
    })
  })
})
