/*
 * Decompiled with CFR 0.152.
 */
package org.jackhuang.hmcl.util.platform;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.logging.Level;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.jackhuang.hmcl.util.Lang;
import org.jackhuang.hmcl.util.Logging;
import org.jackhuang.hmcl.util.StringUtils;
import org.jackhuang.hmcl.util.platform.OperatingSystem;
import org.jackhuang.hmcl.util.platform.Platform;
import org.jackhuang.hmcl.util.versioning.VersionNumber;

public final class JavaVersion {
    private final Path binary;
    private final String longVersion;
    private final Platform platform;
    private final int version;
    private static final Pattern REGEX = Pattern.compile("version \"(?<version>(.*?))\"");
    private static final Pattern VERSION = Pattern.compile("^(?<version>[0-9]+)");
    public static final int UNKNOWN = -1;
    public static final int JAVA_7 = 70;
    public static final int JAVA_8 = 80;
    public static final int JAVA_9_AND_LATER = 90;
    private static final Map<Path, JavaVersion> fromExecutableCache = new ConcurrentHashMap<Path, JavaVersion>();
    public static final JavaVersion CURRENT_JAVA;
    private static List<JavaVersion> JAVAS;
    private static final CountDownLatch LATCH;

    public JavaVersion(Path binary, String longVersion, Platform platform) {
        this.binary = binary;
        this.longVersion = longVersion;
        this.platform = platform;
        this.version = JavaVersion.parseVersion(longVersion);
    }

    public Path getBinary() {
        return this.binary;
    }

    public String getVersion() {
        return this.longVersion;
    }

    public Platform getPlatform() {
        return this.platform;
    }

    public VersionNumber getVersionNumber() {
        return VersionNumber.asVersion(this.longVersion);
    }

    public int getParsedVersion() {
        return this.version;
    }

    private static int parseVersion(String version) {
        int head;
        Matcher matcher = VERSION.matcher(version);
        if (matcher.find() && (head = Lang.parseInt(matcher.group(), -1)) > 1) {
            return 90;
        }
        if (version.contains("1.8")) {
            return 80;
        }
        if (version.contains("1.7")) {
            return 70;
        }
        return -1;
    }

    public static JavaVersion fromExecutable(Path executable) throws IOException {
        JavaVersion cachedJavaVersion = fromExecutableCache.get(executable = executable.toRealPath(new LinkOption[0]));
        if (cachedJavaVersion != null) {
            return cachedJavaVersion;
        }
        Platform platform = Platform.BIT_32;
        String version = null;
        Process process = new ProcessBuilder(executable.toString(), "-version").start();
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getErrorStream()));){
            String line;
            while ((line = reader.readLine()) != null) {
                Matcher m = REGEX.matcher(line);
                if (m.find()) {
                    version = m.group("version");
                }
                if (!line.contains("64-Bit")) continue;
                platform = Platform.BIT_64;
            }
        }
        if (version == null) {
            throw new IOException("No Java version is matched");
        }
        if (JavaVersion.parseVersion(version) == -1) {
            throw new IOException("Unrecognized Java version " + version);
        }
        JavaVersion javaVersion = new JavaVersion(executable, version, platform);
        fromExecutableCache.put(executable, javaVersion);
        return javaVersion;
    }

    private static Path getExecutable(Path javaHome) {
        if (OperatingSystem.CURRENT_OS == OperatingSystem.WINDOWS) {
            return javaHome.resolve("bin").resolve("java.exe");
        }
        return javaHome.resolve("bin").resolve("java");
    }

    public static JavaVersion fromCurrentEnvironment() {
        return CURRENT_JAVA;
    }

    public static List<JavaVersion> getJavas() throws InterruptedException {
        if (JAVAS != null) {
            return JAVAS;
        }
        LATCH.await();
        return JAVAS;
    }

    public static synchronized void initialize() {
        List<JavaVersion> javaVersions;
        if (JAVAS != null) {
            throw new IllegalStateException("JavaVersions have already been initialized.");
        }
        try (Stream<Path> stream = JavaVersion.searchPotentialJavaHomes();){
            javaVersions = JavaVersion.lookupJavas(stream);
        }
        catch (IOException e) {
            Logging.LOG.log(Level.WARNING, "Failed to search Java homes", e);
            javaVersions = new ArrayList<JavaVersion>();
        }
        if (!javaVersions.contains(CURRENT_JAVA)) {
            javaVersions.add(CURRENT_JAVA);
        }
        JAVAS = Collections.unmodifiableList(javaVersions);
        LATCH.countDown();
    }

    private static List<JavaVersion> lookupJavas(Stream<Path> javaHomes) {
        return javaHomes.filter(x$0 -> Files.isDirectory(x$0, new LinkOption[0])).map(JavaVersion::getExecutable).filter(Files::isExecutable).flatMap(executable -> {
            try {
                return Stream.of(executable.toRealPath(new LinkOption[0]));
            }
            catch (IOException e) {
                Logging.LOG.log(Level.WARNING, "Failed to lookup Java executable at " + executable, e);
                return Stream.empty();
            }
        }).distinct().flatMap(executable -> {
            if (executable.equals(CURRENT_JAVA.getBinary())) {
                return Stream.of(CURRENT_JAVA);
            }
            try {
                return Stream.of(JavaVersion.fromExecutable(executable));
            }
            catch (IOException e) {
                Logging.LOG.log(Level.WARNING, "Failed to determine Java at " + executable, e);
                return Stream.empty();
            }
        }).collect(Collectors.toList());
    }

    private static Stream<Path> searchPotentialJavaHomes() throws IOException {
        switch (OperatingSystem.CURRENT_OS) {
            case WINDOWS: {
                ArrayList<Path> locations = new ArrayList<Path>();
                locations.addAll(JavaVersion.queryJavaHomesInRegistryKey("HKEY_LOCAL_MACHINE\\SOFTWARE\\JavaSoft\\Java Runtime Environment\\"));
                locations.addAll(JavaVersion.queryJavaHomesInRegistryKey("HKEY_LOCAL_MACHINE\\SOFTWARE\\JavaSoft\\Java Development Kit\\"));
                locations.addAll(JavaVersion.queryJavaHomesInRegistryKey("HKEY_LOCAL_MACHINE\\SOFTWARE\\JavaSoft\\JRE\\"));
                locations.addAll(JavaVersion.queryJavaHomesInRegistryKey("HKEY_LOCAL_MACHINE\\SOFTWARE\\JavaSoft\\JDK\\"));
                return locations.stream();
            }
            case LINUX: {
                Path linuxJvmDir = Paths.get("/usr/lib/jvm", new String[0]);
                if (Files.isDirectory(linuxJvmDir, new LinkOption[0])) {
                    return Files.list(linuxJvmDir);
                }
                return Stream.empty();
            }
            case OSX: {
                Path osxJvmDir = Paths.get("/Library/Java/JavaVirtualMachines", new String[0]);
                if (Files.isDirectory(osxJvmDir, new LinkOption[0])) {
                    return Files.list(osxJvmDir).map(dir -> dir.resolve("Contents/Home"));
                }
                return Stream.empty();
            }
        }
        return Stream.empty();
    }

    private static List<Path> queryJavaHomesInRegistryKey(String location) throws IOException {
        ArrayList<Path> homes = new ArrayList<Path>();
        for (String java : JavaVersion.querySubFolders(location)) {
            String home;
            if (!JavaVersion.querySubFolders(java).contains(java + "\\MSI") || (home = JavaVersion.queryRegisterValue(java, "JavaHome")) == null) continue;
            try {
                homes.add(Paths.get(home, new String[0]));
            }
            catch (InvalidPathException e) {
                Logging.LOG.log(Level.WARNING, "Invalid Java path in system registry: " + home);
            }
        }
        return homes;
    }

    private static List<String> querySubFolders(String location) throws IOException {
        ArrayList<String> res = new ArrayList<String>();
        Process process = Runtime.getRuntime().exec(new String[]{"cmd", "/c", "reg", "query", location});
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));){
            String line;
            while ((line = reader.readLine()) != null) {
                if (!line.startsWith(location) || line.equals(location)) continue;
                res.add(line);
            }
        }
        return res;
    }

    private static String queryRegisterValue(String location, String name) throws IOException {
        boolean last = false;
        Process process = Runtime.getRuntime().exec(new String[]{"cmd", "/c", "reg", "query", location, "/v", name});
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));){
            String line;
            while ((line = reader.readLine()) != null) {
                String s2;
                int begins;
                if (!StringUtils.isNotBlank(line)) continue;
                if (last && line.trim().startsWith(name) && (begins = line.indexOf(name)) > 0 && (begins = (s2 = line.substring(begins + name.length())).indexOf("REG_SZ")) > 0) {
                    String string = s2.substring(begins + "REG_SZ".length()).trim();
                    return string;
                }
                if (!location.equals(line.trim())) continue;
                last = true;
            }
        }
        return null;
    }

    static {
        Path currentExecutable = JavaVersion.getExecutable(Paths.get(System.getProperty("java.home"), new String[0]));
        try {
            currentExecutable = currentExecutable.toRealPath(new LinkOption[0]);
        }
        catch (IOException e) {
            Logging.LOG.log(Level.WARNING, "Failed to resolve current Java path: " + currentExecutable, e);
        }
        CURRENT_JAVA = new JavaVersion(currentExecutable, System.getProperty("java.version"), Platform.PLATFORM);
        LATCH = new CountDownLatch(1);
    }
}

