/*
 * Decompiled with CFR 0.152.
 */
package com.sk89q.worldedit.command;

import com.google.common.base.Preconditions;
import com.google.common.collect.Multimap;
import com.google.common.io.Files;
import com.sk89q.worldedit.LocalConfiguration;
import com.sk89q.worldedit.LocalSession;
import com.sk89q.worldedit.WorldEdit;
import com.sk89q.worldedit.WorldEditException;
import com.sk89q.worldedit.command.FlattenedClipboardTransform;
import com.sk89q.worldedit.command.util.AsyncCommandBuilder;
import com.sk89q.worldedit.command.util.CommandPermissions;
import com.sk89q.worldedit.command.util.WorldEditAsyncCommandBuilder;
import com.sk89q.worldedit.entity.Player;
import com.sk89q.worldedit.extension.platform.Actor;
import com.sk89q.worldedit.extent.clipboard.BlockArrayClipboard;
import com.sk89q.worldedit.extent.clipboard.Clipboard;
import com.sk89q.worldedit.extent.clipboard.io.BuiltInClipboardFormat;
import com.sk89q.worldedit.extent.clipboard.io.ClipboardFormat;
import com.sk89q.worldedit.extent.clipboard.io.ClipboardFormats;
import com.sk89q.worldedit.extent.clipboard.io.ClipboardReader;
import com.sk89q.worldedit.extent.clipboard.io.ClipboardWriter;
import com.sk89q.worldedit.function.operation.Operations;
import com.sk89q.worldedit.math.transform.Transform;
import com.sk89q.worldedit.session.ClipboardHolder;
import com.sk89q.worldedit.util.formatting.component.CodeFormat;
import com.sk89q.worldedit.util.formatting.component.ErrorFormat;
import com.sk89q.worldedit.util.formatting.component.PaginationBox;
import com.sk89q.worldedit.util.formatting.text.Component;
import com.sk89q.worldedit.util.formatting.text.TextComponent;
import com.sk89q.worldedit.util.formatting.text.event.ClickEvent;
import com.sk89q.worldedit.util.formatting.text.event.HoverEvent;
import com.sk89q.worldedit.util.formatting.text.format.TextColor;
import com.sk89q.worldedit.util.io.Closer;
import com.sk89q.worldedit.util.io.file.FilenameException;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.regex.Pattern;
import org.enginehub.piston.annotation.Command;
import org.enginehub.piston.exception.StopExecutionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SchematicCommands {
    private static final Logger log = LoggerFactory.getLogger(SchematicCommands.class);
    private final WorldEdit worldEdit;

    public SchematicCommands(WorldEdit worldEdit) {
        Preconditions.checkNotNull(worldEdit);
        this.worldEdit = worldEdit;
    }

    @Command(name="load", desc="Load a schematic into your clipboard")
    @CommandPermissions(value={"worldedit.clipboard.load", "worldedit.schematic.load"})
    public void load(Player player, LocalSession session, String filename, String formatName) throws FilenameException {
        LocalConfiguration config = this.worldEdit.getConfiguration();
        File dir = this.worldEdit.getWorkingDirectoryFile(config.saveDir);
        File f = this.worldEdit.getSafeOpenFile(player, dir, filename, BuiltInClipboardFormat.SPONGE_SCHEMATIC.getPrimaryFileExtension(), ClipboardFormats.getFileExtensionArray());
        if (!f.exists()) {
            player.printError("Schematic " + filename + " does not exist!");
            return;
        }
        ClipboardFormat format = ClipboardFormats.findByFile(f);
        if (format == null) {
            format = ClipboardFormats.findByAlias(formatName);
        }
        if (format == null) {
            player.printError("Unknown schematic format: " + formatName);
            return;
        }
        SchematicLoadTask task = new SchematicLoadTask(player, f, format);
        AsyncCommandBuilder.wrap(task, player).registerWithSupervisor(this.worldEdit.getSupervisor(), "Loading schematic " + filename).sendMessageAfterDelay("(Please wait... loading schematic.)").onSuccess(((TextComponent)TextComponent.of((String)filename, (TextColor)TextColor.GOLD).append((Component)TextComponent.of((String)" loaded. Paste it with ", (TextColor)TextColor.LIGHT_PURPLE))).append(CodeFormat.wrap("//paste").clickEvent(ClickEvent.of((ClickEvent.Action)ClickEvent.Action.SUGGEST_COMMAND, (String)"//paste"))), session::setClipboard).onFailure("Failed to load schematic", this.worldEdit.getPlatformManager().getPlatformCommandManager().getExceptionConverter()).buildAndExec(this.worldEdit.getExecutorService());
    }

    @Command(name="save", desc="Save a schematic into your clipboard")
    @CommandPermissions(value={"worldedit.clipboard.save", "worldedit.schematic.save"})
    public void save(Player player, LocalSession session, String filename, String formatName, boolean allowOverwrite) throws WorldEditException {
        File parent;
        LocalConfiguration config = this.worldEdit.getConfiguration();
        File dir = this.worldEdit.getWorkingDirectoryFile(config.saveDir);
        ClipboardFormat format = ClipboardFormats.findByAlias(formatName);
        if (format == null) {
            player.printError("Unknown schematic format: " + formatName);
            return;
        }
        File f = this.worldEdit.getSafeSaveFile(player, dir, filename, format.getPrimaryFileExtension(), new String[0]);
        boolean overwrite = f.exists();
        if (overwrite) {
            if (!player.hasPermission("worldedit.schematic.delete")) {
                throw new StopExecutionException((Component)TextComponent.of((String)"That schematic already exists!"));
            }
            if (!allowOverwrite) {
                player.printError("That schematic already exists. Use the -f flag to overwrite it.");
                return;
            }
        }
        if ((parent = f.getParentFile()) != null && !parent.exists() && !parent.mkdirs()) {
            throw new StopExecutionException((Component)TextComponent.of((String)"Could not create folder for schematics!"));
        }
        ClipboardHolder holder = session.getClipboard();
        SchematicSaveTask task = new SchematicSaveTask(player, f, format, holder, overwrite);
        AsyncCommandBuilder.wrap(task, player).registerWithSupervisor(this.worldEdit.getSupervisor(), "Saving schematic " + filename).sendMessageAfterDelay("(Please wait... saving schematic.)").onSuccess(filename + " saved" + (overwrite ? " (overwriting previous file)." : "."), null).onFailure("Failed to load schematic", this.worldEdit.getPlatformManager().getPlatformCommandManager().getExceptionConverter()).buildAndExec(this.worldEdit.getExecutorService());
    }

    @Command(name="delete", aliases={"d"}, desc="Delete a saved schematic")
    @CommandPermissions(value={"worldedit.schematic.delete"})
    public void delete(Actor actor, String filename) throws WorldEditException {
        File dir;
        LocalConfiguration config = this.worldEdit.getConfiguration();
        File f = this.worldEdit.getSafeOpenFile(actor instanceof Player ? (Player)actor : null, dir = this.worldEdit.getWorkingDirectoryFile(config.saveDir), filename, "schematic", ClipboardFormats.getFileExtensionArray());
        if (!f.exists()) {
            actor.printError("Schematic " + filename + " does not exist!");
            return;
        }
        if (!f.delete()) {
            actor.printError("Deletion of " + filename + " failed! Maybe it is read-only.");
            return;
        }
        actor.print(filename + " has been deleted.");
        try {
            log.info(actor.getName() + " deleted " + f.getCanonicalPath());
        }
        catch (IOException e) {
            log.info(actor.getName() + " deleted " + f.getAbsolutePath());
        }
    }

    @Command(name="formats", aliases={"listformats", "f"}, desc="List available formats")
    @CommandPermissions(value={"worldedit.schematic.formats"})
    public void formats(Actor actor) {
        actor.print("Available clipboard formats (Name: Lookup names)");
        boolean first = true;
        for (ClipboardFormat format : ClipboardFormats.getAll()) {
            StringBuilder builder = new StringBuilder();
            builder.append(format.getName()).append(": ");
            for (String lookupName : format.getAliases()) {
                if (!first) {
                    builder.append(", ");
                }
                builder.append(lookupName);
                first = false;
            }
            first = true;
            actor.print(builder.toString());
        }
    }

    @Command(name="list", aliases={"all", "ls"}, desc="List saved schematics", descFooter="Note: Format is not fully verified until loading.")
    @CommandPermissions(value={"worldedit.schematic.list"})
    public void list(Actor actor, int page, boolean oldFirst, boolean newFirst) {
        int sortType;
        if (oldFirst && newFirst) {
            throw new StopExecutionException((Component)TextComponent.of((String)"Cannot sort by oldest and newest."));
        }
        String saveDir = this.worldEdit.getConfiguration().saveDir;
        int n = oldFirst ? -1 : (sortType = newFirst ? 1 : 0);
        String pageCommand = actor.isPlayer() ? "//schem list -p %page%" + (sortType == -1 ? " -d" : (sortType == 1 ? " -n" : "")) : null;
        WorldEditAsyncCommandBuilder.createAndSendMessage(actor, new SchematicListTask(saveDir, sortType, page, pageCommand), "(Please wait... gathering schematic list.)");
    }

    private static List<File> allFiles(File root) {
        File[] files = root.listFiles();
        if (files == null) {
            return null;
        }
        ArrayList<File> fileList = new ArrayList<File>();
        for (File f : files) {
            if (f.isDirectory()) {
                List<File> subFiles = SchematicCommands.allFiles(f);
                if (subFiles == null) continue;
                fileList.addAll(subFiles);
                continue;
            }
            fileList.add(f);
        }
        return fileList;
    }

    private static class SchematicPaginationBox
    extends PaginationBox {
        private final String prefix;
        private final File[] files;

        SchematicPaginationBox(String rootDir, File[] files, String pageCommand) {
            super("Available schematics", pageCommand);
            this.prefix = rootDir == null ? "" : rootDir;
            this.files = files;
        }

        @Override
        public Component getComponent(int number) {
            Preconditions.checkArgument(number < this.files.length && number >= 0);
            File file = this.files[number];
            Multimap<String, ClipboardFormat> exts = ClipboardFormats.getFileExtensionMap();
            String format = exts.get(Files.getFileExtension(file.getName())).stream().findFirst().map(ClipboardFormat::getName).orElse("Unknown");
            boolean inRoot = file.getParentFile().getName().equals(this.prefix);
            String path = inRoot ? file.getName() : file.getPath().split(Pattern.quote(this.prefix + File.separator))[1];
            return ((TextComponent.Builder)((TextComponent.Builder)((TextComponent.Builder)TextComponent.builder().content("").append(((TextComponent)((TextComponent)TextComponent.of((String)"[L]").color(TextColor.GOLD)).clickEvent(ClickEvent.of((ClickEvent.Action)ClickEvent.Action.RUN_COMMAND, (String)("/schem load \"" + path + "\"")))).hoverEvent(HoverEvent.of((HoverEvent.Action)HoverEvent.Action.SHOW_TEXT, (Component)TextComponent.of((String)"Click to load"))))).append((Component)TextComponent.space())).append(((TextComponent)TextComponent.of((String)path).color(TextColor.DARK_GREEN)).hoverEvent(HoverEvent.of((HoverEvent.Action)HoverEvent.Action.SHOW_TEXT, (Component)TextComponent.of((String)format))))).build();
        }

        @Override
        public int getComponentsSize() {
            return this.files.length;
        }
    }

    private static class SchematicListTask
    implements Callable<Component> {
        private final String prefix;
        private final int sortType;
        private final int page;
        private final File rootDir;
        private final String pageCommand;

        SchematicListTask(String prefix, int sortType, int page, String pageCommand) {
            this.prefix = prefix;
            this.sortType = sortType;
            this.page = page;
            this.rootDir = WorldEdit.getInstance().getWorkingDirectoryFile(prefix);
            this.pageCommand = pageCommand;
        }

        @Override
        public Component call() throws Exception {
            List fileList = SchematicCommands.allFiles(this.rootDir);
            if (fileList == null || fileList.isEmpty()) {
                return ErrorFormat.wrap("No schematics found.");
            }
            File[] files = new File[fileList.size()];
            fileList.toArray(files);
            Arrays.sort(files, (f1, f2) -> {
                int res;
                if (this.sortType == 0) {
                    int p = f1.getParent().compareTo(f2.getParent());
                    res = p == 0 ? f1.getName().compareTo(f2.getName()) : p;
                } else {
                    res = Long.compare(f1.lastModified(), f2.lastModified());
                    if (this.sortType == 1) {
                        res = -res;
                    }
                }
                return res;
            });
            SchematicPaginationBox paginationBox = new SchematicPaginationBox(this.prefix, files, this.pageCommand);
            return paginationBox.create(this.page);
        }
    }

    private static class SchematicSaveTask
    implements Callable<Void> {
        private final Player player;
        private final File file;
        private final ClipboardFormat format;
        private final ClipboardHolder holder;
        private final boolean overwrite;

        SchematicSaveTask(Player player, File file, ClipboardFormat format, ClipboardHolder holder, boolean overwrite) {
            this.player = player;
            this.file = file;
            this.format = format;
            this.holder = holder;
            this.overwrite = overwrite;
        }

        @Override
        public Void call() throws Exception {
            Clipboard target;
            Clipboard clipboard = this.holder.getClipboard();
            Transform transform = this.holder.getTransform();
            if (transform.isIdentity()) {
                target = clipboard;
            } else {
                FlattenedClipboardTransform result = FlattenedClipboardTransform.transform(clipboard, transform);
                target = new BlockArrayClipboard(result.getTransformedRegion());
                target.setOrigin(clipboard.getOrigin());
                Operations.completeLegacy(result.copyTo(target));
            }
            try (Closer closer = Closer.create();){
                FileOutputStream fos = closer.register(new FileOutputStream(this.file));
                BufferedOutputStream bos = closer.register(new BufferedOutputStream(fos));
                ClipboardWriter writer = closer.register(this.format.getWriter(bos));
                writer.write(target);
                log.info(this.player.getName() + " saved " + this.file.getCanonicalPath() + (this.overwrite ? " (overwriting previous file)" : ""));
            }
            return null;
        }
    }

    private static class SchematicLoadTask
    implements Callable<ClipboardHolder> {
        private final Player player;
        private final File file;
        private final ClipboardFormat format;

        SchematicLoadTask(Player player, File file, ClipboardFormat format) {
            this.player = player;
            this.file = file;
            this.format = format;
        }

        @Override
        public ClipboardHolder call() throws Exception {
            try (Closer closer = Closer.create();){
                FileInputStream fis = closer.register(new FileInputStream(this.file));
                BufferedInputStream bis = closer.register(new BufferedInputStream(fis));
                ClipboardReader reader = closer.register(this.format.getReader(bis));
                Clipboard clipboard = reader.read();
                log.info(this.player.getName() + " loaded " + this.file.getCanonicalPath());
                ClipboardHolder clipboardHolder = new ClipboardHolder(clipboard);
                return clipboardHolder;
            }
        }
    }
}

