/// <reference path="../../shared/global.d.ts" />

/**
 * An assignment.
 */
declare class Assignment {
  /**
   * A collection of assigned stories.
   */
  readonly assignedStories: AssignedStories

  /**
   * The status of the assignment file.
   */
  readonly assignmentFileStatus: AssignmentStatus

  /**
   * The path to the document that the hyperlink destination points to.
   */
  readonly documentPath: File

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The content to export in the assignment.
   */
  exportOptions: AssignmentExportOptions

  /**
   * The file path (colon delimited on the Mac OS).
   */
  readonly filePath: string | File

  /**
   * The color of the assignment's frames.
   */
  frameColor: [number, number, number] | UIColors | NothingEnum

  /**
   * The unique ID of the Assignment.
   */
  readonly id: number

  /**
   * If true, includes linked files when packaging the assignment.
   */
  includeLinksWhenPackage: boolean

  /**
   * The index of the Assignment within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The name of the Assignment.
   */
  name: string

  /**
   * If true, the assignment package is up to date.
   */
  readonly packageUpToDate: boolean

  /**
   * If true, the assignment is packaged.
   */
  readonly packaged: boolean

  /**
   * The parent of the Assignment (a Document).
   */
  readonly parent: Document

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * The user name to assign to tracked changes and notes.
   */
  userName: string

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Cancels the package for an assignment.
   */
  cancelPackage(): void

  /**
   * Creates an assignment package.
   * @param filePath The full path name of the assignment package file.
   * @param submit If true, submits assigned stories before packaging the assignment.
   * @param withProperties Initial values for properties of the new Assignment
   */
  createPackage(filePath: File, submit?: boolean, withProperties?: object): File

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): Assignment[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Deletes the assignment and its file.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the Assignment.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string

  /**
   * Updates the assignment file.
   * @param versionComments The comment for this version.
   * @param forceSave If true, forcibly saves a version.
   */
  update(versionComments: string, forceSave?: boolean): void
}

/**
 * A collection of assignments.
 */
declare class Assignments {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Assignment with the specified index.
   * @param index The index.
   */
  [index: number]: Assignment

  /**
   * Creates a new assignment.
   * @param filePath The full path name of the new assignment.
   * @param versionComments The comment for this version.
   * @param forceSave If true, forcibly saves a version.
   * @param withProperties Initial values for properties of the new Assignment
   */
  add(
    filePath: File,
    versionComments: string,
    forceSave?: boolean,
    withProperties?: object,
  ): Assignment

  /**
   * Returns any Assignment in the collection.
   */
  anyItem(): Assignment

  /**
   * Displays the number of elements in the Assignment.
   */
  count(): number

  /**
   * Returns every Assignment in the collection.
   */
  everyItem(): Assignment

  /**
   * Returns the first Assignment in the collection.
   */
  firstItem(): Assignment

  /**
   * Returns the Assignment with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Assignment

  /**
   * Returns the Assignment with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Assignment

  /**
   * Returns the Assignment with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Assignment

  /**
   * Returns the Assignments within the specified range.
   * @param from The Assignment, index, or name at the beginning of the range.
   * @param to The Assignment, index, or name at the end of the range.
   */
  itemByRange(from: Assignment | number | string, to: Assignment | number | string): Assignment[]

  /**
   * Returns the last Assignment in the collection.
   */
  lastItem(): Assignment

  /**
   * Returns the middle Assignment in the collection.
   */
  middleItem(): Assignment

  /**
   * Returns the Assignment whose index follows the specified Assignment in the collection.
   * @param obj The Assignment whose index comes before the desired Assignment.
   */
  nextItem(obj: Assignment): Assignment

  /**
   * Returns the Assignment with the index previous to the specified index.
   * @param obj The index of the Assignment that follows the desired Assignment.
   */
  previousItem(obj: Assignment): Assignment

  /**
   * Generates a string which, if executed, will return the Assignment.
   */
  toSource(): string
}

/**
 * An assigned story.
 */
declare class AssignedStory {
  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The file path (colon delimited on the Mac OS).
   */
  readonly filePath: string | File

  /**
   * The unique ID of the AssignedStory.
   */
  readonly id: number

  /**
   * The index of the AssignedStory within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The name of the AssignedStory.
   */
  name: string

  /**
   * The parent of the AssignedStory (a Assignment).
   */
  readonly parent: Assignment

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * A reference to the assigned story.
   */
  readonly storyReference: Story | PageItem | Oval | Rectangle | Polygon

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): AssignedStory[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Moves the assigned story to the specified location.
   * @param to The location of the assigned story relative to the reference object or within the containing object.
   * @param reference The reference object. Note: Required when the to parameter specifies before or after.
   */
  move(to?: LocationOptions, reference?: Assignment | AssignedStory): AssignedStory

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the AssignedStory.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of assigned stories.
 */
declare class AssignedStories {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the AssignedStory with the specified index.
   * @param index The index.
   */
  [index: number]: AssignedStory

  /**
   * Returns any AssignedStory in the collection.
   */
  anyItem(): AssignedStory

  /**
   * Displays the number of elements in the AssignedStory.
   */
  count(): number

  /**
   * Returns every AssignedStory in the collection.
   */
  everyItem(): AssignedStory

  /**
   * Returns the first AssignedStory in the collection.
   */
  firstItem(): AssignedStory

  /**
   * Returns the AssignedStory with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): AssignedStory

  /**
   * Returns the AssignedStory with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): AssignedStory

  /**
   * Returns the AssignedStory with the specified name.
   * @param name The name.
   */
  itemByName(name: string): AssignedStory

  /**
   * Returns the AssignedStories within the specified range.
   * @param from The AssignedStory, index, or name at the beginning of the range.
   * @param to The AssignedStory, index, or name at the end of the range.
   */
  itemByRange(
    from: AssignedStory | number | string,
    to: AssignedStory | number | string,
  ): AssignedStory[]

  /**
   * Returns the last AssignedStory in the collection.
   */
  lastItem(): AssignedStory

  /**
   * Returns the middle AssignedStory in the collection.
   */
  middleItem(): AssignedStory

  /**
   * Returns the AssignedStory whose index follows the specified AssignedStory in the collection.
   * @param obj The AssignedStory whose index comes before the desired AssignedStory.
   */
  nextItem(obj: AssignedStory): AssignedStory

  /**
   * Returns the AssignedStory with the index previous to the specified index.
   * @param obj The index of the AssignedStory that follows the desired AssignedStory.
   */
  previousItem(obj: AssignedStory): AssignedStory

  /**
   * Generates a string which, if executed, will return the AssignedStory.
   */
  toSource(): string
}

/**
 * A print event
 */
declare class PrintEvent extends Event {
  /**
   * Dispatched after a PrintEvent is printed. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_PRINT: string

  /**
   * Dispatched before a PrintEvent is printed. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_PRINT: string

  /**
   * UI options for print document
   */
  documentPrintUiOption: DocumentPrintUiOptions[]
}

/**
 * Represents the place gun.
 */
declare class PlaceGun extends Preference {
  /**
   * A collection of buttons.
   */
  readonly buttons: Buttons

  /**
   * A collection of checkboxes.
   */
  readonly checkBoxes: CheckBoxes

  /**
   * A collection of comboboxes.
   */
  readonly comboBoxes: ComboBoxes

  /**
   * A collection of EPS files.
   */
  readonly epss: EPSs

  /**
   * EPSTexts
   */
  readonly epstexts: EPSTexts

  /**
   * A collection of form fields.
   */
  readonly formFields: FormFields

  /**
   * A collection of graphic lines.
   */
  readonly graphicLines: GraphicLines

  /**
   * A collection of imported graphics in any graphic file format (vector, metafile, or bitmap).
   */
  readonly graphics: Graphics

  /**
   * A collection of groups.
   */
  readonly groups: Groups

  /**
   * A collection of bitmap images in any bitmap file format (including TIFF, JPEG, or GIF).
   */
  readonly images: Images

  /**
   * Imported InDesign pages.
   */
  readonly importedPages: ImportedPages

  /**
   * A collection of listboxes.
   */
  readonly listBoxes: ListBoxes

  /**
   * Whether the place gun is currently loaded with content for placing.
   */
  readonly loaded: boolean

  /**
   * A collection of multi-state objects.
   */
  readonly multiStateObjects: MultiStateObjects

  /**
   * A collection of ellipses.
   */
  readonly ovals: Ovals

  /**
   * The page items collection, which can be used to process all page items in a container (such as a document, page, or group), regardless of type.
   */
  readonly pageItems: PageItems

  /**
   * A collection of PDF files.
   */
  readonly pdfs: PDFs

  /**
   * A collection of PICT graphics.
   */
  readonly picts: PICTs

  /**
   * A collection of polygons.
   */
  readonly polygons: Polygons

  /**
   * A collection of radio buttons.
   */
  readonly radioButtons: RadioButtons

  /**
   * A collection of rectangles.
   */
  readonly rectangles: Rectangles

  /**
   * A collection of signature fields.
   */
  readonly signatureFields: SignatureFields

  /**
   * A collection of IDML snippets.
   */
  readonly snippets: Snippets

  /**
   * The spline items collection.
   */
  readonly splineItems: SplineItems

  /**
   * A collection of text boxes.
   */
  readonly textBoxes: TextBoxes

  /**
   * A collection of text frames.
   */
  readonly textFrames: TextFrames

  /**
   * A collection of WMF graphics.
   */
  readonly wmfs: WMFs

  /**
   * Delete the contents of the place gun.
   */
  abortPlaceGun(): void

  /**
   * Load the place gun with one or more files.
   * @param fileName One or more files to place.
   * @param showingOptions Whether to display the import options dialog
   * @param withProperties Initial values for properties of the placed object(s)
   */
  loadPlaceGun(fileName: File | File[], showingOptions?: boolean, withProperties?: object): void

  /**
   * Rotate the contents of the place gun.
   * @param direction Which direction to rotate the contents
   */
  rotate(direction?: RotationDirection): void
}

/**
 * An event listener.
 */
declare class EventListener {
  /**
   * The name of the event.
   */
  readonly eventType: string

  /**
   * The handler to invoke when the event occurs.
   */
  readonly handler: File | Function

  /**
   * The unique ID of the EventListener.
   */
  readonly id: number

  /**
   * The index of the EventListener within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The name of the EventListener; this is an alias to the EventListener's label property.
   */
  name: string

  /**
   * The parent of the EventListener (a Object, UIDBasedObject, CellStyleGroup, CellStyle, TableStyleGroup, Article, ConditionSet, HiddenText, Condition, MotionPreset, AssignedStory, Assignment, ObjectStyleGroup, ObjectStyle, NumberingList, Snippet, Dialog, ColorGroup, Swatch, Color, Tint, Gradient, MixedInkGroup, MixedInk, Behavior, SubmitFormBehavior, PrintFormBehavior, ClearFormBehavior, GotoPageBehavior, GotoAnchorBehavior, SoundBehavior, ViewZoomBehavior, GotoStateBehavior, GotoPreviousStateBehavior, GotoNextStateBehavior, OpenFileBehavior, AnimationBehavior, ShowHideFieldsBehavior, MovieBehavior, GotoURLBehavior, GotoPreviousViewBehavior, GotoNextViewBehavior, GotoPreviousPageBehavior, GotoNextPageBehavior, GotoLastPageBehavior, GotoFirstPageBehavior, PreflightProfileRule, PreflightRuleInstance, PreflightProfile, CrossReferenceFormat, HyperlinkURLDestination, HyperlinkExternalPageDestination, HyperlinkPageDestination, HyperlinkTextDestination, ParagraphDestination, HyperlinkTextSource, CrossReferenceSource, HyperlinkPageItemSource, Bookmark, Hyperlink, IndexSection, PageReference, CrossReference, Index, TOCStyle, FlattenerPreset, BookContent, MenuAction, ScriptMenuAction, NamedGrid, CompositeFontEntry, CompositeFont, CharacterStyleGroup, ParagraphStyleGroup, TextVariableInstance, Footnote, XMLRuleProcessor, XMLTag, Note, TableStyle, TextPath, Asset, Link, Section, MojikumiTable, KinsokuTable, Guide, LanguageWithVendors, Language, PageItem, HtmlItem, FormField, SignatureField, TextBox, RadioButton, ListBox, ComboBox, CheckBox, MultiStateObject, Button, MediaItem, Sound, Movie, EPSText, SplineItem, Polygon, GraphicLine, Rectangle, Oval, Graphic, ImportedPage, PICT, WMF, PDF, EPS, Image, Group, TextFrame, EndnoteTextFrame, EndnoteRange, Endnote, MasterSpread, TrapPreset, Ink, DocumentPreset, Page, Spread, Layer, CharacterStyle, ParagraphStyle, Story, XmlStory, IDBasedObject, ArticleChild, ArticleMember, DialogRow, DialogColumn, Widget, BorderPanel, MeasurementEditbox, MeasurementCombobox, RealCombobox, AngleEditbox, PercentEditbox, RealEditbox, PercentCombobox, AngleCombobox, EnablingGroup, IntegerCombobox, IntegerEditbox, RadiobuttonControl, RadiobuttonGroup, CheckboxControl, Dropdown, StaticText, TextEditbox, ColorGroupSwatch, BackgroundTask, State, RuleDataObject, XMLItem, DTD, XMLInstruction, XMLComment, XMLElement, Table, Cell, IdleTask, StrokeStyle, StripedStrokeStyle, DottedStrokeStyle, DashedStrokeStyle, GraphicLayer, NonIDBasedObject, TableStyleMapping, CharStyleMapping, CellStyleMapping, ParaStyleMapping, TimingTarget, TimingGroup, TimingList, OpacityGradientStop, ObjectStyleExportTagMap, DataMergeQrcodePlaceholder, DataMergeField, DataMergeImagePlaceholder, DataMergeTextPlaceholder, Panel, LibraryPanel, PagesPanel, Window, StoryWindow, LayoutWindow, NavigationPoint, PreflightProcess, PreflightRule, BuildingBlock, DisplaySetting, IndexingSortOption, Topic, TOCStyleEntry, MenuElement, MenuSeparator, MenuItem, Submenu, Menu, StyleExportTagMap, TextVariable, XMLRuleMatchData, ValidationError, XMLExportMap, XMLImportMap, XMLAttribute, PrinterPreset, Row, Column, Change, HttpLinkConnectionManagerObject, RootObject, Document, Application, Book, Library, Preference, ContentPlacerObject, LinkedPageItemOption, LinkedStoryOption, PublishExportPreference, HTMLFXLExportPreference, EPubExportPreviewAppPreference, EPubFixedLayoutExportPreference, HTMLExportPreference, EPubExportPreference, ConditionalTextPreference, TimingSetting, AnimationSetting, XFLExportPreference, SWFExportPreference, TransformAttributeOption, AlignDistributePreference, TypeContextualUiPreference, GrabberPreference, ObjectStyleContentEffectsCategorySettings, ObjectStyleFillEffectsCategorySettings, ObjectStyleStrokeEffectsCategorySettings, ObjectStyleObjectEffectsCategorySettings, ChapterNumberPreference, NumberingRestartPolicy, Bullet, DataMerge, DataMergeOption, DataMergePreference, JPEGExportPreference, TrackChangesPreference, NotePreference, TransformPreference, ClipboardPreference, GeneralPreference, WatermarkPreference, ButtonPreference, PNGExportPreference, PreflightBookOption, PreflightOption, XMLViewPreference, GpuPerformancePreference, DisplayPerformancePreference, IndexOptions, LinkMetadata, MetadataPreference, ExcelImportPreference, TaggedTextImportPreference, TaggedTextExportPreference, WordRTFImportPreference, TextExportPreference, TextImportPreference, FindChangeContentTransparencySetting, FindChangeFillTransparencySetting, FindChangeStrokeTransparencySetting, FindChangeTransparencySetting, GradientFeatherSetting, FindChangeGradientFeatherSetting, DirectionalFeatherSetting, FindChangeDirectionalFeatherSetting, ContentTransparencySetting, SatinSetting, FindChangeSatinSetting, BevelAndEmbossSetting, FindChangeBevelAndEmbossSetting, InnerGlowSetting, FindChangeInnerGlowSetting, OuterGlowSetting, FindChangeOuterGlowSetting, InnerShadowSetting, FindChangeInnerShadowSetting, FeatherSetting, FindChangeFeatherSetting, DropShadowSetting, FindChangeDropShadowSetting, BlendingSetting, FindChangeBlendingSetting, FillTransparencySetting, StrokeTransparencySetting, TransparencySetting, TransparencyPreference, FlattenerPreference, GalleyPreference, GridPrintingPreference, CjkGridPreference, StoryGridDataInformation, LayoutGridDataInformation, GridDataInformation, CaptionMetadataVariablePreference, CustomTextVariablePreference, MatchParagraphStylePreference, MatchCharacterStylePreference, FileNameVariablePreference, DateVariablePreference, ChapterNumberVariablePreference, PageNumberVariablePreference, FootnoteOption, BaselineFrameGridOption, AnchoredObjectSetting, AnchoredObjectDefault, ExportForWebPreference, XMLPreference, XMLExportPreference, XMLImportPreference, InCopyExportOption, LinkingPreference, ChangeTransliteratePreference, ChangeObjectPreference, ChangeGlyphPreference, ChangeGrepPreference, ChangeTextPreference, FindTransliteratePreference, FindObjectPreference, FindGlyphPreference, FindGrepPreference, FindTextPreference, FindChangeTransliterateOption, FindChangeObjectOption, FindChangeGlyphOption, FindChangeGrepOption, FindChangeTextOption, ColorSetting, ScriptArg, ScriptPreference, PlaceGun, LayoutAdjustmentPreference, StrokeFillProxySetting, ImportedPageAttribute, EPSImportPreference, SmartGuidePreference, AutoCorrectPreference, SpellPreference, PolygonPreference, DictionaryPreference, FontLockingPreference, MojikumiUiPreference, ContourOption, TextWrapPreference, TextEditingPreference, FrameFittingOption, ObjectExportOption, PageItemDefault, EndnoteOption, TextFrameFootnoteOptionsObject, TaggedPDFPreference, InteractivePDFExportPreference, PDFAttribute, PDFPlacePreference, PDFExportPreference, IMEPreference, GraphicLayerOption, ClippingPathSettings, ImageIOPreference, ImagePreference, ToolBox, EPSExportPreference, PrintBookletPrintPreference, PrintBookletOption, PrintPreference, ViewPreference, PasteboardPreference, MarginPreference, GuidePreference, GridPreference, DocumentPreference, TextDefault, TextPreference, TextFramePreference, StoryPreference, PathPoint, Path, GradientStop, AutoCorrectTable, UserDictionary, HyphenationException, Font, TransformationMatrix, PDFExportPreset, NestedStyle, TabStop, Text, InsertionPoint, TextStyleRange, Paragraph, TextColumn, Line, Word, Character, NestedGrepStyle or NestedLineStyle).
   */
  readonly parent: any

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): EventListener[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Deletes the EventListener.
   */
  remove(): void

  /**
   * Generates a string which, if executed, will return the EventListener.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of event listeners.
 */
declare class EventListeners {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the EventListener with the specified index.
   * @param index The index.
   */
  [index: number]: EventListener

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   * @param withProperties Initial values for properties of the new EventListener
   */
  add(
    eventType: string,
    handler: File | Function,
    captures?: boolean,
    withProperties?: object,
  ): EventListener

  /**
   * Returns any EventListener in the collection.
   */
  anyItem(): EventListener

  /**
   * Displays the number of elements in the EventListener.
   */
  count(): number

  /**
   * Returns every EventListener in the collection.
   */
  everyItem(): EventListener

  /**
   * Returns the first EventListener in the collection.
   */
  firstItem(): EventListener

  /**
   * Returns the EventListener with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): EventListener

  /**
   * Returns the EventListener with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): EventListener

  /**
   * Returns the EventListener with the specified name.
   * @param name The name.
   */
  itemByName(name: string): EventListener

  /**
   * Returns the EventListeners within the specified range.
   * @param from The EventListener, index, or name at the beginning of the range.
   * @param to The EventListener, index, or name at the end of the range.
   */
  itemByRange(
    from: EventListener | number | string,
    to: EventListener | number | string,
  ): EventListener[]

  /**
   * Returns the last EventListener in the collection.
   */
  lastItem(): EventListener

  /**
   * Returns the middle EventListener in the collection.
   */
  middleItem(): EventListener

  /**
   * Returns the EventListener whose index follows the specified EventListener in the collection.
   * @param obj The EventListener whose index comes before the desired EventListener.
   */
  nextItem(obj: EventListener): EventListener

  /**
   * Returns the EventListener with the index previous to the specified index.
   * @param obj The index of the EventListener that follows the desired EventListener.
   */
  previousItem(obj: EventListener): EventListener

  /**
   * Generates a string which, if executed, will return the EventListener.
   */
  toSource(): string
}

/**
 * An event.
 */
declare class Event {
  /**
   * Dispatched after the Event becomes active. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_ACTIVATE: string

  /**
   * Dispatched when a Event is closing. Since the close has been committed, it can no longer be canceled. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_CLOSE: string

  /**
   * Dispatched after the active context changes. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_CONTEXT_CHANGED: string

  /**
   * Dispatched after a Event is deleted. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_DELETE: string

  /**
   * Dispatched after a Event is embedded. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_EMBED: string

  /**
   * Dispatched after the Event is invoked. This event does not bubble. This event is not cancelable.
   */
  static readonly AFTER_INVOKE: string

  /**
   * Dispatched after one or more links in the Event have been added, deleted, or modified. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_LINKS_CHANGED: string

  /**
   * Dispatched after a Event is relocated from one object to another. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_MOVE: string

  /**
   * Dispatched after a Event is created. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_NEW: string

  /**
   * Dispatched after a Event is opened. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_OPEN: string

  /**
   * Dispatched after a Event is placed. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_PLACE: string

  /**
   * Dispatched when the Event is quitting. Since the quit has been committed, it can not be canceled. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_QUIT: string

  /**
   * Dispatched after an attribute on the active selection changes. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_SELECTION_ATTRIBUTE_CHANGED: string

  /**
   * Dispatched after the active selection changes. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_SELECTION_CHANGED: string

  /**
   * Dispatched after a Event is unembedded. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_UNEMBED: string

  /**
   * Dispatched after a Event is updated. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_UPDATE: string

  /**
   * Dispatched before a Event is closed. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_CLOSE: string

  /**
   * Dispatched before the Event becomes inactive. This event bubbles. This event is not cancelable.
   */
  static readonly BEFORE_DEACTIVATE: string

  /**
   * Dispatched before a Event is deleted. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_DELETE: string

  /**
   * Dispatched before the Event is displayed. This event does not bubble. This event is not cancelable.
   */
  static readonly BEFORE_DISPLAY: string

  /**
   * Dispatched before a Event is embedded. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_EMBED: string

  /**
   * Dispatched before the Event is invoked. This event does not bubble. This event is cancelable.
   */
  static readonly BEFORE_INVOKE: string

  /**
   * Dispatched before a Event is relocated from one object to another. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_MOVE: string

  /**
   * Dispatched before a Event is placed. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_PLACE: string

  /**
   * Dispatched before the Event is quit. Allows the quit to be canceled. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_QUIT: string

  /**
   * Dispatched before a Event is unembedded. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_UNEMBED: string

  /**
   * Dispatched before a Event is updated. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_UPDATE: string

  /**
   * Dispatched when the Event is invoked. This event does not bubble. This event is not cancelable.
   */
  static readonly ON_INVOKE: string

  /**
   * If true, the event supports the bubbling phase of propagation.
   */
  readonly bubbles: boolean

  /**
   * If true, the default behavior of the event on its target can be canceled.
   */
  readonly cancelable: boolean

  /**
   * The current propagation target of the event.
   */
  readonly currentTarget: object

  /**
   * If true, the default behavior of the event on its target has been canceled.
   */
  readonly defaultPrevented: boolean

  /**
   * The current propagation phase of the event.
   */
  readonly eventPhase: EventPhases

  /**
   * The name of the event.
   */
  readonly eventType: string

  /**
   * The unique ID of the Event.
   */
  readonly id: number

  /**
   * The index of the Event within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * The parent of the Event (a Object, UIDBasedObject, CellStyleGroup, CellStyle, TableStyleGroup, Article, ConditionSet, HiddenText, Condition, MotionPreset, AssignedStory, Assignment, ObjectStyleGroup, ObjectStyle, NumberingList, Snippet, Dialog, ColorGroup, Swatch, Color, Tint, Gradient, MixedInkGroup, MixedInk, Behavior, SubmitFormBehavior, PrintFormBehavior, ClearFormBehavior, GotoPageBehavior, GotoAnchorBehavior, SoundBehavior, ViewZoomBehavior, GotoStateBehavior, GotoPreviousStateBehavior, GotoNextStateBehavior, OpenFileBehavior, AnimationBehavior, ShowHideFieldsBehavior, MovieBehavior, GotoURLBehavior, GotoPreviousViewBehavior, GotoNextViewBehavior, GotoPreviousPageBehavior, GotoNextPageBehavior, GotoLastPageBehavior, GotoFirstPageBehavior, PreflightProfileRule, PreflightRuleInstance, PreflightProfile, CrossReferenceFormat, HyperlinkURLDestination, HyperlinkExternalPageDestination, HyperlinkPageDestination, HyperlinkTextDestination, ParagraphDestination, HyperlinkTextSource, CrossReferenceSource, HyperlinkPageItemSource, Bookmark, Hyperlink, IndexSection, PageReference, CrossReference, Index, TOCStyle, FlattenerPreset, BookContent, MenuAction, ScriptMenuAction, NamedGrid, CompositeFontEntry, CompositeFont, CharacterStyleGroup, ParagraphStyleGroup, TextVariableInstance, Footnote, XMLRuleProcessor, XMLTag, Note, TableStyle, TextPath, Asset, Link, Section, MojikumiTable, KinsokuTable, Guide, LanguageWithVendors, Language, PageItem, HtmlItem, FormField, SignatureField, TextBox, RadioButton, ListBox, ComboBox, CheckBox, MultiStateObject, Button, MediaItem, Sound, Movie, EPSText, SplineItem, Polygon, GraphicLine, Rectangle, Oval, Graphic, ImportedPage, PICT, WMF, PDF, EPS, Image, Group, TextFrame, EndnoteTextFrame, EndnoteRange, Endnote, MasterSpread, TrapPreset, Ink, DocumentPreset, Page, Spread, Layer, CharacterStyle, ParagraphStyle, Story, XmlStory, IDBasedObject, ArticleChild, ArticleMember, DialogRow, DialogColumn, Widget, BorderPanel, MeasurementEditbox, MeasurementCombobox, RealCombobox, AngleEditbox, PercentEditbox, RealEditbox, PercentCombobox, AngleCombobox, EnablingGroup, IntegerCombobox, IntegerEditbox, RadiobuttonControl, RadiobuttonGroup, CheckboxControl, Dropdown, StaticText, TextEditbox, ColorGroupSwatch, BackgroundTask, State, RuleDataObject, XMLItem, DTD, XMLInstruction, XMLComment, XMLElement, Table, Cell, IdleTask, StrokeStyle, StripedStrokeStyle, DottedStrokeStyle, DashedStrokeStyle, GraphicLayer, NonIDBasedObject, TableStyleMapping, CharStyleMapping, CellStyleMapping, ParaStyleMapping, TimingTarget, TimingGroup, TimingList, OpacityGradientStop, ObjectStyleExportTagMap, DataMergeQrcodePlaceholder, DataMergeField, DataMergeImagePlaceholder, DataMergeTextPlaceholder, Panel, LibraryPanel, PagesPanel, Window, StoryWindow, LayoutWindow, NavigationPoint, PreflightProcess, PreflightRule, BuildingBlock, DisplaySetting, IndexingSortOption, Topic, TOCStyleEntry, MenuElement, MenuSeparator, MenuItem, Submenu, Menu, StyleExportTagMap, TextVariable, XMLRuleMatchData, ValidationError, XMLExportMap, XMLImportMap, XMLAttribute, PrinterPreset, Row, Column, Change, HttpLinkConnectionManagerObject, RootObject, Document, Application, Book, Library, Preference, ContentPlacerObject, LinkedPageItemOption, LinkedStoryOption, PublishExportPreference, HTMLFXLExportPreference, EPubExportPreviewAppPreference, EPubFixedLayoutExportPreference, HTMLExportPreference, EPubExportPreference, ConditionalTextPreference, TimingSetting, AnimationSetting, XFLExportPreference, SWFExportPreference, TransformAttributeOption, AlignDistributePreference, TypeContextualUiPreference, GrabberPreference, ObjectStyleContentEffectsCategorySettings, ObjectStyleFillEffectsCategorySettings, ObjectStyleStrokeEffectsCategorySettings, ObjectStyleObjectEffectsCategorySettings, ChapterNumberPreference, NumberingRestartPolicy, Bullet, DataMerge, DataMergeOption, DataMergePreference, JPEGExportPreference, TrackChangesPreference, NotePreference, TransformPreference, ClipboardPreference, GeneralPreference, WatermarkPreference, ButtonPreference, PNGExportPreference, PreflightBookOption, PreflightOption, XMLViewPreference, GpuPerformancePreference, DisplayPerformancePreference, IndexOptions, LinkMetadata, MetadataPreference, ExcelImportPreference, TaggedTextImportPreference, TaggedTextExportPreference, WordRTFImportPreference, TextExportPreference, TextImportPreference, FindChangeContentTransparencySetting, FindChangeFillTransparencySetting, FindChangeStrokeTransparencySetting, FindChangeTransparencySetting, GradientFeatherSetting, FindChangeGradientFeatherSetting, DirectionalFeatherSetting, FindChangeDirectionalFeatherSetting, ContentTransparencySetting, SatinSetting, FindChangeSatinSetting, BevelAndEmbossSetting, FindChangeBevelAndEmbossSetting, InnerGlowSetting, FindChangeInnerGlowSetting, OuterGlowSetting, FindChangeOuterGlowSetting, InnerShadowSetting, FindChangeInnerShadowSetting, FeatherSetting, FindChangeFeatherSetting, DropShadowSetting, FindChangeDropShadowSetting, BlendingSetting, FindChangeBlendingSetting, FillTransparencySetting, StrokeTransparencySetting, TransparencySetting, TransparencyPreference, FlattenerPreference, GalleyPreference, GridPrintingPreference, CjkGridPreference, StoryGridDataInformation, LayoutGridDataInformation, GridDataInformation, CaptionMetadataVariablePreference, CustomTextVariablePreference, MatchParagraphStylePreference, MatchCharacterStylePreference, FileNameVariablePreference, DateVariablePreference, ChapterNumberVariablePreference, PageNumberVariablePreference, FootnoteOption, BaselineFrameGridOption, AnchoredObjectSetting, AnchoredObjectDefault, ExportForWebPreference, XMLPreference, XMLExportPreference, XMLImportPreference, InCopyExportOption, LinkingPreference, ChangeTransliteratePreference, ChangeObjectPreference, ChangeGlyphPreference, ChangeGrepPreference, ChangeTextPreference, FindTransliteratePreference, FindObjectPreference, FindGlyphPreference, FindGrepPreference, FindTextPreference, FindChangeTransliterateOption, FindChangeObjectOption, FindChangeGlyphOption, FindChangeGrepOption, FindChangeTextOption, ColorSetting, ScriptArg, ScriptPreference, PlaceGun, LayoutAdjustmentPreference, StrokeFillProxySetting, ImportedPageAttribute, EPSImportPreference, SmartGuidePreference, AutoCorrectPreference, SpellPreference, PolygonPreference, DictionaryPreference, FontLockingPreference, MojikumiUiPreference, ContourOption, TextWrapPreference, TextEditingPreference, FrameFittingOption, ObjectExportOption, PageItemDefault, EndnoteOption, TextFrameFootnoteOptionsObject, TaggedPDFPreference, InteractivePDFExportPreference, PDFAttribute, PDFPlacePreference, PDFExportPreference, IMEPreference, GraphicLayerOption, ClippingPathSettings, ImageIOPreference, ImagePreference, ToolBox, EPSExportPreference, PrintBookletPrintPreference, PrintBookletOption, PrintPreference, ViewPreference, PasteboardPreference, MarginPreference, GuidePreference, GridPreference, DocumentPreference, TextDefault, TextPreference, TextFramePreference, StoryPreference, PathPoint, Path, GradientStop, AutoCorrectTable, UserDictionary, HyphenationException, Font, TransformationMatrix, PDFExportPreset, NestedStyle, TabStop, Text, InsertionPoint, TextStyleRange, Paragraph, TextColumn, Line, Word, Character, NestedGrepStyle or NestedLineStyle).
   */
  readonly parent: any

  /**
   * If true, propagation of the event beyond the current target has been stopped.
   */
  readonly propagationStopped: boolean

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * The target of the event.
   */
  readonly target: object

  /**
   * The time the event was initialized.
   */
  readonly timeStamp: Date

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): Event[]

  /**
   * Cancels the default behavior of the event on its target.
   */
  preventDefault(): void

  /**
   * Stops propagation of the event beyond the current target.
   */
  stopPropagation(): void

  /**
   * Generates a string which, if executed, will return the Event.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of events.
 */
declare class Events {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Event with the specified index.
   * @param index The index.
   */
  [index: number]: Event

  /**
   * Returns any Event in the collection.
   */
  anyItem(): Event

  /**
   * Displays the number of elements in the Event.
   */
  count(): number

  /**
   * Returns every Event in the collection.
   */
  everyItem(): Event

  /**
   * Returns the first Event in the collection.
   */
  firstItem(): Event

  /**
   * Returns the Event with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Event

  /**
   * Returns the Event with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Event

  /**
   * Returns the Events within the specified range.
   * @param from The Event, index, or name at the beginning of the range.
   * @param to The Event, index, or name at the end of the range.
   */
  itemByRange(from: Event | number | string, to: Event | number | string): Event[]

  /**
   * Returns the last Event in the collection.
   */
  lastItem(): Event

  /**
   * Returns the middle Event in the collection.
   */
  middleItem(): Event

  /**
   * Returns the Event whose index follows the specified Event in the collection.
   * @param obj The Event whose index comes before the desired Event.
   */
  nextItem(obj: Event): Event

  /**
   * Returns the Event with the index previous to the specified index.
   * @param obj The index of the Event that follows the desired Event.
   */
  previousItem(obj: Event): Event

  /**
   * Generates a string which, if executed, will return the Event.
   */
  toSource(): string
}

/**
 * An idle event.
 */
declare class IdleEvent extends Event {
  /**
   * Dispatched at idle time for this IdleEvent. This event does not bubble. This event is not cancelable.
   */
  static readonly ON_IDLE: string

  /**
   * Amount of time allocated to this task at the time the event was dispatched.
   */
  readonly timeAllocated: number
}

/**
 * A mutation event is dispatched for attribute changes.
 */
declare class MutationEvent extends Event {
  /**
   * Dispatched when the value of a property changes on this MutationEvent. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_ATTRIBUTE_CHANGED: string

  /**
   * The name of the property that changed.
   */
  readonly attributeName: string

  /**
   * The current value of the property that changed.
   */
  readonly attributeValue: any
}

/**
 * An attachable idle task.
 */
declare class IdleTask {
  /**
   * Dispatched at idle time for this IdleTask. This event does not bubble. This event is not cancelable.
   */
  static readonly ON_IDLE: string

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The unique ID of the IdleTask.
   */
  readonly id: number

  /**
   * The index of the IdleTask within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The name of the IdleTask.
   */
  name: string

  /**
   * The parent of the IdleTask (a Application).
   */
  readonly parent: Application

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * Amount of time to sleep (in milliseconds) before calling this task again. Setting this to zero will cause task to be deleted.
   */
  sleep: number

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): IdleTask[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Deletes the IdleTask.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the IdleTask.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * All attachable idle tasks.
 */
declare class IdleTasks {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the IdleTask with the specified index.
   * @param index The index.
   */
  [index: number]: IdleTask

  /**
   * Creates a new IdleTask.
   * @param withProperties Initial values for properties of the new IdleTask
   */
  add(withProperties: object): IdleTask

  /**
   * Returns any IdleTask in the collection.
   */
  anyItem(): IdleTask

  /**
   * Displays the number of elements in the IdleTask.
   */
  count(): number

  /**
   * Returns every IdleTask in the collection.
   */
  everyItem(): IdleTask

  /**
   * Returns the first IdleTask in the collection.
   */
  firstItem(): IdleTask

  /**
   * Returns the IdleTask with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): IdleTask

  /**
   * Returns the IdleTask with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): IdleTask

  /**
   * Returns the IdleTask with the specified name.
   * @param name The name.
   */
  itemByName(name: string): IdleTask

  /**
   * Returns the IdleTasks within the specified range.
   * @param from The IdleTask, index, or name at the beginning of the range.
   * @param to The IdleTask, index, or name at the end of the range.
   */
  itemByRange(from: IdleTask | number | string, to: IdleTask | number | string): IdleTask[]

  /**
   * Returns the last IdleTask in the collection.
   */
  lastItem(): IdleTask

  /**
   * Returns the middle IdleTask in the collection.
   */
  middleItem(): IdleTask

  /**
   * Returns the IdleTask whose index follows the specified IdleTask in the collection.
   * @param obj The IdleTask whose index comes before the desired IdleTask.
   */
  nextItem(obj: IdleTask): IdleTask

  /**
   * Returns the IdleTask with the index previous to the specified index.
   * @param obj The index of the IdleTask that follows the desired IdleTask.
   */
  previousItem(obj: IdleTask): IdleTask

  /**
   * Generates a string which, if executed, will return the IdleTask.
   */
  toSource(): string
}

/**
 * Arguments to pass to a script.
 */
declare class ScriptArg extends Preference {
  /**
   * Clears all script arguments.
   */
  clear(): void

  /**
   * Gets the value of a script argument.
   * @param name The name of the script argument.
   */
  get(name: string): string

  /**
   * Gets the value of a script argument.
   * @param name The name of the script argument.
   */
  getValue(name: string): string

  /**
   * Verifies whether the script argument is defined.
   * @param name The name of the script argument.
   */
  isDefined(name: string): boolean

  /**
   * Restores all script arguments.
   */
  restore(): void

  /**
   * Saves the script arguments.
   */
  save(): void

  /**
   * Sets the value of a script argument.
   * @param name The name of the script argument.
   * @param value The value.
   */
  set(name: string, value: string): void

  /**
   * Sets the value of a script argument.
   * @param name The name of the script argument.
   * @param value The value.
   */
  setValue(name: string, value: string): void
}

/**
 * The application.
 */
declare class Application {
  /**
   * Dispatched after the Application becomes active. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_ACTIVATE: string

  /**
   * Dispatched after a Document is closed. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_CLOSE: string

  /**
   * Dispatched after the active context changes. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_CONTEXT_CHANGED: string

  /**
   * Dispatched when the Application is quitting. Since the quit has been committed, it can not be canceled. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_QUIT: string

  /**
   * Dispatched after an attribute on the active selection changes. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_SELECTION_ATTRIBUTE_CHANGED: string

  /**
   * Dispatched after the active selection changes. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_SELECTION_CHANGED: string

  /**
   * Dispatched before the Application becomes inactive. This event bubbles. This event is not cancelable.
   */
  static readonly BEFORE_DEACTIVATE: string

  /**
   * Dispatched before a Document is created. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_NEW: string

  /**
   * Dispatched before a Document is opened. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_OPEN: string

  /**
   * Dispatched before the Application is quit. Allows the quit to be canceled. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_QUIT: string

  /**
   * The active book.
   */
  activeBook: Book

  /**
   * The front-most document.
   */
  activeDocument: Document

  /**
   * The current script running from the Scripts panel.
   */
  readonly activeScript: File

  /**
   * The undo mode for the current script execution.
   */
  readonly activeScriptUndoMode: UndoModes

  /**
   * The front-most window.
   */
  activeWindow: Window | LayoutWindow | StoryWindow

  /**
   * Preferences for alignment and distribution.
   */
  readonly alignDistributePreferences: AlignDistributePreference

  /**
   * All Cell styles
   */
  readonly allCellStyles: CellStyle[]

  /**
   * Lists all character styles (regardless of their group).
   */
  readonly allCharacterStyles: CharacterStyle[]

  /**
   * All object styles contained by the Application.
   */
  readonly allObjectStyles: ObjectStyle[]

  /**
   * Lists all paragraph styles (regardless of their group).
   */
  readonly allParagraphStyles: ParagraphStyle[]

  /**
   * The list of all object types (strings) a preflight rule can operate on.
   */
  readonly allPreflightObjectTypes: string[]

  /**
   * The list of all categories that have been declared by rules.
   */
  readonly allPreflightRuleCategories: string[]

  /**
   * The list of all known (declared) rule IDs.
   */
  readonly allPreflightRuleIDs: string[]

  /**
   * All Table styles
   */
  readonly allTableStyles: TableStyle[]

  /**
   * Anchored object default settings.
   */
  readonly anchoredObjectDefaults: AnchoredObjectDefault

  /**
   * Anchored object settings.
   */
  readonly anchoredObjectSettings: AnchoredObjectSetting

  /**
   * Auto-correct preferences.
   */
  readonly autoCorrectPreferences: AutoCorrectPreference

  /**
   * A collection of auto-correct tables.
   */
  readonly autoCorrectTables: AutoCorrectTables

  /**
   * A collection of background task objects.
   */
  readonly backgroundTasks: BackgroundTasks

  /**
   * Baseline frame grid option settings.
   */
  readonly baselineFrameGridOptions: BaselineFrameGridOption

  /**
   * A collection of books.
   */
  readonly books: Books

  /**
   * Button preference settings.
   */
  readonly buttonPreferences: ButtonPreference

  /**
   * A collection of cell style groups.
   */
  readonly cellStyleGroups: CellStyleGroups

  /**
   * A collection of cell style mappings.
   */
  readonly cellStyleMappings: CellStyleMappings

  /**
   * A collection of cell styles.
   */
  readonly cellStyles: CellStyles

  /**
   * Change glyph preferences.
   */
  changeGlyphPreferences: ChangeGlyphPreference | NothingEnum

  /**
   * Change grep preferences.
   */
  changeGrepPreferences: ChangeGrepPreference | NothingEnum

  /**
   * Change object preferences.
   */
  changeObjectPreferences: ChangeObjectPreference | NothingEnum

  /**
   * Change text preferences.
   */
  changeTextPreferences: ChangeTextPreference | NothingEnum

  /**
   * Change transliterate preferences.
   */
  changeTransliteratePreferences: ChangeTransliteratePreference | NothingEnum

  /**
   * A collection of char style mappings.
   */
  readonly charStyleMappings: CharStyleMappings

  /**
   * A collection of character style groups.
   */
  readonly characterStyleGroups: CharacterStyleGroups

  /**
   * A collection of character styles.
   */
  readonly characterStyles: CharacterStyles

  /**
   * CJK grid preference settings.
   */
  readonly cjkGridPreferences: CjkGridPreference

  /**
   * Clear overrides first before applying object style
   */
  clearOverridesWhenApplyingStyle: boolean

  /**
   * Clipboard preference properties that define the way that the application interacts with the system clipboard.
   */
  readonly clipboardPreferences: ClipboardPreference

  /**
   * A collection of color groups.
   */
  readonly colorGroups: ColorGroups

  /**
   * Color setting properties that define color management defaults.
   */
  readonly colorSettings: ColorSetting

  /**
   * A collection of colors.
   */
  readonly colors: Colors

  /**
   * A collection of composite fonts.
   */
  readonly compositeFonts: CompositeFonts

  /**
   * A collection of condition sets for conditional text.
   */
  readonly conditionSets: ConditionSets

  /**
   * Conditional text preferences.
   */
  readonly conditionalTextPreferences: ConditionalTextPreference

  /**
   * A collection of conditions for conditional text.
   */
  readonly conditions: Conditions

  /**
   * The Content Placer. Used to get a reference to the content placer object.
   */
  readonly contentPlacer: ContentPlacerObject

  /**
   * A collection of dashed stroke styles.
   */
  readonly dashedStrokeStyles: DashedStrokeStyles

  /**
   * The data merge option properties that define the data merge.
   */
  readonly dataMergeOptions: DataMergeOption

  /**
   * A collection of dialogs.
   */
  readonly dialogs: Dialogs

  /**
   * User dictionary preference settings.
   */
  readonly dictionaryPreferences: DictionaryPreference

  /**
   * Display performance settings.
   */
  readonly displayPerformancePreferences: DisplayPerformancePreference

  /**
   * Display setting properties.
   */
  readonly displaySettings: DisplaySettings

  /**
   * Document preference settings.
   */
  readonly documentPreferences: DocumentPreference

  /**
   * A collection of document presets.
   */
  readonly documentPresets: DocumentPresets

  /**
   * A collection of documents.
   */
  readonly documents: Documents

  /**
   * A collection of dotted stroke styles.
   */
  readonly dottedStrokeStyles: DottedStrokeStyles

  /**
   * Endnote option settings.
   */
  readonly endnoteOptions: EndnoteOption

  /**
   * EPS export preferences.
   */
  readonly epsExportPreferences: EPSExportPreference

  /**
   * EPS import preferences.
   */
  readonly epsImportPreferences: EPSImportPreference

  /**
   * EPub export preview app preference settings.
   */
  readonly epubViewingAppsPreferences: EPubExportPreviewAppPreference

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * Excel import preferences.
   */
  readonly excelImportPreferences: ExcelImportPreference

  /**
   * The default export for web preferences.
   */
  readonly exportForWebPreferences: ExportForWebPreference

  /**
   * The feature set.
   */
  readonly featureSet: FeatureSetOptions

  /**
   * The full path to the file.
   */
  readonly filePath: File

  /**
   * Find/change glyph options.
   */
  findChangeGlyphOptions: FindChangeGlyphOption | NothingEnum

  /**
   * Find/change grep options.
   */
  findChangeGrepOptions: FindChangeGrepOption | NothingEnum

  /**
   * Find/change object options.
   */
  findChangeObjectOptions: FindChangeObjectOption | NothingEnum

  /**
   * Find/change text options.
   */
  findChangeTextOptions: FindChangeTextOption | NothingEnum

  /**
   * Find/change transliterate options.
   */
  findChangeTransliterateOptions: FindChangeTransliterateOption | NothingEnum

  /**
   * Find glyph preferences.
   */
  findGlyphPreferences: FindGlyphPreference | NothingEnum

  /**
   * Find grep preferences.
   */
  findGrepPreferences: FindGrepPreference | NothingEnum

  /**
   * Find object preferences.
   */
  findObjectPreferences: FindObjectPreference | NothingEnum

  /**
   * Find text preferences.
   */
  findTextPreferences: FindTextPreference | NothingEnum

  /**
   * Find transliterate preferences.
   */
  findTransliteratePreferences: FindTransliteratePreference | NothingEnum

  /**
   * A collection of transparency flattener presets.
   */
  readonly flattenerPresets: FlattenerPresets

  /**
   * Font locking preference settings.
   */
  readonly fontLockingPreferences: FontLockingPreference

  /**
   * A collection of fonts.
   */
  readonly fonts: Fonts

  /**
   * Footnote option settings.
   */
  readonly footnoteOptions: FootnoteOption

  /**
   * The frame fitting option to apply to placed or pasted content. Can be applied to a frame, object style, or document or to the application.
   */
  readonly frameFittingOptions: FrameFittingOption

  /**
   * The full path to the Application, including the name of the Application.
   */
  readonly fullName: File

  /**
   * Galley preference settings.
   */
  readonly galleyPreferences: GalleyPreference

  /**
   * General preference settings.
   */
  readonly generalPreferences: GeneralPreference

  /**
   * GPU performance preferences.
   */
  readonly gpuPerformancePreferences: GpuPerformancePreference

  /**
   * Grabber preference properties that define display performance quality during scrolling.
   */
  readonly grabberPreferences: GrabberPreference

  /**
   * A collection of gradients.
   */
  readonly gradients: Gradients

  /**
   * Grid preference settings.
   */
  readonly gridPreferences: GridPreference

  /**
   * Grid printing preference and export settings.
   */
  readonly gridPrintingPreferences: GridPrintingPreference

  /**
   * Guide preference settings.
   */
  readonly guidePreferences: GuidePreference

  /**
   * Experimental: Connection manager property
   */
  httpLinkConnectionManager: HttpLinkConnectionManagerObject

  /**
   * All attachable idle tasks.
   */
  readonly idleTasks: IdleTasks

  /**
   * The image I/O preference properties that define preferences for importing images.
   */
  readonly imageIOPreferences: ImageIOPreference

  /**
   * Image preferences.
   */
  readonly imagePreferences: ImagePreference

  /**
   * IME preference settings.
   */
  readonly imePreferences: IMEPreference

  /**
   * Placed InDesign page attributes.
   */
  readonly importedPageAttributes: ImportedPageAttribute

  /**
   * Export options for InCopy INCX document format.
   */
  readonly incopyExportOptions: InCopyExportOption

  /**
   * The index options properties that define how an index is formatted.
   */
  readonly indexGenerationOptions: IndexOptions

  /**
   * A collection of indexing sort options.
   */
  readonly indexingSortOptions: IndexingSortOptions

  /**
   * A collection of inks.
   */
  readonly inks: Inks

  /**
   * Interactive PDF export preferences.
   */
  readonly interactivePDFExportPreferences: InteractivePDFExportPreference

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * JPEG export preferences.
   */
  readonly jpegExportPreferences: JPEGExportPreference

  /**
   * A collection of kinsoku tables.
   */
  readonly kinsokuTables: KinsokuTables

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * A collection of languages with vendors.
   */
  readonly languagesWithVendors: LanguagesWithVendors

  /**
   * Layout sdjustment preference settings.
   */
  readonly layoutAdjustmentPreferences: LayoutAdjustmentPreference

  /**
   * Default properties specific to layout grids.
   */
  readonly layoutGridData: LayoutGridDataInformation

  /**
   * A collection of layout windows.
   */
  readonly layoutWindows: LayoutWindows

  /**
   * A collection of object libraries.
   */
  readonly libraries: Libraries

  /**
   * Linked Page Item options
   */
  readonly linkedPageItemOptions: LinkedPageItemOption

  /**
   * Linked story options
   */
  readonly linkedStoryOptions: LinkedStoryOption

  /**
   * The links preference properties that define preferences for links.
   */
  readonly linkingPreferences: LinkingPreference

  /**
   * Delay before mouse operations trigger live screen drawing of page items.
   */
  liveScreenDrawing: LiveDrawingOptions

  /**
   * The locale of the application.
   */
  readonly locale: Locale

  /**
   * Margin preference settings.
   */
  readonly marginPreferences: MarginPreference

  /**
   * A collection of menu actions.
   */
  readonly menuActions: MenuActions

  /**
   * A collection of menus.
   */
  readonly menus: Menus

  /**
   * A collection of mixed ink groups.
   */
  readonly mixedInkGroups: MixedInkGroups

  /**
   * A collection of mixed inks.
   */
  readonly mixedInks: MixedInks

  /**
   * If true, a modal dialog or alert is active.
   */
  readonly modalState: boolean

  /**
   * A collection of mojikumi tables.
   */
  readonly mojikumiTables: MojikumiTables

  /**
   * Mojikumi user interface preference settings.
   */
  readonly mojikumiUIPreferences: MojikumiUiPreference

  /**
   * A collection of Motion presets.
   */
  readonly motionPresets: MotionPresets

  /**
   * The name of the Application.
   */
  readonly name: string

  /**
   * A collection of named grids.
   */
  readonly namedGrids: NamedGrids

  /**
   * Note preference settings.
   */
  readonly notePreferences: NotePreference

  /**
   * A collection of numbered lists.
   */
  readonly numberingLists: NumberingLists

  /**
   * A collection of object style groups.
   */
  readonly objectStyleGroups: ObjectStyleGroups

  /**
   * A collection of object styles.
   */
  readonly objectStyles: ObjectStyles

  /**
   * The default page item formatting for the Application.
   */
  readonly pageItemDefaults: PageItemDefault

  /**
   * A collection of panels.
   */
  readonly panels: Panels

  /**
   * A collection of para style mappings.
   */
  readonly paraStyleMappings: ParaStyleMappings

  /**
   * A collection of paragraph style groups.
   */
  readonly paragraphStyleGroups: ParagraphStyleGroups

  /**
   * A collection of paragraph styles.
   */
  readonly paragraphStyles: ParagraphStyles

  /**
   * The parent of the Application (a Application).
   */
  readonly parent: Application

  /**
   * Pasteboard preference settings.
   */
  readonly pasteboardPreferences: PasteboardPreference

  /**
   * A collection of PDF export preferences.
   */
  readonly pdfExportPreferences: PDFExportPreference

  /**
   * A collection of PDF export presets.
   */
  readonly pdfExportPresets: PDFExportPresets

  /**
   * The PDF place preference properties that define how PDF files are placed in the current document.
   */
  readonly pdfPlacePreferences: PDFPlacePreference

  /**
   * The available performance metrics.
   */
  readonly performanceMetrics: number[]

  /**
   * Lists the extensions of file types that can be placed.
   */
  readonly placeableFileExtensions: string[]

  /**
   * Lists the types of files that can be placed.
   */
  readonly placeableFileTypes: string[]

  /**
   * PNG export preferences.
   */
  readonly pngExportPreferences: PNGExportPreference

  /**
   * Polygon preference properties to use to define default settings for creating a polygon.
   */
  readonly polygonPreferences: PolygonPreference

  /**
   * A collection of preferences objects.
   */
  readonly preferences: Preferences

  /**
   * Preflight book option settings.
   */
  readonly preflightBookOptions: PreflightBookOption

  /**
   * Preflight option settings.
   */
  readonly preflightOptions: PreflightOption

  /**
   * A collection of preflight processes.
   */
  readonly preflightProcesses: PreflightProcesses

  /**
   * A collection of preflight profiles.
   */
  readonly preflightProfiles: PreflightProfiles

  /**
   * A collection of preflight rules.
   */
  readonly preflightRules: PreflightRules

  /**
   * A collection of printer presets.
   */
  readonly printerPresets: PrinterPresets

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * The names of the items in the redo stack.
   */
  readonly redoHistory: string[]

  /**
   * The name of the action on the top of the redo stack.
   */
  readonly redoName: string

  /**
   * Arguments to pass to a script.
   */
  readonly scriptArgs: ScriptArg

  /**
   * A collection of script menu actions.
   */
  readonly scriptMenuActions: ScriptMenuActions

  /**
   * Script preferences.
   */
  readonly scriptPreferences: ScriptPreference

  /**
   * The selected object(s).
   */
  selection: object[] | object | NothingEnum

  /**
   * The key object of the selection.
   */
  selectionKeyObject: PageItem | NothingEnum

  /**
   * The user's serial number.
   */
  readonly serialNumber: string

  /**
   * Smart Guide preference properties.
   */
  readonly smartGuidePreferences: SmartGuidePreference

  /**
   * Spell-check preferences.
   */
  readonly spellPreferences: SpellPreference

  /**
   * Default properties specific to frame grids.
   */
  readonly storyGridData: StoryGridDataInformation

  /**
   * Story preference settings.
   */
  readonly storyPreferences: StoryPreference

  /**
   * A collection of story windows.
   */
  readonly storyWindows: StoryWindows

  /**
   * A collection of striped stroke styles.
   */
  readonly stripedStrokeStyles: StripedStrokeStyles

  /**
   * Stroke/fill proxy settings.
   */
  readonly strokeFillProxySettings: StrokeFillProxySetting

  /**
   * A collection of stroke styles.
   */
  readonly strokeStyles: StrokeStyles

  /**
   * A collection of swatches.
   */
  readonly swatches: Swatches

  /**
   * SWF export preferences.
   */
  readonly swfExportPreferences: SWFExportPreference

  /**
   * A collection of table style groups.
   */
  readonly tableStyleGroups: TableStyleGroups

  /**
   * A collection of table style mappings.
   */
  readonly tableStyleMappings: TableStyleMappings

  /**
   * A collection of table styles.
   */
  readonly tableStyles: TableStyles

  /**
   * Tagged PDF preferences.
   */
  readonly taggedPDFPreferences: TaggedPDFPreference

  /**
   * Tagged text export preferences.
   */
  readonly taggedTextExportPreferences: TaggedTextExportPreference

  /**
   * Tagged text import preferences.
   */
  readonly taggedTextImportPreferences: TaggedTextImportPreference

  /**
   * Text default settings.
   */
  readonly textDefaults: TextDefault

  /**
   * Text editing preference settings.
   */
  readonly textEditingPreferences: TextEditingPreference

  /**
   * Text export preferences.
   */
  readonly textExportPreferences: TextExportPreference

  /**
   * Text frame footnote options settings.
   */
  readonly textFrameFootnoteOptions: TextFrameFootnoteOptionsObject

  /**
   * Text frame preference settings.
   */
  readonly textFramePreferences: TextFramePreference

  /**
   * Text import preferences.
   */
  readonly textImportPreferences: TextImportPreference

  /**
   * Text preference settings.
   */
  readonly textPreferences: TextPreference

  /**
   * A collection of text variables.
   */
  readonly textVariables: TextVariables

  /**
   * The text wrap preference properties that define the default formatting for wrapping text around objects.
   */
  readonly textWrapPreferences: TextWrapPreference

  /**
   * A collection of tints.
   */
  readonly tints: Tints

  /**
   * The current tool box states
   */
  readonly toolBoxTools: ToolBox

  /**
   * Track changes preference settings.
   */
  readonly trackChangesPreferences: TrackChangesPreference

  /**
   * Transform preference properties that define default behaviors when transforming objects. Note: Transforming includes rotation, scaling, flipping, and shearing.
   */
  readonly transformPreferences: TransformPreference

  /**
   * A collection of transformation matrices.
   */
  readonly transformationMatrices: TransformationMatrices

  /**
   * Transparency preference settings.
   */
  readonly transparencyPreferences: TransparencyPreference

  /**
   * A collection of trap presets.
   */
  readonly trapPresets: TrapPresets

  /**
   * Preferences for showing contextual ui for alternates.
   */
  readonly typeContextualUiPrefs: TypeContextualUiPreference

  /**
   * The names of the items in the undo stack.
   */
  readonly undoHistory: string[]

  /**
   * The name of the action on the top of the undo stack.
   */
  readonly undoName: string

  /**
   * The swatches that are not being used.
   */
  readonly unusedSwatches: Swatch[]

  /**
   * The current user's adobe id
   */
  readonly userAdobeId: string

  /**
   * The color assigned to the tracked changes and notes created by the user, specified either as an array of three doubles, each in the range 0 to 255 and representing R, G, and B values, or as an InCopy UI color.
   */
  userColor: [number, number, number] | InCopyUIColors

  /**
   * A collection of user dictionaries.
   */
  readonly userDictionaries: UserDictionaries

  /**
   * The current user's GUID
   */
  readonly userGuid: string

  /**
   * The user associated with the tracked changes and notes.
   */
  userName: string

  /**
   * The application version.
   */
  readonly version: string

  /**
   * View preference settings.
   */
  readonly viewPreferences: ViewPreference

  /**
   * If true, the Application is visible.
   */
  readonly visible: boolean

  /**
   * Watermark preferences
   */
  readonly watermarkPreferences: WatermarkPreference

  /**
   * A collection of windows.
   */
  readonly windows: Windows

  /**
   * Word and RTF import preferences.
   */
  readonly wordRTFImportPreferences: WordRTFImportPreference

  /**
   * XFL export preferences.
   */
  readonly xflExportPreferences: XFLExportPreference

  /**
   * A collection of XML export maps.
   */
  readonly xmlExportMaps: XMLExportMaps

  /**
   * XML export preference settings.
   */
  readonly xmlExportPreferences: XMLExportPreference

  /**
   * A collection of XML import maps.
   */
  readonly xmlImportMaps: XMLImportMaps

  /**
   * XML import preference settings.
   */
  readonly xmlImportPreferences: XMLImportPreference

  /**
   * The XML preference settings.
   */
  readonly xmlPreferences: XMLPreference

  /**
   * A collection of XML rule processors.
   */
  readonly xmlRuleProcessors: XMLRuleProcessors

  /**
   * A collection of XML tags.
   */
  readonly xmlTags: XMLTags

  /**
   * XML view preference settings.
   */
  readonly xmlViewPreferences: XMLViewPreference

  /**
   * Makes the application the front-most or active window.
   */
  activate(): void

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Applies the specified menu customization set. An empty string will reset all menus and colorization (Show Full Menus). No string will apply the default menu set.
   * @param name The menu customization set.
   */
  applyMenuCustomization(name: string): void

  /**
   * Applies the specified shortcut set file. No string will apply the default shortcut set.
   * @param name The shortcut set.
   */
  applyShortcutSet(name: string): void

  /**
   * Applies the specified workspace.
   * @param name The workspace.
   */
  applyWorkspace(name: string): void

  /**
   * Cancels all the background tasks.
   */
  cancelAllTasks(): void

  /**
   * Cascades all document windows.
   */
  cascadeWindows(): void

  /**
   * Finds glyphs that match the find what value and replaces the glyphs with the change to value.
   * @param reverseOrder If true, returns the results in reverse order.
   */
  changeGlyph(reverseOrder: boolean): Text[]

  /**
   * Finds text that matches the find what value and replaces the text with the change to value.
   * @param reverseOrder If true, returns the results in reverse order.
   */
  changeGrep(reverseOrder: boolean): Text[]

  /**
   * Finds objects that match the find what value and replace the objects with the change to value.
   * @param reverseOrder If true, returns the results in reverse order.
   */
  changeObject(reverseOrder: boolean): PageItem[]

  /**
   * Finds text that matches the find what value and replaces the text with the change to value.
   * @param reverseOrder If true, returns the results in reverse order.
   */
  changeText(reverseOrder: boolean): Text[]

  /**
   * Finds text that matches the find character type value and replaces the text with the change character type value.
   * @param reverseOrder If true, returns the results in reverse order.
   */
  changeTransliterate(reverseOrder: boolean): Text[]

  /**
   * Removes the frame fittings options and resets it to the initial state.
   */
  clearFrameFittingOptions(): void

  /**
   * Transforms color values
   * @param colorValue source color value
   * @param sourceColorSpace source color space
   * @param destinationColorSpace destination color space
   */
  colorTransform(
    colorValue: number[],
    sourceColorSpace: ColorSpace,
    destinationColorSpace: ColorSpace,
  ): number[]

  /**
   * Copies the selection in the active document window to the clipboard.
   */
  copy(): void

  /**
   * Create a mini-folio out of asset and overlay descriptions.
   *
   * Schema for the mini folio description:
   * || Key || Type || Required? || Description ||
   * | contentstackid | string | yes | The explicit ID to be used for this stack |
   * | tocImage | file | no | The TOC image for the mini-folio |
   * | narrowdimension | int | no | Targetted export size, measured along narrow dimension |
   * | widedimension | int | no | Targetted export size, measured along wide dimension |
   * | smoothscrolling | @enum(SmoothScrollingOptions) | no | The smooth scrolling behavior for this stack (default is noSmoothScroll) |
   * | locationforgeneratedassets | file | no | If included, this is an existing directory where the generated assets should be created. |
   * | overlays | array | no | see the SDK Overlay Builder APIs |
   * | assets | array | yes | The assets for the mini - folio, described below. |
   * | assetDependencies | array | no | Files that are needed by the asset, that will be included in the package but not the manifest. |
   * | metadata | array | no | The metadata for the mini-folio described below. |
   * | showprogressbar | boolean | no | Either true or false to indicate whether we should show a progress bar (default is false) |
   * | targetviewerversion | string | no | If provided (in the form "major.minor.revision") a folio compatible with the viewer version is produced. If omitted, the latest folio format is produced. | CS5 |
   *
   * The following are fields in the array for "assets":
   * || Key || Type || Required? || Description ||
   * | file | file | yes | The asset file |
   * | type | string | yes | The asset file type, either "web" for HTML or "image" for a raster |
   * | width | int | yes | The asset width |
   * | height | int | yes | The asset height |
   * | orientation | string | yes | The asset orientation, either "portrait" or "landscape" |
   * | thumbnail | file | no | A thumbnail of the asset to be used in browse mode |
   * | scrubber | file | no | A thumbnail of the asset to be used in scrubber mode |
   *
   * The following are fields in the array for "assetDependencies":
   * || Key || Type || Required? || Description ||
   * | file | file | no | The dependent file |
   * | relativePath | string | no | The relative path to the file, for use in the package |
   *
   * The following are fields in the array for "metadata":
   * || Key || Type || Required? || Description ||
   * | any string | string or boolean | no | any key/value where the key is a string, and the value is either a string or a boolean, such as those described in "export folio meta data" |
   * @param miniFolioDescription A dictionary describing the custom mini folio to create. Can accept: Ordered array containing key:String, value:Boolean, Long Integer, Long Long Integer, String, File or Array of Any Types.
   * @param destination The location to write the file.
   */
  createCustomMiniFolio(miniFolioDescription: any[], destination: File): void

  /**
   * Creates a temporary copy of the file
   * @param from The file to be copied
   */
  createTemporaryCopy(from: File): string

  /**
   * Cuts the selection in the active document window and stores it in the clipboard.
   */
  cut(): void

  /**
   * Deletes the specified find/change query.
   * @param queryName The query to delete.
   * @param searchMode The search mode.
   */
  deleteFindChangeQuery(queryName: string, searchMode: SearchModes): void

  /**
   * Deletes unused XML markup tags.
   */
  deleteUnusedTags(): void

  /**
   * Executes the script in the specified language as a single transaction.
   * @param script The script to execute.
   * @param language The language of the script to execute. If not specified, uses the language used to call this method.
   * @param withArguments An array of arguments passed to the script.
   * @param undoMode How to undo this script.
   * @param undoName The name of the undo step for entire script undo mode.
   */
  doScript(
    script: File | string | Function,
    language?: ScriptLanguage,
    withArguments?: any[],
    undoMode?: UndoModes,
    undoName?: string,
  ): any

  /**
   * Dumps memory allocations from all marks in the specified range.
   * @param from The first mark in the range.
   * @param to The last mark in the range.
   */
  dumpBetweenMemoryMarks(from: number[], to: number[]): void

  /**
   * Dumps memory allocations from the specified mark.
   * @param from The mark from which to dump memory.
   */
  dumpFromMemoryMark(from: number[]): void

  /**
   * Export the specified documents to an article folio.
   * Note: This method behaves identically to @method(exportMiniFolio), but differs in its return value. @method(exportMiniFolio) returns an array of warning strings whereas this method returns an XML structure.
   * @param destination The location to write the file.
   * @param portraitDocument The InDesign document for the stack's portrait orientation.
   * @param landscapeDocument The InDesign document for the stack's landscape orientation.
   * @param folioMetadata Optional meta data for the mini folio. Can accept: Ordered array containing key:String, value:String.
   * @param miniFolioParams Options for exporting a mini folio. Can accept: Ordered array containing dataField:String, dataValue:Any Type.
   */
  exportArticleFolio(
    destination: File,
    portraitDocument: Document,
    landscapeDocument: Document,
    folioMetadata: any[],
    miniFolioParams: any[],
  ): string

  /**
   * Export the specified document to a DPS article.
   * Note: This method behaves similarly to @method(exportMiniFolio), but differs in its parameters.
   * @param destination The location to write the file.
   * @param document The source InDesign document.
   * @param dpsArticleParams Options for exporting a DPS article. Can accept: Ordered array containing dataField:String, dataValue:Any Type.
   */
  exportDpsArticle(destination: File, document: Document, dpsArticleParams: any[]): string[]

  /**
   * Export the selected documents to a directory.
   * @param destination The directory to write the folio.
   * @param miniFolioList The mini folio(es) to add to the folio.
   * @param folioMetadata Meta data for the folio. Can accept: Ordered array containing key:String, value:String.
   * @param exportFolioParams Additional options for export folios. Can accept: Ordered array containing dataField:String, dataValue:Any Type.
   */
  exportFolioToDirectory(
    destination: File,
    miniFolioList: File | File[],
    folioMetadata: any[],
    exportFolioParams: any[],
  ): void

  /**
   * Export the selected documents to a compressed folio file that contains non-compressed mini folios.
   * @param destination The location to write the package.
   * @param miniFolioList The mini folio(es) to add to the folio.
   * @param folioMetadata Meta data for the folio. Can accept: Ordered array containing key:String, value:String.
   * @param exportFolioParams Additional options for export folios. Can accept: Ordered array containing dataField:String, dataValue:Any Type.
   */
  exportFolioToDirectoryPackage(
    destination: File,
    miniFolioList: File | File[],
    folioMetadata: any[],
    exportFolioParams: any[],
  ): void

  /**
   * Export the selected documents to a compressed folio file that contains compressed mini folios.
   * @param destination The location to write the package.
   * @param miniFolioList The mini folio(es) to add to the folio.
   * @param folioMetadata Meta data for the folio. Can accept: Ordered array containing key:String, value:String.
   * @param exportFolioParams Additional options for export folios. Can accept: Ordered array containing dataField:String, dataValue:Any Type.
   */
  exportFolioToPackage(
    destination: File,
    miniFolioList: File | File[],
    folioMetadata: any[],
    exportFolioParams: any[],
  ): void

  /**
   * Export the specified documents to a mini-folio.
   * @param destination The location to write the file.
   * @param portraitDocument The InDesign document for the stack's portrait orientation.
   * @param landscapeDocument The InDesign document for the stack's landscape orientation.
   * @param folioMetadata Optional meta data for the mini folio. Can accept: Ordered array containing key:String, value:String.
   * @param miniFolioParams Options for exporting a mini folio. Can accept: Ordered array containing dataField:String, dataValue:Any Type.
   */
  exportMiniFolio(
    destination: File,
    portraitDocument: Document,
    landscapeDocument: Document,
    folioMetadata: any[],
    miniFolioParams: any[],
  ): string[]

  /**
   * Export the presets to a file.
   * @param format The preset format.
   * @param to The file to export to.
   * @param versionComments The comments for this version.
   * @param forceSave Forcibly save a version.
   */
  exportPresets(
    format: ExportPresetFormat,
    to: File,
    versionComments: string,
    forceSave?: boolean,
  ): void

  /**
   * Exports selection as assets required for cloud library.
   * @param to The path to the export file.
   */
  exportSelectionForCloudLibrary(to: File): boolean

  /**
   * Exports stroke styles or presets.
   * @param to The file to save to
   * @param strokeStyleList The list of stroke styles to save
   * @param versionComments The comment for this version
   * @param forceSave Forcibly save a version
   */
  exportStrokeStyles(
    to: File,
    strokeStyleList: StrokeStyle[],
    versionComments: string,
    forceSave?: boolean,
  ): void

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Finds glyphs that match the find what value.
   * @param reverseOrder If true, returns the results in reverse order.
   */
  findGlyph(reverseOrder: boolean): Text[]

  /**
   * Finds text that matches the find what value.
   * @param reverseOrder If true, returns the results in reverse order.
   */
  findGrep(reverseOrder: boolean): Text[]

  /**
   * Returns the locale-independent string(s) from the internal string localization database that correspond to the specified string (in the current locale).
   * @param for_ The string to search for.
   */
  findKeyStrings(for_: string): string[]

  /**
   * Finds objects that match the find what value.
   * @param reverseOrder If true, returns the results in reverse order.
   */
  findObject(reverseOrder: boolean): PageItem[]

  /**
   * Finds text that matches the find what value.
   * @param reverseOrder If true, returns the results in reverse order.
   */
  findText(reverseOrder: boolean): Text[]

  /**
   * Finds text that matches the find character type value.
   * @param reverseOrder If true, returns the results in reverse order.
   */
  findTransliterate(reverseOrder: boolean): Text[]

  /**
   * Generate schema for IDML.
   * @param to The folder path of the schema.
   * @param packageFormat If true, generate schema for package format (multiple files). Default value is false.
   */
  generateIDMLSchema(to: File, packageFormat: boolean): void

  /**
   * Get all overlays.
   * @param portraitDocumentForCheckingOverlays The portrait document for checking overlays.
   * @param landscapeDocumentForCheckingOverlays The landscape document for checking overlays.
   * @param miniFolioParams Options for exporting a mini folio. Can accept: Ordered array containing dataField:String, dataValue:Any Type.
   */
  getAllOverlays(
    portraitDocumentForCheckingOverlays: Document,
    landscapeDocumentForCheckingOverlays: Document,
    miniFolioParams: any[],
  ): any[]

  /**
   * Get a JSON string for the CCX Welcome dialog.
   * @param jsondata mode description
   */
  getCCXUserJSONData(jsondata: string): string

  /**
   * Get the current digital publishing article version number for the given parameter.
   * @param digpubArticleVersion Version string(s) to retrieve.
   */
  getDigpubArticleVersion(digpubArticleVersion: DigpubArticleVersion): string[]

  /**
   * Get the current digital publishing version number for the given parameter.
   * @param digpubVersion Version string(s) to retrieve.
   */
  getDigpubVersion(digpubVersion: DigpubVersion): string[]

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): Application[]

  /**
   * Get the resolution strategy for style conflict, false if the user cancels
   * @param charOrParaStyle Style type to look at.
   */
  getStyleConflictResolutionStrategy(charOrParaStyle: StyleType): any

  /**
   * Get the list of article viewer versions the digital publishing plugin supports.
   */
  getSupportedArticleViewerVersions(): string[]

  /**
   * Get the list of viewer versions the digital publishing plugin supports.
   */
  getSupportedViewerVersions(): string[]

  /**
   * Gets the count which will be used in the name of the next untitled document.
   */
  getUntitledCount(): number

  /**
   * Gets the decision from the user to add the content of text frame or the complete story in case of threaded text frame.
   */
  getUserChoiceForCloudTextAddition(): any

  /**
   * Imports a process color swatch from a preloaded Adobe color book.
   * @param name The process color to load.
   */
  importAdobeSwatchbookProcessColor(name: string): Color

  /**
   * Imports a spot color swatch from an Adobe color book.
   * @param name The spot color to load.
   */
  importAdobeSwatchbookSpotColor(name: string): Color

  /**
   * Imports presets from the specified file.
   * @param format The type of preset to import.
   * @param from The file to import presets from.
   */
  importFile(format: ExportPresetFormat, from: File): void

  /**
   * Imports the specified styles.
   * @param format The types of styles to import.
   * @param from The file containing the styles you want to import.
   * @param globalStrategy The resolution strategy to employ for imported styles that have the same names as existing styles.
   */
  importStyles(
    format: ImportFormat,
    from: File,
    globalStrategy?: GlobalClashResolutionStrategy,
  ): void

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Invokes InDesign's Color Picker.
   * @param space Color space RGB, CMYK or LAB
   * @param colorValue Color values
   */
  invokeColorPicker(space: ColorSpace, colorValue: number[]): string

  /**
   * Whether the app is in touch mode or not.
   */
  isAppInTouchMode(): boolean

  /**
   * Whether user has opted-in to share usage data.
   */
  isUserSharingAppUsageData(): boolean

  /**
   * Load conditions from the specified file.
   * @param from The path to the file that contains the conditions.
   * @param loadConditionSets If true, load the condition sets as well.
   */
  loadConditions(from: File, loadConditionSets: boolean): void

  /**
   * Loads the specified find/change query.
   * @param queryName The query to load.
   * @param searchMode The search mode.
   */
  loadFindChangeQuery(queryName: string, searchMode: SearchModes): void

  /**
   * Load motion preset from the specified file.
   * @param from The Flash motion preset file.
   */
  loadMotionPreset(from: File): MotionPreset

  /**
   * Load preflight profile from the specified file.
   * @param from The InDesign preflight profile file or InDesign document.
   */
  loadPreflightProfile(from: File): PreflightProfile

  /**
   * Load swatches from the specified file.
   * @param from The swatch file or InDesign document.
   */
  loadSwatches(from: File): void

  /**
   * Loads a set of XML markup tags from the specified file.
   * @param from The path to the file that contains the tags.
   */
  loadXMLTags(from: File): void

  /**
   * Gets the memory statistics from the server.
   */
  memoryStatistics(): any[]

  /**
   * Mount a Version Cue project.
   * @param serverURL The URL of the Version Cue server containing the project
   * @param projectName The name of the Version Cue project to mount
   */
  mountProject(serverURL: string, projectName: string): void

  /**
   * Opens the specified document, book, or library.
   * @param from The file path(s) to the document, book, or library.
   * @param showingWindow If true, opens the document in a window. If false, the document is opened but is not displayed in a window.
   * @param openOption How to open the document.
   */
  open(from: File | File[], showingWindow?: boolean, openOption?: OpenOptions): any

  /**
   * Opens the cloud library asset for editing.
   * @param jsondata JSON encoded information about the asset to be edited.
   */
  openCloudAssetForEdit(jsondata: string): boolean

  /**
   * Open the panel associated with the action.
   * @param id The ID.
   */
  openPanel(id: number): void

  /**
   * Package a folder into a UCF file.
   * @param sourceFolder The folder to be packaged into an IDML file. Does not validate structure of the folder pursuant to the IDML spec. Caller is responsible for making sure the files in the folder are correctly organized.
   * @param ucfFile The destination UCF file. Will be overwritten if it already exists.
   * @param mimeMediaType The MIME media type, default value identifies package as IDML.
   */
  packageUCF(sourceFolder: File, ucfFile: File, mimeMediaType?: string): void

  /**
   * Pastes data from the clipboard into the active document window.
   */
  paste(): void

  /**
   * Pastes data from the clipboard into the active document window at the same position that the data held in its original document.
   */
  pasteInPlace(): void

  /**
   * Pastes data from the clipboard into the selected object in the active document window.
   */
  pasteInto(): void

  /**
   * Pastes data (minus formatting) from the clipboard into the active document window.
   */
  pasteWithoutFormatting(): void

  /**
   * Gets the current value of the specified performance metric.
   * @param for_ The status to get from InDesign.
   */
  performanceMetric(for_: number | PerformanceMetricOptions): any

  /**
   * Gets the long name of the specified performance metric.
   * @param for_ The status to get from InDesign.
   */
  performanceMetricLongName(for_: number | PerformanceMetricOptions): string

  /**
   * Gets the short name of the specified performance metric.
   * @param for_ The status to get from InDesign.
   */
  performanceMetricShortName(for_: number | PerformanceMetricOptions): string

  /**
   * Place one or more files following the behavior of the place menu item. This may load the place gun or replace the selected object, depending on current preferences.
   * @param fileName One or more files to place.
   * @param showingOptions Whether to display the import options dialog
   * @param withProperties Initial values for properties of the placed object(s)
   */
  place(fileName: File | File[], showingOptions?: boolean, withProperties?: object): void

  /**
   * Prints the specified file(s).
   * @param from One or more file paths.
   * @param printDialog Whether to invoke the print dialog
   * @param using Printer preset to use.
   */
  print(from: File | File[], printDialog: boolean, using: PrinterPresetTypes | PrinterPreset): void

  /**
   * Quits the application.
   * @param saving The option to use for saving changes to open documents before quitting.
   */
  quit(saving?: SaveOptions): void

  /**
   * Redoes the last action.
   */
  redo(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Remove the file from most recently used files
   * @param to The file to be removed
   */
  removeFileFromRecentFiles(to: File): boolean

  /**
   * Saves the specified find/change query.
   * @param queryName The query to save.
   * @param searchMode The search mode.
   */
  saveFindChangeQuery(queryName: string, searchMode: SearchModes): void

  /**
   * Saves the specified swatch(es) to a swatchbook file.
   * @param to The swatchbook file to save to.
   * @param swatchList The swatch(es) to save.
   * @param versionComments The comment for this version.
   * @param forceSave If true, forcibly saves a version.
   */
  saveSwatches(to: File, swatchList: Swatch[], versionComments: string, forceSave?: boolean): void

  /**
   * Saves a set of tags to an external file.
   * @param to The full path to the file in which to save the tags.
   * @param versionComments The comment for this version.
   * @param forceSave If true, forcibly saves a version.
   */
  saveXMLTags(to: File, versionComments: string, forceSave?: boolean): void

  /**
   * Selects the specified object(s).
   * @param selectableItems The objects to select.
   * @param existingSelection The selection status of the Application in relation to previously selected objects.
   */
  select(
    selectableItems: object | object[] | NothingEnum | SelectAll,
    existingSelection?: SelectionOptions,
  ): void

  /**
   * Sets the application's preferences.
   * @param applicationPreferences The IDML defaults file or enumeration.
   */
  setApplicationPreferences(applicationPreferences: File | LanguageAndRegion): void

  /**
   * Set cloud libraries info.
   * @param librariesCollectionInfo JSON encoded information about cloud libraries collection
   */
  setCloudLibraryCollection(librariesCollectionInfo: string): void

  /**
   * Sets the export options for generation of a cloud asset.
   * @param maxwidth The maximum width of the thumbnail generated in pixels.
   * @param maxheight The maximum height of the thumbnail generated in pixels.
   */
  setCloudLibraryOptions(maxwidth: number, maxheight: number): void

  /**
   * Sets the count which will be used in the name of the next untitled document.
   * @param untitledCount The count to be used in the name of the next untitled document. Only positive values are expected
   */
  setUntitledCount(untitledCount: number): void

  /**
   * Tile all document windows
   */
  tileWindows(): void

  /**
   * Generates a string which, if executed, will return the Application.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string

  /**
   * Toggles the visibility of the entire panel system.
   */
  togglePanelSystemVisibility(): void

  /**
   * Translates a key string into localized form based on current application locale.
   * @param for_ The key string to translate
   */
  translateKeyString(for_: string): string

  /**
   * Undoes the last action.
   */
  undo(): void

  /**
   * Unpackage a UCF file into a folder structure.
   * @param ucfFile The UCF file to be unpackaged.
   * @param destinationFolder The folder where you would like the UCF file unpackaged to. Will be created if it does not exist.
   */
  unpackageUCF(ucfFile: File, destinationFolder: File): void

  /**
   * Forces a check for new fonts in the various Fonts folders.
   */
  updateFonts(): void

  /**
   * Waits for all the background tasks to finish.
   */
  waitForAllTasks(): TaskState[]
}

/**
 * background task
 */
declare class BackgroundTask {
  /**
   * The alerts encountered while running this task object. Can return: Array of Array of 2 TaskAlertType enumerators or Strings.
   */
  readonly alerts: any[]

  /**
   * The document name on which this task operates.
   */
  readonly documentName: string

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The unique ID of the BackgroundTask.
   */
  readonly id: number

  /**
   * The index of the BackgroundTask within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * The name of the BackgroundTask.
   */
  readonly name: string

  /**
   * The parent of the BackgroundTask (a Application).
   */
  readonly parent: Application

  /**
   * Progress information for this task.
   */
  readonly percentDone: number

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * The current status of this task object.
   */
  readonly status: TaskState

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Cancels the specified background task.
   */
  cancelTask(): void

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): BackgroundTask[]

  /**
   * Queries for a particular property in the task metadata.
   * @param name The task property being queried
   */
  queryProperty(name: string): any

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the BackgroundTask.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string

  /**
   * Waits for the task to finish.
   */
  waitForTask(): TaskState
}

/**
 * A collection of background task objects.
 */
declare class BackgroundTasks {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the BackgroundTask with the specified index.
   * @param index The index.
   */
  [index: number]: BackgroundTask

  /**
   * Returns any BackgroundTask in the collection.
   */
  anyItem(): BackgroundTask

  /**
   * Displays the number of elements in the BackgroundTask.
   */
  count(): number

  /**
   * Returns every BackgroundTask in the collection.
   */
  everyItem(): BackgroundTask

  /**
   * Returns the first BackgroundTask in the collection.
   */
  firstItem(): BackgroundTask

  /**
   * Returns the BackgroundTask with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): BackgroundTask

  /**
   * Returns the BackgroundTask with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): BackgroundTask

  /**
   * Returns the BackgroundTask with the specified name.
   * @param name The name.
   */
  itemByName(name: string): BackgroundTask

  /**
   * Returns the BackgroundTasks within the specified range.
   * @param from The BackgroundTask, index, or name at the beginning of the range.
   * @param to The BackgroundTask, index, or name at the end of the range.
   */
  itemByRange(
    from: BackgroundTask | number | string,
    to: BackgroundTask | number | string,
  ): BackgroundTask[]

  /**
   * Returns the last BackgroundTask in the collection.
   */
  lastItem(): BackgroundTask

  /**
   * Returns the middle BackgroundTask in the collection.
   */
  middleItem(): BackgroundTask

  /**
   * Returns the BackgroundTask whose index follows the specified BackgroundTask in the collection.
   * @param obj The BackgroundTask whose index comes before the desired BackgroundTask.
   */
  nextItem(obj: BackgroundTask): BackgroundTask

  /**
   * Returns the BackgroundTask with the index previous to the specified index.
   * @param obj The index of the BackgroundTask that follows the desired BackgroundTask.
   */
  previousItem(obj: BackgroundTask): BackgroundTask

  /**
   * Generates a string which, if executed, will return the BackgroundTask.
   */
  toSource(): string
}

/**
 * A document.
 */
declare class Document {
  /**
   * Dispatched after the Document becomes active. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_ACTIVATE: string

  /**
   * Dispatched after a Document is exported. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_EXPORT: string

  /**
   * Dispatched after importing a file into a Document. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_IMPORT: string

  /**
   * Dispatched after one or more links in the Document have been added, deleted, or modified. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_LINKS_CHANGED: string

  /**
   * Dispatched after a Document is created. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_NEW: string

  /**
   * Dispatched after a Document is opened. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_OPEN: string

  /**
   * Dispatched after a Document is printed. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_PRINT: string

  /**
   * Dispatched after a Document is reverted. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_REVERT: string

  /**
   * Dispatched after a Document is saved. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_SAVE: string

  /**
   * Dispatched after a Document is saved under a new name. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_SAVE_AS: string

  /**
   * Dispatched after a copy of a Document is saved. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_SAVE_A_COPY: string

  /**
   * Dispatched before a Document is closed. This event bubbles. This event is not cancelable.
   */
  static readonly BEFORE_CLOSE: string

  /**
   * Dispatched before the Document becomes inactive. This event bubbles. This event is not cancelable.
   */
  static readonly BEFORE_DEACTIVATE: string

  /**
   * Dispatched before a Document is exported. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_EXPORT: string

  /**
   * Dispatched before importing a file into a Document. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_IMPORT: string

  /**
   * Dispatched before a Document is printed. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_PRINT: string

  /**
   * Dispatched before a Document is reverted. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_REVERT: string

  /**
   * Dispatched before a Document is saved. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_SAVE: string

  /**
   * Dispatched before a Document is saved under a new name. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_SAVE_AS: string

  /**
   * Dispatched before a copy of a Document is saved. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_SAVE_A_COPY: string

  /**
   * Dispatched after a Document export is canceled or fails. This event bubbles. This event is not cancelable.
   */
  static readonly FAILED_EXPORT: string

  /**
   * If true, uses LAB alternates for spot colors when available.
   */
  accurateLABSpots: boolean

  /**
   * The active layer.
   */
  activeLayer: Layer | string

  /**
   * The active preflight process for this document.
   */
  readonly activeProcess: PreflightProcess

  /**
   * The rendering intent for colors that result from transparency interactions on the page after blending.
   */
  afterBlendingIntent: RenderingIntent

  /**
   * All Cell styles
   */
  readonly allCellStyles: CellStyle[]

  /**
   * Lists all character styles (regardless of their group).
   */
  readonly allCharacterStyles: CharacterStyle[]

  /**
   * Lists all graphics contained by the Document.
   */
  readonly allGraphics: Graphic[]

  /**
   * All object styles contained by the Document.
   */
  readonly allObjectStyles: ObjectStyle[]

  /**
   * Lists all page items contained by the Document.
   */
  readonly allPageItems: PageItem[]

  /**
   * Lists all paragraph styles (regardless of their group).
   */
  readonly allParagraphStyles: ParagraphStyle[]

  /**
   * All Table styles
   */
  readonly allTableStyles: TableStyle[]

  /**
   * Anchored object default settings.
   */
  readonly anchoredObjectDefaults: AnchoredObjectDefault

  /**
   * Anchored object settings.
   */
  readonly anchoredObjectSettings: AnchoredObjectSetting

  /**
   * A collection of articles.
   */
  readonly articles: Articles

  /**
   * A collection of assignments.
   */
  readonly assignments: Assignments

  /**
   * The XML element associated with the Document.
   */
  readonly associatedXMLElement: XMLItem

  /**
   * Baseline frame grid option settings.
   */
  readonly baselineFrameGridOptions: BaselineFrameGridOption

  /**
   * A collection of bookmarks.
   */
  readonly bookmarks: Bookmarks

  /**
   * Button preference settings.
   */
  readonly buttonPreferences: ButtonPreference

  /**
   * A collection of buttons.
   */
  readonly buttons: Buttons

  /**
   * A collection of cell style groups.
   */
  readonly cellStyleGroups: CellStyleGroups

  /**
   * A collection of cell style mappings.
   */
  readonly cellStyleMappings: CellStyleMappings

  /**
   * A collection of cell styles.
   */
  readonly cellStyles: CellStyles

  /**
   * Chapter numbering preferences.
   */
  readonly chapterNumberPreferences: ChapterNumberPreference

  /**
   * A collection of char style mappings.
   */
  readonly charStyleMappings: CharStyleMappings

  /**
   * A collection of character style groups.
   */
  readonly characterStyleGroups: CharacterStyleGroups

  /**
   * A collection of character styles.
   */
  readonly characterStyles: CharacterStyles

  /**
   * A collection of checkboxes.
   */
  readonly checkBoxes: CheckBoxes

  /**
   * CJK grid preference settings.
   */
  readonly cjkGridPreferences: CjkGridPreference

  /**
   * The policy for handling colors in a CMYK color model, including reading and embedding color profiles, mismatches between embedded color profiles and the working space, and moving colors from one document to another.
   */
  cmykPolicy: ColorSettingsPolicy

  /**
   * The current CMYK profile.
   */
  cmykProfile: string

  /**
   * A list of valid CMYK profiles.
   */
  readonly cmykProfileList: string[]

  /**
   * A collection of color groups.
   */
  readonly colorGroups: ColorGroups

  /**
   * A collection of colors.
   */
  readonly colors: Colors

  /**
   * A collection of comboboxes.
   */
  readonly comboBoxes: ComboBoxes

  /**
   * A collection of composite fonts.
   */
  readonly compositeFonts: CompositeFonts

  /**
   * A collection of condition sets for conditional text.
   */
  readonly conditionSets: ConditionSets

  /**
   * Conditional text preferences.
   */
  readonly conditionalTextPreferences: ConditionalTextPreference

  /**
   * A collection of conditions for conditional text.
   */
  readonly conditions: Conditions

  /**
   * If true, the Document was converted.
   */
  readonly converted: boolean

  /**
   * A collection of cross reference formats.
   */
  readonly crossReferenceFormats: CrossReferenceFormats

  /**
   * A collection of cross reference text sources.
   */
  readonly crossReferenceSources: CrossReferenceSources

  /**
   * A collection of dashed stroke styles.
   */
  readonly dashedStrokeStyles: DashedStrokeStyles

  /**
   * A collection of data merge image placeholders.
   */
  readonly dataMergeImagePlaceholders: DataMergeImagePlaceholders

  /**
   * The data merge option properties that define the data merge.
   */
  readonly dataMergeOptions: DataMergeOption

  /**
   * Data merge properties that define data merge fields and preferences.
   */
  readonly dataMergeProperties: DataMerge

  /**
   * A collection of data merge QR code placeholders.
   */
  readonly dataMergeQrcodePlaceholders: DataMergeQrcodePlaceholders

  /**
   * A collection of data merge text placeholders.
   */
  readonly dataMergeTextPlaceholders: DataMergeTextPlaceholders

  /**
   * The rendering intent for bitmap images.
   */
  defaultImageIntent: RenderingIntent

  /**
   * User dictionary preference settings.
   */
  readonly dictionaryPreferences: DictionaryPreference

  /**
   * Document preference settings.
   */
  readonly documentPreferences: DocumentPreference

  /**
   * A collection of dotted stroke styles.
   */
  readonly dottedStrokeStyles: DottedStrokeStyles

  /**
   * A collection of DTDs.
   */
  readonly dtds: DTDs

  /**
   * The Version Cue editing state of the file.
   */
  readonly editingState: EditingState

  /**
   * Endnote option settings.
   */
  readonly endnoteOptions: EndnoteOption

  /**
   * A collection of endnote text frames.
   */
  readonly endnoteTextFrames: EndnoteTextFrames

  /**
   * EPSTexts
   */
  readonly epstexts: EPSTexts

  /**
   * EPub export preference settings.
   */
  readonly epubExportPreferences: EPubExportPreference

  /**
   * EPub fixed layout export preference settings.
   */
  readonly epubFixedLayoutExportPreferences: EPubFixedLayoutExportPreference

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The default export for web preferences.
   */
  readonly exportForWebPreferences: ExportForWebPreference

  /**
   * The full path to the file.
   */
  readonly filePath: File

  /**
   * A collection of fonts.
   */
  readonly fonts: Fonts

  /**
   * Footnote option settings.
   */
  readonly footnoteOptions: FootnoteOption

  /**
   * A collection of form fields.
   */
  readonly formFields: FormFields

  /**
   * The frame fitting option to apply to placed or pasted content. Can be applied to a frame, object style, or document or to the application.
   */
  readonly frameFittingOptions: FrameFittingOption

  /**
   * The full path to the Document, including the name of the Document.
   */
  readonly fullName: File

  /**
   * Galley preference settings.
   */
  readonly galleyPreferences: GalleyPreference

  /**
   * A collection of gradients.
   */
  readonly gradients: Gradients

  /**
   * A collection of graphic lines.
   */
  readonly graphicLines: GraphicLines

  /**
   * Grid preference settings.
   */
  readonly gridPreferences: GridPreference

  /**
   * A collection of groups.
   */
  readonly groups: Groups

  /**
   * Guide preference settings.
   */
  readonly guidePreferences: GuidePreference

  /**
   * A collection of guides.
   */
  readonly guides: Guides

  /**
   * HTML export preference settings.
   */
  readonly htmlExportPreferences: HTMLExportPreference

  /**
   * HTML FXL export preference settings.
   */
  readonly htmlFXLExportPreferences: HTMLFXLExportPreference

  /**
   * A collection of hyperlink external page destinations.
   */
  readonly hyperlinkExternalPageDestinations: HyperlinkExternalPageDestinations

  /**
   * A collection of hyperlink page destinations.
   */
  readonly hyperlinkPageDestinations: HyperlinkPageDestinations

  /**
   * A collection of hyperlink page item sources.
   */
  readonly hyperlinkPageItemSources: HyperlinkPageItemSources

  /**
   * A collection of hyperlink text destinations.
   */
  readonly hyperlinkTextDestinations: HyperlinkTextDestinations

  /**
   * A collection of hyperlink text sources.
   */
  readonly hyperlinkTextSources: HyperlinkTextSources

  /**
   * A collection of hyperlink URL destinations.
   */
  readonly hyperlinkURLDestinations: HyperlinkURLDestinations

  /**
   * A collection of hyperlinks.
   */
  readonly hyperlinks: Hyperlinks

  /**
   * A collection of hyphenation exceptions lists.
   */
  readonly hyphenationExceptions: HyphenationExceptions

  /**
   * The unique ID of the Document.
   */
  readonly id: number

  /**
   * The index of the Document within its containing object.
   */
  readonly index: number

  /**
   * The index options properties that define how an index is formatted.
   */
  readonly indexGenerationOptions: IndexOptions

  /**
   * A collection of indexes.
   */
  readonly indexes: Indexes

  /**
   * A collection of indexing sort options.
   */
  readonly indexingSortOptions: IndexingSortOptions

  /**
   * A collection of inks.
   */
  readonly inks: Inks

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A collection of kinsoku tables.
   */
  readonly kinsokuTables: KinsokuTables

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * A collection of languages.
   */
  readonly languages: Languages

  /**
   * A collection of layers.
   */
  readonly layers: Layers

  /**
   * Layout sdjustment preference settings.
   */
  readonly layoutAdjustmentPreferences: LayoutAdjustmentPreference

  /**
   * Default properties specific to layout grids.
   */
  readonly layoutGridData: LayoutGridDataInformation

  /**
   * A collection of layout windows.
   */
  readonly layoutWindows: LayoutWindows

  /**
   * Linked Page Item options
   */
  readonly linkedPageItemOptions: LinkedPageItemOption

  /**
   * Linked story options
   */
  readonly linkedStoryOptions: LinkedStoryOption

  /**
   * A collection of links.
   */
  readonly links: Links

  /**
   * A collection of listboxes.
   */
  readonly listBoxes: ListBoxes

  /**
   * Margin preference settings.
   */
  readonly marginPreferences: MarginPreference

  /**
   * A collection of master spreads.
   */
  readonly masterSpreads: MasterSpreads

  /**
   * Metadata preference settings.
   */
  readonly metadataPreferences: MetadataPreference

  /**
   * A collection of mixed ink groups.
   */
  readonly mixedInkGroups: MixedInkGroups

  /**
   * A collection of mixed inks.
   */
  readonly mixedInks: MixedInks

  /**
   * If true, the Document has been modified since it was last saved.
   */
  readonly modified: boolean

  /**
   * A collection of mojikumi tables.
   */
  readonly mojikumiTables: MojikumiTables

  /**
   * Mojikumi user interface preference settings.
   */
  readonly mojikumiUIPreferences: MojikumiUiPreference

  /**
   * A collection of Motion presets.
   */
  readonly motionPresets: MotionPresets

  /**
   * A collection of multi-state objects.
   */
  readonly multiStateObjects: MultiStateObjects

  /**
   * The name of the Document.
   */
  name: string

  /**
   * A collection of named grids.
   */
  readonly namedGrids: NamedGrids

  /**
   * A collection of numbered lists.
   */
  readonly numberingLists: NumberingLists

  /**
   * A collection of object style groups.
   */
  readonly objectStyleGroups: ObjectStyleGroups

  /**
   * A collection of object styles.
   */
  readonly objectStyles: ObjectStyles

  /**
   * A collection of ellipses.
   */
  readonly ovals: Ovals

  /**
   * The default page item formatting for the Document.
   */
  readonly pageItemDefaults: PageItemDefault

  /**
   * The page items collection, which can be used to process all page items in a container (such as a document, page, or group), regardless of type.
   */
  readonly pageItems: PageItems

  /**
   * A collection of pages.
   */
  readonly pages: Pages

  /**
   * A collection of para style mappings.
   */
  readonly paraStyleMappings: ParaStyleMappings

  /**
   * A collection of paragraph destinations.
   */
  readonly paragraphDestinations: ParagraphDestinations

  /**
   * A collection of paragraph style groups.
   */
  readonly paragraphStyleGroups: ParagraphStyleGroups

  /**
   * A collection of paragraph styles.
   */
  readonly paragraphStyles: ParagraphStyles

  /**
   * The parent of the Document (a Application).
   */
  readonly parent: Application

  /**
   * Pasteboard preference settings.
   */
  readonly pasteboardPreferences: PasteboardPreference

  /**
   * The place gun.
   */
  readonly placeGuns: PlaceGun

  /**
   * A collection of polygons.
   */
  readonly polygons: Polygons

  /**
   * A collection of preferences objects.
   */
  readonly preferences: Preferences

  /**
   * Preflight option settings.
   */
  readonly preflightOptions: PreflightOption

  /**
   * A collection of preflight profiles.
   */
  readonly preflightProfiles: PreflightProfiles

  /**
   * Booklet printing options.
   */
  readonly printBookletOptions: PrintBookletOption

  /**
   * Print booklet preferences.
   */
  readonly printBookletPrintPreferences: PrintBookletPrintPreference

  /**
   * Print preference settings.
   */
  readonly printPreferences: PrintPreference

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * Publish export preference settings.
   */
  readonly publishExportPreferences: PublishExportPreference

  /**
   * A collection of radio buttons.
   */
  readonly radioButtons: RadioButtons

  /**
   * If true, the Document is read-only.
   */
  readonly readOnly: boolean

  /**
   * If true, the Document was recovered.
   */
  readonly recovered: boolean

  /**
   * A collection of rectangles.
   */
  readonly rectangles: Rectangles

  /**
   * The names of the items in the redo stack.
   */
  readonly redoHistory: string[]

  /**
   * The name of the action on the top of the redo stack.
   */
  readonly redoName: string

  /**
   * The policy for handling colors in an RGB color model, including reading and embedding color profiles, handling mismatches between embedded color profiles and the working space, and moving colors from one document to another.
   */
  rgbPolicy: ColorSettingsPolicy

  /**
   * The current RGB profile.
   */
  rgbProfile: string

  /**
   * A list of valid RGB profiles.
   */
  readonly rgbProfileList: string[]

  /**
   * If true, the Document has been saved since it was created.
   */
  readonly saved: boolean

  /**
   * A collection of sections.
   */
  readonly sections: Sections

  /**
   * The selected page item(s).
   */
  readonly selectedPageItems: object[] | object | NothingEnum

  /**
   * The selected object(s).
   */
  selection: object[] | object | NothingEnum

  /**
   * The key object of the selection.
   */
  selectionKeyObject: PageItem | NothingEnum

  /**
   * A collection of signature fields.
   */
  readonly signatureFields: SignatureFields

  /**
   * The rendering intent for all vector art (areas of solid color) in native objects.
   */
  solidColorIntent: RenderingIntent

  /**
   * The spline items collection.
   */
  readonly splineItems: SplineItems

  /**
   * A collection of spreads.
   */
  readonly spreads: Spreads

  /**
   * A collection of stories.
   */
  readonly stories: Stories

  /**
   * Default properties specific to frame grids.
   */
  readonly storyGridData: StoryGridDataInformation

  /**
   * Story preference settings.
   */
  readonly storyPreferences: StoryPreference

  /**
   * A collection of story windows.
   */
  readonly storyWindows: StoryWindows

  /**
   * A collection of striped stroke styles.
   */
  readonly stripedStrokeStyles: StripedStrokeStyles

  /**
   * A collection of stroke styles.
   */
  readonly strokeStyles: StrokeStyles

  /**
   * A collection of swatches.
   */
  readonly swatches: Swatches

  /**
   * A collection of table style groups.
   */
  readonly tableStyleGroups: TableStyleGroups

  /**
   * A collection of table style mappings.
   */
  readonly tableStyleMappings: TableStyleMappings

  /**
   * A collection of table styles.
   */
  readonly tableStyles: TableStyles

  /**
   * Tagged PDF preferences.
   */
  readonly taggedPDFPreferences: TaggedPDFPreference

  /**
   * A collection of text boxes.
   */
  readonly textBoxes: TextBoxes

  /**
   * Text default settings.
   */
  readonly textDefaults: TextDefault

  /**
   * Text frame footnote options settings.
   */
  readonly textFrameFootnoteOptions: TextFrameFootnoteOptionsObject

  /**
   * Text frame preference settings.
   */
  readonly textFramePreferences: TextFramePreference

  /**
   * A collection of text frames.
   */
  readonly textFrames: TextFrames

  /**
   * Text preference settings.
   */
  readonly textPreferences: TextPreference

  /**
   * A collection of text variables.
   */
  readonly textVariables: TextVariables

  /**
   * The text wrap preference properties that define the default formatting for wrapping text around objects.
   */
  readonly textWrapPreferences: TextWrapPreference

  /**
   * A collection of tints.
   */
  readonly tints: Tints

  /**
   * A collection of TOC styles.
   */
  readonly tocStyles: TOCStyles

  /**
   * Transparency preference settings.
   */
  readonly transparencyPreferences: TransparencyPreference

  /**
   * A collection of trap presets.
   */
  readonly trapPresets: TrapPresets

  /**
   * The names of the items in the undo stack.
   */
  readonly undoHistory: string[]

  /**
   * The name of the action on the top of the undo stack.
   */
  readonly undoName: string

  /**
   * The swatches that are not being used.
   */
  readonly unusedSwatches: Swatch[]

  /**
   * A collection of XML validation errors.
   */
  readonly validationErrors: ValidationErrors

  /**
   * The Version Cue version state of the file.
   */
  readonly versionState: VersionState

  /**
   * View preference settings.
   */
  readonly viewPreferences: ViewPreference

  /**
   * If true, the Document is visible.
   */
  readonly visible: boolean

  /**
   * Watermark preferences
   */
  readonly watermarkPreferences: WatermarkPreference

  /**
   * A collection of windows.
   */
  readonly windows: Windows

  /**
   * A collection of XML comments.
   */
  readonly xmlComments: XMLComments

  /**
   * A collection of XML elements.
   */
  readonly xmlElements: XMLElements

  /**
   * A collection of XML export maps.
   */
  readonly xmlExportMaps: XMLExportMaps

  /**
   * XML export preference settings.
   */
  readonly xmlExportPreferences: XMLExportPreference

  /**
   * A collection of XML import maps.
   */
  readonly xmlImportMaps: XMLImportMaps

  /**
   * XML import preference settings.
   */
  readonly xmlImportPreferences: XMLImportPreference

  /**
   * A collection of XML instructions.
   */
  readonly xmlInstructions: XMLInstructions

  /**
   * A collection of XML items.
   */
  readonly xmlItems: XMLItems

  /**
   * The XML preference settings.
   */
  readonly xmlPreferences: XMLPreference

  /**
   * A collection of xml stories.
   */
  readonly xmlStories: XmlStories

  /**
   * A collection of XML tags.
   */
  readonly xmlTags: XMLTags

  /**
   * XML view preference settings.
   */
  readonly xmlViewPreferences: XMLViewPreference

  /**
   * The ruler origin, specified as page coordinates in the format [x, y].
   */
  zeroPoint: string[] | number[]

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Align page items.
   * @param alignDistributeItems The page items to align or distribute.
   * @param alignOption The type of alignment to apply.
   * @param alignDistributeBounds The bounds within which to align or distribute the page items.
   * @param reference The reference or key object to align to distribute relative to. Required when 'align distribute bounds' specifies 'key object'.
   */
  align(
    alignDistributeItems: PageItem[],
    alignOption: AlignOptions,
    alignDistributeBounds: AlignDistributeBounds,
    reference: PageItem,
  ): void

  /**
   * asynchronously exports the object(s) to a file.
   * @param format The export format, specified as an enumeration value or as an extension that appears in the Save as type or Format menu in the Export dialog.
   * @param to The path to the export file.
   * @param showingOptions If true, displays the export options dialog.
   * @param using The export style.
   * @param versionComments The comment for this version.
   * @param forceSave If true, forcibly saves a version.
   */
  asynchronousExportFile(
    format: ExportFormat | string,
    to: File,
    showingOptions?: boolean,
    using?: PDFExportPreset,
    versionComments?: string,
    forceSave?: boolean,
  ): BackgroundTask

  /**
   * Change comoser to optyca
   */
  changeComposer(): void

  /**
   * Finds glyphs that match the find what value and replaces the glyphs with the change to value.
   * @param reverseOrder If true, returns the results in reverse order.
   */
  changeGlyph(reverseOrder: boolean): Text[]

  /**
   * Finds text that matches the find what value and replaces the text with the change to value.
   * @param reverseOrder If true, returns the results in reverse order.
   */
  changeGrep(reverseOrder: boolean): Text[]

  /**
   * Finds objects that match the find what value and replace the objects with the change to value.
   * @param reverseOrder If true, returns the results in reverse order.
   */
  changeObject(reverseOrder: boolean): PageItem[]

  /**
   * Finds text that matches the find what value and replaces the text with the change to value.
   * @param reverseOrder If true, returns the results in reverse order.
   */
  changeText(reverseOrder: boolean): Text[]

  /**
   * Finds text that matches the find character type value and replaces the text with the change character type value.
   * @param reverseOrder If true, returns the results in reverse order.
   */
  changeTransliterate(reverseOrder: boolean): Text[]

  /**
   * Check in to Version Cue.
   * @param versionComments The comment for this version
   * @param forceSave Forcibly save a version
   */
  checkIn(versionComments: string, forceSave?: boolean): void

  /**
   * Removes the frame fittings options and resets it to the initial state.
   */
  clearFrameFittingOptions(): void

  /**
   * Close the Document
   * @param saving Whether to save changes before closing the Document
   * @param savingIn The file in which to save the Document
   * @param versionComments The comment for this version
   * @param forceSave Forcibly save a version
   */
  close(saving?: SaveOptions, savingIn?: File, versionComments?: string, forceSave?: boolean): void

  /**
   * Transforms color values
   * @param colorValue source color value
   * @param sourceColorSpace source color space
   * @param destinationColorSpace destination color space
   */
  colorTransform(
    colorValue: number[],
    sourceColorSpace: ColorSpace,
    destinationColorSpace: ColorSpace,
  ): number[]

  /**
   * Creates an alternate layout for a given list of spreads.
   * @param spreadItems The spreads to create an alternate layout for.
   * @param name The name of the alternate layout. Note: This is used for the named layout for the new section for the duplicated spreads.
   * @param width The width of the pages created.
   * @param height The height of the pages created.
   * @param createTextStyles Whether to create new text styles.
   * @param linkTextStories Whether to create linked text for duplicated text stories.
   * @param layoutRule What layout rule to set on the pages.
   */
  createAlternateLayout(
    spreadItems: Spread[],
    name: string,
    width: string | number,
    height: string | number,
    createTextStyles: boolean,
    linkTextStories: boolean,
    layoutRule: LayoutRuleOptions,
  ): void

  /**
   * Create Email QR Code on the page item or document
   * @param emailAddress QR code Email Address
   * @param subject QR code Email Subject
   * @param body QR code Email Body Message
   * @param qrCodeSwatch Swatch to be applied on generated QR Code Graphic .
   * @param withProperties Initial values for properties of the new Document. Above parameters can also be passed as properties
   */
  createEmailQRCode(
    emailAddress: string,
    subject: string,
    body: string,
    qrCodeSwatch: Swatch | string,
    withProperties: string[],
  ): void

  /**
   * Create Hyperlink QR Code on the page item or document
   * @param urlLink QR code Hyperlink URL
   * @param qrCodeSwatch Swatch to be applied on generated QR Code Graphic .
   * @param withProperties Initial values for properties of the new Document. Above parameters can also be passed as properties
   */
  createHyperlinkQRCode(
    urlLink: string,
    qrCodeSwatch: Swatch | string,
    withProperties: string[],
  ): void

  /**
   * Create a missing font object.
   * @param fontFamily The name of the font family
   * @param fontStyleName The name of the font style.
   * @param postscriptName The PostScript name of the font.
   */
  createMissingFontObject(fontFamily: string, fontStyleName: string, postscriptName: string): Font

  /**
   * Create Plain Text QR Code on the page item
   * @param plainText QR code Plain Text
   * @param qrCodeSwatch Swatch to be applied on generated QR Code Graphic .
   * @param withProperties Initial values for properties of the new Document. Above parameters can also be passed as properties
   */
  createPlainTextQRCode(
    plainText: string,
    qrCodeSwatch: Swatch | string,
    withProperties: string[],
  ): void

  /**
   * Creates a table of contents.
   * @param using The TOC style to use to define the content, title, and format of the table of contents.
   * @param replacing If true, replaces the existing TOC.
   * @param fromBook The book whose documents to include in the TOC.
   * @param placePoint The point at which to place the TOC story, specified as page coordinates in the format [x, y].
   * @param includeOverset If true, includes overset text TOC entries in the TOC.
   * @param destinationLayer The layer on which to place the TOC.
   */
  createTOC(
    using: TOCStyle,
    replacing?: boolean,
    fromBook?: Book,
    placePoint?: string[] | number[],
    includeOverset?: boolean,
    destinationLayer?: Layer,
  ): Story[]

  /**
   * Create Text Msg QR Code on the page item or document
   * @param cellNumber QR code Text Phone Number
   * @param textMessage QR code Text Message
   * @param qrCodeSwatch Swatch to be applied on generated QR Code Graphic .
   * @param withProperties Initial values for properties of the new Document. Above parameters can also be passed as properties
   */
  createTextMsgQRCode(
    cellNumber: string,
    textMessage: string,
    qrCodeSwatch: Swatch | string,
    withProperties: string[],
  ): void

  /**
   * Create Business Card QR Code on the page item or load on document's placegun
   * @param firstName QR code Business Card First Name
   * @param lastName QR code Business Card Last Name
   * @param jobTitle QR code Business Card Title
   * @param cellPhone QR code Business Card Cell Phone Number
   * @param phone QR code Business Card Phone Number
   * @param email QR code Business Card Email Address
   * @param organisation QR code Business Card Organisation
   * @param streetAddress QR code Business Card Street Address
   * @param city QR code Business Card City
   * @param adrState QR code Business Card State
   * @param country QR code Business Card Country
   * @param postalCode QR code Business Card Postal Code
   * @param website QR code Business Card URL
   * @param qrCodeSwatch Swatch to be applied on generated QR Code Graphic .
   * @param withProperties Initial values for properties of the new Document. Above parameters can also be passed as properties
   */
  createVCardQRCode(
    firstName: string,
    lastName: string,
    jobTitle: string,
    cellPhone: string,
    phone: string,
    email: string,
    organisation: string,
    streetAddress: string,
    city: string,
    adrState: string,
    country: string,
    postalCode: string,
    website: string,
    qrCodeSwatch: Swatch | string,
    withProperties: string[],
  ): void

  /**
   * Deletes an alternate layout.
   * @param name The name of the alternate layout to delete.
   */
  deleteAlternateLayout(name: string): void

  /**
   * Deletes unused XML markup tags.
   */
  deleteUnusedTags(): void

  /**
   * Distribute page items.
   * @param alignDistributeItems The page items to align or distribute
   * @param distributeOption The type of distribution to apply.
   * @param alignDistributeBounds The bounds within which to align or distribute the page items.
   * @param useDistributeMeasurement If true, distribute space between page items. When this property is true, the bounds setting is ignored.
   * @param absoluteDistributeMeasurement The distance to use when distributing page items. Required when 'align distribute bounds' specifies 'key object'.
   * @param reference The reference or key object to align to distribute relative to. Required when 'align distribute bounds' specifies 'key object'.
   */
  distribute(
    alignDistributeItems: PageItem[],
    distributeOption: DistributeOptions,
    alignDistributeBounds: AlignDistributeBounds,
    useDistributeMeasurement: boolean,
    absoluteDistributeMeasurement: string | number,
    reference: PageItem,
  ): void

  /**
   * Embed this profile to the document.
   * @param using The preflight profile to embed.
   */
  embed(using: string | PreflightProfile): PreflightProfile

  /**
   * Exports the object(s) to a file.
   * @param format The export format, specified as an enumeration value or as an extension that appears in the Save as type or Format menu in the Export dialog.
   * @param to The path to the export file.
   * @param showingOptions If true, displays the export options dialog.
   * @param using The export style.
   * @param versionComments The comment for this version.
   * @param forceSave If true, forcibly saves a version.
   */
  exportFile(
    format: ExportFormat | string,
    to: File,
    showingOptions?: boolean,
    using?: PDFExportPreset,
    versionComments?: string,
    forceSave?: boolean,
  ): void

  /**
   * Exports as assets required for cloud library.
   * @param jsondata JSON encoded information about the export.
   */
  exportForCloudLibrary(jsondata: string): boolean

  /**
   * Exports selected page items to snippet on the destination file path.
   * @param to The path to the export file.
   */
  exportPageItemsSelectionToSnippet(to: File): void

  /**
   * Exports input page item ids to snippet on the destination file path.
   * @param to The path to the export file.
   * @param pageItemIds The array of the page item ids.
   */
  exportPageItemsToSnippet(to: File, pageItemIds: number[]): void

  /**
   * Exports stroke styles or presets.
   * @param to The file to save to
   * @param strokeStyleList The list of stroke styles to save
   * @param versionComments The comment for this version
   * @param forceSave Forcibly save a version
   */
  exportStrokeStyles(
    to: File,
    strokeStyleList: StrokeStyle[],
    versionComments: string,
    forceSave?: boolean,
  ): void

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Finds glyphs that match the find what value.
   * @param reverseOrder If true, returns the results in reverse order.
   */
  findGlyph(reverseOrder: boolean): Text[]

  /**
   * Finds text that matches the find what value.
   * @param reverseOrder If true, returns the results in reverse order.
   */
  findGrep(reverseOrder: boolean): Text[]

  /**
   * Finds objects that match the find what value.
   * @param reverseOrder If true, returns the results in reverse order.
   */
  findObject(reverseOrder: boolean): PageItem[]

  /**
   * Finds text that matches the find what value.
   * @param reverseOrder If true, returns the results in reverse order.
   */
  findText(reverseOrder: boolean): Text[]

  /**
   * Finds text that matches the find character type value.
   * @param reverseOrder If true, returns the results in reverse order.
   */
  findTransliterate(reverseOrder: boolean): Text[]

  /**
   * This will return an array of structs containing information about the alternate layouts.
   * @param resolveMaster Resolves layout policy if setting is "use master" - default is true.
   */
  getAlternateLayoutsForFolio(resolveMaster: boolean): any[]

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): Document[]

  /**
   * Selected text direction
   */
  getSelectedTextDirection(): TextDirection

  /**
   * Get the resolution strategy for style conflict, false if the user cancels
   * @param charOrParaStyle Style type to look at.
   */
  getStyleConflictResolutionStrategy(charOrParaStyle: StyleType): any

  /**
   * Imports a process color swatch from a preloaded Adobe color book.
   * @param name The process color to load.
   */
  importAdobeSwatchbookProcessColor(name: string): Color

  /**
   * Imports a spot color swatch from an Adobe color book.
   * @param name The spot color to load.
   */
  importAdobeSwatchbookSpotColor(name: string): Color

  /**
   * Imports a DTD to use for validation.
   * @param from The path to the DTD file.
   */
  importDtd(from: File): void

  /**
   * Imports the cross reference formats from specified file.
   * @param from The file whose formats to import.
   */
  importFormats(from: File): void

  /**
   * Imports the specified styles.
   * @param format The types of styles to import.
   * @param from The file containing the styles you want to import.
   * @param globalStrategy The resolution strategy to employ for imported styles that have the same names as existing styles.
   */
  importStyles(
    format: ImportFormat,
    from: File,
    globalStrategy?: GlobalClashResolutionStrategy,
  ): void

  /**
   * Imports the specified XML file into an InDesign document.
   * @param from The XML file.
   */
  importXML(from: File): void

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Load conditions from the specified file.
   * @param from The path to the file that contains the conditions.
   * @param loadConditionSets If true, load the condition sets as well.
   */
  loadConditions(from: File, loadConditionSets: boolean): void

  /**
   * Load masterpages from an InDesign file.
   * @param from The InDesign file to load the masters from.
   * @param globalStrategyForMasterPage the global clash resolution strategy for load master page
   */
  loadMasters(
    from: File,
    globalStrategyForMasterPage?: GlobalClashResolutionStrategyForMasterPage,
  ): void

  /**
   * Load swatches from the specified file.
   * @param from The swatch file or InDesign document.
   */
  loadSwatches(from: File): void

  /**
   * Loads a set of XML markup tags from the specified file.
   * @param from The path to the file that contains the tags.
   */
  loadXMLTags(from: File): void

  /**
   * Auto tag the document based on the style to tag mappings
   */
  mapStylesToXMLTags(): void

  /**
   * Auto style the document based on the tag to style mappings
   */
  mapXMLTagsToStyles(): void

  /**
   * Packages the document.
   * @param to The folder, alias, or path in which to place the packaged files.
   * @param copyingFonts If true, copies fonts used in the document to the package folder.
   * @param copyingLinkedGraphics If true, copies linked graphics files to the package folder.
   * @param copyingProfiles If true, copies color profiles to the package folder.
   * @param updatingGraphics If true, updates graphics links to the package folder.
   * @param includingHiddenLayers If true, copies fonts and links from hidden layers to the package.
   * @param ignorePreflightErrors If true, ignores preflight errors and proceeds with the packaging. If false, cancels the packaging when errors exist.
   * @param creatingReport If true, creates a package report that includes printing instructions, print settings, lists of fonts, links and required inks, and other information.
   * @param includeIdml If true, generates and includes IDML in the package folder.
   * @param includePdf If true, generates and includes PDF in the package folder.
   * @param pdfStyle If specified and PDF is to be included, use this style for PDF export if it is valid, otherwise use the last used PDF preset.
   * @param useDocumentHyphenationExceptionsOnly If this option is selected, InDesign flags this document so that it does not reflow when someone else opens or edits it on a computer that has different hyphenation and dictionary settings.
   * @param versionComments The comments for the version.
   * @param forceSave If true, forcibly saves a version.
   */
  packageForPrint(
    to: File,
    copyingFonts: boolean,
    copyingLinkedGraphics: boolean,
    copyingProfiles: boolean,
    updatingGraphics: boolean,
    includingHiddenLayers: boolean,
    ignorePreflightErrors: boolean,
    creatingReport: boolean,
    includeIdml: boolean,
    includePdf: boolean,
    pdfStyle: string,
    useDocumentHyphenationExceptionsOnly: boolean,
    versionComments: string,
    forceSave?: boolean,
  ): boolean

  /**
   * Place one or more files following the behavior of the place menu item. This may load the place gun or replace the selected object, depending on current preferences.
   * @param fileName One or more files to place.
   * @param showingOptions Whether to display the import options dialog
   * @param withProperties Initial values for properties of the placed object(s)
   */
  place(fileName: File | File[], showingOptions?: boolean, withProperties?: object): void

  /**
   * Deprecated: Use ContentPlacerObject load method. Original Description: Place following the behavior of the place and link story menu item. This will load the place gun.
   * @param parentStory The story to place and link from.
   * @param showingOptions Whether to display the link options dialog
   */
  placeAndLink(parentStory: Story, showingOptions?: boolean): void

  /**
   * place a cloud asset on the document
   * @param jsondata JSON data containing metadata about the cloud asset
   */
  placeCloudAsset(jsondata: string): void

  /**
   * Prints the Document(s).
   * @param printDialog Whether to invoke the print dialog
   * @param using Printer preset to use.
   */
  print(printDialog: boolean, using: PrinterPresetTypes | PrinterPreset): void

  /**
   * Print the Booklet using current document and Booklet and Print settings in the document
   * @param printBookletDialog Whether to invoke the print booklet dialog
   * @param using Printer preset to use.
   */
  printBooklet(printBookletDialog?: boolean, using?: PrinterPresetTypes | PrinterPreset): void

  /**
   * Recomposes the text in the Document.
   */
  recompose(): void

  /**
   * Redoes the last action.
   */
  redo(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Resets all the buttons to their Normal state.
   */
  resetAllButtons(): void

  /**
   * Resets all the multi-state objects to their first state.
   */
  resetAllMultiStateObjects(): void

  /**
   * Reverts the document to its state at the last save operation.
   */
  revert(): boolean

  /**
   * Reverts to the version of the document in Version Cue.
   * @param forceRevert Forcibly reverts to the project version.
   */
  revertToProject(forceRevert?: boolean): void

  /**
   * Save the document
   * @param to Where to save the document. If the document is already saved, a copy is saved at this path, the original file is closed the new copy is opened
   * @param stationery Whether to save the file as stationery
   * @param versionComments The comment for this version
   * @param forceSave Forcibly save a version
   */
  save(to: File, stationery?: boolean, versionComments?: string, forceSave?: boolean): Document

  /**
   * Saves a copy of the document.
   * @param to The file path for the copy. Note: Leaves the original document open and does not open the copy.
   * @param stationery If true, saves the file as stationery (Mac OS) or as a template (Windows). Note: The file extension for stationery and templates is different than the extension for regular files.
   */
  saveACopy(to: File, stationery?: boolean): void

  /**
   * Saves the specified swatch(es) to a swatchbook file.
   * @param to The swatchbook file to save to.
   * @param swatchList The swatch(es) to save.
   * @param versionComments The comment for this version.
   * @param forceSave If true, forcibly saves a version.
   */
  saveSwatches(to: File, swatchList: Swatch[], versionComments: string, forceSave?: boolean): void

  /**
   * Saves a set of tags to an external file.
   * @param to The full path to the file in which to save the tags.
   * @param versionComments The comment for this version.
   * @param forceSave If true, forcibly saves a version.
   */
  saveXMLTags(to: File, versionComments: string, forceSave?: boolean): void

  /**
   * Selects the specified object(s).
   * @param selectableItems The objects to select.
   * @param existingSelection The selection status of the Document in relation to previously selected objects.
   */
  select(
    selectableItems: object | object[] | NothingEnum | SelectAll,
    existingSelection?: SelectionOptions,
  ): void

  /**
   * Synchronizes the file with the Version Cue project.
   * @param syncConflictResolution The conflict resolution method to use during synchronization.
   * @param versionComments The comments that describe the version.
   */
  synchronizeWithVersionCue(
    syncConflictResolution?: SyncConflictResolution,
    versionComments?: string,
  ): VersionCueSyncStatus

  /**
   * Generates a string which, if executed, will return the Document.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string

  /**
   * Undoes the last action.
   */
  undo(): void

  /**
   * Updates cross references' text source content in the document.
   */
  updateCrossReferences(): void
}

/**
 * A collection of documents.
 */
declare class Documents {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Document with the specified index.
   * @param index The index.
   */
  [index: number]: Document

  /**
   * Creates a new document.
   * @param showingWindow If true, displays the document.
   * @param documentPreset The document preset to use.
   * @param withProperties Initial values for properties of the new Document
   */
  add(showingWindow?: boolean, documentPreset?: DocumentPreset, withProperties?: object): Document

  /**
   * Returns any Document in the collection.
   */
  anyItem(): Document

  /**
   * Displays the number of elements in the Document.
   */
  count(): number

  /**
   * Returns every Document in the collection.
   */
  everyItem(): Document

  /**
   * Returns the first Document in the collection.
   */
  firstItem(): Document

  /**
   * Returns the Document with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Document

  /**
   * Returns the Document with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Document

  /**
   * Returns the Document with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Document

  /**
   * Returns the Documents within the specified range.
   * @param from The Document, index, or name at the beginning of the range.
   * @param to The Document, index, or name at the end of the range.
   */
  itemByRange(from: Document | number | string, to: Document | number | string): Document[]

  /**
   * Returns the last Document in the collection.
   */
  lastItem(): Document

  /**
   * Returns the middle Document in the collection.
   */
  middleItem(): Document

  /**
   * Returns the Document whose index follows the specified Document in the collection.
   * @param obj The Document whose index comes before the desired Document.
   */
  nextItem(obj: Document): Document

  /**
   * Returns the Document with the index previous to the specified index.
   * @param obj The index of the Document that follows the desired Document.
   */
  previousItem(obj: Document): Document

  /**
   * Generates a string which, if executed, will return the Document.
   */
  toSource(): string
}

/**
 * A window.
 */
declare class Window {
  /**
   * Dispatched after the Window becomes active. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_ACTIVATE: string

  /**
   * Dispatched when the value of a property changes on this Window. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_ATTRIBUTE_CHANGED: string

  /**
   * Dispatched when a Window is closing. Since the close has been committed, it can no longer be canceled. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_CLOSE: string

  /**
   * Dispatched after a Window is opened. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_OPEN: string

  /**
   * Dispatched after an attribute on the active selection changes. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_SELECTION_ATTRIBUTE_CHANGED: string

  /**
   * Dispatched after the active selection changes. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_SELECTION_CHANGED: string

  /**
   * Dispatched before a Window is closed. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_CLOSE: string

  /**
   * Dispatched before the Window becomes inactive. This event bubbles. This event is not cancelable.
   */
  static readonly BEFORE_DEACTIVATE: string

  /**
   * The bounds of the window (specified in pixels) in the form [top, left, bottom, right].
   */
  bounds: number[]

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The index of the Window within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * The name of the Window.
   */
  readonly name: string

  /**
   * The parent of the Window (a Document).
   */
  readonly parent: Document

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * The selected object(s).
   */
  selection: object[] | object | NothingEnum

  /**
   * The key object of the selection.
   */
  selectionKeyObject: PageItem | NothingEnum

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Brings the object to the front.
   */
  bringToFront(): void

  /**
   * Closes the Window.
   */
  close(): void

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): Window[]

  /**
   * Maximizes the window.
   */
  maximize(): void

  /**
   * Minimizes the window.
   */
  minimize(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Restores the window.
   */
  restore(): void

  /**
   * Selects the specified object(s).
   * @param selectableItems The objects to select.
   * @param existingSelection The selection status of the Window in relation to previously selected objects.
   */
  select(
    selectableItems: object | object[] | NothingEnum | SelectAll,
    existingSelection?: SelectionOptions,
  ): void

  /**
   * Generates a string which, if executed, will return the Window.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of windows.
 */
declare class Windows {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Window with the specified index.
   * @param index The index.
   */
  [index: number]: Window

  /**
   * Creates a new Window.
   * @param withProperties Initial values for properties of the new Window
   */
  add(withProperties: object): Window

  /**
   * Returns any Window in the collection.
   */
  anyItem(): Window

  /**
   * Displays the number of elements in the Window.
   */
  count(): number

  /**
   * Returns every Window in the collection.
   */
  everyItem(): Window

  /**
   * Returns the first Window in the collection.
   */
  firstItem(): Window

  /**
   * Returns the Window with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Window

  /**
   * Returns the Window with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Window

  /**
   * Returns the Windows within the specified range.
   * @param from The Window, index, or name at the beginning of the range.
   * @param to The Window, index, or name at the end of the range.
   */
  itemByRange(from: Window | number | string, to: Window | number | string): Window[]

  /**
   * Returns the last Window in the collection.
   */
  lastItem(): Window

  /**
   * Returns the middle Window in the collection.
   */
  middleItem(): Window

  /**
   * Returns the Window whose index follows the specified Window in the collection.
   * @param obj The Window whose index comes before the desired Window.
   */
  nextItem(obj: Window): Window

  /**
   * Returns the Window with the index previous to the specified index.
   * @param obj The index of the Window that follows the desired Window.
   */
  previousItem(obj: Window): Window

  /**
   * Generates a string which, if executed, will return the Window.
   */
  toSource(): string
}

/**
 * A layout window.
 */
declare class LayoutWindow extends Window {
  /**
   * The active layer.
   */
  activeLayer: Layer | string

  /**
   * The front-most page.
   */
  activePage: Page

  /**
   * The front-most spread.
   */
  activeSpread: Spread | MasterSpread

  /**
   * If true, display a simulation of overprinting.
   */
  overprintPreview: boolean

  /**
   * If true, leaves color values unchanged for CMYK objects without embedded profiles and native objects such as line art or type. Note: Converts images whose profiles differ from the profile of the simulated device. Valid only when proofing type is custom.
   */
  preserveColorNumbers: boolean

  /**
   * The profile used for proofing colors.
   */
  proofingProfile: string

  /**
   * The method of proofing colors.
   */
  proofingType: ProofingType

  /**
   * The screen mode for layout view.
   */
  screenMode: ScreenModeOptions

  /**
   * If true, simulates the dark gray produced by many printers in place of solid black, according to the proofing profile. Note: Valid only when proofing type is custom.
   */
  simulateInkBlack: boolean

  /**
   * If true, simulates the dingy white of real paper, according to the proofing profile. Note: Valid only when proofing type is custom.
   */
  simulatePaperWhite: boolean

  /**
   * The default anchor point around which to transform objects.
   */
  transformReferencePoint: AnchorPoint | [number | string, number | string]

  /**
   * The display performance preferences override for the view.
   */
  viewDisplaySetting: ViewDisplaySettings

  /**
   * The size (as a percentage) to which to enlarge or reduce the view of the document. (Range: 5 to 4000)
   */
  zoomPercentage: number

  /**
   * Magnifies or reduces the window to the specified display size.
   * @param given The display size.
   */
  zoom(given: ZoomOptions): void
}

/**
 * A collection of layout windows.
 */
declare class LayoutWindows {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the LayoutWindow with the specified index.
   * @param index The index.
   */
  [index: number]: LayoutWindow

  /**
   * Returns any LayoutWindow in the collection.
   */
  anyItem(): LayoutWindow

  /**
   * Displays the number of elements in the LayoutWindow.
   */
  count(): number

  /**
   * Returns every LayoutWindow in the collection.
   */
  everyItem(): LayoutWindow

  /**
   * Returns the first LayoutWindow in the collection.
   */
  firstItem(): LayoutWindow

  /**
   * Returns the LayoutWindow with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): LayoutWindow

  /**
   * Returns the LayoutWindow with the specified name.
   * @param name The name.
   */
  itemByName(name: string): LayoutWindow

  /**
   * Returns the LayoutWindows within the specified range.
   * @param from The LayoutWindow, index, or name at the beginning of the range.
   * @param to The LayoutWindow, index, or name at the end of the range.
   */
  itemByRange(
    from: LayoutWindow | number | string,
    to: LayoutWindow | number | string,
  ): LayoutWindow[]

  /**
   * Returns the last LayoutWindow in the collection.
   */
  lastItem(): LayoutWindow

  /**
   * Returns the middle LayoutWindow in the collection.
   */
  middleItem(): LayoutWindow

  /**
   * Returns the LayoutWindow whose index follows the specified LayoutWindow in the collection.
   * @param obj The LayoutWindow whose index comes before the desired LayoutWindow.
   */
  nextItem(obj: LayoutWindow): LayoutWindow

  /**
   * Returns the LayoutWindow with the index previous to the specified index.
   * @param obj The index of the LayoutWindow that follows the desired LayoutWindow.
   */
  previousItem(obj: LayoutWindow): LayoutWindow

  /**
   * Generates a string which, if executed, will return the LayoutWindow.
   */
  toSource(): string
}

/**
 * A story window.
 */
declare class StoryWindow extends Window {}

/**
 * A collection of story windows.
 */
declare class StoryWindows {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the StoryWindow with the specified index.
   * @param index The index.
   */
  [index: number]: StoryWindow

  /**
   * Returns any StoryWindow in the collection.
   */
  anyItem(): StoryWindow

  /**
   * Displays the number of elements in the StoryWindow.
   */
  count(): number

  /**
   * Returns every StoryWindow in the collection.
   */
  everyItem(): StoryWindow

  /**
   * Returns the first StoryWindow in the collection.
   */
  firstItem(): StoryWindow

  /**
   * Returns the StoryWindow with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): StoryWindow

  /**
   * Returns the StoryWindow with the specified name.
   * @param name The name.
   */
  itemByName(name: string): StoryWindow

  /**
   * Returns the StoryWindows within the specified range.
   * @param from The StoryWindow, index, or name at the beginning of the range.
   * @param to The StoryWindow, index, or name at the end of the range.
   */
  itemByRange(from: StoryWindow | number | string, to: StoryWindow | number | string): StoryWindow[]

  /**
   * Returns the last StoryWindow in the collection.
   */
  lastItem(): StoryWindow

  /**
   * Returns the middle StoryWindow in the collection.
   */
  middleItem(): StoryWindow

  /**
   * Returns the StoryWindow whose index follows the specified StoryWindow in the collection.
   * @param obj The StoryWindow whose index comes before the desired StoryWindow.
   */
  nextItem(obj: StoryWindow): StoryWindow

  /**
   * Returns the StoryWindow with the index previous to the specified index.
   * @param obj The index of the StoryWindow that follows the desired StoryWindow.
   */
  previousItem(obj: StoryWindow): StoryWindow

  /**
   * Generates a string which, if executed, will return the StoryWindow.
   */
  toSource(): string
}

/**
 * A document event.
 */
declare class DocumentEvent extends Event {
  /**
   * Dispatched after a Document is reverted. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_REVERT: string

  /**
   * Dispatched after a Document is saved. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_SAVE: string

  /**
   * Dispatched after a Document is saved under a new name. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_SAVE_AS: string

  /**
   * Dispatched after a copy of a Document is saved. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_SAVE_A_COPY: string

  /**
   * Dispatched before a Document is created. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_NEW: string

  /**
   * Dispatched before a Document is opened. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_OPEN: string

  /**
   * Dispatched before a Document is reverted. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_REVERT: string

  /**
   * Dispatched before a Document is saved. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_SAVE: string

  /**
   * Dispatched before a Document is saved under a new name. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_SAVE_AS: string

  /**
   * Dispatched before a copy of a Document is saved. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_SAVE_A_COPY: string

  /**
   * The full path to the DocumentEvent, including the name of the DocumentEvent.
   */
  readonly fullName: File

  /**
   * Controls the display of dialogs and alerts during script processing.
   */
  userInteractionLevel: UserInteractionLevels
}

/**
 * An import or export event.
 */
declare class ImportExportEvent extends Event {
  /**
   * Dispatched after a ImportExportEvent is exported. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_EXPORT: string

  /**
   * Dispatched after importing a file into a ImportExportEvent. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_IMPORT: string

  /**
   * Dispatched before a ImportExportEvent is exported. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_EXPORT: string

  /**
   * Dispatched before importing a file into a ImportExportEvent. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_IMPORT: string

  /**
   * Dispatched after a ImportExportEvent export is canceled or fails. This event bubbles. This event is not cancelable.
   */
  static readonly FAILED_EXPORT: string

  /**
   * The import/export file format.
   */
  readonly format: string

  /**
   * The full path to the ImportExportEvent, including the name of the ImportExportEvent.
   */
  readonly fullName: File

  /**
   * Controls the display of dialogs and alerts during script processing.
   */
  userInteractionLevel: UserInteractionLevels
}

/**
 * Represents the content placer object.
 */
declare class ContentPlacerObject extends Preference {
  /**
   * Whether the Content Placer is currently loaded with content for placing.
   */
  readonly loaded: boolean

  /**
   * Load the Content Placer with one or more objects.
   * @param pageItems One or more page items to place or load
   * @param linkPageItems Whether to link pageItems in content placer (if true it will override link stories value)
   * @param linkStories Whether to link stories in content placer (only applicable for single story, pageItem links will also be created in case of more than one item)
   * @param mapStyles Whether to map styles in content placer
   * @param showingOptions Whether to display the link options dialog
   */
  load(
    pageItems: PageItem[],
    linkPageItems?: boolean,
    linkStories?: boolean,
    mapStyles?: boolean,
    showingOptions?: boolean,
  ): void
}

/**
 * A book.
 */
declare class Book {
  /**
   * If true, automatically converts the book content object during repagination and synchronization.
   */
  automaticDocumentConversion: boolean

  /**
   * If true, automatically updates page numbers when pages in book content files are added, deleted, or rearranged.
   */
  automaticPagination: boolean

  /**
   * A collection of book content objects.
   */
  readonly bookContents: BookContents

  /**
   * EPub export preference settings.
   */
  readonly epubExportPreferences: EPubExportPreference

  /**
   * EPub fixed layout export preference settings.
   */
  readonly epubFixedLayoutExportPreferences: EPubFixedLayoutExportPreference

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The full path to the file.
   */
  readonly filePath: File

  /**
   * The full path to the Book, including the name of the Book.
   */
  readonly fullName: File

  /**
   * The index of the Book within its containing object.
   */
  readonly index: number

  /**
   * If true, inserts a blank page as necessary to fill in page number gaps caused by the odd or even specification set in the repagination option.
   */
  insertBlankPage: boolean

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * If true, merges identically named layers when exporting to PDF.
   */
  mergeIdenticalLayers: boolean

  /**
   * If true, the Book has been modified since it was last saved.
   */
  readonly modified: boolean

  /**
   * The name of the Book.
   */
  readonly name: string

  /**
   * The parent of the Book (a Application).
   */
  readonly parent: Application

  /**
   * A collection of preferences objects.
   */
  readonly preferences: Preferences

  /**
   * Preflight book option settings.
   */
  readonly preflightBookOptions: PreflightBookOption

  /**
   * Print preference settings.
   */
  readonly printPreferences: PrintPreference

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * Page numbering options for book content objects within the book.
   */
  repaginationOption: RepaginateOption

  /**
   * If true, the Book has been saved since it was created.
   */
  readonly saved: boolean

  /**
   * How to match styles with same name while synchronizing a book.
   */
  smartMatchStyleGroups: SmartMatchOptions

  /**
   * Current style source document.
   */
  styleSourceDocument: Document | BookContent

  /**
   * If true, synchronize bullets and numbering.
   */
  synchronizeBulletNumberingList: boolean

  /**
   * If true, synchronize cell styles.
   */
  synchronizeCellStyle: boolean

  /**
   * If true, synchronize character styles.
   */
  synchronizeCharacterStyle: boolean

  /**
   * If true, synchronize conditional text.
   */
  synchronizeConditionalText: boolean

  /**
   * If true, synchronize cross reference formats
   */
  synchronizeCrossReferenceFormat: boolean

  /**
   * If true, synchronize master pages.
   */
  synchronizeMasterPage: boolean

  /**
   * If true, synchronize object styles.
   */
  synchronizeObjectStyle: boolean

  /**
   * If true, synchronize paragraph styles.
   */
  synchronizeParagraphStyle: boolean

  /**
   * If true, synchronize swatches.
   */
  synchronizeSwatch: boolean

  /**
   * If true, synchronize table of content styles.
   */
  synchronizeTableOfContentStyle: boolean

  /**
   * If true, synchronize table styles.
   */
  synchronizeTableStyle: boolean

  /**
   * If true, synchronize text variables.
   */
  synchronizeTextVariable: boolean

  /**
   * If true, synchronize trap styles.
   */
  synchronizeTrapStyle: boolean

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Close the Book
   * @param saving Whether to save changes before closing the Book
   * @param savingIn The file in which to save the Book
   * @param versionComments The comment for this version
   * @param forceSave Forcibly save a version
   */
  close(saving?: SaveOptions, savingIn?: File, versionComments?: string, forceSave?: boolean): void

  /**
   * Exports the book to a file.
   * @param format The export format, specified as an enumeration value or as an extension that appears in the Save as type or Format menu in the Export dialog.
   * @param to The file to which to export the book.
   * @param showingOptions Used to specify if the PDF Export Options Dialog needs to be shown or not
   * @param using Used to specify the presets for the export which can be the object of the preset type, enumeration for existing presets or user defined presets, or a string naming the preset to be used, but in case Showing Options is true, the preset specified in the Export Dialog will over ride this parameter
   * @param whichDocuments Used to specify a list of book content references, from the current book, where the list may contain duplicate entries and if the list is not specified then the entire book contents shall be exported
   * @param versionComments The comment for this version.
   * @param forceSave If true, forcibly saves a version.
   */
  exportFile(
    format: ExportFormat | string,
    to: File,
    showingOptions?: boolean,
    using?: PDFExportPreset,
    whichDocuments?: BookContent[],
    versionComments?: string,
    forceSave?: boolean,
  ): void

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): Book[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Packages the document.
   * @param to The folder, alias, or path in which to place the packaged files.
   * @param copyingFonts If true, copies fonts used in the document to the package folder.
   * @param copyingLinkedGraphics If true, copies linked graphics files to the package folder.
   * @param copyingProfiles If true, copies color profiles to the package folder.
   * @param updatingGraphics If true, updates graphics links to the package folder.
   * @param includingHiddenLayers If true, copies fonts and links from hidden layers to the package.
   * @param ignorePreflightErrors If true, ignores preflight errors and proceeds with the packaging. If false, cancels the packaging when errors exist.
   * @param creatingReport If true, creates a package report that includes printing instructions, print settings, lists of fonts, links and required inks, and other information.
   * @param includeIdml If true, generates and includes IDML in the package folder.
   * @param includePdf If true, generates and includes PDF in the package folder.
   * @param pdfStyle If specified and PDF is to be included, use this style for PDF export if it is valid, otherwise use the last used PDF preset.
   * @param useDocumentHyphenationExceptionsOnly If this option is selected, InDesign flags this document so that it does not reflow when someone else opens or edits it on a computer that has different hyphenation and dictionary settings.
   * @param versionComments The comments for the version.
   * @param forceSave If true, forcibly saves a version.
   */
  packageForPrint(
    to: File,
    copyingFonts: boolean,
    copyingLinkedGraphics: boolean,
    copyingProfiles: boolean,
    updatingGraphics: boolean,
    includingHiddenLayers: boolean,
    ignorePreflightErrors: boolean,
    creatingReport: boolean,
    includeIdml: boolean,
    includePdf: boolean,
    pdfStyle: string,
    useDocumentHyphenationExceptionsOnly: boolean,
    versionComments: string,
    forceSave?: boolean,
  ): boolean

  /**
   * Preflight a book and optionally save the resulting report.
   * @param to The preflight report to save to.
   * @param autoOpen If true, automatically open the report after creation.
   */
  preflight(to: File, autoOpen?: boolean): void

  /**
   * Prints the Book(s).
   * @param printDialog Whether to invoke the print dialog
   * @param using Printer preset to use.
   */
  print(printDialog: boolean, using: PrinterPresetTypes | PrinterPreset): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Repaginates the book.
   */
  repaginate(): void

  /**
   * Saves the book.
   * @param to The file path. Note: Required only if the book has not been previously saved. If the book has previously been saved, specifying a path saves a copy and closes the original book.
   * @param versionComments The comment for this version.
   * @param forceSave If true, forcibly saves a version.
   */
  save(to: File, versionComments: string, forceSave?: boolean): void

  /**
   * Synchronizes the entire book to the style source document.
   */
  synchronize(): void

  /**
   * Generates a string which, if executed, will return the Book.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string

  /**
   * Updates the cross references in the entire book.
   */
  updateAllCrossReferences(): void

  /**
   * Update all numbers (e.g. Page numbers, chapter numbers and paragraph numbers) throughout the book.
   */
  updateAllNumbers(): void

  /**
   * Updates chapter numbers and paragraph numbers throughout the book.
   */
  updateChapterAndParagraphNumbers(): void
}

/**
 * A collection of books.
 */
declare class Books {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Book with the specified index.
   * @param index The index.
   */
  [index: number]: Book

  /**
   * Creates a new book.
   * @param fullName The full path name of the new book file, including the book file extension '.indb'.
   * @param withProperties Initial values for properties of the new Book
   */
  add(fullName: File, withProperties: object): Book

  /**
   * Returns any Book in the collection.
   */
  anyItem(): Book

  /**
   * Displays the number of elements in the Book.
   */
  count(): number

  /**
   * Returns every Book in the collection.
   */
  everyItem(): Book

  /**
   * Returns the first Book in the collection.
   */
  firstItem(): Book

  /**
   * Returns the Book with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Book

  /**
   * Returns the Book with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Book

  /**
   * Returns the Books within the specified range.
   * @param from The Book, index, or name at the beginning of the range.
   * @param to The Book, index, or name at the end of the range.
   */
  itemByRange(from: Book | number | string, to: Book | number | string): Book[]

  /**
   * Returns the last Book in the collection.
   */
  lastItem(): Book

  /**
   * Returns the middle Book in the collection.
   */
  middleItem(): Book

  /**
   * Returns the Book whose index follows the specified Book in the collection.
   * @param obj The Book whose index comes before the desired Book.
   */
  nextItem(obj: Book): Book

  /**
   * Returns the Book with the index previous to the specified index.
   * @param obj The index of the Book that follows the desired Book.
   */
  previousItem(obj: Book): Book

  /**
   * Generates a string which, if executed, will return the Book.
   */
  toSource(): string
}

/**
 * A document added to a book.
 */
declare class BookContent {
  /**
   * The date and time the BookContent was created.
   */
  readonly date: Date

  /**
   * The page range of the book content object within the book.
   */
  readonly documentPageRange: string

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The full path to the file.
   */
  readonly filePath: File

  /**
   * The full path to the BookContent, including the name of the BookContent.
   */
  readonly fullName: File

  /**
   * The unique ID of the BookContent.
   */
  readonly id: number

  /**
   * The index of the BookContent within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The name of the BookContent.
   */
  readonly name: string

  /**
   * The parent of the BookContent (a Book).
   */
  readonly parent: Book

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * The size of the BookContent file.
   */
  readonly size: number

  /**
   * The status of the book content object file.
   */
  readonly status: BookContentStatus

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): BookContent[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Moves the book content object.
   * @param to The location relative to the reference object or within the book.
   * @param reference The reference object. Note: Required when the to parameter specifies before or after.
   */
  move(to?: LocationOptions, reference?: BookContent): BookContent

  /**
   * Preflight a book content object and optionally save the resulting report.
   * @param to The preflight report to save to.
   * @param autoOpen If true, automatically open the report after creation.
   */
  preflight(to: File, autoOpen?: boolean): void

  /**
   * Deletes the BookContent.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Replaces a book content object with a new file. Note: If the new file replaces the current style source document, the new file becomes the style source document.
   * @param using The full path name of the new book content object.
   */
  replace(using: File): BookContent

  /**
   * Matches the formatting of the book content object to the style source document.
   */
  synchronize(): void

  /**
   * Generates a string which, if executed, will return the BookContent.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of book content objects.
 */
declare class BookContents {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the BookContent with the specified index.
   * @param index The index.
   */
  [index: number]: BookContent

  /**
   * Creates a new book content object.
   * @param fullName The full path name of the new book content object.
   * @param at The location of the book content object within the book.
   * @param withProperties Initial values for properties of the new BookContent
   */
  add(fullName: File, at?: number, withProperties?: object): BookContent

  /**
   * Returns any BookContent in the collection.
   */
  anyItem(): BookContent

  /**
   * Displays the number of elements in the BookContent.
   */
  count(): number

  /**
   * Returns every BookContent in the collection.
   */
  everyItem(): BookContent

  /**
   * Returns the first BookContent in the collection.
   */
  firstItem(): BookContent

  /**
   * Returns the BookContent with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): BookContent

  /**
   * Returns the BookContent with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): BookContent

  /**
   * Returns the BookContent with the specified name.
   * @param name The name.
   */
  itemByName(name: string): BookContent

  /**
   * Returns the BookContents within the specified range.
   * @param from The BookContent, index, or name at the beginning of the range.
   * @param to The BookContent, index, or name at the end of the range.
   */
  itemByRange(from: BookContent | number | string, to: BookContent | number | string): BookContent[]

  /**
   * Returns the last BookContent in the collection.
   */
  lastItem(): BookContent

  /**
   * Returns the middle BookContent in the collection.
   */
  middleItem(): BookContent

  /**
   * Returns the BookContent whose index follows the specified BookContent in the collection.
   * @param obj The BookContent whose index comes before the desired BookContent.
   */
  nextItem(obj: BookContent): BookContent

  /**
   * Returns the BookContent with the index previous to the specified index.
   * @param obj The index of the BookContent that follows the desired BookContent.
   */
  previousItem(obj: BookContent): BookContent

  /**
   * Generates a string which, if executed, will return the BookContent.
   */
  toSource(): string
}

/**
 * An ink.
 */
declare class Ink {
  /**
   * The ink object to map this ink to.
   */
  aliasInkName: string

  /**
   * The angle of the ink. (Range: 0 to 360)
   */
  angle: number

  /**
   * Converts spot inks to process inks.
   */
  convertToProcess: boolean

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The frequency of the ink. (Range: 1 to 500)
   */
  frequency: number

  /**
   * The unique ID of the Ink.
   */
  readonly id: number

  /**
   * The index of the Ink within its containing object.
   */
  readonly index: number

  /**
   * The trapping type of the ink.
   */
  inkType: InkTypes

  /**
   * If true, the ink is a process ink.
   */
  readonly isProcessInk: boolean

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The name of the Ink.
   */
  readonly name: string

  /**
   * The neutral density of the ink. (Range: 0.001 to 10.0)
   */
  neutralDensity: number

  /**
   * The parent of the Ink (a Application or Document).
   */
  readonly parent: any

  /**
   * If true, prints the ink. Only valid when printing separations.
   */
  printInk: boolean

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * The solidity value of the ink. (Range: 0.0 to 1.0)
   */
  readonly solidity: number

  /**
   * The place of the ink in the trapping sequence.
   */
  trapOrder: number

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): Ink[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the Ink.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of inks.
 */
declare class Inks {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Ink with the specified index.
   * @param index The index.
   */
  [index: number]: Ink

  /**
   * Returns any Ink in the collection.
   */
  anyItem(): Ink

  /**
   * Displays the number of elements in the Ink.
   */
  count(): number

  /**
   * Returns every Ink in the collection.
   */
  everyItem(): Ink

  /**
   * Returns the first Ink in the collection.
   */
  firstItem(): Ink

  /**
   * Returns the Ink with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Ink

  /**
   * Returns the Ink with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Ink

  /**
   * Returns the Ink with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Ink

  /**
   * Returns the Inks within the specified range.
   * @param from The Ink, index, or name at the beginning of the range.
   * @param to The Ink, index, or name at the end of the range.
   */
  itemByRange(from: Ink | number | string, to: Ink | number | string): Ink[]

  /**
   * Returns the last Ink in the collection.
   */
  lastItem(): Ink

  /**
   * Returns the middle Ink in the collection.
   */
  middleItem(): Ink

  /**
   * Returns the Ink whose index follows the specified Ink in the collection.
   * @param obj The Ink whose index comes before the desired Ink.
   */
  nextItem(obj: Ink): Ink

  /**
   * Returns the Ink with the index previous to the specified index.
   * @param obj The index of the Ink that follows the desired Ink.
   */
  previousItem(obj: Ink): Ink

  /**
   * Generates a string which, if executed, will return the Ink.
   */
  toSource(): string
}

/**
 * A trap preset.
 */
declare class TrapPreset {
  /**
   * The minimum amount (as a percentage) of black ink required before the black width setting is applied. (Range: 0 to 100)
   */
  blackColorThreshold: number

  /**
   * The neutral density value at or above which an ink is considered black. (Range: .001 to 10)
   */
  blackDensity: number

  /**
   * The black width. (Range depends on unit. For points: 0.0 to 8.0; picas: 0p0 to 0p8; inches: 0 to 0.1111; mm: 0 to 2.822; cm: 0 to .2822; ciceros: 0c0 to 0c7.507)
   */
  blackWidth: string | number

  /**
   * The degree (as a percentage) to which components from abutting colors are used to reduce the trap color. (Range: 0 to 100) Note: 0% makes a trap whose neutral density is equal to the neutral density of the darker color.
   */
  colorReduction: number

  /**
   * The default width for trapping all colors except those involving solid black. (Range depends on unit. For points: 0.0 to 8.0; picas: 0p0 to 0p8; inches: 0 to 0.1111; mm: 0 to 2.822; cm: 0 to .2822; ciceros: 0c0 to 0c7.507)
   */
  defaultTrapWidth: string | number

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The unique ID of the TrapPreset.
   */
  readonly id: number

  /**
   * The trap placement between vector objects and bitmap images.
   */
  imagePlacement: TrapImagePlacementTypes

  /**
   * If true, turns on trapping along the boundary of overlapping or abutting bitmap images.
   */
  imagesToImages: boolean

  /**
   * The index of the TrapPreset within its containing object.
   */
  readonly index: number

  /**
   * If true, turns on trapping among colors within individual bitmap images.
   */
  internalImages: boolean

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The name of the TrapPreset.
   */
  name: string

  /**
   * If true, ensures that vector objects overlap bitmap images.
   */
  objectsToImages: boolean

  /**
   * If true, ensures that one-bit images trap to abutting objects.
   */
  oneBitImages: boolean

  /**
   * The parent of the TrapPreset (a Application or Document).
   */
  readonly parent: any

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * The difference (as a percentage) between the neutral densities of abutting colors at which the trap is moved from the darker side of a color edge toward the centerline. (Range: 0 to 100)
   */
  slidingTrapThreshold: number

  /**
   * The amount (as a percentage) that components of abutting colors must vary before a trap is created. (Range: 1 to 100)
   */
  stepThreshold: number

  /**
   * The shape to use at the intersection of three-way traps.
   */
  trapEnd: TrapEndTypes

  /**
   * The join type of the trap preset.
   */
  trapJoin: EndJoin

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Duplicates the TrapPreset.
   */
  duplicate(): TrapPreset

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): TrapPreset[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Deletes the trap preset.
   * @param replacingWith The trap preset to apply in place of the deleted preset.
   */
  remove(replacingWith: TrapPreset): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the TrapPreset.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of trap presets.
 */
declare class TrapPresets {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the TrapPreset with the specified index.
   * @param index The index.
   */
  [index: number]: TrapPreset

  /**
   * Creates a new TrapPreset.
   * @param withProperties Initial values for properties of the new TrapPreset
   */
  add(withProperties: object): TrapPreset

  /**
   * Returns any TrapPreset in the collection.
   */
  anyItem(): TrapPreset

  /**
   * Displays the number of elements in the TrapPreset.
   */
  count(): number

  /**
   * Returns every TrapPreset in the collection.
   */
  everyItem(): TrapPreset

  /**
   * Returns the first TrapPreset in the collection.
   */
  firstItem(): TrapPreset

  /**
   * Returns the TrapPreset with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): TrapPreset

  /**
   * Returns the TrapPreset with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): TrapPreset

  /**
   * Returns the TrapPreset with the specified name.
   * @param name The name.
   */
  itemByName(name: string): TrapPreset

  /**
   * Returns the TrapPresets within the specified range.
   * @param from The TrapPreset, index, or name at the beginning of the range.
   * @param to The TrapPreset, index, or name at the end of the range.
   */
  itemByRange(from: TrapPreset | number | string, to: TrapPreset | number | string): TrapPreset[]

  /**
   * Returns the last TrapPreset in the collection.
   */
  lastItem(): TrapPreset

  /**
   * Returns the middle TrapPreset in the collection.
   */
  middleItem(): TrapPreset

  /**
   * Returns the TrapPreset whose index follows the specified TrapPreset in the collection.
   * @param obj The TrapPreset whose index comes before the desired TrapPreset.
   */
  nextItem(obj: TrapPreset): TrapPreset

  /**
   * Returns the TrapPreset with the index previous to the specified index.
   * @param obj The index of the TrapPreset that follows the desired TrapPreset.
   */
  previousItem(obj: TrapPreset): TrapPreset

  /**
   * Generates a string which, if executed, will return the TrapPreset.
   */
  toSource(): string
}

/**
 * A mixed ink swatch.
 */
declare class MixedInk extends Swatch {
  /**
   * The mixed ink group that a mixed ink swatch is based on.
   */
  readonly baseColor: MixedInkGroup

  /**
   * The component inks.
   */
  readonly inkList: Ink[]

  /**
   * The array of tint percentages for inks in the ink list. Note: Specify a value for each ink.
   */
  inkPercentages: number[]

  /**
   * The color model.
   */
  model: ColorModel

  /**
   * The color space.
   */
  space: ColorSpace
}

/**
 * A collection of mixed inks.
 */
declare class MixedInks {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the MixedInk with the specified index.
   * @param index The index.
   */
  [index: number]: MixedInk

  /**
   * Creates a new mixed ink swatch.
   * @param inkList The inks to mix. Note: Must contain at least two process inks and one spot ink.
   * @param inkPercentages The percent to use of each ink in the ink list. (Range: 0 to 100 for each ink)
   * @param withProperties Initial values for properties of the new MixedInk
   */
  add(inkList: Ink[] | MixedInkGroup, inkPercentages: number[], withProperties: object): MixedInk

  /**
   * Returns any MixedInk in the collection.
   */
  anyItem(): MixedInk

  /**
   * Displays the number of elements in the MixedInk.
   */
  count(): number

  /**
   * Returns every MixedInk in the collection.
   */
  everyItem(): MixedInk

  /**
   * Returns the first MixedInk in the collection.
   */
  firstItem(): MixedInk

  /**
   * Returns the MixedInk with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): MixedInk

  /**
   * Returns the MixedInk with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): MixedInk

  /**
   * Returns the MixedInk with the specified name.
   * @param name The name.
   */
  itemByName(name: string): MixedInk

  /**
   * Returns the MixedInks within the specified range.
   * @param from The MixedInk, index, or name at the beginning of the range.
   * @param to The MixedInk, index, or name at the end of the range.
   */
  itemByRange(from: MixedInk | number | string, to: MixedInk | number | string): MixedInk[]

  /**
   * Returns the last MixedInk in the collection.
   */
  lastItem(): MixedInk

  /**
   * Returns the middle MixedInk in the collection.
   */
  middleItem(): MixedInk

  /**
   * Returns the MixedInk whose index follows the specified MixedInk in the collection.
   * @param obj The MixedInk whose index comes before the desired MixedInk.
   */
  nextItem(obj: MixedInk): MixedInk

  /**
   * Returns the MixedInk with the index previous to the specified index.
   * @param obj The index of the MixedInk that follows the desired MixedInk.
   */
  previousItem(obj: MixedInk): MixedInk

  /**
   * Generates a string which, if executed, will return the MixedInk.
   */
  toSource(): string
}

/**
 * A mixed ink group.
 */
declare class MixedInkGroup extends Swatch {
  /**
   * The component inks.
   */
  readonly inkList: Ink[]

  /**
   * The color model.
   */
  model: ColorModel
}

/**
 * A collection of mixed ink groups.
 */
declare class MixedInkGroups {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the MixedInkGroup with the specified index.
   * @param index The index.
   */
  [index: number]: MixedInkGroup

  /**
   * Creates a new mixed ink group.
   * @param inkList The inks to include in the mix.
   * @param inkPercentages The percent of each ink in the ink list. (Range: 0 to 100 for each ink)
   * @param repeatValues The number of repetitions for each ink in the ink list. (Range: 0 to 100 for each ink)
   * @param incrementValues The increment percent per repetition for each ink. (Range: 0 to 100) Note: The cumulative ink percentage per ink cannot exceed 100.)
   * @param withProperties Initial values for properties of the new MixedInkGroup
   */
  add(
    inkList: Ink[],
    inkPercentages: number[],
    repeatValues: number[],
    incrementValues: number[],
    withProperties: object,
  ): MixedInkGroup

  /**
   * Returns any MixedInkGroup in the collection.
   */
  anyItem(): MixedInkGroup

  /**
   * Displays the number of elements in the MixedInkGroup.
   */
  count(): number

  /**
   * Returns every MixedInkGroup in the collection.
   */
  everyItem(): MixedInkGroup

  /**
   * Returns the first MixedInkGroup in the collection.
   */
  firstItem(): MixedInkGroup

  /**
   * Returns the MixedInkGroup with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): MixedInkGroup

  /**
   * Returns the MixedInkGroup with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): MixedInkGroup

  /**
   * Returns the MixedInkGroup with the specified name.
   * @param name The name.
   */
  itemByName(name: string): MixedInkGroup

  /**
   * Returns the MixedInkGroups within the specified range.
   * @param from The MixedInkGroup, index, or name at the beginning of the range.
   * @param to The MixedInkGroup, index, or name at the end of the range.
   */
  itemByRange(
    from: MixedInkGroup | number | string,
    to: MixedInkGroup | number | string,
  ): MixedInkGroup[]

  /**
   * Returns the last MixedInkGroup in the collection.
   */
  lastItem(): MixedInkGroup

  /**
   * Returns the middle MixedInkGroup in the collection.
   */
  middleItem(): MixedInkGroup

  /**
   * Returns the MixedInkGroup whose index follows the specified MixedInkGroup in the collection.
   * @param obj The MixedInkGroup whose index comes before the desired MixedInkGroup.
   */
  nextItem(obj: MixedInkGroup): MixedInkGroup

  /**
   * Returns the MixedInkGroup with the index previous to the specified index.
   * @param obj The index of the MixedInkGroup that follows the desired MixedInkGroup.
   */
  previousItem(obj: MixedInkGroup): MixedInkGroup

  /**
   * Generates a string which, if executed, will return the MixedInkGroup.
   */
  toSource(): string
}

/**
 * A gradient.
 */
declare class Gradient extends Swatch {
  /**
   * A collection of gradient stops.
   */
  readonly gradientStops: GradientStops

  /**
   * The gradient type.
   */
  type: GradientType
}

/**
 * A collection of gradients.
 */
declare class Gradients {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Gradient with the specified index.
   * @param index The index.
   */
  [index: number]: Gradient

  /**
   * Creates a new Gradient.
   * @param withProperties Initial values for properties of the new Gradient
   */
  add(withProperties: object): Gradient

  /**
   * Returns any Gradient in the collection.
   */
  anyItem(): Gradient

  /**
   * Displays the number of elements in the Gradient.
   */
  count(): number

  /**
   * Returns every Gradient in the collection.
   */
  everyItem(): Gradient

  /**
   * Returns the first Gradient in the collection.
   */
  firstItem(): Gradient

  /**
   * Returns the Gradient with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Gradient

  /**
   * Returns the Gradient with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Gradient

  /**
   * Returns the Gradient with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Gradient

  /**
   * Returns the Gradients within the specified range.
   * @param from The Gradient, index, or name at the beginning of the range.
   * @param to The Gradient, index, or name at the end of the range.
   */
  itemByRange(from: Gradient | number | string, to: Gradient | number | string): Gradient[]

  /**
   * Returns the last Gradient in the collection.
   */
  lastItem(): Gradient

  /**
   * Returns the middle Gradient in the collection.
   */
  middleItem(): Gradient

  /**
   * Returns the Gradient whose index follows the specified Gradient in the collection.
   * @param obj The Gradient whose index comes before the desired Gradient.
   */
  nextItem(obj: Gradient): Gradient

  /**
   * Returns the Gradient with the index previous to the specified index.
   * @param obj The index of the Gradient that follows the desired Gradient.
   */
  previousItem(obj: Gradient): Gradient

  /**
   * Generates a string which, if executed, will return the Gradient.
   */
  toSource(): string
}

/**
 * A gradient stop in a gradient.
 */
declare class GradientStop {
  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The index of the GradientStop within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * The starting location (as a percentage of the gradient length) of the gradient stop on the gradient. (Range: 0 to 100).
   */
  location: number

  /**
   * The mid-point (as a percentage of the gradient length) of the gradient stop. (Range: 13 to 87)
   */
  midpoint: number

  /**
   * The parent of the GradientStop (a Gradient).
   */
  readonly parent: Gradient

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * The swatch (color, gradient, tint, or mixed ink) applied to the gradient stop.
   */
  stopColor: MixedInk | Color

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): GradientStop[]

  /**
   * Deletes the GradientStop.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the GradientStop.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of gradient stops.
 */
declare class GradientStops {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the GradientStop with the specified index.
   * @param index The index.
   */
  [index: number]: GradientStop

  /**
   * Creates a new GradientStop.
   * @param withProperties Initial values for properties of the new GradientStop
   */
  add(withProperties: object): GradientStop

  /**
   * Returns any GradientStop in the collection.
   */
  anyItem(): GradientStop

  /**
   * Displays the number of elements in the GradientStop.
   */
  count(): number

  /**
   * Returns every GradientStop in the collection.
   */
  everyItem(): GradientStop

  /**
   * Returns the first GradientStop in the collection.
   */
  firstItem(): GradientStop

  /**
   * Returns the GradientStop with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): GradientStop

  /**
   * Returns the GradientStops within the specified range.
   * @param from The GradientStop, index, or name at the beginning of the range.
   * @param to The GradientStop, index, or name at the end of the range.
   */
  itemByRange(
    from: GradientStop | number | string,
    to: GradientStop | number | string,
  ): GradientStop[]

  /**
   * Returns the last GradientStop in the collection.
   */
  lastItem(): GradientStop

  /**
   * Returns the middle GradientStop in the collection.
   */
  middleItem(): GradientStop

  /**
   * Returns the GradientStop whose index follows the specified GradientStop in the collection.
   * @param obj The GradientStop whose index comes before the desired GradientStop.
   */
  nextItem(obj: GradientStop): GradientStop

  /**
   * Returns the GradientStop with the index previous to the specified index.
   * @param obj The index of the GradientStop that follows the desired GradientStop.
   */
  previousItem(obj: GradientStop): GradientStop

  /**
   * Generates a string which, if executed, will return the GradientStop.
   */
  toSource(): string
}

/**
 * A swatch (color, gradient, tint, or mixed ink).
 */
declare class Swatch {
  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The unique ID of the Swatch.
   */
  readonly id: number

  /**
   * The index of the Swatch within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The name of the Swatch.
   */
  name: string

  /**
   * The parent of the Swatch (a Application or Document).
   */
  readonly parent: any

  /**
   * The color group that a swatch belongs to
   */
  readonly parentColorGroup: ColorGroup

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Duplicates the Swatch.
   */
  duplicate(): Swatch

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): Swatch[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Merges the specified swatches with the swatch.
   * @param with_ The swatches to merge.
   */
  merge(with_: Swatch[] | Swatch | string): Swatch

  /**
   * Deletes the swatch.
   * @param replacingWith The swatch to apply in place of the deleted swatch.
   */
  remove(replacingWith: Swatch): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the Swatch.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of swatches.
 */
declare class Swatches {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Swatch with the specified index.
   * @param index The index.
   */
  [index: number]: Swatch

  /**
   * Returns any Swatch in the collection.
   */
  anyItem(): Swatch

  /**
   * Displays the number of elements in the Swatch.
   */
  count(): number

  /**
   * Returns every Swatch in the collection.
   */
  everyItem(): Swatch

  /**
   * Returns the first Swatch in the collection.
   */
  firstItem(): Swatch

  /**
   * Returns the Swatch with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Swatch

  /**
   * Returns the Swatch with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Swatch

  /**
   * Returns the Swatch with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Swatch

  /**
   * Returns the Swatches within the specified range.
   * @param from The Swatch, index, or name at the beginning of the range.
   * @param to The Swatch, index, or name at the end of the range.
   */
  itemByRange(from: Swatch | number | string, to: Swatch | number | string): Swatch[]

  /**
   * Returns the last Swatch in the collection.
   */
  lastItem(): Swatch

  /**
   * Returns the middle Swatch in the collection.
   */
  middleItem(): Swatch

  /**
   * Returns the Swatch whose index follows the specified Swatch in the collection.
   * @param obj The Swatch whose index comes before the desired Swatch.
   */
  nextItem(obj: Swatch): Swatch

  /**
   * Returns the Swatch with the index previous to the specified index.
   * @param obj The index of the Swatch that follows the desired Swatch.
   */
  previousItem(obj: Swatch): Swatch

  /**
   * Generates a string which, if executed, will return the Swatch.
   */
  toSource(): string
}

/**
 * A color swatch.
 */
declare class Color extends Swatch {
  /**
   * The ink values that create the color, specified as a percentage for each ink. Note: The number of values required and the range depends on the color space. For RGB, specify three values, with each value in the range 0 to 255; for CMYK, specify four values representing C, M, Y, and K, with each value in the range 0 to 100; for LAB, specify three values representing L (Range: 0 to 100), A (Range: -128 to 127), and B (Range: -128 to 127); for mixed ink, specify values for each ink in the ink list, with each value in the range 0 to 100.
   */
  colorValue: number[]

  /**
   * The color model.
   */
  model: ColorModel

  /**
   * The color space.
   */
  space: ColorSpace
}

/**
 * A collection of colors.
 */
declare class Colors {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Color with the specified index.
   * @param index The index.
   */
  [index: number]: Color

  /**
   * Creates a new Color.
   * @param withProperties Initial values for properties of the new Color
   */
  add(withProperties: object): Color

  /**
   * Returns any Color in the collection.
   */
  anyItem(): Color

  /**
   * Displays the number of elements in the Color.
   */
  count(): number

  /**
   * Returns every Color in the collection.
   */
  everyItem(): Color

  /**
   * Returns the first Color in the collection.
   */
  firstItem(): Color

  /**
   * Returns the Color with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Color

  /**
   * Returns the Color with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Color

  /**
   * Returns the Color with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Color

  /**
   * Returns the Colors within the specified range.
   * @param from The Color, index, or name at the beginning of the range.
   * @param to The Color, index, or name at the end of the range.
   */
  itemByRange(from: Color | number | string, to: Color | number | string): Color[]

  /**
   * Returns the last Color in the collection.
   */
  lastItem(): Color

  /**
   * Returns the middle Color in the collection.
   */
  middleItem(): Color

  /**
   * Returns the Color whose index follows the specified Color in the collection.
   * @param obj The Color whose index comes before the desired Color.
   */
  nextItem(obj: Color): Color

  /**
   * Returns the Color with the index previous to the specified index.
   * @param obj The index of the Color that follows the desired Color.
   */
  previousItem(obj: Color): Color

  /**
   * Generates a string which, if executed, will return the Color.
   */
  toSource(): string
}

/**
 * A tint swatch.
 */
declare class Tint extends Color {
  /**
   * The color that the tint is based on.
   */
  readonly baseColor: Color

  /**
   * The percent of the base color. (Range: 0 to 100)
   */
  tintValue: number
}

/**
 * A collection of tints.
 */
declare class Tints {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Tint with the specified index.
   * @param index The index.
   */
  [index: number]: Tint

  /**
   * Creates a new tint swatch.
   * @param baseColor The color that the tint is based upon.
   * @param withProperties Initial values for properties of the new Tint
   */
  add(baseColor: Color, withProperties: object): Tint

  /**
   * Returns any Tint in the collection.
   */
  anyItem(): Tint

  /**
   * Displays the number of elements in the Tint.
   */
  count(): number

  /**
   * Returns every Tint in the collection.
   */
  everyItem(): Tint

  /**
   * Returns the first Tint in the collection.
   */
  firstItem(): Tint

  /**
   * Returns the Tint with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Tint

  /**
   * Returns the Tint with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Tint

  /**
   * Returns the Tint with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Tint

  /**
   * Returns the Tints within the specified range.
   * @param from The Tint, index, or name at the beginning of the range.
   * @param to The Tint, index, or name at the end of the range.
   */
  itemByRange(from: Tint | number | string, to: Tint | number | string): Tint[]

  /**
   * Returns the last Tint in the collection.
   */
  lastItem(): Tint

  /**
   * Returns the middle Tint in the collection.
   */
  middleItem(): Tint

  /**
   * Returns the Tint whose index follows the specified Tint in the collection.
   * @param obj The Tint whose index comes before the desired Tint.
   */
  nextItem(obj: Tint): Tint

  /**
   * Returns the Tint with the index previous to the specified index.
   * @param obj The index of the Tint that follows the desired Tint.
   */
  previousItem(obj: Tint): Tint

  /**
   * Generates a string which, if executed, will return the Tint.
   */
  toSource(): string
}

/**
 * A color group
 */
declare class ColorGroup {
  /**
   * A collection of color group swatches.
   */
  readonly colorGroupSwatches: ColorGroupSwatches

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The unique ID of the ColorGroup.
   */
  readonly id: number

  /**
   * The index of the ColorGroup within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The name of the ColorGroup.
   */
  name: string

  /**
   * The parent of the ColorGroup (a Application or Document).
   */
  readonly parent: any

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Duplicates the ColorGroup.
   */
  duplicate(): ColorGroup

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): ColorGroup[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Deletes the ColorGroup.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the ColorGroup.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string

  /**
   * Ungroups a color group
   */
  ungroup(): void
}

/**
 * A collection of color groups.
 */
declare class ColorGroups {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the ColorGroup with the specified index.
   * @param index The index.
   */
  [index: number]: ColorGroup

  /**
   * Creates a new ColorGroup
   * @param name The color group name
   * @param swatchList The swatches to add in color group.
   * @param withProperties Initial values for properties of the new ColorGroup
   */
  add(name: string, swatchList: Swatch[], withProperties: object): ColorGroup

  /**
   * Returns any ColorGroup in the collection.
   */
  anyItem(): ColorGroup

  /**
   * Displays the number of elements in the ColorGroup.
   */
  count(): number

  /**
   * Returns every ColorGroup in the collection.
   */
  everyItem(): ColorGroup

  /**
   * Returns the first ColorGroup in the collection.
   */
  firstItem(): ColorGroup

  /**
   * Returns the ColorGroup with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): ColorGroup

  /**
   * Returns the ColorGroup with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): ColorGroup

  /**
   * Returns the ColorGroup with the specified name.
   * @param name The name.
   */
  itemByName(name: string): ColorGroup

  /**
   * Returns the ColorGroups within the specified range.
   * @param from The ColorGroup, index, or name at the beginning of the range.
   * @param to The ColorGroup, index, or name at the end of the range.
   */
  itemByRange(from: ColorGroup | number | string, to: ColorGroup | number | string): ColorGroup[]

  /**
   * Returns the last ColorGroup in the collection.
   */
  lastItem(): ColorGroup

  /**
   * Returns the middle ColorGroup in the collection.
   */
  middleItem(): ColorGroup

  /**
   * Returns the ColorGroup whose index follows the specified ColorGroup in the collection.
   * @param obj The ColorGroup whose index comes before the desired ColorGroup.
   */
  nextItem(obj: ColorGroup): ColorGroup

  /**
   * Returns the ColorGroup with the index previous to the specified index.
   * @param obj The index of the ColorGroup that follows the desired ColorGroup.
   */
  previousItem(obj: ColorGroup): ColorGroup

  /**
   * Generates a string which, if executed, will return the ColorGroup.
   */
  toSource(): string
}

/**
 * A color group swatch.
 */
declare class ColorGroupSwatch {
  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The unique ID of the ColorGroupSwatch.
   */
  readonly id: number

  /**
   * The index of the ColorGroupSwatch within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * The parent of the ColorGroupSwatch (a ColorGroup).
   */
  readonly parent: ColorGroup

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * The underlying swatch item
   */
  readonly swatchItemRef: Swatch

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): ColorGroupSwatch[]

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the ColorGroupSwatch.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of color group swatches.
 */
declare class ColorGroupSwatches {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the ColorGroupSwatch with the specified index.
   * @param index The index.
   */
  [index: number]: ColorGroupSwatch

  /**
   * Adds a color group swatch.
   * @param swatchItemRef swatch to be added to color group
   * @param withProperties Initial values for properties of the new ColorGroupSwatch
   */
  add(swatchItemRef: Swatch, withProperties: object): ColorGroupSwatch

  /**
   * Returns any ColorGroupSwatch in the collection.
   */
  anyItem(): ColorGroupSwatch

  /**
   * Displays the number of elements in the ColorGroupSwatch.
   */
  count(): number

  /**
   * Returns every ColorGroupSwatch in the collection.
   */
  everyItem(): ColorGroupSwatch

  /**
   * Returns the first ColorGroupSwatch in the collection.
   */
  firstItem(): ColorGroupSwatch

  /**
   * Returns the ColorGroupSwatch with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): ColorGroupSwatch

  /**
   * Returns the ColorGroupSwatch with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): ColorGroupSwatch

  /**
   * Returns the ColorGroupSwatches within the specified range.
   * @param from The ColorGroupSwatch, index, or name at the beginning of the range.
   * @param to The ColorGroupSwatch, index, or name at the end of the range.
   */
  itemByRange(
    from: ColorGroupSwatch | number | string,
    to: ColorGroupSwatch | number | string,
  ): ColorGroupSwatch[]

  /**
   * Returns the last ColorGroupSwatch in the collection.
   */
  lastItem(): ColorGroupSwatch

  /**
   * Returns the middle ColorGroupSwatch in the collection.
   */
  middleItem(): ColorGroupSwatch

  /**
   * Returns the ColorGroupSwatch whose index follows the specified ColorGroupSwatch in the collection.
   * @param obj The ColorGroupSwatch whose index comes before the desired ColorGroupSwatch.
   */
  nextItem(obj: ColorGroupSwatch): ColorGroupSwatch

  /**
   * Returns the ColorGroupSwatch with the index previous to the specified index.
   * @param obj The index of the ColorGroupSwatch that follows the desired ColorGroupSwatch.
   */
  previousItem(obj: ColorGroupSwatch): ColorGroupSwatch

  /**
   * Generates a string which, if executed, will return the ColorGroupSwatch.
   */
  toSource(): string
}

/**
 * An opacity gradient stop.
 */
declare class OpacityGradientStop {
  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The index of the OpacityGradientStop within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * The location of the opacity gradient stop, as a percentage of the OpacityGradientStop size. (Range: 0 to 100)
   */
  location: number

  /**
   * The mid-point of the opacity gradient stop, as a percentage of the OpacityGradientStop size. (Range: 13 to 87)
   */
  midpoint: number

  /**
   * The opacity of the opacity gradient stop (as a percentage). (Range: 0 to 100)
   */
  opacity: number

  /**
   * The parent of the OpacityGradientStop (a GradientFeatherSetting or FindChangeGradientFeatherSetting).
   */
  readonly parent: any

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): OpacityGradientStop[]

  /**
   * Deletes the OpacityGradientStop.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the OpacityGradientStop.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of opacity gradient stops.
 */
declare class OpacityGradientStops {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the OpacityGradientStop with the specified index.
   * @param index The index.
   */
  [index: number]: OpacityGradientStop

  /**
   * Creates a new OpacityGradientStop.
   * @param withProperties Initial values for properties of the new OpacityGradientStop
   */
  add(withProperties: object): OpacityGradientStop

  /**
   * Returns any OpacityGradientStop in the collection.
   */
  anyItem(): OpacityGradientStop

  /**
   * Displays the number of elements in the OpacityGradientStop.
   */
  count(): number

  /**
   * Returns every OpacityGradientStop in the collection.
   */
  everyItem(): OpacityGradientStop

  /**
   * Returns the first OpacityGradientStop in the collection.
   */
  firstItem(): OpacityGradientStop

  /**
   * Returns the OpacityGradientStop with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): OpacityGradientStop

  /**
   * Returns the OpacityGradientStops within the specified range.
   * @param from The OpacityGradientStop, index, or name at the beginning of the range.
   * @param to The OpacityGradientStop, index, or name at the end of the range.
   */
  itemByRange(
    from: OpacityGradientStop | number | string,
    to: OpacityGradientStop | number | string,
  ): OpacityGradientStop[]

  /**
   * Returns the last OpacityGradientStop in the collection.
   */
  lastItem(): OpacityGradientStop

  /**
   * Returns the middle OpacityGradientStop in the collection.
   */
  middleItem(): OpacityGradientStop

  /**
   * Returns the OpacityGradientStop whose index follows the specified OpacityGradientStop in the collection.
   * @param obj The OpacityGradientStop whose index comes before the desired OpacityGradientStop.
   */
  nextItem(obj: OpacityGradientStop): OpacityGradientStop

  /**
   * Returns the OpacityGradientStop with the index previous to the specified index.
   * @param obj The index of the OpacityGradientStop that follows the desired OpacityGradientStop.
   */
  previousItem(obj: OpacityGradientStop): OpacityGradientStop

  /**
   * Generates a string which, if executed, will return the OpacityGradientStop.
   */
  toSource(): string
}

/**
 * An object that can contain a data merge text field.
 */
declare class DataMergeTextPlaceholder {
  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The data merge field to insert in the placeholder. For information, see data merge text placeholder and data merge image placeholder.
   */
  readonly field: DataMergeField

  /**
   * The index of the DataMergeTextPlaceholder within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * The parent of the DataMergeTextPlaceholder (a Document).
   */
  readonly parent: Document

  /**
   * The story that contains the text.
   */
  readonly parentStory: Story

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * The insertion point before the table in the story containing the table.
   */
  readonly storyOffset: InsertionPoint

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): DataMergeTextPlaceholder[]

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the DataMergeTextPlaceholder.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of data merge text placeholders.
 */
declare class DataMergeTextPlaceholders {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the DataMergeTextPlaceholder with the specified index.
   * @param index The index.
   */
  [index: number]: DataMergeTextPlaceholder

  /**
   * Creates a data merge text placeholder.
   * @param parentStory The story in which to insert the placeholder.
   * @param storyOffset The position within the story at which to insert the placeholder, specified as an offset number or an insertion point.
   * @param field The field to insert.
   * @param withProperties Initial values for properties of the new DataMergeTextPlaceholder
   */
  add(
    parentStory: Story,
    storyOffset: InsertionPoint | number,
    field: DataMergeField,
    withProperties: object,
  ): DataMergeTextPlaceholder

  /**
   * Returns any DataMergeTextPlaceholder in the collection.
   */
  anyItem(): DataMergeTextPlaceholder

  /**
   * Displays the number of elements in the DataMergeTextPlaceholder.
   */
  count(): number

  /**
   * Returns every DataMergeTextPlaceholder in the collection.
   */
  everyItem(): DataMergeTextPlaceholder

  /**
   * Returns the first DataMergeTextPlaceholder in the collection.
   */
  firstItem(): DataMergeTextPlaceholder

  /**
   * Returns the DataMergeTextPlaceholder with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): DataMergeTextPlaceholder

  /**
   * Returns the DataMergeTextPlaceholders within the specified range.
   * @param from The DataMergeTextPlaceholder, index, or name at the beginning of the range.
   * @param to The DataMergeTextPlaceholder, index, or name at the end of the range.
   */
  itemByRange(
    from: DataMergeTextPlaceholder | number | string,
    to: DataMergeTextPlaceholder | number | string,
  ): DataMergeTextPlaceholder[]

  /**
   * Returns the last DataMergeTextPlaceholder in the collection.
   */
  lastItem(): DataMergeTextPlaceholder

  /**
   * Returns the middle DataMergeTextPlaceholder in the collection.
   */
  middleItem(): DataMergeTextPlaceholder

  /**
   * Returns the DataMergeTextPlaceholder whose index follows the specified DataMergeTextPlaceholder in the collection.
   * @param obj The DataMergeTextPlaceholder whose index comes before the desired DataMergeTextPlaceholder.
   */
  nextItem(obj: DataMergeTextPlaceholder): DataMergeTextPlaceholder

  /**
   * Returns the DataMergeTextPlaceholder with the index previous to the specified index.
   * @param obj The index of the DataMergeTextPlaceholder that follows the desired DataMergeTextPlaceholder.
   */
  previousItem(obj: DataMergeTextPlaceholder): DataMergeTextPlaceholder

  /**
   * Generates a string which, if executed, will return the DataMergeTextPlaceholder.
   */
  toSource(): string
}

/**
 * An object that can contain a date merge image field.
 */
declare class DataMergeImagePlaceholder {
  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The data merge field to insert in the placeholder. For information, see data merge text placeholder and data merge image placeholder.
   */
  readonly field: DataMergeField

  /**
   * The index of the DataMergeImagePlaceholder within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * The parent of the DataMergeImagePlaceholder (a Document).
   */
  readonly parent: Document

  /**
   * The page item on which to place the placeholder.
   */
  readonly placeholderPageItem: PageItem

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): DataMergeImagePlaceholder[]

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the DataMergeImagePlaceholder.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of data merge image placeholders.
 */
declare class DataMergeImagePlaceholders {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the DataMergeImagePlaceholder with the specified index.
   * @param index The index.
   */
  [index: number]: DataMergeImagePlaceholder

  /**
   * Creates a data merge image placeholder.
   * @param placeholder The page item on which to place the placeholder.
   * @param field The data merge field to insert.
   * @param withProperties Initial values for properties of the new DataMergeImagePlaceholder
   */
  add(
    placeholder: PageItem,
    field: DataMergeField,
    withProperties: object,
  ): DataMergeImagePlaceholder

  /**
   * Returns any DataMergeImagePlaceholder in the collection.
   */
  anyItem(): DataMergeImagePlaceholder

  /**
   * Displays the number of elements in the DataMergeImagePlaceholder.
   */
  count(): number

  /**
   * Returns every DataMergeImagePlaceholder in the collection.
   */
  everyItem(): DataMergeImagePlaceholder

  /**
   * Returns the first DataMergeImagePlaceholder in the collection.
   */
  firstItem(): DataMergeImagePlaceholder

  /**
   * Returns the DataMergeImagePlaceholder with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): DataMergeImagePlaceholder

  /**
   * Returns the DataMergeImagePlaceholders within the specified range.
   * @param from The DataMergeImagePlaceholder, index, or name at the beginning of the range.
   * @param to The DataMergeImagePlaceholder, index, or name at the end of the range.
   */
  itemByRange(
    from: DataMergeImagePlaceholder | number | string,
    to: DataMergeImagePlaceholder | number | string,
  ): DataMergeImagePlaceholder[]

  /**
   * Returns the last DataMergeImagePlaceholder in the collection.
   */
  lastItem(): DataMergeImagePlaceholder

  /**
   * Returns the middle DataMergeImagePlaceholder in the collection.
   */
  middleItem(): DataMergeImagePlaceholder

  /**
   * Returns the DataMergeImagePlaceholder whose index follows the specified DataMergeImagePlaceholder in the collection.
   * @param obj The DataMergeImagePlaceholder whose index comes before the desired DataMergeImagePlaceholder.
   */
  nextItem(obj: DataMergeImagePlaceholder): DataMergeImagePlaceholder

  /**
   * Returns the DataMergeImagePlaceholder with the index previous to the specified index.
   * @param obj The index of the DataMergeImagePlaceholder that follows the desired DataMergeImagePlaceholder.
   */
  previousItem(obj: DataMergeImagePlaceholder): DataMergeImagePlaceholder

  /**
   * Generates a string which, if executed, will return the DataMergeImagePlaceholder.
   */
  toSource(): string
}

/**
 * A data merge field.
 */
declare class DataMergeField {
  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The name of the field.
   */
  readonly fieldName: string

  /**
   * The source field type.
   */
  readonly fieldType: SourceFieldType

  /**
   * The index of the DataMergeField within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * The parent of the DataMergeField (a DataMerge).
   */
  readonly parent: DataMerge

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): DataMergeField[]

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the DataMergeField.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of data merge fields.
 */
declare class DataMergeFields {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the DataMergeField with the specified index.
   * @param index The index.
   */
  [index: number]: DataMergeField

  /**
   * Returns any DataMergeField in the collection.
   */
  anyItem(): DataMergeField

  /**
   * Displays the number of elements in the DataMergeField.
   */
  count(): number

  /**
   * Returns every DataMergeField in the collection.
   */
  everyItem(): DataMergeField

  /**
   * Returns the first DataMergeField in the collection.
   */
  firstItem(): DataMergeField

  /**
   * Returns the DataMergeField with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): DataMergeField

  /**
   * Returns the DataMergeFields within the specified range.
   * @param from The DataMergeField, index, or name at the beginning of the range.
   * @param to The DataMergeField, index, or name at the end of the range.
   */
  itemByRange(
    from: DataMergeField | number | string,
    to: DataMergeField | number | string,
  ): DataMergeField[]

  /**
   * Returns the last DataMergeField in the collection.
   */
  lastItem(): DataMergeField

  /**
   * Returns the middle DataMergeField in the collection.
   */
  middleItem(): DataMergeField

  /**
   * Returns the DataMergeField whose index follows the specified DataMergeField in the collection.
   * @param obj The DataMergeField whose index comes before the desired DataMergeField.
   */
  nextItem(obj: DataMergeField): DataMergeField

  /**
   * Returns the DataMergeField with the index previous to the specified index.
   * @param obj The index of the DataMergeField that follows the desired DataMergeField.
   */
  previousItem(obj: DataMergeField): DataMergeField

  /**
   * Generates a string which, if executed, will return the DataMergeField.
   */
  toSource(): string
}

/**
 * An object that can contain a date merge QR code field.
 */
declare class DataMergeQrcodePlaceholder {
  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The data merge field to insert in the placeholder. For information, see data merge text placeholder and data merge image placeholder.
   */
  readonly field: DataMergeField

  /**
   * The index of the DataMergeQrcodePlaceholder within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * The parent of the DataMergeQrcodePlaceholder (a Document).
   */
  readonly parent: Document

  /**
   * The page item on which to place the placeholder.
   */
  readonly placeholderPageItem: PageItem

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): DataMergeQrcodePlaceholder[]

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the DataMergeQrcodePlaceholder.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of data merge QR code placeholders.
 */
declare class DataMergeQrcodePlaceholders {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the DataMergeQrcodePlaceholder with the specified index.
   * @param index The index.
   */
  [index: number]: DataMergeQrcodePlaceholder

  /**
   * Creates a data merge QR code placeholder.
   * @param placeholder The page item on which to place the placeholder.
   * @param field The data merge field to insert.
   * @param withProperties Initial values for properties of the new DataMergeQrcodePlaceholder
   */
  add(
    placeholder: PageItem,
    field: DataMergeField,
    withProperties: object,
  ): DataMergeQrcodePlaceholder

  /**
   * Returns any DataMergeQrcodePlaceholder in the collection.
   */
  anyItem(): DataMergeQrcodePlaceholder

  /**
   * Displays the number of elements in the DataMergeQrcodePlaceholder.
   */
  count(): number

  /**
   * Returns every DataMergeQrcodePlaceholder in the collection.
   */
  everyItem(): DataMergeQrcodePlaceholder

  /**
   * Returns the first DataMergeQrcodePlaceholder in the collection.
   */
  firstItem(): DataMergeQrcodePlaceholder

  /**
   * Returns the DataMergeQrcodePlaceholder with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): DataMergeQrcodePlaceholder

  /**
   * Returns the DataMergeQrcodePlaceholders within the specified range.
   * @param from The DataMergeQrcodePlaceholder, index, or name at the beginning of the range.
   * @param to The DataMergeQrcodePlaceholder, index, or name at the end of the range.
   */
  itemByRange(
    from: DataMergeQrcodePlaceholder | number | string,
    to: DataMergeQrcodePlaceholder | number | string,
  ): DataMergeQrcodePlaceholder[]

  /**
   * Returns the last DataMergeQrcodePlaceholder in the collection.
   */
  lastItem(): DataMergeQrcodePlaceholder

  /**
   * Returns the middle DataMergeQrcodePlaceholder in the collection.
   */
  middleItem(): DataMergeQrcodePlaceholder

  /**
   * Returns the DataMergeQrcodePlaceholder whose index follows the specified DataMergeQrcodePlaceholder in the collection.
   * @param obj The DataMergeQrcodePlaceholder whose index comes before the desired DataMergeQrcodePlaceholder.
   */
  nextItem(obj: DataMergeQrcodePlaceholder): DataMergeQrcodePlaceholder

  /**
   * Returns the DataMergeQrcodePlaceholder with the index previous to the specified index.
   * @param obj The index of the DataMergeQrcodePlaceholder that follows the desired DataMergeQrcodePlaceholder.
   */
  previousItem(obj: DataMergeQrcodePlaceholder): DataMergeQrcodePlaceholder

  /**
   * Generates a string which, if executed, will return the DataMergeQrcodePlaceholder.
   */
  toSource(): string
}

/**
 * Story type options.
 */
declare enum StoryTypes {
  /**
   * The story is an index.
   */
  INDEXING_STORY = 1768190836,

  /**
   * The story is a regular text story.
   */
  REGULAR_STORY = 1919382388,

  /**
   * The story is a table of contents.
   */
  TOC_STORY = 1953461108,
}

/**
 * Story direction options.
 */
declare enum StoryDirectionOptions {
  /**
   * Left to right direction.
   */
  LEFT_TO_RIGHT_DIRECTION = 1278366308,

  /**
   * Right to left direction.
   */
  RIGHT_TO_LEFT_DIRECTION = 1379028068,

  /**
   * Unknown direction.
   */
  UNKNOWN_DIRECTION = 1433299812,
}

/**
 * Style import options.
 */
declare enum ImportFormat {
  /**
   * Imports cell styles.
   */
  CELL_STYLES_FORMAT = 1698919284,

  /**
   * Imports character styles.
   */
  CHARACTER_STYLES_FORMAT = 1131565940,

  /**
   * Imports object styles.
   */
  OBJECT_STYLES_FORMAT = 1332368244,

  /**
   * Imports paragraph styles.
   */
  PARAGRAPH_STYLES_FORMAT = 1885885300,

  /**
   * Imports stroke styles.
   */
  STROKE_STYLES_FORMAT = 1817408620,

  /**
   * Imports table and cell styles.
   */
  TABLE_AND_CELL_STYLES_FORMAT = 1700021107,

  /**
   * Imports table styles.
   */
  TABLE_STYLES_FORMAT = 1700033396,

  /**
   * Imports character and paragraph styles.
   */
  TEXT_STYLES_FORMAT = 1668305780,

  /**
   * Imports table of contents styles.
   */
  TOC_STYLES_FORMAT = 1415795572,
}

/**
 * Delimiter options for marking the end of the application of the nested style.
 */
declare enum NestedStyleDelimiters {
  /**
   * Uses the first character or characters other than zero-width markers as the nested style delimiter. Zero-width markers include anchors, index markers, XML tags, and so on.
   */
  ANY_CHARACTER = 1380541539,

  /**
   * Uses the first word or words in the paragraph as the nested style delimiter. The first word is considered all characters preceding the first space or white space character in the paragraph.
   */
  ANY_WORD = 1380541559,

  /**
   * Uses the first automatic page number as the nested style delimiter.
   */
  AUTO_PAGE_NUMBER = 1396797550,

  /**
   * Uses the first numeric character or characters as the nested style delimiter. Note: To specify the number of digits, see repetition.
   */
  DIGITS = 1380541540,

  /**
   * Uses the dropcap as the nested style delimiter.
   */
  DROPCAP = 1380541507,

  /**
   * Uses the first em space as the nested style delimiter.
   */
  EM_SPACE = 1397058899,

  /**
   * Uses the inserted end nested style here character as the nested style delimiter.
   */
  END_NESTED_STYLE = 1396855379,

  /**
   * Uses the first en space as the nested style delimiter.
   */
  EN_SPACE = 1397059155,

  /**
   * Uses the first forced line break as the nested style delimiter.
   */
  FORCED_LINE_BREAK = 1397124194,

  /**
   * Uses the first indent-to-here tab character as the nested style delimiter. Note: Does not use the first indent-to-here tab stop. If no actual indent-to-here tab character has been inserted in the paragraph, the nested style is applied through the end of the paragraph.
   */
  INDENT_HERE_TAB = 1397319796,

  /**
   * Uses the first inline graphic as the nested style delimiter.
   */
  INLINE_GRAPHIC = 1380541545,

  /**
   * Uses the first alpha character or characters as the nested style delimiter. Note: To specify the number of letters, see repetition.
   */
  LETTERS = 1380541548,

  /**
   * Uses the first nonbreaking space as the nested style delimiter.
   */
  NONBREAKING_SPACE = 1397645907,

  /**
   * Repeat
   */
  REPEAT = 1380545132,

  /**
   * Uses the first section name marker as the nested style delimiter.
   */
  SECTION_MARKER = 1400073805,

  /**
   * Uses the first sentence or sentences in the paragraph as the nested style delimiter. The first sentence is considered all text preceding the first period, question mark, or exclamation mark in the paragraph.
   */
  SENTENCE = 1380541555,

  /**
   * Uses the first tab character as the nested style delimiter. Note: Does not use the first tab stop. If no actual tab character has been inserted in the paragraph, the nested style is applied through the end of the paragraph.
   */
  TABS = 1380541556,
}

/**
 * Options for specifying an object on which to base the width of the paragraph rule above.
 */
declare enum RuleWidth {
  /**
   * Makes the rule the width of the column.
   */
  COLUMN_WIDTH = 1265399652,

  /**
   * Makes the paragraph rule above the width of the first line of text in the paragraph.
   */
  TEXT_WIDTH = 1886681207,
}

/**
 * Text alignment options.
 */
declare enum Justification {
  /**
   * Aligns text to the side opposite the binding spine of the page.
   */
  AWAY_FROM_BINDING_SIDE = 1633772147,

  /**
   * Center aligns the text.
   */
  CENTER_ALIGN = 1667591796,

  /**
   * Justifies text text and center aligns the last line in the paragraph.
   */
  CENTER_JUSTIFIED = 1667920756,

  /**
   * Justifies the text, including the last line in the paragraph.
   */
  FULLY_JUSTIFIED = 1718971500,

  /**
   * Left aligns the text.
   */
  LEFT_ALIGN = 1818584692,

  /**
   * Justifies the text and left aligns the last line in the paragraph.
   */
  LEFT_JUSTIFIED = 1818915700,

  /**
   * Right aligns the text.
   */
  RIGHT_ALIGN = 1919379572,

  /**
   * Justifies the text and right aligns the last line in the paragraph.
   */
  RIGHT_JUSTIFIED = 1919578996,

  /**
   * Aligns text to the binding spine of the page or spread.
   */
  TO_BINDING_SIDE = 1630691955,
}

/**
 * Alignment options for lines that contain a single word.
 */
declare enum SingleWordJustification {
  /**
   * Center aligns the word.
   */
  CENTER_ALIGN = 1667591796,

  /**
   * Fully justifies the word.
   */
  FULLY_JUSTIFIED = 1718971500,

  /**
   * Left aligns the word.
   */
  LEFT_ALIGN = 1818584692,

  /**
   * Right aligns the word.
   */
  RIGHT_ALIGN = 1919379572,
}

/**
 * Column and page break options.
 */
declare enum StartParagraph {
  /**
   * Starts in the next available space.
   */
  ANYWHERE = 1851945579,

  /**
   * Starts at the top of the next column.
   */
  NEXT_COLUMN = 1667396203,

  /**
   * Starts at the top of the next even-numbered page.
   */
  NEXT_EVEN_PAGE = 1164993131,

  /**
   * Starts at the top of the next text frame in the thread.
   */
  NEXT_FRAME = 1313235563,

  /**
   * Starts at the top of the next odd-numbered page.
   */
  NEXT_ODD_PAGE = 1332765291,

  /**
   * Starts at the top of the next page.
   */
  NEXT_PAGE = 1885500011,
}

/**
 * Orientation options.
 */
declare enum HorizontalOrVertical {
  /**
   * Horizontal orientation.
   */
  HORIZONTAL = 1752134266,

  /**
   * Vertical orientation.
   */
  VERTICAL = 1986359924,
}

/**
 * Tab stop alignment options.
 */
declare enum TabStopAlignment {
  /**
   * Center.
   */
  CENTER_ALIGN = 1667591796,

  /**
   * Aligns the specified character with the tab stop.
   */
  CHARACTER_ALIGN = 1952604515,

  /**
   * Left.
   */
  LEFT_ALIGN = 1818584692,

  /**
   * Right.
   */
  RIGHT_ALIGN = 1919379572,
}

/**
 * Starting point options for the first baseline of text.
 */
declare enum FirstBaseline {
  /**
   * The tallest character in the font falls below the top inset of the object.
   */
  ASCENT_OFFSET = 1296135023,

  /**
   * The tops of upper case letters touch the top inset of the object.
   */
  CAP_HEIGHT = 1296255087,

  /**
   * The text em box height is the distance between the baseline of the text and the top inset of the object.
   */
  EMBOX_HEIGHT = 1296386159,

  /**
   * Uses the value specified for minimum first baseline offset as the distance between the baseline of the text and the top inset of the object.
   */
  FIXED_HEIGHT = 1313228911,

  /**
   * The text leading value defines the distance between the baseline of the text and the top inset of the object.
   */
  LEADING_OFFSET = 1296852079,

  /**
   * The tops of lower case letters touch the top inset of the object.
   */
  X_HEIGHT = 1299728495,
}

/**
 * Vertical alignment options for text.
 */
declare enum VerticalJustification {
  /**
   * Text is aligned at the bottom of the object.
   */
  BOTTOM_ALIGN = 1651471469,

  /**
   * Text is center aligned vertically in the object.
   */
  CENTER_ALIGN = 1667591796,

  /**
   * Lines of text are evenly distributed vertically between the top and bottom of the object.
   */
  JUSTIFY_ALIGN = 1785951334,

  /**
   * Text is aligned at the top of the object.
   */
  TOP_ALIGN = 1953460256,
}

/**
 * Leading type options.
 */
declare enum Leading {
  /**
   * Apply auto leading.
   */
  AUTO = 1635019116,
}

/**
 * Figure style options for OpenType fonts.
 */
declare enum OTFFigureStyle {
  /**
   * Use the default figure style for the font.
   */
  DEFAULT_VALUE = 1147563124,

  /**
   * Use proportional width lining figures.
   */
  PROPORTIONAL_LINING = 1330932848,

  /**
   * Use proportional width oldstyle figures.
   */
  PROPORTIONAL_OLDSTYLE = 1330933619,

  /**
   * Use monspaced lining figures.
   */
  TABULAR_LINING = 1330931316,

  /**
   * Use monospaced oldstyle figures.
   */
  TABULAR_OLDSTYLE = 1330933620,
}

/**
 * Capitalization options.
 */
declare enum Capitalization {
  /**
   * Use all uppercase letters.
   */
  ALL_CAPS = 1634493296,

  /**
   * Use OpenType small caps.
   */
  CAP_TO_SMALL_CAP = 1664250723,

  /**
   * Do not change the capitalization of the text.
   */
  NORMAL = 1852797549,

  /**
   * Use small caps for lowercase letters.
   */
  SMALL_CAPS = 1936548720,
}

/**
 * Text position options.
 */
declare enum Position {
  /**
   * Normal position
   */
  NORMAL = 1852797549,

  /**
   * For OpenType fonts, shrinks the text but keeps text on the main text baseline. Note: Valid only for numeric characters.
   */
  OT_DENOMINATOR = 1884247140,

  /**
   * For OpenType fonts, shrinks the text but keeps the top of the characters aligned with the top of the main text. Note: Valid only for numeric characters.
   */
  OT_NUMERATOR = 1884247150,

  /**
   * For OpenType fonts, uses--if available--lowered glyphs that are sized correctly relative to the surrounding characters.
   */
  OT_SUBSCRIPT = 1884247138,

  /**
   * For OpenType fonts, uses--if available--raised glyphs that are sized correctly relative to the surrounding characters.
   */
  OT_SUPERSCRIPT = 1884247155,

  /**
   * Subscripts the text.
   */
  SUBSCRIPT = 1935831907,

  /**
   * Superscripts the text.
   */
  SUPERSCRIPT = 1936749411,
}

/**
 * Text frame content type options.
 */
declare enum TextFrameContents {
  /**
   * Fills the text frame with placeholder text.
   */
  PLACEHOLDER_TEXT = 1346925688,

  /**
   * Fills the text frame with arabic placeholder text.
   */
  PLACEHOLDER_TEXT_ARABIC = 1346925665,

  /**
   * Fills the text frame with cyrillic placeholder text.
   */
  PLACEHOLDER_TEXT_CYRILLIC = 1346925667,

  /**
   * Fills the text frame with greek placeholder text.
   */
  PLACEHOLDER_TEXT_GREEK = 1346925671,

  /**
   * Fills the text frame with hebrew placeholder text.
   */
  PLACEHOLDER_TEXT_HEBREW = 1346925672,
}

/**
 * Special character options.
 */
declare enum SpecialCharacters {
  /**
   * Inserts an arabic comma.
   */
  ARABIC_COMMA = 1396798051,

  /**
   * Inserts an arabic kashida.
   */
  ARABIC_KASHIDA = 1396798059,

  /**
   * Inserts an arabic question mark.
   */
  ARABIC_QUESTION_MARK = 1396797805,

  /**
   * Inserts an arabic semicolon.
   */
  ARABIC_SEMICOLON = 1396798307,

  /**
   * Inserts an automatic page number.
   */
  AUTO_PAGE_NUMBER = 1396797550,

  /**
   * Inserts a bullet character.
   */
  BULLET_CHARACTER = 1396862068,

  /**
   * Inserts a column break.
   */
  COLUMN_BREAK = 1396927554,

  /**
   * Inserts a copyright symbol.
   */
  COPYRIGHT_SYMBOL = 1396929140,

  /**
   * Inserts a degree symbol.
   */
  DEGREE_SYMBOL = 1396991858,

  /**
   * Inserts a discretionary hyphen.
   */
  DISCRETIONARY_HYPHEN = 1396983920,

  /**
   * Inserts a discretionary line break.
   */
  DISCRETIONARY_LINE_BREAK = 1397777484,

  /**
   * Inserts a dotted circle.
   */
  DOTTED_CIRCLE = 1399092323,

  /**
   * Inserts a double left quote.
   */
  DOUBLE_LEFT_QUOTE = 1396984945,

  /**
   * Inserts a double right quote.
   */
  DOUBLE_RIGHT_QUOTE = 1396986481,

  /**
   * Inserts a double straight quote.
   */
  DOUBLE_STRAIGHT_QUOTE = 1396986737,

  /**
   * Inserts an ellipsis character.
   */
  ELLIPSIS_CHARACTER = 1397518451,

  /**
   * Inserts an em dash.
   */
  EM_DASH = 1397058884,

  /**
   * Inserts an em space.
   */
  EM_SPACE = 1397058899,

  /**
   * Inserts an end nested style here character.
   */
  END_NESTED_STYLE = 1396855379,

  /**
   * Inserts an en dash.
   */
  EN_DASH = 1397059140,

  /**
   * Inserts an en space.
   */
  EN_SPACE = 1397059155,

  /**
   * Inserts a break to the next even page.
   */
  EVEN_PAGE_BREAK = 1397059650,

  /**
   * Inserts a figure space.
   */
  FIGURE_SPACE = 1397122899,

  /**
   * Inserts a fixed-width nonbreaking space.
   */
  FIXED_WIDTH_NONBREAKING_SPACE = 1399746146,

  /**
   * Inserts a flush space.
   */
  FLUSH_SPACE = 1397124179,

  /**
   * Inserts a footnote symbol.
   */
  FOOTNOTE_SYMBOL = 1399221837,

  /**
   * Inserts a forced line break.
   */
  FORCED_LINE_BREAK = 1397124194,

  /**
   * Inserts a frame break.
   */
  FRAME_BREAK = 1397125698,

  /**
   * Inserts a hair space.
   */
  HAIR_SPACE = 1397256787,

  /**
   * Inserts a hebrew geresh.
   */
  HEBREW_GERESH = 1397253989,

  /**
   * Inserts a hebrew gershayim.
   */
  HEBREW_GERSHAYIM = 1397254003,

  /**
   * Inserts a hebrew maqaf.
   */
  HEBREW_MAQAF = 1397252717,

  /**
   * Inserts a hebrew sof pasuk.
   */
  HEBREW_SOF_PASUK = 1397252723,

  /**
   * Inserts an indent to here character.
   */
  INDENT_HERE_TAB = 1397319796,

  /**
   * Inserts a left to right embedding mark.
   */
  LEFT_TO_RIGHT_EMBEDDING = 1399616101,

  /**
   * Inserts a left to right mark.
   */
  LEFT_TO_RIGHT_MARK = 1399616109,

  /**
   * Inserts a left to right override mark.
   */
  LEFT_TO_RIGHT_OVERRIDE = 1399616111,

  /**
   * Inserts the next page number.
   */
  NEXT_PAGE_NUMBER = 1397649518,

  /**
   * Inserts a nonbreaking hyphen.
   */
  NONBREAKING_HYPHEN = 1397645928,

  /**
   * Inserts a nonbreaking space.
   */
  NONBREAKING_SPACE = 1397645907,

  /**
   * Inserts a break to the next odd page.
   */
  ODD_PAGE_BREAK = 1397715010,

  /**
   * Inserts a page break.
   */
  PAGE_BREAK = 1397778242,

  /**
   * Inserts a paragraph symbol.
   */
  PARAGRAPH_SYMBOL = 1397776754,

  /**
   * Inserts a pop directional formatting mark.
   */
  POP_DIRECTIONAL_FORMATTING = 1399874662,

  /**
   * Inserts the previous page number.
   */
  PREVIOUS_PAGE_NUMBER = 1397780590,

  /**
   * Inserts a punctuation space.
   */
  PUNCTUATION_SPACE = 1397780051,

  /**
   * Inserts a quarter-width space.
   */
  QUARTER_SPACE = 1397847379,

  /**
   * Inserts a registered trademark.
   */
  REGISTERED_TRADEMARK = 1397904493,

  /**
   * Inserts a right indent tab.
   */
  RIGHT_INDENT_TAB = 1397909876,

  /**
   * Inserts a right to left embedding mark.
   */
  RIGHT_TO_LEFT_EMBEDDING = 1400007781,

  /**
   * Inserts a right to left mark.
   */
  RIGHT_TO_LEFT_MARK = 1400007789,

  /**
   * Inserts a right to left override mark.
   */
  RIGHT_TO_LEFT_OVERRIDE = 1400007791,

  /**
   * Inserts a section marker.
   */
  SECTION_MARKER = 1400073805,

  /**
   * Inserts a section symbol.
   */
  SECTION_SYMBOL = 1400073811,

  /**
   * Inserts a single left quote.
   */
  SINGLE_LEFT_QUOTE = 1397967985,

  /**
   * Inserts a single right quote.
   */
  SINGLE_RIGHT_QUOTE = 1397969521,

  /**
   * Inserts a single straight quote.
   */
  SINGLE_STRAIGHT_QUOTE = 1397969777,

  /**
   * Inserts a sixth-width space.
   */
  SIXTH_SPACE = 1397975379,

  /**
   * Inserts the specified text variable.
   */
  TEXT_VARIABLE = 1397781622,

  /**
   * Inserts a thin space.
   */
  THIN_SPACE = 1398042195,

  /**
   * Inserts a third-width space.
   */
  THIRD_SPACE = 1398040659,

  /**
   * Inserts a trademark symbol.
   */
  TRADEMARK_SYMBOL = 1398041963,

  /**
   * Inserts a zero width joiner.
   */
  ZERO_WIDTH_JOINER = 1400534890,

  /**
   * Inserts a zero-width non-joiner.
   */
  ZERO_WIDTH_NONJOINER = 1397780074,
}

/**
 * Text case options.
 */
declare enum ChangecaseMode {
  /**
   * Makes all letters lowercase.
   */
  LOWERCASE = 1667460195,

  /**
   * Makes the first letter of each sentence uppercase.
   */
  SENTENCECASE = 1667461987,

  /**
   * Makes the first letter of each word uppercase.
   */
  TITLECASE = 1667462243,

  /**
   * Makes all letters uppercase.
   */
  UPPERCASE = 1667462499,
}

/**
 * The location of the binding spine in a spread.
 */
declare enum BindingOptions {
  /**
   * Uses the default binding side.
   */
  DEFAULT_VALUE = 1147563124,

  /**
   * Moves the page to the left side of the spread's binding spine.
   */
  LEFT_ALIGN = 1818584692,

  /**
   * Moves the page to the right side of the spread's binding spine.
   */
  RIGHT_ALIGN = 1919379572,
}

/**
 * Page orientation options.
 */
declare enum PageOrientation {
  /**
   * Landscape.
   */
  LANDSCAPE = 2003395685,

  /**
   * Portrait.
   */
  PORTRAIT = 1751738216,
}

/**
 * Zero point placement options.
 */
declare enum RulerOrigin {
  /**
   * The top-left corner of each page is a new zero point on the horizontal ruler.
   */
  PAGE_ORIGIN = 1380143215,

  /**
   * The zero point is at the top-left corner of the left-most page and at the top of the binding spine. The horizontal ruler measures from the leftmost page to the binding edge, and from the binding spine through the right edge of the right-most page. Also locks the zero point and prevents manual overrides.
   */
  SPINE_ORIGIN = 1380143984,

  /**
   * The zero point is at the top-left corner of the spread and the ruler increments continuously across all pages of the spread.
   */
  SPREAD_ORIGIN = 1380143983,
}

/**
 * The unit of measurement.
 */
declare enum MeasurementUnits {
  /**
   * Agates
   */
  AGATES = 2051106676,

  /**
   * American points.
   */
  AMERICAN_POINTS = 1514238068,

  /**
   * Bai
   */
  BAI = 2051170665,

  /**
   * Centimeters.
   */
  CENTIMETERS = 2053336435,

  /**
   * Ciceros.
   */
  CICEROS = 2053335395,

  /**
   * Uses points as the unit of measurement and specifies the number of points between major tick marks on the specified ruler. For information, see horizontal custom points and vertical custom points.
   */
  CUSTOM = 1131639917,

  /**
   * Ha.
   */
  HA = 1516790048,

  /**
   * Inches.
   */
  INCHES = 2053729891,

  /**
   * Decimal inches.
   */
  INCHES_DECIMAL = 2053729892,

  /**
   * Millimeters.
   */
  MILLIMETERS = 2053991795,

  /**
   * Mils
   */
  MILS = 2051893612,

  /**
   * Picas.
   */
  PICAS = 2054187363,

  /**
   * Pixels.
   */
  PIXELS = 2054187384,

  /**
   * Points.
   */
  POINTS = 2054188905,

  /**
   * Q.
   */
  Q = 2054255973,

  /**
   * U
   */
  U = 2051691808,
}

/**
 * The page binding placement.
 */
declare enum PageBindingOptions {
  /**
   * Uses default page binding.
   */
  DEFAULT_VALUE = 1147563124,

  /**
   * Pages are bound on the left.
   */
  LEFT_TO_RIGHT = 1819570786,

  /**
   * Pages are bound on the right.
   */
  RIGHT_TO_LEFT = 1920232546,
}

/**
 * The zero point for the baseline grid offset.
 */
declare enum BaselineGridRelativeOption {
  /**
   * The baseline grid offset zero point is at the top page margin.
   */
  TOP_OF_MARGIN_OF_BASELINE_GRID_RELATIVE_OPTION = 1162766189,

  /**
   * The baseline grid offset zero point is at the top of the page.
   */
  TOP_OF_PAGE_OF_BASELINE_GRID_RELATIVE_OPTION = 1162766196,
}

/**
 * Binding spine placement options.
 */
declare enum PageSideOptions {
  /**
   * The page is on the left side of the binding spine in the spread.
   */
  LEFT_HAND = 1818653800,

  /**
   * The page is on the right side of the binding spine in the spread.
   */
  RIGHT_HAND = 1919382632,

  /**
   * The page is a single-sided page.
   */
  SINGLE_SIDED = 1970496888,
}

/**
 * Scaling options.
 */
declare enum WhenScalingOptions {
  /**
   * Adjust the scaling percentage of the item's transform
   */
  ADJUST_SCALING_PERCENTAGE = 1934587252,

  /**
   * Apply scaling to the item's content
   */
  APPLY_TO_CONTENT = 1934192243,
}

/**
 * Options for page color label.
 */
declare enum PageColorOptions {
  /**
   * No color.
   */
  NOTHING = 1851876449,

  /**
   * Uses the color label of the page's master page.
   */
  USE_MASTER_COLOR = 1346594413,
}

/**
 * Options for document intent.
 */
declare enum DocumentIntentOptions {
  /**
   * Intended purpose of document is for publishing to mobiles.
   */
  MOBILE_INTENT = 1768846445,

  /**
   * Intended purpose of document is for print output.
   */
  PRINT_INTENT = 1768846448,

  /**
   * Intended purpose of document is for web output.
   */
  WEB_INTENT = 1768846455,
}

/**
 * Options for setting the layout rule on a page.
 */
declare enum LayoutRuleOptions {
  /**
   * Use guide slicing to resize objects on the page as it resizes.
   */
  GUIDE_BASED = 1280329538,

  /**
   * Reposition and resize objects on the page as it resizes.
   */
  OBJECT_BASED = 1280331586,

  /**
   * No layout rule on the page as it resizes.
   */
  OFF = 1330005536,

  /**
   * Use existing layout rule setting on the page. Used for creating alternate layouts.
   */
  PRESERVE_EXISTING = 1280331890,

  /**
   * Recenter objects on the page as it resizes.
   */
  RECENTER = 1280332387,

  /**
   * Scale objects on the page as it resizes.
   */
  SCALE = 1280332643,

  /**
   * Use layout rule from the page's applied master page.
   */
  USE_MASTER = 1280333133,
}

/**
 * Modes that control which nearby snapshots, by size and shape, are blended into the new layout.
 */
declare enum SnapshotBlendingModes {
  /**
   * Use up to three nearest snapshots even if a snapshot is in a different class that the new layout.
   */
  FULL_SNAPSHOT_BLENDING = 1399735925,

  /**
   * Turns off the influence of layout snapshots completely.
   */
  IGNORE_LAYOUT_SNAPSHOTS = 1399736679,

  /**
   * Use only layout snapshots within the same class as the new layout.
   */
  LIMITED_SNAPSHOT_BLENDING = 1399737449,

  /**
   * Use the layout snapshot nearest in size and shape to the new layout.
   */
  USE_NEAREST_SNAPSHOT = 1399737957,
}

/**
 * Color space options for representing color in the exported EPS.
 */
declare enum EPSColorSpace {
  /**
   * Creates a separable file by representing all color values using the gamut of CYMK process color inks.
   */
  CMYK = 1129142603,

  /**
   * Converts all color values to high-quality black-and-white images. Gray levels of the converted objects represent the luminosity of the original objects.
   */
  GRAY = 1766290041,

  /**
   * Uses PostScript color management (includes profiles).
   */
  POSTSCRIPT_COLOR_MANAGEMENT = 1164208483,

  /**
   * Represents all color values using the RGB color space. Best suited for documents that will be viewed on-screen.
   */
  RGB = 1666336578,

  /**
   * Leaves each image in its original color space.
   */
  UNCHANGED_COLOR_SPACE = 1970161251,
}

/**
 * Preview image file format options.
 */
declare enum PreviewTypes {
  /**
   * Does not save a preview image.
   */
  NONE = 1852796517,

  /**
   * Saves the preview in PICT format. Note: Valid only for Mac OS.
   */
  PICT_PREVIEW = 1164996724,

  /**
   * Saves the preview in TIFF format.
   */
  TIFF_PREVIEW = 1164997734,
}

/**
 * Options for embedding fonts in the exported EPS.
 */
declare enum FontEmbedding {
  /**
   * Embeds all fonts once per page.
   */
  COMPLETE = 2003332197,

  /**
   * Embeds only references to fonts.
   */
  NONE = 1852796517,

  /**
   * Embeds only the characters (glyphs) used in the document. Glyphs are downloaded once per page.
   */
  SUBSET = 1768842098,
}

/**
 * Page scaling options.
 */
declare enum ScaleModes {
  /**
   * Scales the page to fit the paper. Note: Valid only when tile is false.
   */
  SCALE_TO_FIT = 1935897702,

  /**
   * Scales the page width and height.
   */
  SCALE_WIDTH_HEIGHT = 1935898745,
}

/**
 * Print layer options
 */
declare enum PrintLayerOptions {
  /**
   * Prints all layers.
   */
  ALL_LAYERS = 1634495564,

  /**
   * Prints all visible layers.
   */
  VISIBLE_LAYERS = 1986622284,

  /**
   * Prints only layers that are both visible and printable.
   */
  VISIBLE_PRINTABLE_LAYERS = 1987080780,
}

/**
 * The enumeration of tool box tools.
 */
declare enum UITools {
  /**
   * The add anchor point tool
   */
  ADD_ANCHOR_POINT = 1633767540,

  /**
   * The convert direction point tool
   */
  CONVERT_DIRECTION_POINT = 1667518580,

  /**
   * The delete anchor point tool
   */
  DELETE_ANCHOR_POINT = 1684099188,

  /**
   * The direct selection tool
   */
  DIRECT_SELECTION_TOOL = 1685277812,

  /**
   * The ellipse frame tool
   */
  ELLIPSE_FRAME_TOOL = 1701205100,

  /**
   * The ellipse tool
   */
  ELLIPSE_TOOL = 1701598316,

  /**
   * The erase tool
   */
  ERASE_TOOL = 1701991269,

  /**
   * The eye dropper tool
   */
  EYE_DROPPER_TOOL = 1701074028,

  /**
   * The free transform tool
   */
  FREE_TRANSFORM_TOOL = 1718899820,

  /**
   * The gap tool
   */
  GAP_TOOL = 1734430836,

  /**
   * The gradient feather tool
   */
  GRADIENT_FEATHER_TOOL = 1734759532,

  /**
   * The gradient swatch tool
   */
  GRADIENT_SWATCH_TOOL = 1735611500,

  /**
   * The hand tool
   */
  HAND_TOOL = 1751209068,

  /**
   * The line tool
   */
  LINE_TOOL = 1819169900,

  /**
   * The measure tool
   */
  MEASURE_TOOL = 1835357292,

  /**
   * The motion path tool
   */
  MOTION_PATH_TOOL = 1836078188,

  /**
   * No selection
   */
  NONE = 1852796517,

  /**
   * The note tool
   */
  NOTE_TOOL = 1852789868,

  /**
   * The page tool
   */
  PAGE_TOOL = 1936741484,

  /**
   * The pencil tool
   */
  PENCIL_TOOL = 1886274412,

  /**
   * The pen tool
   */
  PEN_TOOL = 1885687412,

  /**
   * The place cursor tool which gets set after an import via the Place command
   */
  PLACE_CURSOR_TOOL = 1885557868,

  /**
   * The polygon frame tool
   */
  POLYGON_FRAME_TOOL = 1885754476,

  /**
   * The polygon tool
   */
  POLYGON_TOOL = 1886147692,

  /**
   * The rectangle frame tool
   */
  RECTANGLE_FRAME_TOOL = 1919308908,

  /**
   * The rectangle tool
   */
  RECTANGLE_TOOL = 1919243372,

  /**
   * The rotate tool
   */
  ROTATE_TOOL = 1919898732,

  /**
   * The scale tool
   */
  SCALE_TOOL = 1935889516,

  /**
   * The scissors tool
   */
  SCISSORS_TOOL = 1935891060,

  /**
   * The selection tool
   */
  SELECTION_TOOL = 1936018548,

  /**
   * The shear tool
   */
  SHEAR_TOOL = 1936217196,

  /**
   * The smooth tool
   */
  SMOOTH_TOOL = 1936544872,

  /**
   * The table creation tool
   */
  TABLE_TOOL = 1952601196,

  /**
   * The type on a path tool
   */
  TYPE_ON_PATH_TOOL = 1953452148,

  /**
   * The type tool
   */
  TYPE_TOOL = 1954107508,

  /**
   * The zoom tool
   */
  ZOOM_TOOL = 2053985388,
}

/**
 * Resolution options when loaded master pages have the same name as existing masterpages.
 */
declare enum GlobalClashResolutionStrategyForMasterPage {
  /**
   * Overwrites the existing master page.
   */
  LOAD_ALL_WITH_OVERWRITE = 1279350607,

  /**
   * Renames the new master page.
   */
  LOAD_ALL_WITH_RENAME = 1279350610,
}

/**
 * The type of clipping path to create.
 */
declare enum ClippingPathType {
  /**
   * The clipping path is based on an alpha channel defined for the graphic in a graphics application.
   */
  ALPHA_CHANNEL = 1634756707,

  /**
   * The clipping path is based on pixel value threshold and tolerance.
   */
  DETECT_EDGES = 1685349735,

  /**
   * No clipping path applied.
   */
  NONE = 1852796517,

  /**
   * The clipping path is defined for the graphic in Photoshop.
   */
  PHOTOSHOP_PATH = 1886613620,

  /**
   * (Read-only) The clipping path has been manually edited.
   */
  USER_MODIFIED_PATH = 1970106484,
}

/**
 * Link updating options.
 */
declare enum UpdateLinkOptions {
  /**
   * Changes the visiblity settings to match the modified file.
   */
  APPLICATION_SETTINGS = 1819243873,

  /**
   * Ignores the modified settings and maintains those specified in the current document.
   */
  KEEP_OVERRIDES = 1819241327,

  /**
   * Unspecified update option.
   */
  UNKNOWN = 1433299822,
}

/**
 * Icon size options.
 */
declare enum IconSizes {
  /**
   * Extra large icon.
   */
  EXTRA_LARGE_ICON = 1885695079,

  /**
   * Extra small icon.
   */
  EXTRA_SMALL_ICON = 1885696877,

  /**
   * Jumbo icon.
   */
  JUMBO_ICON = 1886943340,

  /**
   * Large icon.
   */
  LARGE_ICON = 1884058215,

  /**
   * Medium icon.
   */
  MEDIUM_ICON = 1886217572,

  /**
   * Small icon.
   */
  SMALL_ICON = 1886612844,
}

/**
 * Palette layout resizing options.
 */
declare enum PanelLayoutResize {
  /**
   * Do not resize master pages panel area when resizing panel.
   */
  MASTERS_FIXED = 1886416230,

  /**
   * Do not resize document pages panel area when resizing panel.
   */
  PAGES_FIXED = 1886414456,

  /**
   * Resize panel areas proportionally.
   */
  PROPORTIONAL_RESIZE = 1886417010,
}

/**
 * options for viewing pages in Pages Panel
 */
declare enum PageViewOptions {
  /**
   * Pages arranged in vertical columns by alternate layout.
   */
  BY_ALTERNATE_LAYOUT = 1987277931,

  /**
   * Pages arranged in horizontal rows.
   */
  HORIZONTALLY = 1752396907,

  /**
   * Pages arranged in a vertical column.
   */
  VERTICALLY = 1987211127,
}

/**
 * Options for specifying how to represent color information in the exported PDF.
 */
declare enum PDFColorSpace {
  /**
   * Represents all color values using CYMK color space.
   */
  CMYK = 1129142603,

  /**
   * Converts all color values to high-quality black-and-white images. Gray levels of the converted objects represent the luminosity of the original objects.
   */
  GRAY = 1766290041,

  /**
   * Repurposes CMYK colors.
   */
  REPURPOSE_CMYK = 1917013337,

  /**
   * Repurposes RGB colors.
   */
  REPURPOSE_RGB = 1917994818,

  /**
   * Represents all color values using the RGB color space. Best suited for documents that will be viewed onscreen.
   */
  RGB = 1666336578,

  /**
   * Leaves each image in its original color space.
   */
  UNCHANGED_COLOR_SPACE = 1970161251,
}

/**
 * The exported PDF document's Acrobat compatibility.
 */
declare enum AcrobatCompatibility {
  /**
   * Makes the file compatible with Acrobat 4.0 and later.
   */
  ACROBAT_4 = 1097020464,

  /**
   * Makes the file compatible with Acrobat 5.0 and later.
   */
  ACROBAT_5 = 1097020720,

  /**
   * Makes the file compatible with Acrobat 6.0 and later.
   */
  ACROBAT_6 = 1097020976,

  /**
   * Makes the file compatible with Acrobat 7.0 or higher.
   */
  ACROBAT_7 = 1097021232,

  /**
   * Acrobat 8.0 compatibility.
   */
  ACROBAT_8 = 1097021488,
}

/**
 * The ICC profiles to include in the PDF document.
 */
declare enum ICCProfiles {
  /**
   * Includes all ICC profiles.
   */
  INCLUDE_ALL = 1229144929,

  /**
   * Does not include ICC profiles.
   */
  INCLUDE_NONE = 1229144942,

  /**
   * Includes RGB and tagged source CMYK profiles.
   */
  INCLUDE_RGB_AND_TAGGED = 1229144946,

  /**
   * Includes tagged source profiles.
   */
  INCLUDE_TAGGED = 1229144948,
}

/**
 * The type of compression for bitmap images.
 */
declare enum BitmapCompression {
  /**
   * The Automatic JPEG 2000 compression method.
   */
  AUTOMATIC_JPEG_2000 = 1634365490,

  /**
   * Uses JPEG compression and automatically determines the best quality type. Valid only when acrobat compatibility is acrobat 6 or higher.
   */
  AUTO_COMPRESSION = 1631808880,

  /**
   * Uses JPEG compression.
   */
  JPEG = 1785751398,

  /**
   * Uses JPEG 2000 compression. Valid only when acrobat compatibility is acrobat 6 or higher.
   */
  JPEG_2000 = 1785751346,

  /**
   * Uses no compression.
   */
  NONE = 1852796517,

  /**
   * Uses ZIP compression.
   */
  ZIP = 2053730371,
}

/**
 * The amount and type of compression to use for bitmap images.
 */
declare enum CompressionQuality {
  /**
   * Uses 8-bit compression. Valid only when bitmap compression is ZIP.
   */
  EIGHT_BIT = 1701722210,

  /**
   * Uses 4-bit compression. Valid only when bitmap compression is ZIP.
   */
  FOUR_BIT = 1701721186,

  /**
   * Uses high compression. Not valid when bitmap compression is ZIP.
   */
  HIGH = 1701726313,

  /**
   * Uses low compression. Not valid when bitmap compression is ZIP.
   */
  LOW = 1701727351,

  /**
   * Uses maximum compression. Not valid when bitmap compression is ZIP.
   */
  MAXIMUM = 1701727608,

  /**
   * Uses medium compression. Not valid when bitmap compression is ZIP.
   */
  MEDIUM = 1701727588,

  /**
   * Uses minimum compression. Not valid when bitmap compression is ZIP.
   */
  MINIMUM = 1701727598,
}

/**
 * The amount and type of compression to apply to monochrome bitmap images.
 */
declare enum MonoBitmapCompression {
  /**
   * Uses CCITT Group 3 compression.
   */
  CCIT3 = 1128879155,

  /**
   * Uses CCITT Group 4 compression.
   */
  CCIT4 = 1128879156,

  /**
   * Uses no compression.
   */
  NONE = 1852796517,

  /**
   * Uses Run Length compression.
   */
  RUN_LENGTH = 1919839299,

  /**
   * Uses ZIP compression.
   */
  ZIP = 2053730371,
}

/**
 * PDF mark weight
 */
declare enum PDFMarkWeight {
  /**
   * Printer mark line weight of 0.125 points.
   */
  P125PT = 825374064,

  /**
   * Printer mark line weight of 0.25 points.
   */
  P25PT = 842346608,

  /**
   * Printer mark line weight of 0.50 points.
   */
  P50PT = 892350576,
}

/**
 * The resampling method.
 */
declare enum Sampling {
  /**
   * Uses a weighted average to determine pixel color.
   */
  BICUBIC_DOWNSAMPLE = 1650742125,

  /**
   * Averages the pixels in a sample area and replaces the entire area with the average pixel color.
   */
  DOWNSAMPLE = 1684951917,

  /**
   * Uses no resampling.
   */
  NONE = 1852796517,

  /**
   * Chooses a pixel in the center of the sample area and replaces the entire area with that pixel color.
   */
  SUBSAMPLE = 1935823725,
}

/**
 * The type of preset to import or export.
 */
declare enum ExportPresetFormat {
  /**
   * Document presets.
   */
  DOCUMENT_PRESETS_FORMAT = 1683190892,

  /**
   * Flattener presets.
   */
  FLATTENER_PRESETS_FORMAT = 1951626348,

  /**
   * PDF export presets.
   */
  PDF_EXPORT_PRESETS_FORMAT = 1716745324,

  /**
   * Printer presets.
   */
  PRINTER_PRESETS_FORMAT = 1918071916,
}

/**
 * The amount of the PDF document to place.
 */
declare enum PDFCrop {
  /**
   * Places only the area defined by the PDF author as placeable artwork.
   */
  CROP_ART = 1131573313,

  /**
   * Places only the area that represents clipped content.
   */
  CROP_BLEED = 1131573314,

  /**
   * Places the page's bounding box using all layers.
   */
  CROP_CONTENT_ALL_LAYERS = 1131561324,

  /**
   * Places the page's bounding box using visible layers only.
   */
  CROP_CONTENT_VISIBLE_LAYERS = 1131566703,

  /**
   * Places the area that represents the physical paper size of the original PDF document.
   */
  CROP_MEDIA = 1131573325,

  /**
   * Places only the area displayed by Acrobat.
   */
  CROP_PDF = 1131573328,

  /**
   * Places only the area that represents the final trim size of the document.
   */
  CROP_TRIM = 1131573332,
}

/**
 * The objects to compress in the PDF document.
 */
declare enum PDFCompressionType {
  /**
   * Uses no compression.
   */
  COMPRESS_NONE = 1131368047,

  /**
   * Compress all objects.
   */
  COMPRESS_OBJECTS = 1131368290,

  /**
   * Compresses only objects related to PDF structure.
   */
  COMPRESS_STRUCTURE = 1131369332,
}

/**
 * The color profile.
 */
declare enum PDFProfileSelector {
  /**
   * Uses the document's CMYK profile.
   */
  USE_DOCUMENT = 1967419235,

  /**
   * Uses the monitor's color profile.
   */
  USE_MONITOR_PROFILE = 1836008528,

  /**
   * Uses no profile.
   */
  USE_NO_PROFILE = 1851868240,

  /**
   * Uses the working CMYK profile.
   */
  WORKING = 1466921579,
}

/**
 * Options for specifying the PDF/X compliance standard.
 */
declare enum PDFXStandards {
  /**
   * Does not check for compliance with a PDF/X standard.
   */
  NONE = 1852796517,

  /**
   * Checks for compliance with the PDF/X-1a:2001 standard.
   */
  PDFX1A2001_STANDARD = 1396912481,

  /**
   * Checks for compliance with the PDF/X-1a:2003 standard.
   */
  PDFX1A2003_STANDARD = 1395745075,

  /**
   * Checks for compliance with the PDF/X-3:2002 standard.
   */
  PDFX32002_STANDARD = 1396922419,

  /**
   * Checks for compliance with the PDF/X-3:2003 standard.
   */
  PDFX32003_STANDARD = 1398289203,

  /**
   * PDFX42010 standard is used.
   */
  PDFX42010_STANDARD = 1398289496,
}

/**
 * Export layer options.
 */
declare enum ExportLayerOptions {
  /**
   * Export all layers
   */
  EXPORT_ALL_LAYERS = 1702388076,

  /**
   * Export visible layers
   */
  EXPORT_VISIBLE_LAYERS = 1702393452,

  /**
   * Export visible and printable layers
   */
  EXPORT_VISIBLE_PRINTABLE_LAYERS = 1702260844,
}

/**
 * PDF export magnification options.
 */
declare enum PdfMagnificationOptions {
  /**
   * Uses the actual size.
   */
  ACTUAL_SIZE = 2053206906,

  /**
   * Uses default magnification.
   */
  DEFAULT_VALUE = 1147563124,

  /**
   * Uses 50 percent magnification option.
   */
  FIFTY_PERCENT = 2053531248,

  /**
   * Uses the fit height magnification option.
   */
  FIT_HEIGHT = 1212437352,

  /**
   * Uses the fit page magnification option.
   */
  FIT_PAGE = 2053534832,

  /**
   * Uses the fit visible magnification option.
   */
  FIT_VISIBLE = 1212437334,

  /**
   * Uses the fit width magnification option.
   */
  FIT_WIDTH = 1212437335,

  /**
   * Uses 100 percent magnification option.
   */
  ONE_HUNDRED_PERCENT = 2053533544,

  /**
   * Uses 75 percent magnification option.
   */
  SEVENTY_FIVE_PERCENT = 2053534566,

  /**
   * Uses 25 percent magnification option.
   */
  TWENTY_FIVE_PERCENT = 2053534822,
}

/**
 * PDF export page layout options.
 */
declare enum PageLayoutOptions {
  /**
   * Uses default page layout.
   */
  DEFAULT_VALUE = 1147563124,

  /**
   * Single page layout.
   */
  SINGLE_PAGE = 1987736432,

  /**
   * Single page continuous layout.
   */
  SINGLE_PAGE_CONTINUOUS = 1884508259,

  /**
   * Two up cover page page layout.
   */
  TWO_UP_COVER_PAGE = 1884570448,

  /**
   * Two up cover page continuous page layout.
   */
  TWO_UP_COVER_PAGE_CONTINUOUS = 1884570467,

  /**
   * Two up facing page layout.
   */
  TWO_UP_FACING = 1884575046,

  /**
   * Two up facing continuous page layout.
   */
  TWO_UP_FACING_CONTINUOUS = 1884571235,
}

/**
 * PDF export raster resolution options.
 */
declare enum RasterResolutionOptions {
  /**
   * 96 pixels per inch.
   */
  NINETY_SIX_PPI = 1853059184,

  /**
   * 144 pixels per inch.
   */
  ONE_HUNDRED_FORTY_FOUR_PPI = 1868984432,

  /**
   * 72 pixels per inch.
   */
  SEVENTY_TWO_PPI = 1937010800,
}

/**
 * PDF raster compression options.
 */
declare enum PDFRasterCompressionOptions {
  /**
   * Uses JPEG compression and automatically determines the best quality type.
   */
  AUTOMATIC_COMPRESSION = 1936875875,

  /**
   * Uses JPEG compression.
   */
  JPEG_COMPRESSION = 1936878179,

  /**
   * Uses the best quality type.
   */
  LOSSLESS_COMPRESSION = 1936878691,
}

/**
 * PDF JPEG Quality options.
 */
declare enum PDFJPEGQualityOptions {
  /**
   * Uses high JPEG compression.
   */
  HIGH = 1701726313,

  /**
   * Uses low JPEG compression.
   */
  LOW = 1701727351,

  /**
   * Uses maximum JPEG compression.
   */
  MAXIMUM = 1701727608,

  /**
   * Uses medium JPEG compression.
   */
  MEDIUM = 1701727588,

  /**
   * Uses minimum JPEG compression.
   */
  MINIMUM = 1701727598,
}

/**
 * Options for how to draw interactive elements.
 */
declare enum InteractiveElementsOptions {
  /**
   * Draw interactive elements appearance only.
   */
  APPEARANCE_ONLY = 1097887823,

  /**
   * Do not include interactive elements.
   */
  DO_NOT_INCLUDE = 1145981283,
}

/**
 * Options for how to draw interactive elements.
 */
declare enum InteractivePDFInteractiveElementsOptions {
  /**
   * Draw interactive elements appearance only.
   */
  APPEARANCE_ONLY = 1097887823,

  /**
   * Include all interactive elements.
   */
  INCLUDE_ALL_MEDIA = 1231241580,
}

/**
 * Page transition to use as an override when exporting.
 */
declare enum PageTransitionOverrideOptions {
  /**
   * The Blinds page transition.
   */
  BLINDS_TRANSITION = 1886667372,

  /**
   * The Blinds page transition.
   */
  BOX_TRANSITION = 1886667384,

  /**
   * The Comb page transition.
   */
  COMB_TRANSITION = 1886667618,

  /**
   * The Cover page transition.
   */
  COVER_TRANSITION = 1886667638,

  /**
   * The Dissolve page transition.
   */
  DISSOLVE_TRANSITION = 1886667891,

  /**
   * The Fade page transition.
   */
  FADE_TRANSITION = 1886668388,

  /**
   * Use the page transition from the document.
   */
  FROM_DOCUMENT = 1718764655,

  /**
   * No page transition applied.
   */
  NONE = 1852796517,

  /**
   * The Page Turn page transition (SWF only).
   */
  PAGE_TURN_TRANSITION = 1886670932,

  /**
   * The Push page transition.
   */
  PUSH_TRANSITION = 1886670963,

  /**
   * The Split page transition.
   */
  SPLIT_TRANSITION = 1886671728,

  /**
   * The Uncover page transition.
   */
  UNCOVER_TRANSITION = 1886672227,

  /**
   * The Wipe page transition.
   */
  WIPE_TRANSITION = 1886672752,

  /**
   * The Zoom In page transition.
   */
  ZOOM_IN_TRANSITION = 1886673481,

  /**
   * The Zoom Out page transition.
   */
  ZOOM_OUT_TRANSITION = 1886673487,
}

/**
 * Tagged PDF structure order options
 */
declare enum TaggedPDFStructureOrderOptions {
  /**
   * Use Articles order for the tagged PDF structure.
   */
  USE_ARTICLES = 1348554610,

  /**
   * Use XML structure and layout heuristic fallback for the tagged PDF structure.
   */
  USE_XML_STRUCTURE = 1350062959,
}

/**
 * PDF export display title options.
 */
declare enum PdfDisplayTitleOptions {
  /**
   * Uses document title.
   */
  DISPLAY_DOCUMENT_TITLE = 1148413044,

  /**
   * Uses file name.
   */
  DISPLAY_FILE_NAME = 1148413550,
}

/**
 * Options for specifying basis of the top origin of the paragraph shading.
 */
declare enum ParagraphShadingTopOriginEnum {
  /**
   * Makes the paragraph shading top origin based on ascent of the text in the paragraph.
   */
  ASCENT_TOP_ORIGIN = 1886606433,

  /**
   * Makes the paragraph shading top origin based on baseline of the text in the paragraph.
   */
  BASELINE_TOP_ORIGIN = 1886606434,

  /**
   * Makes the paragraph shading top origin based on leading of the text in the paragraph.
   */
  LEADING_TOP_ORIGIN = 1885492332,
}

/**
 * Options for specifying basis of the bottom origin of the paragraph shading.
 */
declare enum ParagraphShadingBottomOriginEnum {
  /**
   * Makes the paragraph shading bottom origin based on baseline of the text in the paragraph.
   */
  BASELINE_BOTTOM_ORIGIN = 1886601826,

  /**
   * Makes the paragraph shading bottom origin based on descent of the text in the paragraph.
   */
  DESCENT_BOTTOM_ORIGIN = 1886601828,
}

/**
 * Options for restarting endnote numbering.
 */
declare enum EndnoteRestarting {
  /**
   * Does not restart numbering; numbers endnotes sequentially throughout the document.
   */
  CONTINUOUS = 1164210803,

  /**
   * Restarts endnote numbering on each story.
   */
  STORY_RESTART = 1165193843,
}

/**
 * Options for scope of endnote placement.
 */
declare enum EndnoteScope {
  /**
   * Endnotes specific to each document.
   */
  ENDNOTE_DOCUMENT_SCOPE = 1162765427,

  /**
   * Endnotes specific to each story.
   */
  STORY_SCOPE = 1162769267,
}

/**
 * Options for frame creation of endnotes.
 */
declare enum EndnoteFrameCreate {
  /**
   * Endnotes are loaded in placegun to be placed anywhere in document.
   */
  LOAD_ENDNOTE_PLACE_GUN = 1162768487,

  /**
   * Endnotes created on a new page and frame which are automatically created.
   */
  NEW_PAGE = 1162767984,
}

/**
 * Options for specifying basis of the width of the paragraph border.
 */
declare enum ParagraphBorderEnum {
  /**
   * Makes the paragraph border based on width of the column.
   */
  COLUMN_WIDTH = 1265399652,

  /**
   * Makes the paragraph border based on width of lines of text in the paragraph.
   */
  TEXT_WIDTH = 1886681207,
}

/**
 * Options for specifying basis of the top origin of the paragraph border.
 */
declare enum ParagraphBorderTopOriginEnum {
  /**
   * Makes the paragraph border top origin based on ascent of the text in the paragraph.
   */
  ASCENT_TOP_ORIGIN = 1886606433,

  /**
   * Makes the paragraph border top origin based on baseline of the text in the paragraph.
   */
  BASELINE_TOP_ORIGIN = 1886606434,

  /**
   * Makes the paragraph border top origin based on leading of the text in the paragraph.
   */
  LEADING_TOP_ORIGIN = 1885492332,
}

/**
 * Options for specifying basis of the bottom origin of the paragraph border.
 */
declare enum ParagraphBorderBottomOriginEnum {
  /**
   * Makes the paragraph border bottom origin based on baseline of the text in the paragraph.
   */
  BASELINE_BOTTOM_ORIGIN = 1886601826,

  /**
   * Makes the paragraph border bottom origin based on descent of the text in the paragraph.
   */
  DESCENT_BOTTOM_ORIGIN = 1886601828,
}

/**
 * Provider hyphenation styles. Currently only supported by the Duden hyphenation provider.
 */
declare enum HyphenationStyleEnum {
  /**
   * Hyphenates at aesthetic hyphenation points.
   */
  HYPH_AESTHETIC = 1684104563,

  /**
   * Hyphenates at all possible hyphenation points.
   */
  HYPH_ALL = 1684106348,

  /**
   * Hyphenates at all but unaesthetic hyphenation points.
   */
  HYPH_ALL_BUT_UNAESTHETIC = 1684103797,

  /**
   * Hyphenates at preferred aesthetic hyphenation points.
   */
  HYPH_PREFERRED_AESTHETIC = 1685086565,
}

/**
 * Options for indicating the type of shape to which to covert an object.
 */
declare enum ConvertShapeOptions {
  /**
   * Converts the object to a rectangle with beveled corners.
   */
  CONVERT_TO_BEVELED_RECTANGLE = 1129529938,

  /**
   * Converts the object to a closed path.
   */
  CONVERT_TO_CLOSED_PATH = 1129530224,

  /**
   * Converts the object to a rectangle with inverse rounded corners.
   */
  CONVERT_TO_INVERSE_ROUNDED_RECTANGLE = 1129531730,

  /**
   * Converts the object to a line that connects the upper left and lower right corners of the object's bounding box.
   */
  CONVERT_TO_LINE = 1129532489,

  /**
   * Converts the object to an open path.
   */
  CONVERT_TO_OPEN_PATH = 1129533296,

  /**
   * Converts the object to an ellipse.
   */
  CONVERT_TO_OVAL = 1129533270,

  /**
   * Converts the object to a polygon.
   */
  CONVERT_TO_POLYGON = 1129533519,

  /**
   * Converts the object to a rectangle.
   */
  CONVERT_TO_RECTANGLE = 1129534021,

  /**
   * Converts the object to a rectangle with rounded corners.
   */
  CONVERT_TO_ROUNDED_RECTANGLE = 1129534034,

  /**
   * Converts the object to straight line. If the object is a square, circle, or its bounding box is wider than it is tall, the line is horizontal and connects the center points on the vertical sides of the bounding box. If the object's bounding box is taller than it is wide, the line connects the center points of the horizontal sides of the bounding box.
   */
  CONVERT_TO_STRAIGHT_LINE = 1129534284,

  /**
   * Converts the object to a triangle.
   */
  CONVERT_TO_TRIANGLE = 1129534546,
}

/**
 * Coordinate space options.
 */
declare enum CoordinateSpaces {
  /**
   * Inner coordinates
   */
  INNER_COORDINATES = 2021222766,

  /**
   * Page coordinates.
   */
  PAGE_COORDINATES = 2021224551,

  /**
   * Parent coordinates
   */
  PARENT_COORDINATES = 2021224545,

  /**
   * Pasteboard coordinates
   */
  PASTEBOARD_COORDINATES = 2021224546,

  /**
   * Spread coordinates.
   */
  SPREAD_COORDINATES = 2021225328,
}

/**
 * Matrix content.
 */
declare enum MatrixContent {
  /**
   * Rotation value
   */
  ROTATION_VALUE = 1936746862,

  /**
   * Scale values
   */
  SCALE_VALUES = 1735552887,

  /**
   * Shear value
   */
  SHEAR_VALUE = 1936486004,

  /**
   * Translation values
   */
  TRANSLATION_VALUES = 1936484720,
}

/**
 * Bounding box limits.
 */
declare enum BoundingBoxLimits {
  /**
   * Geometric path bounds
   */
  GEOMETRIC_PATH_BOUNDS = 1768844080,

  /**
   * Outer stroke bounds
   */
  OUTER_STROKE_BOUNDS = 1768844081,
}

/**
 * Resize constraints.
 */
declare enum ResizeConstraints {
  /**
   * Inverse proportions
   */
  INVERSE_PROPORTIONS = 1231976016,

  /**
   * Keep current proportions
   */
  KEEP_CURRENT_PROPORTIONS = 1264939088,

  /**
   * Keep current value
   */
  KEEP_CURRENT_VALUE = 1264939094,

  /**
   * Tall proportions
   */
  TALL_PROPORTIONS = 1415670864,

  /**
   * Wide proportions
   */
  WIDE_PROPORTIONS = 1466524752,
}

/**
 * Resize methods.
 */
declare enum ResizeMethods {
  /**
   * Add additional width and height to current values
   */
  ADDING_CURRENT_DIMENSIONS_TO = 1215264592,

  /**
   * Multiply current width and height by the given factors
   */
  MULTIPLYING_CURRENT_DIMENSIONS_BY = 1215264589,

  /**
   * Change width and height overriding current values
   */
  REPLACING_CURRENT_DIMENSIONS_WITH = 1215264581,

  /**
   * Change width to height ratio keeping the current area
   */
  RESHAPING_AREA_TO_RATIO = 1215264577,

  /**
   * Change width to height ratio keeping the current perimeter
   */
  RESHAPING_BORDER_TO_RATIO = 1215264595,
}

/**
 * Options for fitting content in an empty frame.
 */
declare enum EmptyFrameFittingOptions {
  /**
   * Resizes content to fit the frame. Note: Content that has different proportions than the frame appears stretched or squeezed.
   */
  CONTENT_TO_FRAME = 1668575078,

  /**
   * Resizes content to fill the frame while perserving the content's proportions. If the content and frame have different proportions, some of the content is obscured by the frame.
   */
  FILL_PROPORTIONALLY = 1718185072,

  /**
   * Does not use a fitting option.
   */
  NONE = 1852796517,

  /**
   * Resizes content to fit the frame while preserving content proportions. If the content and frame have different proportions, some empty space appears in the frame.
   */
  PROPORTIONALLY = 1668247152,
}

/**
 * Corner options.
 */
declare enum CornerOptions {
  /**
   * Beveled corner option.
   */
  BEVEL_CORNER = 1667588726,

  /**
   * Fancy corner option.
   */
  FANCY_CORNER = 1667589742,

  /**
   * Inset corner option.
   */
  INSET_CORNER = 1667591795,

  /**
   * Inverted rounded corner option.
   */
  INVERSE_ROUNDED_CORNER = 1667591798,

  /**
   * No corner option.
   */
  NONE = 1852796517,

  /**
   * Rounded corner option.
   */
  ROUNDED_CORNER = 1667592804,
}

/**
 * Dimension constraints for the object-based layout rule
 */
declare enum DimensionsConstraints {
  /**
   * The dimension remains fixed relative to the parent.
   */
  FIXED_DIMENSION = 1145267817,

  /**
   * The dimension can vary relative to the parent.
   */
  FLEXIBLE_DIMENSION = 1145267820,
}

/**
 * The content type of an object.
 */
declare enum ContentType {
  /**
   * The frame is a graphics frame.
   */
  GRAPHIC_TYPE = 1735553140,

  /**
   * The frame is a text frame.
   */
  TEXT_TYPE = 1952412773,

  /**
   * No content type assigned.
   */
  UNASSIGNED = 1970168179,
}

/**
 * End cap types.
 */
declare enum EndCap {
  /**
   * A squared end that stops at the path's endpoint.
   */
  BUTT_END_CAP = 1650680176,

  /**
   * A squared end that extends beyond the endpoint by half the stroke-width.
   */
  PROJECTING_END_CAP = 1886020464,

  /**
   * A semicircular end that extends beyond the endpoint by half the stroke-width.
   */
  ROUND_END_CAP = 1919115632,
}

/**
 * End join types.
 */
declare enum EndJoin {
  /**
   * Beveled end join.
   */
  BEVEL_END_JOIN = 1651142510,

  /**
   * Miter end join.
   */
  MITER_END_JOIN = 1835691886,

  /**
   * Rounded end join.
   */
  ROUND_END_JOIN = 1919577966,
}

/**
 * The automatic adjustment to make to the pattern of a dashed or dotted stroke to cover corner points in a path.
 */
declare enum StrokeCornerAdjustment {
  /**
   * Changes the length of dashes so that dashes always occur at path ends and corners; maintains set gap length. Note: Can cause dashes to be different lengths on shapes whose sides are of different lengths, such as rectangles.
   */
  DASHES = 1162113896,

  /**
   * Adjusts both dashes and gaps to cover corners and end points. Note: Causes dash and gap sizes to be consistent on all sides of the shape.
   */
  DASHES_AND_GAPS = 1148405616,

  /**
   * Changes the length of gaps so that dashes or dots always occur at ends and corners; maintains dash length or dot diameter. Note: Can cause gaps to be different lengths on shapes whose sides are of different lengths, such as rectangles.
   */
  GAPS = 1164406899,

  /**
   * No adjustment.
   */
  NONE = 1852796517,
}

/**
 * The shape of one or both ends of an open path.
 */
declare enum ArrowHead {
  /**
   * A solid arrow head whose pierced end bows sharply toward the point and whose point describes a 45-degree angle.
   */
  BARBED_ARROW_HEAD = 1650553442,

  /**
   * A vertical bar bisected by the stroke, which meets the stroke at a right angle and is the same weight as the stroke. The bar's length is 4.5 times the stroke width.
   */
  BAR_ARROW_HEAD = 1651663208,

  /**
   * A hollow circle whose outline is the same weight as the stroke. The circle's diameter is 5 times the stroke width.
   */
  CIRCLE_ARROW_HEAD = 1668440424,

  /**
   * A solid circle whose diameter is 5 times the stroke width.
   */
  CIRCLE_SOLID_ARROW_HEAD = 1668505960,

  /**
   * A solid arrow head whose pierced end concaves toward the point and whose point describes a 45-degree angle.
   */
  CURVED_ARROW_HEAD = 1668702568,

  /**
   * None.
   */
  NONE = 1852796517,

  /**
   * An arrow head formed by two slanting lines whose intersection forms a 45-degree angle and whose stroke weight is the same as the path's stroke.
   */
  SIMPLE_ARROW_HEAD = 1936289136,

  /**
   * An arrow head formed by two slanting lines whose intersection forms a 90-degree angle and whose stroke weight is the same as the path's stroke.
   */
  SIMPLE_WIDE_ARROW_HEAD = 1937203560,

  /**
   * A hollow square set perpendicular to the path, whose outline is the same weight as the stroke. The length of one side of the square is 5 times the stroke width.
   */
  SQUARE_ARROW_HEAD = 1936810344,

  /**
   * A solid square set perpendicular to the end of the path. The length of one side of the square is 5 times the stroke width.
   */
  SQUARE_SOLID_ARROW_HEAD = 1936941416,

  /**
   * A solid triangle arrow head whose point describes a 45-degree angle.
   */
  TRIANGLE_ARROW_HEAD = 1953655150,

  /**
   * A solid triangle arrow head whose point describes a 90-degree angle.
   */
  TRIANGLE_WIDE_ARROW_HEAD = 1953980776,
}

/**
 * Display performance options.
 */
declare enum DisplaySettingOptions {
  /**
   * Uses the container object's default display performance preferences setting. For information, see default display settings.
   */
  DEFAULT_VALUE = 1147563124,

  /**
   * Slower performance; displays high-resolution graphics and high-quality transparencies and turns on anti-aliasing.
   */
  HIGH_QUALITY = 1346922866,

  /**
   * Best performance; grays out graphics and turns off transparency and anti-aliasing.
   */
  OPTIMIZED = 1349480564,

  /**
   * Moderate performance speed; displays proxy graphics and low-quality transparencies and turns on anti-aliasing.
   */
  TYPICAL = 1349810544,
}

/**
 * Options for fitting content to a frame.
 */
declare enum FitOptions {
  /**
   * Applies the current frame fitting options to the frame and content. Before using this, do confirm that the expected Frame Fitting Options are applied on the frame. For example, the act of placing an image in a frame set to 'Fit Content Proportionally' can change the crop settings in the Frame Fitting Options, which would then get applied for any subsequent image placement when this API is called.
   */
  APPLY_FRAME_FITTING_OPTIONS = 1634100847,

  /**
   * Centers content in the frame; preserves the frame size as well as content size and proportions. Note: If the content is larger than the frame, content around the edges is obscured.
   */
  CENTER_CONTENT = 1667591779,

  /**
   * Resizes content to fit the frame. Note: Content that is a different size than the frame appears stretched or squeezed.
   */
  CONTENT_TO_FRAME = 1668575078,

  /**
   * Resizes content to fill the frame while perserving the proportions of the content. If the content and frame have different proportions, some of the content is obscured by the bounding box of the frame.
   */
  FILL_PROPORTIONALLY = 1718185072,

  /**
   * Resizes the frame so it fits the content.
   */
  FRAME_TO_CONTENT = 1718906723,

  /**
   * Resizes content to fit the frame while preserving content proportions. If the content and frame have different proportions, some empty space appears in the frame.
   */
  PROPORTIONALLY = 1668247152,
}

/**
 * Options for positioning the stroke relative to its path.
 */
declare enum StrokeAlignment {
  /**
   * The stroke straddles the path.
   */
  CENTER_ALIGNMENT = 1936998723,

  /**
   * The stroke is inside the path.
   */
  INSIDE_ALIGNMENT = 1936998729,

  /**
   * The stroke is outside the path, like a picture frame.
   */
  OUTSIDE_ALIGNMENT = 1936998735,
}

/**
 * Guide type options for ruler guides.
 */
declare enum GuideTypeOptions {
  /**
   * Liquid guide.
   */
  LIQUID = 1735617635,

  /**
   * Ruler guide (default).
   */
  RULER = 1735618160,
}

/**
 * The path on which to base the contour text wrap.
 */
declare enum ContourOptionsTypes {
  /**
   * Sets the text wrap shape to the edges of the specified alpha channel. To specify the alpha channel, see contour path name.
   */
  ALPHA_CHANNEL = 1634756707,

  /**
   * Sets the text wrap shape to the object's bounding box.
   */
  BOUNDING_BOX = 1701732962,

  /**
   * Sets the text wrap shape to the edges of the image.
   */
  DETECT_EDGES = 1685349735,

  /**
   * Sets the text wrap shape to the wrapped object's graphics frame.
   */
  GRAPHIC_FRAME = 1701734246,

  /**
   * Sets the text wrap shape to the specified Photoshop path. To specify the Photoshop path, see contour path name.
   */
  PHOTOSHOP_PATH = 1886613620,

  /**
   * Sets the text wrap shape to the clipping path (if any) defined in Photoshop. Note: A path cannot be specified using this enumeration. To set the text wrap shape to a specific path, use the photoshop path contour options type enumeration value.
   */
  SAME_AS_CLIPPING = 1935762288,
}

/**
 * Text wrap side options.
 */
declare enum TextWrapSideOptions {
  /**
   * Both sides text wrap.
   */
  BOTH_SIDES = 1953981043,

  /**
   * Largest side text wrap.
   */
  LARGEST_AREA = 1953975411,

  /**
   * Left side text wrap.
   */
  LEFT_SIDE = 1953983603,

  /**
   * Right side text wrap.
   */
  RIGHT_SIDE = 1953985139,

  /**
   * Away from binding side text wrap.
   */
  SIDE_AWAY_FROM_SPINE = 1953980787,

  /**
   * Binding side text wrap.
   */
  SIDE_TOWARDS_SPINE = 1953985651,
}

/**
 * Options for wrapping text around an object.
 */
declare enum TextWrapModes {
  /**
   * Wraps text around the object's bounding box.
   */
  BOUNDING_BOX_TEXT_WRAP = 1651729523,

  /**
   * Wraps text around the object following the specified contour options.
   */
  CONTOUR = 1835233134,

  /**
   * Forces text to jump above or below the object, so that no text appears on the object's right or left.
   */
  JUMP_OBJECT_TEXT_WRAP = 1650552420,

  /**
   * Forces text to jump to the next available column.
   */
  NEXT_COLUMN_TEXT_WRAP = 1853384306,

  /**
   * No text wrap.
   */
  NONE = 1852796517,
}

/**
 * Font status options.
 */
declare enum FontStatus {
  /**
   * The font has been fauxed.
   */
  FAUXED = 1718830689,

  /**
   * The font is installed.
   */
  INSTALLED = 1718831470,

  /**
   * The font is not available.
   */
  NOT_AVAILABLE = 1718832705,

  /**
   * The font is a substitute.
   */
  SUBSTITUTED = 1718834037,

  /**
   * The font's status is unknown.
   */
  UNKNOWN = 1433299822,
}

/**
 * Font type options.
 */
declare enum FontTypes {
  /**
   * ATC.
   */
  ATC = 1718894932,

  /**
   * Bitmap.
   */
  BITMAP = 1718895209,

  /**
   * CID.
   */
  CID = 1718895433,

  /**
   * OCF.
   */
  OCF = 1718898499,

  /**
   * OpenType CFF.
   */
  OPENTYPE_CFF = 1718898502,

  /**
   * OpenType CID.
   */
  OPENTYPE_CID = 1718898505,

  /**
   * OpenType TT.
   */
  OPENTYPE_TT = 1718898516,

  /**
   * TrueType.
   */
  TRUETYPE = 1718899796,

  /**
   * Type 1.
   */
  TYPE_1 = 1718899761,

  /**
   * The font type is unknown.
   */
  UNKNOWN = 1433299822,
}

/**
 * Supported OpenType feature options.
 */
declare enum OpenTypeFeature {
  /**
   * Provides authentic small caps rather than scaled-down versions of the font's regular caps.
   */
  ALL_SMALL_CAPS_FEATURE = 1664250691,

  /**
   * Activates contextual ligatures and connecting alternates.
   */
  CONTEXTUAL_ALTERNATES_FEATURE = 1330930497,

  /**
   * Applies the default figure style of the current font to figure glyphs.
   */
  DEFAULT_FIGURE_STYLE_FEATURE = 1330931268,

  /**
   * In a series of two numbers separated by a slash that form a non-standard fraction, such as 4/13, reformats the second number as a denominator.
   */
  DENOMINATOR_FEATURE = 1884247108,

  /**
   * Allows the use of optional discretionary ligatures.
   */
  DISCRETIONARY_LIGATURES_FEATURE = 1330930764,

  /**
   * Reformats numbers separated by a slash, such as 1/2, as fractions. Note: In some fonts, the fractions feature reformats only standard fractions. For information on reformatting non-standard fractions such as 4/13, see denominator feature and numerator feature.
   */
  FRACTIONS_FEATURE = 1330931282,

  /**
   * Justification alternate
   */
  JUSTIFICATION_ALTERNATE = 1330932309,

  /**
   * Low.
   */
  LOW = 1701727351,

  /**
   * In a series of two numbers separated by a slash that form a non-standard fraction, such as 4/13, reformats the first number as a numerator.
   */
  NUMERATOR_FEATURE = 1884247118,

  /**
   * Superscripts the alpha characters in ordinal numbers.
   */
  ORDINAL_FEATURE = 1330933586,

  /**
   * Overlap swash
   */
  OVERLAP_SWASH = 1330933590,

  /**
   * Gives full-height figures varying widths.
   */
  PROPORTIONAL_LINING_FEATURE = 1330932816,

  /**
   * Gives varying-height figures varying widths.
   */
  PROPORTIONAL_OLDSTYLE_FEATURE = 1330933587,

  /**
   * Stretched alternate
   */
  STRETCHED_ALTERNATE = 1330934610,

  /**
   * Stylistic alternate
   */
  STYLISTIC_ALTERNATE = 1330934612,

  /**
   * Sizes lowered glyphs correctly relative to the surrounding characters.
   */
  SUBSCRIPT_FEATURE = 1884247106,

  /**
   * Sizes raised glyphs correctly relative to the surrounding characters.
   */
  SUPERSCRIPT_FEATURE = 1884247123,

  /**
   * Provides regular and contextual swashes, which may include alternate caps and end-of-word alternatives.
   */
  SWASH_FEATURE = 1330934615,

  /**
   * Gives full-height figures fixed, equal width.
   */
  TABULAR_LINING_FEATURE = 1330931284,

  /**
   * Gives varying-height figures fixed, equal widths.
   */
  TABULAR_OLDSTYLE_FEATURE = 1330933588,

  /**
   * Activates alternative characters used for uppercase titles.
   */
  TITLING_FEATURE = 1330934857,
}

/**
 * Options for aligning small characters to the largest character in the line.
 */
declare enum CharacterAlignment {
  /**
   * Aligns small characters in a line to the large character.
   */
  ALIGN_BASELINE = 1247896172,

  /**
   * Aligns small characters in horizontal text to the bottom of the em box of large characters. In vertical text, aligns characters to the left of the em box.
   */
  ALIGN_EM_BOTTOM = 1247896173,

  /**
   * Aligns small characters to the center of the em box of large characters.
   */
  ALIGN_EM_CENTER = 1247896436,

  /**
   * Aligns small characters in horizontal text to the top of the em box of large characters. In vertical text, aligns characters to the right of the em box.
   */
  ALIGN_EM_TOP = 1247900784,

  /**
   * Aligns small characters in horizontal text to the bottom of the ICF of large characters. In vertical text, aligns characters to the left of the ICF.
   */
  ALIGN_ICF_BOTTOM = 1248420461,

  /**
   * Aligns small characters in horizontal text to the top of the ICF of large characters. In vertical text, aligns characters to the right of the ICF.
   */
  ALIGN_ICF_TOP = 1248425072,
}

/**
 * Glyph variant substitution options for standard glyphs.
 */
declare enum AlternateGlyphForms {
  /**
   * Uses the expert variant.
   */
  EXPERT_FORM = 1247897445,

  /**
   * Uses the full-width variant.
   */
  FULL_WIDTH_FORM = 1247897446,

  /**
   * Uses the JIS04 variant.
   */
  JIS04_FORM = 1247897396,

  /**
   * Uses the JIS78 variant.
   */
  JIS78_FORM = 1247897399,

  /**
   * Uses the JIS83 variant.
   */
  JIS83_FORM = 1247897400,

  /**
   * Uses the JIS90 variant.
   */
  JIS90_FORM = 1247897401,

  /**
   * Uses the monospaced half-width variant.
   */
  MONOSPACED_HALF_WIDTH_FORM = 1247897453,

  /**
   * Uses the NLC variant.
   */
  NLC_FORM = 1247897454,

  /**
   * Does not use an alternate form.
   */
  NONE = 1852796517,

  /**
   * Substitutes proportional glyphs for half-width and full-width glyphs.
   */
  PROPORTIONAL_WIDTH_FORM = 1247897456,

  /**
   * Uses the quarter-width variant.
   */
  QUARTER_WIDTH_FORM = 1247897457,

  /**
   * Uses the third-width variant.
   */
  THIRD_WIDTH_FORM = 1247897448,

  /**
   * Uses the traditional variant.
   */
  TRADITIONAL_FORM = 1247897460,
}

/**
 * Overprint options for kenten marks.
 */
declare enum AdornmentOverprint {
  /**
   * Uses auto overprint.
   */
  AUTO = 1635019116,

  /**
   * Turns off overprint.
   */
  OVERPRINT_OFF = 1701736294,

  /**
   * Turns on overprint.
   */
  OVERPRINT_ON = 1701736302,
}

/**
 * Style options for kenten characters.
 */
declare enum KentenCharacter {
  /**
   * Uses a custom kenten style.
   */
  CUSTOM = 1131639917,

  /**
   * Uses the kenten black circle.
   */
  KENTEN_BLACK_CIRCLE = 1248551523,

  /**
   * Uses the kenten black triangle.
   */
  KENTEN_BLACK_TRIANGLE = 1248551540,

  /**
   * Uses the kenten bullseye.
   */
  KENTEN_BULLSEYE = 1248551525,

  /**
   * Uses the kenten fisheye.
   */
  KENTEN_FISHEYE = 1248552549,

  /**
   * Uses the kenten sesame dot.
   */
  KENTEN_SESAME_DOT = 1248551795,

  /**
   * Uses the kenten small black circle.
   */
  KENTEN_SMALL_BLACK_CIRCLE = 1248555875,

  /**
   * Uses the kenten small white circle.
   */
  KENTEN_SMALL_WHITE_CIRCLE = 1248555895,

  /**
   * Uses the kenten white circle.
   */
  KENTEN_WHITE_CIRCLE = 1248556899,

  /**
   * Uses the kenten white sesame dot.
   */
  KENTEN_WHITE_SESAME_DOT = 1248551799,

  /**
   * Uses the kenten white triangle.
   */
  KENTEN_WHITE_TRIANGLE = 1248556916,

  /**
   * Does not use kenten.
   */
  NONE = 1852796517,
}

/**
 * Options for specifying the kenten or ruby position relative to the parent character.
 */
declare enum RubyKentenPosition {
  /**
   * Places kenten or ruby to the right and above the parent character.
   */
  ABOVE_RIGHT = 1248551282,

  /**
   * Places kenten or ruby to the left and below the parent character.
   */
  BELOW_LEFT = 1248551532,
}

/**
 * Kenten character set options.
 */
declare enum KentenCharacterSet {
  /**
   * Character input.
   */
  CHARACTER_INPUT = 1248028777,

  /**
   * JIS.
   */
  JIS = 1246382419,

  /**
   * Kuten.
   */
  KUTEN = 1248556404,

  /**
   * Shift JIS.
   */
  SHIFT_JIS = 1249077875,

  /**
   * Unicode.
   */
  UNICODE = 1249209961,
}

/**
 * Ruby type options.
 */
declare enum RubyTypes {
  /**
   * Provides ruby for a group of characters.
   */
  GROUP_RUBY = 1249011570,

  /**
   * Provides ruby for each individual character in the group.
   */
  PER_CHARACTER_RUBY = 1249013859,
}

/**
 * Ruby alignment options.
 */
declare enum RubyAlignments {
  /**
   * Ruby 1 aki.
   */
  RUBY_1_AKI = 1248997729,

  /**
   * Centers ruby relative to the parent text.
   */
  RUBY_CENTER = 1249010548,

  /**
   * Ruby equal aki.
   */
  RUBY_EQUAL_AKI = 1249011041,

  /**
   * Justifies ruby across the parent text.
   */
  RUBY_FULL_JUSTIFY = 1249011306,

  /**
   * Ruby JIS.
   */
  RUBY_JIS = 1249012339,

  /**
   * Aligns ruby with the left-most character in the parent text.
   */
  RUBY_LEFT = 1249012838,

  /**
   * Aligns ruby with the right-most character in the parent text.
   */
  RUBY_RIGHT = 1249014388,
}

/**
 * Options for ruby spacing relative to the parent text.
 */
declare enum RubyParentSpacing {
  /**
   * Ruby parent 121 aki.
   */
  RUBY_PARENT_121_AKI = 1248997682,

  /**
   * Ruby parent both sides.
   */
  RUBY_PARENT_BOTH_SIDES = 1249010291,

  /**
   * Applies the parent text aki to the ruby characters.
   */
  RUBY_PARENT_EQUAL_AKI = 1249014113,

  /**
   * Justifies ruby characters to both edges of the parent text.
   */
  RUBY_PARENT_FULL_JUSTIFY = 1249014634,

  /**
   * Does not base ruby spacing on parent text.
   */
  RUBY_PARENT_NO_ADJUSTMENT = 1249013345,
}

/**
 * Ruby overhang options.
 */
declare enum RubyOverhang {
  /**
   * Does not allow ruby overhang.
   */
  NONE = 1852796517,

  /**
   * Ruby is overhang one-half the size of one chararacter.
   */
  RUBY_OVERHANG_HALF_CHAR = 1249011811,

  /**
   * Ruby overhang is one-half ruby.
   */
  RUBY_OVERHANG_HALF_RUBY = 1249013554,

  /**
   * There is no ruby overhang size limit.
   */
  RUBY_OVERHANG_NO_LIMIT = 1249013621,

  /**
   * Ruby overhang is the size of one character.
   */
  RUBY_OVERHANG_ONE_CHAR = 1249013603,

  /**
   * Ruby overhang is one ruby.
   */
  RUBY_OVERHANG_ONE_RUBY = 1249013553,
}

/**
 * Warichu text alignment options.
 */
declare enum WarichuAlignment {
  /**
   * Automatically aligns warichu characters.
   */
  AUTO = 1635019116,

  /**
   * Aligns warichu in the center of the text frame.
   */
  CENTER_ALIGN = 1667591796,

  /**
   * Justifies warichu lines and center aligns the last line.
   */
  CENTER_JUSTIFIED = 1667920756,

  /**
   * Justifies warichu lines and makes all lines of equal length.
   */
  FULLY_JUSTIFIED = 1718971500,

  /**
   * Aligns warichu on the left side of the text frame.
   */
  LEFT_ALIGN = 1818584692,

  /**
   * Justifies warichu lines and left aligns the last line.
   */
  LEFT_JUSTIFIED = 1818915700,

  /**
   * Warichu on the rigt side of the text frame.
   */
  RIGHT_ALIGN = 1919379572,

  /**
   * Justifies warichu lines and right aligns the last line.
   */
  RIGHT_JUSTIFIED = 1919578996,
}

/**
 * Kinsoku processing options.
 */
declare enum KinsokuType {
  /**
   * The kinsoku prioritize adjustment amount.
   */
  KINSOKU_PRIORITIZE_ADJUSTMENT_AMOUNT = 1248553313,

  /**
   * Attempts to move characters to the previous line; if the push-in is not possible, pushes characters to the next line.
   */
  KINSOKU_PUSH_IN_FIRST = 1248553318,

  /**
   * Attempts to move characters to the next line; if the push-out is not possible, pushes characters to the previous line.
   */
  KINSOKU_PUSH_OUT_FIRST = 1248554854,

  /**
   * Always moves characters to the next line. Does not attempt a push-in.
   */
  KINSOKU_PUSH_OUT_ONLY = 1248554863,
}

/**
 * Hanging punctuation options when a kinsoku set is in effect.
 */
declare enum KinsokuHangTypes {
  /**
   * Enables hanging punctuation but forces hanging punctuation outside the text frame and does not allow the punctuation to be placed on the text frame.
   */
  KINSOKU_HANG_FORCE = 1248553062,

  /**
   * Enables hanging punctuation and allows punctuation marks to be placed on or outside the text frame but allows burasagari characters to hang as little as possible. Note: Differs for justified and nonjustified text. For information on justification, see line alignment.
   */
  KINSOKU_HANG_REGULAR = 1248553074,

  /**
   * Disables hanging punctuation.
   */
  NONE = 1852796517,
}

/**
 * Leading model options.
 */
declare enum LeadingModel {
  /**
   * Measures the space between lines from the aki above.
   */
  LEADING_MODEL_AKI_ABOVE = 1248616801,

  /**
   * Measures the space between lines from the aki below.
   */
  LEADING_MODEL_AKI_BELOW = 1248616802,

  /**
   * Measures the space between the character center points.
   */
  LEADING_MODEL_CENTER = 1248619875,

  /**
   * Center down leading model.
   */
  LEADING_MODEL_CENTER_DOWN = 1248617316,

  /**
   * Measures the space between type baselines.
   */
  LEADING_MODEL_ROMAN = 1248619858,
}

/**
 * Predefined kinsoku set options.
 */
declare enum KinsokuSet {
  /**
   * Uses the hard or maximum kinsoku set, which includes all Japanese characters that should not begin or end a line.
   */
  HARD_KINSOKU = 1248357235,

  /**
   * Uses the Korean kinsoku set.
   */
  KOREAN_KINSOKU = 1263692659,

  /**
   * Does not use a kinsoku set.
   */
  NOTHING = 1851876449,

  /**
   * Uses the simplified Chinese kinsoku set.
   */
  SIMPLIFIED_CHINESE_KINSOKU = 1396927347,

  /**
   * Uses the soft or weak kinsoku set, which omits from the hard kinsoku set long vowel symbols and small hiragana and katakana characters.
   */
  SOFT_KINSOKU = 1249078131,

  /**
   * Uses the traditional Chinese kinsoku set.
   */
  TRADITIONAL_CHINESE_KINSOKU = 1413704563,
}

/**
 * Mojikumi table options.
 */
declare enum MojikumiTableDefaults {
  /**
   * Uses full-width spacing for all characters.
   */
  LINE_END_ALL_ONE_EM_ENUM = 1246572848,

  /**
   * Uses half-width spacing for all characters.
   */
  LINE_END_ALL_ONE_HALF_EM_ENUM = 1246572593,

  /**
   * Uses full-width spacing for punctuation.
   */
  LINE_END_PERIOD_ONE_EM_ENUM = 1246572852,

  /**
   * Uses line end uke no float.
   */
  LINE_END_UKE_NO_FLOAT_ENUM = 1246572849,

  /**
   * Turns off mojikumi.
   */
  NOTHING = 1851876449,

  /**
   * Indents lines one full space and uses no float for all characters.
   */
  ONE_EM_INDENT_LINE_END_ALL_NO_FLOAT_ENUM = 1246572598,

  /**
   * Indents lines one full space and uses full-width spacing for all characters.
   */
  ONE_EM_INDENT_LINE_END_ALL_ONE_EM_ENUM = 1246572597,

  /**
   * Indents lines one full space and uses half-width spacing for all characters.
   */
  ONE_EM_INDENT_LINE_END_ALL_ONE_HALF_EM_ENUM = 1246572601,

  /**
   * Indents lines one full space and uses full-width spacing for punctuation and for the last character in the line.
   */
  ONE_EM_INDENT_LINE_END_PERIOD_ONE_EM_ENUM = 1246572851,

  /**
   * Indents lines one full space and uses line end uke no float.
   */
  ONE_EM_INDENT_LINE_END_UKE_NO_FLOAT_ENUM = 1246572599,

  /**
   * Indents lines one space and uses line end uke one half space.
   */
  ONE_EM_INDENT_LINE_END_UKE_ONE_HALF_EM_ENUM = 1246572594,

  /**
   * Uses full-witdh spacing for all characters except the last character in the line, which uses either full- or half-width spacing.
   */
  ONE_OR_ONE_HALF_EM_INDENT_LINE_END_ALL_ONE_EM_ENUM = 1246572596,

  /**
   * Indents lines one or one-half space and uses full-width spacing for punctuation and for the last character in the line.
   */
  ONE_OR_ONE_HALF_EM_INDENT_LINE_END_PERIOD_ONE_EM_ENUM = 1246572850,

  /**
   * Indents lines one half space or one full space and uses line end uke no float.
   */
  ONE_OR_ONE_HALF_EM_INDENT_LINE_END_UKE_NO_FLOAT_ENUM = 1246572600,

  /**
   * Indents lines one full or half space and uses line end uke one half space.
   */
  ONE_OR_ONE_HALF_EM_INDENT_LINE_END_UKE_ONE_HALF_EM_ENUM = 1246572595,

  /**
   * Uses mojikumi tsume and aki optimized for Simplified Chinese punctuation glyphs.
   */
  SIMP_CHINESE_DEFAULT = 1246572854,

  /**
   * Uses mojikumi tsume and aki optimized for Traditional Chinese centered punctuation glyphs.
   */
  TRAD_CHINESE_DEFAULT = 1246572853,
}

/**
 * Alignment options for frame grids or baseline grids.
 */
declare enum GridAlignment {
  /**
   * Aligns the text baseline to the grid.
   */
  ALIGN_BASELINE = 1247896172,

  /**
   * Aligns the bottom of the em box to the grid.
   */
  ALIGN_EM_BOTTOM = 1247896173,

  /**
   * Aligns the center of the em box to the grid.
   */
  ALIGN_EM_CENTER = 1247896436,

  /**
   * Aligns the top of the em box to the grid.
   */
  ALIGN_EM_TOP = 1247900784,

  /**
   * Aligns the bottom of the ICF box to the grid.
   */
  ALIGN_ICF_BOTTOM = 1248420461,

  /**
   * Aligns the top of the ICF box to the grid.
   */
  ALIGN_ICF_TOP = 1248425072,

  /**
   * Lines are not aligned to the grid.
   */
  NONE = 1852796517,
}

/**
 * Options for aligning kenten characters relative to the parent characters.
 */
declare enum KentenAlignment {
  /**
   * Aligns kenten with the center of parent charactrers.
   */
  ALIGN_KENTEN_CENTER = 1248554595,

  /**
   * Aligns kenten with the left of parent characters.
   */
  ALIGN_KENTEN_LEFT = 1248554604,
}

/**
 * Hyphenation exceptions list options for composing text.
 */
declare enum ComposeUsing {
  /**
   * Uses the lists stored in both the document and the user dictionary.
   */
  BOTH = 1651471464,

  /**
   * Uses the list stored in the document.
   */
  USE_DOCUMENT = 1967419235,

  /**
   * Uses the list stored in the external user dictionary.
   */
  USE_USER_DICTIONARY = 1433629284,
}

/**
 * Page number style options.
 */
declare enum PageNumberStyle {
  /**
   * Uses Arabic numerals.
   */
  ARABIC = 1298231906,

  /**
   * Uses Arabic Abjad
   */
  ARABIC_ABJAD = 1296130410,

  /**
   * Uses Arabic Alif Ba Tah
   */
  ARABIC_ALIF_BA_TAH = 1296130420,

  /**
   * Uses Arabic numerals and formats all page numbers as three digits.
   */
  DOUBLE_LEADING_ZEROS = 1296329850,

  /**
   * Uses Hebrew Biblical
   */
  HEBREW_BIBLICAL = 1296589410,

  /**
   * Uses Hebrew Non Standard
   */
  HEBREW_NON_STANDARD = 1296589422,

  /**
   * Uses Kanji.
   */
  KANJI = 1296788073,

  /**
   * Uses lowercase letters.
   */
  LOWER_LETTERS = 1296855660,

  /**
   * Uses lowercase Roman numerals.
   */
  LOWER_ROMAN = 1297247596,

  /**
   * Uses Arabic numerals and formats all page numbers as two digits.
   */
  SINGLE_LEADING_ZEROS = 1297312890,

  /**
   * Uses Arabic numerals and formats all page numbers as four digits.
   */
  TRIPLE_LEADING_ZEROS = 1297378426,

  /**
   * Uses uppercase letters.
   */
  UPPER_LETTERS = 1296855669,

  /**
   * Uses uppercase Roman numerals.
   */
  UPPER_ROMAN = 1297247605,
}

/**
 * Gradient type options.
 */
declare enum GradientType {
  /**
   * A linear gradient.
   */
  LINEAR = 1635282023,

  /**
   * A radial gradient.
   */
  RADIAL = 1918985319,
}

/**
 * Options for joining two path points.
 */
declare enum JoinOptions {
  /**
   * Combine two end points and replace with an single averaged point.
   */
  COMBINE = 1668113006,

  /**
   * Connect two end points (default).
   */
  CONNECT = 1668178804,
}

/**
 * Path type options.
 */
declare enum PathType {
  /**
   * The path is a closed path.
   */
  CLOSED_PATH = 1668051812,

  /**
   * The path is an open path.
   */
  OPEN_PATH = 1869639280,
}

/**
 * The path point type.
 */
declare enum PointType {
  /**
   * The point is a corner point, it has either one direction line, or two independent direction lines.
   */
  CORNER = 1668443762,

  /**
   * The point is a plain point, it has no direction lines.
   */
  PLAIN = 1886151022,

  /**
   * The point is a smooth point, it has two direction lines which are parallel.
   */
  SMOOTH = 1936553064,

  /**
   * A special type of smooth point with two direction lines of equal length.
   */
  SYMMETRICAL = 1937337709,
}

/**
 * Options for creating preview images.
 */
declare enum CreateProxy {
  /**
   * Always creates preview images.
   */
  ALWAYS = 1699307895,

  /**
   * Creates preview images as needed.
   */
  AS_NEEDED = 1699311204,
}

/**
 * The cropping option of an imported InDesign page.
 */
declare enum ImportedPageCropOptions {
  /**
   * Places the page's bleed area.
   */
  CROP_BLEED = 1131573314,

  /**
   * Places the page's bounding box.
   */
  CROP_CONTENT = 1131573315,

  /**
   * Places the page's slug area.
   */
  CROP_SLUG = 1131565932,
}

/**
 * Options for the active stroke/fill proxy.
 */
declare enum StrokeFillProxyOptions {
  /**
   * Fill proxy is active.
   */
  FILL = 1181314156,

  /**
   * Stroke proxy is active.
   */
  STROKE = 1400140395,
}

/**
 * Options for the target of the active stroke/fill proxy.
 */
declare enum StrokeFillTargetOptions {
  /**
   * Formatting affects the container.
   */
  FORMATTING_AFFECTS_CONTAINER = 1181696323,

  /**
   * Formatting affects the text.
   */
  FORMATTING_AFFECTS_TEXT = 1181696340,
}

/**
 * Options for rotating the contents of the place gun
 */
declare enum RotationDirection {
  /**
   * Rotate the list backward (i.e., move backmost item to front)
   */
  BACKWARD = 1113680759,

  /**
   * Rotate the list forward (i.e., move the front item to end)
   */
  FORWARD = 1181708919,
}

/**
 * Options for specifying location relative to the reference object or within the containing object.
 */
declare enum LocationOptions {
  /**
   * Places the object after the reference object.
   */
  AFTER = 1634104421,

  /**
   * Places the object at the beginning of the containing object.
   */
  AT_BEGINNING = 1650945639,

  /**
   * Places the object at the end of the containing object.
   */
  AT_END = 1701733408,

  /**
   * Places the object before the reference object.
   */
  BEFORE = 1650812527,

  /**
   * No location specified.
   */
  UNKNOWN = 1433299822,
}

/**
 * Standard UI colors.
 */
declare enum UIColors {
  /**
   * Black.
   */
  BLACK = 1765960811,

  /**
   * Blue.
   */
  BLUE = 1765960821,

  /**
   * Brick red.
   */
  BRICK_RED = 1765962340,

  /**
   * Brown.
   */
  BROWN = 1765962350,

  /**
   * Burgundy.
   */
  BURGUNDY = 1765962343,

  /**
   * Charcoal.
   */
  CHARCOAL = 1766025324,

  /**
   * Cute teal.
   */
  CUTE_TEAL = 1766028396,

  /**
   * Cyan.
   */
  CYAN = 1766029678,

  /**
   * Dark blue.
   */
  DARK_BLUE = 1766089324,

  /**
   * Dark green.
   */
  DARK_GREEN = 1766090610,

  /**
   * Fiesta.
   */
  FIESTA = 1766222181,

  /**
   * Gold.
   */
  GOLD = 1766288484,

  /**
   * Grass green.
   */
  GRASS_GREEN = 1766287218,

  /**
   * Gray.
   */
  GRAY = 1766290041,

  /**
   * Green.
   */
  GREEN = 1766290030,

  /**
   * Grid blue.
   */
  GRID_BLUE = 1766285932,

  /**
   * Grid green.
   */
  GRID_GREEN = 1766286439,

  /**
   * Grid orange.
   */
  GRID_ORANGE = 1766289266,

  /**
   * Lavender.
   */
  LAVENDER = 1766618734,

  /**
   * Light blue.
   */
  LIGHT_BLUE = 1766613612,

  /**
   * Light gray.
   */
  LIGHT_GRAY = 1766614898,

  /**
   * Light olive.
   */
  LIGHT_OLIVE = 1766616940,

  /**
   * Lipstick.
   */
  LIPSTICK = 1766615408,

  /**
   * Magenta.
   */
  MAGENTA = 1766680430,

  /**
   * Ochre.
   */
  OCHRE = 1766810482,

  /**
   * Olive green.
   */
  OLIVE_GREEN = 1766812790,

  /**
   * Orange.
   */
  ORANGE = 1766814318,

  /**
   * Peach.
   */
  PEACH = 1766876008,

  /**
   * Pink.
   */
  PINK = 1766878827,

  /**
   * Purple.
   */
  PURPLE = 1766879856,

  /**
   * Red.
   */
  RED = 1767007588,

  /**
   * Sulphur.
   */
  SULPHUR = 1767077228,

  /**
   * Tan.
   */
  TAN = 1767137646,

  /**
   * Teal.
   */
  TEAL = 1767138668,

  /**
   * Violet.
   */
  VIOLET = 1767271540,

  /**
   * White.
   */
  WHITE = 1767336052,

  /**
   * Yellow.
   */
  YELLOW = 1767468151,
}

/**
 * Nothing.
 */
declare enum NothingEnum {
  /**
   * Nothing
   */
  NOTHING = 1851876449,
}

/**
 * The default value.
 */
declare enum AutoEnum {
  /**
   * Uses the default value defined automatically for the object based on a parent or other type of object.
   */
  AUTO_VALUE = 1635087471,
}

/**
 * Phase options for event propagation.
 */
declare enum EventPhases {
  /**
   * The at-target phase of propagation.
   */
  AT_TARGET = 1701724500,

  /**
   * The bubbling phase of propagation.
   */
  BUBBLING_PHASE = 1701724789,

  /**
   * The propagation is complete.
   */
  DONE = 1701725252,

  /**
   * Not yet propagating.
   */
  NOT_DISPATCHING = 1701727812,
}

/**
 * Export format options.
 */
declare enum ExportFormat {
  /**
   * Exports to EPS format.
   */
  EPS_TYPE = 1952400720,

  /**
   * Exports to EPub format.
   */
  EPUB = 1701868898,

  /**
   * Exports to fixed layout EPub format.
   */
  FIXED_LAYOUT_EPUB = 1701865080,

  /**
   * Exports to XHTML format.
   */
  HTML = 1213484364,

  /**
   * Exports to XHTML FXL format.
   */
  HTMLFXL = 1213490808,

  /**
   * Exports to InCopy markup (ICML) format.
   */
  INCOPY_MARKUP = 1768123756,

  /**
   * Exports to InDesign markup (IDML) format.
   */
  INDESIGN_MARKUP = 1768189292,

  /**
   * Exports to InDesign snippet (IDMS) format.
   */
  INDESIGN_SNIPPET = 1936617588,

  /**
   * Exports to Interactive PDF format.
   */
  INTERACTIVE_PDF = 1952409936,

  /**
   * Exports to JPEG format.
   */
  JPG = 1246775072,

  /**
   * Exports to packaged XFL format.
   */
  PACKAGED_XFL = 1702389356,

  /**
   * Exports to PDF format.
   */
  PDF_TYPE = 1952403524,

  /**
   * Exports to PNG format.
   */
  PNG_FORMAT = 1699761735,

  /**
   * Exports to rich text format (RTF).
   */
  RTF = 1381254688,

  /**
   * Exports to SWF format.
   */
  SWF = 1702066022,

  /**
   * Exports to a tagged text file with a TXT extension.
   */
  TAGGED_TEXT = 1416066168,

  /**
   * Exports to text (TXT) format.
   */
  TEXT_TYPE = 1952412773,

  /**
   * Exports the document's tagged content to XML.
   */
  XML = 1481460768,
}

/**
 * The color profile policy for placed vector files (PDF or EPS).
 */
declare enum PlacedVectorProfilePolicy {
  /**
   * Honors all profiles and output intent.
   */
  HONOR_ALL_PROFILES = 1148217441,

  /**
   * Ignores all profiles and output intent.
   */
  IGNORE_ALL = 1148217697,

  /**
   * Ignores output intent; honors calibrated spaces.
   */
  IGNORE_OUTPUT_INTENT = 1148217711,
}

/**
 * The policy for handling mismatched CMYK configurations.
 */
declare enum ColorSettingsPolicy {
  /**
   * Turns off color management for documents whose profiles do not match the working space. For imported colors, numeric values override color appearance.
   */
  COLOR_POLICY_OFF = 1129344870,

  /**
   * Preserves raw color numbers and ignores embedded color profiles.
   */
  COMBINATION_OF_PRESERVE_AND_SAFE_CMYK = 1129345124,

  /**
   * Converts newly opened documents to the current working space. For imported colors, color appearance overrides numeric values.
   */
  CONVERT_TO_WORKING_SPACE = 1129346931,

  /**
   * Preserves embedded color profiles in newly opened documents.
   */
  PRESERVE_EMBEDDED_PROFILES = 1129345136,
}

/**
 * The default rendering intent.
 */
declare enum DefaultRenderingIntent {
  /**
   * Aims to maintain color accuracy at the expense of color relationshps and is suitable for proofing to simulate the output of a particular device. Note: Leaves colors that fall inside the destination gamut unchanged and clips out-of-gamut colors.
   */
  ABSOLUTE_COLORIMETRIC = 1380540771,

  /**
   * Aims to preserve the visual relationship between colors so they are perceived as natural to the human eye, even though the color values themselves may change.
   */
  PERCEPTUAL = 1380544611,

  /**
   * Compares the extreme highlight of the source color space to that of the destination color space and shifts all colors accordingly. Out-of-gamut colors are shifted to the closest reproducible color in the destination color space. Note: Preserves more of the original colors in an image than perceptual rendering intent does.
   */
  RELATIVE_COLORIMETRIC = 1380545123,

  /**
   * Tries to produce vivid colors in an image at the expense of color accuracy.
   */
  SATURATION = 1380545377,
}

/**
 * Options for proofing colors.
 */
declare enum ProofingType {
  /**
   * Allows creation of a custom proofing setup for a specific output condition.
   */
  CUSTOM = 1131639917,

  /**
   * Creates a soft proof of colors using the document's CMYK profile.
   */
  DOCUMENT_CMYK = 1347708003,

  /**
   * Turns off soft proof display.
   */
  PROOF_OFF = 1347710822,

  /**
   * Creates a soft proof of colors using the current CMYK working space.
   */
  WORKING_CMYK = 1347712867,
}

/**
 * Change conditions modes.
 */
declare enum ChangeConditionsModes {
  /**
   * Change adds to applied conditions.
   */
  ADD_TO = 1633969202,

  /**
   * Change replaces applied conditions.
   */
  REPLACE_WITH = 1919250519,
}

/**
 * Search mode options.
 */
declare enum SearchModes {
  /**
   * Glyph search.
   */
  GLYPH_SEARCH = 1181183091,

  /**
   * Grep search.
   */
  GREP_SEARCH = 1181184627,

  /**
   * Object search.
   */
  OBJECT_SEARCH = 1181704819,

  /**
   * Text search.
   */
  TEXT_SEARCH = 1182038131,
}

/**
 * Object type options.
 */
declare enum ObjectTypes {
  /**
   * All frame types.
   */
  ALL_FRAMES_TYPE = 1178682995,

  /**
   * Graphics frame.
   */
  GRAPHIC_FRAMES_TYPE = 1179076211,

  /**
   * Text frame.
   */
  TEXT_FRAMES_TYPE = 1179928178,

  /**
   * Unassigned frame.
   */
  UNASSIGNED_FRAMES_TYPE = 1179993715,
}

/**
 * Find/change transliterate character type options.
 */
declare enum FindChangeTransliterateCharacterTypes {
  /**
   * Arabic Indic(hindi) digits.
   */
  ARABIC_INDIC_DIGITS = 1095328873,

  /**
   * Farsi digits.
   */
  FARSI_DIGITS = 1684629089,

  /**
   * Full-width hiragana.
   */
  FULL_WIDTH_HIRAGANA = 1179023176,

  /**
   * Full-width katakana.
   */
  FULL_WIDTH_KATAKANA = 1179023179,

  /**
   * Full-width Roman symbols.
   */
  FULL_WIDTH_ROMAN_SYMBOLS = 1179023186,

  /**
   * Half-width katakana.
   */
  HALF_WIDTH_KATAKANA = 1179154251,

  /**
   * Half-width Roman symbols.
   */
  HALF_WIDTH_ROMAN_SYMBOLS = 1179154258,

  /**
   * Western Arabic digits (0, 1, 2, 3, ...).
   */
  WESTERN_ARABIC_DIGITS = 1463903337,
}

/**
 * Library panel views
 */
declare enum LibraryPanelViews {
  /**
   * Large thumbnail view
   */
  LARGE_THUMBNAIL_VIEW = 1699501142,

  /**
   * List view
   */
  LIST_VIEW = 1699501673,

  /**
   * Thumbnail view
   */
  THUMBNAIL_VIEW = 1700030550,
}

/**
 * Sort order options
 */
declare enum SortAssets {
  /**
   * Sort by name
   */
  BY_NAME = 1699955278,

  /**
   * Sort newest first
   */
  BY_NEWEST = 1699955310,

  /**
   * Sort oldest first
   */
  BY_OLDEST = 1699955279,

  /**
   * Sort by type
   */
  BY_TYPE = 1699955284,
}

/**
 * Link status options.
 */
declare enum LinkStatus {
  /**
   * The file is embedded in the document.
   */
  LINK_EMBEDDED = 1282237028,

  /**
   * The url link is inaccessible.
   */
  LINK_INACCESSIBLE = 1818848865,

  /**
   * The linked file has been moved, renamed, or deleted.
   */
  LINK_MISSING = 1819109747,

  /**
   * A more recent version of the file exists on the disk.
   */
  LINK_OUT_OF_DATE = 1819242340,

  /**
   * The link is a normal link.
   */
  NORMAL = 1852797549,
}

/**
 * The rendition type of the link resource.
 */
declare enum LinkResourceRenditionType {
  /**
   * The link resource has original rendition.
   */
  ACTUAL = 1282372201,

  /**
   * The link resource has FPO rendition.
   */
  FPO = 1281781871,
}

/**
 * Asset type options.
 */
declare enum AssetType {
  /**
   * The asset is cataloged as an EPS asset.
   */
  EPS_TYPE = 1952400720,

  /**
   * The asset is cataloged as a geometric page item asset.
   */
  GEOMETRY_TYPE = 1952409445,

  /**
   * The asset is cataloged as an image asset.
   */
  IMAGE_TYPE = 1952409965,

  /**
   * InDesign file asset
   */
  INDESIGN_FILE_TYPE = 1952409956,

  /**
   * The asset is cataloged as a page asset.
   */
  PAGE_TYPE = 1952411745,

  /**
   * The asset is cataloged as a PDF asset.
   */
  PDF_TYPE = 1952403524,

  /**
   * Structure asset
   */
  STRUCTURE_TYPE = 1952412532,

  /**
   * The asset is cataloged as a text asset.
   */
  TEXT_TYPE = 1952412773,
}

/**
 * Change type options.
 */
declare enum ChangeTypes {
  /**
   * Deleted text.
   */
  DELETED_TEXT = 1799644524,

  /**
   * Added text.
   */
  INSERTED_TEXT = 1799974515,

  /**
   * Moved text.
   */
  MOVED_TEXT = 1800236918,
}

/**
 * Options for the alignment and appearance of type on a path.
 */
declare enum TextPathEffects {
  /**
   * The center of each character's baseline is on the path while each vertical edge is in line with the path's center point.
   */
  GRAVITY_PATH_EFFECT = 1601201767,

  /**
   * The center of each character's baseline is parallel to the path's tangent. This is the default effect.
   */
  RAINBOW_PATH_EFFECT = 1601201778,

  /**
   * The text characters' horizontal edges are perfectly horizontal regardless of the path shape.
   */
  RIBBON_PATH_EFFECT = 1601201715,

  /**
   * The text characters' vertical edges are perfectly vertical regardless of the path shape.
   */
  SKEW_PATH_EFFECT = 1601201779,

  /**
   * The left edge of each character's baseline is on the path and no characters are rotated.
   */
  STAIR_STEP_PATH_EFFECT = 1601205107,
}

/**
 * The text alignment relative to the path.
 */
declare enum TextTypeAlignments {
  /**
   * The top-edge or right-edge baseline of the em box is aligned to the path.
   */
  ABOVE_RIGHT_EM_BOX_TEXT_ALIGNMENT = 1952543333,

  /**
   * The ideographic character face box top-edge or right-edge baseline is aligned to the path.
   */
  ABOVE_RIGHT_ICF_BOX_TEXT_ALIGNMENT = 1952543337,

  /**
   * Aligns the ascender to the path (not the path's stroke).
   */
  ASCENDER_TEXT_ALIGNMENT = 1952538995,

  /**
   * The text baseline is aligned to the path (not the path's stroke).
   */
  BASELINE_TEXT_ALIGNMENT = 1952539244,

  /**
   * The bottom-edge or left-edge baseline of the em box is aligned to the path.
   */
  BELOW_LEFT_EM_BOX_TEXT_ALIGNMENT = 1952607333,

  /**
   * The ideographic character face box bottom-edge or left-edge baseline is aligned to the path.
   */
  BELOW_LEFT_ICF_BOX_TEXT_ALIGNMENT = 1952607337,

  /**
   * Aligns the midpoint between the ascender and the descender to the path (not the path's stroke).
   */
  CENTER_TEXT_ALIGNMENT = 1952539508,

  /**
   * Aligns descender to the path (not the path's stroke).
   */
  DESCENDER_TEXT_ALIGNMENT = 1952539763,
}

/**
 * Options for aligning text to the path's stroke.
 */
declare enum PathTypeAlignments {
  /**
   * The text is aligned to the bottom of the path stroke.
   */
  BOTTOM_PATH_ALIGNMENT = 1885430367,

  /**
   * The text is aligned to the center of the path stroke.
   */
  CENTER_PATH_ALIGNMENT = 1885430623,

  /**
   * The text is aligned to the top of the path stroke.
   */
  TOP_PATH_ALIGNMENT = 1885434975,
}

/**
 * Options for flipping or unflipping text relative to the path.
 */
declare enum FlipValues {
  /**
   * Flips the text across the path.
   */
  FLIPPED = 2036755568,

  /**
   * No flip effect applied.
   */
  NOT_FLIPPED = 1852206192,

  /**
   * Undefined flip effect.
   */
  UNDEFINED_FLIP_VALUE = 1969646704,
}

/**
 * Location options for XML elements.
 */
declare enum XMLElementLocation {
  /**
   * Locates the element at the end of the containing object.
   */
  ELEMENT_END = 1483042404,

  /**
   * Locates the element at the beginning of the containing object.
   */
  ELEMENT_START = 1482844014,
}

/**
 * Options for specifying position relative to the reference XML element or withing the XML element.
 */
declare enum XMLElementPosition {
  /**
   * Specifies the position after the XML element.
   */
  AFTER_ELEMENT = 1482778228,

  /**
   * Specifies the position before the XML element.
   */
  BEFORE_ELEMENT = 1482843494,

  /**
   * Specifies the position at the end of the XML element.
   */
  ELEMENT_END = 1483042404,

  /**
   * Specifies the position at the beginning of the XML element.
   */
  ELEMENT_START = 1482844014,
}

/**
 * Options for incorporating imported XML content.
 */
declare enum XMLImportStyles {
  /**
   * Appends the imported content.
   */
  APPEND_IMPORT = 1481466217,

  /**
   * Merges the imported content.
   */
  MERGE_IMPORT = 1481469289,
}

/**
 * File encoding options for exported XML content.
 */
declare enum XMLFileEncoding {
  /**
   * Shift-JIS encoding.
   */
  SHIFT_JIS = 1249077875,

  /**
   * UTF-16 encoding.
   */
  UTF16 = 1937125686,

  /**
   * UTF-8 encoding.
   */
  UTF8 = 1937134904,
}

/**
 * File format options for converted images.
 */
declare enum ImageConversion {
  /**
   * Uses the best format based on the image.
   */
  AUTOMATIC = 1768059764,

  /**
   * Uses GIF format for all images.
   */
  GIF = 1734960742,

  /**
   * Uses JPEG format for all images.
   */
  JPEG = 1785751398,

  /**
   * Uses PNG format for all images.
   */
  PNG = 1397059687,
}

/**
 * Color palette options for GIF conversion.
 */
declare enum GIFOptionsPalette {
  /**
   * Uses the adaptive (no dither) palette.
   */
  ADAPTIVE_PALETTE = 1886151024,

  /**
   * Uses the Macintosh palette.
   */
  MACINTOSH_PALETTE = 1886154096,

  /**
   * Uses the Web palette.
   */
  WEB_PALETTE = 1886156656,

  /**
   * Uses the Windows palette.
   */
  WINDOWS_PALETTE = 1886156644,
}

/**
 * Quality options for converted JPEG images.
 */
declare enum JPEGOptionsQuality {
  /**
   * High quality.
   */
  HIGH = 1701726313,

  /**
   * Low quality.
   */
  LOW = 1701727351,

  /**
   * Maximum quality.
   */
  MAXIMUM = 1701727608,

  /**
   * Medium quality.
   */
  MEDIUM = 1701727588,
}

/**
 * Formatting options for converted JPEG images.
 */
declare enum JPEGOptionsFormat {
  /**
   * Uses baseline encoding to download the image in one pass.
   */
  BASELINE_ENCODING = 1785751394,

  /**
   * Uses progressive encoding to download the image in a series of passes, with the first pass at low resolution and each successive pass adding resolution to the image.
   */
  PROGRESSIVE_ENCODING = 1785751408,
}

/**
 * Export options for untagged tables in tagged stories.
 */
declare enum XMLExportUntaggedTablesFormat {
  /**
   * Exports untagged tables as CALS XML.
   */
  CALS = 1484022643,

  /**
   * Does not export untagged tables.
   */
  NONE = 1852796517,
}

/**
 * File used for XML transformation.
 */
declare enum XMLTransformFile {
  /**
   * Use stylesheet specified in XML
   */
  STYLESHEET_IN_XML = 1483961208,
}

/**
 * Options for specifying the position of the anchored object relative to the its anchor.
 */
declare enum AnchorPosition {
  /**
   * Place the anchored object above the line of text that contains the object.
   */
  ABOVE_LINE = 1095716961,

  /**
   * Custom anchor position.
   */
  ANCHORED = 1097814113,

  /**
   * Align the anchored object with the baseline of the line that contains the object.
   */
  INLINE_POSITION = 1095716969,
}

/**
 * The horizontal alignment of an anchored object. Not valid when anchored position is inline.
 */
declare enum HorizontalAlignment {
  /**
   * Place the anchored object at the center of the reference.
   */
  CENTER_ALIGN = 1667591796,

  /**
   * Place the anchored object to the left of the reference.
   */
  LEFT_ALIGN = 1818584692,

  /**
   * Place the anchored object to the right of the reference.
   */
  RIGHT_ALIGN = 1919379572,

  /**
   * Place the anchored object relative to the text alignment.
   */
  TEXT_ALIGN = 1954046316,
}

/**
 * The vertical alignment of an anchored object.
 */
declare enum VerticalAlignment {
  /**
   * Place the anchored object at the bottom of the vertical reference point.
   */
  BOTTOM_ALIGN = 1651471469,

  /**
   * Place the anchored object at the vertical center of the vertical reference point.
   */
  CENTER_ALIGN = 1667591796,

  /**
   * Place the anchored object at the top of the vertical reference point.
   */
  TOP_ALIGN = 1953460256,
}

/**
 * The horizontal alignment point of an anchored object.
 */
declare enum AnchoredRelativeTo {
  /**
   * Align the anchored object to the anchor.
   */
  ANCHOR_LOCATION = 1095786862,

  /**
   * Align the anchored object to the edge of the text or table column.
   */
  COLUMN_EDGE = 1095787375,

  /**
   * Align the anchored object to the edge of the page.
   */
  PAGE_EDGE = 1095790695,

  /**
   * Align the anchored object to the page margin.
   */
  PAGE_MARGINS = 1095789927,

  /**
   * Align the anchored object to the edge of the text frame.
   */
  TEXT_FRAME = 1954051174,
}

/**
 * Options for balancing line endings in the text.
 */
declare enum BalanceLinesStyle {
  /**
   * Balances lines equally.
   */
  FULLY_BALANCED = 1114391921,

  /**
   * Does not balance lines.
   */
  NO_BALANCING = 1114394470,

  /**
   * Prefers longer last lines.
   */
  PYRAMID_SHAPE = 1114394745,

  /**
   * Prefers shorter last lines.
   */
  VEE_SHAPE = 1114396261,
}

/**
 * The vertical alignment point of an anchored object.
 */
declare enum VerticallyRelativeTo {
  /**
   * Align the anchored object to the top of capital letters.
   */
  CAPHEIGHT = 1096185955,

  /**
   * Align the anchored object to the edge of the text or table column.
   */
  COLUMN_EDGE = 1095787375,

  /**
   * Align the anchored object to the bottom of the embox.
   */
  EMBOX_BOTTOM = 1096183106,

  /**
   * Align the anchored object to the middle of the embox.
   */
  EMBOX_MIDDLE = 1096183117,

  /**
   * Align the anchored object to the top of the embox.
   */
  EMBOX_TOP = 1096181101,

  /**
   * Align the anchored object to the top of the tallest letters in the text.
   */
  LINE_ASCENT = 1096180083,

  /**
   * Align the anchored object to the baseline of the line of text.
   */
  LINE_BASELINE = 1096180321,

  /**
   * Align the anchored object to the top of lower case letters with no ascent, such as x.
   */
  LINE_XHEIGHT = 1096185960,

  /**
   * Align the anchored object to the edge of the page.
   */
  PAGE_EDGE = 1095790695,

  /**
   * Align the anchored object to the page margin.
   */
  PAGE_MARGINS = 1095789927,

  /**
   * Align the anchored object to the edge of the text frame.
   */
  TEXT_FRAME = 1954051174,

  /**
   * Align the anchored object to the top of the text leading.
   */
  TOP_OF_LEADING = 1096180332,
}

/**
 * OpenType positional form options.
 */
declare enum PositionalForms {
  /**
   * Calculated forms.
   */
  CALCULATE = 1634756205,

  /**
   * Final form.
   */
  FINAL = 1718185569,

  /**
   * Initial form.
   */
  INITIAL = 1768843636,

  /**
   * Isolated form.
   */
  ISOLATED = 1769172844,

  /**
   * Medial form.
   */
  MEDIAL = 1835361385,

  /**
   * None.
   */
  NONE = 1852796517,
}

/**
 * Stroke alignment options.
 */
declare enum TextStrokeAlign {
  /**
   * The stroke straddles the path.
   */
  CENTER_ALIGNMENT = 1936998723,

  /**
   * The stroke is outside the path, like a picture frame.
   */
  OUTSIDE_ALIGNMENT = 1936998735,
}

/**
 * End join types.
 */
declare enum OutlineJoin {
  /**
   * Beveled end join.
   */
  BEVEL_END_JOIN = 1651142510,

  /**
   * Miter end join.
   */
  MITER_END_JOIN = 1835691886,

  /**
   * Rounded end join.
   */
  ROUND_END_JOIN = 1919577966,
}

/**
 * The starting point used to calculate the baseline frame grid offset.
 */
declare enum BaselineFrameGridRelativeOption {
  /**
   * Offsets the grid from the top of the text frame.
   */
  TOP_OF_FRAME = 1163161446,

  /**
   * Offsets the grid from the top inset of the text frame.
   */
  TOP_OF_INSET = 1163161449,

  /**
   * Offsets the grid from the top margin of the page.
   */
  TOP_OF_MARGIN = 1163161453,

  /**
   * Offsets the grid from the top of the page.
   */
  TOP_OF_PAGE = 1163161458,
}

/**
 * Options for specifying the amount of vertical space between the top of the footnote container and the first line of footnote text.
 */
declare enum FootnoteFirstBaseline {
  /**
   * The tallest character in the font falls below the top of the footnote container.
   */
  ASCENT_OFFSET = 1296135023,

  /**
   * The tops of upper case letters touch the top of the footnote container.
   */
  CAP_HEIGHT = 1296255087,

  /**
   * The em box height of the text defines the distance between the baseline of the text and the top of the footnote container.
   */
  EMBOX_HEIGHT = 1296386159,

  /**
   * The footnote minimum first baseline offset value defines the distance between the baseline of the text and the top of the footnote container.
   */
  FIXED_HEIGHT = 1313228911,

  /**
   * The leading value of the text defines the distance between the baseline of the text and the top of the footnote container.
   */
  LEADING_OFFSET = 1296852079,

  /**
   * The tops of lower case letters without ascents, such as x, touch the top of the footnote container.
   */
  X_HEIGHT = 1299728495,
}

/**
 * Options for restarting footnote numbering.
 */
declare enum FootnoteRestarting {
  /**
   * Does not restart numbering; numbers footnotes sequentially throughout the document.
   */
  DONT_RESTART = 1180988019,

  /**
   * Restarts footnote numbering on each page.
   */
  PAGE_RESTART = 1181774451,

  /**
   * Restarts footnote numbering in each section.
   */
  SECTION_RESTART = 1181053555,

  /**
   * Restarts footnote numbering on each spread.
   */
  SPREAD_RESTART = 1181971059,
}

/**
 * Footnote prefix or suffix placement options.
 */
declare enum FootnotePrefixSuffix {
  /**
   * Does not use a prefix or suffix.
   */
  NO_PREFIX_SUFFIX = 1181774702,

  /**
   * Places the prefix and/or suffix on both the the footnote reference number in the main text and the footnote marker number in the footnote text.
   */
  PREFIX_SUFFIX_BOTH = 1181774690,

  /**
   * Places the prefix and/or suffix on the footnote marker number in the footnote text.
   */
  PREFIX_SUFFIX_MARKER = 1181774708,

  /**
   * Places the prefix and/or suffix on the footnote reference number in the main text.
   */
  PREFIX_SUFFIX_REFERENCE = 1181774706,
}

/**
 * Footnote numbering style options.
 */
declare enum FootnoteNumberingStyle {
  /**
   * Uses Arabic numerals.
   */
  ARABIC = 1298231906,

  /**
   * Uses Arabic Abjad
   */
  ARABIC_ABJAD = 1296130410,

  /**
   * Uses Arabic Alif Ba Tah
   */
  ARABIC_ALIF_BA_TAH = 1296130420,

  /**
   * Asterisks.
   */
  ASTERISKS = 1298232180,

  /**
   * Double leading zeros.
   */
  DOUBLE_LEADING_ZEROS = 1296329850,

  /**
   * Full-width Arabic.
   */
  FULL_WIDTH_ARABIC = 1296455521,

  /**
   * Uses Hebrew Biblical
   */
  HEBREW_BIBLICAL = 1296589410,

  /**
   * Uses Hebrew Non Standard
   */
  HEBREW_NON_STANDARD = 1296589422,

  /**
   * Kanji.
   */
  KANJI = 1296788073,

  /**
   * Uses lowercase letters.
   */
  LOWER_LETTERS = 1296855660,

  /**
   * Uses lowercase Roman numerals.
   */
  LOWER_ROMAN = 1297247596,

  /**
   * Single leading zeros.
   */
  SINGLE_LEADING_ZEROS = 1297312890,

  /**
   * Uses symbols.
   */
  SYMBOLS = 1181971321,

  /**
   * Uses uppercase letters.
   */
  UPPER_LETTERS = 1296855669,

  /**
   * Uses uppercase Roman numerals.
   */
  UPPER_ROMAN = 1297247605,
}

/**
 * Options for positioning footnote reference numbers relative to characters the main text.
 */
declare enum FootnoteMarkerPositioning {
  /**
   * Uses the position defined in the character style applied to footnote reference numbers. For information, see footnote marker style.
   */
  NORMAL_MARKER = 1181576816,

  /**
   * Gives the marker ruby style positioning.
   */
  RUBY_MARKER = 1181577840,

  /**
   * Subscripts footnote reference numbers.
   */
  SUBSCRIPT_MARKER = 1181578096,

  /**
   * Superscripts footnote reference numbers.
   */
  SUPERSCRIPT_MARKER = 1181569904,
}

/**
 * Override clearing options.
 */
declare enum OverrideType {
  /**
   * Clears all types of override.
   */
  ALL = 1634495520,

  /**
   * Clears only character style overrides.
   */
  CHARACTER_ONLY = 1667789423,

  /**
   * Clears only paragraph style overrides.
   */
  PARAGRAPH_ONLY = 1885434479,
}

/**
 * The resolution strategy to employ for imported styles that have the same names as existing styles.
 */
declare enum GlobalClashResolutionStrategy {
  /**
   * Does not import styles whose names clash with existing items.
   */
  DO_NOT_LOAD_THE_STYLE = 1147495276,

  /**
   * Overwrites existing styles whose names clash with imported items.
   */
  LOAD_ALL_WITH_OVERWRITE = 1279350607,

  /**
   * Renames imported styles whose names clash with existing items to preserve existing items.
   */
  LOAD_ALL_WITH_RENAME = 1279350610,
}

/**
 * Text orientation options.
 */
declare enum StoryHorizontalOrVertical {
  /**
   * Orients the text horizontally.
   */
  HORIZONTAL = 1752134266,

  /**
   * The text direction is unknown.
   */
  UNKNOWN = 1433299822,

  /**
   * Orients the text vertically.
   */
  VERTICAL = 1986359924,
}

/**
 * Options for auto page insertion in response to overset text.
 */
declare enum AddPageOptions {
  /**
   * Insert pages at end of document.
   */
  END_OF_DOCUMENT = 1634037604,

  /**
   * Insert pages at end of section.
   */
  END_OF_SECTION = 1634037624,

  /**
   * Insert pages at end of story.
   */
  END_OF_STORY = 1634037619,
}

/**
 * Text variable types.
 */
declare enum VariableTypes {
  /**
   * Chapter number variable.
   */
  CHAPTER_NUMBER_TYPE = 1668183152,

  /**
   * Creation date variable.
   */
  CREATION_DATE_TYPE = 1414947684,

  /**
   * Custom text variable.
   */
  CUSTOM_TEXT_TYPE = 1414947700,

  /**
   * File name variable.
   */
  FILE_NAME_TYPE = 1414948462,

  /**
   * Last page number variable.
   */
  LAST_PAGE_NUMBER_TYPE = 1414952048,

  /**
   * Live Caption variable.
   */
  LIVE_CAPTION_TYPE = 1414947693,

  /**
   * Running header (character style) variable.
   */
  MATCH_CHARACTER_STYLE_TYPE = 1414947667,

  /**
   * Running header (paragraph style) variable.
   */
  MATCH_PARAGRAPH_STYLE_TYPE = 1414950995,

  /**
   * Modification date variable.
   */
  MODIFICATION_DATE_TYPE = 1414950244,

  /**
   * Output date variable.
   */
  OUTPUT_DATE_TYPE = 1414950756,

  /**
   * Private cross reference chapter number variable.
   */
  XREF_CHAPTER_NUMBER_TYPE = 1414947694,

  /**
   * Private cross reference page number variable.
   */
  XREF_PAGE_NUMBER_TYPE = 1414953074,
}

/**
 * Scopes for page number variable.
 */
declare enum VariableScopes {
  /**
   * The scope is limited to the current document.
   */
  DOCUMENT_SCOPE = 1129538671,

  /**
   * The scope is limited to the current section.
   */
  SECTION_SCOPE = 1129542501,
}

/**
 * Number variable numbering styles.
 */
declare enum VariableNumberingStyles {
  /**
   * Arabic numerals.
   */
  ARABIC = 1298231906,

  /**
   * Current
   */
  CURRENT = 1298363762,

  /**
   * Double leading zeros.
   */
  DOUBLE_LEADING_ZEROS = 1296329850,

  /**
   * Full-width Arabic numerals.
   */
  FULL_WIDTH_ARABIC = 1296455521,

  /**
   * Kanji.
   */
  KANJI = 1296788073,

  /**
   * Lowercase letters.
   */
  LOWER_LETTERS = 1296855660,

  /**
   * Lowercase Roman numerals.
   */
  LOWER_ROMAN = 1297247596,

  /**
   * Single leading zero.
   */
  SINGLE_LEADING_ZEROS = 1297312890,

  /**
   * Uppercase letters.
   */
  UPPER_LETTERS = 1296855669,

  /**
   * Uppercase Roman numerals.
   */
  UPPER_ROMAN = 1297247605,
}

/**
 * Search strategy options.
 */
declare enum SearchStrategies {
  /**
   * Searches forward from the start of the current page.
   */
  FIRST_ON_PAGE = 1396794992,

  /**
   * Search backward from the end of the current page.
   */
  LAST_ON_PAGE = 1396796528,
}

/**
 * Change case options.
 */
declare enum ChangeCaseOptions {
  /**
   * Converts all letters to lowercase.
   */
  LOWERCASE = 1667460195,

  /**
   * No conversion.
   */
  NONE = 1852796517,

  /**
   * Converts the first letter of the first word of each sentence to uppercase.
   */
  SENTENCECASE = 1667461987,

  /**
   * Converts the first letter of each word to uppercase.
   */
  TITLECASE = 1667462243,

  /**
   * Converts all letters to uppercase.
   */
  UPPERCASE = 1667462499,
}

/**
 * Auto sizing type options for text.
 */
declare enum AutoSizingTypeEnum {
  /**
   * Text frame will be resized in both(height and width) dimensions.
   */
  HEIGHT_AND_WIDTH = 1752069993,

  /**
   * Text frame will be resized in both(height and width) dimensions proportionally.
   */
  HEIGHT_AND_WIDTH_PROPORTIONALLY = 1752070000,

  /**
   * Text frame will be resized in height dimension only.
   */
  HEIGHT_ONLY = 1751476583,

  /**
   * Text frame auto-sizing is off.
   */
  OFF = 1330005536,

  /**
   * Text frame will be resized in width dimension only.
   */
  WIDTH_ONLY = 2003395700,
}

/**
 * Auto sizing reference points for text.
 */
declare enum AutoSizingReferenceEnum {
  /**
   * Center point on the botom edge of bounding box
   */
  BOTTOM_CENTER_POINT = 1651467109,

  /**
   * Bottom left point of bounding box
   */
  BOTTOM_LEFT_POINT = 1651469413,

  /**
   * Bottom right point of bounding box
   */
  BOTTOM_RIGHT_POINT = 1651470953,

  /**
   * Center point of bounding box
   */
  CENTER_POINT = 1668183154,

  /**
   * Center point on the left edge of bounding box
   */
  LEFT_CENTER_POINT = 1818583909,

  /**
   * Center point on the right edge of bounding box
   */
  RIGHT_CENTER_POINT = 1919509349,

  /**
   * Center point on the top edge of bounding box
   */
  TOP_CENTER_POINT = 1953456997,

  /**
   * Top left point of bounding box
   */
  TOP_LEFT_POINT = 1953459301,

  /**
   * Top right point of bounding box
   */
  TOP_RIGHT_POINT = 1953460841,
}

/**
 * Type of Paragraph span.
 */
declare enum SpanColumnTypeOptions {
  /**
   * Paragraph is a single column
   */
  SINGLE_COLUMN = 1163092844,

  /**
   * Paragraph spans the columns
   */
  SPAN_COLUMNS = 1936745326,

  /**
   * Paragraph splits the columns
   */
  SPLIT_COLUMNS = 1937007470,
}

/**
 * Span Column Count Options.
 */
declare enum SpanColumnCountOptions {
  /**
   * Paragraph spans all columns
   */
  ALL = 1634495520,
}

/**
 * Options for specifying basis of the width of the paragraph shading.
 */
declare enum ParagraphShadingWidthEnum {
  /**
   * Makes the paragraph shading based on width of the column.
   */
  COLUMN_WIDTH = 1265399652,

  /**
   * Makes the paragraph shading based on width of lines of text in the paragraph.
   */
  TEXT_WIDTH = 1886681207,
}

/**
 * Line justification options.
 */
declare enum LineAlignment {
  /**
   * Center aligns the text.
   */
  CENTER_LINE_ALIGN = 1818321774,

  /**
   * Justifies horizontal text on both the right and left and center aligns the last line or justifies vertical text on both the top and bottom and center aligns the last line.
   */
  CENTER_LINE_JUSTIFY = 1818455658,

  /**
   * Justifies horizontal text on both the right and left or justifies vertical text on both the top and bottom and gives all lines a uniform length.
   */
  FULL_LINE_JUSTIFY = 1818651754,

  /**
   * Left aligns horizontal text or top aligns vertical text.
   */
  LEFT_OR_TOP_LINE_ALIGN = 1818324084,

  /**
   * Justifies horizontal text on both the right and left and left aligns the last line or justifies vertical text on both the top and bottom and top aligns the last line.
   */
  LEFT_OR_TOP_LINE_JUSTIFY = 1819047018,

  /**
   * Right aligns horizontal text or bottom aligns vertical text.
   */
  RIGHT_OR_BOTTOM_LINE_ALIGN = 1818325602,

  /**
   * Justifies horizontal text on both the right and left and right aligns the last line or justifies vertical text on both the top and bottom and bottom aligns the last line.
   */
  RIGHT_OR_BOTTOM_LINE_JUSTIFY = 1819435626,
}

/**
 * Grid view options.
 */
declare enum GridViewSettings {
  /**
   * Align view.
   */
  ALIGN_VIEW_ENUM = 1783062902,

  /**
   * Grid and ZN view.
   */
  GRID_AND_ZN_VIEW_ENUM = 1783064442,

  /**
   * Grid view.
   */
  GRID_VIEW_ENUM = 1783064438,

  /**
   * ZN view.
   */
  ZN_VIEW_ENUM = 1783069302,
}

/**
 * Character count location options.
 */
declare enum CharacterCountLocation {
  /**
   * Displays the character count at the bottom of the frame.
   */
  BOTTOM_ALIGN = 1651471469,

  /**
   * Displays the character count on the left side of the frame.
   */
  LEFT_ALIGN = 1818584692,

  /**
   * Hides the character count.
   */
  NONE = 1852796517,

  /**
   * Displays the character count on the right side of the frame.
   */
  RIGHT_ALIGN = 1919379572,

  /**
   * Displays the character count at the top of the frame.
   */
  TOP_ALIGN = 1953460256,
}

/**
 * InCopy UI colors.
 */
declare enum InCopyUIColors {
  /**
   * Amber.
   */
  AMBER = 1765895522,

  /**
   * Aqua.
   */
  AQUA = 1765896545,

  /**
   * Black.
   */
  BLACK = 1765960811,

  /**
   * Blue.
   */
  BLUE = 1765960821,

  /**
   * Blueberry.
   */
  BLUEBERRY = 1765960802,

  /**
   * Brick red.
   */
  BRICK_RED = 1765962340,

  /**
   * Brown.
   */
  BROWN = 1765962350,

  /**
   * Burgundy.
   */
  BURGUNDY = 1765962343,

  /**
   * Canary.
   */
  CANARY = 1766026873,

  /**
   * Carnation.
   */
  CARNATION = 1766023538,

  /**
   * Charcoal.
   */
  CHARCOAL = 1766025324,

  /**
   * Cirrus.
   */
  CIRRUS = 1766025586,

  /**
   * Cornstarch.
   */
  CORNSTARCH = 1766027886,

  /**
   * Cute teal.
   */
  CUTE_TEAL = 1766028396,

  /**
   * Cyan.
   */
  CYAN = 1766029678,

  /**
   * Dark blue.
   */
  DARK_BLUE = 1766089324,

  /**
   * Dark green.
   */
  DARK_GREEN = 1766090610,

  /**
   * Eggplant.
   */
  EGGPLANT = 1766156135,

  /**
   * Electrolyte.
   */
  ELECTROLYTE = 1766157411,

  /**
   * Ether.
   */
  ETHER = 1766159464,

  /**
   * Fiesta.
   */
  FIESTA = 1766222181,

  /**
   * Forest.
   */
  FOREST = 1766224756,

  /**
   * Fuchsia.
   */
  FUCHSIA = 1766225267,

  /**
   * Gold.
   */
  GOLD = 1766288484,

  /**
   * Grape.
   */
  GRAPE = 1766290032,

  /**
   * Graphite.
   */
  GRAPHITE = 1766289512,

  /**
   * Grass green.
   */
  GRASS_GREEN = 1766287218,

  /**
   * Gray.
   */
  GRAY = 1766290041,

  /**
   * Green.
   */
  GREEN = 1766290030,

  /**
   * Grid blue.
   */
  GRID_BLUE = 1766285932,

  /**
   * Grid green.
   */
  GRID_GREEN = 1766286439,

  /**
   * Grid orange.
   */
  GRID_ORANGE = 1766289266,

  /**
   * Gunmetal.
   */
  GUNMETAL = 1766290798,

  /**
   * Iris.
   */
  IRIS = 1766421107,

  /**
   * Jade.
   */
  JADE = 1766482276,

  /**
   * Lavender.
   */
  LAVENDER = 1766618734,

  /**
   * Lemon.
   */
  LEMON = 1766616430,

  /**
   * Lichen.
   */
  LICHEN = 1766615395,

  /**
   * Light blue.
   */
  LIGHT_BLUE = 1766613612,

  /**
   * Light gray.
   */
  LIGHT_GRAY = 1766614898,

  /**
   * Light olive.
   */
  LIGHT_OLIVE = 1766616940,

  /**
   * Lime.
   */
  LIME = 1766615405,

  /**
   * Lipstick.
   */
  LIPSTICK = 1766615408,

  /**
   * Magenta.
   */
  MAGENTA = 1766680430,

  /**
   * Midnight.
   */
  MIDNIGHT = 1766680932,

  /**
   * Mocha.
   */
  MOCHA = 1766682467,

  /**
   * Mustard.
   */
  MUSTARD = 1766684019,

  /**
   * Ochre.
   */
  OCHRE = 1766810482,

  /**
   * Olive green.
   */
  OLIVE_GREEN = 1766812790,

  /**
   * Orange.
   */
  ORANGE = 1766814318,

  /**
   * Peach.
   */
  PEACH = 1766876008,

  /**
   * Pink.
   */
  PINK = 1766878827,

  /**
   * Powder.
   */
  POWDER = 1766879095,

  /**
   * Purple.
   */
  PURPLE = 1766879856,

  /**
   * Red.
   */
  RED = 1767007588,

  /**
   * Slate.
   */
  SLATE = 1767074932,

  /**
   * Smoke.
   */
  SMOKE = 1767075179,

  /**
   * Sulphur.
   */
  SULPHUR = 1767077228,

  /**
   * Tan.
   */
  TAN = 1767137646,

  /**
   * Teal.
   */
  TEAL = 1767138668,

  /**
   * Ultramarine.
   */
  ULTRAMARINE = 1767205997,

  /**
   * Violet.
   */
  VIOLET = 1767271540,

  /**
   * Wheat.
   */
  WHEAT = 1767336037,

  /**
   * White.
   */
  WHITE = 1767336052,

  /**
   * Yellow.
   */
  YELLOW = 1767468151,
}

/**
 * Cursor types.
 */
declare enum CursorTypes {
  /**
   * Uses a barbell cursor.
   */
  BARBELL_CURSOR = 1698841196,

  /**
   * Uses a block cursor.
   */
  BLOCK_CURSOR = 1698851951,

  /**
   * Uses a standard cursor.
   */
  STANDARD_CURSOR = 1699968100,

  /**
   * Uses a thick cursor.
   */
  THICK_CURSOR = 1700029291,
}

/**
 * The anti-aliasing type.
 */
declare enum AntiAliasType {
  /**
   * Color anti-aliasing.
   */
  COLOR_ANTIALIASING = 1665418322,

  /**
   * Gray anti-aliasing.
   */
  GRAY_ANTIALIASING = 1732527186,

  /**
   * Thicker anti-aliasing.
   */
  THICKER_ANTIALIASING = 1950444659,
}

/**
 * Line spacing type.
 */
declare enum LineSpacingType {
  /**
   * Double space
   */
  DOUBLE_SPACE = 1682068336,

  /**
   * One and half space
   */
  ONE_AND_HALF_SPACE = 1866549363,

  /**
   * Single space
   */
  SINGLE_SPACE = 1936282480,

  /**
   * Triple space
   */
  TRIPLE_SPACE = 1951552368,
}

/**
 * Page numbering options for book content objects within the book.
 */
declare enum RepaginateOption {
  /**
   * Starts page numbers for each book content object at the next even-numbered page after the last page of the previous book content object.
   */
  NEXT_EVEN_PAGE = 1164993131,

  /**
   * Starts page numbers for each book content object at the next odd-numbered page after the last page of the previous book content object.
   */
  NEXT_ODD_PAGE = 1332765291,

  /**
   * Continues page numbers sequentially from the previous book content object.
   */
  NEXT_PAGE = 1885500011,
}

/**
 * Book content file status options.
 */
declare enum BookContentStatus {
  /**
   * The book content object is being used by someone else and is therefore locked.
   */
  DOCUMENT_IN_USE = 1148150601,

  /**
   * The book content object is open.
   */
  DOCUMENT_IS_OPEN = 1148150607,

  /**
   * The book content object has been modified after repagination.
   */
  DOCUMENT_OUT_OF_DATE = 1148150596,

  /**
   * The book content object is missing because it has been moved, renamed, or deleted.
   */
  MISSING_DOCUMENT = 1148150605,

  /**
   * The book content object is not open and is unchanged.
   */
  NORMAL = 1852797549,
}

/**
 * Options for matching names when synchronizing styles in a book.
 */
declare enum SmartMatchOptions {
  /**
   * Match only the style name while finding styles in target.
   */
  MATCH_STYLE_NAME = 1936552814,

  /**
   * Match the full path of style while finding styles in target.
   */
  MATCH_STYLE_PATH = 1936549488,
}

/**
 * Specify the type of cell, either text or graphic.
 */
declare enum CellTypeEnum {
  /**
   * Graphic or Page item cell.
   */
  GRAPHIC_TYPE_CELL = 1701728329,

  /**
   * Text cell.
   */
  TEXT_TYPE_CELL = 1701730388,
}

/**
 * The row type.
 */
declare enum RowTypes {
  /**
   * Makes the row a body row.
   */
  BODY_ROW = 1161982583,

  /**
   * Makes the row a footer row. Note: When setting row type as footer row, the row must be either the bottom row in the table or adjacent to an existing footer row.
   */
  FOOTER_ROW = 1162244727,

  /**
   * Makes the row a header row. Note: When setting row type as header row, the row must be either the top row in the table or adjacent to an existing header row.
   */
  HEADER_ROW = 1162375799,

  /**
   * (Read-only) The column's rows are of multiple types.
   */
  MIXED_STATE = 1162703479,
}

/**
 * Placement options for header or footer rows.
 */
declare enum HeaderFooterBreakTypes {
  /**
   * Places headers or footers in each text column.
   */
  IN_ALL_TEXT_COLUMNS = 1231115363,

  /**
   * Places one instance of headers or footers per page.
   */
  ONCE_PER_PAGE = 1332760673,

  /**
   * Repeats headers or footers in each text frame.
   */
  ONCE_PER_TEXT_FRAME = 1332761702,
}

/**
 * Pattern options for alternating fills.
 */
declare enum AlternatingFillsTypes {
  /**
   * Alternates column fills.
   */
  ALTERNATING_COLUMNS = 1097614188,

  /**
   * Alternates row fills.
   */
  ALTERNATING_ROWS = 1097618039,

  /**
   * No alternating pattern.
   */
  NO_ALTERNATING_PATTERN = 1097617007,
}

/**
 * Options for displaying row and column strokes at corners.
 */
declare enum StrokeOrderTypes {
  /**
   * Places row strokes in front of column strokes when row and column strokes are different colors; joins striped strokes and connects crossing points.
   */
  BEST_JOINS = 1935828330,

  /**
   * Places column strokes in front of row strokes.
   */
  COLUMN_ON_TOP = 1935896436,

  /**
   * Places row strokes in front when row and column strokes are different colors; joins striped strokes only at points where strokes cross in a T-shape.
   */
  INDESIGN_2_COMPATIBILITY = 1936286819,

  /**
   * Places row strokes in front of column strokes.
   */
  ROW_ON_TOP = 1936879476,
}

/**
 * Values to specify the order table cells will display in story and galley views.
 */
declare enum DisplayOrderOptions {
  /**
   * Order by columns.
   */
  ORDER_BY_COLUMNS = 1652114254,

  /**
   * Order by rows.
   */
  ORDER_BY_ROWS = 1652118103,
}

/**
 * Rasterization options.
 */
declare enum FlattenerLevel {
  /**
   * Keeps as much artwork as possible vector data.
   */
  HIGH = 1701726313,

  /**
   * Rasterizes all artwork.
   */
  LOW = 1701727351,

  /**
   * Rasterizes a medium amount of artwork.
   */
  MEDIUM = 1701727588,

  /**
   * Rasterizes more than a medium amount of artwork.
   */
  MEDIUM_HIGH = 1718373704,

  /**
   * Rasterizes almost all artwork.
   */
  MEDIUM_LOW = 1718373708,
}

/**
 * Transparency blending space options.
 */
declare enum BlendingSpace {
  /**
   * Uses the CMYK profile.
   */
  CMYK = 1129142603,

  /**
   * Defaults to the current color profile.
   */
  DEFAULT_VALUE = 1147563124,

  /**
   * Uses the RGB color profile.
   */
  RGB = 1666336578,
}

/**
 * Blend mode options.
 */
declare enum BlendMode {
  /**
   * Creates a color with the luminance of the base color and the hue and saturation of the blend color. Note: Preserves gray levels and is useful for coloring monochrome images or tinting color images. Creates the inverse effect of the luminosity blend mode.
   */
  COLOR = 1668246642,

  /**
   * Darkens the base color to reflect the blend color. Note: Blending with white produces no change.
   */
  COLOR_BURN = 2020625768,

  /**
   * Brightens the base color to reflect the blend color. Note: Blending with pure black produces no change.
   */
  COLOR_DODGE = 2020625767,

  /**
   * Selects the darker of the base or blend colors as the resulting color; replaces areas lighter than the blend color but does not change areas darker than the blend color.
   */
  DARKEN = 2020625769,

  /**
   * Subtracts either the blend color from the base color or vice versa, depending on which has the greater brightness value. Note: Blending with white inverts the base color values; blending with black produces no change.
   */
  DIFFERENCE = 2020625771,

  /**
   * Creates an effect similar to--but lower in contrast than--the difference blend mode. Note: Blending with white inverts the base color values; blending with black produces no change
   */
  EXCLUSION = 2020625772,

  /**
   * For blend colors lighter than 50% gray, lightens the artwork as if it were screened; for blend colors darker than 50% gray, darkens the artwork as if it were multiplied. Note: Painting with pure black or white results in pure black or white.
   */
  HARD_LIGHT = 2020625766,

  /**
   * Creates a color with the luminance and saturation of the base color and the hue of the blend color.
   */
  HUE = 2020625773,

  /**
   * Selects the lighter of the base or blend colors as the resulting color; replaces areas darker than the blend color but does not change areas lighter than the blend color
   */
  LIGHTEN = 2020625770,

  /**
   * Creates a color with the hue and saturation of the base color and the luminance of the blend color. Note: Creates the inverse effect of the color blend mode.
   */
  LUMINOSITY = 2020625776,

  /**
   * Multiplies the base color by the blend color, resulting in a darker color. Note: Multiplying with black produces black; multiplying with white leaves the color unchanged.
   */
  MULTIPLY = 2020625762,

  /**
   * Colors the object with the blend color, without interaction with the base color.
   */
  NORMAL = 1852797549,

  /**
   * Multiplies or screens the colors, depending on the base color; patterns or colors overlay the existing artwork, preserving base color highlights and shadows while mixing in the blend color to reflect the lightness or darkness of the original color.
   */
  OVERLAY = 2020625764,

  /**
   * Creates a color with the luminance and hue of the base color and the saturation of the blend color. Note: Does not change areas with no saturation (0% gray).
   */
  SATURATION = 1380545377,

  /**
   * Multiplies the inverse of the blend and base colors, resulting in a lighter color. Note: Screening with white produces white; screening with black leaves the color unchanged.
   */
  SCREEN = 2020625763,

  /**
   * For blend colors lighter than 50% gray, lightens the artwork as if it were dodged; for blend colors darker than 50% gray, darkens the artwork as if it were burned. Note: Painting with pure black or white produces a distinctly darker or lighter area, but does not result in pure black or white.
   */
  SOFT_LIGHT = 2020625765,
}

/**
 * Shadow mode options.
 */
declare enum ShadowMode {
  /**
   * Uses a standard blurred drop shadow.
   */
  DROP = 2020623440,

  /**
   * Does not use a shadow.
   */
  NONE = 1852796517,
}

/**
 * On/off options for feathering.
 */
declare enum FeatherMode {
  /**
   * Does not use feathering.
   */
  NONE = 1852796517,

  /**
   * Uses standard feathering.
   */
  STANDARD = 2020623970,
}

/**
 * Corner type options.
 */
declare enum FeatherCornerType {
  /**
   * The edges of the object fade from opaque to transparent.
   */
  DIFFUSION = 2020623203,

  /**
   * The corners are rounded by the feather radius.
   */
  ROUNDED = 2020623202,

  /**
   * The gradient exactly follows the outer edge of the object, including sharp corners.
   */
  SHARP = 2020623201,
}

/**
 * Flattener level override options.
 */
declare enum SpreadFlattenerLevel {
  /**
   * Uses the specified custom flattening level.
   */
  CUSTOM = 1131639917,

  /**
   * Uses the default level.
   */
  DEFAULT_VALUE = 1147563124,

  /**
   * Turns off flattening.
   */
  NONE = 1852796517,
}

/**
 * Glow technique options.
 */
declare enum GlowTechnique {
  /**
   * Precise.
   */
  PRECISE = 2020618338,

  /**
   * Softer.
   */
  SOFTER = 2020618337,
}

/**
 * Inner glow source options.
 */
declare enum InnerGlowSource {
  /**
   * The glow radiates from the object's center.
   */
  CENTER_SOURCED = 2020618593,

  /**
   * The glow radiates from the edge of the object.
   */
  EDGE_SOURCED = 2020618594,
}

/**
 * Bevel and emboss style options.
 */
declare enum BevelAndEmbossStyle {
  /**
   * An emboss effect is applied to the object.
   */
  EMBOSS = 2020618851,

  /**
   * The inside edges of the object are beveled.
   */
  INNER_BEVEL = 2020618850,

  /**
   * The outside edges of the object are beveled.
   */
  OUTER_BEVEL = 2020618849,

  /**
   * An emboss effect is applied to the edges of the object.
   */
  PILLOW_EMBOSS = 2020618852,
}

/**
 * Bevel and emboss technique options.
 */
declare enum BevelAndEmbossTechnique {
  /**
   * Emboss and bevel contours are chiseled and have hard corners.
   */
  CHISEL_HARD = 2020619106,

  /**
   * Emboss or bevel contours chiseled but softened somewhat.
   */
  CHISEL_SOFT = 2020619107,

  /**
   * Emboss and bevel contours are smooth.
   */
  SMOOTH_CONTOUR = 2020619105,
}

/**
 * Bevel and emboss direction options.
 */
declare enum BevelAndEmbossDirection {
  /**
   * The effect appears depressed.
   */
  DOWN = 1181971556,

  /**
   * The effect appears raised.
   */
  UP = 1181971566,
}

/**
 * Follow-shape options for directional feathering.
 */
declare enum FollowShapeModeOptions {
  /**
   * Feathers all edges that face the specified angle.
   */
  ALL_EDGES = 1701721442,

  /**
   * Feathers only the leading edge facing the specified angle.
   */
  LEADING_EDGE = 1701721441,

  /**
   * Disables shape following and uses the rectangular bounds of the object.
   */
  NONE = 1852796517,
}

/**
 * Page number position options.
 */
declare enum PageNumberPosition {
  /**
   * Places page numbers after entry text.
   */
  AFTER_ENTRY = 1634100590,

  /**
   * Places page numbers before entry text.
   */
  BEFORE_ENTRY = 1650877806,

  /**
   * Turns off page numbers.
   */
  NONE = 1852796517,
}

/**
 * Numbered paragraphs options.
 */
declare enum NumberedParagraphsOptions {
  /**
   * Excludes paragraph numbers.
   */
  EXCLUDE_NUMBERS = 1952804469,

  /**
   * Includes the full paragraph text.
   */
  INCLUDE_FULL_PARAGRAPH = 1953064560,

  /**
   * Includes only the paragraph number.
   */
  INCLUDE_NUMBERS_ONLY = 1953066607,
}

/**
 * Character set options for importing text files.
 */
declare enum TextImportCharacterSet {
  /**
   * The ANSI character set.
   */
  ANSI = 1095652169,

  /**
   * The Arabic ASMO character set.
   */
  ARABIC_ASMO = 1415672685,

  /**
   * The Arabic ASMO transparent character set.
   */
  ARABIC_ASMO_TRANSPARENT = 1415672692,

  /**
   * The Central European (ISO) character set.
   */
  CENTRALEUROPEAN_ISO = 1416184645,

  /**
   * The Chinese Big 5 character set.
   */
  CHINESE_BIG_5 = 1415799349,

  /**
   * The Cyrillic (CP855) character set.
   */
  CYRILLIC_CP855 = 1415788597,

  /**
   * The Cyrillic (ISO) character set.
   */
  CYRILLIC_ISO = 1416184697,

  /**
   * The Cyrillic (KOI8R) character set.
   */
  CYRILLIC_KOI8R = 1416312946,

  /**
   * The Cyrillic (KOI8U) character set.
   */
  CYRILLIC_KOI8U = 1416312949,

  /**
   * The DOS Latin 2 character set.
   */
  DOS_LATIN_2 = 1415867442,

  /**
   * The GB18030 character set.
   */
  GB18030 = 1416061491,

  /**
   * The GB2312 character set.
   */
  GB2312 = 1416061535,

  /**
   * The Greek (ISO) character set.
   */
  GREEK_ISO = 1416185707,

  /**
   * The KSC5601 character set.
   */
  KSC5601 = 1414230883,

  /**
   * The Macintosh Arabic character set.
   */
  MACINTOSH_ARABIC = 1416446322,

  /**
   * The Macintosh CE (Cantral European) character set.
   */
  MACINTOSH_CE = 1416446789,

  /**
   * The Macintosh Croatian character set.
   */
  MACINTOSH_CROATIAN = 1416446834,

  /**
   * The Macintosh Cyrillic character set.
   */
  MACINTOSH_CYRILLIC = 1416446841,

  /**
   * The Macintosh Greek character set.
   */
  MACINTOSH_GREEK = 1416447858,

  /**
   * The Macintosh Greek (Shared caps) character set.
   */
  MACINTOSH_GREEK_SHARED_CAPS = 1416447794,

  /**
   * The Macintosh Hebrew character set.
   */
  MACINTOSH_HEBREW = 1416448098,

  /**
   * The Macintosh Icelandic character set.
   */
  MACINTOSH_ICELANDIC = 1416448355,

  /**
   * The Macintosh Roman character set.
   */
  MACINTOSH_ROMAN = 1416450669,

  /**
   * The Macintosh Romanian character set.
   */
  MACINTOSH_ROMANIAN = 1416450671,

  /**
   * The Macintosh Turkish character set.
   */
  MACINTOSH_TURKISH = 1416451186,

  /**
   * The Recommend:Shift_JIS 83pv character set.
   */
  RECOMMENDSHIFTJIS83PV = 1412969328,

  /**
   * The Shift_JIS 90ms character set.
   */
  SHIFTJIS90MS = 1413034093,

  /**
   * The Shift_JIS 90pv character set.
   */
  SHIFTJIS90PV = 1413034096,

  /**
   * The Turkish (ISO) character set.
   */
  TURKISH_ISO = 1416189045,

  /**
   * The Unicode UTF16 character set.
   */
  UTF16 = 1937125686,

  /**
   * The Unicode UTF8 character set.
   */
  UTF8 = 1937134904,

  /**
   * The Windows Arabic character set.
   */
  WINDOWS_ARABIC = 1417101682,

  /**
   * Tthe Windows Baltic character set.
   */
  WINDOWS_BALTIC = 1417101940,

  /**
   * The Windows CE (Central European) character set.
   */
  WINDOWS_CE = 1417102149,

  /**
   * The Windows Cyrillic character set.
   */
  WINDOWS_CYRILLIC = 1417102201,

  /**
   * The Windows EE (Eastern European) character set.
   */
  WINDOWS_EE = 1417102661,

  /**
   * The Windows Greek character set.
   */
  WINDOWS_GREEK = 1417103218,

  /**
   * The Windows Hebrew character set.
   */
  WINDOWS_HEBREW = 1417103458,

  /**
   * The Windows Turkish character set.
   */
  WINDOWS_TURKISH = 1417106549,
}

/**
 * Import platform options.
 */
declare enum ImportPlatform {
  /**
   * Macintosh.
   */
  MACINTOSH = 1296130931,

  /**
   * Windows.
   */
  PC = 1466852474,
}

/**
 * Options for converting manual page breaks.
 */
declare enum ConvertPageBreaks {
  /**
   * Converts manual page breaks to column breaks.
   */
  COLUMN_BREAK = 1396927554,

  /**
   * Does not preserve page breaks; allows text to flow.
   */
  NONE = 1852796517,

  /**
   * Preserves page breaks.
   */
  PAGE_BREAK = 1397778242,
}

/**
 * Character set options exporting tagged text files.
 */
declare enum TagTextExportCharacterSet {
  /**
   * ANSI.
   */
  ANSI = 1095652169,

  /**
   * ASCII.
   */
  ASCII = 1095975753,

  /**
   * The Central European (ISO) character set.
   */
  CENTRALEUROPEAN_ISO = 1416184645,

  /**
   * Chinese Big 5
   */
  CHINESE_BIG_5 = 1415799349,

  /**
   * The Cyrillic (ISO) character set.
   */
  CYRILLIC_ISO = 1416184697,

  /**
   * Uses GB18030 encoding.
   */
  GB18030 = 1416061491,

  /**
   * The Greek (ISO) character set.
   */
  GREEK_ISO = 1416185707,

  /**
   * Uses KSC5601 encoding.
   */
  KSC5601 = 1414230883,

  /**
   * Shift_JIS.
   */
  SHIFT_JIS = 1249077875,

  /**
   * Unicode.
   */
  UNICODE = 1249209961,

  /**
   * The Windows Arabic character set.
   */
  WINDOWS_ARABIC = 1417101682,

  /**
   * The Windows Hebrew character set.
   */
  WINDOWS_HEBREW = 1417103458,
}

/**
 * Tag form options.
 */
declare enum TagTextForm {
  /**
   * Abbreviates tags; creates smaller text files.
   */
  ABBREVIATED = 1414816098,

  /**
   * Displays tags in long form; creates larger text files.
   */
  VERBOSE = 1414821474,
}

/**
 * Options for resolving style conflicts when importing tagged text.
 */
declare enum StyleConflict {
  /**
   * Uses the publication style.
   */
  PUBLICATION_DEFINITION = 1414819940,

  /**
   * Uses the tag file style.
   */
  TAG_FILE_DEFINITION = 1413903460,
}

/**
 * Character set options for exported text files.
 */
declare enum TextExportCharacterSet {
  /**
   * The default character set for the platform.
   */
  DEFAULT_PLATFORM = 1415865972,

  /**
   * The Unicode UTF16 character set.
   */
  UTF16 = 1937125686,

  /**
   * The Unicode UTF8 character set.
   */
  UTF8 = 1937134904,
}

/**
 * Options for converting tables.
 */
declare enum ConvertTablesOptions {
  /**
   * Convert tables to unformatted, tab-delimited text.
   */
  UNFORMATTED_TABBED_TEXT = 1398101076,

  /**
   * Converts tables to basic, unformatted tables.
   */
  UNFORMATTED_TABLE = 1396921684,
}

/**
 * Formatting options for imported spreadsheets.
 */
declare enum TableFormattingOptions {
  /**
   * Use formatting from the original spreadsheet.
   */
  EXCEL_FORMATTED_TABLE = 2020361812,

  /**
   * Converts the spreadsheet to a table that is formatted only on initial import but not on update.
   */
  EXCEL_FORMAT_ONLY_ONCE = 2017873748,

  /**
   * Convert the spreadsheet to unformatted, tab-delimited text.
   */
  EXCEL_UNFORMATTED_TABBED_TEXT = 2018858068,

  /**
   * Convert the spreadsheet to an unformatted table.
   */
  EXCEL_UNFORMATTED_TABLE = 2020365652,
}

/**
 * Alignment options.
 */
declare enum AlignmentStyleOptions {
  /**
   * Center aligns cells.
   */
  CENTER_ALIGN = 1667591796,

  /**
   * Left aligns cells.
   */
  LEFT_ALIGN = 1818584692,

  /**
   * Right aligns cells.
   */
  RIGHT_ALIGN = 1919379572,

  /**
   * Preserves the spreadsheet's alignment.
   */
  SPREADSHEET = 1936749171,
}

/**
 * Options for resolving clashes that result from matching style names.
 */
declare enum ResolveStyleClash {
  /**
   * Automatically renames the new style.
   */
  RESOLVE_CLASH_AUTO_RENAME = 2001879873,

  /**
   * Uses the existing style.
   */
  RESOLVE_CLASH_USE_EXISTING = 2001879877,

  /**
   * Uses the new style.
   */
  RESOLVE_CLASH_USE_NEW = 2001879886,
}

/**
 * The container type.
 */
declare enum ContainerType {
  /**
   * The container contains alternative values of which only one can be used.
   */
  ALT = 1298424180,

  /**
   * The container contains unordered items.
   */
  BAG = 1298424423,

  /**
   * The container contains ordered or sequential items.
   */
  SEQ = 1298428785,
}

/**
 * The copyright status of the document.
 */
declare enum CopyrightStatus {
  /**
   * The document is in the public domain.
   */
  NO = 1852776480,

  /**
   * The copyright status is unknown.
   */
  UNKNOWN = 1433299822,

  /**
   * The document is copyrighted.
   */
  YES = 2036691744,
}

/**
 * Index entry capitalization options.
 */
declare enum IndexCapitalizationOptions {
  /**
   * Capitalizes all index entries. Note: Capitalizes only topics created before the capitalization statement appears in the script.
   */
  ALL_ENTRIES = 1097624645,

  /**
   * Capitalizes all level 1 entries. Note: Capitalizes only topics created before the capitalization statement appears in the script.
   */
  ALL_LEVEL_1_ENTRIES = 1095517556,

  /**
   * Capitalizes the specified topic and its nested topics. Valid only as parameter of the topic capitalize method; cannot be used as a parameter of the index capitalize method. Note: Must occur after the selected topic and its nested subtopics are created.
   */
  INCLUDE_SUBENTRIES = 1767072325,

  /**
   * Capitalizes the specified topic but does not capitalize its nested topics. Valid only as parameter of the topic capitalize method; cannot be used as a parameter of the index capitalize method. Note: Must occur after the specified topic and its nested topics are created.
   */
  SELECTED_ENTRY = 1398042725,
}

/**
 * Options for formatting level 2 and lower index topics.
 */
declare enum IndexFormat {
  /**
   * Places nested topics on the line below and indented from the parent topic.
   */
  NESTED_FORMAT = 1316243814,

  /**
   * Places nested topics on the same line as their parent topic, separated by the specified separator.
   */
  RUNIN_FORMAT = 1382631782,
}

/**
 * Instructional text options for cross reference.
 */
declare enum CrossReferenceType {
  /**
   * Inserts the specified string in front of the referenced topic.
   */
  CUSTOM_CROSS_REFERENCE = 1131639875,

  /**
   * Inserts the specified following topic separator and the specified string after the referenced topic. If no following topic separator is specified, inserts a space.
   */
  CUSTOM_CROSS_REFERENCE_AFTER = 1131639905,

  /**
   * Inserts the specified string and the specified before cross reference separator in front of the referenced topic. If no before cross reference separator is specified, inserts a space.
   */
  CUSTOM_CROSS_REFERENCE_BEFORE = 1131639906,

  /**
   * Inserts 'See' in front of the referenced topic.
   */
  SEE = 1701729125,

  /**
   * Inserts 'See also' in front of the referenced topic.
   */
  SEE_ALSO = 1399144812,

  /**
   * Inserts 'See also herein' in front of the referenced topic.
   */
  SEE_ALSO_HEREIN = 1398884466,

  /**
   * Inserts 'See herein' in front of the referenced topic.
   */
  SEE_HEREIN = 1397256814,

  /**
   * Inserts 'See also' in front of the referenced topic if the topic has an associated page reference; inserts 'See' if the topic does not have a page reference.
   */
  SEE_OR_ALSO_BRACKET = 1399800172,
}

/**
 * Options for index page references.
 */
declare enum PageReferenceType {
  /**
   * The page on which the index entry is located.
   */
  CURRENT_PAGE = 1668444263,

  /**
   * The range of pages from the page containing the inedex entry to the nth page after that page (where n is the number of pages to include).
   */
  FOR_NEXT_N_PAGES = 1718513767,

  /**
   * The range of pages from the page containing the index entry to the page containing the nth full paragraph from the paragraph containing the index entry (where n is the number of paragraphs to include).
   */
  FOR_NEXT_N_PARAGRAPHS = 1718513778,

  /**
   * Turns off page numbers for the index topic.
   */
  SUPPRESS_PAGE_NUMBERS = 1852863079,

  /**
   * The last page in the document.
   */
  TO_END_OF_DOCUMENT = 1701799011,

  /**
   * The last page in the numbered section containing the index entry.
   */
  TO_END_OF_SECTION = 1701802851,

  /**
   * The last page in the story containing the index entry.
   */
  TO_END_OF_STORY = 1701802868,

  /**
   * The range of pages from the page containing the index entry to the page containing the next paragraph style change.
   */
  TO_NEXT_STYLE_CHANGE = 1953395555,

  /**
   * The range of pages from the page containing the index entry to the page that contains the next occurrance of the specified paragraph style. If no paragraph style is specified, the paragraph style of the index entry paragraph is used.
   */
  TO_NEXT_USE_OF_STYLE = 1953396083,
}

/**
 * Indexing sort option header types
 */
declare enum HeaderTypes {
  /**
   * Basic Latin
   */
  BASIC_LATIN = 1213481548,

  /**
   * Belarusian
   */
  BELARUSIAN = 1213481573,

  /**
   * Bulgarian
   */
  BULGARIAN = 1213481589,

  /**
   * Chinese Pinyin
   */
  CHINESE_PINYIN = 1213481808,

  /**
   * Chinese Stroke Count
   */
  CHINESE_STROKE_COUNT = 1213481811,

  /**
   * Croatian
   */
  CROATIAN = 1213481842,

  /**
   * Czech
   */
  CZECH = 1213481850,

  /**
   * Danish/Norwegian
   */
  DANISH_NORWEGIAN = 1213482062,

  /**
   * Estonian
   */
  ESTONIAN = 1213482355,

  /**
   * Finnish/Swedish
   */
  FINNISH_SWEDISH = 1213485894,

  /**
   * Hiragana: A, I, U, E, O...
   */
  HIRAGANA_ALL = 1213483073,

  /**
   * Hiragana: A, Ka, Sa, Ta, Na...
   */
  HIRAGANA_CONSONANTS_ONLY = 1213483075,

  /**
   * Hungarian
   */
  HUNGARIAN = 1213483125,

  /**
   * Katakana: A, I, U, E, O...
   */
  KATAKANA_ALL = 1213483841,

  /**
   * Katakana: A, Ka, Sa, Ta, Na...
   */
  KATAKANA_CONSONANTS_ONLY = 1213483843,

  /**
   * Korean Consonant
   */
  KOREAN_CONSONANT = 1213483887,

  /**
   * Korean Consonant Plus Vowel
   */
  KOREAN_CONSONANT_PLUS_VOWEL = 1213483862,

  /**
   * Latvian
   */
  LATVIAN = 1213484129,

  /**
   * Lithuanian
   */
  LITHUANIAN = 1213484137,

  /**
   * Polish
   */
  POLISH = 1213485167,

  /**
   * Romanian
   */
  ROMANIAN = 1213485679,

  /**
   * Russian
   */
  RUSSIAN = 1213485685,

  /**
   * Slovak
   */
  SLOVAK = 1213485931,

  /**
   * Slovenian
   */
  SLOVENIAN = 1213485934,

  /**
   * Spanish
   */
  SPANISH = 1213485936,

  /**
   * Turkish
   */
  TURKISH = 1213486197,

  /**
   * Ukrainian
   */
  UKRAINIAN = 1213486443,
}

/**
 * Options for specifying how tranparencies are displayed.
 */
declare enum TagTransparency {
  /**
   * Uses the default setting. For information, see display performance preferences.
   */
  DEFAULT_VALUE = 1147563124,

  /**
   * Displays higher-resolution (144 dpi) drop shadows and feathers, CMYK mattes, and spread isolation.
   */
  HIGH_QUALITY = 1346922866,

  /**
   * Displays basic transparency (opacity and blend modes) and shows transparency effects such as drop shadow and feathering in a low-resolution approximation. Note: Does not isolate page content from the background. Objects with blend modes other than Normal might appear different in other applications and output.
   */
  LOW_QUALITY = 1481666146,

  /**
   * Displays drop shadows and feathering in low resolution.
   */
  MEDIUM_QUALITY = 1481663597,

  /**
   * Turns off the on-screen display of transparency. Note: Does not turn off transparency when printing or exporting the file.
   */
  OFF = 1330005536,
}

/**
 * The display method for vector images.
 */
declare enum TagVector {
  /**
   * Uses the default setting. For information, see display performance preferences.
   */
  DEFAULT_VALUE = 1147563124,

  /**
   * Grays out the image.
   */
  GRAY_OUT = 1917284985,

  /**
   * Displays a high-resolution version of the image.
   */
  HIGH_RESOLUTION = 1917348177,

  /**
   * Displays a low-resolution proxy version of the image.
   */
  PROXY = 1917874808,
}

/**
 * The display method for raster images.
 */
declare enum TagRaster {
  /**
   * Uses the default setting. For information, see display performance preferences.
   */
  DEFAULT_VALUE = 1147563124,

  /**
   * Grays out raster images.
   */
  GRAY_OUT = 1917284985,

  /**
   * Displays a high-resolution version of the image.
   */
  HIGH_RESOLUTION = 1917348177,

  /**
   * Displays a low-resolution proxy image appropriate for identifying and positioning an image.
   */
  PROXY = 1917874808,
}

/**
 * Options for highlighting the hyperlink when selected.
 */
declare enum HyperlinkAppearanceHighlight {
  /**
   * Highlights the hyperlink border inset.
   */
  INSET = 1853056372,

  /**
   * Highlights the hyperlink fill color.
   */
  INVERT = 1853256308,

  /**
   * Does not highlight the hyperlink.
   */
  NONE = 1852796517,

  /**
   * Highlights the hyperlink border.
   */
  OUTLINE = 1869900910,
}

/**
 * Hyperlink border weight options.
 */
declare enum HyperlinkAppearanceWidth {
  /**
   * Uses a medium border.
   */
  MEDIUM = 1701727588,

  /**
   * Uses a thick border.
   */
  THICK = 1952999787,

  /**
   * Uses a thin border.
   */
  THIN = 1952999790,
}

/**
 * Hyperlink border style options.
 */
declare enum HyperlinkAppearanceStyle {
  /**
   * Uses a dashed stroke.
   */
  DASHED = 1684108136,

  /**
   * Uses a solid stroke.
   */
  SOLID = 1936682084,
}

/**
 * Hyperlink destination page display options.
 */
declare enum HyperlinkDestinationPageSetting {
  /**
   * Fits the destination page to the window height; may obscure the right side the page. Note: The magnification changes automatically when the window is resized vertically.
   */
  FIT_HEIGHT = 1212437352,

  /**
   * Displays the visible portion of the destination page as the destination.
   */
  FIT_VIEW = 1212437366,

  /**
   * Fits the the text area of the destination page to the window width; obscures page margins and may obscure the lower portion of the page. Note: The magnification changes automatically when the window is resized horizontally.
   */
  FIT_VISIBLE = 1212437334,

  /**
   * Fits the destination page to the width of the window; may obscure the lower portion of the page. Note: The magnification changes automatically when the window is resized horizontally.
   */
  FIT_WIDTH = 1212437335,

  /**
   * Fits the entire destination page in the document window. Note: The magnification changes automatically when the window is resized.
   */
  FIT_WINDOW = 1212437367,

  /**
   * Fits the destination page within the specified rectangle. For information on specifying the rectangle size and position, see the entry for view bounds.
   */
  FIXED = 1212437350,

  /**
   * The destination page is displayed at the same zoom percent as the previously displayed page. Note: The magnification changes automatically when the window is resized.
   */
  INHERIT_ZOOM = 1212437370,
}

/**
 * Cross reference building block types.
 */
declare enum BuildingBlockTypes {
  /**
   * Bookmark name building block type.
   */
  BOOKMARK_NAME_BUILDING_BLOCK = 1650614894,

  /**
   * Chapter number building block type.
   */
  CHAPTER_NUMBER_BUILDING_BLOCK = 1650615150,

  /**
   * Custom string building block type.
   */
  CUSTOM_STRING_BUILDING_BLOCK = 1650615155,

  /**
   * File name building block type.
   */
  FILE_NAME_BUILDING_BLOCK = 1650615918,

  /**
   * Full paragraph building block type.
   */
  FULL_PARAGRAPH_BUILDING_BLOCK = 1650615920,

  /**
   * Page number building block type.
   */
  PAGE_NUMBER_BUILDING_BLOCK = 1650618478,

  /**
   * Paragraph number building block type.
   */
  PARAGRAPH_NUMBER_BUILDING_BLOCK = 1651533678,

  /**
   * Paragraph text building block type.
   */
  PARAGRAPH_TEXT_BUILDING_BLOCK = 1650618484,
}

/**
 * Sort order for hyperlink ranges.
 */
declare enum RangeSortOrder {
  /**
   * Sort the ranges in ascending order.
   */
  ASCENDING_SORT = 1634952307,

  /**
   * Sort the ranges in descending order.
   */
  DESCENDING_SORT = 1685287796,

  /**
   * Do not sort the ranges.
   */
  NO_SORT = 1852797812,
}

/**
 * The flag indicating whether the rule is disabled, set for error, warning, or just informational.
 */
declare enum PreflightRuleFlag {
  /**
   * Treat as error if rule check failed.
   */
  RETURN_AS_ERROR = 1699890546,

  /**
   * Treat as information only if rule check failed.
   */
  RETURN_AS_INFORMATIONAL = 1699893865,

  /**
   * Treat as warning if rule check failed.
   */
  RETURN_AS_WARNING = 1699893879,

  /**
   * Rule is currently disabled.
   */
  RULE_IS_DISABLED = 1699890274,
}

/**
 * The type of data for this data object.
 */
declare enum RuleDataType {
  /**
   * The data type is a boolean.
   */
  BOOLEAN_DATA_TYPE = 1920221804,

  /**
   * The data type is an int32.
   */
  INTEGER_DATA_TYPE = 1920223598,

  /**
   * The data type is a list.
   */
  LIST_DATA_TYPE = 1920224372,

  /**
   * The data type is an object.
   */
  OBJECT_DATA_TYPE = 1920225122,

  /**
   * The data type is a real.
   */
  REAL_DATA_TYPE = 1920225900,

  /**
   * The data type is an int16.
   */
  SHORT_INTEGER_DATA_TYPE = 1920226153,

  /**
   * The data type is a string.
   */
  STRING_DATA_TYPE = 1920226162,
}

/**
 * Preflight scope options.
 */
declare enum PreflightScopeOptions {
  /**
   * Include all documents in the book preflight.
   */
  PREFLIGHT_ALL_DOCUMENTS = 1885552964,

  /**
   * Include all pages in the preflight.
   */
  PREFLIGHT_ALL_PAGES = 1885552976,

  /**
   * Include only selected document for book preflight.
   */
  PREFLIGHT_SELECTED_DOCUMENTS = 1885557572,
}

/**
 * Preflight layer options.
 */
declare enum PreflightLayerOptions {
  /**
   * Preflight all layers
   */
  PREFLIGHT_ALL_LAYERS = 1886142796,

  /**
   * Preflight visible layers
   */
  PREFLIGHT_VISIBLE_LAYERS = 1886148172,

  /**
   * Preflight visible and printable layers
   */
  PREFLIGHT_VISIBLE_PRINTABLE_LAYERS = 1886148176,
}

/**
 * Preflight Profile options.
 */
declare enum PreflightProfileOptions {
  /**
   * Preflight using the embedded profile.
   */
  USE_EMBEDDED_PROFILE = 1885619533,

  /**
   * Preflight using working profile.
   */
  USE_WORKING_PROFILE = 1885622342,
}

/**
 * PNG page export range options.
 */
declare enum PNGExportRangeEnum {
  /**
   * Exports all pages.
   */
  EXPORT_ALL = 1785742657,

  /**
   * Exports the page range specified in the page string property.
   */
  EXPORT_RANGE = 1785742674,
}

/**
 * Quality options for exported PNG images.
 */
declare enum PNGQualityEnum {
  /**
   * High quality.
   */
  HIGH = 1701726313,

  /**
   * Low quality.
   */
  LOW = 1701727351,

  /**
   * Maximum quality.
   */
  MAXIMUM = 1701727608,

  /**
   * Medium quality.
   */
  MEDIUM = 1701727588,
}

/**
 * Color space options for representing color in the exported PNG.
 */
declare enum PNGColorSpaceEnum {
  /**
   * Converts all color values to high-quality black-and-white images. Gray levels of the converted objects represent the luminosity of the original objects.
   */
  GRAY = 1766290041,

  /**
   * Represents all color values using the RGB color space. Best suited for documents that will be viewed on-screen.
   */
  RGB = 1666336578,
}

/**
 * Options for specifying the events (user actions) that change a button's state.
 */
declare enum StateTypes {
  /**
   * The mouse pointer is clicked on the button's area.
   */
  DOWN = 1181971556,

  /**
   * Down off state.
   */
  DOWN_OFF = 1181967462,

  /**
   * Down on state.
   */
  DOWN_ON = 1181967471,

  /**
   * The mouse pointer moves into the button's area.
   */
  ROLLOVER = 1181971574,

  /**
   * Rollover off state.
   */
  ROLLOVER_OFF = 1181972070,

  /**
   * Rollover on state.
   */
  ROLLOVER_ON = 1181972079,

  /**
   * The default appearance, used when there is no user activity on the button's area.
   */
  UP = 1181971566,

  /**
   * Up off state.
   */
  UP_OFF = 1181970022,

  /**
   * Up on state.
   */
  UP_ON = 1181970031,
}

/**
 * Options for specifying the position of a movie's display window.
 */
declare enum FloatingWindowPosition {
  /**
   * Positions the window in the center of the screen.
   */
  CENTER = 1298359662,

  /**
   * Positions the window on the left side of the screen midway between the top and bottom.
   */
  CENTER_LEFT = 1298361446,

  /**
   * Positions the window on the right side of the screen midway between the top and bottom.
   */
  CENTER_RIGHT = 1298362996,

  /**
   * Positions the window in the lower left corner of the screen.
   */
  LOWER_LEFT = 1298951270,

  /**
   * Positions the window at the bottom of the screen midway between the left and right edges.
   */
  LOWER_MIDDLE = 1298951524,

  /**
   * Positions the window in the lower right corner of the screen.
   */
  LOWER_RIGHT = 1298952820,

  /**
   * Positions the window in the upper left corner of the screen.
   */
  UPPER_LEFT = 1668183118,

  /**
   * Positions the window at the top of the screen midway between the left and right edges.
   */
  UPPER_MIDDLE = 1299541348,

  /**
   * Positions the window in the upper right corner of the screen.
   */
  UPPER_RIGHT = 1299542644,
}

/**
 * The size of the movie's floating window. Valid only when floating window is true.
 */
declare enum FloatingWindowSize {
  /**
   * The floating window is the movie's original display size.
   */
  FULL = 1298560364,

  /**
   * The floating window fills the entire screen.
   */
  MAX = 1299014008,

  /**
   * The floating window is one fifth the length and height of the movie's original display size.
   */
  ONE_FIFTH = 1298557286,

  /**
   * The floating window is one fourth the length and height of the movie's original display size.
   */
  ONE_FOURTH = 1298558834,

  /**
   * The floating window is one half the length and height of the movie's original display size.
   */
  ONE_HALF = 1298686316,

  /**
   * The floating window is quadruple the length and height of the movie's original display size.
   */
  QUADRUPLE = 1299281272,

  /**
   * The floating window is triple the length and height of the movie's original display size.
   */
  TRIPLE = 1299477112,

  /**
   * The floating window is twice the length and height of the movie's original display size.
   */
  TWICE = 1299476344,
}

/**
 * The type of graphic for the movie poster.
 */
declare enum MoviePosterTypes {
  /**
   * Uses an image from the movie file.
   */
  FROM_MOVIE = 1298558310,

  /**
   * None.
   */
  NONE = 1852796517,

  /**
   * (Read-only) Indicates whether the movie poster is not the standard, generic image.
   */
  PROXY_IMAGE = 1299216505,

  /**
   * Uses the generic movie poster image file.
   */
  STANDARD = 2020623970,
}

/**
 * Playback options.
 */
declare enum MoviePlayOperations {
  /**
   * Pauses playback.
   */
  PAUSE = 1885435251,

  /**
   * Starts playback.
   */
  PLAY = 1886151033,

  /**
   * Starts playback from the specified navigation point.
   */
  PLAY_FROM_NAVIGATION_POINT = 1886154358,

  /**
   * Resumes playback.
   */
  RESUME = 1919251317,

  /**
   * Stops playback.
   */
  STOP = 1937010544,

  /**
   * Stops all playback (SWF only).
   */
  STOP_ALL = 1937010785,
}

/**
 * Sound poster graphics options.
 */
declare enum SoundPosterTypes {
  /**
   * No sound poster.
   */
  NONE = 1852796517,

  /**
   * Proxy image sound poster.
   */
  PROXY_IMAGE = 1299216505,

  /**
   * Generic sound poster.
   */
  STANDARD = 2020623970,
}

/**
 * Behavior trigger event options.
 */
declare enum BehaviorEvents {
  /**
   * Triggers the behavior when the mouse button is clicked (without being released).
   */
  MOUSE_DOWN = 1835296118,

  /**
   * Triggers the behavior when the mouse pointer enters the area defined by the bounding box of the object.
   */
  MOUSE_ENTER = 1835361654,

  /**
   * Triggers the behavior when the mouse pointer exits the area defined by the bounding box of the object.
   */
  MOUSE_EXIT = 1836606838,

  /**
   * Triggers the behavior when the mouse is released after a click.
   */
  MOUSE_UP = 1836410230,

  /**
   * Triggers the behavior when the focus moves to a different interactive object.
   */
  ON_BLUR = 1868719478,

  /**
   * Triggers the behavior when the object receives focus, either through a mouse action or by pressing the Tab key.
   */
  ON_FOCUS = 1868981622,
}

/**
 * Zoom options for the goto destination page.
 */
declare enum GoToZoomOptions {
  /**
   * Displays the page at 100% magnification.
   */
  ACTUAL_SIZE = 2053206906,

  /**
   * Fits the text area of the page to the window width; obscures page margins and may obscure the lower portion of the page.
   */
  FIT_VISIBLE = 1212437334,

  /**
   * Fits the page to the width of the window; may obscure the lower portion of the page.
   */
  FIT_WIDTH = 1212437335,

  /**
   * Fits the page in the display window.
   */
  FIT_WINDOW = 1212437367,

  /**
   * Inherits the zoom setting from the previously displayed page.
   */
  INHERIT_ZOOM = 1212437370,
}

/**
 * Playback options.
 */
declare enum PlayOperations {
  /**
   * Pauses playback.
   */
  PAUSE = 1885435251,

  /**
   * Starts playback.
   */
  PLAY = 1886151033,

  /**
   * Resumes playback.
   */
  RESUME = 1919251317,

  /**
   * Stops playback.
   */
  STOP = 1937010544,

  /**
   * Stops all playback (SWF only).
   */
  STOP_ALL = 1937010785,
}

/**
 * Playback options.
 */
declare enum AnimationPlayOperations {
  /**
   * Pauses playback.
   */
  PAUSE = 1885435251,

  /**
   * Starts playback.
   */
  PLAY = 1886151033,

  /**
   * Resumes playback.
   */
  RESUME = 1919251317,

  /**
   * Reverses playback.
   */
  REVERSE_PLAYBACK = 1919252069,

  /**
   * Stops playback.
   */
  STOP = 1937010544,

  /**
   * Stops all playback.
   */
  STOP_ALL = 1937010785,
}

/**
 * View zoom style options.
 */
declare enum ViewZoomStyle {
  /**
   * Displays the page at 100% magnification.
   */
  ACTUAL_SIZE = 2053206906,

  /**
   * Fits the entire page in the window.
   */
  FIT_PAGE = 2053534832,

  /**
   * Fits the the text area of the page to the window width; obscures page margins and may obscure the lower portion of the page.
   */
  FIT_VISIBLE = 1212437334,

  /**
   * Fits the page to the width of the window; may obscure the lower portion of the page.
   */
  FIT_WIDTH = 1212437335,

  /**
   * Fills the screen with the page; hides the toolbar, command bar, menu bar, and window controls.
   */
  FULL_SCREEN = 1987733107,

  /**
   * Arranges the pages in a continuous vertical column that is one page wide.
   */
  ONE_COLUMN = 1987735395,

  /**
   * Displays one page in the document pane at a time.
   */
  SINGLE_PAGE = 1987736432,

  /**
   * Arranges the pages side by side in a continuous vertical column that is two pages wide.
   */
  TWO_COLUMN = 1987736675,

  /**
   * Magnifies the view to the next preset percentage.
   */
  ZOOM_IN = 2053990766,

  /**
   * Reduces the view to the previous preset percentage.
   */
  ZOOM_OUT = 2054124916,
}

/**
 * The version state of the file in Version Cue.
 */
declare enum VersionState {
  /**
   * The version has modifications that make it newer than the project.
   */
  LOCAL_NEWER = 1986221644,

  /**
   * The version is identical to the project.
   */
  LOCAL_PROJECT_MATCH = 1986221645,

  /**
   * No resource and no local file.
   */
  NO_RESOURCE = 1986221646,

  /**
   * The project has a newer file.
   */
  PROJECT_FILE_NEWER = 1986221648,

  /**
   * The version contains local edits but the project file is newer.
   */
  VERSION_CONFLICT = 1986221635,

  /**
   * The version is not known.
   */
  VERSION_UNKNOWN = 1986221653,
}

/**
 * The editing state of the file in Version Cue.
 */
declare enum EditingState {
  /**
   * The file was modified locally or remotely while it was locked and therefore two versions exist.
   */
  EDITING_CONFLICT = 1986217283,

  /**
   * The file has been modified locally and not locked.
   */
  EDITING_LOCALLY = 1986217292,

  /**
   * The file has been locked locally and may be modified.
   */
  EDITING_LOCALLY_LOCKED = 1986217291,

  /**
   * The file is not currently in use and is not locked.
   */
  EDITING_NOWHERE = 1986217294,

  /**
   * Lock held but not usable.
   */
  EDITING_REMOTELY = 1986217298,

  /**
   * The editing status is not known.
   */
  EDITING_UNKNOWN = 1986217301,
}

/**
 * The synchronization status of the file in Version Cue.
 */
declare enum VersionCueSyncStatus {
  /**
   * The project version of the file was downloaded to the local workspace.
   */
  FILE_DOWNLOADED = 1986220868,

  /**
   * The synchronization resulted in no change because the local and project versions were identical.
   */
  FILE_NO_CHANGE = 1986220878,

  /**
   * The file was not synchronized.
   */
  FILE_SKIPPED = 1986220875,

  /**
   * The file was unlocked locally.
   */
  FILE_UNLOCKED = 1986220876,

  /**
   * The local version of the file was uploaded to the project.
   */
  FILE_UPLOADED = 1986220885,
}

/**
 * The type of conflict resolution to employ during Version Cue synchronization.
 */
declare enum SyncConflictResolution {
  /**
   * Asks the user how to resolve conflicts.
   */
  ASK_ABOUT_CONFLICTS = 1986216769,

  /**
   * Uses the local version.
   */
  PREFER_LOCAL = 1986216780,

  /**
   * Uses the project version.
   */
  PREFER_PROJECT = 1986216784,

  /**
   * Skips conflicting files.
   */
  SKIP_CONFLICTS = 1986216787,
}

/**
 * Used to specify a language and region.
 */
declare enum LanguageAndRegion {
  /**
   * Albania: Albanian
   */
  ALBANIA_ALBANIAN = 1936802124,

  /**
   * Belarus: Belarusian
   */
  BELARUS_BELARUSIAN = 1650803289,

  /**
   * Brazil: Portuguese
   */
  BRAZIL_PORTUGUESE = 1886667346,

  /**
   * Bulgaria: Bulgarian
   */
  BULGARIA_BULGARIAN = 1650934343,

  /**
   * Croatia: Croatian
   */
  CROATIA_CROATIAN = 1752320082,

  /**
   * Czech Republic: Czech
   */
  CZECH_REPUBLIC_CZECH = 1668498266,

  /**
   * Denmark: Danish
   */
  DENMARK_DANISH = 1684096075,

  /**
   * Estonia: Estonian
   */
  ESTONIA_ESTONIAN = 1702118725,

  /**
   * Finland: Finnish
   */
  FINLAND_FINNISH = 1718175305,

  /**
   * France: French
   */
  FRANCE_FRENCH = 1718765138,

  /**
   * Germany: German
   */
  GERMANY_GERMAN = 1684358213,

  /**
   * Greece: Greek
   */
  GREECE_GREEK = 1701594962,

  /**
   * Hungary: Hungarian
   */
  HUNGARY_HUNGARIAN = 1752516693,

  /**
   * India: Tamil
   */
  INDIA_TAMIL = 1952532814,

  /**
   * Indic
   */
  INDIC = 1768842345,

  /**
   * Israel: Hebrew
   */
  ISRAEL_HEBREW = 1751468364,

  /**
   * Italy: Italian
   */
  ITALY_ITALIAN = 1769228628,

  /**
   * Japan: Japanese
   */
  JAPAN_JAPANESE = 1784760912,

  /**
   * Latvia: Latvian
   */
  LATVIA_LATVIAN = 1819692118,

  /**
   * Lituania: Lithuanian
   */
  LITUANIA_LITHUANIAN = 1819561044,

  /**
   * Netherlands: Dutch
   */
  NETHERLANDS_DUTCH = 1852591692,

  /**
   * Norway: Norwegian
   */
  NORWAY_NORWEGIAN = 1851936335,

  /**
   * Poland: Polish
   */
  POLAND_POLISH = 1886146636,

  /**
   * Republic Of Korea: Korean
   */
  REPUBLIC_OF_KOREA_KOREAN = 1802455890,

  /**
   * Romania: Romanian
   */
  ROMANIA_ROMANIAN = 1919898191,

  /**
   * Russian Federation: Russian
   */
  RUSSIAN_FEDERATION_RUSSIAN = 1920291413,

  /**
   * Simplified Chinese
   */
  SIMPLIFIED_CHINESE = 2053653326,

  /**
   * Slovakia: Slovak
   */
  SLOVAKIA_SLOVAK = 1936479051,

  /**
   * Slovenia: Slovenian
   */
  SLOVENIA_SLOVENIAN = 1936479049,

  /**
   * Spain: Spanish
   */
  SPAIN_SPANISH = 1702053203,

  /**
   * Standard Arabic
   */
  STANDARD_ARABIC = 1634877765,

  /**
   * Sweden: Swedish
   */
  SWEDEN_SWEDISH = 1937134422,

  /**
   * Thailand: Thai
   */
  THAILAND_THAI = 1952994376,

  /**
   * Traditional Chinese
   */
  TRADITIONAL_CHINESE = 2053657687,

  /**
   * Turkey: Turkish
   */
  TURKEY_TURKISH = 1953649746,

  /**
   * Ukraine: Ukrainian
   */
  UKRAINE_UKRAINIAN = 1969968449,

  /**
   * United Kingdom: English
   */
  UNITED_KINGDOM_ENGLISH = 1701726018,

  /**
   * United States: English
   */
  UNITED_STATES_ENGLISH = 1701729619,

  /**
   * Vietnam: Vietnamese
   */
  VIETNAM_VIETNAMESE = 1986614862,
}

/**
 * User interaction level options.
 */
declare enum UserInteractionLevels {
  /**
   * Displays alerts but not dialogs.
   */
  INTERACT_WITH_ALERTS = 1699311170,

  /**
   * The script displays all dialogs and alerts.
   */
  INTERACT_WITH_ALL = 1699311169,

  /**
   * The script does not display any dialogs or alerts.
   */
  NEVER_INTERACT = 1699640946,
}

/**
 * The locale.
 */
declare enum Locale {
  /**
   * Arabic
   */
  ARABIC_LOCALE = 1279476082,

  /**
   * Czech
   */
  CZECH_LOCALE = 1279476602,

  /**
   * Danish.
   */
  DANISH_LOCALE = 1279476846,

  /**
   * English.
   */
  ENGLISH_LOCALE = 1279477102,

  /**
   * Finnish.
   */
  FINNISH_LOCALE = 1279477358,

  /**
   * French.
   */
  FRENCH_LOCALE = 1279477362,

  /**
   * German.
   */
  GERMAN_LOCALE = 1279477613,

  /**
   * Greek
   */
  GREEK_LOCALE = 1279477618,

  /**
   * Hebrew
   */
  HEBREW_LOCALE = 1279477858,

  /**
   * Hungarian
   */
  HUNGARIAN_LOCALE = 1279477877,

  /**
   * International English.
   */
  INTERNATIONAL_ENGLISH_LOCALE = 1279477097,

  /**
   * Italian.
   */
  ITALIAN_LOCALE = 1279478132,

  /**
   * Japanese.
   */
  JAPANESE_LOCALE = 1279478384,

  /**
   * en_KoreanLocale
   */
  KOREAN_LOCALE = 1279478639,

  /**
   * Polish
   */
  POLISH_LOCALE = 1279479916,

  /**
   * Portuguese.
   */
  PORTUGUESE_LOCALE = 1279479911,

  /**
   * Romanian
   */
  ROMANIAN_LOCALE = 1279480431,

  /**
   * Russian
   */
  RUSSIAN_LOCALE = 1279480437,

  /**
   * simplified chinese
   */
  SIMPLIFIED_CHINESE_LOCALE = 1279476590,

  /**
   * Spanish.
   */
  SPANISH_LOCALE = 1279480688,

  /**
   * Swedish.
   */
  SWEDISH_LOCALE = 1279480695,

  /**
   * traditional chinese
   */
  TRADITIONAL_CHINESE_LOCALE = 1279480951,

  /**
   * Turkish
   */
  TURKISH_LOCALE = 1279480946,

  /**
   * Ukrainian
   */
  UKRAINIAN_LOCALE = 1279481195,
}

/**
 * The language of the script to execute.
 */
declare enum ScriptLanguage {
  /**
   * The AppleScript language.
   */
  APPLESCRIPT_LANGUAGE = 1095978087,

  /**
   * The JavaScript language.
   */
  JAVASCRIPT = 1246973031,

  /**
   * Language not specified.
   */
  UNKNOWN = 1433299822,
}

/**
 * Options for specifying a feature set.
 */
declare enum FeatureSetOptions {
  /**
   * Uses the Japanese feature set and defaults.
   */
  JAPANESE = 1247899758,

  /**
   * Uses the R2L feature set
   */
  RIGHTTOLEFT = 1381265228,

  /**
   * Uses the Roman feature set and defaults.
   */
  ROMAN = 1383034222,
}

/**
 * Undo options for executing a script.
 */
declare enum UndoModes {
  /**
   * Automatically undo the entire script as part of the previous step.
   */
  AUTO_UNDO = 1699963221,

  /**
   * Undo the entire script as a single step.
   */
  ENTIRE_SCRIPT = 1699963733,

  /**
   * Fast undo the entire script as a single step.
   */
  FAST_ENTIRE_SCRIPT = 1699964501,

  /**
   * Undo each script request as a separate step.
   */
  SCRIPT_REQUEST = 1699967573,
}

/**
 * Used to specify how to open a document.
 */
declare enum OpenOptions {
  /**
   * Default based on the file type or extension.
   */
  DEFAULT_VALUE = 1147563124,

  /**
   * Open a copy of the document.
   */
  OPEN_COPY = 1332757360,

  /**
   * Open the document itself.
   */
  OPEN_ORIGINAL = 1332760434,
}

/**
 * The state of a task.
 */
declare enum TaskState {
  /**
   * Task was cancelled (either before it ran or during execution
   */
  CANCELLED = 1700029292,

  /**
   * Task was signalled to cancel but did not stop yet
   */
  CANCELLING = 1700029281,

  /**
   * Task completed execution (successfully or with errors)
   */
  COMPLETED = 1700029296,

  /**
   * Task was queued and is waiting to be scheduled for execution
   */
  QUEUED = 1699837285,

  /**
   * Task is running
   */
  RUNNING = 1700033141,

  /**
   * Task is waiting
   */
  WAITING = 1700225396,
}

/**
 * The type of a task alert.
 */
declare enum TaskAlertType {
  /**
   * error message
   */
  TASK_ERROR = 1699040627,

  /**
   * Information message
   */
  TASK_INFORMATION = 1699302771,

  /**
   * Warning message
   */
  TASK_WARNING = 1700220275,
}

/**
 * Options for saving a document before closing or quitting.
 */
declare enum SaveOptions {
  /**
   * Displays a prompts asking whether to save changes.
   */
  ASK = 1634954016,

  /**
   * Does not save changes.
   */
  NO = 1852776480,

  /**
   * Saves changes.
   */
  YES = 2036691744,
}

/**
 * Display size options.
 */
declare enum ZoomOptions {
  /**
   * Zooms to 100%.
   */
  ACTUAL_SIZE = 2053206906,

  /**
   * Centers the active page in the window.
   */
  FIT_PAGE = 2053534832,

  /**
   * Centers the active spread in the window.
   */
  FIT_SPREAD = 2053534835,

  /**
   * Fits the entire pasteboard in the window.
   */
  SHOW_PASTEBOARD = 2054385762,

  /**
   * Magnifies the view to the next preset percentage.
   */
  ZOOM_IN = 2053990766,

  /**
   * Reduces the view to the next preset percentage.
   */
  ZOOM_OUT = 2054124916,
}

/**
 * Options for display performance settings, which influence the speed and quality with which an object draws and redraws.
 */
declare enum ViewDisplaySettings {
  /**
   * Slower performance; displays high-resolution graphics and high-quality transparencies and turns on anti-aliasing.
   */
  HIGH_QUALITY = 1346922866,

  /**
   * Best performance; grays out graphics and turns off transparency and anti-aliasing.
   */
  OPTIMIZED = 1349480564,

  /**
   * Moderate performance speed; displays proxy graphics and low-quality transparencies and turns on anti-aliasing.
   */
  TYPICAL = 1349810544,
}

/**
 * The reference point on the object's bounding box that does not move during transformation operations. Note: Transformations include rotation, scaling, flipping, and shearing.
 */
declare enum AnchorPoint {
  /**
   * The center point on the bottom of the bounding box.
   */
  BOTTOM_CENTER_ANCHOR = 1095656035,

  /**
   * The bottom left corner.
   */
  BOTTOM_LEFT_ANCHOR = 1095656044,

  /**
   * The bottom right corner.
   */
  BOTTOM_RIGHT_ANCHOR = 1095656050,

  /**
   * The center point in the bounding box.
   */
  CENTER_ANCHOR = 1095656308,

  /**
   * The center point on the left side of the bounding box.
   */
  LEFT_CENTER_ANCHOR = 1095658595,

  /**
   * The center point on the right side of the bounding box.
   */
  RIGHT_CENTER_ANCHOR = 1095660131,

  /**
   * The center point on the top of the bounding box.
   */
  TOP_CENTER_ANCHOR = 1095660643,

  /**
   * The top left corner.
   */
  TOP_LEFT_ANCHOR = 1095660652,

  /**
   * The top right corner.
   */
  TOP_RIGHT_ANCHOR = 1095660658,
}

/**
 * Color model options.
 */
declare enum ColorModel {
  /**
   * Mixed ink color.
   */
  MIXEDINKMODEL = 1768844664,

  /**
   * Process color.
   */
  PROCESS = 1886548851,

  /**
   * Registration color.
   */
  REGISTRATION = 1919248243,

  /**
   * Spot color.
   */
  SPOT = 1936748404,
}

/**
 * Color space options.
 */
declare enum ColorSpace {
  /**
   * CMYK.
   */
  CMYK = 1129142603,

  /**
   * LAB.
   */
  LAB = 1665941826,

  /**
   * Mixed ink.
   */
  MIXEDINK = 1666009432,

  /**
   * RGB.
   */
  RGB = 1666336578,
}

/**
 * Screen mode options.
 */
declare enum ScreenModeOptions {
  /**
   * Preview mode with editing turned off. Mouse clicks and arrow keys will move to previous or next spread.
   */
  PRESENTATION_PREVIEW = 1936552046,

  /**
   * Normal view; displays guides and frame edges.
   */
  PREVIEW_OFF = 1936552047,

  /**
   * Preview mode including the bleed area.
   */
  PREVIEW_TO_BLEED = 1936552034,

  /**
   * Preview mode; displays the document as it will be printed (hides guides and frame edges).
   */
  PREVIEW_TO_PAGE = 1936552048,

  /**
   * Preview mode including the slug area.
   */
  PREVIEW_TO_SLUG = 1936552051,
}

/**
 * Watermark vertical position enum type.
 */
declare enum WatermarkVerticalPositionEnum {
  /**
   * Place watermark vertical bottom.
   */
  WATERMARK_V_BOTTOM = 1884704866,

  /**
   * Place watermark vertical center.
   */
  WATERMARK_V_CENTER = 1884704867,

  /**
   * Place watermark vertical top.
   */
  WATERMARK_V_TOP = 1884704884,
}

/**
 * Watermark horizontal position enum type.
 */
declare enum WatermarkHorizontalPositionEnum {
  /**
   * Place watermark horizontal center.
   */
  WATERMARK_H_CENTER = 1883787363,

  /**
   * Place watermark horizontal left.
   */
  WATERMARK_H_LEFT = 1883787372,

  /**
   * Place watermark horizontal right.
   */
  WATERMARK_H_RIGHT = 1883787378,
}

/**
 * Options for page numbering.
 */
declare enum PageNumberingOptions {
  /**
   * Numbers all pages in the document sequentially.
   */
  ABSOLUTE = 1096971116,

  /**
   * Numbers pages according to page numbering specifications of the section.
   */
  SECTION = 1935897710,
}

/**
 * Tool tip behavior options.
 */
declare enum ToolTipOptions {
  /**
   * Displays tool tips more quickly than normal.
   */
  FAST = 1180791668,

  /**
   * Turns off tool tips.
   */
  NONE = 1852796517,

  /**
   * Displays tool tips.
   */
  NORMAL = 1852797549,
}

/**
 * Options for selection status in relation to previously selected objects.
 */
declare enum SelectionOptions {
  /**
   * Adds the object to the existing selection; if no object was previously selected, makes the object the only selected object.
   */
  ADD_TO = 1633969202,

  /**
   * Deselects the object.
   */
  REMOVE_FROM = 1919249734,

  /**
   * Selects the object and deselects any previously selected objects.
   */
  REPLACE_WITH = 1919250519,

  /**
   * Sets the key object. At least 2 objects must be selected, and the key object specified must be one of them.
   */
  SET_KEY = 1936028779,
}

/**
 * Selection options.
 */
declare enum SelectAll {
  /**
   * Selects all.
   */
  ALL = 1634495520,
}

/**
 * Preview size options.
 */
declare enum PreviewSizeOptions {
  /**
   * Extra large preview (1024 x 1024).
   */
  EXTRA_LARGE = 1162629234,

  /**
   * Large preview (512 x 512).
   */
  LARGE = 1281446002,

  /**
   * Medium preview (256 x 256).
   */
  MEDIUM = 1701727588,

  /**
   * Small preview (128 x 128).
   */
  SMALL = 1399672946,
}

/**
 * Options for Tools panel.
 */
declare enum ToolsPanelOptions {
  /**
   * Double column
   */
  DOUBLE_COLUMN = 1162109804,

  /**
   * Single column
   */
  SINGLE_COLUMN = 1163092844,

  /**
   * Single row
   */
  SINGLE_ROW = 1163096695,
}

/**
 * Live drawing options for when user mouse actions trigger live screen drawing of page items.
 */
declare enum LiveDrawingOptions {
  /**
   * Use live screen drawing during mouse operations after a delay if user pauses before the mouse moves.
   */
  DELAYED = 1347765349,

  /**
   * Use live screen drawing during mouse operations.
   */
  IMMEDIATELY = 1347766637,

  /**
   * Never use live screen drawing during mouse operations, use sprite mode.
   */
  NEVER = 1347767926,
}

/**
 * Options for preview pages.
 */
declare enum PreviewPagesOptions {
  /**
   * All pages.
   */
  ALL_PAGES = 1886547553,

  /**
   * First ten pages.
   */
  FIRST_10_PAGES = 1180192871,

  /**
   * First two pages.
   */
  FIRST_2_PAGES = 1177702503,

  /**
   * First five pages.
   */
  FIRST_5_PAGES = 1177899111,

  /**
   * First page.
   */
  FIRST_PAGE = 1700947536,
}

/**
 * The alignment for static text.
 */
declare enum StaticAlignmentOptions {
  /**
   * Center align the text.
   */
  CENTER_ALIGN = 1667591796,

  /**
   * Left align the text.
   */
  LEFT_ALIGN = 1818584692,

  /**
   * Right align the text.
   */
  RIGHT_ALIGN = 1919379572,
}

/**
 * Rendering intent options.
 */
declare enum RenderingIntent {
  /**
   * Maintains color accuracy at the expense of preserving relationships between colors; most suitable for previewing how paper color affects printed colors.
   */
  ABSOLUTE_COLORIMETRIC = 1380540771,

  /**
   * Preserves the visual relationship between colors at the expense of actual color values; most suitable for photographic images with high percentages of out-of-gamut colors.
   */
  PERCEPTUAL = 1380544611,

  /**
   * Compares the extreme highlight of the source color space to that of the desination color space and shifts all colors accordingly; out-of-gamut colors are shifted to the closest reproducible color in the destination color space.
   */
  RELATIVE_COLORIMETRIC = 1380545123,

  /**
   * Produces vivid colors at the expense of color accuracy; most suitable for business graphics such as graphs or charts.
   */
  SATURATION = 1380545377,

  /**
   * Uses the current color settings.
   */
  USE_COLOR_SETTINGS = 1380541299,
}

/**
 * Stroke weight options for printer marks.
 */
declare enum MarkLineWeight {
  /**
   * 05 mm.
   */
  P05MM = 808807789,

  /**
   * 07 mm.
   */
  P07MM = 808938861,

  /**
   * 10 mm.
   */
  P10MM = 825257325,

  /**
   * 125 pt.
   */
  P125PT = 825374064,

  /**
   * 15 mm.
   */
  P15MM = 825585005,

  /**
   * 20 mm.
   */
  P20MM = 842034541,

  /**
   * 25 pt.
   */
  P25PT = 842346608,

  /**
   * 30 mm.
   */
  P30MM = 858811757,

  /**
   * 50 pt.
   */
  P50PT = 892350576,
}

/**
 * Options for printer marks formats.
 */
declare enum MarkTypes {
  /**
   * Uses the default format.
   */
  DEFAULT_VALUE = 1147563124,

  /**
   * Uses J marks without a circle.
   */
  J_MARK_WITHOUT_CIRCLE = 1785556579,

  /**
   * Uses J marks with a circle.
   */
  J_MARK_WITH_CIRCLE = 1785558883,
}

/**
 * Color output mode options for composites.
 */
declare enum ColorOutputModes {
  /**
   * Sends full-color versions of the specified pages to the printer. Note: Available only for PostScript printers.
   */
  COMPOSITE_CMYK = 1668105035,

  /**
   * Sends grayscale versions of the specified pages to the printer.
   */
  COMPOSITE_GRAY = 1668116583,

  /**
   * Sends a full-color version of the specified pages to the printer, preserving all color values in the original document. Note: Cannot simulate overprint when using this option.
   */
  COMPOSITE_LEAVE_UNCHANGED = 1668107349,

  /**
   * Sends full-color versions of the specified pages to the printer.
   */
  COMPOSITE_RGB = 1668108866,

  /**
   * Allows the printer to create color separations. Note: Valid only with a PostScript raster image processing (RIP) device.
   */
  INRIP_SEPARATIONS = 1919512691,

  /**
   * Sends PostScript information for each of the required separations to the printer. Note: Available only for PostScript printers.
   */
  SEPARATIONS = 1936027745,
}

/**
 * Format options for image data.
 */
declare enum DataFormat {
  /**
   * Uses ASCII format.
   */
  ASCII = 1095975753,

  /**
   * Uses binary format.
   */
  BINARY = 1114534521,
}

/**
 * Flip direction options.
 */
declare enum Flip {
  /**
   * Horizontal and vertical flip (same as rotate 180)
   */
  BOTH = 1651471464,

  /**
   * Flips the printed image horizontally.
   */
  HORIZONTAL = 1752134266,

  /**
   * Flips the printed image horizontally and vertically (same as rotate 180).
   */
  HORIZONTAL_AND_VERTICAL = 1215977068,

  /**
   * The printed image is not flipped.
   */
  NONE = 1852796517,

  /**
   * Flips the printed image vertically.
   */
  VERTICAL = 1986359924,
}

/**
 * Options for downloading fonts to the printer.
 */
declare enum FontDownloading {
  /**
   * Downloads all fonts once per page.
   */
  COMPLETE = 2003332197,

  /**
   * Downloads only references to fonts. Note: Use when fonts reside in the printer.
   */
  NONE = 1852796517,

  /**
   * Downloads only the characters (glyphs) used in the document. Glyphs are downloaded once per page.
   */
  SUBSET = 1768842098,

  /**
   * Downloads only the characters (glyphs) used in the document. Glyphs are downloaded once per page. Note: Use when the number of glyphs exceeds 350.
   */
  SUBSET_LARGE = 1818325607,
}

/**
 * Options for positioning the page on the paper or film.
 */
declare enum PagePositions {
  /**
   * Centers the page horizontally and vertically.
   */
  CENTERED = 1668183106,

  /**
   * Centers the page horizontally.
   */
  CENTER_HORIZONTALLY = 1668183112,

  /**
   * Centers the page vertically.
   */
  CENTER_VERTICALLY = 1668183126,

  /**
   * Places the page in the upper left corner.
   */
  UPPER_LEFT = 1668183118,
}

/**
 * Printer PostScript level options.
 */
declare enum PostScriptLevels {
  /**
   * Level 2 PostScript.
   */
  LEVEL_2 = 1347636274,

  /**
   * Level 3 PostScript.
   */
  LEVEL_3 = 1347636275,
}

/**
 * Page orientation options.
 */
declare enum PrintPageOrientation {
  /**
   * Landscape.
   */
  LANDSCAPE = 2003395685,

  /**
   * Portrait.
   */
  PORTRAIT = 1751738216,

  /**
   * Reverse landscape.
   */
  REVERSE_LANDSCAPE = 1869771372,

  /**
   * Reverse portrait.
   */
  REVERSE_PORTRAIT = 1869771376,
}

/**
 * Options for sending image data to the printer or file.
 */
declare enum ImageDataTypes {
  /**
   * Sends full-resolution data.
   */
  ALL_IMAGE_DATA = 1853058416,

  /**
   * Prints graphics frames with crossbars in place of graphics.
   */
  NONE = 1852796517,

  /**
   * Sends just enough data to print graphics at the best possible resolution for the output device.
   */
  OPTIMIZED_SUBSAMPLING = 1869640563,

  /**
   * Sends screen-resolution versions (72 dpi) of placed bitmap images.
   */
  PROXY_IMAGE_DATA = 1819243130,
}

/**
 * Options for printing page sequences.
 */
declare enum Sequences {
  /**
   * Prints all pages.
   */
  ALL = 1634495520,

  /**
   * Prints only even-numbered pages.
   */
  EVEN = 1702258030,

  /**
   * Prints only odd-numbered pages.
   */
  ODD = 1868850208,
}

/**
 * Color-management system source options.
 */
declare enum SourceSpaces {
  /**
   * Uses the color space of the proof.
   */
  PROOF_SPACE = 1886548848,

  /**
   * Uses the color space of the document.
   */
  USE_DOCUMENT = 1967419235,
}

/**
 * Options for the number of thumbnails per page.
 */
declare enum ThumbsPerPage {
  /**
   * Fits one row on the page; the row contains two thumbnails.
   */
  K1X2 = 1949399090,

  /**
   * Fits two rows of two.
   */
  K2X2 = 1949464626,

  /**
   * Fits three rows of three.
   */
  K3X3 = 1949530163,

  /**
   * Fits four rows of four.
   */
  K4X4 = 1949595700,

  /**
   * Fits five rows of five.
   */
  K5X5 = 1949661237,

  /**
   * Fits six rows of six.
   */
  K6X6 = 1949726774,

  /**
   * Fits seven rows of seven.
   */
  K7X7 = 1949792311,
}

/**
 * Tiling type options.
 */
declare enum TilingTypes {
  /**
   * Automatically calculates the number of tiles required, including the overlap. For information, see tiling overlap.
   */
  AUTO = 1635019116,

  /**
   * Increases the amount of overlap as necessary so that the right sides of the right-most tiles are aligned at the right edge of the document page, and the bottom sides of the bottom-most tiles are aligned at the bottom edge of the document page. For information, see tiling overlap.
   */
  AUTO_JUSTIFIED = 1634366324,

  /**
   * Prints a single tile whose upper left corner is at the zero point of the rulers.
   */
  MANUAL = 1835955308,
}

/**
 * Trapping options.
 */
declare enum Trapping {
  /**
   * Adobe in-RIP.
   */
  ADOBE_INRIP = 1919512660,

  /**
   * Application built-in.
   */
  APPLICATION_BUILTIN = 1114199152,

  /**
   * No trapping.
   */
  OFF = 1330005536,
}

/**
 * Ink trapping type options.
 */
declare enum InkTypes {
  /**
   * Uses traditional process inks and most spot inks.
   */
  NORMAL = 1852797549,

  /**
   * Uses heavy, nontransparent inks to prevent trapping of underlying colors but allow for trapping along the edges of the ink. Best for metallic inks.
   */
  OPAQUE = 1769230192,

  /**
   * Uses heavy, nontransparent inks to prevent trapping of underlying colors but allow for trapping along the edges of the ink. Best for inks that have undesirable interactions with other inks.
   */
  OPAQUE_IGNORE = 1769228647,

  /**
   * Uses clear inks to ensure that underlying items trap. Best for varnishes and dieline inks.
   */
  TRANSPARENT = 1769231474,
}

/**
 * Printer preset options.
 */
declare enum PrinterPresetTypes {
  /**
   * A custom printer preset.
   */
  CUSTOM = 1131639917,

  /**
   * The default printer preset.
   */
  DEFAULT_VALUE = 1147563124,
}

/**
 * The color-rendering dictionary (CRD) to use.
 */
declare enum ColorRenderingDictionary {
  /**
   * Uses the default CRD.
   */
  DEFAULT_VALUE = 1147563124,

  /**
   * Uses the document's CRD.
   */
  USE_DOCUMENT = 1967419235,

  /**
   * Uses the working CRD.
   */
  WORKING = 1466921579,
}

/**
 * Page range options.
 */
declare enum PageRange {
  /**
   * Print or export all pages in the document.
   */
  ALL_PAGES = 1886547553,

  /**
   * Export selected items in the document.
   */
  SELECTED_ITEMS = 1886547571,
}

/**
 * Paper size options.
 */
declare enum PaperSize {
  /**
   * Automatic
   */
  AUTO = 1635019116,
}

/**
 * PPD options.
 */
declare enum PPDValues {
  /**
   * Device-independent.
   */
  DEVICE_INDEPENDENT = 1684367716,
}

/**
 * Color profile options.
 */
declare enum Profile {
  /**
   * No CMS profile is used.
   */
  NO_CMS = 1970499183,

  /**
   * Uses the PostScript CMS profile.
   */
  POSTSCRIPT_CMS = 1970303843,

  /**
   * Uses the document profile.
   */
  USE_DOCUMENT = 1967419235,

  /**
   * Uses the working profile.
   */
  WORKING = 1466921579,
}

/**
 * Options for ink screening for composite gray output in PostScript or PDF format.
 */
declare enum Screeening {
  /**
   * Uses custom screening settings for ink angle and frequency. For information, see composite angle and composite frequency.
   */
  CUSTOM = 1131639917,

  /**
   * Uses the default screening settings.
   */
  DEFAULT_VALUE = 1147563124,
}

/**
 * Paper size optons.
 */
declare enum PaperSizes {
  /**
   * Allows definition of a custom paper size. Note: Not all printers allow custom paper sizes.
   */
  CUSTOM = 1131639917,

  /**
   * Allows the printer driver to define the paper size.
   */
  DEFINED_BY_DRIVER = 1347634290,
}

/**
 * Printer options.
 */
declare enum Printer {
  /**
   * Prints to a PostScript file.
   */
  POSTSCRIPT_FILE = 1886611052,
}

/**
 * Options for trap placement between vector objects and bitmap images.
 */
declare enum TrapImagePlacementTypes {
  /**
   * Creates a trap that straddles the edge between vector objects and bitmap images.
   */
  CENTER_EDGES = 1953522542,

  /**
   * Causes vector objects to overlap abutting images.
   */
  CHOKE = 1953522536,

  /**
   * Causes bitmap images to overlap the abutting objects.
   */
  IMAGES_OVER_SPREAD = 1953526640,

  /**
   * Applies the same trapping rules as used elsewhere in the document. Note: When used to trap an object to a photograph, can result in noticeably uneven edges as the trap moves from one side of the edge to another.
   */
  IMAGE_NEUTRAL_DENSITY = 1953525348,
}

/**
 * Shape options for the intersection of three-way traps.
 */
declare enum TrapEndTypes {
  /**
   * Shapes the end of the trap to keep it away from the intersecting object.
   */
  MITER_TRAP_ENDS = 1953525093,

  /**
   * Reshapes the trap generated by the lightest neutral density object so that it wraps around the point where the three objects intersect.
   */
  OVERLAP_TRAP_ENDS = 1953525612,
}

/**
 * Options for exporting image data to the EPS document.
 */
declare enum EPSImageData {
  /**
   * Exports high-resolution data. Note: Use when the file will be printed on a high-resolution output device.
   */
  ALL_IMAGE_DATA = 1853058416,

  /**
   * Exports only screen-resolution versions (72 dpi) of placed bitmap images. Note: Use in conjunction with OPI image replacement or if the resulting file will be viewed on-screen.
   */
  PROXY_IMAGE_DATA = 1819243130,
}

/**
 * Booklet type options.
 */
declare enum BookletTypeOptions {
  /**
   * Four up consecutive imposition.
   */
  FOUR_UP_CONSECUTIVE = 1110721363,

  /**
   * Three up consecutive imposition.
   */
  THREE_UP_CONSECUTIVE = 1110655827,

  /**
   * Two up consecutive imposition.
   */
  TWO_UP_CONSECUTIVE = 1110590291,

  /**
   * Two up perfect bound imposition.
   */
  TWO_UP_PERFECT_BOUND = 1110593602,

  /**
   * Two up saddle stitch imposition.
   */
  TWO_UP_SADDLE_STITCH = 1110594387,
}

/**
 * Signature size options.
 */
declare enum SignatureSizeOptions {
  /**
   * Signature size 12.
   */
  SIGNATURE_SIZE_12 = 1112748338,

  /**
   * Signature size 16.
   */
  SIGNATURE_SIZE_16 = 1112748342,

  /**
   * Signature size 32.
   */
  SIGNATURE_SIZE_32 = 1112748850,

  /**
   * Signature size 4.
   */
  SIGNATURE_SIZE_4 = 1112748084,

  /**
   * Signature size 8.
   */
  SIGNATURE_SIZE_8 = 1112748088,
}

/**
 * Document print ui options
 */
declare enum DocumentPrintUiOptions {
  /**
   * Do not show file save dialog during printing.
   */
  SUPPRESS_FILE_SAVE_DIALOG = 1936089444,

  /**
   * Do not show print dialog.
   */
  SUPPRESS_PRINT_DIALOG = 1936745575,

  /**
   * Do not show progress bar during printing.
   */
  SUPPRESS_PRINT_PROGRESS = 1936748659,

  /**
   * Do not show warning dialog during printing.
   */
  SUPPRESS_PRINT_WARNINGS = 1936750450,
}

/**
 * Note background color options.
 */
declare enum NoteBackgrounds {
  /**
   * Uses the galley background color.
   */
  GALLEY_BACKGROUND_COLOR = 1699168839,

  /**
   * Uses the note color.
   */
  USE_NOTE_COLOR = 1700020807,
}

/**
 * Note color options.
 */
declare enum NoteColorChoices {
  /**
   * Uses the note color.
   */
  USE_NOTE_PREF_COLOR = 1700089923,

  /**
   * Uses the color assigned to the user.
   */
  USE_USER_COLOR = 1700091203,
}

/**
 * Marking options for changed text.
 */
declare enum ChangeMarkings {
  /**
   * Does not mark changed text.
   */
  NONE = 1852796517,

  /**
   * Outlines changed text.
   */
  OUTLINE = 1869900910,

  /**
   * Uses a strikethrough to mark changed text.
   */
  STRIKETHROUGH = 1699968114,

  /**
   * Underlines changed text.
   */
  UNDERLINE_SINGLE = 1700097636,
}

/**
 * Change bar location options.
 */
declare enum ChangebarLocations {
  /**
   * Change bars are in the left margin.
   */
  LEFT_ALIGN = 1818584692,

  /**
   * Change bars are in the right margin.
   */
  RIGHT_ALIGN = 1919379572,
}

/**
 * Changed text color options.
 */
declare enum ChangeTextColorChoices {
  /**
   * The text color for changed text is the same as the text color defined in track changes preferences. For information, see text color for added text, text color for deleted text, or text color for moved text.
   */
  CHANGE_USES_CHANGE_PREF_COLOR = 1700098147,

  /**
   * The text color for changed text is the same as the galley text color.
   */
  CHANGE_USES_GALLEY_TEXT_COLOR = 1700095843,
}

/**
 * Background color options for changed text.
 */
declare enum ChangeBackgroundColorChoices {
  /**
   * The background color for changed text is the same as the track changes preferences background color. For information, see background color for added text, background color for deleted text, or background color for moved text.
   */
  CHANGE_BACKGROUND_USES_CHANGE_PREF_COLOR = 1700098146,

  /**
   * The background color for changed text is the same as the galley background color.
   */
  CHANGE_BACKGROUND_USES_GALLEY_BACKGROUND_COLOR = 1700095842,

  /**
   * The background color for changed text is the same as the color assigned to the current user.
   */
  CHANGE_BACKGROUND_USES_USER_COLOR = 1700099426,
}

/**
 * Lock state options.
 */
declare enum LockStateValues {
  /**
   * The story has been checked in.
   */
  CHECKED_IN_STORY = 1112695657,

  /**
   * The story has been checked out.
   */
  CHECKED_OUT_STORY = 1112695663,

  /**
   * The story is embedded.
   */
  EMBEDDED_STORY = 1112696173,

  /**
   * The story is locked.
   */
  LOCKED_STORY = 1112697963,

  /**
   * The story file is missing.
   */
  MISSING_LOCK_STATE = 1112698227,

  /**
   * The stories have a mixed lock state.
   */
  MIXED_LOCK_STATE = 1112698232,

  /**
   * No lock state.
   */
  NONE = 1852796517,

  /**
   * The story is unmanaged.
   */
  UNMANAGED_STORY = 1112700269,
}

/**
 * Color space options for representing color in the exported JPEG.
 */
declare enum JpegColorSpaceEnum {
  /**
   * Represents all color values using the CMYK color space.
   */
  CMYK = 1129142603,

  /**
   * Converts all color values to high-quality black-and-white images. Gray levels of the converted objects represent the luminosity of the original objects.
   */
  GRAY = 1766290041,

  /**
   * Represents all color values using the RGB color space. Best suited for documents that will be viewed on-screen.
   */
  RGB = 1666336578,
}

/**
 * Page export options.
 */
declare enum ExportRangeOrAllPages {
  /**
   * Exports all pages.
   */
  EXPORT_ALL = 1785742657,

  /**
   * Exports the page range specified in the page string property.
   */
  EXPORT_RANGE = 1785742674,
}

/**
 * The records to merge.
 */
declare enum RecordSelection {
  /**
   * Merges all records.
   */
  ALL_RECORDS = 1684881778,

  /**
   * Merges the specified record.
   */
  ONE_RECORD = 1684885362,

  /**
   * Merges all records in the specified range.
   */
  RANGE = 1684886130,
}

/**
 * The order in which to arrange records in the target document.
 */
declare enum ArrangeBy {
  /**
   * Arranges records by column.
   */
  COLUMNS_FIRST = 1684882278,

  /**
   * Arranges records by row.
   */
  ROWS_FIRST = 1684886118,
}

/**
 * The number of records per page.
 */
declare enum RecordsPerPage {
  /**
   * Places as many records as fit on a page.
   */
  MULTIPLE_RECORD = 1684884850,

  /**
   * Places each record on a new page.
   */
  SINGLE_RECORD = 1684886386,
}

/**
 * Options for fitting content to a frame.
 */
declare enum Fitting {
  /**
   * Resizes content to fill the frame while preserving content proportions. If the content and frame have different proportions, some content is obscured by the bounding box of the frame.
   */
  FILL_PROPORTIONAL = 1684883056,

  /**
   * Resizes content to fit the frame. Note: Content that is a different size than the frame appears stretched or squeezed.
   */
  FIT_CONTENT_TO_FRAME = 1684883043,

  /**
   * Resizes the frame to fit the content.
   */
  FIT_FRAME_TO_CONTENT = 1684883046,

  /**
   * Preserves the original sizes of the frame and the content. Note: Content that is larger than the frame is obscured around the edges.
   */
  PRESERVE_SIZES = 1684885619,

  /**
   * Resizes content to fit the frame while preserving content proportions. If the content and frame have different proportions, some empty space occurs in the frame.
   */
  PROPORTIONAL = 1684885618,
}

/**
 * Data type options for data merge fields.
 */
declare enum SourceFieldType {
  /**
   * The field can fill a data merge image placeholder.
   */
  IMAGE_FIELD = 1684883814,

  /**
   * The field can fill a data merge QR code placeholder.
   */
  QRCODE_FIELD = 1684885862,

  /**
   * The field can fill a data merge text placeholder.
   */
  TEXT_FIELD = 1684886630,
}

/**
 * List type options.
 */
declare enum ListType {
  /**
   * Bullet list.
   */
  BULLET_LIST = 1280598644,

  /**
   * No list.
   */
  NO_LIST = 1280601711,

  /**
   * Numbered list.
   */
  NUMBERED_LIST = 1280601709,
}

/**
 * Bullet character type.
 */
declare enum BulletCharacterType {
  /**
   * Glyph with font.
   */
  GLYPH_WITH_FONT = 1111713638,

  /**
   * Unicode only.
   */
  UNICODE_ONLY = 1111717231,

  /**
   * Unicode with font.
   */
  UNICODE_WITH_FONT = 1111717222,
}

/**
 * Numbering style
 */
declare enum NumberingStyle {
  /**
   * Arabic
   */
  ARABIC = 1298231906,

  /**
   * Uses Arabic Abjad
   */
  ARABIC_ABJAD = 1296130410,

  /**
   * Uses Arabic Alif Ba Tah
   */
  ARABIC_ALIF_BA_TAH = 1296130420,

  /**
   * Add double leading zeros.
   */
  DOUBLE_LEADING_ZEROS = 1296329850,

  /**
   * Do not add characters.
   */
  FORMAT_NONE = 1701733998,

  /**
   * Uses Hebrew Biblical
   */
  HEBREW_BIBLICAL = 1296589410,

  /**
   * Uses Hebrew Non Standard
   */
  HEBREW_NON_STANDARD = 1296589422,

  /**
   * Kanji
   */
  KANJI = 1296788073,

  /**
   * Katakana (a, i, u, e, o...).
   */
  KATAKANA_MODERN = 1265920877,

  /**
   * Katakana (i, ro, ha, ni...).
   */
  KATAKANA_TRADITIONAL = 1265920884,

  /**
   * Lower letters
   */
  LOWER_LETTERS = 1296855660,

  /**
   * Lower roman
   */
  LOWER_ROMAN = 1297247596,

  /**
   * Add single leading zeros.
   */
  SINGLE_LEADING_ZEROS = 1297312890,

  /**
   * Add triple leading zeros.
   */
  TRIPLE_LEADING_ZEROS = 1297378426,

  /**
   * Upper letters
   */
  UPPER_LETTERS = 1296855669,

  /**
   * Upper roman
   */
  UPPER_ROMAN = 1297247605,
}

/**
 * Restart numbering options.
 */
declare enum RestartPolicy {
  /**
   * Restart numbering after a specific numbering level.
   */
  AFTER_SPECIFIC_LEVEL = 1701737324,

  /**
   * Restart numbering after any previous (higher) numbering level.
   */
  ANY_PREVIOUS_LEVEL = 1701732720,

  /**
   * Restart numbering after any of a range of numbering levels.
   */
  RANGE_OF_LEVELS = 1701737068,
}

/**
 * List alignment options.
 */
declare enum ListAlignment {
  /**
   * Align center.
   */
  CENTER_ALIGN = 1667591796,

  /**
   * Align left.
   */
  LEFT_ALIGN = 1818584692,

  /**
   * Align right.
   */
  RIGHT_ALIGN = 1919379572,
}

/**
 * Chapter number sources.
 */
declare enum ChapterNumberSources {
  /**
   * Continue chapter number from previous document.
   */
  CONTINUE_FROM_PREVIOUS_DOCUMENT = 1668178800,

  /**
   * Chapter number same as previous document.
   */
  SAME_AS_PREVIOUS_DOCUMENT = 1668182896,

  /**
   * User-defined chapter number.
   */
  USER_DEFINED = 1668183396,
}

/**
 * Custom Layout Type Options.
 */
declare enum CustomLayoutTypeEnum {
  /**
   * Alignment And Spacing.
   */
  ALIGNMENT_AND_SPACING = 1097618288,

  /**
   * Float Left.
   */
  FLOAT_LEFT = 1181502565,

  /**
   * Float Right.
   */
  FLOAT_RIGHT = 1181504105,
}

/**
 * Export options for assignment files.
 */
declare enum AssignmentExportOptions {
  /**
   * Exports only spreads with assigned frames.
   */
  ASSIGNED_SPREADS = 1098073459,

  /**
   * Exports frames but does not export content.
   */
  EMPTY_FRAMES = 1098073446,

  /**
   * Exports the entire document.
   */
  EVERYTHING = 1098073441,
}

/**
 * The status of the assigment.
 */
declare enum AssignmentStatus {
  /**
   * The assignment file is missing.
   */
  ASSIGNMENT_FILE_MISSING = 1095126387,

  /**
   * The assignment has been modified and needs to be updated.
   */
  ASSIGNMENT_OUT_OF_DATE = 1095724868,

  /**
   * The assignment has not been modified.
   */
  ASSIGNMENT_UP_TO_DATE = 1096119364,
}

/**
 * The display performance settings to use while scrolling.
 */
declare enum PanningTypes {
  /**
   * While scrolling, greeks newly revealed images until the mouse is released; medium quality display with medium performance speed.
   */
  GREEK_IMAGES = 1699111248,

  /**
   * While scrolling, greeks newly revealed images and text until the mouse is released; highest quality display with the slowest performance.
   */
  GREEK_IMAGES_AND_TEXT = 1699639120,

  /**
   * While scrolling, does not greek images or text; lowest quality display with the fastest performance.
   */
  NO_GREEKING = 1699116368,
}

/**
 * Options for aligning or distributing objects.
 */
declare enum AlignDistributeBounds {
  /**
   * Align or distribute to the bounds of the objects.
   */
  ITEM_BOUNDS = 1416587604,

  /**
   * Align or distribute to a key object.
   */
  KEY_OBJECT = 1699439993,

  /**
   * Align or distribute to the margins of the page.
   */
  MARGIN_BOUNDS = 1416588609,

  /**
   * Align or distribute to the bounds of the page.
   */
  PAGE_BOUNDS = 1416589377,

  /**
   * Align or distribute to the bounds of the spread.
   */
  SPREAD_BOUNDS = 1416590160,
}

/**
 * Options for aligning objects.
 */
declare enum AlignOptions {
  /**
   * Align the bottom edges of the objects.
   */
  BOTTOM_EDGES = 1114916196,

  /**
   * Align the horizontal centers of the objects.
   */
  HORIZONTAL_CENTERS = 1215257187,

  /**
   * Align the left edges of the objects.
   */
  LEFT_EDGES = 1281770852,

  /**
   * Align the right edges of the objects.
   */
  RIGHT_EDGES = 1383351652,

  /**
   * Align the top edges of the objects.
   */
  TOP_EDGES = 1416643940,

  /**
   * Align the vertical centers of the objects.
   */
  VERTICAL_CENTERS = 1449481315,
}

/**
 * Options for distributing objects.
 */
declare enum DistributeOptions {
  /**
   * Distribute based on the bottom edges of the objects.
   */
  BOTTOM_EDGES = 1114916196,

  /**
   * Distribute based on the horizontal centers of the objects.
   */
  HORIZONTAL_CENTERS = 1215257187,

  /**
   * Distribute based on the horizontal spacing of the objects.
   */
  HORIZONTAL_SPACE = 1215257203,

  /**
   * Distribute based on the left edges of the objects.
   */
  LEFT_EDGES = 1281770852,

  /**
   * Distribute based on the right edges of the objects.
   */
  RIGHT_EDGES = 1383351652,

  /**
   * Distribute based on the top edges of the objects.
   */
  TOP_EDGES = 1416643940,

  /**
   * Distribute based on the vertical centers of the objects.
   */
  VERTICAL_CENTERS = 1449481315,

  /**
   * Distribute based on the vertical spacing of the objects.
   */
  VERTICAL_SPACE = 1449489523,
}

/**
 * The source type of alternate or actual text
 */
declare enum SourceType {
  /**
   * Custom Text
   */
  SOURCE_CUSTOM = 1934902125,

  /**
   * XML Structure
   */
  SOURCE_XML_STRUCTURE = 1934907507,

  /**
   * XMP Description
   */
  SOURCE_XMP_DESCRIPTION = 1934907492,

  /**
   * XMP Headline
   */
  SOURCE_XMP_HEADLINE = 1934907496,

  /**
   * User specified XMP metadata property
   */
  SOURCE_XMP_OTHER = 1934907503,

  /**
   * XMP Title
   */
  SOURCE_XMP_TITLE = 1934907508,
}

/**
 * The tag type of page item
 */
declare enum TagType {
  /**
   * Tag as artifact
   */
  TAG_ARTIFACT = 1952924006,

  /**
   * Tag as Story or Figure based on object type
   */
  TAG_BASED_ON_OBJECT = 1952924271,

  /**
   * Determine the tag from XML structure, or fallback to standard tag
   */
  TAG_FROM_STRUCTURE = 1952928613,
}

/**
 * File format options for converted images
 */
declare enum ImageFormat {
  /**
   * Uses GIF format for selected object.
   */
  GIF = 1734960742,

  /**
   * Uses JPEG format for selected object.
   */
  JPEG = 1785751398,

  /**
   * Uses PNG format, for selected object.
   */
  PNG = 1397059687,
}

/**
 * Image size option for a converted object
 */
declare enum ImageSizeOption {
  /**
   * Image size to be used is absolute.
   */
  SIZE_FIXED = 1182295162,

  /**
   * Image size to be used is relative to the text flow.
   */
  SIZE_RELATIVE_TO_TEXT_FLOW = 1383486566,
}

/**
 * Image resolution for converted object
 */
declare enum ImageResolution {
  /**
   * 150 pixels per inch
   */
  PPI_150 = 1920151654,

  /**
   * 300 pixels per inch
   */
  PPI_300 = 1920160872,

  /**
   * 72 pixels per inch
   */
  PPI_72 = 1920160628,

  /**
   * 96 pixels per inch
   */
  PPI_96 = 1920159347,
}

/**
 * Type of Image alignment for objects
 */
declare enum ImageAlignmentType {
  /**
   * image will be aligned center
   */
  ALIGN_CENTER = 1097614194,

  /**
   * image will be aligned left
   */
  ALIGN_LEFT = 1097616486,

  /**
   * image will be aligned right
   */
  ALIGN_RIGHT = 1097618036,
}

/**
 * Image Page Break Options.
 */
declare enum ImagePageBreakType {
  /**
   * Page break after image.
   */
  PAGE_BREAK_AFTER = 1114792294,

  /**
   * Page break before image.
   */
  PAGE_BREAK_BEFORE = 1114792550,

  /**
   * Page break before and after image.
   */
  PAGE_BREAK_BEFORE_AND_AFTER = 1114792545,
}

/**
 * size type options.
 */
declare enum SizeTypeEnum {
  /**
   * Default size.
   */
  DEFAULT_SIZE = 1147491177,

  /**
   * Fixed size.
   */
  FIXED_SIZE = 1181317203,

  /**
   * None size.
   */
  NONE_SIZE = 1315925587,

  /**
   * Relative to text flow.
   */
  RELATIVE_TO_TEXT_FLOW = 1383289940,

  /**
   * Relative to text size.
   */
  RELATIVE_TO_TEXT_SIZE = 1383486579,

  /**
   * Use custom height.
   */
  USE_CUSTOM_HEIGHT = 1430472805,

  /**
   * Use custom width.
   */
  USE_CUSTOM_WIDTH = 1430476649,
}

/**
 * Preserve Appearance from Layout Options
 */
declare enum PreserveAppearanceFromLayoutEnum {
  /**
   * Export preferences will be honoured
   */
  PRESERVE_APPEARANCE_DEFAULT = 1349665893,

  /**
   * Container & content both will be rasterized, if can be
   */
  PRESERVE_APPEARANCE_RASTERIZE_CONTAINER = 1349669490,

  /**
   * Content will be rasterized, if can be
   */
  PRESERVE_APPEARANCE_RASTERIZE_CONTENT = 1349669492,

  /**
   * Exiting image will be used
   */
  PRESERVE_APPEARANCE_USE_EXISTING_IMAGE = 1349670245,
}

/**
 * Arrowhead alignment types.
 */
declare enum ArrowHeadAlignmentEnum {
  /**
   * The arrowhead is inside the path, path geometry changes to accomodate arrow heads.
   */
  INSIDE_PATH = 1634230633,

  /**
   * The arrowhead is outside the path i.e. path geometry remains same.
   */
  OUTSIDE_PATH = 1634230639,
}

/**
 * Options for specifying the reference position for X and Y attributes of style. If set none, it will disable the attribute.
 */
declare enum TransformPositionReference {
  /**
   * Corresponding edge of the page. Left edge for X attribute, Top edge for Y attribute.
   */
  PAGE_EDGE_REFERENCE = 1348945255,

  /**
   * Corresponding page margin of the page.Left margin for X attribute, Top margin for Y attribute.
   */
  PAGE_MARGIN_REFERENCE = 1883721063,
}

/**
 * Dimension attribute which you want to control.
 */
declare enum DimensionAttributes {
  /**
   * Both height and width of dimension.
   */
  BOTH_HEIGHT_WIDTH_ATTRIBUTE = 1700227170,

  /**
   * Height attribute of dimension.
   */
  HEIGHT_ATTRIBUTE = 1699247183,

  /**
   * Width attribute of dimension.
   */
  WIDTH_ATTRIBUTE = 1700226127,
}

/**
 * Position attribute which you want to control.
 */
declare enum PositionAttributes {
  /**
   * Both X and Y of position.
   */
  BOTH_X_Y_ATTRIBUTE = 1698855001,

  /**
   * X attribute of position.
   */
  X_ATTRIBUTE = 1700282735,

  /**
   * Y attribute of position.
   */
  Y_ATTRIBUTE = 1700348271,
}

/**
 * Performance metric options.
 */
declare enum PerformanceMetricOptions {
  /**
   * AGMXShowTime
   */
  AGMXSHOWTIME = 1095191924,

  /**
   * Number of attachable events that have been dispatched.
   */
  ATTACHABLE_EVENT_COUNT = 1095058292,

  /**
   * Number of attached scripts that have been executed.
   */
  ATTACHED_SCRIPTS_COUNT = 1095975796,

  /**
   * BIB Allocations
   */
  BIB_ALLOCATIONS = 1112097345,

  /**
   * BIB Allocations peak
   */
  BIB_ALLOCATIONS_PEAK = 1111576683,

  /**
   * BIB cache
   */
  BIB_CACHE = 1112097379,

  /**
   * BIB cache peak
   */
  BIB_CACHE_PEAK = 1111707755,

  /**
   * change manager update call count
   */
  CHANGE_MANAGER_UPDATE_CALL_COUNT = 1129137012,

  /**
   * change manager update call time
   */
  CHANGE_MANAGER_UPDATE_CALL_TIME = 1129141357,

  /**
   * The core allocation count.
   */
  CORE_ALLOCATION_COUNT = 1128361059,

  /**
   * The core memory size.
   */
  CORE_MEMORY_SIZE = 1129539962,

  /**
   * The CPU time.
   */
  CPU_TIME = 1668314484,

  /**
   * The current memory mark.
   */
  CURRENT_MEMORY_MARK = 1296921195,

  /**
   * database file bytes read
   */
  DATABASE_FILE_BYTES_READ = 1145197156,

  /**
   * database file bytes written
   */
  DATABASE_FILE_BYTES_WRITTEN = 1145198450,

  /**
   * database file page reads
   */
  DATABASE_FILE_PAGE_READS = 1145194098,

  /**
   * database file read time
   */
  DATABASE_FILE_READ_TIME = 1145197172,

  /**
   * database file write time
   */
  DATABASE_FILE_WRITE_TIME = 1145198452,

  /**
   * database instantiate count
   */
  DATABASE_INSTANTIATE_COUNT = 1145194862,

  /**
   * database new UID count
   */
  DATABASE_NEW_UID_COUNT = 1145197929,

  /**
   * Total amount of time spent dispatching attachable events.
   */
  DISPATCH_EVENT_TIME = 1145394285,

  /**
   * draw manager draw time
   */
  DRAW_MANAGER_DRAW_TIME = 1145918573,

  /**
   * draw manager number of interrupts
   */
  DRAW_MANAGER_NUMBER_OF_INTERRUPTS = 1145915758,

  /**
   * drop shadow file read bytes
   */
  DROP_SHADOW_FILE_READ_BYTES = 1145459298,

  /**
   * drop shadow file read time
   */
  DROP_SHADOW_FILE_READ_TIME = 1145459316,

  /**
   * drop shadow file write bytes
   */
  DROP_SHADOW_FILE_WRITE_BYTES = 1145460578,

  /**
   * drop shadow file write time
   */
  DROP_SHADOW_FILE_WRITE_TIME = 1145460596,

  /**
   * drop shadow memory read bytes
   */
  DROP_SHADOW_MEMORY_READ_BYTES = 1146311266,

  /**
   * drop shadow memory read time
   */
  DROP_SHADOW_MEMORY_READ_TIME = 1146311284,

  /**
   * drop shadow memory write bytes
   */
  DROP_SHADOW_MEMORY_WRITE_BYTES = 1146312546,

  /**
   * drop shadow memory write time
   */
  DROP_SHADOW_MEMORY_WRITE_TIME = 1146312564,

  /**
   * galley composition count
   */
  GALLEY_COMPOSITION_COUNT = 1195590516,

  /**
   * galley composition time
   */
  GALLEY_COMPOSITION_TIME = 1195594861,

  /**
   * The GDI object count.
   */
  GDI_OBJECT_COUNT = 1195657582,

  /**
   * The handle count.
   */
  HANDLE_COUNT = 1212378740,

  /**
   * heap allocations
   */
  HEAP_ALLOCATIONS = 1212247148,

  /**
   * heap allocations peak
   */
  HEAP_ALLOCATIONS_PEAK = 1212239979,

  /**
   * image cache allocations
   */
  IMAGE_CACHE_ALLOCATIONS = 1231897409,

  /**
   * image cache allocations peak
   */
  IMAGE_CACHE_ALLOCATIONS_PEAK = 1229144427,

  /**
   * image cache file bytes read
   */
  IMAGE_CACHE_FILE_BYTES_READ = 1231901284,

  /**
   * image cache file bytes written
   */
  IMAGE_CACHE_FILE_BYTES_WRITTEN = 1231902578,

  /**
   * image cache file read time
   */
  IMAGE_CACHE_FILE_READ_TIME = 1229148772,

  /**
   * image cache file write time
   */
  IMAGE_CACHE_FILE_WRITE_TIME = 1229150066,

  /**
   * instance cache purge count
   */
  INSTANCE_CACHE_PURGE_COUNT = 1229148259,

  /**
   * layout composition count
   */
  LAYOUT_COMPOSITION_COUNT = 1279476596,

  /**
   * layout composition time
   */
  LAYOUT_COMPOSITION_TIME = 1279480941,

  /**
   * memory purge count
   */
  MEMORY_PURGE_COUNT = 1297105780,

  /**
   * memory purge time
   */
  MEMORY_PURGE_TIME = 1297110125,

  /**
   * minisave count
   */
  MINISAVE_COUNT = 1297302388,

  /**
   * new snapshot time
   */
  NEW_SNAPSHOT_TIME = 1397651284,

  /**
   * The number of threads.
   */
  NUMBER_OF_THREADS = 1853122674,

  /**
   * The overall system CPU.
   */
  OVERALL_SYSTEM_CPU = 1399025781,

  /**
   * The overall user CPU.
   */
  OVERALL_USER_CPU = 1432580213,

  /**
   * PDF allocactions
   */
  PDF_ALLOCACTIONS = 1346651713,

  /**
   * PDF allocactions peak
   */
  PDF_ALLOCACTIONS_PEAK = 1346650475,

  /**
   * process IO bytes read
   */
  PROCESS_IO_BYTES_READ = 1229935204,

  /**
   * process IO bytes written
   */
  PROCESS_IO_BYTES_WRITTEN = 1229936498,

  /**
   * The resident memory size.
   */
  RESIDENT_MEMORY_SIZE = 1381198202,

  /**
   * snapshot count
   */
  SNAPSHOT_COUNT = 1397965684,

  /**
   * snapshot read write byte count
   */
  SNAPSHOT_READ_WRITE_BYTE_COUNT = 1397905251,

  /**
   * snapshot read write time
   */
  SNAPSHOT_READ_WRITE_TIME = 1397905268,

  /**
   * The virtual memory size.
   */
  VIRTUAL_MEMORY_SIZE = 1448307066,

  /**
   * VXferAlloc
   */
  VXFERALLOC = 1448633921,

  /**
   * VXferAllocPeak
   */
  VXFERALLOCPEAK = 1448624491,

  /**
   * VXferBytesRead
   */
  VXFERBYTESREAD = 1448633938,

  /**
   * VXferBytesWritten
   */
  VXFERBYTESWRITTEN = 1448633943,

  /**
   * VXferFileBytesRead
   */
  VXFERFILEBYTESREAD = 1447445106,

  /**
   * VXFerFileBytesWritten
   */
  VXFERFILEBYTESWRITTEN = 1447445111,

  /**
   * VXferReadTime
   */
  VXFERREADTIME = 1448628852,

  /**
   * VXferWriteTime
   */
  VXFERWRITETIME = 1448630132,

  /**
   * XMP filter time
   */
  XMP_FILTER_TIME = 1481461876,
}

/**
 * SWF export background options.
 */
declare enum SWFBackgroundOptions {
  /**
   * Paper color background.
   */
  PAPER_COLOR = 1935831139,

  /**
   * Transparent background.
   */
  TRANSPARENT_SWF_BACKGROUND = 1935828852,
}

/**
 * XFL rasterize format options.
 */
declare enum XFLRasterizeFormatOptions {
  /**
   * Uses the best format based on the image when rasterizes.
   */
  AUTOMATIC = 1768059764,

  /**
   * Uses GIF format when rasterizes.
   */
  GIF = 1734960742,

  /**
   * Uses JPEG format when rasterizes.
   */
  JPEG = 1785751398,

  /**
   * Uses PNG format when rasterizes.
   */
  PNG = 1397059687,
}

/**
 * The ease option.
 */
declare enum AnimationEaseOptions {
  /**
   * cutom read only ease type.
   */
  CUSTOM_EASE = 2051363407,

  /**
   * simple ease in.
   */
  EASE_IN = 2051371849,

  /**
   * simple ease in and out.
   */
  EASE_IN_OUT = 2051361103,

  /**
   * simple ease out.
   */
  EASE_OUT = 2051371855,

  /**
   * no easing.
   */
  NO_EASE = 2051960645,
}

/**
 * The event that triggers a list of dynamic target objects to play.
 */
declare enum DynamicTriggerEvents {
  /**
   * target is triggered on a button or self click.
   */
  ON_CLICK = 1953317740,

  /**
   * target is triggered on clicking on the page.
   */
  ON_PAGE_CLICK = 1953321027,

  /**
   * target is triggered on loading of the page.
   */
  ON_PAGE_LOAD = 1953321036,

  /**
   * target is triggered on a button release.
   */
  ON_RELEASE = 1953321580,

  /**
   * target is triggered on a button rolloff.
   */
  ON_ROLLOFF = 1953321574,

  /**
   * target is triggered on a button or self rollover.
   */
  ON_ROLLOVER = 1953321590,

  /**
   * target is triggered on self click.
   */
  ON_SELF_CLICK = 1951613804,

  /**
   * target is triggered on self rollover.
   */
  ON_SELF_ROLLOVER = 1951617638,

  /**
   * target is triggered on loading of the state in a multi-state object.
   */
  ON_STATE_LOAD = 1953321804,
}

/**
 * The page transition type options.
 */
declare enum PageTransitionTypeOptions {
  /**
   * The Blinds page transition.
   */
  BLINDS_TRANSITION = 1886667372,

  /**
   * The Blinds page transition.
   */
  BOX_TRANSITION = 1886667384,

  /**
   * The Comb page transition.
   */
  COMB_TRANSITION = 1886667618,

  /**
   * The Cover page transition.
   */
  COVER_TRANSITION = 1886667638,

  /**
   * The Dissolve page transition.
   */
  DISSOLVE_TRANSITION = 1886667891,

  /**
   * The Fade page transition.
   */
  FADE_TRANSITION = 1886668388,

  /**
   * No page transition applied.
   */
  NONE = 1852796517,

  /**
   * The Page Turn page transition.
   */
  PAGE_TURN_TRANSITION = 1886670932,

  /**
   * The Push page transition.
   */
  PUSH_TRANSITION = 1886670963,

  /**
   * The Split page transition.
   */
  SPLIT_TRANSITION = 1886671728,

  /**
   * The Uncover page transition.
   */
  UNCOVER_TRANSITION = 1886672227,

  /**
   * The Wipe page transition.
   */
  WIPE_TRANSITION = 1886672752,

  /**
   * The Zoom In page transition.
   */
  ZOOM_IN_TRANSITION = 1886673481,

  /**
   * The Zoom Out page transition.
   */
  ZOOM_OUT_TRANSITION = 1886673487,
}

/**
 * The page transition direction options.
 */
declare enum PageTransitionDirectionOptions {
  /**
   * The top to bottom direction.
   */
  DOWN = 1181971556,

  /**
   * The horizontal direction.
   */
  HORIZONTAL = 1752134266,

  /**
   * The horizontal inward direction.
   */
  HORIZONTAL_IN = 1886668873,

  /**
   * The horizontal outward direction.
   */
  HORIZONTAL_OUT = 1886668879,

  /**
   * The inward direction.
   */
  IN = 1768824864,

  /**
   * The down and left direction.
   */
  LEFT_DOWN = 1886669892,

  /**
   * The left to right direction.
   */
  LEFT_TO_RIGHT = 1819570786,

  /**
   * The up and left direction.
   */
  LEFT_UP = 1886669909,

  /**
   * Direction does not apply.
   */
  NOT_APPLICABLE = 1886670401,

  /**
   * The outward direction.
   */
  OUT = 1886670708,

  /**
   * The down and right direction.
   */
  RIGHT_DOWN = 1886671428,

  /**
   * The right to left direction.
   */
  RIGHT_TO_LEFT = 1920232546,

  /**
   * The up and right direction.
   */
  RIGHT_UP = 1886671445,

  /**
   * The bottom to top direction.
   */
  UP = 1181971566,

  /**
   * The vertical direction.
   */
  VERTICAL = 1986359924,

  /**
   * The vertical inward direction.
   */
  VERTICAL_IN = 1886672457,

  /**
   * The vertical outward direction.
   */
  VERTICAL_OUT = 1886672463,
}

/**
 * SWF raster compression options.
 */
declare enum RasterCompressionOptions {
  /**
   * Uses JPEG compression and automatically determines the best quality type.
   */
  AUTOMATIC_COMPRESSION = 1936875875,

  /**
   * Uses JPEG compression.
   */
  JPEG_COMPRESSION = 1936878179,

  /**
   * Uses the best quality type.
   */
  LOSSLESS_COMPRESSION = 1936878691,
}

/**
 * Dynamic documents JPEG Quality options.
 */
declare enum DynamicDocumentsJPEGQualityOptions {
  /**
   * Uses high JPEG compression.
   */
  HIGH = 1701726313,

  /**
   * Uses low JPEG compression.
   */
  LOW = 1701727351,

  /**
   * Uses maximum JPEG compression.
   */
  MAXIMUM = 1701727608,

  /**
   * Uses medium JPEG compression.
   */
  MEDIUM = 1701727588,

  /**
   * Uses minimum JPEG compression.
   */
  MINIMUM = 1701727598,
}

/**
 * Dynamic documents text export policy.
 */
declare enum DynamicDocumentsTextExportPolicy {
  /**
   * Text is exported as live text.
   */
  LIVE = 1952804972,

  /**
   * Text is to be rasterized.
   */
  RASTERIZE = 1952804978,

  /**
   * Text is exported as Text Layout Framework text.
   */
  TLF = 1952804980,

  /**
   * Text is to be vectorized.
   */
  VECTORIZE = 1952804982,
}

/**
 * Fit method settings.
 */
declare enum FitMethodSettings {
  /**
   * Fit to given scale percentage.
   */
  FIT_GIVEN_SCALE_PERCENTAGE = 1684304739,

  /**
   * Fit to given width and height.
   */
  FIT_GIVEN_WIDTH_AND_HEIGHT = 1684305768,

  /**
   * Fit to predefined settings.
   */
  FIT_PREDEFINED_SETTINGS = 1684301427,
}

/**
 * Options for fitting to dimension.
 */
declare enum FitDimension {
  /**
   * Fit to 1024x768 dimension.
   */
  FIT1024X768 = 1718906726,

  /**
   * Fit to 1240x620 dimension.
   */
  FIT1240X620 = 1718908023,

  /**
   * Fit to 1280x800 dimension.
   */
  FIT1280X800 = 1718906725,

  /**
   * Fit to 600x300 dimension.
   */
  FIT600X300 = 1718907764,

  /**
   * Fit to 640x480 dimension.
   */
  FIT640X480 = 1718907753,

  /**
   * Fit to 760x420 dimension.
   */
  FIT760X420 = 1718907750,

  /**
   * Fit to 800x600 dimension.
   */
  FIT800X600 = 1718904179,

  /**
   * Fit to 984x588 dimension.
   */
  FIT984X588 = 1718906470,
}

/**
 * The page transition duration options.
 */
declare enum PageTransitionDurationOptions {
  /**
   * Fast duration.
   */
  FAST = 1180791668,

  /**
   * Medium duration.
   */
  MEDIUM = 1701727588,

  /**
   * Slow duration.
   */
  SLOW = 1886671692,
}

/**
 * SWF curve quality values.
 */
declare enum SWFCurveQualityValue {
  /**
   * High curve quality.
   */
  HIGH = 1701726313,

  /**
   * Low curve quality.
   */
  LOW = 1701727351,

  /**
   * Maximum curve quality.
   */
  MAXIMUM = 1701727608,

  /**
   * Medium curve quality.
   */
  MEDIUM = 1701727588,

  /**
   * Minimum curve quality.
   */
  MINIMUM = 1701727598,
}

/**
 * Dynamic media handling options.
 */
declare enum DynamicMediaHandlingOptions {
  /**
   * Draw interactive elements appearance only.
   */
  APPEARANCE_ONLY = 1097887823,

  /**
   * Include all interactive elements.
   */
  INCLUDE_ALL_MEDIA = 1231241580,
}

/**
 * The animation design options.
 */
declare enum DesignOptions {
  /**
   * Uses the current object's properties as the starting appearance of the animation at runtime.
   */
  FROM_CURRENT_APPEARANCE = 1634551405,

  /**
   * Uses the current object's properties as the end appearance of the animation at runtime.
   */
  TO_CURRENT_APPEARANCE = 1634554991,

  /**
   * Uses the current object's properties as the starting appearance, and current position as the end location of the animation at runtime.
   */
  TO_CURRENT_LOCATION = 1634553702,
}

/**
 * Options for the condition indicator method.
 */
declare enum ConditionIndicatorMethod {
  /**
   * Highlights conditional text.
   */
  USE_HIGHLIGHT = 1699244391,

  /**
   * Underlines conditional text.
   */
  USE_UNDERLINE = 1700097644,
}

/**
 * Options for the condition underline indicator appearance.
 */
declare enum ConditionUnderlineIndicatorAppearance {
  /**
   * Dashed underline.
   */
  DASHED = 1684108136,

  /**
   * Solid underline.
   */
  SOLID = 1936682084,

  /**
   * Wavy underline.
   */
  WAVY = 1937208953,
}

/**
 * Condition indicator mode options.
 */
declare enum ConditionIndicatorMode {
  /**
   * Conditions indicators hide.
   */
  HIDE_INDICATORS = 1698908520,

  /**
   * Conditions indicators show and print.
   */
  SHOW_AND_PRINT_INDICATORS = 1698908528,

  /**
   * Condition indicators show only.
   */
  SHOW_INDICATORS = 1698908531,
}

/**
 * Style Type
 */
declare enum StyleType {
  /**
   * Character Style
   */
  CHARACTER_STYLE_TYPE = 1665684340,

  /**
   * Paragraph Style
   */
  PARAGRAPH_STYLE_TYPE = 1883730548,
}

/**
 * Text Directions.
 */
declare enum TextDirection {
  /**
   * Horizontal Text Direction
   */
  HORIZONTAL_TYPE = 1702126696,

  /**
   * Mixed Text Direction
   */
  MIXED_TYPE = 1702126701,

  /**
   * Unknown Text Direction
   */
  UNKNOWN_TYPE = 1702126709,

  /**
   * Vertical Text Direction
   */
  VERTICAL_TYPE = 1702126710,
}

/**
 * Text to consider in case of threaded text frames.
 */
declare enum ThreadedTextFrameTextOptions {
  /**
   * Text for the complete story
   */
  TEXT_FOR_COMPLETE_STORY = 1698911092,

  /**
   * Text within current text frame
   */
  TEXT_WITHIN_TEXTFRAME = 1700021844,
}

/**
 * The smooth scrolling option.
 */
declare enum SmoothScrollingOptions {
  /**
   * horizontal smooth scrolling.
   */
  HORIZONTAL = 1752134266,

  /**
   * no smooth scrolling.
   */
  NO_SMOOTH_SCROLL = 1699959662,

  /**
   * vertical smooth scrolling.
   */
  VERTICAL = 1986359924,

  /**
   * vertical and horizontal smooth scrolling.
   */
  VERTICAL_AND_HORIZONTAL = 1699959650,
}

/**
 * The folio orientation option.
 */
declare enum FolioOrientationOptions {
  /**
   * automatic - determine orientation based on the orientation of the mini folios.
   */
  AUTO = 1635019116,

  /**
   * landscape orientation only.
   */
  LANDSCAPE = 2003395685,

  /**
   * portrait orientation only.
   */
  PORTRAIT = 1751738216,

  /**
   * both portrait and landscape orientations.
   */
  PORTRAIT_AND_LANDSCAPE = 1699303266,
}

/**
 * The folio binding direction option.
 */
declare enum FolioBindingDirectionOptions {
  /**
   * left-edge binding.
   */
  LEFT = 1835102828,

  /**
   * right-edge binding.
   */
  RIGHT = 1835102834,
}

/**
 * The versions that are available.
 */
declare enum DigpubVersion {
  /**
   * plugin, folio and plist versions in that order.
   */
  ALL = 1634495520,

  /**
   * folio version.
   */
  FOLIO = 1685087862,

  /**
   * plist version.
   */
  PLIST = 1685090422,

  /**
   * plugin version.
   */
  PLUGIN = 1685090412,
}

/**
 * The device types that are attached.
 */
declare enum AttachedDevices {
  /**
   * all devices, Android followed by iOS.
   */
  ALL = 1634495520,

  /**
   * Android devices.
   */
  ANDROID = 1685086564,

  /**
   * iOS devices.
   */
  IOS = 1684631411,
}

/**
 * The version of the plugin and article that are available.
 */
declare enum DigpubArticleVersion {
  /**
   * plugin and article versions in that order.
   */
  ALL = 1634495520,

  /**
   * article version.
   */
  ARTICLE = 1685078390,

  /**
   * plugin version.
   */
  PLUGIN = 1685090412,
}

/**
 * Choices for export order of epub and html.
 */
declare enum ExportOrder {
  /**
   * based on article defined in article panel.
   */
  ARTICLE_PANEL_ORDER = 1700946288,

  /**
   * based on document layout.
   */
  LAYOUT_ORDER = 1700949113,

  /**
   * based on XML structure.
   */
  XML_STRUCTURE_ORDER = 1700952179,
}

/**
 * EPub export option for cover image.
 */
declare enum EpubCover {
  /**
   * Use external image as cover image.
   */
  EXTERNAL_IMAGE = 1700952169,

  /**
   * Rasterize first page as cover image.
   */
  FIRST_PAGE = 1700947536,

  /**
   * no cover image.
   */
  NONE = 1852796517,
}

/**
 * Options for export unorder list.
 */
declare enum BulletListExportOption {
  /**
   * convert to text.
   */
  AS_TEXT = 1700946804,

  /**
   * map to html unordered list.
   */
  UNORDERED_LIST = 1700949333,
}

/**
 * Options for export order list.
 */
declare enum NumberedListExportOption {
  /**
   * convert to text.
   */
  AS_TEXT = 1700946804,

  /**
   * map to html ordered list.
   */
  ORDERED_LIST = 1700949359,
}

/**
 * EPub export option for epub version.
 */
declare enum EpubVersion {
  /**
   * EPUB 2.0.1.
   */
  EPUB2 = 1702257970,

  /**
   * EPUB 3.0.
   */
  EPUB3 = 1702257971,
}

/**
 * Choices for export image.
 */
declare enum ImageExportOption {
  /**
   * link to server.
   */
  LINK_TO_SERVER = 1700949107,

  /**
   * optimized image.
   */
  OPTIMIZED_IMAGE = 1700949872,

  /**
   * original image.
   */
  ORIGINAL_IMAGE = 1700949874,
}

/**
 * Choices for footnote placement.
 */
declare enum EPubFootnotePlacement {
  /**
   * Footnote after paragraph.
   */
  FOOTNOTE_AFTER_PARAGRAPH = 1701213296,

  /**
   * Footnote after story.
   */
  FOOTNOTE_AFTER_STORY = 1701213267,

  /**
   * Footnote inside popup.
   */
  FOOTNOTE_INSIDE_POPUP = 1701213235,
}

/**
 * Choices for page range format for export.
 */
declare enum PageRangeFormat {
  /**
   * export all pages.
   */
  EXPORT_ALL_PAGES = 1700883568,

  /**
   * export page ranges.
   */
  EXPORT_PAGE_RANGE = 1700951410,
}

/**
 * Choices for epub navigation style.
 */
declare enum EpubNavigationStyle {
  /**
   * Bookmarks based navigation
   */
  BOOKMARKS_NAVIGATION = 1700949622,

  /**
   * File name based navigation
   */
  FILENAME_NAVIGATION = 1701211766,

  /**
   * No navigation.
   */
  NO_NAVIGATION = 1701736054,

  /**
   * TOC style based navigation
   */
  TOC_STYLE_NAVIGATION = 1702129270,
}

/**
 * Choices for spread control for fixed layout EPub.
 */
declare enum EpubFixedLayoutSpreadControl {
  /**
   * No spreads.
   */
  NO_SPREADS = 1702063727,

  /**
   * Physical spreads.
   */
  PHYSICAL_SPREADS = 1701865593,

  /**
   * Spreads based on document.
   */
  SPREADS_BASED_ON_DOC = 1700949860,

  /**
   * Synthetic spreads.
   */
  SYNTHETIC_SPREADS = 1702066542,
}

/**
 * Choices for publish format.
 */
declare enum PublishFormatEnum {
  /**
   * publish by pages.
   */
  PUBLISH_BY_PAGES = 1700950134,

  /**
   * publish by spread.
   */
  PUBLISH_BY_SPREAD = 1700950902,
}

/**
 * publish export option for cover image.
 */
declare enum PublishCoverEnum {
  /**
   * Rasterize chosen page as cover image.
   */
  CHOSEN_PAGE = 1701013072,

  /**
   * Use external image as cover image.
   */
  EXTERNAL_IMAGE = 1700952169,

  /**
   * Rasterize first page as cover image.
   */
  FIRST_PAGE = 1700947536,
}

/**
 * Mapping type for style mappings.
 */
declare enum MapType {
  /**
   * group to group mapping rule.
   */
  GROUP_MAPPING_RULE = 1735681906,

  /**
   * group to style mapping rule.
   */
  GROUP_TO_STYLE_MAPPING_RULE = 1735684978,

  /**
   * style to style mapping rule.
   */
  STYLE_MAPPING_RULE = 1937011570,

  /**
   * style to group mapping rule.
   */
  STYLE_TO_GROUP_MAPPING_RULE = 1937008498,
}

/**
 * Paragraph direction.
 */
declare enum ParagraphDirectionOptions {
  /**
   * Left to Right paragraph direction
   */
  LEFT_TO_RIGHT_DIRECTION = 1278366308,

  /**
   * Right to Left paragraph direction
   */
  RIGHT_TO_LEFT_DIRECTION = 1379028068,
}

/**
 * Paragraph justification.
 */
declare enum ParagraphJustificationOptions {
  /**
   * Arabic justification
   */
  ARABIC_JUSTIFICATION = 1886019954,

  /**
   * Default justification
   */
  DEFAULT_JUSTIFICATION = 1886020709,

  /**
   * Naskh justification
   */
  NASKH_JUSTIFICATION = 1886023265,

  /**
   * Kashidas. Use naskh justification if you want to also use Justifcation Alternates.
   */
  NASKH_KASHIDA_JUSTIFICATION = 1886023275,

  /**
   * Fractional Kashidas. Use naskh justification if you want to also use Justifcation Alternates.
   */
  NASKH_KASHIDA_JUSTIFICATION_FRAC = 1886021227,

  /**
   * Kashidas without Stretched Connections.
   */
  NASKH_TATWEEL_JUSTIFICATION = 1886023284,

  /**
   * Fractional Kashidas without Stretched Connections.
   */
  NASKH_TATWEEL_JUSTIFICATION_FRAC = 1886021236,
}

/**
 * Character direction.
 */
declare enum CharacterDirectionOptions {
  /**
   * Default direction
   */
  DEFAULT_DIRECTION = 1147496036,

  /**
   * Left to right direction
   */
  LEFT_TO_RIGHT_DIRECTION = 1278366308,

  /**
   * Right to left direction
   */
  RIGHT_TO_LEFT_DIRECTION = 1379028068,
}

/**
 * Digits type options.
 */
declare enum DigitsTypeOptions {
  /**
   * Arabic digits
   */
  ARABIC_DIGITS = 1684627826,

  /**
   * Bengali digits
   */
  BENGALI_DIGITS = 1684628069,

  /**
   * Burmese digits
   */
  BURMESE_DIGITS = 1684628085,

  /**
   * Default digits
   */
  DEFAULT_DIGITS = 1684628581,

  /**
   * Devanagari digits
   */
  DEVANAGARI_DIGITS = 1684628598,

  /**
   * Farsi digits
   */
  FARSI_DIGITS = 1684629089,

  /**
   * Full Farsi digits
   */
  FULL_FARSI_DIGITS = 1684629094,

  /**
   * Gujarati digits
   */
  GUJARATI_DIGITS = 1684629354,

  /**
   * Gurmukhi digits
   */
  GURMUKHI_DIGITS = 1684629357,

  /**
   * Hindi digits
   */
  HINDI_DIGITS = 1684629609,

  /**
   * Kannada digits
   */
  KANNADA_DIGITS = 1684630369,

  /**
   * Khmer digits
   */
  KHMER_DIGITS = 1684630376,

  /**
   * Lao digits
   */
  LAO_DIGITS = 1684630625,

  /**
   * Malayalam digits
   */
  MALAYALAM_DIGITS = 1684630881,

  /**
   * native digits
   */
  NATIVE_DIGITS = 1684631137,

  /**
   * Oriya digits
   */
  ORIYA_DIGITS = 1684631410,

  /**
   * Tamil digits
   */
  TAMIL_DIGITS = 1684632673,

  /**
   * Telugu digits
   */
  TELUGU_DIGITS = 1684632677,

  /**
   * Thai digits
   */
  THAI_DIGITS = 1684632680,

  /**
   * Tibetan digits
   */
  TIBETAN_DIGITS = 1684632681,
}

/**
 * Kashidas.
 */
declare enum KashidasOptions {
  /**
   * Default kashidas
   */
  DEFAULT_KASHIDAS = 1801544805,

  /**
   * Kashidas off
   */
  KASHIDAS_OFF = 1801547622,
}

/**
 * Diacritic position.
 */
declare enum DiacriticPositionOptions {
  /**
   * Default position
   */
  DEFAULT_POSITION = 1685090150,

  /**
   * Loose position
   */
  LOOSE_POSITION = 1685089391,

  /**
   * Medium position
   */
  MEDIUM_POSITION = 1685089637,

  /**
   * OpenType position
   */
  OPENTYPE_POSITION = 1685090164,

  /**
   * OpenType position from baseline
   */
  OPENTYPE_POSITION_FROM_BASELINE = 1685090146,

  /**
   * Tight position
   */
  TIGHT_POSITION = 1685091433,
}

/**
 * Table direction options.
 */
declare enum TableDirectionOptions {
  /**
   * Set left to right table direction
   */
  LEFT_TO_RIGHT_DIRECTION = 1278366308,

  /**
   * Set right to left table direction
   */
  RIGHT_TO_LEFT_DIRECTION = 1379028068,
}

/**
 * A hyperlink.
 */
declare class Hyperlink {
  /**
   * The hyperlink border color.
   */
  borderColor: [number, number, number] | UIColors

  /**
   * The hyperlink border style.
   */
  borderStyle: HyperlinkAppearanceStyle

  /**
   * The text, page, or URL that the hyperlink points to.
   */
  destination:
    | HyperlinkTextDestination
    | HyperlinkPageDestination
    | HyperlinkExternalPageDestination
    | HyperlinkURLDestination
    | ParagraphDestination

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * If true, the hyperlink is hidden.
   */
  readonly hidden: boolean

  /**
   * The hyperlink highlight style.
   */
  highlight: HyperlinkAppearanceHighlight

  /**
   * The unique ID of the Hyperlink.
   */
  readonly id: number

  /**
   * The index of the Hyperlink within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The name of the Hyperlink.
   */
  name: string

  /**
   * The parent of the Hyperlink (a Document).
   */
  readonly parent: Document

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * The hyperlinked text or page item.
   */
  source: HyperlinkPageItemSource | HyperlinkTextSource | CrossReferenceSource

  /**
   * If true, the Hyperlink is visible.
   */
  visible: boolean

  /**
   * The stroke weight of the hyperlink border.
   */
  width: HyperlinkAppearanceWidth

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): Hyperlink[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Deletes the Hyperlink.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Jumps to the hyperlink destination.
   */
  showDestination(): void

  /**
   * Jumps to the hyperlink source.
   */
  showSource(): void

  /**
   * Generates a string which, if executed, will return the Hyperlink.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of hyperlinks.
 */
declare class Hyperlinks {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Hyperlink with the specified index.
   * @param index The index.
   */
  [index: number]: Hyperlink

  /**
   * Creates a new hyperlink.
   * @param hyperlinkSource The hyperlinked object.
   * @param hyperlinkDestination The destination that the hyperlink points to. Can accept: Ordered array containing fileName:String, volumn:String, directoryId:Long Integer, dataLinkClassId:Long Integer, destinationUid:Long Integer, HyperlinkTextDestination, HyperlinkPageDestination, HyperlinkExternalPageDestination, HyperlinkURLDestination or ParagraphDestination.
   * @param withProperties Initial values for properties of the new Hyperlink
   */
  add(
    hyperlinkSource: HyperlinkPageItemSource | HyperlinkTextSource | CrossReferenceSource,
    hyperlinkDestination: any,
    withProperties: object,
  ): Hyperlink

  /**
   * Returns any Hyperlink in the collection.
   */
  anyItem(): Hyperlink

  /**
   * Displays the number of elements in the Hyperlink.
   */
  count(): number

  /**
   * Returns every Hyperlink in the collection.
   */
  everyItem(): Hyperlink

  /**
   * Returns the first Hyperlink in the collection.
   */
  firstItem(): Hyperlink

  /**
   * Returns the Hyperlink with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Hyperlink

  /**
   * Returns the Hyperlink with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Hyperlink

  /**
   * Returns the Hyperlink with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Hyperlink

  /**
   * Returns the Hyperlinks within the specified range.
   * @param from The Hyperlink, index, or name at the beginning of the range.
   * @param to The Hyperlink, index, or name at the end of the range.
   */
  itemByRange(from: Hyperlink | number | string, to: Hyperlink | number | string): Hyperlink[]

  /**
   * Returns the last Hyperlink in the collection.
   */
  lastItem(): Hyperlink

  /**
   * Returns the middle Hyperlink in the collection.
   */
  middleItem(): Hyperlink

  /**
   * Returns the Hyperlink whose index follows the specified Hyperlink in the collection.
   * @param obj The Hyperlink whose index comes before the desired Hyperlink.
   */
  nextItem(obj: Hyperlink): Hyperlink

  /**
   * Returns the Hyperlink with the index previous to the specified index.
   * @param obj The index of the Hyperlink that follows the desired Hyperlink.
   */
  previousItem(obj: Hyperlink): Hyperlink

  /**
   * Generates a string which, if executed, will return the Hyperlink.
   */
  toSource(): string
}

/**
 * A bookmark.
 */
declare class Bookmark {
  /**
   * A collection of bookmarks.
   */
  readonly bookmarks: Bookmarks

  /**
   * The destination that the hyperlink points to.
   */
  readonly destination:
    | HyperlinkTextDestination
    | HyperlinkPageDestination
    | HyperlinkExternalPageDestination
    | Page

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The unique ID of the Bookmark.
   */
  readonly id: number

  /**
   * The indent level of the bookmark.
   */
  readonly indent: number

  /**
   * The index of the Bookmark within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The name of the Bookmark.
   */
  name: string

  /**
   * The parent of the Bookmark (a Document or Bookmark).
   */
  readonly parent: any

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): Bookmark[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Moves the bookmark to the specified location.
   * @param to The bookmark location relative to the reference object or within the containing object.
   * @param reference The reference object. Note: Required when the to parameter specifies before or after.
   */
  move(to?: LocationOptions, reference?: Bookmark | Document): Bookmark

  /**
   * Deletes the Bookmark.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Go to the bookmark.
   */
  showBookmark(): void

  /**
   * Generates a string which, if executed, will return the Bookmark.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of bookmarks.
 */
declare class Bookmarks {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Bookmark with the specified index.
   * @param index The index.
   */
  [index: number]: Bookmark

  /**
   * Creates a new bookmark.
   * @param destination The bookmark destination.
   * @param withProperties Initial values for properties of the new Bookmark
   */
  add(
    destination:
      | HyperlinkTextDestination
      | HyperlinkPageDestination
      | HyperlinkExternalPageDestination
      | Page,
    withProperties: object,
  ): Bookmark

  /**
   * Returns any Bookmark in the collection.
   */
  anyItem(): Bookmark

  /**
   * Displays the number of elements in the Bookmark.
   */
  count(): number

  /**
   * Returns every Bookmark in the collection.
   */
  everyItem(): Bookmark

  /**
   * Returns the first Bookmark in the collection.
   */
  firstItem(): Bookmark

  /**
   * Returns the Bookmark with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Bookmark

  /**
   * Returns the Bookmark with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Bookmark

  /**
   * Returns the Bookmark with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Bookmark

  /**
   * Returns the Bookmarks within the specified range.
   * @param from The Bookmark, index, or name at the beginning of the range.
   * @param to The Bookmark, index, or name at the end of the range.
   */
  itemByRange(from: Bookmark | number | string, to: Bookmark | number | string): Bookmark[]

  /**
   * Returns the last Bookmark in the collection.
   */
  lastItem(): Bookmark

  /**
   * Returns the middle Bookmark in the collection.
   */
  middleItem(): Bookmark

  /**
   * Returns the Bookmark whose index follows the specified Bookmark in the collection.
   * @param obj The Bookmark whose index comes before the desired Bookmark.
   */
  nextItem(obj: Bookmark): Bookmark

  /**
   * Returns the Bookmark with the index previous to the specified index.
   * @param obj The index of the Bookmark that follows the desired Bookmark.
   */
  previousItem(obj: Bookmark): Bookmark

  /**
   * Generates a string which, if executed, will return the Bookmark.
   */
  toSource(): string
}

/**
 * A hyperlinked page item.
 */
declare class HyperlinkPageItemSource {
  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * If true, the hyperlink is hidden.
   */
  readonly hidden: boolean

  /**
   * The unique ID of the HyperlinkPageItemSource.
   */
  readonly id: number

  /**
   * The index of the HyperlinkPageItemSource within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The name of the HyperlinkPageItemSource.
   */
  name: string

  /**
   * The parent of the HyperlinkPageItemSource (a Document).
   */
  readonly parent: Document

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * The hyperlinked page item.
   */
  sourcePageItem: PageItem

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): HyperlinkPageItemSource[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Deletes the HyperlinkPageItemSource.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Jumps to the hyperlink source.
   */
  showSource(): void

  /**
   * Generates a string which, if executed, will return the HyperlinkPageItemSource.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of hyperlink page item sources.
 */
declare class HyperlinkPageItemSources {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the HyperlinkPageItemSource with the specified index.
   * @param index The index.
   */
  [index: number]: HyperlinkPageItemSource

  /**
   * Creates a new hyperlink page item source.
   * @param source The page item to hyperlink.
   * @param withProperties Initial values for properties of the new HyperlinkPageItemSource
   */
  add(source: PageItem, withProperties: object): HyperlinkPageItemSource

  /**
   * Returns any HyperlinkPageItemSource in the collection.
   */
  anyItem(): HyperlinkPageItemSource

  /**
   * Displays the number of elements in the HyperlinkPageItemSource.
   */
  count(): number

  /**
   * Returns every HyperlinkPageItemSource in the collection.
   */
  everyItem(): HyperlinkPageItemSource

  /**
   * Returns the first HyperlinkPageItemSource in the collection.
   */
  firstItem(): HyperlinkPageItemSource

  /**
   * Returns the HyperlinkPageItemSource with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): HyperlinkPageItemSource

  /**
   * Returns the HyperlinkPageItemSource with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): HyperlinkPageItemSource

  /**
   * Returns the HyperlinkPageItemSource with the specified name.
   * @param name The name.
   */
  itemByName(name: string): HyperlinkPageItemSource

  /**
   * Returns the HyperlinkPageItemSources within the specified range.
   * @param from The HyperlinkPageItemSource, index, or name at the beginning of the range.
   * @param to The HyperlinkPageItemSource, index, or name at the end of the range.
   */
  itemByRange(
    from: HyperlinkPageItemSource | number | string,
    to: HyperlinkPageItemSource | number | string,
  ): HyperlinkPageItemSource[]

  /**
   * Returns the last HyperlinkPageItemSource in the collection.
   */
  lastItem(): HyperlinkPageItemSource

  /**
   * Returns the middle HyperlinkPageItemSource in the collection.
   */
  middleItem(): HyperlinkPageItemSource

  /**
   * Returns the HyperlinkPageItemSource whose index follows the specified HyperlinkPageItemSource in the collection.
   * @param obj The HyperlinkPageItemSource whose index comes before the desired HyperlinkPageItemSource.
   */
  nextItem(obj: HyperlinkPageItemSource): HyperlinkPageItemSource

  /**
   * Returns the HyperlinkPageItemSource with the index previous to the specified index.
   * @param obj The index of the HyperlinkPageItemSource that follows the desired HyperlinkPageItemSource.
   */
  previousItem(obj: HyperlinkPageItemSource): HyperlinkPageItemSource

  /**
   * Generates a string which, if executed, will return the HyperlinkPageItemSource.
   */
  toSource(): string
}

/**
 * A hyperlinked text object or insertion point.
 */
declare class HyperlinkTextSource {
  /**
   * Character style of the hyperlink text source.
   */
  appliedCharacterStyle: CharacterStyle

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * If true, the hyperlink is hidden.
   */
  readonly hidden: boolean

  /**
   * The unique ID of the HyperlinkTextSource.
   */
  readonly id: number

  /**
   * The index of the HyperlinkTextSource within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The name of the HyperlinkTextSource.
   */
  name: string

  /**
   * The parent of the HyperlinkTextSource (a Document).
   */
  readonly parent: Document

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * The hyperlinked text or insertion point.
   */
  sourceText: Text

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): HyperlinkTextSource[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Deletes the HyperlinkTextSource.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Jumps to the hyperlink source.
   */
  showSource(): void

  /**
   * Generates a string which, if executed, will return the HyperlinkTextSource.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of hyperlink text sources.
 */
declare class HyperlinkTextSources {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the HyperlinkTextSource with the specified index.
   * @param index The index.
   */
  [index: number]: HyperlinkTextSource

  /**
   * Creates a new hyperlink text source.
   * @param source The text or insertion point to hyperlink.
   * @param withProperties Initial values for properties of the new HyperlinkTextSource
   */
  add(source: Text, withProperties: object): HyperlinkTextSource

  /**
   * Returns any HyperlinkTextSource in the collection.
   */
  anyItem(): HyperlinkTextSource

  /**
   * Displays the number of elements in the HyperlinkTextSource.
   */
  count(): number

  /**
   * Returns every HyperlinkTextSource in the collection.
   */
  everyItem(): HyperlinkTextSource

  /**
   * Returns the first HyperlinkTextSource in the collection.
   */
  firstItem(): HyperlinkTextSource

  /**
   * Returns the HyperlinkTextSource with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): HyperlinkTextSource

  /**
   * Returns the HyperlinkTextSource with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): HyperlinkTextSource

  /**
   * Returns the HyperlinkTextSource with the specified name.
   * @param name The name.
   */
  itemByName(name: string): HyperlinkTextSource

  /**
   * Returns the HyperlinkTextSources within the specified range.
   * @param from The HyperlinkTextSource, index, or name at the beginning of the range.
   * @param to The HyperlinkTextSource, index, or name at the end of the range.
   */
  itemByRange(
    from: HyperlinkTextSource | number | string,
    to: HyperlinkTextSource | number | string,
  ): HyperlinkTextSource[]

  /**
   * Returns the last HyperlinkTextSource in the collection.
   */
  lastItem(): HyperlinkTextSource

  /**
   * Returns the middle HyperlinkTextSource in the collection.
   */
  middleItem(): HyperlinkTextSource

  /**
   * Returns the HyperlinkTextSource whose index follows the specified HyperlinkTextSource in the collection.
   * @param obj The HyperlinkTextSource whose index comes before the desired HyperlinkTextSource.
   */
  nextItem(obj: HyperlinkTextSource): HyperlinkTextSource

  /**
   * Returns the HyperlinkTextSource with the index previous to the specified index.
   * @param obj The index of the HyperlinkTextSource that follows the desired HyperlinkTextSource.
   */
  previousItem(obj: HyperlinkTextSource): HyperlinkTextSource

  /**
   * Generates a string which, if executed, will return the HyperlinkTextSource.
   */
  toSource(): string
}

/**
 * A hyperlink destination that is either text or an insertion point.
 */
declare class HyperlinkTextDestination {
  /**
   * The text or insertion point that the hyperlink points to.
   */
  destinationText: InsertionPoint | Text

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * If true, the hyperlink is hidden.
   */
  readonly hidden: boolean

  /**
   * The unique ID of the HyperlinkTextDestination.
   */
  readonly id: number

  /**
   * The index of the HyperlinkTextDestination within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The name of the HyperlinkTextDestination.
   */
  name: string

  /**
   * The parent of the HyperlinkTextDestination (a Document).
   */
  readonly parent: Document

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): HyperlinkTextDestination[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Deletes the HyperlinkTextDestination.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Jumps to the hyperlink destination.
   */
  showDestination(): void

  /**
   * Generates a string which, if executed, will return the HyperlinkTextDestination.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of hyperlink text destinations.
 */
declare class HyperlinkTextDestinations {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the HyperlinkTextDestination with the specified index.
   * @param index The index.
   */
  [index: number]: HyperlinkTextDestination

  /**
   * Creates a new hyperlink text destination.
   * @param destination The text or insertion point location that the hyperlink points to.
   * @param withProperties Initial values for properties of the new HyperlinkTextDestination
   */
  add(destination: Text, withProperties: object): HyperlinkTextDestination

  /**
   * Returns any HyperlinkTextDestination in the collection.
   */
  anyItem(): HyperlinkTextDestination

  /**
   * Displays the number of elements in the HyperlinkTextDestination.
   */
  count(): number

  /**
   * Returns every HyperlinkTextDestination in the collection.
   */
  everyItem(): HyperlinkTextDestination

  /**
   * Returns the first HyperlinkTextDestination in the collection.
   */
  firstItem(): HyperlinkTextDestination

  /**
   * Returns the HyperlinkTextDestination with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): HyperlinkTextDestination

  /**
   * Returns the HyperlinkTextDestination with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): HyperlinkTextDestination

  /**
   * Returns the HyperlinkTextDestination with the specified name.
   * @param name The name.
   */
  itemByName(name: string): HyperlinkTextDestination

  /**
   * Returns the HyperlinkTextDestinations within the specified range.
   * @param from The HyperlinkTextDestination, index, or name at the beginning of the range.
   * @param to The HyperlinkTextDestination, index, or name at the end of the range.
   */
  itemByRange(
    from: HyperlinkTextDestination | number | string,
    to: HyperlinkTextDestination | number | string,
  ): HyperlinkTextDestination[]

  /**
   * Returns the last HyperlinkTextDestination in the collection.
   */
  lastItem(): HyperlinkTextDestination

  /**
   * Returns the middle HyperlinkTextDestination in the collection.
   */
  middleItem(): HyperlinkTextDestination

  /**
   * Returns the HyperlinkTextDestination whose index follows the specified HyperlinkTextDestination in the collection.
   * @param obj The HyperlinkTextDestination whose index comes before the desired HyperlinkTextDestination.
   */
  nextItem(obj: HyperlinkTextDestination): HyperlinkTextDestination

  /**
   * Returns the HyperlinkTextDestination with the index previous to the specified index.
   * @param obj The index of the HyperlinkTextDestination that follows the desired HyperlinkTextDestination.
   */
  previousItem(obj: HyperlinkTextDestination): HyperlinkTextDestination

  /**
   * Generates a string which, if executed, will return the HyperlinkTextDestination.
   */
  toSource(): string
}

/**
 * A hyperlink destination that is a document page.
 */
declare class HyperlinkPageDestination {
  /**
   * The page that the hyperlink points to.
   */
  destinationPage: Page

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * If true, the hyperlink is hidden.
   */
  readonly hidden: boolean

  /**
   * The unique ID of the HyperlinkPageDestination.
   */
  readonly id: number

  /**
   * The index of the HyperlinkPageDestination within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The name of the HyperlinkPageDestination.
   */
  name: string

  /**
   * If true or unspecified, allows a custom destination name; also does not update the name when the destination is moved to a different page. If false, uses the page number as the page destination name and makes the name property read-only for the destination object; also updates the name when the destination is moved to a different page.
   */
  nameManually: boolean

  /**
   * The parent of the HyperlinkPageDestination (a Document).
   */
  readonly parent: Document

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * The view rectangle, specified in the format [y1, x1, y2, x2]. Note: Valid only when view setting is fixed. Can return: Ordered array containing top:Unit, left:Unit, bottom:Unit, right:Unit.
   */
  viewBounds: any

  /**
   * The zoom percentage. (Range: 5 to 4000) Note: Valid only when view setting is fixed.
   */
  viewPercentage: number

  /**
   * The destination page size when the page is reached by clicking the hyperlink.
   */
  viewSetting: HyperlinkDestinationPageSetting

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): HyperlinkPageDestination[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Deletes the HyperlinkPageDestination.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Jumps to the hyperlink destination.
   */
  showDestination(): void

  /**
   * Generates a string which, if executed, will return the HyperlinkPageDestination.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of hyperlink page destinations.
 */
declare class HyperlinkPageDestinations {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the HyperlinkPageDestination with the specified index.
   * @param index The index.
   */
  [index: number]: HyperlinkPageDestination

  /**
   * Creates a new hyperlink page destination.
   * @param destination The document page that the hyperlink points to.
   * @param withProperties Initial values for properties of the new HyperlinkPageDestination
   */
  add(destination: Page, withProperties: object): HyperlinkPageDestination

  /**
   * Returns any HyperlinkPageDestination in the collection.
   */
  anyItem(): HyperlinkPageDestination

  /**
   * Displays the number of elements in the HyperlinkPageDestination.
   */
  count(): number

  /**
   * Returns every HyperlinkPageDestination in the collection.
   */
  everyItem(): HyperlinkPageDestination

  /**
   * Returns the first HyperlinkPageDestination in the collection.
   */
  firstItem(): HyperlinkPageDestination

  /**
   * Returns the HyperlinkPageDestination with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): HyperlinkPageDestination

  /**
   * Returns the HyperlinkPageDestination with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): HyperlinkPageDestination

  /**
   * Returns the HyperlinkPageDestination with the specified name.
   * @param name The name.
   */
  itemByName(name: string): HyperlinkPageDestination

  /**
   * Returns the HyperlinkPageDestinations within the specified range.
   * @param from The HyperlinkPageDestination, index, or name at the beginning of the range.
   * @param to The HyperlinkPageDestination, index, or name at the end of the range.
   */
  itemByRange(
    from: HyperlinkPageDestination | number | string,
    to: HyperlinkPageDestination | number | string,
  ): HyperlinkPageDestination[]

  /**
   * Returns the last HyperlinkPageDestination in the collection.
   */
  lastItem(): HyperlinkPageDestination

  /**
   * Returns the middle HyperlinkPageDestination in the collection.
   */
  middleItem(): HyperlinkPageDestination

  /**
   * Returns the HyperlinkPageDestination whose index follows the specified HyperlinkPageDestination in the collection.
   * @param obj The HyperlinkPageDestination whose index comes before the desired HyperlinkPageDestination.
   */
  nextItem(obj: HyperlinkPageDestination): HyperlinkPageDestination

  /**
   * Returns the HyperlinkPageDestination with the index previous to the specified index.
   * @param obj The index of the HyperlinkPageDestination that follows the desired HyperlinkPageDestination.
   */
  previousItem(obj: HyperlinkPageDestination): HyperlinkPageDestination

  /**
   * Generates a string which, if executed, will return the HyperlinkPageDestination.
   */
  toSource(): string
}

/**
 * A hyperlink destination that is a page in a document other than the document that contains the hyperlink source. For information on hyperlink sources, hyperlink page item source or hyperlink text source.
 */
declare class HyperlinkExternalPageDestination {
  /**
   * The index of the page that the hyperlink destination points to. Note: Valid only when hyperlink destination is an external page.
   */
  destinationPageIndex: number

  /**
   * The path to the document that the hyperlink destination points to.
   */
  documentPath: File

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * If true, the hyperlink is hidden.
   */
  readonly hidden: boolean

  /**
   * The unique ID of the HyperlinkExternalPageDestination.
   */
  readonly id: number

  /**
   * The index of the HyperlinkExternalPageDestination within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The name of the HyperlinkExternalPageDestination.
   */
  readonly name: string

  /**
   * The parent of the HyperlinkExternalPageDestination (a Document).
   */
  readonly parent: Document

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * The view rectangle, specified in the format [y1, x1, y2, x2]. Note: Valid only when view setting is fixed. Can return: Ordered array containing top:Unit, left:Unit, bottom:Unit, right:Unit.
   */
  viewBounds: any

  /**
   * The zoom percentage. (Range: 5 to 4000) Note: Valid only when view setting is fixed.
   */
  viewPercentage: number

  /**
   * The destination page size when the page is reached by clicking the hyperlink.
   */
  viewSetting: HyperlinkDestinationPageSetting

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): HyperlinkExternalPageDestination[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Deletes the HyperlinkExternalPageDestination.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Jumps to the hyperlink destination.
   */
  showDestination(): void

  /**
   * Generates a string which, if executed, will return the HyperlinkExternalPageDestination.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of hyperlink external page destinations.
 */
declare class HyperlinkExternalPageDestinations {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the HyperlinkExternalPageDestination with the specified index.
   * @param index The index.
   */
  [index: number]: HyperlinkExternalPageDestination

  /**
   * Creates a new hyperlink external page destination.
   * @param destination The destination page.
   * @param withProperties Initial values for properties of the new HyperlinkExternalPageDestination
   */
  add(destination: Page, withProperties: object): HyperlinkExternalPageDestination

  /**
   * Returns any HyperlinkExternalPageDestination in the collection.
   */
  anyItem(): HyperlinkExternalPageDestination

  /**
   * Displays the number of elements in the HyperlinkExternalPageDestination.
   */
  count(): number

  /**
   * Returns every HyperlinkExternalPageDestination in the collection.
   */
  everyItem(): HyperlinkExternalPageDestination

  /**
   * Returns the first HyperlinkExternalPageDestination in the collection.
   */
  firstItem(): HyperlinkExternalPageDestination

  /**
   * Returns the HyperlinkExternalPageDestination with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): HyperlinkExternalPageDestination

  /**
   * Returns the HyperlinkExternalPageDestination with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): HyperlinkExternalPageDestination

  /**
   * Returns the HyperlinkExternalPageDestination with the specified name.
   * @param name The name.
   */
  itemByName(name: string): HyperlinkExternalPageDestination

  /**
   * Returns the HyperlinkExternalPageDestinations within the specified range.
   * @param from The HyperlinkExternalPageDestination, index, or name at the beginning of the range.
   * @param to The HyperlinkExternalPageDestination, index, or name at the end of the range.
   */
  itemByRange(
    from: HyperlinkExternalPageDestination | number | string,
    to: HyperlinkExternalPageDestination | number | string,
  ): HyperlinkExternalPageDestination[]

  /**
   * Returns the last HyperlinkExternalPageDestination in the collection.
   */
  lastItem(): HyperlinkExternalPageDestination

  /**
   * Returns the middle HyperlinkExternalPageDestination in the collection.
   */
  middleItem(): HyperlinkExternalPageDestination

  /**
   * Returns the HyperlinkExternalPageDestination whose index follows the specified HyperlinkExternalPageDestination in the collection.
   * @param obj The HyperlinkExternalPageDestination whose index comes before the desired HyperlinkExternalPageDestination.
   */
  nextItem(obj: HyperlinkExternalPageDestination): HyperlinkExternalPageDestination

  /**
   * Returns the HyperlinkExternalPageDestination with the index previous to the specified index.
   * @param obj The index of the HyperlinkExternalPageDestination that follows the desired HyperlinkExternalPageDestination.
   */
  previousItem(obj: HyperlinkExternalPageDestination): HyperlinkExternalPageDestination

  /**
   * Generates a string which, if executed, will return the HyperlinkExternalPageDestination.
   */
  toSource(): string
}

/**
 * A hyperlink destination that is a URL.
 */
declare class HyperlinkURLDestination {
  /**
   * The URL the hyperlink points to.
   */
  destinationURL: string

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * If true, the hyperlink is hidden.
   */
  readonly hidden: boolean

  /**
   * The unique ID of the HyperlinkURLDestination.
   */
  readonly id: number

  /**
   * The index of the HyperlinkURLDestination within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The name of the HyperlinkURLDestination.
   */
  name: string

  /**
   * The parent of the HyperlinkURLDestination (a Document).
   */
  readonly parent: Document

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): HyperlinkURLDestination[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Deletes the HyperlinkURLDestination.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Jumps to the hyperlink destination.
   */
  showDestination(): void

  /**
   * Generates a string which, if executed, will return the HyperlinkURLDestination.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of hyperlink URL destinations.
 */
declare class HyperlinkURLDestinations {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the HyperlinkURLDestination with the specified index.
   * @param index The index.
   */
  [index: number]: HyperlinkURLDestination

  /**
   * Creates a new hyperlink URL destination.
   * @param destination The URL that the hyperlink points to.
   * @param withProperties Initial values for properties of the new HyperlinkURLDestination
   */
  add(destination: string, withProperties: object): HyperlinkURLDestination

  /**
   * Returns any HyperlinkURLDestination in the collection.
   */
  anyItem(): HyperlinkURLDestination

  /**
   * Displays the number of elements in the HyperlinkURLDestination.
   */
  count(): number

  /**
   * Returns every HyperlinkURLDestination in the collection.
   */
  everyItem(): HyperlinkURLDestination

  /**
   * Returns the first HyperlinkURLDestination in the collection.
   */
  firstItem(): HyperlinkURLDestination

  /**
   * Returns the HyperlinkURLDestination with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): HyperlinkURLDestination

  /**
   * Returns the HyperlinkURLDestination with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): HyperlinkURLDestination

  /**
   * Returns the HyperlinkURLDestination with the specified name.
   * @param name The name.
   */
  itemByName(name: string): HyperlinkURLDestination

  /**
   * Returns the HyperlinkURLDestinations within the specified range.
   * @param from The HyperlinkURLDestination, index, or name at the beginning of the range.
   * @param to The HyperlinkURLDestination, index, or name at the end of the range.
   */
  itemByRange(
    from: HyperlinkURLDestination | number | string,
    to: HyperlinkURLDestination | number | string,
  ): HyperlinkURLDestination[]

  /**
   * Returns the last HyperlinkURLDestination in the collection.
   */
  lastItem(): HyperlinkURLDestination

  /**
   * Returns the middle HyperlinkURLDestination in the collection.
   */
  middleItem(): HyperlinkURLDestination

  /**
   * Returns the HyperlinkURLDestination whose index follows the specified HyperlinkURLDestination in the collection.
   * @param obj The HyperlinkURLDestination whose index comes before the desired HyperlinkURLDestination.
   */
  nextItem(obj: HyperlinkURLDestination): HyperlinkURLDestination

  /**
   * Returns the HyperlinkURLDestination with the index previous to the specified index.
   * @param obj The index of the HyperlinkURLDestination that follows the desired HyperlinkURLDestination.
   */
  previousItem(obj: HyperlinkURLDestination): HyperlinkURLDestination

  /**
   * Generates a string which, if executed, will return the HyperlinkURLDestination.
   */
  toSource(): string
}

/**
 * A cross reference format object.
 */
declare class CrossReferenceFormat {
  /**
   * Character style of the cross reference format.
   */
  appliedCharacterStyle: CharacterStyle

  /**
   * A collection of cross reference building blocks.
   */
  readonly buildingBlocks: BuildingBlocks

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The unique ID of the CrossReferenceFormat.
   */
  readonly id: number

  /**
   * The index of the CrossReferenceFormat within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The name of the CrossReferenceFormat.
   */
  name: string

  /**
   * The parent of the CrossReferenceFormat (a Document).
   */
  readonly parent: Document

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): CrossReferenceFormat[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Deletes the CrossReferenceFormat.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the CrossReferenceFormat.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of cross reference formats.
 */
declare class CrossReferenceFormats {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the CrossReferenceFormat with the specified index.
   * @param index The index.
   */
  [index: number]: CrossReferenceFormat

  /**
   * Creates a new cross reference format.
   * @param name The format name.
   * @param withProperties Initial values for properties of the new CrossReferenceFormat
   */
  add(name: string, withProperties: object): CrossReferenceFormat

  /**
   * Returns any CrossReferenceFormat in the collection.
   */
  anyItem(): CrossReferenceFormat

  /**
   * Displays the number of elements in the CrossReferenceFormat.
   */
  count(): number

  /**
   * Returns every CrossReferenceFormat in the collection.
   */
  everyItem(): CrossReferenceFormat

  /**
   * Returns the first CrossReferenceFormat in the collection.
   */
  firstItem(): CrossReferenceFormat

  /**
   * Returns the CrossReferenceFormat with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): CrossReferenceFormat

  /**
   * Returns the CrossReferenceFormat with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): CrossReferenceFormat

  /**
   * Returns the CrossReferenceFormat with the specified name.
   * @param name The name.
   */
  itemByName(name: string): CrossReferenceFormat

  /**
   * Returns the CrossReferenceFormats within the specified range.
   * @param from The CrossReferenceFormat, index, or name at the beginning of the range.
   * @param to The CrossReferenceFormat, index, or name at the end of the range.
   */
  itemByRange(
    from: CrossReferenceFormat | number | string,
    to: CrossReferenceFormat | number | string,
  ): CrossReferenceFormat[]

  /**
   * Returns the last CrossReferenceFormat in the collection.
   */
  lastItem(): CrossReferenceFormat

  /**
   * Returns the middle CrossReferenceFormat in the collection.
   */
  middleItem(): CrossReferenceFormat

  /**
   * Returns the CrossReferenceFormat whose index follows the specified CrossReferenceFormat in the collection.
   * @param obj The CrossReferenceFormat whose index comes before the desired CrossReferenceFormat.
   */
  nextItem(obj: CrossReferenceFormat): CrossReferenceFormat

  /**
   * Returns the CrossReferenceFormat with the index previous to the specified index.
   * @param obj The index of the CrossReferenceFormat that follows the desired CrossReferenceFormat.
   */
  previousItem(obj: CrossReferenceFormat): CrossReferenceFormat

  /**
   * Generates a string which, if executed, will return the CrossReferenceFormat.
   */
  toSource(): string
}

/**
 * A cross reference text source object.
 */
declare class CrossReferenceSource extends HyperlinkTextSource {
  /**
   * Format used for cross reference source.
   */
  appliedFormat: CrossReferenceFormat

  /**
   * Updates cross reference text source content.
   */
  update(): void
}

/**
 * A collection of cross reference text sources.
 */
declare class CrossReferenceSources {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the CrossReferenceSource with the specified index.
   * @param index The index.
   */
  [index: number]: CrossReferenceSource

  /**
   * Creates a new cross reference text source.
   * @param source The text or insertion point to create the source.
   * @param appliedFormat Format used for cross reference source.
   * @param withProperties Initial values for properties of the new CrossReferenceSource
   */
  add(
    source: Text,
    appliedFormat: CrossReferenceFormat,
    withProperties: object,
  ): CrossReferenceSource

  /**
   * Returns any CrossReferenceSource in the collection.
   */
  anyItem(): CrossReferenceSource

  /**
   * Displays the number of elements in the CrossReferenceSource.
   */
  count(): number

  /**
   * Returns every CrossReferenceSource in the collection.
   */
  everyItem(): CrossReferenceSource

  /**
   * Returns the first CrossReferenceSource in the collection.
   */
  firstItem(): CrossReferenceSource

  /**
   * Returns the CrossReferenceSource with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): CrossReferenceSource

  /**
   * Returns the CrossReferenceSource with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): CrossReferenceSource

  /**
   * Returns the CrossReferenceSource with the specified name.
   * @param name The name.
   */
  itemByName(name: string): CrossReferenceSource

  /**
   * Returns the CrossReferenceSources within the specified range.
   * @param from The CrossReferenceSource, index, or name at the beginning of the range.
   * @param to The CrossReferenceSource, index, or name at the end of the range.
   */
  itemByRange(
    from: CrossReferenceSource | number | string,
    to: CrossReferenceSource | number | string,
  ): CrossReferenceSource[]

  /**
   * Returns the last CrossReferenceSource in the collection.
   */
  lastItem(): CrossReferenceSource

  /**
   * Returns the middle CrossReferenceSource in the collection.
   */
  middleItem(): CrossReferenceSource

  /**
   * Returns the CrossReferenceSource whose index follows the specified CrossReferenceSource in the collection.
   * @param obj The CrossReferenceSource whose index comes before the desired CrossReferenceSource.
   */
  nextItem(obj: CrossReferenceSource): CrossReferenceSource

  /**
   * Returns the CrossReferenceSource with the index previous to the specified index.
   * @param obj The index of the CrossReferenceSource that follows the desired CrossReferenceSource.
   */
  previousItem(obj: CrossReferenceSource): CrossReferenceSource

  /**
   * Generates a string which, if executed, will return the CrossReferenceSource.
   */
  toSource(): string
}

/**
 * A cross reference building block object.
 */
declare class BuildingBlock {
  /**
   * Character style to be applied to the building block.
   */
  appliedCharacterStyle: CharacterStyle

  /**
   * Delimiter character for paragraph text and full paragraph building blocks. It is ignored for other types of building blocks.
   */
  appliedDelimiter: string

  /**
   * Type of the building block.
   */
  blockType: BuildingBlockTypes

  /**
   * Building block custom text. Currently this is only useful in custom string building block. It is ignored for other types of building blocks.
   */
  customText: string

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * If true, the source generated for the building block includes the delimiter. It is ignored if no delimiter is specified on the building block.
   */
  includeDelimiter: boolean

  /**
   * The index of the BuildingBlock within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * The parent of the BuildingBlock (a CrossReferenceFormat).
   */
  readonly parent: CrossReferenceFormat

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): BuildingBlock[]

  /**
   * Deletes the BuildingBlock.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the BuildingBlock.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of cross reference building blocks.
 */
declare class BuildingBlocks {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the BuildingBlock with the specified index.
   * @param index The index.
   */
  [index: number]: BuildingBlock

  /**
   * Creates a new cross reference building block.
   * @param blockType Type of the building block.
   * @param appliedCharacterStyle Character style to be applied to the building block.
   * @param customText Building block custom text. Currently this is only useful in custom string building block. It is ignored for other types of building blocks.
   * @param withProperties Initial values for properties of the new BuildingBlock
   */
  add(
    blockType: BuildingBlockTypes,
    appliedCharacterStyle: CharacterStyle,
    customText: string,
    withProperties: object,
  ): BuildingBlock

  /**
   * Returns any BuildingBlock in the collection.
   */
  anyItem(): BuildingBlock

  /**
   * Displays the number of elements in the BuildingBlock.
   */
  count(): number

  /**
   * Returns every BuildingBlock in the collection.
   */
  everyItem(): BuildingBlock

  /**
   * Returns the first BuildingBlock in the collection.
   */
  firstItem(): BuildingBlock

  /**
   * Returns the BuildingBlock with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): BuildingBlock

  /**
   * Returns the BuildingBlocks within the specified range.
   * @param from The BuildingBlock, index, or name at the beginning of the range.
   * @param to The BuildingBlock, index, or name at the end of the range.
   */
  itemByRange(
    from: BuildingBlock | number | string,
    to: BuildingBlock | number | string,
  ): BuildingBlock[]

  /**
   * Returns the last BuildingBlock in the collection.
   */
  lastItem(): BuildingBlock

  /**
   * Returns the middle BuildingBlock in the collection.
   */
  middleItem(): BuildingBlock

  /**
   * Returns the BuildingBlock whose index follows the specified BuildingBlock in the collection.
   * @param obj The BuildingBlock whose index comes before the desired BuildingBlock.
   */
  nextItem(obj: BuildingBlock): BuildingBlock

  /**
   * Returns the BuildingBlock with the index previous to the specified index.
   * @param obj The index of the BuildingBlock that follows the desired BuildingBlock.
   */
  previousItem(obj: BuildingBlock): BuildingBlock

  /**
   * Generates a string which, if executed, will return the BuildingBlock.
   */
  toSource(): string
}

/**
 * Paragraph destination of a cross reference.
 */
declare class ParagraphDestination extends HyperlinkTextDestination {}

/**
 * A collection of paragraph destinations.
 */
declare class ParagraphDestinations {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the ParagraphDestination with the specified index.
   * @param index The index.
   */
  [index: number]: ParagraphDestination

  /**
   * Creates a new paragraph destination.
   * @param destination The text or insertion point inside the paragraph that the cross reference points to. The insertion point is always adjusted to the beginning of the paragraph.
   * @param withProperties Initial values for properties of the new ParagraphDestination
   */
  add(destination: Text, withProperties: object): ParagraphDestination

  /**
   * Returns any ParagraphDestination in the collection.
   */
  anyItem(): ParagraphDestination

  /**
   * Displays the number of elements in the ParagraphDestination.
   */
  count(): number

  /**
   * Returns every ParagraphDestination in the collection.
   */
  everyItem(): ParagraphDestination

  /**
   * Returns the first ParagraphDestination in the collection.
   */
  firstItem(): ParagraphDestination

  /**
   * Returns the ParagraphDestination with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): ParagraphDestination

  /**
   * Returns the ParagraphDestination with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): ParagraphDestination

  /**
   * Returns the ParagraphDestination with the specified name.
   * @param name The name.
   */
  itemByName(name: string): ParagraphDestination

  /**
   * Returns the ParagraphDestinations within the specified range.
   * @param from The ParagraphDestination, index, or name at the beginning of the range.
   * @param to The ParagraphDestination, index, or name at the end of the range.
   */
  itemByRange(
    from: ParagraphDestination | number | string,
    to: ParagraphDestination | number | string,
  ): ParagraphDestination[]

  /**
   * Returns the last ParagraphDestination in the collection.
   */
  lastItem(): ParagraphDestination

  /**
   * Returns the middle ParagraphDestination in the collection.
   */
  middleItem(): ParagraphDestination

  /**
   * Returns the ParagraphDestination whose index follows the specified ParagraphDestination in the collection.
   * @param obj The ParagraphDestination whose index comes before the desired ParagraphDestination.
   */
  nextItem(obj: ParagraphDestination): ParagraphDestination

  /**
   * Returns the ParagraphDestination with the index previous to the specified index.
   * @param obj The index of the ParagraphDestination that follows the desired ParagraphDestination.
   */
  previousItem(obj: ParagraphDestination): ParagraphDestination

  /**
   * Generates a string which, if executed, will return the ParagraphDestination.
   */
  toSource(): string
}

/**
 * An index.
 */
declare class Index {
  /**
   * The topics in the specified index section.
   */
  readonly allTopics: Topic[]

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The unique ID of the Index.
   */
  readonly id: number

  /**
   * The index of the Index within its containing object.
   */
  readonly index: number

  /**
   * A collection of index sections.
   */
  readonly indexSections: IndexSections

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The name of the Index; this is an alias to the Index's label property.
   */
  name: string

  /**
   * The parent of the Index (a Document).
   */
  readonly parent: Document

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * A collection of index topics.
   */
  readonly topics: Topics

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Makes the initial letter for the specified index topic or group of index topics upper case.
   * @param capitalizationOption The entry or entries to capitalize.
   */
  capitalize(capitalizationOption?: IndexCapitalizationOptions): void

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Generates a new index story.
   * @param on The spread or page on which to place the story.
   * @param placePoint The coordinates of the upper left corner of the story bounding box, in the format [x, y].
   * @param destinationLayer The layer on which to place the story.
   * @param autoflowing If true, allows the story to flow onto subsequent pages if the story does not fit on the specified page. If no subsequent pages exist in the document, creates the necessary pages.
   * @param includeOverset If true, includes topics in overset text in the story.
   */
  generate(
    on: Page | Spread | MasterSpread,
    placePoint: string[] | number[],
    destinationLayer: Layer,
    autoflowing?: boolean,
    includeOverset?: boolean,
  ): Story[]

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): Index[]

  /**
   * Imports a list of index topics.
   * @param from The file from which to import the topics.
   */
  importTopics(from: File): void

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Removes all index topics that do not have any index entries.
   */
  removeUnusedTopics(): void

  /**
   * Generates a string which, if executed, will return the Index.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string

  /**
   * Updates the index preview pane. Note: Does not update the index.
   */
  update(): void
}

/**
 * A collection of indexes.
 */
declare class Indexes {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Index with the specified index.
   * @param index The index.
   */
  [index: number]: Index

  /**
   * Creates a new Index.
   * @param withProperties Initial values for properties of the new Index
   */
  add(withProperties: object): Index

  /**
   * Returns any Index in the collection.
   */
  anyItem(): Index

  /**
   * Displays the number of elements in the Index.
   */
  count(): number

  /**
   * Returns every Index in the collection.
   */
  everyItem(): Index

  /**
   * Returns the first Index in the collection.
   */
  firstItem(): Index

  /**
   * Returns the Index with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Index

  /**
   * Returns the Index with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Index

  /**
   * Returns the Index with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Index

  /**
   * Returns the Indexes within the specified range.
   * @param from The Index, index, or name at the beginning of the range.
   * @param to The Index, index, or name at the end of the range.
   */
  itemByRange(from: Index | number | string, to: Index | number | string): Index[]

  /**
   * Returns the last Index in the collection.
   */
  lastItem(): Index

  /**
   * Returns the middle Index in the collection.
   */
  middleItem(): Index

  /**
   * Returns the Index whose index follows the specified Index in the collection.
   * @param obj The Index whose index comes before the desired Index.
   */
  nextItem(obj: Index): Index

  /**
   * Returns the Index with the index previous to the specified index.
   * @param obj The index of the Index that follows the desired Index.
   */
  previousItem(obj: Index): Index

  /**
   * Generates a string which, if executed, will return the Index.
   */
  toSource(): string
}

/**
 * Index options.
 */
declare class IndexOptions extends Preference {
  /**
   * The character(s) inserted at the start of cross references.
   */
  beforeCrossReferenceSeparator: string

  /**
   * The character(s) inserted between index entries when runin-style index format is used for nested topics.
   */
  betweenEntriesSeparator: string

  /**
   * The character(s) inserted between separate page numbers, page numbers and page ranges, and series of page ranges.
   */
  betweenPageNumbersSeparator: string

  /**
   * The character style applied to cross references.
   */
  crossReferenceStyle: CharacterStyle

  /**
   * The character style applied to cross reference topics.
   */
  crossReferenceTopicStyle: CharacterStyle

  /**
   * The character(s) inserted at the end of each index entry.
   */
  entryEndSeparator: string

  /**
   * The character(s) inserted after each index topic.
   */
  followingTopicSeparator: string

  /**
   * If true, includes topics and page references from all the documents in a book.
   */
  includeBookDocuments: boolean

  /**
   * If true, displays headings for sections with no index topics. Note: Valid only when include section headings is true.
   */
  includeEmptyIndexSections: boolean

  /**
   * If true, includes topics and page references on hidden layers.
   */
  includeHiddenEntries: boolean

  /**
   * If true, displays the letters of the alphabet as index section headings.
   */
  includeSectionHeadings: boolean

  /**
   * The format for level 2 and lower index topics.
   */
  indexFormat: IndexFormat

  /**
   * The paragraph style applied to level 1 index topics.
   */
  level1Style: ParagraphStyle

  /**
   * The paragraph style applied to level 2 index topics.
   */
  level2Style: ParagraphStyle

  /**
   * The paragraph style applied to level 3 index topics.
   */
  level3Style: ParagraphStyle

  /**
   * The paragraph style applied to level 4 index topics.
   */
  level4Style: ParagraphStyle

  /**
   * The character style applied to page numbers in the index.
   */
  pageNumberStyle: CharacterStyle

  /**
   * The character(s) inserted between page numbers to indicate a page range.
   */
  pageRangeSeparator: string

  /**
   * If true, replaces the content of the existing index. Note: Replaces only index content; does not update the index location or other index properties that may have been changed.
   */
  replaceExistingIndex: boolean

  /**
   * The paragraph style applied to index section headings. Note: Valid when include section headings is true.
   */
  sectionHeadingStyle: ParagraphStyle

  /**
   * The title of the generated index.
   */
  title: string

  /**
   * The paragraph style applied to the title of the generated index.
   */
  titleStyle: ParagraphStyle | string
}

/**
 * An index topic.
 */
declare class Topic {
  /**
   * A collection of index cross references. (For cross references in text, use the 'cross reference source' and 'hyperlink' objects.)
   */
  readonly crossReferences: CrossReferences

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The index of the Topic within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * The name of the Topic.
   */
  name: string

  /**
   * A collection of index page references.
   */
  readonly pageReferences: PageReferences

  /**
   * The parent of the Topic (a IndexSection, Topic or Index).
   */
  readonly parent: any

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * The string by which the topic is sorted instead of the topic name is not used. Note: The actual topic text, rather than the sort order text, appears in the index.
   */
  sortOrder: string

  /**
   * A collection of index topics.
   */
  readonly topics: Topics

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Makes the initial letter for the specified index topic or group of index topics upper case.
   * @param capitalizationOption The entry or entries to capitalize.
   */
  capitalize(capitalizationOption?: IndexCapitalizationOptions): void

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): Topic[]

  /**
   * Deletes the Topic.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the Topic.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of index topics.
 */
declare class Topics {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Topic with the specified index.
   * @param index The index.
   */
  [index: number]: Topic

  /**
   * Creates a new index topic.
   * @param name The name of the topic. Note: This is the text that appears in the index.
   * @param sortBy The string to sort this topic by instead of the topic name. Note: The actual topic text, rather than the sort order text, appears in the index.
   * @param withProperties Initial values for properties of the new Topic
   */
  add(name: string, sortBy: string, withProperties: object): Topic

  /**
   * Returns any Topic in the collection.
   */
  anyItem(): Topic

  /**
   * Displays the number of elements in the Topic.
   */
  count(): number

  /**
   * Returns every Topic in the collection.
   */
  everyItem(): Topic

  /**
   * Returns the first Topic in the collection.
   */
  firstItem(): Topic

  /**
   * Returns the Topic with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Topic

  /**
   * Returns the Topic with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Topic

  /**
   * Returns the Topics within the specified range.
   * @param from The Topic, index, or name at the beginning of the range.
   * @param to The Topic, index, or name at the end of the range.
   */
  itemByRange(from: Topic | number | string, to: Topic | number | string): Topic[]

  /**
   * Returns the last Topic in the collection.
   */
  lastItem(): Topic

  /**
   * Returns the middle Topic in the collection.
   */
  middleItem(): Topic

  /**
   * Returns the Topic whose index follows the specified Topic in the collection.
   * @param obj The Topic whose index comes before the desired Topic.
   */
  nextItem(obj: Topic): Topic

  /**
   * Returns the Topic with the index previous to the specified index.
   * @param obj The index of the Topic that follows the desired Topic.
   */
  previousItem(obj: Topic): Topic

  /**
   * Generates a string which, if executed, will return the Topic.
   */
  toSource(): string
}

/**
 * A cross reference to a different index topic. (For cross references in text, use the 'cross reference source' and 'hyperlink' objects.)
 */
declare class CrossReference {
  /**
   * The text that precedes or follows the referenced topic.
   */
  crossReferenceType: CrossReferenceType

  /**
   * The text used for a custom cross reference type. Note: Valid only for custom cross reference types.
   */
  customTypeString: string

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The unique ID of the CrossReference.
   */
  readonly id: number

  /**
   * The index of the CrossReference within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The name of the CrossReference; this is an alias to the CrossReference's label property.
   */
  name: string

  /**
   * The parent of the CrossReference (a Topic).
   */
  readonly parent: Topic

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * The topic to which this CrossReference points.
   */
  referencedTopic: Topic

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): CrossReference[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Deletes the CrossReference.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the CrossReference.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of index cross references. (For cross references in text, use the 'cross reference source' and 'hyperlink' objects.)
 */
declare class CrossReferences {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the CrossReference with the specified index.
   * @param index The index.
   */
  [index: number]: CrossReference

  /**
   * Creates a new cross reference.
   * @param referencedTopic The topic that the cross reference points to.
   * @param crossReferenceType The cross reference type.
   * @param customTypeString The custom string to use in the cross reference. Valid only for custom cross reference types.
   * @param withProperties Initial values for properties of the new CrossReference
   */
  add(
    referencedTopic: Topic,
    crossReferenceType: CrossReferenceType,
    customTypeString: string,
    withProperties: object,
  ): CrossReference

  /**
   * Returns any CrossReference in the collection.
   */
  anyItem(): CrossReference

  /**
   * Displays the number of elements in the CrossReference.
   */
  count(): number

  /**
   * Returns every CrossReference in the collection.
   */
  everyItem(): CrossReference

  /**
   * Returns the first CrossReference in the collection.
   */
  firstItem(): CrossReference

  /**
   * Returns the CrossReference with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): CrossReference

  /**
   * Returns the CrossReference with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): CrossReference

  /**
   * Returns the CrossReference with the specified name.
   * @param name The name.
   */
  itemByName(name: string): CrossReference

  /**
   * Returns the CrossReferences within the specified range.
   * @param from The CrossReference, index, or name at the beginning of the range.
   * @param to The CrossReference, index, or name at the end of the range.
   */
  itemByRange(
    from: CrossReference | number | string,
    to: CrossReference | number | string,
  ): CrossReference[]

  /**
   * Returns the last CrossReference in the collection.
   */
  lastItem(): CrossReference

  /**
   * Returns the middle CrossReference in the collection.
   */
  middleItem(): CrossReference

  /**
   * Returns the CrossReference whose index follows the specified CrossReference in the collection.
   * @param obj The CrossReference whose index comes before the desired CrossReference.
   */
  nextItem(obj: CrossReference): CrossReference

  /**
   * Returns the CrossReference with the index previous to the specified index.
   * @param obj The index of the CrossReference that follows the desired CrossReference.
   */
  previousItem(obj: CrossReference): CrossReference

  /**
   * Generates a string which, if executed, will return the CrossReference.
   */
  toSource(): string
}

/**
 * The page reference for an index topic.
 */
declare class PageReference {
  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The unique ID of the PageReference.
   */
  readonly id: number

  /**
   * The index of the PageReference within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The name of the PageReference; this is an alias to the PageReference's label property.
   */
  name: string

  /**
   * The character style applied to page numbers.
   */
  pageNumberStyleOverride: CharacterStyle

  /**
   * The paragraph style or number of paragraphs or pages that defines the last page in a page range. Note: Valid only when page reference type specifies the next use of a paragraph style or a number of paragraphs or pages. .
   */
  pageReferenceLimit: ParagraphStyle | number

  /**
   * Options for index page references.
   */
  pageReferenceType: PageReferenceType

  /**
   * The parent of the PageReference (a Topic).
   */
  readonly parent: Topic

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * The hyperlinked text or insertion point.
   */
  readonly sourceText: Text

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): PageReference[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Deletes the PageReference.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the PageReference.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of index page references.
 */
declare class PageReferences {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the PageReference with the specified index.
   * @param index The index.
   */
  [index: number]: PageReference

  /**
   * Creates a new page reference.
   * @param source The text or insertion point to which the page reference points.
   * @param pageReferenceType The page number for an index page reference or the last page in an index page reference page range.
   * @param pageReferenceLimit The paragraph style or number of paragraphs or pages that defines the last page in a page range. Valid only when page reference type specifies the next use of a paragraph style or a number of paragraphs or pages. .
   * @param pageNumberStyleOverride The style override for the page number.
   * @param withProperties Initial values for properties of the new PageReference
   */
  add(
    source: Text,
    pageReferenceType?: PageReferenceType,
    pageReferenceLimit?: ParagraphStyle | number,
    pageNumberStyleOverride?: CharacterStyle,
    withProperties?: object,
  ): PageReference

  /**
   * Returns any PageReference in the collection.
   */
  anyItem(): PageReference

  /**
   * Displays the number of elements in the PageReference.
   */
  count(): number

  /**
   * Returns every PageReference in the collection.
   */
  everyItem(): PageReference

  /**
   * Returns the first PageReference in the collection.
   */
  firstItem(): PageReference

  /**
   * Returns the PageReference with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): PageReference

  /**
   * Returns the PageReference with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): PageReference

  /**
   * Returns the PageReference with the specified name.
   * @param name The name.
   */
  itemByName(name: string): PageReference

  /**
   * Returns the PageReferences within the specified range.
   * @param from The PageReference, index, or name at the beginning of the range.
   * @param to The PageReference, index, or name at the end of the range.
   */
  itemByRange(
    from: PageReference | number | string,
    to: PageReference | number | string,
  ): PageReference[]

  /**
   * Returns the last PageReference in the collection.
   */
  lastItem(): PageReference

  /**
   * Returns the middle PageReference in the collection.
   */
  middleItem(): PageReference

  /**
   * Returns the PageReference whose index follows the specified PageReference in the collection.
   * @param obj The PageReference whose index comes before the desired PageReference.
   */
  nextItem(obj: PageReference): PageReference

  /**
   * Returns the PageReference with the index previous to the specified index.
   * @param obj The index of the PageReference that follows the desired PageReference.
   */
  previousItem(obj: PageReference): PageReference

  /**
   * Generates a string which, if executed, will return the PageReference.
   */
  toSource(): string
}

/**
 * A section within an index.
 */
declare class IndexSection {
  /**
   * The topics in the specified index section.
   */
  readonly allTopics: Topic[]

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The unique ID of the IndexSection.
   */
  readonly id: number

  /**
   * The index of the IndexSection within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The name of the IndexSection.
   */
  readonly name: string

  /**
   * The parent of the IndexSection (a Index).
   */
  readonly parent: Index

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * A collection of index topics.
   */
  readonly topics: Topics

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): IndexSection[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the IndexSection.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of index sections.
 */
declare class IndexSections {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the IndexSection with the specified index.
   * @param index The index.
   */
  [index: number]: IndexSection

  /**
   * Returns any IndexSection in the collection.
   */
  anyItem(): IndexSection

  /**
   * Displays the number of elements in the IndexSection.
   */
  count(): number

  /**
   * Returns every IndexSection in the collection.
   */
  everyItem(): IndexSection

  /**
   * Returns the first IndexSection in the collection.
   */
  firstItem(): IndexSection

  /**
   * Returns the IndexSection with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): IndexSection

  /**
   * Returns the IndexSection with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): IndexSection

  /**
   * Returns the IndexSection with the specified name.
   * @param name The name.
   */
  itemByName(name: string): IndexSection

  /**
   * Returns the IndexSections within the specified range.
   * @param from The IndexSection, index, or name at the beginning of the range.
   * @param to The IndexSection, index, or name at the end of the range.
   */
  itemByRange(
    from: IndexSection | number | string,
    to: IndexSection | number | string,
  ): IndexSection[]

  /**
   * Returns the last IndexSection in the collection.
   */
  lastItem(): IndexSection

  /**
   * Returns the middle IndexSection in the collection.
   */
  middleItem(): IndexSection

  /**
   * Returns the IndexSection whose index follows the specified IndexSection in the collection.
   * @param obj The IndexSection whose index comes before the desired IndexSection.
   */
  nextItem(obj: IndexSection): IndexSection

  /**
   * Returns the IndexSection with the index previous to the specified index.
   * @param obj The index of the IndexSection that follows the desired IndexSection.
   */
  previousItem(obj: IndexSection): IndexSection

  /**
   * Generates a string which, if executed, will return the IndexSection.
   */
  toSource(): string
}

/**
 * A form field.
 */
declare class FormField extends PageItem {
  /**
   * The index of the active state in the object's states collection.
   */
  activeStateIndex: number

  /**
   * Anchored object settings.
   */
  readonly anchoredObjectSettings: AnchoredObjectSetting

  /**
   * The description of the FormField.
   */
  description: string
}

/**
 * A collection of form fields.
 */
declare class FormFields {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the FormField with the specified index.
   * @param index The index.
   */
  [index: number]: FormField

  /**
   * Creates a new FormField
   * @param layer The layer on which to create the FormField.
   * @param at The location at which to insert the FormField relative to the reference object or within the container object.
   * @param reference The reference object. Note: Required when the at parameter is before or after.
   * @param withProperties Initial values for properties of the new FormField
   */
  add(
    layer: Layer,
    at?: LocationOptions,
    reference?: Document | Spread | MasterSpread | Page | Layer | PageItem,
    withProperties?: object,
  ): FormField

  /**
   * Returns any FormField in the collection.
   */
  anyItem(): FormField

  /**
   * Displays the number of elements in the FormField.
   */
  count(): number

  /**
   * Returns every FormField in the collection.
   */
  everyItem(): FormField

  /**
   * Returns the first FormField in the collection.
   */
  firstItem(): FormField

  /**
   * Returns the FormField with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): FormField

  /**
   * Returns the FormField with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): FormField

  /**
   * Returns the FormField with the specified name.
   * @param name The name.
   */
  itemByName(name: string): FormField

  /**
   * Returns the FormFields within the specified range.
   * @param from The FormField, index, or name at the beginning of the range.
   * @param to The FormField, index, or name at the end of the range.
   */
  itemByRange(from: FormField | number | string, to: FormField | number | string): FormField[]

  /**
   * Returns the last FormField in the collection.
   */
  lastItem(): FormField

  /**
   * Returns the middle FormField in the collection.
   */
  middleItem(): FormField

  /**
   * Returns the FormField whose index follows the specified FormField in the collection.
   * @param obj The FormField whose index comes before the desired FormField.
   */
  nextItem(obj: FormField): FormField

  /**
   * Returns the FormField with the index previous to the specified index.
   * @param obj The index of the FormField that follows the desired FormField.
   */
  previousItem(obj: FormField): FormField

  /**
   * Generates a string which, if executed, will return the FormField.
   */
  toSource(): string
}

/**
 * A button.
 */
declare class Button extends FormField {
  /**
   * A collection of animation behaviors.
   */
  readonly animationBehaviors: AnimationBehaviors

  /**
   * A collection of behavior objects.
   */
  readonly behaviors: Behaviors

  /**
   * A collection of clear form behavior objects.
   */
  readonly clearFormBehaviors: ClearFormBehaviors

  /**
   * A collection of endnote text frames.
   */
  readonly endnoteTextFrames: EndnoteTextFrames

  /**
   * A collection of EPS files.
   */
  readonly epss: EPSs

  /**
   * EPSTexts
   */
  readonly epstexts: EPSTexts

  /**
   * A collection of goto anchor behavior objects.
   */
  readonly gotoAnchorBehaviors: GotoAnchorBehaviors

  /**
   * A collection of goto first page behavior objects.
   */
  readonly gotoFirstPageBehaviors: GotoFirstPageBehaviors

  /**
   * A collection of goto last page behavior objects.
   */
  readonly gotoLastPageBehaviors: GotoLastPageBehaviors

  /**
   * A collection of goto next page behavior objects.
   */
  readonly gotoNextPageBehaviors: GotoNextPageBehaviors

  /**
   * A collection of goto next state behaviors.
   */
  readonly gotoNextStateBehaviors: GotoNextStateBehaviors

  /**
   * A collection of goto next view behavior objects.
   */
  readonly gotoNextViewBehaviors: GotoNextViewBehaviors

  /**
   * A collection of go to page behavior objects.
   */
  readonly gotoPageBehaviors: GotoPageBehaviors

  /**
   * A collection of goto previous page behavior objects.
   */
  readonly gotoPreviousPageBehaviors: GotoPreviousPageBehaviors

  /**
   * A collection of goto previous state behaviors.
   */
  readonly gotoPreviousStateBehaviors: GotoPreviousStateBehaviors

  /**
   * A collection of goto previous view behavior objects.
   */
  readonly gotoPreviousViewBehaviors: GotoPreviousViewBehaviors

  /**
   * A collection of goto state behaviors.
   */
  readonly gotoStateBehaviors: GotoStateBehaviors

  /**
   * A collection of goto URL behavior objects.
   */
  readonly gotoURLBehaviors: GotoURLBehaviors

  /**
   * A collection of graphic lines.
   */
  readonly graphicLines: GraphicLines

  /**
   * A collection of imported graphics in any graphic file format (vector, metafile, or bitmap).
   */
  readonly graphics: Graphics

  /**
   * A collection of groups.
   */
  readonly groups: Groups

  /**
   * If true, the form field/push button is hidden until triggered in the exported PDF.
   */
  hiddenUntilTriggered: boolean

  /**
   * A collection of bitmap images in any bitmap file format (including TIFF, JPEG, or GIF).
   */
  readonly images: Images

  /**
   * A collection of movie behavior objects.
   */
  readonly movieBehaviors: MovieBehaviors

  /**
   * A collection of open file behavior objects.
   */
  readonly openFileBehaviors: OpenFileBehaviors

  /**
   * A collection of ellipses.
   */
  readonly ovals: Ovals

  /**
   * The page items collection, which can be used to process all page items in a container (such as a document, page, or group), regardless of type.
   */
  readonly pageItems: PageItems

  /**
   * A collection of paths.
   */
  readonly paths: Paths

  /**
   * A collection of PDF files.
   */
  readonly pdfs: PDFs

  /**
   * A collection of PICT graphics.
   */
  readonly picts: PICTs

  /**
   * A collection of polygons.
   */
  readonly polygons: Polygons

  /**
   * A collection of print form behavior objects.
   */
  readonly printFormBehaviors: PrintFormBehaviors

  /**
   * If true, the form field/push button is printable in the exported PDF.
   */
  printableInPdf: boolean

  /**
   * A collection of rectangles.
   */
  readonly rectangles: Rectangles

  /**
   * A collection of show/hide fields behavior objects.
   */
  readonly showHideFieldsBehaviors: ShowHideFieldsBehaviors

  /**
   * A collection of sound behavior objects.
   */
  readonly soundBehaviors: SoundBehaviors

  /**
   * The spline items collection.
   */
  readonly splineItems: SplineItems

  /**
   * A collection of states.
   */
  readonly states: States

  /**
   * A collection of submit form behavior objects.
   */
  readonly submitFormBehaviors: SubmitFormBehaviors

  /**
   * A collection of text frames.
   */
  readonly textFrames: TextFrames

  /**
   * A collection of view zoom behavior objects.
   */
  readonly viewZoomBehaviors: ViewZoomBehaviors

  /**
   * A collection of WMF graphics.
   */
  readonly wmfs: WMFs

  /**
   * Brings the Button forward one level in its layer.
   */
  bringForward(): void

  /**
   * Brings the Button to the front of its layer or in front of a particular item.
   * @param reference The reference object to bring the object in front of (must have same parent)
   */
  bringToFront(reference: PageItem): void

  /**
   * Converts the button object to the page item currently in the active state. Page items from other states will be lost.
   */
  convertToObject(): void

  /**
   * Sends the Button back one level in its layer.
   */
  sendBackward(): void

  /**
   * Sends the Button to the back of its layer or behind a particular item (must have same parent).
   * @param reference The reference object to send the object behind
   */
  sendToBack(reference: PageItem): void
}

/**
 * A collection of buttons.
 */
declare class Buttons {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Button with the specified index.
   * @param index The index.
   */
  [index: number]: Button

  /**
   * Creates a new Button
   * @param layer The layer on which to create the Button.
   * @param at The location at which to insert the Button relative to the reference object or within the container object.
   * @param reference The reference object. Note: Required when the at parameter is before or after.
   * @param withProperties Initial values for properties of the new Button
   */
  add(
    layer: Layer,
    at?: LocationOptions,
    reference?: Document | Spread | MasterSpread | Page | Layer | PageItem,
    withProperties?: object,
  ): Button

  /**
   * Returns any Button in the collection.
   */
  anyItem(): Button

  /**
   * Displays the number of elements in the Button.
   */
  count(): number

  /**
   * Returns every Button in the collection.
   */
  everyItem(): Button

  /**
   * Returns the first Button in the collection.
   */
  firstItem(): Button

  /**
   * Returns the Button with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Button

  /**
   * Returns the Button with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Button

  /**
   * Returns the Button with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Button

  /**
   * Returns the Buttons within the specified range.
   * @param from The Button, index, or name at the beginning of the range.
   * @param to The Button, index, or name at the end of the range.
   */
  itemByRange(from: Button | number | string, to: Button | number | string): Button[]

  /**
   * Returns the last Button in the collection.
   */
  lastItem(): Button

  /**
   * Returns the middle Button in the collection.
   */
  middleItem(): Button

  /**
   * Returns the Button whose index follows the specified Button in the collection.
   * @param obj The Button whose index comes before the desired Button.
   */
  nextItem(obj: Button): Button

  /**
   * Returns the Button with the index previous to the specified index.
   * @param obj The index of the Button that follows the desired Button.
   */
  previousItem(obj: Button): Button

  /**
   * Generates a string which, if executed, will return the Button.
   */
  toSource(): string
}

/**
 * A state (states define the display of the field in an exported PDF or SWF).
 */
declare class State {
  /**
   * If true, the state is active in the exported PDF.
   */
  active: boolean

  /**
   * If true, the state is enabled in PDF documents.
   */
  enabled: boolean

  /**
   * A collection of endnote text frames.
   */
  readonly endnoteTextFrames: EndnoteTextFrames

  /**
   * A collection of EPS files.
   */
  readonly epss: EPSs

  /**
   * EPSTexts
   */
  readonly epstexts: EPSTexts

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * A collection of graphic lines.
   */
  readonly graphicLines: GraphicLines

  /**
   * A collection of imported graphics in any graphic file format (vector, metafile, or bitmap).
   */
  readonly graphics: Graphics

  /**
   * A collection of groups.
   */
  readonly groups: Groups

  /**
   * The unique ID of the State.
   */
  readonly id: number

  /**
   * A collection of bitmap images in any bitmap file format (including TIFF, JPEG, or GIF).
   */
  readonly images: Images

  /**
   * The index of the State within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * The name of the State.
   */
  name: string

  /**
   * A collection of ellipses.
   */
  readonly ovals: Ovals

  /**
   * The page items collection, which can be used to process all page items in a container (such as a document, page, or group), regardless of type.
   */
  readonly pageItems: PageItems

  /**
   * The parent of the State (a Button, MultiStateObject, CheckBox or RadioButton).
   */
  readonly parent: any

  /**
   * A collection of PDF files.
   */
  readonly pdfs: PDFs

  /**
   * A collection of PICT graphics.
   */
  readonly picts: PICTs

  /**
   * A collection of polygons.
   */
  readonly polygons: Polygons

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * A collection of rectangles.
   */
  readonly rectangles: Rectangles

  /**
   * The spline items collection.
   */
  readonly splineItems: SplineItems

  /**
   * For a button, the type of user action that dictates the object's appearance. For a MultiStateObject, which has no user actions associated with states, this property is a numeric value uniquely identifying the state.
   */
  statetype: StateTypes | number

  /**
   * A collection of text frames.
   */
  readonly textFrames: TextFrames

  /**
   * A collection of WMF graphics.
   */
  readonly wmfs: WMFs

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Adds page items to this state.
   * @param pageitems One or more page items to add to this state.
   */
  addItemsToState(pageitems: PageItem[]): void

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): State[]

  /**
   * Moves the state to a new position in its parent collection.
   * @param newPosition the index to move the state to in its parent collection
   */
  move(newPosition: number): void

  /**
   * Releases this state's appearance as a page item, removing the state from its parent object.
   */
  releaseAsObject(): void

  /**
   * Deletes the State.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the State.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of states.
 */
declare class States {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the State with the specified index.
   * @param index The index.
   */
  [index: number]: State

  /**
   * Creates a new State.
   * @param withProperties Initial values for properties of the new State
   */
  add(withProperties: object): State

  /**
   * Returns any State in the collection.
   */
  anyItem(): State

  /**
   * Displays the number of elements in the State.
   */
  count(): number

  /**
   * Returns every State in the collection.
   */
  everyItem(): State

  /**
   * Returns the first State in the collection.
   */
  firstItem(): State

  /**
   * Returns the State with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): State

  /**
   * Returns the State with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): State

  /**
   * Returns the State with the specified name.
   * @param name The name.
   */
  itemByName(name: string): State

  /**
   * Returns the States within the specified range.
   * @param from The State, index, or name at the beginning of the range.
   * @param to The State, index, or name at the end of the range.
   */
  itemByRange(from: State | number | string, to: State | number | string): State[]

  /**
   * Returns the last State in the collection.
   */
  lastItem(): State

  /**
   * Returns the middle State in the collection.
   */
  middleItem(): State

  /**
   * Returns the State whose index follows the specified State in the collection.
   * @param obj The State whose index comes before the desired State.
   */
  nextItem(obj: State): State

  /**
   * Returns the State with the index previous to the specified index.
   * @param obj The index of the State that follows the desired State.
   */
  previousItem(obj: State): State

  /**
   * Generates a string which, if executed, will return the State.
   */
  toSource(): string
}

/**
 * A multi-state object.
 */
declare class MultiStateObject extends FormField {
  /**
   * A collection of endnote text frames.
   */
  readonly endnoteTextFrames: EndnoteTextFrames

  /**
   * A collection of EPS files.
   */
  readonly epss: EPSs

  /**
   * EPSTexts
   */
  readonly epstexts: EPSTexts

  /**
   * A collection of graphic lines.
   */
  readonly graphicLines: GraphicLines

  /**
   * A collection of imported graphics in any graphic file format (vector, metafile, or bitmap).
   */
  readonly graphics: Graphics

  /**
   * A collection of groups.
   */
  readonly groups: Groups

  /**
   * A collection of bitmap images in any bitmap file format (including TIFF, JPEG, or GIF).
   */
  readonly images: Images

  /**
   * Determines if this object is initially hidden when displayed in an exported SWF file.
   */
  initiallyHidden: boolean

  /**
   * A collection of ellipses.
   */
  readonly ovals: Ovals

  /**
   * The page items collection, which can be used to process all page items in a container (such as a document, page, or group), regardless of type.
   */
  readonly pageItems: PageItems

  /**
   * A collection of paths.
   */
  readonly paths: Paths

  /**
   * A collection of PDF files.
   */
  readonly pdfs: PDFs

  /**
   * A collection of PICT graphics.
   */
  readonly picts: PICTs

  /**
   * A collection of polygons.
   */
  readonly polygons: Polygons

  /**
   * A collection of rectangles.
   */
  readonly rectangles: Rectangles

  /**
   * The spline items collection.
   */
  readonly splineItems: SplineItems

  /**
   * A collection of states.
   */
  readonly states: States

  /**
   * A collection of text frames.
   */
  readonly textFrames: TextFrames

  /**
   * A collection of WMF graphics.
   */
  readonly wmfs: WMFs

  /**
   * Adds items to a specific appearance state of this object.
   * @param pageitems One or more page items to add as a state.
   */
  addItemsAsState(pageitems: PageItem[]): void

  /**
   * Brings the MultiStateObject forward one level in its layer.
   */
  bringForward(): void

  /**
   * Brings the MultiStateObject to the front of its layer or in front of a particular item.
   * @param reference The reference object to bring the object in front of (must have same parent)
   */
  bringToFront(reference: PageItem): void

  /**
   * Releases all the states associated with this object and then destroys the parent object.
   */
  releaseAsObjects(): void

  /**
   * Sends the MultiStateObject back one level in its layer.
   */
  sendBackward(): void

  /**
   * Sends the MultiStateObject to the back of its layer or behind a particular item (must have same parent).
   * @param reference The reference object to send the object behind
   */
  sendToBack(reference: PageItem): void
}

/**
 * A collection of multi-state objects.
 */
declare class MultiStateObjects {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the MultiStateObject with the specified index.
   * @param index The index.
   */
  [index: number]: MultiStateObject

  /**
   * Creates a new MultiStateObject
   * @param layer The layer on which to create the MultiStateObject.
   * @param at The location at which to insert the MultiStateObject relative to the reference object or within the container object.
   * @param reference The reference object. Note: Required when the at parameter is before or after.
   * @param withProperties Initial values for properties of the new MultiStateObject
   */
  add(
    layer: Layer,
    at?: LocationOptions,
    reference?: Document | Spread | MasterSpread | Page | Layer | PageItem,
    withProperties?: object,
  ): MultiStateObject

  /**
   * Returns any MultiStateObject in the collection.
   */
  anyItem(): MultiStateObject

  /**
   * Displays the number of elements in the MultiStateObject.
   */
  count(): number

  /**
   * Returns every MultiStateObject in the collection.
   */
  everyItem(): MultiStateObject

  /**
   * Returns the first MultiStateObject in the collection.
   */
  firstItem(): MultiStateObject

  /**
   * Returns the MultiStateObject with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): MultiStateObject

  /**
   * Returns the MultiStateObject with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): MultiStateObject

  /**
   * Returns the MultiStateObject with the specified name.
   * @param name The name.
   */
  itemByName(name: string): MultiStateObject

  /**
   * Returns the MultiStateObjects within the specified range.
   * @param from The MultiStateObject, index, or name at the beginning of the range.
   * @param to The MultiStateObject, index, or name at the end of the range.
   */
  itemByRange(
    from: MultiStateObject | number | string,
    to: MultiStateObject | number | string,
  ): MultiStateObject[]

  /**
   * Returns the last MultiStateObject in the collection.
   */
  lastItem(): MultiStateObject

  /**
   * Returns the middle MultiStateObject in the collection.
   */
  middleItem(): MultiStateObject

  /**
   * Returns the MultiStateObject whose index follows the specified MultiStateObject in the collection.
   * @param obj The MultiStateObject whose index comes before the desired MultiStateObject.
   */
  nextItem(obj: MultiStateObject): MultiStateObject

  /**
   * Returns the MultiStateObject with the index previous to the specified index.
   * @param obj The index of the MultiStateObject that follows the desired MultiStateObject.
   */
  previousItem(obj: MultiStateObject): MultiStateObject

  /**
   * Generates a string which, if executed, will return the MultiStateObject.
   */
  toSource(): string
}

/**
 * A checkbox.
 */
declare class CheckBox extends FormField {
  /**
   * A collection of behavior objects.
   */
  readonly behaviors: Behaviors

  /**
   * If true, the check box/radio button is selected by default in the exported PDF.
   */
  checkedByDefault: boolean

  /**
   * A collection of clear form behavior objects.
   */
  readonly clearFormBehaviors: ClearFormBehaviors

  /**
   * A collection of EPS files.
   */
  readonly epss: EPSs

  /**
   * EPSTexts
   */
  readonly epstexts: EPSTexts

  /**
   * Export value for the check box/radio button in the exported PDF.
   */
  exportValue: string

  /**
   * A collection of goto anchor behavior objects.
   */
  readonly gotoAnchorBehaviors: GotoAnchorBehaviors

  /**
   * A collection of goto first page behavior objects.
   */
  readonly gotoFirstPageBehaviors: GotoFirstPageBehaviors

  /**
   * A collection of goto last page behavior objects.
   */
  readonly gotoLastPageBehaviors: GotoLastPageBehaviors

  /**
   * A collection of goto next page behavior objects.
   */
  readonly gotoNextPageBehaviors: GotoNextPageBehaviors

  /**
   * A collection of goto next view behavior objects.
   */
  readonly gotoNextViewBehaviors: GotoNextViewBehaviors

  /**
   * A collection of goto previous page behavior objects.
   */
  readonly gotoPreviousPageBehaviors: GotoPreviousPageBehaviors

  /**
   * A collection of goto previous view behavior objects.
   */
  readonly gotoPreviousViewBehaviors: GotoPreviousViewBehaviors

  /**
   * A collection of goto URL behavior objects.
   */
  readonly gotoURLBehaviors: GotoURLBehaviors

  /**
   * A collection of graphic lines.
   */
  readonly graphicLines: GraphicLines

  /**
   * A collection of imported graphics in any graphic file format (vector, metafile, or bitmap).
   */
  readonly graphics: Graphics

  /**
   * A collection of groups.
   */
  readonly groups: Groups

  /**
   * If true, the form field/push button is hidden until triggered in the exported PDF.
   */
  hiddenUntilTriggered: boolean

  /**
   * A collection of bitmap images in any bitmap file format (including TIFF, JPEG, or GIF).
   */
  readonly images: Images

  /**
   * A collection of movie behavior objects.
   */
  readonly movieBehaviors: MovieBehaviors

  /**
   * A collection of open file behavior objects.
   */
  readonly openFileBehaviors: OpenFileBehaviors

  /**
   * A collection of ellipses.
   */
  readonly ovals: Ovals

  /**
   * The page items collection, which can be used to process all page items in a container (such as a document, page, or group), regardless of type.
   */
  readonly pageItems: PageItems

  /**
   * A collection of PDF files.
   */
  readonly pdfs: PDFs

  /**
   * A collection of PICT graphics.
   */
  readonly picts: PICTs

  /**
   * A collection of polygons.
   */
  readonly polygons: Polygons

  /**
   * A collection of print form behavior objects.
   */
  readonly printFormBehaviors: PrintFormBehaviors

  /**
   * If true, the form field/push button is printable in the exported PDF.
   */
  printableInPdf: boolean

  /**
   * If true, the form field is read only in the exported PDF.
   */
  readOnly: boolean

  /**
   * A collection of rectangles.
   */
  readonly rectangles: Rectangles

  /**
   * If true, the form field is required in the exported PDF.
   */
  required: boolean

  /**
   * A collection of show/hide fields behavior objects.
   */
  readonly showHideFieldsBehaviors: ShowHideFieldsBehaviors

  /**
   * A collection of sound behavior objects.
   */
  readonly soundBehaviors: SoundBehaviors

  /**
   * The spline items collection.
   */
  readonly splineItems: SplineItems

  /**
   * A collection of states.
   */
  readonly states: States

  /**
   * A collection of submit form behavior objects.
   */
  readonly submitFormBehaviors: SubmitFormBehaviors

  /**
   * A collection of text frames.
   */
  readonly textFrames: TextFrames

  /**
   * A collection of view zoom behavior objects.
   */
  readonly viewZoomBehaviors: ViewZoomBehaviors

  /**
   * A collection of WMF graphics.
   */
  readonly wmfs: WMFs

  /**
   * Brings the CheckBox forward one level in its layer.
   */
  bringForward(): void

  /**
   * Brings the CheckBox to the front of its layer or in front of a particular item.
   * @param reference The reference object to bring the object in front of (must have same parent)
   */
  bringToFront(reference: PageItem): void

  /**
   * Converts the button object to the page item currently in the active state. Page items from other states will be lost.
   */
  convertToObject(): void

  /**
   * Sends the CheckBox back one level in its layer.
   */
  sendBackward(): void

  /**
   * Sends the CheckBox to the back of its layer or behind a particular item (must have same parent).
   * @param reference The reference object to send the object behind
   */
  sendToBack(reference: PageItem): void
}

/**
 * A collection of checkboxes.
 */
declare class CheckBoxes {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the CheckBox with the specified index.
   * @param index The index.
   */
  [index: number]: CheckBox

  /**
   * Creates a new CheckBox
   * @param layer The layer on which to create the CheckBox.
   * @param at The location at which to insert the CheckBox relative to the reference object or within the container object.
   * @param reference The reference object. Note: Required when the at parameter is before or after.
   * @param withProperties Initial values for properties of the new CheckBox
   */
  add(
    layer: Layer,
    at?: LocationOptions,
    reference?: Document | Spread | MasterSpread | Page | Layer | PageItem,
    withProperties?: object,
  ): CheckBox

  /**
   * Returns any CheckBox in the collection.
   */
  anyItem(): CheckBox

  /**
   * Displays the number of elements in the CheckBox.
   */
  count(): number

  /**
   * Returns every CheckBox in the collection.
   */
  everyItem(): CheckBox

  /**
   * Returns the first CheckBox in the collection.
   */
  firstItem(): CheckBox

  /**
   * Returns the CheckBox with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): CheckBox

  /**
   * Returns the CheckBox with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): CheckBox

  /**
   * Returns the CheckBox with the specified name.
   * @param name The name.
   */
  itemByName(name: string): CheckBox

  /**
   * Returns the CheckBoxes within the specified range.
   * @param from The CheckBox, index, or name at the beginning of the range.
   * @param to The CheckBox, index, or name at the end of the range.
   */
  itemByRange(from: CheckBox | number | string, to: CheckBox | number | string): CheckBox[]

  /**
   * Returns the last CheckBox in the collection.
   */
  lastItem(): CheckBox

  /**
   * Returns the middle CheckBox in the collection.
   */
  middleItem(): CheckBox

  /**
   * Returns the CheckBox whose index follows the specified CheckBox in the collection.
   * @param obj The CheckBox whose index comes before the desired CheckBox.
   */
  nextItem(obj: CheckBox): CheckBox

  /**
   * Returns the CheckBox with the index previous to the specified index.
   * @param obj The index of the CheckBox that follows the desired CheckBox.
   */
  previousItem(obj: CheckBox): CheckBox

  /**
   * Generates a string which, if executed, will return the CheckBox.
   */
  toSource(): string
}

/**
 * A combobox.
 */
declare class ComboBox extends FormField {
  /**
   * A collection of behavior objects.
   */
  readonly behaviors: Behaviors

  /**
   * The list of choices for combo and list boxes in the exported PDF.
   */
  choiceList: string[]

  /**
   * A collection of clear form behavior objects.
   */
  readonly clearFormBehaviors: ClearFormBehaviors

  /**
   * A collection of EPS files.
   */
  readonly epss: EPSs

  /**
   * EPSTexts
   */
  readonly epstexts: EPSTexts

  /**
   * The font size for the form field in the exported PDF.
   */
  fontSize: number

  /**
   * A collection of goto anchor behavior objects.
   */
  readonly gotoAnchorBehaviors: GotoAnchorBehaviors

  /**
   * A collection of goto first page behavior objects.
   */
  readonly gotoFirstPageBehaviors: GotoFirstPageBehaviors

  /**
   * A collection of goto last page behavior objects.
   */
  readonly gotoLastPageBehaviors: GotoLastPageBehaviors

  /**
   * A collection of goto next page behavior objects.
   */
  readonly gotoNextPageBehaviors: GotoNextPageBehaviors

  /**
   * A collection of goto next view behavior objects.
   */
  readonly gotoNextViewBehaviors: GotoNextViewBehaviors

  /**
   * A collection of goto previous page behavior objects.
   */
  readonly gotoPreviousPageBehaviors: GotoPreviousPageBehaviors

  /**
   * A collection of goto previous view behavior objects.
   */
  readonly gotoPreviousViewBehaviors: GotoPreviousViewBehaviors

  /**
   * A collection of goto URL behavior objects.
   */
  readonly gotoURLBehaviors: GotoURLBehaviors

  /**
   * A collection of graphic lines.
   */
  readonly graphicLines: GraphicLines

  /**
   * A collection of imported graphics in any graphic file format (vector, metafile, or bitmap).
   */
  readonly graphics: Graphics

  /**
   * A collection of groups.
   */
  readonly groups: Groups

  /**
   * If true, the form field/push button is hidden until triggered in the exported PDF.
   */
  hiddenUntilTriggered: boolean

  /**
   * A collection of bitmap images in any bitmap file format (including TIFF, JPEG, or GIF).
   */
  readonly images: Images

  /**
   * A collection of movie behavior objects.
   */
  readonly movieBehaviors: MovieBehaviors

  /**
   * A collection of open file behavior objects.
   */
  readonly openFileBehaviors: OpenFileBehaviors

  /**
   * A collection of ellipses.
   */
  readonly ovals: Ovals

  /**
   * The page items collection, which can be used to process all page items in a container (such as a document, page, or group), regardless of type.
   */
  readonly pageItems: PageItems

  /**
   * A collection of PDF files.
   */
  readonly pdfs: PDFs

  /**
   * A collection of PICT graphics.
   */
  readonly picts: PICTs

  /**
   * A collection of polygons.
   */
  readonly polygons: Polygons

  /**
   * A collection of print form behavior objects.
   */
  readonly printFormBehaviors: PrintFormBehaviors

  /**
   * If true, the form field/push button is printable in the exported PDF.
   */
  printableInPdf: boolean

  /**
   * If true, the form field is read only in the exported PDF.
   */
  readOnly: boolean

  /**
   * A collection of rectangles.
   */
  readonly rectangles: Rectangles

  /**
   * If true, the form field is required in the exported PDF.
   */
  required: boolean

  /**
   * If true, the form field has right to left text enabled in the exported PDF.
   */
  rightToLeftField: boolean

  /**
   * A collection of show/hide fields behavior objects.
   */
  readonly showHideFieldsBehaviors: ShowHideFieldsBehaviors

  /**
   * If true, the choices would be sorted in the exported PDF.
   */
  sortChoices: boolean

  /**
   * A collection of sound behavior objects.
   */
  readonly soundBehaviors: SoundBehaviors

  /**
   * The spline items collection.
   */
  readonly splineItems: SplineItems

  /**
   * A collection of submit form behavior objects.
   */
  readonly submitFormBehaviors: SubmitFormBehaviors

  /**
   * A collection of text frames.
   */
  readonly textFrames: TextFrames

  /**
   * A collection of view zoom behavior objects.
   */
  readonly viewZoomBehaviors: ViewZoomBehaviors

  /**
   * A collection of WMF graphics.
   */
  readonly wmfs: WMFs

  /**
   * Brings the ComboBox forward one level in its layer.
   */
  bringForward(): void

  /**
   * Brings the ComboBox to the front of its layer or in front of a particular item.
   * @param reference The reference object to bring the object in front of (must have same parent)
   */
  bringToFront(reference: PageItem): void

  /**
   * Converts the button object to the page item currently in the active state. Page items from other states will be lost.
   */
  convertToObject(): void

  /**
   * Sends the ComboBox back one level in its layer.
   */
  sendBackward(): void

  /**
   * Sends the ComboBox to the back of its layer or behind a particular item (must have same parent).
   * @param reference The reference object to send the object behind
   */
  sendToBack(reference: PageItem): void
}

/**
 * A collection of comboboxes.
 */
declare class ComboBoxes {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the ComboBox with the specified index.
   * @param index The index.
   */
  [index: number]: ComboBox

  /**
   * Creates a new ComboBox
   * @param layer The layer on which to create the ComboBox.
   * @param at The location at which to insert the ComboBox relative to the reference object or within the container object.
   * @param reference The reference object. Note: Required when the at parameter is before or after.
   * @param withProperties Initial values for properties of the new ComboBox
   */
  add(
    layer: Layer,
    at?: LocationOptions,
    reference?: Document | Spread | MasterSpread | Page | Layer | PageItem,
    withProperties?: object,
  ): ComboBox

  /**
   * Returns any ComboBox in the collection.
   */
  anyItem(): ComboBox

  /**
   * Displays the number of elements in the ComboBox.
   */
  count(): number

  /**
   * Returns every ComboBox in the collection.
   */
  everyItem(): ComboBox

  /**
   * Returns the first ComboBox in the collection.
   */
  firstItem(): ComboBox

  /**
   * Returns the ComboBox with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): ComboBox

  /**
   * Returns the ComboBox with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): ComboBox

  /**
   * Returns the ComboBox with the specified name.
   * @param name The name.
   */
  itemByName(name: string): ComboBox

  /**
   * Returns the ComboBoxes within the specified range.
   * @param from The ComboBox, index, or name at the beginning of the range.
   * @param to The ComboBox, index, or name at the end of the range.
   */
  itemByRange(from: ComboBox | number | string, to: ComboBox | number | string): ComboBox[]

  /**
   * Returns the last ComboBox in the collection.
   */
  lastItem(): ComboBox

  /**
   * Returns the middle ComboBox in the collection.
   */
  middleItem(): ComboBox

  /**
   * Returns the ComboBox whose index follows the specified ComboBox in the collection.
   * @param obj The ComboBox whose index comes before the desired ComboBox.
   */
  nextItem(obj: ComboBox): ComboBox

  /**
   * Returns the ComboBox with the index previous to the specified index.
   * @param obj The index of the ComboBox that follows the desired ComboBox.
   */
  previousItem(obj: ComboBox): ComboBox

  /**
   * Generates a string which, if executed, will return the ComboBox.
   */
  toSource(): string
}

/**
 * A listbox.
 */
declare class ListBox extends FormField {
  /**
   * A collection of behavior objects.
   */
  readonly behaviors: Behaviors

  /**
   * The list of choices for combo and list boxes in the exported PDF.
   */
  choiceList: string[]

  /**
   * A collection of clear form behavior objects.
   */
  readonly clearFormBehaviors: ClearFormBehaviors

  /**
   * A collection of EPS files.
   */
  readonly epss: EPSs

  /**
   * EPSTexts
   */
  readonly epstexts: EPSTexts

  /**
   * The font size for the form field in the exported PDF.
   */
  fontSize: number

  /**
   * A collection of goto anchor behavior objects.
   */
  readonly gotoAnchorBehaviors: GotoAnchorBehaviors

  /**
   * A collection of goto first page behavior objects.
   */
  readonly gotoFirstPageBehaviors: GotoFirstPageBehaviors

  /**
   * A collection of goto last page behavior objects.
   */
  readonly gotoLastPageBehaviors: GotoLastPageBehaviors

  /**
   * A collection of goto next page behavior objects.
   */
  readonly gotoNextPageBehaviors: GotoNextPageBehaviors

  /**
   * A collection of goto next view behavior objects.
   */
  readonly gotoNextViewBehaviors: GotoNextViewBehaviors

  /**
   * A collection of goto previous page behavior objects.
   */
  readonly gotoPreviousPageBehaviors: GotoPreviousPageBehaviors

  /**
   * A collection of goto previous view behavior objects.
   */
  readonly gotoPreviousViewBehaviors: GotoPreviousViewBehaviors

  /**
   * A collection of goto URL behavior objects.
   */
  readonly gotoURLBehaviors: GotoURLBehaviors

  /**
   * A collection of graphic lines.
   */
  readonly graphicLines: GraphicLines

  /**
   * A collection of imported graphics in any graphic file format (vector, metafile, or bitmap).
   */
  readonly graphics: Graphics

  /**
   * A collection of groups.
   */
  readonly groups: Groups

  /**
   * If true, the form field/push button is hidden until triggered in the exported PDF.
   */
  hiddenUntilTriggered: boolean

  /**
   * A collection of bitmap images in any bitmap file format (including TIFF, JPEG, or GIF).
   */
  readonly images: Images

  /**
   * A collection of movie behavior objects.
   */
  readonly movieBehaviors: MovieBehaviors

  /**
   * If true, the list box can have multiple items selected simultaneously in the exported PDF.
   */
  multipleSelection: boolean

  /**
   * A collection of open file behavior objects.
   */
  readonly openFileBehaviors: OpenFileBehaviors

  /**
   * A collection of ellipses.
   */
  readonly ovals: Ovals

  /**
   * The page items collection, which can be used to process all page items in a container (such as a document, page, or group), regardless of type.
   */
  readonly pageItems: PageItems

  /**
   * A collection of PDF files.
   */
  readonly pdfs: PDFs

  /**
   * A collection of PICT graphics.
   */
  readonly picts: PICTs

  /**
   * A collection of polygons.
   */
  readonly polygons: Polygons

  /**
   * A collection of print form behavior objects.
   */
  readonly printFormBehaviors: PrintFormBehaviors

  /**
   * If true, the form field/push button is printable in the exported PDF.
   */
  printableInPdf: boolean

  /**
   * If true, the form field is read only in the exported PDF.
   */
  readOnly: boolean

  /**
   * A collection of rectangles.
   */
  readonly rectangles: Rectangles

  /**
   * If true, the form field is required in the exported PDF.
   */
  required: boolean

  /**
   * If true, the form field has right to left text enabled in the exported PDF.
   */
  rightToLeftField: boolean

  /**
   * A collection of show/hide fields behavior objects.
   */
  readonly showHideFieldsBehaviors: ShowHideFieldsBehaviors

  /**
   * If true, the choices would be sorted in the exported PDF.
   */
  sortChoices: boolean

  /**
   * A collection of sound behavior objects.
   */
  readonly soundBehaviors: SoundBehaviors

  /**
   * The spline items collection.
   */
  readonly splineItems: SplineItems

  /**
   * A collection of submit form behavior objects.
   */
  readonly submitFormBehaviors: SubmitFormBehaviors

  /**
   * A collection of text frames.
   */
  readonly textFrames: TextFrames

  /**
   * A collection of view zoom behavior objects.
   */
  readonly viewZoomBehaviors: ViewZoomBehaviors

  /**
   * A collection of WMF graphics.
   */
  readonly wmfs: WMFs

  /**
   * Brings the ListBox forward one level in its layer.
   */
  bringForward(): void

  /**
   * Brings the ListBox to the front of its layer or in front of a particular item.
   * @param reference The reference object to bring the object in front of (must have same parent)
   */
  bringToFront(reference: PageItem): void

  /**
   * Converts the button object to the page item currently in the active state. Page items from other states will be lost.
   */
  convertToObject(): void

  /**
   * Sends the ListBox back one level in its layer.
   */
  sendBackward(): void

  /**
   * Sends the ListBox to the back of its layer or behind a particular item (must have same parent).
   * @param reference The reference object to send the object behind
   */
  sendToBack(reference: PageItem): void
}

/**
 * A collection of listboxes.
 */
declare class ListBoxes {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the ListBox with the specified index.
   * @param index The index.
   */
  [index: number]: ListBox

  /**
   * Creates a new ListBox
   * @param layer The layer on which to create the ListBox.
   * @param at The location at which to insert the ListBox relative to the reference object or within the container object.
   * @param reference The reference object. Note: Required when the at parameter is before or after.
   * @param withProperties Initial values for properties of the new ListBox
   */
  add(
    layer: Layer,
    at?: LocationOptions,
    reference?: Document | Spread | MasterSpread | Page | Layer | PageItem,
    withProperties?: object,
  ): ListBox

  /**
   * Returns any ListBox in the collection.
   */
  anyItem(): ListBox

  /**
   * Displays the number of elements in the ListBox.
   */
  count(): number

  /**
   * Returns every ListBox in the collection.
   */
  everyItem(): ListBox

  /**
   * Returns the first ListBox in the collection.
   */
  firstItem(): ListBox

  /**
   * Returns the ListBox with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): ListBox

  /**
   * Returns the ListBox with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): ListBox

  /**
   * Returns the ListBox with the specified name.
   * @param name The name.
   */
  itemByName(name: string): ListBox

  /**
   * Returns the ListBoxes within the specified range.
   * @param from The ListBox, index, or name at the beginning of the range.
   * @param to The ListBox, index, or name at the end of the range.
   */
  itemByRange(from: ListBox | number | string, to: ListBox | number | string): ListBox[]

  /**
   * Returns the last ListBox in the collection.
   */
  lastItem(): ListBox

  /**
   * Returns the middle ListBox in the collection.
   */
  middleItem(): ListBox

  /**
   * Returns the ListBox whose index follows the specified ListBox in the collection.
   * @param obj The ListBox whose index comes before the desired ListBox.
   */
  nextItem(obj: ListBox): ListBox

  /**
   * Returns the ListBox with the index previous to the specified index.
   * @param obj The index of the ListBox that follows the desired ListBox.
   */
  previousItem(obj: ListBox): ListBox

  /**
   * Generates a string which, if executed, will return the ListBox.
   */
  toSource(): string
}

/**
 * A radio button.
 */
declare class RadioButton extends FormField {
  /**
   * A collection of behavior objects.
   */
  readonly behaviors: Behaviors

  /**
   * If true, the check box/radio button is selected by default in the exported PDF.
   */
  checkedByDefault: boolean

  /**
   * A collection of clear form behavior objects.
   */
  readonly clearFormBehaviors: ClearFormBehaviors

  /**
   * A collection of EPS files.
   */
  readonly epss: EPSs

  /**
   * EPSTexts
   */
  readonly epstexts: EPSTexts

  /**
   * Export value for the check box/radio button in the exported PDF.
   */
  exportValue: string

  /**
   * A collection of goto anchor behavior objects.
   */
  readonly gotoAnchorBehaviors: GotoAnchorBehaviors

  /**
   * A collection of goto first page behavior objects.
   */
  readonly gotoFirstPageBehaviors: GotoFirstPageBehaviors

  /**
   * A collection of goto last page behavior objects.
   */
  readonly gotoLastPageBehaviors: GotoLastPageBehaviors

  /**
   * A collection of goto next page behavior objects.
   */
  readonly gotoNextPageBehaviors: GotoNextPageBehaviors

  /**
   * A collection of goto next view behavior objects.
   */
  readonly gotoNextViewBehaviors: GotoNextViewBehaviors

  /**
   * A collection of goto previous page behavior objects.
   */
  readonly gotoPreviousPageBehaviors: GotoPreviousPageBehaviors

  /**
   * A collection of goto previous view behavior objects.
   */
  readonly gotoPreviousViewBehaviors: GotoPreviousViewBehaviors

  /**
   * A collection of goto URL behavior objects.
   */
  readonly gotoURLBehaviors: GotoURLBehaviors

  /**
   * A collection of graphic lines.
   */
  readonly graphicLines: GraphicLines

  /**
   * A collection of imported graphics in any graphic file format (vector, metafile, or bitmap).
   */
  readonly graphics: Graphics

  /**
   * A collection of groups.
   */
  readonly groups: Groups

  /**
   * If true, the form field/push button is hidden until triggered in the exported PDF.
   */
  hiddenUntilTriggered: boolean

  /**
   * A collection of bitmap images in any bitmap file format (including TIFF, JPEG, or GIF).
   */
  readonly images: Images

  /**
   * A collection of movie behavior objects.
   */
  readonly movieBehaviors: MovieBehaviors

  /**
   * A collection of open file behavior objects.
   */
  readonly openFileBehaviors: OpenFileBehaviors

  /**
   * A collection of ellipses.
   */
  readonly ovals: Ovals

  /**
   * The page items collection, which can be used to process all page items in a container (such as a document, page, or group), regardless of type.
   */
  readonly pageItems: PageItems

  /**
   * A collection of PDF files.
   */
  readonly pdfs: PDFs

  /**
   * A collection of PICT graphics.
   */
  readonly picts: PICTs

  /**
   * A collection of polygons.
   */
  readonly polygons: Polygons

  /**
   * A collection of print form behavior objects.
   */
  readonly printFormBehaviors: PrintFormBehaviors

  /**
   * If true, the form field/push button is printable in the exported PDF.
   */
  printableInPdf: boolean

  /**
   * If true, the form field is read only in the exported PDF.
   */
  readOnly: boolean

  /**
   * A collection of rectangles.
   */
  readonly rectangles: Rectangles

  /**
   * If true, the form field is required in the exported PDF.
   */
  required: boolean

  /**
   * A collection of show/hide fields behavior objects.
   */
  readonly showHideFieldsBehaviors: ShowHideFieldsBehaviors

  /**
   * A collection of sound behavior objects.
   */
  readonly soundBehaviors: SoundBehaviors

  /**
   * The spline items collection.
   */
  readonly splineItems: SplineItems

  /**
   * A collection of states.
   */
  readonly states: States

  /**
   * A collection of submit form behavior objects.
   */
  readonly submitFormBehaviors: SubmitFormBehaviors

  /**
   * A collection of text frames.
   */
  readonly textFrames: TextFrames

  /**
   * A collection of view zoom behavior objects.
   */
  readonly viewZoomBehaviors: ViewZoomBehaviors

  /**
   * A collection of WMF graphics.
   */
  readonly wmfs: WMFs

  /**
   * Brings the RadioButton forward one level in its layer.
   */
  bringForward(): void

  /**
   * Brings the RadioButton to the front of its layer or in front of a particular item.
   * @param reference The reference object to bring the object in front of (must have same parent)
   */
  bringToFront(reference: PageItem): void

  /**
   * Converts the button object to the page item currently in the active state. Page items from other states will be lost.
   */
  convertToObject(): void

  /**
   * Sends the RadioButton back one level in its layer.
   */
  sendBackward(): void

  /**
   * Sends the RadioButton to the back of its layer or behind a particular item (must have same parent).
   * @param reference The reference object to send the object behind
   */
  sendToBack(reference: PageItem): void
}

/**
 * A collection of radio buttons.
 */
declare class RadioButtons {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the RadioButton with the specified index.
   * @param index The index.
   */
  [index: number]: RadioButton

  /**
   * Creates a new RadioButton
   * @param layer The layer on which to create the RadioButton.
   * @param at The location at which to insert the RadioButton relative to the reference object or within the container object.
   * @param reference The reference object. Note: Required when the at parameter is before or after.
   * @param withProperties Initial values for properties of the new RadioButton
   */
  add(
    layer: Layer,
    at?: LocationOptions,
    reference?: Document | Spread | MasterSpread | Page | Layer | PageItem,
    withProperties?: object,
  ): RadioButton

  /**
   * Returns any RadioButton in the collection.
   */
  anyItem(): RadioButton

  /**
   * Displays the number of elements in the RadioButton.
   */
  count(): number

  /**
   * Returns every RadioButton in the collection.
   */
  everyItem(): RadioButton

  /**
   * Returns the first RadioButton in the collection.
   */
  firstItem(): RadioButton

  /**
   * Returns the RadioButton with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): RadioButton

  /**
   * Returns the RadioButton with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): RadioButton

  /**
   * Returns the RadioButton with the specified name.
   * @param name The name.
   */
  itemByName(name: string): RadioButton

  /**
   * Returns the RadioButtons within the specified range.
   * @param from The RadioButton, index, or name at the beginning of the range.
   * @param to The RadioButton, index, or name at the end of the range.
   */
  itemByRange(from: RadioButton | number | string, to: RadioButton | number | string): RadioButton[]

  /**
   * Returns the last RadioButton in the collection.
   */
  lastItem(): RadioButton

  /**
   * Returns the middle RadioButton in the collection.
   */
  middleItem(): RadioButton

  /**
   * Returns the RadioButton whose index follows the specified RadioButton in the collection.
   * @param obj The RadioButton whose index comes before the desired RadioButton.
   */
  nextItem(obj: RadioButton): RadioButton

  /**
   * Returns the RadioButton with the index previous to the specified index.
   * @param obj The index of the RadioButton that follows the desired RadioButton.
   */
  previousItem(obj: RadioButton): RadioButton

  /**
   * Generates a string which, if executed, will return the RadioButton.
   */
  toSource(): string
}

/**
 * A text box.
 */
declare class TextBox extends FormField {
  /**
   * A collection of behavior objects.
   */
  readonly behaviors: Behaviors

  /**
   * A collection of clear form behavior objects.
   */
  readonly clearFormBehaviors: ClearFormBehaviors

  /**
   * A collection of EPS files.
   */
  readonly epss: EPSs

  /**
   * EPSTexts
   */
  readonly epstexts: EPSTexts

  /**
   * The font size for the form field in the exported PDF.
   */
  fontSize: number

  /**
   * A collection of goto anchor behavior objects.
   */
  readonly gotoAnchorBehaviors: GotoAnchorBehaviors

  /**
   * A collection of goto first page behavior objects.
   */
  readonly gotoFirstPageBehaviors: GotoFirstPageBehaviors

  /**
   * A collection of goto last page behavior objects.
   */
  readonly gotoLastPageBehaviors: GotoLastPageBehaviors

  /**
   * A collection of goto next page behavior objects.
   */
  readonly gotoNextPageBehaviors: GotoNextPageBehaviors

  /**
   * A collection of goto next view behavior objects.
   */
  readonly gotoNextViewBehaviors: GotoNextViewBehaviors

  /**
   * A collection of goto previous page behavior objects.
   */
  readonly gotoPreviousPageBehaviors: GotoPreviousPageBehaviors

  /**
   * A collection of goto previous view behavior objects.
   */
  readonly gotoPreviousViewBehaviors: GotoPreviousViewBehaviors

  /**
   * A collection of goto URL behavior objects.
   */
  readonly gotoURLBehaviors: GotoURLBehaviors

  /**
   * A collection of graphic lines.
   */
  readonly graphicLines: GraphicLines

  /**
   * A collection of imported graphics in any graphic file format (vector, metafile, or bitmap).
   */
  readonly graphics: Graphics

  /**
   * A collection of groups.
   */
  readonly groups: Groups

  /**
   * If true, the form field/push button is hidden until triggered in the exported PDF.
   */
  hiddenUntilTriggered: boolean

  /**
   * A collection of bitmap images in any bitmap file format (including TIFF, JPEG, or GIF).
   */
  readonly images: Images

  /**
   * A collection of movie behavior objects.
   */
  readonly movieBehaviors: MovieBehaviors

  /**
   * If true, the text field is multiline in the exported PDF.
   */
  multiline: boolean

  /**
   * A collection of open file behavior objects.
   */
  readonly openFileBehaviors: OpenFileBehaviors

  /**
   * A collection of ellipses.
   */
  readonly ovals: Ovals

  /**
   * The page items collection, which can be used to process all page items in a container (such as a document, page, or group), regardless of type.
   */
  readonly pageItems: PageItems

  /**
   * If true, the text field is a password field in the exported PDF.
   */
  password: boolean

  /**
   * A collection of PDF files.
   */
  readonly pdfs: PDFs

  /**
   * A collection of PICT graphics.
   */
  readonly picts: PICTs

  /**
   * A collection of polygons.
   */
  readonly polygons: Polygons

  /**
   * A collection of print form behavior objects.
   */
  readonly printFormBehaviors: PrintFormBehaviors

  /**
   * If true, the form field/push button is printable in the exported PDF.
   */
  printableInPdf: boolean

  /**
   * If true, the form field is read only in the exported PDF.
   */
  readOnly: boolean

  /**
   * A collection of rectangles.
   */
  readonly rectangles: Rectangles

  /**
   * If true, the form field is required in the exported PDF.
   */
  required: boolean

  /**
   * If true, the form field has right to left text enabled in the exported PDF.
   */
  rightToLeftField: boolean

  /**
   * If true, the text field is scrollable in the exported PDF.
   */
  scrollable: boolean

  /**
   * A collection of show/hide fields behavior objects.
   */
  readonly showHideFieldsBehaviors: ShowHideFieldsBehaviors

  /**
   * A collection of sound behavior objects.
   */
  readonly soundBehaviors: SoundBehaviors

  /**
   * The spline items collection.
   */
  readonly splineItems: SplineItems

  /**
   * A collection of submit form behavior objects.
   */
  readonly submitFormBehaviors: SubmitFormBehaviors

  /**
   * A collection of text frames.
   */
  readonly textFrames: TextFrames

  /**
   * A collection of view zoom behavior objects.
   */
  readonly viewZoomBehaviors: ViewZoomBehaviors

  /**
   * A collection of WMF graphics.
   */
  readonly wmfs: WMFs

  /**
   * Brings the TextBox forward one level in its layer.
   */
  bringForward(): void

  /**
   * Brings the TextBox to the front of its layer or in front of a particular item.
   * @param reference The reference object to bring the object in front of (must have same parent)
   */
  bringToFront(reference: PageItem): void

  /**
   * Converts the button object to the page item currently in the active state. Page items from other states will be lost.
   */
  convertToObject(): void

  /**
   * Sends the TextBox back one level in its layer.
   */
  sendBackward(): void

  /**
   * Sends the TextBox to the back of its layer or behind a particular item (must have same parent).
   * @param reference The reference object to send the object behind
   */
  sendToBack(reference: PageItem): void
}

/**
 * A collection of text boxes.
 */
declare class TextBoxes {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the TextBox with the specified index.
   * @param index The index.
   */
  [index: number]: TextBox

  /**
   * Creates a new TextBox
   * @param layer The layer on which to create the TextBox.
   * @param at The location at which to insert the TextBox relative to the reference object or within the container object.
   * @param reference The reference object. Note: Required when the at parameter is before or after.
   * @param withProperties Initial values for properties of the new TextBox
   */
  add(
    layer: Layer,
    at?: LocationOptions,
    reference?: Document | Spread | MasterSpread | Page | Layer | PageItem,
    withProperties?: object,
  ): TextBox

  /**
   * Returns any TextBox in the collection.
   */
  anyItem(): TextBox

  /**
   * Displays the number of elements in the TextBox.
   */
  count(): number

  /**
   * Returns every TextBox in the collection.
   */
  everyItem(): TextBox

  /**
   * Returns the first TextBox in the collection.
   */
  firstItem(): TextBox

  /**
   * Returns the TextBox with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): TextBox

  /**
   * Returns the TextBox with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): TextBox

  /**
   * Returns the TextBox with the specified name.
   * @param name The name.
   */
  itemByName(name: string): TextBox

  /**
   * Returns the TextBoxes within the specified range.
   * @param from The TextBox, index, or name at the beginning of the range.
   * @param to The TextBox, index, or name at the end of the range.
   */
  itemByRange(from: TextBox | number | string, to: TextBox | number | string): TextBox[]

  /**
   * Returns the last TextBox in the collection.
   */
  lastItem(): TextBox

  /**
   * Returns the middle TextBox in the collection.
   */
  middleItem(): TextBox

  /**
   * Returns the TextBox whose index follows the specified TextBox in the collection.
   * @param obj The TextBox whose index comes before the desired TextBox.
   */
  nextItem(obj: TextBox): TextBox

  /**
   * Returns the TextBox with the index previous to the specified index.
   * @param obj The index of the TextBox that follows the desired TextBox.
   */
  previousItem(obj: TextBox): TextBox

  /**
   * Generates a string which, if executed, will return the TextBox.
   */
  toSource(): string
}

/**
 * A signature field.
 */
declare class SignatureField extends FormField {
  /**
   * A collection of behavior objects.
   */
  readonly behaviors: Behaviors

  /**
   * A collection of clear form behavior objects.
   */
  readonly clearFormBehaviors: ClearFormBehaviors

  /**
   * A collection of EPS files.
   */
  readonly epss: EPSs

  /**
   * EPSTexts
   */
  readonly epstexts: EPSTexts

  /**
   * A collection of goto anchor behavior objects.
   */
  readonly gotoAnchorBehaviors: GotoAnchorBehaviors

  /**
   * A collection of goto first page behavior objects.
   */
  readonly gotoFirstPageBehaviors: GotoFirstPageBehaviors

  /**
   * A collection of goto last page behavior objects.
   */
  readonly gotoLastPageBehaviors: GotoLastPageBehaviors

  /**
   * A collection of goto next page behavior objects.
   */
  readonly gotoNextPageBehaviors: GotoNextPageBehaviors

  /**
   * A collection of goto next view behavior objects.
   */
  readonly gotoNextViewBehaviors: GotoNextViewBehaviors

  /**
   * A collection of goto previous page behavior objects.
   */
  readonly gotoPreviousPageBehaviors: GotoPreviousPageBehaviors

  /**
   * A collection of goto previous view behavior objects.
   */
  readonly gotoPreviousViewBehaviors: GotoPreviousViewBehaviors

  /**
   * A collection of goto URL behavior objects.
   */
  readonly gotoURLBehaviors: GotoURLBehaviors

  /**
   * A collection of graphic lines.
   */
  readonly graphicLines: GraphicLines

  /**
   * A collection of imported graphics in any graphic file format (vector, metafile, or bitmap).
   */
  readonly graphics: Graphics

  /**
   * A collection of groups.
   */
  readonly groups: Groups

  /**
   * If true, the form field/push button is hidden until triggered in the exported PDF.
   */
  hiddenUntilTriggered: boolean

  /**
   * A collection of bitmap images in any bitmap file format (including TIFF, JPEG, or GIF).
   */
  readonly images: Images

  /**
   * A collection of movie behavior objects.
   */
  readonly movieBehaviors: MovieBehaviors

  /**
   * A collection of open file behavior objects.
   */
  readonly openFileBehaviors: OpenFileBehaviors

  /**
   * A collection of ellipses.
   */
  readonly ovals: Ovals

  /**
   * The page items collection, which can be used to process all page items in a container (such as a document, page, or group), regardless of type.
   */
  readonly pageItems: PageItems

  /**
   * A collection of PDF files.
   */
  readonly pdfs: PDFs

  /**
   * A collection of PICT graphics.
   */
  readonly picts: PICTs

  /**
   * A collection of polygons.
   */
  readonly polygons: Polygons

  /**
   * A collection of print form behavior objects.
   */
  readonly printFormBehaviors: PrintFormBehaviors

  /**
   * If true, the form field/push button is printable in the exported PDF.
   */
  printableInPdf: boolean

  /**
   * If true, the form field is read only in the exported PDF.
   */
  readOnly: boolean

  /**
   * A collection of rectangles.
   */
  readonly rectangles: Rectangles

  /**
   * If true, the form field is required in the exported PDF.
   */
  required: boolean

  /**
   * A collection of show/hide fields behavior objects.
   */
  readonly showHideFieldsBehaviors: ShowHideFieldsBehaviors

  /**
   * A collection of sound behavior objects.
   */
  readonly soundBehaviors: SoundBehaviors

  /**
   * The spline items collection.
   */
  readonly splineItems: SplineItems

  /**
   * A collection of submit form behavior objects.
   */
  readonly submitFormBehaviors: SubmitFormBehaviors

  /**
   * A collection of text frames.
   */
  readonly textFrames: TextFrames

  /**
   * A collection of view zoom behavior objects.
   */
  readonly viewZoomBehaviors: ViewZoomBehaviors

  /**
   * A collection of WMF graphics.
   */
  readonly wmfs: WMFs

  /**
   * Brings the SignatureField forward one level in its layer.
   */
  bringForward(): void

  /**
   * Brings the SignatureField to the front of its layer or in front of a particular item.
   * @param reference The reference object to bring the object in front of (must have same parent)
   */
  bringToFront(reference: PageItem): void

  /**
   * Converts the button object to the page item currently in the active state. Page items from other states will be lost.
   */
  convertToObject(): void

  /**
   * Sends the SignatureField back one level in its layer.
   */
  sendBackward(): void

  /**
   * Sends the SignatureField to the back of its layer or behind a particular item (must have same parent).
   * @param reference The reference object to send the object behind
   */
  sendToBack(reference: PageItem): void
}

/**
 * A collection of signature fields.
 */
declare class SignatureFields {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the SignatureField with the specified index.
   * @param index The index.
   */
  [index: number]: SignatureField

  /**
   * Creates a new SignatureField
   * @param layer The layer on which to create the SignatureField.
   * @param at The location at which to insert the SignatureField relative to the reference object or within the container object.
   * @param reference The reference object. Note: Required when the at parameter is before or after.
   * @param withProperties Initial values for properties of the new SignatureField
   */
  add(
    layer: Layer,
    at?: LocationOptions,
    reference?: Document | Spread | MasterSpread | Page | Layer | PageItem,
    withProperties?: object,
  ): SignatureField

  /**
   * Returns any SignatureField in the collection.
   */
  anyItem(): SignatureField

  /**
   * Displays the number of elements in the SignatureField.
   */
  count(): number

  /**
   * Returns every SignatureField in the collection.
   */
  everyItem(): SignatureField

  /**
   * Returns the first SignatureField in the collection.
   */
  firstItem(): SignatureField

  /**
   * Returns the SignatureField with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): SignatureField

  /**
   * Returns the SignatureField with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): SignatureField

  /**
   * Returns the SignatureField with the specified name.
   * @param name The name.
   */
  itemByName(name: string): SignatureField

  /**
   * Returns the SignatureFields within the specified range.
   * @param from The SignatureField, index, or name at the beginning of the range.
   * @param to The SignatureField, index, or name at the end of the range.
   */
  itemByRange(
    from: SignatureField | number | string,
    to: SignatureField | number | string,
  ): SignatureField[]

  /**
   * Returns the last SignatureField in the collection.
   */
  lastItem(): SignatureField

  /**
   * Returns the middle SignatureField in the collection.
   */
  middleItem(): SignatureField

  /**
   * Returns the SignatureField whose index follows the specified SignatureField in the collection.
   * @param obj The SignatureField whose index comes before the desired SignatureField.
   */
  nextItem(obj: SignatureField): SignatureField

  /**
   * Returns the SignatureField with the index previous to the specified index.
   * @param obj The index of the SignatureField that follows the desired SignatureField.
   */
  previousItem(obj: SignatureField): SignatureField

  /**
   * Generates a string which, if executed, will return the SignatureField.
   */
  toSource(): string
}

/**
 * A movie.
 */
declare class Movie extends MediaItem {
  /**
   * The video controller skin name.
   */
  controllerSkin: string

  /**
   * The description of the Movie.
   */
  description: string

  /**
   * If true, the file is embedded in the PDF. If false, the file is linked to the PDF. Note: To embed movie files, acrobat compatibility must be acrobat 6 or higher.
   */
  embedInPDF: boolean

  /**
   * The file path (colon delimited on the Mac OS).
   */
  filePath: string | File

  /**
   * If true, opens a new window to play the movie. If false, plays the movie in the poster frame on the PDF document page.
   */
  floatingWindow: boolean

  /**
   * The position of the floating window that displays the movie.
   */
  floatingWindowPosition: FloatingWindowPosition

  /**
   * The size of the floating window that displays the movie.
   */
  floatingWindowSize: FloatingWindowSize

  /**
   * A collection of imported graphics in any graphic file format (vector, metafile, or bitmap).
   */
  readonly graphics: Graphics

  /**
   * A collection of bitmap images in any bitmap file format (including TIFF, JPEG, or GIF).
   */
  readonly images: Images

  /**
   * The source file of the link.
   */
  readonly itemLink: Link

  /**
   * If true, movie loops forever.
   */
  movieLoop: boolean

  /**
   * The type of poster for the movie.
   */
  moviePosterType: MoviePosterTypes

  /**
   * A collection of navigation points.
   */
  readonly navigationPoints: NavigationPoints

  /**
   * The page items collection, which can be used to process all page items in a container (such as a document, page, or group), regardless of type.
   */
  readonly pageItems: PageItems

  /**
   * If true, the movie plays automatically when a user views the page that contains the movie poster in the PDF document.
   */
  playOnPageTurn: boolean

  /**
   * The poster file.
   */
  posterFile: string

  /**
   * If true, displays controller skin with mouse rollover.
   */
  showController: boolean

  /**
   * If true, displays playback controls at the bottom of the movie display window.
   */
  showControls: boolean

  /**
   * The URL.
   */
  url: string

  /**
   * Brings the Movie forward one level in its layer.
   */
  bringForward(): void

  /**
   * Brings the Movie to the front of its layer or in front of a particular item.
   * @param reference The reference object to bring the object in front of (must have same parent)
   */
  bringToFront(reference: PageItem): void

  /**
   * Sends the Movie back one level in its layer.
   */
  sendBackward(): void

  /**
   * Sends the Movie to the back of its layer or behind a particular item (must have same parent).
   * @param reference The reference object to send the object behind
   */
  sendToBack(reference: PageItem): void

  /**
   * Verifies that the specified URL is valid and contains the specified movie file. Valid only when the movie file is specified by a URL rather than a file path.
   */
  verifyURL(): boolean
}

/**
 * A collection of movies.
 */
declare class Movies {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Movie with the specified index.
   * @param index The index.
   */
  [index: number]: Movie

  /**
   * Creates a new Movie
   * @param layer The layer on which to create the Movie.
   * @param at The location at which to insert the Movie relative to the reference object or within the container object.
   * @param reference The reference object. Note: Required when the at parameter is before or after.
   * @param withProperties Initial values for properties of the new Movie
   */
  add(
    layer: Layer,
    at?: LocationOptions,
    reference?: Document | Spread | MasterSpread | Page | Layer | PageItem,
    withProperties?: object,
  ): Movie

  /**
   * Returns any Movie in the collection.
   */
  anyItem(): Movie

  /**
   * Displays the number of elements in the Movie.
   */
  count(): number

  /**
   * Returns every Movie in the collection.
   */
  everyItem(): Movie

  /**
   * Returns the first Movie in the collection.
   */
  firstItem(): Movie

  /**
   * Returns the Movie with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Movie

  /**
   * Returns the Movie with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Movie

  /**
   * Returns the Movie with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Movie

  /**
   * Returns the Movies within the specified range.
   * @param from The Movie, index, or name at the beginning of the range.
   * @param to The Movie, index, or name at the end of the range.
   */
  itemByRange(from: Movie | number | string, to: Movie | number | string): Movie[]

  /**
   * Returns the last Movie in the collection.
   */
  lastItem(): Movie

  /**
   * Returns the middle Movie in the collection.
   */
  middleItem(): Movie

  /**
   * Returns the Movie whose index follows the specified Movie in the collection.
   * @param obj The Movie whose index comes before the desired Movie.
   */
  nextItem(obj: Movie): Movie

  /**
   * Returns the Movie with the index previous to the specified index.
   * @param obj The index of the Movie that follows the desired Movie.
   */
  previousItem(obj: Movie): Movie

  /**
   * Generates a string which, if executed, will return the Movie.
   */
  toSource(): string
}

/**
 * A navigation point.
 */
declare class NavigationPoint {
  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * Unique internally-generated identifier (read only).
   */
  readonly id: number

  /**
   * The index of the NavigationPoint within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * The UI display name.
   */
  name: string

  /**
   * The parent of the NavigationPoint (a Movie).
   */
  readonly parent: Movie

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * The time in seconds rounded up to two decimal places (e.g., 3.115 rounded to 3.12).
   */
  time: number

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): NavigationPoint[]

  /**
   * Deletes the NavigationPoint.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the NavigationPoint.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of navigation points.
 */
declare class NavigationPoints {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the NavigationPoint with the specified index.
   * @param index The index.
   */
  [index: number]: NavigationPoint

  /**
   * Creates a new NavigationPoint.
   * @param withProperties Initial values for properties of the new NavigationPoint
   */
  add(withProperties: object): NavigationPoint

  /**
   * Returns any NavigationPoint in the collection.
   */
  anyItem(): NavigationPoint

  /**
   * Displays the number of elements in the NavigationPoint.
   */
  count(): number

  /**
   * Returns every NavigationPoint in the collection.
   */
  everyItem(): NavigationPoint

  /**
   * Returns the first NavigationPoint in the collection.
   */
  firstItem(): NavigationPoint

  /**
   * Returns the NavigationPoint with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): NavigationPoint

  /**
   * Returns the NavigationPoint with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): NavigationPoint

  /**
   * Returns the NavigationPoint with the specified name.
   * @param name The name.
   */
  itemByName(name: string): NavigationPoint

  /**
   * Returns the NavigationPoints within the specified range.
   * @param from The NavigationPoint, index, or name at the beginning of the range.
   * @param to The NavigationPoint, index, or name at the end of the range.
   */
  itemByRange(
    from: NavigationPoint | number | string,
    to: NavigationPoint | number | string,
  ): NavigationPoint[]

  /**
   * Returns the last NavigationPoint in the collection.
   */
  lastItem(): NavigationPoint

  /**
   * Returns the middle NavigationPoint in the collection.
   */
  middleItem(): NavigationPoint

  /**
   * Returns the NavigationPoint whose index follows the specified NavigationPoint in the collection.
   * @param obj The NavigationPoint whose index comes before the desired NavigationPoint.
   */
  nextItem(obj: NavigationPoint): NavigationPoint

  /**
   * Returns the NavigationPoint with the index previous to the specified index.
   * @param obj The index of the NavigationPoint that follows the desired NavigationPoint.
   */
  previousItem(obj: NavigationPoint): NavigationPoint

  /**
   * Generates a string which, if executed, will return the NavigationPoint.
   */
  toSource(): string
}

/**
 * A sound clip.
 */
declare class Sound extends MediaItem {
  /**
   * The description of the Sound.
   */
  description: string

  /**
   * If true, the sound poster does not print with the document.
   */
  doNotPrintPoster: boolean

  /**
   * If true, the file is embedded in the PDF. If false, the file is linked to the PDF. Note: To embed movie files, acrobat compatibility must be acrobat 6 or higher.
   */
  embedInPDF: boolean

  /**
   * The file path (colon delimited on the Mac OS).
   */
  filePath: string | File

  /**
   * A collection of imported graphics in any graphic file format (vector, metafile, or bitmap).
   */
  readonly graphics: Graphics

  /**
   * A collection of bitmap images in any bitmap file format (including TIFF, JPEG, or GIF).
   */
  readonly images: Images

  /**
   * The source file of the link.
   */
  readonly itemLink: Link

  /**
   * The page items collection, which can be used to process all page items in a container (such as a document, page, or group), regardless of type.
   */
  readonly pageItems: PageItems

  /**
   * If true, the movie plays automatically when a user views the page that contains the movie poster in the PDF document.
   */
  playOnPageTurn: boolean

  /**
   * The poster file.
   */
  posterFile: string

  /**
   * If true, sound loops forever (SWF only).
   */
  soundLoop: boolean

  /**
   * The type of sound poster.
   */
  soundPosterType: SoundPosterTypes

  /**
   * If true, sounds stops playing when its page turns.
   */
  stopOnPageTurn: boolean

  /**
   * Brings the Sound forward one level in its layer.
   */
  bringForward(): void

  /**
   * Brings the Sound to the front of its layer or in front of a particular item.
   * @param reference The reference object to bring the object in front of (must have same parent)
   */
  bringToFront(reference: PageItem): void

  /**
   * Sends the Sound back one level in its layer.
   */
  sendBackward(): void

  /**
   * Sends the Sound to the back of its layer or behind a particular item (must have same parent).
   * @param reference The reference object to send the object behind
   */
  sendToBack(reference: PageItem): void
}

/**
 * A collection of sound clips.
 */
declare class Sounds {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Sound with the specified index.
   * @param index The index.
   */
  [index: number]: Sound

  /**
   * Creates a new Sound
   * @param layer The layer on which to create the Sound.
   * @param at The location at which to insert the Sound relative to the reference object or within the container object.
   * @param reference The reference object. Note: Required when the at parameter is before or after.
   * @param withProperties Initial values for properties of the new Sound
   */
  add(
    layer: Layer,
    at?: LocationOptions,
    reference?: Document | Spread | MasterSpread | Page | Layer | PageItem,
    withProperties?: object,
  ): Sound

  /**
   * Returns any Sound in the collection.
   */
  anyItem(): Sound

  /**
   * Displays the number of elements in the Sound.
   */
  count(): number

  /**
   * Returns every Sound in the collection.
   */
  everyItem(): Sound

  /**
   * Returns the first Sound in the collection.
   */
  firstItem(): Sound

  /**
   * Returns the Sound with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Sound

  /**
   * Returns the Sound with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Sound

  /**
   * Returns the Sound with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Sound

  /**
   * Returns the Sounds within the specified range.
   * @param from The Sound, index, or name at the beginning of the range.
   * @param to The Sound, index, or name at the end of the range.
   */
  itemByRange(from: Sound | number | string, to: Sound | number | string): Sound[]

  /**
   * Returns the last Sound in the collection.
   */
  lastItem(): Sound

  /**
   * Returns the middle Sound in the collection.
   */
  middleItem(): Sound

  /**
   * Returns the Sound whose index follows the specified Sound in the collection.
   * @param obj The Sound whose index comes before the desired Sound.
   */
  nextItem(obj: Sound): Sound

  /**
   * Returns the Sound with the index previous to the specified index.
   * @param obj The index of the Sound that follows the desired Sound.
   */
  previousItem(obj: Sound): Sound

  /**
   * Generates a string which, if executed, will return the Sound.
   */
  toSource(): string
}

/**
 * A behavior object.
 */
declare class Behavior {
  /**
   * The event which triggers the behavior.
   */
  behaviorEvent: BehaviorEvents

  /**
   * If true, the behavior is enabled.
   */
  enableBehavior: boolean

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The unique ID of the Behavior.
   */
  readonly id: number

  /**
   * The index of the Behavior within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The name of the Behavior.
   */
  readonly name: string

  /**
   * The parent of the Behavior (a Button, CheckBox, ComboBox, ListBox, RadioButton, TextBox or SignatureField).
   */
  readonly parent: any

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): Behavior[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Deletes the Behavior.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the Behavior.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of behavior objects.
 */
declare class Behaviors {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Behavior with the specified index.
   * @param index The index.
   */
  [index: number]: Behavior

  /**
   * Returns any Behavior in the collection.
   */
  anyItem(): Behavior

  /**
   * Displays the number of elements in the Behavior.
   */
  count(): number

  /**
   * Returns every Behavior in the collection.
   */
  everyItem(): Behavior

  /**
   * Returns the first Behavior in the collection.
   */
  firstItem(): Behavior

  /**
   * Returns the Behavior with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Behavior

  /**
   * Returns the Behavior with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Behavior

  /**
   * Returns the Behavior with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Behavior

  /**
   * Returns the Behaviors within the specified range.
   * @param from The Behavior, index, or name at the beginning of the range.
   * @param to The Behavior, index, or name at the end of the range.
   */
  itemByRange(from: Behavior | number | string, to: Behavior | number | string): Behavior[]

  /**
   * Returns the last Behavior in the collection.
   */
  lastItem(): Behavior

  /**
   * Returns the middle Behavior in the collection.
   */
  middleItem(): Behavior

  /**
   * Returns the Behavior whose index follows the specified Behavior in the collection.
   * @param obj The Behavior whose index comes before the desired Behavior.
   */
  nextItem(obj: Behavior): Behavior

  /**
   * Returns the Behavior with the index previous to the specified index.
   * @param obj The index of the Behavior that follows the desired Behavior.
   */
  previousItem(obj: Behavior): Behavior

  /**
   * Generates a string which, if executed, will return the Behavior.
   */
  toSource(): string
}

/**
 * A behavior object that jumps to the first page of the document.
 */
declare class GotoFirstPageBehavior extends Behavior {
  /**
   * The zoom setting.
   */
  zoomSetting: GoToZoomOptions
}

/**
 * A collection of goto first page behavior objects.
 */
declare class GotoFirstPageBehaviors {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the GotoFirstPageBehavior with the specified index.
   * @param index The index.
   */
  [index: number]: GotoFirstPageBehavior

  /**
   * Creates a new GotoFirstPageBehavior.
   * @param withProperties Initial values for properties of the new GotoFirstPageBehavior
   */
  add(withProperties: object): GotoFirstPageBehavior

  /**
   * Returns any GotoFirstPageBehavior in the collection.
   */
  anyItem(): GotoFirstPageBehavior

  /**
   * Displays the number of elements in the GotoFirstPageBehavior.
   */
  count(): number

  /**
   * Returns every GotoFirstPageBehavior in the collection.
   */
  everyItem(): GotoFirstPageBehavior

  /**
   * Returns the first GotoFirstPageBehavior in the collection.
   */
  firstItem(): GotoFirstPageBehavior

  /**
   * Returns the GotoFirstPageBehavior with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): GotoFirstPageBehavior

  /**
   * Returns the GotoFirstPageBehavior with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): GotoFirstPageBehavior

  /**
   * Returns the GotoFirstPageBehavior with the specified name.
   * @param name The name.
   */
  itemByName(name: string): GotoFirstPageBehavior

  /**
   * Returns the GotoFirstPageBehaviors within the specified range.
   * @param from The GotoFirstPageBehavior, index, or name at the beginning of the range.
   * @param to The GotoFirstPageBehavior, index, or name at the end of the range.
   */
  itemByRange(
    from: GotoFirstPageBehavior | number | string,
    to: GotoFirstPageBehavior | number | string,
  ): GotoFirstPageBehavior[]

  /**
   * Returns the last GotoFirstPageBehavior in the collection.
   */
  lastItem(): GotoFirstPageBehavior

  /**
   * Returns the middle GotoFirstPageBehavior in the collection.
   */
  middleItem(): GotoFirstPageBehavior

  /**
   * Returns the GotoFirstPageBehavior whose index follows the specified GotoFirstPageBehavior in the collection.
   * @param obj The GotoFirstPageBehavior whose index comes before the desired GotoFirstPageBehavior.
   */
  nextItem(obj: GotoFirstPageBehavior): GotoFirstPageBehavior

  /**
   * Returns the GotoFirstPageBehavior with the index previous to the specified index.
   * @param obj The index of the GotoFirstPageBehavior that follows the desired GotoFirstPageBehavior.
   */
  previousItem(obj: GotoFirstPageBehavior): GotoFirstPageBehavior

  /**
   * Generates a string which, if executed, will return the GotoFirstPageBehavior.
   */
  toSource(): string
}

/**
 * A behavior object that jumps to the last page of the document.
 */
declare class GotoLastPageBehavior extends Behavior {
  /**
   * The zoom setting.
   */
  zoomSetting: GoToZoomOptions
}

/**
 * A collection of goto last page behavior objects.
 */
declare class GotoLastPageBehaviors {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the GotoLastPageBehavior with the specified index.
   * @param index The index.
   */
  [index: number]: GotoLastPageBehavior

  /**
   * Creates a new GotoLastPageBehavior.
   * @param withProperties Initial values for properties of the new GotoLastPageBehavior
   */
  add(withProperties: object): GotoLastPageBehavior

  /**
   * Returns any GotoLastPageBehavior in the collection.
   */
  anyItem(): GotoLastPageBehavior

  /**
   * Displays the number of elements in the GotoLastPageBehavior.
   */
  count(): number

  /**
   * Returns every GotoLastPageBehavior in the collection.
   */
  everyItem(): GotoLastPageBehavior

  /**
   * Returns the first GotoLastPageBehavior in the collection.
   */
  firstItem(): GotoLastPageBehavior

  /**
   * Returns the GotoLastPageBehavior with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): GotoLastPageBehavior

  /**
   * Returns the GotoLastPageBehavior with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): GotoLastPageBehavior

  /**
   * Returns the GotoLastPageBehavior with the specified name.
   * @param name The name.
   */
  itemByName(name: string): GotoLastPageBehavior

  /**
   * Returns the GotoLastPageBehaviors within the specified range.
   * @param from The GotoLastPageBehavior, index, or name at the beginning of the range.
   * @param to The GotoLastPageBehavior, index, or name at the end of the range.
   */
  itemByRange(
    from: GotoLastPageBehavior | number | string,
    to: GotoLastPageBehavior | number | string,
  ): GotoLastPageBehavior[]

  /**
   * Returns the last GotoLastPageBehavior in the collection.
   */
  lastItem(): GotoLastPageBehavior

  /**
   * Returns the middle GotoLastPageBehavior in the collection.
   */
  middleItem(): GotoLastPageBehavior

  /**
   * Returns the GotoLastPageBehavior whose index follows the specified GotoLastPageBehavior in the collection.
   * @param obj The GotoLastPageBehavior whose index comes before the desired GotoLastPageBehavior.
   */
  nextItem(obj: GotoLastPageBehavior): GotoLastPageBehavior

  /**
   * Returns the GotoLastPageBehavior with the index previous to the specified index.
   * @param obj The index of the GotoLastPageBehavior that follows the desired GotoLastPageBehavior.
   */
  previousItem(obj: GotoLastPageBehavior): GotoLastPageBehavior

  /**
   * Generates a string which, if executed, will return the GotoLastPageBehavior.
   */
  toSource(): string
}

/**
 * A behavior object that jumps to the next page in the document.
 */
declare class GotoNextPageBehavior extends Behavior {
  /**
   * The zoom setting.
   */
  zoomSetting: GoToZoomOptions
}

/**
 * A collection of goto next page behavior objects.
 */
declare class GotoNextPageBehaviors {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the GotoNextPageBehavior with the specified index.
   * @param index The index.
   */
  [index: number]: GotoNextPageBehavior

  /**
   * Creates a new GotoNextPageBehavior.
   * @param withProperties Initial values for properties of the new GotoNextPageBehavior
   */
  add(withProperties: object): GotoNextPageBehavior

  /**
   * Returns any GotoNextPageBehavior in the collection.
   */
  anyItem(): GotoNextPageBehavior

  /**
   * Displays the number of elements in the GotoNextPageBehavior.
   */
  count(): number

  /**
   * Returns every GotoNextPageBehavior in the collection.
   */
  everyItem(): GotoNextPageBehavior

  /**
   * Returns the first GotoNextPageBehavior in the collection.
   */
  firstItem(): GotoNextPageBehavior

  /**
   * Returns the GotoNextPageBehavior with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): GotoNextPageBehavior

  /**
   * Returns the GotoNextPageBehavior with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): GotoNextPageBehavior

  /**
   * Returns the GotoNextPageBehavior with the specified name.
   * @param name The name.
   */
  itemByName(name: string): GotoNextPageBehavior

  /**
   * Returns the GotoNextPageBehaviors within the specified range.
   * @param from The GotoNextPageBehavior, index, or name at the beginning of the range.
   * @param to The GotoNextPageBehavior, index, or name at the end of the range.
   */
  itemByRange(
    from: GotoNextPageBehavior | number | string,
    to: GotoNextPageBehavior | number | string,
  ): GotoNextPageBehavior[]

  /**
   * Returns the last GotoNextPageBehavior in the collection.
   */
  lastItem(): GotoNextPageBehavior

  /**
   * Returns the middle GotoNextPageBehavior in the collection.
   */
  middleItem(): GotoNextPageBehavior

  /**
   * Returns the GotoNextPageBehavior whose index follows the specified GotoNextPageBehavior in the collection.
   * @param obj The GotoNextPageBehavior whose index comes before the desired GotoNextPageBehavior.
   */
  nextItem(obj: GotoNextPageBehavior): GotoNextPageBehavior

  /**
   * Returns the GotoNextPageBehavior with the index previous to the specified index.
   * @param obj The index of the GotoNextPageBehavior that follows the desired GotoNextPageBehavior.
   */
  previousItem(obj: GotoNextPageBehavior): GotoNextPageBehavior

  /**
   * Generates a string which, if executed, will return the GotoNextPageBehavior.
   */
  toSource(): string
}

/**
 * A behavior object that jumps to the previous page in the document.
 */
declare class GotoPreviousPageBehavior extends Behavior {
  /**
   * The zoom setting.
   */
  zoomSetting: GoToZoomOptions
}

/**
 * A collection of goto previous page behavior objects.
 */
declare class GotoPreviousPageBehaviors {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the GotoPreviousPageBehavior with the specified index.
   * @param index The index.
   */
  [index: number]: GotoPreviousPageBehavior

  /**
   * Creates a new GotoPreviousPageBehavior.
   * @param withProperties Initial values for properties of the new GotoPreviousPageBehavior
   */
  add(withProperties: object): GotoPreviousPageBehavior

  /**
   * Returns any GotoPreviousPageBehavior in the collection.
   */
  anyItem(): GotoPreviousPageBehavior

  /**
   * Displays the number of elements in the GotoPreviousPageBehavior.
   */
  count(): number

  /**
   * Returns every GotoPreviousPageBehavior in the collection.
   */
  everyItem(): GotoPreviousPageBehavior

  /**
   * Returns the first GotoPreviousPageBehavior in the collection.
   */
  firstItem(): GotoPreviousPageBehavior

  /**
   * Returns the GotoPreviousPageBehavior with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): GotoPreviousPageBehavior

  /**
   * Returns the GotoPreviousPageBehavior with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): GotoPreviousPageBehavior

  /**
   * Returns the GotoPreviousPageBehavior with the specified name.
   * @param name The name.
   */
  itemByName(name: string): GotoPreviousPageBehavior

  /**
   * Returns the GotoPreviousPageBehaviors within the specified range.
   * @param from The GotoPreviousPageBehavior, index, or name at the beginning of the range.
   * @param to The GotoPreviousPageBehavior, index, or name at the end of the range.
   */
  itemByRange(
    from: GotoPreviousPageBehavior | number | string,
    to: GotoPreviousPageBehavior | number | string,
  ): GotoPreviousPageBehavior[]

  /**
   * Returns the last GotoPreviousPageBehavior in the collection.
   */
  lastItem(): GotoPreviousPageBehavior

  /**
   * Returns the middle GotoPreviousPageBehavior in the collection.
   */
  middleItem(): GotoPreviousPageBehavior

  /**
   * Returns the GotoPreviousPageBehavior whose index follows the specified GotoPreviousPageBehavior in the collection.
   * @param obj The GotoPreviousPageBehavior whose index comes before the desired GotoPreviousPageBehavior.
   */
  nextItem(obj: GotoPreviousPageBehavior): GotoPreviousPageBehavior

  /**
   * Returns the GotoPreviousPageBehavior with the index previous to the specified index.
   * @param obj The index of the GotoPreviousPageBehavior that follows the desired GotoPreviousPageBehavior.
   */
  previousItem(obj: GotoPreviousPageBehavior): GotoPreviousPageBehavior

  /**
   * Generates a string which, if executed, will return the GotoPreviousPageBehavior.
   */
  toSource(): string
}

/**
 * A behavior object that jumps to the next view.
 */
declare class GotoNextViewBehavior extends Behavior {
  /**
   * The zoom setting.
   */
  zoomSetting: GoToZoomOptions
}

/**
 * A collection of goto next view behavior objects.
 */
declare class GotoNextViewBehaviors {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the GotoNextViewBehavior with the specified index.
   * @param index The index.
   */
  [index: number]: GotoNextViewBehavior

  /**
   * Creates a new GotoNextViewBehavior.
   * @param withProperties Initial values for properties of the new GotoNextViewBehavior
   */
  add(withProperties: object): GotoNextViewBehavior

  /**
   * Returns any GotoNextViewBehavior in the collection.
   */
  anyItem(): GotoNextViewBehavior

  /**
   * Displays the number of elements in the GotoNextViewBehavior.
   */
  count(): number

  /**
   * Returns every GotoNextViewBehavior in the collection.
   */
  everyItem(): GotoNextViewBehavior

  /**
   * Returns the first GotoNextViewBehavior in the collection.
   */
  firstItem(): GotoNextViewBehavior

  /**
   * Returns the GotoNextViewBehavior with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): GotoNextViewBehavior

  /**
   * Returns the GotoNextViewBehavior with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): GotoNextViewBehavior

  /**
   * Returns the GotoNextViewBehavior with the specified name.
   * @param name The name.
   */
  itemByName(name: string): GotoNextViewBehavior

  /**
   * Returns the GotoNextViewBehaviors within the specified range.
   * @param from The GotoNextViewBehavior, index, or name at the beginning of the range.
   * @param to The GotoNextViewBehavior, index, or name at the end of the range.
   */
  itemByRange(
    from: GotoNextViewBehavior | number | string,
    to: GotoNextViewBehavior | number | string,
  ): GotoNextViewBehavior[]

  /**
   * Returns the last GotoNextViewBehavior in the collection.
   */
  lastItem(): GotoNextViewBehavior

  /**
   * Returns the middle GotoNextViewBehavior in the collection.
   */
  middleItem(): GotoNextViewBehavior

  /**
   * Returns the GotoNextViewBehavior whose index follows the specified GotoNextViewBehavior in the collection.
   * @param obj The GotoNextViewBehavior whose index comes before the desired GotoNextViewBehavior.
   */
  nextItem(obj: GotoNextViewBehavior): GotoNextViewBehavior

  /**
   * Returns the GotoNextViewBehavior with the index previous to the specified index.
   * @param obj The index of the GotoNextViewBehavior that follows the desired GotoNextViewBehavior.
   */
  previousItem(obj: GotoNextViewBehavior): GotoNextViewBehavior

  /**
   * Generates a string which, if executed, will return the GotoNextViewBehavior.
   */
  toSource(): string
}

/**
 * A behavior object that jumps to the previous view.
 */
declare class GotoPreviousViewBehavior extends Behavior {
  /**
   * The zoom setting.
   */
  zoomSetting: GoToZoomOptions
}

/**
 * A collection of goto previous view behavior objects.
 */
declare class GotoPreviousViewBehaviors {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the GotoPreviousViewBehavior with the specified index.
   * @param index The index.
   */
  [index: number]: GotoPreviousViewBehavior

  /**
   * Creates a new GotoPreviousViewBehavior.
   * @param withProperties Initial values for properties of the new GotoPreviousViewBehavior
   */
  add(withProperties: object): GotoPreviousViewBehavior

  /**
   * Returns any GotoPreviousViewBehavior in the collection.
   */
  anyItem(): GotoPreviousViewBehavior

  /**
   * Displays the number of elements in the GotoPreviousViewBehavior.
   */
  count(): number

  /**
   * Returns every GotoPreviousViewBehavior in the collection.
   */
  everyItem(): GotoPreviousViewBehavior

  /**
   * Returns the first GotoPreviousViewBehavior in the collection.
   */
  firstItem(): GotoPreviousViewBehavior

  /**
   * Returns the GotoPreviousViewBehavior with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): GotoPreviousViewBehavior

  /**
   * Returns the GotoPreviousViewBehavior with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): GotoPreviousViewBehavior

  /**
   * Returns the GotoPreviousViewBehavior with the specified name.
   * @param name The name.
   */
  itemByName(name: string): GotoPreviousViewBehavior

  /**
   * Returns the GotoPreviousViewBehaviors within the specified range.
   * @param from The GotoPreviousViewBehavior, index, or name at the beginning of the range.
   * @param to The GotoPreviousViewBehavior, index, or name at the end of the range.
   */
  itemByRange(
    from: GotoPreviousViewBehavior | number | string,
    to: GotoPreviousViewBehavior | number | string,
  ): GotoPreviousViewBehavior[]

  /**
   * Returns the last GotoPreviousViewBehavior in the collection.
   */
  lastItem(): GotoPreviousViewBehavior

  /**
   * Returns the middle GotoPreviousViewBehavior in the collection.
   */
  middleItem(): GotoPreviousViewBehavior

  /**
   * Returns the GotoPreviousViewBehavior whose index follows the specified GotoPreviousViewBehavior in the collection.
   * @param obj The GotoPreviousViewBehavior whose index comes before the desired GotoPreviousViewBehavior.
   */
  nextItem(obj: GotoPreviousViewBehavior): GotoPreviousViewBehavior

  /**
   * Returns the GotoPreviousViewBehavior with the index previous to the specified index.
   * @param obj The index of the GotoPreviousViewBehavior that follows the desired GotoPreviousViewBehavior.
   */
  previousItem(obj: GotoPreviousViewBehavior): GotoPreviousViewBehavior

  /**
   * Generates a string which, if executed, will return the GotoPreviousViewBehavior.
   */
  toSource(): string
}

/**
 * A behavior object that jumps to a URL.
 */
declare class GotoURLBehavior extends Behavior {
  /**
   * The URL.
   */
  url: string
}

/**
 * A collection of goto URL behavior objects.
 */
declare class GotoURLBehaviors {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the GotoURLBehavior with the specified index.
   * @param index The index.
   */
  [index: number]: GotoURLBehavior

  /**
   * Creates a new GotoURLBehavior.
   * @param withProperties Initial values for properties of the new GotoURLBehavior
   */
  add(withProperties: object): GotoURLBehavior

  /**
   * Returns any GotoURLBehavior in the collection.
   */
  anyItem(): GotoURLBehavior

  /**
   * Displays the number of elements in the GotoURLBehavior.
   */
  count(): number

  /**
   * Returns every GotoURLBehavior in the collection.
   */
  everyItem(): GotoURLBehavior

  /**
   * Returns the first GotoURLBehavior in the collection.
   */
  firstItem(): GotoURLBehavior

  /**
   * Returns the GotoURLBehavior with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): GotoURLBehavior

  /**
   * Returns the GotoURLBehavior with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): GotoURLBehavior

  /**
   * Returns the GotoURLBehavior with the specified name.
   * @param name The name.
   */
  itemByName(name: string): GotoURLBehavior

  /**
   * Returns the GotoURLBehaviors within the specified range.
   * @param from The GotoURLBehavior, index, or name at the beginning of the range.
   * @param to The GotoURLBehavior, index, or name at the end of the range.
   */
  itemByRange(
    from: GotoURLBehavior | number | string,
    to: GotoURLBehavior | number | string,
  ): GotoURLBehavior[]

  /**
   * Returns the last GotoURLBehavior in the collection.
   */
  lastItem(): GotoURLBehavior

  /**
   * Returns the middle GotoURLBehavior in the collection.
   */
  middleItem(): GotoURLBehavior

  /**
   * Returns the GotoURLBehavior whose index follows the specified GotoURLBehavior in the collection.
   * @param obj The GotoURLBehavior whose index comes before the desired GotoURLBehavior.
   */
  nextItem(obj: GotoURLBehavior): GotoURLBehavior

  /**
   * Returns the GotoURLBehavior with the index previous to the specified index.
   * @param obj The index of the GotoURLBehavior that follows the desired GotoURLBehavior.
   */
  previousItem(obj: GotoURLBehavior): GotoURLBehavior

  /**
   * Generates a string which, if executed, will return the GotoURLBehavior.
   */
  toSource(): string
}

/**
 * A movie behavior object.
 */
declare class MovieBehavior extends Behavior {
  /**
   * The movie page item.
   */
  movieItem: Movie

  /**
   * The id of the navigation point to play from. This property is ignored for all operations other than Play From Navigation Point.
   */
  navigationPointID: number

  /**
   * The playback mode.
   */
  operation: MoviePlayOperations
}

/**
 * A collection of movie behavior objects.
 */
declare class MovieBehaviors {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the MovieBehavior with the specified index.
   * @param index The index.
   */
  [index: number]: MovieBehavior

  /**
   * Creates a new MovieBehavior.
   * @param withProperties Initial values for properties of the new MovieBehavior
   */
  add(withProperties: object): MovieBehavior

  /**
   * Returns any MovieBehavior in the collection.
   */
  anyItem(): MovieBehavior

  /**
   * Displays the number of elements in the MovieBehavior.
   */
  count(): number

  /**
   * Returns every MovieBehavior in the collection.
   */
  everyItem(): MovieBehavior

  /**
   * Returns the first MovieBehavior in the collection.
   */
  firstItem(): MovieBehavior

  /**
   * Returns the MovieBehavior with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): MovieBehavior

  /**
   * Returns the MovieBehavior with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): MovieBehavior

  /**
   * Returns the MovieBehavior with the specified name.
   * @param name The name.
   */
  itemByName(name: string): MovieBehavior

  /**
   * Returns the MovieBehaviors within the specified range.
   * @param from The MovieBehavior, index, or name at the beginning of the range.
   * @param to The MovieBehavior, index, or name at the end of the range.
   */
  itemByRange(
    from: MovieBehavior | number | string,
    to: MovieBehavior | number | string,
  ): MovieBehavior[]

  /**
   * Returns the last MovieBehavior in the collection.
   */
  lastItem(): MovieBehavior

  /**
   * Returns the middle MovieBehavior in the collection.
   */
  middleItem(): MovieBehavior

  /**
   * Returns the MovieBehavior whose index follows the specified MovieBehavior in the collection.
   * @param obj The MovieBehavior whose index comes before the desired MovieBehavior.
   */
  nextItem(obj: MovieBehavior): MovieBehavior

  /**
   * Returns the MovieBehavior with the index previous to the specified index.
   * @param obj The index of the MovieBehavior that follows the desired MovieBehavior.
   */
  previousItem(obj: MovieBehavior): MovieBehavior

  /**
   * Generates a string which, if executed, will return the MovieBehavior.
   */
  toSource(): string
}

/**
 * A show/hide fields behavior object.
 */
declare class ShowHideFieldsBehavior extends Behavior {
  /**
   * The hidden interactive objects.
   */
  fieldsToHide:
    | Button[]
    | CheckBoxes
    | ComboBoxes
    | ListBoxes
    | RadioButtons
    | TextBoxes
    | SignatureFields

  /**
   * The visible interactive objects.
   */
  fieldsToShow:
    | Button[]
    | CheckBoxes
    | ComboBoxes
    | ListBoxes
    | RadioButtons
    | TextBoxes
    | SignatureFields
}

/**
 * A collection of show/hide fields behavior objects.
 */
declare class ShowHideFieldsBehaviors {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the ShowHideFieldsBehavior with the specified index.
   * @param index The index.
   */
  [index: number]: ShowHideFieldsBehavior

  /**
   * Creates a new ShowHideFieldsBehavior.
   * @param withProperties Initial values for properties of the new ShowHideFieldsBehavior
   */
  add(withProperties: object): ShowHideFieldsBehavior

  /**
   * Returns any ShowHideFieldsBehavior in the collection.
   */
  anyItem(): ShowHideFieldsBehavior

  /**
   * Displays the number of elements in the ShowHideFieldsBehavior.
   */
  count(): number

  /**
   * Returns every ShowHideFieldsBehavior in the collection.
   */
  everyItem(): ShowHideFieldsBehavior

  /**
   * Returns the first ShowHideFieldsBehavior in the collection.
   */
  firstItem(): ShowHideFieldsBehavior

  /**
   * Returns the ShowHideFieldsBehavior with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): ShowHideFieldsBehavior

  /**
   * Returns the ShowHideFieldsBehavior with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): ShowHideFieldsBehavior

  /**
   * Returns the ShowHideFieldsBehavior with the specified name.
   * @param name The name.
   */
  itemByName(name: string): ShowHideFieldsBehavior

  /**
   * Returns the ShowHideFieldsBehaviors within the specified range.
   * @param from The ShowHideFieldsBehavior, index, or name at the beginning of the range.
   * @param to The ShowHideFieldsBehavior, index, or name at the end of the range.
   */
  itemByRange(
    from: ShowHideFieldsBehavior | number | string,
    to: ShowHideFieldsBehavior | number | string,
  ): ShowHideFieldsBehavior[]

  /**
   * Returns the last ShowHideFieldsBehavior in the collection.
   */
  lastItem(): ShowHideFieldsBehavior

  /**
   * Returns the middle ShowHideFieldsBehavior in the collection.
   */
  middleItem(): ShowHideFieldsBehavior

  /**
   * Returns the ShowHideFieldsBehavior whose index follows the specified ShowHideFieldsBehavior in the collection.
   * @param obj The ShowHideFieldsBehavior whose index comes before the desired ShowHideFieldsBehavior.
   */
  nextItem(obj: ShowHideFieldsBehavior): ShowHideFieldsBehavior

  /**
   * Returns the ShowHideFieldsBehavior with the index previous to the specified index.
   * @param obj The index of the ShowHideFieldsBehavior that follows the desired ShowHideFieldsBehavior.
   */
  previousItem(obj: ShowHideFieldsBehavior): ShowHideFieldsBehavior

  /**
   * Generates a string which, if executed, will return the ShowHideFieldsBehavior.
   */
  toSource(): string
}

/**
 * A animation behavior.
 */
declare class AnimationBehavior extends Behavior {
  /**
   * The animation page item.
   */
  animatedPageItem: PageItem

  /**
   * If true, will automatically play the animation in reverse on roll off of the rollover event.
   */
  autoReverseOnRollOff: boolean

  /**
   * The playback mode.
   */
  operation: AnimationPlayOperations
}

/**
 * A collection of animation behaviors.
 */
declare class AnimationBehaviors {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the AnimationBehavior with the specified index.
   * @param index The index.
   */
  [index: number]: AnimationBehavior

  /**
   * Creates a new AnimationBehavior.
   * @param withProperties Initial values for properties of the new AnimationBehavior
   */
  add(withProperties: object): AnimationBehavior

  /**
   * Returns any AnimationBehavior in the collection.
   */
  anyItem(): AnimationBehavior

  /**
   * Displays the number of elements in the AnimationBehavior.
   */
  count(): number

  /**
   * Returns every AnimationBehavior in the collection.
   */
  everyItem(): AnimationBehavior

  /**
   * Returns the first AnimationBehavior in the collection.
   */
  firstItem(): AnimationBehavior

  /**
   * Returns the AnimationBehavior with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): AnimationBehavior

  /**
   * Returns the AnimationBehavior with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): AnimationBehavior

  /**
   * Returns the AnimationBehavior with the specified name.
   * @param name The name.
   */
  itemByName(name: string): AnimationBehavior

  /**
   * Returns the AnimationBehaviors within the specified range.
   * @param from The AnimationBehavior, index, or name at the beginning of the range.
   * @param to The AnimationBehavior, index, or name at the end of the range.
   */
  itemByRange(
    from: AnimationBehavior | number | string,
    to: AnimationBehavior | number | string,
  ): AnimationBehavior[]

  /**
   * Returns the last AnimationBehavior in the collection.
   */
  lastItem(): AnimationBehavior

  /**
   * Returns the middle AnimationBehavior in the collection.
   */
  middleItem(): AnimationBehavior

  /**
   * Returns the AnimationBehavior whose index follows the specified AnimationBehavior in the collection.
   * @param obj The AnimationBehavior whose index comes before the desired AnimationBehavior.
   */
  nextItem(obj: AnimationBehavior): AnimationBehavior

  /**
   * Returns the AnimationBehavior with the index previous to the specified index.
   * @param obj The index of the AnimationBehavior that follows the desired AnimationBehavior.
   */
  previousItem(obj: AnimationBehavior): AnimationBehavior

  /**
   * Generates a string which, if executed, will return the AnimationBehavior.
   */
  toSource(): string
}

/**
 * An open file behavior object.
 */
declare class OpenFileBehavior extends Behavior {
  /**
   * The file path (colon delimited on the Mac OS).
   */
  filePath: string | File
}

/**
 * A collection of open file behavior objects.
 */
declare class OpenFileBehaviors {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the OpenFileBehavior with the specified index.
   * @param index The index.
   */
  [index: number]: OpenFileBehavior

  /**
   * Creates a new OpenFileBehavior.
   * @param withProperties Initial values for properties of the new OpenFileBehavior
   */
  add(withProperties: object): OpenFileBehavior

  /**
   * Returns any OpenFileBehavior in the collection.
   */
  anyItem(): OpenFileBehavior

  /**
   * Displays the number of elements in the OpenFileBehavior.
   */
  count(): number

  /**
   * Returns every OpenFileBehavior in the collection.
   */
  everyItem(): OpenFileBehavior

  /**
   * Returns the first OpenFileBehavior in the collection.
   */
  firstItem(): OpenFileBehavior

  /**
   * Returns the OpenFileBehavior with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): OpenFileBehavior

  /**
   * Returns the OpenFileBehavior with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): OpenFileBehavior

  /**
   * Returns the OpenFileBehavior with the specified name.
   * @param name The name.
   */
  itemByName(name: string): OpenFileBehavior

  /**
   * Returns the OpenFileBehaviors within the specified range.
   * @param from The OpenFileBehavior, index, or name at the beginning of the range.
   * @param to The OpenFileBehavior, index, or name at the end of the range.
   */
  itemByRange(
    from: OpenFileBehavior | number | string,
    to: OpenFileBehavior | number | string,
  ): OpenFileBehavior[]

  /**
   * Returns the last OpenFileBehavior in the collection.
   */
  lastItem(): OpenFileBehavior

  /**
   * Returns the middle OpenFileBehavior in the collection.
   */
  middleItem(): OpenFileBehavior

  /**
   * Returns the OpenFileBehavior whose index follows the specified OpenFileBehavior in the collection.
   * @param obj The OpenFileBehavior whose index comes before the desired OpenFileBehavior.
   */
  nextItem(obj: OpenFileBehavior): OpenFileBehavior

  /**
   * Returns the OpenFileBehavior with the index previous to the specified index.
   * @param obj The index of the OpenFileBehavior that follows the desired OpenFileBehavior.
   */
  previousItem(obj: OpenFileBehavior): OpenFileBehavior

  /**
   * Generates a string which, if executed, will return the OpenFileBehavior.
   */
  toSource(): string
}

/**
 * A goto next state behavior.
 */
declare class GotoNextStateBehavior extends Behavior {
  /**
   * The associated multi-state object page item.
   */
  associatedMultiStateObject: MultiStateObject

  /**
   * If true, will loop to the next or previous state.
   */
  loopsToNextOrPrevious: boolean
}

/**
 * A collection of goto next state behaviors.
 */
declare class GotoNextStateBehaviors {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the GotoNextStateBehavior with the specified index.
   * @param index The index.
   */
  [index: number]: GotoNextStateBehavior

  /**
   * Creates a new GotoNextStateBehavior.
   * @param withProperties Initial values for properties of the new GotoNextStateBehavior
   */
  add(withProperties: object): GotoNextStateBehavior

  /**
   * Returns any GotoNextStateBehavior in the collection.
   */
  anyItem(): GotoNextStateBehavior

  /**
   * Displays the number of elements in the GotoNextStateBehavior.
   */
  count(): number

  /**
   * Returns every GotoNextStateBehavior in the collection.
   */
  everyItem(): GotoNextStateBehavior

  /**
   * Returns the first GotoNextStateBehavior in the collection.
   */
  firstItem(): GotoNextStateBehavior

  /**
   * Returns the GotoNextStateBehavior with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): GotoNextStateBehavior

  /**
   * Returns the GotoNextStateBehavior with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): GotoNextStateBehavior

  /**
   * Returns the GotoNextStateBehavior with the specified name.
   * @param name The name.
   */
  itemByName(name: string): GotoNextStateBehavior

  /**
   * Returns the GotoNextStateBehaviors within the specified range.
   * @param from The GotoNextStateBehavior, index, or name at the beginning of the range.
   * @param to The GotoNextStateBehavior, index, or name at the end of the range.
   */
  itemByRange(
    from: GotoNextStateBehavior | number | string,
    to: GotoNextStateBehavior | number | string,
  ): GotoNextStateBehavior[]

  /**
   * Returns the last GotoNextStateBehavior in the collection.
   */
  lastItem(): GotoNextStateBehavior

  /**
   * Returns the middle GotoNextStateBehavior in the collection.
   */
  middleItem(): GotoNextStateBehavior

  /**
   * Returns the GotoNextStateBehavior whose index follows the specified GotoNextStateBehavior in the collection.
   * @param obj The GotoNextStateBehavior whose index comes before the desired GotoNextStateBehavior.
   */
  nextItem(obj: GotoNextStateBehavior): GotoNextStateBehavior

  /**
   * Returns the GotoNextStateBehavior with the index previous to the specified index.
   * @param obj The index of the GotoNextStateBehavior that follows the desired GotoNextStateBehavior.
   */
  previousItem(obj: GotoNextStateBehavior): GotoNextStateBehavior

  /**
   * Generates a string which, if executed, will return the GotoNextStateBehavior.
   */
  toSource(): string
}

/**
 * A goto previous state behavior.
 */
declare class GotoPreviousStateBehavior extends Behavior {
  /**
   * The associated multi-state object page item.
   */
  associatedMultiStateObject: MultiStateObject

  /**
   * If true, will loop to the next or previous state.
   */
  loopsToNextOrPrevious: boolean
}

/**
 * A collection of goto previous state behaviors.
 */
declare class GotoPreviousStateBehaviors {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the GotoPreviousStateBehavior with the specified index.
   * @param index The index.
   */
  [index: number]: GotoPreviousStateBehavior

  /**
   * Creates a new GotoPreviousStateBehavior.
   * @param withProperties Initial values for properties of the new GotoPreviousStateBehavior
   */
  add(withProperties: object): GotoPreviousStateBehavior

  /**
   * Returns any GotoPreviousStateBehavior in the collection.
   */
  anyItem(): GotoPreviousStateBehavior

  /**
   * Displays the number of elements in the GotoPreviousStateBehavior.
   */
  count(): number

  /**
   * Returns every GotoPreviousStateBehavior in the collection.
   */
  everyItem(): GotoPreviousStateBehavior

  /**
   * Returns the first GotoPreviousStateBehavior in the collection.
   */
  firstItem(): GotoPreviousStateBehavior

  /**
   * Returns the GotoPreviousStateBehavior with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): GotoPreviousStateBehavior

  /**
   * Returns the GotoPreviousStateBehavior with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): GotoPreviousStateBehavior

  /**
   * Returns the GotoPreviousStateBehavior with the specified name.
   * @param name The name.
   */
  itemByName(name: string): GotoPreviousStateBehavior

  /**
   * Returns the GotoPreviousStateBehaviors within the specified range.
   * @param from The GotoPreviousStateBehavior, index, or name at the beginning of the range.
   * @param to The GotoPreviousStateBehavior, index, or name at the end of the range.
   */
  itemByRange(
    from: GotoPreviousStateBehavior | number | string,
    to: GotoPreviousStateBehavior | number | string,
  ): GotoPreviousStateBehavior[]

  /**
   * Returns the last GotoPreviousStateBehavior in the collection.
   */
  lastItem(): GotoPreviousStateBehavior

  /**
   * Returns the middle GotoPreviousStateBehavior in the collection.
   */
  middleItem(): GotoPreviousStateBehavior

  /**
   * Returns the GotoPreviousStateBehavior whose index follows the specified GotoPreviousStateBehavior in the collection.
   * @param obj The GotoPreviousStateBehavior whose index comes before the desired GotoPreviousStateBehavior.
   */
  nextItem(obj: GotoPreviousStateBehavior): GotoPreviousStateBehavior

  /**
   * Returns the GotoPreviousStateBehavior with the index previous to the specified index.
   * @param obj The index of the GotoPreviousStateBehavior that follows the desired GotoPreviousStateBehavior.
   */
  previousItem(obj: GotoPreviousStateBehavior): GotoPreviousStateBehavior

  /**
   * Generates a string which, if executed, will return the GotoPreviousStateBehavior.
   */
  toSource(): string
}

/**
 * A goto state behavior.
 */
declare class GotoStateBehavior extends Behavior {
  /**
   * The associated multi-state object page item.
   */
  associatedMultiStateObject: MultiStateObject

  /**
   * If true, will automatically go back to the prior state on roll off of the rollover event.
   */
  goBackOnRollOff: boolean

  /**
   * The name of the state in the associated multi-state object.
   */
  stateName: string
}

/**
 * A collection of goto state behaviors.
 */
declare class GotoStateBehaviors {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the GotoStateBehavior with the specified index.
   * @param index The index.
   */
  [index: number]: GotoStateBehavior

  /**
   * Creates a new GotoStateBehavior.
   * @param withProperties Initial values for properties of the new GotoStateBehavior
   */
  add(withProperties: object): GotoStateBehavior

  /**
   * Returns any GotoStateBehavior in the collection.
   */
  anyItem(): GotoStateBehavior

  /**
   * Displays the number of elements in the GotoStateBehavior.
   */
  count(): number

  /**
   * Returns every GotoStateBehavior in the collection.
   */
  everyItem(): GotoStateBehavior

  /**
   * Returns the first GotoStateBehavior in the collection.
   */
  firstItem(): GotoStateBehavior

  /**
   * Returns the GotoStateBehavior with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): GotoStateBehavior

  /**
   * Returns the GotoStateBehavior with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): GotoStateBehavior

  /**
   * Returns the GotoStateBehavior with the specified name.
   * @param name The name.
   */
  itemByName(name: string): GotoStateBehavior

  /**
   * Returns the GotoStateBehaviors within the specified range.
   * @param from The GotoStateBehavior, index, or name at the beginning of the range.
   * @param to The GotoStateBehavior, index, or name at the end of the range.
   */
  itemByRange(
    from: GotoStateBehavior | number | string,
    to: GotoStateBehavior | number | string,
  ): GotoStateBehavior[]

  /**
   * Returns the last GotoStateBehavior in the collection.
   */
  lastItem(): GotoStateBehavior

  /**
   * Returns the middle GotoStateBehavior in the collection.
   */
  middleItem(): GotoStateBehavior

  /**
   * Returns the GotoStateBehavior whose index follows the specified GotoStateBehavior in the collection.
   * @param obj The GotoStateBehavior whose index comes before the desired GotoStateBehavior.
   */
  nextItem(obj: GotoStateBehavior): GotoStateBehavior

  /**
   * Returns the GotoStateBehavior with the index previous to the specified index.
   * @param obj The index of the GotoStateBehavior that follows the desired GotoStateBehavior.
   */
  previousItem(obj: GotoStateBehavior): GotoStateBehavior

  /**
   * Generates a string which, if executed, will return the GotoStateBehavior.
   */
  toSource(): string
}

/**
 * A view zoom behavior object.
 */
declare class ViewZoomBehavior extends Behavior {
  /**
   * The view zoom style.
   */
  viewZoomStyle: ViewZoomStyle
}

/**
 * A collection of view zoom behavior objects.
 */
declare class ViewZoomBehaviors {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the ViewZoomBehavior with the specified index.
   * @param index The index.
   */
  [index: number]: ViewZoomBehavior

  /**
   * Creates a new ViewZoomBehavior.
   * @param withProperties Initial values for properties of the new ViewZoomBehavior
   */
  add(withProperties: object): ViewZoomBehavior

  /**
   * Returns any ViewZoomBehavior in the collection.
   */
  anyItem(): ViewZoomBehavior

  /**
   * Displays the number of elements in the ViewZoomBehavior.
   */
  count(): number

  /**
   * Returns every ViewZoomBehavior in the collection.
   */
  everyItem(): ViewZoomBehavior

  /**
   * Returns the first ViewZoomBehavior in the collection.
   */
  firstItem(): ViewZoomBehavior

  /**
   * Returns the ViewZoomBehavior with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): ViewZoomBehavior

  /**
   * Returns the ViewZoomBehavior with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): ViewZoomBehavior

  /**
   * Returns the ViewZoomBehavior with the specified name.
   * @param name The name.
   */
  itemByName(name: string): ViewZoomBehavior

  /**
   * Returns the ViewZoomBehaviors within the specified range.
   * @param from The ViewZoomBehavior, index, or name at the beginning of the range.
   * @param to The ViewZoomBehavior, index, or name at the end of the range.
   */
  itemByRange(
    from: ViewZoomBehavior | number | string,
    to: ViewZoomBehavior | number | string,
  ): ViewZoomBehavior[]

  /**
   * Returns the last ViewZoomBehavior in the collection.
   */
  lastItem(): ViewZoomBehavior

  /**
   * Returns the middle ViewZoomBehavior in the collection.
   */
  middleItem(): ViewZoomBehavior

  /**
   * Returns the ViewZoomBehavior whose index follows the specified ViewZoomBehavior in the collection.
   * @param obj The ViewZoomBehavior whose index comes before the desired ViewZoomBehavior.
   */
  nextItem(obj: ViewZoomBehavior): ViewZoomBehavior

  /**
   * Returns the ViewZoomBehavior with the index previous to the specified index.
   * @param obj The index of the ViewZoomBehavior that follows the desired ViewZoomBehavior.
   */
  previousItem(obj: ViewZoomBehavior): ViewZoomBehavior

  /**
   * Generates a string which, if executed, will return the ViewZoomBehavior.
   */
  toSource(): string
}

/**
 * A sound behavior object.
 */
declare class SoundBehavior extends Behavior {
  /**
   * The playback mode.
   */
  operation: PlayOperations

  /**
   * The sound page item.
   */
  soundItem: Sound
}

/**
 * A collection of sound behavior objects.
 */
declare class SoundBehaviors {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the SoundBehavior with the specified index.
   * @param index The index.
   */
  [index: number]: SoundBehavior

  /**
   * Creates a new SoundBehavior.
   * @param withProperties Initial values for properties of the new SoundBehavior
   */
  add(withProperties: object): SoundBehavior

  /**
   * Returns any SoundBehavior in the collection.
   */
  anyItem(): SoundBehavior

  /**
   * Displays the number of elements in the SoundBehavior.
   */
  count(): number

  /**
   * Returns every SoundBehavior in the collection.
   */
  everyItem(): SoundBehavior

  /**
   * Returns the first SoundBehavior in the collection.
   */
  firstItem(): SoundBehavior

  /**
   * Returns the SoundBehavior with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): SoundBehavior

  /**
   * Returns the SoundBehavior with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): SoundBehavior

  /**
   * Returns the SoundBehavior with the specified name.
   * @param name The name.
   */
  itemByName(name: string): SoundBehavior

  /**
   * Returns the SoundBehaviors within the specified range.
   * @param from The SoundBehavior, index, or name at the beginning of the range.
   * @param to The SoundBehavior, index, or name at the end of the range.
   */
  itemByRange(
    from: SoundBehavior | number | string,
    to: SoundBehavior | number | string,
  ): SoundBehavior[]

  /**
   * Returns the last SoundBehavior in the collection.
   */
  lastItem(): SoundBehavior

  /**
   * Returns the middle SoundBehavior in the collection.
   */
  middleItem(): SoundBehavior

  /**
   * Returns the SoundBehavior whose index follows the specified SoundBehavior in the collection.
   * @param obj The SoundBehavior whose index comes before the desired SoundBehavior.
   */
  nextItem(obj: SoundBehavior): SoundBehavior

  /**
   * Returns the SoundBehavior with the index previous to the specified index.
   * @param obj The index of the SoundBehavior that follows the desired SoundBehavior.
   */
  previousItem(obj: SoundBehavior): SoundBehavior

  /**
   * Generates a string which, if executed, will return the SoundBehavior.
   */
  toSource(): string
}

/**
 * A behavior object that jumps to an anchor.
 */
declare class GotoAnchorBehavior extends Behavior {
  /**
   * The anchor item, specified as a bookmark or a hyperlink text or page destination.
   */
  anchorItem: Bookmark | HyperlinkTextDestination | HyperlinkPageDestination

  /**
   * The anchor name.
   */
  readonly anchorName: string

  /**
   * The file path (colon delimited on the Mac OS).
   */
  filePath: string | File

  /**
   * The zoom setting.
   */
  zoomSetting: GoToZoomOptions
}

/**
 * A collection of goto anchor behavior objects.
 */
declare class GotoAnchorBehaviors {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the GotoAnchorBehavior with the specified index.
   * @param index The index.
   */
  [index: number]: GotoAnchorBehavior

  /**
   * Creates a new GotoAnchorBehavior.
   * @param withProperties Initial values for properties of the new GotoAnchorBehavior
   */
  add(withProperties: object): GotoAnchorBehavior

  /**
   * Returns any GotoAnchorBehavior in the collection.
   */
  anyItem(): GotoAnchorBehavior

  /**
   * Displays the number of elements in the GotoAnchorBehavior.
   */
  count(): number

  /**
   * Returns every GotoAnchorBehavior in the collection.
   */
  everyItem(): GotoAnchorBehavior

  /**
   * Returns the first GotoAnchorBehavior in the collection.
   */
  firstItem(): GotoAnchorBehavior

  /**
   * Returns the GotoAnchorBehavior with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): GotoAnchorBehavior

  /**
   * Returns the GotoAnchorBehavior with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): GotoAnchorBehavior

  /**
   * Returns the GotoAnchorBehavior with the specified name.
   * @param name The name.
   */
  itemByName(name: string): GotoAnchorBehavior

  /**
   * Returns the GotoAnchorBehaviors within the specified range.
   * @param from The GotoAnchorBehavior, index, or name at the beginning of the range.
   * @param to The GotoAnchorBehavior, index, or name at the end of the range.
   */
  itemByRange(
    from: GotoAnchorBehavior | number | string,
    to: GotoAnchorBehavior | number | string,
  ): GotoAnchorBehavior[]

  /**
   * Returns the last GotoAnchorBehavior in the collection.
   */
  lastItem(): GotoAnchorBehavior

  /**
   * Returns the middle GotoAnchorBehavior in the collection.
   */
  middleItem(): GotoAnchorBehavior

  /**
   * Returns the GotoAnchorBehavior whose index follows the specified GotoAnchorBehavior in the collection.
   * @param obj The GotoAnchorBehavior whose index comes before the desired GotoAnchorBehavior.
   */
  nextItem(obj: GotoAnchorBehavior): GotoAnchorBehavior

  /**
   * Returns the GotoAnchorBehavior with the index previous to the specified index.
   * @param obj The index of the GotoAnchorBehavior that follows the desired GotoAnchorBehavior.
   */
  previousItem(obj: GotoAnchorBehavior): GotoAnchorBehavior

  /**
   * Generates a string which, if executed, will return the GotoAnchorBehavior.
   */
  toSource(): string
}

/**
 * A behavior object that jumps to a specific page of the document.
 */
declare class GotoPageBehavior extends Behavior {
  /**
   * The page number to go to expressed as an index where 1 is the first page.
   */
  pageNumber: number

  /**
   * The zoom setting.
   */
  zoomSetting: GoToZoomOptions
}

/**
 * A collection of go to page behavior objects.
 */
declare class GotoPageBehaviors {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the GotoPageBehavior with the specified index.
   * @param index The index.
   */
  [index: number]: GotoPageBehavior

  /**
   * Creates a new GotoPageBehavior.
   * @param withProperties Initial values for properties of the new GotoPageBehavior
   */
  add(withProperties: object): GotoPageBehavior

  /**
   * Returns any GotoPageBehavior in the collection.
   */
  anyItem(): GotoPageBehavior

  /**
   * Displays the number of elements in the GotoPageBehavior.
   */
  count(): number

  /**
   * Returns every GotoPageBehavior in the collection.
   */
  everyItem(): GotoPageBehavior

  /**
   * Returns the first GotoPageBehavior in the collection.
   */
  firstItem(): GotoPageBehavior

  /**
   * Returns the GotoPageBehavior with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): GotoPageBehavior

  /**
   * Returns the GotoPageBehavior with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): GotoPageBehavior

  /**
   * Returns the GotoPageBehavior with the specified name.
   * @param name The name.
   */
  itemByName(name: string): GotoPageBehavior

  /**
   * Returns the GotoPageBehaviors within the specified range.
   * @param from The GotoPageBehavior, index, or name at the beginning of the range.
   * @param to The GotoPageBehavior, index, or name at the end of the range.
   */
  itemByRange(
    from: GotoPageBehavior | number | string,
    to: GotoPageBehavior | number | string,
  ): GotoPageBehavior[]

  /**
   * Returns the last GotoPageBehavior in the collection.
   */
  lastItem(): GotoPageBehavior

  /**
   * Returns the middle GotoPageBehavior in the collection.
   */
  middleItem(): GotoPageBehavior

  /**
   * Returns the GotoPageBehavior whose index follows the specified GotoPageBehavior in the collection.
   * @param obj The GotoPageBehavior whose index comes before the desired GotoPageBehavior.
   */
  nextItem(obj: GotoPageBehavior): GotoPageBehavior

  /**
   * Returns the GotoPageBehavior with the index previous to the specified index.
   * @param obj The index of the GotoPageBehavior that follows the desired GotoPageBehavior.
   */
  previousItem(obj: GotoPageBehavior): GotoPageBehavior

  /**
   * Generates a string which, if executed, will return the GotoPageBehavior.
   */
  toSource(): string
}

/**
 * A behavior object that clears the form fields in the document.
 */
declare class ClearFormBehavior extends Behavior {}

/**
 * A collection of clear form behavior objects.
 */
declare class ClearFormBehaviors {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the ClearFormBehavior with the specified index.
   * @param index The index.
   */
  [index: number]: ClearFormBehavior

  /**
   * Creates a new ClearFormBehavior.
   * @param withProperties Initial values for properties of the new ClearFormBehavior
   */
  add(withProperties: object): ClearFormBehavior

  /**
   * Returns any ClearFormBehavior in the collection.
   */
  anyItem(): ClearFormBehavior

  /**
   * Displays the number of elements in the ClearFormBehavior.
   */
  count(): number

  /**
   * Returns every ClearFormBehavior in the collection.
   */
  everyItem(): ClearFormBehavior

  /**
   * Returns the first ClearFormBehavior in the collection.
   */
  firstItem(): ClearFormBehavior

  /**
   * Returns the ClearFormBehavior with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): ClearFormBehavior

  /**
   * Returns the ClearFormBehavior with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): ClearFormBehavior

  /**
   * Returns the ClearFormBehavior with the specified name.
   * @param name The name.
   */
  itemByName(name: string): ClearFormBehavior

  /**
   * Returns the ClearFormBehaviors within the specified range.
   * @param from The ClearFormBehavior, index, or name at the beginning of the range.
   * @param to The ClearFormBehavior, index, or name at the end of the range.
   */
  itemByRange(
    from: ClearFormBehavior | number | string,
    to: ClearFormBehavior | number | string,
  ): ClearFormBehavior[]

  /**
   * Returns the last ClearFormBehavior in the collection.
   */
  lastItem(): ClearFormBehavior

  /**
   * Returns the middle ClearFormBehavior in the collection.
   */
  middleItem(): ClearFormBehavior

  /**
   * Returns the ClearFormBehavior whose index follows the specified ClearFormBehavior in the collection.
   * @param obj The ClearFormBehavior whose index comes before the desired ClearFormBehavior.
   */
  nextItem(obj: ClearFormBehavior): ClearFormBehavior

  /**
   * Returns the ClearFormBehavior with the index previous to the specified index.
   * @param obj The index of the ClearFormBehavior that follows the desired ClearFormBehavior.
   */
  previousItem(obj: ClearFormBehavior): ClearFormBehavior

  /**
   * Generates a string which, if executed, will return the ClearFormBehavior.
   */
  toSource(): string
}

/**
 * A behavior object that triggers print for the document.
 */
declare class PrintFormBehavior extends Behavior {}

/**
 * A collection of print form behavior objects.
 */
declare class PrintFormBehaviors {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the PrintFormBehavior with the specified index.
   * @param index The index.
   */
  [index: number]: PrintFormBehavior

  /**
   * Creates a new PrintFormBehavior.
   * @param withProperties Initial values for properties of the new PrintFormBehavior
   */
  add(withProperties: object): PrintFormBehavior

  /**
   * Returns any PrintFormBehavior in the collection.
   */
  anyItem(): PrintFormBehavior

  /**
   * Displays the number of elements in the PrintFormBehavior.
   */
  count(): number

  /**
   * Returns every PrintFormBehavior in the collection.
   */
  everyItem(): PrintFormBehavior

  /**
   * Returns the first PrintFormBehavior in the collection.
   */
  firstItem(): PrintFormBehavior

  /**
   * Returns the PrintFormBehavior with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): PrintFormBehavior

  /**
   * Returns the PrintFormBehavior with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): PrintFormBehavior

  /**
   * Returns the PrintFormBehavior with the specified name.
   * @param name The name.
   */
  itemByName(name: string): PrintFormBehavior

  /**
   * Returns the PrintFormBehaviors within the specified range.
   * @param from The PrintFormBehavior, index, or name at the beginning of the range.
   * @param to The PrintFormBehavior, index, or name at the end of the range.
   */
  itemByRange(
    from: PrintFormBehavior | number | string,
    to: PrintFormBehavior | number | string,
  ): PrintFormBehavior[]

  /**
   * Returns the last PrintFormBehavior in the collection.
   */
  lastItem(): PrintFormBehavior

  /**
   * Returns the middle PrintFormBehavior in the collection.
   */
  middleItem(): PrintFormBehavior

  /**
   * Returns the PrintFormBehavior whose index follows the specified PrintFormBehavior in the collection.
   * @param obj The PrintFormBehavior whose index comes before the desired PrintFormBehavior.
   */
  nextItem(obj: PrintFormBehavior): PrintFormBehavior

  /**
   * Returns the PrintFormBehavior with the index previous to the specified index.
   * @param obj The index of the PrintFormBehavior that follows the desired PrintFormBehavior.
   */
  previousItem(obj: PrintFormBehavior): PrintFormBehavior

  /**
   * Generates a string which, if executed, will return the PrintFormBehavior.
   */
  toSource(): string
}

/**
 * A behavior object that submits the document.
 */
declare class SubmitFormBehavior extends Behavior {
  /**
   * The URL.
   */
  url: string
}

/**
 * A collection of submit form behavior objects.
 */
declare class SubmitFormBehaviors {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the SubmitFormBehavior with the specified index.
   * @param index The index.
   */
  [index: number]: SubmitFormBehavior

  /**
   * Creates a new SubmitFormBehavior.
   * @param withProperties Initial values for properties of the new SubmitFormBehavior
   */
  add(withProperties: object): SubmitFormBehavior

  /**
   * Returns any SubmitFormBehavior in the collection.
   */
  anyItem(): SubmitFormBehavior

  /**
   * Displays the number of elements in the SubmitFormBehavior.
   */
  count(): number

  /**
   * Returns every SubmitFormBehavior in the collection.
   */
  everyItem(): SubmitFormBehavior

  /**
   * Returns the first SubmitFormBehavior in the collection.
   */
  firstItem(): SubmitFormBehavior

  /**
   * Returns the SubmitFormBehavior with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): SubmitFormBehavior

  /**
   * Returns the SubmitFormBehavior with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): SubmitFormBehavior

  /**
   * Returns the SubmitFormBehavior with the specified name.
   * @param name The name.
   */
  itemByName(name: string): SubmitFormBehavior

  /**
   * Returns the SubmitFormBehaviors within the specified range.
   * @param from The SubmitFormBehavior, index, or name at the beginning of the range.
   * @param to The SubmitFormBehavior, index, or name at the end of the range.
   */
  itemByRange(
    from: SubmitFormBehavior | number | string,
    to: SubmitFormBehavior | number | string,
  ): SubmitFormBehavior[]

  /**
   * Returns the last SubmitFormBehavior in the collection.
   */
  lastItem(): SubmitFormBehavior

  /**
   * Returns the middle SubmitFormBehavior in the collection.
   */
  middleItem(): SubmitFormBehavior

  /**
   * Returns the SubmitFormBehavior whose index follows the specified SubmitFormBehavior in the collection.
   * @param obj The SubmitFormBehavior whose index comes before the desired SubmitFormBehavior.
   */
  nextItem(obj: SubmitFormBehavior): SubmitFormBehavior

  /**
   * Returns the SubmitFormBehavior with the index previous to the specified index.
   * @param obj The index of the SubmitFormBehavior that follows the desired SubmitFormBehavior.
   */
  previousItem(obj: SubmitFormBehavior): SubmitFormBehavior

  /**
   * Generates a string which, if executed, will return the SubmitFormBehavior.
   */
  toSource(): string
}

/**
 * Timing settings.
 */
declare class TimingSetting extends Preference {
  /**
   * A collection of timing lists.
   */
  readonly timingLists: TimingLists

  /**
   * Dynamic targets on the spread that are not assigned.
   */
  readonly unassignedDynamicTargets: object[]
}

/**
 * a timing list.
 */
declare class TimingList {
  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The index of the TimingList within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * The parent of the TimingList (a TimingSetting).
   */
  readonly parent: TimingSetting

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * A collection of timing data objects.
   */
  readonly timingGroups: TimingGroups

  /**
   * trigger event.
   */
  readonly triggerEvent: DynamicTriggerEvents

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): TimingList[]

  /**
   * Deletes the TimingList.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the TimingList.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of timing lists.
 */
declare class TimingLists {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the TimingList with the specified index.
   * @param index The index.
   */
  [index: number]: TimingList

  /**
   * Adds a new event-triggered timing list object.
   * @param triggerEvent Trigger Event
   * @param withProperties Initial values for properties of the new TimingList
   */
  add(triggerEvent: DynamicTriggerEvents, withProperties: object): TimingList

  /**
   * Returns any TimingList in the collection.
   */
  anyItem(): TimingList

  /**
   * Displays the number of elements in the TimingList.
   */
  count(): number

  /**
   * Returns every TimingList in the collection.
   */
  everyItem(): TimingList

  /**
   * Returns the first TimingList in the collection.
   */
  firstItem(): TimingList

  /**
   * Returns the TimingList with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): TimingList

  /**
   * Returns the TimingLists within the specified range.
   * @param from The TimingList, index, or name at the beginning of the range.
   * @param to The TimingList, index, or name at the end of the range.
   */
  itemByRange(from: TimingList | number | string, to: TimingList | number | string): TimingList[]

  /**
   * Returns the last TimingList in the collection.
   */
  lastItem(): TimingList

  /**
   * Returns the middle TimingList in the collection.
   */
  middleItem(): TimingList

  /**
   * Returns the TimingList whose index follows the specified TimingList in the collection.
   * @param obj The TimingList whose index comes before the desired TimingList.
   */
  nextItem(obj: TimingList): TimingList

  /**
   * Returns the TimingList with the index previous to the specified index.
   * @param obj The index of the TimingList that follows the desired TimingList.
   */
  previousItem(obj: TimingList): TimingList

  /**
   * Generates a string which, if executed, will return the TimingList.
   */
  toSource(): string
}

/**
 * a timing group.
 */
declare class TimingGroup {
  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The index of the TimingGroup within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * The parent of the TimingGroup (a TimingList).
   */
  readonly parent: TimingList

  /**
   * The placement of the timing group in the timing list.
   */
  placement: number

  /**
   * The number of times this timing group plays.
   */
  plays: number

  /**
   * Set to true if timing group loops.
   */
  playsLoop: boolean

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * A collection of timing target.
   */
  readonly timingTargets: TimingTargets

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): TimingGroup[]

  /**
   * Moves the timing group or target to the specified location.
   * @param to The location in relation to the reference object or within the containing object.
   * @param reference The reference object. Note: Required when the to parameter specifies before or after. .
   */
  move(to: LocationOptions, reference: TimingGroup | TimingTarget | TimingList): TimingGroup

  /**
   * Deletes the TimingGroup.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the TimingGroup.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string

  /**
   * Unlink all targets in the group into separate groups in this timing list.
   */
  unlink(): void
}

/**
 * A collection of timing data objects.
 */
declare class TimingGroups {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the TimingGroup with the specified index.
   * @param index The index.
   */
  [index: number]: TimingGroup

  /**
   * Adds a new timing group.
   * @param dynamicTarget A page item target that is an animation, media, or mso.
   * @param delaySeconds The time delay in seconds for a target.
   * @param withProperties Initial values for properties of the new TimingGroup
   */
  add(
    dynamicTarget: PageItem | Graphic | Behavior | MediaItem,
    delaySeconds?: number,
    withProperties?: object,
  ): TimingGroup

  /**
   * Returns any TimingGroup in the collection.
   */
  anyItem(): TimingGroup

  /**
   * Displays the number of elements in the TimingGroup.
   */
  count(): number

  /**
   * Returns every TimingGroup in the collection.
   */
  everyItem(): TimingGroup

  /**
   * Returns the first TimingGroup in the collection.
   */
  firstItem(): TimingGroup

  /**
   * Returns the TimingGroup with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): TimingGroup

  /**
   * Returns the TimingGroups within the specified range.
   * @param from The TimingGroup, index, or name at the beginning of the range.
   * @param to The TimingGroup, index, or name at the end of the range.
   */
  itemByRange(from: TimingGroup | number | string, to: TimingGroup | number | string): TimingGroup[]

  /**
   * Returns the last TimingGroup in the collection.
   */
  lastItem(): TimingGroup

  /**
   * Returns the middle TimingGroup in the collection.
   */
  middleItem(): TimingGroup

  /**
   * Returns the TimingGroup whose index follows the specified TimingGroup in the collection.
   * @param obj The TimingGroup whose index comes before the desired TimingGroup.
   */
  nextItem(obj: TimingGroup): TimingGroup

  /**
   * Returns the TimingGroup with the index previous to the specified index.
   * @param obj The index of the TimingGroup that follows the desired TimingGroup.
   */
  previousItem(obj: TimingGroup): TimingGroup

  /**
   * Generates a string which, if executed, will return the TimingGroup.
   */
  toSource(): string
}

/**
 * a timing target.
 */
declare class TimingTarget {
  /**
   * The time delay in seconds for a single target or a group of targets after the previoius group has finished.
   */
  delaySeconds: number

  /**
   * A page item target that is an animation, media, or mso.
   */
  dynamicTarget: PageItem | Graphic | Behavior | MediaItem

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The index of the TimingTarget within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * The parent of the TimingTarget (a TimingGroup).
   */
  readonly parent: TimingGroup

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * Set to true if reversing animation on rolloff. Valid only for self rolloff trigger event.
   */
  reverseAnimation: boolean

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): TimingTarget[]

  /**
   * Moves the timing group or target to the specified location.
   * @param to The location in relation to the reference object or within the containing object.
   * @param reference The reference object. Note: Required when the to parameter specifies before or after. .
   */
  move(to: LocationOptions, reference: TimingGroup | TimingTarget | TimingList): TimingTarget

  /**
   * Deletes the TimingTarget.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the TimingTarget.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string

  /**
   * Unlink target from this group and append to the end of this timing list.
   */
  unlink(): void
}

/**
 * A collection of timing target.
 */
declare class TimingTargets {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the TimingTarget with the specified index.
   * @param index The index.
   */
  [index: number]: TimingTarget

  /**
   * Adds a new target item.
   * @param dynamicTarget A page item target that is an animation, media, or mso.
   * @param delaySeconds The time delay in seconds for a target.
   * @param withProperties Initial values for properties of the new TimingTarget
   */
  add(
    dynamicTarget: PageItem | Graphic | Behavior | MediaItem,
    delaySeconds?: number,
    withProperties?: object,
  ): TimingTarget

  /**
   * Returns any TimingTarget in the collection.
   */
  anyItem(): TimingTarget

  /**
   * Displays the number of elements in the TimingTarget.
   */
  count(): number

  /**
   * Returns every TimingTarget in the collection.
   */
  everyItem(): TimingTarget

  /**
   * Returns the first TimingTarget in the collection.
   */
  firstItem(): TimingTarget

  /**
   * Returns the TimingTarget with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): TimingTarget

  /**
   * Returns the TimingTargets within the specified range.
   * @param from The TimingTarget, index, or name at the beginning of the range.
   * @param to The TimingTarget, index, or name at the end of the range.
   */
  itemByRange(
    from: TimingTarget | number | string,
    to: TimingTarget | number | string,
  ): TimingTarget[]

  /**
   * Returns the last TimingTarget in the collection.
   */
  lastItem(): TimingTarget

  /**
   * Returns the middle TimingTarget in the collection.
   */
  middleItem(): TimingTarget

  /**
   * Returns the TimingTarget whose index follows the specified TimingTarget in the collection.
   * @param obj The TimingTarget whose index comes before the desired TimingTarget.
   */
  nextItem(obj: TimingTarget): TimingTarget

  /**
   * Returns the TimingTarget with the index previous to the specified index.
   * @param obj The index of the TimingTarget that follows the desired TimingTarget.
   */
  previousItem(obj: TimingTarget): TimingTarget

  /**
   * Generates a string which, if executed, will return the TimingTarget.
   */
  toSource(): string
}

/**
 * A layer.
 */
declare class Layer {
  /**
   * Lists all graphics contained by the Layer.
   */
  readonly allGraphics: Graphic[]

  /**
   * Lists all page items contained by the Layer.
   */
  readonly allPageItems: PageItem[]

  /**
   * A collection of buttons.
   */
  readonly buttons: Buttons

  /**
   * A collection of checkboxes.
   */
  readonly checkBoxes: CheckBoxes

  /**
   * A collection of comboboxes.
   */
  readonly comboBoxes: ComboBoxes

  /**
   * A collection of endnote text frames.
   */
  readonly endnoteTextFrames: EndnoteTextFrames

  /**
   * EPSTexts
   */
  readonly epstexts: EPSTexts

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * A collection of form fields.
   */
  readonly formFields: FormFields

  /**
   * A collection of graphic lines.
   */
  readonly graphicLines: GraphicLines

  /**
   * A collection of groups.
   */
  readonly groups: Groups

  /**
   * A collection of guides.
   */
  readonly guides: Guides

  /**
   * The unique ID of the Layer.
   */
  readonly id: number

  /**
   * If true, text wrap settings applied to objects on the layer will not affect text on other layers when the layer is hidden.
   */
  ignoreWrap: boolean

  /**
   * The index of the Layer within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The color of the layer, specified either as an array of three doubles, each in the range 0 to 255 and representing R, G, and B values, or as a UI color.
   */
  layerColor: [number, number, number] | UIColors

  /**
   * A collection of listboxes.
   */
  readonly listBoxes: ListBoxes

  /**
   * If true, the guide positions on the layer are locked.
   */
  lockGuides: boolean

  /**
   * If true, the Layer is locked.
   */
  locked: boolean

  /**
   * A collection of multi-state objects.
   */
  readonly multiStateObjects: MultiStateObjects

  /**
   * The name of the Layer.
   */
  name: string

  /**
   * A collection of ellipses.
   */
  readonly ovals: Ovals

  /**
   * The page items collection, which can be used to process all page items in a container (such as a document, page, or group), regardless of type.
   */
  readonly pageItems: PageItems

  /**
   * The parent of the Layer (a Document).
   */
  readonly parent: Document

  /**
   * A collection of polygons.
   */
  readonly polygons: Polygons

  /**
   * If true, the layer will print.
   */
  printable: boolean

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * A collection of radio buttons.
   */
  readonly radioButtons: RadioButtons

  /**
   * A collection of rectangles.
   */
  readonly rectangles: Rectangles

  /**
   * If true, guides are visible on the layer.
   */
  showGuides: boolean

  /**
   * A collection of signature fields.
   */
  readonly signatureFields: SignatureFields

  /**
   * The spline items collection.
   */
  readonly splineItems: SplineItems

  /**
   * A collection of text boxes.
   */
  readonly textBoxes: TextBoxes

  /**
   * A collection of text frames.
   */
  readonly textFrames: TextFrames

  /**
   * If true, the Layer is visible.
   */
  visible: boolean

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Duplicates the Layer.
   */
  duplicate(): Layer

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): Layer[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Merges the layer with other layer(s).
   * @param with_ The layer(s) with which to merge.
   */
  merge(with_: Layer[]): Layer

  /**
   * Moves the Layer to the specified location.
   * @param to The location relative to the reference object or within the containing object.
   * @param reference The reference object. Note: Required when the to value specifies before or after.
   */
  move(to: LocationOptions, reference: Layer): Layer

  /**
   * Deletes the Layer.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the Layer.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of layers.
 */
declare class Layers {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Layer with the specified index.
   * @param index The index.
   */
  [index: number]: Layer

  /**
   * Creates a new Layer.
   * @param withProperties Initial values for properties of the new Layer
   */
  add(withProperties: object): Layer

  /**
   * Returns any Layer in the collection.
   */
  anyItem(): Layer

  /**
   * Displays the number of elements in the Layer.
   */
  count(): number

  /**
   * Returns every Layer in the collection.
   */
  everyItem(): Layer

  /**
   * Returns the first Layer in the collection.
   */
  firstItem(): Layer

  /**
   * Returns the Layer with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Layer

  /**
   * Returns the Layer with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Layer

  /**
   * Returns the Layer with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Layer

  /**
   * Returns the Layers within the specified range.
   * @param from The Layer, index, or name at the beginning of the range.
   * @param to The Layer, index, or name at the end of the range.
   */
  itemByRange(from: Layer | number | string, to: Layer | number | string): Layer[]

  /**
   * Returns the last Layer in the collection.
   */
  lastItem(): Layer

  /**
   * Returns the middle Layer in the collection.
   */
  middleItem(): Layer

  /**
   * Returns the Layer whose index follows the specified Layer in the collection.
   * @param obj The Layer whose index comes before the desired Layer.
   */
  nextItem(obj: Layer): Layer

  /**
   * Returns the Layer with the index previous to the specified index.
   * @param obj The index of the Layer that follows the desired Layer.
   */
  previousItem(obj: Layer): Layer

  /**
   * Generates a string which, if executed, will return the Layer.
   */
  toSource(): string
}

/**
 * A spread.
 */
declare class Spread {
  /**
   * Lists all graphics contained by the Spread.
   */
  readonly allGraphics: Graphic[]

  /**
   * Lists all page items contained by the Spread.
   */
  readonly allPageItems: PageItem[]

  /**
   * If true, guarantees that when pages are added to a spread it will contain a maximum of two pages. If false, allows pages to be added or moved into existing spreads. For override information, see preserve layout when shuffling.
   */
  allowPageShuffle: boolean

  /**
   * The master spread applied to the Spread.
   */
  appliedMaster: MasterSpread | NothingEnum

  /**
   * A collection of buttons.
   */
  readonly buttons: Buttons

  /**
   * A collection of checkboxes.
   */
  readonly checkBoxes: CheckBoxes

  /**
   * A collection of comboboxes.
   */
  readonly comboBoxes: ComboBoxes

  /**
   * A collection of endnote text frames.
   */
  readonly endnoteTextFrames: EndnoteTextFrames

  /**
   * EPSTexts
   */
  readonly epstexts: EPSTexts

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The transparency flattener preferences override for the spread.
   */
  flattenerOverride: SpreadFlattenerLevel

  /**
   * Flattener preference settings.
   */
  readonly flattenerPreferences: FlattenerPreference

  /**
   * A collection of form fields.
   */
  readonly formFields: FormFields

  /**
   * A collection of graphic lines.
   */
  readonly graphicLines: GraphicLines

  /**
   * A collection of groups.
   */
  readonly groups: Groups

  /**
   * A collection of guides.
   */
  readonly guides: Guides

  /**
   * The unique ID of the Spread.
   */
  readonly id: number

  /**
   * The IDML component name of the Spread.
   */
  idmlComponentName: string

  /**
   * The index of the Spread within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * A collection of listboxes.
   */
  readonly listBoxes: ListBoxes

  /**
   * A collection of multi-state objects.
   */
  readonly multiStateObjects: MultiStateObjects

  /**
   * The name of the Spread; this is an alias to the Spread's label property.
   */
  name: string

  /**
   * A collection of ellipses.
   */
  readonly ovals: Ovals

  /**
   * The page items collection, which can be used to process all page items in a container (such as a document, page, or group), regardless of type.
   */
  readonly pageItems: PageItems

  /**
   * The direction of the page transition.
   */
  pageTransitionDirection: PageTransitionDirectionOptions

  /**
   * The duration of the page transition.
   */
  pageTransitionDuration: PageTransitionDurationOptions

  /**
   * The type of page transition.
   */
  pageTransitionType: PageTransitionTypeOptions

  /**
   * A collection of pages.
   */
  readonly pages: Pages

  /**
   * The parent of the Spread (a Document).
   */
  readonly parent: Document

  /**
   * A collection of polygons.
   */
  readonly polygons: Polygons

  /**
   * A collection of preferences objects.
   */
  readonly preferences: Preferences

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * A collection of radio buttons.
   */
  readonly radioButtons: RadioButtons

  /**
   * A collection of rectangles.
   */
  readonly rectangles: Rectangles

  /**
   * If true, displays master page items on document pages in the spread.
   */
  showMasterItems: boolean

  /**
   * A collection of signature fields.
   */
  readonly signatureFields: SignatureFields

  /**
   * The spline items collection.
   */
  readonly splineItems: SplineItems

  /**
   * A collection of text boxes.
   */
  readonly textBoxes: TextBoxes

  /**
   * A collection of text frames.
   */
  readonly textFrames: TextFrames

  /**
   * The object timing settings.
   */
  readonly timingSettings: TimingSetting

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Duplicate an object and place it into the target.
   * @param pageItems One or more page items to place or load
   * @param linkPageItems Whether to link pageItems in content placer (if true it will override link stories value)
   * @param linkStories Whether to link stories in content placer (only applicable for single story, pageItem links will also be created in case of more than one item)
   * @param mapStyles Whether to map styles in content placer
   * @param placePoint The point at which to place
   * @param destinationLayer The layer on which to place
   * @param showingOptions Whether to display the link options dialog
   */
  contentPlace(
    pageItems: PageItem[],
    linkPageItems?: boolean,
    linkStories?: boolean,
    mapStyles?: boolean,
    placePoint?: string[] | number[],
    destinationLayer?: Layer,
    showingOptions?: boolean,
  ): any[]

  /**
   * Creates multiple guides on all pages of the spread.
   * @param numberOfRows The number of rows to create on each page.
   * @param numberOfColumns The number of columns to create on each page.
   * @param rowGutter The height of the gutter between rows.
   * @param columnGutter The width of the gutter between columns.
   * @param guideColor The color to make the guides, specified either as an array of three doubles, each in the range 0 to 255 and representing R, G, and B values, or as a UI color.
   * @param fitMargins If true, the row height and column width are calculated based on the space within the page margins. If false, row height and column width are calculated based on the full page.
   * @param removeExisting If true, removes existing guides when creating new ones.
   * @param layer The layer on which to create the guides.
   */
  createGuides(
    numberOfRows?: number,
    numberOfColumns?: number,
    rowGutter?: string | number,
    columnGutter?: string | number,
    guideColor?: [number, number, number] | UIColors,
    fitMargins?: boolean,
    removeExisting?: boolean,
    layer?: Layer,
  ): void

  /**
   * Detaches an overridden master page item from the master page.
   */
  detach(): void

  /**
   * Duplicates the spread.
   * @param to The location of the spread relative to the reference object or within the document.
   * @param reference The reference object. Note: Required only when the to parameter specifies before or after.
   */
  duplicate(to?: LocationOptions, reference?: Spread | Document | MasterSpread): any

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): Spread[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Moves the spread.
   * @param to The location of the spread relative to the reference object or within the document.
   * @param reference The reference object. Note: Required when the to parameter specifies before or after.
   */
  move(to?: LocationOptions, reference?: Spread | Page | Document): Spread

  /**
   * Places the file.
   * @param fileName The file to place
   * @param placePoint The point at which to place
   * @param destinationLayer The layer on which to place
   * @param showingOptions Whether to display the import options dialog
   * @param autoflowing Whether to autoflow placed text
   * @param withProperties Initial values for properties of the placed object(s)
   */
  place(
    fileName: File,
    placePoint: string[] | number[],
    destinationLayer: Layer,
    showingOptions?: boolean,
    autoflowing?: boolean,
    withProperties?: object,
  ): any[]

  /**
   * Deprecated: Use contentPlace method. Original Description: Create a linked story and place it into the target.
   * @param parentStory The story to place and link from.
   * @param placePoint The point at which to place
   * @param destinationLayer The layer on which to place
   * @param showingOptions Whether to display the link options dialog
   */
  placeAndLink(
    parentStory: Story,
    placePoint: string[] | number[],
    destinationLayer: Layer,
    showingOptions?: boolean,
  ): Story

  /**
   * Places the XML element onto a page. If the place point is above an existing page item, place the XML element into the page item.
   * @param using The XML element to place.
   * @param placePoint The point at which to place the object, specified in the format [x, y].
   * @param autoflowing If true, autoflows placed text.
   */
  placeXML(using: XMLElement, placePoint: string[] | number[], autoflowing?: boolean): PageItem

  /**
   * Deletes the Spread.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Removes the override from a previously overridden master page item.
   */
  removeOverride(): void

  /**
   * Get the coordinates of the given location in the specified coordinate system.
   * @param location The location requested. Can accept: Array of 2 Reals, AnchorPoint enumerator or Array of Arrays of 2 Reals, CoordinateSpaces enumerators, AnchorPoint enumerators, BoundingBoxLimits enumerators or Long Integers.
   * @param in_ The coordinate space to use.
   * @param consideringRulerUnits If true then a ruler location is interpreted using ruler units rather than points. The default value is false. This parameter has no effect unless the reference point is specified relative to a page.
   */
  resolve(location: any, in_: CoordinateSpaces, consideringRulerUnits?: boolean): any

  /**
   * Selects the object.
   * @param existingSelection The selection status of the Spread in relation to previously selected objects.
   */
  select(existingSelection?: SelectionOptions): void

  /**
   * Replaces the content of XML element with content imported from a file.
   * @param using The file path to the import file.
   * @param relativeBasePath Base path used to resolve relative paths.
   */
  setContent(using: string, relativeBasePath: string): PageItem

  /**
   * Generates a string which, if executed, will return the Spread.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string

  /**
   * Transform the page item.
   * @param in_ The coordinate space to use
   * @param from The temporary origin during the transformation. Can accept: Array of 2 Reals, AnchorPoint enumerator or Array of Arrays of 2 Reals, CoordinateSpaces enumerators, AnchorPoint enumerators, BoundingBoxLimits enumerators or Long Integers.
   * @param withMatrix Transform matrix.
   * @param replacingCurrent Transform components to consider; providing this optional parameter causes the target's existing transform components to be replaced with new values.Without this parameter, the given matrix is concatenated onto the target's existing transform combining the effect of the two.
   * @param consideringRulerUnits If true then a ruler based origin is interpreted using ruler units rather than points. The default value is false. This parameter has no effect unless the reference point is specified relative to a page.
   */
  transform(
    in_: CoordinateSpaces,
    from: any,
    withMatrix: [number, number, number, number, number, number] | TransformationMatrix,
    replacingCurrent: MatrixContent | MatrixContent[] | number,
    consideringRulerUnits?: boolean,
  ): void

  /**
   * Get the transformation values of the page item.
   * @param in_ The coordinate space to use
   */
  transformValuesOf(in_: CoordinateSpaces): TransformationMatrix[]
}

/**
 * A collection of spreads.
 */
declare class Spreads {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Spread with the specified index.
   * @param index The index.
   */
  [index: number]: Spread

  /**
   * Creates a new spread.
   * @param at The location of the spread relative to the reference object or within the document.
   * @param reference The reference object. Note: Required when the at parameter specifies before or after.
   * @param withProperties Initial values for properties of the new Spread
   */
  add(at?: LocationOptions, reference?: Spread | Document, withProperties?: object): Spread

  /**
   * Returns any Spread in the collection.
   */
  anyItem(): Spread

  /**
   * Displays the number of elements in the Spread.
   */
  count(): number

  /**
   * Returns every Spread in the collection.
   */
  everyItem(): Spread

  /**
   * Returns the first Spread in the collection.
   */
  firstItem(): Spread

  /**
   * Returns the Spread with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Spread

  /**
   * Returns the Spread with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Spread

  /**
   * Returns the Spread with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Spread

  /**
   * Returns the Spreads within the specified range.
   * @param from The Spread, index, or name at the beginning of the range.
   * @param to The Spread, index, or name at the end of the range.
   */
  itemByRange(from: Spread | number | string, to: Spread | number | string): Spread[]

  /**
   * Returns the last Spread in the collection.
   */
  lastItem(): Spread

  /**
   * Returns the middle Spread in the collection.
   */
  middleItem(): Spread

  /**
   * Returns the Spread whose index follows the specified Spread in the collection.
   * @param obj The Spread whose index comes before the desired Spread.
   */
  nextItem(obj: Spread): Spread

  /**
   * Returns the Spread with the index previous to the specified index.
   * @param obj The index of the Spread that follows the desired Spread.
   */
  previousItem(obj: Spread): Spread

  /**
   * Generates a string which, if executed, will return the Spread.
   */
  toSource(): string
}

/**
 * A page.
 */
declare class Page {
  /**
   * Lists all graphics contained by the Page.
   */
  readonly allGraphics: Graphic[]

  /**
   * Lists all page items contained by the Page.
   */
  readonly allPageItems: PageItem[]

  /**
   * The alternate layout section to which the page belongs.
   */
  readonly appliedAlternateLayout: Section

  /**
   * The master spread applied to the Page.
   */
  appliedMaster: MasterSpread | NothingEnum

  /**
   * The section to which the page belongs.
   */
  readonly appliedSection: Section

  /**
   * The trap preset applied to the page.
   */
  appliedTrapPreset: TrapPreset | string

  /**
   * The bounds of the Page, in the format [y1, x1, y2, x2].
   */
  readonly bounds: string[] | number[]

  /**
   * A collection of buttons.
   */
  readonly buttons: Buttons

  /**
   * A collection of checkboxes.
   */
  readonly checkBoxes: CheckBoxes

  /**
   * A collection of comboboxes.
   */
  readonly comboBoxes: ComboBoxes

  /**
   * The sequential number of the page within the document.
   */
  readonly documentOffset: number

  /**
   * EPSTexts
   */
  readonly epstexts: EPSTexts

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * A collection of form fields.
   */
  readonly formFields: FormFields

  /**
   * A collection of graphic lines.
   */
  readonly graphicLines: GraphicLines

  /**
   * Default grid properties. Note: Applies to named, layout, and frame (story) grids.
   */
  readonly gridData: GridDataInformation

  /**
   * A collection of groups.
   */
  readonly groups: Groups

  /**
   * A collection of guides.
   */
  readonly guides: Guides

  /**
   * The unique ID of the Page.
   */
  readonly id: number

  /**
   * The index of the Page within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * layout rule
   */
  layoutRule: LayoutRuleOptions

  /**
   * A collection of listboxes.
   */
  readonly listBoxes: ListBoxes

  /**
   * Margin preference settings.
   */
  readonly marginPreferences: MarginPreference

  /**
   * The items on a specified document page that originated on the applied master page and have not been overridden or detached.
   */
  readonly masterPageItems: PageItem[] | Guides | Graphics | Movies | Sounds

  /**
   * The transform applied to the master page before it is applied to Page.
   */
  masterPageTransform: TransformationMatrix

  /**
   * A collection of multi-state objects.
   */
  readonly multiStateObjects: MultiStateObjects

  /**
   * The name of the Page.
   */
  readonly name: string

  /**
   * optional page for HTML5 pagination. Obsolete after CS6
   */
  optionalPage: boolean

  /**
   * A collection of ellipses.
   */
  readonly ovals: Ovals

  /**
   * The color label of the Page, specified either asan array of three doubles, each in the range 0 to 255 and representing R, G, and B values, or as a UI color.
   */
  pageColor: [number, number, number] | UIColors | PageColorOptions

  /**
   * The page items collection, which can be used to process all page items in a container (such as a document, page, or group), regardless of type.
   */
  readonly pageItems: PageItems

  /**
   * The parent of the Page (a Spread or MasterSpread).
   */
  readonly parent: any

  /**
   * A collection of polygons.
   */
  readonly polygons: Polygons

  /**
   * A collection of preferences objects.
   */
  readonly preferences: Preferences

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * A collection of radio buttons.
   */
  readonly radioButtons: RadioButtons

  /**
   * A collection of rectangles.
   */
  readonly rectangles: Rectangles

  /**
   * The side of the binding spine on which to place the page within the spread.
   */
  readonly side: PageSideOptions

  /**
   * A collection of signature fields.
   */
  readonly signatureFields: SignatureFields

  /**
   * snapshot blending mode
   */
  snapshotBlendingMode: SnapshotBlendingModes

  /**
   * The spline items collection.
   */
  readonly splineItems: SplineItems

  /**
   * The order in which the focus moves to different form fields in the PDF when the tab key is pressed.
   */
  tabOrder:
    | Button[]
    | CheckBoxes
    | ComboBoxes
    | ListBoxes
    | RadioButtons
    | TextBoxes
    | SignatureFields

  /**
   * A collection of text boxes.
   */
  readonly textBoxes: TextBoxes

  /**
   * A collection of text frames.
   */
  readonly textFrames: TextFrames

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Duplicate an object and place it into the target.
   * @param pageItems One or more page items to place or load
   * @param linkPageItems Whether to link pageItems in content placer (if true it will override link stories value)
   * @param linkStories Whether to link stories in content placer (only applicable for single story, pageItem links will also be created in case of more than one item)
   * @param mapStyles Whether to map styles in content placer
   * @param placePoint The point at which to place
   * @param destinationLayer The layer on which to place
   * @param showingOptions Whether to display the link options dialog
   */
  contentPlace(
    pageItems: PageItem[],
    linkPageItems?: boolean,
    linkStories?: boolean,
    mapStyles?: boolean,
    placePoint?: string[] | number[],
    destinationLayer?: Layer,
    showingOptions?: boolean,
  ): any[]

  /**
   * Delete all layout snapshots for this Page.
   */
  deleteAllLayoutSnapshots(): void

  /**
   * Delete the snapshot of the layout for the current Page size and shape.
   */
  deleteLayoutSnapshot(): void

  /**
   * Detaches an overridden master page item from the master page.
   */
  detach(): void

  /**
   * Duplicates the page.
   * @param to The location at which to place the duplicate page relative to the reference object or within the document or spread.
   * @param reference The reference object. Note: Required when the to value specifies before or after.
   */
  duplicate(to?: LocationOptions, reference?: Page | Spread): Page

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): Page[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Moves the page.
   * @param to The new location of the page relative to the reference object or within the document or spread.
   * @param reference The reference object. Note: Required when the to parameter value specifies before or after.
   * @param binding The location of the binding spine in spreads.
   */
  move(to?: LocationOptions, reference?: Page | Spread, binding?: BindingOptions): Page

  /**
   * Places the file.
   * @param fileName The file to place
   * @param placePoint The point at which to place
   * @param destinationLayer The layer on which to place
   * @param showingOptions Whether to display the import options dialog
   * @param autoflowing Whether to autoflow placed text
   * @param withProperties Initial values for properties of the placed object(s)
   */
  place(
    fileName: File,
    placePoint: string[] | number[],
    destinationLayer: Layer,
    showingOptions?: boolean,
    autoflowing?: boolean,
    withProperties?: object,
  ): any[]

  /**
   * Deprecated: Use contentPlace method. Original Description: Create a linked story and place it into the target.
   * @param parentStory The story to place and link from.
   * @param placePoint The point at which to place
   * @param destinationLayer The layer on which to place
   * @param showingOptions Whether to display the link options dialog
   */
  placeAndLink(
    parentStory: Story,
    placePoint: string[] | number[],
    destinationLayer: Layer,
    showingOptions?: boolean,
  ): Story

  /**
   * Places the XML element onto a page. If the place point is above an existing page item, place the XML element into the page item.
   * @param using The XML element to place.
   * @param placePoint The point at which to place the object, specified in the format [x, y].
   * @param autoflowing If true, autoflows placed text.
   */
  placeXML(using: XMLElement, placePoint: string[] | number[], autoflowing?: boolean): PageItem

  /**
   * Move the bounding box of the page item
   * @param in_ The bounding box to resize. Can accept: CoordinateSpaces enumerator or Ordered array containing coordinateSpace:CoordinateSpaces enumerator, boundsKind:BoundingBoxLimits enumerator.
   * @param opposingCorners Opposing corners of new bounding box in the given coordinate space
   */
  reframe(in_: any, opposingCorners: any[]): void

  /**
   * Deletes the Page.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Removes the override from a previously overridden master page item.
   */
  removeOverride(): void

  /**
   * Resize the page item.
   * @param in_ The bounding box to resize. Can accept: CoordinateSpaces enumerator, BoundingBoxLimits enumerator or Ordered array containing coordinateSpace:CoordinateSpaces enumerator, boundsKind:BoundingBoxLimits enumerator.
   * @param from The transform origin. Legal specifications: relative to bounding box: anchor | {anchor | {x,y}, bounds kind [, coordinate space]}; relative to coordinate space: {x,y} | {{x,y}[, coordinate space]}; relative to layout window ruler: {{x,y}, page index | bounds kind}. Can accept: Array of 2 Reals, AnchorPoint enumerator or Array of Arrays of 2 Reals, CoordinateSpaces enumerators, AnchorPoint enumerators, BoundingBoxLimits enumerators or Long Integers.
   * @param by How the current dimensions are affected by the given values
   * @param values The width and height values. Legal dimensions specifications: {x, y [, coordinate space]}, {x, resize constraint [, coordinate space]}, or {resize constraint, y [, coordinate space]}; where x and y are real numbers and coordinate space is used to determine _only_ the unit of length for x and y; coordinate space is ignored for the 'current dimensions times' resize method).
   * @param resizeIndividually If false and multiple page items are targeted, the new dimensions are attained only by moving the individual items rather than resizing them.
   * @param consideringRulerUnits If true then a ruler location is interpreted using ruler units rather than points. The default value is false. This parameter has no effect unless the reference point is specified relative to a page.
   */
  resize(
    in_: any,
    from: any,
    by: ResizeMethods,
    values: number[] | ResizeConstraints | CoordinateSpaces,
    resizeIndividually?: boolean,
    consideringRulerUnits?: boolean,
  ): void

  /**
   * Get the coordinates of the given location in the specified coordinate system.
   * @param location The location requested. Can accept: Array of 2 Reals, AnchorPoint enumerator or Array of Arrays of 2 Reals, CoordinateSpaces enumerators, AnchorPoint enumerators, BoundingBoxLimits enumerators or Long Integers.
   * @param in_ The coordinate space to use.
   * @param consideringRulerUnits If true then a ruler location is interpreted using ruler units rather than points. The default value is false. This parameter has no effect unless the reference point is specified relative to a page.
   */
  resolve(location: any, in_: CoordinateSpaces, consideringRulerUnits?: boolean): any

  /**
   * Selects the object.
   * @param existingSelection The selection status of the Page in relation to previously selected objects.
   */
  select(existingSelection?: SelectionOptions): void

  /**
   * Replaces the content of XML element with content imported from a file.
   * @param using The file path to the import file.
   * @param relativeBasePath Base path used to resolve relative paths.
   */
  setContent(using: string, relativeBasePath: string): PageItem

  /**
   * Create a snapshot of the layout for the current Page size and shape.
   */
  snapshotCurrentLayout(): void

  /**
   * Generates a string which, if executed, will return the Page.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string

  /**
   * Transform the page item.
   * @param in_ The coordinate space to use
   * @param from The temporary origin during the transformation. Can accept: Array of 2 Reals, AnchorPoint enumerator or Array of Arrays of 2 Reals, CoordinateSpaces enumerators, AnchorPoint enumerators, BoundingBoxLimits enumerators or Long Integers.
   * @param withMatrix Transform matrix.
   * @param replacingCurrent Transform components to consider; providing this optional parameter causes the target's existing transform components to be replaced with new values.Without this parameter, the given matrix is concatenated onto the target's existing transform combining the effect of the two.
   * @param consideringRulerUnits If true then a ruler based origin is interpreted using ruler units rather than points. The default value is false. This parameter has no effect unless the reference point is specified relative to a page.
   */
  transform(
    in_: CoordinateSpaces,
    from: any,
    withMatrix: [number, number, number, number, number, number] | TransformationMatrix,
    replacingCurrent: MatrixContent | MatrixContent[] | number,
    consideringRulerUnits?: boolean,
  ): void

  /**
   * Get the transformation values of the page item.
   * @param in_ The coordinate space to use
   */
  transformValuesOf(in_: CoordinateSpaces): TransformationMatrix[]
}

/**
 * A collection of pages.
 */
declare class Pages {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Page with the specified index.
   * @param index The index.
   */
  [index: number]: Page

  /**
   * Creates a new page.
   * @param at The location of the new page relative to the reference object or within the document or spread.
   * @param reference The reference object. Note: Required when the at parameter value specifies before or after.
   * @param withProperties Initial values for properties of the new Page
   */
  add(
    at?: LocationOptions,
    reference?: Page | Spread | MasterSpread | Document,
    withProperties?: object,
  ): Page

  /**
   * Returns any Page in the collection.
   */
  anyItem(): Page

  /**
   * Displays the number of elements in the Page.
   */
  count(): number

  /**
   * Returns every Page in the collection.
   */
  everyItem(): Page

  /**
   * Returns the first Page in the collection.
   */
  firstItem(): Page

  /**
   * Returns the Page with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Page

  /**
   * Returns the Page with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Page

  /**
   * Returns the Page with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Page

  /**
   * Returns the Pages within the specified range.
   * @param from The Page, index, or name at the beginning of the range.
   * @param to The Page, index, or name at the end of the range.
   */
  itemByRange(from: Page | number | string, to: Page | number | string): Page[]

  /**
   * Returns the last Page in the collection.
   */
  lastItem(): Page

  /**
   * Returns the middle Page in the collection.
   */
  middleItem(): Page

  /**
   * Returns the Page whose index follows the specified Page in the collection.
   * @param obj The Page whose index comes before the desired Page.
   */
  nextItem(obj: Page): Page

  /**
   * Returns the Page with the index previous to the specified index.
   * @param obj The index of the Page that follows the desired Page.
   */
  previousItem(obj: Page): Page

  /**
   * Generates a string which, if executed, will return the Page.
   */
  toSource(): string
}

/**
 * A master spread.
 */
declare class MasterSpread {
  /**
   * Lists all graphics contained by the MasterSpread.
   */
  readonly allGraphics: Graphic[]

  /**
   * Lists all page items contained by the MasterSpread.
   */
  readonly allPageItems: PageItem[]

  /**
   * The master spread applied to the MasterSpread.
   */
  appliedMaster: MasterSpread | NothingEnum

  /**
   * The name of the master spread.
   */
  baseName: string

  /**
   * A collection of buttons.
   */
  readonly buttons: Buttons

  /**
   * A collection of checkboxes.
   */
  readonly checkBoxes: CheckBoxes

  /**
   * A collection of comboboxes.
   */
  readonly comboBoxes: ComboBoxes

  /**
   * A collection of endnote text frames.
   */
  readonly endnoteTextFrames: EndnoteTextFrames

  /**
   * EPSTexts
   */
  readonly epstexts: EPSTexts

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * A collection of form fields.
   */
  readonly formFields: FormFields

  /**
   * A collection of graphic lines.
   */
  readonly graphicLines: GraphicLines

  /**
   * A collection of groups.
   */
  readonly groups: Groups

  /**
   * A collection of guides.
   */
  readonly guides: Guides

  /**
   * The unique ID of the MasterSpread.
   */
  readonly id: number

  /**
   * The IDML component name of the MasterSpread.
   */
  idmlComponentName: string

  /**
   * The index of the MasterSpread within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * A collection of listboxes.
   */
  readonly listBoxes: ListBoxes

  /**
   * A collection of multi-state objects.
   */
  readonly multiStateObjects: MultiStateObjects

  /**
   * The name of the MasterSpread.
   */
  readonly name: string

  /**
   * The prefix of the master spread name.
   */
  namePrefix: string

  /**
   * A collection of ellipses.
   */
  readonly ovals: Ovals

  /**
   * The color label of the MasterSpread, specified either asan array of three doubles, each in the range 0 to 255 and representing R, G, and B values, or as a UI color.
   */
  pageColor: [number, number, number] | UIColors | PageColorOptions

  /**
   * The page items collection, which can be used to process all page items in a container (such as a document, page, or group), regardless of type.
   */
  readonly pageItems: PageItems

  /**
   * A collection of pages.
   */
  readonly pages: Pages

  /**
   * The parent of the MasterSpread (a Document).
   */
  readonly parent: Document

  /**
   * A collection of polygons.
   */
  readonly polygons: Polygons

  /**
   * A collection of preferences objects.
   */
  readonly preferences: Preferences

  /**
   * The primary text frame through which text flows on the MasterSpread. Must be a text frame or a type on a path spline.
   */
  primaryTextFrame: PageItem | NothingEnum

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * A collection of radio buttons.
   */
  readonly radioButtons: RadioButtons

  /**
   * A collection of rectangles.
   */
  readonly rectangles: Rectangles

  /**
   * If true, displays master page items on document pages in the spread.
   */
  showMasterItems: boolean

  /**
   * A collection of signature fields.
   */
  readonly signatureFields: SignatureFields

  /**
   * The spline items collection.
   */
  readonly splineItems: SplineItems

  /**
   * A collection of text boxes.
   */
  readonly textBoxes: TextBoxes

  /**
   * A collection of text frames.
   */
  readonly textFrames: TextFrames

  /**
   * The object timing settings.
   */
  readonly timingSettings: TimingSetting

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Duplicate an object and place it into the target.
   * @param pageItems One or more page items to place or load
   * @param linkPageItems Whether to link pageItems in content placer (if true it will override link stories value)
   * @param linkStories Whether to link stories in content placer (only applicable for single story, pageItem links will also be created in case of more than one item)
   * @param mapStyles Whether to map styles in content placer
   * @param placePoint The point at which to place
   * @param destinationLayer The layer on which to place
   * @param showingOptions Whether to display the link options dialog
   */
  contentPlace(
    pageItems: PageItem[],
    linkPageItems?: boolean,
    linkStories?: boolean,
    mapStyles?: boolean,
    placePoint?: string[] | number[],
    destinationLayer?: Layer,
    showingOptions?: boolean,
  ): any[]

  /**
   * Creates multiple guides on all pages of the spread.
   * @param numberOfRows The number of rows to create on each page.
   * @param numberOfColumns The number of columns to create on each page.
   * @param rowGutter The height of the gutter between rows.
   * @param columnGutter The width of the gutter between columns.
   * @param guideColor The color to make the guides, specified either as an array of three doubles, each in the range 0 to 255 and representing R, G, and B values, or as a UI color.
   * @param fitMargins If true, the row height and column width are calculated based on the space within the page margins. If false, row height and column width are calculated based on the full page.
   * @param removeExisting If true, removes existing guides when creating new ones.
   * @param layer The layer on which to create the guides.
   */
  createGuides(
    numberOfRows?: number,
    numberOfColumns?: number,
    rowGutter?: string | number,
    columnGutter?: string | number,
    guideColor?: [number, number, number] | UIColors,
    fitMargins?: boolean,
    removeExisting?: boolean,
    layer?: Layer,
  ): void

  /**
   * Detaches an overridden master page item from the master page.
   */
  detach(): void

  /**
   * Duplicates the spread.
   * @param to The location of the spread relative to the reference object or within the document.
   * @param reference The reference object. Note: Required only when the to parameter specifies before or after.
   */
  duplicate(to?: LocationOptions, reference?: MasterSpread | Document): any

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): MasterSpread[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Places the file.
   * @param fileName The file to place
   * @param placePoint The point at which to place
   * @param destinationLayer The layer on which to place
   * @param showingOptions Whether to display the import options dialog
   * @param autoflowing Whether to autoflow placed text
   * @param withProperties Initial values for properties of the placed object(s)
   */
  place(
    fileName: File,
    placePoint: string[] | number[],
    destinationLayer: Layer,
    showingOptions?: boolean,
    autoflowing?: boolean,
    withProperties?: object,
  ): any[]

  /**
   * Deprecated: Use contentPlace method. Original Description: Create a linked story and place it into the target.
   * @param parentStory The story to place and link from.
   * @param placePoint The point at which to place
   * @param destinationLayer The layer on which to place
   * @param showingOptions Whether to display the link options dialog
   */
  placeAndLink(
    parentStory: Story,
    placePoint: string[] | number[],
    destinationLayer: Layer,
    showingOptions?: boolean,
  ): Story

  /**
   * Deletes the MasterSpread.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Removes the override from a previously overridden master page item.
   */
  removeOverride(): void

  /**
   * Get the coordinates of the given location in the specified coordinate system.
   * @param location The location requested. Can accept: Array of 2 Reals, AnchorPoint enumerator or Array of Arrays of 2 Reals, CoordinateSpaces enumerators, AnchorPoint enumerators, BoundingBoxLimits enumerators or Long Integers.
   * @param in_ The coordinate space to use.
   * @param consideringRulerUnits If true then a ruler location is interpreted using ruler units rather than points. The default value is false. This parameter has no effect unless the reference point is specified relative to a page.
   */
  resolve(location: any, in_: CoordinateSpaces, consideringRulerUnits?: boolean): any

  /**
   * Selects the object.
   * @param existingSelection The selection status of the MasterSpread in relation to previously selected objects.
   */
  select(existingSelection?: SelectionOptions): void

  /**
   * Generates a string which, if executed, will return the MasterSpread.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string

  /**
   * Transform the page item.
   * @param in_ The coordinate space to use
   * @param from The temporary origin during the transformation. Can accept: Array of 2 Reals, AnchorPoint enumerator or Array of Arrays of 2 Reals, CoordinateSpaces enumerators, AnchorPoint enumerators, BoundingBoxLimits enumerators or Long Integers.
   * @param withMatrix Transform matrix.
   * @param replacingCurrent Transform components to consider; providing this optional parameter causes the target's existing transform components to be replaced with new values.Without this parameter, the given matrix is concatenated onto the target's existing transform combining the effect of the two.
   * @param consideringRulerUnits If true then a ruler based origin is interpreted using ruler units rather than points. The default value is false. This parameter has no effect unless the reference point is specified relative to a page.
   */
  transform(
    in_: CoordinateSpaces,
    from: any,
    withMatrix: [number, number, number, number, number, number] | TransformationMatrix,
    replacingCurrent: MatrixContent | MatrixContent[] | number,
    consideringRulerUnits?: boolean,
  ): void

  /**
   * Get the transformation values of the page item.
   * @param in_ The coordinate space to use
   */
  transformValuesOf(in_: CoordinateSpaces): TransformationMatrix[]
}

/**
 * A collection of master spreads.
 */
declare class MasterSpreads {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the MasterSpread with the specified index.
   * @param index The index.
   */
  [index: number]: MasterSpread

  /**
   * Creates a new master spread.
   * @param pagesPerSpread The number of pages to include in the master spread.
   * @param withProperties Initial values for properties of the new MasterSpread
   */
  add(pagesPerSpread: number, withProperties: object): MasterSpread

  /**
   * Returns any MasterSpread in the collection.
   */
  anyItem(): MasterSpread

  /**
   * Displays the number of elements in the MasterSpread.
   */
  count(): number

  /**
   * Returns every MasterSpread in the collection.
   */
  everyItem(): MasterSpread

  /**
   * Returns the first MasterSpread in the collection.
   */
  firstItem(): MasterSpread

  /**
   * Returns the MasterSpread with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): MasterSpread

  /**
   * Returns the MasterSpread with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): MasterSpread

  /**
   * Returns the MasterSpread with the specified name.
   * @param name The name.
   */
  itemByName(name: string): MasterSpread

  /**
   * Returns the MasterSpreads within the specified range.
   * @param from The MasterSpread, index, or name at the beginning of the range.
   * @param to The MasterSpread, index, or name at the end of the range.
   */
  itemByRange(
    from: MasterSpread | number | string,
    to: MasterSpread | number | string,
  ): MasterSpread[]

  /**
   * Returns the last MasterSpread in the collection.
   */
  lastItem(): MasterSpread

  /**
   * Returns the middle MasterSpread in the collection.
   */
  middleItem(): MasterSpread

  /**
   * Returns the MasterSpread whose index follows the specified MasterSpread in the collection.
   * @param obj The MasterSpread whose index comes before the desired MasterSpread.
   */
  nextItem(obj: MasterSpread): MasterSpread

  /**
   * Returns the MasterSpread with the index previous to the specified index.
   * @param obj The index of the MasterSpread that follows the desired MasterSpread.
   */
  previousItem(obj: MasterSpread): MasterSpread

  /**
   * Generates a string which, if executed, will return the MasterSpread.
   */
  toSource(): string
}

/**
 * PDF attributes.
 */
declare class PDFAttribute extends Preference {
  /**
   * The page number of the PDF document page to place.
   */
  readonly pageNumber: number

  /**
   * The type of cropping to apply.
   */
  readonly pdfCrop: PDFCrop

  /**
   * If true, the background of the PDF is transparent.
   */
  readonly transparentBackground: boolean
}

/**
 * An item on a page, including rectangles, ellipses, graphic lines, polygons, groups, text frames, and buttons.
 */
declare class PageItem {
  /**
   * Dispatched after a PageItem is placed. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_PLACE: string

  /**
   * Dispatched before a PageItem is placed. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_PLACE: string

  /**
   * Indicates whether the PageItem has been flipped independently of its parent object and, if yes, the direction in which the PageItem was flipped.
   */
  absoluteFlip: Flip

  /**
   * The horizontal scale of the PageItem relative to its containing object.
   */
  absoluteHorizontalScale: number

  /**
   * The rotation angle of the PageItem relative to its containing object. (Range: -360 to 360)
   */
  absoluteRotationAngle: number

  /**
   * The skewing angle of the PageItem relative to its containing object. (Range: -360 to 360)
   */
  absoluteShearAngle: number

  /**
   * The vertical scale of the PageItem relative to its containing object.
   */
  absoluteVerticalScale: number

  /**
   * The list of all articles this page item is part of
   */
  readonly allArticles: Article[]

  /**
   * Lists all graphics contained by the PageItem.
   */
  readonly allGraphics: Graphic[]

  /**
   * Lists all page items contained by the PageItem.
   */
  readonly allPageItems: PageItem[]

  /**
   * If true, the master page item can be overridden.
   */
  allowOverrides: boolean

  /**
   * The page item animation settings.
   */
  readonly animationSettings: AnimationSetting

  /**
   * The object style applied to the PageItem.
   */
  appliedObjectStyle: ObjectStyle

  /**
   * The arrowhead alignment applied to the PageItem.
   */
  arrowHeadAlignment: ArrowHeadAlignmentEnum

  /**
   * The XML element associated with the PageItem.
   */
  readonly associatedXMLElement: XMLItem

  /**
   * The shape to apply to the bottom left corner of rectangular shapes.
   */
  bottomLeftCornerOption: CornerOptions

  /**
   * The radius in measurement units of the corner effect applied to the bottom left corner of rectangular shapes
   */
  bottomLeftCornerRadius: string | number

  /**
   * The shape to apply to the bottom right corner of rectangular shapes.
   */
  bottomRightCornerOption: CornerOptions

  /**
   * The radius in measurement units of the corner effect applied to the bottom right corner of rectangular shapes
   */
  bottomRightCornerRadius: string | number

  /**
   * Transparency settings for the content of the PageItem.
   */
  readonly contentTransparencySettings: ContentTransparencySetting

  /**
   * The end shape of an open path.
   */
  endCap: EndCap

  /**
   * The corner join applied to the PageItem.
   */
  endJoin: EndJoin

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The swatch (color, gradient, tint, or mixed ink) applied to the fill of the PageItem. .
   */
  fillColor: Swatch | string

  /**
   * The percent of tint to use in the PageItem's fill color. (To specify a tint percent, use a number in the range of 0 to 100; to use the inherited or overridden value, use -1.)
   */
  fillTint: number

  /**
   * Transparency settings for the fill applied to the PageItem.
   */
  readonly fillTransparencySettings: FillTransparencySetting

  /**
   * The direction in which to flip the printed image.
   */
  flip: Flip

  /**
   * The swatch (color, gradient, tint, or mixed ink) applied to the gap of a dashed, dotted, or striped stroke. For information, see stroke type.
   */
  gapColor: Swatch

  /**
   * The tint as a percentage of the gap color. (To specify a tint percent, use a number in the range of 0 to 100; to use the inherited or overridden value, use -1.)
   */
  gapTint: number

  /**
   * The bounds of the PageItem excluding the stroke width, in the format [y1, x1, y2, x2], which give the coordinates of the top-left and bottom-right corners of the bounding box.
   */
  geometricBounds: string[] | number[]

  /**
   * The angle of a linear gradient applied to the fill of the PageItem. (Range: -180 to 180)
   */
  gradientFillAngle: number

  /**
   * The length (for a linear gradient) or radius (for a radial gradient) applied to the fill of the PageItem.
   */
  gradientFillLength: string | number

  /**
   * The starting point (in page coordinates) of a gradient applied to the fill of the PageItem, in the format [x, y].
   */
  gradientFillStart: string[] | number[]

  /**
   * The angle of a linear gradient applied to the stroke of the PageItem. (Range: -180 to 180)
   */
  gradientStrokeAngle: number

  /**
   * The length (for a linear gradient) or radius (for a radial gradient) applied to the stroke of the PageItem.
   */
  gradientStrokeLength: string | number

  /**
   * The starting point (in page coordinates) of a gradient applied to the stroke of the PageItem, in the format [x, y].
   */
  gradientStrokeStart: string[] | number[]

  /**
   * The left margin, width, and right margin constraints this item is subject to when using the object-based layout rule.
   */
  horizontalLayoutConstraints: DimensionsConstraints[]

  /**
   * The horizontal scaling applied to the PageItem.
   */
  horizontalScale: number

  /**
   * The unique ID of the PageItem.
   */
  readonly id: number

  /**
   * The index of the PageItem within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * The layer that the PageItem is on.
   */
  itemLayer: Layer

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The scaling applied to the arrowhead at the start of the path. (Range: 1 to 1000)
   */
  leftArrowHeadScale: number

  /**
   * The arrowhead applied to the start of the path.
   */
  leftLineEnd: ArrowHead

  /**
   * Linked Page Item options
   */
  readonly linkedPageItemOptions: LinkedPageItemOption

  /**
   * Display performance options for the PageItem.
   */
  localDisplaySetting: DisplaySettingOptions

  /**
   * If true, the PageItem is locked.
   */
  locked: boolean

  /**
   * The limit of the ratio of stroke width to miter length before a miter (pointed) join becomes a bevel (squared-off) join.
   */
  miterLimit: number

  /**
   * The name of the PageItem.
   */
  name: string

  /**
   * If true, the PageItem does not print.
   */
  nonprinting: boolean

  /**
   * If true, the PageItem's fill color overprints any underlying objects. If false, the fill color knocks out the underlying colors.
   */
  overprintFill: boolean

  /**
   * If true, the gap color overprints any underlying colors. If false, the gap color knocks out the underlying colors.
   */
  overprintGap: boolean

  /**
   * If true, the PageItem's stroke color overprints any underlying objects. If false, the stroke color knocks out theunderlying colors.
   */
  overprintStroke: boolean

  /**
   * If true, the object originated on a master spread and was overridden. If false, the object either originated on a master spread and was not overridden, or the object did not originate on a master page.
   */
  readonly overridden: boolean

  /**
   * An object that originated on a master page and has been overridden.
   */
  readonly overriddenMasterPageItem: PageItem | Guide | Graphic | Movie | Sound

  /**
   * The parent of the PageItem (a XMLElement, ComboBox, ListBox, TextBox, SignatureField, Spread, MasterSpread, Polygon, GraphicLine, Rectangle, Oval, SplineItem, Group, State, Cell, Character, Sound, PlaceGun, Movie or Snippet).
   */
  readonly parent: any

  /**
   * The page on which this page item appears.
   */
  readonly parentPage: Page

  /**
   * A collection of preferences objects.
   */
  readonly preferences: Preferences

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * The scaling applied to the arrowhead at the end of the path. (Range: 1 to 1000)
   */
  rightArrowHeadScale: number

  /**
   * The arrowhead applied to the end of the path.
   */
  rightLineEnd: ArrowHead

  /**
   * The rotatation angle of the PageItem. (Range: -360 to 360)
   */
  rotationAngle: number

  /**
   * The skewing angle applied to the PageItem. (Range: -360 to 360)
   */
  shearAngle: number

  /**
   * The stroke alignment applied to the PageItem.
   */
  strokeAlignment: StrokeAlignment

  /**
   * The swatch (color, gradient, tint, or mixed ink) applied to the stroke of the PageItem.
   */
  strokeColor: Swatch | string

  /**
   * The corner adjustment applied to the PageItem.
   */
  strokeCornerAdjustment: StrokeCornerAdjustment

  /**
   * The dash and gap measurements that define the pattern of a custom dashed line. Define up to six values (in points) in the format [dash1, gap1, dash2, gap2, dash3, gap3].
   */
  strokeDashAndGap: string[] | number[]

  /**
   * The percent of tint to use in object's stroke color. (To specify a tint percent, use a number in the range of 0 to 100; to use the inherited or overridden value, use -1.)
   */
  strokeTint: number

  /**
   * Transparency settings for the stroke.
   */
  readonly strokeTransparencySettings: StrokeTransparencySetting

  /**
   * The name of the stroke style to apply.
   */
  strokeType: StrokeStyle | string

  /**
   * The weight (in points) to apply to the PageItem's stroke.
   */
  strokeWeight: string | number

  /**
   * The text wrap preference properties that define the default formatting for wrapping text around objects.
   */
  readonly textWrapPreferences: TextWrapPreference

  /**
   * The object timing settings.
   */
  readonly timingSettings: TimingSetting

  /**
   * The shape to be applied to the top left corner of rectangular shapes and all corners of non-rectangular shapes.Note: corner option differs from end join in which you can set a radius for a corner option, whereas the rounded or beveled effect of an end join depends on the stroke weight.
   */
  topLeftCornerOption: CornerOptions

  /**
   * The radius in measurement units of the corner effect applied to the top left corner of rectangular shapes and all corners of non-rectangular shapes
   */
  topLeftCornerRadius: string | number

  /**
   * The shape to apply to the top right corner of rectangular shapes
   */
  topRightCornerOption: CornerOptions

  /**
   * The radius in measurement units of the corner effect applied to the top right corner of rectangular shapes
   */
  topRightCornerRadius: string | number

  /**
   * Transparency settings.
   */
  readonly transparencySettings: TransparencySetting

  /**
   * The top margin, height, and bottom margin constraints this item is subject to when using the object-based layout rule.
   */
  verticalLayoutConstraints: DimensionsConstraints[]

  /**
   * The vertical scaling applied to the PageItem.
   */
  verticalScale: number

  /**
   * If true, the PageItem is visible.
   */
  visible: boolean

  /**
   * The bounds of the PageItem including the stroke width, in the format [y1, x1, y2, x2], which give the coordinates of the top-left and bottom-right corners of the bounding box.
   */
  visibleBounds: string[] | number[]

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Applies the specified object style.
   * @param using The object style to apply.
   * @param clearingOverrides If true, clears the PageItem's existing attributes before applying the style.
   * @param clearingOverridesThroughRootObjectStyle If true, clears attributes and formatting applied to the PageItem that are not defined in the object style.
   */
  applyObjectStyle(
    using: ObjectStyle,
    clearingOverrides?: boolean,
    clearingOverridesThroughRootObjectStyle?: boolean,
  ): void

  /**
   * asynchronously exports the object(s) to a file.
   * @param format The export format, specified as an enumeration value or as an extension that appears in the Save as type or Format menu in the Export dialog.
   * @param to The path to the export file.
   * @param showingOptions If true, displays the export options dialog.
   * @param using The export style.
   * @param versionComments The comment for this version.
   * @param forceSave If true, forcibly saves a version.
   */
  asynchronousExportFile(
    format: ExportFormat | string,
    to: File,
    showingOptions?: boolean,
    using?: PDFExportPreset,
    versionComments?: string,
    forceSave?: boolean,
  ): BackgroundTask

  /**
   * Tag the object or the parent story using default tags defined in XML preference.
   */
  autoTag(): void

  /**
   * Finds objects that match the find what value and replace the objects with the change to value.
   * @param reverseOrder If true, returns the results in reverse order.
   */
  changeObject(reverseOrder: boolean): PageItem[]

  /**
   * Clear overrides for object style
   */
  clearObjectStyleOverrides(): void

  /**
   * Clears transformations from the PageItem. Transformations include rotation, scaling, flipping, fitting, and shearing.
   */
  clearTransformations(): void

  /**
   * Duplicate an object and place it into the target page item.
   * @param pageItems One or more page items to place or load
   * @param linkPageItems Whether to link pageItems in content placer (if true it will override link stories value)
   * @param linkStories Whether to link stories in content placer (only applicable for single story, pageItem links will also be created in case of more than one item)
   * @param mapStyles Whether to map styles in content placer
   * @param showingOptions Whether to display the link options dialog
   */
  contentPlace(
    pageItems: PageItem[],
    linkPageItems?: boolean,
    linkStories?: boolean,
    mapStyles?: boolean,
    showingOptions?: boolean,
  ): any[]

  /**
   * Converts the PageItem to a different shape.
   * @param given The PageItem's new shape.
   * @param numberOfSides The number of sides for the resulting polygon. (Range: 3 to 100)
   * @param insetPercentage The star inset percentage for the resulting polygon. (Range: 0.0 to 100.0)
   * @param cornerRadius The corner radius of the resulting rectangle.
   */
  convertShape(
    given: ConvertShapeOptions,
    numberOfSides: number,
    insetPercentage: number,
    cornerRadius: string | number,
  ): void

  /**
   * Create Email QR Code on the page item or document
   * @param emailAddress QR code Email Address
   * @param subject QR code Email Subject
   * @param body QR code Email Body Message
   * @param qrCodeSwatch Swatch to be applied on generated QR Code Graphic .
   * @param withProperties Initial values for properties of the new PageItem. Above parameters can also be passed as properties
   */
  createEmailQRCode(
    emailAddress: string,
    subject: string,
    body: string,
    qrCodeSwatch: Swatch | string,
    withProperties: string[],
  ): void

  /**
   * Create Hyperlink QR Code on the page item or document
   * @param urlLink QR code Hyperlink URL
   * @param qrCodeSwatch Swatch to be applied on generated QR Code Graphic .
   * @param withProperties Initial values for properties of the new PageItem. Above parameters can also be passed as properties
   */
  createHyperlinkQRCode(
    urlLink: string,
    qrCodeSwatch: Swatch | string,
    withProperties: string[],
  ): void

  /**
   * Create Plain Text QR Code on the page item
   * @param plainText QR code Plain Text
   * @param qrCodeSwatch Swatch to be applied on generated QR Code Graphic .
   * @param withProperties Initial values for properties of the new PageItem. Above parameters can also be passed as properties
   */
  createPlainTextQRCode(
    plainText: string,
    qrCodeSwatch: Swatch | string,
    withProperties: string[],
  ): void

  /**
   * Create Text Msg QR Code on the page item or document
   * @param cellNumber QR code Text Phone Number
   * @param textMessage QR code Text Message
   * @param qrCodeSwatch Swatch to be applied on generated QR Code Graphic .
   * @param withProperties Initial values for properties of the new PageItem. Above parameters can also be passed as properties
   */
  createTextMsgQRCode(
    cellNumber: string,
    textMessage: string,
    qrCodeSwatch: Swatch | string,
    withProperties: string[],
  ): void

  /**
   * Create Business Card QR Code on the page item or load on document's placegun
   * @param firstName QR code Business Card First Name
   * @param lastName QR code Business Card Last Name
   * @param jobTitle QR code Business Card Title
   * @param cellPhone QR code Business Card Cell Phone Number
   * @param phone QR code Business Card Phone Number
   * @param email QR code Business Card Email Address
   * @param organisation QR code Business Card Organisation
   * @param streetAddress QR code Business Card Street Address
   * @param city QR code Business Card City
   * @param adrState QR code Business Card State
   * @param country QR code Business Card Country
   * @param postalCode QR code Business Card Postal Code
   * @param website QR code Business Card URL
   * @param qrCodeSwatch Swatch to be applied on generated QR Code Graphic .
   * @param withProperties Initial values for properties of the new PageItem. Above parameters can also be passed as properties
   */
  createVCardQRCode(
    firstName: string,
    lastName: string,
    jobTitle: string,
    cellPhone: string,
    phone: string,
    email: string,
    organisation: string,
    streetAddress: string,
    city: string,
    adrState: string,
    country: string,
    postalCode: string,
    website: string,
    qrCodeSwatch: Swatch | string,
    withProperties: string[],
  ): void

  /**
   * Detaches an overridden master page item from the master page.
   */
  detach(): void

  /**
   * Duplicates the PageItem at the specified location or offset.
   * @param to The location of the new PageItem, specified in coordinates in the format [x, y].
   * @param by Amount by which to offset the new PageItem from the original PageItem's position.
   */
  duplicate(
    to: [number | string, number | string] | Spread | Page | Layer,
    by: string[] | number[],
  ): PageItem

  /**
   * Exports the object(s) to a file.
   * @param format The export format, specified as an enumeration value or as an extension that appears in the Save as type or Format menu in the Export dialog.
   * @param to The path to the export file.
   * @param showingOptions If true, displays the export options dialog.
   * @param using The export style.
   * @param versionComments The comment for this version.
   * @param forceSave If true, forcibly saves a version.
   */
  exportFile(
    format: ExportFormat | string,
    to: File,
    showingOptions?: boolean,
    using?: PDFExportPreset,
    versionComments?: string,
    forceSave?: boolean,
  ): void

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Finds objects that match the find what value.
   * @param reverseOrder If true, returns the results in reverse order.
   */
  findObject(reverseOrder: boolean): PageItem[]

  /**
   * Applies the specified fit option to content in a frame.
   * @param given The fit option to use.
   */
  fit(given: FitOptions): void

  /**
   * Flips the PageItem.
   * @param given The axis around which to flip the PageItem.
   * @param around The point around which to flip the PageItem.
   */
  flipItem(given: Flip, around: [number | string, number | string] | AnchorPoint): void

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): PageItem[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Associates the page item with the specified XML element while preserving existing content.
   * @param using The XML element.
   */
  markup(using: XMLElement): void

  /**
   * Moves the PageItem to a new location. Note: Either the 'to' or 'by' parameter is required; if both parameters are defined, only the to value is used.
   * @param to The new location of the PageItem,in the format (x, y).
   * @param by The amount (in measurement units) to move the PageItem relative to its current position, in the format (x, y).
   */
  move(
    to: [number | string, number | string] | Spread | Page | Layer,
    by: string[] | number[],
  ): void

  /**
   * Overrides a master page item and places the item on the document page as a new object.
   * @param destinationPage The document page that contains the master page item to override.
   */
  override(destinationPage: Page): any

  /**
   * Places the file.
   * @param fileName The file to place
   * @param showingOptions Whether to display the import options dialog
   * @param withProperties Initial values for properties of the placed object(s)
   */
  place(fileName: File, showingOptions?: boolean, withProperties?: object): any[]

  /**
   * Places XML content into the specified object. Note: Replaces any existing content.
   * @param using The XML element whose content you want to place.
   */
  placeXML(using: XMLElement): void

  /**
   * Apply an item's scaling to its content if possible.
   * @param to The scale factors to be left on the item.The default is {1.0, 1.0}.
   */
  redefineScaling(to: number[]): void

  /**
   * Move the bounding box of the page item
   * @param in_ The bounding box to resize. Can accept: CoordinateSpaces enumerator or Ordered array containing coordinateSpace:CoordinateSpaces enumerator, boundsKind:BoundingBoxLimits enumerator.
   * @param opposingCorners Opposing corners of new bounding box in the given coordinate space
   */
  reframe(in_: any, opposingCorners: any[]): void

  /**
   * Deletes the PageItem.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Removes the override from a previously overridden master page item.
   */
  removeOverride(): void

  /**
   * Resize the page item.
   * @param in_ The bounding box to resize. Can accept: CoordinateSpaces enumerator, BoundingBoxLimits enumerator or Ordered array containing coordinateSpace:CoordinateSpaces enumerator, boundsKind:BoundingBoxLimits enumerator.
   * @param from The transform origin. Legal specifications: relative to bounding box: anchor | {anchor | {x,y}, bounds kind [, coordinate space]}; relative to coordinate space: {x,y} | {{x,y}[, coordinate space]}; relative to layout window ruler: {{x,y}, page index | bounds kind}. Can accept: Array of 2 Reals, AnchorPoint enumerator or Array of Arrays of 2 Reals, CoordinateSpaces enumerators, AnchorPoint enumerators, BoundingBoxLimits enumerators or Long Integers.
   * @param by How the current dimensions are affected by the given values
   * @param values The width and height values. Legal dimensions specifications: {x, y [, coordinate space]}, {x, resize constraint [, coordinate space]}, or {resize constraint, y [, coordinate space]}; where x and y are real numbers and coordinate space is used to determine _only_ the unit of length for x and y; coordinate space is ignored for the 'current dimensions times' resize method).
   * @param resizeIndividually If false and multiple page items are targeted, the new dimensions are attained only by moving the individual items rather than resizing them.
   * @param consideringRulerUnits If true then a ruler location is interpreted using ruler units rather than points. The default value is false. This parameter has no effect unless the reference point is specified relative to a page.
   */
  resize(
    in_: any,
    from: any,
    by: ResizeMethods,
    values: number[] | ResizeConstraints | CoordinateSpaces,
    resizeIndividually?: boolean,
    consideringRulerUnits?: boolean,
  ): void

  /**
   * Get the coordinates of the given location in the specified coordinate system.
   * @param location The location requested. Can accept: Array of 2 Reals, AnchorPoint enumerator or Array of Arrays of 2 Reals, CoordinateSpaces enumerators, AnchorPoint enumerators, BoundingBoxLimits enumerators or Long Integers.
   * @param in_ The coordinate space to use.
   * @param consideringRulerUnits If true then a ruler location is interpreted using ruler units rather than points. The default value is false. This parameter has no effect unless the reference point is specified relative to a page.
   */
  resolve(location: any, in_: CoordinateSpaces, consideringRulerUnits?: boolean): any

  /**
   * Selects the object.
   * @param existingSelection The selection status of the PageItem in relation to previously selected objects.
   */
  select(existingSelection?: SelectionOptions): void

  /**
   * Stores the object in the specified library.
   * @param using The library in which to store the object.
   * @param withProperties Initial values for properties of the new PageItem
   */
  store(using: Library, withProperties: object): Asset

  /**
   * Generates a string which, if executed, will return the PageItem.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string

  /**
   * Transform the page item.
   * @param in_ The coordinate space to use
   * @param from The temporary origin during the transformation. Can accept: Array of 2 Reals, AnchorPoint enumerator or Array of Arrays of 2 Reals, CoordinateSpaces enumerators, AnchorPoint enumerators, BoundingBoxLimits enumerators or Long Integers.
   * @param withMatrix Transform matrix.
   * @param replacingCurrent Transform components to consider; providing this optional parameter causes the target's existing transform components to be replaced with new values.Without this parameter, the given matrix is concatenated onto the target's existing transform combining the effect of the two.
   * @param consideringRulerUnits If true then a ruler based origin is interpreted using ruler units rather than points. The default value is false. This parameter has no effect unless the reference point is specified relative to a page.
   */
  transform(
    in_: CoordinateSpaces,
    from: any,
    withMatrix: [number, number, number, number, number, number] | TransformationMatrix,
    replacingCurrent: MatrixContent | MatrixContent[] | number,
    consideringRulerUnits?: boolean,
  ): void

  /**
   * Transforms the PageItem using the last transformation performed on any object. Transformations include moving, rotating, shearing, scaling, and flipping.
   */
  transformAgain(): string[]

  /**
   * Transforms the PageItem using the last transformation performed on any PageItem. Transformations include moving, rotating, shearing, scaling, and flipping.
   */
  transformAgainIndividually(): string[]

  /**
   * Transforms the PageItem using the last sequence of transform operations performed on any single object or performed at the same time on any group of objects. Transformations include moving, rotating, shearing, scaling, and flipping.
   */
  transformSequenceAgain(): string[]

  /**
   * Transforms the PageItem using the last sequence of transformations performed on any single object or performed at the same time on any group of objects. Transformations include moving, rotating, shearing, scaling, and flipping.
   */
  transformSequenceAgainIndividually(): string[]

  /**
   * Get the transformation values of the page item.
   * @param in_ The coordinate space to use
   */
  transformValuesOf(in_: CoordinateSpaces): TransformationMatrix[]
}

/**
 * The page items collection, which can be used to process all page items in a container (such as a document, page, or group), regardless of type.
 */
declare class PageItems {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the PageItem with the specified index.
   * @param index The index.
   */
  [index: number]: PageItem

  /**
   * Returns any PageItem in the collection.
   */
  anyItem(): PageItem

  /**
   * Displays the number of elements in the PageItem.
   */
  count(): number

  /**
   * Returns every PageItem in the collection.
   */
  everyItem(): PageItem

  /**
   * Returns the first PageItem in the collection.
   */
  firstItem(): PageItem

  /**
   * Returns the PageItem with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): PageItem

  /**
   * Returns the PageItem with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): PageItem

  /**
   * Returns the PageItem with the specified name.
   * @param name The name.
   */
  itemByName(name: string): PageItem

  /**
   * Returns the PageItems within the specified range.
   * @param from The PageItem, index, or name at the beginning of the range.
   * @param to The PageItem, index, or name at the end of the range.
   */
  itemByRange(from: PageItem | number | string, to: PageItem | number | string): PageItem[]

  /**
   * Returns the last PageItem in the collection.
   */
  lastItem(): PageItem

  /**
   * Returns the middle PageItem in the collection.
   */
  middleItem(): PageItem

  /**
   * Returns the PageItem whose index follows the specified PageItem in the collection.
   * @param obj The PageItem whose index comes before the desired PageItem.
   */
  nextItem(obj: PageItem): PageItem

  /**
   * Returns the PageItem with the index previous to the specified index.
   * @param obj The index of the PageItem that follows the desired PageItem.
   */
  previousItem(obj: PageItem): PageItem

  /**
   * Generates a string which, if executed, will return the PageItem.
   */
  toSource(): string
}

/**
 * An ellipse.
 */
declare class Oval extends SplineItem {
  /**
   * The frame fitting option to apply to placed or pasted content. Can be applied to a frame, object style, or document or to the application.
   */
  readonly frameFittingOptions: FrameFittingOption

  /**
   * Export options for InCopy INCX document format.
   */
  readonly incopyExportOptions: InCopyExportOption

  /**
   * Title for this InCopy story.
   */
  storyTitle: string

  /**
   * Removes the frame fittings options and resets it to the initial state.
   */
  clearFrameFittingOptions(): void
}

/**
 * A collection of ellipses.
 */
declare class Ovals {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Oval with the specified index.
   * @param index The index.
   */
  [index: number]: Oval

  /**
   * Creates a new Oval
   * @param layer The layer on which to create the Oval.
   * @param at The location at which to insert the Oval relative to the reference object or within the container object.
   * @param reference The reference object. Note: Required when the at parameter is before or after.
   * @param withProperties Initial values for properties of the new Oval
   */
  add(
    layer: Layer,
    at?: LocationOptions,
    reference?: Document | Spread | MasterSpread | Page | Layer | PageItem,
    withProperties?: object,
  ): Oval

  /**
   * Returns any Oval in the collection.
   */
  anyItem(): Oval

  /**
   * Displays the number of elements in the Oval.
   */
  count(): number

  /**
   * Returns every Oval in the collection.
   */
  everyItem(): Oval

  /**
   * Returns the first Oval in the collection.
   */
  firstItem(): Oval

  /**
   * Returns the Oval with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Oval

  /**
   * Returns the Oval with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Oval

  /**
   * Returns the Oval with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Oval

  /**
   * Returns the Ovals within the specified range.
   * @param from The Oval, index, or name at the beginning of the range.
   * @param to The Oval, index, or name at the end of the range.
   */
  itemByRange(from: Oval | number | string, to: Oval | number | string): Oval[]

  /**
   * Returns the last Oval in the collection.
   */
  lastItem(): Oval

  /**
   * Returns the middle Oval in the collection.
   */
  middleItem(): Oval

  /**
   * Returns the Oval whose index follows the specified Oval in the collection.
   * @param obj The Oval whose index comes before the desired Oval.
   */
  nextItem(obj: Oval): Oval

  /**
   * Returns the Oval with the index previous to the specified index.
   * @param obj The index of the Oval that follows the desired Oval.
   */
  previousItem(obj: Oval): Oval

  /**
   * Generates a string which, if executed, will return the Oval.
   */
  toSource(): string
}

/**
 * A rectangle.
 */
declare class Rectangle extends SplineItem {
  /**
   * The frame fitting option to apply to placed or pasted content. Can be applied to a frame, object style, or document or to the application.
   */
  readonly frameFittingOptions: FrameFittingOption

  /**
   * Export options for InCopy INCX document format.
   */
  readonly incopyExportOptions: InCopyExportOption

  /**
   * Title for this InCopy story.
   */
  storyTitle: string

  /**
   * Removes the frame fittings options and resets it to the initial state.
   */
  clearFrameFittingOptions(): void
}

/**
 * A collection of rectangles.
 */
declare class Rectangles {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Rectangle with the specified index.
   * @param index The index.
   */
  [index: number]: Rectangle

  /**
   * Creates a new Rectangle
   * @param layer The layer on which to create the Rectangle.
   * @param at The location at which to insert the Rectangle relative to the reference object or within the container object.
   * @param reference The reference object. Note: Required when the at parameter is before or after.
   * @param withProperties Initial values for properties of the new Rectangle
   */
  add(
    layer: Layer,
    at?: LocationOptions,
    reference?: Document | Spread | MasterSpread | Page | Layer | PageItem,
    withProperties?: object,
  ): Rectangle

  /**
   * Returns any Rectangle in the collection.
   */
  anyItem(): Rectangle

  /**
   * Displays the number of elements in the Rectangle.
   */
  count(): number

  /**
   * Returns every Rectangle in the collection.
   */
  everyItem(): Rectangle

  /**
   * Returns the first Rectangle in the collection.
   */
  firstItem(): Rectangle

  /**
   * Returns the Rectangle with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Rectangle

  /**
   * Returns the Rectangle with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Rectangle

  /**
   * Returns the Rectangle with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Rectangle

  /**
   * Returns the Rectangles within the specified range.
   * @param from The Rectangle, index, or name at the beginning of the range.
   * @param to The Rectangle, index, or name at the end of the range.
   */
  itemByRange(from: Rectangle | number | string, to: Rectangle | number | string): Rectangle[]

  /**
   * Returns the last Rectangle in the collection.
   */
  lastItem(): Rectangle

  /**
   * Returns the middle Rectangle in the collection.
   */
  middleItem(): Rectangle

  /**
   * Returns the Rectangle whose index follows the specified Rectangle in the collection.
   * @param obj The Rectangle whose index comes before the desired Rectangle.
   */
  nextItem(obj: Rectangle): Rectangle

  /**
   * Returns the Rectangle with the index previous to the specified index.
   * @param obj The index of the Rectangle that follows the desired Rectangle.
   */
  previousItem(obj: Rectangle): Rectangle

  /**
   * Generates a string which, if executed, will return the Rectangle.
   */
  toSource(): string
}

/**
 * A straight line consisting of two points.
 */
declare class GraphicLine extends SplineItem {}

/**
 * A collection of graphic lines.
 */
declare class GraphicLines {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the GraphicLine with the specified index.
   * @param index The index.
   */
  [index: number]: GraphicLine

  /**
   * Creates a new GraphicLine
   * @param layer The layer on which to create the GraphicLine.
   * @param at The location at which to insert the GraphicLine relative to the reference object or within the container object.
   * @param reference The reference object. Note: Required when the at parameter is before or after.
   * @param withProperties Initial values for properties of the new GraphicLine
   */
  add(
    layer: Layer,
    at?: LocationOptions,
    reference?: Document | Spread | MasterSpread | Page | Layer | PageItem,
    withProperties?: object,
  ): GraphicLine

  /**
   * Returns any GraphicLine in the collection.
   */
  anyItem(): GraphicLine

  /**
   * Displays the number of elements in the GraphicLine.
   */
  count(): number

  /**
   * Returns every GraphicLine in the collection.
   */
  everyItem(): GraphicLine

  /**
   * Returns the first GraphicLine in the collection.
   */
  firstItem(): GraphicLine

  /**
   * Returns the GraphicLine with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): GraphicLine

  /**
   * Returns the GraphicLine with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): GraphicLine

  /**
   * Returns the GraphicLine with the specified name.
   * @param name The name.
   */
  itemByName(name: string): GraphicLine

  /**
   * Returns the GraphicLines within the specified range.
   * @param from The GraphicLine, index, or name at the beginning of the range.
   * @param to The GraphicLine, index, or name at the end of the range.
   */
  itemByRange(from: GraphicLine | number | string, to: GraphicLine | number | string): GraphicLine[]

  /**
   * Returns the last GraphicLine in the collection.
   */
  lastItem(): GraphicLine

  /**
   * Returns the middle GraphicLine in the collection.
   */
  middleItem(): GraphicLine

  /**
   * Returns the GraphicLine whose index follows the specified GraphicLine in the collection.
   * @param obj The GraphicLine whose index comes before the desired GraphicLine.
   */
  nextItem(obj: GraphicLine): GraphicLine

  /**
   * Returns the GraphicLine with the index previous to the specified index.
   * @param obj The index of the GraphicLine that follows the desired GraphicLine.
   */
  previousItem(obj: GraphicLine): GraphicLine

  /**
   * Generates a string which, if executed, will return the GraphicLine.
   */
  toSource(): string
}

/**
 * A polygon. Any shape that is not a rectangle, ellipse, or graphic line. When you add a polygon, InDesign creates a regular polygon based on the current polygon preferences settings.
 */
declare class Polygon extends SplineItem {
  /**
   * The frame fitting option to apply to placed or pasted content. Can be applied to a frame, object style, or document or to the application.
   */
  readonly frameFittingOptions: FrameFittingOption

  /**
   * Export options for InCopy INCX document format.
   */
  readonly incopyExportOptions: InCopyExportOption

  /**
   * Title for this InCopy story.
   */
  storyTitle: string

  /**
   * Removes the frame fittings options and resets it to the initial state.
   */
  clearFrameFittingOptions(): void
}

/**
 * A collection of polygons.
 */
declare class Polygons {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Polygon with the specified index.
   * @param index The index.
   */
  [index: number]: Polygon

  /**
   * Creates a new Polygon.
   * @param layer The layer on which to create the Polygon.
   * @param numberOfSides The number of sides to give the Polygon. (Range: 3 to 100)
   * @param insetPercentage The star inset percentage applied to the Polygon.
   * @param at The location at which to insert the Polygon relative to the reference object or within the container object.
   * @param reference The reference object. Note: Required when the at parameter is before or after.
   * @param withProperties Initial values for properties of the new Polygon
   */
  add(
    layer: Layer,
    numberOfSides: number,
    insetPercentage: number,
    at?: LocationOptions,
    reference?: Document | Spread | MasterSpread | Page | Layer | PageItem,
    withProperties?: object,
  ): Polygon

  /**
   * Returns any Polygon in the collection.
   */
  anyItem(): Polygon

  /**
   * Displays the number of elements in the Polygon.
   */
  count(): number

  /**
   * Returns every Polygon in the collection.
   */
  everyItem(): Polygon

  /**
   * Returns the first Polygon in the collection.
   */
  firstItem(): Polygon

  /**
   * Returns the Polygon with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Polygon

  /**
   * Returns the Polygon with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Polygon

  /**
   * Returns the Polygon with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Polygon

  /**
   * Returns the Polygons within the specified range.
   * @param from The Polygon, index, or name at the beginning of the range.
   * @param to The Polygon, index, or name at the end of the range.
   */
  itemByRange(from: Polygon | number | string, to: Polygon | number | string): Polygon[]

  /**
   * Returns the last Polygon in the collection.
   */
  lastItem(): Polygon

  /**
   * Returns the middle Polygon in the collection.
   */
  middleItem(): Polygon

  /**
   * Returns the Polygon whose index follows the specified Polygon in the collection.
   * @param obj The Polygon whose index comes before the desired Polygon.
   */
  nextItem(obj: Polygon): Polygon

  /**
   * Returns the Polygon with the index previous to the specified index.
   * @param obj The index of the Polygon that follows the desired Polygon.
   */
  previousItem(obj: Polygon): Polygon

  /**
   * Generates a string which, if executed, will return the Polygon.
   */
  toSource(): string
}

/**
 * A group.
 */
declare class Group extends PageItem {
  /**
   * Anchored object settings.
   */
  readonly anchoredObjectSettings: AnchoredObjectSetting

  /**
   * A collection of group items that are also part of an article.
   */
  readonly articleChildren: ArticleChildren

  /**
   * A collection of buttons.
   */
  readonly buttons: Buttons

  /**
   * A collection of checkboxes.
   */
  readonly checkBoxes: CheckBoxes

  /**
   * A collection of comboboxes.
   */
  readonly comboBoxes: ComboBoxes

  /**
   * A collection of endnote text frames.
   */
  readonly endnoteTextFrames: EndnoteTextFrames

  /**
   * EPSTexts
   */
  readonly epstexts: EPSTexts

  /**
   * A collection of form fields.
   */
  readonly formFields: FormFields

  /**
   * A collection of graphic lines.
   */
  readonly graphicLines: GraphicLines

  /**
   * A collection of groups.
   */
  readonly groups: Groups

  /**
   * A collection of listboxes.
   */
  readonly listBoxes: ListBoxes

  /**
   * A collection of multi-state objects.
   */
  readonly multiStateObjects: MultiStateObjects

  /**
   * Export options for the object
   */
  readonly objectExportOptions: ObjectExportOption

  /**
   * A collection of ellipses.
   */
  readonly ovals: Ovals

  /**
   * The page items collection, which can be used to process all page items in a container (such as a document, page, or group), regardless of type.
   */
  readonly pageItems: PageItems

  /**
   * A collection of polygons.
   */
  readonly polygons: Polygons

  /**
   * A collection of radio buttons.
   */
  readonly radioButtons: RadioButtons

  /**
   * A collection of rectangles.
   */
  readonly rectangles: Rectangles

  /**
   * A collection of signature fields.
   */
  readonly signatureFields: SignatureFields

  /**
   * The spline items collection.
   */
  readonly splineItems: SplineItems

  /**
   * A collection of text boxes.
   */
  readonly textBoxes: TextBoxes

  /**
   * A collection of text frames.
   */
  readonly textFrames: TextFrames

  /**
   * Brings the Group forward one level in its layer.
   */
  bringForward(): void

  /**
   * Brings the Group to the front of its layer or in front of a particular item.
   * @param reference The reference object to bring the object in front of (must have same parent)
   */
  bringToFront(reference: PageItem): void

  /**
   * Sends the Group back one level in its layer.
   */
  sendBackward(): void

  /**
   * Sends the Group to the back of its layer or behind a particular item (must have same parent).
   * @param reference The reference object to send the object behind
   */
  sendToBack(reference: PageItem): void

  /**
   * Ungroups the group.
   */
  ungroup(): void
}

/**
 * A collection of groups.
 */
declare class Groups {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Group with the specified index.
   * @param index The index.
   */
  [index: number]: Group

  /**
   * Creates a new Group.
   * @param groupItems The objects to group.
   * @param layer The layer on which to create the Group.
   * @param at The location at which to insert the Group relative to the reference object or within the container object.
   * @param reference The reference object. Note: Required when the at parameter is before or after.
   * @param withProperties Initial values for properties of the new Group
   */
  add(
    groupItems: PageItem[],
    layer: Layer,
    at?: LocationOptions,
    reference?: Document | Spread | MasterSpread | Page | Layer | PageItem,
    withProperties?: object,
  ): Group

  /**
   * Returns any Group in the collection.
   */
  anyItem(): Group

  /**
   * Displays the number of elements in the Group.
   */
  count(): number

  /**
   * Returns every Group in the collection.
   */
  everyItem(): Group

  /**
   * Returns the first Group in the collection.
   */
  firstItem(): Group

  /**
   * Returns the Group with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Group

  /**
   * Returns the Group with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Group

  /**
   * Returns the Group with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Group

  /**
   * Returns the Groups within the specified range.
   * @param from The Group, index, or name at the beginning of the range.
   * @param to The Group, index, or name at the end of the range.
   */
  itemByRange(from: Group | number | string, to: Group | number | string): Group[]

  /**
   * Returns the last Group in the collection.
   */
  lastItem(): Group

  /**
   * Returns the middle Group in the collection.
   */
  middleItem(): Group

  /**
   * Returns the Group whose index follows the specified Group in the collection.
   * @param obj The Group whose index comes before the desired Group.
   */
  nextItem(obj: Group): Group

  /**
   * Returns the Group with the index previous to the specified index.
   * @param obj The index of the Group that follows the desired Group.
   */
  previousItem(obj: Group): Group

  /**
   * Generates a string which, if executed, will return the Group.
   */
  toSource(): string
}

/**
 * An imported bitmap image in any bitmap file format (including TIFF, JPEG, or GIF).
 */
declare class Image extends Graphic {
  /**
   * The native resolution of a placed graphic.
   */
  readonly actualPpi: number[]

  /**
   * Clipping path settings.
   */
  readonly clippingPath: ClippingPathSettings

  /**
   * The resolution of a graphic after it has been resized.
   */
  readonly effectivePpi: number[]

  /**
   * Graphic layer option settings.
   */
  readonly graphicLayerOptions: GraphicLayerOption

  /**
   * The image I/O preference properties that define preferences for importing images.
   */
  readonly imageIOPreferences: ImageIOPreference

  /**
   * The rendering intent override applied to the image.
   */
  imageRenderingIntent: RenderingIntent

  /**
   * The color profile.
   */
  profile: Profile | string

  /**
   * A list of valid RGB profiles.
   */
  readonly profileList: string[]

  /**
   * The color space.
   */
  readonly space: string
}

/**
 * A collection of bitmap images in any bitmap file format (including TIFF, JPEG, or GIF).
 */
declare class Images {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Image with the specified index.
   * @param index The index.
   */
  [index: number]: Image

  /**
   * Returns any Image in the collection.
   */
  anyItem(): Image

  /**
   * Displays the number of elements in the Image.
   */
  count(): number

  /**
   * Returns every Image in the collection.
   */
  everyItem(): Image

  /**
   * Returns the first Image in the collection.
   */
  firstItem(): Image

  /**
   * Returns the Image with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Image

  /**
   * Returns the Image with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Image

  /**
   * Returns the Image with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Image

  /**
   * Returns the Images within the specified range.
   * @param from The Image, index, or name at the beginning of the range.
   * @param to The Image, index, or name at the end of the range.
   */
  itemByRange(from: Image | number | string, to: Image | number | string): Image[]

  /**
   * Returns the last Image in the collection.
   */
  lastItem(): Image

  /**
   * Returns the middle Image in the collection.
   */
  middleItem(): Image

  /**
   * Returns the Image whose index follows the specified Image in the collection.
   * @param obj The Image whose index comes before the desired Image.
   */
  nextItem(obj: Image): Image

  /**
   * Returns the Image with the index previous to the specified index.
   * @param obj The index of the Image that follows the desired Image.
   */
  previousItem(obj: Image): Image

  /**
   * Generates a string which, if executed, will return the Image.
   */
  toSource(): string
}

/**
 * A placed EPS file.
 */
declare class EPS extends Graphic {
  /**
   * The native resolution of a placed graphic.
   */
  readonly actualPpi: number[]

  /**
   * Clipping path settings.
   */
  readonly clippingPath: ClippingPathSettings

  /**
   * The color profile policy for CMYK content in a placed vector.
   */
  cmykVectorPolicy: PlacedVectorProfilePolicy

  /**
   * The resolution of a graphic after it has been resized.
   */
  readonly effectivePpi: number[]

  /**
   * The color profile policy for grayscale content in a placed vector.
   */
  grayVectorPolicy: PlacedVectorProfilePolicy

  /**
   * The color profile policy for RGB content in a placed vector.
   */
  rgbVectorPolicy: PlacedVectorProfilePolicy

  /**
   * The color space.
   */
  readonly space: string
}

/**
 * A collection of EPS files.
 */
declare class EPSs {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the EPS with the specified index.
   * @param index The index.
   */
  [index: number]: EPS

  /**
   * Returns any EPS in the collection.
   */
  anyItem(): EPS

  /**
   * Displays the number of elements in the EPS.
   */
  count(): number

  /**
   * Returns every EPS in the collection.
   */
  everyItem(): EPS

  /**
   * Returns the first EPS in the collection.
   */
  firstItem(): EPS

  /**
   * Returns the EPS with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): EPS

  /**
   * Returns the EPS with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): EPS

  /**
   * Returns the EPS with the specified name.
   * @param name The name.
   */
  itemByName(name: string): EPS

  /**
   * Returns the EPSs within the specified range.
   * @param from The EPS, index, or name at the beginning of the range.
   * @param to The EPS, index, or name at the end of the range.
   */
  itemByRange(from: EPS | number | string, to: EPS | number | string): EPS[]

  /**
   * Returns the last EPS in the collection.
   */
  lastItem(): EPS

  /**
   * Returns the middle EPS in the collection.
   */
  middleItem(): EPS

  /**
   * Returns the EPS whose index follows the specified EPS in the collection.
   * @param obj The EPS whose index comes before the desired EPS.
   */
  nextItem(obj: EPS): EPS

  /**
   * Returns the EPS with the index previous to the specified index.
   * @param obj The index of the EPS that follows the desired EPS.
   */
  previousItem(obj: EPS): EPS

  /**
   * Generates a string which, if executed, will return the EPS.
   */
  toSource(): string
}

/**
 * A placed PDF file.
 */
declare class PDF extends Graphic {
  /**
   * Clipping path settings.
   */
  readonly clippingPath: ClippingPathSettings

  /**
   * The color profile policy for CMYK content in a placed vector.
   */
  cmykVectorPolicy: PlacedVectorProfilePolicy

  /**
   * Graphic layer option settings.
   */
  readonly graphicLayerOptions: GraphicLayerOption

  /**
   * The color profile policy for grayscale content in a placed vector.
   */
  grayVectorPolicy: PlacedVectorProfilePolicy

  /**
   * PDF attribute settings.
   */
  readonly pdfAttributes: PDFAttribute

  /**
   * The color profile policy for RGB content in a placed vector.
   */
  rgbVectorPolicy: PlacedVectorProfilePolicy
}

/**
 * A collection of PDF files.
 */
declare class PDFs {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the PDF with the specified index.
   * @param index The index.
   */
  [index: number]: PDF

  /**
   * Returns any PDF in the collection.
   */
  anyItem(): PDF

  /**
   * Displays the number of elements in the PDF.
   */
  count(): number

  /**
   * Returns every PDF in the collection.
   */
  everyItem(): PDF

  /**
   * Returns the first PDF in the collection.
   */
  firstItem(): PDF

  /**
   * Returns the PDF with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): PDF

  /**
   * Returns the PDF with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): PDF

  /**
   * Returns the PDF with the specified name.
   * @param name The name.
   */
  itemByName(name: string): PDF

  /**
   * Returns the PDFs within the specified range.
   * @param from The PDF, index, or name at the beginning of the range.
   * @param to The PDF, index, or name at the end of the range.
   */
  itemByRange(from: PDF | number | string, to: PDF | number | string): PDF[]

  /**
   * Returns the last PDF in the collection.
   */
  lastItem(): PDF

  /**
   * Returns the middle PDF in the collection.
   */
  middleItem(): PDF

  /**
   * Returns the PDF whose index follows the specified PDF in the collection.
   * @param obj The PDF whose index comes before the desired PDF.
   */
  nextItem(obj: PDF): PDF

  /**
   * Returns the PDF with the index previous to the specified index.
   * @param obj The index of the PDF that follows the desired PDF.
   */
  previousItem(obj: PDF): PDF

  /**
   * Generates a string which, if executed, will return the PDF.
   */
  toSource(): string
}

/**
 * A placed WMF graphic.
 */
declare class WMF extends Graphic {
  /**
   * Clipping path settings.
   */
  readonly clippingPath: ClippingPathSettings
}

/**
 * A collection of WMF graphics.
 */
declare class WMFs {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the WMF with the specified index.
   * @param index The index.
   */
  [index: number]: WMF

  /**
   * Returns any WMF in the collection.
   */
  anyItem(): WMF

  /**
   * Displays the number of elements in the WMF.
   */
  count(): number

  /**
   * Returns every WMF in the collection.
   */
  everyItem(): WMF

  /**
   * Returns the first WMF in the collection.
   */
  firstItem(): WMF

  /**
   * Returns the WMF with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): WMF

  /**
   * Returns the WMF with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): WMF

  /**
   * Returns the WMF with the specified name.
   * @param name The name.
   */
  itemByName(name: string): WMF

  /**
   * Returns the WMFs within the specified range.
   * @param from The WMF, index, or name at the beginning of the range.
   * @param to The WMF, index, or name at the end of the range.
   */
  itemByRange(from: WMF | number | string, to: WMF | number | string): WMF[]

  /**
   * Returns the last WMF in the collection.
   */
  lastItem(): WMF

  /**
   * Returns the middle WMF in the collection.
   */
  middleItem(): WMF

  /**
   * Returns the WMF whose index follows the specified WMF in the collection.
   * @param obj The WMF whose index comes before the desired WMF.
   */
  nextItem(obj: WMF): WMF

  /**
   * Returns the WMF with the index previous to the specified index.
   * @param obj The index of the WMF that follows the desired WMF.
   */
  previousItem(obj: WMF): WMF

  /**
   * Generates a string which, if executed, will return the WMF.
   */
  toSource(): string
}

/**
 * A placed PICT graphic.
 */
declare class PICT extends Graphic {
  /**
   * Clipping path settings.
   */
  readonly clippingPath: ClippingPathSettings
}

/**
 * A collection of PICT graphics.
 */
declare class PICTs {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the PICT with the specified index.
   * @param index The index.
   */
  [index: number]: PICT

  /**
   * Returns any PICT in the collection.
   */
  anyItem(): PICT

  /**
   * Displays the number of elements in the PICT.
   */
  count(): number

  /**
   * Returns every PICT in the collection.
   */
  everyItem(): PICT

  /**
   * Returns the first PICT in the collection.
   */
  firstItem(): PICT

  /**
   * Returns the PICT with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): PICT

  /**
   * Returns the PICT with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): PICT

  /**
   * Returns the PICT with the specified name.
   * @param name The name.
   */
  itemByName(name: string): PICT

  /**
   * Returns the PICTs within the specified range.
   * @param from The PICT, index, or name at the beginning of the range.
   * @param to The PICT, index, or name at the end of the range.
   */
  itemByRange(from: PICT | number | string, to: PICT | number | string): PICT[]

  /**
   * Returns the last PICT in the collection.
   */
  lastItem(): PICT

  /**
   * Returns the middle PICT in the collection.
   */
  middleItem(): PICT

  /**
   * Returns the PICT whose index follows the specified PICT in the collection.
   * @param obj The PICT whose index comes before the desired PICT.
   */
  nextItem(obj: PICT): PICT

  /**
   * Returns the PICT with the index previous to the specified index.
   * @param obj The index of the PICT that follows the desired PICT.
   */
  previousItem(obj: PICT): PICT

  /**
   * Generates a string which, if executed, will return the PICT.
   */
  toSource(): string
}

/**
 * An imported graphic in any graphic file format (including vector, metafile, and bitmap formats).
 */
declare class Graphic extends PageItem {
  /**
   * The type of the image.
   */
  readonly imageTypeName: string

  /**
   * The source file of the link.
   */
  readonly itemLink: Link

  /**
   * Exports the Graphic for the web.
   * @param to The full path name of the exported file.
   */
  exportForWeb(to: File): string[]
}

/**
 * A collection of imported graphics in any graphic file format (vector, metafile, or bitmap).
 */
declare class Graphics {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Graphic with the specified index.
   * @param index The index.
   */
  [index: number]: Graphic

  /**
   * Returns any Graphic in the collection.
   */
  anyItem(): Graphic

  /**
   * Displays the number of elements in the Graphic.
   */
  count(): number

  /**
   * Returns every Graphic in the collection.
   */
  everyItem(): Graphic

  /**
   * Returns the first Graphic in the collection.
   */
  firstItem(): Graphic

  /**
   * Returns the Graphic with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Graphic

  /**
   * Returns the Graphic with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Graphic

  /**
   * Returns the Graphic with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Graphic

  /**
   * Returns the Graphics within the specified range.
   * @param from The Graphic, index, or name at the beginning of the range.
   * @param to The Graphic, index, or name at the end of the range.
   */
  itemByRange(from: Graphic | number | string, to: Graphic | number | string): Graphic[]

  /**
   * Returns the last Graphic in the collection.
   */
  lastItem(): Graphic

  /**
   * Returns the middle Graphic in the collection.
   */
  middleItem(): Graphic

  /**
   * Returns the Graphic whose index follows the specified Graphic in the collection.
   * @param obj The Graphic whose index comes before the desired Graphic.
   */
  nextItem(obj: Graphic): Graphic

  /**
   * Returns the Graphic with the index previous to the specified index.
   * @param obj The index of the Graphic that follows the desired Graphic.
   */
  previousItem(obj: Graphic): Graphic

  /**
   * Generates a string which, if executed, will return the Graphic.
   */
  toSource(): string
}

/**
 * A transformation matrix.
 */
declare class TransformationMatrix {
  /**
   * The shear angle of the transformation matrix.
   */
  readonly clockwiseShearAngle: number

  /**
   * The rotation angle of the transformation matrix.
   */
  readonly counterclockwiseRotationAngle: number

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The horizontal scale factor of the transformation matrix.
   */
  readonly horizontalScaleFactor: number

  /**
   * The horizontal translation of the transformation matrix.
   */
  readonly horizontalTranslation: number

  /**
   * The index of the TransformationMatrix within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * The mapping the transformation matrix performs on the unit triangle. Can return: Array of Array of 2 Arrays of 2 Reals.
   */
  readonly matrixMapping: any[]

  /**
   * The values of the transformation matrix.
   */
  readonly matrixValues: number[]

  /**
   * The name of the TransformationMatrix.
   */
  readonly name: string

  /**
   * The parent of the TransformationMatrix (a Application).
   */
  readonly parent: Application

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * The vertical scale factor of the transformation matrix.
   */
  readonly verticalScaleFactor: number

  /**
   * The vertical translation of the transformation matrix.
   */
  readonly verticalTranslation: number

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Multiply the transformation matrix by another.
   * @param withMatrix The right hand matrix factor
   */
  catenateMatrix(withMatrix: TransformationMatrix): TransformationMatrix

  /**
   * Multiply the point by the matrix.
   * @param point The point to transform
   */
  changeCoordinates(point: number[]): number[]

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): TransformationMatrix[]

  /**
   * Invert the transformation matrix.
   */
  invertMatrix(): TransformationMatrix

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Rotate the transformation matrix.
   * @param byAngle The counterclockwise rotation angle
   * @param byCosine The cosine of the desired rotation
   * @param bySine The sine of the desired rotation
   */
  rotateMatrix(byAngle: number, byCosine: number, bySine: number): TransformationMatrix

  /**
   * Scale the transformation matrix.
   * @param horizontallyBy The horizontal scale factor
   * @param verticallyBy The vertical scale factor
   */
  scaleMatrix(horizontallyBy: number, verticallyBy: number): TransformationMatrix

  /**
   * Shear the transformation matrix.
   * @param byAngle The horizontal shear angle
   * @param bySlope The horizontal shear slope
   */
  shearMatrix(byAngle: number, bySlope: number): TransformationMatrix

  /**
   * Generates a string which, if executed, will return the TransformationMatrix.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string

  /**
   * Translate the transformation matrix.
   * @param horizontallyBy The horizontal translation distance
   * @param verticallyBy The vertical translation distance
   */
  translateMatrix(horizontallyBy: number, verticallyBy: number): TransformationMatrix
}

/**
 * A collection of transformation matrices.
 */
declare class TransformationMatrices {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the TransformationMatrix with the specified index.
   * @param index The index.
   */
  [index: number]: TransformationMatrix

  /**
   * Create a new TransformationMatrix.
   * @param horizontalScaleFactor The horizontal scale factor of the transformation matrix
   * @param verticalScaleFactor The vertical scale factor of the transformation matrix
   * @param clockwiseShearAngle The shear angle of the transformation matrix
   * @param counterclockwiseRotationAngle The rotation angle of the transformation matrix
   * @param horizontalTranslation The horizontal translation of the transformation matrix
   * @param verticalTranslation The vertical translation of the transformation matrix
   * @param matrixValues The values of the transformation matrix
   * @param matrixMapping The mapping the transformation matrix performs on the unit triangle. Can accept: Array of Array of 2 Arrays of 2 Reals.
   * @param withProperties Initial values for properties of the new TransformationMatrix
   */
  add(
    horizontalScaleFactor: number,
    verticalScaleFactor: number,
    clockwiseShearAngle: number,
    counterclockwiseRotationAngle: number,
    horizontalTranslation: number,
    verticalTranslation: number,
    matrixValues: number[],
    matrixMapping: any[],
    withProperties: object,
  ): TransformationMatrix

  /**
   * Returns any TransformationMatrix in the collection.
   */
  anyItem(): TransformationMatrix

  /**
   * Displays the number of elements in the TransformationMatrix.
   */
  count(): number

  /**
   * Returns every TransformationMatrix in the collection.
   */
  everyItem(): TransformationMatrix

  /**
   * Returns the first TransformationMatrix in the collection.
   */
  firstItem(): TransformationMatrix

  /**
   * Returns the TransformationMatrix with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): TransformationMatrix

  /**
   * Returns the TransformationMatrix with the specified name.
   * @param name The name.
   */
  itemByName(name: string): TransformationMatrix

  /**
   * Returns the TransformationMatrices within the specified range.
   * @param from The TransformationMatrix, index, or name at the beginning of the range.
   * @param to The TransformationMatrix, index, or name at the end of the range.
   */
  itemByRange(
    from: TransformationMatrix | number | string,
    to: TransformationMatrix | number | string,
  ): TransformationMatrix[]

  /**
   * Returns the last TransformationMatrix in the collection.
   */
  lastItem(): TransformationMatrix

  /**
   * Returns the middle TransformationMatrix in the collection.
   */
  middleItem(): TransformationMatrix

  /**
   * Returns the TransformationMatrix whose index follows the specified TransformationMatrix in the collection.
   * @param obj The TransformationMatrix whose index comes before the desired TransformationMatrix.
   */
  nextItem(obj: TransformationMatrix): TransformationMatrix

  /**
   * Returns the TransformationMatrix with the index previous to the specified index.
   * @param obj The index of the TransformationMatrix that follows the desired TransformationMatrix.
   */
  previousItem(obj: TransformationMatrix): TransformationMatrix

  /**
   * Generates a string which, if executed, will return the TransformationMatrix.
   */
  toSource(): string
}

/**
 * Options for fitting placed or pasted content in a frame.
 */
declare class FrameFittingOption extends Preference {
  /**
   * If true, the last saved fitting options will be applied to the contents of a frame when it is resized.
   */
  autoFit: boolean

  /**
   * The amount in measurement units to crop the bottom edge of a graphic.
   */
  bottomCrop: string | number

  /**
   * The point with which to align the image empty when fitting in a frame. For information, see frame fitting options.
   */
  fittingAlignment: AnchorPoint

  /**
   * The frame fitting option to apply to placed or pasted content if the frame is empty. Can be applied to a frame, object style, or document or to the application.
   */
  fittingOnEmptyFrame: EmptyFrameFittingOptions

  /**
   * The amount in measurement units to crop the left edge of a graphic.
   */
  leftCrop: string | number

  /**
   * The amount in measurement units to crop the right edge of a graphic.
   */
  rightCrop: string | number

  /**
   * The amount in measurement units to crop the top edge of a graphic.
   */
  topCrop: string | number
}

/**
 * A guide.
 */
declare class Guide {
  /**
   * Dispatched after a Guide is placed. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_PLACE: string

  /**
   * If true, the master page item can be overridden.
   */
  allowOverrides: boolean

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * If true, horizontal orientation guides stop at the edges of the specified page. If false, the guides extends across the width of the spread and into the pasteboard area.
   */
  fitToPage: boolean

  /**
   * The color of the guide, specified either as an array of three doubles, each in the range 0 to 255 and representing R, G, and B values, or as a UI color.
   */
  guideColor: [number, number, number] | UIColors

  /**
   * The type of the guide.
   */
  guideType: GuideTypeOptions

  /**
   * The zone of the guide.
   */
  guideZone: string | number

  /**
   * The unique ID of the Guide.
   */
  readonly id: number

  /**
   * The index of the Guide within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * The layer that the Guide is on.
   */
  itemLayer: Layer

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The location at which to place the guide relative to the current ruler zero point.
   */
  location: string | number

  /**
   * If true, the Guide is locked.
   */
  locked: boolean

  /**
   * The name of the Guide; this is an alias to the Guide's label property.
   */
  name: string

  /**
   * The orientation of the guide.
   */
  orientation: HorizontalOrVertical

  /**
   * If true, the object originated on a master spread and was overridden. If false, the object either originated on a master spread and was not overridden, or the object did not originate on a master page.
   */
  readonly overridden: boolean

  /**
   * An object that originated on a master page and has been overridden.
   */
  readonly overriddenMasterPageItem: PageItem | Guide | Graphic | Movie | Sound

  /**
   * The parent of the Guide (a Spread or MasterSpread).
   */
  readonly parent: any

  /**
   * The page on which this page item appears.
   */
  readonly parentPage: Page

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * The view magnification as a percentage below which guides are no longer displayed. (Range: 5.0 to 4000.0)
   */
  viewThreshold: number

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Detaches an overridden master page item from the master page.
   */
  detach(): void

  /**
   * Duplicates the Guide.
   */
  duplicate(): Guide

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): Guide[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Moves the guide to a new location. Note: Either the to or the by parameter is required; if both parameters are defined, only the to value is used.
   * @param to The new location of the guide, in the format [x, y].
   * @param by The amount to move the guide relative to its current position, in the format [x, y].
   */
  move(to: string[] | number[], by: string[] | number[]): void

  /**
   * Overrides a master page item and places the item on the document page as a new object.
   * @param destinationPage The document page that contains the master page item to override.
   */
  override(destinationPage: Page): any

  /**
   * Deletes the Guide.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Removes the override from a previously overridden master page item.
   */
  removeOverride(): void

  /**
   * Get the coordinates of the given location in the specified coordinate system.
   * @param location The location requested. Can accept: Array of 2 Reals, AnchorPoint enumerator or Array of Arrays of 2 Reals, CoordinateSpaces enumerators, AnchorPoint enumerators, BoundingBoxLimits enumerators or Long Integers.
   * @param in_ The coordinate space to use.
   * @param consideringRulerUnits If true then a ruler location is interpreted using ruler units rather than points. The default value is false. This parameter has no effect unless the reference point is specified relative to a page.
   */
  resolve(location: any, in_: CoordinateSpaces, consideringRulerUnits?: boolean): any

  /**
   * Selects the object.
   * @param existingSelection The selection status of the Guide in relation to previously selected objects.
   */
  select(existingSelection?: SelectionOptions): void

  /**
   * Generates a string which, if executed, will return the Guide.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string

  /**
   * Get the transformation values of the page item.
   * @param in_ The coordinate space to use
   */
  transformValuesOf(in_: CoordinateSpaces): TransformationMatrix[]
}

/**
 * A collection of guides.
 */
declare class Guides {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Guide with the specified index.
   * @param index The index.
   */
  [index: number]: Guide

  /**
   * Creates a new guide.
   * @param layer The layer on which to create the guide.
   * @param withProperties Initial values for properties of the new Guide
   */
  add(layer: Layer, withProperties: object): Guide

  /**
   * Returns any Guide in the collection.
   */
  anyItem(): Guide

  /**
   * Displays the number of elements in the Guide.
   */
  count(): number

  /**
   * Returns every Guide in the collection.
   */
  everyItem(): Guide

  /**
   * Returns the first Guide in the collection.
   */
  firstItem(): Guide

  /**
   * Returns the Guide with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Guide

  /**
   * Returns the Guide with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Guide

  /**
   * Returns the Guide with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Guide

  /**
   * Returns the Guides within the specified range.
   * @param from The Guide, index, or name at the beginning of the range.
   * @param to The Guide, index, or name at the end of the range.
   */
  itemByRange(from: Guide | number | string, to: Guide | number | string): Guide[]

  /**
   * Returns the last Guide in the collection.
   */
  lastItem(): Guide

  /**
   * Returns the middle Guide in the collection.
   */
  middleItem(): Guide

  /**
   * Returns the Guide whose index follows the specified Guide in the collection.
   * @param obj The Guide whose index comes before the desired Guide.
   */
  nextItem(obj: Guide): Guide

  /**
   * Returns the Guide with the index previous to the specified index.
   * @param obj The index of the Guide that follows the desired Guide.
   */
  previousItem(obj: Guide): Guide

  /**
   * Generates a string which, if executed, will return the Guide.
   */
  toSource(): string
}

/**
 * Text wrap preferences.
 */
declare class TextWrapPreference extends Preference {
  /**
   * If true, text wraps on the master spread apply to that spread only, and not to any pages the master spread has been applied to.
   */
  applyToMasterPageOnly: boolean

  /**
   * The contour of the text wrap. Valid only when when text wrap type is contour.
   */
  readonly contourOptions: ContourOption

  /**
   * If true, inverts the text wrap.
   */
  inverse: boolean

  /**
   * A collection of paths.
   */
  readonly paths: Paths

  /**
   * A collection of preferences objects.
   */
  readonly preferences: Preferences

  /**
   * The text wrap mode.
   */
  textWrapMode: TextWrapModes

  /**
   * The minimum space between text and the edges of the wrapped object. The format for defining text wrap offset values depends on the text wrap type. If text wrap type is jump object text wrap, specify 2 values in the format [top, bottom]. If text wrap type is next column text wrap or contour, specify a single value. For bounding box text wrap, specify 4 values in the format in the format [top, left, bottom, right]. .
   */
  textWrapOffset: number | number[] | NothingEnum

  /**
   * Text wrap side options.
   */
  textWrapSide: TextWrapSideOptions

  /**
   * If true, the text wrap path has been explicitly modified by the user.
   */
  readonly userModifiedWrap: boolean
}

/**
 * A contour option.
 */
declare class ContourOption extends Preference {
  /**
   * A list of the alpha channels stored in the graphic.
   */
  readonly alphaChannelPathNames: string[]

  /**
   * The alpha channel or Photoshop path to use for the contour option. Valid only when the contour options is photoshop path or alpha channel.
   */
  contourPathName: string

  /**
   * The contour type.
   */
  contourType: ContourOptionsTypes

  /**
   * If true, creates interior clipping paths within the surrounding clipping path. Note: Valid only when clipping type is alpha channel or detect edges.
   */
  includeInsideEdges: boolean

  /**
   * A list of the clipping paths stored in the graphic.
   */
  readonly photoshopPathNames: string[]
}

/**
 * A section.
 */
declare class Section {
  /**
   * The alternate layout name for a set of pages.
   */
  alternateLayout: string

  /**
   * The number of pages in the alternate layout section.
   */
  readonly alternateLayoutLength: number

  /**
   * If true, continues page numbers sequentially from the previous section.
   */
  continueNumbering: boolean

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The unique ID of the Section.
   */
  readonly id: number

  /**
   * If true, places the specified prefix before page numbers on all pages in the section.
   */
  includeSectionPrefix: boolean

  /**
   * The index of the Section within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The number of pages in the section.
   */
  readonly length: number

  /**
   * The section marker.
   */
  marker: string

  /**
   * The name of the Section.
   */
  name: string

  /**
   * The page number assigned to the first page in the section. Note: Valid only when continue numbering is false.
   */
  pageNumberStart: number

  /**
   * The page number style.
   */
  pageNumberStyle: PageNumberStyle | string

  /**
   * The start page for the section.
   */
  pageStart: Page

  /**
   * The parent of the Section (a Document).
   */
  readonly parent: Document

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * The prefix to place before page numbers on pages in the section. May include up to 8 characters. Note: Valid only when include section prefix is true.
   */
  sectionPrefix: string

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): Section[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Deletes the Section.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the Section.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of sections.
 */
declare class Sections {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Section with the specified index.
   * @param index The index.
   */
  [index: number]: Section

  /**
   * Creates a new section.
   * @param reference The page on which the section begins.
   * @param withProperties Initial values for properties of the new Section
   */
  add(reference: Page, withProperties: object): Section

  /**
   * Returns any Section in the collection.
   */
  anyItem(): Section

  /**
   * Displays the number of elements in the Section.
   */
  count(): number

  /**
   * Returns every Section in the collection.
   */
  everyItem(): Section

  /**
   * Returns the first Section in the collection.
   */
  firstItem(): Section

  /**
   * Returns the Section with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Section

  /**
   * Returns the Section with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Section

  /**
   * Returns the Section with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Section

  /**
   * Returns the Sections within the specified range.
   * @param from The Section, index, or name at the beginning of the range.
   * @param to The Section, index, or name at the end of the range.
   */
  itemByRange(from: Section | number | string, to: Section | number | string): Section[]

  /**
   * Returns the last Section in the collection.
   */
  lastItem(): Section

  /**
   * Returns the middle Section in the collection.
   */
  middleItem(): Section

  /**
   * Returns the Section whose index follows the specified Section in the collection.
   * @param obj The Section whose index comes before the desired Section.
   */
  nextItem(obj: Section): Section

  /**
   * Returns the Section with the index previous to the specified index.
   * @param obj The index of the Section that follows the desired Section.
   */
  previousItem(obj: Section): Section

  /**
   * Generates a string which, if executed, will return the Section.
   */
  toSource(): string
}

/**
 * A path.
 */
declare class Path {
  /**
   * A list of the coordinates of all of the path points on the path, including anchor points and left- and right-direction points. When creating a path using this property, supply either a list of anchor point coordinates ([[x1, y1], [x2, y2], ...]) or a list of anchor point, left-direction point, and right-direction point coordinates ([[[x1, y1], [x2, y2], [x3, y3]], [[x4, y4], [x5, y5], [x6, y6]], ...]). Note: Providing only anchor points results in a path on which all of the path points are connected with straight line segments; supplying the positions of left- and right-direction points specifies curved line segments. Can return: Array of Arrays of 2 Units.
   */
  entirePath: any[]

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The index of the Path within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * The parent of the Path (a SplineItem, Polygon, GraphicLine, Rectangle, Oval, TextFrame, EndnoteTextFrame, MediaItem, Sound, Movie, Button, MultiStateObject, ClippingPathSettings or TextWrapPreference).
   */
  readonly parent: any

  /**
   * A collection of path points.
   */
  readonly pathPoints: PathPoints

  /**
   * The path type.
   */
  pathType: PathType

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): Path[]

  /**
   * Deletes the Path.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Reverses the path.
   */
  reverse(): void

  /**
   * Generates a string which, if executed, will return the Path.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of paths.
 */
declare class Paths {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Path with the specified index.
   * @param index The index.
   */
  [index: number]: Path

  /**
   * Creates a new Path.
   * @param withProperties Initial values for properties of the new Path
   */
  add(withProperties: object): Path

  /**
   * Returns any Path in the collection.
   */
  anyItem(): Path

  /**
   * Displays the number of elements in the Path.
   */
  count(): number

  /**
   * Returns every Path in the collection.
   */
  everyItem(): Path

  /**
   * Returns the first Path in the collection.
   */
  firstItem(): Path

  /**
   * Returns the Path with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Path

  /**
   * Returns the Paths within the specified range.
   * @param from The Path, index, or name at the beginning of the range.
   * @param to The Path, index, or name at the end of the range.
   */
  itemByRange(from: Path | number | string, to: Path | number | string): Path[]

  /**
   * Returns the last Path in the collection.
   */
  lastItem(): Path

  /**
   * Returns the middle Path in the collection.
   */
  middleItem(): Path

  /**
   * Returns the Path whose index follows the specified Path in the collection.
   * @param obj The Path whose index comes before the desired Path.
   */
  nextItem(obj: Path): Path

  /**
   * Returns the Path with the index previous to the specified index.
   * @param obj The index of the Path that follows the desired Path.
   */
  previousItem(obj: Path): Path

  /**
   * Generates a string which, if executed, will return the Path.
   */
  toSource(): string
}

/**
 * A path point of a path.
 */
declare class PathPoint {
  /**
   * The location of the path point on the page, in the format [x, y].
   */
  anchor: string[] | number[]

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The index of the PathPoint within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * The left-direction point, which controls the curve of the line segment preceding the path point on the path, in the format [x, y].
   */
  leftDirection: string[] | number[]

  /**
   * The parent of the PathPoint (a Path).
   */
  readonly parent: Path

  /**
   * The path point type.
   */
  pointType: PointType

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * The right-direction point, which controls the curve of the line segment following the path point on the path, in the format [x, y].
   */
  rightDirection: string[] | number[]

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): PathPoint[]

  /**
   * Join this path point to another path point. The two points must be end points and their paths combined into a single path on a single page item.
   * @param reference The reference object. The path point to join to
   * @param given The join option to use.
   */
  join(reference: PathPoint, given: JoinOptions): void

  /**
   * Deletes the PathPoint.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the PathPoint.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of path points.
 */
declare class PathPoints {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the PathPoint with the specified index.
   * @param index The index.
   */
  [index: number]: PathPoint

  /**
   * Creates a new PathPoint.
   * @param withProperties Initial values for properties of the new PathPoint
   */
  add(withProperties: object): PathPoint

  /**
   * Returns any PathPoint in the collection.
   */
  anyItem(): PathPoint

  /**
   * Displays the number of elements in the PathPoint.
   */
  count(): number

  /**
   * Returns every PathPoint in the collection.
   */
  everyItem(): PathPoint

  /**
   * Returns the first PathPoint in the collection.
   */
  firstItem(): PathPoint

  /**
   * Returns the PathPoint with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): PathPoint

  /**
   * Returns the PathPoints within the specified range.
   * @param from The PathPoint, index, or name at the beginning of the range.
   * @param to The PathPoint, index, or name at the end of the range.
   */
  itemByRange(from: PathPoint | number | string, to: PathPoint | number | string): PathPoint[]

  /**
   * Returns the last PathPoint in the collection.
   */
  lastItem(): PathPoint

  /**
   * Returns the middle PathPoint in the collection.
   */
  middleItem(): PathPoint

  /**
   * Returns the PathPoint whose index follows the specified PathPoint in the collection.
   * @param obj The PathPoint whose index comes before the desired PathPoint.
   */
  nextItem(obj: PathPoint): PathPoint

  /**
   * Returns the PathPoint with the index previous to the specified index.
   * @param obj The index of the PathPoint that follows the desired PathPoint.
   */
  previousItem(obj: PathPoint): PathPoint

  /**
   * Generates a string which, if executed, will return the PathPoint.
   */
  toSource(): string
}

/**
 * A path-based page item, such as a rectangle, oval, polygon, or graphic line.
 */
declare class SplineItem extends PageItem {
  /**
   * Anchored object settings.
   */
  readonly anchoredObjectSettings: AnchoredObjectSetting

  /**
   * A collection of buttons.
   */
  readonly buttons: Buttons

  /**
   * A collection of checkboxes.
   */
  readonly checkBoxes: CheckBoxes

  /**
   * A collection of comboboxes.
   */
  readonly comboBoxes: ComboBoxes

  /**
   * The type of content that a frame can contain.
   */
  contentType: ContentType

  /**
   * A collection of endnote text frames.
   */
  readonly endnoteTextFrames: EndnoteTextFrames

  /**
   * A collection of EPS files.
   */
  readonly epss: EPSs

  /**
   * EPSTexts
   */
  readonly epstexts: EPSTexts

  /**
   * A collection of form fields.
   */
  readonly formFields: FormFields

  /**
   * A collection of graphic lines.
   */
  readonly graphicLines: GraphicLines

  /**
   * A collection of imported graphics in any graphic file format (vector, metafile, or bitmap).
   */
  readonly graphics: Graphics

  /**
   * A collection of groups.
   */
  readonly groups: Groups

  /**
   * A collection of embedded HTML page items.
   */
  readonly htmlItems: HtmlItems

  /**
   * A collection of bitmap images in any bitmap file format (including TIFF, JPEG, or GIF).
   */
  readonly images: Images

  /**
   * Imported InDesign pages.
   */
  readonly importedPages: ImportedPages

  /**
   * A collection of listboxes.
   */
  readonly listBoxes: ListBoxes

  /**
   * The lock state.
   */
  readonly lockState: LockStateValues

  /**
   * The media items collection.
   */
  readonly mediaItems: MediaItems

  /**
   * A collection of movies.
   */
  readonly movies: Movies

  /**
   * A collection of multi-state objects.
   */
  readonly multiStateObjects: MultiStateObjects

  /**
   * Export options for the object
   */
  readonly objectExportOptions: ObjectExportOption

  /**
   * A collection of ellipses.
   */
  readonly ovals: Ovals

  /**
   * The page items collection, which can be used to process all page items in a container (such as a document, page, or group), regardless of type.
   */
  readonly pageItems: PageItems

  /**
   * A collection of paths.
   */
  readonly paths: Paths

  /**
   * A collection of PDF files.
   */
  readonly pdfs: PDFs

  /**
   * A collection of PICT graphics.
   */
  readonly picts: PICTs

  /**
   * A collection of polygons.
   */
  readonly polygons: Polygons

  /**
   * A collection of radio buttons.
   */
  readonly radioButtons: RadioButtons

  /**
   * A collection of rectangles.
   */
  readonly rectangles: Rectangles

  /**
   * A collection of signature fields.
   */
  readonly signatureFields: SignatureFields

  /**
   * A collection of sound clips.
   */
  readonly sounds: Sounds

  /**
   * The spline items collection.
   */
  readonly splineItems: SplineItems

  /**
   * A collection of text boxes.
   */
  readonly textBoxes: TextBoxes

  /**
   * A collection of text frames.
   */
  readonly textFrames: TextFrames

  /**
   * A collection of text paths.
   */
  readonly textPaths: TextPaths

  /**
   * A collection of WMF graphics.
   */
  readonly wmfs: WMFs

  /**
   * Creates a new page item by combining the SplineItem with other objects. Deletes the objects if they do not intersect.
   * @param with_ The object(s) to add.
   */
  addPath(with_: PageItem[]): PageItem

  /**
   * Brings the SplineItem forward one level in its layer.
   */
  bringForward(): void

  /**
   * Brings the SplineItem to the front of its layer or in front of a particular item.
   * @param reference The reference object to bring the object in front of (must have same parent)
   */
  bringToFront(reference: PageItem): void

  /**
   * Checks in the story or stories.
   * @param versionComments The comment for this version.
   * @param forceSave If true, forcibly saves a version.
   */
  checkIn(versionComments: string, forceSave?: boolean): boolean

  /**
   * Checks out the story.
   */
  checkOut(): boolean

  /**
   * Creates a new page item by excluding the overlapping areas of the SplineItem and other objects.
   * @param with_ The object(s) to exclude.
   */
  excludeOverlapPath(with_: PageItem[]): PageItem

  /**
   * Creates a new page item by intersecting the SplineItem with other objects. Returns an error if the objects do not intersect.
   * @param with_ The object(s) with which to intersect.
   */
  intersectPath(with_: PageItem[]): PageItem

  /**
   * Creates a compound path by combining the path(s) of the SplineItem with the paths of other objects.
   * @param with_ The other objects whose paths to include in the new compound path.
   */
  makeCompoundPath(with_: PageItem[]): PageItem

  /**
   * Creates a new page item by reverse subtracting the overlapping areas of the SplineItem and other objects.
   * @param with_ The object(s) to reverse subtract.
   */
  minusBack(with_: PageItem[]): PageItem

  /**
   * Releases a compound path.
   */
  releaseCompoundPath(): PageItem[]

  /**
   * Reverts the document to its state at the last save operation.
   */
  revert(): boolean

  /**
   * Sends the SplineItem back one level in its layer.
   */
  sendBackward(): void

  /**
   * Sends the SplineItem to the back of its layer or behind a particular item (must have same parent).
   * @param reference The reference object to send the object behind
   */
  sendToBack(reference: PageItem): void

  /**
   * Creates a new page item by subtracting the overlapping areas of the SplineItem and other objects.
   * @param with_ The object(s) to subtract.
   */
  subtractPath(with_: PageItem[]): PageItem
}

/**
 * The spline items collection.
 */
declare class SplineItems {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the SplineItem with the specified index.
   * @param index The index.
   */
  [index: number]: SplineItem

  /**
   * Returns any SplineItem in the collection.
   */
  anyItem(): SplineItem

  /**
   * Displays the number of elements in the SplineItem.
   */
  count(): number

  /**
   * Returns every SplineItem in the collection.
   */
  everyItem(): SplineItem

  /**
   * Returns the first SplineItem in the collection.
   */
  firstItem(): SplineItem

  /**
   * Returns the SplineItem with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): SplineItem

  /**
   * Returns the SplineItem with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): SplineItem

  /**
   * Returns the SplineItem with the specified name.
   * @param name The name.
   */
  itemByName(name: string): SplineItem

  /**
   * Returns the SplineItems within the specified range.
   * @param from The SplineItem, index, or name at the beginning of the range.
   * @param to The SplineItem, index, or name at the end of the range.
   */
  itemByRange(from: SplineItem | number | string, to: SplineItem | number | string): SplineItem[]

  /**
   * Returns the last SplineItem in the collection.
   */
  lastItem(): SplineItem

  /**
   * Returns the middle SplineItem in the collection.
   */
  middleItem(): SplineItem

  /**
   * Returns the SplineItem whose index follows the specified SplineItem in the collection.
   * @param obj The SplineItem whose index comes before the desired SplineItem.
   */
  nextItem(obj: SplineItem): SplineItem

  /**
   * Returns the SplineItem with the index previous to the specified index.
   * @param obj The index of the SplineItem that follows the desired SplineItem.
   */
  previousItem(obj: SplineItem): SplineItem

  /**
   * Generates a string which, if executed, will return the SplineItem.
   */
  toSource(): string
}

/**
 * EPSText.
 */
declare class EPSText extends PageItem {
  /**
   * Anchored object settings.
   */
  readonly anchoredObjectSettings: AnchoredObjectSetting

  /**
   * A collection of text paths.
   */
  readonly textPaths: TextPaths

  /**
   * Brings the EPSText forward one level in its layer.
   */
  bringForward(): void

  /**
   * Brings the EPSText to the front of its layer or in front of a particular item.
   * @param reference The reference object to bring the object in front of (must have same parent)
   */
  bringToFront(reference: PageItem): void

  /**
   * Sends the EPSText back one level in its layer.
   */
  sendBackward(): void

  /**
   * Sends the EPSText to the back of its layer or behind a particular item (must have same parent).
   * @param reference The reference object to send the object behind
   */
  sendToBack(reference: PageItem): void
}

/**
 * EPSTexts
 */
declare class EPSTexts {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the EPSText with the specified index.
   * @param index The index.
   */
  [index: number]: EPSText

  /**
   * Returns any EPSText in the collection.
   */
  anyItem(): EPSText

  /**
   * Displays the number of elements in the EPSText.
   */
  count(): number

  /**
   * Returns every EPSText in the collection.
   */
  everyItem(): EPSText

  /**
   * Returns the first EPSText in the collection.
   */
  firstItem(): EPSText

  /**
   * Returns the EPSText with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): EPSText

  /**
   * Returns the EPSText with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): EPSText

  /**
   * Returns the EPSText with the specified name.
   * @param name The name.
   */
  itemByName(name: string): EPSText

  /**
   * Returns the EPSTexts within the specified range.
   * @param from The EPSText, index, or name at the beginning of the range.
   * @param to The EPSText, index, or name at the end of the range.
   */
  itemByRange(from: EPSText | number | string, to: EPSText | number | string): EPSText[]

  /**
   * Returns the last EPSText in the collection.
   */
  lastItem(): EPSText

  /**
   * Returns the middle EPSText in the collection.
   */
  middleItem(): EPSText

  /**
   * Returns the EPSText whose index follows the specified EPSText in the collection.
   * @param obj The EPSText whose index comes before the desired EPSText.
   */
  nextItem(obj: EPSText): EPSText

  /**
   * Returns the EPSText with the index previous to the specified index.
   * @param obj The index of the EPSText that follows the desired EPSText.
   */
  previousItem(obj: EPSText): EPSText

  /**
   * Generates a string which, if executed, will return the EPSText.
   */
  toSource(): string
}

/**
 * An imported InDesign page.
 */
declare class ImportedPage extends Graphic {
  /**
   * Clipping path settings.
   */
  readonly clippingPath: ClippingPathSettings

  /**
   * Graphic layer option settings.
   */
  readonly graphicLayerOptions: GraphicLayerOption

  /**
   * Specifies the cropping of the imported InDesign page. Read only for page items.
   */
  readonly importedPageCrop: ImportedPageCropOptions

  /**
   * Which page of the InDesign document should be imported. Read only for page items.
   */
  readonly pageNumber: number

  /**
   * PDF attribute settings.
   */
  readonly pdfAttributes: PDFAttribute
}

/**
 * Imported InDesign pages.
 */
declare class ImportedPages {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the ImportedPage with the specified index.
   * @param index The index.
   */
  [index: number]: ImportedPage

  /**
   * Creates a new ImportedPage.
   * @param withProperties Initial values for properties of the new ImportedPage
   */
  add(withProperties: object): ImportedPage

  /**
   * Returns any ImportedPage in the collection.
   */
  anyItem(): ImportedPage

  /**
   * Displays the number of elements in the ImportedPage.
   */
  count(): number

  /**
   * Returns every ImportedPage in the collection.
   */
  everyItem(): ImportedPage

  /**
   * Returns the first ImportedPage in the collection.
   */
  firstItem(): ImportedPage

  /**
   * Returns the ImportedPage with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): ImportedPage

  /**
   * Returns the ImportedPage with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): ImportedPage

  /**
   * Returns the ImportedPage with the specified name.
   * @param name The name.
   */
  itemByName(name: string): ImportedPage

  /**
   * Returns the ImportedPages within the specified range.
   * @param from The ImportedPage, index, or name at the beginning of the range.
   * @param to The ImportedPage, index, or name at the end of the range.
   */
  itemByRange(
    from: ImportedPage | number | string,
    to: ImportedPage | number | string,
  ): ImportedPage[]

  /**
   * Returns the last ImportedPage in the collection.
   */
  lastItem(): ImportedPage

  /**
   * Returns the middle ImportedPage in the collection.
   */
  middleItem(): ImportedPage

  /**
   * Returns the ImportedPage whose index follows the specified ImportedPage in the collection.
   * @param obj The ImportedPage whose index comes before the desired ImportedPage.
   */
  nextItem(obj: ImportedPage): ImportedPage

  /**
   * Returns the ImportedPage with the index previous to the specified index.
   * @param obj The index of the ImportedPage that follows the desired ImportedPage.
   */
  previousItem(obj: ImportedPage): ImportedPage

  /**
   * Generates a string which, if executed, will return the ImportedPage.
   */
  toSource(): string
}

/**
 * An sound or movie page item.
 */
declare class MediaItem extends PageItem {
  /**
   * Dispatched when the value of a property changes on this MediaItem. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_ATTRIBUTE_CHANGED: string

  /**
   * A collection of paths.
   */
  readonly paths: Paths
}

/**
 * The media items collection.
 */
declare class MediaItems {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the MediaItem with the specified index.
   * @param index The index.
   */
  [index: number]: MediaItem

  /**
   * Returns any MediaItem in the collection.
   */
  anyItem(): MediaItem

  /**
   * Displays the number of elements in the MediaItem.
   */
  count(): number

  /**
   * Returns every MediaItem in the collection.
   */
  everyItem(): MediaItem

  /**
   * Returns the first MediaItem in the collection.
   */
  firstItem(): MediaItem

  /**
   * Returns the MediaItem with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): MediaItem

  /**
   * Returns the MediaItem with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): MediaItem

  /**
   * Returns the MediaItem with the specified name.
   * @param name The name.
   */
  itemByName(name: string): MediaItem

  /**
   * Returns the MediaItems within the specified range.
   * @param from The MediaItem, index, or name at the beginning of the range.
   * @param to The MediaItem, index, or name at the end of the range.
   */
  itemByRange(from: MediaItem | number | string, to: MediaItem | number | string): MediaItem[]

  /**
   * Returns the last MediaItem in the collection.
   */
  lastItem(): MediaItem

  /**
   * Returns the middle MediaItem in the collection.
   */
  middleItem(): MediaItem

  /**
   * Returns the MediaItem whose index follows the specified MediaItem in the collection.
   * @param obj The MediaItem whose index comes before the desired MediaItem.
   */
  nextItem(obj: MediaItem): MediaItem

  /**
   * Returns the MediaItem with the index previous to the specified index.
   * @param obj The index of the MediaItem that follows the desired MediaItem.
   */
  previousItem(obj: MediaItem): MediaItem

  /**
   * Generates a string which, if executed, will return the MediaItem.
   */
  toSource(): string
}

/**
 * An IDML snippet.
 */
declare class Snippet {
  /**
   * Dispatched before a Snippet is placed. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_PLACE: string

  /**
   * A collection of buttons.
   */
  readonly buttons: Buttons

  /**
   * A collection of checkboxes.
   */
  readonly checkBoxes: CheckBoxes

  /**
   * A collection of comboboxes.
   */
  readonly comboBoxes: ComboBoxes

  /**
   * A collection of EPS files.
   */
  readonly epss: EPSs

  /**
   * EPSTexts
   */
  readonly epstexts: EPSTexts

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * A collection of form fields.
   */
  readonly formFields: FormFields

  /**
   * A collection of graphic lines.
   */
  readonly graphicLines: GraphicLines

  /**
   * A collection of imported graphics in any graphic file format (vector, metafile, or bitmap).
   */
  readonly graphics: Graphics

  /**
   * A collection of groups.
   */
  readonly groups: Groups

  /**
   * The unique ID of the Snippet.
   */
  readonly id: number

  /**
   * A collection of bitmap images in any bitmap file format (including TIFF, JPEG, or GIF).
   */
  readonly images: Images

  /**
   * The index of the Snippet within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * A collection of listboxes.
   */
  readonly listBoxes: ListBoxes

  /**
   * A collection of multi-state objects.
   */
  readonly multiStateObjects: MultiStateObjects

  /**
   * The name of the Snippet; this is an alias to the Snippet's label property.
   */
  name: string

  /**
   * A collection of ellipses.
   */
  readonly ovals: Ovals

  /**
   * The page items collection, which can be used to process all page items in a container (such as a document, page, or group), regardless of type.
   */
  readonly pageItems: PageItems

  /**
   * The parent of the Snippet (a PlaceGun).
   */
  readonly parent: PlaceGun

  /**
   * A collection of PDF files.
   */
  readonly pdfs: PDFs

  /**
   * A collection of PICT graphics.
   */
  readonly picts: PICTs

  /**
   * A collection of polygons.
   */
  readonly polygons: Polygons

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * A collection of radio buttons.
   */
  readonly radioButtons: RadioButtons

  /**
   * A collection of rectangles.
   */
  readonly rectangles: Rectangles

  /**
   * A collection of signature fields.
   */
  readonly signatureFields: SignatureFields

  /**
   * The spline items collection.
   */
  readonly splineItems: SplineItems

  /**
   * A collection of text boxes.
   */
  readonly textBoxes: TextBoxes

  /**
   * A collection of text frames.
   */
  readonly textFrames: TextFrames

  /**
   * A collection of WMF graphics.
   */
  readonly wmfs: WMFs

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): Snippet[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Deletes the Snippet.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the Snippet.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of IDML snippets.
 */
declare class Snippets {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Snippet with the specified index.
   * @param index The index.
   */
  [index: number]: Snippet

  /**
   * Returns any Snippet in the collection.
   */
  anyItem(): Snippet

  /**
   * Displays the number of elements in the Snippet.
   */
  count(): number

  /**
   * Returns every Snippet in the collection.
   */
  everyItem(): Snippet

  /**
   * Returns the first Snippet in the collection.
   */
  firstItem(): Snippet

  /**
   * Returns the Snippet with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Snippet

  /**
   * Returns the Snippet with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Snippet

  /**
   * Returns the Snippet with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Snippet

  /**
   * Returns the Snippets within the specified range.
   * @param from The Snippet, index, or name at the beginning of the range.
   * @param to The Snippet, index, or name at the end of the range.
   */
  itemByRange(from: Snippet | number | string, to: Snippet | number | string): Snippet[]

  /**
   * Returns the last Snippet in the collection.
   */
  lastItem(): Snippet

  /**
   * Returns the middle Snippet in the collection.
   */
  middleItem(): Snippet

  /**
   * Returns the Snippet whose index follows the specified Snippet in the collection.
   * @param obj The Snippet whose index comes before the desired Snippet.
   */
  nextItem(obj: Snippet): Snippet

  /**
   * Returns the Snippet with the index previous to the specified index.
   * @param obj The index of the Snippet that follows the desired Snippet.
   */
  previousItem(obj: Snippet): Snippet

  /**
   * Generates a string which, if executed, will return the Snippet.
   */
  toSource(): string
}

/**
 * Options for applying layout attributes to any page item.
 */
declare class TransformAttributeOption extends Preference {
  /**
   * The height of the object, defined by the object style.
   */
  transformAttrHeight: string | number

  /**
   * The reference point to be used while setting the X attribute of object style.
   */
  transformAttrLeftReference: TransformPositionReference

  /**
   * Option to specify the achor point to be used by the style for anchoring the object while applying the position.
   */
  transformAttrRefAnchorPoint: AnchorPoint

  /**
   * The reference point to be used while setting the Y attribute of object style.
   */
  transformAttrTopReference: TransformPositionReference

  /**
   * The width of the object, defined by the object style.
   */
  transformAttrWidth: string | number

  /**
   * The left position of the object, defined by the object style.
   */
  transformAttrX: string | number

  /**
   * The top position of the object, defined by the object style.
   */
  transformAttrY: string | number
}

/**
 * A Motion preset Object.
 */
declare class MotionPreset {
  /**
   * Motion preset raw data
   */
  contents: string

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The unique ID of the MotionPreset.
   */
  readonly id: number

  /**
   * The index of the MotionPreset within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The name of the MotionPreset.
   */
  name: string

  /**
   * The parent of the MotionPreset (a Application or Document).
   */
  readonly parent: any

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Duplicates the motion preset.
   * @param name The name for the duplicated motion preset.
   */
  duplicate(name: string): MotionPreset

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): MotionPreset[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Deletes the MotionPreset.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Save a copy of this motion preset to a InDesign motion preset file.
   * @param to The Flash motion preset file to export to.
   */
  saveACopy(to: File): void

  /**
   * Generates a string which, if executed, will return the MotionPreset.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of Motion presets.
 */
declare class MotionPresets {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the MotionPreset with the specified index.
   * @param index The index.
   */
  [index: number]: MotionPreset

  /**
   * Creates a new MotionPreset.
   * @param withProperties Initial values for properties of the new MotionPreset
   */
  add(withProperties: object): MotionPreset

  /**
   * Returns any MotionPreset in the collection.
   */
  anyItem(): MotionPreset

  /**
   * Displays the number of elements in the MotionPreset.
   */
  count(): number

  /**
   * Returns every MotionPreset in the collection.
   */
  everyItem(): MotionPreset

  /**
   * Returns the first MotionPreset in the collection.
   */
  firstItem(): MotionPreset

  /**
   * Returns the MotionPreset with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): MotionPreset

  /**
   * Returns the MotionPreset with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): MotionPreset

  /**
   * Returns the MotionPreset with the specified name.
   * @param name The name.
   */
  itemByName(name: string): MotionPreset

  /**
   * Returns the MotionPresets within the specified range.
   * @param from The MotionPreset, index, or name at the beginning of the range.
   * @param to The MotionPreset, index, or name at the end of the range.
   */
  itemByRange(
    from: MotionPreset | number | string,
    to: MotionPreset | number | string,
  ): MotionPreset[]

  /**
   * Returns the last MotionPreset in the collection.
   */
  lastItem(): MotionPreset

  /**
   * Returns the middle MotionPreset in the collection.
   */
  middleItem(): MotionPreset

  /**
   * Returns the MotionPreset whose index follows the specified MotionPreset in the collection.
   * @param obj The MotionPreset whose index comes before the desired MotionPreset.
   */
  nextItem(obj: MotionPreset): MotionPreset

  /**
   * Returns the MotionPreset with the index previous to the specified index.
   * @param obj The index of the MotionPreset that follows the desired MotionPreset.
   */
  previousItem(obj: MotionPreset): MotionPreset

  /**
   * Generates a string which, if executed, will return the MotionPreset.
   */
  toSource(): string
}

/**
 * An article
 */
declare class Article {
  /**
   * The export status of the Article
   */
  articleExportStatus: boolean

  /**
   * A collection of article members.
   */
  readonly articleMembers: ArticleMembers

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The unique ID of the Article.
   */
  readonly id: number

  /**
   * The index of the Article within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The name of the Article.
   */
  name: string

  /**
   * The parent of the Article (a Document).
   */
  readonly parent: Document

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * Populates the article with all page items in the document.
   */
  addDocumentContent(): void

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): Article[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Moves the Article to the specified location.
   * @param to The location relative to the reference object or within the containing object.
   * @param reference The reference object. Note: Required when the to value specifies before or after.
   */
  move(to: LocationOptions, reference: Article): Article

  /**
   * Deletes the Article.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the Article.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of articles.
 */
declare class Articles {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Article with the specified index.
   * @param index The index.
   */
  [index: number]: Article

  /**
   * Creates a new Article
   * @param name The article name
   * @param articleExportStatus The article's export status
   * @param at The location relative to the reference object or within the containing object.
   * @param reference The reference object. Note: Required when the to value specifies before or after.
   * @param withProperties Initial values for properties of the new Article
   */
  add(
    name: string,
    articleExportStatus: boolean,
    at?: LocationOptions,
    reference?: Article,
    withProperties?: object,
  ): Article

  /**
   * Returns any Article in the collection.
   */
  anyItem(): Article

  /**
   * Displays the number of elements in the Article.
   */
  count(): number

  /**
   * Returns every Article in the collection.
   */
  everyItem(): Article

  /**
   * Returns the first Article in the collection.
   */
  firstItem(): Article

  /**
   * Returns the Article with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Article

  /**
   * Returns the Article with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Article

  /**
   * Returns the Article with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Article

  /**
   * Returns the Articles within the specified range.
   * @param from The Article, index, or name at the beginning of the range.
   * @param to The Article, index, or name at the end of the range.
   */
  itemByRange(from: Article | number | string, to: Article | number | string): Article[]

  /**
   * Returns the last Article in the collection.
   */
  lastItem(): Article

  /**
   * Returns the middle Article in the collection.
   */
  middleItem(): Article

  /**
   * Returns the Article whose index follows the specified Article in the collection.
   * @param obj The Article whose index comes before the desired Article.
   */
  nextItem(obj: Article): Article

  /**
   * Returns the Article with the index previous to the specified index.
   * @param obj The index of the Article that follows the desired Article.
   */
  previousItem(obj: Article): Article

  /**
   * Generates a string which, if executed, will return the Article.
   */
  toSource(): string
}

/**
 * An article member.
 */
declare class ArticleMember {
  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The unique ID of the ArticleMember.
   */
  readonly id: number

  /**
   * The index of the ArticleMember within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * The underlying page item
   */
  readonly itemRef: PageItem

  /**
   * The parent of the ArticleMember (a Article).
   */
  readonly parent: Article

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): ArticleMember[]

  /**
   * Moves the ArticleMember to the specified location.
   * @param to The location relative to the reference object or within the containing object.
   * @param reference The reference object. Note: Required when the to value specifies before or after.
   */
  move(to: LocationOptions, reference: ArticleMember): ArticleMember

  /**
   * Deletes the ArticleMember.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the ArticleMember.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of article members.
 */
declare class ArticleMembers {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the ArticleMember with the specified index.
   * @param index The index.
   */
  [index: number]: ArticleMember

  /**
   * Adds a new member.
   * @param itemRef page item to be added to article
   * @param at The location relative to the reference object or within the containing object.
   * @param reference The reference object. Note: Required when the to value specifies before or after.
   * @param withProperties Initial values for properties of the new ArticleMember
   */
  add(
    itemRef: PageItem,
    at?: LocationOptions,
    reference?: ArticleMember,
    withProperties?: object,
  ): ArticleMember

  /**
   * Returns any ArticleMember in the collection.
   */
  anyItem(): ArticleMember

  /**
   * Displays the number of elements in the ArticleMember.
   */
  count(): number

  /**
   * Returns every ArticleMember in the collection.
   */
  everyItem(): ArticleMember

  /**
   * Returns the first ArticleMember in the collection.
   */
  firstItem(): ArticleMember

  /**
   * Returns the ArticleMember with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): ArticleMember

  /**
   * Returns the ArticleMember with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): ArticleMember

  /**
   * Returns the ArticleMembers within the specified range.
   * @param from The ArticleMember, index, or name at the beginning of the range.
   * @param to The ArticleMember, index, or name at the end of the range.
   */
  itemByRange(
    from: ArticleMember | number | string,
    to: ArticleMember | number | string,
  ): ArticleMember[]

  /**
   * Returns the last ArticleMember in the collection.
   */
  lastItem(): ArticleMember

  /**
   * Returns the middle ArticleMember in the collection.
   */
  middleItem(): ArticleMember

  /**
   * Returns the ArticleMember whose index follows the specified ArticleMember in the collection.
   * @param obj The ArticleMember whose index comes before the desired ArticleMember.
   */
  nextItem(obj: ArticleMember): ArticleMember

  /**
   * Returns the ArticleMember with the index previous to the specified index.
   * @param obj The index of the ArticleMember that follows the desired ArticleMember.
   */
  previousItem(obj: ArticleMember): ArticleMember

  /**
   * Generates a string which, if executed, will return the ArticleMember.
   */
  toSource(): string
}

/**
 * Member of a group that is part of an article.
 */
declare class ArticleChild {
  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The unique ID of the ArticleChild.
   */
  readonly id: number

  /**
   * The index of the ArticleChild within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * The underlying page item
   */
  readonly itemRef: PageItem

  /**
   * The parent of the ArticleChild (a Group).
   */
  readonly parent: Group

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): ArticleChild[]

  /**
   * Moves the ArticleChild to the specified location.
   * @param to The location relative to the reference object or within the containing object.
   * @param reference The reference object. Note: Required when the to value specifies before or after.
   */
  move(to: LocationOptions, reference: ArticleChild): ArticleChild

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the ArticleChild.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of group items that are also part of an article.
 */
declare class ArticleChildren {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the ArticleChild with the specified index.
   * @param index The index.
   */
  [index: number]: ArticleChild

  /**
   * Returns any ArticleChild in the collection.
   */
  anyItem(): ArticleChild

  /**
   * Displays the number of elements in the ArticleChild.
   */
  count(): number

  /**
   * Returns every ArticleChild in the collection.
   */
  everyItem(): ArticleChild

  /**
   * Returns the first ArticleChild in the collection.
   */
  firstItem(): ArticleChild

  /**
   * Returns the ArticleChild with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): ArticleChild

  /**
   * Returns the ArticleChild with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): ArticleChild

  /**
   * Returns the ArticleChildren within the specified range.
   * @param from The ArticleChild, index, or name at the beginning of the range.
   * @param to The ArticleChild, index, or name at the end of the range.
   */
  itemByRange(
    from: ArticleChild | number | string,
    to: ArticleChild | number | string,
  ): ArticleChild[]

  /**
   * Returns the last ArticleChild in the collection.
   */
  lastItem(): ArticleChild

  /**
   * Returns the middle ArticleChild in the collection.
   */
  middleItem(): ArticleChild

  /**
   * Returns the ArticleChild whose index follows the specified ArticleChild in the collection.
   * @param obj The ArticleChild whose index comes before the desired ArticleChild.
   */
  nextItem(obj: ArticleChild): ArticleChild

  /**
   * Returns the ArticleChild with the index previous to the specified index.
   * @param obj The index of the ArticleChild that follows the desired ArticleChild.
   */
  previousItem(obj: ArticleChild): ArticleChild

  /**
   * Generates a string which, if executed, will return the ArticleChild.
   */
  toSource(): string
}

/**
 * Embedded HTML.
 */
declare class HtmlItem extends PageItem {
  /**
   * Is this HTML able to adapt its width and height based on changes to the parent div or does it have fixed dimensions?
   */
  fixedDimensions: boolean

  /**
   * The embedded HTML text
   */
  htmlContent: string
}

/**
 * A collection of embedded HTML page items.
 */
declare class HtmlItems {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the HtmlItem with the specified index.
   * @param index The index.
   */
  [index: number]: HtmlItem

  /**
   * Creates a new HtmlItem
   * @param layer The layer on which to create the HtmlItem.
   * @param at The location at which to insert the HtmlItem relative to the reference object or within the container object.
   * @param reference The reference object. Note: Required when the at parameter is before or after.
   * @param withProperties Initial values for properties of the new HtmlItem
   */
  add(
    layer: Layer,
    at?: LocationOptions,
    reference?: Document | Spread | MasterSpread | Page | Layer | PageItem,
    withProperties?: object,
  ): HtmlItem

  /**
   * Returns any HtmlItem in the collection.
   */
  anyItem(): HtmlItem

  /**
   * Displays the number of elements in the HtmlItem.
   */
  count(): number

  /**
   * Returns every HtmlItem in the collection.
   */
  everyItem(): HtmlItem

  /**
   * Returns the first HtmlItem in the collection.
   */
  firstItem(): HtmlItem

  /**
   * Returns the HtmlItem with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): HtmlItem

  /**
   * Returns the HtmlItem with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): HtmlItem

  /**
   * Returns the HtmlItem with the specified name.
   * @param name The name.
   */
  itemByName(name: string): HtmlItem

  /**
   * Returns the HtmlItems within the specified range.
   * @param from The HtmlItem, index, or name at the beginning of the range.
   * @param to The HtmlItem, index, or name at the end of the range.
   */
  itemByRange(from: HtmlItem | number | string, to: HtmlItem | number | string): HtmlItem[]

  /**
   * Returns the last HtmlItem in the collection.
   */
  lastItem(): HtmlItem

  /**
   * Returns the middle HtmlItem in the collection.
   */
  middleItem(): HtmlItem

  /**
   * Returns the HtmlItem whose index follows the specified HtmlItem in the collection.
   * @param obj The HtmlItem whose index comes before the desired HtmlItem.
   */
  nextItem(obj: HtmlItem): HtmlItem

  /**
   * Returns the HtmlItem with the index previous to the specified index.
   * @param obj The index of the HtmlItem that follows the desired HtmlItem.
   */
  previousItem(obj: HtmlItem): HtmlItem

  /**
   * Generates a string which, if executed, will return the HtmlItem.
   */
  toSource(): string
}

/**
 * An object library.
 */
declare class Library {
  /**
   * A collection of object library assets.
   */
  readonly assets: Assets

  /**
   * The associated panel.
   */
  readonly associatedPanel: Panel

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The full path to the file.
   */
  readonly filePath: File

  /**
   * The full path to the Library, including the name of the Library.
   */
  readonly fullName: File

  /**
   * The index of the Library within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * The name of the Library.
   */
  readonly name: string

  /**
   * The parent of the Library (a Application).
   */
  readonly parent: Application

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Closes the Library.
   */
  close(): void

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): Library[]

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Stores the specified object in the library.
   * @param using The page item(s) to store.
   * @param withProperties Initial values for properties of the new Library
   */
  store(using: PageItem[] | Movies | Sounds | Graphics | XMLElements, withProperties: object): Asset

  /**
   * Generates a string which, if executed, will return the Library.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of object libraries.
 */
declare class Libraries {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Library with the specified index.
   * @param index The index.
   */
  [index: number]: Library

  /**
   * Creates a new object libary.
   * @param fullName The library's path and file name.
   * @param withProperties Initial values for properties of the new Library
   */
  add(fullName: File, withProperties: object): Library

  /**
   * Returns any Library in the collection.
   */
  anyItem(): Library

  /**
   * Displays the number of elements in the Library.
   */
  count(): number

  /**
   * Returns every Library in the collection.
   */
  everyItem(): Library

  /**
   * Returns the first Library in the collection.
   */
  firstItem(): Library

  /**
   * Returns the Library with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Library

  /**
   * Returns the Library with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Library

  /**
   * Returns the Libraries within the specified range.
   * @param from The Library, index, or name at the beginning of the range.
   * @param to The Library, index, or name at the end of the range.
   */
  itemByRange(from: Library | number | string, to: Library | number | string): Library[]

  /**
   * Returns the last Library in the collection.
   */
  lastItem(): Library

  /**
   * Returns the middle Library in the collection.
   */
  middleItem(): Library

  /**
   * Returns the Library whose index follows the specified Library in the collection.
   * @param obj The Library whose index comes before the desired Library.
   */
  nextItem(obj: Library): Library

  /**
   * Returns the Library with the index previous to the specified index.
   * @param obj The index of the Library that follows the desired Library.
   */
  previousItem(obj: Library): Library

  /**
   * Generates a string which, if executed, will return the Library.
   */
  toSource(): string
}

/**
 * An object library asset.
 */
declare class Asset {
  /**
   * The type of object library asset.
   */
  assetType: AssetType

  /**
   * The date and time the Asset was created.
   */
  readonly date: Date

  /**
   * The description of the Asset.
   */
  description: string

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The unique ID of the Asset.
   */
  readonly id: number

  /**
   * The index of the Asset within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The name of the Asset.
   */
  name: string

  /**
   * The parent of the Asset (a Library).
   */
  readonly parent: Library

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): Asset[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Places the asset in the specified document or text.
   * @param on The document or text object in which to place the asset.
   */
  placeAsset(on: Document | Text): any[]

  /**
   * Deletes the Asset.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Selects the object.
   * @param existingSelection The selection status of the Asset in relation to previously selected objects.
   */
  select(existingSelection?: SelectionOptions): void

  /**
   * Generates a string which, if executed, will return the Asset.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of object library assets.
 */
declare class Assets {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Asset with the specified index.
   * @param index The index.
   */
  [index: number]: Asset

  /**
   * Returns any Asset in the collection.
   */
  anyItem(): Asset

  /**
   * Displays the number of elements in the Asset.
   */
  count(): number

  /**
   * Returns every Asset in the collection.
   */
  everyItem(): Asset

  /**
   * Returns the first Asset in the collection.
   */
  firstItem(): Asset

  /**
   * Returns the Asset with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Asset

  /**
   * Returns the Asset with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Asset

  /**
   * Returns the Asset with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Asset

  /**
   * Returns the Assets within the specified range.
   * @param from The Asset, index, or name at the beginning of the range.
   * @param to The Asset, index, or name at the end of the range.
   */
  itemByRange(from: Asset | number | string, to: Asset | number | string): Asset[]

  /**
   * Returns the last Asset in the collection.
   */
  lastItem(): Asset

  /**
   * Returns the middle Asset in the collection.
   */
  middleItem(): Asset

  /**
   * Returns the Asset whose index follows the specified Asset in the collection.
   * @param obj The Asset whose index comes before the desired Asset.
   */
  nextItem(obj: Asset): Asset

  /**
   * Returns the Asset with the index previous to the specified index.
   * @param obj The index of the Asset that follows the desired Asset.
   */
  previousItem(obj: Asset): Asset

  /**
   * Generates a string which, if executed, will return the Asset.
   */
  toSource(): string
}

/**
 * A link to a placed file.
 */
declare class Link {
  /**
   * Dispatched when the value of a property changes on this Link. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_ATTRIBUTE_CHANGED: string

  /**
   * Dispatched after a Link is deleted. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_DELETE: string

  /**
   * Dispatched after a Link is embedded. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_EMBED: string

  /**
   * Dispatched after a Link is relocated from one object to another. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_MOVE: string

  /**
   * Dispatched after a Link is created. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_NEW: string

  /**
   * Dispatched after a Link is unembedded. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_UNEMBED: string

  /**
   * Dispatched after a Link is updated. This event bubbles. This event is not cancelable.
   */
  static readonly AFTER_UPDATE: string

  /**
   * Dispatched before a Link is deleted. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_DELETE: string

  /**
   * Dispatched before a Link is embedded. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_EMBED: string

  /**
   * Dispatched before a Link is relocated from one object to another. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_MOVE: string

  /**
   * Dispatched before a Link is unembedded. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_UNEMBED: string

  /**
   * Dispatched before a Link is updated. This event bubbles. This event is cancelable.
   */
  static readonly BEFORE_UPDATE: string

  /**
   * The asset ID of the linked object.
   */
  readonly assetID: string

  /**
   * The asset URL of the linked object.
   */
  readonly assetURL: string

  /**
   * The date and time the Link was created.
   */
  readonly date: Date

  /**
   * If true, indicates the linked object has been edited in the current document but the source file has not been updated.
   */
  readonly edited: boolean

  /**
   * The Version Cue editing state of the file.
   */
  readonly editingState: EditingState

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The file path (colon delimited on the Mac OS).
   */
  readonly filePath: string | File

  /**
   * The unique ID of the Link.
   */
  readonly id: number

  /**
   * The index of the link in the links collection.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The URI of the linked resource.
   */
  readonly linkResourceURI: string

  /**
   * The file type of the linked object.
   */
  readonly linkType: string

  /**
   * XMP data for the link source file.
   */
  readonly linkXmp: LinkMetadata

  /**
   * A collection of links.
   */
  readonly links: Links

  /**
   * The name of the Link.
   */
  readonly name: string

  /**
   * If true, indicates a link to a full-resolution version of the source file is needed. If false, indicates the object is embedded.
   */
  readonly needed: boolean

  /**
   * The linked object.
   */
  readonly parent: Story | Graphic | Movie | Sound

  /**
   * A collection of preferences objects.
   */
  readonly preferences: Preferences

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * The rendition data of the link resource.
   */
  readonly renditionData: LinkResourceRenditionType

  /**
   * The size of the Link file.
   */
  readonly size: number

  /**
   * The status of the link.
   */
  readonly status: LinkStatus

  /**
   * The Version Cue version state of the file.
   */
  readonly versionState: VersionState

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Check in to Version Cue.
   * @param versionComments The comment for this version
   * @param forceSave Forcibly save a version
   */
  checkIn(versionComments: string, forceSave?: boolean): void

  /**
   * Copies the link file to the specified location.
   * @param to The file or folder to which to copy the file.
   * @param versionComments The comment for this version.
   * @param forceSave If true, forcibly saves a version.
   */
  copyLink(to: File, versionComments: string, forceSave?: boolean): void

  /**
   * Opens the source file of the link in the default editor for the source file type.
   */
  editOriginal(): void

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): Link[]

  /**
   * Opens the source file of the link in InDesign for SharedContent links.
   */
  goToSource(): void

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Experimental: Reinitialize the link to a new uri
   * @param linkResourceURI Resource URI to relink.
   */
  reinitLink(linkResourceURI: string): void

  /**
   * Points the link to a new source file.
   * @param to The full path name of the new source file.
   */
  relink(to: File | string): void

  /**
   * Experimental: Relink the text fragment link to a new uri
   * @param linkResourceURI Resource URI to relink to.
   * @param name The tag name for the key.
   */
  relinkTextFragmentLink(linkResourceURI: string, name: string): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Experimental: Download the original asset and replace FPO with it.
   */
  replaceWithOriginal(): void

  /**
   * Opens Adobe Bridge and selects the source file of the link.
   */
  revealInBridge(): void

  /**
   * Opens the file system to the folder that contains the source file of the link, and selects the file.
   */
  revealInSystem(): void

  /**
   * Selects the link.
   */
  show(): void

  /**
   * Generates a string which, if executed, will return the Link.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string

  /**
   * Unembeds the source file. If no folder is specified, creates a link to the original source file; if a folder is specified, copies the file to the folder and creates a link to the copied file.
   * @param to The folder to which to copy the unembedded file.
   * @param versionComments The comment for this version.
   * @param forceSave If true, forcibly saves a version.
   */
  unembed(to: File, versionComments: string, forceSave?: boolean): void

  /**
   * Embeds the source file in the document.
   */
  unlink(): void

  /**
   * Updates the link if the source file has been changed.
   */
  update(): Link
}

/**
 * A collection of links.
 */
declare class Links {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the Link with the specified index.
   * @param index The index.
   */
  [index: number]: Link

  /**
   * Returns any Link in the collection.
   */
  anyItem(): Link

  /**
   * Displays the number of elements in the Link.
   */
  count(): number

  /**
   * Returns every Link in the collection.
   */
  everyItem(): Link

  /**
   * Returns the first Link in the collection.
   */
  firstItem(): Link

  /**
   * Returns the Link with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): Link

  /**
   * Returns the Link with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): Link

  /**
   * Returns the Link with the specified name.
   * @param name The name.
   */
  itemByName(name: string): Link

  /**
   * Returns the Links within the specified range.
   * @param from The Link, index, or name at the beginning of the range.
   * @param to The Link, index, or name at the end of the range.
   */
  itemByRange(from: Link | number | string, to: Link | number | string): Link[]

  /**
   * Returns the last Link in the collection.
   */
  lastItem(): Link

  /**
   * Returns the middle Link in the collection.
   */
  middleItem(): Link

  /**
   * Returns the Link whose index follows the specified Link in the collection.
   * @param obj The Link whose index comes before the desired Link.
   */
  nextItem(obj: Link): Link

  /**
   * Returns the Link with the index previous to the specified index.
   * @param obj The index of the Link that follows the desired Link.
   */
  previousItem(obj: Link): Link

  /**
   * Generates a string which, if executed, will return the Link.
   */
  toSource(): string
}

/**
 * Experimental: An http link connection manager.
 */
declare class HttpLinkConnectionManagerObject {
  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The index of the HttpLinkConnectionManagerObject within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * The parent of the HttpLinkConnectionManagerObject (a Application).
   */
  readonly parent: Application

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): HttpLinkConnectionManagerObject[]

  /**
   * Experimental: Create a url connection for the given server address
   * @param serverurl Server URL to connect
   * @param jsonData JSON data
   */
  httpConnect(serverurl: string, jsonData: string): void

  /**
   * Experimental: Checks if the given server URL is connected or not.
   * @param serverurl Server URL to check
   */
  isConnected(serverurl: string): boolean

  /**
   * Experimental: Logout from the given URL
   * @param serverurl Server URL to logout from
   */
  logout(serverurl: string): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the HttpLinkConnectionManagerObject.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * MetaData properties for the link source file.
 */
declare class LinkMetadata extends Preference {
  /**
   * The author of the document.
   */
  readonly author: string

  /**
   * The URL of the file that contains the linked copyright statement.
   */
  readonly copyrightInfoURL: string

  /**
   * The text to use as a copyright notice.
   */
  readonly copyrightNotice: string

  /**
   * The copyright status of the document.
   */
  readonly copyrightStatus: CopyrightStatus

  /**
   * The creation date of the document.
   */
  readonly creationDate: Date

  /**
   * The name of the application used to create the document.
   */
  readonly creator: string

  /**
   * The description of the LinkMetadata.
   */
  readonly description: string

  /**
   * The title of the document.
   */
  readonly documentTitle: string

  /**
   * The format of the document.
   */
  readonly format: string

  /**
   * The job name.
   */
  readonly jobName: string

  /**
   * The list of keywords associated with the document.
   */
  readonly keywords: string[]

  /**
   * The most recent modification date of the document.
   */
  readonly modificationDate: Date

  /**
   * The location of the document on the asset management server.
   */
  readonly serverURL: string

  /**
   * Counts the number of items in the container.
   * @param namespace The namespace of the container.
   * @param path The path to the container.
   */
  countContainer(namespace: string, path: string): number

  /**
   * Gets the XMP property value associated with the specified path.
   * @param namespace The namespace of the property.
   * @param path The specified path.
   */
  getProperty(namespace: string, path: string): string
}

/**
 * An object style.
 */
declare class ObjectStyle {
  /**
   * Anchored object settings.
   */
  anchoredObjectSettings: AnchoredObjectSetting

  /**
   * The paragraph style applied to the text.
   */
  appliedParagraphStyle: ParagraphStyle | string

  /**
   * If true, applies paragraph styles using Next Paragraph Style settings, beginning with the Next Paragraph Style defined in the paragraph style associated with the object style (if any).
   */
  applyNextParagraphStyle: boolean

  /**
   * The arrowhead alignment applied to the ObjectStyle.
   */
  arrowHeadAlignment: ArrowHeadAlignmentEnum

  /**
   * The style that this style is based on.
   */
  basedOn: ObjectStyle | string

  /**
   * Baseline frame grid option settings.
   */
  baselineFrameGridOptions: BaselineFrameGridOption

  /**
   * The shape to apply to the bottom left corner of rectangular shapes.
   */
  bottomLeftCornerOption: CornerOptions

  /**
   * The radius in measurement units of the corner effect applied to the bottom left corner of rectangular shapes
   */
  bottomLeftCornerRadius: string | number

  /**
   * The shape to apply to the bottom right corner of rectangular shapes.
   */
  bottomRightCornerOption: CornerOptions

  /**
   * The radius in measurement units of the corner effect applied to the bottom right corner of rectangular shapes
   */
  bottomRightCornerRadius: string | number

  /**
   * The content effects enabling settings.
   */
  readonly contentEffectsEnablingSettings: ObjectStyleContentEffectsCategorySettings

  /**
   * Transparency settings for the content of the ObjectStyle.
   */
  readonly contentTransparencySettings: ContentTransparencySetting

  /**
   * Emit CSS
   */
  emitCss: boolean

  /**
   * If true, the object style will apply an anchored object setting.
   */
  enableAnchoredObjectOptions: boolean

  /**
   * If true, the object style will apply an epub tag and class.
   */
  enableExportTagging: boolean

  /**
   * If true, the object style will apply a fill.
   */
  enableFill: boolean

  /**
   * If true, the object style will apply frame fitting options.
   */
  enableFrameFittingOptions: boolean

  /**
   * If true, the object style will apply alt text export options.
   */
  enableObjectExportAltTextOptions: boolean

  /**
   * If true, the object style will apply epub export options.
   */
  enableObjectExportEpubOptions: boolean

  /**
   * If true, the object style will apply tagged pdf export options.
   */
  enableObjectExportTaggedPdfOptions: boolean

  /**
   * If true, the object style will apply a paragraph style.
   */
  enableParagraphStyle: boolean

  /**
   * If true, the object style will apply story options.
   */
  enableStoryOptions: boolean

  /**
   * If true, the object style will apply a stroke.
   */
  enableStroke: boolean

  /**
   * If true, the object style will apply stroke options and corner options.
   */
  enableStrokeAndCornerOptions: boolean

  /**
   * If true, the object style will apply auto-sizing text frame options.
   */
  enableTextFrameAutoSizingOptions: boolean

  /**
   * If true, the object style will apply baseline text frame options.
   */
  enableTextFrameBaselineOptions: boolean

  /**
   * Enable the text frame footnote options category of object style
   */
  enableTextFrameFootnoteOptions: boolean

  /**
   * If true, the object style will apply general text frame options.
   */
  enableTextFrameGeneralOptions: boolean

  /**
   * If true, the object style will apply text wrap, contour, and non-printing settings.
   */
  enableTextWrapAndOthers: boolean

  /**
   * If true, enables the Dimension and Position attributes
   */
  enableTransformAttributes: boolean

  /**
   * The end shape of an open path.
   */
  endCap: EndCap

  /**
   * The corner join applied to the ObjectStyle.
   */
  endJoin: EndJoin

  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The swatch (color, gradient, tint, or mixed ink) applied to the fill of the ObjectStyle. .
   */
  fillColor: Swatch | string

  /**
   * The fill effects enabling settings.
   */
  readonly fillEffectsEnablingSettings: ObjectStyleFillEffectsCategorySettings

  /**
   * The percent of tint to use in the ObjectStyle's fill color. (To specify a tint percent, use a number in the range of 0 to 100; to use the inherited or overridden value, use -1.)
   */
  fillTint: number

  /**
   * Transparency settings for the fill applied to the ObjectStyle.
   */
  readonly fillTransparencySettings: FillTransparencySetting

  /**
   * The frame fitting option to apply to placed or pasted content. Can be applied to a frame, object style, or document or to the application.
   */
  frameFittingOptions: FrameFittingOption

  /**
   * The swatch (color, gradient, tint, or mixed ink) applied to the gap of a dashed, dotted, or striped stroke. For information, see stroke type.
   */
  gapColor: Swatch

  /**
   * The tint as a percentage of the gap color. (To specify a tint percent, use a number in the range of 0 to 100; to use the inherited or overridden value, use -1.)
   */
  gapTint: number

  /**
   * The angle of a linear gradient applied to the fill of the ObjectStyle. (Range: -180 to 180)
   */
  gradientFillAngle: number

  /**
   * The angle of a linear gradient applied to the stroke of the ObjectStyle. (Range: -180 to 180)
   */
  gradientStrokeAngle: number

  /**
   * The unique ID of the ObjectStyle.
   */
  readonly id: number

  /**
   * If true, class attribute will be generated for the style
   */
  includeClass: boolean

  /**
   * The index of the ObjectStyle within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * A property that can be set to any string.
   */
  label: string

  /**
   * The scaling applied to the arrowhead at the start of the path. (Range: 1 to 1000)
   */
  leftArrowHeadScale: number

  /**
   * The arrowhead applied to the start of the path.
   */
  leftLineEnd: ArrowHead

  /**
   * The limit of the ratio of stroke width to miter length before a miter (pointed) join becomes a bevel (squared-off) join.
   */
  miterLimit: number

  /**
   * The name of the ObjectStyle.
   */
  name: string

  /**
   * If true, the ObjectStyle does not print.
   */
  nonprinting: boolean

  /**
   * The object effects enabling settings.
   */
  readonly objectEffectsEnablingSettings: ObjectStyleObjectEffectsCategorySettings

  /**
   * Export options for the object
   */
  readonly objectExportOptions: ObjectExportOption

  /**
   * A collection of object style export tag maps.
   */
  readonly objectStyleExportTagMaps: ObjectStyleExportTagMaps

  /**
   * If true, the ObjectStyle's fill color overprints any underlying objects. If false, the fill color knocks out the underlying colors.
   */
  overprintFill: boolean

  /**
   * If true, the gap color overprints any underlying colors. If false, the gap color knocks out the underlying colors.
   */
  overprintGap: boolean

  /**
   * If true, the ObjectStyle's stroke color overprints any underlying objects. If false, the stroke color knocks out theunderlying colors.
   */
  overprintStroke: boolean

  /**
   * The parent of the ObjectStyle (a Document, Application or ObjectStyleGroup).
   */
  readonly parent: any

  /**
   * A collection of preferences objects.
   */
  readonly preferences: Preferences

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * The scaling applied to the arrowhead at the end of the path. (Range: 1 to 1000)
   */
  rightArrowHeadScale: number

  /**
   * The arrowhead applied to the end of the path.
   */
  rightLineEnd: ArrowHead

  /**
   * Story preference settings.
   */
  storyPreferences: StoryPreference

  /**
   * The stroke alignment applied to the ObjectStyle.
   */
  strokeAlignment: StrokeAlignment

  /**
   * The swatch (color, gradient, tint, or mixed ink) applied to the stroke of the ObjectStyle.
   */
  strokeColor: Swatch | string

  /**
   * The stroke effects enabling settings.
   */
  readonly strokeEffectsEnablingSettings: ObjectStyleStrokeEffectsCategorySettings

  /**
   * The percent of tint to use in object's stroke color. (To specify a tint percent, use a number in the range of 0 to 100; to use the inherited or overridden value, use -1.)
   */
  strokeTint: number

  /**
   * Transparency settings for the stroke.
   */
  readonly strokeTransparencySettings: StrokeTransparencySetting

  /**
   * The name of the stroke style to apply.
   */
  strokeType: StrokeStyle | string

  /**
   * The weight (in points) to apply to the ObjectStyle's stroke.
   */
  strokeWeight: string | number

  /**
   * Text frame footnote options settings.
   */
  textFrameFootnoteOptions: TextFrameFootnoteOptionsObject

  /**
   * Text frame preference settings.
   */
  textFramePreferences: TextFramePreference

  /**
   * The text wrap preference properties that define the default formatting for wrapping text around objects.
   */
  textWrapPreferences: TextWrapPreference

  /**
   * The shape to be applied to the top left corner of rectangular shapes and all corners of non-rectangular shapes.Note: corner option differs from end join in which you can set a radius for a corner option, whereas the rounded or beveled effect of an end join depends on the stroke weight.
   */
  topLeftCornerOption: CornerOptions

  /**
   * The radius in measurement units of the corner effect applied to the top left corner of rectangular shapes and all corners of non-rectangular shapes
   */
  topLeftCornerRadius: string | number

  /**
   * The shape to apply to the top right corner of rectangular shapes
   */
  topRightCornerOption: CornerOptions

  /**
   * The radius in measurement units of the corner effect applied to the top right corner of rectangular shapes
   */
  topRightCornerRadius: string | number

  /**
   * The layout attribute options to apply to any page item.
   */
  transformAttributeOptions: TransformAttributeOption

  /**
   * Transparency settings.
   */
  readonly transparencySettings: TransparencySetting

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Duplicates the ObjectStyle.
   */
  duplicate(): ObjectStyle

  /**
   * Gets the label value associated with the specified key.
   * @param key The key.
   */
  extractLabel(key: string): string

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): ObjectStyle[]

  /**
   * Sets the label to the value associated with the specified key.
   * @param key The key.
   * @param value The value.
   */
  insertLabel(key: string, value: string): void

  /**
   * Moves the ObjectStyle to the specified location.
   * @param to The new location relative to the reference object or within the container object.
   * @param reference The reference object. Note: Required when the to parameter is before or after.
   */
  move(
    to: LocationOptions,
    reference: ObjectStyle | ObjectStyleGroup | Document | Application,
  ): ObjectStyle

  /**
   * Deletes the style.
   * @param replacingWith The style to apply in place of the deleted style.
   */
  remove(replacingWith: ObjectStyle): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Sets the given dimension attribute state to enabled or disabled state
   * @param whichAttributes Which dimension attributes do you want to enable or disable.
   * @param attributeState Attribute state to be set, set true to enable and false to disable the attributes
   */
  setDimensionAttributeState(whichAttributes: DimensionAttributes, attributeState: boolean): boolean

  /**
   * Sets the given position attribute state to enabled or disabled state
   * @param whichAttributes Which position attributes do you want to enable or disable.
   * @param attributeState Attribute state to be set, set true to enable and false to disable the attributes
   */
  setPositionAttributeState(whichAttributes: PositionAttributes, attributeState: boolean): boolean

  /**
   * Generates a string which, if executed, will return the ObjectStyle.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of object styles.
 */
declare class ObjectStyles {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the ObjectStyle with the specified index.
   * @param index The index.
   */
  [index: number]: ObjectStyle

  /**
   * Creates a new ObjectStyle.
   * @param withProperties Initial values for properties of the new ObjectStyle
   */
  add(withProperties: object): ObjectStyle

  /**
   * Returns any ObjectStyle in the collection.
   */
  anyItem(): ObjectStyle

  /**
   * Displays the number of elements in the ObjectStyle.
   */
  count(): number

  /**
   * Returns every ObjectStyle in the collection.
   */
  everyItem(): ObjectStyle

  /**
   * Returns the first ObjectStyle in the collection.
   */
  firstItem(): ObjectStyle

  /**
   * Returns the ObjectStyle with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): ObjectStyle

  /**
   * Returns the ObjectStyle with the specified ID.
   * @param id The ID.
   */
  itemByID(id: number): ObjectStyle

  /**
   * Returns the ObjectStyle with the specified name.
   * @param name The name.
   */
  itemByName(name: string): ObjectStyle

  /**
   * Returns the ObjectStyles within the specified range.
   * @param from The ObjectStyle, index, or name at the beginning of the range.
   * @param to The ObjectStyle, index, or name at the end of the range.
   */
  itemByRange(from: ObjectStyle | number | string, to: ObjectStyle | number | string): ObjectStyle[]

  /**
   * Returns the last ObjectStyle in the collection.
   */
  lastItem(): ObjectStyle

  /**
   * Returns the middle ObjectStyle in the collection.
   */
  middleItem(): ObjectStyle

  /**
   * Returns the ObjectStyle whose index follows the specified ObjectStyle in the collection.
   * @param obj The ObjectStyle whose index comes before the desired ObjectStyle.
   */
  nextItem(obj: ObjectStyle): ObjectStyle

  /**
   * Returns the ObjectStyle with the index previous to the specified index.
   * @param obj The index of the ObjectStyle that follows the desired ObjectStyle.
   */
  previousItem(obj: ObjectStyle): ObjectStyle

  /**
   * Generates a string which, if executed, will return the ObjectStyle.
   */
  toSource(): string
}

/**
 * A mapping object that maps an object export type to an export tag.
 */
declare class ObjectStyleExportTagMap {
  /**
   * A collection of event listeners.
   */
  readonly eventListeners: EventListeners

  /**
   * A collection of events.
   */
  readonly events: Events

  /**
   * The attributes to map.
   */
  exportAttributes: string

  /**
   * The class to map.
   */
  exportClass: string

  /**
   * The tag to map.
   */
  exportTag: string

  /**
   * The type of export.
   */
  readonly exportType: string

  /**
   * The index of the ObjectStyleExportTagMap within its containing object.
   */
  readonly index: number

  /**
   * Returns true if the object specifier resolves to valid objects.
   */
  readonly isValid: boolean

  /**
   * The parent of the ObjectStyleExportTagMap (a ObjectStyle).
   */
  readonly parent: ObjectStyle

  /**
   * A property that allows setting of several properties at the same time.
   */
  properties: object

  /**
   * Adds an event listener.
   * @param eventType The event type.
   * @param handler The event handler.
   * @param captures This parameter is obsolete.
   */
  addEventListener(eventType: string, handler: File | Function, captures?: boolean): EventListener

  /**
   * Resolves the object specifier, creating an array of object references.
   */
  getElements(): ObjectStyleExportTagMap[]

  /**
   * Deletes the ObjectStyleExportTagMap.
   */
  remove(): void

  /**
   * Removes the event listener.
   * @param eventType The registered event type.
   * @param handler The registered event handler.
   * @param captures This parameter is obsolete.
   */
  removeEventListener(eventType: string, handler: File | Function, captures?: boolean): boolean

  /**
   * Generates a string which, if executed, will return the ObjectStyleExportTagMap.
   */
  toSource(): string

  /**
   * Retrieves the object specifier.
   */
  toSpecifier(): string
}

/**
 * A collection of object style export tag maps.
 */
declare class ObjectStyleExportTagMaps {
  /**
   * The number of objects in the collection.
   */
  readonly length: number

  /**
   * Returns the ObjectStyleExportTagMap with the specified index.
   * @param index The index.
   */
  [index: number]: ObjectStyleExportTagMap

  /**
   * Create a new mapping
   * @param exportType The type of export.
   * @param exportTag The tag to map.
   * @param exportClass The class to map.
   * @param exportAttributes The attributes to map.
   * @param withProperties Initial values for properties of the new ObjectStyleExportTagMap
   */
  add(
    exportType: string,
    exportTag: string,
    exportClass: string,
    exportAttributes: string,
    withProperties: object,
  ): ObjectStyleExportTagMap

  /**
   * Returns any ObjectStyleExportTagMap in the collection.
   */
  anyItem(): ObjectStyleExportTagMap

  /**
   * Displays the number of elements in the ObjectStyleExportTagMap.
   */
  count(): number

  /**
   * Returns every ObjectStyleExportTagMap in the collection.
   */
  everyItem(): ObjectStyleExportTagMap

  /**
   * Returns the first ObjectStyleExportTagMap in the collection.
   */
  firstItem(): ObjectStyleExportTagMap

  /**
   * Returns the ObjectStyleExportTagMap with the specified index or name.
   * @param index The index or name.
   */
  item(index: number | string): ObjectStyleExportTagMap

  /**
   * Returns the ObjectStyleExportTagMaps within the specified range.
   * @param from The ObjectStyleExportTagMap, index, or name at the beginning of the range.
   * @param to The ObjectStyleExportTagMap, index, or name at the end of the range.
   */
  itemByRange(
    from: ObjectStyleExportTagMap | number | string,
    to: ObjectStyleExportTagMap | number | string,
  ): ObjectStyleExportTagMap[]

  /**
   * Returns the last ObjectStyleExportTagMap in the collection.
   */
  lastItem(): ObjectStyleExportTagMap

  /**
   * Returns the middle ObjectStyleExportTagMap in the collection.
   */
  middleItem(): ObjectStyleExportTagMap

  /**
   * Returns the ObjectStyleExportTagMap whose index follows the specified ObjectStyleExportTagMap in the collection.
   * @param obj The ObjectStyleExportTagMap whose index comes before the desired ObjectStyleExportTagMap.
   */
  nextItem(obj: ObjectStyleExportTagMap): ObjectStyleExportTagMap

  /**
   * Returns the ObjectStyleExportTagMap with the index previous to the specified index.
   * @param obj The index of the ObjectStyleExportTagMap that follows the desired ObjectStyleExportTagMap.
   */
  previousItem(obj: ObjectStyleExportTagMap): ObjectStyleExportTagMap

  /**
   * Generates a string which, if executed, will return the ObjectStyleExportTagMap.
   */
  toSource(): string
}

/**
 * Story preferences.
 */
declare class StoryPreference extends Preference {
  /**
   * If true, adjust the position of characters at the edges of the frame to provide a better appearance.
   */
  opticalMarginAlignment: boolean

  /**
   * The point size used as the basis for calculating optical margin alignment. (Range: 0.1 to 1296)
   */
  opticalMarginSize: string | number

  /**
   * The direction of the story.
   */
  storyDirection: StoryDirectionOptions
}

/**
 * Text frame preferences.
 */
declare class TextFramePreference extends Preference {
  /**
   * The reference point for auto sizing of text frame. Reference point is automatically adjusted to the suitable value depending on the auto-sizing type value. As an example, top left reference point becomes top center for height only dimension
   */
  autoSizingReferencePoint: AutoSizingReferenceEnum

  /**
   * Auto-sizing type of text frame. Based on type, reference value is automatically adjusted. For example, for height only type, top-left reference point becomes top-center. Recommended to change auto-sizing type, after setting other auto-sizing attributes
   */
  autoSizingType: AutoSizingTypeEnum

  /**
   * The distance between the baseline of the text and the top inset of the text frame or cell.
   */
  firstBaselineOffset: FirstBaseline

  /**
   * If true, ignores text wrap settings for drawn or placed objects in the text frame.
   */
  ignoreWrap: boolean

  /**
   * The amount to offset text from the edges of the text frame, specified either as a single value applied uniformly to all sides of the text frame or as an array of 4 values in the format [top inset, left inset, bottom inset, right inset].
   */
  insetSpacing: number | [number, number, number, number]

  /**
   * The minimum distance between the baseline of the text and the top inset of the text frame or cell.
   */
  minimumFirstBaselineOffset: string | number

  /**
   * The minimum height for auto-sizing of the text frame.
   */
  minimumHeightForAutoSizing: string | number

  /**
   * The minimum width for auto-sizing of the text frame.
   */
  minimumWidthForAutoSizing: string | number

  /**
   * The number of columns in the text frame. Note: Depending on the value of use fixed column width, the number of columns can change automatically when the text frame size changes.
   */
  textColumnCount: number

  /**
   * The column width of the columns in the text frame.
   */
  textColumnFixedWidth: string | number

  /**
   * The space between columns in the text frame.
   */
  textColumnGutter: string | number

  /**
   * The maximum column width of the columns in the text frame. Use 0 to indicate no upper limit.
   */
  textColumnMaxWidth: string | number

  /**
   * If true, maintains column width when the text frame is resized. If false, causes columns to resize when the text frame is resized. Note: When true, resizing the frame can change the number of columns in the frame.
   */
  useFixedColumnWidth: boolean

  /**
   * If true, maintains column width between a min and max range when the text frame is resized. If false, causes columns to resize when the text frame is resized. Note: When true, resizing the frame can change the number of columns in the frame.
   */
  useFlexibleColumnWidth: boolean

  /**
   * If true, minimum height value is used during the auto-sizing of text frame.
   */
  useMinimumHeightForAutoSizing: boolean

  /**
   * If true, minimum width value is used during the auto-sizing of text frame.
   */
  useMinimumWidthForAutoSizing: boolean

  /**
   * If true, line-breaks are not introduced after auto sizing.
   */
  useNoLineBreaksForAutoSizing: boolean

  /**
   * Vertically justify balanced across all columns.
   */
  verticalBalanceColumns: boolean

  /**
   * The vertical alignment of the text content.
   */
  verticalJustification: VerticalJustification

  /**
   * The maximum amount of vertical space between two paragraphs. Note: Valid only when vertical justification is justified; the specified amount is applied in addition to the space before or space after values defined for the paragraph.
   */
  verticalThreshold: string | number
}

/**
 * Text preferences.
 */
declare class TextPreference extends Preference {
  /**
   * If true, moves wrapped text to the next available leading increment below the text wrap objects (skip by leading).
   */
  abutTextToTextWrap: boolean

  /**
   * Specifies where to insert new pages in response to overset text.
   */
  addPages: AddPageOptions

  /**
   * The amount that the baseline shift increases each time the user presses the option/alt-shift-up arrow keys or decreases each time the user presses the option/alt-shift-down arrow keys. (Range: .001 to 100)
   */
  baselineShiftKeyIncrement: string | number

  /**
   * Enable auto-deletion of pages containing empty threaded text frames.
   */
  deleteEmptyPages: boolean

  /**
   * If true, highlights character and paragraph styles with colored backgrounds.
   */
  enableStylePreviewMode: boolean

  /**
   * If true, highlights custom kerned or tracked characters.
   */
  highlightCustomSpacing: boolean

  /**
   * If true, highlights hyphenation and justification rule violations in the text.
   */
  highlightHjViolations: boolean

  /**
   * If true, highlights paragraphs that violate keep options.
   */
  highlightKeeps: boolean

  /**
   * If true, uses on-screen highlighting to identify kinsoku.
   */
  highlightKinsoku: boolean

  /**
   * If true, highlights missing fonts.
   */
  highlightSubstitutedFonts: boolean

  /**
   * If true, highlights substituted glyphs.
   */
  highlightSubstitutedGlyphs: boolean

  /**
   * If true, justifies text around text wrap objects.
   */
  justifyTextWraps: boolean

  /**
   * The amount the kerning value per 1000 ems increases each time the user presses of the option/alt-right arrow keys or decreases each time the user presses the option/alt-left arrow keys. (Range: 1 to 100)
   */
  kerningKeyIncrement: number

  /**
   * The amount that leading increases each time the user presses the option/alt-up arrow keys or decreases each time the user presses the option/alt-down arrow keys. (Range: .001 to 100)
   */
  leadingKeyIncrement: string | number

  /**
   * Restrict the adding of pages during smart text reflow to overridden master text frames.
   */
  limitToMasterTextFrames: boolean

  /**
   * If true, links placed text files and spreadsheet files. If false, embeds the files.
   */
  linkTextFilesWhenImporting: boolean

  /**
   * Preserve left-hand and right-and pages when facing pages are enabled during smart text reflow.
   */
  preserveFacingPageSpreads: boolean

  /**
   * If true, Japanese composer treats quotes as half width and rotates them in vertical.
   */
  quoteCharactersRotatedInVertical: boolean

  /**
   * If true, shows hidden characters.
   */
  showInvisibles: boolean

  /**
   * The size of text formatted as small caps, specified as a percentage of the font size. (Range: 1 to 200)
   */
  smallCap: number

  /**
   * If true, enable automatic adding and deleting of pages in response to text reflow.
   */
  smartTextReflow: boolean

  /**
   * The position of subscript characters, specified as a percentage of the regular leading. (Range: -500 to 500)
   */
  subscriptPosition: number

  /**
   * The size of subscript characters, specified as a percentage of the font size. (Range: 0 to 200)
   */
  subscriptSize: number

  /**
   * The position of superscript characters, specified as a percentage of the regular leading. (Range: -500 to 500)
   */
  superscriptPosition: number

  /**
   * The size of superscript characters, specified as a percentage of the font size. (Range: 0 to 200)
   */
  superscriptSize: number

  /**
   * If true, converts straight quotes to typographic quotes.
   */
  typographersQuotes: boolean

  /**
   * If true, uses the glyph CID to get the mojikumi class of the character.
   */
  useCidMojikumi: boolean

  /**
   * If true, reverses X and Y scaling on Roman characters in vertical text.
   */
  useNewVerticalScaling: boolean

  /**
   * If true, automatically selects the correct optical size.
   */
  useOpticalSize: boolean

  /**
   * If true, applies the leading changes made to a text range to the entire paragraph. If false, applies leading changes only to the text range.
   */
  useParagraphLeading: boolean

  /**
   * If true, text wrap does not affect text on layers above the layer that contains the text wrap object. If false, text wrap affects text on all visible layers.
   */
  zOrderTextWrap: boolean
}

/**
 * Text defaults.
 */
declare class TextDefault extends Preference {
  /**
   * If true, aligns the baseline of the text to the baseline grid.
   */
  alignToBaseline: boolean

  /**
   * If true, words unassociated with a hyphenation dictionary can break to the next line on any character.
   */
  allowArbitraryHyphenation: boolean

  /**
   * The character style applied to the text.
   */
  appliedCharacterStyle: CharacterStyle | string

  /**
   * The font applied to the TextDefault, specified as either a font object or the name of font family.
   */
  appliedFont: Font | string

  /**
   * The language of the text.
   */
  appliedLanguage: LanguageWithVendors | Language | string

  /**
   * The named grid in use.
   */
  appliedNamedGrid: NamedGrid

  /**
   * The list to be part of.
   */
  appliedNumberingList: NumberingList | string

  /**
   * The paragraph style applied to the text.
   */
  appliedParagraphStyle: ParagraphStyle | string

  /**
   * The percent of the type size to use for auto leading. (Range: 0 to 500).
   */
  autoLeading: number

  /**
   * The number of half-width characters at or below which the characters automatically run horizontally in vertical text.
   */
  autoTcy: number

  /**
   * If true, auto tcy includes Roman characters.
   */
  autoTcyIncludeRoman: boolean

  /**
   * If true or set to an enumeration value, balances ragged lines. Note: Not valid with a single-line text composer.
   */
  balanceRaggedLines: boolean | BalanceLinesStyle

  /**
   * The baseline shift applied to the text.
   */
  baselineShift: string | number

  /**
   * Bullet character.
   */
  readonly bulletChar: Bullet

  /**
   * The alignment of the bullet character.
   */
  bulletsAlignment: ListAlignment

  /**
   * List type for bullets and numbering.
   */
  bulletsAndNumberingListType: ListType

  /**
   * The character style to be used for the text after string.
   */
  bulletsCharacterStyle: CharacterStyle | string

  /**
   * The text after string expression for bullets.
   */
  bulletsTextAfter: string

  /**
   * If true, adds the double period (..), ellipse (...), and double hyphen (--) to the selected kinsoku set. Note: Valid only when a kinsoku set is in effect.
   */
  bunriKinshi: boolean

  /**
   * The capitalization scheme.
   */
  capitalization: Capitalization

  /**
   * The alignment of small characters to the largest character in the line.
   */
  characterAlignment: CharacterAlignment

  /**
   * The direction of the character.
   */
  characterDirection: CharacterDirectionOptions

  /**
   * The rotation angle (in degrees) of individual characters. Note: The rotation is counterclockwise.
   */
  characterRotation: number

  /**
   * If true, uses grid tracking to track non-Roman characters in CJK grids.
   */
  cjkGridTracking: boolean

  /**
   * The text composer to use to compose the text.
   */
  composer: string

  /**
   * The desired width (as a percentage) of individual characters. (Range: 50 to 200)
   */
  desiredGlyphScaling: number

  /**
   * The desired letter spacing, specified as a percentge of the built-in space between letters in the font. (Range: -100 to 500)
   */
  desiredLetterSpacing: number

  /**
   * The desired word spacing, specified as a percentage of the font word space value. (Range: 0 to 1000)
   */
  desiredWordSpacing: number

  /**
   * Position of diacriticical characters.
   */
  diacriticPosition: DiacriticPositionOptions

  /**
   * The digits type.
   */
  digitsType: DigitsTypeOptions

  /**
   * The number of characters to drop cap.
   */
  dropCapCharacters: number

  /**
   * The number of lines to drop cap.
   */
  dropCapLines: number

  /**
   * The character style to apply to the drop cap.
   */
  dropCapStyle: CharacterStyle | string

  /**
   * Details about the drop cap based on the glyph outlines. 1 = left side bearing. 2 = descenders. 0x100,0x200,0x400 are used for Japanese frame grid.
   */
  dropcapDetail: number

  /**
   * The stroke join type applied to the characters of the text.
   */
  endJoin: OutlineJoin

  /**
   * The swatch (color, gradient, tint, or mixed ink) applied to the fill of the TextDefault. .
   */
  fillColor: Swatch | string

  /**
   * The tint (as a percentage) of the fill color of the TextDefault. (To specify a tint percentage, use a number in the range of 0 to 100; to use the inherited or overridden value, use -1.)
   */
  fillTint: number

  /**
   * The amount to indent the first line.
   */
  firstLineIndent: string | number

  /**
   * The name of the font style.
   */
  fontStyle: string

  /**
   * The glyph variant to substitute for standard glyphs.
   */
  glyphForm: AlternateGlyphForms

  /**
   * The angle of a linear gradient applied to the fill of the text. (Range: -180 to 180)
   */
  gradientFillAngle: number

  /**
   * The length (for a linear gradient) or radius (for a radial gradient) applied to the fill of the text.
   */
  gradientFillLength: number

  /**
   * The starting point (in page coordinates) of a gradient applied to the fill of the text, in the format [x, y].
   */
  gradientFillStart: string[] | number[]

  /**
   * The angle of a linear gradient applied to the stroke of the text. (Range: -180 to 180)
   */
  gradientStrokeAngle: number

  /**
   * The length (for a linear gradient) or radius (for a radial gradient) applied to the stroke of the text.
   */
  gradientStrokeLength: number

  /**
   * The starting point (in page coordinates) of a gradient applied to the stroke of the text, in the format [x, y].
   */
  gradientStrokeStart: string[] | number[]

  /**
   * If true, aligns only the first line to the frame grid or baseline grid. If false, aligns all lines to the grid.
   */
  gridAlignFirstLineOnly: boolean

  /**
   * The alignment to the frame grid or baseline grid.
   */
  gridAlignment: GridAlignment

  /**
   * The manual gyoudori setting.
   */
  gridGyoudori: number

  /**
   * The horizontal scaling applied to the TextDefault.
   */
  horizontalScale: number

  /**
   * The relative desirability of better spacing vs. fewer hyphens. A lower value results in greater use of hyphens. (Range: 0 to 100)
   */
  hyphenWeight: number

  /**
   * If true, allows the last word in a text column to be hyphenated.
   */
  hyphenateAcrossColumns: boolean

  /**
   * The mininum number of letters at the beginning of a word that can be broken by a hyphen.
   */
  hyphenateAfterFirst: number

  /**
   * The minimum number of letters at the end of a word that can be broken by a hyphen.
   */
  hyphenateBeforeLast: number

  /**
   * If true, allows hyphenation of capitalized words.
   */
  hyphenateCapitalizedWords: boolean

  /**
   * The maximum number of hyphens that can appear on consecutive lines. To specify unlimited consecutive lines, use zero.
   */
  hyphenateLadderLimit: number

  /**
   * If true, allows hyphenation in the last word in a paragraph. Note: Valid only when hyphenation is true.
   */
  hyphenateLastWord: boolean

  /**
   * The minimum number of letters a word must have in order to qualify for hyphenation.
   */
  hyphenateWordsLongerThan: number

  /**
   * If true, allows hyphenation.
   */
  hyphenation: boolean

  /**
   * The amount of white space allowed at the end of a line of non-justified text before hypenation begins. Note: Valid when composer is single-line composer.
   */
  hyphenationZone: string | number

  /**
   * If true, ignores optical edge alignment for the paragraph.
   */
  ignoreEdgeAlignment: boolean

  /**
   * The number of grid squares in which to arrange the text.
   */
  jidori: number

  /**
   * The paragraph alignment.
   */
  justification: Justification

  /**
   * Use of Kashidas for justification
   */
  kashidas: KashidasOptions

  /**
   * If true, keeps all lines of the paragraph together. If false, allows paragraphs to break across pages or columns.
   */
  keepAllLinesTogether: boolean

  /**
   * The minimum number of lines to keep together in a paragraph before allowing a page break.
   */
  keepFirstLines: number

  /**
   * The minimum number of lines to keep together in a paragraph after a page break.
   */
  keepLastLines: number

  /**
   * If true, keeps a specified number of lines together when the paragraph breaks across columns or text frames.
   */
  keepLinesTogether: boolean

  /**
   * If true, forces the rule above the paragraph to remain in the frame bounds. Note: Valid only when rule above is true.
   */
  keepRuleAboveInFrame: boolean

  /**
   * The minimum number of lines to keep with the next paragraph.
   */
  keepWithNext: number

  /**
   * If the first line in the paragraph should be kept with the last line of previous paragraph.
   */
  keepWithPrevious: boolean

  /**
   * The alignment of kenten characters relative to the parent characters.
   */
  kentenAlignment: KentenAlignment

  /**
   * The character set used for the custom kenten character. Note: Valid only when kenten kind is custom.
   */
  kentenCharacterSet: KentenCharacterSet

  /**
   * The character used for kenten. Note: Valid only when kenten kind is custom.
   */
  kentenCustomCharacter: string

  /**
   * The swatch (color, gradient, tint, or mixed ink) applied to the fill of kenten characters.
   */
  kentenFillColor: Swatch | string

  /**
   * The font to use for kenten characters.
   */
  kentenFont: Font | string

  /**
   * The size (in points) of kenten characters.
   */
  kentenFontSize: number

  /**
   * The font style of kenten characters.
   */
  kentenFontStyle: string | NothingEnum

  /**
   * The style of kenten characters.
   */
  kentenKind: KentenCharacter

  /**
   * The method of overprinting the kenten fill.
   */
  kentenOverprintFill: AdornmentOverprint

  /**
   * The method of overprinting the kenten stroke.
   */
  kentenOverprintStroke: AdornmentOverprint

  /**
   * The distance between kenten characters and their parent characters.
   */
  kentenPlacement: number

  /**
   * The kenten position relative to the parent character.
   */
  kentenPosition: RubyKentenPosition

  /**
   * The swatch (color, gradient, tint, or mixed ink) applied to the stroke of kenten characters.
   */
  kentenStrokeColor: Swatch | string

  /**
   * The stroke tint (as a percentage) of kenten characters. (Range: 0 to 100)
   */
  kentenStrokeTint: number

  /**
   * The fill tint (as a percentage) of kenten characters. (Range: 0 to 100)
   */
  kentenTint: number

  /**
   * The stroke weight (in points) of kenten characters.
   */
  kentenWeight: number

  /**
   * The horizontal size of kenten characters as a percent of the original size.
   */
  kentenXScale: number

  /**
   * The vertical size of kenten charachers as a percent of the original size.
   */
  kentenYScale: number

  /**
   * The type of pair kerning.
   */
  kerningMethod: string

  /**
   * The keyboard direction of the character.
   */
  keyboardDirection: CharacterDirectionOptions

  /**
   * The type of hanging punctuation to allow. Note: Valid only when a kinsoku set is in effect.
   */
  kinsokuHangType: KinsokuHangTypes

  /**
   * The kinsoku set that determines legitimate line breaks.
   */
  kinsokuSet: KinsokuTable | KinsokuSet | string

  /**
   * The type of kinsoku processing for preventing kinsoku characters from beginning or ending a line. Note: Valid only when a kinsoku set is defined.
   */
  kinsokuType: KinsokuType

  /**
   * The amount to indent the last line in the paragraph.
   */
  lastLineIndent: string | number

  /**
   * The leading applied to the text.
   */
  leading: number | Leading

  /**
   * The amount of space before each character.
   */
  leadingAki: number

  /**
   * The point from which leading is measured from line to line.
   */
  leadingModel: LeadingModel

  /**
   * The width of the left indent.
   */
  leftIndent: string | number

  /**
   * If true, replaces specific character combinations (e.g., fl, fi) with ligature characters.
   */
  ligatures: boolean

  /**
   * The maximum width (as a percentage) of individual characters. (Range: 50 to 200)
   */
  maximumGlyphScaling: number

  /**
   * The maximum letter spacing, specified as a percentge of the built-in space between letters in the font. (Range: -100 to 500) Note: Valid only when text is justified.
   */
  maximumLetterSpacing: number

  /**
   * The maximum word spacing, specified as a percentage of the font word space value. Note: Valid only when text is justified. (Range: 0 to 1000)
   */
  maximumWordSpacing: number

  /**
   * The minimum width (as a percentage) of individual characters. (Range: 50 to 200)
   */
  minimumGlyphScaling: number

  /**
   * The minimum letter spacing, specified as a percentge of the built-in space between letters in the font. (Range: -100 to 500) Note: Valid only when text is justified.
   */
  minimumLetterSpacing: number

  /**
   * The minimum word spacing, specified as a percentage of the font word space value. Note: Valid only when text is justified. (Range: 0 to 1000)
   */
  minimumWordSpacing: number

  /**
   * The limit of the ratio of stroke width to miter length before a miter (pointed) join becomes a bevel (squared-off) join.
   */
  miterLimit: number

  /**
   * The mojikumi table. For information, see mojikumi table defaults.
   */
  mojikumi: MojikumiTable | string | MojikumiTableDefaults

  /**
   * A collection of nested GREP styles.
   */
  readonly nestedGrepStyles: NestedGrepStyles

  /**
   * A collection of nested line styles.
   */
  readonly nestedLineStyles: NestedLineStyles

  /**
   * A collection of nested styles.
   */
  readonly nestedStyles: NestedStyles

  /**
   * If true, keeps the text on the same line.
   */
  noBreak: boolean

  /**
   * The alignment of the number.
   */
  numberingAlignment: ListAlignment

  /**
   * If true, apply the numbering restart policy.
   */
  numberingApplyRestartPolicy: boolean

  /**
   * The character style to be used for the number string.
   */
  numberingCharacterStyle: CharacterStyle | string

  /**
   * Continue the numbering at this level.
   */
  numberingContinue: boolean

  /**
   * The number string expression for numbering.
   */
  numberingExpression: string

  /**
   * Numbering format options.
   */
  numberingFormat: NumberingStyle | string

  /**
   * The level of the paragraph.
   */
  numberingLevel: number

  /**
   * Numbering restart policies.
   */
  readonly numberingRestartPolicies: NumberingRestartPolicy

  /**
   * Determines starting number in a numbered list.
   */
  numberingStartAt: number

  /**
   * If true, uses contextual alternate forms in OpenType fonts.
   */
  otfContextualAlternate: boolean

  /**
   * If true, uses discretionary ligatures in OpenType fonts.
   */
  otfDiscretionaryLigature: boolean

  /**
   * The figure style in OpenType fonts.
   */
  otfFigureStyle: OTFFigureStyle

  /**
   * If true, uses fractions in OpenType fonts.
   */
  otfFraction: boolean

  /**
   * If true, switches hiragana fonts, which have different glyphs for horizontal and vertical.
   */
  otfHVKana: boolean

  /**
   * If true, use historical forms in OpenType fonts.
   */
  otfHistorical: boolean

  /**
   * If true, use alternate justification forms in OpenType fonts
   */
  otfJustificationAlternate: boolean

  /**
   * If true, uses localized forms in OpenType fonts.
   */
  otfLocale: boolean

  /**
   * If true, uses mark positioning in OpenType fonts.
   */
  otfMark: boolean

  /**
   * If true, uses ordinals in OpenType fonts.
   */
  otfOrdinal: boolean

  /**
   * If true, use overlapping swash forms in OpenType fonts
   */
  otfOverlapSwash: boolean

  /**
   * If true, kerns according to proportional CJK metrics in OpenType fonts.
   */
  otfProportionalMetrics: boolean

  /**
   * If true, applies italics to half-width alphanumerics.
   */
  otfRomanItalics: boolean

  /**
   * If true, use a slashed zeroes in OpenType fonts.
   */
  otfSlashedZero: boolean

  /**
   * If true, use stretched alternate forms in OpenType fonts
   */
  otfStretchedAlternate: boolean

  /**
   * If true, use stylistic alternate forms in OpenType fonts
   */
  otfStylisticAlternate: boolean

  /**
   * The stylistic sets to use in OpenType fonts.
   */
  otfStylisticSets: number

  /**
   * If true, uses swash forms in OpenType fonts.
   */
  otfSwash: boolean

  /**
   * If true, uses titling forms in OpenType fonts.
   */
  otfTitling: boolean

  /**
   * If true, the fill color of the characters will overprint.
   */
  overprintFill: boolean

  /**
   * If true, the stroke of the characters will overprint.
   */
  overprintStroke: boolean

  /**
   * The shape to apply to the bottom left corner of rectangular shapes.
   */
  paragraphBorderBottomLeftCornerOption: CornerOptions

  /**
   * The radius in measurement units of the corner effect applied to the bottom left corner of rectangular shapes
   */
  paragraphBorderBottomLeftCornerRadius: string | number

  /**
   * The bottom line weight of the border of paragraph.
   */
  paragraphBorderBottomLineWeight: string | number

  /**
   * The distance to offset the bottom edge of the paragraph border.
   */
  paragraphBorderBottomOffset: string | number

  /**
   * The basis (descent or baseline) used to calculate the bottom origin of the paragraph border.
   */
  paragraphBorderBottomOrigin: ParagraphBorderBottomOriginEnum

  /**
   * The shape to apply to the bottom right corner of rectangular shapes.
   */
  paragraphBorderBottomRightCornerOption: CornerOptions

  /**
   * The radius in measurement units of the corner effect applied to the bottom right corner of rectangular shapes
   */
  paragraphBorderBottomRightCornerRadius: string | number

  /**
   * The swatch (color, gradient, tint, or mixed ink) applied to the paragraph stroke.
   */
  paragraphBorderColor: Swatch | string

  /**
   * If true, then paragraph border is also displayed at the points where the paragraph splits across frames or columns.
   */
  paragraphBorderDisplayIfSplits: boolean

  /**
   * The swatch (color, gradient, tint, or mixed ink) applied to the paragraph border gap. Note: Valid only when the border type is not solid.
   */
  paragraphBorderGapColor: Swatch | string

  /**
   * If true, the paragraph border gap will overprint. Note: Valid only when border type is not solid.
   */
  paragraphBorderGapOverprint: boolean

  /**
   * The tint (as a percentage) of the paragraph border gap. Note: Valid only when the border type is not solid. (Range: 0 to 100)
   */
  paragraphBorderGapTint: numb