#include "gamerule.h"
#include "serverplayer.h"
#include "room.h"
#include "standard.h"
#include "engine.h"
#include "settings.h"

#include <QTime>

GameRule::GameRule()
    :TriggerSkill("game_rule")
{
    events << GameStart << TurnStart << PhaseChange << CardUsed << CardFinished
            << CardEffected << HpRecover << HpLost << AskForPeachesDone
            << AskForPeaches << PreDeath << Death << Dying << GameOverJudge << RewardAndPunish
            << SlashHit << SlashMissed << SlashEffected << SlashProceed
            << Interrupt << DamageDone << DamageComplete
            << StartJudge << FinishJudge << Pindian;
}

bool GameRule::triggerable(const ServerPlayer *) const{
    return true;
}

int GameRule::getPriority() const{
    return 0;
}

void GameRule::beforeNext(ServerPlayer *player) const{
    Room *room = player->getRoom();
    if(player->hasFlag("drank")){
        LogMessage log;
        log.type = "#UnsetDrankEndOfTurn";
        log.from = player;
        room->sendLog(log);

        room->setPlayerFlag(player, "-drank");
    }

    player->clearFlags();
    if(player->hasMark("rende"))
        room->setPlayerMark(player, "rende", 0);
    if(player->hasMark("fl_jizhi"))
        room->setPlayerMark(player, "fl_jizhi", 0);
}

void GameRule::onPhaseChange(ServerPlayer *player) const{
    Room *room = player->getRoom();
    switch(player->getPhase()){
    case Player::RoundStart:{
            break;
        }
    case Player::Start: {
            player->setMark("SlashCount", 0);

            if(room->getMode() == "05_2v3")
                if(player->isLord() || player->getRole() == "loyalist"){
                    int card_id = room->drawCard();
                    room->showCard(player, card_id);
                    room->getThread()->delay();
                    player->addToPile("Angers", card_id);
                    LogMessage log;
                    log.type = "$Juqi";
                    log.from = player;
                    log.card_str = QString::number(card_id);
                    room->sendLog(log);
                    if(player->getPile("Angers").length() > 4){
                        const QList<int> angs = player->getPile("Angers");
                        room->fillAG(angs, player);
                        int card_id = room->askForAG(player, angs, false, "cbangercollect");
                        if(card_id < 0)
                            card_id = player->getPile("Angers").first();
                        room->throwCard(card_id);
                        player->invoke("clearAG");
                    }
                }

            break;
        }
    case Player::Judge: {
            QList<const DelayedTrick *> tricks = player->delayedTricks();
            while(!tricks.isEmpty() && player->isAlive()){
                const DelayedTrick *trick = tricks.takeLast();
                bool on_effect = room->cardEffect(trick, NULL, player);
                if(!on_effect)
                    trick->onNullified(player);
            }

            break;
        }
    case Player::Draw: {
            int nUm = 2;
            if(room->getTag("FirstRound").toBool()){
                room->setTag("FirstRound", false);
                if(room->getMode() == "02_1v1")
                    nUm --;
            }

            QVariant num = nUm;
            room->getThread()->trigger(DrawNCards, player, num);
            int n = num.toInt();
            if(n > 0)
                player->drawCards(n, false);
            room->getThread()->trigger(DrawNCardsDone, player, num);
            break;
        }

    case Player::Play: {
            player->clearHistory();

            while(player->isAlive()){
                CardUseStruct card_use;
                room->activate(player, card_use);
                if(card_use.isValid()){
                    room->useCard(card_use);
                }else
                    break;
            }
            break;
        }

    case Player::Discard:{
            int discard_num = player->getHandcardNum() - player->getMaxCards();
            if(player->hasFlag("jilei")){
                QSet<const Card *> jilei_cards;
                QList<const Card *> handcards = player->getHandcards();
                foreach(const Card *card, handcards){
                    if(player->isJilei(card))
                        jilei_cards << card;
                }

                if(jilei_cards.size() > player->getMaxCards()){
                    // show all his cards
                    room->showAllCards(player, true);

                    DummyCard *dummy_card = new DummyCard;
                    foreach(const Card *card, handcards.toSet() - jilei_cards){
                        dummy_card->addSubcard(card);
                    }
                    room->throwCard(dummy_card, player);

                    return;
                }
            }

            if(discard_num > 0)
                room->askForDiscard(player, "gamerule", discard_num);
            break;
        }
    case Player::Finish: {
            break;
        }

    case Player::NotActive:{
            beforeNext(player);
            return;
        }
    }
}

void GameRule::setGameProcess(Room *room) const{
    int good = 0, bad = 0;
    QList<ServerPlayer *> players = room->getAlivePlayers();
    foreach(ServerPlayer *player, players){
        switch(player->getRoleEnum()){
        case Player::Lord:
        case Player::Loyalist: good ++; break;
        case Player::Rebel: bad++; break;
        case Player::Careerist:
        case Player::Renegade: break;
        }
    }

    QString process;
    if(good == bad)
        process = "Balance";
    else if(good > bad)
        process = "LordSuperior";
    else
        process = "RebelSuperior";

    room->setTag("GameProcess", process);
}

bool GameRule::trigger(TriggerEvent event,Room *room, ServerPlayer *player, QVariant &data) const{
    if(room->getTag("SkipGameRule").toBool()){
        room->removeTag("SkipGameRule");
        return false;
    }

    switch(event){
    case GameStart: {
            if(player->getGeneral()->getKingdom() == "god" && player->getGeneralName() != "anjiang"){
                QString new_kingdom = room->askForKingdom(player);
                room->setPlayerProperty(player, "kingdom", new_kingdom);

                LogMessage log;
                log.type = "#ChooseKingdom";
                log.from = player;
                log.arg = new_kingdom;
                room->sendLog(log);
            }
            if(player->getMaxHp() < 1){
                room->setPlayerProperty(player, "maxhp", 1);
                room->setPlayerProperty(player, "hp", 1);
            }

            if(player->isLord())
                setGameProcess(room);

            room->setTag("FirstRound", true);
            player->drawCards(4, false);
             if(Config.value("Cheat/GamblingCards", false).toBool()){
			    int i=0;
			    for(i=0; i<998; i++){
                if ((player->getState() != "online") || (!player->askForSkillInvoke("gambling"))) {
				return false;
				}
                    player->throwAllHandCards();
                    player->drawCards(4, false);
                }
            }
            if(setjmp(danshou_env))
                player = room->getCurrent()->getNextAlive();
            break;
        }

    case TurnStart:{
            player = room->getCurrent();
            if(!player->faceUp())
                player->turnOver();
            else if(player->isAlive())
                player->play();

            break;
        }

    case PhaseChange: onPhaseChange(player); break;
    case CardUsed: {
            if(data.canConvert<CardUseStruct>()){
                CardUseStruct card_use = data.value<CardUseStruct>();
                const Card *card = card_use.card;

                if(card_use.from->getArmor() && card_use.from->getArmor()->isKindOf("Fiveline")
                    && card->isNDTrick()){ // fivelines
                    if(card_use.from->getMark("fl_jizhi") < 998 && card_use.from->getHp() == 2
                            && room->askForSkillInvoke(card_use.from, "jizhi")){
                        room->playSkillEffect("jizhi");
                        card_use.from->drawCards(1);
                        card_use.from->addMark("fl_jizhi");
                    }
                }

                if(!card->getSkillName().isEmpty())
                    room->setEmotion(player, "skill/" + card->getSkillName());

                card_use.from->playCardEffect(card);
                card->use(room, card_use.from, card_use.to);
            }

            break;
        }

    case CardFinished: {
            CardUseStruct use = data.value<CardUseStruct>();
            bool hasyanxiao = use.card->hasFlag("yanxiao");
            use.card->clearFlags();
            if(hasyanxiao)
                room->setCardFlag(use.card, "yanxiao");

            break;
    }

    case HpRecover:{
            RecoverStruct recover_struct = data.value<RecoverStruct>();
            int recover = recover_struct.recover;

            room->setPlayerStatistics(player, "recover", recover);

            int new_hp = qMin(player->getHp() + recover, player->getMaxHP());
            room->setPlayerProperty(player, "hp", new_hp);
            room->broadcastInvoke("hpChange", QString("%1:%2").arg(player->objectName()).arg(recover));

            break;
        }

    case HpLost:{
            int lose = data.toInt();

            LogMessage log;
            log.type = "#LoseHp";
            log.from = player;
            log.arg = QString::number(lose);
            room->sendLog(log);

            room->setPlayerProperty(player, "hp", player->getHp() - lose);
            QString str = QString("%1:%2").arg(player->objectName()).arg(-lose);
            str.append("L");
            room->broadcastInvoke("hpChange", str);

            if(player->getHp() <= 0)
                room->enterDying(player, NULL);

            break;
    }

    case Dying:{
            if(player->getHp() > 0){
                player->setFlags("-dying");
                break;
            }

            DyingStruct dying = data.value<DyingStruct>();

            LogMessage log;
            log.type = "#AskForPeaches";
            log.from = player;
            log.to = dying.savers;
            log.arg = QString::number(1 - player->getHp());
            room->sendLog(log);

            RoomThread *thread = room->getThread();
            foreach(ServerPlayer *saver, dying.savers){
                if(player->getHp() > 0)
                    break;

                thread->trigger(AskForPeaches, saver, data);
            }

            player->setFlags("-dying");
            thread->trigger(AskForPeachesDone, player, data);

            break;
        }

    case AskForPeaches:{
            DyingStruct dying = data.value<DyingStruct>();

            while(dying.who->getHp() <= 0){
                if(dying.who->isDead())
                    break;
                const Card *peach = room->askForSinglePeach(player, dying.who);
                if(peach == NULL)
                    break;

                CardUseStruct use;
                use.card = peach;
                use.from = player;
                if(player != dying.who)
                    use.to << dying.who;

                room->useCard(use, false);

                if(player != dying.who && dying.who->getHp() > 0)
                    room->setPlayerStatistics(player, "save", 1);
            }

            break;
        }

    case AskForPeachesDone:{
            if(player->getHp() <= 0 && player->isAlive()){
                DyingStruct dying = data.value<DyingStruct>();
                room->killPlayer(player, dying.damage);
            }

            break;
        }

    case Interrupt:{
            //player->setFlags("ShutUp");
            player->skip();
            beforeNext(player);
            longjmp(danshou_env, 1);
            break;
        }
    case DamageDone:{
            DamageStruct damage = data.value<DamageStruct>();
            room->sendDamageLog(damage);

            if(damage.from)
                room->setPlayerStatistics(damage.from, "damage", damage.damage);

            room->applyDamage(player, damage);
            if(player->getHp() <= 0){
                room->enterDying(player, &damage);
            }

            break;
        }

    case DamageComplete:{
            if(room->getMode() == "02_1v1" && player->isDead()){
                QString new_general = player->tag["1v1ChangeGeneral"].toString();
                if(!new_general.isEmpty())
                    changeGeneral1v1(player);
            }

            bool chained = player->isChained();
            if(!chained)
                break;

            DamageStruct damage = data.value<DamageStruct>();
            if(damage.nature != DamageStruct::Normal){
                room->setPlayerProperty(player, "chained", false);

                // iron chain effect
                QList<ServerPlayer *> chained_players = room->getOtherPlayers(player);
                foreach(ServerPlayer *chained_player, chained_players){
                    if(chained_player->isChained()){
                        room->getThread()->delay();
                        room->setPlayerProperty(chained_player, "chained", false);

                        LogMessage log;
                        log.type = "#IronChainDamage";
                        log.from = chained_player;
                        room->sendLog(log);

                        DamageStruct chain_damage = damage;
                        chain_damage.to = chained_player;
                        chain_damage.chain = true;

                        room->damage(chain_damage);
                    }
                }
            }

            break;
        }

    case CardEffected:{
            if(data.canConvert<CardEffectStruct>()){
                CardEffectStruct effect = data.value<CardEffectStruct>();
                if(room->isCanceled(effect))
                    return true;

                effect.card->onEffect(effect);
            }

            break;
        }

    case SlashEffected:{
            SlashEffectStruct effect = data.value<SlashEffectStruct>();

            QVariant data = QVariant::fromValue(effect);
            room->getThread()->trigger(SlashProceed, effect.from, data);

            break;
        }

    case SlashProceed:{
        SlashEffectStruct effect = data.value<SlashEffectStruct>();

        QString slasher = effect.from->objectName();
        const Card *jink = room->askForCard(effect.to, "jink", "slash-jink:" + slasher, data, JinkUsed);
        room->slashResult(effect, jink);

        break;
    }

    case SlashHit:{
            SlashEffectStruct effect = data.value<SlashEffectStruct>();

            DamageStruct damage;
            damage.card = effect.slash;

            damage.damage = 1;
            if(effect.drank){
                LogMessage log;
                log.type = "#AnalepticBuff";
                log.from = effect.from;
                log.to << effect.to;
                log.arg = "analeptic";
                room->sendLog(log);

                damage.damage ++;
            }

            damage.from = effect.from;
            damage.to = effect.to;
            damage.nature = effect.nature;

            room->damage(damage);

            effect.to->removeMark("qinggang");

            break;
        }

    case SlashMissed:{
            SlashEffectStruct effect = data.value<SlashEffectStruct>();
            effect.to->removeMark("qinggang");

            break;
        }

    case GameOverJudge:{
            if(room->getMode() == "02_1v1"){
                QStringList list = player->tag["1v1Arrange"].toStringList();

                if(!list.isEmpty())
                    return false;
            }

            QString winner = getWinner(player);
            if(!winner.isNull()){
                room->gameOver(winner);
                return true;
            }
            break;
        }

    case PreDeath:{
        //DamageStar damage = data.value<DamageStar>();
        //ServerPlayer *killer = damage ? damage->from : NULL;

        if(Config.EnableEndless){
            if(player->getMaxHp() <= 0)
                room->setPlayerProperty(player, "maxhp", player->getGeneral()->getMaxHp());
            if(player->getHp() <= 0)
                room->setPlayerProperty(player, "hp", 1);
            return true;
        }

        if(player->getState() == "online"){
            if(Config.value("Cheat/FreeUnDead", false).toBool()){
                if(player->getMaxHp() <= 0)
                    room->setPlayerProperty(player, "maxhp", player->getGeneral()->getMaxHp());
                if(player->getHp() <= 0)
                    room->setPlayerProperty(player, "hp", 1);
                LogMessage log;
                log.type = "#Undead";
                log.from = player;
                room->sendLog(log);
                return true;
            }
            if(Config.value("Cheat/HandsUp", false).toBool() && room->getMode() != "02_1v1"){
                bool allrobot = true;
                QStringList winners;
                foreach(ServerPlayer *robot, room->getOtherPlayers(player)){
                    if(robot->getState() != "robot" && allrobot)
                        allrobot = false;
                    if(robot->isAlive())
                        winners << robot->objectName();
                }
                if(allrobot && player->askForSkillInvoke("goaway")){
                    room->gameOver(winners.join("+"));
                    return true;
                }
            }
        }
        break;
    }

    case Death:{
            player->bury();

            if(room->getTag("SkipNormalDeathProcess").toBool())
                return false;

            DamageStar damage = data.value<DamageStar>();
            ServerPlayer *killer = damage ? damage->from : NULL;
            if(killer)
            	room->getThread()->trigger(RewardAndPunish, player, data);

            setGameProcess(room);

            if(room->getMode() == "02_1v1"){
                QStringList list = player->tag["1v1Arrange"].toStringList();

                if(!list.isEmpty()){
                    player->tag["1v1ChangeGeneral"] = list.takeFirst();
                    player->tag["1v1Arrange"] = list;

                    DamageStar damage = data.value<DamageStar>();

                if(damage == NULL){
                    PlayerStar target = room->getTag("LihunTarget").value<PlayerStar>();
                    if(target && target == player)
                        room->removeTag("LihunTarget");
                    changeGeneral1v1(player);
                    return false;
                }
            }
        }
        break;
    }
    case RewardAndPunish:{
        if(Config.value("ReincaPersist", false).toBool() && player->property("isDead").toBool())
            return true;

        DamageStar damage = data.value<DamageStar>();
        PlayerStar killer = damage->from;
        PlayerStar victim = player;

        if(killer->isDead())
            return true;

        if(room->getMode() == "06_3v3")
            killer->drawCards(3);
        else{
            if(room->getMode() == "contract")
                return true;
            if(victim->getRole() == "rebel" && killer != victim){
                killer->drawCards(3);
            }else if(victim->getRole() == "loyalist" && killer->getRole() == "lord"){
                killer->throwAllEquips();
                killer->throwAllHandCards();
            }
        }
        break;
    }
    case StartJudge:{
            int card_id = room->drawCard();

            JudgeStar judge = data.value<JudgeStar>();
            judge->card = Sanguosha->getCard(card_id);
            room->moveCardTo(judge->card, NULL, Player::Special);

            LogMessage log;
            log.type = "$InitialJudge";
            log.from = player;
            log.card_str = judge->card->getEffectIdString();
            room->sendLog(log);

            room->sendJudgeResult(judge);

            int delay = Config.AIDelay;
            if(judge->time_consuming)
                delay /= 4;
            room->getThread()->delay(delay);

            break;
        }

    case FinishJudge:{
            JudgeStar judge = data.value<JudgeStar>();
            room->throwCard(judge->card);

            LogMessage log;
            log.type = "$JudgeResult";
            log.from = player;
            log.card_str = judge->card->getEffectIdString();
            room->sendLog(log);

            room->sendJudgeResult(judge);

            room->getThread()->delay();

            break;
        }

    case Pindian:{
            PindianStar pindian = data.value<PindianStar>();

            room->setEmotion(pindian->from, "pindian");
            room->setEmotion(pindian->to, "pindian");
            room->broadcastInvoke("playAudio", "pindian");

            room->moveCardTo(pindian->from_card, NULL, Player::DiscardedPile);
            LogMessage log;
            log.type = "$PindianResult";
            log.from = pindian->from;
            log.card_str = pindian->from_card->getEffectIdString();
            room->sendLog(log);
            room->getThread()->delay();

            room->moveCardTo(pindian->to_card, NULL, Player::DiscardedPile);
            log.type = "$PindianResult";
            log.from = pindian->to;
            log.card_str = pindian->to_card->getEffectIdString();
            room->sendLog(log);
            room->getThread()->delay();

            break;
        }

    default:
        ;
    }

    return false;
}

void GameRule::changeGeneral1v1(ServerPlayer *player) const{
    Room *room = player->getRoom();
    QString new_general = player->tag["1v1ChangeGeneral"].toString();
    player->tag.remove("1v1ChangeGeneral");
    room->transfigure(player, new_general, true, true);
    room->revivePlayer(player);

    if(player->getKingdom() != player->getGeneral()->getKingdom())
        room->setPlayerProperty(player, "kingdom", player->getGeneral()->getKingdom());

    room->broadcastInvoke("revealGeneral",
                          QString("%1:%2").arg(player->objectName()).arg(new_general),
                          player);

    if(!player->faceUp())
        player->turnOver();

    if(player->isChained())
        room->setPlayerProperty(player, "chained", false);

    player->drawCards(4);
}

QString GameRule::getWinner(ServerPlayer *victim) const{
    Room *room = victim->getRoom();
    QString winner;

    if(room->getMode() == "06_3v3"){
        switch(victim->getRoleEnum()){
        case Player::Lord: winner = "renegade+rebel"; break;
        case Player::Renegade: winner = "lord+loyalist"; break;
        default:
            break;
        }
    }else if(Config.EnableHegemony){
        bool has_anjiang = false, has_diff_kingdoms = false, has_careerist = false;
        QString init_kingdom;
        foreach(ServerPlayer *p, room->getAlivePlayers()){
            if(room->getTag(p->objectName()).toStringList().size()){
                has_anjiang = true;
            }

            if(p->getKingdom() == "careerist" && room->getAlivePlayers().length() > 1)
                has_careerist = true;

            if(init_kingdom.isEmpty()){
                init_kingdom = p->getKingdom();
            }
            else if(init_kingdom != p->getKingdom()){
                has_diff_kingdoms = true;
            }
        }

        if(!has_anjiang && !has_diff_kingdoms && !has_careerist){
            if(room->getAlivePlayers().length() == 1 && room->getAlivePlayers().first()->getKingdom() == "careerist"){
                winner = room->getAlivePlayers().first()->objectName();
            }
            else{
                QStringList winners;
                QString aliveKingdom = room->getAlivePlayers().first()->getKingdom();
                foreach(ServerPlayer *p, room->getPlayers()){
                    if(p->isAlive()) winners << p->objectName();
                    if(p->getKingdom() == aliveKingdom)
                    {
                        if(room->getTag("RecordVictimsOfHegemony").toStringList().contains(p->objectName()))
                            winners << p->objectName();
                    }
                }
                winner = winners.join("+");
            }
        }
    }else{
        QStringList alive_roles = room->aliveRoles(victim);
        switch(victim->getRoleEnum()){
        case Player::Lord:{
                if(alive_roles.length() == 1 && alive_roles.first() == "renegade")
                    winner = room->getAlivePlayers().first()->objectName();
                else
                    winner = "rebel";
                break;
            }

        case Player::Rebel:
        case Player::Renegade:
            {
                if(!alive_roles.contains("rebel") && !alive_roles.contains("renegade")){
                    winner = "lord+loyalist";
                    if(victim->getRole() == "renegade" && !alive_roles.contains("loyalist"))
                        room->setTag("RenegadeInFinalPK", true);
                }
                break;
            }

        default:
            break;
        }
    }

    return winner;
}

HulaoPassMode::HulaoPassMode()
{
    setObjectName("hulaopass_mode");

    events << HpChanged;
    default_choice = "recover";
}

static int Transfiguration = 1;

bool HulaoPassMode::trigger(TriggerEvent event,Room *room, ServerPlayer *player, QVariant &data) const{
    switch(event){
    case GameStart:{
            if(player->isLord()){
                if(setjmp(env) == Transfiguration){
                    player = room->getLord();

                    room->broadcastInvoke("animate", "lightbox:$Tshenlvbu:3000");
                    room->playSkillEffect("Tshenlvbu");
                    room->getThread()->delay(1500);
                    room->transfigure(player, "shenlvbu2", true, true);

                    QList<const Card *> tricks = player->getJudgingArea();
                    foreach(const Card *trick, tricks)
                        room->throwCard(trick);

                }else{
                    player->drawCards(8, false);
                }
            }else
                player->drawCards(player->getSeat() + 1, false);

            return false;
        }

    case CardUsed:{
            CardUseStruct use = data.value<CardUseStruct>();
            if(use.card->inherits("Weapon") && player->askForSkillInvoke("weapon_recast", data)){
                player->playCardEffect("@recast");
                room->throwCard(use.card);
                player->drawCards(1, false);
                return false;
            }

            break;
        }

    case HpChanged:{
            if(player->getGeneralName() == "shenlvbu1" && player->getHp() <= 4){
                longjmp(env, Transfiguration);
            }

            return false;
        }

    case Death:{
            if(player->isLord()){
                room->gameOver("rebel");
            }else{
                if(room->aliveRoles(player).length() == 1)
                    room->gameOver("lord");

                LogMessage log;
                log.type = "#Reforming";
                log.from = player;
                room->sendLog(log);

                player->bury();
                room->setPlayerProperty(player, "hp", 0);

                foreach(ServerPlayer *player, room->getOtherPlayers(room->getLord())){
                    if(player->askForSkillInvoke("draw_1v3"))
                        player->drawCards(1, false);
                }
            }

            return false;
        }

    case TurnStart:{
            if(player->isLord()){
                if(!player->faceUp())
                    player->turnOver();
                else
                    player->play();
            }else{
                if(player->isDead()){
                    if(player->getHp() + player->getHandcardNum() == 6){
                        LogMessage log;
                        log.type = "#ReformingRevive";
                        log.from = player;
                        room->sendLog(log);

                        room->revivePlayer(player);
                    }else if(player->isWounded()){
                        if(player->getHp() > 0 && (room->askForChoice(player, "Hulaopass", "recover+draw") == "draw")){
                            LogMessage log;
                            log.type = "#ReformingDraw";
                            log.from = player;
                            room->sendLog(log);
                            player->drawCards(1, false);
                            return false;
                        }

                        LogMessage log;
                        log.type = "#ReformingRecover";
                        log.from = player;
                        room->sendLog(log);

                        room->setPlayerProperty(player, "hp", player->getHp() + 1);
                    }else
                        player->drawCards(1, false);
                }else if(!player->faceUp())
                    player->turnOver();
                else
                    player->play();
            }

            return false;
        }

    default:
        break;
    }

    return GameRule::trigger(event, room, player, data);
}

BasaraMode::BasaraMode()
{
    setObjectName("basara_mode");

    events << CardLost << Predamaged;

    skill_mark["niepan"] = "@nirvana";
    skill_mark["smallyeyan"] = "@flame";
    skill_mark["luanwu"] = "@chaos";
    skill_mark["laoji"] = "@laoji";
    skill_mark["zuixiang"] = "@sleep";
}

QString BasaraMode::getMappedRole(const QString &role){
    static QMap<QString, QString> roles;
    if(roles.isEmpty()){
        roles["wei"] = "lord";
        roles["shu"] = "loyalist";
        roles["wu"] = "rebel";
        roles["qun"] = "renegade";
        roles["careerist"] = "careerist";
    }
    return roles[role];
}

int BasaraMode::getPriority() const{
    return 5;
}

void BasaraMode::playerShowed(ServerPlayer *player) const{
    Room *room = player->getRoom();
    QStringList names = room->getTag(player->objectName()).toStringList();
    if(names.isEmpty())
        return;

    if(Config.EnableHegemony){
        QMap<QString, int> kingdom_roles;
        foreach(ServerPlayer *p, room->getOtherPlayers(player)){
            kingdom_roles[p->getKingdom()]++;
        }
/*
        if(kingdom_roles[Sanguosha->getGeneral(names.first())->getKingdom()] >= 2
                && player->getGeneralName() == "anjiang")
            return;*/
    }

    QString answer = room->askForChoice(player, "RevealGeneral", "yes+no");
    if(answer == "yes"){
        QString general_name = room->askForGeneral(player,names);

        generalShowed(player,general_name);
        if(Config.EnableHegemony)
            room->getThread()->trigger(GameOverJudge, player);
        playerShowed(player);
    }
}

void BasaraMode::generalShowed(ServerPlayer *player, QString general_name) const
{
    Room * room = player->getRoom();
    QStringList names = room->getTag(player->objectName()).toStringList();
    if(names.isEmpty())return;

    bool is_careerist = false;
    if(Config.EnableHegemony){
        QMap<QString, int> kingdom_roles;
        foreach(ServerPlayer *p, room->getOtherPlayers(player)){
            kingdom_roles[p->getKingdom()]++;
        }

        if(kingdom_roles[Sanguosha->getGeneral(names.first())->getKingdom()] >= room->getAlivePlayers().length()/2
                && player->getGeneralName() == "anjiang")
            is_careerist = true;
    }

    if(player->getGeneralName() == "anjiang")
    {
        QString transfigure_str = QString("%1:%2").arg(player->getGeneralName()).arg(general_name);
        player->invoke("transfigure", transfigure_str);
        room->setPlayerProperty(player,"general",general_name);

        foreach(QString skill_name, skill_mark.keys()){
            if(player->hasSkill(skill_name))
                room->setPlayerMark(player, skill_mark[skill_name], 1);
        }
    }
    else{
        QString transfigure_str = QString("%1:%2").arg(player->getGeneral2Name()).arg(general_name);
        player->invoke("transfigure", transfigure_str);
        room->setPlayerProperty(player,"general2",general_name);
    }

    QString kingdom = is_careerist || player->getRole() == "careerist" ? "careerist" : player->getGeneral()->getKingdom();
    room->setPlayerProperty(player, "kingdom", kingdom);
    if(Config.EnableHegemony)
        room->setPlayerProperty(player, "role", getMappedRole(kingdom));

    names.removeOne(general_name);
    room->setTag(player->objectName(),QVariant::fromValue(names));

    LogMessage log;
    log.type = "#BasaraReveal";
    log.from = player;
    log.arg  = player->getGeneralName();
    log.arg2 = player->getGeneral2Name();

    room->sendLog(log);
    room->broadcastInvoke("playAudio","choose-item");
}

bool BasaraMode::trigger(TriggerEvent event,Room *room, ServerPlayer *player, QVariant &data) const{
    player->tag["event"] = event;
    player->tag["event_data"] = data;

    switch(event){
    case GameStart:{
        if(player->isLord()){
            if(Config.EnableHegemony)
                room->setTag("SkipNormalDeathProcess", true);

            foreach(ServerPlayer* sp, room->getAlivePlayers())
            {
                QString transfigure_str = QString("%1:%2").arg(sp->getGeneralName()).arg("anjiang");
                sp->invoke("transfigure", transfigure_str);
                room->setPlayerProperty(sp,"general","anjiang");
                room->setPlayerProperty(sp,"kingdom","god");

                LogMessage log;
                log.type = "#BasaraGeneralChosen";
                log.arg = room->getTag(sp->objectName()).toStringList().at(0);

                if(Config.Enable2ndGeneral)
                {

                    transfigure_str = QString("%1:%2").arg(sp->getGeneral2Name()).arg("anjiang");
                    sp->invoke("transfigure", transfigure_str);
                    room->setPlayerProperty(sp,"general2","anjiang");

                    log.arg2 = room->getTag(sp->objectName()).toStringList().at(1);
                }

                sp->invoke("log",log.toString());
                sp->tag["roles"] = room->getTag(sp->objectName()).toStringList().join("+");
            }
        }

        break;
    }
    case CardEffected:{
        if(player->getPhase() == Player::NotActive){
            CardEffectStruct ces = data.value<CardEffectStruct>();
            if(ces.card)
                if(ces.card->inherits("TrickCard") ||
                        ces.card->inherits("Slash"))
                playerShowed(player);

            const ProhibitSkill* prohibit = room->isProhibited(ces.from,ces.to,ces.card);
            if(prohibit)
            {
                LogMessage log;
                log.type = "#SkillAvoid";
                log.from = ces.to;
                log.arg  = prohibit->objectName();
                log.arg2 = ces.card->objectName();

                room->sendLog(log);

                return true;
            }
        }
        break;
    }

    case PhaseChange:{
        if(player->getPhase() == Player::Start)
            playerShowed(player);

        break;
    }
    case Predamaged:{
        playerShowed(player);
        break;
    }
    case GameOverJudge:{
        if(Config.EnableHegemony){
            if(player->getGeneralName() == "anjiang"){
                QStringList generals = room->getTag(player->objectName()).toStringList();
                room->setPlayerProperty(player, "general", generals.at(0));
                if(Config.Enable2ndGeneral)room->setPlayerProperty(player, "general2", generals.at(1));
                room->setPlayerProperty(player, "kingdom", player->getGeneral()->getKingdom());
                room->setPlayerProperty(player, "role", getMappedRole(player->getKingdom()));
            }
        }
        break;
    }

    case Death:{
        if(Config.EnableHegemony){
            DamageStar damage = data.value<DamageStar>();
            ServerPlayer *killer = damage ? damage->from : NULL;
            if(killer && killer->getKingdom() == damage->to->getKingdom()){
                killer->throwAllEquips();
                killer->throwAllHandCards();
            }
            else if(killer && killer->isAlive()){
                killer->drawCards(3);
            }
        }

        break;
    }

    default:
        break;
    }

    return false;
}

ChangbanSlopeMode::ChangbanSlopeMode()
{
    setObjectName("changbanslope_mode");
    events << HpChanged ;
}

static int TransfigurationCB = 1;

bool ChangbanSlopeMode::trigger(TriggerEvent event, Room* room, ServerPlayer *player, QVariant &data) const{
    switch(event){
    case GameStart:{
            room->setTag("SkipNormalDeathProcess", true);
            if(player->isLord()){
                if(setjmp(env) == TransfigurationCB){
                    player = room->getLord();

                    room->broadcastInvoke("animate", "lightbox:$Tshenzhaoyun:3000");
                    room->playSkillEffect("Tshenzhaoyun");
                    room->transfigure(player, "cbzhaoyun2", true, true);
                    ServerPlayer *cbzhangfei = player;
                    foreach(ServerPlayer *p, room->getAlivePlayers()){
                        if(p->getRole() == "loyalist")
                            cbzhangfei = p;
                    }
                    if(cbzhangfei == player){
                        cbzhangfei = room->findPlayer("cbzhangfei2", true);
                        if(cbzhangfei == NULL)
                            cbzhangfei = room->findPlayer("cbzhangfei1", true);
                        room->revivePlayer(cbzhangfei);
                    }
                    room->transfigure(cbzhangfei, "cbzhangfei2", true, true);
                    QList<ServerPlayer *> cbgod;
                    cbgod << player << cbzhangfei;
                    foreach(ServerPlayer *p, cbgod){
                        QList<const Card *> tricks = p->getJudgingArea();
                        foreach(const Card *trick, tricks)
                            room->throwCard(trick);

                        if(!p->faceUp())
                            p->turnOver();

                        if(p->isChained())
                            room->setPlayerProperty(p, "chained", false);
                    }
                }else{
                    player->drawCards(4, false);
                }
            }else
                player->drawCards(4, false);


            return false;
        }

    case HpChanged:{
            if(player->getGeneralName() == "cbzhaoyun1" && player->getHp() <= 4){
                longjmp(env, TransfigurationCB);
            }

            if(player->getGeneralName() == "cbzhangfei1" && player->getHp() <= 5){
                room->broadcastInvoke("animate", "lightbox:$Tshenzhangfei:3000");
                room->playSkillEffect("Tshenzhangfei");
                room->transfigure(player, "cbzhangfei2", true, true);
            }
            return false;
        }

    case TurnStart:{
            if(player->isLord()){
                if(!player->faceUp())
                    player->turnOver();
                else
                    player->play();
            }else{
                if(player->isDead()){
                    QStringList generals = room->getTag(player->objectName()).toStringList();
                    if(!generals.isEmpty()){
                        changeGeneral(player);
                        player->play();
                    }
                }
                else if(!player->faceUp())
                    player->turnOver();
                else
                    player->play();
            }

            return false;
        }

    case Death:{
            player->bury();

            if(player->getRoleEnum() == Player::Rebel){
                QStringList generals = room->getTag(player->objectName()).toStringList();
                if(generals.isEmpty()){
                    QStringList alive_roles = room->aliveRoles(player);
                    if(!alive_roles.contains("rebel"))
                        room->gameOver("lord+loyalist");
                }
            }

            if(player->isDead()){
                DamageStar damage = data.value<DamageStar>();
                ServerPlayer *killer = damage ? damage->from : NULL;
                if(killer && killer->isAlive()){
                    if(player->getRole() == "rebel" && killer != player){
                        killer->drawCards(2);
                    }else if(player->getRole() == "loyalist" && killer->getRole() == "lord"){
                        killer->throwAllEquips();
                        killer->throwAllHandCards();
                    }
                }
            }

            if(player->isLord()){
                room->gameOver("rebel");
            }

            if(player->getGeneralName() == "cbzhangfei2"){
                ServerPlayer *cbzhaoyun = room->getLord();
                if(cbzhaoyun->getHp() < 4)
                    room->setPlayerProperty(cbzhaoyun, "hp", 4);
            }

            return false;
        }

    case GameOverJudge:{
            if(player->getRole() == "rebel"){
                QStringList list = room->getTag(player->objectName()).toStringList();

                if(!list.isEmpty())
                    return false;
            }

            break;
        }

    default:
        break;
    }

    return GameRule::trigger(event, room, player, data);
}

void ChangbanSlopeMode::changeGeneral(ServerPlayer *player) const{
    Room *room = player->getRoom();
    QStringList generals = room->getTag(player->objectName()).toStringList();
    QString new_general = generals.takeFirst();
    room->setTag(player->objectName(), QVariant(generals));
    room->transfigure(player, new_general, true, true);
    room->revivePlayer(player);

    if(player->getKingdom() != player->getGeneral()->getKingdom())
        room->setPlayerProperty(player, "kingdom", player->getGeneral()->getKingdom());

    room->broadcastInvoke("revealGeneral",
                          QString("%1:%2").arg(player->objectName()).arg(new_general),
                          player);

    if(!player->faceUp())
        player->turnOver();

    if(player->isChained())
        room->setPlayerProperty(player, "chained", false);

    player->drawCards(4);
}

ReincarnationRule::ReincarnationRule()
{
    setObjectName("reincarnation_rule");
}

int ReincarnationRule::getPriority() const{
    return -1;
}

bool ReincarnationRule::trigger(TriggerEvent event, Room* room, ServerPlayer *player, QVariant &) const{
    switch(event){
    case GameStart:{
        int count = Sanguosha->getPlayerCount(room->getMode());
        if(count > 3)
            room->attachSkillToPlayer(player, "sacrifice");
        break;
    }
    case PhaseChange:{
        if(player->getPhase() == Player::NotActive){
            int count = Sanguosha->getPlayerCount(room->getMode());
            if(count < 4)
                break;
            int max = count > 5 ? 4 : 3;
            ServerPlayer *next = player->getNext();
            while(next->isDead()){
                if(next->getHandcardNum() >= max){
                    LogMessage log;
                    log.type = "#ReincarnRevive";
                    log.from = next;
                    room->sendLog(log);

                    room->broadcastInvoke("playAudio", "mode/reincarnation");
                    room->revivePlayer(next);

                    if(!Config.value("ReincaPersist", false).toBool()){
                        QStringList names;
                        foreach(ServerPlayer *tmp, room->getAllPlayers()){
                            names << tmp->getGeneralName();
                            if(tmp->getGeneral2())
                                names << tmp->getGeneral2Name();
                        }
                        if(!names.isEmpty()){
                            QSet<QString> names_set = names.toSet();
                            QString newname = Sanguosha->getRandomGenerals(1, names_set).first();
                            room->transfigure(next, newname, false, true);
                        }
                    }
                    if(next->getMaxHp() == 0)
                        room->setPlayerProperty(next, "maxhp", 1);
                    room->setPlayerProperty(next, "hp", 1);

                    room->getThread()->delay(1500);
                    room->attachSkillToPlayer(next, "sacrifice");
                    room->setPlayerMark(next, "@skull", 1);
                    room->setPlayerProperty(next, "isDead", true);
                }
                next = next->getNext();
            }
        }
        break;
    }

    default:
        break;
    }

    return false;
}
