#include "clientstruct.h"
#include "engine.h"
#include "client.h"
#include "settings.h"

ServerInfoStruct ServerInfo;

#include <QFormLayout>
#include <QLabel>
#include <QListWidget>
#include <QCheckBox>

bool ServerInfoStruct::parse(const QString &str){
    QRegExp rx("(.*):(@?\\w+):(\\d+):([+\\w]*):([FSRCTEBHAM12]*)");
    if(!rx.exactMatch(str)){
        // older version, just take the player count
        int count = str.split(":").at(1).toInt();
        GameMode = QString("%1p").arg(count, 2, 10, QChar('0'));

        return false;
    }

    QStringList texts = rx.capturedTexts();
    isPlay = !texts.isEmpty();

    QString server_name = texts.at(1);
    Name = QString::fromUtf8(QByteArray::fromBase64(server_name.toLatin1()));

    GameMode = texts.at(2);
    OperationTimeout = texts.at(3).toInt();

    QStringList ban_packages = texts.at(4).split("+");
    QList<const Package *> packages = Sanguosha->findChildren<const Package *>();
    foreach(const Package *package, packages){
        if(package->inherits("Scenario"))
            continue;

        QString package_name = package->objectName();
        if(ban_packages.contains(package_name))
            package_name = "!" + package_name;

        Extensions << package_name;
    }

    QString flags = texts.at(5);

    FreeChoose = flags.contains("F");
    Enable2ndGeneral = flags.contains("S");
    EnableReincarnation = flags.contains("R");
    EnableScene = flags.contains("C");
    EnableSame = flags.contains("T");
    EnableEndless = flags.contains("E");
    EnableBasara= flags.contains("B");
    EnableHegemony = flags.contains("H");
    EnableAI = flags.contains("A");
    DisableChat = flags.contains("M");

    if(flags.contains("1"))
        MaxHPScheme = 1;
    else if(flags.contains("2"))
        MaxHPScheme = 2;
    else
        MaxHPScheme = 0;

    return true;
}

ServerInfoWidget::ServerInfoWidget(bool show_lack)
{
    name_label = new QLabel;
    address_label = new QLabel;
    port_label = new QLabel;
    game_mode_label = new QLabel;
    player_count_label = new QLabel;
    two_general_label = new QLabel;
    reincarnation_label = new QLabel;
    scene_label = new QLabel;
    same_label = new QLabel;
    endless_label = new QLabel;
    basara_label = new QLabel;
    hegemony_label = new QLabel;
    free_choose_label = new QLabel;
    enable_ai_label = new QLabel;
    time_limit_label = new QLabel;
    max_hp_label = new QLabel;

    list_widget = new QListWidget;
    list_widget->setViewMode(QListView::IconMode);
    list_widget->setMovement(QListView::Static);

    QFormLayout *layout = new QFormLayout;
    layout->addRow(tr("Server name"), name_label);
    layout->addRow(tr("Address"), address_label);
    layout->addRow(tr("Port"), port_label);
    layout->addRow(tr("Game mode"), game_mode_label);
    layout->addRow(tr("Player count"), player_count_label);
    layout->addRow(tr("2nd general mode"), two_general_label);
    if(ServerInfo.Enable2ndGeneral)
        layout->addRow(tr("Max HP scheme"), max_hp_label);
    if(ServerInfo.EnableReincarnation)
        layout->addRow(tr("Reincarnation Rule"), reincarnation_label);
    if(ServerInfo.EnableEndless)
        layout->addRow(tr("Endless Mode"), endless_label);
    if(ServerInfo.EnableBasara)
        layout->addRow(tr("Basara Mode"), basara_label);
    if(ServerInfo.EnableHegemony)
        layout->addRow(tr("Hegemony Mode"), hegemony_label);
    layout->addRow(tr("Free choose"), free_choose_label);
    if(ServerInfo.EnableScene)
        layout->addRow(tr("Scene Mode"), scene_label);
    if(ServerInfo.EnableSame)
        layout->addRow(tr("Same Mode"), same_label);
    layout->addRow(tr("Enable AI"), enable_ai_label);
    layout->addRow(tr("Operation time"), time_limit_label);
    layout->addRow(tr("Extension packages"), list_widget);

    if(show_lack){
        lack_label = new QLabel;
        layout->addRow(tr("Lack"), lack_label);
    }else
        lack_label = NULL;

    setLayout(layout);
}

void ServerInfoWidget::fill(const ServerInfoStruct &info, const QString &address){
    name_label->setText(info.Name);
    address_label->setText(address);
    game_mode_label->setText(Sanguosha->getModeName(info.GameMode));
    int player_count = Sanguosha->getPlayerCount(info.GameMode);
    player_count_label->setText(QString::number(player_count));
    port_label->setText(QString::number(Config.ServerPort));
    two_general_label->setText(info.Enable2ndGeneral ? tr("Enabled") : tr("Disabled"));
    reincarnation_label->setText(info.EnableReincarnation ? tr("Enabled") : tr("Disabled"));
    scene_label->setText(info.EnableScene ? tr("Enabled") : tr("Disabled"));
    same_label->setText(info.EnableSame ? tr("Enabled") : tr("Disabled"));
    endless_label->setText(info.EnableEndless ? tr("Enabled") : tr("Disabled"));
    basara_label->setText(info.EnableBasara ? tr("Enabled") : tr("Disabled"));
    hegemony_label->setText(info.EnableHegemony ? tr("Enabled") : tr("Disabled"));

    if(info.Enable2ndGeneral){
        switch(info.MaxHPScheme){
        case 0: max_hp_label->setText(tr("Sum - 3")); break;
        case 1: max_hp_label->setText(tr("Minimum")); break;
        case 2: max_hp_label->setText(tr("Average")); break;
        }
    }else{
        max_hp_label->setText(tr("2nd general is disabled"));
        max_hp_label->setEnabled(false);
    }

    free_choose_label->setText(info.FreeChoose ? tr("Enabled") : tr("Disabled"));
    enable_ai_label->setText(info.EnableAI ? tr("Enabled") : tr("Disabled"));

    if(info.OperationTimeout == 0)
        time_limit_label->setText(tr("No limit"));
    else
        time_limit_label->setText(tr("%1 seconds").arg(info.OperationTimeout));

    list_widget->clear();

    static QIcon enabled_icon("image/system/enabled.png");
    static QIcon disabled_icon("image/system/disabled.png");

    foreach(QString extension, info.Extensions){
        bool checked = ! extension.startsWith("!");
        if(!checked)
            extension.remove("!");

        QString package_name = Sanguosha->translate(extension);
        QCheckBox *checkbox = new QCheckBox(package_name);
        checkbox->setChecked(checked);

        new QListWidgetItem(checked ? enabled_icon : disabled_icon, package_name, list_widget);
    }
}

void ServerInfoWidget::updateLack(int count){
    if(lack_label){
        QString path = QString("image/system/number/%1.png").arg(count);
        lack_label->setPixmap(QPixmap(path));
    }
}

void ServerInfoWidget::clear(){
    name_label->clear();
    address_label->clear();
    port_label->clear();
    game_mode_label->clear();
    player_count_label->clear();
    two_general_label->clear();
    reincarnation_label->clear();
    scene_label->clear();
    same_label->clear();
    endless_label->clear();
    basara_label->clear();
    hegemony_label->clear();
    free_choose_label->clear();
    time_limit_label->clear();
    list_widget->clear();
}


bool CardMoveStructForClient::parse(const QString &str){
    static QMap<QString, Player::Place> place_map;
    if(place_map.isEmpty()){
        place_map["hand"] = Player::Hand;
        place_map["equip"] = Player::Equip;
        place_map["judging"] = Player::Judging;
        place_map["special"] = Player::Special;
        place_map["_"] = Player::DiscardedPile;
        place_map["="] = Player::DrawPile;
    }

    // example: 12:tenshi@equip->moligaloo@hand
    QRegExp pattern("(-?\\d+):(.+)@(.+)->(.+)@(.+)");
    if(!pattern.exactMatch(str)){
        return false;
    }

    QStringList words = pattern.capturedTexts();

    card_id = words.at(1).toInt();

    if(words.at(2) == "_")
        from = NULL;
    else
        from = ClientInstance->getPlayer(words.at(2));
    from_place = place_map.value(words.at(3), Player::DiscardedPile);

    if(words.at(4) == "_")
        to = NULL;
    else
        to = ClientInstance->getPlayer(words.at(4));
    to_place = place_map.value(words.at(5), Player::DiscardedPile);

    return true;
}
