/**
 * This file is part of Talkie -- text-to-speech browser extension button.
 * <https://joelpurra.com/projects/talkie/>
 *
 * Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>
 *
 * Talkie is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Talkie is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
 *
 * ---
 *
 * # About the package "talkie"
 *
 * - Name: talkie
 * - Generated: 2021-01-21T21:06:51+01:00
 * - Version: 6.0.0
 * - License: GPL-3.0
 * - Author: Joel Purra <code+github@joelpurra.com> (https://joelpurra.com/)
 * - Homepage: https://joelpurra.com/projects/talkie/
 *
 *
 * ## Detected dependencies for this file:
 *
 * - Count: 0
 */

(function (factory) {
    typeof define === 'function' && define.amd ? define(factory) :
    factory();
}((function () { 'use strict';

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    const promiseTry = (fn) => new Promise(
        (resolve, reject) => {
            try {
                const result = fn();

                resolve(result);
            } catch (error) {
                reject(error);
            }
        },
    );

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class WebExtensionEnvironmentManifestProvider {
        getSync() {
            // NOTE: synchronous call.
            // https://developer.mozilla.org/en-US/Add-ons/WebExtensions/API/runtime/getManifest
            const manifest = browser.runtime.getManifest();

            return manifest;
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    const manifestProvider = new WebExtensionEnvironmentManifestProvider();

    // TODO: configuration.
    const extensionShortName = "Talkie";

    // https://stackoverflow.com/questions/12830649/check-if-chrome-extension-installed-in-unpacked-mode
    // https://stackoverflow.com/a/20227975
    /* eslint-disable no-sync */
    const isDevMode = () => !("update_url" in manifestProvider.getSync());
    /* eslint-enable no-sync */

    // NOTE: 0, 1, ...
    const loggingLevels = [
        "TRAC",
        "DEBG",
        "INFO",
        "WARN",
        "ERRO",

        // NOTE: should "always" be logged, presumably for technical reasons.
        "ALWA",

        // NOTE: turns off logging output.
        "NONE",
    ];

    const parseLevelName = (nextLevelName) => {
        if (typeof nextLevelName !== "string") {
            throw new TypeError("nextLevelName");
        }

        const normalizedLevelName = nextLevelName.toUpperCase();

        const levelIndex = loggingLevels.indexOf(normalizedLevelName);

        if (typeof levelIndex === "number" && Math.floor(levelIndex) === Math.ceil(levelIndex) && levelIndex >= 0 && levelIndex < loggingLevels.length) {
            return levelIndex;
        }

        throw new TypeError("nextLevel");
    };

    const parseLevel = (nextLevel) => {
        if (typeof nextLevel === "number" && Math.floor(nextLevel) === Math.ceil(nextLevel) && nextLevel >= 0 && nextLevel < loggingLevels.length) {
            return nextLevel;
        }

        const levelIndex = parseLevelName(nextLevel);

        return levelIndex;
    };

    // NOTE: default logging level differs for developers using the unpacked extension, and "normal" usage.
    let currentLevelIndex = isDevMode() ? parseLevel("DEBG") : parseLevel("WARN");

    const generateLogger = (loggingLevelName, consoleFunctioName) => {
        const functionLevelIndex = parseLevel(loggingLevelName);

        const logger = (...args) => {
            if (functionLevelIndex < currentLevelIndex) {
                return;
            }

            const now = new Date().toISOString();

            let loggingArgs = [
                loggingLevels[functionLevelIndex],
                now,
                extensionShortName,
                ...args,
            ];

            /* eslint-disable no-console */
            console[consoleFunctioName](...loggingArgs);
            /* eslint-enable no-console */
        };

        return logger;
    };

    const logTrace = generateLogger("TRAC", "log");
    const logDebug = generateLogger("DEBG", "log");
    const logInfo = generateLogger("INFO", "info");
    const logWarn = generateLogger("WARN", "warn");
    const logError = generateLogger("ERRO", "error");
    const logAlways = generateLogger("ALWA", "log");

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    const getBackgroundPage = () => promiseTry(
        // https://developer.browser.com/extensions/runtime.html#method-getBackgroundPage
        () => browser.runtime.getBackgroundPage()
            .then((backgroundPage) => {
                if (backgroundPage) {
                    return backgroundPage;
                }

                return null;
            }),
    );

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class DualLogger {
        constructor(localScriptName) {
            this.localScriptName = localScriptName;

            this.dualLogTrace = this._generateLogger(logTrace, "logTrace");
            this.dualLogDebug = this._generateLogger(logDebug, "logDebug");
            this.dualLogInfo = this._generateLogger(logInfo, "logInfo");
            this.dualLogWarn = this._generateLogger(logWarn, "logWarn");
            this.dualLogError = this._generateLogger(logError, "logError");
        }

        _generateLogger(localLoggerFunctionName, backgroundLoggerFunctionName) {
            const logger = (...args) => Promise.all([
                localLoggerFunctionName(this.localScriptName, ...args),

                getBackgroundPage()
                    .then((background) => {
                        background[backgroundLoggerFunctionName](this.localScriptName, ...args);

                        return undefined;
                    })
                    .catch((error) => {
                        logError(this.localScriptName, "backgroundLoggerFunctionName", "Error logging to background page", "Swallowing error", error, "arguments", ...args);

                        return undefined;
                    }),
            ]);

            return logger;
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    const dualLogger = new DualLogger("shared-frontend.js");

    const eventToPromise = (eventHandler, event) => promiseTry(
        () => {
            dualLogger.dualLogDebug("Start", "eventToPromise", event && event.type, event);

            return Promise.resolve()
                .then(() => eventHandler(event))
                .then((result) => dualLogger.dualLogDebug("Done", "eventToPromise", event && event.type, event, result))
                .catch((error) => dualLogger.dualLogError("eventToPromise", event && event.type, event, error));
        },
    );

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class SuspensionListenerManager {
        constructor() {
            // NOTE: could be made configurable, in case there are multiple reasons to manage suspension.
            this.preventSuspensionPortName = "talkie-prevents-suspension";

            this._onConnectHandlerBound = null;
            this.connectionCounter = 0;
            this.activeSuspendConnections = 0;
            this.suspendConnectionPorts = [];

            this._preventSuspendPromise = null;
            this._preventSuspendPromiseResolve = null;

            this._onConnectHandlerBound = (...args) => this._onConnectHandler(...args);
        }

        _onConnectHandler(port) {
            // NOTE: not only suspension managers connect here.
            this.connectionCounter++;

            const currentConnectionId = this.connectionCounter;

            logDebug("Start", "_onConnectHandler", currentConnectionId, this.activeSuspendConnections, port);

            if (port.name !== this.preventSuspensionPortName) {
                // NOTE: ignore non-matching ports.
                logDebug("Non-matching port", "_onConnectHandler", currentConnectionId, this.activeSuspendConnections, port);

                return;
            }

            this.activeSuspendConnections++;

            logDebug("Matching port", "_onConnectHandler", currentConnectionId, this.activeSuspendConnections, port);

            const _onMessageHandler = (msg) => {
                logDebug("_onMessageHandler", "_onConnectHandler", currentConnectionId, this.activeSuspendConnections, msg);

                return this._preventSuspendPromise;
            };

            const _messageProducer = () => {
                this.suspendConnectionPorts[currentConnectionId].port.postMessage(`Ah, ha, ha, ha, stayin' alive, stayin' alive: ${currentConnectionId}, ${this.activeSuspendConnections}`);
            };

            // NOTE: this message producer is unneccessary.
            const preventSuspensionIntervalId = setInterval(_messageProducer, 1000);

            const _onDisconnectHandler = () => {
                logDebug("Start", "_onDisconnectHandler", currentConnectionId, this.activeSuspendConnections, port);

                clearInterval(preventSuspensionIntervalId);

                delete this.suspendConnectionPorts[currentConnectionId];

                this.activeSuspendConnections--;

                logDebug("Done", "_onDisconnectHandler", currentConnectionId, this.activeSuspendConnections, port);
            };

            // NOTE: the background connects once per suspension prevention.
            this.suspendConnectionPorts[currentConnectionId] = {
                currentConnectionId: currentConnectionId,
                port: port,
                preventSuspensionIntervalId: preventSuspensionIntervalId,
                _onDisconnectHandler: _onDisconnectHandler,
            };

            // NOTE: this disconnect listener is unneccessary.
            this.suspendConnectionPorts[currentConnectionId].port.onDisconnect.addListener(_onDisconnectHandler);

            // NOTE: this message listener is unneccessary.
            this.suspendConnectionPorts[currentConnectionId].port.onMessage.addListener(_onMessageHandler);

            logDebug("Done", "_onConnectHandler", currentConnectionId, this.activeSuspendConnections, port);
        };

        _cleanUpConnections() {
            return promiseTry(
                () => {
                    logDebug("Start", "_cleanUpConnections");

                    const portDisconnectPromises = this.suspendConnectionPorts
                        .map((suspendConnectionPort) => {
                            if (suspendConnectionPort) {
                                if (suspendConnectionPort.port && typeof suspendConnectionPort.port.disconnect === "function") {
                                    suspendConnectionPort.port.disconnect();
                                }

                                if (typeof suspendConnectionPort._onDisconnectHandler === "function") {
                                    suspendConnectionPort._onDisconnectHandler();
                                }
                            }

                            return undefined;
                        });

                    return Promise.all(portDisconnectPromises)
                        .then(() => {
                            logDebug("Done", "_cleanUpConnections");

                            return undefined;
                        });
                },
            );
        }

        _createOnSuspendPromise() {
            return promiseTry(
                () => {
                    logDebug("Start", "_createOnSuspendPromise");

                    this._preventSuspendPromise = Promise.resolve()
                        .then(() => {
                            logDebug("Start", "_preventSuspendPromise", "creating");

                            return undefined;
                        })
                        .then(() => new Promise(
                            (resolve) => {
                                // NOTE: should keep the channel alive until it disconnects.
                                // https://developer.mozilla.org/en-US/Add-ons/WebExtensions/API/runtime/onMessage
                                // https://developer.chrome.com/extensions/runtime
                                this._preventSuspendPromiseResolve = resolve;
                            }),
                        )
                        .then(() => {
                            logDebug("Done", "_preventSuspendPromise", "resolved");

                            return undefined;
                        });
                },
            );
        }

        _finishOnSuspendPromise() {
            return promiseTry(
                () => {
                    logDebug("Start", "_finishOnSuspendPromise");

                    return Promise.resolve()
                        .then(() => this._preventSuspendPromiseResolve())
                        .then(() => this._preventSuspendPromise)
                        .then(() => {
                            logDebug("Done", "_finishOnSuspendPromise");

                            return undefined;
                        });
                },
            );
        }

        start() {
            return promiseTry(
                () => {
                    logDebug("Start", "SuspensionListenerManager.start");

                    if (browser.runtime.onConnect.hasListener(this._onConnectHandlerBound)) {
                        throw new Error("SuspensionListenerManager: Already initialized.");
                    }

                    browser.runtime.onConnect.addListener(this._onConnectHandlerBound);

                    return this._createOnSuspendPromise()
                        .then(() => {
                            logDebug("Done", "SuspensionListenerManager.start");

                            return undefined;
                        });
                },
            );
        }

        stop() {
            return promiseTry(
                () => {
                    logDebug("Start", "SuspensionListenerManager.stop");

                    if (!browser.runtime.onConnect.hasListener(this._onConnectHandlerBound)) {
                        throw new Error("SuspensionListenerManager: Not initialized.");
                    }

                    browser.runtime.onConnect.removeListener(this._onConnectHandlerBound);
                    this._onConnectHandlerBound = null;

                    return this._finishOnSuspendPromise()
                        .then(() => {
                            if (this.activeSuspendConnections !== 0) {
                                logWarn(`this.activeSuspendConnections is not 0: ${this.activeSuspendConnections}. Attempting to disconnect.`);

                                return this._cleanUpConnections();
                            }

                            return undefined;
                        })
                        .then(() => {
                            logDebug("Done", "SuspensionListenerManager.stop");

                            return undefined;
                        });
                },
            );
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    const dualLogger$1 = new DualLogger("stay-alive.js");

    const suspensionListenerManager = new SuspensionListenerManager();

    const startStayAliveListener = () => {
        return suspensionListenerManager.start();
    };

    const stopStayAliveListener = () => {
        return suspensionListenerManager.stop()
            .catch((error) => {
                dualLogger$1.dualLogError("stopStayAliveListener", "Swallowing error", error);

                // NOTE: swallowing errors.
                return undefined;
            });
    };

    const start = () => promiseTry(
        () => startStayAliveListener()
            .then(() => undefined),
    );

    const stop = () => promiseTry(
        // NOTE: probably won't be correctly executed as before/unload doesn't guarantee asynchronous calls.
        () => stopStayAliveListener()
            .then(() => undefined),
    );

    document.addEventListener("DOMContentLoaded", eventToPromise.bind(null, start));
    window.addEventListener("beforeunload", eventToPromise.bind(null, stop));

})));
//# sourceMappingURL=stay-alive.js.map
