/**
 * This file is part of Talkie -- text-to-speech browser extension button.
 * <https://joelpurra.com/projects/talkie/>
 *
 * Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>
 *
 * Talkie is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Talkie is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
 *
 * ---
 *
 * # About the package "talkie"
 *
 * - Name: talkie
 * - Generated: 2021-01-21T21:06:46+01:00
 * - Version: 6.0.0
 * - License: GPL-3.0
 * - Author: Joel Purra <code+github@joelpurra.com> (https://joelpurra.com/)
 * - Homepage: https://joelpurra.com/projects/talkie/
 *
 *
 * ## Detected dependencies for this file:
 *
 * - Count: 9
 *
 *
 * ### About the dependency "styletron-core"
 *
 * - Name styletron-core
 * - Version 3.0.4
 * - License MIT
 * - Author: Ryan Tsao <ryan.j.tsao@gmail.com> ()
 * - Homepage: https://github.com/rtsao/styletron
 *
 *
 *
 * ### About the dependency "styletron-client"
 *
 * - Name styletron-client
 * - Version 3.0.4
 * - License MIT
 * - Author: Ryan Tsao <ryan.j.tsao@gmail.com> ()
 * - Homepage: https://github.com/rtsao/styletron
 *
 *
 *
 * ### About the dependency "rollup-plugin-node-globals"
 *
 * - Name rollup-plugin-node-globals
 * - Version 1.4.0
 * - License MIT
 * - Author: Calvin Metcalf <calvin.metcalf@gmail.com> ()
 * - Homepage:
 *
 *
 *
 * ### About the dependency "process-es6"
 *
 * - Name process-es6
 * - Version 0.11.6
 * - License MIT
 * - Author: Roman Shtylman <shtylman@gmail.com> ()
 * - Homepage:
 *
 *
 *
 * ### About the dependency "inline-style-prefixer"
 *
 * - Name inline-style-prefixer
 * - Version 4.0.2
 * - License MIT
 * - Author: Robin Frischmann <> ()
 * - Homepage:
 *
 *
 *
 * ### About the dependency "css-in-js-utils"
 *
 * - Name css-in-js-utils
 * - Version 2.0.1
 * - License MIT
 * - Author: Robin Frischmann <robin@rofrischmann.de> ()
 * - Homepage:
 *
 *
 *
 * ### About the dependency "hyphenate-style-name"
 *
 * - Name hyphenate-style-name
 * - Version 1.0.4
 * - License BSD-3-Clause
 * - Author: Espen Hovlandsdal <espen@hovlandsdal.com> ()
 * - Homepage: https://github.com/rexxars/hyphenate-style-name#readme
 *
 *
 *
 * ### About the dependency "styletron-utils"
 *
 * - Name styletron-utils
 * - Version 3.0.4
 * - License MIT
 * - Author: Ryan Tsao <ryan.j.tsao@gmail.com> ()
 * - Homepage: https://github.com/rtsao/styletron
 *
 *
 *
 * ### About the dependency "styletron-react"
 *
 * - Name styletron-react
 * - Version 3.0.4
 * - License MIT
 * - Author: Ryan Tsao <ryan.j.tsao@gmail.com> ()
 * - Homepage: https://github.com/rtsao/styletron
 */

(function (ReactDOM, React, redux, thunk, PropTypes, reactRedux, reselect) {
    'use strict';

    function _interopDefaultLegacy (e) { return e && typeof e === 'object' && 'default' in e ? e : { 'default': e }; }

    var ReactDOM__default = /*#__PURE__*/_interopDefaultLegacy(ReactDOM);
    var React__default = /*#__PURE__*/_interopDefaultLegacy(React);
    var thunk__default = /*#__PURE__*/_interopDefaultLegacy(thunk);
    var PropTypes__default = /*#__PURE__*/_interopDefaultLegacy(PropTypes);

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const promiseTry = fn => new Promise((resolve, reject) => {
      try {
        const result = fn();
        resolve(result);
      } catch (error) {
        reject(error);
      }
    });

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    class WebExtensionEnvironmentManifestProvider {
      getSync() {
        // NOTE: synchronous call.
        // https://developer.mozilla.org/en-US/Add-ons/WebExtensions/API/runtime/getManifest
        const manifest = browser.runtime.getManifest();
        return manifest;
      }

    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const manifestProvider = new WebExtensionEnvironmentManifestProvider(); // TODO: configuration.

    const extensionShortName = "Talkie"; // https://stackoverflow.com/questions/12830649/check-if-chrome-extension-installed-in-unpacked-mode
    // https://stackoverflow.com/a/20227975

    /* eslint-disable no-sync */

    const isDevMode = () => !("update_url" in manifestProvider.getSync());
    /* eslint-enable no-sync */
    // NOTE: 0, 1, ...


    const loggingLevels = ["TRAC", "DEBG", "INFO", "WARN", "ERRO", // NOTE: should "always" be logged, presumably for technical reasons.
    "ALWA", // NOTE: turns off logging output.
    "NONE"];

    const parseLevelName = nextLevelName => {
      if (typeof nextLevelName !== "string") {
        throw new TypeError("nextLevelName");
      }

      const normalizedLevelName = nextLevelName.toUpperCase();
      const levelIndex = loggingLevels.indexOf(normalizedLevelName);

      if (typeof levelIndex === "number" && Math.floor(levelIndex) === Math.ceil(levelIndex) && levelIndex >= 0 && levelIndex < loggingLevels.length) {
        return levelIndex;
      }

      throw new TypeError("nextLevel");
    };

    const parseLevel = nextLevel => {
      if (typeof nextLevel === "number" && Math.floor(nextLevel) === Math.ceil(nextLevel) && nextLevel >= 0 && nextLevel < loggingLevels.length) {
        return nextLevel;
      }

      const levelIndex = parseLevelName(nextLevel);
      return levelIndex;
    }; // NOTE: default logging level differs for developers using the unpacked extension, and "normal" usage.


    let currentLevelIndex = isDevMode() ? parseLevel("DEBG") : parseLevel("WARN");

    const generateLogger = (loggingLevelName, consoleFunctioName) => {
      const functionLevelIndex = parseLevel(loggingLevelName);

      const logger = (...args) => {
        if (functionLevelIndex < currentLevelIndex) {
          return;
        }

        const now = new Date().toISOString();
        let loggingArgs = [loggingLevels[functionLevelIndex], now, extensionShortName, ...args];
        /* eslint-disable no-console */


        console[consoleFunctioName](...loggingArgs);
        /* eslint-enable no-console */
      };

      return logger;
    };

    const logTrace = generateLogger("TRAC", "log");
    const logDebug = generateLogger("DEBG", "log");
    const logInfo = generateLogger("INFO", "info");
    const logWarn = generateLogger("WARN", "warn");
    const logError = generateLogger("ERRO", "error");
    const logAlways = generateLogger("ALWA", "log");

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    const handleUnhandledRejection = event => {
      logWarn("Unhandled rejection", "Error", event.reason, event.promise, event);
      logInfo("Starting debugger, if attached.");
      /* eslint-disable no-debugger */

      debugger;
      /* eslint-enable no-debugger */
    };

    const registerUnhandledRejectionHandler = () => {
      window.addEventListener("unhandledrejection", handleUnhandledRejection);
    };

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const getBackgroundPage = () => promiseTry( // https://developer.browser.com/extensions/runtime.html#method-getBackgroundPage
    () => browser.runtime.getBackgroundPage().then(backgroundPage => {
      if (backgroundPage) {
        return backgroundPage;
      }

      return null;
    }));

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    class DualLogger {
      constructor(localScriptName) {
        this.localScriptName = localScriptName;
        this.dualLogTrace = this._generateLogger(logTrace, "logTrace");
        this.dualLogDebug = this._generateLogger(logDebug, "logDebug");
        this.dualLogInfo = this._generateLogger(logInfo, "logInfo");
        this.dualLogWarn = this._generateLogger(logWarn, "logWarn");
        this.dualLogError = this._generateLogger(logError, "logError");
      }

      _generateLogger(localLoggerFunctionName, backgroundLoggerFunctionName) {
        const logger = (...args) => Promise.all([localLoggerFunctionName(this.localScriptName, ...args), getBackgroundPage().then(background => {
          background[backgroundLoggerFunctionName](this.localScriptName, ...args);
          return undefined;
        }).catch(error => {
          logError(this.localScriptName, "backgroundLoggerFunctionName", "Error logging to background page", "Swallowing error", error, "arguments", ...args);
          return undefined;
        })]);

        return logger;
      }

    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const dualLogger = new DualLogger("shared-frontend.js");

    const reflow = () => promiseTry(() => {
      document.body.style.marginBottom = "0";
    });

    const eventToPromise = (eventHandler, event) => promiseTry(() => {
      dualLogger.dualLogDebug("Start", "eventToPromise", event && event.type, event);
      return Promise.resolve().then(() => eventHandler(event)).then(result => dualLogger.dualLogDebug("Done", "eventToPromise", event && event.type, event, result)).catch(error => dualLogger.dualLogError("eventToPromise", event && event.type, event, error));
    });

    const focusFirstLink = () => promiseTry(() => {
      const links = document.getElementsByTagName("a");

      if (links.length > 0) {
        const firstLinkElement = links[0];
        firstLinkElement.focus();
      }
    });

    const startReactFrontend = () => promiseTry(() => Promise.all([focusFirstLink(), reflow()]));
    const stopReactFrontend = () => promiseTry(() => {// TODO: unregister listeners.
    });

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const dispatchAll = (store, actionsToDispatch) => promiseTry(() => {
      return Promise.all(actionsToDispatch.map(action => store.dispatch(action)));
    });

    /**
     * The core styletron module
     * @packagename styletron-core
     */
    class StyletronCore {
      /**
       * Create a new StyletronCore instance
       * @param {object} [opts]           An object containing options
       * @param {string} [opts.prefix=''] A prefix for generated CSS class names
       */
      constructor({
        prefix = ''
      } = {}) {
        this.cache = {
          media: {},
          pseudo: {}
        };
        this.prefix = prefix === '' ? false : prefix;
        this.uniqueCount = 0;
        this.offset = 10; // skip 0-9

        this.msb = 35;
        this.power = 1;
      }

      static assignDecl(target, decl, className) {
        const {
          block,
          media,
          pseudo
        } = decl;
        let targetEntry;

        if (media) {
          if (!target.media[media]) {
            target.media[media] = {
              pseudo: {}
            };
          }

          targetEntry = target.media[media];
        } else {
          targetEntry = target;
        }

        if (pseudo) {
          if (!targetEntry.pseudo[pseudo]) {
            targetEntry.pseudo[pseudo] = {};
          }

          targetEntry = targetEntry.pseudo[pseudo];
        }

        targetEntry[block] = className;
      }
      /**
       * Injects a declaration (if not already injected) and returns a class name
       * @param  {object} decl          The CSS declaration object
       * @param  {string} decl.prop     The property name
       * @param  {string} decl.val      The property value
       * @param  {string} [decl.media]  The media query
       * @param  {string} [decl.pseudo] The pseudo selector
       * @return {string|undefined}     The class name for the declaration
       */


      injectDeclaration({
        prop,
        val,
        media,
        pseudo
      }) {
        return this.injectRawDeclaration({
          block: `${prop}:${val}`,
          media,
          pseudo
        });
      }
      /**
       * Injects a raw declaration (if not already injected) and returns a class name
       * @param  {object} decl          The CSS declaration object
       * @param  {string} decl.block    The declaration block
       * @param  {string} [decl.media]  The media query
       * @param  {string} [decl.pseudo] The pseudo selector
       * @return {string|undefined}     The class name for the declaration
       */


      injectRawDeclaration(decl) {
        const cached = this.getCachedDeclaration(decl);

        if (cached) {
          return cached;
        }

        const virtualCount = this.incrementVirtualCount();
        const hash = virtualCount.toString(36);
        const className = this.prefix ? this.prefix + hash : hash;
        StyletronCore.assignDecl(this.cache, decl, className);
        return className;
      }
      /**
       * Get the next virtual class number, while setting
       * the uniqueCount, offset, and msb counters appropriately.
       * @return {number} The virtual class count
       * @private
       */


      incrementVirtualCount() {
        const virtualCount = this.uniqueCount + this.offset;

        if (virtualCount === this.msb) {
          this.offset += (this.msb + 1) * 9;
          this.msb = Math.pow(36, ++this.power) - 1;
        }

        this.uniqueCount++;
        return virtualCount;
      }
      /**
       * Gets the class name for an already injected declaration
       * @param  {object} decl          The CSS declaration object
       * @param  {string} decl.block    The declaration block
       * @param  {string} [decl.media]  The media query
       * @param  {string} [decl.pseudo] The pseudo selector
       * @return {string|undefined}     The class name for the declaration
       * @private
       */


      getCachedDeclaration({
        block,
        media,
        pseudo
      }) {
        let entry;

        if (media) {
          entry = this.cache.media[media];

          if (!entry) {
            return false;
          }
        } else {
          entry = this.cache;
        }

        if (pseudo) {
          entry = entry.pseudo[pseudo];

          if (!entry) {
            return false;
          }
        }

        return entry[block];
      }

    }

    /* eslint-env browser */
    const DECL_REGEX = /.([^:{]+)(:[^{]+)?{([^}]+)}/g;
    /**
     * A Styletron class for rendering styles in the browser
     * @extends StyletronCore
     * @packagename styletron-client
     * @example
     * const elements = document.getElementsByClassName('_styletron_hydrate_');
     * const styletron = new StyletronClient(elements);
     */

    class StyletronClient extends StyletronCore {
      /**
       * Create a new StyletronClient instance
       * @param {NodeList|HTMLCollection|HTMLStyleElement[]} [serverStyles] - List of server style elements
       * @param {object} [opts] - StyletronCore options
       */
      constructor(serverStyles, opts) {
        super(opts);
        this.uniqueCount = 0;
        this.mediaSheets = {};

        if (serverStyles && serverStyles.length > 0) {
          for (let i = 0; i < serverStyles.length; i++) {
            const element = serverStyles[i];

            if (element.media) {
              this.mediaSheets[element.media] = element;
            } else {
              this.mainSheet = element;
            }

            this.hydrateCacheFromCssString(element.textContent, element.media);
          }
        } else {
          const styleSheet = document.createElement('style');
          document.head.appendChild(styleSheet);
          this.mainSheet = styleSheet;
        }
      }
      /*
       * Hydrate the cache from a css string and media string
       * @param {string} css   - The stylesheet css content
       * @param {string} media - The stylesheet media string
       */


      hydrateCacheFromCssString(css, media) {
        let decl; // {
        //  1: className,
        //  2: pseudo,
        //  3: block,
        // }

        while (decl = DECL_REGEX.exec(css)) {
          super.incrementVirtualCount();
          StyletronCore.assignDecl(this.cache, {
            block: decl[3],
            pseudo: decl[2],
            media
          }, decl[1]);
        }
      }
      /**
       * Inject declaration into the stylesheet and return the unique class name
       * @return {string}      class name
       * @example
       * // <style id="styletron">.a{color:red}</style>
       * const styletron = new StyletronClient(document.getElementsByClassName('_styletron_hydrate_'));
       * styletron.injectDeclaration({prop: 'color', val: 'blue'});
       * // → 'b'
       * styletron.injectDeclaration({prop: 'color', val: 'red', media: '(min-width: 800px)'});
       * // → 'c'
       * styletron.injectDeclaration({prop: 'color', val: 'red'});
       * // → 'a'
       */


      injectDeclaration({
        prop,
        val,
        media,
        pseudo
      }) {
        return this.injectRawDeclaration({
          block: `${prop}:${val}`,
          media,
          pseudo
        });
      }
      /**
       * Inject raw declaration into the stylesheet and return the unique class name
       * @return {string}      class name
       * @example
       * // <style id="styletron">.a{color:red}</style>
       * const styletron = new StyletronClient(document.getElementsByClassName('_styletron_hydrate_'));
       * styletron.injectRawDeclaration({block: 'color:blue'});
       * // → 'b'
       * styletron.injectRawDeclaration({block: 'color:red', media: '(min-width: 800px)'});
       * // → 'c'
       * styletron.injectRawDeclaration({block: 'color:red'});
       * // → 'a'
       */


      injectRawDeclaration(decl) {
        const oldCount = this.uniqueCount;
        const className = super.injectRawDeclaration(decl);

        if (oldCount !== this.uniqueCount) {
          const rule = declarationToRule(className, decl);
          let sheet;

          if (decl.media) {
            if (!this.mediaSheets[decl.media]) {
              const mediaSheet = document.createElement('style');
              mediaSheet.media = decl.media;
              this.mediaSheets[decl.media] = mediaSheet;
              this.mainSheet.parentNode.appendChild(mediaSheet);
            }

            sheet = this.mediaSheets[decl.media].sheet;
          } else {
            sheet = this.mainSheet.sheet;
          }

          sheet.insertRule(rule, sheet.cssRules.length);
        }

        return className;
      }

    }

    /*
     * Injection helpers
     */

    function declarationToRule(className, {
      block,
      pseudo
    }) {
      let selector = `.${className}`;

      if (pseudo) {
        selector += pseudo;
      }

      return `${selector}{${block}}`;
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    class WebExtensionEnvironmentTranslator {
      getSync() {
        const styletron = new StyletronClient();
        return styletron;
      }

    }

    var shared = {
    	urls: {
    		cla: "https://joelpurra.com/projects/talkie/CLA.md",
    		github: "https://github.com/joelpurra/talkie",
    		gpl: "https://www.gnu.org/licenses/gpl.html",
    		main: "https://joelpurra.com/projects/talkie/",
    		project: "https://joelpurra.com/projects/talkie/",
    		"shortcut-keys": "https://joelpurra.com/projects/talkie/#shortcut-keys",
    		"support-feedback": "https://joelpurra.com/support/",
    		chromewebstore: "https://chrome.google.com/webstore/detail/enfbcfmmdpdminapkflljhbfeejjhjjk",
    		"firefox-amo": "https://addons.mozilla.org/en-US/firefox/addon/talkie/",
    		"primary-payment": "https://www.paypal.me/joelpurrade",
    		"alternative-payment": "https://joelpurra.com/donate/",
    		share: {
    			twitter: "https://twitter.com/intent/tweet?text=Using%20Talkie%20to%20read%20text%20for%20me%20-%20text%20to%20speech%20works%20great!&url=https://joelpurra.com/projects/talkie/&via=joelpurra&related=&hashtags=texttospeech,tts",
    			facebook: "https://www.facebook.com/sharer/sharer.php?u=https%3A//joelpurra.com/projects/talkie/",
    			googleplus: "https://plus.google.com/share?url=https%3A//joelpurra.com/projects/talkie/",
    			linkedin: "https://www.linkedin.com/shareArticle?url=https%3A//joelpurra.com/projects/talkie/"
    		}
    	}
    };
    var chrome = {
    	urls: {
    		rate: "https://chrome.google.com/webstore/detail/enfbcfmmdpdminapkflljhbfeejjhjjk/reviews",
    		"support-feedback": "https://chrome.google.com/webstore/detail/enfbcfmmdpdminapkflljhbfeejjhjjk/support"
    	}
    };
    var webextension = {
    	urls: {
    		rate: "https://addons.mozilla.org/en-US/firefox/addon/talkie/reviews/"
    	}
    };
    var configurationObject = {
    	shared: shared,
    	chrome: chrome,
    	webextension: webextension
    };

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    class WebExtensionEnvironmentLocaleProvider {
      getUILocale() {
        const locale = browser.i18n.getMessage("@@ui_locale");
        return locale;
      }

      getTranslationLocale() {
        const locale = browser.i18n.getMessage("extensionLocale");
        return locale;
      }

    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    class WebExtensionEnvironmentTranslatorProvider {
      constructor(localeProvider) {
        // TODO REMOVE: unused.
        this.localeProvider = localeProvider;
      }

      translate(key, extras) {
        // const locale = this.localeProvider.getTranslationLocale();
        // TODO: use same translation system in frontend and backend?
        const translated = browser.i18n.getMessage(key, extras);
        return translated;
      }

    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    class WebExtensionEnvironmentBroadcasterProvider {
      registerListeningAction(event, handler) {
        return getBackgroundPage().then(background => background.broadcaster().registerListeningAction(event, handler));
      }

    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    class WebExtensionEnvironmentStorageProvider {
      get(key) {
        return promiseTry(() => getBackgroundPage().then(background => {
          const valueJson = background.localStorage.getItem(key);

          if (valueJson === null) {
            return null;
          }

          const value = JSON.parse(valueJson);
          return value;
        }));
      }

      set(key, value) {
        return promiseTry(() => getBackgroundPage().then(background => {
          const valueJson = JSON.stringify(value);
          background.localStorage.setItem(key, valueJson);
          return undefined;
        }));
      }

    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    class ReduxStoreProvider {
      createStore(initialState, rootReducer, api) {
        const middlewares = redux.applyMiddleware(thunk__default['default'].withExtraArgument(api));
        const enhancer = redux.compose(middlewares);
        const store = redux.createStore(rootReducer, initialState, enhancer); // store.subscribe(() => {
        //     const state = store.getState();
        //
        //     console.log(
        //         "subscribe",
        //         "state",
        //         JSON.stringify(state).length,
        //         state
        //     );
        // });

        return store;
      }

    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    class StorageManager {
      constructor(storageProvider) {
        this.storageProvider = storageProvider;
        this.currentStorageFormatVersion = "v1.4.0";
        this.storageMetadataId = "_storage-metadata";
        this.allKnownStorageKeys = {};
        this.allKnownStorageKeys["v1.0.0"] = {
          "options-popup-donate-buttons-hide": "options-popup-donate-buttons-hide"
        };
        this.allKnownStorageKeys["v1.1.0"] = {
          "language-voice-overrides": "language-voice-overrides",
          "options-popup-donate-buttons-hide": "options-popup-donate-buttons-hide",
          "voice-pitch-overrides": "voice-pitch-overrides",
          "voice-rate-overrides": "voice-rate-overrides"
        };
        this.allKnownStorageKeys["v1.2.0"] = {
          "language-voice-overrides": "language-voice-overrides",
          "voice-pitch-overrides": "voice-pitch-overrides",
          "voice-rate-overrides": "voice-rate-overrides"
        };
        this.allKnownStorageKeys["v1.3.0"] = {
          "language-voice-overrides": "language-voice-overrides",
          "speak-long-texts": "speak-long-texts",
          "voice-pitch-overrides": "voice-pitch-overrides",
          "voice-rate-overrides": "voice-rate-overrides"
        };
        this.allKnownStorageKeys["v1.4.0"] = {
          "is-premium-edition": "is-premium-edition",
          "language-voice-overrides": "language-voice-overrides",
          "speak-long-texts": "speak-long-texts",
          "voice-pitch-overrides": "voice-pitch-overrides",
          "voice-rate-overrides": "voice-rate-overrides"
        }; // TODO: sort by semantic version.

        this.allKnownStorageFormatVersions = Object.keys(this.allKnownStorageKeys);
        this.allKnownStorageFormatVersions.sort();
        this.allKnownUpgradePaths = {};
        this.allKnownUpgradePaths["v1.0.0"] = {};
        this.allKnownUpgradePaths["v1.0.0"]["v1.1.0"] = {
          upgradeKey: this._createIdentityUpgrader("v1.0.0", "v1.1.0")
        };
        this.allKnownUpgradePaths["v1.0.0"]["v1.2.0"] = {
          upgradeKey: this._createIdentityUpgrader("v1.0.0", "v1.2.0")
        };
        this.allKnownUpgradePaths["v1.1.0"] = {};
        this.allKnownUpgradePaths["v1.1.0"]["v1.2.0"] = {
          upgradeKey: this._createIdentityUpgrader("v1.1.0", "v1.2.0")
        };
        this.allKnownUpgradePaths["v1.2.0"] = {};
        this.allKnownUpgradePaths["v1.2.0"]["v1.3.0"] = {
          upgradeKey: this._createIdentityUpgrader("v1.2.0", "v1.3.0")
        };
        this.allKnownUpgradePaths["v1.3.0"] = {};
        this.allKnownUpgradePaths["v1.3.0"]["v1.4.0"] = {
          upgradeKey: this._createIdentityUpgrader("v1.3.0", "v1.4.0")
        };
      }

      _getStorageKey(storageFormatVersion, key) {
        return promiseTry(() => {
          if (!this.allKnownStorageKeys[storageFormatVersion]) {
            throw new Error(`Unknown storage format version: (${storageFormatVersion})`);
          }

          if (key !== this.storageMetadataId && !this.allKnownStorageKeys[storageFormatVersion][key]) {
            throw new Error(`Unknown storage key (${storageFormatVersion}): ${key}`);
          }

          return `${storageFormatVersion}_${key}`;
        });
      }

      _isStorageKeyValid(storageFormatVersion, key) {
        return promiseTry(() => {
          return this._getStorageKey(storageFormatVersion, key).then(() => {
            return true;
          }).catch(() => {
            // TODO: check for the specific storageKey errors.
            return false;
          });
        });
      }

      _setStoredValue(storageFormatVersion, key, value) {
        return promiseTry(() => {
          logTrace("Start", "_setStoredValue", storageFormatVersion, key, typeof value, value);
          return this._getStorageKey(storageFormatVersion, key).then(storageKey => {
            return this.storageProvider.set(storageKey, value).then(() => {
              logTrace("Done", "_setStoredValue", storageFormatVersion, key, typeof value, value);
              return undefined;
            });
          });
        });
      }

      setStoredValue(key, value) {
        return promiseTry(() => {
          logDebug("Start", "setStoredValue", key, typeof value, value);
          return this._setStoredValue(this.currentStorageFormatVersion, key, value).then(() => {
            logDebug("Done", "setStoredValue", key, typeof value, value);
            return undefined;
          });
        });
      }

      _getStoredValue(storageFormatVersion, key) {
        return promiseTry(() => {
          logTrace("Start", "_getStoredValue", storageFormatVersion, key);
          return this._getStorageKey(storageFormatVersion, key).then(storageKey => {
            return this.storageProvider.get(storageKey).then(value => {
              logTrace("Done", "_getStoredValue", storageFormatVersion, key, value);
              return value;
            });
          });
        });
      }

      getStoredValue(key) {
        return promiseTry(() => {
          logTrace("Start", "getStoredValue", key);
          return this._getStoredValue(this.currentStorageFormatVersion, key).then(value => {
            logTrace("Done", "getStoredValue", key, value);
            return value;
          });
        });
      }

      _createIdentityUpgrader(fromStorageFormatVersion, toStorageFormatVersion) {
        const identityUpgrader = key => promiseTry(() => {
          return this._isStorageKeyValid(fromStorageFormatVersion, key).then(isStorageKeyValid => {
            if (!isStorageKeyValid) {
              return false;
            }

            return this._getStoredValue(fromStorageFormatVersion, key).then(fromValue => {
              if (fromValue === undefined || fromValue === null) {
                return false;
              }

              const toValue = fromValue;
              return this._setStoredValue(toStorageFormatVersion, key, toValue).then(() => true);
            });
          });
        });

        return identityUpgrader;
      }

      _upgradeKey(fromStorageFormatVersion, toStorageFormatVersion, key) {
        return this.allKnownUpgradePaths[fromStorageFormatVersion][toStorageFormatVersion].upgradeKey(key);
      }

      _upgrade(fromStorageFormatVersion, toStorageFormatVersion) {
        return promiseTry(() => {
          const storageKeysForVersion = Object.keys(this.allKnownStorageKeys[toStorageFormatVersion]);
          const upgradePromises = storageKeysForVersion.map(key => this._upgradeKey(fromStorageFormatVersion, toStorageFormatVersion, key));
          return Promise.all(upgradePromises);
        });
      }

      _getStorageMetadata(storageFormatVersion) {
        return promiseTry(() => {
          return this._getStoredValue(storageFormatVersion, this.storageMetadataId);
        });
      }

      _isStorageFormatVersionInitialized(storageFormatVersion) {
        return promiseTry(() => {
          return this._getStorageMetadata(storageFormatVersion).then(storageMetadata => {
            if (storageMetadata !== null) {
              return true;
            }

            return false;
          });
        });
      }

      _setStorageMetadataAsInitialized(storageFormatVersion, fromStorageFormatVersion) {
        return promiseTry(() => {
          return this._isStorageFormatVersionInitialized(storageFormatVersion).then(storageFormatVersionIsInitialized => {
            if (storageFormatVersionIsInitialized) {
              throw new Error(`Already initialized: ${storageFormatVersion}`);
            }

            return undefined;
          }).then(() => {
            const storageMetadata = {
              version: storageFormatVersion,
              "upgraded-from-version": fromStorageFormatVersion,
              "upgraded-at": Date.now()
            };
            return this._setStoredValue(storageFormatVersion, this.storageMetadataId, storageMetadata);
          });
        });
      }

      _findUpgradePaths(toStorageFormatVersion) {
        return promiseTry(() => {
          return this.allKnownStorageFormatVersions.filter(knownStorageFormatVersion => knownStorageFormatVersion !== toStorageFormatVersion).reverse().filter(knownStorageFormatVersion => {
            if (!this.allKnownUpgradePaths[knownStorageFormatVersion]) {
              return false;
            }

            if (!this.allKnownUpgradePaths[knownStorageFormatVersion][toStorageFormatVersion]) {
              return false;
            }

            const upgradePath = this.allKnownUpgradePaths[knownStorageFormatVersion][toStorageFormatVersion];

            if (upgradePath) {
              return true;
            }

            return false;
          });
        });
      }

      _findUpgradePath(toStorageFormatVersion) {
        return promiseTry(() => {
          return this._findUpgradePaths(toStorageFormatVersion).then(upgradePaths => {
            if (!upgradePaths || !Array.isArray(upgradePaths) || upgradePaths.length === 0) {
              return null;
            }

            const possiblyInitializedUpgradePathPromises = upgradePaths.map(upgradePath => {
              return this._isStorageFormatVersionInitialized(upgradePath).then(storageFormatVersionIsInitialized => {
                if (storageFormatVersionIsInitialized) {
                  return upgradePath;
                }

                return null;
              });
            });
            return Promise.all(possiblyInitializedUpgradePathPromises).then(possiblyInitializedUpgradePaths => {
              return possiblyInitializedUpgradePaths.filter(possiblyInitializedUpgradePath => !!possiblyInitializedUpgradePath);
            }).then(initializedUpgradePaths => {
              if (initializedUpgradePaths.length === 0) {
                return null;
              }

              const firstInitializedUpgradePath = initializedUpgradePaths[0];
              return firstInitializedUpgradePath;
            });
          });
        });
      }

      _upgradeIfNecessary(storageFormatVersion) {
        return promiseTry(() => {
          return this._isStorageFormatVersionInitialized(storageFormatVersion).then(storageFormatVersionIsInitialized => {
            if (!storageFormatVersionIsInitialized) {
              return this._findUpgradePath(storageFormatVersion).then(firstInitializedUpgradePath => {
                return promiseTry(() => {
                  if (!firstInitializedUpgradePath) {
                    return false;
                  }

                  return this._upgrade(firstInitializedUpgradePath, storageFormatVersion).then(() => true);
                }).then(result => {
                  this._setStorageMetadataAsInitialized(storageFormatVersion, firstInitializedUpgradePath);

                  return result;
                });
              });
            }

            return false;
          });
        });
      }

      _upgradeV1x0x0IfNecessary() {
        return promiseTry(() => {
          const storageFormatVersion1x0x0 = "v1.0.0";
          const keyToCheck = "options-popup-donate-buttons-hide"; // NOTE: return v1.0.0 as initialized if it had the single setting set,
          // as it didn't have initialization code yet.

          return this._isStorageFormatVersionInitialized(storageFormatVersion1x0x0).then(storageFormatVersionIsInitialized => {
            if (!storageFormatVersionIsInitialized) {
              return this._getStoredValue(storageFormatVersion1x0x0, keyToCheck).then(storageValueToCheck => {
                if (storageValueToCheck !== null) {
                  return this._setStorageMetadataAsInitialized(storageFormatVersion1x0x0, null).then(() => true);
                }

                return false;
              });
            }

            return false;
          });
        });
      }

      upgradeIfNecessary() {
        return promiseTry(() => {
          logDebug("Start", "upgradeIfNecessary");
          return this._upgradeV1x0x0IfNecessary().then(() => this._upgradeIfNecessary(this.currentStorageFormatVersion)).then(result => {
            logDebug("Done", "upgradeIfNecessary");
            return result;
          }).catch(error => {
            logError("upgradeIfNecessary", error);
            throw error;
          });
        });
      }

    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    class SettingsManager {
      constructor(storageManager) {
        this.storageManager = storageManager; // TODO: shared place for stored value constants.

        this._isPremiumEditionStorageKey = "is-premium-edition";
        this._speakLongTextsStorageKey = "speak-long-texts"; // TODO: shared place for default/fallback values for booleans etcetera.

        this._isPremiumEditionDefaultValue = false;
        this._speakLongTextsStorageKeyDefaultValue = false;
      }

      setIsPremiumEdition(isPremiumEdition) {
        return promiseTry(() => this.storageManager.setStoredValue(this._isPremiumEditionStorageKey, isPremiumEdition === true));
      }

      getIsPremiumEdition() {
        return promiseTry(() => this.storageManager.getStoredValue(this._isPremiumEditionStorageKey).then(isPremiumEdition => isPremiumEdition || this._isPremiumEditionDefaultValue));
      }

      setSpeakLongTexts(speakLongTexts) {
        return promiseTry(() => this.storageManager.setStoredValue(this._speakLongTextsStorageKey, speakLongTexts === true));
      }

      getSpeakLongTexts() {
        return promiseTry(() => this.storageManager.getStoredValue(this._speakLongTextsStorageKey).then(speakLongTexts => speakLongTexts || this._speakLongTextsStorageKeyDefaultValue));
      }

    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    class MetadataManager {
      constructor(manifestProvider, settingsManager) {
        this.manifestProvider = manifestProvider;
        this.settingsManager = settingsManager;
        this._editionTypePremium = "premium";
        this._editionTypeFree = "free";
        this._systemTypeChrome = "chrome";
        this._systemTypeWebExtension = "webextension";
      }

      isPremiumEdition() {
        return promiseTry(() => this.settingsManager.getIsPremiumEdition());
      }

      getExtensionId() {
        return promiseTry(() => browser.runtime.id);
      }

      getManifestSync() {
        /* eslint-disable no-sync */
        return this.manifestProvider.getSync();
        /* eslint-enable no-sync */
      }

      getManifest() {
        return promiseTry(
        /* eslint-disable no-sync */
        () => this.getManifestSync()
        /* eslint-enable no-sync */
        );
      }

      getVersionNumber() {
        return promiseTry(() => this.getManifest().then(manifest => {
          return manifest.version || null;
        }));
      }

      getVersionName() {
        return promiseTry(() => this.getManifest().then(manifest => {
          return manifest.version_name || null;
        }));
      }

      getEditionType() {
        return promiseTry(() => this.isPremiumEdition().then(isPremiumEdition => {
          if (isPremiumEdition) {
            return this._editionTypePremium;
          }

          return this._editionTypeFree;
        }));
      }

      isChromeVersion() {
        return promiseTry(() => this.getVersionName().then(versionName => {
          if (versionName.includes(" Chrome Extension ")) {
            return true;
          }

          return false;
        }));
      }

      isWebExtensionVersion() {
        return promiseTry(() => this.getVersionName().then(versionName => {
          if (versionName.includes(" WebExtension ")) {
            return true;
          }

          return false;
        }));
      }

      getSystemType() {
        return promiseTry(() => this.isChromeVersion().then(isChrome => {
          if (isChrome) {
            return this._systemTypeChrome;
          }

          return this._systemTypeWebExtension;
        }));
      }

      getOsType() {
        return promiseTry(() => browser.runtime.getPlatformInfo().then(platformInfo => {
          if (platformInfo && typeof platformInfo.os === "string") {
            // https://developer.chrome.com/extensions/runtime#type-PlatformOs
            return platformInfo.os;
          }

          return null;
        }));
      }

    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    class Configuration {
      // NOTE: keep SynchronousConfiguration and Configuration in... sync.
      constructor(metadataManager, configurationObject) {
        this.metadataManager = metadataManager;
        this.configurationObject = configurationObject;

        this._initialize();
      }

      _initialize() {
        this.configurationObject.shared.urls.root = "/";
        this.configurationObject.shared.urls.demo = "/src/demo/demo.html";
        this.configurationObject.shared.urls.options = "/src/options/options.html";
        this.configurationObject.shared.urls.popup = "/src/popup/popup.html"; // NOTE: direct links to individual tabs.

        this.configurationObject.shared.urls["demo-about"] = this.configurationObject.shared.urls.demo + "#about";
        this.configurationObject.shared.urls["demo-features"] = this.configurationObject.shared.urls.demo + "#features";
        this.configurationObject.shared.urls["demo-support"] = this.configurationObject.shared.urls.demo + "#support";
        this.configurationObject.shared.urls["demo-usage"] = this.configurationObject.shared.urls.demo + "#usage";
        this.configurationObject.shared.urls["demo-voices"] = this.configurationObject.shared.urls.demo + "#voices";
        this.configurationObject.shared.urls["demo-welcome"] = this.configurationObject.shared.urls.demo + "#welcome"; // NOTE: direct links to individual tabs.
        // NOTE: need to pass a parameter to the options page.

        ["popup", "demo"].forEach(from => {
          this.configurationObject.shared.urls[`options-from-${from}`] = this.configurationObject.shared.urls.options + `?from=${from}`;
          this.configurationObject.shared.urls[`options-about-from-${from}`] = this.configurationObject.shared.urls[`options-from-${from}`] + "#about";
          this.configurationObject.shared.urls[`options-upgrade-from-${from}`] = this.configurationObject.shared.urls[`options-from-${from}`] + "#upgrade";
        });
        this.configurationObject.shared.urls["popup-passclick-false"] = this.configurationObject.shared.urls.popup + "?passclick=false";
      }

      _resolvePath(obj, path) {
        // NOTE: doesn't handle arrays nor properties of "any" non-object objects.
        if (!obj || typeof obj !== "object") {
          throw new Error();
        }

        if (!path || typeof path !== "string" || path.length === 0) {
          throw new Error();
        } // NOTE: doesn't handle path["subpath"].


        const parts = path.split(".");
        const part = parts.shift();

        if ({}.hasOwnProperty.call(obj, part)) {
          if (parts.length === 0) {
            return obj[part];
          }

          return this._resolvePath(obj[part], parts.join("."));
        }

        return null;
      }

      get(path) {
        return promiseTry(() => this.metadataManager.getSystemType().then(systemType =>
        /* eslint-disable no-sync */
        this.getSync(systemType, path)
        /* eslint-enable no-sync */
        ));
      }

      getSync(systemType, path) {
        const systemValue = this._resolvePath(this.configurationObject[systemType], path);

        const sharedValue = this._resolvePath(this.configurationObject.shared, path);

        const value = systemValue || sharedValue || null;
        return value;
      }

    }

    var base = {
    	direction: "ltr",
    	sample: "The quick brown fox jumps over the lazy dog."
    };
    var languages = {
    	ar: {
    		direction: "rtl",
    		sample: "الثعلب البني السريع يقفز فوق الكلب الكسول."
    	},
    	bg: {
    		direction: "ltr",
    		sample: "Бърката кафява лисица скача над мързеливото куче."
    	},
    	cs: {
    		direction: "ltr",
    		sample: "Rychlá hnědá liška skočí přes líný pes."
    	},
    	da: {
    		direction: "ltr",
    		sample: "Den hurtige brune ræv hopper over den dovne hund."
    	},
    	de: {
    		direction: "ltr",
    		sample: "Zwölf Boxkämpfer jagen Viktor quer über den großen Sylter Deich."
    	},
    	el: {
    		direction: "ltr",
    		sample: "Η γρήγορη καφέ αλεπού πηδάει πάνω από το τεμπέλικο σκυλί."
    	},
    	en: {
    		direction: "ltr",
    		sample: "The quick brown fox jumps over the lazy dog."
    	},
    	es: {
    		direction: "ltr",
    		sample: "El zorro marrón rápido salta sobre el perro perezoso."
    	},
    	fi: {
    		direction: "ltr",
    		sample: "Nopea ruskea kettu hyppää laiskan koiran yli."
    	},
    	fr: {
    		direction: "ltr",
    		sample: "Portez ce vieux whisky au juge blond qui fume."
    	},
    	he: {
    		direction: "rtl",
    		sample: "השועל החום המהיר קופץ מעל הכלב העצל."
    	},
    	hi: {
    		direction: "ltr",
    		sample: "तेज, भूरी लोमडी आलसी कुत्ते के उपर कूद गई।"
    	},
    	hu: {
    		direction: "ltr",
    		sample: "A gyors barna róka átugorja a lusta kutyát."
    	},
    	id: {
    		direction: "ltr",
    		sample: "Rubah coklat cepat melompati anjing malas itu."
    	},
    	it: {
    		direction: "ltr",
    		sample: "The quick brown fox jumps over the lazy dog."
    	},
    	ja: {
    		direction: "ltr",
    		sample: "クイックブラウンキツネは怠惰な犬の上を飛ぶ。"
    	},
    	ko: {
    		direction: "ltr",
    		sample: "빠른 갈색 여우는 게으른 개를 뛰어 넘습니다."
    	},
    	nb: {
    		direction: "ltr",
    		sample: "Den raske brune reven hopper over den late hunden."
    	},
    	nl: {
    		direction: "ltr",
    		sample: "De snelle bruine vos springt over de luie hond."
    	},
    	pl: {
    		direction: "ltr",
    		sample: "Szybkie brązowe lis skacze nad leniwym psem."
    	},
    	pt: {
    		direction: "ltr",
    		sample: "A rápida raposa marrom salta sobre o cão preguiçoso."
    	},
    	ro: {
    		direction: "ltr",
    		sample: "Vulpea brună se scurge peste câinele leneș."
    	},
    	ru: {
    		direction: "ltr",
    		sample: "Быстрая коричневая лиса прыгает через ленивую собаку."
    	},
    	sk: {
    		direction: "ltr",
    		sample: "Rýchla hnedá liška preskakuje lenivého psa."
    	},
    	sv: {
    		direction: "ltr",
    		sample: "Flygande bäckasiner söka hwila på mjuka tuvor."
    	},
    	th: {
    		direction: "ltr",
    		sample: "สุนัขจิ้งจอกสีน้ำตาลเร็วกระโดดข้ามสุนัขขี้เกียจ"
    	},
    	tr: {
    		direction: "ltr",
    		sample: "Hızlı kahverengi tilki tembel köpeği atlar."
    	},
    	zh: {
    		direction: "ltr",
    		sample: "敏捷的棕色狐狸跳过了懒狗。"
    	}
    };
    var languages$1 = {
    	base: base,
    	languages: languages
    };

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    class TalkieLocaleHelper {
      _getSync(languageCode, name) {
        const languageGroup = languageCode.substring(0, 2);
        const value = languages$1.languages[languageCode] && languages$1.languages[languageCode][name] || languages$1.languages[languageGroup] && languages$1.languages[languageGroup][name] || languages$1.base[name];
        return value;
      }

      getBidiDirectionSync(languageCode) {
        /* eslint-disable no-sync */
        return this._getSync(languageCode, "direction");
        /* eslint-enable no-sync */
      }

      getBidiDirection(languageCode) {
        /* eslint-disable no-sync */
        return promiseTry(() => this.getBidiDirectionSync(languageCode));
        /* eslint-enable no-sync */
      }

      getSampleTextSync(languageCode) {
        /* eslint-disable no-sync */
        return this._getSync(languageCode, "sample");
        /* eslint-enable no-sync */
      }

      getSampleText(languageCode) {
        /* eslint-disable no-sync */
        return promiseTry(() => this.getSampleTextSync(languageCode));
        /* eslint-enable no-sync */
      }

      getTranslatedLanguagesSync() {
        const allLanguages = Object.keys(languages$1.languages);
        return allLanguages;
      }

      getTranslatedLanguages() {
        /* eslint-disable no-sync */
        return promiseTry(() => this.getTranslatedLanguagesSync());
        /* eslint-enable no-sync */
      }

    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const openUrlInNewTab = url => promiseTry(() => {
      if (typeof url !== "string") {
        throw new Error("Bad url: " + url);
      } // NOTE: only https urls.


      if (!url.startsWith("https://")) {
        throw new Error("Bad url, only https:// allowed: " + url);
      }

      return browser.tabs.create({
        active: true,
        url: url
      });
    });
    const openShortKeysConfiguration = () => promiseTry(() => {
      const url = "chrome://extensions/configureCommands";
      return browser.tabs.create({
        active: true,
        url: url
      });
    });
    const openOptionsPage = () => promiseTry(() => {
      return browser.runtime.openOptionsPage();
    });

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const getVoices = () => promiseTry(() => {
      return getBackgroundPage().then(background => background.getAllVoices()).then(voices => {
        if (!Array.isArray(voices)) {
          // NOTE: the list of voices could still be empty, either due to slow loading (cold cache) or that there actually are no voices loaded.
          throw new Error("Could not load list of voices from browser.");
        }

        return voices;
      });
    });

    const getMappedVoice = voice => {
      return {
        default: voice.default,
        lang: voice.lang,
        localService: voice.localService,
        name: voice.name,
        voiceURI: voice.voiceURI
      };
    };

    const getMappedVoices = () => promiseTry(() => {
      return getVoices().then(voices => {
        const mappedVoices = voices.map(getMappedVoice);
        return mappedVoices;
      });
    });
    // checkVoices() {
    //     return this.getSynthesizer()
    //         .then((synthesizer) => {
    //             logDebug("Start", "Voices check");
    //
    //             return getMappedVoices()
    //                 .then((voices) => {
    //                     logDebug("Variable", "voices[]", voices.length, voices);
    //
    //                     logDebug("Done", "Voices check");
    //
    //                     return synthesizer;
    //                 });
    //         });
    // }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    // https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Math/round
    // TODO: don't add non-standard functions to Math.
    // Closure

    (function () {
      /**
      * Decimal adjustment of a number.
      *
      * @param {String}  type  The type of adjustment.
      * @param {Number}  value The number.
      * @param {Integer} exp   The exponent (the 10 logarithm of the adjustment base).
      * @returns {Number} The adjusted value.
      */
      function decimalAdjust(type, value, exp) {
        // If the exp is undefined or zero...
        if (typeof exp === "undefined" || +exp === 0) {
          return Math[type](value);
        }

        value = +value;
        exp = +exp; // If the value is not a number or the exp is not an integer...

        if (isNaN(value) || !(typeof exp === "number" && exp % 1 === 0)) {
          return NaN;
        } // If the value is negative...


        if (value < 0) {
          return -decimalAdjust(type, -value, exp);
        } // Shift


        value = value.toString().split("e");
        value = Math[type](+(value[0] + "e" + (value[1] ? +value[1] - exp : -exp))); // Shift back

        value = value.toString().split("e");
        return +(value[0] + "e" + (value[1] ? +value[1] + exp : exp));
      } // Decimal round


      if (!Math.round10) {
        Math.round10 = function (value, exp) {
          return decimalAdjust("round", value, exp);
        };
      } // Decimal floor


      if (!Math.floor10) {
        Math.floor10 = function (value, exp) {
          return decimalAdjust("floor", value, exp);
        };
      } // Decimal ceil


      if (!Math.ceil10) {
        Math.ceil10 = function (value, exp) {
          return decimalAdjust("ceil", value, exp);
        };
      }
    })();

    const debounce = (fn, limit) => {
      let timeout = null;
      let args = null;

      const limiter = (...limiterArgs) => {
        args = limiterArgs;
        clearTimeout(timeout);
        timeout = setTimeout(() => {
          timeout = null;
          Promise.resolve().then(() => fn(...args)).catch(error => {
            // TODO: log/handle success/errors?
            throw error;
          });
        }, limit);
        return undefined;
      };

      return limiter;
    };

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    class Api {
      constructor(metadataManager, configuration, translator, broadcastProvider, talkieLocaleHelper) {
        this.metadataManager = metadataManager;
        this.configuration = configuration;
        this.translator = translator;
        this.broadcastProvider = broadcastProvider;
        this.talkieLocaleHelper = talkieLocaleHelper;
        this.debouncedSpeak = debounce(this.speak.bind(this), 200);
        this.debouncedSpeakTextInLanguageWithOverrides = debounce(this.speakTextInLanguageWithOverrides.bind(this), 200);
      }

      getConfigurationValueSync(systemType, path) {
        /* eslint-disable no-sync */
        return this.configuration.getSync(systemType, path);
        /* eslint-enable no-sync */
      }

      getConfigurationValue(configurationPath) {
        return this.configuration.get(configurationPath);
      }

      iconClick() {
        return getBackgroundPage().then(background => background.iconClick());
      }

      speak(text, voice) {
        return getBackgroundPage().then(background => background.stopSpeakFromFrontend().then(() => background.startSpeakFromFrontend(text, voice)));
      }

      speakTextInLanguageWithOverrides(text, languageCode) {
        return getBackgroundPage().then(background => background.stopSpeakFromFrontend().then(() => background.startSpeakInLanguageWithOverridesFromFrontend(text, languageCode)));
      }

      getVoices() {
        return getMappedVoices();
      }

      getIsPremiumEditionOption() {
        return getBackgroundPage().then(background => background.getIsPremiumEditionOption());
      }

      setIsPremiumEditionOption(isPremiumEdition) {
        return getBackgroundPage().then(background => background.setIsPremiumEditionOption(isPremiumEdition === true));
      }

      getSpeakLongTextsOption() {
        return getBackgroundPage().then(background => background.getSpeakLongTextsOption());
      }

      setSpeakLongTextsOption(speakLongTexts) {
        return getBackgroundPage().then(background => background.setSpeakLongTextsOption(speakLongTexts === true));
      }

      getSampleText() {
        return promiseTry(() => this.translator.translate("frontend_voicesSampleText"));
      }

      getEffectiveVoiceForLanguage(languageCode) {
        return getBackgroundPage().then(background => background.getEffectiveVoiceForLanguage(languageCode)).then(effectiveVoiceForLanguage => effectiveVoiceForLanguage.name);
      }

      getEffectiveRateForVoice(voiceName) {
        return getBackgroundPage().then(background => background.getEffectiveRateForVoice(voiceName));
      }

      setVoiceRateOverride(voiceName, rate) {
        return getBackgroundPage().then(background => background.setVoiceRateOverride(voiceName, rate));
      }

      getEffectivePitchForVoice(voiceName) {
        return getBackgroundPage().then(background => background.getEffectivePitchForVoice(voiceName));
      }

      setVoicePitchOverride(voiceName, pitch) {
        return getBackgroundPage().then(background => background.setVoicePitchOverride(voiceName, pitch));
      }

      toggleLanguageVoiceOverrideName(languageCode, voiceName) {
        return getBackgroundPage().then(background => background.toggleLanguageVoiceOverrideName(languageCode, voiceName));
      }

      getTranslatedLanguages() {
        return this.talkieLocaleHelper.getTranslatedLanguages();
      }

      isPremiumEdition() {
        return this.metadataManager.isPremiumEdition();
      }

      getVersionName() {
        return this.metadataManager.getVersionName();
      }

      getVersionNumber() {
        return this.metadataManager.getVersionNumber();
      }

      getEditionType() {
        return this.metadataManager.getEditionType();
      }

      getSystemType() {
        return this.metadataManager.getSystemType();
      }

      getOsType() {
        return this.metadataManager.getOsType();
      }

      openUrlInNewTab(url) {
        return openUrlInNewTab(url);
      }

      openShortKeysConfiguration() {
        return openShortKeysConfiguration();
      }

      openOptionsPage() {
        return openOptionsPage();
      }

      registerListeningAction(event, handler) {
        return this.broadcastProvider.registerListeningAction(event, handler);
      }

    }

    function _defineProperty(obj, key, value) {
      if (key in obj) {
        Object.defineProperty(obj, key, {
          value: value,
          enumerable: true,
          configurable: true,
          writable: true
        });
      } else {
        obj[key] = value;
      }

      return obj;
    }

    function _extends() {
      _extends = Object.assign || function (target) {
        for (var i = 1; i < arguments.length; i++) {
          var source = arguments[i];

          for (var key in source) {
            if (Object.prototype.hasOwnProperty.call(source, key)) {
              target[key] = source[key];
            }
          }
        }

        return target;
      };

      return _extends.apply(this, arguments);
    }

    function ownKeys(object, enumerableOnly) {
      var keys = Object.keys(object);

      if (Object.getOwnPropertySymbols) {
        var symbols = Object.getOwnPropertySymbols(object);
        if (enumerableOnly) symbols = symbols.filter(function (sym) {
          return Object.getOwnPropertyDescriptor(object, sym).enumerable;
        });
        keys.push.apply(keys, symbols);
      }

      return keys;
    }

    function _objectSpread2(target) {
      for (var i = 1; i < arguments.length; i++) {
        var source = arguments[i] != null ? arguments[i] : {};

        if (i % 2) {
          ownKeys(Object(source), true).forEach(function (key) {
            _defineProperty(target, key, source[key]);
          });
        } else if (Object.getOwnPropertyDescriptors) {
          Object.defineProperties(target, Object.getOwnPropertyDescriptors(source));
        } else {
          ownKeys(Object(source)).forEach(function (key) {
            Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key));
          });
        }
      }

      return target;
    }

    const manifestProvider$1 = new WebExtensionEnvironmentManifestProvider();
    /* eslint-disable no-sync */

    const manifest = manifestProvider$1.getSync();
    /* eslint-enable no-sync */

    class ErrorBoundary extends React__default['default'].PureComponent {
      constructor(props) {
        super(props);
        this.state = {
          hasError: false,
          message: null,
          stacktrace: null,
          componentStack: null
        };
      }

      componentDidCatch(error, info) {
        // TODO: use DualLogger?

        /* eslint-disable no-console */
        console.error("ErrorBoundary", error, info);
        /* eslint-enable no-console */

        this.setState({
          hasError: true,
          message: error.message,
          stacktrace: error.stack && error.stack.toString(),
          componentStack: info.componentStack
        });
      }

      prettyPrintForEmailBody(value, limit) {
        let pretty = null;

        if (value) {
          pretty = value.toString().trim().replace(/\n/g, "\n> ");

          if (pretty.length > limit) {
            pretty = pretty.substring(0, limit) + "...";
          }
        } else {
          pretty = value;
        }

        return pretty;
      }

      render() {
        if (this.state.hasError) {
          const recipient = "code@joelpurra.com";
          const subject = "Something went wrong in Talkie";
          const body = `Hello Joel,

Something went wrong while using Talkie! This is my error report — can you please have a look at it?

(Optional) My description of the problem is:



Below are some techical details.

Talkie ${manifest.version_name}
https://joelpurra.com/projects/talkie/


Error message:

> ${this.prettyPrintForEmailBody(this.state.message, 128)}


Component stack:

> ${this.prettyPrintForEmailBody(this.state.componentStack, 128)}


Error stack trace:

> ${this.prettyPrintForEmailBody(this.state.stacktrace, 512)}



Hope this helps =)

`;
          const mailto = `mailto:${recipient}?subject=${encodeURIComponent(subject)}&body=${encodeURIComponent(body)}`;
          return /*#__PURE__*/React__default['default'].createElement("div", null, /*#__PURE__*/React__default['default'].createElement("h1", null, "Something went wrong"), /*#__PURE__*/React__default['default'].createElement("p", null, "Sorry! This really should not happen. If you would like, email me an error report using the link below, and I will try to fix it for ", /*#__PURE__*/React__default['default'].createElement("a", {
            href: "https://joelpurra.com/projects/talkie/"
          }, "the next version of Talkie"), "!"), /*#__PURE__*/React__default['default'].createElement("p", null, /*#__PURE__*/React__default['default'].createElement("a", {
            href: "https://joelpurra.com/",
            rel: "noopener noreferrer",
            target: "_blank",
            lang: "sv"
          }, "Joel Purra")), /*#__PURE__*/React__default['default'].createElement("hr", null), /*#__PURE__*/React__default['default'].createElement("p", null, "Talkie ", manifest.version_name), /*#__PURE__*/React__default['default'].createElement("blockquote", null, /*#__PURE__*/React__default['default'].createElement("pre", null, this.state.message)), /*#__PURE__*/React__default['default'].createElement("p", null, /*#__PURE__*/React__default['default'].createElement("a", {
            href: mailto,
            rel: "noopener noreferrer",
            target: "_blank"
          }, "Email error report to ", recipient)), /*#__PURE__*/React__default['default'].createElement("details", null, /*#__PURE__*/React__default['default'].createElement("summary", null, "Component stack"), /*#__PURE__*/React__default['default'].createElement("blockquote", null, /*#__PURE__*/React__default['default'].createElement("pre", null, this.state.componentStack))), /*#__PURE__*/React__default['default'].createElement("details", null, /*#__PURE__*/React__default['default'].createElement("summary", null, "Error stack trace"), /*#__PURE__*/React__default['default'].createElement("blockquote", null, /*#__PURE__*/React__default['default'].createElement("pre", null, this.state.stacktrace))));
        }

        return this.props.children;
      }

    }

    _defineProperty(ErrorBoundary, "propTypes", {
      children: PropTypes__default['default'].node.isRequired
    });

    var global$1 = (typeof global !== "undefined" ? global :
                typeof self !== "undefined" ? self :
                typeof window !== "undefined" ? window : {});

    // from https://github.com/kumavis/browser-process-hrtime/blob/master/index.js
    var performance = global$1.performance || {};
    var performanceNow =
      performance.now        ||
      performance.mozNow     ||
      performance.msNow      ||
      performance.oNow       ||
      performance.webkitNow  ||
      function(){ return (new Date()).getTime() };

    function getDefaultExportFromCjs (x) {
    	return x && x.__esModule && Object.prototype.hasOwnProperty.call(x, 'default') ? x['default'] : x;
    }

    function createCommonjsModule(fn, basedir, module) {
    	return module = {
    		path: basedir,
    		exports: {},
    		require: function (path, base) {
    			return commonjsRequire(path, (base === undefined || base === null) ? module.path : base);
    		}
    	}, fn(module, module.exports), module.exports;
    }

    function getAugmentedNamespace(n) {
    	if (n.__esModule) return n;
    	var a = Object.defineProperty({}, '__esModule', {value: true});
    	Object.keys(n).forEach(function (k) {
    		var d = Object.getOwnPropertyDescriptor(n, k);
    		Object.defineProperty(a, k, d.get ? d : {
    			enumerable: true,
    			get: function () {
    				return n[k];
    			}
    		});
    	});
    	return a;
    }

    function commonjsRequire () {
    	throw new Error('Dynamic requires are not currently supported by @rollup/plugin-commonjs');
    }

    var capitalizeString_1 = createCommonjsModule(function (module, exports) {

    Object.defineProperty(exports, "__esModule", {
      value: true
    });
    exports.default = capitalizeString;
    function capitalizeString(str) {
      return str.charAt(0).toUpperCase() + str.slice(1);
    }
    module.exports = exports["default"];
    });

    var prefixProperty_1 = createCommonjsModule(function (module, exports) {

    Object.defineProperty(exports, "__esModule", {
      value: true
    });
    exports.default = prefixProperty;



    var _capitalizeString2 = _interopRequireDefault(capitalizeString_1);

    function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

    function prefixProperty(prefixProperties, property, style) {
      if (prefixProperties.hasOwnProperty(property)) {
        var newStyle = {};
        var requiredPrefixes = prefixProperties[property];
        var capitalizedProperty = (0, _capitalizeString2.default)(property);
        var keys = Object.keys(style);
        for (var i = 0; i < keys.length; i++) {
          var styleProperty = keys[i];
          if (styleProperty === property) {
            for (var j = 0; j < requiredPrefixes.length; j++) {
              newStyle[requiredPrefixes[j] + capitalizedProperty] = style[property];
            }
          }
          newStyle[styleProperty] = style[styleProperty];
        }
        return newStyle;
      }
      return style;
    }
    module.exports = exports['default'];
    });

    var prefixValue_1 = createCommonjsModule(function (module, exports) {

    Object.defineProperty(exports, "__esModule", {
      value: true
    });
    exports.default = prefixValue;
    function prefixValue(plugins, property, value, style, metaData) {
      for (var i = 0, len = plugins.length; i < len; ++i) {
        var processedValue = plugins[i](property, value, style, metaData);

        // we can stop processing if a value is returned
        // as all plugin criteria are unique
        if (processedValue) {
          return processedValue;
        }
      }
    }
    module.exports = exports["default"];
    });

    var addNewValuesOnly_1 = createCommonjsModule(function (module, exports) {

    Object.defineProperty(exports, "__esModule", {
      value: true
    });
    exports.default = addNewValuesOnly;
    function addIfNew(list, value) {
      if (list.indexOf(value) === -1) {
        list.push(value);
      }
    }

    function addNewValuesOnly(list, values) {
      if (Array.isArray(values)) {
        for (var i = 0, len = values.length; i < len; ++i) {
          addIfNew(list, values[i]);
        }
      } else {
        addIfNew(list, values);
      }
    }
    module.exports = exports["default"];
    });

    var isObject_1 = createCommonjsModule(function (module, exports) {

    Object.defineProperty(exports, "__esModule", {
      value: true
    });
    exports.default = isObject;
    function isObject(value) {
      return value instanceof Object && !Array.isArray(value);
    }
    module.exports = exports["default"];
    });

    var createPrefixer_1 = createCommonjsModule(function (module, exports) {

    Object.defineProperty(exports, "__esModule", {
      value: true
    });
    exports.default = createPrefixer;



    var _prefixProperty2 = _interopRequireDefault(prefixProperty_1);



    var _prefixValue2 = _interopRequireDefault(prefixValue_1);



    var _addNewValuesOnly2 = _interopRequireDefault(addNewValuesOnly_1);



    var _isObject2 = _interopRequireDefault(isObject_1);

    function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

    function createPrefixer(_ref) {
      var prefixMap = _ref.prefixMap,
          plugins = _ref.plugins;

      function prefixAll(style) {
        for (var property in style) {
          var value = style[property];

          // handle nested objects
          if ((0, _isObject2.default)(value)) {
            style[property] = prefixAll(value);
            // handle array values
          } else if (Array.isArray(value)) {
            var combinedValue = [];

            for (var i = 0, len = value.length; i < len; ++i) {
              var processedValue = (0, _prefixValue2.default)(plugins, property, value[i], style, prefixMap);
              (0, _addNewValuesOnly2.default)(combinedValue, processedValue || value[i]);
            }

            // only modify the value if it was touched
            // by any plugin to prevent unnecessary mutations
            if (combinedValue.length > 0) {
              style[property] = combinedValue;
            }
          } else {
            var _processedValue = (0, _prefixValue2.default)(plugins, property, value, style, prefixMap);

            // only modify the value if it was touched
            // by any plugin to prevent unnecessary mutations
            if (_processedValue) {
              style[property] = _processedValue;
            }

            style = (0, _prefixProperty2.default)(prefixMap, property, style);
          }
        }

        return style;
      }

      return prefixAll;
    }
    module.exports = exports['default'];
    });

    var staticData = createCommonjsModule(function (module, exports) {

    Object.defineProperty(exports, "__esModule", {
      value: true
    });

    var w = ["Webkit"];
    var m = ["Moz"];
    var ms = ["ms"];
    var wm = ["Webkit", "Moz"];
    var wms = ["Webkit", "ms"];
    var wmms = ["Webkit", "Moz", "ms"];

    exports.default = {
      plugins: [],
      prefixMap: { "appearance": wm, "textEmphasisPosition": w, "textEmphasis": w, "textEmphasisStyle": w, "textEmphasisColor": w, "boxDecorationBreak": w, "maskImage": w, "maskMode": w, "maskRepeat": w, "maskPosition": w, "maskClip": w, "maskOrigin": w, "maskSize": w, "maskComposite": w, "mask": w, "maskBorderSource": w, "maskBorderMode": w, "maskBorderSlice": w, "maskBorderWidth": w, "maskBorderOutset": w, "maskBorderRepeat": w, "maskBorder": w, "maskType": w, "textDecorationStyle": w, "textDecorationSkip": w, "textDecorationLine": w, "textDecorationColor": w, "userSelect": wmms, "backdropFilter": w, "fontKerning": w, "scrollSnapType": wms, "scrollSnapPointsX": wms, "scrollSnapPointsY": wms, "scrollSnapDestination": wms, "scrollSnapCoordinate": wms, "clipPath": w, "shapeImageThreshold": w, "shapeImageMargin": w, "shapeImageOutside": w, "filter": w, "hyphens": wms, "flowInto": wms, "flowFrom": wms, "breakBefore": wms, "breakAfter": wms, "breakInside": wms, "regionFragment": wms, "writingMode": wms, "textOrientation": w, "tabSize": m, "fontFeatureSettings": w, "columnCount": w, "columnFill": w, "columnGap": w, "columnRule": w, "columnRuleColor": w, "columnRuleStyle": w, "columnRuleWidth": w, "columns": w, "columnSpan": w, "columnWidth": w, "wrapFlow": ms, "wrapThrough": ms, "wrapMargin": ms, "gridTemplateColumns": ms, "gridTemplateRows": ms, "gridTemplateAreas": ms, "gridTemplate": ms, "gridAutoColumns": ms, "gridAutoRows": ms, "gridAutoFlow": ms, "grid": ms, "gridRowStart": ms, "gridColumnStart": ms, "gridRowEnd": ms, "gridRow": ms, "gridColumn": ms, "gridColumnEnd": ms, "gridColumnGap": ms, "gridRowGap": ms, "gridArea": ms, "gridGap": ms, "textSizeAdjust": wms }
    };
    module.exports = exports["default"];
    });

    var cursor_1 = createCommonjsModule(function (module, exports) {

    Object.defineProperty(exports, "__esModule", {
      value: true
    });
    exports.default = cursor;
    var prefixes = ['-webkit-', '-moz-', ''];

    var values = {
      'zoom-in': true,
      'zoom-out': true,
      grab: true,
      grabbing: true
    };

    function cursor(property, value) {
      if (property === 'cursor' && values.hasOwnProperty(value)) {
        return prefixes.map(function (prefix) {
          return prefix + value;
        });
      }
    }
    module.exports = exports['default'];
    });

    var isPrefixedValue_1 = createCommonjsModule(function (module, exports) {

    Object.defineProperty(exports, "__esModule", {
      value: true
    });
    exports.default = isPrefixedValue;
    var regex = /-webkit-|-moz-|-ms-/;

    function isPrefixedValue(value) {
      return typeof value === 'string' && regex.test(value);
    }
    module.exports = exports['default'];
    });

    var crossFade_1 = createCommonjsModule(function (module, exports) {

    Object.defineProperty(exports, "__esModule", {
      value: true
    });
    exports.default = crossFade;



    var _isPrefixedValue2 = _interopRequireDefault(isPrefixedValue_1);

    function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

    // http://caniuse.com/#search=cross-fade
    var prefixes = ['-webkit-', ''];
    function crossFade(property, value) {
      if (typeof value === 'string' && !(0, _isPrefixedValue2.default)(value) && value.indexOf('cross-fade(') > -1) {
        return prefixes.map(function (prefix) {
          return value.replace(/cross-fade\(/g, prefix + 'cross-fade(');
        });
      }
    }
    module.exports = exports['default'];
    });

    var filter_1 = createCommonjsModule(function (module, exports) {

    Object.defineProperty(exports, "__esModule", {
      value: true
    });
    exports.default = filter;



    var _isPrefixedValue2 = _interopRequireDefault(isPrefixedValue_1);

    function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

    // http://caniuse.com/#feat=css-filter-function
    var prefixes = ['-webkit-', ''];
    function filter(property, value) {
      if (typeof value === 'string' && !(0, _isPrefixedValue2.default)(value) && value.indexOf('filter(') > -1) {
        return prefixes.map(function (prefix) {
          return value.replace(/filter\(/g, prefix + 'filter(');
        });
      }
    }
    module.exports = exports['default'];
    });

    var flex_1 = createCommonjsModule(function (module, exports) {

    Object.defineProperty(exports, "__esModule", {
      value: true
    });
    exports.default = flex;
    var values = {
      flex: ['-webkit-box', '-moz-box', '-ms-flexbox', '-webkit-flex', 'flex'],
      'inline-flex': ['-webkit-inline-box', '-moz-inline-box', '-ms-inline-flexbox', '-webkit-inline-flex', 'inline-flex']
    };

    function flex(property, value) {
      if (property === 'display' && values.hasOwnProperty(value)) {
        return values[value];
      }
    }
    module.exports = exports['default'];
    });

    var flexboxOld_1 = createCommonjsModule(function (module, exports) {

    Object.defineProperty(exports, "__esModule", {
      value: true
    });
    exports.default = flexboxOld;
    var alternativeValues = {
      'space-around': 'justify',
      'space-between': 'justify',
      'flex-start': 'start',
      'flex-end': 'end',
      'wrap-reverse': 'multiple',
      wrap: 'multiple',
      flex: 'box',
      'inline-flex': 'inline-box'
    };

    var alternativeProps = {
      alignItems: 'WebkitBoxAlign',
      justifyContent: 'WebkitBoxPack',
      flexWrap: 'WebkitBoxLines',
      flexGrow: 'WebkitBoxFlex'
    };

    function flexboxOld(property, value, style) {
      if (property === 'flexDirection' && typeof value === 'string') {
        if (value.indexOf('column') > -1) {
          style.WebkitBoxOrient = 'vertical';
        } else {
          style.WebkitBoxOrient = 'horizontal';
        }
        if (value.indexOf('reverse') > -1) {
          style.WebkitBoxDirection = 'reverse';
        } else {
          style.WebkitBoxDirection = 'normal';
        }
      }
      if (alternativeProps.hasOwnProperty(property)) {
        style[alternativeProps[property]] = alternativeValues[value] || value;
      }
    }
    module.exports = exports['default'];
    });

    var gradient_1 = createCommonjsModule(function (module, exports) {

    Object.defineProperty(exports, "__esModule", {
      value: true
    });
    exports.default = gradient;



    var _isPrefixedValue2 = _interopRequireDefault(isPrefixedValue_1);

    function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

    var prefixes = ['-webkit-', '-moz-', ''];

    var values = /linear-gradient|radial-gradient|repeating-linear-gradient|repeating-radial-gradient/gi;

    function gradient(property, value) {
      if (typeof value === 'string' && !(0, _isPrefixedValue2.default)(value) && values.test(value)) {
        return prefixes.map(function (prefix) {
          return value.replace(values, function (grad) {
            return prefix + grad;
          });
        });
      }
    }
    module.exports = exports['default'];
    });

    var imageSet_1 = createCommonjsModule(function (module, exports) {

    Object.defineProperty(exports, "__esModule", {
      value: true
    });
    exports.default = imageSet;



    var _isPrefixedValue2 = _interopRequireDefault(isPrefixedValue_1);

    function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

    // http://caniuse.com/#feat=css-image-set
    var prefixes = ['-webkit-', ''];
    function imageSet(property, value) {
      if (typeof value === 'string' && !(0, _isPrefixedValue2.default)(value) && value.indexOf('image-set(') > -1) {
        return prefixes.map(function (prefix) {
          return value.replace(/image-set\(/g, prefix + 'image-set(');
        });
      }
    }
    module.exports = exports['default'];
    });

    var position_1 = createCommonjsModule(function (module, exports) {

    Object.defineProperty(exports, "__esModule", {
      value: true
    });
    exports.default = position;
    function position(property, value) {
      if (property === 'position' && value === 'sticky') {
        return ['-webkit-sticky', 'sticky'];
      }
    }
    module.exports = exports['default'];
    });

    var sizing_1 = createCommonjsModule(function (module, exports) {

    Object.defineProperty(exports, "__esModule", {
      value: true
    });
    exports.default = sizing;
    var prefixes = ['-webkit-', '-moz-', ''];

    var properties = {
      maxHeight: true,
      maxWidth: true,
      width: true,
      height: true,
      columnWidth: true,
      minWidth: true,
      minHeight: true
    };
    var values = {
      'min-content': true,
      'max-content': true,
      'fill-available': true,
      'fit-content': true,
      'contain-floats': true
    };

    function sizing(property, value) {
      if (properties.hasOwnProperty(property) && values.hasOwnProperty(value)) {
        return prefixes.map(function (prefix) {
          return prefix + value;
        });
      }
    }
    module.exports = exports['default'];
    });

    /* eslint-disable no-var, prefer-template */
    var uppercasePattern = /[A-Z]/g;
    var msPattern = /^ms-/;
    var cache = {};

    function toHyphenLower(match) {
      return '-' + match.toLowerCase()
    }

    function hyphenateStyleName(name) {
      if (cache.hasOwnProperty(name)) {
        return cache[name]
      }

      var hName = name.replace(uppercasePattern, toHyphenLower);
      return (cache[name] = msPattern.test(hName) ? '-' + hName : hName)
    }

    var hyphenateStyleName$1 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        'default': hyphenateStyleName
    });

    var _hyphenateStyleName = /*@__PURE__*/getAugmentedNamespace(hyphenateStyleName$1);

    var hyphenateProperty_1 = createCommonjsModule(function (module, exports) {

    Object.defineProperty(exports, "__esModule", {
      value: true
    });
    exports.default = hyphenateProperty;



    var _hyphenateStyleName2 = _interopRequireDefault(_hyphenateStyleName);

    function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

    function hyphenateProperty(property) {
      return (0, _hyphenateStyleName2.default)(property);
    }
    module.exports = exports['default'];
    });

    var transition_1 = createCommonjsModule(function (module, exports) {

    Object.defineProperty(exports, "__esModule", {
      value: true
    });
    exports.default = transition;



    var _hyphenateProperty2 = _interopRequireDefault(hyphenateProperty_1);



    var _isPrefixedValue2 = _interopRequireDefault(isPrefixedValue_1);



    var _capitalizeString2 = _interopRequireDefault(capitalizeString_1);

    function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

    var properties = {
      transition: true,
      transitionProperty: true,
      WebkitTransition: true,
      WebkitTransitionProperty: true,
      MozTransition: true,
      MozTransitionProperty: true
    };


    var prefixMapping = {
      Webkit: '-webkit-',
      Moz: '-moz-',
      ms: '-ms-'
    };

    function prefixValue(value, propertyPrefixMap) {
      if ((0, _isPrefixedValue2.default)(value)) {
        return value;
      }

      // only split multi values, not cubic beziers
      var multipleValues = value.split(/,(?![^()]*(?:\([^()]*\))?\))/g);

      for (var i = 0, len = multipleValues.length; i < len; ++i) {
        var singleValue = multipleValues[i];
        var values = [singleValue];
        for (var property in propertyPrefixMap) {
          var dashCaseProperty = (0, _hyphenateProperty2.default)(property);

          if (singleValue.indexOf(dashCaseProperty) > -1 && dashCaseProperty !== 'order') {
            var prefixes = propertyPrefixMap[property];
            for (var j = 0, pLen = prefixes.length; j < pLen; ++j) {
              // join all prefixes and create a new value
              values.unshift(singleValue.replace(dashCaseProperty, prefixMapping[prefixes[j]] + dashCaseProperty));
            }
          }
        }

        multipleValues[i] = values.join(',');
      }

      return multipleValues.join(',');
    }

    function transition(property, value, style, propertyPrefixMap) {
      // also check for already prefixed transitions
      if (typeof value === 'string' && properties.hasOwnProperty(property)) {
        var outputValue = prefixValue(value, propertyPrefixMap);
        // if the property is already prefixed
        var webkitOutput = outputValue.split(/,(?![^()]*(?:\([^()]*\))?\))/g).filter(function (val) {
          return !/-moz-|-ms-/.test(val);
        }).join(',');

        if (property.indexOf('Webkit') > -1) {
          return webkitOutput;
        }

        var mozOutput = outputValue.split(/,(?![^()]*(?:\([^()]*\))?\))/g).filter(function (val) {
          return !/-webkit-|-ms-/.test(val);
        }).join(',');

        if (property.indexOf('Moz') > -1) {
          return mozOutput;
        }

        style['Webkit' + (0, _capitalizeString2.default)(property)] = webkitOutput;
        style['Moz' + (0, _capitalizeString2.default)(property)] = mozOutput;
        return outputValue;
      }
    }
    module.exports = exports['default'];
    });

    var _static = createCommonjsModule(function (module, exports) {

    Object.defineProperty(exports, "__esModule", {
      value: true
    });



    var _createPrefixer2 = _interopRequireDefault(createPrefixer_1);



    var _staticData2 = _interopRequireDefault(staticData);



    var _cursor2 = _interopRequireDefault(cursor_1);



    var _crossFade2 = _interopRequireDefault(crossFade_1);



    var _filter2 = _interopRequireDefault(filter_1);



    var _flex2 = _interopRequireDefault(flex_1);



    var _flexboxOld2 = _interopRequireDefault(flexboxOld_1);



    var _gradient2 = _interopRequireDefault(gradient_1);



    var _imageSet2 = _interopRequireDefault(imageSet_1);



    var _position2 = _interopRequireDefault(position_1);



    var _sizing2 = _interopRequireDefault(sizing_1);



    var _transition2 = _interopRequireDefault(transition_1);

    function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

    var plugins = [_crossFade2.default, _cursor2.default, _filter2.default, _flexboxOld2.default, _gradient2.default, _imageSet2.default, _position2.default, _sizing2.default, _transition2.default, _flex2.default];

    exports.default = (0, _createPrefixer2.default)({
      prefixMap: _staticData2.default.prefixMap,
      plugins: plugins
    });
    module.exports = exports['default'];
    });

    var prefixAll = /*@__PURE__*/getDefaultExportFromCjs(_static);

    const uppercasePattern$1 = /[A-Z]/g;
    const msPattern$1 = /^ms-/;
    const cache$1 = {};
    function hyphenateStyleName$2(prop) {
      return prop in cache$1 ? cache$1[prop] : cache$1[prop] = prop.replace(uppercasePattern$1, '-$&').toLowerCase().replace(msPattern$1, '-ms-');
    }

    const prefixedBlockCache = {};
    function injectStylePrefixed(styletron, styles, media, pseudo, cache = prefixedBlockCache) {
      let classString = '';

      for (const originalKey in styles) {
        const originalVal = styles[originalKey];
        const originalValType = typeof originalVal;
        const isPrimitiveVal = originalValType === 'string' || originalValType === 'number';

        if (isPrimitiveVal || Array.isArray(originalVal)) {
          let block = '';

          if (isPrimitiveVal && cache.hasOwnProperty(originalKey) && cache[originalKey].hasOwnProperty(originalVal)) {
            block = cache[originalKey][originalVal];
          } else {
            const prefixed = prefixAll({
              [originalKey]: originalVal
            });

            for (const prefixedKey in prefixed) {
              const prefixedVal = prefixed[prefixedKey];
              const prefixedValType = typeof prefixedVal;

              if (prefixedValType === 'string' || prefixedValType === 'number') {
                block += `${hyphenateStyleName$2(prefixedKey)}:${prefixedVal};`;
                continue;
              }

              if (Array.isArray(prefixedVal)) {
                const hyphenated = hyphenateStyleName$2(prefixedKey);

                for (let i = 0; i < prefixedVal.length; i++) {
                  block += `${hyphenated}:${prefixedVal[i]};`;
                }

                continue;
              }
            }

            block = block.slice(0, -1); // Remove trailing semicolon

            if (isPrimitiveVal) {
              if (!cache.hasOwnProperty(originalKey)) {
                cache[originalKey] = {};
              }

              cache[originalKey][originalVal] = block;
            }
          }

          classString += ' ' + styletron.injectRawDeclaration({
            block,
            media,
            pseudo
          });
        }

        if (originalValType === 'object') {
          if (originalKey[0] === ':') {
            classString += ' ' + injectStylePrefixed(styletron, originalVal, media, originalKey, cache);
            continue;
          }

          if (originalKey.substring(0, 6) === '@media') {
            classString += ' ' + injectStylePrefixed(styletron, originalVal, originalKey.substr(7), pseudo, cache);
            continue;
          }
        }
      } // remove leading space on way out


      return classString.slice(1);
    }

    /**
     * @class StyletronProvider
     * @packagename styletron-react
     * @description Provides a Styletron instance to descendant styled components via context
     * @example
     * const Styletron = require('styletron');
     *
     * function render() {
     *   return React.renderToString(
     *     <StyletronProvider styletron={new Styletron()}>
     *       <App/>
     *     </StyletronProvider>
     *   );
     * }
     *
     * @property {object} styletron - Styletron instance
     * @property {ReactElement} children - children
     * @extends ReactClass
     */

    class StyletronProvider extends React__default['default'].Component {
      getChildContext() {
        return {
          styletron: this.styletron
        };
      }

      constructor(props, context) {
        super(props, context);
        this.styletron = props.styletron;
      }

      render() {
        return React__default['default'].Children.only(this.props.children);
      }

    }

    StyletronProvider.propTypes = {
      styletron: PropTypes__default['default'].object.isRequired,
      children: PropTypes__default['default'].element.isRequired
    };
    StyletronProvider.childContextTypes = {
      styletron: PropTypes__default['default'].object.isRequired
    };

    const STYLETRON_KEY = '__STYLETRON';
    /**
     * Helper function to create styled element components
     * @packagename styletron-react
     * @param  {String|function} base        Tag name or component
     * @param  {function|object} styleFn     Style object or function that returns a style object
     * @param  {function}        assignProps Function that consumes the style result and props and returns an object with new props
     * @return {function}                    Component
     * @example
     */

    function core(base, style, assignProps) {
      if (typeof base === 'function' && base[STYLETRON_KEY]) {
        const {
          tag,
          styles
        } = base[STYLETRON_KEY]; // Styled component

        return createStyledElementComponent(tag, styles.concat(style), assignProps);
      }

      if (typeof base === 'string' || typeof base === 'function') {
        // Tag name or non-styled component
        return createStyledElementComponent(base, [style], assignProps);
      }

      throw new Error('`styled` takes either a DOM element name or a component');
    }

    function createStyledElementComponent(base, stylesArray, assignProps) {
      function StyledElement(props, context) {
        const ownProps = assign({}, props);
        delete ownProps.innerRef;
        const styleResult = {};
        StyledElement[STYLETRON_KEY].styles.forEach(style => {
          if (typeof style === 'function') {
            assign(styleResult, style(ownProps, context));
          } else if (typeof style === 'object') {
            assign(styleResult, style);
          }
        });
        let elementProps = assignProps(context.styletron, styleResult, ownProps);
        elementProps = omit$Props(elementProps);

        if (props.innerRef) {
          elementProps.ref = props.innerRef;
        }

        return React__default['default'].createElement(StyledElement[STYLETRON_KEY].tag, elementProps);
      }

      StyledElement[STYLETRON_KEY] = {
        tag: base,
        styles: stylesArray
      };
      StyledElement.contextTypes = {
        styletron: PropTypes__default['default'].object
      };

      {
        const name = base.displayName ? base.displayName : typeof base === 'function' ? base.name : base;
        StyledElement.displayName = `Styled${name ? `(${name})` : ''}`;
      }

      return StyledElement;
    }

    function assign(target, source) {
      for (const key in source) {
        target[key] = source[key];
      }

      return target;
    }

    function omit$Props(source) {
      const result = {};

      for (const key in source) {
        if (key[0] !== '$') {
          result[key] = source[key];
        }
      }

      return result;
    }

    /**
     * Helper function to create styled element components
     * @packagename styletron-react
     * @param  {String|function} base     Tag name or styled element component
     * @param  {function|object} styleFn  Style object or function that returns a style object
     * @return {function}                 Styled element component
     * @example
     * import {styled} from 'styletron-react';
     *
     * const Panel = styled('div', {
     *   backgroundColor: 'lightblue',
     *   fontSize: '12px'
     * });
     *
     * <Panel>Hello World</Panel>
     * @example
     * import {styled} from 'styletron-react';
     *
     * const Panel = styled('div', (props) => ({
     *   backgroundColor: props.alert ? 'orange' : 'lightblue',
     *   fontSize: '12px'
     * }));
     *
     * <Panel alert>Danger!</Panel>
     * @example
     * import {styled} from 'styletron-react';
     *
     * const DeluxePanel = styled(Panel, (props) => ({
     *   backgroundColor: props.alert ? 'firebrick' : 'rebeccapurple',
     *   color: 'white',
     *   boxShadow: '3px 3px 3px darkgray'
     * }));
     *
     * <DeluxePanel>Bonjour Monde</DeluxePanel>
     */

    function styled(base, style) {
      return core(base, style, assignProps);
    }

    function assignProps(styletron, styleResult, ownProps) {
      const styletronClassName = injectStylePrefixed(styletron, styleResult); // Skipping cloning of `ownProps` since that's already done internally

      if (ownProps.styleProps) {
        delete ownProps.styleProps;
      }

      ownProps.className = ownProps.className ? `${ownProps.className} ${styletronClassName}` : styletronClassName;
      return ownProps;
    }

    class ConfigurationProvider extends React__default['default'].PureComponent {
      constructor(props) {
        super(props);
        this._listeners = {};
        this._counter = 0;
      }

      UNSAFE_componentWillReceiveProps(nextProps) {
        if (this.state.systemType !== nextProps.systemType) {
          this.onChange({
            systemType: nextProps.systemType
          });
        }
      }

      getChildContext() {
        const {
          systemType
        } = this.props;
        return {
          onConfigurationChange: listener => this.registerListener(listener),

          /* eslint-disable no-sync */
          configure: path => this.props.configuration.getSync(systemType, path)
          /* eslint-enable no-sync */

        };
      }

      registerListener(listener) {
        const id = `listener-${this._counter.toString().padStart(4, "0")}}`;
        this._counter++;
        this._listeners[id] = listener;

        const unregisterListener = () => {
          if (!(id in this._listeners)) {
            throw new Error(`Listener id not found: ${JSON.stringify(id)}`);
          }

          delete this._listeners[id];
        };

        return unregisterListener;
      }

      render() {
        return React__default['default'].Children.only(this.props.children);
      }

    }

    _defineProperty(ConfigurationProvider, "propTypes", {
      children: PropTypes__default['default'].element.isRequired,
      configuration: PropTypes__default['default'].object.isRequired,
      systemType: PropTypes__default['default'].string.isRequired
    });

    _defineProperty(ConfigurationProvider, "childContextTypes", {
      // NOTE: uses hacky workaround for dynamically updating "semi-static" context values and updating "deep" components
      // which may be obscured descendants of a PureComponent (or otherwise set shouldComponentUpdate to false).
      // https://medium.com/@mweststrate/how-to-safely-use-react-context-b7e343eff076
      // TODO: find a better solution to look up arbitrary (although it's a short list) configuration values.
      // TODO: use proper event listener system?
      onConfigurationChange: PropTypes__default['default'].func.isRequired,
      configure: PropTypes__default['default'].func.isRequired
    });

    class TranslationProvider extends React__default['default'].PureComponent {
      getChildContext() {
        return {
          translate: (key, ...args) => this.props.translator.translate(key, ...args)
        };
      }

      render() {
        return React__default['default'].Children.only(this.props.children);
      }

    }

    _defineProperty(TranslationProvider, "propTypes", {
      children: PropTypes__default['default'].element.isRequired,
      translator: PropTypes__default['default'].object.isRequired
    });

    _defineProperty(TranslationProvider, "childContextTypes", {
      translate: PropTypes__default['default'].func.isRequired
    });

    class BroadcasterProvider extends React__default['default'].PureComponent {
      getChildContext() {
        return {
          broadcaster: this.props.broadcaster
        };
      }

      render() {
        return React__default['default'].Children.only(this.props.children);
      }

    }

    _defineProperty(BroadcasterProvider, "propTypes", {
      children: PropTypes__default['default'].element.isRequired,
      broadcaster: PropTypes__default['default'].object.isRequired
    });

    _defineProperty(BroadcasterProvider, "childContextTypes", {
      broadcaster: PropTypes__default['default'].object.isRequired
    });

    class StyleRoot extends React__default['default'].PureComponent {
      getStateClasses() {
        const {
          isSpeaking,
          isPremiumEdition
        } = this.props;
        const stateClasses = [];

        if (isSpeaking) {
          stateClasses.push("talkie-speaking");
        } else {
          stateClasses.push("talkie-not-speaking");
        }

        if (isPremiumEdition) {
          stateClasses.push("talkie-premium");
        } else {
          stateClasses.push("talkie-free");
        }

        return stateClasses;
      }

      render() {
        const {
          children
        } = this.props;
        const stateClasses = this.getStateClasses();
        const stateClassNames = stateClasses.join(" ");
        return /*#__PURE__*/React__default['default'].createElement("div", {
          className: stateClassNames
        }, React__default['default'].Children.only(children));
      }

    }

    _defineProperty(StyleRoot, "defaultProps", {
      isSpeaking: false,
      isPremiumEdition: false
    });

    _defineProperty(StyleRoot, "propTypes", {
      isSpeaking: PropTypes__default['default'].bool.isRequired,
      isPremiumEdition: PropTypes__default['default'].bool.isRequired,
      children: PropTypes__default['default'].element.isRequired
    });

    var _dec, _class, _class2, _temp;

    const mapStateToProps = state => {
      return {
        isSpeaking: state.shared.speaking.isSpeaking,
        isPremiumEdition: state.shared.metadata.isPremiumEdition,
        versionName: state.shared.metadata.versionName
      };
    };

    const mapDispatchToProps = (
    /* eslint-disable no-unused-vars */
    dispatch
    /* eslint-enable no-unused-vars */
    ) => {
      return {};
    };

    let StateRoot = (_dec = reactRedux.connect(mapStateToProps, mapDispatchToProps), _dec(_class = (_temp = _class2 = class StateRoot extends React__default['default'].PureComponent {
      render() {
        const {
          isSpeaking,
          isPremiumEdition,
          versionName
        } = this.props;
        return /*#__PURE__*/React__default['default'].createElement(StyleRoot, {
          isSpeaking: isSpeaking,
          isPremiumEdition: isPremiumEdition,
          versionName: versionName
        }, React__default['default'].Children.only(this.props.children));
      }

    }, _defineProperty(_class2, "defaultProps", {
      isSpeaking: false,
      isPremiumEdition: false,
      versionName: null
    }), _defineProperty(_class2, "propTypes", {
      isSpeaking: PropTypes__default['default'].bool.isRequired,
      isPremiumEdition: PropTypes__default['default'].bool.isRequired,
      versionName: PropTypes__default['default'].string,
      children: PropTypes__default['default'].element.isRequired
    }), _temp)) || _class);

    var _dec$1, _class$1, _class2$1, _temp$1;

    const mapStateToProps$1 = state => {
      return {
        systemType: state.shared.metadata.systemType
      };
    };

    const mapDispatchToProps$1 = (
    /* eslint-disable no-unused-vars */
    dispatch
    /* eslint-enable no-unused-vars */
    ) => {
      return {};
    };

    let Providers = (_dec$1 = reactRedux.connect(mapStateToProps$1, mapDispatchToProps$1), _dec$1(_class$1 = (_temp$1 = _class2$1 = class Providers extends React__default['default'].PureComponent {
      render() {
        const {
          broadcaster,
          configuration,
          styletron,
          translator,
          systemType
        } = this.props;
        return /*#__PURE__*/React__default['default'].createElement(ConfigurationProvider, {
          configuration: configuration,
          systemType: systemType
        }, /*#__PURE__*/React__default['default'].createElement(TranslationProvider, {
          translator: translator
        }, /*#__PURE__*/React__default['default'].createElement(BroadcasterProvider, {
          broadcaster: broadcaster
        }, /*#__PURE__*/React__default['default'].createElement(StyletronProvider, {
          styletron: styletron
        }, /*#__PURE__*/React__default['default'].createElement(StateRoot, null, React__default['default'].Children.only(this.props.children))))));
      }

    }, _defineProperty(_class2$1, "propTypes", {
      broadcaster: PropTypes__default['default'].object.isRequired,
      children: PropTypes__default['default'].element.isRequired,
      configuration: PropTypes__default['default'].object.isRequired,
      styletron: PropTypes__default['default'].object.isRequired,
      translator: PropTypes__default['default'].object.isRequired,
      systemType: PropTypes__default['default'].string.isRequired
    }), _temp$1)) || _class$1);

    class Root extends React__default['default'].PureComponent {
      render() {
        const {
          broadcaster,
          configuration,
          store,
          styletron,
          translator
        } = this.props;
        return /*#__PURE__*/React__default['default'].createElement(ErrorBoundary, null, /*#__PURE__*/React__default['default'].createElement(reactRedux.Provider, {
          store: store
        }, /*#__PURE__*/React__default['default'].createElement(Providers, {
          configuration: configuration,
          translator: translator,
          broadcaster: broadcaster,
          styletron: styletron
        }, React__default['default'].Children.only(this.props.children))));
      }

    }

    _defineProperty(Root, "propTypes", {
      store: PropTypes__default['default'].object.isRequired,
      configuration: PropTypes__default['default'].object.isRequired,
      translator: PropTypes__default['default'].object.isRequired,
      broadcaster: PropTypes__default['default'].object.isRequired,
      styletron: PropTypes__default['default'].object.isRequired,
      children: PropTypes__default['default'].element.isRequired
    });

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    const getRoot = (store, translator, configuration, styletron, broadcaster, ChildComponent) => promiseTry(() => {
      const root = /*#__PURE__*/React__default['default'].createElement(Root, {
        store: store,
        configuration: configuration,
        translator: translator,
        broadcaster: broadcaster,
        styletron: styletron
      }, /*#__PURE__*/React__default['default'].createElement(ChildComponent, null));
      return root;
    });

    const autoRoot = (initialState, rootReducer, ChildComponent) => promiseTry(() => {
      const storageProvider = new WebExtensionEnvironmentStorageProvider();
      const storageManager = new StorageManager(storageProvider);
      const settingsManager = new SettingsManager(storageManager);
      const manifestProvider = new WebExtensionEnvironmentManifestProvider();
      const metadataManager = new MetadataManager(manifestProvider, settingsManager);
      const configuration = new Configuration(metadataManager, configurationObject);
      const localeProvider = new WebExtensionEnvironmentLocaleProvider();
      const translatorProvider = new WebExtensionEnvironmentTranslatorProvider(localeProvider);
      const broadcasterProvider = new WebExtensionEnvironmentBroadcasterProvider();
      const talkieLocaleHelper = new TalkieLocaleHelper();
      const api = new Api(metadataManager, configuration, translatorProvider, broadcasterProvider, talkieLocaleHelper);
      const reduxStoreProvider = new ReduxStoreProvider();
      const store = reduxStoreProvider.createStore(initialState, rootReducer, api);
      const styletronProvider = new WebExtensionEnvironmentTranslator();
      /* eslint-disable no-sync */

      const styletron = styletronProvider.getSync();
      /* eslint-enable no-sync */
      // TODO: create a generic static/default/render-time preloaded state action provider attached to the component hierarchy?

      return getRoot(store, translatorProvider, configuration, styletron, broadcasterProvider, ChildComponent).then(root => ({
        localeProvider,
        root,
        store,
        styletron
      }));
    });

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const SET_IS_PREMIUM_EDITION = "SET_IS_PREMIUM_EDITION";
    const SET_VERSION_NAME = "SET_VERSION_NAME";
    const SET_VERSION_NUMBER = "SET_VERSION_NUMBER";
    const SET_EDITION_TYPE = "SET_EDITION_TYPE";
    const SET_SYSTEM_TYPE = "SET_SYSTEM_TYPE";
    const SET_OS_TYPE = "SET_OS_TYPE";

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    /*eslint no-unused-vars: ["warn", { "args": "after-used" }] */

    const loadIsPremiumEdition = () => (dispatch, getState, api) => api.isPremiumEdition().then(isPremiumEdition => dispatch(setIsPremium(isPremiumEdition)));
    const storeIsPremiumEdition = isPremiumEdition => (dispatch, getState, api) => api.setIsPremiumEditionOption(isPremiumEdition).then(() => dispatch(loadIsPremiumEdition()));
    const setIsPremium = isPremiumEdition => {
      return {
        type: SET_IS_PREMIUM_EDITION,
        isPremiumEdition
      };
    };
    const loadVersionName = () => (dispatch, getState, api) => api.getVersionName().then(versionName => dispatch(setVersionName(versionName)));
    const setVersionName = versionName => {
      return {
        type: SET_VERSION_NAME,
        versionName
      };
    };
    const loadVersionNumber = () => (dispatch, getState, api) => api.getVersionNumber().then(versionNumber => dispatch(setVersionNumber(versionNumber)));
    const setVersionNumber = versionNumber => {
      return {
        type: SET_VERSION_NUMBER,
        versionNumber
      };
    };
    const loadEditionType = () => (dispatch, getState, api) => api.getEditionType().then(editionType => dispatch(setEditionType(editionType)));
    const setEditionType = editionType => {
      return {
        type: SET_EDITION_TYPE,
        editionType
      };
    };
    const loadSystemType = () => (dispatch, getState, api) => api.getSystemType().then(systemType => dispatch(setSystemType(systemType)));
    const setSystemType = systemType => {
      return {
        type: SET_SYSTEM_TYPE,
        systemType
      };
    };
    const loadOsType = () => (dispatch, getState, api) => api.getOsType().then(osType => dispatch(setOsType(osType)));
    const setOsType = osType => {
      return {
        type: SET_OS_TYPE,
        osType
      };
    };

    var metadata = /*#__PURE__*/Object.freeze({
        __proto__: null,
        loadIsPremiumEdition: loadIsPremiumEdition,
        storeIsPremiumEdition: storeIsPremiumEdition,
        setIsPremium: setIsPremium,
        loadVersionName: loadVersionName,
        setVersionName: setVersionName,
        loadVersionNumber: loadVersionNumber,
        setVersionNumber: setVersionNumber,
        loadEditionType: loadEditionType,
        setEditionType: setEditionType,
        loadSystemType: loadSystemType,
        setSystemType: setSystemType,
        loadOsType: loadOsType,
        setOsType: setOsType
    });

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    // import * as actionTypes from "../constants/action-types-navigation";

    /*eslint no-unused-vars: ["warn", { "args": "after-used" }] */
    const openUrlInNewTab$1 = url => (dispatch, getState, api) => api.openUrlInNewTab(url);
    const openShortKeysConfiguration$1 = () => (dispatch, getState, api) => api.openShortKeysConfiguration();
    const openOptionsPage$1 = () => (dispatch, getState, api) => api.openOptionsPage();

    var navigation = /*#__PURE__*/Object.freeze({
        __proto__: null,
        openUrlInNewTab: openUrlInNewTab$1,
        openShortKeysConfiguration: openShortKeysConfiguration$1,
        openOptionsPage: openOptionsPage$1
    });

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const SET_MIN = "SET_MIN";
    const SET_CURRENT = "SET_CURRENT";
    const SET_MAX = "SET_MAX";

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    /*eslint no-unused-vars: ["warn", { "args": "after-used" }] */

    const setMin = min => {
      return {
        type: SET_MIN,
        min
      };
    };
    const setCurrent = current => {
      return {
        type: SET_CURRENT,
        current
      };
    };
    const setMax = max => {
      return {
        type: SET_MAX,
        max
      };
    };

    var progress = /*#__PURE__*/Object.freeze({
        __proto__: null,
        setMin: setMin,
        setCurrent: setCurrent,
        setMax: setMax
    });

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const SET_IS_SPEAKING = "SET_IS_SPEAKING";

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    /*eslint no-unused-vars: ["warn", { "args": "after-used" }] */

    const iconClick = () => (dispatch, getState, api) => api.iconClick();
    const setIsSpeaking = isSpeaking => {
      return {
        type: SET_IS_SPEAKING,
        isSpeaking
      };
    };
    const speakTextInLanguageWithOverrides = (text, languageCode) => (dispatch, getState, api) => api.debouncedSpeakTextInLanguageWithOverrides(text, languageCode);

    var speaking = /*#__PURE__*/Object.freeze({
        __proto__: null,
        iconClick: iconClick,
        setIsSpeaking: setIsSpeaking,
        speakTextInLanguageWithOverrides: speakTextInLanguageWithOverrides
    });

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const SET_VOICES = "SET_VOICES";
    const SET_SPEAK_LONG_TEXTS = "SET_SPEAK_LONG_TEXTS";
    const SET_NAVIGATOR_LANGUAGE = "SET_NAVIGATOR_LANGUAGE";
    const SET_NAVIGATOR_LANGUAGES = "SET_NAVIGATOR_LANGUAGES";
    const SET_TRANSLATED_LANGUAGES = "SET_TRANSLATED_LANGUAGES";

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    /*eslint no-unused-vars: ["warn", { "args": "after-used" }] */

    const loadVoices = () => (dispatch, getState, api) => api.getVoices().then(voices => dispatch(setVoices(voices)));
    const setVoices = voices => {
      return {
        type: SET_VOICES,
        voices
      };
    };
    const loadSpeakLongTexts = () => (dispatch, getState, api) => api.getSpeakLongTextsOption().then(speakLongTexts => dispatch(setSpeakLongTexts(speakLongTexts)));
    const storeSpeakLongTexts = speakLongTexts => (dispatch, getState, api) => api.setSpeakLongTextsOption(speakLongTexts).then(() => dispatch(loadSpeakLongTexts()));
    const setSpeakLongTexts = speakLongTexts => {
      return {
        type: SET_SPEAK_LONG_TEXTS,
        speakLongTexts
      };
    };

    function getNavigatorLanguage() {
      let lang = null;

      try {
        lang = navigator.language;
      } catch (error) {// NOTE: swallowing errors.
      }

      return lang;
    }

    function getNavigatorLanguages() {
      let langs = null;

      try {
        langs = navigator.languages;
      } catch (error) {
        // NOTE: swallowing errors.
        langs = [];
      }

      return langs;
    }
    const loadNavigatorLanguage = () => dispatch => Promise.resolve().then(() => getNavigatorLanguage()).then(navigatorLanguage => dispatch(setNavigatorLanguage(navigatorLanguage)));
    const setNavigatorLanguage = navigatorLanguage => {
      return {
        type: SET_NAVIGATOR_LANGUAGE,
        navigatorLanguage
      };
    };
    const loadNavigatorLanguages = () => dispatch => Promise.resolve().then(() => getNavigatorLanguages()).then(navigatorLanguages => dispatch(setNavigatorLanguages(navigatorLanguages)));
    const setNavigatorLanguages = navigatorLanguages => {
      return {
        type: SET_NAVIGATOR_LANGUAGES,
        navigatorLanguages
      };
    };
    const loadTranslatedLanguages = () => (dispatch, getState, api) => api.getTranslatedLanguages().then(translatedLanguages => dispatch(setTranslatedLanguages(translatedLanguages)));
    const setTranslatedLanguages = translatedLanguages => {
      return {
        type: SET_TRANSLATED_LANGUAGES,
        translatedLanguages
      };
    };
    const speak = (text, voice) => (dispatch, getState, api) => api.debouncedSpeak(text, voice);

    var voices = /*#__PURE__*/Object.freeze({
        __proto__: null,
        loadVoices: loadVoices,
        setVoices: setVoices,
        loadSpeakLongTexts: loadSpeakLongTexts,
        storeSpeakLongTexts: storeSpeakLongTexts,
        setSpeakLongTexts: setSpeakLongTexts,
        loadNavigatorLanguage: loadNavigatorLanguage,
        setNavigatorLanguage: setNavigatorLanguage,
        loadNavigatorLanguages: loadNavigatorLanguages,
        setNavigatorLanguages: setNavigatorLanguages,
        loadTranslatedLanguages: loadTranslatedLanguages,
        setTranslatedLanguages: setTranslatedLanguages,
        speak: speak
    });

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    var shared$1 = {
      metadata,
      navigation,
      progress,
      speaking,
      voices
    };

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    const getPrerenderActionsToDispatch = prerenderedActionsToDispatch => {
      const styleRootActionsToDispatch = [];
      const allActionsToDispatch = [].concat(styleRootActionsToDispatch).concat(prerenderedActionsToDispatch);
      return allActionsToDispatch;
    };

    const getPostrenderActionsToDispatch = postrenderActionsToDispatch => {
      // TODO: simplify.
      const clientSideActionsToDispatch = [// NOTE: don't want to keep track of when to load these, preemptively loading.
      shared$1.metadata.loadIsPremiumEdition(), shared$1.metadata.loadOsType()];
      const allActionsToDispatch = [].concat(clientSideActionsToDispatch).concat(postrenderActionsToDispatch);
      return allActionsToDispatch;
    };

    const renderHtml = root => promiseTry(() => {
      const rootElement = document.getElementById("react-root");
      ReactDOM__default['default'].hydrate(root, rootElement);
    });

    const hydrateHtml = (rootReducer, customPrerenderedActionsToDispatch, customPostrenderActionsToDispatch, ChildComponent) => promiseTry(() => {
      // NOTE: use preloaded state from the pre-rendered html.
      const prerenderedState = JSON.parse(document.getElementById("__PRERENDERED_STATE__").textContent);
      const prerenderedActionsToDispatch = getPrerenderActionsToDispatch(customPrerenderedActionsToDispatch);
      const postrenderActionsToDispatch = getPostrenderActionsToDispatch(customPostrenderActionsToDispatch);
      return autoRoot(prerenderedState, rootReducer, ChildComponent).then(({
        root,
        store
      }) => dispatchAll(store, prerenderedActionsToDispatch).then(() => renderHtml(root)).then(() => dispatchAll(store, postrenderActionsToDispatch)));
    });

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    // TODO: use library?
    const jsonClone = obj => {
      const json = JSON.stringify(obj);
      const restored = JSON.parse(json);
      return restored;
    };

    const assignActionKeyToState = (previousState, action, key) => {
      return _objectSpread2(_objectSpread2({}, previousState), {}, {
        // TODO: replace generic deep clone with faster custom code per key.
        [key]: jsonClone(action[key])
      });
    };

    const assignActionTypeToKey = key => (previousState, action) => {
      return assignActionKeyToState(previousState, action, key);
    };

    const createAssignmentActionMapReducer = (initialState, customActionsMap, assignActionsMap) => {
      return (previousState = initialState, action) => {
        let reduceFn = null;

        if (customActionsMap[action.type]) {
          reduceFn = customActionsMap[action.type];
        } else if (assignActionsMap[action.type]) {
          reduceFn = assignActionTypeToKey(assignActionsMap[action.type]);
        }

        if (!reduceFn) {
          return previousState;
        }

        return reduceFn(previousState, action);
      };
    };

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const initialState = {
      isPremiumEdition: false,
      versionName: null,
      versionNumber: null,
      systemType: null,
      osType: null
    };
    const customActionsMap = {};
    const assignActionsMap = {
      [SET_IS_PREMIUM_EDITION]: "isPremiumEdition",
      [SET_VERSION_NAME]: "versionName",
      [SET_VERSION_NUMBER]: "versionNumber",
      [SET_EDITION_TYPE]: "editionType",
      [SET_SYSTEM_TYPE]: "systemType",
      [SET_OS_TYPE]: "osType"
    };
    var metadata$1 = createAssignmentActionMapReducer(initialState, customActionsMap, assignActionsMap);

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    const initialState$1 = {// xxx: null,
    };
    const customActionsMap$1 = {};
    const assignActionsMap$1 = {// [actionTypes.SET_XXX]: "xxx",
    };
    var navigation$1 = createAssignmentActionMapReducer(initialState$1, customActionsMap$1, assignActionsMap$1);

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const initialState$2 = {
      min: 0,
      current: 0,
      max: 0
    };
    const customActionsMap$2 = {};
    const assignActionsMap$2 = {
      [SET_MIN]: "min",
      [SET_CURRENT]: "current",
      [SET_MAX]: "max"
    };
    var progress$1 = createAssignmentActionMapReducer(initialState$2, customActionsMap$2, assignActionsMap$2);

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const initialState$3 = {
      isSpeaking: false
    };
    const customActionsMap$3 = {};
    const assignActionsMap$3 = {
      [SET_IS_SPEAKING]: "isSpeaking"
    };
    var speaking$1 = createAssignmentActionMapReducer(initialState$3, customActionsMap$3, assignActionsMap$3);

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const initialState$4 = {
      voices: [],
      speakLongTexts: false,
      navigatorLanguage: null,
      navigatorLanguages: [],
      translatedLanguages: []
    };
    const customActionsMap$4 = {};
    const assignActionsMap$4 = {
      [SET_SPEAK_LONG_TEXTS]: "speakLongTexts",
      [SET_VOICES]: "voices",
      [SET_NAVIGATOR_LANGUAGE]: "navigatorLanguage",
      [SET_NAVIGATOR_LANGUAGES]: "navigatorLanguages",
      [SET_TRANSLATED_LANGUAGES]: "translatedLanguages"
    };
    var voices$1 = createAssignmentActionMapReducer(initialState$4, customActionsMap$4, assignActionsMap$4);

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    var shared$2 = redux.combineReducers({
      metadata: metadata$1,
      navigation: navigation$1,
      progress: progress$1,
      speaking: speaking$1,
      voices: voices$1
    });

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const SET_ACTIVE_TAB_ID = "SET_ACTIVE_TAB_ID";

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const initialState$5 = {
      activeTabId: null
    };
    const customActionsMap$5 = {};
    const assignActionsMap$5 = {
      [SET_ACTIVE_TAB_ID]: "activeTabId"
    };
    var navigation$2 = createAssignmentActionMapReducer(initialState$5, customActionsMap$5, assignActionsMap$5);

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    var unshared = redux.combineReducers({
      navigation: navigation$2
    });

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    var rootReducer = redux.combineReducers({
      shared: shared$2,
      unshared
    });

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    /*eslint no-unused-vars: ["warn", { "args": "after-used" }] */

    const setActiveTabId = activeTabId => {
      return {
        type: SET_ACTIVE_TAB_ID,
        activeTabId
      };
    };

    var navigation$3 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        setActiveTabId: setActiveTabId
    });

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    var actionCreators = {
      navigation: navigation$3
    };

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    var actions = {
      shared: shared$1,
      unshared: actionCreators
    };

    function translateHoc(ComponentToWrap) {
      var _class, _temp;

      return _temp = _class = class TranslationHoc extends React__default['default'].PureComponent {
        render() {
          return /*#__PURE__*/React__default['default'].createElement(ComponentToWrap, _extends({}, this.props, {
            translate: this.context.translate
          }));
        }

      }, _defineProperty(_class, "contextTypes", {
        translate: PropTypes__default['default'].func.isRequired
      }), _temp;
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    function styledHoc(styles) {
      return ComponentToWrap => styled(ComponentToWrap, styles);
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const knownEvents = {
      beforeSpeaking: "beforeSpeaking",
      stopSpeaking: "stopSpeaking",
      afterSpeaking: "afterSpeaking",
      beforeSpeakingPart: "beforeSpeakingPart",
      afterSpeakingPart: "afterSpeakingPart",
      updateProgress: "updateProgress",
      resetProgress: "resetProgress",
      addProgress: "addProgress",
      finishProgress: "finishProgress",
      passSelectedTextToBackground: "passSelectedTextToBackground"
    };

    //
    //const dualLogger = new DualLogger("status-container.jsx");

    function passSelectedTextToBackgroundHoc(ComponentToWrap) {
      var _class, _temp;

      return _temp = _class = class PassSelectedTextToBackgroundHoc extends React__default['default'].Component {
        constructor(props) {
          super(props);
          this.componentCleanup = this.componentCleanup.bind(this);
          this.handleFocus = this.handleFocus.bind(this);
          this.enable = this.enable.bind(this);
          this.disable = this.disable.bind(this);
          this.getSelectedTextWithFocusTimestamp = this.getSelectedTextWithFocusTimestamp.bind(this);
          this.isListeningToBroadcasts = false;
          this.killSwitches = [];
          this.mostRecentUse = 0;
        }

        componentDidMount() {
          window.addEventListener("beforeunload", this.componentCleanup);
          window.addEventListener("focus", this.handleFocus);
          this.gotFocus();
          this.enable();
        }

        componentWillUnmount() {
          this.componentCleanup();
        }

        shouldComponentUpdate(nextProps, nextState) {
          // NOTE: always update.
          // TODO: optimize by comparing old and new props/state.
          return this.isListeningToBroadcasts;
        }

        componentCleanup() {
          window.removeEventListener("beforeunload", this.componentCleanup);
          window.removeEventListener("focus", this.handleFocus);
          this.disable();
        }

        handleFocus() {
          this.gotFocus();
        }

        gotFocus() {
          this.mostRecentUse = Date.now();
        }

        enable() {
          // TODO: properly avoid race conditions when enabling/disabling.
          if (this.isListeningToBroadcasts) {
            return;
          }

          this.registerBroadcastListeners();
          this.isListeningToBroadcasts = true;
        }

        disable() {
          // TODO: properly avoid race conditions when enabling/disabling.
          if (!this.isListeningToBroadcasts) {
            return;
          }

          this.isListeningToBroadcasts = false;
          this.executeKillSwitches();
        }

        render() {
          return /*#__PURE__*/React__default['default'].createElement(ComponentToWrap, this.props);
        }

        getSelectedTextWithFocusTimestamp() {
          if (!this.isListeningToBroadcasts) {
            return null;
          } // NOTE: duplicated elsewhere in the codebase.

          /* eslint-disable no-inner-declarations */


          const executeGetFramesSelectionTextAndLanguageCode = function () {
            try {
              function talkieGetParentElementLanguages(element) {
                return [].concat((element || null) && element.getAttribute && element.getAttribute("lang")).concat((element || null) && element.parentElement && talkieGetParentElementLanguages(element.parentElement));
              }

              ;
              const talkieSelectionData = {
                text: (document || null) && (document.getSelection || null) && (document.getSelection() || null) && document.getSelection().toString(),
                htmlTagLanguage: (document || null) && (document.getElementsByTagName || null) && (document.getElementsByTagName("html") || null) && (document.getElementsByTagName("html").length > 0 || null) && (document.getElementsByTagName("html")[0].getAttribute("lang") || null),
                parentElementsLanguages: talkieGetParentElementLanguages((document || null) && (document.getSelection || null) && (document.getSelection() || null) && (document.getSelection().rangeCount > 0 || null) && (document.getSelection().getRangeAt || null) && (document.getSelection().getRangeAt(0) || null) && (document.getSelection().getRangeAt(0).startContainer || null))
              };
              return talkieSelectionData;
            } catch (error) {
              return null;
            }
          }();
          /* eslint-enable no-inner-declarations */


          const selectedTextWithFocusTimestamp = {
            mostRecentUse: this.mostRecentUse,
            selectionTextAndLanguageCode: executeGetFramesSelectionTextAndLanguageCode
          };
          return selectedTextWithFocusTimestamp;
        }

        executeKillSwitches() {
          // NOTE: expected to have only synchronous methods for the relevant parts.
          const killSwitchesToExecute = this.killSwitches;
          this.killSwitches = [];
          killSwitchesToExecute.forEach(killSwitch => {
            try {
              killSwitch();
            } catch (error) {
            }
          });
        }

        registerBroadcastListeners() {
          return promiseTry(() => {
            return Promise.all([
            /* eslint-disable no-unused-vars */
            this.context.broadcaster.registerListeningAction(knownEvents.passSelectedTextToBackground, (actionName, actionData) => this.getSelectedTextWithFocusTimestamp(actionName, actionData)).then(killSwitch => this.killSwitches.push(killSwitch))
            /* eslint-enable no-unused-vars */
            ]);
          });
        }

      }, _defineProperty(_class, "contextTypes", {
        broadcaster: PropTypes__default['default'].object.isRequired
      }), _temp;
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const layoutWithMargins = {
      marginTop: "0.5em",
      marginBottom: "0.5em",
      marginLeft: "1em",
      marginRight: "1em",
      fontSize: "0.75em"
    };
    const header = layoutWithMargins;
    const main = layoutWithMargins;
    const nav = layoutWithMargins;
    const footer = layoutWithMargins;
    const hr = Object.assign({}, layoutWithMargins, {
      color: "#dddddd",
      borderStyle: "solid",
      borderWidth: 0,
      borderTopWidth: "1px"
    });
    const details = {};
    const summary = {
      cursor: "pointer"
    };
    const hero = {
      borderRadius: "0.4em",
      fontSize: "2em",
      paddingLeft: "1em",
      paddingRight: "1em",
      paddingTop: "1em",
      paddingBottom: "1em",
      marginLeft: "2em",
      marginRight: "2em",
      marginTop: "2em",
      marginBottom: "2em",
      backgroundColor: "#eceff5"
    };

    /*
    This file is part of Talkie -- button-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const header$1 = styledHoc(header)("header");
    const main$1 = styledHoc(main)("main");
    const nav$1 = styledHoc(nav)("nav");
    const footer$1 = styledHoc(footer)("footer");
    const hr$1 = styledHoc(hr)("hr");
    const details$1 = styledHoc(details)("details");
    const summary$1 = styledHoc(summary)("summary");
    const hero$1 = styledHoc(hero)("div");

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    class WebExtensionEnvironmentDynamicEnvironment {
      isNode() {
        const isNode = false;
        return isNode;
      }

      isWebExtension() {
        const isWebExtension = true;
        return isWebExtension;
      }

    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const textColor = "#000000";
    const text = {
      color: textColor
    };
    const linkHighlightColor = "#3497ff";
    const highlight = {
      color: linkHighlightColor
    };
    const a = {
      color: textColor,
      ":focus": highlight,
      ":hover": highlight,
      ":active": highlight
    };
    const h1 = {};
    const h2 = {};
    const h3 = {};
    const h4 = {};
    const h5 = {};
    const kbd = {
      display: "inline-block",
      color: "#555",
      verticalAlign: "middle",
      paddingLeft: "0.3em",
      paddingRight: "0.3em",
      paddingTop: "0.1em",
      paddingBottom: "0.1em",
      backgroundColor: "#fcfcfc",
      border: "solid 1px #ccc",
      borderBottomColor: "#bbb",
      borderRadius: "0.3em",
      boxShadow: "inset 0 -1px 0 #bbb",
      fontFamily: "Helvetica, Verdana, sans-serif"
    };
    const blockquote = {
      background: "#fafafa",
      borderLeft: "0.5em solid #cccccc",
      marginLeft: "0.5em",
      marginTop: "1em",
      marginBottom: "1em",
      paddingLeft: "0.5em",
      paddingRight: "0.5em",
      paddingTop: "0.5em",
      paddingBottom: "0.5em"
    };

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const p = styledHoc(text)("p");
    const span = styledHoc(text)("span");
    const a$1 = styledHoc(a)("a");
    const h1$1 = styledHoc(h1)("h1");
    const h2$1 = styledHoc(h2)("h2");
    const h3$1 = styledHoc(h3)("h3");
    const h4$1 = styledHoc(h4)("h4");
    const h5$1 = styledHoc(h5)("h5");
    const kbd$1 = styledHoc(kbd)("kbd");
    const blockquote$1 = styledHoc(blockquote)("blockquote");

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const table = {
      width: "100%"
    };
    const tbody = {};
    const th = {};
    const tr = {};
    const td = {
      paddingLeft: "0.5em",
      paddingRight: "0.5em",
      paddingTop: "0.5em",
      paddingBottom: "0.5em"
    };

    /*
    This file is part of Talkie -- button-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const table$1 = styledHoc(table)("table");
    const tbody$1 = styledHoc(tbody)("tbody");
    const tr$1 = styledHoc(tr)("tr");
    const th$1 = styledHoc(th)("th");
    const td$1 = styledHoc(td)("td");

    class Nav extends React__default['default'].PureComponent {
      constructor(props) {
        super(props);
        this.handleClick = this.handleClick.bind(this);
        this.styled = {
          nav: styledHoc({
            lineHeight: "1.5em",
            textAlign: "center"
          })(nav$1),
          navTable: styledHoc({
            lineHeight: "1.5em",
            textAlign: "center"
          })(table$1),
          navTableTd: styledHoc({
            marginLeft: 0,
            marginRight: 0,
            marginTop: 0,
            marginBottom: 0,
            paddingLeft: 0,
            paddingRight: 0,
            paddingTop: 0,
            paddingBottom: 0
          })(td$1),
          selectedLink: styledHoc({
            color: "#3497ff",
            fontWeight: "bold"
          })(a$1)
        };
      }

      handleClick(e) {
        if (e.target.tagName === "A") {
          const href = e.target.getAttribute("href");

          if (typeof href === "string" && href.startsWith("#")) {
            const tabId = href.replace("#", "");
            e.preventDefault();
            e.stopPropagation();
            this.props.onTabChange(tabId);
            return false;
          } // TODO: warn about mismatched link style?

        }
      }

      render() {
        const {
          links,
          initialActiveTabId
        } = this.props;
        const linkCells = links.map(link => {
          const SelectedLinkType = initialActiveTabId === link.tabId ? this.styled.selectedLink : a$1;
          const url = link.url || "#" + link.tabId;
          return /*#__PURE__*/React__default['default'].createElement(this.styled.navTableTd, {
            key: link.tabId,
            onClick: this.handleClick
          }, /*#__PURE__*/React__default['default'].createElement(SelectedLinkType, {
            href: url
          }, link.text));
        });
        const colCount = linkCells.length;
        const colWidth = `${100 / linkCells.length}%`;
        return /*#__PURE__*/React__default['default'].createElement(this.styled.nav, {
          className: "columns"
        }, /*#__PURE__*/React__default['default'].createElement(this.styled.navTable, null, /*#__PURE__*/React__default['default'].createElement("colgroup", null, /*#__PURE__*/React__default['default'].createElement("col", {
          width: colWidth,
          colSpan: colCount
        })), /*#__PURE__*/React__default['default'].createElement(tbody$1, null, /*#__PURE__*/React__default['default'].createElement(tr$1, null, linkCells))));
      }

    }

    _defineProperty(Nav, "propTypes", {
      initialActiveTabId: PropTypes__default['default'].string.isRequired,
      onTabChange: PropTypes__default['default'].func.isRequired,
      links: PropTypes__default['default'].arrayOf(PropTypes__default['default'].shape({
        url: PropTypes__default['default'].string,
        tabId: PropTypes__default['default'].string,
        text: PropTypes__default['default'].string.isRequired
      })).isRequired
    });

    var _dec$2, _class$2, _class2$2, _temp$2;
    const dynamicEnvironment = new WebExtensionEnvironmentDynamicEnvironment();

    const mapStateToProps$2 = state => {
      return {
        activeTabId: state.unshared.navigation.activeTabId
      };
    };

    const mapDispatchToProps$2 = dispatch => {
      return {
        actions: redux.bindActionCreators(actionCreators.navigation, dispatch)
      };
    };

    let NavContainer = (_dec$2 = reactRedux.connect(mapStateToProps$2, mapDispatchToProps$2), _dec$2(_class$2 = (_temp$2 = _class2$2 = class NavContainer extends React__default['default'].PureComponent {
      constructor(props) {
        super(props);
        this.handleTabChange = this.handleTabChange.bind(this);
      }

      getLocationHash() {
        let locationHash = null;

        if (dynamicEnvironment.isWebExtension() && document.location && typeof document.location.hash === "string" && document.location.hash.length > 0) {
          locationHash = "#" + decodeURIComponent(document.location.hash.replace("#", ""));
        }

        return locationHash;
      }

      setLocationHash(locationHash) {
        // HACK: don't do browser detection.
        // https://stackoverflow.com/questions/41819284/how-to-determine-in-which-browser-your-extension-background-script-is-executing
        // https://stackoverflow.com/a/41820692
        const isFirefox = typeof InstallTrigger !== "undefined"; // NOTE: this breaks navigation in the options page in the preferences pane in firefox, so skip.
        // TODO: don't disable on other pages (demo, options in normal tab) in firefox.

        if (!isFirefox) {
          document.location.hash = locationHash;
        }
      }

      componentDidMount() {
        const locationHash = this.getLocationHash();

        if (typeof locationHash === "string") {
          const activeTabId = locationHash.replace("#", "");
          this.props.actions.setActiveTabId(activeTabId);
        }
      }

      handleTabChange(activeTabId) {
        this.props.actions.setActiveTabId(activeTabId);
        this.setLocationHash(activeTabId);
      }

      render() {
        const {
          activeTabId,
          links
        } = this.props;
        return /*#__PURE__*/React__default['default'].createElement(Nav, {
          initialActiveTabId: activeTabId,
          onTabChange: this.handleTabChange,
          links: links
        });
      }

    }, _defineProperty(_class2$2, "propTypes", {
      actions: PropTypes__default['default'].object.isRequired,
      activeTabId: PropTypes__default['default'].string.isRequired,
      links: PropTypes__default['default'].arrayOf(PropTypes__default['default'].shape({
        url: PropTypes__default['default'].string,
        tabId: PropTypes__default['default'].string,
        text: PropTypes__default['default'].string.isRequired
      })).isRequired
    }), _temp$2)) || _class$2);

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    function handleBubbledLinkClick(handleUrl, e) {
      let element = e.target;

      do {
        if (element.tagName === "A") {
          const href = element.getAttribute("href");

          if (typeof href === "string" && href.startsWith("https://")) {
            e.preventDefault();
            e.stopPropagation();
            handleUrl(href);
            return false;
          } // TODO: warn about mismatched link style?

        }

        element = element.parentElement;
      } while (element);
    }

    class TabContents extends React__default['default'].PureComponent {
      constructor(props) {
        super(props);
        this.handleClick = this.handleClick.bind(this);
      }

      handleClick(e) {
        // TODO: use an api call which has handleBubbledLinkClick?
        return handleBubbledLinkClick(this.props.onLinkClick, e);
      }

      render() {
        if (this.props.id !== this.props.activeTabId) {
          return null;
        }

        return /*#__PURE__*/React__default['default'].createElement("div", {
          id: this.props.id,
          onClick: this.handleClick
        }, this.props.children);
      }

    }

    _defineProperty(TabContents, "propTypes", {
      id: PropTypes__default['default'].string.isRequired,
      activeTabId: PropTypes__default['default'].string.isRequired,
      children: PropTypes__default['default'].node.isRequired,
      onLinkClick: PropTypes__default['default'].func.isRequired
    });

    function configureHoc(ComponentToWrap) {
      var _class, _temp;

      return _temp = _class = class ConfigurationHoc extends React__default['default'].PureComponent {
        render() {
          return /*#__PURE__*/React__default['default'].createElement(ComponentToWrap, _extends({}, this.props, {
            configure: this.context.configure,
            onConfigurationChange: this.context.onConfigurationChange
          }));
        }

      }, _defineProperty(_class, "contextTypes", {
        configure: PropTypes__default['default'].func.isRequired,
        onConfigurationChange: PropTypes__default['default'].func.isRequired
      }), _temp;
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const hover = {
      color: "#ffffff"
    };
    const focus = {
      backgroundImage: "linear-gradient(to bottom, #087eff, #087eff)",
      color: "#ffffff"
    };
    const button = {
      borderRadius: "0.3em",
      backgroundImage: "linear-gradient(to bottom, #6bb3fa, #087eff)",
      color: "#ffffff",
      paddingTop: "0.3em",
      paddingBottom: "0.3em",
      paddingLeft: "0.75em",
      paddingRight: "0.75em",
      textDecoration: "none",
      textAlign: "center",
      ":hover": hover,
      ":focus": focus
    };
    const a$2 = button;

    /*
    This file is part of Talkie -- button-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const button$1 = styledHoc(button)("button");
    const a$3 = styledHoc(a$2)("a");

    class Discretional extends React__default['default'].PureComponent {
      render() {
        const {
          enabled,
          children
        } = this.props;

        if (enabled) {
          return children;
        }

        return null;
      }

    }

    _defineProperty(Discretional, "defaultProps", {
      enabled: false
    });

    _defineProperty(Discretional, "propTypes", {
      enabled: PropTypes__default['default'].bool.isRequired,
      children: PropTypes__default['default'].node.isRequired
    });

    class TalkieEditionIcon extends React__default['default'].PureComponent {
      render() {
        const {
          mode,
          size,
          marginLeft,
          marginRight,
          className
        } = this.props;
        const iconStyle = {
          ":before": {
            content: "''",
            backgroundRepeat: "no-repeat",
            backgroundPosition: "center",
            backgroundSize: "contain",
            display: "inline-block"
          }
        };

        if (mode === "inline") {
          iconStyle[":before"].verticalAlign = "sub";
        }

        iconStyle[":before"].width = size;
        iconStyle[":before"].height = size;
        iconStyle[":before"].marginLeft = marginLeft;
        iconStyle[":before"].marginRight = marginRight;
        const StyledIcon = styledHoc(iconStyle)("span"); // TODO: fully replace with css-in-js?

        return /*#__PURE__*/React__default['default'].createElement(StyledIcon, {
          className: `icon icon-${mode} icon-${size} ${className}`
        });
      }

    }

    _defineProperty(TalkieEditionIcon, "defaultProps", {
      mode: "inline",
      size: "1.3em",
      marginLeft: "0.3em",
      marginRight: "0.3em",
      className: ""
    });

    _defineProperty(TalkieEditionIcon, "propTypes", {
      mode: PropTypes__default['default'].oneOf(["inline", "standalone"]).isRequired,
      size: PropTypes__default['default'].string.isRequired,
      marginLeft: PropTypes__default['default'].string.isRequired,
      marginRight: PropTypes__default['default'].string.isRequired,
      className: PropTypes__default['default'].string.isRequired
    });

    class TalkieEditionIcon$1 extends React__default['default'].PureComponent {
      render() {
        const {
          mode,
          size,
          marginLeft,
          marginRight,
          className,
          isPremiumEdition
        } = this.props;
        const isPremiumEditionClassName = isPremiumEdition ? "premium" : "free";
        const classNames = ["icon-talkie", isPremiumEditionClassName, className].join(" ").trim();
        return /*#__PURE__*/React__default['default'].createElement(TalkieEditionIcon, {
          mode: mode,
          size: size,
          marginLeft: marginLeft,
          marginRight: marginRight,
          className: classNames
        });
      }

    }

    _defineProperty(TalkieEditionIcon$1, "defaultProps", {
      mode: "inline",
      size: "1.3em",
      marginLeft: "0.3em",
      marginRight: "0.3em",
      className: "",
      isPremiumEdition: false
    });

    _defineProperty(TalkieEditionIcon$1, "propTypes", {
      mode: PropTypes__default['default'].oneOf(["inline", "standalone"]).isRequired,
      size: PropTypes__default['default'].string.isRequired,
      marginLeft: PropTypes__default['default'].string.isRequired,
      marginRight: PropTypes__default['default'].string.isRequired,
      className: PropTypes__default['default'].string.isRequired,
      isPremiumEdition: PropTypes__default['default'].bool.isRequired
    });

    var _class$3, _class2$3, _temp$3;

    let ExtensionShortName = translateHoc(_class$3 = (_temp$3 = _class2$3 = class ExtensionShortName extends React__default['default'].PureComponent {
      render() {
        const {
          isPremiumEdition,
          translate
        } = this.props; // TODO: move resolving the name to the state, like edition type?

        const extensionShortName = isPremiumEdition ? translate("extensionShortName_Premium") : translate("extensionShortName_Free");
        return /*#__PURE__*/React__default['default'].createElement(span, null, extensionShortName);
      }

    }, _defineProperty(_class2$3, "defaultProps", {
      isPremiumEdition: false
    }), _defineProperty(_class2$3, "propTypes", {
      isPremiumEdition: PropTypes__default['default'].bool.isRequired,
      translate: PropTypes__default['default'].func.isRequired
    }), _temp$3)) || _class$3;

    var _dec$3, _class$4, _class2$4, _temp$4;
    const styles = {};
    let Header = (_dec$3 = styledHoc(styles), configureHoc(_class$4 = translateHoc(_class$4 = _dec$3(_class$4 = (_temp$4 = _class2$4 = class Header extends React__default['default'].PureComponent {
      constructor(props) {
        super(props);
        this.styled = {
          extensionName: styledHoc({
            fontWeight: "bold",
            textDecoration: "none",
            ":focus": {
              outline: 0
            }
          })(a$1),
          button: styledHoc({
            lineHeight: "1.5em",
            // float: __MSG_@@bidi_end_edge__;
            ":focus": {
              outline: 0
            }
          })(a$3)
        };
      }

      componentDidMount() {
        this._unregisterConfigurationListener = this.props.onConfigurationChange(() => this.forceUpdate());
      }

      componentWillUnmount() {
        this._unregisterConfigurationListener();
      }

      render() {
        const {
          className,
          isPremiumEdition,
          translate,
          configure
        } = this.props;
        return /*#__PURE__*/React__default['default'].createElement(header$1, {
          className: className
        }, /*#__PURE__*/React__default['default'].createElement(Discretional, {
          enabled: !isPremiumEdition
        }, /*#__PURE__*/React__default['default'].createElement(this.styled.button, {
          href: configure("urls.options-upgrade-from-demo"),
          id: "header-premium-button",
          lang: "en"
        }, translate("extensionShortName_Premium"))), /*#__PURE__*/React__default['default'].createElement(h1$1, null, /*#__PURE__*/React__default['default'].createElement(TalkieEditionIcon$1, {
          isPremiumEdition: isPremiumEdition
        }), /*#__PURE__*/React__default['default'].createElement(this.styled.extensionName, {
          href: configure("urls.main"),
          lang: "en"
        }, /*#__PURE__*/React__default['default'].createElement(ExtensionShortName, {
          isPremiumEdition: isPremiumEdition
        }))));
      }

    }, _defineProperty(_class2$4, "defaultProps", {
      isPremiumEdition: false
    }), _defineProperty(_class2$4, "propTypes", {
      isPremiumEdition: PropTypes__default['default'].bool.isRequired,
      translate: PropTypes__default['default'].func.isRequired,
      configure: PropTypes__default['default'].func.isRequired,
      className: PropTypes__default['default'].string.isRequired,
      onConfigurationChange: PropTypes__default['default'].func.isRequired
    }), _temp$4)) || _class$4) || _class$4) || _class$4);

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const text$1 = {
      color: "#9b9b9b"
    };
    const highlight$1 = {
      color: "#3497ff",
      textDecoration: "none"
    };
    const a$4 = {
      color: "#9b9b9b",
      textDecoration: "none",
      ":focus": highlight$1,
      ":hover": highlight$1,
      ":active": highlight$1
    };

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const p$1 = styledHoc(text$1)(p);
    const span$1 = styledHoc(text$1)(span);
    const a$5 = styledHoc(a$4)(a$1);

    var _class$5, _class2$5, _temp$5;

    let Footer = configureHoc(_class$5 = (_temp$5 = _class2$5 = class Footer extends React__default['default'].PureComponent {
      constructor(props) {
        super(props);
        this.styled = {
          footer: styledHoc({
            lineHeight: "2em",
            verticalAlign: "middle"
          })(footer$1),
          footerFirstLink: styledHoc({
            fontSize: "1.75em"
          })(a$5),
          footerSecondLink: styledHoc({// float: __MSG_@@bidi_end_edge__;
          })(a$5)
        };
      }

      componentDidMount() {
        this._unregisterConfigurationListener = this.props.onConfigurationChange(() => this.forceUpdate());
      }

      componentWillUnmount() {
        this._unregisterConfigurationListener();
      }

      render() {
        const {
          configure,
          versionNumber,
          optionsPageClick
        } = this.props;
        return /*#__PURE__*/React__default['default'].createElement(this.styled.footer, null, /*#__PURE__*/React__default['default'].createElement(this.styled.footerFirstLink, {
          href: configure("urls.options"),
          rel: "noopener noreferrer",
          target: "_blank",
          onClick: optionsPageClick
        }, /*#__PURE__*/React__default['default'].createElement(TalkieEditionIcon, {
          mode: "standalone",
          size: "0.75em",
          className: "icon-settings"
        })), /*#__PURE__*/React__default['default'].createElement(this.styled.footerSecondLink, {
          href: configure("urls.options-about-from-demo"),
          id: "footer-about-link"
        }, "v", versionNumber));
      }

    }, _defineProperty(_class2$5, "defaultProps", {
      versionNumber: false
    }), _defineProperty(_class2$5, "propTypes", {
      versionNumber: PropTypes__default['default'].string.isRequired,
      optionsPageClick: PropTypes__default['default'].func.isRequired,
      configure: PropTypes__default['default'].func.isRequired,
      onConfigurationChange: PropTypes__default['default'].func.isRequired
    }), _temp$5)) || _class$5;

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const sharedList = {
      marginTop: "0.25em",
      marginBottom: "0.25em",
      paddingLeft: "2em"
    };
    const ul = sharedList;
    const ol = sharedList;
    const dl = sharedList;
    const li = {
      marginBottom: "0.25em"
    };
    const dt = {
      fontWeight: "bold",
      marginTop: "1em"
    };
    const dd = {
      marginBottom: "0.5em"
    };
    const sharedInlineList = {
      listStyleType: "none",
      marginTop: "0.25em",
      marginBottom: "0.25em",
      paddingLeft: 0,
      paddingRight: 0,
      paddingTop: 0,
      paddingBottom: 0
    };
    const inlineUl = sharedInlineList;
    const inlineOl = sharedInlineList;
    const inlineLi = {
      listStyleType: "none",
      paddingLeft: 0,
      paddingRight: 0,
      paddingTop: 0,
      paddingBottom: 0,
      marginLeft: 0,
      marginRight: 0,
      marginTop: 0,
      marginBottom: "0.25em",
      display: "inline-block"
    };

    /*
    This file is part of Talkie -- button-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const ul$1 = styledHoc(ul)("ul");
    const ol$1 = styledHoc(ol)("ol");
    const li$1 = styledHoc(li)("li");
    const dl$1 = styledHoc(dl)("dl");
    const dt$1 = styledHoc(dt)("dt");
    const dd$1 = styledHoc(dd)("dd");
    const inlineUl$1 = styledHoc(inlineUl)("ul");
    const inlineOl$1 = styledHoc(inlineOl)("ol");
    const inlineLi$1 = styledHoc(inlineLi)("li");

    class TalkieEditionIcon$2 extends React__default['default'].PureComponent {
      render() {
        const {
          mode,
          size,
          marginLeft,
          marginRight,
          className,
          network
        } = this.props;
        const networkClassName = `icon-${network}`;
        const classNames = ["icon-share", networkClassName, className].join(" ").trim();
        return /*#__PURE__*/React__default['default'].createElement(TalkieEditionIcon, {
          mode: mode,
          size: size,
          marginLeft: marginLeft,
          marginRight: marginRight,
          className: classNames
        });
      }

    }

    _defineProperty(TalkieEditionIcon$2, "defaultProps", {
      mode: "inline",
      size: "1.3em",
      marginLeft: "0.3em",
      marginRight: "0.3em",
      className: "",
      network: null
    });

    _defineProperty(TalkieEditionIcon$2, "propTypes", {
      mode: PropTypes__default['default'].oneOf(["inline", "standalone"]).isRequired,
      size: PropTypes__default['default'].string.isRequired,
      marginLeft: PropTypes__default['default'].string.isRequired,
      marginRight: PropTypes__default['default'].string.isRequired,
      className: PropTypes__default['default'].string.isRequired,
      network: PropTypes__default['default'].oneOf(["twitter", "facebook", "googleplus", "linkedin"]).isRequired
    });

    var _class$6, _class2$6, _temp$6;

    let SharingIcons = configureHoc(_class$6 = (_temp$6 = _class2$6 = class SharingIcons extends React__default['default'].PureComponent {
      componentDidMount() {
        this._unregisterConfigurationListener = this.props.onConfigurationChange(() => this.forceUpdate());
      }

      componentWillUnmount() {
        this._unregisterConfigurationListener();
      }

      render() {
        const {
          className,
          configure
        } = this.props;
        return /*#__PURE__*/React__default['default'].createElement(inlineUl$1, {
          className: className
        }, /*#__PURE__*/React__default['default'].createElement(inlineLi$1, null, /*#__PURE__*/React__default['default'].createElement(a$1, {
          href: configure("urls.share.twitter")
        }, /*#__PURE__*/React__default['default'].createElement(TalkieEditionIcon$2, {
          mode: "standalone",
          size: "2em",
          network: "twitter"
        }))), /*#__PURE__*/React__default['default'].createElement(inlineLi$1, null, /*#__PURE__*/React__default['default'].createElement(a$1, {
          href: configure("urls.share.facebook")
        }, /*#__PURE__*/React__default['default'].createElement(TalkieEditionIcon$2, {
          mode: "standalone",
          size: "2em",
          network: "facebook"
        }))), /*#__PURE__*/React__default['default'].createElement(inlineLi$1, null, /*#__PURE__*/React__default['default'].createElement(a$1, {
          href: configure("urls.share.googleplus")
        }, /*#__PURE__*/React__default['default'].createElement(TalkieEditionIcon$2, {
          mode: "standalone",
          size: "2em",
          network: "googleplus"
        }))), /*#__PURE__*/React__default['default'].createElement(inlineLi$1, null, /*#__PURE__*/React__default['default'].createElement(a$1, {
          href: configure("urls.share.linkedin")
        }, /*#__PURE__*/React__default['default'].createElement(TalkieEditionIcon$2, {
          mode: "standalone",
          size: "2em",
          network: "linkedin"
        }))));
      }

    }, _defineProperty(_class2$6, "defaultProps", {
      className: ""
    }), _defineProperty(_class2$6, "propTypes", {
      configure: PropTypes__default['default'].func.isRequired,
      onConfigurationChange: PropTypes__default['default'].func.isRequired,
      className: PropTypes__default['default'].string.isRequired
    }), _temp$6)) || _class$6;

    var _class$7, _class2$7, _temp$7;

    let About = translateHoc(_class$7 = configureHoc(_class$7 = (_temp$7 = _class2$7 = class About extends React__default['default'].PureComponent {
      constructor(props) {
        super(props);
        this.styled = {
          sharingIcons: styledHoc({
            display: "inline-block",
            verticalAlign: "middle"
          })(SharingIcons)
        };
      }

      componentDidMount() {
        this._unregisterConfigurationListener = this.props.onConfigurationChange(() => this.forceUpdate());
      }

      componentWillUnmount() {
        this._unregisterConfigurationListener();
      }

      render() {
        const {
          translate,
          configure
        } = this.props;
        return /*#__PURE__*/React__default['default'].createElement("section", null, /*#__PURE__*/React__default['default'].createElement(h2$1, null, translate("frontend_storyHeading")), /*#__PURE__*/React__default['default'].createElement(p, null, translate("frontend_storyDescription")), /*#__PURE__*/React__default['default'].createElement(p, null, translate("frontend_storyThankYou")), /*#__PURE__*/React__default['default'].createElement(p, null, "\u2014", /*#__PURE__*/React__default['default'].createElement(a$1, {
          href: "https://joelpurra.com/",
          lang: "sv"
        }, "Joel Purra")), /*#__PURE__*/React__default['default'].createElement(h2$1, null, translate("frontend_shareHeading")), /*#__PURE__*/React__default['default'].createElement(p, null, translate("frontend_sharePitch", [translate("extensionShortName")])), /*#__PURE__*/React__default['default'].createElement("div", null, /*#__PURE__*/React__default['default'].createElement(this.styled.sharingIcons, null), /*#__PURE__*/React__default['default'].createElement(a$1, {
          href: configure("urls.rate")
        }, translate("frontend_rateIt"))));
      }

    }, _defineProperty(_class2$7, "propTypes", {
      translate: PropTypes__default['default'].func.isRequired,
      configure: PropTypes__default['default'].func.isRequired,
      onConfigurationChange: PropTypes__default['default'].func.isRequired
    }), _temp$7)) || _class$7) || _class$7;

    var _class$8, _class2$8, _temp$8;

    let EditionSection = translateHoc(_class$8 = configureHoc(_class$8 = (_temp$8 = _class2$8 = class EditionSection extends React__default['default'].PureComponent {
      componentDidMount() {
        this._unregisterConfigurationListener = this.props.onConfigurationChange(() => this.forceUpdate());
      }

      componentWillUnmount() {
        this._unregisterConfigurationListener();
      }

      render() {
        const {
          mode,
          isPremiumEdition,
          className,
          translate,
          configure
        } = this.props; // TODO: move resolving the name to the state, like edition type?

        const text = isPremiumEdition ? translate("extensionShortName_Premium") : translate("extensionShortName_Free");
        const versionClassName = isPremiumEdition ? "premium-section" : "free-section";
        const classNames = [versionClassName, className].join(" ").trim();
        let HeadingElement = null;

        switch (mode) {
          case "p":
            HeadingElement = p;
            break;

          case "h2":
            HeadingElement = h2$1;
            break;

          default:
            throw new Error("Uknown mode");
        }

        return /*#__PURE__*/React__default['default'].createElement("div", {
          className: classNames
        }, /*#__PURE__*/React__default['default'].createElement(HeadingElement, null, /*#__PURE__*/React__default['default'].createElement(a$1, {
          href: configure("urls.options-upgrade-from-demo"),
          lang: "en"
        }, /*#__PURE__*/React__default['default'].createElement(TalkieEditionIcon$1, {
          isPremiumEdition: isPremiumEdition
        }), text)), this.props.children);
      }

    }, _defineProperty(_class2$8, "defaultProps", {
      mode: "h2",
      isPremiumEdition: false,
      className: ""
    }), _defineProperty(_class2$8, "propTypes", {
      mode: PropTypes__default['default'].oneOf(["p", "h2"]).isRequired,
      isPremiumEdition: PropTypes__default['default'].bool.isRequired,
      children: PropTypes__default['default'].node.isRequired,
      className: PropTypes__default['default'].string.isRequired,
      configure: PropTypes__default['default'].func.isRequired,
      onConfigurationChange: PropTypes__default['default'].func.isRequired,
      translate: PropTypes__default['default'].func.isRequired
    }), _temp$8)) || _class$8) || _class$8;

    class FreeSection extends React__default['default'].PureComponent {
      render() {
        const {
          mode,
          className
        } = this.props;
        const isPremiumEdition = false;
        return /*#__PURE__*/React__default['default'].createElement(EditionSection, {
          mode: mode,
          isPremiumEdition: isPremiumEdition,
          className: className
        }, this.props.children);
      }

    }

    _defineProperty(FreeSection, "defaultProps", {
      mode: "h2",
      className: ""
    });

    _defineProperty(FreeSection, "propTypes", {
      mode: PropTypes__default['default'].oneOf(["p", "h2"]).isRequired,
      children: PropTypes__default['default'].node.isRequired,
      className: PropTypes__default['default'].string.isRequired
    });

    class PremiumSection extends React__default['default'].PureComponent {
      render() {
        const {
          mode,
          className
        } = this.props;
        const isPremiumEdition = true;
        return /*#__PURE__*/React__default['default'].createElement(EditionSection, {
          mode: mode,
          isPremiumEdition: isPremiumEdition,
          className: className
        }, this.props.children);
      }

    }

    _defineProperty(PremiumSection, "defaultProps", {
      mode: "h2",
      className: ""
    });

    _defineProperty(PremiumSection, "propTypes", {
      mode: PropTypes__default['default'].oneOf(["p", "h2"]).isRequired,
      children: PropTypes__default['default'].node.isRequired,
      className: PropTypes__default['default'].string.isRequired
    });

    class TalkieFreeIcon extends React__default['default'].PureComponent {
      render() {
        const {
          mode,
          size,
          marginLeft,
          marginRight,
          className
        } = this.props;
        return /*#__PURE__*/React__default['default'].createElement(TalkieEditionIcon$1, {
          mode: mode,
          size: size,
          marginLeft: marginLeft,
          marginRight: marginRight,
          className: className,
          isPremiumEdition: false
        });
      }

    }

    _defineProperty(TalkieFreeIcon, "defaultProps", {
      mode: "inline",
      size: "1.3em",
      marginLeft: "0.3em",
      marginRight: "0.3em",
      className: ""
    });

    _defineProperty(TalkieFreeIcon, "propTypes", {
      mode: PropTypes__default['default'].oneOf(["inline", "standalone"]).isRequired,
      size: PropTypes__default['default'].string.isRequired,
      marginLeft: PropTypes__default['default'].string.isRequired,
      marginRight: PropTypes__default['default'].string.isRequired,
      className: PropTypes__default['default'].string.isRequired
    });

    class TalkiePremiumIcon extends React__default['default'].PureComponent {
      render() {
        const {
          mode,
          size,
          marginLeft,
          marginRight,
          className
        } = this.props;
        return /*#__PURE__*/React__default['default'].createElement(TalkieEditionIcon$1, {
          mode: mode,
          size: size,
          marginLeft: marginLeft,
          marginRight: marginRight,
          className: className,
          isPremiumEdition: true
        });
      }

    }

    _defineProperty(TalkiePremiumIcon, "defaultProps", {
      mode: "inline",
      size: "1.3em",
      marginLeft: "0.3em",
      marginRight: "0.3em",
      className: ""
    });

    _defineProperty(TalkiePremiumIcon, "propTypes", {
      mode: PropTypes__default['default'].oneOf(["inline", "standalone"]).isRequired,
      size: PropTypes__default['default'].string.isRequired,
      marginLeft: PropTypes__default['default'].string.isRequired,
      marginRight: PropTypes__default['default'].string.isRequired,
      className: PropTypes__default['default'].string.isRequired
    });

    var _class$9, _class2$9, _temp$9;

    let Features = configureHoc(_class$9 = translateHoc(_class$9 = (_temp$9 = _class2$9 = class Features extends React__default['default'].PureComponent {
      constructor(props) {
        super(props);
        this.styled = {
          storeLink: styledHoc({
            textAlign: "center",
            marginTop: "0.5em"
          })("div"),
          storeLinks: styledHoc({
            textAlign: "center",
            marginTop: "0.5em",
            "@media (min-width: 450px)": {
              columns: 2
            }
          })("div"),
          storeLinksP: styledHoc({
            marginBottom: "0.5em"
          })(p),
          storeLinksPFirst: styledHoc({
            marginBottom: "0.5em",
            "@media (min-width: 450px)": {
              marginTop: 0
            }
          })(p)
        };
      }

      componentDidMount() {
        this._unregisterConfigurationListener = this.props.onConfigurationChange(() => this.forceUpdate());
      }

      componentWillUnmount() {
        this._unregisterConfigurationListener();
      }

      render() {
        const {
          isPremiumEdition,
          systemType,
          translate,
          configure
        } = this.props;
        return /*#__PURE__*/React__default['default'].createElement("section", null, /*#__PURE__*/React__default['default'].createElement("p", null, translate("frontend_featuresEditions")), /*#__PURE__*/React__default['default'].createElement(Discretional, {
          enabled: !isPremiumEdition
        }, /*#__PURE__*/React__default['default'].createElement("p", null, translate("frontend_featuresEdition_Free"))), /*#__PURE__*/React__default['default'].createElement(Discretional, {
          enabled: isPremiumEdition
        }, /*#__PURE__*/React__default['default'].createElement("p", null, translate("frontend_featuresEdition_Premium"))), /*#__PURE__*/React__default['default'].createElement(PremiumSection, null, /*#__PURE__*/React__default['default'].createElement(ul$1, null, /*#__PURE__*/React__default['default'].createElement(li$1, null, translate("frontend_featuresPremium_List01")), /*#__PURE__*/React__default['default'].createElement(li$1, null, translate("frontend_featuresPremium_List02")), /*#__PURE__*/React__default['default'].createElement(Discretional, {
          enabled: systemType === "chrome"
        }, /*#__PURE__*/React__default['default'].createElement(li$1, null, translate("frontend_featuresPremium_List05"))), /*#__PURE__*/React__default['default'].createElement(li$1, null, translate("frontend_featuresPremium_List03")), /*#__PURE__*/React__default['default'].createElement(li$1, null, translate("frontend_featuresPremium_List04"))), /*#__PURE__*/React__default['default'].createElement(this.styled.storeLink, null, /*#__PURE__*/React__default['default'].createElement(a$1, {
          href: configure("urls.options-upgrade-from-demo"),
          lang: "en"
        }, /*#__PURE__*/React__default['default'].createElement(TalkiePremiumIcon, null), translate("frontend_featuresUpgradeToTalkiePremiumLinkText")))), /*#__PURE__*/React__default['default'].createElement(FreeSection, null, /*#__PURE__*/React__default['default'].createElement(ul$1, null, /*#__PURE__*/React__default['default'].createElement(li$1, null, translate("frontend_featuresFree_List01")), /*#__PURE__*/React__default['default'].createElement(li$1, null, translate("frontend_featuresFree_List02")), /*#__PURE__*/React__default['default'].createElement(li$1, null, translate("frontend_featuresFree_List03")), /*#__PURE__*/React__default['default'].createElement(li$1, null, translate("frontend_featuresFree_List04")), /*#__PURE__*/React__default['default'].createElement(li$1, null, translate("frontend_featuresFree_List05")), /*#__PURE__*/React__default['default'].createElement(li$1, null, translate("frontend_featuresFree_List06"))), /*#__PURE__*/React__default['default'].createElement(this.styled.storeLinks, null, /*#__PURE__*/React__default['default'].createElement(this.styled.storeLinksPFirst, null, /*#__PURE__*/React__default['default'].createElement(a$1, {
          href: configure("urls.chromewebstore"),
          lang: "en"
        }, /*#__PURE__*/React__default['default'].createElement("img", {
          src: "../../resources/chrome-web-store/ChromeWebStore_Badge_v2_496x150.png",
          alt: "Talkie is available for installation from the Chrome Web Store",
          width: "248",
          height: "75"
        }), /*#__PURE__*/React__default['default'].createElement("br", null), /*#__PURE__*/React__default['default'].createElement(TalkieFreeIcon, null), translate("extensionShortName_Free"))), /*#__PURE__*/React__default['default'].createElement(this.styled.storeLinksP, null, /*#__PURE__*/React__default['default'].createElement(a$1, {
          href: configure("urls.firefox-amo"),
          lang: "en"
        }, /*#__PURE__*/React__default['default'].createElement("img", {
          src: "../../resources/firefox-amo/AMO-button_1.png",
          alt: "Talkie is available for installation from the Chrome Web Store",
          width: "172",
          height: "60"
        }), /*#__PURE__*/React__default['default'].createElement("br", null), /*#__PURE__*/React__default['default'].createElement(TalkieFreeIcon, null), translate("extensionShortName_Free"))))));
      }

    }, _defineProperty(_class2$9, "defaultProps", {
      isPremiumEdition: false,
      systemType: false
    }), _defineProperty(_class2$9, "propTypes", {
      isPremiumEdition: PropTypes__default['default'].bool.isRequired,
      systemType: PropTypes__default['default'].string.isRequired,
      translate: PropTypes__default['default'].func.isRequired,
      configure: PropTypes__default['default'].func.isRequired,
      onConfigurationChange: PropTypes__default['default'].func.isRequired
    }), _temp$9)) || _class$9) || _class$9;

    var _class$a, _class2$a, _temp$a;

    let Support = configureHoc(_class$a = translateHoc(_class$a = (_temp$a = _class2$a = class Support extends React__default['default'].PureComponent {
      constructor(props) {
        super(props);
        this.styled = {
          summaryHeading: styledHoc({
            display: "inline-block",
            marginLeft: 0,
            marginRight: 0,
            marginTop: 0,
            marginBottom: 0,
            paddingLeft: "0.5em",
            paddingRight: "0.5em",
            paddingTop: "0.5em",
            paddingBottom: "0.5em"
          })(h4$1),
          sharingIcons: styledHoc({
            display: "inline-block",
            verticalAlign: "middle",
            fontSize: "0.5em"
          })(SharingIcons)
        };
      }

      componentDidMount() {
        this._unregisterConfigurationListener = this.props.onConfigurationChange(() => this.forceUpdate());
      }

      componentWillUnmount() {
        this._unregisterConfigurationListener();
      }

      standardFaqEntry(id) {
        const {
          translate
        } = this.props;
        const paddedId = id.toString(10).padStart(3, "0");
        return /*#__PURE__*/React__default['default'].createElement(details$1, null, /*#__PURE__*/React__default['default'].createElement(summary$1, null, /*#__PURE__*/React__default['default'].createElement(this.styled.summaryHeading, null, translate(`frontend_faq${paddedId}Q`))), /*#__PURE__*/React__default['default'].createElement(p, null, translate(`frontend_faq${paddedId}A`)));
      }

      render() {
        const {
          systemType,
          osType,
          translate,
          configure
        } = this.props; // TODO: configuration.
        return /*#__PURE__*/React__default['default'].createElement("section", null, /*#__PURE__*/React__default['default'].createElement(p, null, translate("frontend_supportDescription", [translate("extensionShortName")])), /*#__PURE__*/React__default['default'].createElement(ul$1, null, /*#__PURE__*/React__default['default'].createElement(li$1, null, /*#__PURE__*/React__default['default'].createElement(a$1, {
          href: configure("urls.support-feedback")
        }, translate("frontend_supportAndFeedback"))), /*#__PURE__*/React__default['default'].createElement(li$1, null, /*#__PURE__*/React__default['default'].createElement(a$1, {
          href: configure("urls.project")
        }, translate("frontend_aboutProjectPageLinkText")))), /*#__PURE__*/React__default['default'].createElement(h2$1, null, translate("frontend_faqHeading")), /*#__PURE__*/React__default['default'].createElement(h3$1, null, translate("frontend_faqVoicesHeading")), this.standardFaqEntry(1), /*#__PURE__*/React__default['default'].createElement(Discretional, {
          enabled:  osType === "win"
        }, /*#__PURE__*/React__default['default'].createElement(details$1, null, /*#__PURE__*/React__default['default'].createElement(summary$1, null, /*#__PURE__*/React__default['default'].createElement(this.styled.summaryHeading, null, translate("frontend_faq002Q"))), /*#__PURE__*/React__default['default'].createElement(p, null, translate("frontend_faq002A")), /*#__PURE__*/React__default['default'].createElement(ul$1, null, /*#__PURE__*/React__default['default'].createElement(li$1, null, /*#__PURE__*/React__default['default'].createElement(a$1, {
          href: "https://support.office.com/en-us/article/How-to-download-Text-to-Speech-languages-for-Windows-10-d5a6b612-b3ae-423f-afa5-4f6caf1ec5d3",
          lang: "en"
        }, "Windows 10"), ": Settings >\xA0Time\xA0&\xA0Language >\xA0Language"), /*#__PURE__*/React__default['default'].createElement(li$1, null, /*#__PURE__*/React__default['default'].createElement(a$1, {
          href: "https://support.office.com/en-us/article/How-to-download-Text-to-Speech-languages-for-Windows-4c83a8d8-7486-42f7-8e46-2b0fdf753130",
          lang: "en"
        }, "Windows 8")), /*#__PURE__*/React__default['default'].createElement(li$1, null, /*#__PURE__*/React__default['default'].createElement(a$1, {
          href: "https://www.microsoft.com/en-us/download/details.aspx?id=27224",
          lang: "en"
        }, "Windows 7"))))), /*#__PURE__*/React__default['default'].createElement(Discretional, {
          enabled:  osType === "cros"
        }, /*#__PURE__*/React__default['default'].createElement(details$1, null, /*#__PURE__*/React__default['default'].createElement(summary$1, null, /*#__PURE__*/React__default['default'].createElement(this.styled.summaryHeading, null, translate("frontend_faq003Q"))), /*#__PURE__*/React__default['default'].createElement(p, null, translate("frontend_faq003A")), /*#__PURE__*/React__default['default'].createElement(ul$1, null, /*#__PURE__*/React__default['default'].createElement(li$1, null, /*#__PURE__*/React__default['default'].createElement(a$1, {
          href: "https://chrome.google.com/webstore/detail/us-english-female-text-to/pkidpnnapnfgjhfhkpmjpbckkbaodldb",
          lang: "en"
        }, "US English Female Text-to-speech (by Google)"))))), /*#__PURE__*/React__default['default'].createElement(Discretional, {
          enabled:  osType === "mac"
        }, /*#__PURE__*/React__default['default'].createElement(details$1, null, /*#__PURE__*/React__default['default'].createElement(summary$1, null, /*#__PURE__*/React__default['default'].createElement(this.styled.summaryHeading, null, translate("frontend_faq004Q"))), /*#__PURE__*/React__default['default'].createElement(p, null, translate("frontend_faq004A")), /*#__PURE__*/React__default['default'].createElement(ul$1, null, /*#__PURE__*/React__default['default'].createElement(li$1, null, /*#__PURE__*/React__default['default'].createElement(a$1, {
          href: "https://support.apple.com/kb/index?page=search&q=VoiceOver+language&product=PF6&doctype=PRODUCT_HELP,HOWTO_ARTICLES&locale=en_US",
          lang: "en"
        }, "macOS"), ": System\xA0Preferences >\xA0Accessibility >\xA0Speech >\xA0System\xA0voice >\xA0Customize...")))), /*#__PURE__*/React__default['default'].createElement(Discretional, {
          enabled:  osType === "linux"
        }, /*#__PURE__*/React__default['default'].createElement(details$1, null, /*#__PURE__*/React__default['default'].createElement(summary$1, null, /*#__PURE__*/React__default['default'].createElement(this.styled.summaryHeading, null, translate("frontend_faq005Q"))), /*#__PURE__*/React__default['default'].createElement(p, null, translate("frontend_faq005A")))), this.standardFaqEntry(6), this.standardFaqEntry(7), this.standardFaqEntry(8), this.standardFaqEntry(9), /*#__PURE__*/React__default['default'].createElement(h3$1, null, translate("frontend_faqGeneralHeading")), this.standardFaqEntry(14), this.standardFaqEntry(15), this.standardFaqEntry(16), /*#__PURE__*/React__default['default'].createElement(Discretional, {
          enabled:  systemType === "chrome"
        }, /*#__PURE__*/React__default['default'].createElement(details$1, null, /*#__PURE__*/React__default['default'].createElement(summary$1, null, /*#__PURE__*/React__default['default'].createElement(this.styled.summaryHeading, null, translate("frontend_faq017Q"))), /*#__PURE__*/React__default['default'].createElement(p, null, translate("frontend_faq017A")), /*#__PURE__*/React__default['default'].createElement(ul$1, null, /*#__PURE__*/React__default['default'].createElement(li$1, null, /*#__PURE__*/React__default['default'].createElement(a$1, {
          href: configure("urls.shortcut-keys"),
          onClick: this.handleOpenShortKeysConfigurationClick
        }, translate("frontend_usageShortcutKeyAlternative04")))))), /*#__PURE__*/React__default['default'].createElement(Discretional, {
          enabled:  systemType === "webextension"
        }, /*#__PURE__*/React__default['default'].createElement(details$1, null, /*#__PURE__*/React__default['default'].createElement(summary$1, null, /*#__PURE__*/React__default['default'].createElement(this.styled.summaryHeading, null, translate("frontend_faq018Q"))), /*#__PURE__*/React__default['default'].createElement(p, null, translate("frontend_faq018A")))), this.standardFaqEntry(19), /*#__PURE__*/React__default['default'].createElement(details$1, null, /*#__PURE__*/React__default['default'].createElement(summary$1, null, /*#__PURE__*/React__default['default'].createElement(this.styled.summaryHeading, null, translate("frontend_faq020Q"))), /*#__PURE__*/React__default['default'].createElement(p, null, translate("frontend_faq020A")), /*#__PURE__*/React__default['default'].createElement("div", null, /*#__PURE__*/React__default['default'].createElement(this.styled.sharingIcons, null), /*#__PURE__*/React__default['default'].createElement(a$1, {
          href: configure("urls.rate")
        }, translate("frontend_rateIt")))), this.standardFaqEntry(25), /*#__PURE__*/React__default['default'].createElement(h3$1, null, translate("frontend_faqTalkiePremiumHeading")), this.standardFaqEntry(21), this.standardFaqEntry(33), this.standardFaqEntry(24), this.standardFaqEntry(22), this.standardFaqEntry(23), this.standardFaqEntry(26), this.standardFaqEntry(27), this.standardFaqEntry(28), this.standardFaqEntry(29), this.standardFaqEntry(30), this.standardFaqEntry(31), this.standardFaqEntry(32), /*#__PURE__*/React__default['default'].createElement(h3$1, null, translate("frontend_faqBugsHeading")), this.standardFaqEntry(10), this.standardFaqEntry(11), this.standardFaqEntry(12), this.standardFaqEntry(13));
      }

    }, _defineProperty(_class2$a, "defaultProps", {
      systemType: null,
      osType: null
    }), _defineProperty(_class2$a, "propTypes", {
      systemType: PropTypes__default['default'].string.isRequired,
      osType: PropTypes__default['default'].string,
      translate: PropTypes__default['default'].func.isRequired,
      configure: PropTypes__default['default'].func.isRequired,
      onConfigurationChange: PropTypes__default['default'].func.isRequired
    }), _temp$a)) || _class$a) || _class$a;

    var _class$b, _class2$b, _temp$b;

    let Usage = configureHoc(_class$b = translateHoc(_class$b = (_temp$b = _class2$b = class Usage extends React__default['default'].PureComponent {
      constructor(props) {
        super(props);
        this.handleOpenShortKeysConfigurationClick = this.handleOpenShortKeysConfigurationClick.bind(this);
        this.styled = {
          shortcutKeysTable: styledHoc({
            borderSpacing: 0
          })(table$1),
          shortcutKeysTd: styledHoc({
            // text-align: __MSG_@@bidi_end_edge__;
            whiteSpace: "nowrap"
          })(td$1)
        };
      }

      handleOpenShortKeysConfigurationClick(e) {
        e.preventDefault();
        e.stopPropagation();
        this.props.onOpenShortKeysConfigurationClick();
        return false;
      }

      componentDidMount() {
        this._unregisterConfigurationListener = this.props.onConfigurationChange(() => this.forceUpdate());
      }

      componentWillUnmount() {
        this._unregisterConfigurationListener();
      }

      render() {
        const {
          isPremiumEdition,
          systemType,
          osType,
          configure,
          translate
        } = this.props;
        return /*#__PURE__*/React__default['default'].createElement("section", null, /*#__PURE__*/React__default['default'].createElement(ul$1, null, /*#__PURE__*/React__default['default'].createElement(li$1, null, translate("frontend_usageStep01")), /*#__PURE__*/React__default['default'].createElement(li$1, null, translate("frontend_usageStep02"), /*#__PURE__*/React__default['default'].createElement(TalkieEditionIcon$1, {
          isPremiumEdition: isPremiumEdition
        }))), /*#__PURE__*/React__default['default'].createElement("p", null, translate("frontend_usageSelectionContextMenuDescription")), /*#__PURE__*/React__default['default'].createElement(Discretional, {
          enabled: systemType === "chrome"
        }, /*#__PURE__*/React__default['default'].createElement(PremiumSection, {
          mode: "p"
        }, /*#__PURE__*/React__default['default'].createElement("p", null, translate("frontend_usageReadclipboard")))), /*#__PURE__*/React__default['default'].createElement(h2$1, null, translate("frontend_usageShortcutHeading")), /*#__PURE__*/React__default['default'].createElement("p", null, translate("frontend_usageShortcutKeyDescription")), /*#__PURE__*/React__default['default'].createElement("div", {
          className: "talkie-block"
        }, /*#__PURE__*/React__default['default'].createElement(this.styled.shortcutKeysTable, null, /*#__PURE__*/React__default['default'].createElement("colgroup", null, /*#__PURE__*/React__default['default'].createElement("col", {
          width: "100%"
        }), /*#__PURE__*/React__default['default'].createElement("col", {
          width: "0*"
        })), /*#__PURE__*/React__default['default'].createElement(tbody$1, null, /*#__PURE__*/React__default['default'].createElement(tr$1, null, /*#__PURE__*/React__default['default'].createElement(td$1, null, /*#__PURE__*/React__default['default'].createElement(TalkieEditionIcon, {
          className: "icon-small-play"
        }), /*#__PURE__*/React__default['default'].createElement(span$1, null, "/"), /*#__PURE__*/React__default['default'].createElement(TalkieEditionIcon, {
          className: "icon-small-stop"
        }), translate("frontend_usageShortcutKeyDescriptionStartStopWithMenu")), /*#__PURE__*/React__default['default'].createElement(this.styled.shortcutKeysTd, null, /*#__PURE__*/React__default['default'].createElement(Discretional, {
          enabled: osType === "mac"
        }, /*#__PURE__*/React__default['default'].createElement(kbd$1, null, "\u2325")), /*#__PURE__*/React__default['default'].createElement(Discretional, {
          enabled: osType !== "mac"
        }, /*#__PURE__*/React__default['default'].createElement(kbd$1, null, "Alt")), "+", /*#__PURE__*/React__default['default'].createElement(kbd$1, null, "Shift"), "+", /*#__PURE__*/React__default['default'].createElement(kbd$1, null, "A"))), /*#__PURE__*/React__default['default'].createElement(Discretional, {
          enabled: systemType === "chrome"
        }, /*#__PURE__*/React__default['default'].createElement(tr$1, null, /*#__PURE__*/React__default['default'].createElement(td$1, null, /*#__PURE__*/React__default['default'].createElement(TalkieEditionIcon, {
          className: "icon-small-play"
        }), /*#__PURE__*/React__default['default'].createElement(span$1, null, "/"), /*#__PURE__*/React__default['default'].createElement(TalkieEditionIcon, {
          className: "icon-small-stop"
        }), translate("frontend_usageShortcutKeyDescriptionStartStopWithoutMenu")), /*#__PURE__*/React__default['default'].createElement(this.styled.shortcutKeysTd, null, /*#__PURE__*/React__default['default'].createElement(Discretional, {
          enabled: osType === "mac"
        }, /*#__PURE__*/React__default['default'].createElement(kbd$1, null, "\u2318")), /*#__PURE__*/React__default['default'].createElement(Discretional, {
          enabled: osType !== "mac"
        }, /*#__PURE__*/React__default['default'].createElement(kbd$1, null, "Ctrl")), "+", /*#__PURE__*/React__default['default'].createElement(kbd$1, null, "Shift"), "+", /*#__PURE__*/React__default['default'].createElement(kbd$1, null, "A")))), /*#__PURE__*/React__default['default'].createElement(Discretional, {
          enabled: systemType === "chrome"
        }, /*#__PURE__*/React__default['default'].createElement(tr$1, {
          className: "premium-section"
        }, /*#__PURE__*/React__default['default'].createElement(td$1, {
          colSpan: "2"
        }, /*#__PURE__*/React__default['default'].createElement(a$1, {
          href: configure("urls.options-upgrade-from-demo"),
          lang: "en"
        }, /*#__PURE__*/React__default['default'].createElement(TalkiePremiumIcon, null), translate("extensionShortName_Premium"))))), /*#__PURE__*/React__default['default'].createElement(Discretional, {
          enabled: systemType === "chrome"
        }, /*#__PURE__*/React__default['default'].createElement(tr$1, {
          className: "premium-section"
        }, /*#__PURE__*/React__default['default'].createElement(td$1, null, /*#__PURE__*/React__default['default'].createElement(TalkieEditionIcon, {
          className: "icon-small-speaker"
        }), translate("frontend_usageShortcutKeyDescriptionReadFromClipboard")), /*#__PURE__*/React__default['default'].createElement(this.styled.shortcutKeysTd, null, /*#__PURE__*/React__default['default'].createElement(Discretional, {
          enabled: osType === "mac"
        }, /*#__PURE__*/React__default['default'].createElement(kbd$1, null, "\u2318")), /*#__PURE__*/React__default['default'].createElement(Discretional, {
          enabled: osType !== "mac"
        }, /*#__PURE__*/React__default['default'].createElement(kbd$1, null, "Ctrl")), "+", /*#__PURE__*/React__default['default'].createElement(kbd$1, null, "Shift"), "+", /*#__PURE__*/React__default['default'].createElement(kbd$1, null, "1"))))))), /*#__PURE__*/React__default['default'].createElement(p$1, null, translate("frontend_usageShortcutKeyAlternative03")), /*#__PURE__*/React__default['default'].createElement(Discretional, {
          enabled: systemType === "chrome"
        }, /*#__PURE__*/React__default['default'].createElement("p", null, /*#__PURE__*/React__default['default'].createElement(a$1, {
          href: configure("urls.shortcut-keys"),
          onClick: this.handleOpenShortKeysConfigurationClick
        }, translate("frontend_usageShortcutKeyAlternative04")))));
      }

    }, _defineProperty(_class2$b, "defaultProps", {
      isPremiumEdition: false,
      systemType: null,
      osType: null
    }), _defineProperty(_class2$b, "propTypes", {
      isPremiumEdition: PropTypes__default['default'].bool.isRequired,
      systemType: PropTypes__default['default'].string.isRequired,
      osType: PropTypes__default['default'].string,
      onOpenShortKeysConfigurationClick: PropTypes__default['default'].func.isRequired,
      translate: PropTypes__default['default'].func.isRequired,
      configure: PropTypes__default['default'].func.isRequired,
      onConfigurationChange: PropTypes__default['default'].func.isRequired
    }), _temp$b)) || _class$b) || _class$b;

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const getLanguageGroupsFromLanguages = languages => {
      const languageGroupsObject = languages.reduce((obj, language) => {
        let languageGroupCode = null;

        if (language.length === 2) {
          languageGroupCode = language;
        } else {
          languageGroupCode = language.substring(0, 2);
        }

        obj[languageGroupCode] = (obj[languageGroupCode] || 0) + 1;
        return obj;
      }, {});
      const languageGroups = Object.keys(languageGroupsObject);
      languageGroups.sort();
      return languageGroups;
    };
    const getLanguagesFromVoices = voices => {
      const languagesAsKeys = voices.reduce((obj, voice) => {
        obj[voice.lang] = (obj[voice.lang] || 0) + 1;
        return obj;
      }, {});
      const languages = Object.keys(languagesAsKeys);
      languages.sort();
      return languages;
    };
    const getVoicesByLanguageFromVoices = voices => {
      const voicesByLanguage = voices.reduce((obj, voice) => {
        obj[voice.lang] = (obj[voice.lang] || []).concat(Object.assign({}, voice));
        return obj;
      }, {});
      return voicesByLanguage;
    };
    const getVoicesByLanguageGroupFromVoices = voices => {
      const voicesByLanguageGroup = voices.reduce((obj, voice) => {
        const group = voice.lang.substr(0, 2);
        obj[group] = (obj[group] || []).concat(Object.assign({}, voice));
        return obj;
      }, {});
      return voicesByLanguageGroup;
    };
    const getVoicesByLanguagesByLanguageGroupFromVoices = voices => {
      const voicesByLanguage = getVoicesByLanguageFromVoices(voices);
      const languagesByLanguageGroup = voices.reduce((obj, voice) => {
        const group = voice.lang.substr(0, 2);
        obj[group] = obj[group] || {};
        obj[group][voice.lang] = voicesByLanguage[voice.lang];
        return obj;
      }, {});
      return languagesByLanguageGroup;
    };
    const getLanguagesByLanguageGroupFromVoices = voices => {
      const voicesByLanguagesByLanguageGroup = getVoicesByLanguagesByLanguageGroupFromVoices(voices);
      const languageGroups = Object.keys(voicesByLanguagesByLanguageGroup);
      const languagesByLanguageGroup = languageGroups.reduce((obj, group) => {
        const languages = Object.keys(voicesByLanguagesByLanguageGroup[group]);
        obj[group] = languages;
        return obj;
      }, {});
      Object.keys(languagesByLanguageGroup).forEach(languageGroup => languagesByLanguageGroup[languageGroup].sort());
      return languagesByLanguageGroup;
    };
    const getLanguageForVoiceNameFromVoices = (voices, voiceName) => {
      const matchingVoices = voices.filter(voice => voice.name === voiceName);

      if (matchingVoices.length !== 1) {
        throw new Error(`Mismatching number of voices found: ${matchingVoices.length}`);
      }

      const voice = matchingVoices[0];
      return voice;
    };
    const getAvailableBrowserLanguageWithInstalledVoiceFromNavigatorLanguagesAndLanguagesAndLanguageGroups = (navigatorLanguages, languages, languageGroups) => [] // NOTE: preferring language groups over languages/dialects.
    .concat(navigatorLanguages.filter(navigatorLanguage => languageGroups.includes(navigatorLanguage))).concat(navigatorLanguages.filter(navigatorLanguage => languages.includes(navigatorLanguage)));

    var _class$c, _class2$c, _temp$c;

    let Loading = translateHoc(_class$c = (_temp$c = _class2$c = class Loading extends React__default['default'].PureComponent {
      render() {
        const {
          enabled,
          children,
          translate
        } = this.props;

        if (enabled) {
          return children;
        }

        return /*#__PURE__*/React__default['default'].createElement("div", null, /*#__PURE__*/React__default['default'].createElement(span$1, null, translate("frontend_loading")));
      }

    }, _defineProperty(_class2$c, "defaultProps", {
      enabled: false
    }), _defineProperty(_class2$c, "propTypes", {
      enabled: PropTypes__default['default'].bool.isRequired,
      translate: PropTypes__default['default'].func.isRequired,
      children: PropTypes__default['default'].node.isRequired
    }), _temp$c)) || _class$c;

    var _class$d, _class2$d, _temp$d;

    let Voices = translateHoc(_class$d = (_temp$d = _class2$d = class Voices extends React__default['default'].PureComponent {
      constructor(props) {
        super(props);
        this.handleSpeakClickForLanguage = this.handleSpeakClickForLanguage.bind(this);
        this.handleSpeakClickForVoice = this.handleSpeakClickForVoice.bind(this);
        this.styled = {
          summaryH3: styledHoc({
            display: "inline-block",
            marginLeft: 0,
            marginRight: 0,
            marginTop: 0,
            marginBottom: 0,
            paddingLeft: "0.5em",
            paddingRight: "0.5em",
            paddingTop: "0.5em",
            paddingBottom: "0.5em"
          })(h3$1),
          hr: styledHoc({
            marginLeft: 0,
            marginRight: 0
          })(hr$1),
          clickableLi: styledHoc({
            cursor: "pointer"
          })(li$1)
        };
        this.styled.clickableNoBulletLi = styledHoc({
          listStyle: "none",
          marginLeft: "-2em"
        })(this.styled.clickableLi);
      }

      getSampleTextForLanguage(languageCode) {
        /* eslint-disable no-sync */
        return this.props.talkieLocaleHelper.getSampleTextSync(languageCode);
        /* eslint-enable no-sync */
      }

      getSampleTextForVoicename(voiceName) {
        const voices = this.props.voices;
        const voice = getLanguageForVoiceNameFromVoices(voices, voiceName);
        const languageCode = voice.lang;
        /* eslint-disable no-sync */

        return this.props.talkieLocaleHelper.getSampleTextSync(languageCode);
        /* eslint-enable no-sync */
      }

      getTextDirectionForLanguageGroup(languageGroup) {
        /* eslint-disable no-sync */
        return this.props.talkieLocaleHelper.getBidiDirectionSync(languageGroup);
        /* eslint-enable no-sync */
      }

      getTextDirectionClassNameForLanguageGroup(languageGroup) {
        const direction = this.getTextDirectionForLanguageGroup(languageGroup);
        let className = null;

        switch (direction) {
          case "ltr":
            className = "text-direction-ltr";
            break;

          case "rtl":
            className = "text-direction-rtl";
            break;

          default:
            throw new Error("Unknown text direction");
        }

        return className;
      }

      handleSpeakClickForLanguage(languageCode, e) {
        e.preventDefault();
        e.stopPropagation();
        const text = this.getSampleTextForLanguage(languageCode);
        const voice = {
          lang: languageCode // name: null,
          // rate: ...,
          // pitch: ...,

        };
        this.props.actions.sharedVoices.speak(text, voice);
        return false;
      }

      handleSpeakClickForVoice(voiceName, e) {
        e.preventDefault();
        e.stopPropagation();
        const text = this.getSampleTextForVoicename(voiceName);
        const voice = {
          // lang: null,
          name: voiceName // rate: ...,
          // pitch: ...,

        };
        this.props.actions.sharedVoices.speak(text, voice);
        return false;
      }

      getVoicesListItems(voices, showCode) {
        const {
          translate
        } = this.props;
        const translatedVoiceFeatureOnline = translate("frontend_voiceFeatureOnline");
        return voices.map(voice => {
          let voiceNameAndFeaturesText = voice.name;
          const voiceFeatures = [];

          if (showCode) {
            voiceFeatures.push(voice.lang);
          }

          if (voice.localService === false) {
            voiceFeatures.push(translatedVoiceFeatureOnline);
          }

          if (voiceFeatures.length > 0) {
            voiceNameAndFeaturesText += " (";
            voiceNameAndFeaturesText += voiceFeatures.join(", ");
            voiceNameAndFeaturesText += ")";
          }

          return /*#__PURE__*/React__default['default'].createElement(this.styled.clickableNoBulletLi, {
            key: voice.name,
            onClick: this.handleSpeakClickForVoice.bind(null, voice.name)
          }, /*#__PURE__*/React__default['default'].createElement(TalkieEditionIcon, {
            className: "icon-voices"
          }), voiceNameAndFeaturesText);
        });
      }

      getVoicesListItemsWithCode(voices) {
        return this.getVoicesListItems(voices, true);
      }

      getVoicesListItemsWithoutCode(voices) {
        return this.getVoicesListItems(voices, false);
      }

      getLanguagesListItems(languages) {
        return languages.map(language => /*#__PURE__*/React__default['default'].createElement(this.styled.clickableLi, {
          key: language,
          onClick: this.handleSpeakClickForLanguage.bind(null, language)
        }, language));
      }

      getFilteredLanguagesAndVoicesTree(voicesByLanguagesByLanguageGroup, languagesPerGroup, languagesFilter, languageGroup) {
        const filteredLanguagesPerGroup = languagesPerGroup.filter(language => !languagesFilter || languagesFilter.includes(language));
        return filteredLanguagesPerGroup.map(language => {
          const voicesPerLanguage = voicesByLanguagesByLanguageGroup[languageGroup][language];
          return /*#__PURE__*/React__default['default'].createElement(this.styled.clickableLi, {
            key: language,
            onClick: this.handleSpeakClickForLanguage.bind(null, language)
          }, language, /*#__PURE__*/React__default['default'].createElement(ul$1, null, this.getVoicesListItemsWithoutCode(voicesPerLanguage)));
        });
      }

      getFilteredLanguageGroupsAndLanguagesAndVoicesTree(voicesByLanguagesByLanguageGroup, languagesFilter) {
        const languageGroupsFilter = languagesFilter && getLanguageGroupsFromLanguages(languagesFilter) || null;
        const languageGroups = Object.keys(voicesByLanguagesByLanguageGroup);
        languageGroups.sort();
        const filteredLanguageGroups = languageGroups.filter(languageGroup => !languageGroupsFilter || languageGroupsFilter.includes(languageGroup));
        return filteredLanguageGroups.map((languageGroup, index) => {
          const languagesPerGroup = Object.keys(voicesByLanguagesByLanguageGroup[languageGroup]);
          languagesPerGroup.sort();
          const sampleTextForLanguage = this.getSampleTextForLanguage(languageGroup);
          let sampleTextBlockQuote = null;

          if (sampleTextForLanguage) {
            sampleTextBlockQuote = /*#__PURE__*/React__default['default'].createElement(blockquote$1, {
              lang: languageGroup,
              className: this.getTextDirectionClassNameForLanguageGroup(languageGroup),
              onClick: this.handleSpeakClickForLanguage.bind(null, languageGroup)
            }, sampleTextForLanguage);
          }

          return /*#__PURE__*/React__default['default'].createElement("div", {
            key: languageGroup
          }, /*#__PURE__*/React__default['default'].createElement(details$1, null, /*#__PURE__*/React__default['default'].createElement(summary$1, null, /*#__PURE__*/React__default['default'].createElement(this.styled.summaryH3, null, languageGroup)), sampleTextBlockQuote, /*#__PURE__*/React__default['default'].createElement(p, null, /*#__PURE__*/React__default['default'].createElement(a$1, {
            href: `https://${languageGroup}.wikipedia.org/`
          }, /*#__PURE__*/React__default['default'].createElement(TalkieEditionIcon, {
            className: "icon-wikipedia-w"
          }), languageGroup, ".wikipedia.org")), /*#__PURE__*/React__default['default'].createElement(ul$1, null, this.getFilteredLanguagesAndVoicesTree(voicesByLanguagesByLanguageGroup, languagesPerGroup, languagesFilter, languageGroup))), index + 1 < filteredLanguageGroups.length && /*#__PURE__*/React__default['default'].createElement(this.styled.hr, null));
        });
      }

      render() {
        const {
          languageGroupsCount,
          languagesCount,
          navigatorLanguages,
          translate,
          voicesByLanguagesByLanguageGroup,
          voicesCount
        } = this.props;
        const haveVoices = voicesCount > 0;
        return /*#__PURE__*/React__default['default'].createElement("section", null, /*#__PURE__*/React__default['default'].createElement(p, null, translate("frontend_voicesDescription")), /*#__PURE__*/React__default['default'].createElement(PremiumSection, null, translate("frontend_voicesTalkiePremiumPitch")), /*#__PURE__*/React__default['default'].createElement(h2$1, null, translate("frontend_voicesPreferredHeading")), /*#__PURE__*/React__default['default'].createElement(Loading, {
          enabled: haveVoices
        }, this.getFilteredLanguageGroupsAndLanguagesAndVoicesTree(voicesByLanguagesByLanguageGroup, navigatorLanguages)), /*#__PURE__*/React__default['default'].createElement(h2$1, null, translate("frontend_voicesInstalledHeading", [languageGroupsCount, languagesCount, voicesCount])), /*#__PURE__*/React__default['default'].createElement(Loading, {
          enabled: haveVoices
        }, this.getFilteredLanguageGroupsAndLanguagesAndVoicesTree(voicesByLanguagesByLanguageGroup, null)));
      }

    }, _defineProperty(_class2$d, "defaultProps", {
      voices: [],
      voicesByLanguagesByLanguageGroup: {},
      navigatorLanguages: [],
      voicesCount: 0,
      languagesCount: 0,
      languageGroupsCount: 0
    }), _defineProperty(_class2$d, "propTypes", {
      actions: PropTypes__default['default'].object.isRequired,
      voices: PropTypes__default['default'].arrayOf(PropTypes__default['default'].shape({
        default: PropTypes__default['default'].bool.isRequired,
        lang: PropTypes__default['default'].string.isRequired,
        localService: PropTypes__default['default'].bool.isRequired,
        name: PropTypes__default['default'].string.isRequired,
        voiceURI: PropTypes__default['default'].string.isRequired
      })).isRequired,
      voicesByLanguagesByLanguageGroup: PropTypes__default['default'].objectOf(PropTypes__default['default'].objectOf(PropTypes__default['default'].arrayOf(PropTypes__default['default'].shape({
        default: PropTypes__default['default'].bool.isRequired,
        lang: PropTypes__default['default'].string.isRequired,
        localService: PropTypes__default['default'].bool.isRequired,
        name: PropTypes__default['default'].string.isRequired,
        voiceURI: PropTypes__default['default'].string.isRequired
      })).isRequired).isRequired).isRequired,
      navigatorLanguages: PropTypes__default['default'].arrayOf(PropTypes__default['default'].string.isRequired).isRequired,
      voicesCount: PropTypes__default['default'].number.isRequired,
      languagesCount: PropTypes__default['default'].number.isRequired,
      languageGroupsCount: PropTypes__default['default'].number.isRequired,
      translate: PropTypes__default['default'].func.isRequired,
      talkieLocaleHelper: PropTypes__default['default'].object.isRequired
    }), _temp$d)) || _class$d;

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    const getVoices$1 = state => state.shared.voices.voices;
    const getVoicesCount = reselect.createSelector([getVoices$1], voices => voices.length);
    const getLanguages = reselect.createSelector([getVoices$1], voices => getLanguagesFromVoices(voices));
    const getLanguagesCount = reselect.createSelector([getLanguages], languages => languages.length);
    const getLanguageGroups = reselect.createSelector([getLanguages], languages => getLanguageGroupsFromLanguages(languages));
    const getLanguageGroupsCount = reselect.createSelector([getLanguageGroups], languageGroups => languageGroups.length);
    const getVoicesByLanguage = reselect.createSelector([getVoices$1], voices => getVoicesByLanguageFromVoices(voices));
    const getVoicesByLanguageGroup = reselect.createSelector([getVoices$1], voices => getVoicesByLanguageGroupFromVoices(voices));
    const getLanguagesByLanguageGroup = reselect.createSelector([getVoices$1], voices => getLanguagesByLanguageGroupFromVoices(voices));
    const getVoicesByLanguagesByLanguageGroup = reselect.createSelector([getVoices$1], voices => getVoicesByLanguagesByLanguageGroupFromVoices(voices));
    const getNavigatorLanguages$1 = state => state.shared.voices.navigatorLanguages;
    const getAvailableBrowserLanguageWithInstalledVoice = reselect.createSelector([getNavigatorLanguages$1, getLanguages, getLanguageGroups], (navigatorLanguages, languages, languageGroups) => getAvailableBrowserLanguageWithInstalledVoiceFromNavigatorLanguagesAndLanguagesAndLanguageGroups(navigatorLanguages, languages, languageGroups));

    var voices$2 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        getVoices: getVoices$1,
        getVoicesCount: getVoicesCount,
        getLanguages: getLanguages,
        getLanguagesCount: getLanguagesCount,
        getLanguageGroups: getLanguageGroups,
        getLanguageGroupsCount: getLanguageGroupsCount,
        getVoicesByLanguage: getVoicesByLanguage,
        getVoicesByLanguageGroup: getVoicesByLanguageGroup,
        getLanguagesByLanguageGroup: getLanguagesByLanguageGroup,
        getVoicesByLanguagesByLanguageGroup: getVoicesByLanguagesByLanguageGroup,
        getNavigatorLanguages: getNavigatorLanguages$1,
        getAvailableBrowserLanguageWithInstalledVoice: getAvailableBrowserLanguageWithInstalledVoice
    });

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    var shared$3 = {
      voices: voices$2
    };

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    var selectors = {
      shared: shared$3
    };

    var _dec$4, _class$e, _class2$e, _temp$e;
    const talkieLocaleHelper = new TalkieLocaleHelper();

    const mapStateToProps$3 = state => {
      return {
        voices: selectors.shared.voices.getVoices(state),
        voicesByLanguagesByLanguageGroup: selectors.shared.voices.getVoicesByLanguagesByLanguageGroup(state),
        voicesCount: selectors.shared.voices.getVoicesCount(state),
        languagesCount: selectors.shared.voices.getLanguagesCount(state),
        languageGroupsCount: selectors.shared.voices.getLanguageGroupsCount(state),
        navigatorLanguages: selectors.shared.voices.getNavigatorLanguages(state)
      };
    };

    const mapDispatchToProps$3 = dispatch => {
      return {
        actions: {
          sharedVoices: redux.bindActionCreators(actions.shared.voices, dispatch)
        }
      };
    };

    let VoicesContainer = (_dec$4 = reactRedux.connect(mapStateToProps$3, mapDispatchToProps$3), _dec$4(_class$e = (_temp$e = _class2$e = class VoicesContainer extends React__default['default'].PureComponent {
      render() {
        const {
          actions,
          languageGroupsCount,
          languagesCount,
          navigatorLanguages,
          voices,
          voicesByLanguagesByLanguageGroup,
          voicesCount
        } = this.props;
        return /*#__PURE__*/React__default['default'].createElement(Voices, {
          actions: actions,
          voices: voices,
          voicesByLanguagesByLanguageGroup: voicesByLanguagesByLanguageGroup,
          navigatorLanguages: navigatorLanguages,
          voicesCount: voicesCount,
          languagesCount: languagesCount,
          languageGroupsCount: languageGroupsCount,
          talkieLocaleHelper: talkieLocaleHelper
        });
      }

    }, _defineProperty(_class2$e, "propTypes", {
      actions: PropTypes__default['default'].object.isRequired,
      voices: PropTypes__default['default'].arrayOf(PropTypes__default['default'].shape({
        default: PropTypes__default['default'].bool.isRequired,
        lang: PropTypes__default['default'].string.isRequired,
        localService: PropTypes__default['default'].bool.isRequired,
        name: PropTypes__default['default'].string.isRequired,
        voiceURI: PropTypes__default['default'].string.isRequired
      })).isRequired,
      voicesByLanguagesByLanguageGroup: PropTypes__default['default'].objectOf(PropTypes__default['default'].objectOf(PropTypes__default['default'].arrayOf(PropTypes__default['default'].shape({
        default: PropTypes__default['default'].bool.isRequired,
        lang: PropTypes__default['default'].string.isRequired,
        localService: PropTypes__default['default'].bool.isRequired,
        name: PropTypes__default['default'].string.isRequired,
        voiceURI: PropTypes__default['default'].string.isRequired
      })).isRequired).isRequired).isRequired,
      navigatorLanguages: PropTypes__default['default'].arrayOf(PropTypes__default['default'].string.isRequired).isRequired,
      voicesCount: PropTypes__default['default'].number.isRequired,
      languagesCount: PropTypes__default['default'].number.isRequired,
      languageGroupsCount: PropTypes__default['default'].number.isRequired
    }), _temp$e)) || _class$e);

    class HeroSection extends React__default['default'].PureComponent {
      render() {
        const {
          className
        } = this.props;
        return /*#__PURE__*/React__default['default'].createElement(hero$1, {
          className: className
        }, this.props.children);
      }

    }

    _defineProperty(HeroSection, "defaultProps", {
      className: ""
    });

    _defineProperty(HeroSection, "propTypes", {
      children: PropTypes__default['default'].node.isRequired,
      className: PropTypes__default['default'].string.isRequired
    });

    var _class$f, _class2$f, _temp$f;

    let Welcome = configureHoc(_class$f = translateHoc(_class$f = (_temp$f = _class2$f = class Welcome extends React__default['default'].PureComponent {
      constructor(props) {
        super(props);
        this.playWelcomeMessage = this.playWelcomeMessage.bind(this);
        this.selectTextInElement = this.selectTextInElement.bind(this);
        this.spokeSample = false;
        this.welcomeSampleTextElement = null;
        this.styled = {
          heroDiv: styledHoc({
            marginBottom: "2em"
          })("div"),
          HeroEditionSection: styledHoc({
            // NOTE: atomic css class ordering seems to not work well in this case.
            marginBottom: 0
          })(HeroSection),
          sampleHeroP: styledHoc({
            marginTop: 0
          })(p),
          welcomeHeroP: styledHoc({
            marginTop: 0,
            marginBottom: 0
          })(p),
          sharingDiv: styledHoc({
            marginTop: "-4em",
            marginLeft: "6em",
            marginRight: "6em"
          })("div"),
          sharingIcons: styledHoc({
            display: "inline-block",
            verticalAlign: "middle"
          })(SharingIcons),
          summaryHeading: styledHoc({
            display: "inline-block",
            marginLeft: 0,
            marginRight: 0,
            marginTop: 0,
            marginBottom: 0,
            paddingLeft: "0.5em",
            paddingRight: "0.5em",
            paddingTop: "0.5em",
            paddingBottom: "0.5em"
          })(h3$1)
        };
      }

      componentDidMount() {
        this._unregisterConfigurationListener = this.props.onConfigurationChange(() => this.forceUpdate());
        this.playWelcomeMessage();
      }

      componentDidUpdate() {
        this.playWelcomeMessage();
      }

      componentWillUnmount() {
        this._unregisterConfigurationListener();
      }

      playWelcomeMessage() {
        // TODO: take sample text language code into account.
        if (this.props.voicesCount === 0) {
          return;
        }

        if (!this.welcomeSampleTextElement) {
          return;
        }

        if (this.spokeSample) {
          return;
        }

        this.spokeSample = true;
        this.selectTextInElement(this.welcomeSampleTextElement);
        const welcomeSample = this.getWelcomeSample();
        const text = welcomeSample.text;
        const languageCode = welcomeSample.languageCode;
        this.props.speakTextInLanguageWithOverrides(text, languageCode);
      }

      selectTextInElement(element) {
        const selection = document.getSelection();
        selection.removeAllRanges();
        selection.selectAllChildren(element);
      }

      getWelcomeSample() {
        const hasSampleText = !!this.props.sampleText;
        const welcomeSampleText = this.props.sampleText;
        const welcomeSampleTextLanguage = this.props.sampleTextLanguageCode;
        const welcomeSample = {
          hasSampleText: hasSampleText,
          text: welcomeSampleText,
          languageCode: welcomeSampleTextLanguage
        };
        return welcomeSample;
      }

      render() {
        const {
          isPremiumEdition,
          systemType,
          osType,
          voicesCount,
          languagesCount,
          languageGroupsCount,
          canSpeakInTranslatedLocale,
          translate,
          configure
        } = this.props; // TODO: configuration.

        const systemTypePrettyName = systemType; // TODO: pretty name.

        const osTypePrettyName = osType;
        const welcomeSample = this.getWelcomeSample();
        const haveVoices = voicesCount > 0;
        return /*#__PURE__*/React__default['default'].createElement("section", null, /*#__PURE__*/React__default['default'].createElement(this.styled.heroDiv, null, /*#__PURE__*/React__default['default'].createElement(this.styled.HeroEditionSection, {
          isPremiumEdition: isPremiumEdition
        }, /*#__PURE__*/React__default['default'].createElement(Discretional, {
          enabled: welcomeSample.hasSampleText
        }, /*#__PURE__*/React__default['default'].createElement(this.styled.sampleHeroP, null, /*#__PURE__*/React__default['default'].createElement("span", {
          lang: welcomeSample.languageCode,
          ref: welcomeSampleTextElement => {
            this.welcomeSampleTextElement = welcomeSampleTextElement;
          }
        }, welcomeSample.text))), /*#__PURE__*/React__default['default'].createElement(this.styled.welcomeHeroP, null, translate("frontend_welcomeHero01", [translate("extensionShortName")]), /*#__PURE__*/React__default['default'].createElement(Discretional, {
          enabled: canSpeakInTranslatedLocale
        }, " ", translate("frontend_welcomeHero02")))), /*#__PURE__*/React__default['default'].createElement(this.styled.sharingDiv, null, /*#__PURE__*/React__default['default'].createElement(this.styled.sharingIcons, null), /*#__PURE__*/React__default['default'].createElement(a$1, {
          href: configure("urls.rate")
        }, translate("frontend_rateIt")))), /*#__PURE__*/React__default['default'].createElement(h2$1, null, translate("frontend_welcomeInstallMoreVoicesHeading")), /*#__PURE__*/React__default['default'].createElement(p, null, /*#__PURE__*/React__default['default'].createElement(Loading, {
          enabled: haveVoices
        }, translate("frontend_welcomeInstallMoreVoicesDescription", [voicesCount, languageGroupsCount, languagesCount, systemTypePrettyName, osTypePrettyName]))), /*#__PURE__*/React__default['default'].createElement(Discretional, {
          enabled:  osType === "win"
        }, /*#__PURE__*/React__default['default'].createElement(details$1, null, /*#__PURE__*/React__default['default'].createElement(summary$1, null, /*#__PURE__*/React__default['default'].createElement(this.styled.summaryHeading, null, translate("frontend_faq002Q"))), /*#__PURE__*/React__default['default'].createElement(p, null, translate("frontend_faq002A")), /*#__PURE__*/React__default['default'].createElement(ul$1, null, /*#__PURE__*/React__default['default'].createElement(li$1, null, /*#__PURE__*/React__default['default'].createElement(a$1, {
          href: "https://support.office.com/en-us/article/How-to-download-Text-to-Speech-languages-for-Windows-10-d5a6b612-b3ae-423f-afa5-4f6caf1ec5d3",
          lang: "en"
        }, "Windows 10"), ": Settings >\xA0Time\xA0&\xA0Language >\xA0Language"), /*#__PURE__*/React__default['default'].createElement(li$1, null, /*#__PURE__*/React__default['default'].createElement(a$1, {
          href: "https://support.office.com/en-us/article/How-to-download-Text-to-Speech-languages-for-Windows-4c83a8d8-7486-42f7-8e46-2b0fdf753130",
          lang: "en"
        }, "Windows 8")), /*#__PURE__*/React__default['default'].createElement(li$1, null, /*#__PURE__*/React__default['default'].createElement(a$1, {
          href: "https://www.microsoft.com/en-us/download/details.aspx?id=27224",
          lang: "en"
        }, "Windows 7"))))), /*#__PURE__*/React__default['default'].createElement(Discretional, {
          enabled:  osType === "cros"
        }, /*#__PURE__*/React__default['default'].createElement(details$1, null, /*#__PURE__*/React__default['default'].createElement(summary$1, null, /*#__PURE__*/React__default['default'].createElement(this.styled.summaryHeading, null, translate("frontend_faq003Q"))), /*#__PURE__*/React__default['default'].createElement(p, null, translate("frontend_faq003A")), /*#__PURE__*/React__default['default'].createElement(ul$1, null, /*#__PURE__*/React__default['default'].createElement(li$1, null, /*#__PURE__*/React__default['default'].createElement(a$1, {
          href: "https://chrome.google.com/webstore/detail/us-english-female-text-to/pkidpnnapnfgjhfhkpmjpbckkbaodldb",
          lang: "en"
        }, "US English Female Text-to-speech (by Google)"))))), /*#__PURE__*/React__default['default'].createElement(Discretional, {
          enabled:  osType === "mac"
        }, /*#__PURE__*/React__default['default'].createElement(details$1, null, /*#__PURE__*/React__default['default'].createElement(summary$1, null, /*#__PURE__*/React__default['default'].createElement(this.styled.summaryHeading, null, translate("frontend_faq004Q"))), /*#__PURE__*/React__default['default'].createElement(p, null, translate("frontend_faq004A")), /*#__PURE__*/React__default['default'].createElement(ul$1, null, /*#__PURE__*/React__default['default'].createElement(li$1, null, /*#__PURE__*/React__default['default'].createElement(a$1, {
          href: "https://support.apple.com/kb/index?page=search&q=VoiceOver+language&product=PF6&doctype=PRODUCT_HELP,HOWTO_ARTICLES&locale=en_US",
          lang: "en"
        }, "macOS"), ": System\xA0Preferences >\xA0Accessibility >\xA0Speech >\xA0System\xA0voice >\xA0Customize...")))), /*#__PURE__*/React__default['default'].createElement(Discretional, {
          enabled:  osType === "linux"
        }, /*#__PURE__*/React__default['default'].createElement(details$1, null, /*#__PURE__*/React__default['default'].createElement(summary$1, null, /*#__PURE__*/React__default['default'].createElement(this.styled.summaryHeading, null, translate("frontend_faq005Q"))), /*#__PURE__*/React__default['default'].createElement(p, null, translate("frontend_faq005A")))));
      }

    }, _defineProperty(_class2$f, "defaultProps", {
      isPremiumEdition: false,
      systemType: null,
      osType: null,
      voicesCount: 0,
      languagesCount: 0,
      languageGroupsCount: 0,
      sampleText: null,
      sampleTextLanguageCode: null,
      speakTextInLanguageWithOverrides: null,
      canSpeakInTranslatedLocale: false
    }), _defineProperty(_class2$f, "propTypes", {
      isPremiumEdition: PropTypes__default['default'].bool.isRequired,
      systemType: PropTypes__default['default'].string.isRequired,
      osType: PropTypes__default['default'].string,
      voicesCount: PropTypes__default['default'].number.isRequired,
      languagesCount: PropTypes__default['default'].number.isRequired,
      languageGroupsCount: PropTypes__default['default'].number.isRequired,
      sampleText: PropTypes__default['default'].string,
      sampleTextLanguageCode: PropTypes__default['default'].string,
      speakTextInLanguageWithOverrides: PropTypes__default['default'].func.isRequired,
      canSpeakInTranslatedLocale: PropTypes__default['default'].bool.isRequired,
      translate: PropTypes__default['default'].func.isRequired,
      configure: PropTypes__default['default'].func.isRequired,
      onConfigurationChange: PropTypes__default['default'].func.isRequired
    }), _temp$f)) || _class$f) || _class$f;

    var _dec$5, _class$g, _class2$g, _temp$g;
    const talkieLocaleHelper$1 = new TalkieLocaleHelper();
    const localeProvider = new WebExtensionEnvironmentLocaleProvider();

    const mapStateToProps$4 = state => {
      return {
        languages: selectors.shared.voices.getLanguages(state),
        languageGroups: selectors.shared.voices.getLanguageGroups(state),
        voicesCount: selectors.shared.voices.getVoicesCount(state),
        languagesCount: selectors.shared.voices.getLanguagesCount(state),
        languageGroupsCount: selectors.shared.voices.getLanguageGroupsCount(state),
        availableBrowserLanguageWithInstalledVoice: selectors.shared.voices.getAvailableBrowserLanguageWithInstalledVoice(state),
        isPremiumEdition: state.shared.metadata.isPremiumEdition,
        systemType: state.shared.metadata.systemType,
        osType: state.shared.metadata.osType
      };
    };

    const mapDispatchToProps$4 = dispatch => {
      return {
        actions: {
          sharedSpeaking: redux.bindActionCreators(actions.shared.speaking, dispatch),
          sharedVoices: redux.bindActionCreators(actions.shared.voices, dispatch)
        }
      };
    };

    let WelcomeContainer = (_dec$5 = reactRedux.connect(mapStateToProps$4, mapDispatchToProps$4), _dec$5(_class$g = (_temp$g = _class2$g = class WelcomeContainer extends React__default['default'].PureComponent {
      UNSAFE_componentWillReceiveProps(nextProps) {
        if (nextProps.voicesCount === 0) {
          // NOTE: since this welcome page is the first thing users see when installing Talkie, it's important that the voice list loads.
          // NOTE: sometimes the browser (Firefox?) has not actually loaded the voices (cold cache), and will instead synchronously return an empty array.
          // NOTE: wait a bit between retries, both to allow any voices to load, and to not bog down the system with a loop if there actually are no voices.
          const loadVoicesRetryDelay = 250;
          setTimeout(() => {
            // TODO: is this the best place to load data?
            this.props.actions.sharedVoices.loadVoices();
          }, loadVoicesRetryDelay);
        }
      }

      render() {
        const {
          actions,
          isPremiumEdition,
          systemType,
          osType,
          voicesCount,
          languagesCount,
          languageGroupsCount,
          languages,
          languageGroups,
          availableBrowserLanguageWithInstalledVoice
        } = this.props; // TODO: create action and store as redux state?

        const availableBrowserLanguageWithInstalledVoiceAndSampleText = availableBrowserLanguageWithInstalledVoice.filter(languageCode => {
          /* eslint-disable no-sync */
          return !!talkieLocaleHelper$1.getSampleTextSync(languageCode);
          /* eslint-enable no-sync */
        });
        const firstAvailableBrowserLanguageWithInstalledVoiceAndSampleText = availableBrowserLanguageWithInstalledVoiceAndSampleText[0] || null; // TODO: create action and store as redux state?

        const sampleTextLanguageCode = firstAvailableBrowserLanguageWithInstalledVoiceAndSampleText || null;
        let sampleText = null;

        if (sampleTextLanguageCode) {
          /* eslint-disable no-sync */
          sampleText = talkieLocaleHelper$1.getSampleTextSync(firstAvailableBrowserLanguageWithInstalledVoiceAndSampleText);
          /* eslint-enable no-sync */
        } // TODO: create action and store as redux state?


        const translationLocale = localeProvider.getTranslationLocale();
        const canSpeakInTranslatedLocale = languages.includes(translationLocale) || languageGroups.includes(translationLocale);
        return /*#__PURE__*/React__default['default'].createElement(Welcome, {
          isPremiumEdition: isPremiumEdition,
          systemType: systemType,
          osType: osType,
          voicesCount: voicesCount,
          languagesCount: languagesCount,
          languageGroupsCount: languageGroupsCount,
          canSpeakInTranslatedLocale: canSpeakInTranslatedLocale,
          sampleTextLanguageCode: sampleTextLanguageCode,
          sampleText: sampleText,
          speakTextInLanguageWithOverrides: actions.sharedSpeaking.speakTextInLanguageWithOverrides
        });
      }

    }, _defineProperty(_class2$g, "defaultProps", {
      isPremiumEdition: false,
      systemType: false,
      osType: false,
      languages: [],
      languageGroups: [],
      availableBrowserLanguageWithInstalledVoice: [],
      voicesCount: 0,
      languagesCount: 0,
      languageGroupsCount: 0,
      speakTextInLanguageWithOverrides: null
    }), _defineProperty(_class2$g, "propTypes", {
      actions: PropTypes__default['default'].object.isRequired,
      languages: PropTypes__default['default'].arrayOf(PropTypes__default['default'].string.isRequired).isRequired,
      languageGroups: PropTypes__default['default'].arrayOf(PropTypes__default['default'].string.isRequired).isRequired,
      availableBrowserLanguageWithInstalledVoice: PropTypes__default['default'].arrayOf(PropTypes__default['default'].string.isRequired).isRequired,
      voicesCount: PropTypes__default['default'].number.isRequired,
      languagesCount: PropTypes__default['default'].number.isRequired,
      languageGroupsCount: PropTypes__default['default'].number.isRequired,
      isPremiumEdition: PropTypes__default['default'].bool.isRequired,
      systemType: PropTypes__default['default'].string.isRequired,
      osType: PropTypes__default['default'].string
    }), _temp$g)) || _class$g);

    var _dec$6, _class$h, _class2$h, _temp$h;
    const widthStyles = {
      minWidth: "400px",
      maxWidth: "1000px"
    };
    const styles$1 = Object.assign({}, widthStyles, {
      minHeight: "450px",
      paddingBottom: "2em"
    });
    let Main = (_dec$6 = styledHoc(styles$1), translateHoc(_class$h = _dec$6(_class$h = passSelectedTextToBackgroundHoc(_class$h = (_temp$h = _class2$h = class Main extends React__default['default'].PureComponent {
      constructor(props) {
        super(props);
        this.handleLinkClick = this.handleLinkClick.bind(this);
        this.handleOpenShortKeysConfigurationClick = this.handleOpenShortKeysConfigurationClick.bind(this);
        this.handleOptionsPageClick = this.handleOptionsPageClick.bind(this); // TODO: better place to put navigation menu links?

        this.links = [{
          tabId: "welcome",
          text: this.props.translate("frontend_welcomeLinkText")
        }, {
          tabId: "voices",
          text: this.props.translate("frontend_voicesLinkText")
        }, {
          tabId: "usage",
          text: this.props.translate("frontend_usageLinkText")
        }, {
          tabId: "features",
          text: this.props.translate("frontend_featuresLinkText")
        }, {
          tabId: "support",
          text: this.props.translate("frontend_supportLinkText")
        }, {
          tabId: "about",
          text: this.props.translate("frontend_aboutLinkText")
        }];
        this.styled = {
          navHeader: styledHoc(Object.assign({}, widthStyles, {
            position: "fixed",
            top: 0,
            left: 0,
            right: 0,
            backgroundColor: "#ffffff"
          }))("div"),
          main: styledHoc({
            paddingTop: "8em"
          })(main$1),
          footerHr: styledHoc({
            marginTop: "3em"
          })(hr$1)
        };
      }

      componentDidMount() {
        // NOTE: execute outside the synchronous rendering.
        setTimeout(() => this.scrollToTop(), 100);
      }

      UNSAFE_componentWillReceiveProps(nextProps) {
        if (this.props.activeTabId !== nextProps.activeTabId) {
          this.scrollToTop();
        }
      }

      scrollToTop() {
        // NOTE: feels like this might be the wrong place to put this? Is there a better place?
        // NOTE: due to schuffling around elements, there's some confusion regarding which element to apply scrolling to.
        document.body.scrollTop = 0;
        window.scroll(0, 0);
      }

      handleOpenShortKeysConfigurationClick() {
        this.props.actions.sharedNavigation.openShortKeysConfiguration();
      }

      handleLinkClick(url) {
        this.props.actions.sharedNavigation.openUrlInNewTab(url);
      }

      handleOptionsPageClick(e) {
        e.preventDefault();
        e.stopPropagation();
        this.props.actions.sharedNavigation.openOptionsPage();
        return false;
      }

      render() {
        const {
          activeTabId,
          isPremiumEdition,
          versionNumber,
          systemType,
          osType,
          className
        } = this.props;
        const linksToShow = this.links;
        return /*#__PURE__*/React__default['default'].createElement("div", {
          className: className
        }, /*#__PURE__*/React__default['default'].createElement(this.styled.navHeader, null, /*#__PURE__*/React__default['default'].createElement(Header, {
          isPremiumEdition: isPremiumEdition
        }), /*#__PURE__*/React__default['default'].createElement(NavContainer, {
          links: linksToShow
        }), /*#__PURE__*/React__default['default'].createElement(hr$1, null)), /*#__PURE__*/React__default['default'].createElement(hr$1, null), /*#__PURE__*/React__default['default'].createElement(this.styled.main, {
          onClick: this.handleClick
        }, /*#__PURE__*/React__default['default'].createElement(TabContents, {
          id: "welcome",
          activeTabId: activeTabId,
          onLinkClick: this.handleLinkClick
        }, /*#__PURE__*/React__default['default'].createElement(WelcomeContainer, null)), /*#__PURE__*/React__default['default'].createElement(TabContents, {
          id: "voices",
          activeTabId: activeTabId,
          onLinkClick: this.handleLinkClick
        }, /*#__PURE__*/React__default['default'].createElement(VoicesContainer, null)), /*#__PURE__*/React__default['default'].createElement(TabContents, {
          id: "usage",
          activeTabId: activeTabId,
          onLinkClick: this.handleLinkClick
        }, /*#__PURE__*/React__default['default'].createElement(Usage, {
          isPremiumEdition: isPremiumEdition,
          systemType: systemType,
          osType: osType,
          onOpenShortKeysConfigurationClick: this.handleOpenShortKeysConfigurationClick
        })), /*#__PURE__*/React__default['default'].createElement(TabContents, {
          id: "features",
          activeTabId: activeTabId,
          onLinkClick: this.handleLinkClick
        }, /*#__PURE__*/React__default['default'].createElement(Features, {
          isPremiumEdition: isPremiumEdition,
          systemType: systemType
        })), /*#__PURE__*/React__default['default'].createElement(TabContents, {
          id: "support",
          activeTabId: activeTabId,
          onLinkClick: this.handleLinkClick
        }, /*#__PURE__*/React__default['default'].createElement(Support, {
          systemType: systemType,
          osType: osType
        })), /*#__PURE__*/React__default['default'].createElement(TabContents, {
          id: "about",
          activeTabId: activeTabId,
          onLinkClick: this.handleLinkClick
        }, /*#__PURE__*/React__default['default'].createElement(About, null))), /*#__PURE__*/React__default['default'].createElement(this.styled.footerHr, null), /*#__PURE__*/React__default['default'].createElement(Footer, {
          versionNumber: versionNumber,
          optionsPageClick: this.handleOptionsPageClick
        }));
      }

    }, _defineProperty(_class2$h, "defaultProps", {
      isPremiumEdition: false,
      versionNumber: null,
      systemType: null,
      osType: null,
      activeTabId: null
    }), _defineProperty(_class2$h, "propTypes", {
      actions: PropTypes__default['default'].object.isRequired,
      isPremiumEdition: PropTypes__default['default'].bool.isRequired,
      versionNumber: PropTypes__default['default'].string.isRequired,
      systemType: PropTypes__default['default'].string.isRequired,
      osType: PropTypes__default['default'].string,
      activeTabId: PropTypes__default['default'].string.isRequired,
      className: PropTypes__default['default'].string.isRequired,
      translate: PropTypes__default['default'].func.isRequired
    }), _temp$h)) || _class$h) || _class$h) || _class$h);

    var _dec$7, _class$i, _class2$i, _temp$i;

    const mapStateToProps$5 = state => {
      return {
        isPremiumEdition: state.shared.metadata.isPremiumEdition,
        versionNumber: state.shared.metadata.versionNumber,
        systemType: state.shared.metadata.systemType,
        osType: state.shared.metadata.osType,
        activeTabId: state.unshared.navigation.activeTabId
      };
    };

    const mapDispatchToProps$5 = dispatch => {
      return {
        actions: {
          sharedNavigation: redux.bindActionCreators(actions.shared.navigation, dispatch)
        }
      };
    };

    let App = (_dec$7 = reactRedux.connect(mapStateToProps$5, mapDispatchToProps$5), _dec$7(_class$i = (_temp$i = _class2$i = class App extends React__default['default'].PureComponent {
      render() {
        const {
          actions,
          isPremiumEdition,
          versionNumber,
          systemType,
          osType,
          activeTabId
        } = this.props;
        return /*#__PURE__*/React__default['default'].createElement(Main, {
          actions: actions,
          isPremiumEdition: isPremiumEdition,
          versionNumber: versionNumber,
          systemType: systemType,
          osType: osType,
          activeTabId: activeTabId
        });
      }

    }, _defineProperty(_class2$i, "defaultProps", {
      isPremiumEdition: false,
      versionNumber: null,
      systemType: null,
      osType: null,
      activeTabId: null
    }), _defineProperty(_class2$i, "propTypes", {
      actions: PropTypes__default['default'].object.isRequired,
      isPremiumEdition: PropTypes__default['default'].bool.isRequired,
      versionNumber: PropTypes__default['default'].string.isRequired,
      systemType: PropTypes__default['default'].string.isRequired,
      osType: PropTypes__default['default'].string,
      activeTabId: PropTypes__default['default'].string.isRequired
    }), _temp$i)) || _class$i);

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */
    // TODO: generalize preloading?

    const prerenderActionsToDispatch = [];
    const postrenderActionsToDispatch = [actions.shared.voices.loadVoices(), actions.shared.voices.loadNavigatorLanguages()];
    var loadRoot = (() => hydrateHtml(rootReducer, prerenderActionsToDispatch, postrenderActionsToDispatch, App));

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    const start = () => promiseTry(() => Promise.resolve().then(() => startReactFrontend()).then(() => loadRoot()).then(() => undefined));

    const stop = () => promiseTry( // NOTE: probably won't be correctly executed as before/unload doesn't guarantee asynchronous calls.
    () => stopReactFrontend().then(() => undefined));

    registerUnhandledRejectionHandler();
    document.addEventListener("DOMContentLoaded", eventToPromise.bind(null, start));
    window.addEventListener("beforeunload", eventToPromise.bind(null, stop));

}(ReactDOM, React, Redux, ReduxThunk, PropTypes, ReactRedux, Reselect));
//# sourceMappingURL=demo.js.map
