/**
 * This file is part of Talkie -- text-to-speech browser extension button.
 * <https://joelpurra.com/projects/talkie/>
 *
 * Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>
 *
 * Talkie is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Talkie is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
 *
 * ---
 *
 * # About the package "talkie"
 *
 * - Name: talkie
 * - Generated: 2021-01-21T21:06:26+01:00
 * - Version: 6.0.0
 * - License: GPL-3.0
 * - Author: Joel Purra <code+github@joelpurra.com> (https://joelpurra.com/)
 * - Homepage: https://joelpurra.com/projects/talkie/
 *
 *
 * ## Detected dependencies for this file:
 *
 * - Count: 0
 */

(function (factory) {
    typeof define === 'function' && define.amd ? define(factory) :
    factory();
}((function () { 'use strict';

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    const promiseTry = (fn) => new Promise(
        (resolve, reject) => {
            try {
                const result = fn();

                resolve(result);
            } catch (error) {
                reject(error);
            }
        },
    );

    const promiseSeries = (promiseFunctions, state) => promiseTry(
        () => {
            if (promiseFunctions.length === 0) {
                return undefined;
            }

            const first = promiseFunctions[0];

            if (promiseFunctions.length === 1) {
                return Promise.resolve(first(state));
            }

            const rest = promiseFunctions.slice(1);

            return Promise.resolve(first(state))
                .then((result) => promiseSeries(rest, result));
        },
    );

    const promiseTimeout = (promise, limit) => {
        let timeoutId = null;

        const timeoutPromise = new Promise((resolve) => {
            timeoutId = setTimeout(() => resolve(), limit);
        })
            .then(() => {
                timeoutId = null;

                const timeoutError = new Error(`Timeout after ${limit} milliseconds.`);
                timeoutError.name = "PromiseTimeout";

                throw timeoutError;
            });

        const originalPromise = promise
            .then((result) => {
                // NOTE: timeout has already happened.
                if (timeoutId === null) {
                    return undefined;
                }

                // NOTE: timeout has not yet happened.
                clearTimeout(timeoutId);
                timeoutId = null;

                return result;
            })
            .catch((error) => {
                // NOTE: timeout has already happened.
                if (timeoutId === null) {
                    return undefined;
                }

                // NOTE: timeout has not yet happened.
                clearTimeout(timeoutId);
                timeoutId = null;

                throw error;
            });

        return Promise.race([
            originalPromise,
            timeoutPromise,
        ]);
    };

    const promiseSleep = (fn, sleep) => {
        return new Promise((resolve, reject) => {
            setTimeout(() => {
                try {
                    resolve(fn());
                } catch (error) {
                    reject(error);
                }
            }, sleep);
        });
    };

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class WebExtensionEnvironmentManifestProvider {
        getSync() {
            // NOTE: synchronous call.
            // https://developer.mozilla.org/en-US/Add-ons/WebExtensions/API/runtime/getManifest
            const manifest = browser.runtime.getManifest();

            return manifest;
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    const manifestProvider = new WebExtensionEnvironmentManifestProvider();

    // TODO: configuration.
    const extensionShortName = "Talkie";

    // https://stackoverflow.com/questions/12830649/check-if-chrome-extension-installed-in-unpacked-mode
    // https://stackoverflow.com/a/20227975
    /* eslint-disable no-sync */
    const isDevMode = () => !("update_url" in manifestProvider.getSync());
    /* eslint-enable no-sync */

    // NOTE: 0, 1, ...
    const loggingLevels = [
        "TRAC",
        "DEBG",
        "INFO",
        "WARN",
        "ERRO",

        // NOTE: should "always" be logged, presumably for technical reasons.
        "ALWA",

        // NOTE: turns off logging output.
        "NONE",
    ];

    const parseLevelName = (nextLevelName) => {
        if (typeof nextLevelName !== "string") {
            throw new TypeError("nextLevelName");
        }

        const normalizedLevelName = nextLevelName.toUpperCase();

        const levelIndex = loggingLevels.indexOf(normalizedLevelName);

        if (typeof levelIndex === "number" && Math.floor(levelIndex) === Math.ceil(levelIndex) && levelIndex >= 0 && levelIndex < loggingLevels.length) {
            return levelIndex;
        }

        throw new TypeError("nextLevel");
    };

    const parseLevel = (nextLevel) => {
        if (typeof nextLevel === "number" && Math.floor(nextLevel) === Math.ceil(nextLevel) && nextLevel >= 0 && nextLevel < loggingLevels.length) {
            return nextLevel;
        }

        const levelIndex = parseLevelName(nextLevel);

        return levelIndex;
    };

    // NOTE: default logging level differs for developers using the unpacked extension, and "normal" usage.
    let currentLevelIndex = isDevMode() ? parseLevel("DEBG") : parseLevel("WARN");

    const setLevel = (nextLevel) => {
        currentLevelIndex = parseLevel(nextLevel);
    };

    // NOTE: allows switching logging to strings only, to allow terminal output logging (where only one string argument is shown).
    let stringOnlyOutput = false;

    const setStringOnlyOutput = (stringOnly) => {
        stringOnlyOutput = (stringOnly === true);
    };

    const generateLogger = (loggingLevelName, consoleFunctioName) => {
        const functionLevelIndex = parseLevel(loggingLevelName);

        const logger = (...args) => {
            if (functionLevelIndex < currentLevelIndex) {
                return;
            }

            const now = new Date().toISOString();

            let loggingArgs = [
                loggingLevels[functionLevelIndex],
                now,
                extensionShortName,
                ...args,
            ];

            if (stringOnlyOutput) {
                // NOTE: for chrome command line console debugging.
                // NOTE: has to be an array.
                loggingArgs = [
                    JSON.stringify(loggingArgs),
                ];
            }

            /* eslint-disable no-console */
            console[consoleFunctioName](...loggingArgs);
            /* eslint-enable no-console */
        };

        return logger;
    };

    const logTrace = generateLogger("TRAC", "log");
    const logDebug = generateLogger("DEBG", "log");
    const logInfo = generateLogger("INFO", "info");
    const logWarn = generateLogger("WARN", "warn");
    const logError = generateLogger("ERRO", "error");
    const logAlways = generateLogger("ALWA", "log");

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    const loggedPromise = (...args) => {
        const fn = args.pop();

        return (...fnArgs) => {
            return Promise.resolve()
                .then(() => {
                    logDebug("Start", "loggedPromise", ...args, ...fnArgs);

                    return undefined;
                })
                .then(() => fn(...fnArgs))
                .then((result) => {
                    logDebug("Done", "loggedPromise", ...args, ...fnArgs, result);

                    return result;
                })
                .catch((error) => {
                    logError("loggedPromise", ...args, ...fnArgs, error);

                    throw error;
                });
        };
    };

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    const handleUnhandledRejection = (event) => {
        logWarn("Unhandled rejection", "Error", event.reason, event.promise, event);

        logInfo("Starting debugger, if attached.");
        /* eslint-disable no-debugger */
        debugger;
        /* eslint-enable no-debugger */
    };

    const registerUnhandledRejectionHandler = () => {
        window.addEventListener("unhandledrejection", handleUnhandledRejection);
    };

    var shared = {
    	urls: {
    		cla: "https://joelpurra.com/projects/talkie/CLA.md",
    		github: "https://github.com/joelpurra/talkie",
    		gpl: "https://www.gnu.org/licenses/gpl.html",
    		main: "https://joelpurra.com/projects/talkie/",
    		project: "https://joelpurra.com/projects/talkie/",
    		"shortcut-keys": "https://joelpurra.com/projects/talkie/#shortcut-keys",
    		"support-feedback": "https://joelpurra.com/support/",
    		chromewebstore: "https://chrome.google.com/webstore/detail/enfbcfmmdpdminapkflljhbfeejjhjjk",
    		"firefox-amo": "https://addons.mozilla.org/en-US/firefox/addon/talkie/",
    		"primary-payment": "https://www.paypal.me/joelpurrade",
    		"alternative-payment": "https://joelpurra.com/donate/",
    		share: {
    			twitter: "https://twitter.com/intent/tweet?text=Using%20Talkie%20to%20read%20text%20for%20me%20-%20text%20to%20speech%20works%20great!&url=https://joelpurra.com/projects/talkie/&via=joelpurra&related=&hashtags=texttospeech,tts",
    			facebook: "https://www.facebook.com/sharer/sharer.php?u=https%3A//joelpurra.com/projects/talkie/",
    			googleplus: "https://plus.google.com/share?url=https%3A//joelpurra.com/projects/talkie/",
    			linkedin: "https://www.linkedin.com/shareArticle?url=https%3A//joelpurra.com/projects/talkie/"
    		}
    	}
    };
    var chrome = {
    	urls: {
    		rate: "https://chrome.google.com/webstore/detail/enfbcfmmdpdminapkflljhbfeejjhjjk/reviews",
    		"support-feedback": "https://chrome.google.com/webstore/detail/enfbcfmmdpdminapkflljhbfeejjhjjk/support"
    	}
    };
    var webextension = {
    	urls: {
    		rate: "https://addons.mozilla.org/en-US/firefox/addon/talkie/reviews/"
    	}
    };
    var configurationObject = {
    	shared: shared,
    	chrome: chrome,
    	webextension: webextension
    };

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class Configuration {
        // NOTE: keep SynchronousConfiguration and Configuration in... sync.
        constructor(metadataManager, configurationObject) {
            this.metadataManager = metadataManager;
            this.configurationObject = configurationObject;

            this._initialize();
        }

        _initialize() {
            this.configurationObject.shared.urls.root = "/";
            this.configurationObject.shared.urls.demo = "/src/demo/demo.html";
            this.configurationObject.shared.urls.options = "/src/options/options.html";
            this.configurationObject.shared.urls.popup = "/src/popup/popup.html";

            // NOTE: direct links to individual tabs.
            this.configurationObject.shared.urls["demo-about"] = this.configurationObject.shared.urls.demo + "#about";
            this.configurationObject.shared.urls["demo-features"] = this.configurationObject.shared.urls.demo + "#features";
            this.configurationObject.shared.urls["demo-support"] = this.configurationObject.shared.urls.demo + "#support";
            this.configurationObject.shared.urls["demo-usage"] = this.configurationObject.shared.urls.demo + "#usage";
            this.configurationObject.shared.urls["demo-voices"] = this.configurationObject.shared.urls.demo + "#voices";
            this.configurationObject.shared.urls["demo-welcome"] = this.configurationObject.shared.urls.demo + "#welcome";

            // NOTE: direct links to individual tabs.
            // NOTE: need to pass a parameter to the options page.
            [
                "popup",
                "demo",
            ].forEach((from) => {
                this.configurationObject.shared.urls[`options-from-${from}`] = this.configurationObject.shared.urls.options + `?from=${from}`;
                this.configurationObject.shared.urls[`options-about-from-${from}`] = this.configurationObject.shared.urls[`options-from-${from}`] + "#about";
                this.configurationObject.shared.urls[`options-upgrade-from-${from}`] = this.configurationObject.shared.urls[`options-from-${from}`] + "#upgrade";
            });

            this.configurationObject.shared.urls["popup-passclick-false"] = this.configurationObject.shared.urls.popup + "?passclick=false";
        }

        _resolvePath(obj, path) {
            // NOTE: doesn't handle arrays nor properties of "any" non-object objects.
            if (!obj || typeof obj !== "object") {
                throw new Error();
            }

            if (!path || typeof path !== "string" || path.length === 0) {
                throw new Error();
            }

            // NOTE: doesn't handle path["subpath"].
            const parts = path.split(".");
            const part = parts.shift();

            if (({}).hasOwnProperty.call(obj, part)) {
                if (parts.length === 0) {
                    return obj[part];
                }
                return this._resolvePath(obj[part], parts.join("."));
            }

            return null;
        }

        get(path) {
            return promiseTry(
                () => this.metadataManager.getSystemType()
                    .then((systemType) =>
                        /* eslint-disable no-sync */
                        this.getSync(systemType, path),
                        /* eslint-enable no-sync */
                    ),
            );
        }

        getSync(systemType, path) {
            const systemValue = this._resolvePath(this.configurationObject[systemType], path);
            const sharedValue = this._resolvePath(this.configurationObject.shared, path);

            const value = systemValue
                            || sharedValue
                            || null;

            return value;
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class WebExtensionEnvironmentLocaleProvider {
        getUILocale() {
            const locale = browser.i18n.getMessage("@@ui_locale");

            return locale;
        }

        getTranslationLocale() {
            const locale = browser.i18n.getMessage("extensionLocale");

            return locale;
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class WebExtensionEnvironmentTranslatorProvider {
        constructor(localeProvider) {
            // TODO REMOVE: unused.
            this.localeProvider = localeProvider;
        }

        translate(key, extras) {
            // const locale = this.localeProvider.getTranslationLocale();

            // TODO: use same translation system in frontend and backend?
            const translated = browser.i18n.getMessage(key, extras);

            return translated;
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class WebExtensionEnvironmentInternalUrlProvider {
        getSync(url) {
            // NOTE: synchronous call.
            // https://developer.mozilla.org/en-US/Add-ons/WebExtensions/API/runtime/getURL
            const internalUrl = browser.runtime.getURL(url);

            return internalUrl;
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    const getBackgroundPage = () => promiseTry(
        // https://developer.browser.com/extensions/runtime.html#method-getBackgroundPage
        () => browser.runtime.getBackgroundPage()
            .then((backgroundPage) => {
                if (backgroundPage) {
                    return backgroundPage;
                }

                return null;
            }),
    );

    const getCurrentActiveTab = () => promiseTry(
        () => {
            const queryOptions = {
                "active": true,
                "currentWindow": true,
            };

            // https://developer.browser.com/extensions/tabs#method-query
            return browser.tabs.query(queryOptions)
                .then((tabs) => {
                    if (!tabs) {
                        return null;
                    }

                    const singleTabResult = tabs.length === 1;

                    const tab = tabs[0] || null;

                    logDebug("getCurrentActiveTab", tabs, tab, singleTabResult);

                    if (singleTabResult) {
                        return tab;
                    }

                    return null;
                });
        },
    );

    const getCurrentActiveTabId = () => getCurrentActiveTab()
        .then((activeTab) => {
            if (activeTab) {
                return activeTab.id;
            }

            // NOTE: some tabs can't be retreived.
            return null;
        });

    const isCurrentPageInternalToTalkie = (internalUrlProvider) => promiseTry(
        () => getCurrentActiveTab()
            .then((tab) => {
                if (tab) {
                    const url = tab.url;

                    if (
                        typeof url === "string"
                        && (
                            /* eslint-disable no-sync */
                            url.startsWith(internalUrlProvider.getSync("/src/"))
                            || url.startsWith(internalUrlProvider.getSync("/dist/"))
                            /* eslint-enable no-sync */
                        )
                    ) {
                        return true;
                    }

                    return false;
                }

                return false;
            }),
    );

    const getCurrentActiveNormalLoadedTab = () => promiseTry(
        () => {
            const queryOptions = {
                "active": true,
                "currentWindow": true,
                "windowType": "normal",
                "status": "complete",
            };

            // https://developer.browser.com/extensions/tabs#method-query
            return browser.tabs.query(queryOptions)
                .then((tabs) => {
                    const singleTabResult = tabs.length === 1;

                    const tab = tabs[0] || null;

                    logDebug("getCurrentActiveNormalLoadedTab", tabs, tab, singleTabResult);

                    if (singleTabResult) {
                        return tab;
                    }

                    return null;
                });
        },
    );

    const canTalkieRunInTab = () => promiseTry(
        () => getCurrentActiveNormalLoadedTab()
            .then((tab) => {
                if (tab) {
                    const url = tab.url;

                    if (typeof url === "string") {
                        if (
                            (
                                // NOTE: whitelisting schemes.
                                // TODO: can the list be extended?
                                url.startsWith("http://")
                                || url.startsWith("https://")
                                || url.startsWith("ftp://")
                                || url.startsWith("file:")
                            )
                            && !(
                                // NOTE: blacklisting known (per-browser store) urls.
                                // TODO: should the list be extended?
                                // TODO: move to configuration.
                                url.startsWith("https://chrome.google.com/")
                                || url.startsWith("https://addons.mozilla.org/")
                            )
                        ) {
                            return true;
                        }

                        return false;
                    }

                    return false;
                }

                return false;
            }),
    );

    // NOTE: used to check if a DOM element cross-page (background, popup, options, ...) reference was used after it was supposed to be unreachable (memory leak).
    // https://developer.mozilla.org/de/docs/Web/JavaScript/Reference/Errors/Dead_object
    const isDeadWrapper = (domElementReference) => {
        try {
            String(domElementReference);

            return false;
        }
        catch (e) {
            return true;
        }
    };

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class WebExtensionEnvironmentStorageProvider {
        get(key) {
            return promiseTry(() => getBackgroundPage()
                .then((background) => {
                    const valueJson = background.localStorage.getItem(key);

                    if (valueJson === null) {
                        return null;
                    }

                    const value = JSON.parse(valueJson);

                    return value;
                }),
            );
        }

        set(key, value) {
            return promiseTry(() => getBackgroundPage()
                .then((background) => {
                    const valueJson = JSON.stringify(value);

                    background.localStorage.setItem(key, valueJson);

                    return undefined;
                }),
            );
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    const knownEvents = {
        beforeSpeaking: "beforeSpeaking",
        stopSpeaking: "stopSpeaking",
        afterSpeaking: "afterSpeaking",
        beforeSpeakingPart: "beforeSpeakingPart",
        afterSpeakingPart: "afterSpeakingPart",
        updateProgress: "updateProgress",
        resetProgress: "resetProgress",
        addProgress: "addProgress",
        finishProgress: "finishProgress",
        passSelectedTextToBackground: "passSelectedTextToBackground",
    };

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    const openUrlInNewTab = (url) => promiseTry(
        () => {
            if (typeof url !== "string") {
                throw new Error("Bad url: " + url);
            }

            // NOTE: only https urls.
            if (!url.startsWith("https://")) {
                throw new Error("Bad url, only https:// allowed: " + url);
            }

            return browser.tabs.create({
                active: true,
                url: url,
            });
        },
    );

    const openInternalUrlInNewTab = (url) => promiseTry(
        () => {
            if (typeof url !== "string") {
                throw new Error("Bad url: " + url);
            }

            // NOTE: only root-relative internal urls.
            // NOTE: double-slash is protocol relative, checking just in case.
            if (!url.startsWith("/") || url[1] === "/") {
                throw new Error("Bad url, only internally rooted allowed: " + url);
            }

            return browser.tabs.create({
                active: true,
                url: url,
            });
        },
    );

    const openUrlFromConfigurationInNewTab = (id) => promiseTry(
        () => getBackgroundPage()
            .then((background) => background.getConfigurationValue(`urls.${id}`))
            .then((url) => {
                if (typeof url !== "string") {
                    throw new Error("Bad url for id: " + id);
                }

                return openUrlInNewTab(url);
            }),
    );

    const openInternalUrlFromConfigurationInNewTab = (id) => promiseTry(
        () => getBackgroundPage()
            .then((background) => background.getConfigurationValue(`urls.${id}`))
            .then((url) => {
                if (typeof url !== "string") {
                    throw new Error("Bad url for id: " + id);
                }

                return openInternalUrlInNewTab(url);
            }),
    );

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class TalkieProgress {
        constructor(broadcaster, min, max, current) {
            this.broadcaster = broadcaster;
            this.interval = null;
            this.intervalDelay = 100;
            this.minSpeed = 0.015;

            this.resetProgress(min, max, current);
        }

        getEventData() {
            const eventData = {
                min: this.min,
                max: this.max,
                current: this.current,
            };

            return eventData;
        }

        broadcastEvent(eventName) {
            const eventData = this.getEventData();

            return this.broadcaster.broadcastEvent(eventName, eventData);
        }

        getPercent() {
            if (this.max === 0) {
                return 0;
            }

            const pct = (this.current / this.max) * 100;

            return pct;
        }

        updateProgress() {
            this.broadcastEvent(knownEvents.updateProgress);
        }

        resetProgress(min, max, current) {
            const now = Date.now();

            this.resetTime = now;
            this.min = min || 0;
            this.max = max || 0;
            this.current = current || 0;
            this.segmentSum = this.min;
            // this.segments = [];

            this.broadcastEvent(knownEvents.resetProgress);

            this.updateProgress();
        }

        addProgress(n) {
            const segmentLimited = Math.min(this.segmentSum, this.current + n);

            this.current = segmentLimited;

            this.broadcastEvent(knownEvents.addProgress);

            this.updateProgress();
        }

        getSpeed() {
            const now = Date.now();

            const timeDiff = now - this.resetTime;

            if (timeDiff === 0) {
                return this.minSpeed;
            }

            const speed = this.current / timeDiff;

            const adjustedSpeed = Math.max(speed, this.minSpeed);

            return adjustedSpeed;
        }

        intervalIncrement() {
            const now = Date.now();
            const intervalDiff = now - this.previousInterval;
            const speed = this.getSpeed();
            const increment = intervalDiff * speed;

            this.previousInterval = now;

            this.addProgress(increment);
        }

        startSegment(n) {
            const now = Date.now();

            this.previousInterval = now;

            this.segmentSum += n;

            this.interval = setInterval(() => this.intervalIncrement(), this.intervalDelay);
        }

        endSegment() {
            clearInterval(this.interval);

            this.interval = null;

            this.current = this.segmentSum;

            this.updateProgress();
        }

        finishProgress() {
            this.current = this.max;

            this.broadcastEvent(knownEvents.finishProgress);

            this.updateProgress();
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class Broadcaster {
        constructor() {
            this.actionListeningMap = {};
        }

        unregisterListeningAction(actionName, listeningActionHandler) {
            return promiseTry(
                () => {
                    if (!this.actionListeningMap[actionName]) {
                        throw new Error("No listening action(s) registered for action: " + actionName);
                    }

                    const countBefore = this.actionListeningMap[actionName].length;

                    this.actionListeningMap[actionName] = this.actionListeningMap[actionName].filter((registeredListeningActionHandler) => registeredListeningActionHandler !== listeningActionHandler);

                    const countAfter = this.actionListeningMap[actionName].length;

                    if (countBefore === countAfter) {
                        throw new Error("The specific listening action handler was not registered for action: " + actionName);
                    }
                },
            );
        }

        registerListeningAction(actionName, listeningActionHandler) {
            return promiseTry(
                () => {
                    this.actionListeningMap[actionName] = (this.actionListeningMap[actionName] || []).concat(listeningActionHandler);

                    const killSwitch = () => {
                        // NOTE: the promise chain probably won't be completed (by the caller, outside of this function), as the kill switch might be executed during the "onunload" event.
                        return this.unregisterListeningAction(actionName, listeningActionHandler);
                    };

                    return killSwitch;
                },
            );
        }

        broadcastEvent(actionName, actionData) {
            return promiseTry(
                () => {
                    logTrace("Start", "Sending message", actionName, actionData);

                    const listeningActions = this.actionListeningMap[actionName];

                    if (!listeningActions || listeningActions.length === 0) {
                        logTrace("Skipping", "Sending message", actionName, actionData);

                        return [];
                    }

                    const listeningActionPromises = listeningActions.map((action) => {
                        return promiseTry(
                            () => {
                                // NOTE: check for dead objects from cross-page (background, popup, options, ...) memory leaks.
                                // NOTE: this is just in case the killSwitch hasn't been called.
                                // https://developer.mozilla.org/en-US/docs/Extensions/Common_causes_of_memory_leaks_in_extensions#Failing_to_clean_up_event_listeners
                                // TODO: throw error instead of cleaning up?
                                // TODO: clean up code to avoid memory leaks, primarly in Firefox as it doesn't have onSuspend at the moment.
                                if (isDeadWrapper(action)) {
                                    logWarn("Dead wrapper (detected)", "Sending message", actionName, actionData);

                                    return this.unregisterListeningAction(actionName, action);
                                }

                                return action(actionName, actionData);
                            })
                            .catch((error) => {
                                if (error && typeof error.message === "string" && error.message.includes("access dead object")) {
                                    // NOTE: it's a dead wrapper, but it wasn't detected by isDeadWrapper() above. Ignore.
                                    logWarn("Dead wrapper (caught)", "Sending message", actionName, actionData);

                                    return this.unregisterListeningAction(actionName, action);
                                }

                                throw error;
                            });
                    });

                    return Promise.all(listeningActionPromises)
                        .then((responses) => {
                            logTrace("Done", "Sending message", actionName, actionData, responses);

                            return responses;
                        })
                        .catch((error) => {
                            logError("Sending message", actionName, actionData);

                            throw error;
                        });
                });
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class ContentLogger {
        constructor(execute, configuration) {
            this.execute = execute;
            this.configuration = configuration;

            this.executeLogToPageCode = "(function(){ try { console.log(%a); } catch (error) { console.error('Talkie', 'logToPage', error); } }());";
            this.executeLogToPageWithColorCode = "(function(){ try { console.log(%a); } catch (error) { console.error('Talkie', 'logToPageWithColor', error); } }());";
        }

        logToPage(...args) {
            return promiseTry(
                () => {
                    const now = new Date().toISOString();

                    const logValues = [
                        now,
                        // TODO: configuration.
                        "Talkie",
                        ...args,
                    ]
                        .map((arg) => JSON.stringify(arg))
                        .join(", ");

                    const code = this.executeLogToPageCode.replace("%a", logValues);

                    return this.execute.scriptInTopFrame(code)
                        .catch((error) => {
                            // NOTE: reduced logging for known tab/page access problems.
                            if (error && typeof error.message === "string" && error.message.startsWith("Cannot access")) {
                                logInfo("this.execute.logToPage", "Error", error, ...args);
                            } else {
                                logWarn("this.execute.logToPage", "Error", error, ...args);
                            }

                            throw error;
                        });
                },
            );
        }

        logToPageWithColor(...args) {
            return promiseTry(
                () => {
                    const now = new Date().toISOString();

                    // NOTE: create one long console.log() string argument, then add the color argument second.
                    const logValuesArrayAsString = [
                        now,
                        // TODO: configuration.
                        "Talkie",
                        "%c",
                        ...args,
                    ]
                        .join(" ");

                    const logValues = JSON.stringify(logValuesArrayAsString)
                        + ", "
                        + JSON.stringify("background: #007F41; color: #FFFFFF; padding: 0.3em;");

                    const code = this.executeLogToPageWithColorCode.replace("%a", logValues);

                    return this.execute.scriptInTopFrame(code)
                        .catch((error) => {
                            // NOTE: reduced logging for known tab/page access problems.
                            if (error && typeof error.message === "string" && error.message.startsWith("Cannot access")) {
                                logInfo("this.execute.logToPageWithColor", ...args);
                            } else {
                                logWarn("this.execute.logToPageWithColor", ...args);
                            }

                            throw error;
                        });
                },
            );
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class Plug {
        constructor(contentLogger, execute, configuration) {
            this.contentLogger = contentLogger;
            this.execute = execute;
            this.configuration = configuration;

            this.executeGetTalkieWasPluggedCode = "(function(){ return window.talkieWasPlugged; }());";
            this.executeSetTalkieWasPluggedCode = "(function(){ window.talkieWasPlugged = true; }());";
        }

        executePlug() {
            return promiseTry(
                () => {
                    return Promise.resolve()
                        // TODO: premium version of the same message?
                        .then(() => this.contentLogger.logToPageWithColor("Thank you for using Talkie!"))
                        .then(() => this.contentLogger.logToPageWithColor("https://joelpurra.com/projects/talkie/"))
                        .then(() => this.contentLogger.logToPageWithColor("Created by Joel Purra. Released under GNU General Public License version 3.0 (GPL-3.0)"))
                        .then(() => this.contentLogger.logToPageWithColor("https://joelpurra.com/"))
                        .then(() => this.contentLogger.logToPageWithColor("If you like Talkie, send a link to your friends -- and consider upgrading to Talkie Premium to support further open source development."));
                },
            );
        }

        executeGetTalkieWasPlugged() {
            return this.execute.scriptInTopFrameWithTimeout(this.executeGetTalkieWasPluggedCode, 1000);
        }

        executeSetTalkieWasPlugged() {
            return this.execute.scriptInTopFrameWithTimeout(this.executeSetTalkieWasPluggedCode, 1000);
        }

        once() {
            return this.executeGetTalkieWasPlugged()
                .then((talkieWasPlugged) => {
                    if (talkieWasPlugged && talkieWasPlugged.toString() !== "true") {
                        return this.executePlug()
                            .then(() => this.executeSetTalkieWasPlugged());
                    }

                    return undefined;
                });
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class SuspensionConnectorManager {
        constructor() {
            // NOTE: could be made configurable, in case there are multiple reasons to manage suspension.
            this.preventSuspensionPortName = "talkie-prevents-suspension";

            this.talkiePreventSuspensionPort = null;
        }

        _connectToStayAlive() {
            return promiseTry(
                () => {
                    logDebug("Start", "_connectToStayAlive");

                    const preventSuspensionConnectOptions = {
                        name: this.preventSuspensionPortName,
                    };

                    this.talkiePreventSuspensionPort = browser.runtime.connect(preventSuspensionConnectOptions);

                    if (this.talkiePreventSuspensionPort === null) {
                        throw new Error(`Could not connect to ${this.preventSuspensionPortName}.`);
                    }

                    const onDisconnectHandler = () => {
                        logDebug("onDisconnect", "_connectToStayAlive");

                        this.talkiePreventSuspensionPort = null;
                    };

                    this.talkiePreventSuspensionPort.onDisconnect.addListener(onDisconnectHandler);

                    const _onMessageHandler = (msg) => {
                        logDebug("_onMessageHandler", "_connectToStayAlive", msg); ;
                    };

                    // NOTE: this message listener is unneccessary.
                    this.talkiePreventSuspensionPort.onMessage.addListener(_onMessageHandler);

                    this.talkiePreventSuspensionPort.postMessage("Hello from the SuspensionConnectorManager.");

                    logDebug("Done", "_connectToStayAlive");
                },
            );
        }

        _disconnectToDie() {
            return promiseTry(
                () => {
                    logDebug("Start", "_disconnectToDie");

                    if (this.talkiePreventSuspensionPort === null) {
                        // TODO: investigate if this should happen during normal operation, or not.
                        // throw new Error("this.talkiePreventSuspensionPort is null");
                        logDebug("Done", "_disconnectToDie", "already null");

                        return;
                    }

                    this.talkiePreventSuspensionPort.postMessage("Goodbye from the SuspensionConnectorManager.");

                    // https://developer.browser.com/extensions/runtime#type-Port
                    // NOTE: should work irregardless if the port was connected or not.
                    this.talkiePreventSuspensionPort.disconnect();

                    this.talkiePreventSuspensionPort = null;

                    logDebug("Done", "_disconnectToDie");
                },
            );
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class SuspensionManager {
        constructor(suspensionConnectorManager) {
            // NOTE: the iframe takes care of the SuspensionListenerManager.
            this.suspensionConnectorManager = suspensionConnectorManager;

            this.stayAliveElementId = "stay-alive-iframe";
            this.stayAliveHtmlPath = "/src/stay-alive/stay-alive.html";
        }

        _getExistingIframe() {
            return promiseTry(
                () => {
                    const existingIframe = document.getElementById(this.stayAliveElementId);

                    return existingIframe;
                },
            );
        }

        _isInitialized() {
            return this._getExistingIframe()
                .then((existingIframe) => existingIframe !== null);
        }

        _ensureIsInitialized() {
            return this._isInitialized()
                .then((isInitialized) => {
                    if (isInitialized === true) {
                        return undefined;
                    }

                    throw new Error("this.stayAliveElementId did not exist.");
                });
        }

        _ensureIsNotInitialized() {
            return this._isInitialized()
                .then((isInitialized) => {
                    if (isInitialized === false) {
                        return undefined;
                    }

                    throw new Error("this.stayAliveElementId exists.");
                });
        }

        _injectBackgroundFrame() {
            return this._ensureIsNotInitialized()
                .then(() => {
                    const iframe = document.createElement("iframe");
                    iframe.id = this.stayAliveElementId;
                    /* eslint-disable no-sync */
                    iframe.src = this.stayAliveHtmlPath;
                    /* eslint-enable no-sync */
                    document.body.appendChild(iframe);

                    return undefined;
                });
        }

        _removeBackgroundFrame() {
            return this._ensureIsInitialized()
                .then(() => this._getExistingIframe())
                .then((existingIframe) => {
                    // NOTE: trigger onunload.
                    // https://stackoverflow.com/questions/8677113/how-to-trigger-onunload-event-when-removing-iframe
                    existingIframe.src = "about:blank";
                    existingIframe.src = "the-id-does-not-matter-now";

                    // NOTE: ensure the src change has time to take effect.
                    return promiseSleep(() => {
                        existingIframe.parentNode.removeChild(existingIframe);
                    }, 10);
                });
        }

        initialize() {
            return promiseTry(
                () => {
                    logDebug("Start", "SuspensionManager.initialize");

                    return this._injectBackgroundFrame()
                        .then(() => {
                            logDebug("Done", "SuspensionManager.initialize");

                            return undefined;
                        });
                },
            );
        }

        unintialize() {
            return promiseTry(
                () => {
                    logDebug("Start", "SuspensionManager.unintialize");

                    return this._removeBackgroundFrame()
                        .then(() => {
                            logDebug("Done", "SuspensionManager.unintialize");

                            return undefined;
                        });
                },
            );
        }

        preventExtensionSuspend() {
            return promiseTry(
                () => {
                    logInfo("SuspensionManager.preventExtensionSuspend");

                    return this._ensureIsInitialized()
                        .then(() => this.suspensionConnectorManager._connectToStayAlive());
                },
            );
        }

        allowExtensionSuspend() {
            return promiseTry(
                () => {
                    logInfo("SuspensionManager.allowExtensionSuspend");

                    return this._ensureIsInitialized()
                        .then(() => this.suspensionConnectorManager._disconnectToDie());
                },
            );
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    const shallowCopy = (...objs) => Object.assign({}, ...objs);

    const last = (indexable) => indexable[indexable.length - 1];

    const flatten = (deepArray) => {
        if (!Array.isArray(deepArray)) {
            return deepArray;
        }

        if (deepArray.length === 0) {
            return [];
        }

        if (deepArray.length === 1) {
            return [].concat(flatten(deepArray[0]));
        }

        return [].concat(flatten(deepArray[0])).concat(flatten(deepArray.slice(1)));
    };

    const isUndefinedOrNullOrEmptyOrWhitespace = (str) => !(str && typeof str === "string" && str.length > 0 && str.trim().length > 0);

    // Polyfill for Math.round()
    // https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Math/round
    // TODO: don't add non-standard functions to Math.
    // Closure
    (function() {
        /**
       * Decimal adjustment of a number.
       *
       * @param {String}  type  The type of adjustment.
       * @param {Number}  value The number.
       * @param {Integer} exp   The exponent (the 10 logarithm of the adjustment base).
       * @returns {Number} The adjusted value.
       */
        function decimalAdjust(type, value, exp) {
            // If the exp is undefined or zero...
            if (typeof exp === "undefined" || +exp === 0) {
                return Math[type](value);
            }
            value = +value;
            exp = +exp;
            // If the value is not a number or the exp is not an integer...
            if (isNaN(value) || !(typeof exp === "number" && exp % 1 === 0)) {
                return NaN;
            }
            // If the value is negative...
            if (value < 0) {
                return -decimalAdjust(type, -value, exp);
            }
            // Shift
            value = value.toString().split("e");
            value = Math[type](+(value[0] + "e" + (value[1] ? (+value[1] - exp) : -exp)));
            // Shift back
            value = value.toString().split("e");
            return +(value[0] + "e" + (value[1] ? (+value[1] + exp) : exp));
        }

        // Decimal round
        if (!Math.round10) {
            Math.round10 = function(value, exp) {
                return decimalAdjust("round", value, exp);
            };
        }
        // Decimal floor
        if (!Math.floor10) {
            Math.floor10 = function(value, exp) {
                return decimalAdjust("floor", value, exp);
            };
        }
        // Decimal ceil
        if (!Math.ceil10) {
            Math.ceil10 = function(value, exp) {
                return decimalAdjust("ceil", value, exp);
            };
        }
    })();

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    const getVoices = () => promiseTry(
        () => {
            return getBackgroundPage()
                .then((background) => background.getAllVoices())
                .then((voices) => {
                    if (!Array.isArray(voices)) {
                        // NOTE: the list of voices could still be empty, either due to slow loading (cold cache) or that there actually are no voices loaded.
                        throw new Error("Could not load list of voices from browser.");
                    }

                    return voices;
                });
        },
    );

    const getMappedVoice = voice => {
        return {
            default: voice.default,
            lang: voice.lang,
            localService: voice.localService,
            name: voice.name,
            voiceURI: voice.voiceURI,
        };
    };

    const resolveVoice = (mappedVoice) => {
        return promiseTry(
            () => {
                return getVoices()
                    .then((voices) => {
                        const actualMatchingVoicesByName = voices.filter((voice) => mappedVoice.name && (voice.name === mappedVoice.name));
                        const actualMatchingVoicesByLanguage = voices.filter((voice) => mappedVoice.lang && (voice.lang === mappedVoice.lang));
                        const actualMatchingVoicesByLanguagePrefix = voices.filter((voice) => mappedVoice.lang && (voice.lang.substr(0, 2) === mappedVoice.lang.substr(0, 2)));

                        const resolvedVoices = []
                            .concat(actualMatchingVoicesByName)
                            .concat(actualMatchingVoicesByLanguage)
                            .concat(actualMatchingVoicesByLanguagePrefix);

                        if (resolvedVoices.length === 0) {
                            return null;
                        }

                        // NOTE: while there might be more than one voice for the particular voice name/language/language prefix, just consistently pick the first one.
                        // if (actualMatchingVoices.length !== 1) {
                        //     throw new Error(`Found other matching voices: ${JSON.stringify(mappedVoice)} ${actualMatchingVoices.length}`);
                        // }

                        const resolvedVoice = resolvedVoices[0];

                        return resolvedVoice;
                    });
            },
        );
    };

    const resolveVoiceAsMappedVoice = (mappedVoice) => {
        return promiseTry(
            () => {
                return resolveVoice(mappedVoice)
                    .then((resolvedVoice) => {
                        if (!resolvedVoice) {
                            return null;
                        }

                        const resolvedVoiceAsMappedVoice = getMappedVoice(resolvedVoice);

                        return resolvedVoiceAsMappedVoice;
                    });
            },
        );
    };

    const rateRange = {
        min: 0.1,
        default: 1,
        max: 10,
        step: 0.1,
    };

    const pitchRange = {
        min: 0,
        default: 1,
        max: 2,
        step: 0.1,
    };

    // TODO: check if there are any voices installed, alert user if not.
    // checkVoices() {
    //     return this.getSynthesizer()
    //         .then((synthesizer) => {
    //             logDebug("Start", "Voices check");
    //
    //             return getMappedVoices()
    //                 .then((voices) => {
    //                     logDebug("Variable", "voices[]", voices.length, voices);
    //
    //                     logDebug("Done", "Voices check");
    //
    //                     return synthesizer;
    //                 });
    //         });
    // }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class TextHelper {}

    TextHelper.splitTextToParagraphs = (text) => {
        // NOTE: in effect discarding empty paragraphs.
        return text.split(/[\n\r\u2028\u2029]+/);
    };

    TextHelper.splitTextToSentencesOfMaxLength = (text, maxPartLength) => {
        // NOTE: in effect merging multiple whitespaces in row to a single separator/space.
        const spacedTextParts = text.split(/\s+/);

        const naturalPauseRx = /(^--?$|[.,!?:;]$)/;

        const textParts = spacedTextParts.reduce((newParts, spacedTextPart) => {
            const appendToText = (ttt) => {
                if (last(newParts) === "") {
                    newParts[newParts.length - 1] = ttt;
                } else {
                    newParts[newParts.length - 1] += " " + ttt;
                }
            };

            const appendPart = (ttt) => {
                newParts[newParts.length] = ttt;
            };

            if (naturalPauseRx.test(spacedTextPart)) {
                appendToText(spacedTextPart);

                appendPart("");
            } else if ((last(newParts).length + 1 + spacedTextPart.length) < maxPartLength) {
                appendToText(spacedTextPart);
            } else {
                appendPart(spacedTextPart);
            }

            return newParts;
        }, [""]);

        // NOTE: cleaning empty strings "just in case".
        const cleanTextParts = textParts.filter((textPart) => textPart.trim().length > 0);

        return cleanTextParts;
    };

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class TalkieSpeaker {
        // https://dvcs.w3.org/hg/speech-api/raw-file/tip/speechapi.html#tts-section
        // https://dvcs.w3.org/hg/speech-api/raw-file/tip/speechapi.html#examples-synthesis
        // https://developer.mozilla.org/en-US/docs/Web/API/Web_Speech_API/Using_the_Web_Speech_API#Speech_synthesis
        constructor(broadcaster, shouldContinueSpeakingProvider, contentLogger, settingsManager) {
            this.broadcaster = broadcaster;
            this.shouldContinueSpeakingProvider = shouldContinueSpeakingProvider;
            this.contentLogger = contentLogger;
            this.settingsManager = settingsManager;

            this.resetSynthesizer();

            this.MAX_UTTERANCE_TEXT_LENGTH = 100;
        }

        getSynthesizerFromBrowser() {
            return promiseTry(
                () => {
                    logDebug("Start", "getSynthesizerFromBrowser", "Pre-requisites check");

                    if (!("speechSynthesis" in window) || typeof window.speechSynthesis.getVoices !== "function" || typeof window.speechSynthesis.speak !== "function") {
                        throw new Error("The browser does not support speechSynthesis.");
                    }

                    if (!("SpeechSynthesisUtterance" in window)) {
                        throw new Error("The browser does not support SpeechSynthesisUtterance.");
                    }

                    logDebug("Done", "getSynthesizerFromBrowser", "Pre-requisites check");
                })
                .then(() => {
                    logDebug("Start", "getSynthesizerFromBrowser");

                    // NOTE: the speech synthesizer can only be used in Chrome after the voices have been loaded.
                    const synthesizer = window.speechSynthesis;

                    // https://github.com/mdn/web-speech-api/blob/gh-pages/speak-easy-synthesis/script.js#L33-L36
                    // NOTE: The synthesizer will work right away in Firefox.
                    const voices = synthesizer.getVoices();

                    if (Array.isArray(voices) && voices.length > 0) {
                        logDebug("Done", "getSynthesizerFromBrowser (direct)");

                        return synthesizer;
                    }

                    const asyncSynthesizerInitialization = new Promise(
                        (resolve, reject) => {
                            try {
                                logDebug("Start", "getSynthesizerFromBrowser (event-based)");

                                const handleVoicesChanged = () => {
                                    synthesizer.removeEventListener("error", handleError);
                                    synthesizer.removeEventListener("voiceschanged", handleVoicesChanged);

                                    logDebug("Variable", "synthesizer", synthesizer);

                                    logDebug("Done", "getSynthesizerFromBrowser (event-based)");

                                    return resolve(synthesizer);
                                };

                                const handleError = (event) => {
                                    synthesizer.removeEventListener("error", handleError);
                                    synthesizer.removeEventListener("voiceschanged", handleVoicesChanged);

                                    logError("getSynthesizerFromBrowser", event);

                                    return reject(event.error);
                                };

                                // NOTE: Chrome needs to wait for the onvoiceschanged event before using the synthesizer.
                                synthesizer.addEventListener("voiceschanged", handleVoicesChanged);
                                synthesizer.addEventListener("error", handleError);
                            } catch (error) {
                                return reject(error);
                            }
                        },
                    );

                    return promiseTimeout(asyncSynthesizerInitialization, 1000)
                        .catch((error) => {
                            // TODO: remove the specific listeners previously registered.
                            if (error && error.name === "PromiseTimeout") {
                                logDebug("Done", "getSynthesizerFromBrowser (timeout)", "asyncSynthesizerInitialization", error);

                                // NOTE: assume the synthesizer has somehow been initialized without triggering the onvoiceschanged event.
                                return synthesizer;
                            }

                            logError("getSynthesizerFromBrowser", "asyncSynthesizerInitialization", error);

                            throw error;
                        });
                })
                .then((synthesizer) => {
                    // NOTE: only for logging purposes.
                    const voices = synthesizer.getVoices();

                    logDebug("Variable", "getSynthesizerFromBrowser", "voices[]", voices.length, voices);

                    return synthesizer;
                });
        }

        resetSynthesizer() {
            return promiseTry(
                () => {
                    delete this.cachedSynthesizer;
                    this.cachedSynthesizer = null;

                    return undefined;
                },
            );
        }

        getSynthesizer() {
            return promiseTry(
                () => {
                    if (this.cachedSynthesizer !== null) {
                        return this.cachedSynthesizer;
                    }

                    return this.getSynthesizerFromBrowser()
                        .then((synthesizer) => {
                            this.cachedSynthesizer = synthesizer;

                            return this.cachedSynthesizer;
                        });
                },
            );
        }

        getAllVoices() {
            return promiseTry(
                () => this.getSynthesizer()
                    .then((synthesizer) => {
                        return synthesizer.getVoices();
                    }),
            );
        }

        stopSpeaking() {
            return promiseTry(
                () => {
                    logDebug("Start", "stopSpeaking");

                    return this.getSynthesizer()
                        .then((synthesizer) => {
                            const eventData = {};

                            return Promise.resolve()
                                .then(() => this.broadcaster.broadcastEvent(knownEvents.stopSpeaking, eventData))
                                .then(() => {
                                    synthesizer.cancel();

                                    // NOTE: reset the system to resume playback, just to be nice to the world.
                                    synthesizer.resume();

                                    logDebug("Done", "stopSpeaking");

                                    return undefined;
                                });
                        });
                },
            );
        }

        speakPartOfText(utterance) {
            return this.getSynthesizer()
                .then((synthesizer) => new Promise(
                    (resolve, reject) => {
                        try {
                            logDebug("Start", "speakPartOfText", `Speak text part (length ${utterance.text.length}): "${utterance.text}"`);

                            const handleEnd = (event) => {
                                utterance.removeEventListener("end", handleEnd);
                                utterance.removeEventListener("error", handleEnd);

                                logDebug("End", "speakPartOfText", `Speak text part (length ${utterance.text.length}) spoken in ${event.elapsedTime} milliseconds.`);

                                return resolve();
                            };

                            const handleError = (event) => {
                                utterance.removeEventListener("end", handleEnd);
                                utterance.removeEventListener("error", handleEnd);

                                logError("speakPartOfText", `Speak text part (length ${utterance.text.length})`, event);

                                return reject(event.error);
                            };

                            utterance.addEventListener("end", handleEnd);
                            utterance.addEventListener("error", handleError);

                            // The actual act of speaking the text.
                            synthesizer.speak(utterance);

                            // NOTE: pause/resume suggested (for longer texts) in Chrome bug reports, trying it for shorter texts as well.
                            // https://bugs.chromium.org/p/chromium/issues/detail?id=335907
                            // https://bugs.chromium.org/p/chromium/issues/detail?id=369472
                            synthesizer.pause();
                            synthesizer.resume();

                            logDebug("Variable", "synthesizer", synthesizer);

                            logDebug("Done", "speakPartOfText", `Speak text part (length ${utterance.text.length})`);
                        } catch (error) {
                            return reject(error);
                        }
                    },
                ),
                );
        }

        getActualVoice(mappedVoice) {
            return promiseTry(
                () => {
                    logDebug("Start", "getActualVoice", mappedVoice);

                    return resolveVoice(mappedVoice)
                        .then((actualVoice) => {
                            logDebug("Done", "getActualVoice", mappedVoice, actualVoice);

                            return actualVoice;
                        })
                        .catch((error) => {
                            logError("getActualVoice", mappedVoice, error);

                            throw error;
                        });
                },
            );
        }

        splitAndSpeak(text, voice) {
            return promiseTry(
                () => {
                    logDebug("Start", "splitAndSpeak", `Speak text (length ${text.length}): "${text}"`);

                    const speakingEventData = {
                        text: text,
                        voice: voice.name,
                        language: voice.lang,
                    };

                    return Promise.resolve()
                        .then(() => this.broadcaster.broadcastEvent(knownEvents.beforeSpeaking, speakingEventData))
                        .then(() => {
                            // HACK: keep a reference to the utterance attached to the window. Not sure why this helps, but might prevent garbage collection or something.
                            delete window.talkieUtterance;

                            return undefined;
                        })
                        .then(() => Promise.all([
                            this.getActualVoice(voice),
                            this.settingsManager.getSpeakLongTexts(),
                        ]))
                        .then(([actualVoice, speakLongTexts]) => {
                            const paragraphs = TextHelper.splitTextToParagraphs(text);

                            let textParts = null;

                            if (speakLongTexts === true) {
                                textParts = paragraphs;
                            } else {
                                const cleanTextParts = paragraphs.map((paragraph) => TextHelper.splitTextToSentencesOfMaxLength(paragraph, this.MAX_UTTERANCE_TEXT_LENGTH));
                                textParts = flatten(cleanTextParts);
                            }

                            logDebug("Variable", "textParts.length", textParts.length);

                            const shouldContinueSpeaking = this.shouldContinueSpeakingProvider.getShouldContinueSpeakingProvider();

                            const textPartsPromises = textParts
                                .map((textPart) => () => {
                                    const speakingPartEventData = {
                                        textPart: textPart,
                                        voice: voice.name,
                                        language: voice.lang,
                                    };

                                    return shouldContinueSpeaking()
                                        .then((continueSpeaking) => {
                                            if (continueSpeaking) {
                                                return Promise.resolve()
                                                    .then(() => this.broadcaster.broadcastEvent(knownEvents.beforeSpeakingPart, speakingPartEventData))
                                                    .then(() => {
                                                        const utterance = new window.SpeechSynthesisUtterance(textPart);

                                                        utterance.voice = actualVoice;
                                                        utterance.rate = voice.rate || rateRange.default;
                                                        utterance.pitch = voice.pitch || pitchRange.default;

                                                        logDebug("Variable", "utterance", utterance);

                                                        // HACK: keep a reference to the utterance attached to the window. Not sure why this helps, but might prevent garbage collection or something.
                                                        window.talkieUtterance = utterance;

                                                        return utterance;
                                                    })
                                                    .then((utterance) => this.speakPartOfText(utterance))
                                                    .then(() => this.broadcaster.broadcastEvent(knownEvents.afterSpeakingPart, speakingPartEventData));
                                            }

                                            return undefined;
                                        });
                                });

                            return promiseSeries(textPartsPromises);
                        })
                        .then(() => {
                            // HACK: keep a reference to the utterance attached to the window. Not sure why this helps, but might prevent garbage collection or something.
                            delete window.talkieUtterance;

                            return undefined;
                        })
                        .then(() => this.broadcaster.broadcastEvent(knownEvents.afterSpeaking, speakingEventData));
                },
            );
        }

        speakTextInVoice(text, voice) {
            return promiseTry(
                () => Promise.resolve()
                    .then(() => {
                        this.contentLogger.logToPage(`Speaking text (length ${text.length}, ${voice.name}, ${voice.lang}): ${text}`)
                            .catch((error) => {
                                // NOTE: swallowing any logToPage() errors.
                                // NOTE: reduced logging for known tab/page access problems.
                                if (error && typeof error.message === "string" && error.message.startsWith("Cannot access")) {
                                    logDebug("getSelectionsWithValidTextAndDetectedLanguageAndEffectiveLanguage", "Error", error);
                                } else {
                                    logInfo("getSelectionsWithValidTextAndDetectedLanguageAndEffectiveLanguage", "Error", error);
                                }

                                return undefined;
                            });

                        return this.splitAndSpeak(text, voice);
                    })
                    .then(() => {
                        logDebug("Done", "speakTextInVoice", `Speak text (length ${text.length}, ${voice.name}, ${voice.lang})`);

                        return undefined;
                    }),
            );
        }

        speakTextInLanguage(text, language) {
            return promiseTry(
                () => {
                    const voice = {
                        name: null,
                        lang: language,
                    };

                    return Promise.resolve()
                        .then(() => {
                            return this.speakTextInVoice(text, voice);
                        })
                        .then(() => logDebug("Done", "speakTextInLanguage", `Speak text (length ${text.length}, ${language})`));
                },
            );
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class VoiceLanguageManager {
        constructor(storageManager, metadataManager) {
            this.storageManager = storageManager;
            this.metadataManager = metadataManager;

            this.languageLanguageVoiceOverrideNamesStorageKey = "language-voice-overrides";
        }

        getLanguageVoiceDefault(languageName) {
            return promiseTry(
                () => {
                    const mappedVoice = {
                        name: null,
                        lang: languageName,
                    };

                    return resolveVoiceAsMappedVoice(mappedVoice);
                },
            );
        }

        hasLanguageVoiceDefault(languageName) {
            return promiseTry(
                () => this.getLanguageVoiceDefault(languageName)
                    .then((languageVoiceDefault) => {
                        if (languageVoiceDefault) {
                            return true;
                        }

                        return false;
                    }),
            );
        }

        _getLanguageLanguageVoiceOverrideNames() {
            return promiseTry(
                () => this.metadataManager.isPremiumEdition()
                    .then((isPremiumEdition) => {
                        if (isPremiumEdition) {
                            return this.storageManager.getStoredValue(this.languageLanguageVoiceOverrideNamesStorageKey)
                                .then((languageLanguageVoiceOverrideNames) => {
                                    if (languageLanguageVoiceOverrideNames !== null && typeof languageLanguageVoiceOverrideNames === "object") {
                                        return languageLanguageVoiceOverrideNames;
                                    }

                                    return {};
                                });
                        }

                        return {};
                    }),
            );
        }

        _setLanguageLanguageVoiceOverrideNames(languageLanguageVoiceOverrideNames) {
            return promiseTry(
                () => this.metadataManager.isPremiumEdition()
                    .then((isPremiumEdition) => {
                        if (isPremiumEdition) {
                            return this.storageManager.setStoredValue(this.languageLanguageVoiceOverrideNamesStorageKey, languageLanguageVoiceOverrideNames);
                        }

                        return undefined;
                    }),
            );
        }

        getLanguageVoiceOverrideName(languageName) {
            return promiseTry(
                () => this._getLanguageLanguageVoiceOverrideNames()
                    .then((languageLanguageVoiceOverrideNames) => {
                        return languageLanguageVoiceOverrideNames[languageName] || null;
                    }),
            );
        }

        setLanguageVoiceOverrideName(languageName, voiceName) {
            return promiseTry(
                () => this._getLanguageLanguageVoiceOverrideNames()
                    .then((languageLanguageVoiceOverrideNames) => {
                        languageLanguageVoiceOverrideNames[languageName] = voiceName;

                        return this._setLanguageLanguageVoiceOverrideNames(languageLanguageVoiceOverrideNames);
                    }),
            );
        }

        removeLanguageVoiceOverrideName(languageName) {
            return promiseTry(
                () => this._getLanguageLanguageVoiceOverrideNames()
                    .then((languageLanguageVoiceOverrideNames) => {
                        delete languageLanguageVoiceOverrideNames[languageName];

                        return this._setLanguageLanguageVoiceOverrideNames(languageLanguageVoiceOverrideNames);
                    }),
            );
        }

        hasLanguageVoiceOverrideName(languageName) {
            return promiseTry(
                () => this.getLanguageVoiceOverrideName(languageName)
                    .then((languageVoiceOverride) => {
                        if (languageVoiceOverride) {
                            return true;
                        }

                        return false;
                    }),
            );
        }

        isLanguageVoiceOverrideName(languageName, voiceName) {
            return promiseTry(
                () => this.getLanguageVoiceOverrideName(languageName)
                    .then((languageVoiceOverride) => {
                        if (languageVoiceOverride) {
                            return languageVoiceOverride === voiceName;
                        }

                        return false;
                    }),
            );
        }

        toggleLanguageVoiceOverrideName(languageName, voiceName) {
            return promiseTry(
                () => this.isLanguageVoiceOverrideName(languageName, voiceName)
                    .then((isLanguageVoiceOverrideName) => {
                        if (isLanguageVoiceOverrideName) {
                            return this.removeLanguageVoiceOverrideName(languageName);
                        }

                        return this.setLanguageVoiceOverrideName(languageName, voiceName);
                    }),
            );
        }

        getEffectiveVoiceForLanguage(languageName) {
            return promiseTry(
                () => this.hasLanguageVoiceOverrideName(languageName)
                    .then((hasLanguageVoiceOverrideName) => {
                        if (hasLanguageVoiceOverrideName) {
                            return this.getLanguageVoiceOverrideName(languageName)
                                .then((languageOverrideName) => {
                                    const voice = {
                                        name: languageOverrideName,
                                        lang: null,
                                    };

                                    return voice;
                                });
                        }

                        return this.getLanguageVoiceDefault(languageName);
                    }),
            );
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class VoiceRateManager {
        constructor(storageManager, metadataManager) {
            this.storageManager = storageManager;
            this.metadataManager = metadataManager;

            this.voiceRateRateOverridesStorageKey = "voice-rate-overrides";
        }

        getVoiceRateDefault(/* eslint-disable no-unused-vars */voiceName/* eslint-enable no-unused-vars */) {
            return promiseTry(
                // TODO: initialize a "real" synthesizer voice, then read out the rate value.
                () => rateRange.default,
            );
        }

        hasVoiceRateDefault(voiceName) {
            return promiseTry(
                () => this.getVoiceRateDefault(voiceName)
                    .then((voiceRateDefault) => {
                        if (voiceRateDefault) {
                            return true;
                        }

                        return false;
                    }),
            );
        }

        _getVoiceRateOverrides() {
            return promiseTry(
                () => this.metadataManager.isPremiumEdition()
                    .then((isPremiumEdition) => {
                        if (isPremiumEdition) {
                            return this.storageManager.getStoredValue(this.voiceRateRateOverridesStorageKey)
                                .then((voiceRateRateOverrides) => {
                                    if (voiceRateRateOverrides !== null && typeof voiceRateRateOverrides === "object") {
                                        return voiceRateRateOverrides;
                                    }

                                    return {};
                                });
                        }

                        return {};
                    }),
            );
        }

        _setVoiceRateOverrides(voiceRateRateOverrides) {
            return promiseTry(
                () => this.metadataManager.isPremiumEdition()
                    .then((isPremiumEdition) => {
                        if (isPremiumEdition) {
                            return this.storageManager.setStoredValue(this.voiceRateRateOverridesStorageKey, voiceRateRateOverrides);
                        }

                        return undefined;
                    }),
            );
        }

        getVoiceRateOverride(voiceName) {
            return promiseTry(
                () => this._getVoiceRateOverrides()
                    .then((voiceRateRateOverrides) => {
                        return voiceRateRateOverrides[voiceName] || null;
                    }),
            );
        }

        setVoiceRateOverride(voiceName, rate) {
            return promiseTry(
                () => this._getVoiceRateOverrides()
                    .then((voiceRateRateOverrides) => {
                        voiceRateRateOverrides[voiceName] = rate;

                        return this._setVoiceRateOverrides(voiceRateRateOverrides);
                    }),
            );
        }

        removeVoiceRateOverride(voiceName) {
            return promiseTry(
                () => this._getVoiceRateOverrides()
                    .then((voiceRateRateOverrides) => {
                        delete voiceRateRateOverrides[voiceName];

                        return this._setVoiceRateOverrides(voiceRateRateOverrides);
                    }),
            );
        }

        hasVoiceRateOverride(voiceName) {
            return promiseTry(
                () => this.getVoiceRateOverride(voiceName)
                    .then((voiceRateOverride) => {
                        if (voiceRateOverride) {
                            return true;
                        }

                        return false;
                    }),
            );
        }

        isVoiceRateOverride(voiceName, rate) {
            return promiseTry(
                () => this.getVoiceRateOverride(voiceName)
                    .then((voiceRateOverride) => {
                        if (voiceRateOverride) {
                            return voiceRateOverride === rate;
                        }

                        return false;
                    }),
            );
        }

        getEffectiveRateForVoice(voiceName) {
            return promiseTry(
                () => this.hasVoiceRateOverride(voiceName)
                    .then((hasVoiceRateOverride) => {
                        if (hasVoiceRateOverride) {
                            return this.getVoiceRateOverride(voiceName);
                        }

                        return this.getVoiceRateDefault(voiceName);
                    }),
            );
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class VoicePitchManager {
        constructor(storageManager, metadataManager) {
            this.storageManager = storageManager;
            this.metadataManager = metadataManager;

            this.voicePitchPitchOverridesStorageKey = "voice-pitch-overrides";
        }

        getVoicePitchDefault(/* eslint-disable no-unused-vars */voiceName/* eslint-enable no-unused-vars */) {
            return promiseTry(
                // TODO: initialize a "real" synthesizer voice, then read out the pitch value.
                () => pitchRange.default,
            );
        }

        hasVoicePitchDefault(voiceName) {
            return promiseTry(
                () => this.getVoicePitchDefault(voiceName)
                    .then((voicePitchDefault) => {
                        if (voicePitchDefault) {
                            return true;
                        }

                        return false;
                    }),
            );
        }

        _getVoicePitchOverrides() {
            return promiseTry(
                () => this.metadataManager.isPremiumEdition()
                    .then((isPremiumEdition) => {
                        if (isPremiumEdition) {
                            return this.storageManager.getStoredValue(this.voicePitchPitchOverridesStorageKey)
                                .then((voicePitchPitchOverrides) => {
                                    if (voicePitchPitchOverrides !== null && typeof voicePitchPitchOverrides === "object") {
                                        return voicePitchPitchOverrides;
                                    }

                                    return {};
                                });
                        }

                        return {};
                    }),
            );
        }

        _setVoicePitchOverrides(voicePitchPitchOverrides) {
            return promiseTry(
                () => this.metadataManager.isPremiumEdition()
                    .then((isPremiumEdition) => {
                        if (isPremiumEdition) {
                            return this.storageManager.setStoredValue(this.voicePitchPitchOverridesStorageKey, voicePitchPitchOverrides);
                        }

                        return undefined;
                    }),
            );
        }

        getVoicePitchOverride(voiceName) {
            return promiseTry(
                () => this._getVoicePitchOverrides()
                    .then((voicePitchPitchOverrides) => {
                        return voicePitchPitchOverrides[voiceName] || null;
                    }),
            );
        }

        setVoicePitchOverride(voiceName, pitch) {
            return promiseTry(
                () => this._getVoicePitchOverrides()
                    .then((voicePitchPitchOverrides) => {
                        voicePitchPitchOverrides[voiceName] = pitch;

                        return this._setVoicePitchOverrides(voicePitchPitchOverrides);
                    }),
            );
        }

        removeVoicePitchOverride(voiceName) {
            return promiseTry(
                () => this._getVoicePitchOverrides()
                    .then((voicePitchPitchOverrides) => {
                        delete voicePitchPitchOverrides[voiceName];

                        return this._setVoicePitchOverrides(voicePitchPitchOverrides);
                    }),
            );
        }

        hasVoicePitchOverride(voiceName) {
            return promiseTry(
                () => this.getVoicePitchOverride(voiceName)
                    .then((voicePitchOverride) => {
                        if (voicePitchOverride) {
                            return true;
                        }

                        return false;
                    }),
            );
        }

        isVoicePitchOverride(voiceName, pitch) {
            return promiseTry(
                () => this.getVoicePitchOverride(voiceName)
                    .then((voicePitchOverride) => {
                        if (voicePitchOverride) {
                            return voicePitchOverride === pitch;
                        }

                        return false;
                    }),
            );
        }

        getEffectivePitchForVoice(voiceName) {
            return promiseTry(
                () => this.hasVoicePitchOverride(voiceName)
                    .then((hasVoicePitchOverride) => {
                        if (hasVoicePitchOverride) {
                            return this.getVoicePitchOverride(voiceName);
                        }

                        return this.getVoicePitchDefault(voiceName);
                    }),
            );
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class VoiceManager {
        constructor(voiceLanguageManager, voiceRateManager, voicePitchManager) {
            this.voiceLanguageManager = voiceLanguageManager;
            this.voiceRateManager = voiceRateManager;
            this.voicePitchManager = voicePitchManager;
        }

        getEffectiveVoiceForLanguage(...args) {
            return this.voiceLanguageManager.getEffectiveVoiceForLanguage(...args);
        }

        isLanguageVoiceOverrideName(...args) {
            return this.voiceLanguageManager.isLanguageVoiceOverrideName(...args);
        }

        toggleLanguageVoiceOverrideName(...args) {
            return this.voiceLanguageManager.toggleLanguageVoiceOverrideName(...args);
        }

        getVoiceRateDefault(...args) {
            return this.voiceRateManager.getVoiceRateDefault(...args);
        }

        getEffectiveRateForVoice(...args) {
            return this.voiceRateManager.getEffectiveRateForVoice(...args);
        }

        setVoiceRateOverride(...args) {
            return this.voiceRateManager.setVoiceRateOverride(...args);
        }

        getVoicePitchDefault(...args) {
            return this.voicePitchManager.getVoicePitchDefault(...args);
        }

        getEffectivePitchForVoice(...args) {
            return this.voicePitchManager.getEffectivePitchForVoice(...args);
        }

        setVoicePitchOverride(...args) {
            return this.voicePitchManager.setVoicePitchOverride(...args);
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class SpeakingStatus {
        constructor() {
            this.currentSpeakingTab = null;
        }

        getSpeakingTabId() {
            return promiseTry(
                () => this.currentSpeakingTab,
            );
        }

        setSpeakingTabId(tabId) {
            return this.isSpeakingTabId(tabId)
                .then((isTabSpeaking) => {
                    if (isTabSpeaking) {
                        throw new Error(`Tried to set tab ${tabId} as speaking, but another tab was already speaking.`);
                    }

                    this.currentSpeakingTab = tabId;

                    return undefined;
                });
        }

        setDoneSpeaking() {
            return promiseTry(
                () => {
                    this.currentSpeakingTab = null;
                },
            );
        }

        setTabIsDoneSpeaking(tabId) {
            return this.isSpeakingTabId(tabId)
                .then((isTabSpeaking) => {
                    // TODO: throw if it's not the same tabId as the currently speaking tab?
                    if (isTabSpeaking) {
                        return this.setDoneSpeaking();
                    }

                    return undefined;
                });
        }

        isSpeakingTabId(tabId) {
            return promiseTry(
                () => this.currentSpeakingTab !== null && tabId === this.currentSpeakingTab,
            );
        }

        isSpeaking() {
            return this.getSpeakingTabId()
                // TODO: check synthesizer.speaking === true?
                .then((speakingTabId) => speakingTabId !== null);
        }

        setActiveTabIsDoneSpeaking() {
            return getCurrentActiveTabId()
                .then((activeTabId) => this.setTabIsDoneSpeaking(activeTabId));
        }

        setActiveTabAsSpeaking() {
            return getCurrentActiveTab()
                .then((activeTab) => {
                    // NOTE: some tabs can't be retreived.
                    if (!activeTab) {
                        return undefined;
                    }

                    const activeTabId = activeTab.id;

                    return this.setSpeakingTabId(activeTabId);
                });
        }

        isActiveTabSpeaking() {
            return getCurrentActiveTabId()
                .then((activeTabId) => this.isSpeakingTabId(activeTabId));
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class IconManager {
        constructor(metadataManager) {
            this.metadataManager = metadataManager;
        }

        getIconModePaths(editionType, name) {
            return {
                // NOTE: icons in use before Chrome 53 were 19x19 and 38x38.
                // NOTE: icons in use from Chrome 53 (switching to Material design) are 16x16 and 32x32.
                // NOTE: keeping larger icons to accomodate future changes.
                "16": `resources/icon/${editionType}/icon-${name}/icon-16x16.png`,
                "32": `resources/icon/${editionType}/icon-${name}/icon-32x32.png`,
                "48": `resources/icon/${editionType}/icon-${name}/icon-48x48.png`,
                "64": `resources/icon/${editionType}/icon-${name}/icon-64x64.png`,

                // NOTE: passing the larger icons slowed down the UI by several hundred milliseconds per icon switch.
                // "128": `resources/icon/${editionType}/icon-${name}/icon-128x128.png`,
                // "256": `resources/icon/${editionType}/icon-${name}/icon-256x256.png`,
                // "512": `resources/icon/${editionType}/icon-${name}/icon-512x512.png`,
                // "1024": `resources/icon/${editionType}/icon-${name}/icon-1024x1024.png`,
            };
        };

        setIconMode(name) {
            return promiseTry(
                () => {
                    logDebug("Start", "Changing icon to", name);

                    return this.metadataManager.getEditionType()
                        .then((editionType) => {
                            const paths = this.getIconModePaths(editionType, name);
                            const details = {
                                path: paths,
                            };

                            return browser.browserAction.setIcon(details)
                                .then((result) => {
                                    logDebug("Done", "Changing icon to", name);

                                    return result;
                                });
                        });
                },
            );
        }

        setIconModePlaying() {
            return this.setIconMode("stop");
        }

        setIconModeStopped() {
            return this.setIconMode("play");
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class ButtonPopupManager {
        constructor(translator, metadataManager) {
            this.translator = translator;
            this.metadataManager = metadataManager;
        }

        _getExtensionShortName() {
            return promiseTry(
                () => this.metadataManager.isPremiumEdition()
                    .then((isPremiumEdition) => {
                        // TODO: move resolving the name to a layer on top of the translator?
                        const extensionShortName = isPremiumEdition
                            ? this.translator.translate("extensionShortName_Premium")
                            : this.translator.translate("extensionShortName_Free");

                        return extensionShortName;
                    }),
            );
        }

        _getButtonDefaultTitle() {
            return promiseTry(
                () => this._getExtensionShortName()
                    .then((extensionShortName) => this.translator.translate("buttonDefaultTitle", [extensionShortName])),
            );
        }

        _getButtonStopTitle() {
            return promiseTry(
                () => this.translator.translate("buttonStopTitle"),
            );
        }

        _disablePopupSync(buttonStopTitle) {
            const disablePopupOptions = {
                popup: "",
            };

            browser.browserAction.setPopup(disablePopupOptions);

            const disableIconTitleOptions = {
                title: buttonStopTitle,
            };

            browser.browserAction.setTitle(disableIconTitleOptions);
        }

        _enablePopupSync(buttonDefaultTitle) {
            const enablePopupOptions = {
                popup: "src/popup/popup.html",
            };

            browser.browserAction.setPopup(enablePopupOptions);

            const enableIconTitleOptions = {
                title: buttonDefaultTitle,
            };

            browser.browserAction.setTitle(enableIconTitleOptions);
        }

        disablePopup() {
            return promiseTry(
                /* eslint-disable no-sync */
                () => this._getButtonStopTitle()
                    .then((buttonStopTitle) => this._disablePopupSync(buttonStopTitle)),
                /* eslint-enable no-sync */
            );
        }

        enablePopup() {
            return promiseTry(
                /* eslint-disable no-sync */
                () => this._getButtonDefaultTitle()
                    .then((buttonDefaultTitle) => this._enablePopupSync(buttonDefaultTitle)),
                /* eslint-enable no-sync */
            );
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class CommandHandler {
        constructor(commandMap) {
            this.commandMap = commandMap;
        }

        handle(command, ...args) {
            logDebug("Start", "commandHandler", command);

            const commandAction = this.commandMap[command];

            if (typeof commandAction !== "function") {
                throw new Error("Bad command action for command: " + command);
            }

            return commandAction(...args)
                .then((result) => {
                    logDebug("Done", "commandHandler", command, result);

                    return undefined;
                })
                .catch((error) => {
                    logError("commandHandler", command, error);

                    throw error;
                });
        }

        handleCommandEvent(command, ...args) {
            logDebug("Start", "handleCommandEvent", command);

            // NOTE: straight mapping from command to action.
            return this.handle(command, ...args)
                .then((result) => {
                    logDebug("Done", "handleCommandEvent", command, result);

                    return undefined;
                })
                .catch((error) => {
                    logError("handleCommandEvent", command, error);

                    throw error;
                });
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class OnlyLastCaller {
        constructor() {
            this.lastCallerId = 0;
        }

        incrementCallerId() {
            this.lastCallerId++;
        }

        getShouldContinueSpeakingProvider() {
            this.incrementCallerId();
            const callerOnTheAirId = this.lastCallerId;

            logDebug("Start", "getShouldContinueSpeakingProvider", callerOnTheAirId);

            return () => promiseTry(
                () => {
                    const isLastCallerOnTheAir = callerOnTheAirId === this.lastCallerId;

                    logDebug("Status", "getShouldContinueSpeakingProvider", callerOnTheAirId, isLastCallerOnTheAir);

                    return isLastCallerOnTheAir;
                },
            );
        };
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class Chain {
        constructor() {
            this.chainPromise = Promise.resolve();
            this.length = 0;
        }

        link(promise) {
            this.length++;
            const currentLength = this.length;

            logDebug("Start", "Chain", currentLength);

            this.chainPromise = this.chainPromise
                .then(promise)
                .then((result) => {
                    logDebug("Done", "Chain", currentLength);

                    return result;
                })
                .catch((error) => {
                    logError("Chain", currentLength, error);

                    throw error;
                });

            return this.chainPromise;
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class TalkieBackground {
        constructor(speechChain, broadcaster, talkieSpeaker, speakingStatus, voiceManager, languageHelper, configuration, execute, translator, internalUrlProvider) {
            this.speechChain = speechChain;
            this.broadcaster = broadcaster;
            this.talkieSpeaker = talkieSpeaker;
            this.speakingStatus = speakingStatus;
            this.voiceManager = voiceManager;
            this.languageHelper = languageHelper;
            this.configuration = configuration;
            this.execute = execute;
            this.translator = translator;
            this.internalUrlProvider = internalUrlProvider;

            this.notAbleToSpeakTextFromThisSpecialTab = {
                text: this.translator.translate("notAbleToSpeakTextFromThisSpecialTab"),
                effectiveLanguage: this.translator.translate("extensionLocale"),
            };

            // NOTE: duplicated elsewhere in the codebase.
            this.executeGetFramesSelectionTextAndLanguageCode = `
            (function() {
                try {
                    function talkieGetParentElementLanguages(element) {
                        return []
                            .concat((element || null) && element.getAttribute && element.getAttribute("lang"))
                            .concat((element || null) && element.parentElement && talkieGetParentElementLanguages(element.parentElement));
                    };

                    var talkieSelectionData = {
                        text: ((document || null) && (document.getSelection || null) && (document.getSelection() || null) && document.getSelection().toString()),
                        htmlTagLanguage: ((document || null) && (document.getElementsByTagName || null) && (document.getElementsByTagName("html") || null) && (document.getElementsByTagName("html").length > 0 || null) && (document.getElementsByTagName("html")[0].getAttribute("lang") || null)),
                        parentElementsLanguages: (talkieGetParentElementLanguages((document || null) && (document.getSelection || null) && (document.getSelection() || null) && (document.getSelection().rangeCount > 0 || null) && (document.getSelection().getRangeAt || null) && (document.getSelection().getRangeAt(0) || null) && (document.getSelection().getRangeAt(0).startContainer || null))),
                    };

                    return talkieSelectionData;
                } catch (error) {
                    return null;
                }
            }());`
                .replace(/\n/g, "")
                .replace(/\s{2,}/g, " ");
        }

        speakSelectionOnPage() {
            return promiseTry(
                () => Promise.all([
                    canTalkieRunInTab(),
                    isCurrentPageInternalToTalkie(this.internalUrlProvider),
                ])
                    .then(([canRun, isInternalPage]) => {
                        // NOTE: can't perform (most) actions if it's not a "normal" tab.
                        if (!canRun) {
                            logDebug("iconClickAction", "Did not detect a normal tab.");

                            if (isInternalPage) {
                                logDebug("iconClickAction", "Requesting text selection from internal page.");

                                const eventData = null;

                                return this.broadcaster.broadcastEvent(knownEvents.passSelectedTextToBackground, eventData)
                                    .then((selectedTextsFromFrontend) => {
                                        logDebug("iconClickAction", "Received text selections from internal pages.", selectedTextsFromFrontend);

                                        const filteredSelectedTextsFromFrontend = selectedTextsFromFrontend
                                            .filter((selectedTextWithFocusTimestamp) => selectedTextWithFocusTimestamp !== null);

                                        return filteredSelectedTextsFromFrontend;
                                    })
                                    .then((filteredSelectedTextsFromFrontend) => {
                                        const selectedTextFromFrontend = filteredSelectedTextsFromFrontend
                                            .reduce(
                                                (prev, selectedTextWithFocusTimestamp) => {
                                                    if (prev === null || prev.mostRecentUse < selectedTextWithFocusTimestamp.mostRecentUse) {
                                                        return selectedTextWithFocusTimestamp;
                                                    }

                                                    return prev;
                                                },
                                                null,
                                            );

                                        return selectedTextFromFrontend;
                                    })
                                    .then((selectedTextFromFrontend) => {
                                        if (selectedTextFromFrontend === null) {
                                            logDebug("iconClickAction", "Did not receive text selection from internal page, doing nothing.");

                                            return undefined;
                                        }

                                        const selections = [
                                            selectedTextFromFrontend.selectionTextAndLanguageCode,
                                        ];

                                        // NOTE: assumes that internal pages have at least proper <html lang=""> attributes.
                                        const detectedPageLanguage = null;

                                        return this.detectLanguagesAndSpeakAllSelections(selections, detectedPageLanguage);
                                    });
                            }

                            logDebug("iconClickAction", "Skipping speaking selection.");

                            const text = this.notAbleToSpeakTextFromThisSpecialTab.text;
                            const lang = this.notAbleToSpeakTextFromThisSpecialTab.effectiveLanguage;

                            return this.startSpeakingTextInLanguageWithOverridesAction(text, lang);
                        }

                        return this.speakUserSelection();
                    }),
            );
        }

        startStopSpeakSelectionOnPage() {
            return promiseTry(
                () => this.speakingStatus.isSpeaking()
                    .then((wasSpeaking) => this.talkieSpeaker.stopSpeaking()
                        .then(() => {
                            if (!wasSpeaking) {
                                return this.speakSelectionOnPage();
                            }

                            return undefined;
                        })),
            );
        }

        stopSpeakingAction() {
            return promiseTry(
                () => this.talkieSpeaker.stopSpeaking(),
            );
        }

        startSpeakingTextInVoiceAction(text, voice) {
            return promiseTry(
                () => this.talkieSpeaker.stopSpeaking()
                    .then(() => {
                        // NOTE: keeping the root chain separate from this chain.
                        this.speechChain.link(() => this.talkieSpeaker.speakTextInVoice(text, voice))
                            .catch((error) => {
                                logError("Caught error on the speechChain. Swallowing. Resetting synthesizer just in case.", error);

                                // TODO: handle internally in talkieSpeaker?
                                return this.talkieSpeaker.resetSynthesizer();
                            });

                        return undefined;
                    }),
            );
        }

        addRateAndPitchToSpecificVoice(voice) {
            return promiseTry(
                () => {
                    return Promise.all([
                        this.voiceManager.getEffectiveRateForVoice(voice.name),
                        this.voiceManager.getEffectivePitchForVoice(voice.name),
                    ])
                        .then(([effectiveRateForVoice, effectivePitchForVoice]) => {
                            const voiceWithPitchAndRate = shallowCopy(
                                voice,
                                {
                                    rate: effectiveRateForVoice,
                                    pitch: effectivePitchForVoice,
                                },
                            );

                            return voiceWithPitchAndRate;
                        });
                },
            );
        }

        startSpeakingTextInVoiceWithOverridesAction(text, voice) {
            return promiseTry(
                () => this.addRateAndPitchToSpecificVoice(voice)
                    .then((voiceWithPitchAndRate) => this.startSpeakingTextInVoiceAction(text, voiceWithPitchAndRate)),
            );
        }

        startSpeakingTextInLanguageAction(text, language) {
            return promiseTry(
                () => this.talkieSpeaker.stopSpeaking()
                    .then(() => {
                        // NOTE: keeping the root chain separate from this chain.
                        this.speechChain.link(() => this.talkieSpeaker.speakTextInLanguage(text, language))
                            .catch((error) => {
                                logError("Caught error on the speechChain. Swallowing. Resetting synthesizer just in case.", error);

                                // TODO: handle internally in talkieSpeaker?
                                return this.talkieSpeaker.resetSynthesizer();
                            });

                        return undefined;
                    }),
            );
        }

        startSpeakingTextInLanguageWithOverridesAction(text, language) {
            return promiseTry(
                () => {
                    return this.voiceManager.getEffectiveVoiceForLanguage(language)
                        .then((effectiveVoiceForLanguage) => this.startSpeakingTextInVoiceWithOverridesAction(text, effectiveVoiceForLanguage));
                },
            );
        }

        startSpeakingCustomTextDetectLanguage(text) {
            return promiseTry(
                () => this.talkieSpeaker.stopSpeaking()
                    .then(() => canTalkieRunInTab())
                    .then((canRun) => {
                        if (canRun) {
                            return this.languageHelper.detectPageLanguage();
                        }

                        logDebug("startSpeakingCustomTextDetectLanguage", "Did not detect a normal tab.", "Skipping page language detection.");

                        return null;
                    })
                    .then((detectedPageLanguage) => {
                        const selections = [
                            {
                                text: text,
                                htmlTagLanguage: null,
                                parentElementsLanguages: [],
                            },
                        ];

                        return this.detectLanguagesAndSpeakAllSelections(selections, detectedPageLanguage);
                    }),
            );
        }

        onTabRemovedHandler(tabId) {
            return this.speakingStatus.isSpeakingTabId(tabId)
                .then((isTabSpeaking) => {
                    if (isTabSpeaking) {
                        return this.talkieSpeaker.stopSpeaking();
                    }

                    return undefined;
                });
        }

        onTabUpdatedHandler(tabId, changeInfo) {
            return this.speakingStatus.isSpeakingTabId(tabId)
                .then((isTabSpeaking) => {
                    // NOTE: changeInfo only has properties which have changed.
                    // https://developer.browser.com/extensions/tabs#event-onUpdated
                    if (isTabSpeaking && changeInfo.url) {
                        return this.talkieSpeaker.stopSpeaking();
                    }

                    return undefined;
                });
        }

        onExtensionSuspendHandler() {
            return promiseTry(
                () => {
                    logDebug("Start", "onExtensionSuspendHandler");

                    return this.speakingStatus.isSpeaking()
                        .then((talkieIsSpeaking) => {
                            // NOTE: Clear all text if Talkie was speaking.
                            if (talkieIsSpeaking) {
                                return this.talkieSpeaker.stopSpeaking();
                            }

                            return undefined;
                        })
                        .then(() => {
                            logDebug("Done", "onExtensionSuspendHandler");

                            return undefined;
                        });
                },
            );
        }

        executeGetFramesSelectionTextAndLanguage() {
            return this.execute.scriptInAllFramesWithTimeout(this.executeGetFramesSelectionTextAndLanguageCode, 1000)
                .then((framesSelectionTextAndLanguage) => {
                    logDebug("Variable", "framesSelectionTextAndLanguage", framesSelectionTextAndLanguage);

                    if (!framesSelectionTextAndLanguage || !Array.isArray(framesSelectionTextAndLanguage)) {
                        throw new Error("framesSelectionTextAndLanguage");
                    }

                    return framesSelectionTextAndLanguage;
                });
        }

        detectLanguagesAndSpeakAllSelections(selections, detectedPageLanguage) {
            return promiseTry(() => {
                logDebug("Start", "Speaking all selections");

                logDebug("Variable", `selections (length ${(selections && selections.length) || 0})`, selections);

                return promiseTry(
                    () => getVoices(),
                )
                    .then((allVoices) => this.languageHelper.cleanupSelections(allVoices, detectedPageLanguage, selections))
                    .then((cleanedupSelections) => {
                        logDebug("Variable", `cleanedupSelections (length ${(cleanedupSelections && cleanedupSelections.length) || 0})`, cleanedupSelections);

                        const speakPromises = cleanedupSelections.map((selection) => {
                            logDebug("Text", `Speaking selection (length ${selection.text.length}, effectiveLanguage ${selection.effectiveLanguage})`, selection);

                            return this.startSpeakingTextInLanguageWithOverridesAction(selection.text, selection.effectiveLanguage);
                        });

                        logDebug("Done", "Speaking all selections");

                        return Promise.all(speakPromises);
                    });
            });
        }

        speakUserSelection() {
            return promiseTry(
                () => {
                    logDebug("Start", "Speaking selection");

                    return Promise.all(
                        [
                            this.executeGetFramesSelectionTextAndLanguage(),
                            this.languageHelper.detectPageLanguage(),
                        ],
                    )
                        .then(([framesSelectionTextAndLanguage, detectedPageLanguage]) => {
                            return this.detectLanguagesAndSpeakAllSelections(framesSelectionTextAndLanguage, detectedPageLanguage);
                        })
                        .then(() => logDebug("Done", "Speaking selection"));
                });
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class PermissionsManager {
        constructor() {
            // NOTE: this is obfuscation to avoid errors regarding the permissions API not yet being implemented in Firefox (WebExtensions).
            // NOTE: am doing feature detection, and this code should not even be called as the menus are not enabled for WebExtensions (Firefox).
            // TODO: remove once Firefox is the main/only browser, and/or update strict_min_version in manifest.json.
            const something = browser;
            /* eslint-disable dot-notation */
            this._pms = something["permissions"];
            /* eslint-enable dot-notation */
        }

        browserHasPermissionsFeature() {
            return promiseTry(
                () => !!this._pms,
            );
        }

        hasPermissions(permissionNames, origins) {
            return promiseTry(
                () => this._pms.contains({
                    permissions: permissionNames,
                    origins: origins,
                }),
            );
        }

        acquirePermissions(permissionNames, origins) {
            return promiseTry(
                () => this._pms.request({
                    permissions: permissionNames,
                    origins: origins,
                }),
            );
        }

        releasePermissions(permissionNames, origins) {
            return promiseTry(
                () => this._pms.remove({
                    permissions: permissionNames,
                    origins: origins,
                }),
            );
        }

        useOptionalPermissions(permissionNames, origins, fn) {
            return promiseTry(
                () => {
                    logDebug("Start", "useOptionalPermissions", permissionNames.length, permissionNames, origins.length, origins);

                    const hasPermissionsPromises = permissionNames.map((permissionName) => {
                        // TODO: be more fine-grained per origin as well?
                        return this.hasPermissions([permissionName], origins);
                    });

                    return Promise.all(hasPermissionsPromises)
                        .then((hasPermissionsStates) => {
                            const activePermissionNames = [];
                            const inactivePermissionNames = [];

                            hasPermissionsStates.forEach((hasPermissionsState, index) => {
                                const permissionName = permissionNames[index];

                                if (hasPermissionsState) {
                                    activePermissionNames.push(permissionName);
                                } else {
                                    inactivePermissionNames.push(permissionName);
                                }
                            });

                            logDebug("useOptionalPermissions", permissionNames.length, origins.length, "Already acquired", activePermissionNames);
                            logDebug("useOptionalPermissions", permissionNames.length, origins.length, "Not yet acquired", inactivePermissionNames);

                            return this.acquirePermissions(inactivePermissionNames, origins)
                                .then((granted) => {
                                    if (granted) {
                                        logDebug("useOptionalPermissions", permissionNames.length, origins.length, "All permissions acquired");
                                    } else {
                                        logDebug("useOptionalPermissions", permissionNames.length, origins.length, "Permissions not acquired");
                                    }

                                    return Promise.resolve()
                                        .then(() => fn(granted))
                                        .then((result) => {
                                            return this.releasePermissions(inactivePermissionNames)
                                                .then(() => {
                                                    logDebug("Done", "useOptionalPermissions", permissionNames.length, origins.length);

                                                    return result;
                                                });
                                        })
                                        .catch((error) => {
                                            logError("useOptionalPermissions", permissionNames.length, origins.length, error);

                                            return this.releasePermissions(inactivePermissionNames, origins)
                                                .then(() => {
                                                    throw error;
                                                });
                                        });
                                });
                        });
                },
            );
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class ClipboardManager {
        constructor(talkieBackground, permissionsManager) {
            this.talkieBackground = talkieBackground;
            this.permissionsManager = permissionsManager;

            this.copyPasteTargetElementId = "copy-paste-textarea";
        }

        _getExistingTextarea() {
            return promiseTry(
                () => {
                    const existingTextarea = document.getElementById(this.copyPasteTargetElementId);

                    return existingTextarea;
                },
            );
        }

        _isInitialized() {
            return this._getExistingTextarea()
                .then((existingTextarea) => existingTextarea !== null);
        }

        _ensureIsInitialized() {
            return this._isInitialized()
                .then((isInitialized) => {
                    if (isInitialized === true) {
                        return undefined;
                    }

                    throw new Error("this.copyPasteTargetElementId did not exist.");
                });
        }

        _ensureIsNotInitialized() {
            return this._isInitialized()
                .then((isInitialized) => {
                    if (isInitialized === false) {
                        return undefined;
                    }

                    throw new Error("this.copyPasteTargetElementId exists.");
                });
        }

        _injectBackgroundTextarea() {
            return this._ensureIsNotInitialized()
                .then(() => {
                    const textarea = document.createElement("textarea");
                    textarea.id = this.copyPasteTargetElementId;
                    document.body.appendChild(textarea);

                    return undefined;
                });
        }

        _initializeIfNecessary() {
            return this._isInitialized()
                .then((isInitialized) => {
                    if (isInitialized !== true) {
                        return this.initialize();
                    }

                    return undefined;
                });
        }

        _removeBackgroundTextarea() {
            return this._ensureIsInitialized()
                .then(() => this._getExistingTextarea())
                .then((existingTextarea) => {
                    existingTextarea.parentNode.removeChild(existingTextarea);

                    return undefined;
                });
        }

        initialize() {
            return promiseTry(
                () => {
                    logDebug("Start", "ClipboardManager.initialize");

                    return this._injectBackgroundTextarea()
                        .then(() => {
                            logDebug("Done", "ClipboardManager.initialize");

                            return undefined;
                        });
                },
            );
        }

        unintialize() {
            return promiseTry(
                () => {
                    logDebug("Start", "ClipboardManager.unintialize");

                    return this._removeBackgroundTextarea()
                        .then(() => {
                            logDebug("Done", "ClipboardManager.unintialize");

                            return undefined;
                        });
                },
            );
        }

        getClipboardText() {
            return promiseTry(
                () => {
                    logDebug("Start", "getClipboardText");

                    return this._initializeIfNecessary()
                        .then(() => this.permissionsManager.useOptionalPermissions(
                            [
                                "clipboardRead",
                            ],
                            [],
                            (granted) => {
                                if (granted) {
                                    return this._getExistingTextarea()
                                        .then((textarea) => {
                                            textarea.value = "";
                                            textarea.focus();

                                            const success = document.execCommand("Paste");

                                            if (!success) {
                                                return null;
                                            }

                                            return textarea.value;
                                        });
                                }

                                return null;
                            }),
                        )
                        .then((text) => {
                            logDebug("Done", "getClipboardText", text);

                            return text;
                        });
                },
            );
        }}

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class ReadClipboardManager {
        constructor(clipboardManager, talkieBackground, permissionsManager, metadataManager, translator) {
            this.clipboardManager = clipboardManager;
            this.talkieBackground = talkieBackground;
            this.permissionsManager = permissionsManager;
            this.metadataManager = metadataManager;
            this.translator = translator;

            this.copyPasteTargetElementId = "copy-paste-textarea";
        }

        startSpeaking() {
            return promiseTry(
                () => {
                    logDebug("Start", "startSpeaking");

                    return this.metadataManager.isPremiumEdition()
                        .then((isPremiumEdition) => {
                            if (!isPremiumEdition) {
                                const text = this.translator.translate("readClipboardIsAPremiumFeature");

                                return text;
                            }

                            return this.permissionsManager.browserHasPermissionsFeature()
                                .then((hasPermissionsFeature) => {
                                    if (!hasPermissionsFeature) {
                                        const text = this.translator.translate("readClipboardNeedsBrowserSupport");

                                        return text;
                                    }

                                    return this.clipboardManager.getClipboardText()
                                        .then((clipboardText) => {
                                            let text = clipboardText;

                                            if (typeof text !== "string") {
                                                text = this.translator.translate("readClipboardNeedsPermission");
                                            }

                                            if (text.length === 0 || text.trim().length === 0) {
                                                text = this.translator.translate("readClipboardNoSuitableText");
                                            }

                                            return text;
                                        });
                                });
                        })
                        .then((text) => this.talkieBackground.startSpeakingCustomTextDetectLanguage(text))
                        .then((result) => {
                            logDebug("Done", "startSpeaking");

                            return result;
                        });
                },
            );
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class ContextMenuManager {
        constructor(commandHandler, metadataManager, translator) {
            this.commandHandler = commandHandler;
            this.metadataManager = metadataManager;
            this.translator = translator;

            if (!isNaN(browser.contextMenus.ACTION_MENU_TOP_LEVEL_LIMIT) && browser.contextMenus.ACTION_MENU_TOP_LEVEL_LIMIT > 0) {
                this.actionMenuLimit = browser.contextMenus.ACTION_MENU_TOP_LEVEL_LIMIT;
            } else {
                this.actionMenuLimit = Number.MAX_SAFE_INTEGER;
            }

            this.contextMenuOptionsCollection = [
                {
                    free: true,
                    premium: true,
                    chrome: true,
                    webextension: true,
                    item: {
                        id: "talkie-context-menu-start-stop",
                        title: this.translator.translate("contextMenuStartStopText"),
                        contexts: [
                            "selection",
                        ],
                    },
                },
                {
                    free: true,
                    premium: true,
                    chrome: true,
                    webextension: true,
                    item: {
                        id: "start-stop",
                        title: this.translator.translate("commandStartStopDescription"),
                        contexts: [
                            "browser_action",
                        ],
                    },
                },
                {
                    free: true,
                    premium: true,
                    chrome: true,
                    // TODO: enable after Firefox 55 has landed?
                    webextension: false,
                    item: {
                        id: "read-clipboard",
                        title: this.translator.translate("commandReadClipboardDescription"),
                        contexts: [
                            "browser_action",
                            "page",
                        ],
                    },
                },
                {
                    free: true,
                    premium: true,
                    chrome: true,
                    webextension: true,
                    item: {
                        id: "buttonContextMenuSeparator01",
                        type: "separator",
                        contexts: [
                            "browser_action",
                        ],
                    },
                },
                {
                    free: true,
                    premium: true,
                    chrome: true,
                    webextension: true,
                    item: {
                        id: "open-website-main",
                        title: this.translator.translate("commandOpenWebsiteMainDescription"),
                        contexts: [
                            "browser_action",
                        ],
                    },
                },
                {
                    free: true,
                    premium: false,
                    chrome: true,
                    webextension: true,
                    item: {
                        id: "open-website-upgrade",
                        title: this.translator.translate("commandOpenWebsiteUpgradeDescription"),
                        contexts: [
                            "browser_action",
                        ],
                    },
                },
            ];
        }

        removeAll() {
            return promiseTry(
                () => {
                    logDebug("Start", "Removing all context menus");

                    return browser.contextMenus.removeAll()
                        .then((result) => {
                            logDebug("Done", "Removing all context menus");

                            return result;
                        });
                },
            );
        }

        createContextMenu(contextMenuOptions) {
            return new Promise(
                (resolve, reject) => {
                    try {
                        logDebug("Start", "Creating context menu", contextMenuOptions);

                        // NOTE: apparently Chrome modifies the context menu object after it has been passed in, by adding generatedId.
                        // NOTE: Need to pass a clean object to avoid object reuse reference problems.
                        const contextMenu = shallowCopy(contextMenuOptions);

                        // NOTE: Can't directly use a promise chain here, as the id is returned instead.
                        // https://github.com/mozilla/webextension-polyfill/pull/26
                        const contextMenuId = browser.contextMenus.create(
                            contextMenu,
                            () => {
                                if (browser.runtime.lastError) {
                                    return reject(browser.runtime.lastError);
                                }

                                logDebug("Done", "Creating context menu", contextMenu);

                                return resolve(contextMenuId);
                            },
                        );
                    } catch (error) {
                        return reject(error);
                    }
                },
            );
        }

        contextMenuClickAction(info) {
            return promiseTry(
                () => {
                    logDebug("Start", "contextMenuClickAction", info);

                    if (!info) {
                        throw new Error("Unknown context menu click action info object.");
                    }

                    return promiseTry(
                        () => {
                            const id = info.menuItemId;

                            const selectionContextMenuStartStop = this.contextMenuOptionsCollection
                                .reduce(
                                    (prev, obj) => {
                                        if (obj.item.id === "talkie-context-menu-start-stop") {
                                            return obj;
                                        }

                                        return prev;
                                    },
                                    null);

                            // TODO: use assertions?
                            if (!selectionContextMenuStartStop) {
                                throw new Error("Not found: selectionContextMenuStartStop");
                            }

                            if (id === selectionContextMenuStartStop.item.id) {
                                const selection = info.selectionText || null;

                                if (!selection || typeof selection !== "string" || selection.length === 0) {
                                    throw new Error("Unknown context menu click action selection was empty.");
                                }

                                return this.commandHandler.handle("start-text", selection);
                            }

                            // NOTE: context menu items default to being commands.
                            return this.commandHandler.handle(id);
                        },
                    )
                        .then(() => {
                            logDebug("Done", "contextMenuClickAction", info);

                            return undefined;
                        });
                },
            );
        }

        createContextMenus() {
            return promiseTry(
                () => {
                    return Promise.all([
                        this.metadataManager.getEditionType(),
                        this.metadataManager.getSystemType(),
                    ])
                        .then(([editionType, systemType]) => {
                            const applicableContextMenuOptions = this.contextMenuOptionsCollection
                                .filter((contextMenuOption) => contextMenuOption[editionType] === true && contextMenuOption[systemType] === true);

                            // // TODO: group by selected contexts before checking against limit.
                            // if (applicableContextMenuOptions > this.actionMenuLimit) {
                            //     throw new Error("Maximum number of menu items reached.");
                            // }

                            const contextMenuOptionsCollectionPromises = applicableContextMenuOptions
                                .map((contextMenuOption) => this.createContextMenu(contextMenuOption.item));

                            return Promise.all(contextMenuOptionsCollectionPromises);
                        });
                },
            );
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class ShortcutKeyManager {
        constructor(commandHandler) {
            this.commandHandler = commandHandler;
        }

        handler(command) {
            return promiseTry(
                () => {
                    logDebug("Start", "handler", command);

                    // NOTE: straight mapping from command to action.
                    return this.commandHandler.handle(command)
                        .then((result) => {
                            logDebug("Done", "handler", command, result);

                            return undefined;
                        })
                        .catch((error) => {
                            logError("handler", command, error);

                            throw error;
                        });
                });
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class MetadataManager {
        constructor(manifestProvider, settingsManager) {
            this.manifestProvider = manifestProvider;
            this.settingsManager = settingsManager;

            this._editionTypePremium = "premium";
            this._editionTypeFree = "free";
            this._systemTypeChrome = "chrome";
            this._systemTypeWebExtension = "webextension";
        }

        isPremiumEdition() {
            return promiseTry(
                () => this.settingsManager.getIsPremiumEdition(),
            );
        }

        getExtensionId() {
            return promiseTry(
                () => browser.runtime.id,
            );
        }

        getManifestSync() {
            /* eslint-disable no-sync */
            return this.manifestProvider.getSync();
            /* eslint-enable no-sync */
        }

        getManifest() {
            return promiseTry(
                /* eslint-disable no-sync */
                () => this.getManifestSync(),
                /* eslint-enable no-sync */
            );
        }

        getVersionNumber() {
            return promiseTry(
                () => this.getManifest()
                    .then((manifest) => {
                        return manifest.version || null;
                    }),
            );
        }

        getVersionName() {
            return promiseTry(
                () => this.getManifest()
                    .then((manifest) => {
                        return manifest.version_name || null;
                    }),
            );
        }

        getEditionType() {
            return promiseTry(
                () => this.isPremiumEdition()
                    .then((isPremiumEdition) => {
                        if (isPremiumEdition) {
                            return this._editionTypePremium;
                        }

                        return this._editionTypeFree;
                    }),
            );
        }

        isChromeVersion() {
            return promiseTry(
                () => this.getVersionName()
                    .then((versionName) => {
                        if (versionName.includes(" Chrome Extension ")) {
                            return true;
                        }

                        return false;
                    }),
            );
        }

        isWebExtensionVersion() {
            return promiseTry(
                () => this.getVersionName()
                    .then((versionName) => {
                        if (versionName.includes(" WebExtension ")) {
                            return true;
                        }

                        return false;
                    }),
            );
        }

        getSystemType() {
            return promiseTry(
                () => this.isChromeVersion()
                    .then((isChrome) => {
                        if (isChrome) {
                            return this._systemTypeChrome;
                        }

                        return this._systemTypeWebExtension;
                    }),
            );
        }

        getOsType() {
            return promiseTry(
                () => browser.runtime.getPlatformInfo()
                    .then((platformInfo) => {
                        if (platformInfo && typeof platformInfo.os === "string") {
                            // https://developer.chrome.com/extensions/runtime#type-PlatformOs
                            return platformInfo.os;
                        }

                        return null;
                    }),
            );
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class WelcomeManager {
        openWelcomePage() {
            // TODO: focus the tab's window to ensure that the welcome text selection works?
            // https://developer.mozilla.org/en-US/Add-ons/WebExtensions/API/tabs/Tab
            // https://developer.mozilla.org/en-US/Add-ons/WebExtensions/API/windows/update
            return openInternalUrlFromConfigurationInNewTab("demo-welcome");
        };
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    // NOTE: https://developer.chrome.com/extensions/runtime#type-OnInstalledReason
    const REASON_INSTALL = "install";

    class OnInstalledManager {
        constructor(storageManager, settingsManager, metadataManager, contextMenuManager, welcomeManager, onInstallListenerEventQueue) {
            // TODO: use broadcast listeners instead.
            this.storageManager = storageManager;
            this.settingsManager = settingsManager;
            this.metadataManager = metadataManager;
            this.contextMenuManager = contextMenuManager;
            this.welcomeManager = welcomeManager;
            this.onInstallListenerEventQueue = onInstallListenerEventQueue;
        }

        _setSettingsManagerDefaults() {
            // TODO: move this function elsewhere?
            return promiseTry(
                () => {
                    logDebug("Start", "_setSettingsManagerDefaults");

                    return this.metadataManager.isWebExtensionVersion()
                        .then((isWebExtensionVersion) => {
                            // NOTE: enabling speaking long texts by default on in WebExtensions (Firefox).
                            const speakLongTexts = isWebExtensionVersion;

                            // TODO: move setting the default settings to the SettingsManager?
                            return this.settingsManager.setSpeakLongTexts(speakLongTexts);
                        })
                        .then((result) => {
                            logDebug("Done", "_setSettingsManagerDefaults");

                            return result;
                        })
                        .catch((error) => {
                            logError("_setSettingsManagerDefaults", error);

                            throw error;
                        });
                },
            );
        }

        onExtensionInstalledHandler(event) {
            return promiseTry(
                () => Promise.resolve()
                    .then(() => this.storageManager.upgradeIfNecessary())
                    // NOTE: removing all context menus in case the menus have changed since the last install/update.
                    .then(() => this.contextMenuManager.removeAll())
                    .then(() => this.contextMenuManager.createContextMenus())
                    .then(() => {
                        if (event.reason === REASON_INSTALL) {
                            return this._setSettingsManagerDefaults()
                                .then(() => this.welcomeManager.openWelcomePage());
                        }

                        return undefined;
                    })
                    .catch((error) => logError("onExtensionInstalledHandler", error)),
            );
        }

        onInstallListenerEventQueueHandler() {
            return promiseTry(
                () => {
                    while (this.onInstallListenerEventQueue.length > 0) {
                        const onInstallListenerEvent = this.onInstallListenerEventQueue.shift();

                        logDebug("onInstallListenerEventQueueHandler", "Start", onInstallListenerEvent);

                        return this.onExtensionInstalledHandler(onInstallListenerEvent.event)
                            .then(() => {
                                logDebug("onInstallListenerEventQueueHandler", "Done", onInstallListenerEvent);

                                return undefined;
                            });
                    }
                },
            );
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class StorageManager {
        constructor(storageProvider) {
            this.storageProvider = storageProvider;

            this.currentStorageFormatVersion = "v1.4.0";

            this.storageMetadataId = "_storage-metadata";

            this.allKnownStorageKeys = {};

            this.allKnownStorageKeys["v1.0.0"] = {
                "options-popup-donate-buttons-hide": "options-popup-donate-buttons-hide",
            };

            this.allKnownStorageKeys["v1.1.0"] = {
                "language-voice-overrides": "language-voice-overrides",
                "options-popup-donate-buttons-hide": "options-popup-donate-buttons-hide",
                "voice-pitch-overrides": "voice-pitch-overrides",
                "voice-rate-overrides": "voice-rate-overrides",
            };

            this.allKnownStorageKeys["v1.2.0"] = {
                "language-voice-overrides": "language-voice-overrides",
                "voice-pitch-overrides": "voice-pitch-overrides",
                "voice-rate-overrides": "voice-rate-overrides",
            };

            this.allKnownStorageKeys["v1.3.0"] = {
                "language-voice-overrides": "language-voice-overrides",
                "speak-long-texts": "speak-long-texts",
                "voice-pitch-overrides": "voice-pitch-overrides",
                "voice-rate-overrides": "voice-rate-overrides",
            };

            this.allKnownStorageKeys["v1.4.0"] = {
                "is-premium-edition": "is-premium-edition",
                "language-voice-overrides": "language-voice-overrides",
                "speak-long-texts": "speak-long-texts",
                "voice-pitch-overrides": "voice-pitch-overrides",
                "voice-rate-overrides": "voice-rate-overrides",
            };

            // TODO: sort by semantic version.
            this.allKnownStorageFormatVersions = Object.keys(this.allKnownStorageKeys);
            this.allKnownStorageFormatVersions.sort();

            this.allKnownUpgradePaths = {};
            this.allKnownUpgradePaths["v1.0.0"] = {};
            this.allKnownUpgradePaths["v1.0.0"]["v1.1.0"] = {
                upgradeKey: this._createIdentityUpgrader("v1.0.0", "v1.1.0"),
            };
            this.allKnownUpgradePaths["v1.0.0"]["v1.2.0"] = {
                upgradeKey: this._createIdentityUpgrader("v1.0.0", "v1.2.0"),
            };
            this.allKnownUpgradePaths["v1.1.0"] = {};
            this.allKnownUpgradePaths["v1.1.0"]["v1.2.0"] = {
                upgradeKey: this._createIdentityUpgrader("v1.1.0", "v1.2.0"),
            };
            this.allKnownUpgradePaths["v1.2.0"] = {};
            this.allKnownUpgradePaths["v1.2.0"]["v1.3.0"] = {
                upgradeKey: this._createIdentityUpgrader("v1.2.0", "v1.3.0"),
            };
            this.allKnownUpgradePaths["v1.3.0"] = {};
            this.allKnownUpgradePaths["v1.3.0"]["v1.4.0"] = {
                upgradeKey: this._createIdentityUpgrader("v1.3.0", "v1.4.0"),
            };
        }

        _getStorageKey(storageFormatVersion, key) {
            return promiseTry(
                () => {
                    if (!this.allKnownStorageKeys[storageFormatVersion]) {
                        throw new Error(`Unknown storage format version: (${storageFormatVersion})`);
                    }

                    if (key !== this.storageMetadataId && !this.allKnownStorageKeys[storageFormatVersion][key]) {
                        throw new Error(`Unknown storage key (${storageFormatVersion}): ${key}`);
                    }

                    return `${storageFormatVersion}_${key}`;
                },
            );
        }

        _isStorageKeyValid(storageFormatVersion, key) {
            return promiseTry(
                () => {
                    return this._getStorageKey(storageFormatVersion, key)
                        .then(() => {
                            return true;
                        })
                        .catch(() => {
                            // TODO: check for the specific storageKey errors.
                            return false;
                        });
                },
            );
        }

        _setStoredValue(storageFormatVersion, key, value) {
            return promiseTry(
                () => {
                    logTrace("Start", "_setStoredValue", storageFormatVersion, key, typeof value, value);

                    return this._getStorageKey(storageFormatVersion, key)
                        .then((storageKey) => {
                            return this.storageProvider.set(storageKey, value)
                                .then(() => {
                                    logTrace("Done", "_setStoredValue", storageFormatVersion, key, typeof value, value);

                                    return undefined;
                                });
                        });
                },
            );
        }

        setStoredValue(key, value) {
            return promiseTry(
                () => {
                    logDebug("Start", "setStoredValue", key, typeof value, value);

                    return this._setStoredValue(this.currentStorageFormatVersion, key, value)
                        .then(() => {
                            logDebug("Done", "setStoredValue", key, typeof value, value);

                            return undefined;
                        });
                },
            );
        }

        _getStoredValue(storageFormatVersion, key) {
            return promiseTry(
                () => {
                    logTrace("Start", "_getStoredValue", storageFormatVersion, key);

                    return this._getStorageKey(storageFormatVersion, key)
                        .then((storageKey) => {
                            return this.storageProvider.get(storageKey)
                                .then((value) => {
                                    logTrace("Done", "_getStoredValue", storageFormatVersion, key, value);

                                    return value;
                                });
                        });
                },
            );
        }

        getStoredValue(key) {
            return promiseTry(
                () => {
                    logTrace("Start", "getStoredValue", key);

                    return this._getStoredValue(this.currentStorageFormatVersion, key)
                        .then((value) => {
                            logTrace("Done", "getStoredValue", key, value);

                            return value;
                        });
                },
            );
        }

        _createIdentityUpgrader(fromStorageFormatVersion, toStorageFormatVersion) {
            const identityUpgrader = (key) => promiseTry(
                () => {
                    return this._isStorageKeyValid(fromStorageFormatVersion, key)
                        .then((isStorageKeyValid) => {
                            if (!isStorageKeyValid) {
                                return false;
                            }

                            return this._getStoredValue(fromStorageFormatVersion, key)
                                .then((fromValue) => {
                                    if (fromValue === undefined || fromValue === null) {
                                        return false;
                                    }

                                    const toValue = fromValue;

                                    return this._setStoredValue(toStorageFormatVersion, key, toValue)
                                        .then(() => true);
                                });
                        });
                },
            );

            return identityUpgrader;
        }

        _upgradeKey(fromStorageFormatVersion, toStorageFormatVersion, key) {
            return this.allKnownUpgradePaths[fromStorageFormatVersion][toStorageFormatVersion].upgradeKey(key);
        }

        _upgrade(fromStorageFormatVersion, toStorageFormatVersion) {
            return promiseTry(
                () => {
                    const storageKeysForVersion = Object.keys(this.allKnownStorageKeys[toStorageFormatVersion]);

                    const upgradePromises = storageKeysForVersion.map((key) => this._upgradeKey(fromStorageFormatVersion, toStorageFormatVersion, key));

                    return Promise.all(upgradePromises);
                },
            );
        }

        _getStorageMetadata(storageFormatVersion) {
            return promiseTry(
                () => {
                    return this._getStoredValue(storageFormatVersion, this.storageMetadataId);
                },
            );
        }

        _isStorageFormatVersionInitialized(storageFormatVersion) {
            return promiseTry(
                () => {
                    return this._getStorageMetadata(storageFormatVersion)
                        .then((storageMetadata) => {
                            if (storageMetadata !== null) {
                                return true;
                            }

                            return false;
                        });
                },
            );
        }

        _setStorageMetadataAsInitialized(storageFormatVersion, fromStorageFormatVersion) {
            return promiseTry(
                () => {
                    return this._isStorageFormatVersionInitialized(storageFormatVersion)
                        .then((storageFormatVersionIsInitialized) => {
                            if (storageFormatVersionIsInitialized) {
                                throw new Error(`Already initialized: ${storageFormatVersion}`);
                            }

                            return undefined;
                        })
                        .then(() => {
                            const storageMetadata = {
                                version: storageFormatVersion,
                                "upgraded-from-version": fromStorageFormatVersion,
                                "upgraded-at": Date.now(),
                            };

                            return this._setStoredValue(storageFormatVersion, this.storageMetadataId, storageMetadata);
                        });
                },
            );
        }

        _findUpgradePaths(toStorageFormatVersion) {
            return promiseTry(
                () => {
                    return this.allKnownStorageFormatVersions
                        .filter((knownStorageFormatVersion) => knownStorageFormatVersion !== toStorageFormatVersion)
                        .reverse()
                        .filter((knownStorageFormatVersion) => {
                            if (!this.allKnownUpgradePaths[knownStorageFormatVersion]) {
                                return false;
                            }

                            if (!this.allKnownUpgradePaths[knownStorageFormatVersion][toStorageFormatVersion]) {
                                return false;
                            }

                            const upgradePath = this.allKnownUpgradePaths[knownStorageFormatVersion][toStorageFormatVersion];

                            if (upgradePath) {
                                return true;
                            }

                            return false;
                        });
                },
            );
        }

        _findUpgradePath(toStorageFormatVersion) {
            return promiseTry(
                () => {
                    return this._findUpgradePaths(toStorageFormatVersion)
                        .then((upgradePaths) => {
                            if (!upgradePaths || !Array.isArray(upgradePaths) || upgradePaths.length === 0) {
                                return null;
                            }

                            const possiblyInitializedUpgradePathPromises = upgradePaths.map((upgradePath) => {
                                return this._isStorageFormatVersionInitialized(upgradePath)
                                    .then((storageFormatVersionIsInitialized) => {
                                        if (storageFormatVersionIsInitialized) {
                                            return upgradePath;
                                        }

                                        return null;
                                    });
                            });

                            return Promise.all(possiblyInitializedUpgradePathPromises)
                                .then((possiblyInitializedUpgradePaths) => {
                                    return possiblyInitializedUpgradePaths
                                        .filter((possiblyInitializedUpgradePath) => !!possiblyInitializedUpgradePath);
                                })
                                .then((initializedUpgradePaths) => {
                                    if (initializedUpgradePaths.length === 0) {
                                        return null;
                                    }

                                    const firstInitializedUpgradePath = initializedUpgradePaths[0];

                                    return firstInitializedUpgradePath;
                                });
                        });
                },
            );
        }

        _upgradeIfNecessary(storageFormatVersion) {
            return promiseTry(
                () => {
                    return this._isStorageFormatVersionInitialized(storageFormatVersion)
                        .then((storageFormatVersionIsInitialized) => {
                            if (!storageFormatVersionIsInitialized) {
                                return this._findUpgradePath(storageFormatVersion)
                                    .then((firstInitializedUpgradePath) => {
                                        return promiseTry(
                                            () => {
                                                if (!firstInitializedUpgradePath) {
                                                    return false;
                                                }

                                                return this._upgrade(firstInitializedUpgradePath, storageFormatVersion)
                                                    .then(() => true);
                                            })
                                            .then((result) => {
                                                this._setStorageMetadataAsInitialized(storageFormatVersion, firstInitializedUpgradePath);

                                                return result;
                                            });
                                    });
                            }

                            return false;
                        });
                },
            );
        }

        _upgradeV1x0x0IfNecessary() {
            return promiseTry(
                () => {
                    const storageFormatVersion1x0x0 = "v1.0.0";
                    const keyToCheck = "options-popup-donate-buttons-hide";

                    // NOTE: return v1.0.0 as initialized if it had the single setting set,
                    // as it didn't have initialization code yet.
                    return this._isStorageFormatVersionInitialized(storageFormatVersion1x0x0)
                        .then((storageFormatVersionIsInitialized) => {
                            if (!storageFormatVersionIsInitialized) {
                                return this._getStoredValue(storageFormatVersion1x0x0, keyToCheck)
                                    .then((storageValueToCheck) => {
                                        if (storageValueToCheck !== null) {
                                            return this._setStorageMetadataAsInitialized(storageFormatVersion1x0x0, null)
                                                .then(() => true);
                                        }

                                        return false;
                                    });
                            }

                            return false;
                        });
                },
            );
        }

        upgradeIfNecessary() {
            return promiseTry(
                () => {
                    logDebug("Start", "upgradeIfNecessary");

                    return this._upgradeV1x0x0IfNecessary()
                        .then(() => this._upgradeIfNecessary(this.currentStorageFormatVersion))
                        .then((result) => {
                            logDebug("Done", "upgradeIfNecessary");

                            return result;
                        })
                        .catch((error) => {
                            logError("upgradeIfNecessary", error);

                            throw error;
                        });
                },
            );
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class SettingsManager {
        constructor(storageManager) {
            this.storageManager = storageManager;

            // TODO: shared place for stored value constants.
            this._isPremiumEditionStorageKey = "is-premium-edition";
            this._speakLongTextsStorageKey = "speak-long-texts";

            // TODO: shared place for default/fallback values for booleans etcetera.
            this._isPremiumEditionDefaultValue = false;
            this._speakLongTextsStorageKeyDefaultValue = false;
        }

        setIsPremiumEdition(isPremiumEdition) {
            return promiseTry(
                () => this.storageManager.setStoredValue(this._isPremiumEditionStorageKey, isPremiumEdition === true),
            );
        }

        getIsPremiumEdition() {
            return promiseTry(
                () => this.storageManager.getStoredValue(this._isPremiumEditionStorageKey)
                    .then((isPremiumEdition) => isPremiumEdition || this._isPremiumEditionDefaultValue),
            );
        }

        setSpeakLongTexts(speakLongTexts) {
            return promiseTry(
                () => this.storageManager.setStoredValue(this._speakLongTextsStorageKey, speakLongTexts === true),
            );
        }

        getSpeakLongTexts() {
            return promiseTry(
                () => this.storageManager.getStoredValue(this._speakLongTextsStorageKey)
                    .then((speakLongTexts) => speakLongTexts || this._speakLongTextsStorageKeyDefaultValue),
            );
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class LanguageHelper {
        constructor(contentLogger, configuration, translator) {
            this.contentLogger = contentLogger;
            this.configuration = configuration;
            this.translator = translator;

            this.noTextSelectedMessage = {
                text: this.translator.translate("noTextSelectedMessage"),
                effectiveLanguage: this.translator.translate("extensionLocale"),
            };

            this.noVoiceForLanguageDetectedMessage = {
                text: this.translator.translate("noVoiceForLanguageDetectedMessage"),
                effectiveLanguage: this.translator.translate("noVoiceForLanguageDetectedMessageLanguage"),
            };

            // https://www.iso.org/obp/ui/#iso:std:iso:639:-1:ed-1:v1:en
            // https://en.wikipedia.org/wiki/ISO_639-1
            // https://en.wikipedia.org/wiki/List_of_ISO_639-1_codes
            // http://xml.coverpages.org/iso639a.html
            // NOTE: discovered because Twitter seems to still use "iw".
            this.iso639Dash1Aliases1988To2002 = {
                "in": "id",
                "iw": "he",
                "ji": "yi",
            };
        }

        detectPageLanguage() {
            return promiseTry(
                () => {
                    // https://developer.browser.com/extensions/tabs#method-detectLanguage
                    return browser.tabs.detectLanguage()
                        .then((language) => {
                            logDebug("detectPageLanguage", "Browser detected primary page language", language);

                            // The language fallback value is "und", so treat it as no language.
                            if (!language || typeof language !== "string" || language === "und") {
                                return null;
                            }

                            return language;
                        })
                        .catch((error) => {
                            // https://github.com/joelpurra/talkie/issues/3
                            // NOTE: It seems the Vivaldi browser doesn't (yet/always) support detectLanguage.
                            // As this is not critical, just log the error and resolve with null.
                            logError("detectPageLanguage", error);

                            return null;
                        });
                },
            );
        }

        detectTextLanguage(text) {
            return promiseTry(
                () => {
                    if (!("detectLanguage" in browser.i18n)) {
                        // NOTE: text-based language detection is only used as a fallback.
                        logDebug("detectTextLanguage", "Browser does not support detecting text language");

                        return null;
                    }

                    // https://developer.browser.com/extensions/i18n#method-detectLanguage
                    return browser.i18n.detectLanguage(text)
                        .then((result) => {
                            const MINIMUM_RELIABILITY_PERCENTAGE = 50;

                            if (
                                !result
                                    // NOTE: the "isReliable" flag can apparently be false for languages with 100% reliabilty.
                                    // NOTE: using the percentage instead.
                                    // || !result.isReliable
                                    || !result.languages
                                    || !(result.languages.length > 0)
                                    || typeof result.languages[0].language !== "string"
                                    || !(result.languages[0].language.trim().length > 0)
                                    // NOTE: The language fallback value is "und", so treat it as no language.
                                    || result.languages[0].language === "und"
                                    || result.languages[0].percentage < MINIMUM_RELIABILITY_PERCENTAGE
                            ) {
                                // NOTE: text-based language detection is only used as a fallback.
                                logDebug("detectTextLanguage", "Browser did not detect reliable text language", result);

                                return null;
                            }

                            const primaryDetectedTextLanguage = result.languages[0].language;

                            logDebug("detectTextLanguage", "Browser detected reliable text language", result, primaryDetectedTextLanguage);

                            return primaryDetectedTextLanguage;
                        });
                },
            );
        }

        getSelectionsWithValidText(selections) {
            return promiseTry(
                () => {
                    const isNonNullObject = (selection) => !!selection && typeof selection === "object";

                    const hasValidText = (selection) => !isUndefinedOrNullOrEmptyOrWhitespace(selection.text);

                    const trimText = (selection) => {
                        const copy = shallowCopy(selection);

                        copy.text = copy.text.trim();

                        return copy;
                    };

                    const selectionsWithValidText = selections
                        .filter(isNonNullObject)
                        .filter(hasValidText)
                        .map(trimText)
                        .filter(hasValidText);

                    return selectionsWithValidText;
                },
            );
        }

        detectAndAddLanguageForSelections(selectionsWithValidText) {
            return promiseTry(
                () => Promise.all(
                    selectionsWithValidText.map(
                        (selection) => {
                            const copy = shallowCopy(selection);

                            return this.detectTextLanguage(copy.text)
                                .then((detectedTextLanguage) => {
                                    copy.detectedTextLanguage = detectedTextLanguage;

                                    return copy;
                                });
                        }),
                ),
            );
        }

        isKnownVoiceLanguage(allVoices, elementLanguage) {
            return allVoices.some((voice) => voice.lang.startsWith(elementLanguage));
        }

        mapIso639Aliases(language) {
            return this.iso639Dash1Aliases1988To2002[language] || language;
        }

        isValidString(str) {
            return !isUndefinedOrNullOrEmptyOrWhitespace(str);
        }

        cleanupLanguagesArray(allVoices, languages) {
            const copy = (languages || [])
                .filter((str) => this.isValidString(str))
                .map((language) => this.mapIso639Aliases(language))
                .filter((elementLanguage) => this.isKnownVoiceLanguage(allVoices, elementLanguage));

            return copy;
        }

        getSelectionsWithValidTextAndDetectedLanguageAndEffectiveLanguage(allVoices, detectedPageLanguage, selectionsWithValidTextAndDetectedLanguage) {
            return promiseTry(
                () => {
                    const cleanupParentElementsLanguages = (selection) => {
                        const copy = shallowCopy(selection);

                        copy.parentElementsLanguages = this.cleanupLanguagesArray(allVoices, copy.parentElementsLanguages);

                        return copy;
                    };

                    const getMoreSpecificLanguagesWithPrefix = (prefix) => {
                        return (language) => language.startsWith(prefix) && language.length > prefix.length;
                    };

                    const setEffectiveLanguage = (selection) => {
                        const copy = shallowCopy(selection);

                        const detectedLanguages = [
                            copy.detectedTextLanguage,
                            copy.parentElementsLanguages[0] || null,
                            copy.htmlTagLanguage,
                            detectedPageLanguage,
                        ];

                        logDebug("setEffectiveLanguage", "detectedLanguages", detectedLanguages);

                        const cleanedLanguages = this.cleanupLanguagesArray(allVoices, detectedLanguages);

                        logDebug("setEffectiveLanguage", "cleanedLanguages", cleanedLanguages);

                        const primaryLanguagePrefix = cleanedLanguages[0] || null;

                        logDebug("setEffectiveLanguage", "primaryLanguagePrefix", primaryLanguagePrefix);

                        // NOTE: if there is a more specific language with the same prefix among the detected languages, prefer it.
                        const cleanedLanguagesWithPrimaryPrefix = cleanedLanguages.filter(getMoreSpecificLanguagesWithPrefix(primaryLanguagePrefix));

                        logDebug("setEffectiveLanguage", "cleanedLanguagesWithPrimaryPrefix", cleanedLanguagesWithPrimaryPrefix);

                        const effectiveLanguage = cleanedLanguagesWithPrimaryPrefix[0] || cleanedLanguages[0] || null;

                        logDebug("setEffectiveLanguage", "effectiveLanguage", effectiveLanguage);

                        copy.effectiveLanguage = effectiveLanguage;

                        // TODO: report language results and move logging elsewhere?
                        promiseSeries([
                            () => this.contentLogger.logToPage("Language", "Selected text language:", copy.detectedTextLanguage),
                            () => this.contentLogger.logToPage("Language", "Selected text element language:", copy.parentElementsLanguages[0] || null),
                            () => this.contentLogger.logToPage("Language", "HTML tag language:", copy.htmlTagLanguage),
                            () => this.contentLogger.logToPage("Language", "Detected page language:", detectedPageLanguage),
                            () => this.contentLogger.logToPage("Language", "Effective language:", copy.effectiveLanguage),
                        ])
                            .catch((error) => {
                                // NOTE: swallowing any logToPage() errors.
                                // NOTE: reduced logging for known tab/page access problems.
                                if (error && typeof error.message === "string" && error.message.startsWith("Cannot access")) {
                                    logDebug("getSelectionsWithValidTextAndDetectedLanguageAndEffectiveLanguage", "Error", error);
                                } else {
                                    logInfo("getSelectionsWithValidTextAndDetectedLanguageAndEffectiveLanguage", "Error", error);
                                }

                                return undefined;
                            });

                        return copy;
                    };

                    const selectionsWithValidTextAndDetectedLanguageAndEffectiveLanguage = selectionsWithValidTextAndDetectedLanguage
                        .map(cleanupParentElementsLanguages)
                        .map(setEffectiveLanguage);

                    return selectionsWithValidTextAndDetectedLanguageAndEffectiveLanguage;
                },
            );
        }

        useFallbackMessageIfNoLanguageDetected(selectionsWithValidTextAndDetectedLanguageAndEffectiveLanguage) {
            return promiseTry(
                () => {
                    const fallbackMessageForNoLanguageDetected = (selection) => {
                        if (selection.effectiveLanguage === null) {
                            return this.noVoiceForLanguageDetectedMessage;
                        }

                        return selection;
                    };

                    const mapResults = (selection) => {
                        return {
                            text: selection.text,
                            effectiveLanguage: selection.effectiveLanguage,
                        };
                    };

                    const results = selectionsWithValidTextAndDetectedLanguageAndEffectiveLanguage
                        .map(fallbackMessageForNoLanguageDetected)
                        .map(mapResults);

                    if (results.length === 0) {
                        logDebug("Empty filtered selections");

                        results.push(this.noTextSelectedMessage);
                    }

                    return results;
                },
            );
        }

        cleanupSelections(allVoices, detectedPageLanguage, selections) {
            return Promise.resolve()
                .then(() => this.getSelectionsWithValidText(selections))
                .then((selectionsWithValidText) => this.detectAndAddLanguageForSelections(selectionsWithValidText))
                .then((selectionsWithValidTextAndDetectedLanguage) => this.getSelectionsWithValidTextAndDetectedLanguageAndEffectiveLanguage(allVoices, detectedPageLanguage, selectionsWithValidTextAndDetectedLanguage))
                .then((selectionsWithValidTextAndDetectedLanguageAndEffectiveLanguage) => this.useFallbackMessageIfNoLanguageDetected(selectionsWithValidTextAndDetectedLanguageAndEffectiveLanguage));
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    class Execute {
        constructor(configuration) {
            this.configuration = configuration;
        }

        scriptInTopFrame(code) {
            return promiseTry(
                () => {
                    logDebug("Start", "scriptInTopFrame", code.length, code);

                    return browser.tabs.executeScript(
                        {
                            allFrames: false,
                            code: code,
                        },
                    )
                        .then((result) => {
                            logDebug("Done", "scriptInTopFrame", code.length);

                            return result;
                        })
                        .catch((error) => {
                            logInfo("scriptInTopFrame", code.length, "Error", error);

                            throw error;
                        });
                },
            );
        }

        scriptInAllFrames(code) {
            return promiseTry(
                () => {
                    logDebug("Start", "scriptInAllFrames", code.length, code);

                    return browser.tabs.executeScript(
                        {
                            allFrames: true,
                            code: code,
                        },
                    )
                        .then((result) => {
                            logDebug("Done", "scriptInAllFrames", code.length);

                            return result;
                        })
                        .catch((error) => {
                            logInfo("scriptInAllFrames", code.length, "Error", error);

                            throw error;
                        });
                },
            );
        }

        scriptInTopFrameWithTimeout(code, timeout) {
            return promiseTry(
                () => {
                    logDebug("Start", "scriptInTopFrameWithTimeout", code.length, "code.length", timeout, "milliseconds");

                    return promiseTimeout(
                        this.scriptInTopFrame(code),
                        timeout,
                    )
                        .then((result) => {
                            logDebug("Done", "scriptInTopFrameWithTimeout", code.length, "code.length", timeout, "milliseconds");

                            return result;
                        })
                        .catch((error) => {
                            if (error && typeof error.name === "PromiseTimeout") {
                                // NOTE: this is how to check for a timeout.
                            }

                            throw error;
                        });
                },
            );
        }

        scriptInAllFramesWithTimeout(code, timeout) {
            return promiseTry(
                () => {
                    logDebug("Start", "scriptInAllFramesWithTimeout", code.length, "code.length", timeout, "milliseconds");

                    return promiseTimeout(
                        this.scriptInAllFrames(code),
                        timeout,
                    )
                        .then((result) => {
                            logDebug("Done", "scriptInAllFramesWithTimeout", code.length, "code.length", timeout, "milliseconds");

                            return result;
                        })
                        .catch((error) => {
                            if (error && typeof error.name === "PromiseTimeout") {
                            // NOTE: this is how to check for a timeout.
                            }

                            throw error;
                        });
                },
            );
        }
    }

    /*
    This file is part of Talkie -- text-to-speech browser extension button.
    <https://joelpurra.com/projects/talkie/>

    Copyright (c) 2016, 2017, 2018, 2019, 2020, 2021 Joel Purra <https://joelpurra.com/>

    Talkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Talkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Talkie.  If not, see <https://www.gnu.org/licenses/>.
    */

    registerUnhandledRejectionHandler();

    // NOTE: synchronous handling of the onInstall event through a separate, polled queue handled by the OnInstalledManager.
    const onInstallListenerEventQueue = [];

    const startOnInstallListener = () => {
        const onInstallListener = (event) => {
            const onInstallEvent = {
                source: "event",
                event: event,
            };

            onInstallListenerEventQueue.push(onInstallEvent);
        };

        // NOTE: "This event is not triggered for temporarily installed add-ons."
        // https://developer.mozilla.org/en-US/Add-ons/WebExtensions/API/runtime/onInstalled#Compatibility_notes
        // NOTE: When using the WebExtensions polyfill, this check doesn't seem to work as browser.runtime.onInstalled always exists.
        // https://github.com/mozilla/webextension-polyfill
        if (browser.runtime.onInstalled) {
            // NOTE: the onInstalled listener can't be added asynchronously
            browser.runtime.onInstalled.addListener(onInstallListener);
        } else {
            const onInstallEvent = {
                source: "fallback",
                event: null,
            };

            onInstallListenerEventQueue.push(onInstallEvent);
        }
    };

    function main() {
        logDebug("Start", "Main background function");

        const storageProvider = new WebExtensionEnvironmentStorageProvider();
        const storageManager = new StorageManager(storageProvider);
        const settingsManager = new SettingsManager(storageManager);
        const manifestProvider = new WebExtensionEnvironmentManifestProvider();
        const metadataManager = new MetadataManager(manifestProvider, settingsManager);
        const internalUrlProvider = new WebExtensionEnvironmentInternalUrlProvider();
        const configuration = new Configuration(metadataManager, configurationObject);

        const broadcaster = new Broadcaster();

        const onlyLastCaller = new OnlyLastCaller();
        const shouldContinueSpeakingProvider = onlyLastCaller;
        const execute = new Execute();
        const contentLogger = new ContentLogger(execute, configuration);
        const talkieSpeaker = new TalkieSpeaker(broadcaster, shouldContinueSpeakingProvider, contentLogger, settingsManager);
        const speakingStatus = new SpeakingStatus();

        const voiceLanguageManager = new VoiceLanguageManager(storageManager, metadataManager);
        const voiceRateManager = new VoiceRateManager(storageManager, metadataManager);
        const voicePitchManager = new VoicePitchManager(storageManager, metadataManager);
        const voiceManager = new VoiceManager(voiceLanguageManager, voiceRateManager, voicePitchManager);
        const localeProvider = new WebExtensionEnvironmentLocaleProvider();
        const translatorProvider = new WebExtensionEnvironmentTranslatorProvider(localeProvider);
        const languageHelper = new LanguageHelper(contentLogger, configuration, translatorProvider);

        // NOTE: using a chainer to be able to add user (click/shortcut key/context menu) initialized speech events one after another.
        const speechChain = new Chain();
        const talkieBackground = new TalkieBackground(speechChain, broadcaster, talkieSpeaker, speakingStatus, voiceManager, languageHelper, configuration, execute, translatorProvider, internalUrlProvider);
        const permissionsManager = new PermissionsManager();
        const clipboardManager = new ClipboardManager(talkieBackground, permissionsManager);
        const readClipboardManager = new ReadClipboardManager(clipboardManager, talkieBackground, permissionsManager, metadataManager, translatorProvider);

        const commandMap = {
            // NOTE: implicitly set by the browser, and actually "clicks" the Talkie icon.
            // Handled by the popup handler (popup contents) and icon click handler.
            // "_execute_browser_action": talkieBackground.startStopSpeakSelectionOnPage(),
            "start-stop": () => talkieBackground.startStopSpeakSelectionOnPage(),
            "start-text": (text) => talkieBackground.startSpeakingCustomTextDetectLanguage(text),
            "read-clipboard": () => readClipboardManager.startSpeaking(),
            "open-website-main": () => openUrlFromConfigurationInNewTab("main"),
            "open-website-upgrade": () => openUrlFromConfigurationInNewTab("upgrade"),
        };

        const commandHandler = new CommandHandler(commandMap);
        const contextMenuManager = new ContextMenuManager(commandHandler, metadataManager, translatorProvider);
        const shortcutKeyManager = new ShortcutKeyManager(commandHandler);

        const suspensionConnectorManager = new SuspensionConnectorManager();
        const suspensionManager = new SuspensionManager(suspensionConnectorManager);
        const iconManager = new IconManager(metadataManager);
        const buttonPopupManager = new ButtonPopupManager(translatorProvider, metadataManager);

        const progress = new TalkieProgress(broadcaster);

        const plug = new Plug(contentLogger, execute);

        const welcomeManager = new WelcomeManager();
        const onInstalledManager = new OnInstalledManager(storageManager, settingsManager, metadataManager, contextMenuManager, welcomeManager, onInstallListenerEventQueue);

        (function addOnInstalledEventQueuePolling() {
            // NOTE: run the function once first, to allow for a very long interval.
            const ONE_SECOND_IN_MILLISECONDS = 1 * 1000;
            const ON_INSTALL_LISTENER_EVENT_QUEUE_HANDLER_TIMEOUT = ONE_SECOND_IN_MILLISECONDS;

            /* eslint-disable no-unused-vars */
            const onInstallListenerEventQueueHandlerTimeoutId = setTimeout(
                () => onInstalledManager.onInstallListenerEventQueueHandler(),
                ON_INSTALL_LISTENER_EVENT_QUEUE_HANDLER_TIMEOUT,
            );
            /* eslint-enable no-unused-vars */

            // NOTE: run the function with a very long interval.
            const ONE_HOUR_IN_MILLISECONDS = 60 * 60 * 1000;
            const ON_INSTALL_LISTENER_EVENT_QUEUE_HANDLER_INTERVAL = ONE_HOUR_IN_MILLISECONDS;

            /* eslint-disable no-unused-vars */
            const onInstallListenerEventQueueHandlerIntervalId = setInterval(
                () => onInstalledManager.onInstallListenerEventQueueHandler(),
                ON_INSTALL_LISTENER_EVENT_QUEUE_HANDLER_INTERVAL,
            );
            /* eslint-enable no-unused-vars */
        }());

        // NOTE: cache listeners so they can be added and removed by reference before/after speaking.
        const onTabRemovedListener = loggedPromise("onRemoved", () => talkieBackground.onTabRemovedHandler());
        const onTabUpdatedListener = loggedPromise("onUpdated", () => talkieBackground.onTabUpdatedHandler());

        // TODO: put initialization promise on the root chain?
        return Promise.resolve()
            .then(() => suspensionManager.initialize())
            .then(() => {
                const killSwitches = [];

                const executeKillSwitches = () => {
                    // NOTE: expected to have only synchronous methods for the relevant parts.
                    killSwitches.forEach((killSwitch) => {
                        try {
                            killSwitch();
                        } catch (error) {
                            logError("executeKillSwitches", error);
                        }
                    });
                };

                // NOTE: synchronous version.
                window.addEventListener("beforeunload", () => {
                    executeKillSwitches();
                });

                return Promise.all([
                    broadcaster.registerListeningAction(knownEvents.stopSpeaking, () => onlyLastCaller.incrementCallerId()),
                    broadcaster.registerListeningAction(knownEvents.afterSpeaking, () => onlyLastCaller.incrementCallerId()),

                    broadcaster.registerListeningAction(knownEvents.afterSpeaking, () => plug.once()
                        .catch((error) => {
                            // NOTE: swallowing any plug.once() errors.
                            // NOTE: reduced logging for known tab/page access problems.
                            if (error && typeof error.message === "string" && error.message.startsWith("Cannot access")) {
                                logDebug("plug.once", "Error swallowed", error);
                            } else {
                                logInfo("plug.once", "Error swallowed", error);
                            }

                            return undefined;
                        })),

                    broadcaster.registerListeningAction(knownEvents.beforeSpeaking, () => speakingStatus.setActiveTabAsSpeaking()),
                    broadcaster.registerListeningAction(knownEvents.afterSpeaking, () => speakingStatus.setDoneSpeaking()),

                    // NOTE: setting icons async.
                    broadcaster.registerListeningAction(knownEvents.beforeSpeaking, () => promiseSleep(() => iconManager.setIconModePlaying(), 10)),
                    broadcaster.registerListeningAction(knownEvents.afterSpeaking, () => promiseSleep(() => iconManager.setIconModeStopped(), 10)),

                    // NOTE: a feeble attempt to make the popup window render properly, instead of only a tiny box flashing away, as the reflow() has questionable effect.
                    broadcaster.registerListeningAction(knownEvents.beforeSpeaking, () => promiseSleep(() => buttonPopupManager.disablePopup(), 200)),
                    broadcaster.registerListeningAction(knownEvents.afterSpeaking, () => promiseSleep(() => buttonPopupManager.enablePopup(), 200)),

                    broadcaster.registerListeningAction(knownEvents.beforeSpeaking, () => suspensionManager.preventExtensionSuspend()),
                    broadcaster.registerListeningAction(knownEvents.afterSpeaking, () => suspensionManager.allowExtensionSuspend()),

                    broadcaster.registerListeningAction(knownEvents.beforeSpeaking, () => browser.tabs.onRemoved.addListener(onTabRemovedListener)),
                    broadcaster.registerListeningAction(knownEvents.afterSpeaking, () => browser.tabs.onRemoved.removeListener(onTabRemovedListener)),

                    broadcaster.registerListeningAction(knownEvents.beforeSpeaking, () => browser.tabs.onUpdated.addListener(onTabUpdatedListener)),
                    broadcaster.registerListeningAction(knownEvents.afterSpeaking, () => browser.tabs.onUpdated.removeListener(onTabUpdatedListener)),

                    broadcaster.registerListeningAction(knownEvents.beforeSpeaking, (/* eslint-disable no-unused-vars*/actionName/* eslint-enable no-unused-vars*/, actionData) => progress.resetProgress(0, actionData.text.length, 0)),
                    broadcaster.registerListeningAction(knownEvents.beforeSpeakingPart, (/* eslint-disable no-unused-vars*/actionName/* eslint-enable no-unused-vars*/, actionData) => progress.startSegment(actionData.textPart.length)),
                    broadcaster.registerListeningAction(knownEvents.afterSpeakingPart, () => progress.endSegment()),
                    broadcaster.registerListeningAction(knownEvents.afterSpeaking, () => progress.finishProgress()),
                ])
                    .then((registeredKillSwitches) => {
                        // NOTE: don't want to replace the existing killSwitches array.
                        registeredKillSwitches.forEach((registeredKillSwitch) => killSwitches.push(registeredKillSwitch));

                        return undefined;
                    });
            })
            .then(() => {
                // NOTE: not supported in Firefox (2017-04-28).
                // https://developer.mozilla.org/en-US/Add-ons/WebExtensions/API/runtime/onSuspend#Browser_compatibility
                if ("onSuspend" in browser.runtime) {
                    browser.runtime.onSuspend.addListener(loggedPromise("onSuspend", () => talkieBackground.onExtensionSuspendHandler()));
                    // browser.runtime.onSuspend.addListener(loggedPromise("onSuspend", () => suspensionManager.unintialize()));
                }

                // NOTE: not supported in Firefox (2017-04-28).
                // https://developer.mozilla.org/en-US/Add-ons/WebExtensions/API/runtime/onSuspend#Browser_compatibility
                // if ("onSuspendCanceled" in browser.runtime) {
                //     browser.runtime.onSuspendCanceled.addListener(loggedPromise("onSuspendCanceled", () => suspensionManager.initialize()));
                // }

                // NOTE: used when the popup has been disabled.
                browser.browserAction.onClicked.addListener(loggedPromise("onClicked", () => talkieBackground.startStopSpeakSelectionOnPage()));

                browser.contextMenus.onClicked.addListener(loggedPromise("onClicked", (info) => contextMenuManager.contextMenuClickAction(info)));

                // NOTE: might throw an unexpected error in Firefox due to command configuration in manifest.json.
                // Does not seem to happen in Chrome.
                // https://developer.mozilla.org/en-US/Add-ons/WebExtensions/API/commands/onCommand
                try {
                    browser.commands.onCommand.addListener(loggedPromise("onCommand", (command) => shortcutKeyManager.handler(command)));
                } catch (error) {
                    logError("browser.commands.onCommand.addListener(...)", error);
                }

                return undefined;
            })
            .then(() => {
                window.broadcaster = () => broadcaster;

                window.logTrace = (...args) => logTrace(...args);
                window.logDebug = (...args) => logDebug(...args);
                window.logInfo = (...args) => logInfo(...args);
                window.logWarn = (...args) => logWarn(...args);
                window.logError = (...args) => logError(...args);
                window.setLoggingLevel = (...args) => setLevel(...args);
                window.setLoggingStringOnlyOutput = (...args) => setStringOnlyOutput(...args);

                window.getAllVoices = () => talkieSpeaker.getAllVoices();
                window.iconClick = () => talkieBackground.startStopSpeakSelectionOnPage();
                window.stopSpeakFromFrontend = () => talkieBackground.stopSpeakingAction();
                window.startSpeakFromFrontend = (frontendText, frontendVoice) => {
                    // NOTE: not sure if copying these variables have any effect.
                    // NOTE: Hope it helps avoid some vague "TypeError: can't access dead object" in Firefox.
                    const text = String(frontendText);
                    const voice = {
                        name: typeof frontendVoice.name === "string" ? String(frontendVoice.name) : undefined,
                        lang: typeof frontendVoice.lang === "string" ? String(frontendVoice.lang) : undefined,
                        rate: !isNaN(frontendVoice.rate) ? (0 + frontendVoice.rate) : undefined,
                        pitch: !isNaN(frontendVoice.pitch) ? (0 + frontendVoice.pitch) : undefined,
                    };

                    talkieBackground.startSpeakingTextInVoiceAction(text, voice);
                };
                window.startSpeakInLanguageWithOverridesFromFrontend = (frontendText, frontendLanguageCode) => {
                    // NOTE: not sure if copying these variables have any effect.
                    // NOTE: Hope it helps avoid some vague "TypeError: can't access dead object" in Firefox.
                    const text = String(frontendText);
                    const languageCode = String(frontendLanguageCode);

                    talkieBackground.startSpeakingTextInLanguageWithOverridesAction(text, languageCode);
                };

                window.getVersionNumber = () => metadataManager.getVersionNumber();
                window.getVersionName = () => metadataManager.getVersionName();
                window.getEditionType = () => metadataManager.getEditionType();
                window.isPremiumEdition = () => metadataManager.isPremiumEdition();
                window.getSystemType = () => metadataManager.getSystemType();
                window.getOsType = () => metadataManager.getOsType();

                window.getIsPremiumEditionOption = () => settingsManager.getIsPremiumEdition();
                window.setIsPremiumEditionOption = (isPremiumEdition) => settingsManager.setIsPremiumEdition(isPremiumEdition);
                window.getSpeakLongTextsOption = () => settingsManager.getSpeakLongTexts();
                window.setSpeakLongTextsOption = (speakLongTexts) => settingsManager.setSpeakLongTexts(speakLongTexts);

                window.setVoiceRateOverride = (voiceName, rate) => voiceManager.setVoiceRateOverride(voiceName, rate);
                window.getEffectiveVoiceForLanguage = (languageName) => voiceManager.getEffectiveVoiceForLanguage(languageName);
                window.isLanguageVoiceOverrideName = (languageName, voiceName) => voiceManager.isLanguageVoiceOverrideName(languageName, voiceName);
                window.toggleLanguageVoiceOverrideName = (languageName, voiceName) => voiceManager.toggleLanguageVoiceOverrideName(languageName, voiceName);
                window.getVoiceRateDefault = (voiceName) => voiceManager.getVoiceRateDefault(voiceName);
                window.setVoiceRateOverride = (voiceName, rate) => voiceManager.setVoiceRateOverride(voiceName, rate);
                window.getEffectiveRateForVoice = (voiceName) => voiceManager.getEffectiveRateForVoice(voiceName);
                window.getVoicePitchDefault = (voiceName) => voiceManager.getVoicePitchDefault(voiceName);
                window.setVoicePitchOverride = (voiceName, pitch) => voiceManager.setVoicePitchOverride(voiceName, pitch);
                window.getEffectivePitchForVoice = (voiceName) => voiceManager.getEffectivePitchForVoice(voiceName);
                window.getStoredValue = (key) => storageManager.getStoredValue(key);
                window.setStoredValue = (key, value) => storageManager.setStoredValue(key, value);
                window.getConfigurationValue = (path) => configuration.get(path);

                return undefined;
            })
            .then(() => {
                buttonPopupManager.enablePopup();

                logDebug("Done", "Main background function");

                return undefined;
            });
    }

    try {
        startOnInstallListener();

        main();
    } catch (error) {
        logError("background", error);
    }

})));
//# sourceMappingURL=background.js.map
