# -*- coding:utf-8 -*-

# Speedflow Add-on
# Copyright (C) 2018 Cedric Lepiller aka Pitiwazou & Legigan Jeremy AKA Pistiwique and Stephen Leger
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

# <pep8 compliant>


import bpy, os
import bmesh
from bpy.types import Menu, Operator
from os.path import isfile, isdir, join, basename, split, exists, basename, dirname, abspath
from bpy.props import PointerProperty, StringProperty, BoolProperty, \
    EnumProperty, IntProperty, FloatProperty, FloatVectorProperty, \
    CollectionProperty, BoolVectorProperty
from math import log,sqrt
from mathutils import *
import math
from math import ceil
import random
from .functions import *
import colorsys
from math import radians
from .ui import *
#=================================================================#
#  Object Mode
#=================================================================#

class SFC_OT_vgroups(bpy.types.Operator):
    """    VERTEX GROUPS

    CLICK - Create Vertex Group
    SHIFT - Assign to Active Vertex Group
    CTRL  - Remove from Active Vertex Group
    CTRL + SHIFT - Remove from All Vertex Group
    ALT    - Select vertex

    """
    bl_idname = "object.sfc_vgroups"
    bl_label = "Code"
    bl_options = {"REGISTER"}

    @classmethod
    def poll(cls, context):
        return True

    bevel_from_solidify : StringProperty(default="")

    def invoke(self, context, event):
        self.obj = context.active_object

        if event.shift and event.ctrl:
            bpy.ops.object.vertex_group_remove_from(use_all_groups=True, use_all_verts=False)
        # Add selection to Vgroup
        elif event.shift:
            context.scene.tool_settings.vertex_group_weight = 1

            bpy.ops.object.vertex_group_assign()


        # Remove Vgroup
        elif event.ctrl:
            if context.object.mode == "OBJECT":
                for vgroup in self.obj.vertex_groups:
                    if vgroup.name == "Group":
                        bpy.ops.object.vertex_groups.remove(vgroup)
            elif context.object.mode == "EDIT":
                bpy.ops.object.vertex_group_remove_from()


        # Select
        elif event.alt:
            bpy.ops.mesh.select_all(action='DESELECT')
            bpy.ops.object.vertex_group_select()

        # Add modifier and Edit
        else:
            bpy.ops.object.vertex_group_add()
            context.scene.tool_settings.vertex_group_weight = 1

            bpy.ops.object.vertex_group_assign()

            # new_bevel = self.obj.modifiers.new("Bevel", "BEVEL")
            # new_bevel.width = 0.3
            # new_bevel.segments = 4
            # new_bevel.profile = 0.5
            # new_bevel.use_clamp_overlap = False
            # new_bevel.loop_slide = True
            # new_bevel.limit_method = 'VGROUP'
            # new_bevel.use_only_vertices = True
            #
            # vgroup_active = self.obj.vertex_groups.active.name
            # new_bevel.vertex_group = vgroup_active
            #
            # bpy.ops.object.mode_set(mode='OBJECT')
            #
            # bpy.ops.object.modifier_move_up(modifier=new_bevel.name)

            # for bevel_mod in self.obj.modifiers:
            #     if bevel_mod.type == 'BEVEL' and bevel_mod.limit_method == 'ANGLE':
            #         break
            #
            # bpy.ops.speedflow.bevel('INVOKE_DEFAULT', bevel=bevel_mod.name)


        return {"FINISHED"}



class SFC_OT_fast_rename(bpy.types.Operator):
    """    FAST RENAME

    CLICK - Rename selection from active object
    SHIFT - Show Popup to Rename all the selection"""
    bl_idname = 'object.sc_fast_rename'
    bl_label = "Fast Rename"
    bl_options = {'REGISTER'}

    @classmethod
    def poll(cls, context):
        return True

    empty_name : StringProperty()

    def check(self, context):
        return True

    def draw(self, context):
        layout = self.layout
        layout.prop(self, "empty_name", text="Name")

    """
    - Si on a un seul object           > on lance le popup pour renommer l'objet
    - si on a plusieurs objets         > on clic et ça les renomme par rapport à l'objet actif
    - Si on a plusieurs objets + shift > On lance le popup et on renomme tout
    - Si le nom existe                 > On ajoute un _1 et on continue
    """

    def Check_Name_Exist(self):
        if bpy.context.scene.objects.get(self.empty_name):
            self.empty_name = self.empty_name + "_1"

    #            print("Ca existe")

    def Code_Popup(self, selection, OB):  # Pour lancer le popup

        if len(selection) == 1:
            OB.name = self.empty_name

        else:
            for i, obj in enumerate(selection):
                if obj == OB:
                    obj.name = self.empty_name
                else:
                    obj.name = "{}_{}".format(self.empty_name, str(i).zfill(3))

    def Code_direct(self, selection, OB):  # Quand on clic direct sur le bouton

        for i, obj in enumerate(selection):
            if obj == OB:
                continue
            obj.name = "{}_{}".format(OB.name, str(i).zfill(3))

    def invoke(self, context, event):
        C = context
        self.OB = C.active_object
        self.selection = C.selected_objects

        if len(self.selection) == 1 or event.shift:
            dpi_value = context.preferences.view.ui_scale
            return context.window_manager.invoke_props_dialog(self, width=dpi_value * 400, height=100)

        self.Code_direct(self.selection, self.OB)
        return {'FINISHED'}

    def execute(self, context):
        self.Check_Name_Exist()

        self.Code_Popup(self.selection, self.OB)

        return {'FINISHED'}


class SFC_OT_parent_asset(bpy.types.Operator):
    """    Parent Asset

    CLICK - Popup for Options
    SHIFT - Parent to Active or Vertex/Edge/Face Selection
    CTRL  - Parent to Center
    ALT    - Parent to Cursor
    CTRL + Shift - Parent to Selection + To center of the Grid
    """
    bl_idname = "object.sc_parent_asset"
    bl_label = "Add Empty to Cursor"
    bl_options = {"REGISTER", "UNDO"}

    @classmethod
    def poll(cls, context):
        return len([obj for obj in context.selected_objects if context.object is not None]) > 1

    empty_pivot : EnumProperty(
        items=(('pivot_to_center', "Center", "Center of the Selection"),
               ('pivot_to_cursor', "Cursor", "Cursor"),
               ('pivot_to_selection', "Selection", "Active Object or Selected FaceVertex/Edge/Face")),
        default='pivot_to_cursor'
    )

    empty_center_object : BoolProperty(
        name="Center object",
        description="Place the object at the center of the scene",
        default=False)

    empty_scale : EnumProperty(
        items=(('1', "1", "1"),
               ('2', "2", "2"),
               ('3', "3", "3"),
               ('4', "4", "4"),
               ('5', "5", "5"),
               ('6', "6", "6")),
        default='2'
    )

    empty_name : StringProperty()

    selection_list : []

    active_object : StringProperty()

    bool_event : BoolProperty()

    def invoke(self, context, event):

        # if not event.shift or event.ctrl or event.alt :
        #
        #     if self.empty_pivot == 'pivot_to_selection':
        #         bpy.ops.view3d.snap_cursor_to_active()
        #     elif self.empty_pivot == 'pivot_to_center':
        #         bpy.ops.view3d.snap_cursor_to_selected()
        #
        #     if bpy.app.version == (2, 78, 0):
        #         dpi_value = context.preferences.system.dpi
        #         return context.window_manager.invoke_props_dialog(self, width=dpi_value * 400, height=100)
        #
        #     elif bpy.app.version >= (2, 79, 0):
        #         dpi_value = context.preferences.view.ui_scale
        #         return context.window_manager.invoke_props_dialog(self, width=dpi_value * 400, height=100)
        #
        #
        #
        # else:
        #     #To Selection and To Center of the scene
        #     if event.ctrl and event.shift:
        #         bpy.ops.view3d.snap_cursor_to_selected()
        #         self.empty_center_object = True
        #
        #     #To Selection
        #     elif event.shift:
        #         bpy.ops.view3d.snap_cursor_to_active()
        #
        #     #To Center
        #     elif event.ctrl:
        #         bpy.ops.view3d.snap_cursor_to_selected()
        #
        #     return self.execute(context)
        dpi_value = context.preferences.view.ui_scale
        return context.window_manager.invoke_props_dialog(self, width=dpi_value * 400, height=100)

    def draw(self, context):
        layout = self.layout
        layout.prop(self, "empty_name", text="Name")
        layout.prop(self, "empty_scale", text="Empty Scale")
        layout.prop(self, "empty_pivot", expand=True)
        layout.prop(self, "empty_center_object", text="Center The Object")

    # def invoke(self, context, event):
    def execute(self, context):
        obj = context.active_object

        # Store cursor location
        saved_location = context.scene.cursor.location.copy()

        if self.empty_pivot == 'pivot_to_selection':
            bpy.ops.view3d.snap_cursor_to_active()
        elif self.empty_pivot == 'pivot_to_center':
            bpy.ops.view3d.snap_cursor_to_selected()

        if context.object.mode == "EDIT":
            bpy.ops.object.mode_set(mode='OBJECT')
            bpy.ops.object.select_last_parent()

        # Create a list of the selection
        obj_list = context.selected_objects

        # Create empty and place pivot to cursor
        bpy.ops.object.empty_add(type='PLAIN_AXES', radius=int(self.empty_scale))

        # our new empty is now the active object, so we can keep track of it in a variable:
        empty_created = context.active_object
        empty_created.name = self.empty_name

        # Pivot to Center
        if self.empty_center_object == 'to_center':
            bpy.ops.object.location_clear(clear_delta=False)

        # Increment name
        if self.empty_name:
            context.active_object.name = self.empty_name

        # Parenting
        for obj in obj_list:
            obj.select_set(state=True)
            if not obj.parent:
                bpy.ops.object.parent_set(type='OBJECT', keep_transform=True)

        # Deselect the list
        for obj in obj_list:
            obj.select_set(state=False)

        # Center the object to the scene
        if self.empty_center_object:
            bpy.ops.object.location_clear(clear_delta=False)

        # Restore cursor location
        context.scene.cursor.location = saved_location
        return {"FINISHED"}

class SFC_OT_replace_mesh_data(bpy.types.Operator):
    bl_idname = 'object.sc_replace_mesh_data'
    bl_label = "Replace Mesh Data"
    bl_description = "Replace the Mesh Data by the Active object and make it single"
    bl_options = {'REGISTER'}

    @classmethod
    def poll(cls, context):
        return len([obj for obj in context.selected_objects if context.object is not None if context.object.mode == "OBJECT"]) > 1

    def execute(self, context):
        act_obj = context.active_object
        obj_list = [obj for obj in context.selected_objects if obj != act_obj]

        bpy.ops.object.select_all(action='DESELECT')

        print(obj_list)
        for obj in obj_list:
            obj.select_set(state=True)
            ob_matrix = context.object.matrix_world.copy()

            act_obj.select_set(state=True)
            context.view_layer.objects.active = act_obj
            bpy.ops.object.make_links_data(type='OBDATA')
            # bpy.ops.object.make_single_user(type='SELECTED_OBJECTS', object=True, obdata=True, material=False,
            #                                 texture=False, animation=False)
            bpy.ops.object.make_single_user(type='SELECTED_OBJECTS', object=True, obdata=True, material=True,
                                            animation=True)

            context.active_object.matrix_world = ob_matrix
            bpy.ops.object.select_all(action='DESELECT')

        return {'FINISHED'}

class SFC_OT_fast_intersect(bpy.types.Operator):
    """    FAST INTERSECT

    CLICK - Intersect and Delete Objects
    SHIFT - Intersect and keep Objects"""
    bl_idname = 'object.sc_fast_intersect'
    bl_label = "Fast Intersect"
    bl_options = {'REGISTER'}

    @classmethod
    def poll(cls, context):
        return len([obj for obj in context.selected_objects if context.object is not None if obj.type == 'MESH' if context.object.mode == "OBJECT"]) > 1

    def invoke(self, context, event):
        act_obj = context.active_object
        intersect_obj = [obj for obj in context.selected_objects if obj != act_obj]
        new_objs_list = []

        # Remove Vgroup
        for vgroup in act_obj.vertex_groups:
            if vgroup.name.startswith("Intersect_Group"):
                act_obj.vertex_groups.remove(vgroup)

        act_obj.select_set(state=False)


        for ob in intersect_obj:
            ob.select_set(state=True)
            context.view_layer.objects.active = ob
            bpy.ops.object.mode_set(mode='EDIT')
            bpy.ops.mesh.select_all(action='SELECT')
            bpy.ops.object.mode_set(mode='OBJECT')




        if event.shift:
            bpy.ops.object.duplicate()

        for new_obj in context.selected_objects:
            context.view_layer.objects.active = new_obj
            new_objs_list.append(new_obj)
            # print(new_objs_list)

        bpy.ops.object.select_all(action='DESELECT')

        act_obj.select_set(state=True)
        context.view_layer.objects.active = act_obj
        bpy.ops.object.mode_set(mode='EDIT')
        bpy.ops.mesh.select_all(action='DESELECT')
        bpy.ops.object.mode_set(mode='OBJECT')

        for obj in new_objs_list:
            obj.select_set(state=True)

        bpy.ops.object.join()
        bpy.ops.object.mode_set(mode='EDIT')

        #create vertex group and assign vertices
        vgroup = act_obj.vertex_groups.new(name="Intersect_Group")
        for vgroup in act_obj.vertex_groups:
            if vgroup.name == "Intersect_Group":
                bpy.ops.object.vertex_group_assign()

        bpy.ops.mesh.intersect()
        bpy.ops.mesh.select_all(action='DESELECT')


        for vgroup in act_obj.vertex_groups:
            if vgroup.name.startswith("Intersect_Group"):
                bpy.ops.object.vertex_group_select()

        bpy.ops.mesh.select_more()

        bpy.ops.mesh.delete(type='FACE')

        bpy.ops.object.mode_set(mode='OBJECT')

        # Remove Vgroup
        for vgroup in act_obj.vertex_groups:
            if vgroup.name.startswith("Intersect_Group"):
                act_obj.vertex_groups.remove(vgroup)
        return {'FINISHED'}

class SFC_OT_clean_mesh(bpy.types.Operator):
    """   Clean the Mesh

    CLICK - Clean the Mesh
    SHIFT - Clean Faces and remove Ngons
    """
    bl_idname = 'object.sfc_clean_mesh'
    bl_label = "Clean Meshes"
    bl_options = {'REGISTER', 'UNDO'}

    event_enum : EnumProperty(
        items=(('shift', "Shift", ""),
               ('ctrl', "Ctrl", ""),
               ('alt', "Alt", "")),
        default='ctrl'
    )

    @classmethod
    def poll(cls, context):
        return getattr(context.active_object, "type", "") == "MESH"

    dissolve_faces : FloatProperty(name="Limited Dissolve Angle", default=radians(0.05),
                                   min=radians(0), max=radians(30), subtype="ANGLE")

    remove_double : FloatProperty(name="Remove Double",
                                  description="Merge distance",
                                  min=0.0001,
                                  max=1.00,
                                  default=0.001)

    delete_interior_faces : BoolProperty(default=True, description="Delete Interior Faces")
    delete_isolate_vertices : BoolProperty(default=True, description="Remove Isolate Vertices")
    clean_faces : BoolProperty(default=False, description="Add Edges to remove Ngons")

    def Clean_Meshes(self):
        for obj in bpy.context.selected_objects:
            bpy.context.view_layer.objects.active = obj

            bpy.ops.object.mode_set(mode='EDIT')
            bpy.ops.mesh.reveal()

            # Clean isolated vertices
            if self.delete_isolate_vertices:
                bpy.ops.mesh.select_mode(use_extend=False, use_expand=False, type='VERT')
                bpy.ops.mesh.select_all(action='SELECT')
                bpy.ops.mesh.delete_loose(use_verts=True, use_edges=False, use_faces=False)

            # Clean Faces (Dissolve)
            bpy.ops.mesh.select_mode(use_extend=False, use_expand=False, type='FACE')
            bpy.ops.mesh.select_all(action='SELECT')
            bpy.ops.mesh.dissolve_limited(angle_limit=self.dissolve_faces)

            # Clean Interior Faces
            if self.delete_interior_faces:
                bpy.ops.mesh.select_all(action='DESELECT')
                bpy.ops.mesh.select_interior_faces()
                bpy.ops.mesh.delete(type='FACE')

            # Remove Double
            bpy.ops.mesh.remove_doubles(threshold=self.remove_double)

            # Clean Faces
            if self.clean_faces:
                bpy.ops.mesh.select_mode(use_extend=False, use_expand=False, type='FACE')
                bpy.ops.mesh.select_all(action='SELECT')
                bpy.ops.mesh.quads_convert_to_tris(quad_method='BEAUTY', ngon_method='BEAUTY')
                bpy.ops.mesh.tris_convert_to_quads(shape_threshold=2.18166)

            if self.mode == "EDIT":
                bpy.ops.object.mode_set(mode='EDIT')
            else:
                bpy.ops.object.mode_set(mode='OBJECT')


    def draw(self, context):
        layout = self.layout

        layout.prop(self, 'remove_double', text="Remove Double")
        layout.prop(self, 'dissolve_faces', text="Disolve Faces")
        layout.prop(self, 'delete_interior_faces', text="Delete Interior Faces")
        layout.prop(self, 'delete_isolate_vertices', text="Delete Isolate Vertices")
        layout.prop(self, 'clean_faces', text="Clean Faces")

    def execute(self, context):

        if self.event_enum == 'shift':
            self.clean_faces = True
            self.Clean_Meshes()
            self.clean_faces = False
        #
        # elif event.ctrl:
        #
        # elif event.alt:
        #
        else:
            self.Clean_Meshes()

        return {'FINISHED'}

    def invoke(self, context, event):
        self.mode = context.object.mode

        if event.shift:
            self.event_enum = 'shift'
        #
        #
        # elif event.ctrl:
        #     self.event_enum = 'ctrl'
        #
        # elif event.alt:
        #     self.event_enum = 'alt'
        #

        return self.execute(context)


# Show text
class SFC_OT_show_text_options(bpy.types.Operator):
    """   Show Text Option

    CLICK - Show/Hide Text
    SHIFT - Show Popup
    CTRL  - Simple Mode for Modifiers Text"""

    bl_idname = 'object.show_text_options'
    bl_label = "Show Text Options"
    bl_options = {'REGISTER', 'UNDO'}


    def invoke(self, context, event):
        # self.addon_prefs = get_addon_preferences()

        drawText = get_addon_preferences().drawText
        simple_text_mode = get_addon_preferences().simple_text_mode
        # self.simple_text_mode = self.addon_prefs.simple_text_mode
        # self.simple_text_mode = get_addon_preferences().simple_text_mode

        if event.shift:
            bpy.ops.object.sfc_text_options_popup('INVOKE_DEFAULT')

        elif event.ctrl:
            get_addon_preferences().simple_text_mode = not simple_text_mode


        else:
            get_addon_preferences().drawText  = not drawText
            bpy.context.area.tag_redraw()
        return {'FINISHED'}



# Solo
def Create_Transparent_Shader():
    addon_pref = get_addon_preferences()
    solo_color = addon_pref.solo_color
    solo_alpha = addon_pref.solo_alpha


    material = bpy.data.materials.new(name="Transparent_shader")
    material.use_nodes = True

    material_output = material.node_tree.nodes.get('Material Output')
    shader = material.node_tree.nodes['Diffuse BSDF']

    material.node_tree.links.new(material_output.inputs[0], shader.outputs[0])

    material.alpha = solo_alpha
    material.diffuse_color = solo_color

def Add_Solo(obj):
    addon_pref = get_addon_preferences()
    solo_alpha = addon_pref.solo_alpha

    bpy.context.space_data.viewport_shade = 'SOLID'

    sculpt = False
    if bpy.context.object.mode == "SCULPT":
        sculpt = True

    if bpy.context.object.mode != "OBJECT":
        bpy.ops.object.mode_set(mode='OBJECT')

    bpy.context.preferences.edit.use_duplicate_material = False
    bpy.context.space_data.show_backface_culling = True
    bpy.context.object.show_transparent = False

    # on recup la selection et l'objet actif
    act_obj_list = [obj for obj in bpy.context.selected_objects]

    # On selectionne les objets visible de la scene
    bpy.ops.object.select_grouped(type='LAYER')

    # On deselectionne les objets selectionnes
    for obj in act_obj_list:
        obj.select_set(state=False)

    # On cree une liste avec les objets restants
    obj_list = [obj for obj in bpy.context.selected_objects]

    # On check si le shader existe dans la scene
    if not bpy.data.materials.get('Transparent_shader'):
        Create_Transparent_Shader()

    # On sélectionne les objets de la liste et on leur attribue le transparent shader, la transparence et l'alpha
    for obj in obj_list:

        bpy.context.view_layer.objects.active = obj
        if hasattr(obj.data, "materials") and not obj.type == 'EMPTY_IMAGE':
            obj.show_transparent = True

            if obj.material_slots:
                mats = [mat.name for mat in obj.material_slots]
                if not 'Transparent_shader' in mats:

                    bpy.ops.object.material_slot_add()
                    obj.material_slots[-1].material = bpy.data.materials['Transparent_shader']
                    for f in obj.data.polygons:
                        f.material_index = 1
            else:
                obj.active_material = bpy.data.materials['Transparent_shader']

            obj.active_material.alpha = solo_alpha

    bpy.ops.object.mode_set(mode='OBJECT')
    bpy.ops.object.select_all(action='DESELECT')

    for obj in act_obj_list:
        bpy.context.view_layer.objects.active = obj
        obj.select_set(state=True)
        if hasattr(obj.data, "materials"):
            mats = [mat.name for mat in obj.material_slots]
            if 'Transparent_shader' in mats:
                obj.data.materials.pop(mats.index('Transparent_shader'), update_data=True)

    if sculpt:
        bpy.ops.object.mode_set(mode='SCULPT')

# Add Solo On Selection
def Add_Solo_On_Selection(obj):
    addon_pref = get_addon_preferences()
    solo_alpha = addon_pref.solo_alpha
    bpy.context.preferences.edit.use_duplicate_material = False

    for obj in bpy.context.selected_objects:
        bpy.context.view_layer.objects.active = obj
        if hasattr(obj.data, "materials") and not obj.type == 'EMPTY_IMAGE':
            obj.show_transparent = True
            # On check si le shader existe dans la scene
            if not bpy.data.materials.get('Transparent_shader'):
                bpy.context.space_data.show_backface_culling = True
                Create_Transparent_Shader()

            if obj.material_slots:
                mats = [mat.name for mat in obj.material_slots]
                if not 'Transparent_shader' in mats:

                    bpy.ops.object.material_slot_add()
                    obj.material_slots[1].material = bpy.data.materials['Transparent_shader']
                    for f in obj.data.polygons:
                        f.material_index = 1

            else:
                obj.active_material = bpy.data.materials['Transparent_shader']

            obj.active_material.alpha = solo_alpha

#Remove Transparent Shader
def Remove_Transparent_shader(obj):
    if hasattr(obj.data, "materials"):

        mats = [mat.name for mat in obj.material_slots]
        if 'Transparent_shader' in mats:

            obj.data.materials.pop(mats.index('Transparent_shader'), update_data=True)
            obj.show_transparent = False

        else:
            if obj.data.materials:
                obj.active_material.alpha = 1
            obj.show_transparent = False

#Remove_Solo
def Remove_Solo(obj):
    duplicate_material = bpy.context.preferences.edit.use_duplicate_material

    if bpy.data.materials.get('Transparent_shader'):
        bpy.context.space_data.show_backface_culling = False
        for obj in bpy.context.visible_objects:
            if hasattr(obj.data, "materials"):
                Remove_Transparent_shader(obj)

            obj.hide_select = False

        bpy.data.materials.remove(bpy.data.materials['Transparent_shader'], do_unlink=True)

    bpy.context.preferences.edit.use_duplicate_material = duplicate_material

#Remove Solo From Selection
def Remove_Solo_From_Selection(obj):
    duplicate_material = bpy.context.preferences.edit.use_duplicate_material
    if hasattr(obj.data, "materials"):
        Remove_Transparent_shader(obj)

    else:
        bpy.context.space_data.show_backface_culling = False
        for obj in bpy.context.visible_objects:
            if hasattr(obj.data, "materials"):
                Remove_Transparent_shader(obj)
        bpy.data.materials.remove(bpy.data.materials['Transparent_shader'], do_unlink=True)

    bpy.context.preferences.edit.use_duplicate_material = duplicate_material

# class SFC_OT_make_solo(bpy.types.Operator):
#     """    SOLO
#
#     CLICK - Solo the selection
#     SHIFT - Add Selection to Transparency
#     CTRL  - Remove Selection from Transparency
#     ALT    - UnSolo"""
#
#     bl_idname = 'object.sfc_make_solo'
#     bl_label = "Make Solo"
#     bl_options = {'REGISTER'}
#
#     def invoke(self, context, event):
#         obj = context.active_object
#
#         if context.object is not None and hasattr(context.object.data, "materials"):
#
#             if event.shift:
#                 Add_Solo_On_Selection(obj)
#
#             elif event.ctrl:
#                 Remove_Solo_From_Selection(obj)
#
#             elif event.alt:
#                 Remove_Solo(obj)
#
#             else:
#                 Add_Solo(obj)
#         return {'FINISHED'}
#
#
# class SFC_OT_make_solo_old(bpy.types.Operator):
#     """    SOLO
#
#     CLICK - Solo the selection
#     SHIFT - Add Selection to Transparency
#     CTRL  - Remove Selection from Transparency
#     ALT    - UnSolo"""
#
#     bl_idname = 'object.sfc_make_solo_old'
#     bl_label = "Make Solo"
#     bl_options = {'REGISTER'}
#
#     def __init__(self):
#         self.duplicate_material = bpy.context.preferences.edit.use_duplicate_material
#
#     def Remove_Transparent_shader(self, obj):
#         if hasattr(obj.data, "materials"):
#
#             mats = [mat.name for mat in obj.material_slots]
#             if 'Transparent_shader' in mats:
#
#                 obj.data.materials.pop(mats.index('Transparent_shader'), update_data=True)
#                 obj.show_transparent = False
#
#             else:
#                 if obj.data.materials:
#                     obj.active_material.alpha = 1
#                 obj.show_transparent = False
#
#     def Create_Transparent_Shader(self):
#         addon_pref = get_addon_preferences()
#         solo_color = addon_pref.solo_color
#         solo_alpha = addon_pref.solo_alpha
#
#         material = bpy.data.materials.new(name="Transparent_shader")
#         material.use_nodes = True
#
#         material_output = material.node_tree.nodes.get('Material Output')
#         shader = material.node_tree.nodes['Diffuse BSDF']
#
#         material.node_tree.links.new(material_output.inputs[0], shader.outputs[0])
#
#         material.alpha = solo_alpha
#         material.diffuse_color = solo_color
#
#     def invoke(self, context, event):
#         addon_pref = get_addon_preferences()
#         solo_alpha = addon_pref.solo_alpha
#
#         if context.object is not None and hasattr(context.object.data, "materials"):
#             if event.ctrl and event.alt:
#                 for obj in context.selected_objects:
#                     obj.hide_select = True
#
#             if event.shift:
#                 context.preferences.edit.use_duplicate_material = False
#
#                 for obj in context.selected_objects:
#                     context.view_layer.objects.active = obj
#                     if hasattr(obj.data, "materials") and not obj.type == 'EMPTY_IMAGE':
#                         obj.show_transparent = True
#                         # On check si le shader existe dans la scene
#                         if not bpy.data.materials.get('Transparent_shader'):
#                             context.space_data.show_backface_culling = True
#                             self.Create_Transparent_Shader()
#
#                         if obj.material_slots:
#                             mats = [mat.name for mat in obj.material_slots]
#                             if not 'Transparent_shader' in mats:
#
#                                 bpy.ops.object.material_slot_add()
#                                 obj.material_slots[1].material = bpy.data.materials['Transparent_shader']
#                                 for f in obj.data.polygons:
#                                     f.material_index = 1
#                         else:
#                             obj.active_material = bpy.data.materials['Transparent_shader']
#
#                         obj.active_material.alpha = solo_alpha
#
#             elif event.ctrl:
#                 # Pour les objets sélectionnés, on check si ils ont le transp shader et on clean
#                 if context.selected_objects:
#                     for obj in context.selected_objects:
#                         if hasattr(obj.data, "materials"):
#                             self.Remove_Transparent_shader(obj)
#
#                 else:
#                     context.space_data.show_backface_culling = False
#                     for obj in context.visible_objects:
#                         if hasattr(obj.data, "materials"):
#                             self.Remove_Transparent_shader(obj)
#                     bpy.data.materials.remove(bpy.data.materials['Transparent_shader'], do_unlink=True)
#
#                 context.preferences.edit.use_duplicate_material = self.duplicate_material
#                 # print(self.duplicate_material)
#
#             elif event.alt:
#                 if bpy.data.materials.get('Transparent_shader'):
#                     context.space_data.show_backface_culling = False
#                     for obj in context.visible_objects:
#                         if hasattr(obj.data, "materials"):
#                             self.Remove_Transparent_shader(obj)
#
#                         obj.hide_select = False
#
#                     bpy.data.materials.remove(bpy.data.materials['Transparent_shader'], do_unlink=True)
#
#                 context.preferences.edit.use_duplicate_material = self.duplicate_material
#                 # print(self.duplicate_material)
#
#             else:
#                 context.preferences.edit.use_duplicate_material = False
#                 context.space_data.show_backface_culling = True
#
#                 # on recup la selection et l'objet actif
#                 act_obj_list = [obj for obj in context.selected_objects]
#
#                 # On met l'objet/s actif transparent
#                 context.object.show_transparent = False
#
#                 # On selectionne les objets visible de la scene
#                 bpy.ops.object.select_grouped(type='LAYER')
#
#                 # On deselectionne les objets selectionnes
#                 for obj in act_obj_list:
#                     obj.select_set(state=False)
#
#                 # On cree une liste avec les objets restants
#                 obj_list = [obj for obj in context.selected_objects]
#
#                 # On check si le shader existe dans la scene
#                 if not bpy.data.materials.get('Transparent_shader'):
#                     self.Create_Transparent_Shader()
#
#                 # On sélectionne les objets de la liste et on leur attribue le transparent shader, la transparence et l'alphe
#                 for obj in obj_list:
#
#                     context.view_layer.objects.active = obj
#                     if hasattr(obj.data, "materials") and not obj.type == 'EMPTY_IMAGE':
#                         obj.show_transparent = True
#
#                         if obj.material_slots:
#                             mats = [mat.name for mat in obj.material_slots]
#                             if not 'Transparent_shader' in mats:
#
#                                 bpy.ops.object.material_slot_add()
#                                 obj.material_slots[1].material = bpy.data.materials['Transparent_shader']
#                                 for f in obj.data.polygons:
#                                     f.material_index = 1
#
#                         else:
#                             obj.active_material = bpy.data.materials['Transparent_shader']
#
#                         obj.active_material.alpha = solo_alpha
#
#                 bpy.ops.object.select_all(action='DESELECT')
#
#                 for obj in act_obj_list:
#                     context.view_layer.objects.active = obj
#                     obj.select_set(state=True)
#                     if hasattr(obj.data, "materials"):
#                         mats = [mat.name for mat in obj.material_slots]
#                         if 'Transparent_shader' in mats:
#                             obj.data.materials.pop(mats.index('Transparent_shader'), update_data=True)
#
#         return {'FINISHED'}


#Apply/Remove
def Apply_Remove_Lattive(self, context, event):
    act_obj = context.active_object
    selection = context.selected_objects

    for obj in context.selected_objects:
        bpy.ops.object.select_hierarchy(direction='PARENT', extend=False)
        context.object.hide_viewport = False

    lattice_selection = [obj for obj in context.selected_objects if obj.type == 'LATTICE']

    for obj in context.selected_objects:

        if obj.type == 'LATTICE':
            # Apply
            if event.shift:
                for children in obj.children:
                    context.view_layer.objects.active = children
                    for mod in children.modifiers:
                        if mod.type == 'LATTICE':
                            children.select_set(state=True)
                            bpy.ops.object.parent_clear(type='CLEAR_KEEP_TRANSFORM')
                            bpy.ops.object.modifier_apply(modifier=mod.name)
                            children.select_set(state=False)
                            clean = True

            # Remove
            elif event.ctrl:
                for children in obj.children:
                    context.view_layer.objects.active = children
                    for mod in children.modifiers:
                        if mod.type == 'DATA_TRANSFERT':
                            children.select_set(state=True)
                            bpy.ops.object.modifier_remove(modifier=mod.name)
                            children.select_set(state=False)
                            clean = True
                        if mod.type == 'SHRINKWRAP':
                            children.select_set(state=True)
                            bpy.ops.object.modifier_remove(modifier=mod.name)
                            children.select_set(state=False)
                            clean = True

                        if mod.type == 'LATTICE':
                            children.select_set(state=True)
                            bpy.ops.object.modifier_remove(modifier=mod.name)
                            children.select_set(state=False)
                            clean = True


            else:
                self.report({'INFO'}, "SHIFT or CTRL to make an action")
                bpy.ops.object.select_all(action='DESELECT')

                for obj in selection:
                    obj.select_set(state=True)

                    context.view_layer.objects.active = obj
                return {'FINISHED'}




#Show Hide booleans
class SFC_OT_show_bool_objects(bpy.types.Operator):
    """    SHOW/HIDE BOOL OBJECTS

    CLICK - Show Booleans Objects
    SHIFT - Show and Select Booleans Objects
    CTRL  - Hide booleans Objects
    ALT    - Put the Booleans Objects in Solid Mode
    CTRL + SHIFT - Show and Select Booleans Objects and Active Object"""
    bl_idname = 'object.sfc_show_bool_objects'
    bl_label = "Show/Hide Objetc Bool"
    bl_options = {'REGISTER'}

    def get_modifier_list(self, obj, mod_type):
        return [mod.name for mod in obj.modifiers if mod.type == mod_type]

    def display_boolean_object(self, bool_obj, hide_bool):
        """ Pass the object to visible if it wasn't in the viewport """

        # for bool_obj in bool_list:
        # if bool_obj.hide_viewport:
        #     bool_obj.hide_viewport = False
        # else:
        #     bool_obj.hide_viewport = True

        # active_layer = bpy.context.scene.active_layer
        if not hide_bool:
            # if not bool_obj.layers[active_layer]:
            #     bool_obj.layers[active_layer] = True

            if bool_obj.hide_viewport:
                bool_obj.hide_viewport = False
        else:
            bool_obj.hide_viewport = True
            # layers = [i for i in range(20) if bool_obj.layers[i]]
            # if len(layers) > 1:
            #     bool_obj.layers[active_layer] = False

    def invoke(self, context, event):
        act_obj = context.active_object
        bool_objects_sel = []

        for obj in context.selected_objects:
            bool_modifiers = self.get_modifier_list(obj, 'BOOLEAN')
            if bool_modifiers:
                for bool in bool_modifiers:
                    obj_bool = obj.modifiers[bool].object
                    self.display_boolean_object(obj_bool, event.ctrl == True)
                    if event.shift or event.alt :
                        bool_objects_sel.append(obj_bool)

        # Show and Select Booleans Objects and actobj
        if event.ctrl and event.shift:
            for obj in bool_objects_sel:
                obj.select_set(state=True)

        #Show and Select Booleans Objects
        if event.shift:
            act_obj.select_set(state=False)
            for obj in bool_objects_sel:
                context.view_layer.objects.active = obj
                obj.select_set(state=True)


        #Put the Booleans Objects in Solid Mode
        if event.alt:
            for obj in bool_objects_sel:
                obj.select_set(state=True)
                if obj.display_type == 'TEXTURED':
                    obj.cycles_visibility.diffuse = False
                    obj.cycles_visibility.glossy = False
                    obj.cycles_visibility.transmission = False
                    obj.cycles_visibility.scatter = False
                    obj.cycles_visibility.shadow = False
                    obj.display_type = 'WIRE'

                else:
                    obj.cycles_visibility.diffuse = True
                    obj.cycles_visibility.glossy = True
                    obj.cycles_visibility.transmission = True
                    obj.cycles_visibility.scatter = True
                    obj.cycles_visibility.shadow = True
                    obj.display_type = 'TEXTURED'

            bpy.ops.object.select_all(action='DESELECT')
            act_obj = context.active_object
            act_obj.select_set(state=True)


        del (bool_objects_sel[:])


        return {'FINISHED'}


 # select_hierarchy
def find_child(ob, lst):
    for x in ob.children:
        lst.append(x)
        find_child(x, lst)


def find_parent(ob, lst):
    lst.append(ob)
    if ob.parent:
        find_parent(ob.parent, lst)


class SFC_OT_select_hierarchy(bpy.types.Operator):
    bl_idname = "object.sfc_select_hierarchy"
    bl_label = "select hierarchy"
    bl_options = {"UNDO"}

    def execute(self, context):
        lst = []  # list for all objects
        find_parent(context.object, lst)  # find parent

        if len(lst) > 0:

            last_parent = lst[0]
            for p in lst:
                # parent of everything has no parent itself
                if not p.parent: last_parent = p

            # find children for each parent
            for c in lst: find_child(c, lst)

            # select all
            for x in lst:
                x.select_set(state=True)

            # make parent active
            context.view_layer.objects.active = last_parent

        # print(lst)
        context.scene.tool_settings.transform_pivot_point = 'ACTIVE_ELEMENT'

        # context.space_data.pivot_point = 'ACTIVE_ELEMENT'
        return {'FINISHED'}

#Apply Projection
def Apply_Projection(self, context):
    lattice = context.active_object

    for obj in lattice.children:
        context.view_layer.objects.active = obj
        if obj.modifiers:
            if "Shrinkwrap" in obj.modifiers:
                bpy.ops.object.modifier_remove(modifier="Shrinkwrap")

            if "Lattice" in obj.modifiers:
                obj.select_set(state=True)
                bpy.ops.object.parent_clear(type='CLEAR_KEEP_TRANSFORM')
                bpy.ops.object.modifier_apply(apply_as='DATA', modifier="Lattice")
                obj.select_set(state=False)
                bpy.data.objects.remove(lattice, do_unlink=True)


    context.scene.tool_settings.use_snap_align_rotation = False
    context.scene.tool_settings.use_snap = False


#Clean Projection
def Clean_Projection(self, context):
    bpy.ops.object.select_hierarchy(direction='PARENT', extend=False)
    context.object.hide_viewport = False

    lattice = context.active_object



    for obj in lattice.children:
        context.view_layer.objects.active = obj
        if obj.modifiers:
            if "Lattice" in obj.modifiers:
                if "Shrinkwrap" in obj.modifiers:
                    bpy.ops.object.modifier_remove(modifier="Shrinkwrap")
                if "DataTransfer" in obj.modifiers:
                    bpy.ops.object.modifier_remove(modifier="DataTransfer")

                bpy.ops.object.modifier_remove(modifier="Lattice")
                bpy.data.objects.remove(lattice, do_unlink=True)

    context.scene.tool_settings.use_snap_align_rotation = False
    context.scene.tool_settings.use_snap = False

#Create Projection
def Create_Projection(self, context):


    obj = context.active_object
    
    self.act_obj = context.active_object
    sel_obj = context.selected_objects[0] if context.selected_objects[0] != self.act_obj else context.selected_objects[1]

    #Lattice Creation
    bpy.ops.object.add(type='LATTICE', location = self.act_obj.location)
    context.object.data.points_u = 10
    context.object.data.points_v = 10
    context.object.data.points_w = 1
    lattice_object = context.active_object
    lattice_object.dimensions = self.act_obj.dimensions * 1.2
 
    new_lattice = self.act_obj.modifiers.new(name = 'Lattice', type = 'LATTICE')
    new_lattice.object = lattice_object
    new_lattice.show_expanded = False

    # If Vgroup
    for vgroup in obj.vertex_groups:
        if vgroup.name == "Group":
            new_shrinkwarp = self.act_obj.modifiers.new(name='Shrinkwrap', type='SHRINKWRAP')
            new_shrinkwarp.target = sel_obj
            new_shrinkwarp.vertex_group = "Group"
            new_shrinkwarp.show_expanded = False

            new_data_transfert = self.act_obj.modifiers.new(name='DataTransfer', type='DATA_TRANSFER')
            new_data_transfert.object = sel_obj
            new_data_transfert.vertex_group = "Group"
            new_data_transfert.use_loop_data = True
            new_data_transfert.data_types_loops = {'CUSTOM_NORMAL'}
            new_data_transfert.show_expanded = False

    self.act_obj.select_set(state=True)
    bpy.ops.object.parent_set(type='OBJECT', keep_transform=True)
    self.act_obj.select_set(state=False)

    new_shrinkwarp = lattice_object.modifiers.new(name = 'Shrinkwrap', type = 'SHRINKWRAP')
    new_shrinkwarp.target = sel_obj



    context.window_manager.modal_handler_add(self)
    # self.report({'INFO'}, "Projection Started")
    bpy.ops.transform.translate('INVOKE_DEFAULT', True) 
    
def get_last_parent():
    last_parent = []
 
    def get_parent(obj):
        if obj.parent:
            if not obj.parent in last_parent:
                get_parent(obj.parent)
 
        else:
            if obj.children:
                last_parent.append(obj)
 
    for obj in bpy.context.selected_objects:
        get_parent(obj)
 
    return last_parent  
 
class SFC_OT_project_modal(bpy.types.Operator):
    """    PROJECTION
    
    SELECT SURFACE AND OBJECT TO PROJECT
    G - MOVE THE OBJECT ON THE SURFACE
    R - ROTATE 
    S - SCALE
    D - DUPLICATE PROJECTION
    
    IF SELECTION
    CLICK - ENTER IN THE MODAL TO EDIT
    SHIFT - APPLY PROJECTION
    CTRL  - REMOVE PROJECTION"""
    bl_idname = "object.sfc_project_modal"
    bl_label = "Simple Modal Operator"

    def modal(self, context, event): 
        context.area.header_text_set("G: Move the Object on the Surface, R: Rotate, S: Scale, A: Apply, D: Duplicate Object")


        #Move
        if event.type == 'G' and event.value == 'PRESS':
            context.scene.tool_settings.use_snap = True
            context.scene.tool_settings.use_snap_align_rotation = True
            bpy.ops.transform.translate('INVOKE_DEFAULT', True)
        
        #Rotate
        if event.type == 'R' and event.value == 'PRESS':
            context.scene.tool_settings.use_snap = False
            # bpy.ops.transform.rotate('INVOKE_DEFAULT', True, constraint_axis=(False, False, True), constraint_orientation='LOCAL')

            bpy.ops.transform.rotate('INVOKE_DEFAULT', orient_axis='Z', orient_type='LOCAL',
                                     orient_matrix_type='LOCAL', constraint_axis=(False, False, True))

        #Scale
        if event.type == 'S' and event.value == 'PRESS':
            context.scene.tool_settings.use_snap = False
            bpy.ops.transform.resize('INVOKE_DEFAULT', True)

 
        # Apply
        if event.type == 'A' and event.value == 'PRESS':
            Apply_Projection(self, context)
            context.scene.tool_settings.use_snap_align_rotation = False
            context.scene.tool_settings.use_snap = False
            # self.report({'INFO'}, "Projection Applied")
            context.area.header_text_set(text='Applied')
            return {'FINISHED'}

        # Duplicate
        if event.type == 'D' and event.value == 'PRESS':
            act_obj = context.active_object
            bpy.ops.object.sfc_select_hierarchy()
            bpy.ops.object.duplicate_move()
            bpy.ops.object.select_hierarchy(direction='PARENT', extend=False)
            bpy.data.objects[act_obj.name].hide_viewport = True


            for obj in context.selected_objects:
                context.view_layer.objects.active = obj

            context.scene.tool_settings.use_snap = True
            context.scene.tool_settings.use_snap_align_rotation = True
            bpy.ops.transform.translate('INVOKE_DEFAULT', True)

            # self.report({'INFO'}, "Projection Duplicated")
            context.area.header_text_set(text='Duplicated')


        # Accept
        if event.type in {'RIGHTMOUSE','SPACE'}:
            obj = context.active_object
            bpy.data.objects[obj.name].hide_viewport = True
            context.scene.tool_settings.use_snap_align_rotation = False
            context.scene.tool_settings.use_snap = False
            # self.report({'INFO'}, "Projection Exited")
            context.area.header_text_set(text='toto')
            return {'FINISHED'}
            
        if event.type in {'MIDDLEMOUSE', 'WHEELUPMOUSE', 'WHEELDOWNMOUSE'}:
            return {'PASS_THROUGH'}
 
        #Remove
        if event.type in {'DEL', 'BACK_SPACE'} and event.value == 'PRESS':
            Clean_Projection(self, context)
            # self.report({'INFO'}, "Projection Removed")
            context.area.header_text_set(text='toto')
            return {'FINISHED'}
        
        # Cancel
        elif event.type == 'ESC':
            Clean_Projection(self, context)
            # self.report({'INFO'}, "Projection Canceled")
            context.area.header_text_set(text='toto')

            return {'CANCELLED'}
 
        return {'RUNNING_MODAL'}


#---INVOKE 
    def invoke(self, context, event):
#        obj = context.active_object
        act_obj = context.active_object
        selection = context.selected_objects
        
        #Snapping Activation

        context.scene.tool_settings.use_snap = True
        context.scene.tool_settings.snap_elements = {'FACE'}
        context.scene.tool_settings.snap_target = 'MEDIAN'
        context.scene.tool_settings.use_snap_align_rotation = True


        if context.object:
#-----------Creation
            if len(context.selected_objects) >= 2 :
                clean = False

                #Apply Or Remove Projection from Selection
#                parent_list = get_last_parent()
#                for obj in context.scene.objects:
#                    if obj not in parent_list:
#                        obj.select = False
#                    else :
#                        obj.select_set(state=True)
#
#                if parent_list:
#                    final_selection = parent_list
#                    context.view_layer.objects.active = parent_list[0]
#                else:
#                    final_selection = selection



                for obj in context.selected_objects:
                    bpy.ops.object.select_hierarchy(direction='PARENT', extend=False)
                    context.object.hide_viewport = False

                lattice_selection = [obj for obj in context.selected_objects if obj.type == 'LATTICE']


                for obj in context.selected_objects :

                    if obj.type == 'LATTICE':
                        #Apply
                        if event.shift:
                            for children in obj.children:
                                context.view_layer.objects.active = children
                                for mod in children.modifiers :
                                    if mod.type == 'LATTICE':
                                        children.select_set(state=True)
                                        bpy.ops.object.parent_clear(type='CLEAR_KEEP_TRANSFORM')
                                        bpy.ops.object.modifier_apply(modifier=mod.name)
                                        children.select_set(state=False)
                                        clean = True

                        #Remove
                        elif event.ctrl:
                            for children in obj.children:
                                context.view_layer.objects.active = children
                                for mod in children.modifiers :
                                    if mod.type == 'DATA_TRANSFERT':
                                        children.select_set(state=True)
                                        bpy.ops.object.modifier_remove(modifier=mod.name)
                                        children.select_set(state=False)
                                        clean = True
                                    if mod.type == 'SHRINKWRAP':
                                        children.select_set(state=True)
                                        bpy.ops.object.modifier_remove(modifier=mod.name)
                                        children.select_set(state=False)
                                        clean = True

                                    if mod.type == 'LATTICE':
                                        children.select_set(state=True)
                                        bpy.ops.object.modifier_remove(modifier=mod.name)
                                        children.select_set(state=False)
                                        clean = True


                        else:
#                            clean = False
#                             self.report({'INFO'}, "SHIFT or CTRL to make an action")
                            bpy.ops.object.select_all(action='DESELECT')

                            for obj in selection:
                                obj.select_set(state=True)

                            context.view_layer.objects.active = act_obj
                            return {'FINISHED'}


                    #Create Projection
                    elif  obj.type != 'LATTICE' :
                        if len(context.selected_objects) == 2 :
                            Create_Projection(self, context)

                if clean :
                    #Remove Lattices
                    for lattice in lattice_selection:
                        bpy.data.objects.remove(lattice, do_unlink = True)


                    del(lattice_selection[:])
                context.scene.tool_settings.use_snap = False


            
#-----------If Lattices / Apply And Remove
#            elif len(context.selected_objects) >= 2 and obj.type == 'LATTICE':  
#            elif len(context.selected_objects) >= 2 :    
#                obj = context.active_object
#                for obj in context.selected_objects:
#                    bpy.ops.object.select_hierarchy(direction='PARENT', extend=False)

#                    
#                    lattice = context.active_object
#                    
#                    for lattice in context.selected_objects:
#                        bpy.ops.object.select_all(action='DESELECT')
#                        lattice.select_set(state=True)
#                        context.view_layer.objects.active=lattice
#                        
#                        #Apply
#                        if event.shift:
#                            for obj in lattice.children:
#                                context.view_layer.objects.active = obj
#                                if obj.modifiers:
#                                    if "Lattice" in obj.modifiers:
#                                        obj.select_set(state=True)
#                                        bpy.ops.object.parent_clear(type='CLEAR_KEEP_TRANSFORM')
#                                        bpy.ops.object.modifier_apply(apply_as='DATA', modifier="Lattice")
#                                        bpy.ops.object.select_all(action='DESELECT')
#                                 
#                        #Remove
#                        elif event.ctrl:
#                            for obj in lattice.children:
#                                context.view_layer.objects.active = obj
#                                if obj.modifiers:
#                                    if "Lattice" in obj.modifiers:
#                                        bpy.ops.object.modifier_remove(modifier="Lattice")
#                    
#                        bpy.data.objects.remove(lattice, do_unlink = True)  
#                        context.scene.tool_settings.use_snap = False
    
#-----------Edition
            elif len(context.selected_objects) == 1:
                bpy.ops.object.select_hierarchy(direction='PARENT', extend=False)
                context.object.hide_viewport = False


                lattice = context.active_object
 
                for obj in lattice.children:
                    context.view_layer.objects.active = obj
                    if obj.modifiers:
                        if "Lattice" in obj.modifiers:
                            #Apply
                            if event.shift:
                                obj.select_set(state=True)
                                bpy.ops.object.parent_clear(type='CLEAR_KEEP_TRANSFORM')
                                bpy.ops.object.modifier_apply(apply_as='DATA', modifier="Lattice")
                                obj.select_set(state=False)
                                bpy.data.objects.remove(lattice, do_unlink = True) 
                                # self.report({'INFO'}, "Projection Applied")
                                return {'FINISHED'} 
                                
                            #Remove
                            elif event.ctrl:

                                if "Shrinkwrap" in obj.modifiers:
                                    bpy.ops.object.modifier_remove(modifier="Shrinkwrap")
                                if "DataTransfer" in obj.modifiers:
                                    bpy.ops.object.modifier_remove(modifier="DataTransfer")

                                bpy.ops.object.modifier_remove(modifier="Lattice")
                                bpy.data.objects.remove(lattice, do_unlink = True)

                                # self.report({'INFO'}, "Projection Removed")
                                context.scene.tool_settings.use_snap_align_rotation = False
                                context.scene.tool_settings.use_snap = False
                                return {'FINISHED'}
 
                            else:

                                context.view_layer.objects.active = lattice
                                lattice.hide = False

                                context.window_manager.modal_handler_add(self)
                                # self.report({'INFO'}, "Projection Started")
                                context.area.header_text_set("G: Move the Object on the Surface, R: Rotate, S: Scale, A: Apply, D: Duplicate Object")

            return {'RUNNING_MODAL'}
 
        else:
            self.report({'WARNING'}, "No active object, could not finish")
            return {'CANCELLED'}
        

class SFC_OT_clean_datas(Operator):
    """ CLEAN ALL UNUSED DATAS """
    bl_idname = "object.sfc_clean_unused_data"
    bl_label = "Clean Datas"
    bl_options = {'REGISTER', 'UNDO'}
    
    def execute(self, context):
#        AM = context.window_manager.asset_m
        
        persistent_datas = [obj for obj in bpy.data.objects if obj.name not in context.scene.objects]
        
        if persistent_datas:
            selection = [obj for obj in context.selected_objects]
            
            bpy.ops.object.select_all(action = 'DESELECT')
             
            for obj in persistent_datas:
                context.scene.objects.link(obj)
                obj.select_set(state=True)
            
            bpy.ops.object.delete()
            
            for obj in selection:
                obj.select_set(state=True)
    
        for images in bpy.data.images:
            if not images.use_fake_user:
                if images.name != 'Render Result':
                    if not images.packed_file:
                        if not isfile(bpy.path.abspath(images.filepath)):
                            images.user_clear()
                            bpy.data.images.remove(images)
         
        data_list = ['objects', 'curves',
                    'texts', 'metaballs',
                    'lamps', 'cameras',
                    'fonts', 'lattices',
                    'meshes', 'groups',
                    'armatures', 'materials',
                    'textures', 'images',
                    'node_groups', 'grease_pencil',
                    'actions', 'libraries',
                    'movieclips', 'sounds', 
                    'speakers'
                    ] 
         
        for data in data_list:
            target_coll = eval("bpy.data." + data)
            for item in target_coll:
                if item.name != 'Render Result':
                    if item and item.users == 0:
                            target_coll.remove(item)
        
        return {'FINISHED'} 

#====================#
#  Display Mode
#====================#

class SFC_OT_display_mode(bpy.types.Operator):
    """    DISPLAY MODE
    
    CLICK - Wire
    SHIFT - Textured
    CTRL  - Bounds
    ALT    - Solid"""
    bl_idname = "object.sfc_display_mode"
    bl_label = "Sfc Display Mode"
    bl_options = {"REGISTER", "UNDO"}



    def invoke(self, context, event):
        act_obj = context.active_object
        obj = context.active_object
        
        if event.shift:
            for obj in context.selected_objects:
                context.view_layer.objects.active = obj
                obj.select_set(state=True)
                obj.display_type = 'TEXTURED'
                obj.cycles_visibility.camera = True
                obj.cycles_visibility.shadow = True
                obj.cycles_visibility.transmission = True
                obj.cycles_visibility.scatter = True
                obj.cycles_visibility.diffuse = True
                obj.cycles_visibility.glossy = True
                # obj.hide_viewport_render = True

            # self.report({'INFO'}, "Display Textured Mode")
            
        elif event.ctrl:
            for obj in context.selected_objects:
                context.view_layer.objects.active = obj
                obj.select_set(state=True)
                obj.display_type = 'BOUNDS'
                obj.cycles_visibility.camera = False
                obj.cycles_visibility.shadow = False
                obj.cycles_visibility.transmission = False
                obj.cycles_visibility.scatter = False
                obj.cycles_visibility.diffuse = False
                obj.cycles_visibility.glossy = False
                obj.hide_render = False
            # self.report({'INFO'}, "Display Bounds Mode")
        
        elif event.alt:  
            for obj in context.selected_objects:
                context.view_layer.objects.active = obj
                obj.select_set(state=True)
                obj.display_type = 'SOLID'
                obj.cycles_visibility.camera = True
                obj.cycles_visibility.shadow = True
                obj.cycles_visibility.transmission = True
                obj.cycles_visibility.scatter = True
                obj.cycles_visibility.diffuse = True
                obj.cycles_visibility.glossy = True
                obj.hide_render = True
            # self.report({'INFO'}, "Display Solid Mode")
        
        else:      
            for obj in context.selected_objects:
                context.view_layer.objects.active = obj
                obj.select_set(state=True)
                obj.display_type = 'WIRE'
                obj.cycles_visibility.camera = False
                obj.cycles_visibility.shadow = False
                obj.cycles_visibility.transmission = False
                obj.cycles_visibility.scatter = False
                obj.cycles_visibility.diffuse = False
                obj.cycles_visibility.glossy = False
                obj.hide_render = False
            # self.report({'INFO'}, "Display Wire Mode")

        for obj in context.selected_objects:
            context.view_layer.objects.active = act_obj
            act_obj.select_set(state=True)
                
        return {"FINISHED"}

class SFC_OT_wire_mode(bpy.types.Operator):
    bl_idname = "object.sfc_wire_mode"
    bl_label = "Sfc Wire Display Mode"
    bl_description = "WIRE DISPLAY MODE"
    bl_options = {"REGISTER", "UNDO"}

    def execute(self, context):
        act_obj = context.active_object
        obj = context.active_object
        
        for obj in context.selected_objects:
            context.view_layer.objects.active = obj
            obj.select_set(state=True)
            context.object.display_type = 'WIRE'
        
        # self.report({'INFO'}, "Display Wire Mode")

        for obj in context.selected_objects:
            context.view_layer.objects.active = act_obj
            act_obj.select_set(state=True)
                
        return {"FINISHED"}

#solid
class SFC_OT_solid_mode(bpy.types.Operator):
    bl_idname = "object.sfc_solid_mode"
    bl_label = "Sfc Solid Display Mode"
    bl_description = "SOLID  DISPLAY MODE"
    bl_options = {"REGISTER", "UNDO"}

    def execute(self, context):
        act_obj = context.active_object
        obj = context.active_object
        
        for obj in context.selected_objects:
            context.view_layer.objects.active = obj
            obj.select_set(state=True)
            context.object.display_type = 'SOLID'
        
        # self.report({'INFO'}, "Display Solid Mode")
        
        for obj in context.selected_objects:
            context.view_layer.objects.active = act_obj
            act_obj.select_set(state=True)
        return {"FINISHED"}
#Bounds
class SFC_OT_bounds_mode(bpy.types.Operator):
    bl_idname = "object.sfc_bounds_mode"
    bl_label = "Sfc Bounds Display Mode"
    bl_description = "BOUND DISPLAY MODE"
    bl_options = {"REGISTER", "UNDO"}

    def execute(self, context):
        act_obj = context.active_object
        obj = context.active_object
        
        for obj in context.selected_objects:
            context.view_layer.objects.active = obj
            obj.select_set(state=True)
            context.object.display_type = 'BOUNDS'
        
        # self.report({'INFO'}, "Display Bounds Mode")
        
        for obj in context.selected_objects:
            context.view_layer.objects.active = act_obj
            act_obj.select_set(state=True)
        return {"FINISHED"}
 
#====================#
#  Materials
#====================#

# class SFC_OT_apply_material(bpy.types.Operator):
#     bl_idname = "object.sfc_apply_material"
#     bl_label = "Apply material"
#
#     mat_to_assign : bpy.props.StringProperty(default="")
#
#     def execute(self, context):
#
#
#
#         if context.object.mode == 'EDIT':
#             obj = context.object
#             bm = bmesh.from_edit_mesh(obj.data)
#
#             selected_face = [f for f in bm.faces if f.select]  # si des faces sont sélectionnées, elles sont stockées dans la liste "selected_faces"
#
#             mat_name = [mat.name for mat in context.object.material_slots if len(context.object.material_slots)] # pour tout les material_slots, on stock les noms des mat de chaque slots dans la liste "mat_name"
#
#             if self.mat_to_assign in mat_name: # on test si le nom du mat sélectionné dans le menu est présent dans la liste "mat_name" (donc, si un des slots possède le materiau du même nom). Si oui:
#                 context.object.active_material_index = mat_name.index(self.mat_to_assign) # on definit le slot portant le nom du comme comme étant le slot actif
#                 bpy.ops.object.material_slot_assign() # on assigne le matériau à la sélection
#             else: # sinon
#                 bpy.ops.object.material_slot_add() # on ajout un slot
#                 context.object.active_material = bpy.data.materials[self.mat_to_assign] # on lui assigne le materiau choisi
#                 bpy.ops.object.material_slot_assign() # on assigne le matériau à la sélection
#
#             return {'FINISHED'}
#
#         elif context.object.mode == 'OBJECT':
#
#             obj_list = [obj.name for obj in context.selected_objects]
#
#             for obj in obj_list:
#                 bpy.ops.object.select_all(action='DESELECT')
#                 bpy.data.objects[obj].select_set(state=True)
#                 # bpy.data.objects[obj].select = True
#                 context.view_layer.objects.active = bpy.data.objects[obj]
#                 context.object.active_material_index = 0
#
#                 if self.mat_to_assign == bpy.data.materials:
#                     context.active_object.active_material = bpy.data.materials[mat_name]
#
#                 else:
#                     if not len(context.object.material_slots):
#                         bpy.ops.object.material_slot_add()
#
#                     context.active_object.active_material = bpy.data.materials[self.mat_to_assign]
#
#             for obj in obj_list:
#                 # bpy.data.objects[obj].select = True
#                 bpy.data.objects[obj].select_set(state=True)
#
#             return {'FINISHED'}
    
#SFC Material List
# class SFC_OT_material_list_menu(bpy.types.Operator):
#     bl_idname = "object.sfc_material_list_menu"
#     bl_label = "Material List"
#     bl_options = {"REGISTER"}
#
#     sortElement : bpy.props.StringProperty(
#         default = ""
#         )
#
#     @classmethod
#     def poll(cls, context):
#         return context.object is not None and context.selected_objects and len(bpy.data.materials)
#
#     def check(self, context):
#         return True
#
#     def execute(self, context):
#         return {"FINISHED"}
#
#     def invoke(self, context, event):
#         if self.sortElement:
#             self.materialList = [mat for mat in bpy.data.materials if mat.name.lower().startswith(self.sortElement.lower())]
#         else:
#             self.materialList = [mat for mat in bpy.data.materials]
#         self.materialCount = len(self.materialList)
#         self.maxMaterial = 10
#         self.maxColumns = 6
#         self.columns = 1
#         self.start = 0
#         self.steps = 0
#         self.end = 0
#
#         # Calcul du nombre de colonne en arrondissant a l'entier superieur (fonction ceil du module math)
#         # pour une quantite de ligne maximum de 10 par colonne
#         self.colCount = ceil(self.materialCount / self.maxMaterial)
#
#         # Si le nombre de colonne trouve est superieur au nombre maximum de colonne "autorise",
#         # on limite le nombre de colonne au miximum de colonne autorise
#         # Repartition de la quantite de datas de facon homogene dans les colonnes
#         # Definition du premier stop pour changer de colonne
#         if self.colCount > self.maxColumns:
#             self.columns = self.maxColumns
#             self.steps = ceil(self.materialCount / self.maxColumns)
#             self.end = self.steps
#
#         # Sinon, le nombre de colonne sera le nombre trouve lors du calcul
#         else:
#             self.columns = self.colCount
#             self.steps = ceil(self.materialCount / self.colCount)
#             self.end = self.steps
#
#         dpi_value = context.preferences.system.dpi
#         return context.window_manager.invoke_props_dialog(self, width = dpi_value * 3 * self.columns, height = 100)
#
#     def draw(self, context):
#         layout = self.layout
#         layout.prop(self, "sortElement", text = "Sort Element")
#         row = layout.row()
#
#         if self.sortElement:
#             col = row.column()
#             box = col.box()
#             for mat in self.materialList:
#                 name = mat.name
#                 try:
#                     icon_val = layout.icon(mat)
#                 except:
#                     icon_val = 1
#                     print ("WARNING [Mat Panel]: Could not get icon value for %s" % name)
#
#                 if mat.name.lower().startswith(self.sortElement.lower()):
#                     box.operator("object.sfc_apply_material", text=name, icon_value=icon_val).mat_to_assign = name
#
#             # on doit reinitialiser self.steps, self.start et self.end
#             if self.colCount > self.maxColumns:
#                 self.steps = ceil(self.materialCount / self.maxColumns)
#
#             else:
#                 self.steps = ceil(self.materialCount / self.colCount)
#
#             self.start = 0
#             self.end = self.steps
#
#         else:
#             for i in range(self.columns):
#                 col = row.column()
#                 box = col.box()
#                 for idx in range(self.start, self.end):
#                     mat = self.materialList[idx]
#                     name = mat.name
#                     try:
#                         icon_val = layout.icon(mat)
#                     except:
#                         icon_val = 1
#                         print ("WARNING [Mat Panel]: Could not get icon value for %s" % name)
#
#                     box.operator("object.sfc_apply_material", text=name, icon_value=icon_val).mat_to_assign = name
#
#                 self.start += self.steps
#                 if self.end + self.steps <= self.materialCount:
#                     self.end += self.steps
#                 else:
#                     self.end = self.materialCount
#
#             # on doit reinitialiser self.steps, self.start et self.end
#             if self.colCount > self.maxColumns:
#                 self.steps = ceil(self.materialCount / self.maxColumns)
#
#             else:
#                 self.steps = ceil(self.materialCount / self.colCount)
#
#             self.start = 0
#             self.end = self.steps
#
# def Set_Material_Color(self, context):
#     selection = context.selected_objects
#     WM = context.window_manager
#     alpha = WM.material_alpha
#
#     for obj in selection:
#         obj.select_set(state=True)
#         context.view_layer.objects.active = obj
#
#
#
#         if len(obj.material_slots):
#             context.object.active_material.diffuse_color = WM.material_color
#             # if material.node_tree.nodes['Principled BSDF']:
#             context.object.active_material.node_tree.nodes['Principled BSDF'].inputs[0].default_value = (
#             WM.material_color[0], WM.material_color[1], WM.material_color[2], 1)
#             # elif
#             #
#             #     context.object.active_material.node_tree.nodes['Diffuse BSDF'].inputs[0].default_value = (
#             #     diffuse_color[0], diffuse_color[1], diffuse_color[2], 1)
#             context.object.active_material.alpha = alpha
#
#
# def Set_Material_Alpha(self, context):
#     selection = context.selected_objects
#     WM = context.window_manager
#     alpha = WM.material_alpha
#
#     for obj in selection:
#         obj.select_set(state=True)
#         context.view_layer.objects.active = obj
#         if len(obj.material_slots):
#             obj.active_material.alpha = alpha
#             context.object.show_transparent = True
#
# def material_presets():
#     obj = bpy.context.active_object
#     WM = bpy.context.window_manager
#     material = obj.active_material
#
#     shader = material.node_tree.nodes['Principled BSDF']
#
#     if WM.shader_type == 'metal':
#         # Metallic
#         shader.inputs[4].default_value = 1
#         # Roughness
#         shader.inputs[7].default_value = 0.2
#
#     if WM.shader_type == 'plastic':
#         # Metallic
#         shader.inputs[4].default_value = 0
#         # Specular
#         shader.inputs[5].default_value = 0.5
#         # Roughness
#         shader.inputs[7].default_value = 0.2
#
#     if WM.shader_type == 'glass':
#         # Metallic
#         shader.inputs[4].default_value = 0
#         # Specular
#         shader.inputs[5].default_value = 1
#         # Roughness
#         shader.inputs[7].default_value = 0
#         # Transmittance
#         shader.inputs[15].default_value = 1
#
#     WM.random_or_color = 'random'
#     WM.material_alpha = 1
# def set_material_color_or_random(self, context):
#     WM = context.window_manager
#     if WM.random_or_color == 'color':
#         Set_Material_Color(self, context)
#
#     else:
#         val = lambda: random.random()
#         C = (val(), val(), val())
#         context.object.active_material.diffuse_color = (C[0], C[1], C[2])
#         context.object.active_material.node_tree.nodes['Principled BSDF'].inputs[0].default_value = (
#             C[0], C[1], C[2], 1)

# # Create Material
# def SFC_Create_Material(self, context):
#     obj = context.active_object
#     addon_pref = get_addon_preferences()
#     use_bevel_node = addon_pref.use_bevel_node
#     WM = context.window_manager
#     context.scene.render.engine = 'CYCLES'
#
#     # if context.object.mode == "OBJECT":
#     #     obj.data.materials.clear()
#
#     if context.object.mode == "EDIT":
#         bpy.ops.object.material_slot_add()
#
#     # newmat = bpy.data.materials.new('Material')
#     # context.active_object.data.materials.append(newmat)
#     # context.object.active_material_index = len(context.object.data.materials) - 1
#     # context.active_object.active_material.use_nodes = True
#
#     # Create a new material
#     material = bpy.data.materials.new(name=obj.name)
#     material.use_nodes = True
#     material_output = material.node_tree.nodes.get('Material Output')
#     # remove diffuse
#     diffuse = material.node_tree.nodes['Diffuse BSDF']
#     material.node_tree.nodes.remove(diffuse)
#     # add Principled
#     shader = material.node_tree.nodes.new('ShaderNodeBsdfPrincipled')
#     shader.location[0] = 50
#     shader.location[1] = 306
#
#     # add bevel node
#     if use_bevel_node and bpy.app.version >= (2, 79, 1):
#         bevel = material.node_tree.nodes.new(type='ShaderNodeBevel')
#         bevel.samples = 16
#         bevel.inputs[0].default_value = 0.01
#         bevel.location[0] = -200
#         bevel.location[1] = -95
#         # connect Bevel to Principled
#         material.node_tree.links.new(bevel.outputs[0], shader.inputs[17])
#     # link shader to material
#     material.node_tree.links.new(material_output.inputs[0], shader.outputs[0])
#     # set active material to your new material
#     obj.active_material = material
#
#     set_material_color_or_random(self, context)
#     material_presets()
#
#     if context.object.mode == "EDIT":
#         bpy.ops.object.material_slot_assign()
#
#
#     WM.random_or_color = 'random'
#     WM.material_alpha = 1




# class SFC_OT_material_dif_colors(bpy.types.Operator):
#     """    ADD MATERIALS
#
#     CLICK - Plastic
#     SHIFT - Metallic
#     CTRL  - Glass
#     ALT    - With Transparency"""
#     bl_idname = "object.sfc_material_dif_colors"
#     bl_label = "Custom Add Custom Primitives"
#     bl_options = {"REGISTER"}
#
#     material_dif_colors : EnumProperty(
#         items=(("red", "Red", ""),
#                ("orange", "Orange", ""),
#                ("yellow", "Yellow", ""),
#                ('green', "Green", ""),
#                ('cian', "cian", ""),
#                ('blue', "Blue", ""),
#                ('purple', "Purple", ""),
#                ('pink', "pink", ""),
#                ('nb_1', "nb_1", ""),
#                ('nb_2', "nb_2", ""),
#                ('nb_3', "nb_3", ""),
#                ('nb_4', "nb_4", ""),
#                ('nb_5', "nb_5", ""),
#                ('nb_6', "nb_6", ""),
#                ('nb_7', "nb_7", ""),
#                ('nb_8', "nb_8", "")
#                ),
#         default='red')
#
#     def invoke(self, context,event):
#         WM = context.window_manager
#         obj = context.active_object
#         # acto_material = obj.active_material
#
#
#
#         create_and_assign_material(self, context)
#
#         WM.random_or_color = 'color'
#
#         if self.material_dif_colors == 'red':
#             WM.material_color = (1,0,0)
#
#         if self.material_dif_colors == 'orange':
#             WM.material_color = (1, 0.3, 0)
#
#         if self.material_dif_colors == 'yellow':
#             WM.material_color = (1, 0.8, 0)
#
#         if self.material_dif_colors == 'green':
#             WM.material_color = (0.01, 1, 0)
#
#         if self.material_dif_colors == 'cian':
#             WM.material_color = (0, 1, 1)
#
#         if self.material_dif_colors == 'blue':
#             WM.material_color = (0, 0.15, 1)
#
#         if self.material_dif_colors == 'purple':
#             WM.material_color = (0.2, 0, 1)
#
#         if self.material_dif_colors == 'pink':
#             WM.material_color = (1, 0, 1)
#
#         if self.material_dif_colors == 'nb_1':
#             WM.material_color = (0.004, 0.004, 0.004)
#
#         if self.material_dif_colors == 'nb_2':
#             WM.material_color = (0.2, 0.2, 0.2)
#
#         if self.material_dif_colors == 'nb_3':
#             WM.material_color = (0.073, 0.073, 0.073)
#
#         if self.material_dif_colors == 'nb_4':
#             WM.material_color = (0.171, 0.171, 0.171)
#
#         if self.material_dif_colors == 'nb_5':
#             WM.material_color = (0.319, 0.319, 0.319)
#
#         if self.material_dif_colors == 'nb_6':
#             WM.material_color = (0.523, 0.523, 0.523)
#
#         if self.material_dif_colors == 'nb_7':
#             WM.material_color = (0.787, 0.787, 0.787)
#
#         if self.material_dif_colors == 'nb_8':
#             WM.material_color = (1, 1, 1)
#
#
#         activeMaterial = context.active_object.active_material
#         currentNodeTree = activeMaterial.node_tree
#         shader = currentNodeTree.nodes["Principled BSDF"]
#
#         # if shader:
#         if event.shift:
#             # Metallic
#             shader.inputs[4].default_value = 1
#             # Roughness
#             shader.inputs[7].default_value = 0.2
#             # WM.shader_type == 'metal'
#
#         elif event.ctrl:
#             # Metallic
#             shader.inputs[4].default_value = 0
#             # Specular
#             shader.inputs[5].default_value = 0.5
#             # Roughness
#             shader.inputs[7].default_value = 0.01
#             # Transmittance
#             shader.inputs[15].default_value = 1
#             # WM.shader_type == 'glass'
#
#         else:
#             # Metallic
#             shader.inputs[4].default_value = 0
#             # Specular
#             shader.inputs[5].default_value = 0.5
#             # Roughness
#             shader.inputs[7].default_value = 0.2
#             # WM.shader_type == 'plastic'
#
#         if event.alt:
#             WM.material_alpha = 0.3
#
#         return {"FINISHED"}
#
# def create_and_assign_material(self, context):
#     selection = context.selected_objects
#
#     for obj in selection:
#         obj.select_set(state=True)
#         context.view_layer.objects.active = obj
#         context.object.show_transparent = True
#         context.space_data.show_backface_culling = True
#
#         if context.object.mode == "OBJECT":
#
#             if len(obj.material_slots):
#                 if context.active_object.active_material:
#                     activeMaterial = context.active_object.active_material
#                     currentNodeTree = activeMaterial.node_tree
#                     if not currentNodeTree.nodes == "Principled_BSDF":
#                         SFC_Create_Material(self, context)
#
#                     else:
#                         set_material_color_or_random(self, context)
#                         material_presets()
#                 else:
#                     bpy.ops.object.material_slot_remove()
#                     SFC_Create_Material(self, context)
#
#             else:
#                 SFC_Create_Material(self, context)
#
#         elif context.object.mode == "EDIT":
#             SFC_Create_Material(self, context)
#
# #Random Shader/Color
# class SFC_OT_random_color(bpy.types.Operator):
#     """    MATERIAL
#
#     CLICK - Add Random Color
#     SHIFT - Add the same Shader to Selection
#     CTRL  - Remove Shader
#     ALT    - Link Shader
#     CTRL + SHIFT - Select objects with the same material
#     CTRL + ALT - Single User"""
#
#     bl_idname = "object.sfc_random_color"
#     bl_label = "Random Color"
#     bl_options = {"REGISTER", "UNDO"}
#
#
#     def invoke(self, context, event):
#         selection = context.selected_objects
#
#         context.object.show_transparent = True
#         context.space_data.show_backface_culling = True
#
#         if event.ctrl and event.shift:
#             bpy.ops.object.select_linked(type='MATERIAL')
#
#         elif event.ctrl and event.alt:
#             bpy.ops.object.make_single_user(type='SELECTED_OBJECTS', object=False, obdata=False, material=True,
#                                                 texture=True, animation=False)
#
#         elif event.shift:
#             for obj in selection:
#                 obj.select_set(state=True)
#                 context.view_layer.objects.active =obj
#                 SFC_Create_Material(self, context)
#                 bpy.ops.object.make_links_data(type='MATERIAL')
#
#         elif event.ctrl:
#             for obj in selection:
#                 obj.select_set(state=True)
#                 context.view_layer.objects.active =obj
#                 if obj.material_slots:
#                     bpy.ops.object.material_slot_remove()
#
#         elif event.alt:
#             bpy.ops.object.make_links_data(type='MATERIAL')
#
#         else:
#             create_and_assign_material(self, context)
#
#         return {"FINISHED"}

def sfc_get_modifier_list(obj, mod_type):
    return [mod.name for mod in obj.modifiers if mod.type == mod_type]

def sfc_check_weighted_normals(obj):
    weighted_normals = any([mod for mod in bpy.types.Modifier.bl_rna.properties['type'].enum_items if
                            mod.identifier == 'WEIGHTED_NORMAL'])

    if weighted_normals:
        mod_weighted_normals = sfc_get_modifier_list(obj, 'WEIGHTED_NORMAL')
        if not mod_weighted_normals:
            new_weighted_normals = obj.modifiers.new("Weighted Normal", 'WEIGHTED_NORMAL')
            new_weighted_normals.keep_sharp = True

        else:

            bpy.ops.object.modifier_move_down(modifier="Weighted Normal")
            bpy.ops.object.modifier_move_down(modifier="Weighted Normal")
            bpy.ops.object.modifier_move_down(modifier="Weighted Normal")
            bpy.ops.object.modifier_move_down(modifier="Weighted Normal")

#====================#
#  Primitives
#====================#


class SFC_OT_custom_primitives(bpy.types.Operator):
    """    ADD BOOLEAN PRIMITIVES

    SHIFT - On selection
    CTRL  - On mouse
    ALT    - Align To View
    CTRL + SHIFT - Add Mirror Modifier"""
    bl_idname = "object.sfc_custom_primitives"
    bl_label = "Custom Add Custom Primitives"
    bl_options = {"REGISTER"}

    custom_primitives : EnumProperty(
        items=(('rounded_cube', "Rounded Cube", ""),
               ('rounded_plane', "Rounded Plane", ""),
               ('rounded_plane_round', "Rounded Plane Round", ""),
               ('cross', "cross", ""),
               ('rounded_cross', "Rounded cross", ""),
               ('long_cylinder', "Long cylinder", ""),),
                default='rounded_cube'
                )

    def invoke(self, context, event):
        SFC = context.window_manager.SFC
        obj = False
        ref = context.active_object

        if context.selected_objects and context.object is not None:
            mirror_object = context.active_object
        else:
            mirror_object = [obj for obj in context.selected_objects if context.object]

        saved_location = context.scene.cursor.location.copy()
        cursor_rot = bpy.context.scene.cursor.rotation_euler.copy()



        if event.shift:
            bpy.ops.view3d.snap_cursor_to_selected()

        if event.ctrl:
            bpy.ops.view3d.cursor3d('INVOKE_DEFAULT')

        # Rounded Cube
        if self.custom_primitives == 'rounded_cube':
            if context.object is not None:
                bpy.ops.object.mode_set(mode='OBJECT')
            bpy.ops.object.select_all(action='DESELECT')

            #add boolean
            if event.alt:
                self.act_obj = context.active_object

                self.new_bool_mod = self.act_obj.modifiers.new("Boolean", 'BOOLEAN')
                self.new_bool_mod.operation = 'DIFFERENCE'
                bpy.ops.object.modifier_move_up(modifier=self.new_bool_mod.name)
                bpy.ops.object.modifier_move_up(modifier=self.new_bool_mod.name)

                if SFC.align_to_view:
                    bpy.ops.mesh.primitive_cube_add(True, enter_editmode=False, align='VIEW')
                else:
                    bpy.ops.mesh.primitive_cube_add(True, enter_editmode=False, align='WORLD')
                self.new_prim = context.active_object
                self.new_prim.display_type = 'BOUNDS'
                self.new_bool_mod.object = self.new_prim

            else:
                if SFC.align_to_view:
                    bpy.ops.mesh.primitive_cube_add(True, enter_editmode=False, align='VIEW')
                else:
                    bpy.ops.mesh.primitive_cube_add(True, enter_editmode=False, align='WORLD')
                self.new_prim = context.active_object


            context.object.data.use_auto_smooth = True
            context.object.data.auto_smooth_angle = 1.0472
            bpy.ops.object.shade_smooth()



            # bevel
            new_bevel = self.new_prim.modifiers.new("Bevel", 'BEVEL')
            new_bevel.width = 0.3
            new_bevel.segments = 6
            new_bevel.profile = 0.5
            new_bevel.use_only_vertices = False
            new_bevel.limit_method = 'ANGLE'
            new_bevel.angle_limit = 1.535889744758606
            new_bevel.use_clamp_overlap = False

            # Add Mirror
            if event.ctrl and event.shift:
                self.act_obj = context.active_object
                newMod = self.act_obj.modifiers.new("Mirror", 'MIRROR')
                newMod.show_on_cage = True
                newMod.show_in_editmode = True
                if not mirror_object:
                    bpy.ops.object.empty_add(type='PLAIN_AXES')
                    bpy.ops.object.location_clear(clear_delta=False)
                    empty_for_mirror = context.active_object
                    bpy.ops.object.select_all(action='DESELECT')
                    context.view_layer.objects.active = self.act_obj
                    self.act_obj.select_set(state=True)
                    newMod.mirror_object = empty_for_mirror
                else:
                    newMod.mirror_object = mirror_object

        # Cross
        if self.custom_primitives == 'cross':
            if context.object is not None :
                bpy.ops.object.mode_set(mode = 'OBJECT')
            bpy.ops.object.select_all(action='DESELECT')

            # add boolean
            if event.alt:
                self.act_obj = context.active_object

                self.new_bool_mod = self.act_obj.modifiers.new("Boolean", 'BOOLEAN')
                self.new_bool_mod.operation = 'DIFFERENCE'
                bpy.ops.object.modifier_move_up(modifier=self.new_bool_mod.name)
                bpy.ops.object.modifier_move_up(modifier=self.new_bool_mod.name)

                if SFC.align_to_view:
                    bpy.ops.mesh.primitive_plane_add(True, enter_editmode=False, align='VIEW')
                else:
                    bpy.ops.mesh.primitive_plane_add(True, enter_editmode=False, align='WORLD')

                self.new_prim = context.active_object
                self.new_prim.display_type = 'BOUNDS'
                self.new_bool_mod.object = self.new_prim
            else:
                if SFC.align_to_view:
                    bpy.ops.mesh.primitive_plane_add(True, enter_editmode=False, align='VIEW')
                else:
                    bpy.ops.mesh.primitive_plane_add(True, enter_editmode=False, align='WORLD')
                self.new_prim = context.active_object

            context.object.data.use_auto_smooth = True
            context.object.data.auto_smooth_angle = 1.0472
            bpy.ops.object.shade_smooth()


            #bevel
            new_bevel = self.new_prim.modifiers.new("Bevel", 'BEVEL')
            new_bevel.width = 0.6
            new_bevel.segments = 2
            new_bevel.profile = 0.0
            new_bevel.use_only_vertices = True
            new_bevel.limit_method = 'ANGLE'
            new_bevel.angle_limit = 1.535889744758606
            new_bevel.use_clamp_overlap = False

            #Solidify
            new_solidify = self.new_prim.modifiers.new("Solidify", 'SOLIDIFY')
            new_solidify.thickness = 2
            new_solidify.use_even_offset = True
            new_solidify.use_quality_normals = True
            new_solidify.offset = 0

            # Add Mirror
            if event.ctrl and event.shift:
                newMod = self.new_prim.modifiers.new("Mirror", 'MIRROR')
                newMod.show_on_cage = True
                newMod.show_in_editmode = True
                if not mirror_object:
                    bpy.ops.object.empty_add(type='PLAIN_AXES')
                    bpy.ops.object.location_clear(clear_delta=False)
                    empty_for_mirror = context.active_object
                    bpy.ops.object.select_all(action='DESELECT')
                    context.view_layer.objects.active = self.new_prim
                    self.new_prim.select_set(state=True)
                    newMod.mirror_object = empty_for_mirror
                else:
                    newMod.mirror_object = mirror_object

        # Rounded Cross
        if self.custom_primitives == 'rounded_cross':
            if context.object is not None :
                bpy.ops.object.mode_set(mode = 'OBJECT')
            bpy.ops.object.select_all(action='DESELECT')

            # add boolean
            if event.alt:
                self.act_obj = context.active_object

                self.new_bool_mod = self.act_obj.modifiers.new("Boolean", 'BOOLEAN')
                self.new_bool_mod.operation = 'DIFFERENCE'
                bpy.ops.object.modifier_move_up(modifier=self.new_bool_mod.name)
                bpy.ops.object.modifier_move_up(modifier=self.new_bool_mod.name)

                if SFC.align_to_view:
                    bpy.ops.mesh.primitive_plane_add(True, enter_editmode=False, align='VIEW')
                else:
                    bpy.ops.mesh.primitive_plane_add(True, enter_editmode=False, align='WORLD')

                self.new_prim = context.active_object
                self.new_prim.display_type = 'BOUNDS'
                self.new_bool_mod.object = self.new_prim
            else:
                if SFC.align_to_view:
                    bpy.ops.mesh.primitive_plane_add(True, enter_editmode=False, align='VIEW')
                else:
                    bpy.ops.mesh.primitive_plane_add(True, enter_editmode=False, align='WORLD')
                self.new_prim = context.active_object

            context.object.data.use_auto_smooth = True
            context.object.data.auto_smooth_angle = 1.0472
            bpy.ops.object.shade_smooth()


            # bevel
            new_bevel = self.new_prim.modifiers.new("Bevel", 'BEVEL')
            new_bevel.width = 0.6
            new_bevel.segments = 2
            new_bevel.profile = 0.0
            new_bevel.use_only_vertices = True
            new_bevel.limit_method = 'ANGLE'
            new_bevel.angle_limit = 1.535889744758606
            new_bevel.use_clamp_overlap = False

            # bevel round
            new_bevel_round = self.new_prim.modifiers.new("Bevel", 'BEVEL')
            new_bevel_round.width = 0.15
            new_bevel_round.segments = 4
            new_bevel_round.profile = 0.5
            new_bevel_round.use_only_vertices = True
            new_bevel_round.limit_method = 'ANGLE'
            new_bevel_round.angle_limit = 1.535889744758606
            new_bevel_round.use_clamp_overlap = False

            # Solidify
            new_solidify = self.new_prim.modifiers.new("Solidify", 'SOLIDIFY')
            new_solidify.thickness = 2
            new_solidify.use_even_offset = True
            new_solidify.use_quality_normals = True
            new_solidify.offset = 0

            # Add Mirror
            if event.ctrl and event.shift:

                newMod = self.new_prim.modifiers.new("Mirror", 'MIRROR')
                newMod.show_on_cage = True
                newMod.show_in_editmode = True
                if not mirror_object:
                    bpy.ops.object.empty_add(type='PLAIN_AXES')
                    bpy.ops.object.location_clear(clear_delta=False)
                    empty_for_mirror = context.active_object
                    bpy.ops.object.select_all(action='DESELECT')
                    context.view_layer.objects.active = self.new_prim
                    self.new_prim.select_set(state=True)
                    newMod.mirror_object = empty_for_mirror
                else:
                    newMod.mirror_object = mirror_object

        # Rounded Plane
        if self.custom_primitives == 'rounded_plane':
            if context.object is not None :
                bpy.ops.object.mode_set(mode = 'OBJECT')
            bpy.ops.object.select_all(action='DESELECT')

            # add boolean
            if event.alt:
                self.act_obj = context.active_object

                self.new_bool_mod = self.act_obj.modifiers.new("Boolean", 'BOOLEAN')
                self.new_bool_mod.operation = 'DIFFERENCE'
                bpy.ops.object.modifier_move_up(modifier=self.new_bool_mod.name)
                bpy.ops.object.modifier_move_up(modifier=self.new_bool_mod.name)

                if SFC.align_to_view:
                    bpy.ops.mesh.primitive_plane_add(True, enter_editmode=False, align='VIEW')
                else:
                    bpy.ops.mesh.primitive_plane_add(True, enter_editmode=False, align='WORLD')
                self.new_prim = context.active_object
                self.new_prim.display_type = 'BOUNDS'
                self.new_bool_mod.object = self.new_prim
            else:
                if SFC.align_to_view:
                    bpy.ops.mesh.primitive_plane_add(True, enter_editmode=False, align='VIEW')
                else:
                    bpy.ops.mesh.primitive_plane_add(True, enter_editmode=False, align='WORLD')

                self.new_prim = context.active_object

            context.object.data.use_auto_smooth = True
            context.object.data.auto_smooth_angle = 1.0472
            bpy.ops.object.shade_smooth()

            # bevel
            new_bevel = self.new_prim.modifiers.new("Bevel", 'BEVEL')
            new_bevel.width = 0.4
            new_bevel.segments = 6
            new_bevel.profile = 0.5
            new_bevel.use_only_vertices = True
            new_bevel.limit_method = 'ANGLE'
            new_bevel.angle_limit = 1.535889744758606
            new_bevel.use_clamp_overlap = False

            # Solidify
            new_solidify = self.new_prim.modifiers.new("Solidify", 'SOLIDIFY')
            new_solidify.thickness = 2
            new_solidify.use_even_offset = True
            new_solidify.use_quality_normals = True
            new_solidify.offset = 0

            # Add Mirror
            if event.ctrl and event.shift:
                newMod = self.new_prim.modifiers.new("Mirror", 'MIRROR')
                newMod.show_on_cage = True
                newMod.show_in_editmode = True
                if not mirror_object:
                    bpy.ops.object.empty_add(type='PLAIN_AXES')
                    bpy.ops.object.location_clear(clear_delta=False)
                    empty_for_mirror = context.active_object
                    bpy.ops.object.select_all(action='DESELECT')
                    context.view_layer.objects.active = self.new_prim
                    self.new_prim.select_set(state=True)
                    newMod.mirror_object = empty_for_mirror
                else:
                    newMod.mirror_object = mirror_object

        # Rounded Plane Round
        if self.custom_primitives == 'rounded_plane_round':
            if context.object is not None :
                bpy.ops.object.mode_set(mode = 'OBJECT')
            bpy.ops.object.select_all(action='DESELECT')

            # add boolean
            if event.alt:
                self.act_obj = context.active_object

                self.new_bool_mod = self.act_obj.modifiers.new("Boolean", 'BOOLEAN')
                self.new_bool_mod.operation = 'DIFFERENCE'
                bpy.ops.object.modifier_move_up(modifier=self.new_bool_mod.name)
                bpy.ops.object.modifier_move_up(modifier=self.new_bool_mod.name)

                if SFC.align_to_view:
                    bpy.ops.mesh.primitive_plane_add(True, enter_editmode=False, align='VIEW')
                else:
                    bpy.ops.mesh.primitive_plane_add(True, enter_editmode=False, align='WORLD')
                self.new_prim = context.active_object
                self.new_prim.display_type = 'BOUNDS'
                self.new_bool_mod.object = self.new_prim
            else:
                if SFC.align_to_view:
                    bpy.ops.mesh.primitive_plane_add(True, enter_editmode=False, align='VIEW')
                else:
                    bpy.ops.mesh.primitive_plane_add(True, enter_editmode=False, align='WORLD')
                self.new_prim = context.active_object

            context.object.data.use_auto_smooth = True
            context.object.data.auto_smooth_angle = 1.0472
            bpy.ops.object.shade_smooth()

            # bevel
            new_bevel = self.new_prim.modifiers.new("Bevel", 'BEVEL')
            new_bevel.width = 0.4
            new_bevel.segments = 6
            new_bevel.profile = 0.5
            new_bevel.use_only_vertices = True
            new_bevel.limit_method = 'ANGLE'
            new_bevel.angle_limit = 1.535889744758606
            new_bevel.use_clamp_overlap = False

            # Solidify
            new_solidify = self.new_prim.modifiers.new("Solidify", 'SOLIDIFY')
            new_solidify.thickness = 2
            new_solidify.use_even_offset = True
            new_solidify.use_quality_normals = True
            new_solidify.offset = 0

            # bevel round
            new_bevel_round = self.new_prim.modifiers.new("Bevel", 'BEVEL')
            new_bevel_round.width = 0.15
            new_bevel_round.segments = 4
            new_bevel_round.profile = 0.5
            new_bevel_round.use_only_vertices = False
            new_bevel_round.limit_method = 'ANGLE'
            new_bevel_round.angle_limit = 1.535889744758606
            new_bevel_round.use_clamp_overlap = False

            # Add Mirror
            if event.ctrl and event.shift:
                newMod = self.new_prim.modifiers.new("Mirror", 'MIRROR')
                newMod.show_on_cage = True
                newMod.show_in_editmode = True
                if not mirror_object:
                    bpy.ops.object.empty_add(type='PLAIN_AXES')
                    bpy.ops.object.location_clear(clear_delta=False)
                    empty_for_mirror = context.active_object
                    bpy.ops.object.select_all(action='DESELECT')
                    context.view_layer.objects.active = self.new_prim
                    self.new_prim.select_set(state=True)
                    newMod.mirror_object = empty_for_mirror
                else:
                    newMod.mirror_object = mirror_object

        # Long Cylinder
        if self.custom_primitives == 'long_cylinder':
            if context.object is not None :
                bpy.ops.object.mode_set(mode = 'OBJECT')
            bpy.ops.object.select_all(action='DESELECT')

            # add boolean
            if event.alt:
                self.act_obj = context.active_object

                self.new_bool_mod = self.act_obj.modifiers.new("Boolean", 'BOOLEAN')
                self.new_bool_mod.operation = 'DIFFERENCE'
                bpy.ops.object.modifier_move_up(modifier=self.new_bool_mod.name)
                bpy.ops.object.modifier_move_up(modifier=self.new_bool_mod.name)

                if SFC.align_to_view:
                    bpy.ops.mesh.primitive_plane_add(True, enter_editmode=False, align='VIEW')
                else:
                    bpy.ops.mesh.primitive_plane_add(True, enter_editmode=False, align='WORLD')
                self.new_prim = context.active_object
                self.new_prim.display_type = 'BOUNDS'
                self.new_bool_mod.object = self.new_prim
            else:
                if SFC.align_to_view:
                    bpy.ops.mesh.primitive_plane_add(True, enter_editmode=False, align='VIEW')
                else:
                    bpy.ops.mesh.primitive_plane_add(True, enter_editmode=False, align='WORLD')
                self.new_prim = context.active_object

            bpy.ops.transform.resize(value=(1, 2, 1))
            bpy.ops.object.transform_apply(location=False, rotation=False, scale=True)
            context.object.data.use_auto_smooth = True
            context.object.data.auto_smooth_angle = 1.0472
            bpy.ops.object.shade_smooth()

            # bevel
            new_bevel = self.new_prim.modifiers.new("Bevel", 'BEVEL')
            new_bevel.width = 0.888
            new_bevel.segments = 6
            new_bevel.profile = 0.46
            new_bevel.use_only_vertices = True
            new_bevel.limit_method = 'ANGLE'
            new_bevel.angle_limit = 1.535889744758606
            new_bevel.use_clamp_overlap = False

            # Solidify
            new_solidify = self.new_prim.modifiers.new("Solidify", 'SOLIDIFY')
            new_solidify.thickness = 2
            new_solidify.use_even_offset = True
            new_solidify.use_quality_normals = True
            new_solidify.offset = 0

            # Add Mirror
            if event.ctrl and event.shift:
                newMod = self.new_prim.modifiers.new("Mirror", 'MIRROR')
                newMod.show_on_cage = True
                newMod.show_in_editmode = True
                if not mirror_object:
                    bpy.ops.object.empty_add(type='PLAIN_AXES')
                    bpy.ops.object.location_clear(clear_delta=False)
                    empty_for_mirror = context.active_object
                    bpy.ops.object.select_all(action='DESELECT')
                    context.view_layer.objects.active = self.new_prim
                    self.new_prim.select_set(state=True)
                    newMod.mirror_object = empty_for_mirror
                else:
                    newMod.mirror_object = mirror_object

        # context.object.data.show_double_sided = True

        if SFC.align_cursor_rot:
            bpy.context.object.rotation_euler = cursor_rot


        # weighted = False
        # for mod in ref.modifiers:
        #     if mod.type == 'WEIGHTED_NORMAL':
        #         weighted = True
        #
        #     if weighted:
        #         bpy.ops.object.select_all(action='DESELECT')
        #         context.view_layer.objects.active = ref
        #         ref.select_set(state=True)
        #         bpy.ops.object.modifier_move_down(modifier="Weighted Normal")
        #         bpy.ops.object.modifier_move_down(modifier="Weighted Normal")
        #         bpy.ops.object.modifier_move_down(modifier="Weighted Normal")
        #         bpy.ops.object.modifier_move_down(modifier="Weighted Normal")
        #         bpy.ops.object.modifier_move_down(modifier="Weighted Normal")
        #         bpy.ops.object.select_all(action='DESELECT')
        #
        #
        #         context.view_layer.objects.active = self.new_prim
        #         self.new_prim.select_set(state=True)

        if event.ctrl:
            bpy.ops.transform.translate('INVOKE_DEFAULT')

        context.scene.cursor.location = saved_location
        return {"FINISHED"}




class SFC_OT_primitives(bpy.types.Operator):
    """    ADD PRIMITIVES
    
    SHIFT - On selection
    CTRL  - On mouse
    ALT    - In Edit Mode
    CTRL + SHIFT - Add Mirror Modifier"""
    bl_idname = "object.sfc_primitives"
    bl_label = "Custom Add Primitives"
    bl_options = {"REGISTER"}
    
    primitive : EnumProperty(
        items = (('cube', "Cube", ""),
                 ('sphere', "Sphere", ""),
                 ('plan', "Plan", ""),
                 ('grid', "Grid", ""),
                 ('cylinder', "Cylinder", ""),
                 ('cone', "Cone", ""),
                 ('torus', "Torus", ""),
                 ('bezier', "Bezier", ""),
                 ('vertex', "Vertex", ""),
                 ('mesh_circle', "Mesh Circle", ""),
                 ('circle', "Circle", ""),
                 ('curve_line', "Curve Line", "")),
                 default = 'cube'
                 )
    
    def invoke(self, context, event):
        obj = False      
        self.act_obj = context.active_object
        self.mirror_object = self.act_obj

        # if context.selected_objects and context.object is not None :
        #     mirror_object = context.active_object
        # else:
        #     mirror_object = [obj for obj in context.selected_objects if context.object]
        
        
        saved_location = context.scene.cursor.location.copy()
        
        if event.shift:
            bpy.ops.view3d.snap_cursor_to_selected()

        if event.ctrl:
            bpy.ops.view3d.cursor3d('INVOKE_DEFAULT')
        
        #Cube
        if self.primitive == 'cube':
            if event.alt:
                if context.object is not None and context.object.mode == "OBJECT":
                    bpy.ops.object.select_all(action='DESELECT')
                bpy.ops.mesh.primitive_cube_add(size=2, enter_editmode=True, align='WORLD')
            else:
                if context.object is not None :
                    bpy.ops.object.mode_set(mode = 'OBJECT')
                bpy.ops.mesh.primitive_cube_add(True, size=2, enter_editmode=False, align='WORLD')
                

        # Circle
        elif self.primitive == 'mesh_circle':
            if event.alt:
                if context.object is not None and context.object.mode == "OBJECT":
                    bpy.ops.object.select_all(action='DESELECT')
                bpy.ops.mesh.primitive_circle_add(vertices=16, radius=1, fill_type='NOTHING', enter_editmode=True,align='WORLD')
            else:
                if context.object is not None:
                    bpy.ops.object.mode_set(mode='OBJECT')
                bpy.ops.mesh.primitive_circle_add(True, vertices=16, radius=1, fill_type='NOTHING', enter_editmode=False,
                                                  align='WORLD')

        #Plan
        elif self.primitive == 'plan':
            if event.alt:
                if context.object is not None and context.object.mode == "OBJECT":
                    bpy.ops.object.select_all(action='DESELECT')
                bpy.ops.mesh.primitive_plane_add(size=2, enter_editmode=True, align='WORLD')

            else:
                if context.object is not None :
                    bpy.ops.object.mode_set(mode = 'OBJECT')
                bpy.ops.mesh.primitive_plane_add(True, size=2, enter_editmode=False, align='WORLD')

                        
        #Grid
        elif self.primitive == 'grid':
            if event.alt:
                if context.object is not None and context.object.mode == "OBJECT":
                    bpy.ops.object.select_all(action='DESELECT')
                bpy.ops.mesh.primitive_grid_add(x_subdivisions=10, y_subdivisions=10, size=2, enter_editmode=True,align='WORLD')

            else:
                if context.object is not None :
                    bpy.ops.object.mode_set(mode = 'OBJECT')
                bpy.ops.mesh.primitive_grid_add(True, x_subdivisions=10, y_subdivisions=10, size=2, enter_editmode=False,align='WORLD')

                        
        #Sphere                                
        elif self.primitive == 'sphere':
            if event.alt:
                if context.object is not None and context.object.mode == "OBJECT":
                    bpy.ops.object.select_all(action='DESELECT')
                bpy.ops.mesh.primitive_uv_sphere_add(segments=24, ring_count=12, radius=1, enter_editmode=True,align='WORLD')

            else:
                if context.object is not None :
                    bpy.ops.object.mode_set(mode = 'OBJECT')
                bpy.ops.mesh.primitive_uv_sphere_add(True, segments=24, ring_count=12, radius=1, enter_editmode=False,
                                                     align='WORLD')

                        
        #Cylinder    
        elif self.primitive == 'cylinder':
            if event.alt:
                if context.object is not None and context.object.mode == "OBJECT":
                    bpy.ops.object.select_all(action='DESELECT')
                bpy.ops.mesh.primitive_cylinder_add(vertices=16, radius=1, depth=2, end_fill_type='NGON',
                                                    enter_editmode=True, align='WORLD')

            else:
                if context.object is not None :
                    bpy.ops.object.mode_set(mode = 'OBJECT')
                bpy.ops.mesh.primitive_cylinder_add(True, vertices=16, radius=1, depth=2, end_fill_type='NGON',
                                                    enter_editmode=False, align='WORLD')

                        
        #cone    
        elif self.primitive == 'cone':
            if event.alt:
                if context.object is not None and context.object.mode == "OBJECT":
                    bpy.ops.object.select_all(action='DESELECT')
                bpy.ops.mesh.primitive_cone_add(vertices=16, radius1=1, radius2=0, depth=2, end_fill_type='NGON',
                                                enter_editmode=True, align='WORLD')
            else:
                if context.object is not None :
                    bpy.ops.object.mode_set(mode = 'OBJECT')
                bpy.ops.mesh.primitive_cone_add(True, vertices=16, radius1=1, radius2=0, depth=2, end_fill_type='NGON',
                                                enter_editmode=False, align='WORLD')

                        
        #Empty Axe
        elif self.primitive == 'empty_axe':
            if context.object is not None :
                bpy.ops.object.mode_set(mode = 'OBJECT')
            bpy.ops.object.empty_add(type='PLAIN_AXES', radius=1, align='WORLD')


        #Bezier
        elif self.primitive == 'bezier':
            if event.alt:
                if context.object is not None and context.object.mode == "OBJECT":
                    bpy.ops.object.select_all(action='DESELECT')
                bpy.ops.curve.primitive_bezier_curve_add(radius=1, enter_editmode=True, align='WORLD')

            else:
                if context.object is not None :
                    bpy.ops.object.mode_set(mode = 'OBJECT')
                bpy.ops.curve.primitive_bezier_curve_add(True, radius=1, enter_editmode=False, align='WORLD')


        #Circle
        elif self.primitive == 'circle':
            if event.alt:
                if context.object is not None and context.object.mode == "OBJECT":
                    bpy.ops.object.select_all(action='DESELECT')
                bpy.ops.curve.primitive_bezier_circle_add(radius=1, enter_editmode=True, align='WORLD')

            else:
                if context.object is not None :
                    bpy.ops.object.mode_set(mode = 'OBJECT')
                bpy.ops.curve.primitive_bezier_circle_add(True, radius=1, enter_editmode=False, align='WORLD')



        #Text
        elif self.primitive == 'text': 
            if event.alt:
                if context.object is not None and context.object.mode == "OBJECT":
                    bpy.ops.object.select_all(action='DESELECT')
                bpy.ops.object.text_add(radius=1, enter_editmode=True, align='WORLD')
            else:       
                if context.object is not None :
                    bpy.ops.object.mode_set(mode = 'OBJECT')
                    bpy.ops.object.text_add(True, radius=1, enter_editmode=False, align='WORLD')

                        
        #Torus         
        elif self.primitive == 'torus': 
            if event.alt:
                if context.object is not None and context.object.mode == "OBJECT":
                    bpy.ops.object.select_all(action='DESELECT')
                bpy.ops.mesh.primitive_torus_add(align='WORLD', major_segments=32, minor_segments=12, mode='MAJOR_MINOR',
                                                 major_radius=1.25, minor_radius=0.25, abso_major_rad=1.25,
                                                 abso_minor_rad=0.75)
                bpy.ops.object.mode_set(mode = 'EDIT')
            else:
                if context.object is not None :
                    bpy.ops.object.mode_set(mode = 'OBJECT')
                bpy.ops.mesh.primitive_torus_add(True, align='WORLD', major_segments=32, minor_segments=12,
                                                 mode='MAJOR_MINOR',
                                                 major_radius=1.25, minor_radius=0.25, abso_major_rad=1.25,
                                                 abso_minor_rad=0.75)

        # #Vertex
        # elif self.primitive == 'vertex':
        #     context.scene.tool_settings.snap_element = 'INCREMENT'
        #     context.scene.tool_settings.use_snap_grid_absolute = True
        #
        #     if context.object is not None :
        #         bpy.ops.object.mode_set(mode = 'OBJECT')
        #
        #     bpy.ops.view3d.cursor3d('INVOKE_DEFAULT')
        #
        #     self.bevel_angle = False
        #     self.weigthed_normals = False
        #
        #     # add boolean
        #     if event.shift and event.alt :
        #         if len([obj for obj in context.selected_objects if obj.type == 'MESH']) == 1:
        #             self.act_obj = context.active_object
        #
        #             # Add boolean to Act_obj
        #             self.new_bool_mod = self.act_obj.modifiers.new("Boolean", 'BOOLEAN')
        #             self.new_bool_mod.operation = 'DIFFERENCE'
        #             # new_bool_mod.show_viewport = False
        #             bool_name = self.new_bool_mod.name
        #
        #
        #             for mod in self.act_obj.modifiers:
        #                 if mod.type == 'BEVEL' and mod.limit_method == 'ANGLE':
        #                     self.bevel_angle = True
        #
        #             if self.bevel_angle:
        #                 bpy.ops.object.modifier_move_up(modifier=self.new_bool_mod.name)
        #
        #             sfc_get_modifier_list(self.act_obj)
        #
        #             # Add plane
        #             bpy.ops.mesh.primitive_plane_add()
        #
        #
        #             plane_obj = context.active_object
        #             plane_obj.display_type = 'WIRE'
        #             self.new_bool_mod.object = plane_obj
        #
        #             bpy.ops.object.select_all(action='DESELECT')
        #
        #             # Duplicate and change the mode
        #             context.view_layer.objects.active = self.act_obj
        #             self.act_obj.select_set(state=True)
        #             bpy.ops.object.duplicate_move()
        #             rebool_obj = context.active_object
        #             rebool_obj.modifiers[bool_name].operation = 'INTERSECT'
        #
        #             bpy.ops.object.select_all(action='DESELECT')
        #
        #             #select plane
        #             context.view_layer.objects.active = plane_obj
        #             plane_obj.select_set(state=True)
        #
        #         else:
        #             bpy.ops.mesh.primitive_plane_add()
        #
        #
        #
        #     elif event.shift:
        #         if len([obj for obj in context.selected_objects if obj.type == 'MESH']) == 1:
        #             self.act_obj = context.active_object
        #
        #             self.new_bool_mod = self.act_obj.modifiers.new("Boolean", 'BOOLEAN')
        #             self.new_bool_mod.operation = 'DIFFERENCE'
        #
        #             for mod in self.act_obj.modifiers:
        #                 if mod.type == 'BEVEL' and mod.limit_method == 'ANGLE':
        #                     self.bevel_angle = True
        #                 if mod.type == 'WEIGHTED_NORMAL':
        #                     self.weigthed_normals = True
        #
        #             if self.bevel_angle:
        #                 bpy.ops.object.modifier_move_up(modifier=self.new_bool_mod.name)
        #
        #             if self.weigthed_normals:
        #                 bpy.ops.object.modifier_move_up(modifier=self.new_boolean.name)
        #
        #             sfc_get_modifier_list(self.act_obj)
        #
        #             bpy.ops.mesh.primitive_plane_add()
        #
        #
        #             new_obj = context.active_object
        #             new_obj.display_type = 'WIRE'
        #             self.new_bool_mod.object = new_obj
        #         else:
        #             bpy.ops.mesh.primitive_plane_add()
        #
        #     else :
        #         bpy.ops.mesh.primitive_plane_add()



        # bpy.ops.object.transform_apply(location=True, rotation=True, scale=True)



        # bpy.ops.transform.translate('INVOKE_DEFAULT')
                     
        #Curve Line
        elif self.primitive == 'curve_line': 
            if context.object is not None and obj.type == 'MESH':
                bpy.ops.object.mode_set(mode = 'OBJECT')
            bpy.ops.curve.primitive_bezier_curve_add(enter_editmode=False, align='VIEW')
            bpy.ops.object.mode_set(mode = 'EDIT')
            bpy.ops.curve.delete(type='VERT')
            bpy.ops.object.mode_set(mode = 'OBJECT')
            bpy.ops.object.transform_apply(location=True, rotation=True, scale=True)
            bpy.ops.object.mode_set(mode = 'EDIT')
            context.space_data.overlay.normal_face = False
            
            #Add Mirror
            bpy.ops.object.mode_set(mode = 'OBJECT') 
            if event.ctrl and event.shift:
                act_obj = context.active_object
                newMod = act_obj.modifiers.new("Mirror", 'MIRROR')
                newMod.show_on_cage = True
                newMod.show_in_editmode = True
                if not self.mirror_object:
                    bpy.ops.object.empty_add(type='PLAIN_AXES')
                    bpy.ops.object.location_clear(clear_delta=False)
                    empty_for_mirror = context.active_object
                    bpy.ops.object.select_all(action='DESELECT')
                    context.view_layer.objects.active = act_obj
                    act_obj.select_set(state=True)
                    newMod.mirror_object = empty_for_mirror  
                else:
                    newMod.mirror_object = self.mirror_object
                
            bpy.ops.object.mode_set(mode = 'EDIT')
            bpy.ops.mesh.select_all(action='DESELECT')
            # bpy.ops.object.mode_set(mode='OBJECT')
                    
            bpy.ops.curve.draw('INVOKE_DEFAULT')

        # Add Mirror
        elif event.ctrl and event.shift:
            self.new_obj = context.active_object
            newMod = self.new_obj.modifiers.new("Mirror", 'MIRROR')
            newMod.show_on_cage = True
            newMod.show_in_editmode = True
            newMod.mirror_object = self.mirror_object



        # Add boolean
        elif event.oskey and event.shift:
            self.new_obj = context.active_object
            self.new_bool_mod = self.act_obj.modifiers.new("Boolean", 'BOOLEAN')
            self.new_bool_mod.operation = 'DIFFERENCE'
            self.new_bool_mod.object = self.new_obj
            context.object.display_type = 'BOUNDS'

        # if context.space_data.use_occlude_geometry == True:
        #     context.space_data.use_occlude_geometry = False
        # else:
        #     pass

        for obj in context.selected_objects:
            if obj.type =='MESH':
                # context.object.data.show_double_sided = True
                context.object.data.use_auto_smooth = True
                context.object.data.auto_smooth_angle = 0.541052
                bpy.ops.object.mode_set(mode='OBJECT')
                bpy.ops.object.shade_smooth()


        if event.ctrl:
            bpy.ops.transform.translate('INVOKE_DEFAULT')
            
        context.scene.cursor.location = saved_location
        return {"FINISHED"}

# def sfc_get_modifier_list(act_obj, mod_type, limit_method):
#     return [mod.name for mod in act_obj.modifiers if mod.type == mod_type if mod.limit_method == limit_method]

# def sfc_get_modifier_index(self, obj, target):
#
#     for idx, mod in enumerate(obj.modifiers):
#         if mod == target:
#             return idx

class SFC_OT_primitives_vertex(bpy.types.Operator):
    """    ADD PRIMITIVES

    SHIFT - Boolean
    ALT    - Rebool
    CTRL  - Add Mirror Modifier
    You can combine CTRL+ALT & CTRL+SHIFT"""
    bl_idname = "object.sfc_primitives_vertex"
    bl_label = "Custom Add Primitives Vertex"
    bl_options = {"REGISTER"}



    def invoke(self, context, event):
        obj = False
        self.act_obj = context.active_object

        # context.scene.tool_settings.use_snap = True
        # context.scene.tool_settings.snap_element = 'VERTEX'

        mirror_object = self.act_obj

        # if context.selected_objects and context.object is not None:
        #     mirror_object = self.act_obj
        # else:
        #     mirror_object = [obj for obj in context.selected_objects if context.object]

        saved_location = context.scene.cursor.location.copy()

        # if event.shift:
        #     bpy.ops.view3d.snap_cursor_to_selected()


        if context.object is not None:
            bpy.ops.object.mode_set(mode='OBJECT')

        bpy.ops.view3d.cursor3d('INVOKE_DEFAULT')

        # add boolean
        if event.alt:
            if len([obj for obj in context.selected_objects if obj.type == 'MESH']) == 1:

                # Add boolean to Act_obj
                self.new_bool_mod = self.act_obj.modifiers.new("Boolean", 'BOOLEAN')
                self.new_bool_mod.operation = 'DIFFERENCE'
                # new_bool_mod.show_viewport = False
                bool_name = self.new_bool_mod.name

                self.bevel_angle = False
                self.weigthed_normals = False
                for mod in self.act_obj.modifiers:
                    if mod.type == 'BEVEL' and mod.limit_method == 'ANGLE':
                        self.bevel_angle = True

                # We make sure that the Bevel is placed before the solidify
                # self.bevel_name = sfc_get_modifier_list(self.act_obj, 'BEVEL')
                #
                # bool_idx = self.sfc_get_modifier_index(self.act_obj, bool_name)
                # bevel_idx = self.sfc_get_modifier_index(self.act_obj, self.bevel_name)
                #
                # while bool_idx != bevel_idx - 1:
                #     bpy.ops.object.modifier_move_up(
                #         modifier=bool_name.name)
                #     bevel_idx -= 1

                if self.bevel_angle:
                    bpy.ops.object.modifier_move_up(modifier=bool_name)
                    bpy.ops.object.modifier_move_up(modifier=bool_name)

                if self.weigthed_normals:
                    bpy.ops.object.modifier_move_up(modifier=bool_name)



                # Add plane
                bpy.ops.mesh.primitive_plane_add()

                plane_obj = context.active_object
                # context.object.data.show_double_sided = True
                plane_obj.display_type = 'WIRE'
                self.new_bool_mod.object = plane_obj

                bpy.ops.object.select_all(action='DESELECT')

                # Duplicate and change the mode
                context.view_layer.objects.active = self.act_obj
                self.act_obj.select_set(state=True)
                bpy.ops.object.duplicate_move()
                rebool_obj = context.active_object
                rebool_obj.modifiers[bool_name].operation = 'INTERSECT'

                if self.bevel_angle:
                    bpy.ops.object.modifier_move_up(modifier=bool_name)


                if self.weigthed_normals:
                    bpy.ops.object.modifier_move_up(modifier=bool_name)

                bpy.ops.object.select_all(action='DESELECT')

                # select plane
                context.view_layer.objects.active = plane_obj
                plane_obj.select_set(state=True)

            else:
                bpy.ops.mesh.primitive_plane_add()




        elif event.shift:
            if len([obj for obj in context.selected_objects if context.object.type == 'MESH']) == 1:

                self.new_bool_mod = self.act_obj.modifiers.new("Boolean", 'BOOLEAN')
                self.new_bool_mod.operation = 'DIFFERENCE'
                bool_name = self.new_bool_mod.name

                # # We make sure that the Bevel is placed before the solidify
                # self.bevel_name = sfc_get_modifier_list(self.act_obj, 'BEVEL')
                #
                # bool_idx = self.sfc_get_modifier_index(self.act_obj, bool_name)
                # bevel_idx = self.sfc_get_modifier_index(self.act_obj, self.bevel_name)
                #
                # while bool_idx != bevel_idx - 1:
                #     bpy.ops.object.modifier_move_up(
                #         modifier=bool_name.name)
                #     bevel_idx -= 1

                self.bevel_angle = False
                self.weigthed_normals = False
                for mod in self.act_obj.modifiers:
                    if mod.type == 'BEVEL' and mod.limit_method == 'ANGLE':
                        self.bevel_angle = True
                    if mod.type == 'WEIGHTED_NORMAL':
                        self.weigthed_normals = True

                if self.bevel_angle:
                    bpy.ops.object.modifier_move_up(modifier=bool_name)


                if self.weigthed_normals:
                    bpy.ops.object.modifier_move_up(modifier=bool_name)

                # sfc_get_modifier_list(self.act_obj)

                bpy.ops.mesh.primitive_plane_add()
                new_obj = context.active_object
                new_obj.display_type = 'WIRE'
                self.new_bool_mod.object = new_obj
            else:
                bpy.ops.mesh.primitive_plane_add()

        else:
            bpy.ops.mesh.primitive_plane_add()

        bpy.ops.object.transform_apply(location=True, rotation=True, scale=True)

        self.new_obj = context.active_object

        # Add Mirror
        if event.ctrl :

            newMod = self.new_obj.modifiers.new("Mirror", 'MIRROR')
            newMod.show_on_cage = True
            newMod.show_in_editmode = True
            # if not mirror_object:
            #     bpy.ops.object.empty_add(type='PLAIN_AXES')
            #     bpy.ops.object.location_clear(clear_delta=False)
            #     empty_for_mirror = context.active_object
            #     bpy.ops.object.select_all(action='DESELECT')
            #     context.view_layer.objects.active = self.act_obj
            #     self.act_obj.select_set(state=True)
            #     newMod.mirror_object = empty_for_mirror
            # else:
            newMod.mirror_object = self.act_obj #mirror_object




        #shading
        context.object.data.use_auto_smooth = True
        context.object.data.auto_smooth_angle = 0.541052

        if event.shift and event.alt:
            context.object.display_type = 'BOUNDS'

        bpy.ops.object.mode_set(mode='EDIT')
        bpy.ops.mesh.select_mode(use_extend=False, use_expand=False, type='VERT')
        bpy.ops.mesh.merge(type='CENTER')

        # if context.space_data.use_occlude_geometry == True:
        #     context.space_data.use_occlude_geometry = False
        # else:
        #     pass




        bpy.ops.transform.translate('INVOKE_DEFAULT')


        context.scene.cursor.location = saved_location
        return {"FINISHED"}

#====================#
#  Modifiers
#====================#
    
# Remove Mirror
class SFC_OT_remove_mirror(bpy.types.Operator):
    bl_idname = "object.sfc_remove_mirror"
    bl_label = "Remove Mirror modifier"
    bl_description = "REMOVE MIRROR MODIFIER"
    bl_options = {"REGISTER", "UNDO"}

    def execute(self, context):
        act_obj = context.active_object
        obj = context.active_object
        selection = context.selected_objects
        
        for obj in selection:
            context.view_layer.objects.active = obj
            obj.select_set(state=True)
            if obj.modifiers.get('Mirror'):
                bpy.ops.object.modifier_remove(modifier="Mirror")
                   
        for obj in selection:
            context.view_layer.objects.active = act_obj
            act_obj.select_set(state=True)
        
        # self.report({'INFO'}, "Mirror Modifier Removed")
        return {"FINISHED"}    

#Apply_Remove_Modifiers    
class SFC_OT_apply_remove_hide_modifiers(bpy.types.Operator):
    """    MODIFIERS 
    
    CLICK - Show Modifiers Popup
    SHIFT - Apply
    CTRL  - Remove
    ALT    - Show/Hide"""
    bl_idname = "object.apply_remove_hide_modifiers"
    bl_label = "Apply/Remove/Hide Modifiers"
    bl_options = {"REGISTER",'UNDO'}

    def invoke(self, context, event):
        obj = context.active_object
        selection = context.selected_objects
        edit_mode = False
        
        #Apply   
        if event.shift:
            if context.object.mode == "EDIT":
                edit_mode = True
                bpy.ops.object.mode_set(mode = 'OBJECT')
                
            for obj in selection:
                obj.select_set(state=True)
                if obj.modifiers:
                    context.view_layer.objects.active=obj
                    for mod in obj.modifiers :
                        if mod.show_viewport == False :
                            bpy.ops.object.modifier_remove(modifier=mod.name)
                        else:
                            bpy.ops.object.modifier_apply(apply_as='DATA', modifier=mod.name) 
                
            if edit_mode == True:
                bpy.ops.object.mode_set(mode = 'EDIT') 
            
            # self.report({'INFO'}, "Modifiers Applied")
            
        
        #Remove
        elif event.ctrl:
            for obj in selection:
                obj.select_set(state=True)
                if obj.modifiers:
                    context.view_layer.objects.active=obj
                    for mod in obj.modifiers :
                        bpy.ops.object.modifier_remove(modifier=mod.name)  
            # self.report({'INFO'}, "Modifiers Removed")
        #Hide
        elif event.alt:
            for obj in selection:
                obj.select_set(state=True)
                if obj.modifiers:
                    context.view_layer.objects.active=obj
                    for mod in obj.modifiers :
                        obj.modifiers[mod.name].show_viewport = not obj.modifiers[mod.name].show_viewport 

        
        #Show Popup Modifiers
        else:
            bpy.ops.sfc.modifiers_popup('INVOKE_DEFAULT')
              
                       
        return {"FINISHED"} 

#Apply Mirror Modifier
class SFC_OT_apply_mirror_modifiers(bpy.types.Operator):
    bl_idname = "object.sfc_apply_mirror_modifiers"
    bl_label = "Sfc Apply Mirror Modifiers"
    bl_description = "APPLY MIRROR MODIFIER"
    bl_options = {"REGISTER","UNDO"}

    @classmethod
    def poll(cls, context):
        return True

    def execute(self, context):
        obj = context.active_object
        selection = context.selected_objects
        edit_mode = False
        
        #Apply   
        if context.object.mode == "EDIT":
            edit_mode = True
            bpy.ops.object.mode_set(mode = 'OBJECT')
        
        for obj in selection:
            obj.select_set(state=True)
            if obj.modifiers:
                context.view_layer.objects.active=obj
                for mod in obj.modifiers :
                    if "Mirror" in obj.modifiers:
                        obj.select_set(state=True)
                        bpy.ops.object.modifier_apply(apply_as='DATA', modifier="Mirror")
                       
        if edit_mode == True:
            bpy.ops.object.mode_set(mode = 'EDIT')  
        
        # self.report({'INFO'}, "Mirror Modifier Applied")
        return {"FINISHED"}

#Hide Mirror Modifier
class SFC_OT_hide_mirror_modifier(bpy.types.Operator):
    bl_idname = "object.sfc_hide_mirror_modifier"
    bl_label = "Sfc Hide Mirror Modifier"
    bl_description = "HIDE MIRROR MODIFIER"
    bl_options = {"REGISTER","UNDO"}

    def execute(self, context):
        obj = context.active_object
        selection = context.selected_objects
     
        for obj in selection:
            obj.select_set(state=True)
            if obj.modifiers:
                context.view_layer.objects.active=obj
                if "Mirror" in obj.modifiers:
                    obj.select_set(state=True)
                    obj.modifiers["Mirror"].show_viewport = not obj.modifiers["Mirror"].show_viewport                    
        return {"FINISHED"}
    
    
#Apply Modifiers
class SFC_OT_apply_modifiers(bpy.types.Operator):
    bl_idname = "object.sfc_apply_modifiers"
    bl_label = "Sfc Apply Modifiers"
    bl_description = "APPLY MODIFIERS"
    bl_options = {"REGISTER","UNDO"}

    def execute(self, context):
        obj = context.active_object
        selection = context.selected_objects
        edit_mode = False
        
        #Apply   
        if context.object.mode == "EDIT":
            edit_mode = True
            bpy.ops.object.mode_set(mode = 'OBJECT')
                
        for obj in selection:
            obj.select_set(state=True)
            if obj.modifiers:
                context.view_layer.objects.active=obj
                for mod in obj.modifiers :
                    if mod.show_viewport == False :
                        bpy.ops.object.modifier_remove(modifier=mod.name)
                    else:
                        bpy.ops.object.modifier_apply(apply_as='DATA', modifier=mod.name) 
        
        if edit_mode == True:
            bpy.ops.object.mode_set(mode = 'EDIT')  
            
        # self.report({'INFO'}, "Modifiers applied")
        return {"FINISHED"}

#Remove Modifiers        
class SFC_OT_remove_modifiers(bpy.types.Operator):
    bl_idname = "object.sfc_remove_modifiers"
    bl_label = "Sfc Remove Modifiers"
    bl_description = "REMOVE MODIFIERS"
    bl_options = {"REGISTER","UNDO"}

    @classmethod
    def poll(cls, context):
        return True

    def execute(self, context):
        obj = context.active_object
        selection = context.selected_objects
        edit_mode = False
        
        #Apply   
        if context.object.mode == "EDIT":
            edit_mode = True
            bpy.ops.object.mode_set(mode = 'OBJECT')
                
        for obj in selection:
            obj.select_set(state=True)
            if obj.modifiers:
                context.view_layer.objects.active=obj
                for mod in obj.modifiers :
                    bpy.ops.object.modifier_remove(modifier=mod.name)
        
        if edit_mode == True:
            bpy.ops.object.mode_set(mode = 'EDIT')   
        
        # self.report({'INFO'}, "Modifiers Removed")
        return {"FINISHED"}

#Hide Modifiers
class SFC_OT_hide_modifiers(bpy.types.Operator):
    bl_idname = "object.sfc_hide_modifiers"
    bl_label = "Sfc Hide Modifiers"
    bl_description = "HIDE MODIFIERS"
    bl_options = {"REGISTER","UNDO"}

    @classmethod
    def poll(cls, context):
        return True

    def execute(self, context):
        obj = context.active_object
        selection = context.selected_objects
     
        for obj in selection:
            obj.select_set(state=True)
            if obj.modifiers:
                context.view_layer.objects.active=obj
                for mod in obj.modifiers :
                    obj.modifiers[mod.name].show_viewport = not obj.modifiers[mod.name].show_viewport                    
        return {"FINISHED"}


    
#====================#
#  Subdiv Booleans
#====================#

#Prepare Subdiv booleans
def Prepare_Mesh_Subdiv_Booleans():
    obj = bpy.context.active_object
    for obj in bpy.context.selected_objects:
        bpy.context.view_layer.objects.active = obj
        
        if bpy.context.object.mode == "OBJECT":
            bpy.ops.object.mode_set(mode = 'EDIT')
            bpy.ops.mesh.select_all(action='DESELECT')
            bpy.ops.mesh.edges_select_sharp()
        
        bpy.ops.mesh.mark_sharp()
        bpy.ops.transform.edge_crease(value=1)
        bpy.ops.object.mode_set(mode = 'OBJECT')

        #shading smooth
        if bpy.context.object.mode == "OBJECT":
            bpy.ops.object.shade_smooth()

        elif bpy.context.object.mode == "EDIT":
            bpy.ops.object.mode_set(mode='OBJECT')
            bpy.ops.object.shade_smooth()
            bpy.ops.object.mode_set(mode='EDIT')

        bpy.context.object.data.use_auto_smooth = True
        bpy.context.object.data.auto_smooth_angle = 0.523599

        # for mod in obj.modifiers:
        #     if mod.type == 'SUBSURF' is None:

        # for mod in obj.modifiers:
        #     if not mod.type == 'SUBSURF':
        # for mod in [m for m in obj.modifiers if m.type == 'MIRROR' is None]:
        subsurf_modifier = obj.modifiers.new("Subsurf", 'SUBSURF')
        subsurf_modifier.levels = 2
        subsurf_modifier.name = "Subsurf - %s" % (subsurf_modifier.levels)
        subsurf_modifier.show_on_cage = True
        subsurf_modifier.show_only_control_edges = True


#Prepare Subdiv booleans
class SFC_OT_subdiv_booleans_prepare(bpy.types.Operator):
    """    SUBDIV BOOLEANS
    
    CLICK - Prepare for Subdiv Booleans
    SHIFT - Add and Apply
    CTRL  - Clean Mesh"""
    bl_idname = "object.sfc_subdiv_booleans_prepare"
    bl_label = "Subdiv Booleans Prepare"
    bl_options = {"REGISTER", "UNDO"}

    @classmethod
    def poll(cls, context):
        return context.object is not None and context.object.type == 'MESH'

    def invoke(self, context, event):
        obj = context.active_object
        
        if context.object.mode == "EDIT":
            self.mode = 'EDIT'
            bpy.ops.object.mode_set(mode = 'OBJECT')
        else:
            self.mode = 'OBJECT'
        
        if event.shift:
            for obj in context.selected_objects:
                context.view_layer.objects.active = obj
                if "Subsurf" in obj.modifiers: 
                    bpy.ops.object.modifier_apply(apply_as='DATA', modifier="Subsurf")
                else:
                    Prepare_Mesh_Subdiv_Booleans()
                    bpy.ops.object.modifier_apply(apply_as='DATA', modifier="Subsurf")
            # self.report({'INFO'}, "Subdiv Booleans Added and Applied")

        elif event.ctrl:
            for obj in context.selected_objects:
                context.view_layer.objects.active = obj
                if "Subsurf" in obj.modifiers: 
                    bpy.ops.object.modifier_remove(modifier="Subsurf")

                bpy.ops.object.mode_set(mode = 'EDIT')
                bpy.ops.mesh.select_all(action='SELECT')
                bpy.ops.transform.edge_crease(value=-1)
                bpy.ops.mesh.mark_sharp(clear=True)
                bpy.ops.mesh.select_all(action='DESELECT')
                bpy.ops.object.mode_set(mode = 'OBJECT')
                context.object.data.use_auto_smooth = False
                bpy.ops.shading.flat()
                # self.report({'INFO'}, "Subdiv Booleans Removed")

        else:
            Prepare_Mesh_Subdiv_Booleans()
            # self.report({'INFO'}, "Subdiv Booleans Added")

        
        if self.mode == 'EDIT':
            bpy.ops.object.mode_set(mode = 'EDIT')
        
            
        return {"FINISHED"}


#Clean Subdiv booleans
class SFC_OT_clean_subdiv_booleans_prepare(bpy.types.Operator):
    bl_idname = "object.sfc_clean_subdiv_booleans"
    bl_label = "Clean Subdiv Booleans"
    bl_description = "REMOVE SUBDIV MODIFIER"
    bl_options = {"REGISTER", "UNDO"}

    @classmethod
    def poll(cls, context):
        return context.object is not None and context.object.type == 'MESH'

    def invoke(self, context, event):
        obj = context.active_object
        
        for obj in context.selected_objects:
            context.view_layer.objects.active = obj
            
            
            if context.object.mode == "EDIT":
                self.mode = 'EDIT'
                bpy.ops.object.mode_set(mode = 'OBJECT')
            else:
                self.mode = 'OBJECT'
            
            if "Subsurf" in obj.modifiers: 
                bpy.ops.object.modifier_remove(modifier="Subsurf")

            bpy.ops.object.mode_set(mode = 'EDIT')
            bpy.ops.mesh.select_all(action='SELECT')
            bpy.ops.transform.edge_crease(value=-1)
            bpy.ops.mesh.mark_sharp(clear=True)
            bpy.ops.mesh.select_all(action='DESELECT')
            bpy.ops.object.mode_set(mode = 'OBJECT')
            context.object.data.use_auto_smooth = False
            bpy.ops.shading.flat()
        
        if self.mode == 'EDIT':
            bpy.ops.object.mode_set(mode = 'EDIT')
        
        # self.report({'INFO'}, "Subdiv Booleans Removed")
        return {"FINISHED"}
    

#====================#
#  Auto Mirror
#====================#

# Auto Mirror Popup
class SFC_PT_auto_mirror_popup(bpy.types.Operator):
    bl_idname = "object.sfc_auto_mirror_popup"
    bl_label = "Auto Mirror Popup"
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return context.active_object is not None

    def check(self, context):

        return True

    def execute(self, context):
        return {'FINISHED'}

    def draw(self, context):
        bpy.types.BisectMirror.draw(self, context)

    def invoke(self, context, event):
        # if bpy.app.version == (2, 78, 0):
        #     dpi_value = context.preferences.system.dpi
        #     return context.window_manager.invoke_props_dialog(self, width=dpi_value * 400, height=100)
        #
        # elif bpy.app.version >= (2, 79, 0):
        dpi_value = context.preferences.view.ui_scale
        return context.window_manager.invoke_props_dialog(self, width=dpi_value * 400, height=100)


class SFC_OT_auto_mirror(bpy.types.Operator):
    """    AUTO MIRROR 
    
    CLICK - Add
    SHIFT - Apply
    CTRL  - Remove
    ALT    - Show/Hide
    CTRL+SHIFT - Auto Mirror Popup"""
    bl_idname = "object.sfc_auto_mirror"
    bl_label = "Auto Mirror"
    bl_options = {"REGISTER", "UNDO"}

    @classmethod
    def poll(cls, context):
        return context.object is not None and context.object.type == 'MESH'
    
    def invoke(self, context, event):
        obj = context.active_object
        for obj in context.selected_objects:
            context.view_layer.objects.active = obj

            #Popup
            if event.shift and event.ctrl:
                bpy.ops.object.sfc_auto_mirror_popup('INVOKE_DEFAULT')

            #apply
            elif event.shift:
                for mod in [m for m in obj.modifiers if m.type == 'MIRROR']:
                    if context.object.mode == "EDIT":
                        bpy.ops.object.mode_set(mode = 'OBJECT')
                        bpy.ops.object.modifier_apply( modifier = mod.name )
                        bpy.ops.object.mode_set(mode = 'EDIT')
                    else:
                        bpy.ops.object.modifier_apply( modifier = mod.name )
                # self.report({'INFO'}, "Auto Mirror Applied")
            
            #Remove
            elif event.ctrl:
                for mod in [m for m in obj.modifiers if m.type == 'MIRROR']:
                    bpy.ops.object.modifier_remove( modifier = mod.name )
                # self.report({'INFO'}, "Auto Mirror Removed")
            
            #Hide
            elif event.alt:
                for mod in [m for m in obj.modifiers if m.type == 'MIRROR']:
                    obj.modifiers[mod.name].show_viewport = not obj.modifiers[mod.name].show_viewport
            
            #Add        
            else:
                if not obj.type == 'MIRROR':
                    if context.object.mode == "EDIT":
                        bpy.ops.object.mode_set(mode = 'OBJECT')
                        bpy.ops.object.automirror() 
                        bpy.ops.object.mode_set(mode = 'EDIT')
                        
                    elif context.object.mode == "OBJECT":
                        bpy.ops.object.automirror() 
                        bpy.ops.object.mode_set(mode = 'OBJECT')  
                        
                    else:
                        bpy.ops.object.automirror() 
                    # self.report({'INFO'}, "Auto Mirror Added")
                         
        return {"FINISHED"}

#==============================#
#  Activate snap Utility Line
#==============================#
class SFC_OT_activate_snap_utilities_line(bpy.types.Operator):
    bl_idname = "object.sfc_activate_line"
    bl_label = "Activate Snap Utilities Line"
    bl_description = "ACTIVATE SNAP UTILITIES LINE"
    bl_options = {"REGISTER"}

    def execute(self, context):
        bpy.ops.wm.addon_enable(module="mesh_snap_utilities_line")
        bpy.ops.wm.save_userpref()
        # self.report({'INFO'}, "Snap Utilities Activated")
        return {"FINISHED"}    
    

#====================#
#  Toggle Smooth
#====================#



class SFC_OT_toggle_smooth(bpy.types.Operator):
    """    AUTOSMOOTH

    CLICK - AutoSmooth ON
    CTRL  - AutoSmooth OFF / Shade Flat
    SHIFT  - AutoSmooth OFF / Shade Smooth"""
    bl_idname = "object.sfc_toggle_smooth"
    bl_label = "Toggle Smooth"
    bl_options = {'REGISTER', 'UNDO'}
    
    @classmethod
    def poll(cls, context):
        return context.object is not None and context.object.type == 'MESH'
    
    mode : StringProperty(default="")


    def invoke(self, context, event):
        self.auto_smooth = get_addon_preferences().auto_smooth_value

        act_obj = context.active_object

        for obj in context.selected_objects:
            context.view_layer.objects.active=obj
            obj.select_set(state=True)

            # if len(context.selected_objects) == 1:
            if context.object.mode == "EDIT":
                self.mode = 'EDIT'
                bpy.ops.object.mode_set(mode='OBJECT')
            else:
                self.mode = 'OBJECT'
            # else:
            #     bpy.ops.object.mode_set(mode='OBJECT')


            if event.shift:
                context.object.data.use_auto_smooth = False

                bpy.ops.object.shade_smooth()

            elif event.ctrl:
                context.object.data.use_auto_smooth = False

                bpy.ops.object.shade_flat()

            else:
                context.object.data.use_auto_smooth = True
                # context.object.data.auto_smooth_angle = 1.0472
                context.object.data.auto_smooth_angle = radians(self.auto_smooth)
                bpy.ops.object.shade_smooth()


            if self.mode == 'EDIT':
                bpy.ops.object.mode_set(mode = 'EDIT')



        return {'FINISHED'} 
         
#=================================================================#
#  Edit Mode
#=================================================================#



# class Curve_Radius(bpy.types.Operator):
#     bl_idname = 'object.curve_radius'
#     bl_label = ""
#     bl_options = {'REGISTER'}
#
#     @classmethod
#     def poll(cls, context):
#         return True
#
#     def execute(self, context):
#         bpy.ops.curve.radius_set('INVOKE_DEFAULT', True)
#
#         return {'FINISHED'}



#====================#
#  Select Ngons
#====================#
class SFC_OT_ngons_select(Operator):
    """    SELECT NGONS    """
    bl_idname = "object.sfc_facetype_select"
    bl_label = "SELECT NGONS"
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return context.object is not None and context.object.type == 'MESH'

    def execute(self, context):
        select_mode : StringProperty()
 
        #Go in edit mode and Deselect
        bpy.ops.object.mode_set(mode='EDIT')
        context.tool_settings.mesh_select_mode=(False, False, True)
        bpy.ops.mesh.select_all(action='DESELECT')
        
 
        #Select Faces
        if tuple(context.tool_settings.mesh_select_mode) == (True, False, False):
            self.select_mode = "VERT"
        elif tuple(context.tool_settings.mesh_select_mode) == (False, True, False):
            self.select_mode = "EDGE"
        elif tuple(context.tool_settings.mesh_select_mode) == (False, False, True):
            self.select_mode = "FACE"
 
        context.tool_settings.mesh_select_mode=(False, False, True)
 
        #Select Faces with more than 4 vertices
        bpy.ops.mesh.select_face_by_sides(number=4, type='GREATER')
        bpy.ops.mesh.select_mode(type=self.select_mode) 
        return {'CANCELLED'}    

#====================#
#  Clean Faces
#====================#    
class SFC_OT_clean_faces(bpy.types.Operator):
    bl_idname = "object.sfc_clean_faces"
    bl_label = "Clean Faces"
    bl_description = "CLEAN FACES"
    bl_options = {"REGISTER", "UNDO"}

    @classmethod
    def poll(cls, context):
        return context.object is not None and context.object.type == 'MESH'

    def execute(self, context):
        bpy.ops.mesh.quads_convert_to_tris(quad_method='BEAUTY', ngon_method='BEAUTY')
        bpy.ops.mesh.tris_convert_to_quads(shape_threshold=2.18166)
        return {"FINISHED"}



#====================#
#  Sharps
#====================#  

#Sharp All
class SFC_OT_sharp_all(bpy.types.Operator):
    """    SHARPS (Creases, Sharps, Bevel Weight)
    
    CLICK - Add
    CTRL  - Remove
    ALT    - Select Sharps by Angle
    SHIFT  - Clean normals by adding sharp on faces    """
    bl_idname = "object.sfc_sharp_all"
    bl_label = "Sharp All"
    bl_options = {"REGISTER", "UNDO"}
    
    @classmethod
    def poll(cls, context):
        return context.object is not None and context.object.type == 'MESH'

    bevel_weight : BoolProperty(
        name="Center object",
        description="Place the object at the center of the scene",
        default=True)

    crease : BoolProperty(
        name="Center object",
        description="Place the object at the center of the scene",
        default=True)

    sharps : BoolProperty(
        name="Center object",
        description="Place the object at the center of the scene",
        default=True)

    seams  : BoolProperty(
        name="Center object",
        description="Place the object at the center of the scene",
        default=False)

    select_sharps : BoolProperty(
        name="Select Sharps",
        description="Select Sharps Edges",
        default=True)

    event_enum : EnumProperty(
        items=(('shift', "Shift", ""),
               ('ctrl', "Ctrl", ""),
               ('alt', "Alt", "")),
        default='ctrl')



    def draw(self, context):
        layout = self.layout

        # if self.add_sharps :
        layout.prop(self, "bevel_weight", text="Bevel Weight")
        layout.prop(self, "crease", text="Crease")
        layout.prop(self, "sharps", text="Sharps")
        layout.prop(self, "seams", text="Seams")
        layout.prop(self, "select_sharps", text="Mark Sharps")


        # layout.prop(mesh, "edges_select_sharp", text="Seams")



    def execute(self, context):
        # self.add_sharps =

        obj_mode = False
        # if self.event_enum == 'shift' and self.event_enum == 'ctrl':
        #     obj = context.object
        #     me = obj.data
        #     bm = bmesh.from_edit_mesh(me)
        #     edge_selection = False
        #
        #     if any([e.select for e in bm.edges]):
        #         edge_selection = True
        #
        #
        #     #clean
        #     bpy.ops.transform.edge_bevelweight(value=-1)
        #     bpy.ops.mesh.mark_sharp(clear=True)
        #     bpy.ops.transform.edge_crease(value=-1)
        #     bpy.ops.mesh.mark_seam(clear=True)
        #
        #     bpy.ops.mesh.select_all(action='DESELECT')
        #     bpy.ops.mesh.select_mode(use_extend=False, use_expand=False, type='EDGE')
        #     bpy.ops.mesh.edges_select_sharp(True)
        #
        #     #add
        #     if not edge_selection:
        #         bpy.ops.mesh.select_all(action='SELECT')
        #         if self.select_sharps:
        #             bpy.ops.mesh.select_all(action='DESELECT')
        #             bpy.ops.mesh.select_mode(use_extend=False, use_expand=False, type='EDGE')
        #             bpy.ops.mesh.edges_select_sharp(True)
        #
        #     if self.bevel_weight:
        #         bpy.ops.transform.edge_bevelweight(value=1)
        #     if self.sharps:
        #         bpy.ops.mesh.mark_sharp()
        #     if self.crease:
        #         bpy.ops.transform.edge_crease(value=1)
        #     if self.seams:
        #         bpy.ops.mesh.mark_seam(clear=False)
        #
        #     bpy.ops.mesh.select_all(action='DESELECT')


        # clean normals
        if self.event_enum == 'shift':
            obj = context.object
            me = obj.data
            bm = bmesh.from_edit_mesh(me)
            face_selection = False

            if any([e.select for e in bm.faces]):
                face_selection = True

            if face_selection:
                self.crease = False
                self.bevel_weight = False
                bpy.ops.mesh.select_similar(type='COPLANAR', threshold=0.01)
                bpy.ops.mesh.region_to_loop()
                bpy.ops.mesh.mark_sharp()
                obj_mode = True

        #Remove
        elif self.event_enum == 'ctrl':
            obj = context.object
            me = obj.data
            bm = bmesh.from_edit_mesh(me)
            edge_selection = False

            if any([e.select for e in bm.edges]):
                edge_selection = True

            if not edge_selection:
                bpy.ops.mesh.select_all(action='SELECT')
            if self.bevel_weight:
                bpy.ops.transform.edge_bevelweight(value=-1)
            if self.sharps:
                bpy.ops.mesh.mark_sharp(clear=True)
            if self.crease:
                bpy.ops.transform.edge_crease(value=-1)
            if self.seams:
                bpy.ops.mesh.mark_seam(clear=True)

            bpy.ops.mesh.select_all(action='DESELECT')

        #Select
        elif self.event_enum == 'alt':
            bpy.ops.mesh.select_all(action='DESELECT')
            bpy.ops.mesh.select_mode(use_extend=False, use_expand=False, type='EDGE')
            bpy.ops.mesh.edges_select_sharp(True)

            # self.select_sharps = True

        #Add
        else:
            obj = context.object
            me = obj.data
            bm = bmesh.from_edit_mesh(me)
            edge_selection = False

            if any([e.select for e in bm.edges]):
                edge_selection = True

            if not edge_selection:
                bpy.ops.mesh.select_all(action='SELECT')
                if self.select_sharps:
                    bpy.ops.mesh.select_all(action='DESELECT')
                    bpy.ops.mesh.select_mode(use_extend=False, use_expand=False, type='EDGE')
                    bpy.ops.mesh.edges_select_sharp(True)

            if self.bevel_weight:
                bpy.ops.transform.edge_bevelweight(value=1)
            if self.sharps:
                bpy.ops.mesh.mark_sharp()
            if self.crease:
                bpy.ops.transform.edge_crease(value=1)
            if self.seams:
                bpy.ops.mesh.mark_seam(clear=False)

            bpy.ops.mesh.select_all(action='DESELECT')


        if obj_mode :
            bpy.ops.object.mode_set(mode='OBJECT')
        return {"FINISHED"}


    def invoke(self, context, event) :
        # if not self.select_sharps
        self.mode = context.object.mode
        if event.alt:
            self.event_enum = 'alt'

        elif event.ctrl:
            self.event_enum = 'ctrl'

        else:
            self.event_enum = 'shift'

        return self.execute(context)

#UnSharp All
class SFC_OT_unsharp_all(bpy.types.Operator):
    bl_idname = "object.sfc_unsharp_all"
    bl_label = "UnSharp All"
    bl_description = "UnSharps"
    bl_options = {"REGISTER"}
    
    @classmethod
    def poll(cls, context):
        return context.object is not None and context.object.type == 'MESH'
    
    def execute(self, context):
        bpy.ops.mesh.mark_sharp(clear=True)
        bpy.ops.transform.edge_crease(value=-1)
        bpy.ops.transform.edge_bevelweight(value=-1)

        return {"FINISHED"}
    
#Bevel Weight        
class SFC_OT_bevel_weight(bpy.types.Operator):
    """    BEVEL WEIGHT
    
    CLICK - Add
    SHIFT - Add and Edit
    CTRL  - Remove
    ALT    - Select    """
    bl_idname = "object.sfc_bevel_weight"
    bl_label = "SFC Bevel Weight"
    bl_options = {"REGISTER"}
    
    @classmethod
    def poll(cls, context):
        return context.object is not None and context.object.type == 'MESH'
    
    def invoke(self, context, event):
        obj = context.object

        #Add and Edit
        if event.shift:
            if tuple(context.tool_settings.mesh_select_mode) == (False, False, True):
                bpy.ops.mesh.region_to_loop()
                bpy.ops.transform.edge_bevelweight('INVOKE_DEFAULT')
                bpy.ops.mesh.select_mode(use_extend=False, use_expand=False, type='FACE')
            else:
                bpy.ops.transform.edge_bevelweight('INVOKE_DEFAULT')
            
        #Remove
        elif event.ctrl:
            bpy.ops.transform.edge_bevelweight(value=-1)

        #Select
        elif event.alt:
            bpy.ops.mesh.select_all(action='DESELECT')
            obj = context.object
            me = obj.data
            bm = bmesh.from_edit_mesh(me)
            cr = bm.edges.layers.bevel_weight.verify()

            me.show_edge_bevel_weight = True

            for e in bm.edges:
                if e[cr]:
                    e.select = True

            bmesh.update_edit_mesh(me, False, False)

        #Add
        else:
            if tuple(context.tool_settings.mesh_select_mode) == (False, False, True):
                bpy.ops.mesh.region_to_loop()
                bpy.ops.transform.edge_bevelweight(value=1)
                bpy.ops.mesh.select_mode(use_extend=False, use_expand=False, type='FACE')
            else:
                bpy.ops.transform.edge_bevelweight(value=1)

        return {"FINISHED"}
    
        
# Sharps Edges 
class SFC_OT_sharps(bpy.types.Operator):
    """    SHARP EDGES
    
    CLICK - Add
    CTRL  - Remove
    ALT    - Select    """
    bl_idname = "object.sfc_sharps"
    bl_label = "SFC Sharps"
    bl_options = {"REGISTER"}
    
    @classmethod
    def poll(cls, context):
        return context.object is not None and context.object.type == 'MESH'
    
    def invoke(self, context, event):

        #Remove
        if event.ctrl:
            bpy.ops.mesh.mark_sharp(clear=True, use_verts=False)

        #Select
        elif event.alt:
            bpy.ops.mesh.select_all(action='DESELECT')
            obj = context.edit_object
            me = obj.data
            bm = bmesh.from_edit_mesh(me)

            for e in bm.edges:
                if not e.smooth:
                    e.select = True

            bmesh.update_edit_mesh(me, False)

        #Add
        else:
            bpy.ops.mesh.mark_sharp(clear=False, use_verts=False)  
        return {"FINISHED"}

# UnSharps Edges 
class SFC_OT_unsharps(bpy.types.Operator):
    bl_idname = "object.sfc_unsharps"
    bl_label = "UnSharps"
    bl_description = "UNSHARP EDGES"
    bl_options = {"REGISTER"}
    
    @classmethod
    def poll(cls, context):
        return context.object is not None and context.object.type == 'MESH'
    
    def execute(self, context):
        bpy.ops.mesh.mark_sharp(clear=True, use_verts=False)
        return {"FINISHED"}
    
# Creases
class SFC_OT_creases(bpy.types.Operator):
    """    CREASES
    
    CLICK - Add
    SHIFT - Add and Edit
    CTRL  - Remove
    ALT    - Select    """
    bl_idname = "object.sfc_creases"
    bl_label = "SFC Creases"
    bl_options = {"REGISTER"}
    
    @classmethod
    def poll(cls, context):
        return context.object is not None and context.object.type == 'MESH'
    
    def invoke(self, context, event):
        obj = context.object

        #Add and Edit
        if event.shift:
            bpy.ops.transform.edge_crease('INVOKE_DEFAULT')

        #Remove
        elif event.ctrl:
            bpy.ops.transform.edge_crease(value=-1)

        # Select
        elif event.alt:
            bpy.ops.mesh.select_all(action='DESELECT')
            obj = context.object
            me = obj.data
            bm = bmesh.from_edit_mesh(me)
            cr = bm.edges.layers.crease.verify()

            me.show_edge_crease = True

            for e in bm.edges:
                if e[cr]:
                    e.select = True

            bmesh.update_edit_mesh(me, False, False)

        #Add
        else:
            bpy.ops.transform.edge_crease(value=1)


        return {"FINISHED"}

# Seams
class SFC_OT_seams(bpy.types.Operator):
    """    SEAMS

    CLICK - Add
    CTRL  - Remove
    ALT    - Select    """
    bl_idname = "object.sfc_seams"
    bl_label = "SFC SEAMS"
    bl_options = {"REGISTER"}

    @classmethod
    def poll(cls, context):
        return context.object is not None and context.object.type == 'MESH'

    def invoke(self, context, event):

        #Add and Edit
        if event.shift:
            bpy.ops.mesh.select_similar(True,type='SEAM', threshold=0.01)

        #Remove
        if event.ctrl:
            bpy.ops.mesh.mark_seam(True, clear=True)

        #Select
        elif event.alt:
            bpy.ops.mesh.select_all(action='DESELECT')
            obj = context.edit_object
            me = obj.data
            bm = bmesh.from_edit_mesh(me)

            for e in bm.edges:
                if e.seam:
                    e.select = True

            bmesh.update_edit_mesh(me, False)

        #Add
        else:
            bpy.ops.mesh.mark_seam(True, clear=False)

        return {"FINISHED"}

# UnSharps Edges
class SFC_OT_unseam(bpy.types.Operator):
    bl_idname = "object.sfc_unseam"
    bl_label = "UnSeam"
    bl_description = "UNSEAM EDGES"
    bl_options = {"REGISTER"}

    @classmethod
    def poll(cls, context):
        return context.object is not None and context.object.type == 'MESH'

    def execute(self, context):
        bpy.ops.mesh.mark_seam(True, clear=True)
        return {"FINISHED"}

#Select sharps 
class SFC_OT_select_sharp(bpy.types.Operator):
    """    SELECT SHARPS
    
    CLICK - Select sharps bye Angle
    SHIFT - Select Bevel Weight
    CTRL  - Select Crease
    ALT   - Select Marked Sharps
    CTRL + SHIFT - Select Marked Seams    """
    bl_idname = "object.sfc_select_sharp"
    bl_label = "Select Sharp"
    bl_options = {"REGISTER"}
    
    @classmethod
    def poll(cls, context):
        return context.object is not None and context.object.type == 'MESH'
        
    def invoke(self, context, event):
        bpy.ops.mesh.select_all(action='DESELECT')

        if event.ctrl and event.shift:
            obj = context.object
            me = obj.data
            bm = bmesh.from_edit_mesh(me)
            cr = bm.edges.layers.uv.verify()


            me.show_uv = True

            for e in bm.edges:
                if e[cr]:
                    e.select = True

            bmesh.update_edit_mesh(me, False, False)

        elif event.shift:
            obj = context.object
            me = obj.data
            bm = bmesh.from_edit_mesh(me)
            cr = bm.edges.layers.bevel_weight.verify()

            me.show_edge_bevel_weight = True

            for e in bm.edges:
                if e[cr]:
                    e.select = True

            bmesh.update_edit_mesh(me, False, False)


        elif event.ctrl:

            obj = context.object
            me = obj.data
            bm = bmesh.from_edit_mesh(me)
            cr = bm.edges.layers.crease.verify()

            me.show_edge_crease = True

            for e in bm.edges:
                if e[cr]:
                    e.select = True

            bmesh.update_edit_mesh(me, False, False)

        elif event.alt:
            obj = context.edit_object
            me = obj.data
            bm = bmesh.from_edit_mesh(me)

            for e in bm.edges:
                if not e.smooth:
                    e.select = True

            bmesh.update_edit_mesh(me, False)


        else:    
            bpy.ops.mesh.select_all(action='DESELECT')
            bpy.ops.mesh.select_mode(use_extend=False, use_expand=False, type='EDGE')
            bpy.ops.mesh.edges_select_sharp(True)
            
        return {"FINISHED"}


#Show/hide Sharps 
class SFC_OT_show_hide_sharps(bpy.types.Operator):
    """    SHOW/HIDE SHARPS
    
    CLICK - Show
    CTRL  - Hide    """
    bl_idname = "object.sfc_show_hide_sharps"
    bl_label = "Show/Hide Sharps"
    bl_options = {"REGISTER"}
    
    @classmethod
    def poll(cls, context):
        return context.object is not None and context.object.type == 'MESH'
    
    def invoke(self, context, event):
        if event.ctrl:
            context.space_data.overlay.show_edge_bevel_weight = False
            context.space_data.overlay.show_edge_crease = False
            context.space_data.overlay.show_edge_sharp = False
        else:
            context.space_data.overlay.show_edge_bevel_weight = True
            context.space_data.overlay.show_edge_crease = True
            context.space_data.overlay.show_edge_sharp = True
        return {"FINISHED"}            

#hide Sharps 
class SFC_OT_hide_sharps(bpy.types.Operator):
    bl_idname = "object.sfc_hide_sharps"
    bl_label = "Hide Sharps"
    bl_description = "HIDE SHARPS"
    bl_options = {"REGISTER"}

    def invoke(self, context, event):
        context.space_data.overlay.show_edge_bevel_weight = False
        context.space_data.overlay.show_edge_crease = False
        context.space_data.overlay.edge_sharp = False

        return {"FINISHED"}  


#====================#
#  #LapRelax
#====================#   
class SFC_OT_laprelax(bpy.types.Operator):
	bl_idname = "mesh.sfc_laprelax"
	bl_label = "LapRelax"
	bl_description = "SMOOTH THE MESH, KEEP THE VOLUME"
	bl_options = {'REGISTER'}
	
	Repeat : bpy.props.IntProperty(
		name = "Repeat", 
		description = "Repeat how many times",
		default = 1,
		min = 1,
		max = 100)

	@classmethod
	def poll(cls, context):
		return context.object is not None and context.object.type == 'MESH'

	def invoke(self, context, event):
		
		# smooth #Repeat times
		for i in range(self.Repeat):
			self.do_laprelax()
		
		return {'FINISHED'}


	def do_laprelax(self):
	
		context = bpy.context
		region = context.region  
		area = context.area
		selobj = bpy.context.active_object
		mesh = selobj.data
		bm = bmesh.from_edit_mesh(mesh)
		bmprev = bm.copy()
	
		for v in bmprev.verts:
			if v.select:
				tot = Vector((0, 0, 0))
				cnt = 0
				for e in v.link_edges:
					for f in e.link_faces:
						if not(f.select):
							cnt = 1
					if len(e.link_faces) == 1:
						cnt = 1
						break
				if cnt:
					# dont affect border edges: they cause shrinkage
					continue
					
				# find Laplacian mean
				for e in v.link_edges:
					tot += e.other_vert(v).co
				tot /= len(v.link_edges)
				
				# cancel movement in direction of vertex normal
				delta = (tot - v.co)
				if delta.length != 0:
					ang = delta.angle(v.normal)
					deltanor = math.cos(ang) * delta.length
					nor = v.normal
					nor.length = abs(deltanor)
					bm.verts[v.index].co = tot + nor
			
			
		mesh.update()
		bm.free()
		bmprev.free()
		bpy.ops.object.editmode_toggle()
		bpy.ops.object.editmode_toggle()



CLASSES = [SFC_OT_vgroups,
           SFC_OT_fast_rename,
           SFC_OT_parent_asset,
           SFC_OT_replace_mesh_data,
           SFC_OT_fast_intersect,
           SFC_OT_clean_mesh,
           SFC_OT_show_text_options,
           SFC_OT_show_bool_objects,
           SFC_OT_select_hierarchy,
           SFC_OT_project_modal,
           SFC_OT_clean_datas,
           SFC_OT_display_mode,
           SFC_OT_wire_mode,
           SFC_OT_solid_mode,
           SFC_OT_bounds_mode,
           SFC_OT_custom_primitives,
           SFC_OT_primitives,
           SFC_OT_primitives_vertex,
           SFC_OT_remove_mirror,
           SFC_OT_apply_remove_hide_modifiers,
           SFC_OT_apply_mirror_modifiers,
           SFC_OT_hide_mirror_modifier,
           SFC_OT_apply_modifiers,
           SFC_OT_remove_modifiers,
           SFC_OT_hide_modifiers,
           SFC_OT_subdiv_booleans_prepare,
           SFC_OT_clean_subdiv_booleans_prepare,
           SFC_PT_auto_mirror_popup,
           SFC_OT_auto_mirror,
           SFC_OT_activate_snap_utilities_line,
           SFC_OT_toggle_smooth,
           SFC_OT_ngons_select,
           SFC_OT_clean_faces,
           SFC_OT_sharp_all,
           SFC_OT_bevel_weight,
           SFC_OT_sharps,
           SFC_OT_unsharps,
           SFC_OT_creases,
           SFC_OT_seams,
           SFC_OT_unseam,
           SFC_OT_select_sharp,
           SFC_OT_show_hide_sharps,
           SFC_OT_hide_sharps,
           SFC_OT_laprelax,]

def register():
    for cls in CLASSES:
        try:
            bpy.utils.register_class(cls)
        except:
            print(f"{cls.__name__} already registred")


def unregister():
    for cls in CLASSES:
        bpy.utils.unregister_class(cls)