# -*- coding:utf-8 -*-

# Speedflow Add-on
# Copyright (C) 2018 Cedric Lepiller aka Pitiwazou & Legigan Jeremy AKA Pistiwique and Stephen Leger
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

# <pep8 compliant>

import bpy
from mathutils import Vector
from bpy.props import (StringProperty,
                       BoolProperty,
                       FloatVectorProperty,
                       FloatProperty,
                       EnumProperty,
                       IntProperty)



#Add Lattice        
class SFC_OT_add_lattice(bpy.types.Operator):
    """    ADD/EDIT LATTICE

    CLICK - Add Lattice on selection
    SHIFT - Don't take mirror in account
    CTRL  - Apply Transforms
    """
    bl_idname = "sfc.add_lattice"
    bl_label = "Add Lattice"
    bl_options = {"REGISTER","UNDO"}

    def invoke(self, context, event):
    # def execute(self, context):
        SFC = context.window_manager.SFC
        act_obj = context.active_object
        obj_list = [obj for obj in context.selected_objects]
        edit_mode = False
 
        # Remove Vgroup
        for obj in obj_list:
            bpy.context.view_layer.objects.active = obj
            obj.select_set(state=True)
            for vgroup in obj.vertex_groups:
                if vgroup.name == "SpeedLattice_Group":
                    obj.vertex_groups.remove(vgroup)

        # #Check Mirror
        if event.shift:
            for obj in obj_list:
                for mod in obj.modifiers :
                    if mod.type == 'MIRROR':
                        mod.show_viewport = False
 
        #Object 1 selection
        if context.object.mode == 'OBJECT':

            # Create BBox_object
            bpy.ops.object.duplicate_move()
            bpy.ops.object.convert(target='MESH')
            bpy.ops.object.join()
            context.active_object.name= "BBox_Object"

            if len([obj for obj in context.selected_objects ]) >= 2:
                bpy.ops.object.transform_apply(location=False, rotation=True, scale=False)
            else:
                if event.ctrl:
                    bpy.ops.object.transform_apply(location=False, rotation=True, scale=False)
                else:
                    pass
 
        #EDIT
        elif context.object.mode == 'EDIT':
            edit_mode = True

            act_obj.vertex_groups.new("SpeedLattice_Group")

            for vgroup in act_obj.vertex_groups:
                if vgroup.name == "SpeedLattice_Group":
                    bpy.ops.object.vertex_group_assign()

            bpy.ops.mesh.duplicate_move()
            bpy.ops.mesh.separate(type="SELECTED")
            bpy.ops.object.mode_set(mode = 'OBJECT')
            act_obj.select_set(state=True)
            context.view_layer.objects.active = context.selected_objects[0]
            context.active_object.name = "BBox_Object"

            if SFC.sfc_global_orientation :
                bpy.ops.object.transform_apply(location=False, rotation=True, scale=False)
 
        bbox_obj = context.active_object

        #ensure origin is centered on bounding box center
        bpy.ops.object.origin_set(type='ORIGIN_GEOMETRY', center='BOUNDS')
        #create a cube for the bounding box
        location = bbox_obj.location
        bpy.ops.object.add(radius=1, type='LATTICE', location=location)
 
        #our new cube is now the active object, so we can keep track of it in a variable:
        lattice_object = context.active_object
 
        #copy transforms
        lattice_object.dimensions = bbox_obj.dimensions
        lattice_object.rotation_euler = bbox_obj.rotation_euler

        for obj in obj_list:                       
            bpy.context.view_layer.objects.active = obj
            obj.select_set(state=True)
            newMod = obj.modifiers.new(lattice_object.name, 'LATTICE')
            newMod.object = lattice_object
            # if obj.modifiers[0].name != lattice_object.name:
            #     while obj.modifiers[0].name != lattice_object.name:
            #         bpy.ops.object.modifier_move_up(modifier="Lattice")
 
            # Assign Vertex Group
            if edit_mode:
                newMod.vertex_group = "SpeedLattice_Group"
 
        #Affiche le mirror
        if event.shift:
            for obj in obj_list:
                for mod in obj.modifiers :
                    if mod.type == 'MIRROR':
                        mod.show_viewport = True
 
        # on supprime le BBox_Object 
        bbox_object = bpy.data.objects.get("BBox_Object")
        if bbox_object:
            bpy.data.objects.remove(bbox_object, do_unlink = True)
 
        #Make Lattice Active object        
        bpy.ops.object.select_all(action='DESELECT')
        bpy.context.view_layer.objects.active = lattice_object
        lattice_object.select_set(state=True)
 
        bpy.ops.object.mode_set(mode = 'EDIT')  

        del(obj_list[:])                                               
 
        # bpy.context.space_data.cursor_location = (0, 0, 0)
 
        return {"FINISHED"}


# Add Lattice To Selection
class SFC_OT_add_lattice_to_selection(bpy.types.Operator):
    """    LATTICE

    CLICK - Add Lattice To Selection
    SHIFT - Use Global Orientation
    """
    bl_idname = "object.sfc_add_lattice_to_selection"
    bl_label = "Sfc Add Lattice To Selection"
    bl_options = {"REGISTER"}

    def invoke(self, context, event):
        SFC = bpy.context.window_manager.SFC

        if event.shift:
            SFC.sfc_global_orientation = True

        # elif event.alt:
        #     WM.sfc_mirror = True

        bpy.ops.sfc.add_lattice()

        return {"FINISHED"}

# Hide Lattice
def SFC_Hide_Lattice_Modifier(self, context):
    bl_description = "Hide the Lattice Modifier on selected objects"

    act_obj = context.active_object
    obj_list = [obj for obj in bpy.context.selected_objects]

    for obj in obj_list:
        bpy.context.view_layer.objects.active = obj
        for mod in obj.modifiers:
            if mod.type == "LATTICE":
                bpy.context.object.modifiers[mod.name].show_viewport = not bpy.context.object.modifiers[
                    mod.name].show_viewport

    for obj in obj_list:
        obj.select_set(state=True)
    bpy.context.view_layer.objects.active = act_obj
    act_obj.select_set(state=True)

#Remove Lattice Modifier
def SFC_Remove_Lattice_Modifier(self, context):
    act_obj = bpy.context.active_object
    obj_list = [obj for obj in bpy.context.selected_objects]
    
    bpy.ops.object.mode_set(mode = 'OBJECT')
    bpy.ops.object.select_all(action='DESELECT')
    for obj in obj_list:
        bpy.context.view_layer.objects.active = obj
        for mod in obj.modifiers:
            if mod.type == "LATTICE":
                bpy.ops.object.modifier_remove(modifier=mod.name)
        

    for obj in obj_list:
        obj.select_set(state=True)
    bpy.context.view_layer.objects.active = act_obj
    act_obj.select_set(state=True)

    for vgroup in obj.vertex_groups:
        if vgroup.name == "SpeedLattice_Group":
            obj.vertex_groups.remove(vgroup)

    del (obj_list[:])


#Apply Lattice Modifier
def SFC_Apply_Lattice_Modifier(self, context):
    act_obj = bpy.context.active_object
    obj_list = [obj for obj in bpy.context.selected_objects]
    
    bpy.ops.object.mode_set(mode = 'OBJECT')
    bpy.ops.object.select_all(action='DESELECT')
    for obj in obj_list:
        bpy.context.view_layer.objects.active = obj
        for mod in obj.modifiers:
            if mod.type == "LATTICE":
                bpy.ops.object.modifier_apply(apply_as='DATA', modifier=mod.name)

    del(obj_list[:])    



#Apply Lattice on all objects
class SFC_OT_apply_lattice_objects(bpy.types.Operator):
    bl_idname = "object.sfc_apply_lattice_objects"
    bl_label = "Apply Lattice Objects"
    bl_description = "Apply Lattice on all objects and Delete the Lattice"
    bl_options = {"REGISTER","UNDO"}

    @classmethod
    def poll(cls, context):
        return True

    def execute(self, context):
        act_obj = bpy.context.active_object.name
        active_lattice = bpy.context.active_object
        
        bpy.ops.object.mode_set(mode = 'OBJECT')
        bpy.ops.object.select_all(action='DESELECT')
        for obj in bpy.data.objects:
            if obj.type in ["MESH", "CURVE", "FONT", "LATTICE"]:

                lattice = False
                for mod in obj.modifiers:
                    if mod.type == "LATTICE":
                        lattice = True
                        
                    if lattice :
                        bpy.context.view_layer.objects.active = obj
                        obj.select_set(state=True)
                        if act_obj in bpy.context.object.modifiers:
                            bpy.ops.object.modifier_apply(apply_as='DATA', modifier=act_obj)

                            #Remove Vertex Group
                            for vgroup in obj.vertex_groups:
                                if vgroup.name.startswith("S"):
                                    obj.vertex_groups.remove(vgroup)
                        
        bpy.ops.object.select_all(action='DESELECT') 

        bpy.context.view_layer.objects.active = active_lattice
        active_lattice.select_set(state=True)
        bpy.ops.object.delete(use_global=False)
 
        
        return {"FINISHED"}

#Remove Lattice on all objects
class SFC_OT_remove_lattice_objects(bpy.types.Operator):
    bl_idname = "object.sfc_remove_lattice_objects"
    bl_label = "Remove Lattice Objects"
    bl_description = "Remove Lattice on all objects and Delete the Lattice"
    bl_options = {"REGISTER","UNDO"}

    @classmethod
    def poll(cls, context):
        return True

    def execute(self, context):
        act_obj = bpy.context.active_object.name
        active_lattice = bpy.context.active_object

        bpy.ops.object.mode_set(mode = 'OBJECT')
        bpy.ops.object.select_all(action='DESELECT')
        for obj in bpy.data.objects:

            if obj.type in ["MESH", "CURVE", "FONT", "LATTICE"]:

                lattice = False
                for mod in obj.modifiers:
                    if mod.type == "LATTICE":
                        lattice = True
                if lattice :
                    bpy.context.view_layer.objects.active = obj
                    obj.select_set(state=True)
                    if act_obj in bpy.context.object.modifiers:
                        bpy.ops.object.modifier_remove(modifier=act_obj)

                        for vgroup in obj.vertex_groups:
                            if vgroup.name.startswith("S"):
                                obj.vertex_groups.remove(vgroup)

        bpy.ops.object.select_all(action='DESELECT')
        bpy.context.view_layer.objects.active = active_lattice
        active_lattice.select_set(state=True)
        bpy.ops.object.delete(use_global=False)
 
        
        return {"FINISHED"}   

#Connect Object To Lattice
class SFC_OT_connect_lattice(bpy.types.Operator):
    """    CONNECT SELECTION TO LATTICE    """
    bl_idname = "object.sfc_connect_lattice"
    bl_label = "Connect Lattice"
    bl_options = {"REGISTER","UNDO"}

    def execute(self, context):
        lattice = context.active_object
        obj_list = [obj for obj in bpy.context.selected_objects]

        bpy.ops.object.select_all(action='DESELECT')
        for obj in obj_list:
            if obj != context.active_object and(obj.type in ["MESH", "CURVE", "FONT", "LATTICE"]):
                if not "Lattice" in obj.modifiers:
                    newMod = obj.modifiers.new(lattice.name, 'LATTICE')
                    newMod.object = lattice
                bpy.context.view_layer.objects.active = obj
                bpy.ops.object.select_all(action='DESELECT')


        bpy.ops.object.select_all(action='DESELECT')
        # Select Lattice
        bpy.context.view_layer.objects.active = lattice
        lattice.select_set(state=True)

        # detele list
        del (obj_list[:])

        return {"FINISHED"}
        

        

class SFC_OT_edit_lattice(bpy.types.Operator):
    """    LATTICE
    
    CLICK - Show Lattice Popup
    SHIFT - Apply Lattice
    CTRL  - Remove Lattice
    """
    bl_idname = "object.sfc_edit_lattice"
    bl_label = "Edit Lattice"
    bl_options = {"REGISTER", "UNDO"}

    def invoke(self, context, event):
        obj = context.active_object   
        
        if event.shift:
            bpy.ops.object.sfc_apply_lattice_objects()  
        elif event.ctrl:
            bpy.ops.object.sfc_remove_lattice_objects()
        else:
            bpy.ops.object.sfc_lattice_popup('INVOKE_DEFAULT')
            
        return {"FINISHED"}

# class SFC_OT_edit_objects(bpy.types.Operator):
#     """    LATTICE
#
#     CLICK - Show Popup to Edit
#     SHIFT - Apply Lattice
#     CTRL  - Delete Lattice
#     ALT    - Hide Lattice Modifier
#     """
#     bl_idname = "object.sfc_edit_objects"
#     bl_label = "Edit Lattice"
#     bl_options = {"REGISTER", "UNDO"}
#
#     @classmethod
#     def poll(cls, context):
#         return True
#
#     def invoke(self, context, event):
#         obj = context.active_object
#
#         if event.shift :
#             SFC_Apply_Lattice_Modifier(self, context)
#
#         elif event.ctrl:
#             SFC_Remove_Lattice_Modifier(self, context)
#
#         elif event.alt:
#             SFC_Hide_Lattice_Modifier(self, context)
#
#         else:
#             lattice = False
#             for mod in obj.modifiers:
#                 if mod.type == "LATTICE":
#                     lattice = True
#
#                 if lattice:
#                     lattice = bpy.context.object.modifiers[mod.name].object
#                     bpy.context.view_layer.objects.active = lattice
#                     lattice.select_set(state=True)
#                     obj.select_set(state=False)
#                     bpy.ops.object.mode_set(mode = 'EDIT')
#                     bpy.ops.object.sfc_lattice_popup('INVOKE_DEFAULT')
#
#         return {"FINISHED"}


class SFC_OT_edit_objects(bpy.types.Operator):
    """    LATTICE

    CLICK - Show Popup to Edit
    SHIFT - Apply Lattice
    CTRL  - Delete Lattice
    ALT    - Hide Lattice Modifier
    """
    bl_idname = "object.sfc_edit_objects"
    bl_label = "Edit Lattice"
    bl_options = {"REGISTER", "UNDO"}

    @classmethod
    def poll(cls, context):
        return True

    def invoke(self, context, event):
        act_obj = bpy.context.active_object
        obj_list = [obj for obj in bpy.context.selected_objects]

        bpy.ops.object.mode_set(mode='OBJECT')
        bpy.ops.object.select_all(action='DESELECT')

        for obj in obj_list:
            bpy.context.view_layer.objects.active = obj
            for mod in obj.modifiers:
                if mod.type == "LATTICE":
                    # Apply
                    if event.shift:
                        bpy.ops.object.modifier_apply(apply_as='DATA', modifier=mod.name)
                    # Remove
                    elif event.ctrl:
                        bpy.ops.object.modifier_remove(modifier=mod.name)
                    # Hide
                    elif event.alt:
                        bpy.context.object.modifiers[mod.name].show_viewport = not bpy.context.object.modifiers[
                            mod.name].show_viewport
                    # Edit
                    else:
                        lattice = bpy.context.object.modifiers[mod.name].object
                        bpy.context.view_layer.objects.active = lattice
                        lattice.select_set(state=True)
                        obj.select_set(state=False)
                        bpy.ops.object.mode_set(mode='EDIT')
                        bpy.ops.object.sfc_lattice_popup('INVOKE_DEFAULT')

        # Restore Selection
        for obj in obj_list:
            obj.select_set(state=True)
        bpy.context.view_layer.objects.active = act_obj
        act_obj.select_set(state=True)

        # Delete Vgroup if exist
        for vgroup in obj.vertex_groups:
            if vgroup.name == "SpeedLattice_Group":
                obj.vertex_groups.remove(vgroup)

        # detele list
        del (obj_list[:])
        return {"FINISHED"}

CLASSES = [SFC_OT_add_lattice,
           SFC_OT_add_lattice_to_selection,
           SFC_OT_apply_lattice_objects,
           SFC_OT_remove_lattice_objects,
           SFC_OT_connect_lattice,
           SFC_OT_edit_lattice,
           SFC_OT_edit_objects]

def register():
    for cls in CLASSES:
        try:
            bpy.utils.register_class(cls)
        except:
            print(f"{cls.__name__} already registred")


def unregister():
    for cls in CLASSES:
        bpy.utils.unregister_class(cls)


                