# -*- coding:utf-8 -*-

# Speedflow Add-on
# Copyright (C) 2018 Cedric Lepiller aka Pitiwazou & Legigan Jeremy AKA Pistiwique and Stephen Leger
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

# <pep8 compliant>

import bmesh
import bpy
from bpy.types import Operator
from bpy.props import StringProperty, FloatVectorProperty
from .utils.text_to_draw import *
from .utils.functions import *


# -*- coding:utf-8 -*-

from bpy.types import Operator
from bpy.props import StringProperty
from math import degrees, radians
from .utils.text_to_draw import *
from .utils.functions import *
import bmesh


class SPEEDFLOW_OT_edge_split(SpeedApi, Operator):
    """
    EDGE SPLIT

    CLICK - Add and adjust
    SHIFT - Apply all
    CTRL  - Remove all
    ALT    - Hide all

    Use default exit method
    Does change display of objects
    """
    bl_idname = "speedflow.edge_split"
    bl_label = "Modal Edge Split"
    bl_options = {'REGISTER', 'UNDO', 'GRAB_CURSOR', 'BLOCKING'}

    modal_action : StringProperty(default='free')

    @classmethod
    def poll(cls, context):
        return SpeedApi.can_add_modifier(context.object, types={'MESH', 'CURVE'})

    def __init__(self):

        SpeedApi.__init__(self)

        # NOTE: les valeurs par défaut sont définies dans les addon prefs
        # idéalement dans un propertyGroup pour chaque modifier
        # et set à la volée à la création du modifier seulement



        # xxx_action dicts here make an abstraction level between key and action
        # this way we may change keys without changing code !!
        # keys may contains modifiers in definition like CTRL+ALT+SHIFT+key
        # action_name should be exact modifier parameters names
        # No space allowed between modifiers+key
        # {[CTRL+|ALT+|SHIFT+]key: action_name}

        # Mouse action list (float and int values), allow keyboard values input
        self.mouse_actions = {
            'S': 'split_angle',
        }

        # Generic modifier toggle action list (boolean vars)
        self.toggle_actions = self.setup_modal_keymap({
            'D': 'use_edge_angle',
            'F': 'use_edge_sharp',
        })

        # "special actions" not changing modifier attributes in a regular way (mouse/toggle)
        self.special_actions = self.setup_modal_keymap({
            'F1': 'set_origin',
            'SHIFT+F1': 'set_origin_1',
            'SHIFT+RIGHT_ARROW': 'show_modifier',
            'SHIFT+LEFT_ARROW': 'hide_modifier',
            'QUOTE': 'show_hide_all_modifiers',
            'NUMPAD_SLASH': 'localview',
            'NUMPAD_PLUS': 'boolean_operation',
        })

        # default actions available for all modifiers based modals
        # default actions available for all modifiers based modals
        self.default_actions = self.setup_modal_keymap({
            # {[CTRL+|ALT+|SHIFT+]key: action_name}
            'A': 'apply_modifier',
            'SHIFT+SPACE': 'add_modifier',
            'DEL': 'remove_modifier',
            'BACK_SPACE': 'remove_modifier',

            'H': 'show_viewport',
            'CTRL+H': 'show_hide_keymap',
            'SHIFT+H': 'beginner_mode',

            'UP_ARROW': 'move_modifier_up',
            'DOWN_ARROW': 'move_modifier_down',

            'GRLESS': 'same_modifier_up',
            'RIGHT_ARROW': 'same_modifier_up',
            'SHIFT+GRLESS': 'same_modifier_down',
            'LEFT_ARROW': 'same_modifier_down',

            'CTRL+GRLESS': 'next_modifier_up',
            'CTRL+RIGHT_ARROW': 'next_modifier_up',
            'CTRL+SHIFT+GRLESS': 'next_modifier_down',
            'CTRL+LEFT_ARROW': 'next_modifier_down',

            'SPACE': 'call_pie',
            'ONE': 'keep_wire',
            'TWO': 'show_wire_in_modal',
            'THREE': 'set_xray',
            'FOUR': 'shading_mode',
            'FIVE': 'random',
            'SIX': 'overlays',
            'SEVEN': 'face_orientation',
            'EIGHT' : 'hide_grid',
            'NINE': 'bool_mode',
            'F2': 'active_snap',
            'F3': 'snap_vertex',
            'F4': 'snap_face',
            'F5': 'snap_grid',
            'F6': 'origin_to_selection',
            'F7': 'apply_location',

            'ESC': 'cancel',
            self.key_cancel: 'cancel',
            'SHIFT+%s' % self.key_cancel: 'cancel',
            'CTRL+SHIFT+%s' % self.key_cancel: 'cancel',
            'CTRL+%s' % self.key_cancel: 'cancel',

            'RET': 'confirm',
            'NUMPAD_ENTER': 'confirm',

            self.key_confirm: 'confirm',
            'SHIFT+%s' % self.key_confirm: 'confirm',
            'CTRL+SHIFT+%s' % self.key_confirm: 'confirm',
            'CTRL+%s' % self.key_confirm: 'confirm',

            # 'SPACE': 'confirm'
        })

    # ----------------------------------------------------------------------------------------------
    #
    # ----------------------------------------------------------------------------------------------

    def sf_mark_sharp(self):

        me = self.act_obj.data
        bm = bmesh.from_edit_mesh(me)

        if any([e.select for e in bm.edges]):
            bpy.ops.mesh.mark_sharp()
            bpy.ops.mesh.select_all(action='DESELECT')

    # ----------------------------------------------------------------------------------------------
    #
    # ----------------------------------------------------------------------------------------------

    def sf_clear_sharp(self):

        me = self.act_obj.data
        bm = bmesh.from_edit_mesh(me)

        if any([e.select for e in bm.edges]):
            bpy.ops.mesh.mark_sharp(clear=True)
            bpy.ops.mesh.select_all(action='DESELECT')
    # ----------------------------------------------------------------------------------------------
    #
    # ----------------------------------------------------------------------------------------------

    def modal(self, context, event):

        context.area.tag_redraw()

        modal_action = self.modal_action
        act_obj = self.act_obj

        if modal_action == 'free':
            if event.type in {'NUMPAD_1','NUMPAD_3','NUMPAD_7','NUMPAD_5'} or event.ctrl and event.type in {'NUMPAD_1','NUMPAD_3','NUMPAD_7','NUMPAD_5'}:
                return {'PASS_THROUGH'}

        if event.alt:
            return {'PASS_THROUGH'}

        if modal_action == 'free':
            if event.alt and event.type in {'LEFTMOUSE','RIGHTMOUSE','MIDDLEMOUSE'}:
                return {'PASS_THROUGH'}


        if event.type in { 'A', 'B','C','D','E','F','G','H','I','J','K','L','M','N','O','P','Q','R','S','T',
                           'U','V','W','X','Y','Z' 'SHIFT', 'LEFTMOUSE', 'RIGHTMOUSE','CTRL','ALT', 'ONE', 'TWO',
                           'SLASH'} and act_obj.mode == 'EDIT':
            return {'PASS_THROUGH'}

        # VERTEX
        if event.type == 'TAB' and event.value == 'PRESS':
            bpy.ops.wm.tool_set_by_id(name="builtin.select", cycle=False, space_type='VIEW_3D')
            obj_mode = any([obj.mode == 'OBJECT' for obj in self.sel])
            if obj_mode:
                bpy.ops.object.mode_set(mode='EDIT')
                if self.act_obj.type == 'MESH':
                    context.scene.tool_settings.use_mesh_automerge = False
            else:
                bpy.ops.object.mode_set(mode='OBJECT')
                return {'RUNNING_MODAL'}
            self.mode = self.act_obj.mode


        # pass through events
        if event.type == 'MIDDLEMOUSE' or (
                # allow zoom in free mode and for pen mode
                (modal_action == 'free' or self.work_tool != 'mouse') and
                event.type in {'WHEELUPMOUSE', 'WHEELDOWNMOUSE'}):
            return {'PASS_THROUGH'}

        # event ctrl/shift/alt state last time a key was pressed
        event_alt = self.event_alt

        # ---------------------------------
        # Mouse actions
        # ---------------------------------

        if modal_action in self.mouse_actions.values():

            if event.unicode in self.input_list:
                self.get_input_value(event.unicode)
                # No need to go further
                return {'RUNNING_MODAL'}

            # mouse action enabled
            if event.type in {'LEFT_SHIFT', 'RIGHT_SHIFT'}:

                # Sauve la dernière valeur pour modifier les steps
                value = self.get_property_value(self.act_mod, modal_action)

                # disallow slow_down for integer values
                slow_down = type(value).__name__ != 'int'

                if slow_down:
                    if event.value == 'PRESS':
                        if not self.slow_down:
                            self.mouse_x = event.mouse_x
                            # self.wheel_delta = 0
                            self.last_value = value
                            self.slow_down = True

                    elif event.value == 'RELEASE':
                        self.mouse_x = event.mouse_x
                        # self.wheel_delta = 0
                        self.last_value = value
                        self.slow_down = False

            if event.type in {'MOUSEMOVE', 'WHEELUPMOUSE', 'WHEELDOWNMOUSE'}:

                if event.type == 'MOUSEMOVE':

                    # prevent mouse move in 'mouse' mode (we could allow mouse in both modes)
                    if self.work_tool == 'mouse':
                        return {'PASS_THROUGH'}

                    # relative diff between last action and current
                    delta = event.mouse_x - self.mouse_x

                else:

                    # absolute delta for this step
                    wheel_delta = 1
                    if event.type == 'WHEELDOWNMOUSE':
                        wheel_delta = -wheel_delta

                    # relative diff between last action and current
                    self.wheel_delta += wheel_delta

                    delta = self.wheel_delta * 40 * self.modal_speed

                if self.slow_down:
                    delta /= 8

                pixels = self.get_pixels_from_rna(self.act_mod, modal_action)

                value = self.limit_value(self.last_value + delta / pixels, self.act_mod, modal_action)

                self.set_same_modifiers_value(modal_action, value)
                # if modal_action == 'levels':
                #     self.set_same_modifiers_value('render_levels', value)

        # -----------------------------------------
        # Select another object while modal running
        # -----------------------------------------

        event_alt = (event.ctrl, event.shift, event.alt)

        if modal_action == "free" and event.type == self.key_select and any(event_alt):
            if event.shift:
                return self.copy_modifier_if_missing(context, event, types={'MESH', 'CURVE', 'FONT'})
            elif event.ctrl:
                return self.remove_same_modifier(context, event)
            else:
                action = 'free'
                return self.run_modal_with_another_object(context, event, action, types={'MESH', 'CURVE', 'FONT'})

        # ---------------------------------
        # Keyboard events, check PRESS state once
        # ---------------------------------

        if event.value == 'PRESS':

            # Store event alternatives
            self.event_alt = event_alt = (event.ctrl, event.shift, event.alt)

            # set default value for action
            action = 'free'

            # ---------------------------------
            # Step 1 : Process events : turn events into actions
            # ---------------------------------

            # ---------------------------------
            # Mouse actions (startup - store current values)
            # ---------------------------------

            if event.type in self.mouse_actions.keys() and not self.input:

                # Mouse actions
                action = self.mouse_actions[event.type]
                self.store_init_value(event, action)

                # next modal run will handle this action so store action name
                self.modal_action = action

                return {'RUNNING_MODAL'}

            # ---------------------------------
            # Special actions
            # ---------------------------------

            elif event.type in self.special_actions[event_alt].keys():
                action = self.special_actions[event_alt][event.type]
                # let process actions below once

            elif event.type == 'BACK_SPACE' and self.input:
                self.input = self.input[:-1]
                # No need to process action
                return {'RUNNING_MODAL'}

            # ---------------------------------
            # Default actions
            # ---------------------------------

            elif event.type in self.default_actions[event_alt].keys():
                action = self.default_actions[event_alt][event.type]
                # let process actions below once

            # ---------------------------------
            # Toggle actions
            # ---------------------------------

            elif event.type in self.toggle_actions[event_alt].keys():
                action = self.toggle_actions[event_alt][event.type]
                self.toggle_same_modifiers_value(action)
                # No need to process action
                return {'RUNNING_MODAL'}

            # ---------------------------------
            # Step 2 : Process actions
            # ---------------------------------

            # ---------------------------------
            # Confirm (Generic)
            # ---------------------------------

            if action == 'confirm':

                if self.input:

                    value = self.input_as_value(self.act_mod, modal_action)
                    self.set_same_modifiers_value(modal_action, value)

                    self.input = ""

                if modal_action == 'free':
                    self.exit(context)

                    # restore Overlays
                    context.space_data.overlay.show_overlays = self.overlay_state
                    context.space_data.overlay.show_wireframes = self.overlay_wire_state
                    return {'FINISHED'}

                self.modal_action = 'free'

            # ---------------------------------
            # Cancel (Generic)
            # ---------------------------------

            elif action == 'cancel':

                if self.input:
                    self.input = ""

                if modal_action in self.mouse_actions.values():
                    self.set_same_modifiers_value(modal_action, self.init_value)

                if modal_action == 'free':
                    self.exit(context)

                    # restore Overlays
                    context.space_data.overlay.show_overlays = self.overlay_state
                    context.space_data.overlay.show_wireframes = self.overlay_wire_state
                    return {'CANCELLED'}

                self.modal_action = 'free'

            # ---------------------------------
            # Add Modifier  (Generic)
            # ---------------------------------

            elif action == 'add_modifier':

                # On peut imaginer une action similaire, pour
                # rajouter les modifiers manquants dans les objets selectionnés

                for obj in self.sel:
                    mod = self.add_modifier(obj, self.mod_type)
                    self.same_modifs[obj] = mod
                    if obj.type == 'MESH':
                        self.move_weighted_normals_down(context, obj)

                # Update same modifiers cache
                self.act_mod = self.same_modifs[act_obj]

                self.modal_action = 'split_angle'
                self.store_init_value(event, self.modal_action)

            # ---------------------------------
            # Apply Modifier
            # ---------------------------------

            elif action == 'apply_modifier':

                next = self.next_supported_modifier_down(act_obj, self.act_mod)

                for obj, mod in self.same_modifs.items():

                    if mod.levels == 0:
                        self.remove_modifier(obj, mod)
                    else:
                        # this will make object unique !!!
                        self.apply_modifier(context, obj, mod)

                self.run_modal(act_obj, next, call_from_pie=False)
                self.exit(context)

                # restore Overlays
                context.space_data.overlay.show_overlays = self.overlay_state
                context.space_data.overlay.show_wireframes = self.overlay_wire_state
                return {'FINISHED'}

            # ---------------------------------
            # Remove Modifier (Generic)
            # ---------------------------------

            elif action == 'remove_modifier':

                if self.act_obj.mode == 'OBJECT':
                    next = self.next_supported_modifier_down(act_obj, self.act_mod)

                    for obj, mod in self.same_modifs.items():
                        self.remove_modifier(obj, mod)

                    self.run_modal(act_obj, next)
                    self.exit(context)

                    # restore Overlays
                    context.space_data.overlay.show_overlays = self.overlay_state
                    context.space_data.overlay.show_wireframes = self.overlay_wire_state
                return {'FINISHED'}

            # ---------------------------------
            # Change Modifier Position (Generic)
            # ---------------------------------

            elif action == 'move_modifier_up':
                for obj, mod in self.same_modifs.items():
                    self.move_modifier_up(context, obj, mod)
                self.set_index_for_companion(self.act_mod)

            elif action == 'move_modifier_down':
                for obj, mod in self.same_modifs.items():
                    self.move_modifier_down(context, obj, mod)
                self.set_index_for_companion(self.act_mod)

            # ---------------------------------
            # Switch Between modifiers (Generic)
            # ---------------------------------

            elif action == 'next_modifier_up':
                res = self.run_modal_up(act_obj, self.act_mod)
                if res:
                    self.exit(context)

                    # restore Overlays
                    context.space_data.overlay.show_overlays = self.overlay_state
                    context.space_data.overlay.show_wireframes = self.overlay_wire_state
                    return {'FINISHED'}

            elif action == 'next_modifier_down':
                res = self.run_modal_down(act_obj, self.act_mod)
                if res:
                    self.exit(context)

                    # restore Overlays
                    context.space_data.overlay.show_overlays = self.overlay_state
                    context.space_data.overlay.show_wireframes = self.overlay_wire_state
                    return {'FINISHED'}

            elif action == 'same_modifier_up':
                self.act_mod = self.get_modifier_up_same_type(act_obj, self.act_mod)

            elif action == 'same_modifier_down':
                self.act_mod = self.get_modifier_down_same_type(act_obj, self.act_mod)

            # ---------------------------------
            # Call pie menu (Generic)
            # ---------------------------------

            elif action == 'call_pie':
                self.exit(context)
                if self.prefs.choose_menu_type == 'pie':
                    bpy.ops.wm.call_menu_pie('INVOKE_DEFAULT', name="SPEEDFLOW_MT_pie_menu")
                else:
                    bpy.ops.wm.call_menu('INVOKE_DEFAULT', name="SPEEDFLOW_MT_simple_menu")
                return {'FINISHED'}

            # ---------------------------------
            # Toggle Keymaps (Generic)
            # ---------------------------------

            elif action == 'show_hide_keymap':
                self.prefs.display_keymaps = not self.prefs.display_keymaps

            # ---------------------------------
            # SHIFT + H Toggle Beginner Mode
            # ---------------------------------
            elif action == 'beginner_mode':
                self.prefs.beginner_mode = not self.prefs.beginner_mode

            # ---------------------------------------
            # Toggle Show / Hide Modifier (Generic)
            # ---------------------------------------
            elif action == 'show_viewport':
                # same modifiers
                self.toggle_same_modifiers_value('show_viewport')


# ---------------------------------
# Special actions
# ---------------------------------

            # ---------------------------------
            # Change Boolean operation
            # ---------------------------------
            elif action == 'boolean_operation':
                self.edit_boolean_operation(context, self)

            # ---------------------------------
            # Toggle local view in Modal
            # ---------------------------------
            elif action == 'localview':
                bpy.ops.view3d.localview()

            # ---------------------------------
            # Toggle Hide all modifiers
            # ---------------------------------
            elif action == 'show_hide_all_modifiers':
                self.show_hide_all_modifiers(act_obj, self.act_mod)

            # ---------------------------------
            # Show modifiers
            # ---------------------------------
            elif action == 'show_modifier':
                for obj in context.selected_objects:
                    to_show = self.get_first_modifier_by_dict(obj, {'show_viewport' : False})
                    if to_show is not None:
                        to_show.show_viewport = True

            # ---------------------------------
            # Hide modifiers
            # ---------------------------------
            elif action == 'hide_modifier':
                for obj in context.selected_objects:
                    to_show = self.get_last_modifier_by_dict(obj, {'show_viewport' : True})
                    if to_show is not None:
                        to_show.show_viewport = False

    # ---------------------------------
    # Shading / Snap
    # ---------------------------------

            # ---------------------------------------
            # 1 - Toggle Keep Wire in Modal
            # ---------------------------------------
            elif action == 'keep_wire':
                self.prefs.keep_wire = not self.prefs.keep_wire

            # --------------------------------------
            # 2 - Toggle wire in the modal
            # --------------------------------------
            elif action == 'show_wire_in_modal':
                self.shading_wire_in_modal(context, self)

            # ---------------------------------
            # 3 - Toggle XRAY
            # ---------------------------------
            elif action == 'set_xray':
                self.shading_xray(context, self)

            # ---------------------------------
            # 4 - Toggle shading mode type
            # ---------------------------------
            elif action == 'shading_mode':
                self.shading_mode_type(context, self)

            # ---------------------------------
            # 5 - Toggle Random Mode
            # ---------------------------------
            elif action == 'random':
                self.shading_random_mode(context, self)

            # ---------------------------------
            # 6 - Toggle OVERLAYS
            # ---------------------------------
            elif action == 'overlays':
                self.shading_overlays(context, self)

            # ---------------------------------
            # 7 - Toggle FACE ORIENTATION
            # ---------------------------------
            elif action == 'face_orientation':
                self.shading_face_orientation(context, self)

            # ---------------------------------
            # 8 - Toggle Hide Grid
            # ---------------------------------
            elif action == 'hide_grid':
                self.shading_hide_grid(context, self)

            # ---------------------------------
            # 9 - Toggle Bool Shading Mode
            # ---------------------------------
            elif action == 'bool_mode':
                self.shading_bool_mode(context, self)

            # ---------------------------------
            # F2 - Toggle Activate Snap
            # ---------------------------------
            elif action == 'active_snap':
                self.snap_activate_snap(context, self)

            # ---------------------------------
            # F3 - Toggle Snap Vertex
            # ---------------------------------
            elif action == 'snap_vertex':
                self.snap_vertex(context, self)

            # ---------------------------------
            # F4 - Toggle Snap Face
            # ---------------------------------
            elif action == 'snap_face':
                self.snap_face(context, self)

            # ---------------------------------
            # F5 - Toggle Snap Grid
            # ---------------------------------
            elif action == 'snap_grid':
                self.snap_grid(context, self)

            # ---------------------------------
            # F6 - Origin to selection
            # ---------------------------------
            elif action == 'origin_to_selection':
                self.snap_origin_to_selection(context)

            # ---------------------------------
            # F7 - Toggle Apply Origin
            # ---------------------------------
            elif action == 'apply_location':
                self.snap_origin_to_grid(context)

            # ---------------------------------
            # START Origin
            # ---------------------------------
            if action == 'set_origin':
                # self.act_mod.show_viewport = False

                bpy.ops.wm.tool_set_by_id(name="builtin.select", cycle=False, space_type='VIEW_3D')

                self.mode_snap = self.act_obj.mode
                context.scene.tool_settings.use_snap = True
                # context.scene.tool_settings.snap_elements = {'VERTEX'}
                context.scene.tool_settings.use_snap_align_rotation = False

                bpy.ops.view3d.snap_cursor_to_selected()

                if act_obj.mode != 'OBJECT':
                    bpy.ops.object.mode_set(mode='OBJECT')

                bpy.ops.transform.translate('INVOKE_DEFAULT', cursor_transform=True)

                self.modal_action = 'origin'

            if action == 'set_origin_1':
                # self.act_mod.show_viewport = False

                self.mode_snap = self.act_obj.mode
                context.scene.tool_settings.use_snap = True
                context.scene.tool_settings.snap_elements = {'INCREMENT'}
                context.scene.tool_settings.use_snap_grid_absolute = True

                bpy.ops.view3d.snap_cursor_to_selected()

                if act_obj.mode != 'OBJECT':
                    bpy.ops.object.mode_set(mode='OBJECT')

                bpy.ops.transform.translate('INVOKE_DEFAULT', cursor_transform=True)

                self.modal_action = 'origin'

            # ---------------------------------
            # END ORIGIN
            # ---------------------------------
            if modal_action == 'origin':

                bpy.ops.object.origin_set(type='ORIGIN_CURSOR')
                # self.act_mod.show_viewport = True

                bpy.ops.object.mode_set(mode=self.mode_snap)
                self.modal_action = 'free'

        return {'RUNNING_MODAL'}

    # ----------------------------------------------------------------------------------------------
    #
    # ----------------------------------------------------------------------------------------------

    def invoke(self, context, event):

        if context.area and context.area.type == 'VIEW_3D':
            self.saved_location = context.scene.cursor.location.copy()

            # self.key_select = "%sMOUSE" % context.window_manager.keyconfigs.active.preferences.select_mouse

            # Get modifier type from modal class name
            mod_type = self.mod_type

            # retrieve addon prefs
            prefs = self.prefs

            act_obj = context.object

            # filter objects from selection, act_obj is part of sel
            sel = self.filter_objects_by_type(context.selected_objects, {'MESH', 'CURVE', 'FONT'})

            # store context mode, selected and active
            self.mode = act_obj.mode
            self.act_obj = act_obj
            self.sel = sel
            self.shading_viewport_type = context.space_data.shading.type
            self.overlay_state = context.space_data.overlay.show_overlays
            self.overlay_wire_state = context.space_data.overlay.show_wireframes
            self.shading_random = context.space_data.shading.color_type

            # flag indicate call method, through pie menu or object selection change or <-> with specified modifier
            call_through_pie = self.call_from_pie

            for obj in self.sel:
                if obj.type == 'MESH':
                    if self.prefs.shading_smooth:
                        context.object.data.use_auto_smooth = True

                    if self.act_obj.mode == 'EDIT':
                        # if call_through_pie and (event.ctrl):
                        #     if event.ctrl:
                        #
                        #         self.sf_clear_sharp()
                        #     else:

                            self.sf_mark_sharp()

                            # return {'FINISHED'}


            #
            if act_obj.mode != 'OBJECT':
                bpy.ops.object.mode_set(mode="OBJECT")


            # flag when call through selection change using alt key, prevent special actions
            allow_special_actions = self.modifier_index == -2

            # Find if we do have a modifier of this modal type
            has_mod = self.has_modifier_of_type(act_obj, mod_type)

            if self.prefs.default_modal_action and has_mod:
                self.modal_action = 'free'
            else:
                self.modal_action = 'split_angle'

            if has_mod and call_through_pie and self.work_tool == 'pen' and prefs.default_modal_action:
                self.modal_action = 'free'

            # NOTE: act_obj est contenu dans sel
            if call_through_pie:
                act_mod = self.get_or_create_modifier(act_obj, mod_type, mod_type.capitalize())
                # act_mod.show_viewport = prefs.show_wire_in_modal

            else:
                # use specified act_mod
                act_mod = self.get_modifier_by_index(act_obj, self.modifier_index)

            # specials actions
            special = (event.shift, event.ctrl, event.alt)

            if allow_special_actions and any(special):

                if check_valid_input(special):

                    for obj in sel:
                        fun, args = {
                            (True, False, False): [self.apply_modifiers_by_type, (context, obj, 'EDGE_SPLIT')],
                            (False, True, False): [self.remove_modifiers_by_type, (obj, 'EDGE_SPLIT')],
                            (False, False, True): [self.toggle_modifiers_visibility_by_type, (obj, 'EDGE_SPLIT')],
                        }[special]

                        # hi hi hi, have some fun
                        fun(*args)

                    bpy.ops.object.mode_set(mode=self.mode)

                    return {'FINISHED'}

            # Add modifiers to selected objects
            for obj in sel:

                # find existing modifier with same data than act_mod
                mod = self.find_same_in_list(act_mod, obj.modifiers)

                # when not found, add a new one
                if mod is None:
                    # condition to add a modifier on selected objects:
                    # if we are browsing act_obj stack using <->
                    # and no similar modifier is found then do nothing (ignore object)

                    # if we call through pie, and no modifier of this kind
                    # is found then add a new one
                    if call_through_pie and not self.has_modifier_of_type(obj, mod_type):
                        mod = self.add_modifier(obj, mod_type)
                        # selection may change between 2 pie calls
                        # then maybe act_obj already hold a non default modifier
                        # so we copy params from it
                        if has_mod:
                            self.copy_modifier_params(act_mod, mod)

                    # subsequent modifiers must be added through modal commands

                # if mod is not None:

                    # maybe move this right after add_modifier only ??
                    # self.move_weighted_normals_down(context, obj)


                if obj.type == 'MESH':
                    # self.sf_mark_sharp()


                    self.move_weighted_normals_down(context, obj)

            # Store same modifiers in cache
            self.act_mod = act_mod

            # store initial value for mouse action to restore on exit and last to change mouse steps on the fly
            if self.modal_action in self.mouse_actions.values():
                self.store_init_value(event, self.modal_action)

            # Setup display of objects
            display_type = None
            if prefs.show_wire_in_modal and act_obj.display_type == 'BOUNDS':
                display_type = 'WIRE'

            if prefs.show_wire_in_modal:
                context.space_data.overlay.show_overlays = True
                context.space_data.overlay.show_wireframes = False

            self.setup_display(display_type)

            # setup draw handler
            self.setup_draw_handler(context)

            context.window_manager.modal_handler_add(self)

            return {'RUNNING_MODAL'}

        else:
            self.report({'WARNING'}, "View3D not found, cannot run operator")
            return {'CANCELLED'}

def register():
    try:
        bpy.utils.register_class(SPEEDFLOW_OT_edge_split)
    except:
        print("Edge Split already registred")

def unregister():
    bpy.utils.unregister_class(SPEEDFLOW_OT_edge_split)