# -*- coding:utf-8 -*-

# Speedflow Add-on
# Copyright (C) 2018 Cedric Lepiller aka Pitiwazou & Legigan Jeremy AKA Pistiwique and Stephen Leger
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

# <pep8 compliant>

import bpy
from bpy.types import Operator
from bpy.props import (StringProperty,
                       BoolProperty,
                       FloatVectorProperty,
                       FloatProperty,
                       EnumProperty,
                       IntProperty,
                       PointerProperty)

import bmesh
from .utils.functions import *
from .utils.properties import *
import math
from mathutils import Matrix, Vector
import gpu
from gpu_extras.batch import batch_for_shader
import bgl

def speedflow_primitives_default(speedflow_default_enum):

    if speedflow_default_enum == 'cylinder':
        bpy.ops.mesh.primitive_cylinder_add(vertices=24, radius=1, depth=2, enter_editmode=False)
        bpy.ops.transform.resize(value=(0.0005, 0.0005, 0.0005))

    if speedflow_default_enum == 'cube':
        bpy.ops.mesh.primitive_cube_add(size=1, enter_editmode=False)
        bpy.ops.transform.resize(value=(0.0005, 0.0005, 0.0005))


def prim_popup_menu(self, context):
    layout = self.layout
    split = layout.split()
    col = split.column(align=True)
    row = col.row(align=True)
    row.scale_y = 1.5
    op = row.operator("speedflow.cutter", text="", icon='MESH_CUBE')
    op.speedflow_primitives_enum = "default_prim"
    op.speedflow_default_enum = 'cube'

    op = row.operator("speedflow.cutter", text="", icon='MESH_CYLINDER')
    op.speedflow_primitives_enum = "default_prim"
    op.speedflow_default_enum = 'cylinder'

    op = row.operator("speedflow.cutter", text="", icon='META_CUBE')
    op.speedflow_primitives_enum = "custom_prim"
    op.speedflow_custom_enum = 'rounded_cube'

    op = row.operator("speedflow.cutter", text="", icon='META_PLANE')
    op.speedflow_primitives_enum = "custom_prim"
    op.speedflow_custom_enum = 'rounded_plane'

class speedflow_OT_custom_primitives(bpy.types.Operator):
    bl_idname = "view3d.speedflow_custom_primitives"
    bl_label = "Speedflow Custom Primitives"
    bl_options = {'REGISTER'}

    def execute(self, context):
        context.window_manager.popup_menu(prim_popup_menu, title="")
        return {'FINISHED'}


def speedflow_primitives_custom(speedflow_custom_enum):
    # Rounded Cube
    if speedflow_custom_enum == 'rounded_cube':
        bpy.ops.mesh.primitive_cube_add(True,size=1,  enter_editmode=False)
        new_prim = bpy.context.active_object
        new_prim.name = "Custom Rounded Cube"

        # bevel
        new_bevel = new_prim.modifiers.new("Bevel", 'BEVEL')
        new_bevel.limit_method = 'ANGLE'
        new_bevel.name = "Bevel - %s" % (new_bevel.limit_method)
        new_bevel.width = 0.3
        new_bevel.segments = 6
        new_bevel.profile = 0.5
        new_bevel.use_only_vertices = False
        new_bevel.angle_limit = 1.535889744758606
        new_bevel.use_clamp_overlap = False
        bpy.ops.transform.resize(value=(0.005, 0.005, 0.005))

    # Rounded Plane
    elif speedflow_custom_enum == 'rounded_plane':
        bpy.ops.mesh.primitive_plane_add(True,size=0.01,  enter_editmode=False)
        new_prim = bpy.context.active_object
        new_prim.name = "Custom Rounded Plane"

        bpy.ops.object.mode_set(mode='EDIT')
        bpy.ops.mesh.select_all(action='SELECT')
        bpy.ops.mesh.subdivide(quadcorner='INNERVERT')
        bpy.ops.mesh.region_to_loop()
        bpy.ops.mesh.select_less(use_face_step=False)

        # Add Vgroup
        bpy.ops.object.vertex_group_add()
        bpy.context.scene.tool_settings.vertex_group_weight = 1
        bpy.ops.object.vertex_group_assign()
        vgroup = new_prim.vertex_groups

        # bevel
        new_bevel = new_prim.modifiers.new("Bevel", 'BEVEL')
        new_bevel.limit_method = 'VGROUP'
        new_bevel.name = "Bevel - %s" % (new_bevel.limit_method)
        new_bevel.width = 0.4
        new_bevel.segments = 6
        new_bevel.profile = 0.5
        new_bevel.use_only_vertices = True
        new_bevel.angle_limit = 1.535889744758606
        new_bevel.use_clamp_overlap = False
        new_bevel.vertex_group = vgroup.active.name

        # Solidify
        new_solidify = new_prim.modifiers.new("Solidify", 'SOLIDIFY')
        new_solidify.thickness = 2
        new_solidify.use_even_offset = True
        new_solidify.use_quality_normals = True
        new_solidify.offset = 0

# class SPEEDFLOW_MT_custom_primitives(bpy.types.Menu):
#     bl_label = "Speedflow Custom Primitives"
#
#     def draw(self, context):
#         layout = self.layout
#         # SF = bpy.context.window_manager.SF
#         # speedflow_primitives_enum = SF.speedflow_primitives_enum
#         # speedflow_default_enum = SF.speedflow_default_enum
#
#         op = layout.operator("speedflow.cutter", text="Cylinder", icon='MESH_CYLINDER')
#         op.speedflow_primitives_enum = "default_prim"
#         op.speedflow_default_enum = 'cylinder'
#
#         op = layout.operator("speedflow.cutter", text="Cube", icon='MESH_CUBE')
#         op.speedflow_primitives_enum = "default_prim"
#         op.speedflow_default_enum = 'cube'
#
#         # op = layout.operator("speedflow.cutter", text="Rounded Cube", icon='META_CUBE')
#         # op.speedflow_primitives_enum = "custom_prim"
#         # op.speedflow_custom_enum = 'rounded_cube'
#
#         # op = layout.operator("speedflow.cutter", text="Rounded Plane", icon='META_PLANE')
#         # op.speedflow_primitives_enum = "custom_prim"
#         # op.speedflow_custom_enum = 'rounded_plane'

def local_rotate(obj, axis, angle):
    obj.matrix_world @= Matrix.Rotation(math.radians(angle), 4, axis)

def draw_callback_px_rectangle(self, context):
    self.cutter_face_color = self.prefs.cutter.cutter_face_color
    self.cutter_lines_color = self.prefs.cutter.cutter_lines_color

    tM = self.plan.matrix_world
    face_index = (0, 3, 7, 4)
    indices = [(0, 1, 2), (0, 2, 3)]
    coords = [tM @ Vector(self.plan.bound_box[idx]) for idx in face_index]

    lines_index = [0, 1, 2, 3, 0]
    lines_coords = [coords[idx] for idx in lines_index]
    lines_shader = gpu.shader.from_builtin('3D_UNIFORM_COLOR')
    lines_batch = batch_for_shader(lines_shader, 'LINE_STRIP', {"pos": lines_coords})
    bgl.glEnable(bgl.GL_LINE_SMOOTH)
    lines_shader.bind()
    lines_shader.uniform_float("color", self.cutter_lines_color)
    # lines_shader.uniform_float("color", (0, 0.65, 1, 1))
    bgl.glLineWidth(4)
    lines_batch.draw(lines_shader)


    face_shader = gpu.shader.from_builtin('3D_UNIFORM_COLOR')
    face_batch = batch_for_shader(face_shader, 'TRIS', {"pos": coords}, indices=indices)
    face_shader.bind()
    face_shader.uniform_float("color", self.cutter_face_color)
    # face_shader.uniform_float("color", (0, 0.65, 1, 0.3))
    bgl.glEnable(bgl.GL_BLEND)
    face_batch.draw(face_shader)
    bgl.glDisable(bgl.GL_BLEND)

class SPEEDFLOW_OT_cutter(SpeedApi, Operator):
    """ CUTTER

        CLICK - Cut
        SHIFT - Union
        CTRL  - Rebool
        ALT    - Create primitive

        IN MODAL
        CLICK - Rectangle
        SHIFT - Cylinder
        CTRL  - Line
        ALT    - On Cursor
        """
    bl_idname = "speedflow.cutter"
    bl_label = "Speedflow Cutter"

    first_mouse_x: IntProperty()
    first_mouse_y: IntProperty()

    @classmethod
    def poll(cls, context):
        return True
        # return context.object is not None and context.object.type == 'MESH' and context.object.mode == 'OBJECT'

    def __init__(self):
        SpeedApi.__init__(self)
        CommonProperties.__init__(self)

        self.new_bool = False
        self.prim_plan = False
        self.prim_screw = False
        self.prim_line_curve = False
        self.prim_vertex_line = False
        self.prim_bisect = False
        self.is_perspective = False
        self.new_prim = False
        self.bool_op = False
        self.orient = False
        self.cursor_Z_0 = False
        self.z_axis = False
        self.custom_prim = False
        self.snap_grid = False
        self.cutter_screw_steps = self.prefs.cutter.cutter_screw_steps

    speedflow_primitives_enum: EnumProperty(
        items=(('default_prim', "Default Primitives", ""),
               ('custom_prim', "Custom Primitives", "")),
        default='default_prim')

    speedflow_default_enum: EnumProperty(
        items=(('cube', "Cube", ""),
               ('cylinder', "Cylinder", ""),
               ),
        default='cylinder')

    speedflow_custom_enum: EnumProperty(
        items=(('rounded_cube', "Rounded Cube", ""),
               ('rounded_plane', "Rounded Plane", "")),
        default='rounded_cube'
    )

# -------------------------------------
# CUSTOM PRIMITIVES
# -------------------------------------
    def custom_prim_cutter(self, context):
        cursor_rot = bpy.context.scene.cursor.rotation_euler.copy()

        # Mesh Creation
        if self.speedflow_primitives_enum == 'default_prim':
            speedflow_primitives_default(self.speedflow_default_enum)

        if self.speedflow_primitives_enum == 'custom_prim':
            speedflow_primitives_custom(self.speedflow_custom_enum)

        self.bbox_obj = bpy.context.active_object
        self.bbox_obj.rotation_euler = cursor_rot

        # Shading
        bpy.context.object.data.use_auto_smooth = True
        bpy.context.object.data.auto_smooth_angle = 0.523599
        bpy.ops.object.mode_set(mode='OBJECT')
        bpy.ops.object.shade_smooth()

        if not self.new_prim:
            self.bbox_obj.display_type = 'BOUNDS'

        # Create Boolean
        if self.has_selection and self.bool_op:
            self.create_boolean(context)

        if self.rebool:
            self.create_rebool()

        bpy.ops.object.select_all(action='DESELECT')
        self.bbox_obj.select_set(state=True)
        bpy.context.view_layer.objects.active = self.bbox_obj

# -------------------------------------
# SCREW
# -------------------------------------
    def create_screw(self):
        bpy.ops.mesh.primitive_plane_add(size=1, enter_editmode=True, align='CURSOR')
        bpy.context.tool_settings.mesh_select_mode = (True, False, False)

        self.screw = bpy.context.active_object
        self.new_bool = True

        bpy.ops.mesh.merge(type='CENTER')
        bpy.ops.mesh.extrude_region_move()

        bpy.ops.object.vertex_group_add()
        bpy.context.scene.tool_settings.vertex_group_weight = 1
        bpy.ops.object.vertex_group_assign()

        self.mod_displace = self.screw.modifiers.new("Displace - X", 'DISPLACE')
        self.mod_displace.mid_level = 0
        self.mod_displace.direction = 'X'
        self.mod_displace.strength = 0
        self.mod_displace.vertex_group = "Group"
        self.mod_displace.show_in_editmode = True
        self.mod_displace.show_on_cage = True
        if self.prefs.show_expanded:
            self.mod_displace.show_expanded = True

        self.strength = self.mod_displace.strength

        mod_screw = self.screw.modifiers.new("Screw", 'SCREW')
        mod_screw.steps = self.cutter_screw_steps
        mod_screw.render_steps = self.cutter_screw_steps
        mod_screw.use_merge_vertices = True
        mod_screw.use_normal_calculate = True
        mod_screw.show_in_editmode = True
        mod_screw.show_on_cage = True
        if self.prefs.show_expanded:
            mod_screw.show_expanded = True

        bpy.context.object.data.use_auto_smooth = True
        bpy.context.object.data.auto_smooth_angle = 0.523599
        bpy.ops.object.mode_set(mode='OBJECT')
        bpy.ops.object.shade_smooth()
        bpy.context.object.show_in_front = True
        self.prim_screw = True
# -------------------------------------
# CORE MESH FOR PLAN
# -------------------------------------
    def create_plan(self):
        bpy.ops.mesh.primitive_plane_add(size=1, enter_editmode=False, align='CURSOR')

        bpy.context.object.display_type = 'BOUNDS'
        self.plan = bpy.context.active_object
        self.new_bool = True
        self.prim_plan = True

        self.mod_mirror = self.plan.modifiers.new("Mirror", 'MIRROR')
        self.mod_mirror.use_axis[0] = True
        self.mod_mirror.use_axis[1] = True
        self.mod_mirror.show_on_cage = False
        if self.on_cursor:
            self.mod_mirror.show_viewport = True
        else:
            self.mod_mirror.show_viewport = False

        bpy.ops.object.mode_set(mode='EDIT')
        bpy.context.tool_settings.mesh_select_mode = (True, False, False)
        bpy.ops.mesh.merge(type='CENTER')
        bpy.ops.mesh.duplicate_move()
# -------------------------------------
# VERTEX LINE
# -------------------------------------
    def create_vertex_line(self):
        bpy.ops.mesh.primitive_plane_add(size=1, enter_editmode=True, align='CURSOR')
        self.vertex_line = bpy.context.active_object
        self.prim_vertex_line = True
        self.new_bool = True

        bpy.context.tool_settings.mesh_select_mode = (True, False, False)
        bpy.ops.mesh.merge(type='CENTER')

        bpy.ops.mesh.extrude_region_move()
        bpy.context.scene.tool_settings.use_snap = True
        bpy.context.scene.tool_settings.snap_elements = {'VERTEX'}
        bpy.context.scene.tool_settings.use_snap_align_rotation = False

        bpy.context.object.show_in_front = True

    # -------------------------------------
# CURVE LINE
# -------------------------------------
    def create_line_curve(self):

        # set cursor
        if not self.on_cursor:
            bpy.ops.wm.tool_set_by_id(name="builtin.cursor")
            bpy.ops.view3d.cursor3d('INVOKE_DEFAULT', use_depth=True, orientation='GEOM')
        self.cursor_rot = bpy.context.scene.cursor.rotation_euler.copy()
        bpy.ops.curve.primitive_bezier_curve_add(enter_editmode=True, align='CURSOR')

        self.line_curve = bpy.context.active_object
        self.new_bool = True
        self.prim_line_curve = True

        bpy.context.space_data.overlay.show_curve_handles = False
        bpy.ops.curve.delete(type='VERT')
        bpy.ops.object.mode_set(mode='OBJECT')
        bpy.ops.object.transform_apply(location=True, rotation=True, scale=True)

        bpy.ops.object.mode_set(mode='EDIT')
        bpy.context.space_data.overlay.show_curve_normals = False

        bpy.ops.curve.draw('INVOKE_DEFAULT', error_threshold=0.05, fit_method='REFIT', corner_angle=60, use_cyclic=True,
                           stroke=[],
                           wait_for_input=True)

# -------------------------------------
# FINAL PLAN
# -------------------------------------
    def create_bbox(self):
        bpy.ops.object.convert(target='MESH')
        bpy.ops.object.mode_set(mode='EDIT')
        bpy.ops.mesh.select_all(action='SELECT')
        bpy.ops.view3d.snap_cursor_to_selected()
        bpy.ops.object.mode_set(mode='OBJECT')
        bpy.ops.object.origin_set(type='ORIGIN_GEOMETRY', center='BOUNDS')
        bpy.ops.mesh.primitive_plane_add(enter_editmode=False, align='CURSOR', location=(self.plan.location))

        self.bbox_obj = bpy.context.active_object

        # copy transforms
        self.bbox_obj.dimensions = self.plan.dimensions
        self.bbox_obj.rotation_euler = self.plan.rotation_euler

        bpy.ops.object.transform_apply(location=False, rotation=False, scale=True)
        bpy.data.objects.remove(self.plan, do_unlink=True)

# -------------------------------------
# BOOLEAN
# -------------------------------------
    def create_boolean(self, context):

        if self.last_ref_object :
            for obj in self.last_ref_object:
                if obj.get("is_last_ref"):
                    del obj["is_last_ref"]

        # Check si ref_object et mettre l'objet en actif si il est dans la liste
        # if self.has_ref and len(self.sel) == 1:
        if self.ref_object and len(self.sel) == 1:
            if self.act_obj != self.ref_object[0]:
                del (self.sel[:])
                bpy.ops.object.select_all(action='DESELECT')
                for obj in self.ref_object:
                    obj.select_set(state=True)
                    context.view_layer.objects.active = obj
                    self.sel.append(obj)
                    print(self.sel)
                    print(self.ref_object)

        bpy.ops.object.select_all(action='DESELECT')
        for obj in self.sel:
            obj.select_set(state=True)
            bpy.context.view_layer.objects.active = obj
            obj["is_last_ref"] = True

            act_mod = self.add_modifier(obj, 'BOOLEAN')
            act_mod.object = self.bbox_obj
            if self.boolean_union:
                act_mod.operation = 'UNION'
            else:
                act_mod.operation = 'DIFFERENCE'


            has_mirror = False
            for mod in obj.modifiers:
                if mod.type == 'MIRROR':
                    if any(mod.use_bisect_axis):
                        has_mirror = True

            # has_mirror = self.get_modifiers_by_type_and_attr(obj, 'MIRROR', 'use_bisect_axis[0]', True)

            has_bevel = self.get_modifiers_by_type_and_attr(obj, 'BEVEL', 'limit_method', 'ANGLE')
            has_triangulate = self.get_modifiers_by_type(obj, 'TRIANGULATE')
            has_weighted = self.get_modifiers_by_type(self.act_obj, 'WEIGHTED_NORMAL')
            if has_weighted:
                self.move_weighted_normals_down(context, obj)

            if has_bevel:
                # print("YES, je monte")
                bpy.ops.object.modifier_move_up(modifier=act_mod.name)

            if has_mirror:
                print("YES, j'ai un mirror avec Bisect")
                bpy.ops.object.modifier_move_up(modifier=act_mod.name)

            if has_triangulate:
                bpy.ops.object.modifier_move_up(modifier=act_mod.name)

#-------------------------------------
# REBOOL
#-------------------------------------
    def create_rebool(self):

        bpy.ops.object.select_all(action='DESELECT')

        for obj in self.sel:
            obj.select_set(state=True)
            bpy.context.view_layer.objects.active = obj

        bpy.ops.object.duplicate_move()

        for obj in bpy.context.selected_objects:
            has_boolean = self.get_last_modifier_by_type(obj, 'BOOLEAN')
            has_boolean.operation = 'INTERSECT'
            if self.last_ref_object:
                if obj.get("is_last_ref"):
                    del obj["is_last_ref"]

# -------------------------------------
# LOCAL MOVE
# -------------------------------------
    def cutter_move(self):

        if self.perspective:
            if self.z_axis:
                bpy.ops.transform.translate('INVOKE_DEFAULT', True, orient_matrix_type='LOCAL',
                                            constraint_axis=(False, False, True), release_confirm=False)
            else:
                bpy.ops.transform.translate('INVOKE_DEFAULT', True, orient_matrix_type='LOCAL',
                                            constraint_axis=(True, True, False), release_confirm=False)
        else:
            bpy.ops.transform.translate('INVOKE_DEFAULT', release_confirm=False)

# -------------------------------------
# FINALISE CUTTING
# -------------------------------------
    def finilize_cutting(self, context):
        # Create Face
        if self.prim_vertex_line:
            if not self.prim_bisect:
                # Remove last vertex
                bpy.ops.mesh.delete(type='VERT')
            bpy.ops.mesh.select_all(action='SELECT')
            bpy.ops.mesh.edge_face_add()
            bpy.ops.mesh.normals_make_consistent(inside=False)

        bpy.ops.object.mode_set(mode='OBJECT')

        if self.prim_vertex_line:
            bpy.context.object.show_in_front = False

        if self.prim_line_curve:
            bpy.ops.object.convert(target='MESH')

        # Create Plan from bounding box
        if self.prim_plan:
            self.create_bbox()

        # Set act_obj for boolean
        self.bbox_obj = bpy.context.active_object
        bpy.context.object.show_in_front = False

        if not self.custom_prim:
            # Create Boolean
            if self.has_selection and self.bool_op:
                self.create_boolean(context)

            if self.rebool:
                self.create_rebool()

        # Set act_obj and Shading
        bpy.ops.object.select_all(action='DESELECT')
        self.bbox_obj.select_set(state=True)

        # Add To collection
        if not self.new_prim:
            if self.prefs.add_bool_objects_to_collection:
                in_collection = bpy.data.collections['Bool_Objects'].objects.get(self.bbox_obj.name)
                if not in_collection:
                    self.unlink_object_from_scene(self.bbox_obj)
                    bpy.data.collections['Bool_Objects'].objects.link(self.bbox_obj)

        if not self.new_prim:
            if self.parent_bool_objects:
                bpy.ops.object.parent_set(type='OBJECT', keep_transform=True)

        bpy.context.view_layer.objects.active = self.bbox_obj

        if self.bool_op:
            self.bbox_obj.display_type = 'BOUNDS'
        else:
            self.bbox_obj.display_type = 'TEXTURED'


        # Shading
        bpy.context.object.data.use_auto_smooth = True
        bpy.context.object.data.auto_smooth_angle = 0.523599
        bpy.ops.object.shade_smooth()

        # Exit to Solidify
        bpy.ops.wm.tool_set_by_id(name="builtin.select")

        if self.prim_line_curve:
            bpy.context.space_data.overlay.show_curve_handles = True

        # restore
        bpy.context.scene.tool_settings.use_snap = self.snap
        bpy.context.space_data.overlay.show_overlays = self.overlay_state
        bpy.context.space_data.overlay.show_wireframes = self.overlay_wire_state
        bpy.context.scene.transform_orientation_slots[0].type = str(self.orientation_mode)

        if self.prim_plan:
            bpy.types.SpaceView3D.draw_handler_remove(self._handle, 'WINDOW')
        bpy.types.SpaceView3D.draw_handler_remove(self._draw_handler, 'WINDOW')

        # if bpy.context.scene.cursor.rotation_euler[0:3] == 0:
        #     bpy.ops.object.transform_apply(location=False, rotation=True, scale=False)

        # args = {"modal_action": 'thickness'}
        # bpy.ops.speedflow.solidify('INVOKE_DEFAULT', **args)
        if not self.custom_prim:
            bpy.ops.speedflow.solidify('INVOKE_DEFAULT')

# -------------------------------------
# CANCEL & EXIT
# -------------------------------------
    def cancel_exit(self, context):
        for obj in context.selected_objects:
            if obj != self.act_obj:
                bpy.ops.object.mode_set(mode='OBJECT')
                bpy.data.objects.remove(obj, do_unlink=True)

        if self.has_selection:
            context.view_layer.objects.active = self.act_obj
            self.act_obj.select_set(state=True)

        bpy.ops.wm.tool_set_by_id(name="builtin.select")

        if self.prim_plan:
            bpy.types.SpaceView3D.draw_handler_remove(self._handle, 'WINDOW')
        bpy.types.SpaceView3D.draw_handler_remove(self._draw_handler, 'WINDOW')

        # restore
        context.scene.tool_settings.use_snap = self.snap
        context.space_data.overlay.show_overlays = self.overlay_state
        context.space_data.overlay.show_wireframes = self.overlay_wire_state
        context.scene.transform_orientation_slots[0].type = str(self.orientation_mode)

    # def Check_Ref_object_Exist(self):
    #     if not bpy.context.scene.objects.get(self.ref_object_name):
    #         del (self.ref_object[:])

# -------------------------------------
# MODAL
# -------------------------------------
    # MODAL
    def modal(self, context, event):
        context.area.tag_redraw()

# -----  PASS TRHOUG EVENTS
        if event.type == 'MIDDLEMOUSE':
            return {'PASS_THROUGH'}

        # if event.type == 'RIGHTMOUSE' and event.value == 'PRESS':
        #     return {'PASS_THROUGH'}

        if event.type == 'ONE' and event.value == 'PRESS' and not self.new_bool:
            # bpy.ops.wm.call_menu(name="SPEEDFLOW_MT_custom_primitives")
            bpy.ops.view3d.speedflow_custom_primitives()
            self.custom_prim = True


# ----- CREATE PRIMITIVES
#         if event.type == 'LEFTMOUSE' and event.value == 'PRESS' and not self.new_bool and self.custom_prim:
#             if event.alt:
#                 self.on_cursor = True
#
#             if not self.on_cursor:
#                 bpy.ops.wm.tool_set_by_id(name="builtin.cursor")
#                 bpy.ops.view3d.cursor3d('INVOKE_DEFAULT', use_depth=True, orientation='GEOM')
#
#             if self.perspective and not self.orient and self.new_prim:
#                 bpy.context.scene.cursor.rotation_euler[1] = 0
#                 bpy.context.scene.cursor.rotation_euler[2] = 0
#                 bpy.context.scene.cursor.rotation_euler[0] = 0
#
#             self.custom_prim_cutter(context)
#             self.new_bool = True

        elif event.type == 'LEFTMOUSE' and event.value == 'PRESS' and not self.new_bool :
            if event.alt:
                self.on_cursor = True

            if not self.on_cursor:
                bpy.ops.wm.tool_set_by_id(name="builtin.cursor")
                bpy.ops.view3d.cursor3d('INVOKE_DEFAULT', use_depth=True, orientation='GEOM')

            if self.perspective and not self.orient and self.new_prim:
                bpy.context.scene.cursor.rotation_euler[1] = 0
                bpy.context.scene.cursor.rotation_euler[2] = 0
                bpy.context.scene.cursor.rotation_euler[0] = 0

                if self.cursor_Z_0:
                    bpy.context.scene.cursor.location[2] = 0

            if self.custom_prim:
                self.custom_prim_cutter(context)

            else:
                self.custom_prim = False

                if event.shift:
                    self.create_screw()

                elif event.ctrl:
                    self.create_vertex_line()
                    self.cutter_move()

                else :
                    self.create_plan()
                    self.cutter_move()
                    args = (self, None)
                    self._handle = bpy.types.SpaceView3D.draw_handler_add(draw_callback_px_rectangle, args, 'WINDOW', 'POST_VIEW')
                # self.z_axis = True

            self.new_bool = True

        # elif event.type == 'LEFTMOUSE' and event.value == 'RELEASE' and self.new_bool and self.z_axis:
        #     self.cutter_move()

# ----- Vertex
        elif event.type == 'LEFTMOUSE' and event.value == 'RELEASE' and self.new_bool:
            if self.prim_vertex_line:
                bpy.ops.mesh.extrude_region_move()
                self.cutter_move()

# ----- TRANSLATE VERTEX PLAN
        elif event.type == 'MOUSEMOVE' and self.new_bool:
            if self.prim_screw:
                delta = (self.first_mouse_x - event.mouse_x) / 10 if event.shift else self.first_mouse_x - event.mouse_x
                bpy.context.active_object.modifiers["Displace - X"].strength = self.strength + min(100,-(delta / 100) / 5)
            elif self.custom_prim:

                bpy.ops.transform.resize('INVOKE_DEFAULT')

# ----- rotation
        if event.value == 'PRESS' and event.type == 'Z':
            bool_obj = context.object
            if event.ctrl and event.shift:
                local_rotate(bool_obj, 'Z', 5)
            elif event.shift:
                local_rotate(bool_obj, 'Z', 45)
            else:
                local_rotate(bool_obj, 'Z', 90)
        if event.value == 'PRESS' and event.type == 'Y':
            bool_obj = context.object
            if event.ctrl and event.shift:
                local_rotate(bool_obj, 'Y', 5)
            elif event.shift:
                local_rotate(bool_obj, 'Y', 45)
            else:
                local_rotate(bool_obj, 'Y', 90)
        if event.value == 'PRESS' and event.type == 'X':
            bool_obj = context.object
            if event.ctrl and event.shift:
                local_rotate(bool_obj, 'X', 5)
            elif event.shift:
                local_rotate(bool_obj, 'X', 45)
            else:
                local_rotate(bool_obj, 'X', 90)

# ----- MIRROR PLAN
        if event.type == 'Q' and event.value == 'RELEASE' and self.new_bool:
            if self.prim_plan:
                self.cutter_move()
                self.mod_mirror.show_viewport = not self.mod_mirror.show_viewport

# ----- TRANSLATE PLAN (son vertex) EDIT MODE
        elif event.type == 'G' and event.value == 'RELEASE' and self.new_bool:
            if self.prim_plan:
                if self.new_bool:
                    self.cutter_move()

# ----- TRANSLATE PLAN OBJECT MODE
        if event.shift and event.type == 'SPACE' and event.value == 'PRESS':
            bpy.ops.object.mode_set(mode='OBJECT')
            self.cutter_move()
            bpy.ops.object.mode_set(mode='EDIT')

# ----- CANCEL AND EXIT
        if event.type == 'RIGHTMOUSE':
            self.cancel_exit(context)
            return {'CANCELLED'}

        if event.type == 'ESC' and not self.new_bool:
            self.cancel_exit(context)
            return {'CANCELLED'}

# ----- ECHAP
        if event.type == 'ESC' and self.new_bool:
            # Reset variables
            self.new_bool = False
            self.prim_bisect = False
            self.prim_vertex_line = False
            self.prim_screw = False
            self.on_cursor = False

            for obj in context.selected_objects:
                if obj != self.act_obj:
                    bpy.ops.object.mode_set(mode='OBJECT')
                    bpy.data.objects.remove(obj, do_unlink=True)

            if self.has_selection:
                context.view_layer.objects.active = self.act_obj
                self.act_obj.select_set(state=True)

            if self.prim_plan:
                bpy.types.SpaceView3D.draw_handler_remove(self._handle, 'WINDOW')

            self.prim_plan = False

            bpy.ops.wm.tool_set_by_id(name="builtin.select")

# ----- FINILIZE CUTTING
        elif event.type == 'SPACE' and event.value == 'RELEASE' and self.new_bool:
            self.finilize_cutting(context)
            return {'FINISHED'}

        return {'RUNNING_MODAL'}

    def invoke(self, context, event):
        if context.area.type == 'VIEW_3D':

            SF = context.window_manager.SF
            # self.ref_object = self.ref_object[0]
            # self.ref_object_name = SF.ref_object_name


            self.act_obj = context.active_object

            self.sel = self.filter_objects_by_type(context.selected_objects, {'MESH', 'CURVE', 'FONT'})
            self.sel_rebool = self.filter_objects_by_type(context.selected_objects, {'MESH', 'CURVE', 'FONT'})

            self.shading_viewport_type = context.space_data.shading.type
            self.overlay_state = context.space_data.overlay.show_overlays
            self.overlay_wire_state = context.space_data.overlay.show_wireframes
            self.parent_bool_objects = self.prefs.boolean.parent_bool_objects
            self.boolean_union = False
            self.on_cursor = False
            self.rebool = False
            self.perspective = context.area.spaces.active.region_3d.is_perspective
            self.shading = bpy.context.space_data.shading
            self.snap = bpy.context.scene.tool_settings.use_snap
            self.orientation_mode = bpy.context.scene.transform_orientation_slots[0].type
            bpy.context.scene.transform_orientation_slots[0].type = 'CURSOR'
            bpy.context.scene.tool_settings.use_snap = False

            # Disable Auto Merge
            tool_settings = context.tool_settings
            tool_settings.use_mesh_automerge = False

            # obj = context.view_layer.objects.get(self.ref_object[0].name)
            # if obj is None:
            #     del (self.ref_object[:])

            # if self.ref_object[0] in context.scene.objects:
            #     print("yes")
            # else:
            #     del (self.ref_object[:])
            # try:
            #     if len(self.ref_object) == 0:
            #         del (self.ref_object[:])
            # except IndexError:
            #     pass

            #     print("Il y a un objet dans la liste")
            # else:
            #     for obj in self.ref_object:
            #         if obj.data.users < 1:
            #             del (self.ref_object[:])
            #             context.objects.remove(obj, do_unlink=True)
            #     print("Il n'y a plus d'objet dans la liste")
            # has_ref_object = context.view_layer.objects.get(self.ref_object_name)
            # has_ref_object = context.scene.objects.get(self.ref_object_name)
            # if not bpy.context.scene.objects.get(self.ref_object_name):
            # # if not has_ref_object:
            #     del (self.ref_object[:])
            # self.Check_Ref_object_Exist()

            # has_ref_object = [obj for obj in bpy.data.objects if obj.name not in context.scene.objects]
            # if not has_ref_object == self.ref_object_name:
            #     del (self.ref_object[:])

            self.ref_object = [obj for obj in context.scene.objects if obj.get("is_ref")]
            self.last_ref_object = [obj for obj in context.scene.objects if obj.get("is_last_ref")]

            # Check if Selection
            self.has_selection = False
            if context.selected_objects:
                self.has_selection = True
                if context.object.mode != 'OBJECT':
                    bpy.ops.object.mode_set(mode='OBJECT')

                # Si un modifier 'BOOLEAN' ne possede pas d'objet, on le supprime
                for obj in self.sel:
                    for mod in obj.modifiers:
                        if mod.type == 'BOOLEAN' and not mod.object:
                            self.remove_modifier(obj, mod)

                # Add / Replace / Clear Self.ref_object
                if event.ctrl and event.shift and event.alt:
                    for obj in self.ref_object:
                        if obj.get("is_ref"):
                            del obj["is_ref"]

                    self.report({'INFO'}, "Ref Object Removed")

                    return {'FINISHED'}

                elif event.ctrl and event.shift:

                    if len(self.sel) == 1:
                        if self.ref_object:
                            if self.act_obj == self.ref_object[0]:
                                del self.act_obj["is_ref"]
                                self.report({'INFO'}, "Ref object removed")

                            else:
                                del self.ref_object[0]["is_ref"]
                                self.act_obj["is_ref"] = True
                                self.report({'INFO'}, "Ref object replaced")

                        else:
                            self.act_obj["is_ref"] = True
                            self.report({'INFO'}, "Ref object added")

                    return {'FINISHED'}

                elif event.shift:
                    SF.override_solidify_z_o = False
                    self.boolean_union = True
                    self.bool_op = True

                elif event.ctrl:
                    SF.override_solidify_z_o = False
                    self.rebool = True
                    self.bool_op = True

                elif event.alt :
                    SF.override_solidify_z_o = True
                    self.orient = True
                    self.bool_op = False
                    self.new_prim = True
                    self.has_selection = False
                else:
                    SF.override_solidify_z_o = False
                    self.bool_op = True
            else:
                SF.override_solidify_z_o = True
                self.orient = False
                self.bool_op = False
                self.new_prim = True
                self.has_selection = False

                if event.shift:
                    self.cursor_Z_0 = True

                # bpy.context.scene.tool_settings.use_snap = True
                # bpy.context.scene.tool_settings.snap_elements = {'INCREMENT'}
                # bpy.context.scene.tool_settings.use_snap_grid_absolute = True

            self.cursor_rot = bpy.context.scene.cursor.rotation_euler.copy()

            # Check si collection existe, si non, on la cree
            if self.prefs.add_bool_objects_to_collection:
                if not bpy.data.collections.get('Bool_Objects'):
                    coll = bpy.data.collections.new("Bool_Objects")
                    bpy.context.collection.children.link(coll)

            self.first_mouse_x = event.mouse_x
            self.first_mouse_y = event.mouse_y

            self.setup_draw_handler(context)
            context.window_manager.modal_handler_add(self)
            return {'RUNNING_MODAL'}
        else:
            self.report({'WARNING'}, "View3D not found, cannot run operator")
            return {'CANCELLED'}


def register():
    try:
        bpy.utils.register_class(speedflow_OT_custom_primitives)
        bpy.utils.register_class(SPEEDFLOW_OT_cutter)
    except:
        print("Cutter already registred")

def unregister():
    bpy.utils.unregister_class(speedflow_OT_custom_primitives)
    bpy.utils.unregister_class(SPEEDFLOW_OT_cutter)