/*------------------------------------------------------------------------------*
 * File Name: mscomm.h															*
 * Creation:  JCG 8/22/2001														*
 * Purpose: Origin C serial communication functions								*
 * Copyright (C) OriginLab Corp.2001											*
 * All Rights Reserved															*
 *------------------------------------------------------------------------------*/


#ifndef _MSCOMM_
#define _MSCOMM_

#pragma dll(kernel32, system)


// Baud rates at which the communication device operates
#define CBR_110             110
#define CBR_300             300
#define CBR_600             600
#define CBR_1200            1200
#define CBR_2400            2400
#define CBR_4800            4800
#define CBR_9600            9600
#define CBR_14400           14400
#define CBR_19200           19200
#define CBR_38400           38400
#define CBR_56000           56000
#define CBR_57600           57600
#define CBR_115200          115200
#define CBR_128000          128000
#define CBR_256000          256000

typedef struct _DCB {
    DWORD DCBlength;     		// sizeof(DCB)                    
    DWORD BaudRate;       		// Baudrate at which running      
    DWORD fBinary: 1;     		// Binary Mode (skip EOF check)    
    DWORD fParity: 1;     		// Enable parity checking         
    DWORD fOutxCtsFlow:1; 		// CTS handshaking on output       
    DWORD fOutxDsrFlow:1; 		// DSR handshaking on output      
    DWORD fDtrControl:2;  		// DTR Flow control              
    DWORD fDsrSensitivity:1; 	// DSR Sensitivity              
    DWORD fTXContinueOnXoff: 1; // Continue TX when Xoff sent 
    DWORD fOutX: 1;       		// Enable output X-ON/X-OFF        
    DWORD fInX: 1;        		// Enable input X-ON/X-OFF        
    DWORD fErrorChar: 1;  		// Enable Err Replacement          
    DWORD fNull: 1;       		// Enable Null stripping           
    DWORD fRtsControl:2;  		// Rts Flow control                
    DWORD fAbortOnError:1; 		// Abort all reads and writes on Error 
    DWORD fDummy2:17;     		// Reserved                       
    WORD wReserved;       		// Not currently used              
    WORD XonLim;          		// Transmit X-ON threshold         
    WORD XoffLim;         		// Transmit X-OFF threshold       
    BYTE ByteSize;        		// Number of bits/byte, 4-8        
    BYTE Parity;          		// 0-4=None,Odd,Even,Mark,Space    
    BYTE StopBits;        		// 0,1,2 = 1, 1.5, 2               
    char XonChar;         		// Tx and Rx X-ON character        
    char XoffChar;        		// Tx and Rx X-OFF character       
    char ErrorChar;       		// Error replacement char          
    char EofChar;         		// End of Input character          
    char EvtChar;         		// Received Event character        
    WORD wReserved1;      		// Fill for now.                
} DCB, *LPDCB;

typedef struct _COMMTIMEOUTS {
    DWORD ReadIntervalTimeout;          // Maximum time between read chars. 
    DWORD ReadTotalTimeoutMultiplier;   // Multiplier of characters.        
    DWORD ReadTotalTimeoutConstant;     // Constant in milliseconds.        
    DWORD WriteTotalTimeoutMultiplier;  // Multiplier of characters.       
    DWORD WriteTotalTimeoutConstant;    // Constant in milliseconds.        
} COMMTIMEOUTS, *LPCOMMTIMEOUTS;

#define NOPARITY            0
#define ODDPARITY           1
#define EVENPARITY          2
#define MARKPARITY          3
#define SPACEPARITY         4

#define ONESTOPBIT          0
#define ONE5STOPBITS        1
#define TWOSTOPBITS         2

#define	MAXWORD    		 	0xffff      
typedef DWORD  				*LPDWORD;

/** >Communications 
	Remarks:
		The SetCommState function configures a communications device according to the specifications 
		in a device-control block (a DCB structure). The function reinitializes all hardware and control 
		settings, but it does not empty output or input queues. 
	Example:
		file ff;

		//Open comm port
		if (!ff.Open("COM1:",  file::modeReadWrite ))
		{
			ASSERT(FALSE);
			return FALSE;
		}

		UINT hCom = ff.GetHandle();
		
		if (hCom == file::hFileNull ) 
		{
			ASSERT(FALSE);
			return FALSE;
		}
		
		DCB dcb;

		if (!GetCommState((HANDLE)hCom, &dcb)) 
		{
			ASSERT(FALSE);
			return FALSE;
		}

		//dcb parameters for user
		dcb.BaudRate = CBR_9600;  		// set the baud rate
		dcb.ByteSize = 8;   			// data size, xmit, and rcv
		dcb.Parity = NOPARITY;        	// no parity bit
		dcb.StopBits = ONESTOPBIT;    	// one stop bit
		
		//dcb fixed parameters 
		dcb.fBinary=1; 
		dcb.fParity=0; 
		dcb.fOutxCtsFlow=0; 
		dcb.fOutxDsrFlow=0; 
		dcb.fDtrControl=0; 
		dcb.fDsrSensitivity=0; 
		
		dcb.fTXContinueOnXoff=0; 
		dcb.fRtsControl=0;

		if (!SetCommState((HANDLE)hCom, &dcb)) 
		{
			ASSERT(FALSE);
			return FALSE;
		}
		
		//...
		
		if( !ff.Close() ) // Close() function will happen automatically by the file class destructor.
		{
			ASSERT(FALSE);
			return FALSE;
		}
	Parameters:
		hFile = handle to the communications device. 
		lpDCB = Pointer to a DCB structure that contains the configuration information for the specified communications device. 
	Return:
		If the function succeeds, the return value is nonzero.
		If the function fails, the return value is zero. 
	SeeAlso:
		file::Open, file::GetHandle, GetCommState.
*/
BOOL WINAPI SetCommState(
    HANDLE hFile,
    LPDCB lpDCB
    );
    
/** >Communications
	Remarks:
		The GetCommState function retrieves the current control settings for a specified communications device. 
	Example:
		file ff;

		//Open comm port
		if (!ff.Open("COM1:",  file::modeReadWrite ))
		{
			ASSERT(FALSE);
			return FALSE;
		}

		UINT hCom = ff.GetHandle();
		
		if (hCom == file::hFileNull ) 
		{
			ASSERT(FALSE);
			return FALSE;
		}
		
		DCB dcb;

		if (!GetCommState((HANDLE)hCom, &dcb)) 
		{
			ASSERT(FALSE);
			return FALSE;
		}

		//dcb parameters for user
		dcb.BaudRate = CBR_9600;  		// set the baud rate
		dcb.ByteSize = 8;   			// data size, xmit, and rcv
		dcb.Parity = NOPARITY;        	// no parity bit
		dcb.StopBits = ONESTOPBIT;    	// one stop bit
		
		//dcb fixed parameters 
		dcb.fBinary=1; 
		dcb.fParity=0; 
		dcb.fOutxCtsFlow=0; 
		dcb.fOutxDsrFlow=0; 
		dcb.fDtrControl=0; 
		dcb.fDsrSensitivity=0; 
		
		dcb.fTXContinueOnXoff=0; 
		dcb.fRtsControl=0;

		if (!SetCommState((HANDLE)hCom, &dcb)) 
		{
			ASSERT(FALSE);
			return FALSE;
		}

		//...
		
		if( !ff.Close() ) // Close() function will happen automatically by the file class destructor.
		{
			ASSERT(FALSE);
			return FALSE;
		}
	Parameters
		hFile = Handle to the communications device. 
		lpDCB = Pointer to a DCB structure that receives the control settings information. 
	Return:
		If the function succeeds, the return value is nonzero.
		If the function fails, the return value is zero. 	
	SeeAlso:
		file::Open, file::GetHandle, SetCommState.
		
*/
BOOL WINAPI GetCommState(
    HANDLE hFile,
    LPDCB lpDCB
    );

/** >Communications
	Remarks:
		The SetCommTimeouts function sets the time-out parameters for all read and write operations on a specified 
		communications device. 
	Example:
		file ff;

		//Open comm port
		if (!ff.Open("COM1:",  file::modeReadWrite ))
		{
			ASSERT(FALSE);
			return FALSE;
		}
		
		UINT hCom = ff.GetHandle();
		
		if ( hCom == file::hFileNull ) 
		{
			ASSERT(FALSE);
			return FALSE;
		}
		
		COMMTIMEOUTS tTimeout; 
		tTimeout.ReadIntervalTimeout = MAXWORD; 
		tTimeout.ReadTotalTimeoutMultiplier = 0; 
		tTimeout.ReadTotalTimeoutConstant = 500; // pas de time out = 0 
		tTimeout.WriteTotalTimeoutMultiplier = 0; 
		tTimeout.WriteTotalTimeoutConstant = 0; 
		
		// config the timeout 
		if( !SetCommTimeouts((HANDLE)hCom,&tTimeout) )
		{
			ASSERT(FALSE);
			return FALSE;
		}
		
		//...
		
		if( !ff.Close() ) // Close() function will happen automatically by the file class destructor.
		{
			ASSERT(FALSE);
			return FALSE;
		}
	Parameters:
		hFile = Handle to the communications device. 
		lpCommTimeouts = Pointer to a COMMTIMEOUTS structure that contains the new time-out values. 
	Return:
		If the function succeeds, the return value is nonzero.
		If the function fails, the return value is zero. 
	SeeAlso:
		file::Open, file::GetHandle, GetCommTimeouts.
*/
BOOL WINAPI SetCommTimeouts(
    HANDLE hFile,
    LPCOMMTIMEOUTS lpCommTimeouts
    );
    
/** >Communications
	Remarks:
		The GetCommTimeouts function retrieves the time-out parameters for all read and write operations on a 
		specified communications device. 
	Example:
		file ff;

		//Open comm port
		if (!ff.Open("COM1:",  file::modeReadWrite ))
		{
			ASSERT(FALSE);
			return FALSE;
		}
		
		UINT hCom = ff.GetHandle();
		
		if ( hCom == file::hFileNull ) 
		{
			ASSERT(FALSE);
			return FALSE;
		}
		
		COMMTIMEOUTS tTimeout; 
		// get the timeout 
		if( !GetCommTimeouts((HANDLE)hCom,&tTimeout) )
		{
			ASSERT(FALSE);  
			return FALSE;
		}

		//...
		
		if( !ff.Close() ) // Close() function will happen automatically by the file class destructor.
		{
			ASSERT(FALSE);
			return FALSE;
		}		
	Parameters:
		hFile = Handle to the communications device.
		lpCommTimeouts = Pointer to a COMMTIMEOUTS structure in which the time-out information is returned. 
	Return:
		If the function succeeds, the return value is nonzero.
		If the function fails, the return value is zero. 
	SeeAlso:
		file::Open, file::GetHandle, SetCommTimeouts.
*/   
BOOL WINAPI GetCommTimeouts(
    HANDLE hFile,
    LPCOMMTIMEOUTS lpCommTimeouts
    );

#endif //_MSCOMM_
