/*------------------------------------------------------------------------------*
 *	File Name: 	Stat.h	     													*
 *	Purpose:	Stat function prototypes for calling statistical functions		*
 *	Creation:	April 11, 2001 by GJL											* 
 *	Copyright (c) OriginLab Corp. 2001, 2002, 2003								*
 *	All Rights Reserved															*
 *	Modification log                                                        	*
 *------------------------------------------------------------------------------*/  

#ifndef _STAT_H
#define _STAT_H

// Origin C specific command to include an external DLL
// All function declarations will be for the importdll
// until another importdll is specified or when the end
// of the header file is reached

#pragma dll(@OUTL)

// Existing Statistical Distribution Functions

/** >Statistics
		Returns Student's t distribution for significance level 0<dP<1 and nDof>0.
	Example: 
		double dP=0.95;
		UINT nDof=14;
		double dt;
		dt=tTable(dP,nDof); // results in dt==1.761
		dt=tTable(0.05, 14); // results in dt==-1.761
	Parameters:
		dP=Significance level where 0<dP<1.  dP equals the area under the t distribution curve to the left of the
		returned value of t.
		nDof=Degrees of freedom where nDof>0.    
	Return:
		Value of t for given values of dP and nDof.  Returns a t value such that the area under the t distribution
		curve to the left of t equals dP.  
	SeeAlso:
		invt, probt 
*/
double	tTable(double dP, UINT nDof); // Student's t distribution.

/** >Statistics
		Student's inverse t distribution for value of t (all Reals) and nDof>0.
	Example: 
		double dt=1.761;
		UINT nDof=14;
		double dP;
		dP=invt(dt,nDof); // results in dP==0.95
		dt=-1.761;
		dP=invt(dt,nDof); // results in dP==0.05 
	Parameter:
		dt=Value of t (t can be any Real number) for which inverse t is returned.
		nDof=Degrees of freedom where nDof>0.
	Return:
		Student's inverse t distribution for abs(dt) and nDof.  Reurns the area under the t distribution curve
		to the left of abs(dt).
	SeeAlso:
		tTable, probt
*/
double	invt(double dt, UINT nDof); // Student's inverse t distribution.

/** >Statistics

		Returns F distribution for significance level 0<dP<1, nNumDof>0, and nDenDof>0.
	Example: 
		double dP=0.95;
		UINT nNumDof=2;
		UINT nDenDof=15;
		double dF;
		dF=Ftable(dP,nNumDof,nDenDof); // results in dF==3.68232
		nNumDof=4;
		nDenDof=10;
		dF=Ftable(dP,nNumDof,nDenDof); // results in dF==3.47805
	Parameters:
		dP=Significance level where 0<dP<1.  dP equals the area under the F distribution curve to the left
		of the returned value of F.
		nNumDof=Numerator degrees of freedom where nNumDof>0.
		nDenDof=Denominator degrees of freedom where nDenDof>0.
	Return:
		Value of F for given values of dP, nNumDof, and nDenDof.  Returns F value such that the area under
		the F distribution curve to the left of F equals dP.
	SeeAlso:
		invf
*/
double	Ftable(double dP, UINT nNumDof, UINT nDenDof); // F distribution.

/** >Statistics
		Inverse F distribution.
	Example: 
		double dF=3.68232;
		UINT nNumDof=2;
		UINT nDenDof=15;
		double dP;
		dP=invf(dF,nNumDof,nDenDof); // results in dP==0.95
	Parameters:
		dF=Value of F (F>0) for which inverse F is returned.
		nNumDof=Numerator degrees of freedom where nNumDof>0.
		nDenDof=Denominator degrees of freedom where nDenDof>0.
	Return:
		Inverse F distribution for dF, nNumDof, and nDenDof.  Reurns the area under the F distribution curve
		to the left of dF.   
	SeeAlso:
		Ftable
*/
double	invf(double dF, UINT nNumDof, UINT nDenDof); // Inverse F distribution.

/** >Statistics
		Error function (or normal error integral).
	Example:
		double dy,dx=1;
		dy=erf(dx); // results in dy==0.8427008
	Parameters:
		dx=Upper limit (dx is any Real) of the definite integral in the erf function.
	Return:
		erf(x)=2/sqrt(pi)*(definite integral of exp(-u^2)du from 0 to +x).
	SeeAlso:
		inverf
*/
double	erf(double dx); // Error function.

/** >Statistics
		Inverse error function.
	Example:
		double dx,dy=0.8427008;
		dx=inverf(dy); // results in dx==1
	Parameters:
		dy=Normal integral error (dy is any Real).
	Return:
		Upper limit of the definite integral in the erf function that returns a value of dy.
	SeeAlso:
		erf
*/
double	inverf(double dy); // Inverse error function.

/** >Statistics
		Probability density for a normal distribution.
	Example:
		double dA,dz=1;
		dA=prob(dz); // results in dA==0.6826895
	Parameters:
		dz=Range of z values (-z to +z) for which the area under the normal curve is returned (dz can be any Real number).
	Return:
		Area under the normal curve between -z and +z.  prob(x)=1/sqrt(2*pi)*(definite integral of exp((-t^2)/2)dt
		from -x to +x).
	SeeAlso:
		invprob
*/
double	prob(double dz); // Probability density for a normal distribution.

/** >Statistics
		Inverse probability density for a normal distribution.
	Example:
		double dz,dA=0.6826895;
		dz=invprob(dA); // results in dz==1
	Parameters:
		dA=Area under the normal curve between -z and +z where 0<dA<1.
	Return:
		The z value such that the area under the normal curve between -z and +z is dA.
	SeeAlso:
		prob
*/
double	invprob(double dA); // Inverse probability density for a normal distribution.

/** >Statistics
		Inverse cumulative probability density for a normal distribution.
		This function is related to invprob by the following relationship
		invprob(x) = invncp((x+1)/2)
	Parameters:
		dA=Area under the normal curve between negative infinity and z where 0<dA<1.
	Example:
		double dz,dA=0.8413447;
		dz=invncp(dA); // results in dz==1
	Return:
		The z value such that the area under the normal curve between -INF and z is dA.
	SeeAlso:
		invprob
*/
double invncp(double dA);
	/**#
	Example:
		void test_invncp1()
		{
			double dz,dA=0.8413447;
			dz=invncp1(dA); // results in dz==1   
	
			out_double("The value is ",dz);
		}
	*/
double invncp1(double x);  // different algorithm, does not work as well in small x
	/**#
	Example:
		void test_invncp2()
		{
				double dz,dA=0.8413447;
				dz=invncp2(dA); // results in dz==1   
		
				out_double("The value is ",dz);
		}
	*/
double invncp2(double x);  // different algorithm, does not work as well in small x

/** >Statistics
		Quality Control D2 Factor.
	Example:
		UINT nn=4;
		double dD2;
		dD2=QCD2(nn); // results in dD2==2.058750
	Parameters:
		nn=Sample or subgroup size where nn>0.
	Return:
		Quality Control D2 Factor which is used to estimate the standard deviation of a parent distribution
		(or population) from an average range.  The standard deviation of the parent distribution=average range/factor.
		The returned factors assume sampling from a normal population.
	SeeAlso:
		QCD3, QCD4
*/
double	QCD2(UINT nn); // Quality Control D2 Factor.

/** >Statistics
		Quality Control D3 Factor.
	Example:
		UINT nn=11;
		double dD3;
		dD3=QCD3(nn); // results in dD3==0.256
	Parameters:
		nn=Sample or subgroup size where nn>0. 
	Return:
		Quality Control D3 Factor for determining the 3-sigma lower control limit for R charts (Range of Sample Charts)
		from the average range.  The Lower Control Limit for R=(factor)*(Average Range).  The calculations for the factors
		are based on the normal distribution.
	SeeAlso:
		QCD2, QCD4
*/
double	QCD3(UINT nn); // Quality Control D3 Factor.

/** >Statistics
		Quality Control D4 Factor.
	Example:
		UINT nn=11;
		double dD4;
		dD4=QCD4(nn); // results in dD4==1.744
	Parameters:
		nn=Sample or subgroup size where nn>0.
	Return:
		Quality Control D4 Factor for determining the 3-sigma upper control limit for R charts (Range of Sample Charts)
		from the average range.  The Upper Control Limit for R=(factor)*(Average Range).  The calculations for the factors
		are based on the normal distribution.
	SeeAlso:
		QCD2, QCD3
*/
double	QCD4(UINT nn); // Quality Control D4 Factor.

// New Statistical Functions
/** >Statistics
		Student's inverse t distribution.
	Example: 
		double dt=1.761;
		UINT nDof=14;
		double dP;
		dP=probt(dt,nDof); // results in dP==0.95
		dt=-1.761;
		dP=probt(dt,nDof); // results in dP==0.05
	Parameters:
		dt=Value of t (t can be any real number) for which inverse t is returned.
		nDof=Degrees of freedom where nDof>0.
	Return:
		Student's inverse t distribution for dt [not abs(dt)] and nDof.  Reurns the area under the t distribution
		curve to the left of dt.   
	SeeAlso:
		tTable, invt
*/
double	probt(double dt, UINT nDof); // Student's inverse t distribution.

#endif  //_STAT_H