/*------------------------------------------------------------------------------*
 * File Name: Page.h															*
 * Creation: TD 10/01/2001													    *
 * Purpose: Origin C header	for Page class and other related functions			*
 * Copyright (c) OriginLab Corp.2001											*
 * All Rights Reserved															*
 * 																				*
 * Modification Log:															*
 *------------------------------------------------------------------------------*/
#ifndef _PAGE_H
#define _PAGE_H
#include <OC_const.h> // consts used in Origin internal functions
#include <Collection.h>
#include <OrgObj.h>

/** >Internal Origin Objects
		The PageBase class provides methods and properties common to all internal Origin
		pages (windows). One use of this class is to write functions that accept a PageBase
		object rather than a page object of specific type. Another use is to attach a PageBase
		object to an active page whose page type is unknown. These uses facilitate writing
		general code that can then branch to handle specific page types. The PageBase object
		can be cast to an object of its page type in order to access methods and properties
		specific to its class. In general, however, the Origin C PageBase class is intended
		to be an abstract base class to be used for polymorphic handling of derived page types.
		Derived classes, such as Note, GraphPage, WorksheetPage, LayoutPage, and MatrixPage
		inherit PageBase class methods and should be used instead.
	Example:
		// Get active window...if not valid...
		PageBase pbActiveWindow;
		string strWindowName;
		int iPageType;
		
		pbActiveWindow = Project.Pages();
		if( !pbActiveWindow.IsValid() )
			return 1; // Return an error code	
	
		// Get window name and type
		strWindowName = pbActiveWindow.GetName();
		iPageType = pbActiveWindow.GetType();
	
		// If window is not a worksheet or workbook...
		if( iPageType != EXIST_WKS && iPageType != EXIST_EXTERN_WKS )
			return 1; // Return an error code
*/
class PageBase : public OriginObject
{
public:
	/**
		Default constructor for a PageBase object.
	Examples:
		PageBase pbTemp;
		pbTemp = Project.Pages(); // Get the project's active page
		if( pbTemp.IsValid() )
			printf("Active page is of type %d\n", pbTemp.GetType());
		else
			printf("Active page is invalid\n");
	*/
	PageBase(); // Default constructor.

	/**
		Construct a PageBase object using the name of an existing page.
	Parameters:
		lpcszName = The name of an existing page.
	Example:
		void output_page_type(string strName)
		{
			PageBase pb(strName);
			if( pb.IsValid() )
				printf("Page %s is of type %d\n", strName, pb.GetType());
			else
				printf("Page %s is invalid\n", strName);
		}

		void test_output_page_type()
		{
			MatrixPage mp;
			mp.Create("origin.otm");
			if( mp.IsValid() )
				output_page_type(mp.GetName());
		}
	*/
	PageBase(LPCSTR lpcszName);

	/**
		Construct a PageBase object using another PageBase object.
	Parameters:
		page = An existing PageBase object.
	Example:
		void output_page_type(PageBase &pb)
		{
			PageBase pbTemp(pb);
			if( pbTemp.IsValid() )
				printf("Page is of type %d\n", pbTemp.GetType());
			else
				printf("Page is invalid\n");
		}

		void test_output_page_type()
		{
			MatrixPage mp;
			mp.Create("origin.otm");
			if( mp.IsValid() )
				output_page_type(mp);
		}
	*/
	PageBase(PageBase &page);

	/**
		Get the page type.
	Example:
		PageBase pbTemp;
		pbTemp = Project.Pages(); // Get the project's active page
		if( pbTemp.IsValid() )
			printf("Active page is of type %d\n", pbTemp.GetType());
		else
			printf("Active page is not valid\n");
	Return:
		A value representing the page type.  See the EXIST constants defined in oc_const.h, they are:
			EXIST_WKS       2       
			EXIST_PLOT      3
			EXIST_MATRIX    5
			EXIST_LAYOUT	11
			EXIST_EXTERN_WKS	12  (Excel workbook)
	*/
	int GetType(); // Get the page type.

	/**
		Get the name of the page.
	Example:
		PageBase pb;
		pb = Project.Pages(); // Get the project's active page
		if( pb.IsValid() )
		{
			string strName;
			if( pb.GetName(strName) )
				printf("Active page is named %s\n", strName);
			else
				printf("Failed to get page name\n");
		}
		else
			printf("Active page is not valid\n");
	Parameters:
		strName = The string that will recieve the name of the page.
	Return:
		TRUE for success or FALSE for failure.
	SeeAlso:
		PageBase::Rename
	*/
	BOOL GetName(string &strName);

#if  _OC_VER < 0x0750
	/**
		Get the name of the page.
	Example:
		PageBase pb;
		pb = Project.Pages(); // Get the project's active page
		if( pb.IsValid() )
			printf("Active page is named %s\n", pb.GetName());
		else
			printf("Active page is not valid\n");
	Return:
		The name of the page.
	SeeAlso:
		PageBase::Rename
	*/
	string GetName();
#endif //_OC_VER < 0x0750

	/**
		Change the name of the page.
	Parameters:
		lpcszNewName = Pointer to the string that holds the new name of the page.
		bAskIfAlreadyUsed = TRUE will bring up a dialog box to ask for a new name, FALSE will use internal enumeration for the next available name
	Return:
		1 = given name is used to rename the page successfully, 
		0 = another name was used to rename the page, either through user input or through internal enumeration.
		-1 = user click Cancel when ask for a different name to rename the page when bAskIfAlreadyUsed is set to TRUE.
	Example:
		void	run_this()
		{
			PageBase	pg = Project.Pages(); // Get the active page
			if( pg )
			{
				string		strOldName = pg.GetName();
				string		strNewName = strOldName + "AA";
				if( pg.Rename(strNewName) >=0 )
					printf("Page renamed from %s to %s\n", strOldName, strNewName);
				else
					printf("Failed to rename page\n");
			}
			else
				printf("There is no Active page\n");
		}
	Remarks:
		Please note that this function was changed from its original form

			BOOL  Rename(string strNewName);

		to this new form in Origin 7 SR4, or Origin 8 Alpha3. The return values were setup such that old codes will still work
		correctly.
	SeeAlso:
		PageBase::GetName
	*/
	int Rename(LPCSTR lpcszNewName, BOOL bAskIfAlreadyUsed = FALSE);
	
	/**
		Detach a Page object from a page.  This will make the object invalid.
	Retun:
		TRUE for success or FALSE for failure.
	Example:
		GraphPage gp;
		if( gp.Create("origin.otp", CREATE_VISIBLE_SAME) )
		{
			string strName = gp.GetName();
			printf("Created a graph named %s\n", strName);
			
			if( gp.Detach() )
				printf("Detached %s, IsValid == %d\n", strName, gp.IsValid());
			else				
				printf("Failed to detach %s\n", strName);
		}
		else
			printf("Failed to create graph page.\n");
	*/	
	BOOL Detach(); 
	
	/**
		This data member contains the labet text associated with the PageBase object.
	Example:
		WorksheetPage wp;
		if( wp.Create("origin.otw", CREATE_VISIBLE_SAME) )
		{
			wp.Label = "My Label";
			printf("Worksheet '%s' has label '%s'\n", wp.GetName(), wp.Label);
		}
		else
			printf("Failed to create worksheet page.\n");
	*/
	string Label;

	/**
		This data member controls the display of window names and labels of the PageBase object.
		It can be the following values, which defined OC_Const.h:
			WIN_TITLE_SHOW_LABEL	1
			WIN_TITLE_SHOW_NAME		2
			WIN_TITLE_SHOW_BOTH		3

	Example:
		WorksheetPage wp;
		if( wp.Create("origin.otw", CREATE_VISIBLE_SAME) )
		{
			wp.Label = "My Label";
			wp.TitleShow = WIN_TITLE_SHOW_LABEL;
		}
	*/
	int TitleShow;

	/**
		To open a hidden page, or to show a minimized window
	Retun:
		the original show state of the page if successful. Returns -1 if operation failed.
	Remarks:
		This function does the same thing as if double-clicking on the page in the Project explorer window if
		PAGE_ACTIVATE is used, which is not one of the show state that can be obtained from GetShow.
	Example:
		Worksheet wksTemp;
		if(wksTemp.Create(NULL, CREATE_HIDDEN))
		{
			// codes till fill wks with data

			// then show it at the end when all is done
			wksTemp.GetPage().Rename("MyNewWks");
			wksTemp.GetPage().SetShow();
		}
	*/
	int SetShow(int nNewShow = PAGE_ACTIVATE);

	/**
		To get the show state of the page
	Return:
		One of the values 	PAGE_NORMAL, PAGE_MINIMIZED, PAGE_MAXIMIZED, PAGE_HIDDEN,

	Example:
		void all_pages_show_info()
		{
			
			foreach(PageBase pg in Project.Pages)
			{
				if(pg.GetShow() == PAGE_HIDDEN)
					printf("%s is hidden\n", pg.GetName());
				else
					printf("%s Show state is %d\n", pg.GetName(), pg.GetShow());
			}
		}
	*/
	int	GetShow();

	#if  _OC_VER > 0x0703
	/**#
		This function returns a Window object associated with a page object.
	Parameters:
		nChildID = Resource ID of a window in the page. Use 0 to get the page's frame window (MDI ChildFrame)
	Example:	
		void test_PageBase_GetWindow()
		{
			WorksheetPage wpMy("Data1");
			Window winMyNN = wpMy.GetWindow();
			if (winMyNN)
				printf("Get window successfully!\n");
			else 
				printf("Get window error!\n");
		}
	*/
	Window		GetWindow(int nChildID = 0);

	/**
		Print the page.
	Parameters:
		tn = tree node which stores settings related to printing.
	Return:
		TRUE if succeded, FALSE if failed.
	Example:
		GraphPage page("Graph1");								// page to print
		Tree tree;
		tree.Printing.PrinterName.strVal = "Canon Bubble-Jet BJC-2000";	// ignore default printer, use Canon instead
		tree.Printing.NumberOfCopies.nVal = 2;							// print 2 copies
		tree.Printing.DPI.nVal = 300;									// resolution 300 dpi
		BOOL bResult = page.Print(tree);
	*/
	BOOL Print(TreeNode& tn);

	#endif // _OC_VER > 0x0703
	
	/**
	   get the folder of the page
	 Return:
	     A valid Folder object if successful, otherwise a NULL object
	Example:
		PageBase pbTemp = Project.Pages();
		if(pbTemp)
		{
		   Folder fd = pbTemp.GetFolder();
		   if(fd)
		     printf("%s is at %s\n", pbTemp.GetName(), fd.GetPath());
		   else
		      printf("Error: %s is not in any folder, impossible!\n", pbTemp.GetName());
		}
	*/
	Folder GetFolder();

	#if  _OC_VER >= 0x0750
	/**
		Get the page's creation and last modified time, size, and number of dependents
	Return:
		TRUE if succeded, FALSE if failed.
	Parameters:
       pInfo=the pointer to a structure to receive the information:
       typedef struct tagPageSystemInfo
       {
           double dCreated;
           double dModified;
           int nSize;
           int nDependents;
       } PageSystemInfo;
	Example:
		PageBase		pg = Project.Pages();
		PageSystemInfo	PgInfo;

		bool	bb = pg.GetPageSystemInfo(&PgInfo);

		double	dCreated = PgInfo.dCreated;
		double	dModified = PgInfo.dModified;
		int		nSize = PgInfo.nSize;
		int		nDependents = PgInfo.nDependents;
	*/
	BOOL GetPageSystemInfo(PageSystemInfo *pInfo);
	
	
	/**
		Saves the page to file. It corresponds to the individual window saving from menu. For
		Excel it saves it as a native Excel workbook file. For Note it saves it as a text file.
	Parameters:
		lpcszPathName=full pathname to the file location. The extension must be appropriate for type:
							WorksheetPage:				.OGW
							WorksheetPage if Excel:		.XLS
							GraphPage:					.OGG
							Note						extension not important
							
	Return:
		TRUE if succeded, FALSE if failed.
	Example:
		void	test_SaveToFile()
		{
			// Make sure that a graph with the name "Graph1" exists.
			GraphPage			pg("Graph1");
			if (!pg)
			{
				out_str("Invalid page!");
				return;
			}
			
			
			// For this to succeed, the folder c:\mydatafolder\ must exist in advance.
			BOOL				bOK = pg.SaveToFile("c:\\mydatafolder\\save test.ogg");
			
			out_int("OK = ", bOK);
		}
	*/
	BOOL	SaveToFile(LPCSTR lpcszPathName);
	#endif //_OC_VER >= 0x0750

};

/** >Internal Origin Objects
		The Page class provides methods and properties common to all internal Origin pages
		that contain one or more layers (all Origin windows except Note windows). The Page
		class contains a collection of all layers in the page. An Origin C Page object is a
		wrapper object that is a reference to an internal Origin page object. Origin C wrapper
		objects do not actually exist in Origin and merely refer to the internal Origin object.
		Consequently, multiple Origin C wrapper objects can refer to the same internal Origin
		object. The Page class is derived from the PageBase and OriginObject classes from which
		it inherits methods and properties.
	Example:
		// Assumes Origin page with graph (text, arrow, rectangle, etc.) object is active window
		Page pg;
		pg = (Page) Project.Pages(); // Get the project's active page
		if( pg.IsValid() )
		{
			Layer lyr;
			lyr = pg.Layers(0); // Get first layer in page
			if( pg.IsValid() )
			{
				GraphObject go;
				go = lyr.GraphObjects(0); // Get first GraphObject in layer
				printf("Name of GraphObject is %s\n", go.GetName());
			}
		}
*/
class Page : public PageBase
{
public:
	/**
		Default constructor for a Page object.
	Examples:
		Page pg;
		pg = (Page) Project.Pages(); // Get the project's active page
		if( pg.IsValid() )
			printf("Active page is of type %d\n", pg.GetType());
		else
			printf("Active page is invalid\n");
	SeeAlso:
		Page::Create
	*/
	Page();

	/**
		Construct a Page object using the name of an existing page.
	Parameters:
		lpcszName = The name of an existing page.
	Example:
		void output_page_type(string strName)
		{
			Page pg(strName);
			if( pg.IsValid() )
				printf("Page %s is of type %d\n", strName, pg.GetType());
			else
				printf("Page %s is invalid\n", strName);
		}

		void test_output_page_type()
		{
			MatrixPage mp;
			mp.Create("origin.otm");
			if( mp.IsValid() )
				output_page_type(mp.GetName());
		}
	SeeAlso:
		Page::Create
	*/
	Page(LPCSTR lpcszName);

	/**
		Construct a Page object using another Page object.
	Parameters:
		page = An existing Page object.
	Example:
		void output_page_type(Page &pg)
		{
			Page pgTemp(pg);
			if( pgTemp.IsValid() )
				printf("Page is of type %d\n", pgTemp.GetType());
			else
				printf("Page is invalid\n");
		}

		void test_output_page_type()
		{
			MatrixPage mp;
			mp.Create("origin.otm");
			if( mp.IsValid() )
				output_page_type(mp);
		}
	SeeAlso:
		Page::Create
	*/
	Page(PageBase &page);

	/**
		Create a new page using the supplied template and attach it to the object.
		The Create method should only be called from derived classes.
	Example:
		GraphPage gp;
		if( gp.Create("origin.otp", CREATE_VISIBLE_SAME) )
			printf("Created a graph named %s\n", gp.GetName());
		else
			printf("Failed to create graph page\n");
	Parameters: 
		lpcszTemplate = The template file name to create the page from.
			Pass an empty string or NULL to use the default template.
			Pass "0" to create without a template.
		iOption = How the window should be created.

			Must be one of the following:
			CREATE_TEMP = Create invisible and destroy on exit of scope.
			CREATE_VISIBLE_SAME = Create with a visibility setting the same as what is stored in the template.
			CREATE_VISIBLE = Create visible.
			CREATE_HIDDEN = Create the page so it appears hidden in Proejct Explorer.

			The following flags can be OR'ed with the above options:
			CREATE_NO_REMOVE_TEMPLATEPICT = Do not remove template preview picture (graphic object) on loading.
			CREATE_NO_GUI_ACCESS = Do not allow access from Project Explorer (only valid with CREATE_HIDDEN).
	Return:
		TRUE for success or FALSE for failure.
	SeeAlso:
		PageBase::PageBase
	*/
	BOOL Create(LPCSTR lpcszTemplate = NULL, int iOption = CREATE_VISIBLE);
 
	/**
		A Collection of all layers in a page
	Example:
		GraphPage gp;
		gp.Create("origin.otp");
		if( gp.IsValid() )
		{
			int iLayers;
			foreach(GraphLayer gl in gp.Layers)
				iLayers++;
			printf("%s has %d layer(s)\n", gp.GetName(), iLayers);
		}
	*/
	Collection<Layer> Layers;

	/**
		Get a layer object by index.
	Parameters: 
		iIndex = The index of the layer requested to be accessed.
		If -1 then the active layer is returned.
	Return:
		If iIndex is valid then the layer to which the index refers is returned otherwise an invalid layer object is returned.
	Example:
		// For this example to run, a Graph window must be the active window
		// in the project.
		void	run_Layers()
		{
			GraphPage	gp = Project.Pages();
			
			if( gp)
			{
				GraphLayer		gl = gp.Layers(); // Get active layer
				printf("%s has %d layer(s) with active layer %d\n", gp.GetName(), gp.Layers.Count(), gl.GetIndex() + 1); // show LabTalk index
			}
		}
	*/	
	Layer Layers(int iIndex = -1);
	
	/**
		Get a layer object by name.
	Parameters: 
		lpcszName = The name of the Layer requested to be accessed.
	Return:
		If the name is valid then the layer to which the name refers is returned otherwise an invalid layer object is returned.
	Example:
		// Assume Book1 is an Excel workbook with a Sheet1
		Page pg("Book1");
		Worksheet wks;
		wks = (Worksheet)pg.Layers("Sheet1");
	*/	
	Layer Layers(LPCSTR lpcszName);

	/**
		Execute a LabTalk script temporarily setting this page as the active page.
		This command is equivalent to LabTalk's "win -o" command.
		The original active page is restored after the script completes execution.
	Example:
		// Output the template name for all the graphs in the project.
		foreach(GraphPage gp in Project.GraphPages)
			gp.LT_execute("win -g; type \"%h uses template %a\";");
	Parameters:
		lpcszScript = LabTalk script to execute.
		wCtrl = This argument is currently ignored.
	Retun:
		TRUE for success or FALSE for failure.
	*/
	BOOL LT_execute(LPCSTR lpcszScript, int wCtrl = 0);

	/**
		Refresh the page.
	Example:
		Page pg = Project.Pages();
		pg.Refresh(TRUE);// slow refresh, redraw everything from data
	Parameters:
		bRedraw = indicate if Origin will need to destroy all the internal drawing cache and recreate the graph.
		FALSE will tell the page to simply repaint itself using existing cache.
	*/
	void Refresh(BOOL bRedraw = FALSE);
};

/** >Internal Origin Objects
		The MatrixPage class provides methods and properties common to all internal Origin
		matrix pages (windows). The Project class contains a collection of MatrixPage objects.
		An Origin C MatrixPage object is a wrapper object that is a reference to an internal
		Origin matrix page object. Origin C wrapper objects do not actually exist in Origin
		and merely refer to the internal Origin object. Consequently, multiple Origin C wrapper
		objects can refer to the same internal Origin object. The MatrixPage class is derived
		from the Page, PageBase, and OriginObject classes from which it inherits methods and
		properties.
	Example:
		// Assumes Matrix1 window exists in Origin
		MatrixPage mp;
		MatrixLayer ml;
		
		mp = Project.MatrixPages("Matrix1");     // Get Matrix1 page from MatrixPages collection
		if(mp)                                   // If mp IsValid...
		{
			ml = (MatrixLayer) mp.Layers(0);     // Get Matrix1 Layer from Layers collection
			if(ml)                               // If ml IsValid...
			{
				MatrixObject mo(ml,0);           // Get first MatrixObject in Layer
				if(mo)                           // If mo IsValid...
				{
					printf("%s is %d columns x %d rows\n", mp.GetName(), mo.GetNumCols(), mo.GetNumRows() );
				}
			}
		}
*/
class MatrixPage : public Page
{
public:
	/**
		Construct a MatrixPage object using the name of an existing page.
	Parameters:
		lpcszName = The name of an existing page.
	Example:
		MatrixPage mp;
		mp.Create("origin.otm");
		if( mp.IsValid() )
		{
			MatrixPage mp2(mp.GetName());
			if( mp2.IsValid() )
				printf("mp and mp2 are both attached to '%s'\n", mp2.GetName());
		}
	SeeAlso:
		Page::Create
	*/
	MatrixPage(LPCTSTR lpcszName);

	/**
		Construct a MatrixPage object using an existing page object.
	Parameters:
		page = An existing page object.
	Example:
		MatrixPage mp;
		mp.Create("origin.otm");
		if( mp.IsValid() )
		{
			MatrixPage mp2(mp);
			if( mp2.IsValid() )
				printf("mp and mp2 are both attached to '%s'\n", mp2.GetName());
		}
	SeeAlso:
		Page::Create
	*/
	MatrixPage(PageBase &page);
};

/** >Internal Origin Objects
		The GraphPage class provides methods and properties common to all internal Origin
		graph pages (windows). The Project class contains a collection of GraphPage objects.
		An Origin C GraphPage object is a wrapper object that is a reference to an internal
		Origin graph page object. Origin C wrapper objects do not actually exist in Origin
		and merely refer to the internal Origin object. Consequently, multiple Origin C wrapper
		objects can refer to the same internal Origin object. The GraphPage class is derived
		from the Page, PageBase, and OriginObject classes from which it inherits methods and
		properties.
	Example:
		// Assumes Graph1 window exists in Origin
		GraphPage gp;
		GraphLayer gl;
		GraphObject go;
		
		gp = Project.GraphPages("Graph1");       // Get Graph1 page from GraphPages collection
		if(gp)                                   // If gp IsValid...
		{
			gl = (GraphLayer) gp.Layers(0);      // Get Graph1 Layer from Layers collection
			if(gl)                               // If gl IsValid...
			{
				go = gl.GraphObjects(0);         // Get first GraphObject in Layer
				if(go)                           // If go IsValid...
				{
					printf("Name of first GraphObject is %s\n", go.GetName() );
				}
			}
		}
*/
class GraphPage : public Page
{
public:
	/**
		Construct a GraphPage object using the name of an existing page.
	Parameters:
		lpcszName = The name of an existing page.
	Example:
		GraphPage gp;
		gp.Create("origin.otm");
		if( gp.IsValid() )
		{
			GraphPage gp2(gp.GetName());
			if( gp2.IsValid() )
				printf("gp and gp2 are both attached to '%s'\n", gp2.GetName());
		}
	SeeAlso:
		Page::Create
	*/
	GraphPage(LPCTSTR lpcszName);

	/**
		Construct a GraphPage object using an existing page object.
	Parameters:
		page = An existing page object.
	Example:
		GraphPage gp;
		gp.Create("origin.otm");
		if( gp.IsValid() )
		{
			GraphPage gp2(gp);
			if( gp2.IsValid() )
				printf("gp and gp2 are both attached to '%s'\n", gp2.GetName());
		}
	SeeAlso:
		Page::Create
	*/
	GraphPage(PageBase &page);
	
public:
	/**
		Removes the graph object which is used to store the picture of 
		what the graph created from the template will look like.
	Return:
		TRUE for success or FALSE for failure.
	Example:
		GraphPage gp;
		gp.Create("origin.otm");
		if( gp.IsValid() )
			gp.RemoveTemplatePict();
	*/
	BOOL RemoveTemplatePict();

	/**
		Append layers from the supplied template to the page.
	Parameters: 
		lpcszName = Template name
	Return:
		TRUE for success or FALSE for failure.
	Example:
		GraphPage gp;
		gp.Create();
		if( gp.IsValid() )
		{
			printf("%s has %d layers\n", gp.GetName(), gp.Layers.Count());
			gp.AppendLayers("origin");
			printf("Now %s has %d layers\n", gp.GetName(), gp.Layers.Count());
		}
	SeeAlso:
		GraphPage::LoadTemplate
	*/
	BOOL AppendLayers(LPCSTR lpcszName);

	/**
		Load a template into the page.
	Parameters: 
		lpcszName = Template name
	Return:
		TRUE for success or FALSE for failure.
	Example:
		GraphPage gp;
		gp.Create();
		if( gp.IsValid() )
		{
			printf("%s has %d layers\n", gp.GetName(), gp.Layers.Count());
			gp.AppendLayers("origin");
			printf("Now %s has %d layers\n", gp.GetName(), gp.Layers.Count());
			gp.LoadTemplate("origin");
			printf("Now %s has %d layers\n", gp.GetName(), gp.Layers.Count());
		}
	SeeAlso:
		GraphPage::AppendLayers
	*/
	BOOL LoadTemplate(LPCSTR lpcszName);

	/**
		A Collection of all graph layers in a page.
	Example:
		GraphPage gp;
		gp.Create("origin.otp");
		if( gp.IsValid() )
		{
			int iLayers;
			foreach(GraphLayer gl in gp.Layers)
				iLayers++;
			printf("%s has %d layer(s)\n", gp.GetName(), iLayers);
		}
	*/
	Collection<GraphLayer> Layers;

	/**
		Get a GraphLayer object by index.
	Parameters: 
		iIndex = The index of the layer requested to be accessed.
		If -1 then the active layer is returned.
	Return:
		If iIndex is valid then the layer to which the index refers is returned otherwise an invalid layer object is returned.
	Example:
		// For this example to run, the graph window with the name "Graph1" must exist
		// in the project.
		void	run_Layers()
		{
			GraphPage	gp("Graph1");
			
			if( gp.IsValid() )
			{
				GraphLayer		gl = gp.Layers(); // Get active layer
				printf("%s has %d layer(s)\n", gp.GetName(), gp.Layers.Count());
			}
		}
	*/	
	GraphLayer Layers(int iIndex = -1);
	
	/**
		Get a GraphLayer object by name.
	Parameters: 
		lpcszName = The name of the layer requested to be accessed.
	Return:
		If the name is valid then the layer to which the name refers is returned otherwise an invalid layer object is returned.
	Example:
		// For this example to run, the graph window with the name "Graph1" must exist
		// in the project.
		void	run_Layers()
		{
			GraphPage	gp("Graph1");
			string		strLayerName = "ABC";
			GraphLayer	gl = gp.Layers(strLayerName);
			if (gl.IsValid())
				out_str("The layer exists.");
			else
				out_str("The layer does not exist.");
		}
	*/	
	GraphLayer Layers(LPCSTR lpcszName);
};


/** >Internal Origin Objects
		The LayoutPage class provides methods and properties common to all internal Origin
		layout pages (windows). The Project class contains a collection of LayoutPage objects.
		An Origin C LayoutPage object is a wrapper object that is a reference to an internal
		Origin layout page object. Origin C wrapper objects do not actually exist in Origin
		and merely refer to the internal Origin object. Consequently, multiple Origin C wrapper
		objects can refer to the same internal Origin object. The LayoutPage class is derived
		from the Page, PageBase, and OriginObject classes from which it inherits methods and
		properties.
	Example:
		// Assumes Layout1 window with at least one Graph Object exists in Origin
		LayoutPage lpg;
		Layout lo;
		GraphObject go;
		
		lpg = Project.LayoutPages("Layout1");       // Get Layout1 page from LayoutPages collection
		if(lpg)                                     // If lpg IsValid...
		{
			lo = (Layout) lpg.Layers(0);            // Get Layout1 Layer from Layers collection
			if(lo)                                  // If lo IsValid...
			{
				go = lo.GraphObjects(0);            // Get first GraphObject in Layout Layer
				if(go)                              // If go IsValid...
				{
					printf("Name of first GraphObject is %s\n", go.GetName() );
				}
			}
		}
*/
class LayoutPage : public Page
{
public:
	/**
		Construct a LayoutPage object using the name of an existing page.
	Parameters:
		lpcszName = The name of an existing page.
	Example:
		LayoutPage lp;
		lp.Create("origin.otp");
		if( lp.IsValid() )
		{
			LayoutPage lp2(lp.GetName());
			if( lp2.IsValid() )
				printf("lp and lp2 are both attached to '%s'\n", lp2.GetName());
		}
	SeeAlso:
		Page::Create
	*/
	LayoutPage(LPCTSTR lpcszName);

	/**
		Construct a LayoutPage object using an existing LayoutPage object.
	Parameters:
		page = An existing LayoutPage object.
	Example:
		LayoutPage lp;
		lp.Create("origin.otp");
		if( lp.IsValid() )
		{
			LayoutPage lp2(lp);
			if( lp2.IsValid() )
				printf("lp and lp2 are both attached to '%s'\n", lp2.GetName());
		}
	SeeAlso:
		Page::Create
	*/
	LayoutPage(PageBase &page);
};

/** >Internal Origin Objects
		The WorksheetPage class provides methods and properties common to all internal Origin
		worksheet pages (windows). The Project class contains a collection of WorksheetPage
		objects. An Origin C WorksheetPage object is a wrapper object that is a reference to
		an internal Origin worksheet page object. Origin C wrapper objects do not actually exist
		in Origin and merely refer to the internal Origin object. Consequently, multiple Origin
		C wrapper objects can refer to the same internal Origin object. The WorksheetPage class
		is derived from the Page, PageBase, and OriginObject classes from which it inherits
		methods and properties.
	Example:
		// Assumes Data1 window with at least one Column exists in Origin
		WorksheetPage wp;
		Worksheet wks;
		Column col;
		
		wp = Project.WorksheetPages("Data1");           // Get Data1 page from WorksheetPages collection
		if(wp)                                          // If wp IsValid...
		{
			wks = (Worksheet) wp.Layers(0);             // Get Worksheet from Layers collection
			if(wks)                                     // If wks IsValid...
			{
				col = wks.Columns(0);                   // Get first Column in Worksheet
				if(col)                                 // If col IsValid...
				{
					Dataset ds(col);				    // Create Dataset object and attach to column
					if(ds)                              // If ds IsValid...
					{
						ds.SetSize(10);                 // Set size of data set
						for(int ii = 0; ii < 10; ii++ )
							ds[ii]=2*ii+5;              // Fill data set/column with values
					}
				}
			}
		}
*/
class WorksheetPage : public Page
{
public:
	/**
		Construct a WorksheetPage object using the name of an existing page.
	Parameters:
		lpcszName = The name of an existing page.
	Example:
		WorksheetPage wp;
		wp.Create("origin.otw");
		if( wp.IsValid() )
		{
			WorksheetPage wp2(wp.GetName());
			if( wp2.IsValid() )
				printf("wp and wp2 are both attached to '%s'\n", wp2.GetName());
		}
	SeeAlso:
		Page::Create
	*/
	WorksheetPage(LPCTSTR lpcszName);

	/**
		Construct a WorksheetPage object using the name of an existing page.
	Parameters:
		page = A reference to an existing page.
	Example:
		WorksheetPage wp;
		wp.Create("origin.otw");
		if( wp.IsValid() )
		{
			WorksheetPage wp2(wp);
			if( wp2.IsValid() )
				printf("wp and wp2 are both attached to '%s'\n", wp2.GetName());
		}
	SeeAlso:
		Page::Create
	*/
	WorksheetPage(PageBase &page);
	
	/**
		It opens an .XLS file lpcszWorkbookFile as either an Excel workbook or as Origin worksheet.
   Parameters:
		lpcszWorkbookFile=pathname of the .XLS file.
		lpcszSheetName=the name of the Excel sheet inside the workbook to open.
      				If NULL, it opens the workbook as Excel.
      				If not NULL, it opens only the sheet with this name as Origin worksheet
      				(other sheets are ignored).
	Return:
    	TRUE if success, otherwise FALSE.
	Example:
		void	test_OpenExcel(string strWorkbookPathname)
		{
			WorksheetPage		wpg;
			
			BOOL				bOK = wpg.OpenExcel(strWorkbookPathname, "Sheet2");
			
			out_int("OK = ", bOK);
			if (!wpg)
			{
				out_str("Invalid page!");
			}
			
			out_str(wpg.GetName());
		}
	*/
	BOOL    OpenExcel(LPCSTR lpcszWorkbookFile, LPCSTR lpcszSheetName = NULL);
	
	
	/**
		It opens an .XLS file lpcszWorkbookFile as an Origin worksheet.
	Parameters:
		lpcszWorkbookFile=pathname of the .XLS file.
		nSheetIndex=the index of the Excel sheet to open as Origin worksheet (other sheets are ignored).
		If negative, it opens only the active ("top") sheet.
	Return:
    	TRUE if success, otherwise FALSE.
	Example:
		void	test_OpenExcel_2(string strWorkbookPathname)
		{
			WorksheetPage		wpg;
			
			BOOL				bOK = wpg.OpenExcel(strWorkbookPathname, -1);
			
			out_int("OK = ", bOK);
			if (!wpg)
			{
				out_str("Invalid page!");
			}
			
			out_str(wpg.GetName());
		}
	*/
	BOOL    OpenExcel(LPCSTR lpcszWorkbookFile, int nSheetIndex);


	/**
		It connects to the Excel sheet with the name lpcszSheetName and makes sure that there are at least
		nColsNeeded columns in the underlying layer object.
	Parameters:
		lpcszSheetName=the name of the sheet to connect.
		nColsNeeded=[optional] specifies the minumum number of the underlying Origin columns to create
					(the method will automatically create at least as many columns as there are data in
					Excel sheet)
	Return:
		TRUE if success, otherwise FALSE.
	Example:
		// For this example, to run, a worbook with the name "Book1" must already exist
		// in the current project and have at least three sheets.
		void	ggg()
		{
			WorksheetPage		wpg("Book1");
			if (!wpg)
			{
				out_str("Invalid page!");
				return;
			}
			
			string				strSheet = "Sheet2";
			BOOL				bOK = wpg.ExcelSheet(strSheet, 25);
			out_int("OK = ", bOK);
			
			Worksheet			wks = wpg.Layers(strSheet);
			if (!wks)
			{
				out_str("Invalid sheet!");
				return;
			}
			
			Column				col = wks.Columns(23);
			if (!col)
			{
				out_str("Invalid column!");
				return;
			}
			
			Dataset				ds(col);
			if (!ds)
			{
				out_str("Invalid datasset!");
				return;
			}
			
			ds.SetSize(100);
			ds[3] = 765.234;
		}
	*/
	BOOL	ExcelSheet(LPCSTR lpcszSheetName, int nColsNeeded = 0);
	
	/**
		It connects to the Excel sheet with index nSheetNumber and makes sure that there are at least
		nColsNeeded columns in the underlying layer object.
	Parameters:
		nSheetNumber=the (0-offset) index of the sheet to connect.
		nColsNeeded=[optional] specifies the minumum number of the underlying Origin columns to create
					(the method will automatically create at least as many columns as there are data in
					Excel sheet)
	Return:
		the name of the sheet if success, otherwise empty string.
	Example:
		// For this example, to run, a worbook with the name "Book1" must already exist
		// in the current project and have the sheet with the name "Sheet2".
		void	ggg()
		{
			WorksheetPage		wpg("Book1");
			if (!wpg)
			{
				out_str("Invalid page!");
				return;
			}
			
			string				strSheet;
			// connect to the third sheet and ensure that there are at least 25 columns.
			strSheet = wpg.ExcelSheet(2, 25);
			out_str(strSheet);
			
			Worksheet			wks = wpg.Layers(strSheet);
			if (!wks)
			{
				out_str("Invalid sheet!");
				return;
			}
			
			// The 24th column
			Column				col = wks.Columns(23);
			if (!col)
			{
				out_str("Invalid column!");
				return;
			}
			
			// The dataset attached to this column:
			Dataset				ds(col);
			if (!ds)
			{
				out_str("Invalid datasset!");
				return;
			}
			
			// Set its size:
			ds.SetSize(100);
			
			// Set one value:
			ds[3] = 765.234;
			
		}
	*/
	string	ExcelSheet(int nSheetNumber, int nColsNeeded = 0);
};

/** >Internal Origin Objects
		The Note class provides methods and properties common to all internal Origin Note
		pages (windows). The Project class contains a collection of Note objects. An Origin
		C Note object is a wrapper object that is a reference to an internal Origin Note
		page. Origin C wrapper objects do not actually exist in Origin and merely refer to
		the internal Origin object. Consequently, multiple Origin C wrapper objects can refer
		to the same internal Origin object. The Note class is derived from the PageBase and
		OriginObject classes from which it inherits methods and properties.
		
	Example:
		// Assumes Notes window exists in Origin
		Note no;
		no = Project.Notes("Notes");           // Get Notes page from Notes collection
		if(no)                                 // If no IsValid...
		{
			no.Rename("MyNotes");              // Rename Notes window to MyNotes
		}
*/
class Note : public PageBase
{
public:
	/**
		Construct a Note object using the name of an existing page.
	Parameters:
		lpcszName = The name of an existing page.
	Example:
		LT_execute("win -n n MyNotes;");
		Note note("MyNotes");
		if( note.IsValid() )
		{
			Note note2(note.GetName());
			if( note2.IsValid() )
				printf("note and note2 are both attached to '%s'\n", note2.GetName());
		}
	SeeAlso:
		Page::Create
	*/
	Note(LPCTSTR lpcszName);

	/**
		Construct a Note object using an existing page object.
	Parameters:
		page = An existing page object.
	Example:
		LT_execute("win -n n MyNotes;");
		Note note("MyNotes");
		if( note.IsValid() )
		{
			Note note2(note);
			if( note2.IsValid() )
				printf("note and note2 are both attached to '%s'\n", note2.GetName());
		}
	SeeAlso:
		Page::Create
	*/
	Note(PageBase &page);
	
	
	/**
		Creates either a new empty Notes (page) window, or a new Notes
		window with the contents of a sepcified text file.
	Parameters:
		nOption = (optional) creating options. Possible values are:
					CREATE_VISIBLE - creates a visible Notes page (window)
					CREATE_HIDDEN - creates a hidden Notes page
		lpcszFilePathName = (optional) the full pathname of the text file
							to load into the new page. If NULL, an empty
							page is created.
	Returns:
		TRUE for success, otherwise FALSE.
	Example:
		void	run_this()
		{
			Note		pg;
			// Create a new empty Notes window:
			if (pg.Create())
			{
				printf("The name of the new Notes page is %s\n", pg.GetName());
			}
			else
				out_str("Creating a Notes page failed!");
		}
	*/
	BOOL	Create(int nOption = CREATE_VISIBLE, LPCSTR lpcszFilePathName = NULL);
	
	/**
		This data member is a string containing the notes text.
	Example:
		LT_execute("win -n n MyNotes;");
		Note note("MyNotes");
		note.Text = "These are my notes.";
		printf("MyNotes contains: %s\n", note.Text);
	*/
	string Text;
};

#endif //_PAGE_H
