/*------------------------------------------------------------------------------*
 * File Name: ExternApps.h														*
 * Creation: DVT 12/24/2001														*
 * Purpose: Origin C header file for communication with third party applications*
 * Copyright (c) OriginLab Corp.	2002 - 2007									*
 * All Rights Reserved															*
 * 																				*
 *	DVT 8/14/03 MATLAB_SINGLE_INSTANCE_FOR_MENU_IMPORT							*
 *------------------------------------------------------------------------------*/

#ifndef _EXTERNAPPS_H
#define _EXTERNAPPS_H

#include <common.h>

#ifndef _STRING_H
#include <string.h>		// Most likely will need strings
#endif // _STRING_H

#include <OC_types.h>	// Structures used in Origin internal functions

#ifndef _WKSHEET_H
#include <Wksheet.h>
#endif // _WKSHEET_H

/** >Application Communication
		Matlab class provides communication between Origin and Matlab.
	Examples:
		#include <origin.h>
		#include <externApps.h> // this file is not included in origin.h
		////////////////////////////////////////////////////////////////////////////////////
		
		void mltest()
		{
			Matlab matObj;
			if(!matObj)
			{
				out_str("No Matlab found");
				return;
			}
			
			//defines 3x5 matrix named ma
			string strRet = matObj.Execute("ma=[1 2 3 4 5; 4 5 6 7 8;10.3 4.5 -4.7 -23.2 -6.7]");	
			out_str(strRet);// show str from Matlab
			
			// put matrix into Origin matrix
			MatrixLayer matLayer;
			matLayer.Create();
			Matrix mao(matLayer);
			
			//Transfer Matlab's matrix (ma) to Origin's mao matrix.
			BOOL bRet = matObj.GetMatrix("ma", &mao);
		}
*/
class Matlab
{
public:
	
	/**
			Constructor for Matlab class. It will connect to Matlab if called with default argument(TRUE).
		Example:
			//Assumes Matlab version R12 or greater is installed on client's computer (DCOM not tested)
			Matlab matObj; 					// Use constructor to create Origin C Matlab object
			if (!matObj) return FALSE;		//return FALSE if communication is not established
			Matlab matObj0(FALSE);			//Use to create Origin C Matlab object without connecting to Matlab.
			matObj0.Attach();				//To actually connect to Matlab, Attach() have to be called.
		SeeAlso:
			Matlab::Attach
	*/
	Matlab( BOOL bAttach = TRUE, BOOL bSingle = FALSE, BOOL bKeepMatlabRunning = FALSE ); // Constructor for Matlab class.
	//Matlab(); // Default constructor for Matlab class.
	//Matlab(BOOL bAttach =TRUE); // Constructor for Matlab class.

	/**
			Establishes connection to Matlab.
		Example:
			// Assumes	Matlab object successfully created.
			if ( !matObj.Attach() ) return;	//return if connection can not be made
			matObj.Attach(FALSE);			//Establish new connection destroying any previous connection first
		Parameters:
			bUseRunningInstance=if TRUE (default) attempt is made to reuse already established connection. 
								If that is not possible, or if the parameter is FALSE, new connection is made.
		Return:
			Returns TRUE if connection is being made, FALSE otherwise. 
		SeeAlso: 
			Matlab::Detach
	*/
	BOOL	Attach( BOOL bUseRunningInstance=TRUE, BOOL bSingle = FALSE, BOOL bKeepMatlabRunning = FALSE );
	//BOOL	Attach(BOOL bUseRunningInstance=TRUE);

	/**
			Destroys the connection with Matlab
		Example:
			// Assumes	Matlab object successfully created, and communication established
			matObj.Detach();	//destroys the connection
			
			matObj.Detach();
			matObj.Attach()		//the same effect as matObj.Attach(FALSE);
			
		Return:
			Returns TRUE if connection is being destroyed, FALSE if the operation failed. 
		SeeAlso: 
			Matlab::Attach
	*/
	BOOL	Detach();
	
	/**
			Executes command in Matlab workspace
		Example:
			// Assumes	Matlab object successfully created, i.e. communication established
			string strRet = matObj.Execute("ma=[1 2 3; 4 5 6]");	//defines 2x3 matrix named ma
			string strRet = matObj.Execute("filename");				//runs Matlab's script filename.m if in Matlab's search path
		Parameters:
			lpcszCommand=Matlab command to execute. 
			
		Return:
			Returns the string containing Matlab command window output. 
			//Comment: ";" at the end of the command suppresses output in Matlab.
	*/
	string	Execute(LPCSTR lpcszCommand);

	/**
			Executes command in Matlab workspace
		Example:
			// Assumes	Matlab object successfully created, i.e. communication established
			string strRet;
			BOOL bRet = matObj.Execute("ma=[1 2 3; 4 5 6]", strRet);	//defines 2x3 matrix named ma
			out_str(strRet);
		Parameters:
			lpcszCommand=Matlab command to execute. 
			strRet=string to receive Matlab output,
					or OriginC output "??? FAILED TO EXECUTE MAtlab" if execution failed
		Return:
			Returns TRUE on successful exit and FALSE on error.
	*/
	BOOL	Execute(LPCSTR lpcszCommand, string &strReturn);

	/**
			Transfer Origin Matrix to Matlab matrix. OriginC matrix not currently supported. Complex matrices not currently supported
		Example:
			// Assumes	Matlab object successfully created, i.e. communication established
			//			Matrix1 with sample data exists in Origin
			Matrix mReal("Matrix1");
			BOOL bRet = matObj.PutMatrix("mat", &mReal);				//Transfer mReal data to mat Matlab's matrix. mat will be created if it didn't exist before
			if (bRet) 
				string strRet = matObj.Execute("mat");				//strRet will contain display of mat
		Parameters:
			lpcszMatrixName=The name of the Matlab matrix to receive the data
			pmmReal=Origin matrix is the source of the data
			pmmImag=To create complex matrix in Matlab, define Origin matrices for real and imaginary parts
			lpcszMatLabWorkspace=default value "base" defines lpcszMatrixName in default Matlab workspace. The use of "global" for global Matlab workspace is not currently supported
		Return:
			Returns TRUE on successful exit and FALSE on error.
		SeeAlso: 
			Matlab::GetMatrix
	*/
	BOOL	PutMatrix(LPCSTR lpcszMatrixName, Matrix *pmmReal, Matrix *pmmImag = NULL, LPCSTR lpcszMatLabWorkspace = NULL);

	/**
			Transfer Matlab matrix to Origin Matrix. OriginC matrix not currently supported.
		Example:
			// Assumes	Matlab object successfully created, i.e. communication established
			//			Matrix1 exists in Origin
			Matrix mReal("Matrix1");
			BOOL bRet = matObj.GetMatrix("mat", &mReal);				//Transfer mat Matlab's matrix to mReal.
		Parameters:
			lpcszMatrixName=The name of the Matlab matrix source
			pmmReal=Origin matrix as destination for the data
			pmmImag=To receive complex matrix from Matlab, define Origin matrices for real and imaginary parts. Currently not implemented - 
					leave without this optional argument, or set NULL to access the fourth parameter
			lpcszMatLabWorkspace=default value "base" defines lpcszMatrixName in default Matlab workspace. The use of "global" for global Matlab workspace is not currently supported
		Return:
			Returns TRUE on successful exit and FALSE on error.
		SeeAlso: 
			Matlab::PutMatrix
	*/
	BOOL	GetMatrix(LPCSTR lpcszMatrixName, Matrix *pmmReal, Matrix *pmmImag = NULL, LPCSTR lpcszMatLabWorkspace = NULL);
	
	/**
			Transfer Matlab character array to Origin string.
		Example:
			// Assumes	Matlab object successfully created, i.e. communication established
			string strMcharname;
			strMcharname = matObj.GetString( "mcharname", "base" );
		Parameters:
			lpcszStringName=The name of the Matlab character array source
			lpcszMatLabWorkspace=default value "base" defines lpcszMatrixName in default Matlab workspace. The use of "global" for global Matlab workspace is not currently supported
		Return:
			Returns Matlab's character array as a string.
		SeeAlso: 
			Matlab::PutString
	*/
	string	GetString( LPCSTR lpcszStringName, LPCSTR lpcszMatLabWorkspace = NULL );
	
	/**
			Transfer Origin string to Matlab character array.
		Example:
			// Assumes	Matlab object successfully created, i.e. communication established
			matObj.PutString( "mcharname", "OC::Matlab Origin Text", "base" );
		Parameters:
			lpcszStringName=The name of the Matlab character array source
			lpcszMatLabWorkspace=default value "base" defines lpcszMatrixName in default Matlab workspace. The use of "global" for global Matlab workspace is not currently supported
		Return:
			Returns TRUE on successful exit and FALSE on error.
		SeeAlso: 
			Matlab::GetString
	*/
	BOOL	PutString( LPCSTR lpcszStringName, LPCSTR lpcszString, LPCSTR lpcszMatLabWorkspace = NULL );

	/**
			Minimizes Matlab command window.
		Example:
			// Assumes	Matlab object successfully created, i.e. communication established
			matObj.MinimizeCommandWindow();
		SeeAlso:
			Matlab::MaximizeCommandWindow
	*/
	void	MinimizeCommandWindow();

	/**#
			Maximizes Matlab command window.
			
			Note: Not supported in Matlab 6.5 and earlier.
		Example:
			// Assumes	Matlab object successfully created, i.e. communication established
			matObj.MaximizeCommandWindow();
		SeeAlso:
			Matlab::MinimizeCommandWindow
	*/
	//void	MaximizeCommandWindow();

	/**#
			Closes Matlab.
			
			Note: Not supported in Matlab 6.5 and earlier
		Example:
			// Assumes	Matlab object successfully created, i.e. communication established
			matObj.Quit();
	*/
	//void	Quit();

	/**
			Shows the Visible/Hidden state of Matlab window.
		Example:
			// Assumes	Matlab object successfully created, i.e. communication established
			matObj.Visible = FALSE;
	*/
	int		Visible;
	
	/**
			Controls if Origin should keep Matlab session alive after closing the connection.
			When Origin exits, Matlab session created by Origin, or Matlab session started started prior to Origin one, 
			with command line: Matlab /Automation, will close.
		Example:
			// Assumes	Matlab object successfully created, i.e. communication established
			BOOL bIsKeep = matObj.KeepMatlabRunning;
			matObj.KeepMatlabRunning = !bIsKeep;
	*/
	BOOL	KeepMatlabRunning;
};

#endif //_EXTERNAPPS_H