/*------------------------------------------------------------------------------*
 * File Name LT_PE.c: Contains Origin C source code for Labtalk PE commands		*
 * Creation: August 05, 2003, Prithvi Maddi.									*
 * Purpose: OriginC Source C file												*
 * Copyright (c) OriginLab Corp. 2000, 2001, 2002, 2003, 2004, 2005, 2006, 2007	*
 * All Rights Reserved															*
 * 																				*
 * Modification Log:															*
 *------------------------------------------------------------------------------*/
 
////////////////////////////////////////////////////////////////////////////////////
// you can include just this typical header file for most Origin built-in functions and classes
// and it takes a reasonable amount of time to compile, 
#include <Folder.h>
#include <page.h>
#include <Project.h>

////////////////////////////////////////////////////////////////////////////////////
// A sample program to illustrate how to use Project Explore from Origin C by
// creating labtalk commands that are similar to DOS commands
// the following commands are supported once this file is compiled
// pe_dir		 :	List pages and folders
// pe_mkdir		 :	Creat a new folder
// pe_rmdir		 :	Remove an existing folder
// pe_cd		 :	Change the current path(folder)
// pe_move		 :	Move a page or subfolder in current folder to another folder
// pe_search_page:	Find a page in current folder and subfolder, and printf full path of location if find.
// pe_get_path	 :	Get the path of the location a page.
// pe_rename	 :	Rename the Page or subfoler 
////////////////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////////////////
// static functions in OriginC are not visible from Labtalk
static int PageTypeStr2Int(string strType)
{
	if( strType.IsEmpty() )
		return 0; // any page
	
	strType.MakeUpper();
	switch( strType[0] )
	{
	case 'W':
		return EXIST_WKS;
	case 'P': case 'G':
		return EXIST_GRAPH;
	case 'M':
		return EXIST_MATRIX;
	case 'N':
		return EXIST_NOTES;
	case 'L':
		return EXIST_LAYOUT;
	}
	return 0;
}

///////////////////////////////////////////////////////////////////////	
// PEDir name    		List  all pages matching name
// PEDir name type		List all matched pages of given type, "W', "P", "M", "N", "L". etc

/** 
	Display pages/subfolders inside the current folder.
Example: 
	pe_dir() 		 // gives a list of all subfolders as well as pages
	pe_dir("*") 	 // gives a list of all pages
	pe_dir("*", "w") // gives a list of all pages of type w or wks
	pe_dir("", "w")  // gives a list of all pages of type w or wks
	
Parameters:
	pagename = name of the page; "*" will match all the pages

Return:

SeeAlso:
	
*/
void pe_dir(string pagename = "", string strPageType = "")
{
	string strName;
	Folder fld = Project.ActiveFolder();
		
	if( strPageType.IsEmpty() && pagename.IsEmpty() )// no win_type/page_name specified, will show subfolders
	{
		foreach(Folder sub in fld.Subfolders)
		{
			strName = sub.GetName();
			printf("<Folder> %s\n", strName);
		}
	}
	
	int nWinType = PageTypeStr2Int(strPageType);
	PageBase page;		
	foreach(page in fld.Pages)
	{
		if( 0 == nWinType || nWinType == page.GetType() )
		{
			strName = page.GetName();
			if(!pagename.IsEmpty())
			{
				if(strName.Match(pagename) )
					printf("%s\n", strName);
			}
			else printf("%s\n", strName);
		}
	}
}

/**
		Create a new directory
	Example: 
		pe_mkdir("myDir") //a new directory named myDir is created	
	Parameters:
		subfolderName = name of the new folder to be created; if the name already exists, it will add number after name.
	Return:
	SeeAlso:
	
*/ 	
void pe_mkdir(string subfolderName)
{
	Folder fld = Project.ActiveFolder();
	Folder subfld = fld.AddSubfolder(subfolderName);
	if( !subfld.IsValid() )
	{
		printf("Error!!! Can't Create new subfolder %s !\n", subfolderName);
	}
}


/**
		Remove or delete a directory
	Example: 
		pe_rmdir("myDir") //a directory named myDir is deleted	
	Parameters:
		subfolderName = name of the folder to be deleted
		bFolderPrompt = TRUE, Origin will prompt the user to verify before deleting the folder
		bPagePrompt   = TRUE, Origin will prompt the user to verify before deleting the page
	Return:
	SeeAlso:
*/
void pe_rmdir(string subfolderName, BOOL bFolderPrompt = FALSE , BOOL bPagePrompt = FALSE)
{
	Folder fld = Project.ActiveFolder();
	if( !fld.RemoveSubFolder(subfolderName) )
	{
		printf("Error!!! Can't remove subfolder %s !\n", subfolderName);
	}
}



/**
		Change the current path. You can use both absolute or relative way to change current path, like "/a/b" or "../b"
		When absolute way is used, the project name can be omitted, like "/" can return to root folder.
	Example: 
		pe_cd("../") 	//move out of current directory
		pe_cd("myDir")  //move into a directory named myDir
		pe_cd("../myDir") //move out of the current directory into a directory named myDir
	Parameters:
		strPathName = path of the directory to move into
	Return:
	SeeAlso:
*/	
void pe_cd(string strPathName) 
{
	if( !Project.ActivateFolder(strPathName) )	
	{
		printf("Error!!! Can't change path to %s !", strPathName);
	}
}

/**
		Move a page or subfolder in current folder to another folder.
	Example: 
		pe_move("myPage", "../myDir") 	//move myPage to the directory myDir
	Parameters:
		strItemName = name of the existing page
		strPathName = path of the location where the page should be moved to
	Return:
	SeeAlso:
*/	
void pe_move(string strItemName, string strPathName)
{
	Folder fld = Project.ActiveFolder();
	if( !fld.Move(strItemName, strPathName) )
	{
		printf( " Error!!! Can't move %s to %s !", strItemName, strPathName );
	}
}

/**#
//////////////////////////////////////////////////////////////////////////
// pe_search: Search through the Project's folders for a specified page.

void pe_search(string strPageName)
{
	Folder fldLoaction = Project.GetFolderWithPage(strPageName);
	if( fldLoaction.IsValid() )
	{
		string path = fldLoaction.GetPath();
		printf("%s",path);
	}
	else
	{
		printf("Error!!! Can't find page %s !", strPageName); 
	}
}
*/


/**
		Search through the Project's folders (recursively) for a specified page.
	Example: 
		int i = pe_search_page("myPage") 	//search for a page named myPage
	Parameters:
		strPageName = name of the page to be searched for.
	Return: 1 if the page is found, 0 otherwise.
	SeeAlso:
*/	

int pe_search_page(string strPageName)
{
	Folder fldLoaction = Project.GetFolderWithPage(strPageName);
	if( fldLoaction.IsValid() )
	{
		return(1); 
	}
	else
	{
		return(0); 
	}
}

/**
		Gets the path of the folder where the specified page is located.  If no page is specified, the
		current path will be returned.
	Example: 
		a$ = pe_get_path("myPage")$ 	//get the path of a page named myPage
	Parameters:
		strPageName = name of the page to be searched for.
	Return: if a page name is given and found, returns a string containing the path of the page.  If the
		page is given and not found, returns a null string and prints an error message.  If no page is
		given, returns the active folder name.
	SeeAlso:
*/	
string pe_get_path(string strPageName = "")
{
	if( strPageName.IsEmpty() )
		return Project.ActiveFolder().GetPath();
	Folder fldLocation = Project.GetFolderWithPage(strPageName);
	if( fldLocation.IsValid() )
	{
		return fldLocation.GetPath();
	}
	else
	{
		printf("Error!!! Can't find page %s ! \n", strPageName);
		return("\0");
	}
}

/**
		Rename a page or subfolder.
	Example: 
		pe_rename("myPage", "myPage1") 	//renames a page named myPage to myPage1
	Parameters:
		strItemName = name of the existing page
		strPathName = path of the location where the page should be moved to
	Return:
	SeeAlso:
*/	 
void pe_rename(string strOldName, string strNewName)
{
	string 	strName;
	Folder fld = Project.ActiveFolder();
	
	foreach(PageBase page in fld.Pages)
	{
		strName = page.GetName();
		if ( strName.Match(strOldName) )
		{
			if ( !page.Rename( strNewName ) )
			{
				printf( " Error!!! Can't Rename Page %s to %s !", strOldName, strNewName );
				return;
			}
			else
				return;
		}
	}
	foreach(Folder sub in fld.Subfolders)
	{
		strName = sub.GetName();
		if( strName.Match(strOldName) )
		{
			if( !sub.Rename(strNewName) )
			{
				printf("Error!!! Can't Rename subfolder %s to %s !", strOldName, strNewName);
				return;
			}
			else
				return;
		}
	}
	printf("Error!!! Can't find %s in current folder !", strOldName);
}