/*--------------------------------------------------------------------------*
 * File Name:	datasetObj.cpp												*
 * Purpose:		Demonstrate how to access an Origin dataset					*
 * Creation:	March 24, 2000												*
 * Copyright Microcal Software Inc. 2000									*
 *																			*
 * Modification Log:														*
 *--------------------------------------------------------------------------*/      

#include "datasetObj.h"

//---------------------------------------------------------------------------
// MOCA_ENTRY_POINT( <main object class> )
//
// The MOCA_ENTRY_POINT macro is used to define the exported function that
// Origin will use to communicate with your DLL.  The macro takes a single
// argument, the class name of your main object.
//---------------------------------------------------------------------------
MOCA_ENTRY_POINT(CDataSetDemo)


//---------------------------------------------------------------------------
// Property Map:
//
// The Property Map is for declaring the properties of your LabTalk object.
//
// A property is mapped to Get and Set function.  These functions allow
// you to do error checking or any necessary conversions.  A read-only
// property can be declared by using the _GET macro.
//
// MOCA_PROP_INT( <GetFunction>,<SetFunction>,<PropertyNameStr> )
// MOCA_PROP_REAL( <GetFunction>,<SetFunction>,<PropertyNameStr> )
// MOCA_PROP_STR( <GetFunction>,<SetFunction>,<PropertyNameStr> )
// MOCA_PROP_INT_GET( <GetFunction>,<PropertyNameStr> )
// MOCA_PROP_REAL_GET( <GetFunction>,<PropertyNameStr> )
// MOCA_PROP_STR_GET( <GetFunction>,<PropertyNameStr> )
//
// A Simple Property is mapped to a data member.  There are no Get/Set
// functions for a Simple Property.  MOCA will take care of the assignment.
// A simple read-only property can be declared by using the _GET macro.
//
// MOCA_SIMPLE_PROP_INT( <DataMember>,<PropertyNameStr> )
// MOCA_SIMPLE_PROP_REAL( <DataMember>,<PropertyNameStr> )
// MOCA_SIMPLE_PROP_STR( <DataMember>,<PropertyNameStr> )
// MOCA_SIMPLE_PROP_INT_GET( <DataMember>,<PropertyNameStr> )
// MOCA_SIMPLE_PROP_REAL_GET( <DataMember>,<PropertyNameStr> )
// MOCA_SIMPLE_PROP_STR_GET( <DataMember>,<PropertyNameStr> )
//---------------------------------------------------------------------------
MOCA_BEGIN_PROP_MAP(CDataSetDemo, CMOCAObjBase)
	MOCA_SIMPLE_PROP_STR(m_DataSetName, "DatasetName")
	MOCA_SIMPLE_PROP_STR(m_WksName, "WksName")
	MOCA_SIMPLE_PROP_STR(m_ColName, "ColumnName")
	MOCA_SIMPLE_PROP_INT(m_ColNumber, "ColumnIndex")
	MOCA_PROP_INT_GET(GetLowRange, "LowRange")
	MOCA_PROP_INT_GET(GetHighRange,  "HighRange")
MOCA_END_PROP_MAP(CDataSetDemo, CMOCAObjBase)


//---------------------------------------------------------------------------
// Method Map:
//
// INPORTANT: To have a Method Map or a SubObject Map, you must
// have a property map.  A property map can be empty.
//
// MOCA_METH_ENTRY( <MemberFunction>,<MethodNameStr> )
//
// <MemberFunction> must be declared as "BOOL foo(double &, CStringArray &);"
// The double is used for storing LabTalk's return value.
// The CStringArray contains the arguments passed from LabTalk.
//---------------------------------------------------------------------------
MOCA_BEGIN_METH_MAP(CDataSetDemo, CMOCAObjBase)
	MOCA_METH_ENTRY(MethodSetValue, "SetValue")	
	MOCA_METH_ENTRY(MethodGetValue, "GetValue")	
	MOCA_METH_ENTRY(MethodSetRange, "SetRange")	
	MOCA_METH_ENTRY(MethodLowRange, "GetLowRange")	
	MOCA_METH_ENTRY(MethodHighRange, "GetHighRange")	
	MOCA_METH_ENTRY(MethodValueString, "GetCellAsStr")	
MOCA_END_METH_MAP(CDataSetDemo, CMOCAObjBase)


//---------------------------------------------------------------------------
// SubObject Map:
//
// IMPORTANT: To have a Method Map or a SubObject Map, you must
// have a property map.  A property map can be empty.
//
// MOCA_SUBOBJ_ENTRY( <m_SubObject>,<SubObjectNameStr> )
//
//---------------------------------------------------------------------------
// This MOCA example has no subobjects.
// The following is an example of how a SubObject table is declared.
//MOCA_BEGIN_SUBOBJ_MAP(CLASSNAME, BASECLASSNAME)
//	MOCA_SUBOBJ_ENTRY(m_SubObject, "SUBOBJECT")
// 	MOCA_SUBOBJ_ENTRY(m_SubObject2, "SUBOBJECT2")
//MOCA_END_SUBOBJ_MAP(CLASSNAME, BASECLASSNAME)


//---------------------------------------------------------------------------
// CDataSetDemo::CDataSetDemo
//
//---------------------------------------------------------------------------
CDataSetDemo::CDataSetDemo()
{
	m_ColNumber = 1;
}

CDataSetDemo::~CDataSetDemo()
{
}

//---------------------------------------------------------------------------
// CDataSetDemo::GetLowRange
//
// LabTalk Property: LowRange
//---------------------------------------------------------------------------
BOOL CDataSetDemo::GetLowRange(int &iValue)
{
	MoSourceData myDataSet(m_DataSetName);

	if( myDataSet.IsValid() )
		// iRange1() returns the low index of the data set range
		iValue = myDataSet.iRange1() + 1;
	else
		iValue = 0;

	return TRUE;
}

//---------------------------------------------------------------------------
// CDataSetDemo::GetHighRange
//
// LabTalk Property: HighRange
//---------------------------------------------------------------------------
BOOL CDataSetDemo::GetHighRange(int &iValue)
{
	MoSourceData myDataSet(m_DataSetName);

	if( myDataSet.IsValid() )
		// iRange2() returns the high index of the data set range
		iValue = myDataSet.iRange2() + 1;
	else
		iValue = 0;

	return TRUE;
}


//-------------------------------------------------------------------
// CDataSetDemo::AttachToRelevant
//
// Returns a MoSourceData object attached to the most relevant
// dataset (if any).
//------------------------------------------------------------------
MoSourceData CDataSetDemo::AttachToRelevant(LPCSTR lpszDataSetName)
{
	if( lpszDataSetName && *lpszDataSetName != '\0' )
		return MoSourceData(lpszDataSetName);
	else if( *m_DataSetName != '\0' )
		return MoSourceData(m_DataSetName);
	else if( *m_WksName != '\0' && !m_ColName.IsEmpty() )
		return MoSourceData(m_WksName, m_ColName);
	else if( *m_WksName != '\0' && m_ColNumber>0 )
		return MoSourceData(m_WksName, m_ColNumber);

	return MoSourceData((LPCSTR)NULL);
}

//---------------------------------------------------------------------------
// CDataSetDemo::MethodSetValue
//
//---------------------------------------------------------------------------
BOOL CDataSetDemo::MethodSetValue(double &dReturn, CStringArray &argarray)
{
	dReturn = 0; // 0 indicates the cell's value was NOT set

	// Method requires 3 arguments: dataset name, row index, and value
	if( argarray.GetSize() != 3 )
		return FALSE; // LabTalk command error

	// Get and check the row index
	int iRowIndex = LabTalkStr2int(argarray[1]);
	if( iRowIndex <= 0 )
		return FALSE; // LabTalk command error
	iRowIndex--; // LabTalk starts with 1 but C starts with 0.

	// Create a reference to the dataset by name.
	MoSourceData myDataSet = AttachToRelevant(argarray[0]);
	if( !myDataSet.IsValid() )
	{
		MessageBox(NULL, "SetValue method received invalid dataset", "MOCA Dataset Example", MB_OK);
		return FALSE; // LabTalk command error
	}
	
	double dValue = LabTalkStr2double(argarray[2]);
	myDataSet.SetValue((MOINDEX)iRowIndex, dValue);

	dReturn = 1; // 1 indicates the cell's value was set

	return TRUE; // LabTalk command succeeded
}

//---------------------------------------------------------------------------
// CDataSetDemo::MethodGetValue
//
//---------------------------------------------------------------------------
BOOL CDataSetDemo::MethodGetValue(double &dReturn, CStringArray &argarray)
{
	dReturn = 0; // init to 0 to avoid returning junk on error

	// Method requires 2 arguments: dataset name, and row index
	if( argarray.GetSize() != 2 )
		return FALSE; // LabTalk command error

	// Get and check the row index
	int iRowIndex = LabTalkStr2int(argarray[1]);
	if( iRowIndex <= 0 )
		return FALSE; // LabTalk command error

	// Create a reference to the dataset by name.
	MoSourceData myDataSet = AttachToRelevant(argarray[0]);
	if( !myDataSet.IsValid() )
	{
		MessageBox(NULL, "GetValue method received invalid dataset", "MOCA Dataset Example", MB_OK);
		return FALSE; // LabTalk command error
	}

	// Get the value from the dataset
	dReturn = myDataSet[(MOINDEX)iRowIndex - 1];

	return TRUE; // LabTalk command succeeded
}

//---------------------------------------------------------------------------
// CDataSetDemo::MethodSetRange
//
//---------------------------------------------------------------------------
BOOL CDataSetDemo::MethodSetRange(double &dReturn, CStringArray &argarray)
{
	dReturn = 0; // 0 indicates the range was NOT set

	// Method requires 3 arguments: dataset name, low index, and high index
	if( argarray.GetSize() != 3 )
		return FALSE; // LabTalk command error

	// Get and check the low and high index values
	int iLowIndex =	LabTalkStr2int(argarray[1]); 
	int iHighIndex = LabTalkStr2int(argarray[2]); 
	if( iLowIndex <= 0 || iHighIndex <= 0 )
		return FALSE; // LabTalk command error
	
	// Create a reference to the dataset by name.
	MoSourceData myDataSet	=	AttachToRelevant(argarray[0]);
	if( !myDataSet.IsValid() )
	{
		MessageBox(NULL, "SetRange method received invalid dataset", "MOCA Dataset Example", MB_OK);
		return FALSE; // LabTalk command error
	}

	myDataSet.SetRange((MOINDEX)iLowIndex, (MOINDEX)iHighIndex);
	
	dReturn = 1; // 1 indicates the range was set

	return TRUE; // LabTalk command succeeded
}

//---------------------------------------------------------------------------
// CDataSetDemo::MethodLowRange
//
//---------------------------------------------------------------------------
BOOL CDataSetDemo::MethodLowRange(double &dReturn, CStringArray &argarray)
{
	dReturn = 0; // init to 0 to avoid returning junk on error

	// Method requires 1 argument: dataset name
	if( argarray.GetSize() != 1 )
		return FALSE; // LabTalk command error

	// Create a reference to the dataset by name.
	MoSourceData myDataSet = AttachToRelevant(argarray[0]);
	if( !myDataSet.IsValid() )
	{
		MessageBox(NULL, "GetLowRange method received invalid dataset", "MOCA Dataset Example", MB_OK);
		return FALSE; // LabTalk command error
	}

	// Get the low index of the dataset range from iRange1()
	dReturn = myDataSet.iRange1();

	return TRUE; // LabTalk command succeeded
}

//---------------------------------------------------------------------------
// CDataSetDemo::MethodHighRange
//
//---------------------------------------------------------------------------
BOOL CDataSetDemo::MethodHighRange(double &dReturn, CStringArray &argarray)
{
	dReturn = 0; // init to 0 to avoid returning junk on error

	// Method requires 1 argument: dataset name
	if( argarray.GetSize() != 1 )
		return FALSE; // LabTalk command error

	// Create a reference to the dataset by name.
	MoSourceData myDataSet = AttachToRelevant(argarray[0]);
	if( !myDataSet.IsValid() )
	{
		MessageBox(NULL, "GetHighRange method received invalid dataset", "MOCA Dataset Example", MB_OK);
		return FALSE; // LabTalk command error
	}

	// Get the high index of the dataset range from iRange2()
	dReturn = myDataSet.iRange2();

	return TRUE; // LabTalk command succeeded
}

//---------------------------------------------------------------------------
// CDataSetDemo::MethodValueString
//
//---------------------------------------------------------------------------
BOOL CDataSetDemo::MethodValueString(double &dReturn, CStringArray &argarray)
{
	dReturn = 0; // 0 indicates LabTalk str variable did NOT receive cell value

	// Method requires 3 arguments: dataset name, row index, and LabTalk str variable
	if( argarray.GetSize() != 3 )
		return FALSE; // LabTalk command error

	// Get and check the row index
	int iRowIndex = LabTalkStr2int(argarray[1]);
	if( iRowIndex <= 0 )
		return FALSE; // LabTalk command error

	// Get and check LabTalk str variable
	if( argarray[2].GetLength() != 1 )
		return FALSE; // LabTalk command error
	argarray[2].MakeUpper();
	char cStrVar = argarray[2].GetAt(0);
	if( cStrVar < 'A' || cStrVar > 'Z' )
		return FALSE; // LabTalk command error

	// Create a reference to the dataset by name.
	MoSourceData myDataSet = AttachToRelevant(argarray[0]);
	if( !myDataSet.IsValid() )
	{
		MessageBox(NULL, "GetValueStr method received invalid dataset", "MOCA Dataset Example", MB_OK);
		return FALSE; // LabTalk command error
	}

	// Get dataset cell value as a string
	char szValStr[256];
	myDataSet.GetNumericValueAsText((MOINDEX)iRowIndex - 1, szValStr);
	
	// Set LabTalk str variable
	if( SetOriginString(cStrVar, szValStr) == EGOR_YES )
		dReturn = 1; // 1 indicates LabTalk str variable did receive cell value

	return TRUE; // LabTalk command succeeded
}
