/**
 * Copyright 2020 Huawei Technologies Co., Ltd
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef YUV_SCALER_PARA_SET
#define YUV_SCALER_PARA_SET

#include <functional>
#include <string>
#include "minddata/dataset/kernels/image/soft_dvpp/utils/soft_dp_log.h"

const uint32_t maxFileCount = 10;
const uint32_t kMaxParaCount = 2048;
const uint32_t kScalerCoffNb4 = 32;
const uint32_t kScalerCoffNb6 = 48;
const uint32_t paramterInterval = 6;  // scaler Number of bits between each group of filtering coefficients
const uint32_t scalerRadio1Time = 0X10000;

struct ScalerCoefficientIndex {
  uint32_t first_index;
  uint32_t second_index;

  ScalerCoefficientIndex() : first_index(0), second_index() {}
};

struct YuvScalerPara {
  struct {
    uint32_t start;
    uint32_t end;
  } range;
  int16_t taps_4[kScalerCoffNb4];
  int16_t taps_6[kScalerCoffNb6];
};

struct YuvWPara {
  int32_t real_count;
  YuvScalerPara scale[kMaxParaCount];
};

class YuvScalerParaSet {
 public:
  static YuvWPara *GetInstance(std::string *paraset_name = nullptr, uint32_t yuv_scaler_paraset_size = 1);

  YuvScalerParaSet(const YuvScalerParaSet &) = delete;

  YuvScalerParaSet &operator=(const YuvScalerParaSet &) = delete;

  static YuvWPara *g_m_instance_;

 protected:
  YuvScalerParaSet() {}

  ~YuvScalerParaSet() {}

 private:
  class GarbageCollector {
   public:
    ~GarbageCollector() {
      if (YuvScalerParaSet::g_m_instance_) {
#ifndef DVPP_UTST
        (void)pthread_mutex_lock(&g_mutex_);
        free(YuvScalerParaSet::g_m_instance_);
        YuvScalerParaSet::g_m_instance_ = nullptr;
        (void)pthread_mutex_unlock(&g_mutex_);
#endif
      }
    }
  };

  static GarbageCollector g_collector_;
  static pthread_mutex_t g_mutex_;
};

int32_t GetScalerParamterIndex(uint32_t paramter, YuvWPara *paramterset);

constexpr YuvWPara YUV_W_PARA = {
  1024,
  {
    // Band_8_1024
    {
      // Scaler inc range
      {1023, 1024},
      // Coefficients 4 taps
      {1,   254, 1,   0,   -20, 253, 29,  -6,  -30, 231, 68,  -13, -33, 199, 112, -22,
       -29, 157, 157, -29, -22, 112, 199, -33, -13, 68,  231, -30, -6,  29,  253, -20},
      // Coefficients 6 taps
      {0,  1,   254, 1,   0,   0,  8,  -24, 248, 32,  -11, 3,  13, -38, 225, 71,  -22, 7,  15, -43, 193, 113, -33, 11,
       14, -40, 154, 154, -40, 14, 11, -33, 113, 193, -43, 15, 7,  -22, 71,  225, -38, 13, 3,  -11, 32,  248, -24, 8},
    },

    // Band_8_1025
    {
      // Scaler inc range
      {1024, 1025},
      // Coefficients 4 taps
      {1,   254, 1,   0,   -20, 252, 30,  -6,  -30, 231, 68,  -13, -33, 199, 112, -22,
       -29, 157, 157, -29, -22, 112, 199, -33, -13, 68,  231, -30, -6,  30,  252, -20},
      // Coefficients 6 taps
      {0,  1,   254, 1,   0,   0,  8,  -23, 247, 32,  -11, 3,  13, -37, 224, 71,  -22, 7,  15, -43, 193, 113, -33, 11,
       14, -40, 154, 154, -40, 14, 11, -33, 113, 193, -43, 15, 7,  -22, 71,  224, -37, 13, 3,  -11, 32,  247, -23, 8},
    },

    // Band_8_1026
    {
      // Scaler inc range
      {1025, 1026},
      // Coefficients 4 taps
      {1,   254, 1,   0,   -19, 251, 30,  -6,  -30, 231, 68,  -13, -33, 199, 112, -22,
       -29, 157, 157, -29, -22, 112, 199, -33, -13, 68,  231, -30, -6,  30,  251, -19},
      // Coefficients 6 taps
      {0,  1,   254, 1,   0,   0,  8,  -23, 246, 33,  -11, 3,  13, -37, 224, 71,  -22, 7,  15, -43, 193, 113, -33, 11,
       14, -40, 154, 154, -40, 14, 11, -33, 113, 193, -43, 15, 7,  -22, 71,  224, -37, 13, 3,  -11, 33,  246, -23, 8},
    },

    // Band_8_1027
    {
      // Scaler inc range
      {1026, 1027},
      // Coefficients 4 taps
      {1,   254, 1,   0,   -19, 251, 30,  -6,  -30, 231, 68,  -13, -33, 199, 112, -22,
       -29, 157, 157, -29, -22, 112, 199, -33, -13, 68,  231, -30, -6,  30,  251, -19},
      // Coefficients 6 taps
      {0,  1,   254, 1,   0,   0,  8,  -23, 245, 33,  -11, 4,  13, -37, 225, 71,  -23, 7,  15, -42, 192, 113, -33, 11,
       14, -40, 154, 154, -40, 14, 11, -33, 113, 192, -42, 15, 7,  -23, 71,  225, -37, 13, 4,  -11, 33,  245, -23, 8},
    },

    // Band_8_1028
    {
      // Scaler inc range
      {1027, 1028},
      // Coefficients 4 taps
      {1,   254, 1,   0,   -19, 251, 30,  -6,  -30, 231, 68,  -13, -33, 199, 112, -22,
       -29, 157, 157, -29, -22, 112, 199, -33, -13, 68,  231, -30, -6,  30,  251, -19},
      // Coefficients 6 taps
      {-1, 2,   254, 2,   -1,  0,  8,  -23, 245, 33,  -11, 4,  13, -37, 225, 71,  -23, 7,  15, -42, 192, 113, -33, 11,
       14, -40, 154, 154, -40, 14, 11, -33, 113, 192, -42, 15, 7,  -23, 71,  225, -37, 13, 4,  -11, 33,  245, -23, 8},
    },

    // Band_8_1029
    {
      // Scaler inc range
      {1028, 1029},
      // Coefficients 4 taps
      {1,   254, 1,   0,   -19, 251, 30,  -6,  -30, 231, 68,  -13, -32, 198, 112, -22,
       -29, 157, 157, -29, -22, 112, 198, -32, -13, 68,  231, -30, -6,  30,  251, -19},
      // Coefficients 6 taps
      {-1, 2,   254, 2,   -1,  0,  8,  -23, 245, 33,  -11, 4,  13, -37, 224, 72,  -23, 7,  15, -42, 192, 113, -33, 11,
       14, -40, 154, 154, -40, 14, 11, -33, 113, 192, -42, 15, 7,  -23, 72,  224, -37, 13, 4,  -11, 33,  245, -23, 8},
    },

    // Band_8_1030
    {
      // Scaler inc range
      {1029, 1030},
      // Coefficients 4 taps
      {1,   254, 1,   0,   -19, 251, 30,  -6,  -30, 232, 68,  -14, -32, 198, 112, -22,
       -29, 157, 157, -29, -22, 112, 198, -32, -14, 68,  232, -30, -6,  30,  251, -19},
      // Coefficients 6 taps
      {-1, 2,   254, 2,   -1,  0,  7,  -22, 245, 34,  -12, 4,  13, -37, 223, 72,  -23, 8,  14, -42, 193, 113, -33, 11,
       14, -40, 154, 154, -40, 14, 11, -33, 113, 193, -42, 14, 8,  -23, 72,  223, -37, 13, 4,  -12, 34,  245, -22, 7},
    },

    // Band_8_1031
    {
      // Scaler inc range
      {1030, 1031},
      // Coefficients 4 taps
      {1,   254, 1,   0,   -19, 250, 31,  -6,  -30, 231, 69,  -14, -32, 198, 112, -22,
       -29, 157, 157, -29, -22, 112, 198, -32, -14, 69,  231, -30, -6,  31,  250, -19},
      // Coefficients 6 taps
      {-1, 2,   254, 2,   -1,  0,  7,  -22, 245, 34,  -12, 4,  12, -37, 224, 72,  -23, 8,  14, -42, 193, 113, -33, 11,
       14, -40, 154, 154, -40, 14, 11, -33, 113, 193, -42, 14, 8,  -23, 72,  224, -37, 12, 4,  -12, 34,  245, -22, 7},
    },

    // Band_8_1032
    {
      // Scaler inc range
      {1031, 1032},
      // Coefficients 4 taps
      {2,   252, 2,   0,   -18, 249, 31,  -6,  -29, 230, 69,  -14, -32, 198, 112, -22,
       -29, 157, 157, -29, -22, 112, 198, -32, -14, 69,  230, -29, -6,  31,  249, -18},
      // Coefficients 6 taps
      {-1, 2,   254, 2,   -1,  0,  7,  -22, 245, 34,  -12, 4,  12, -36, 223, 72,  -23, 8,  14, -42, 193, 113, -33, 11,
       14, -40, 154, 154, -40, 14, 11, -33, 113, 193, -42, 14, 8,  -23, 72,  223, -36, 12, 4,  -12, 34,  245, -22, 7},
    },

    // Band_8_1033
    {
      // Scaler inc range
      {1032, 1033},
      // Coefficients 4 taps
      {2,   252, 2,   0,   -18, 249, 31,  -6,  -29, 230, 69,  -14, -32, 198, 112, -22,
       -29, 157, 157, -29, -22, 112, 198, -32, -14, 69,  230, -29, -6,  31,  249, -18},
      // Coefficients 6 taps
      {-1, 2,   254, 2,   -1,  0,  7,  -22, 245, 34,  -12, 4,  12, -36, 223, 72,  -23, 8,  14, -42, 193, 113, -33, 11,
       14, -40, 154, 154, -40, 14, 11, -33, 113, 193, -42, 14, 8,  -23, 72,  223, -36, 12, 4,  -12, 34,  245, -22, 7},
    },

    // Band_8_1034
    {
      // Scaler inc range
      {1033, 1034},
      // Coefficients 4 taps
      {2,   252, 2,   0,   -18, 249, 31,  -6,  -29, 230, 69,  -14, -32, 198, 112, -22,
       -29, 157, 157, -29, -22, 112, 198, -32, -14, 69,  230, -29, -6,  31,  249, -18},
      // Coefficients 6 taps
      {-2, 3,   254, 3,   -2,  0,  7,  -22, 245, 34,  -12, 4,  12, -36, 223, 72,  -23, 8,  14, -42, 193, 113, -33, 11,
       14, -40, 154, 154, -40, 14, 11, -33, 113, 193, -42, 14, 8,  -23, 72,  223, -36, 12, 4,  -12, 34,  245, -22, 7},
    },

    // Band_8_1035
    {
      // Scaler inc range
      {1034, 1035},
      // Coefficients 4 taps
      {2,   252, 2,   0,   -18, 249, 31,  -6,  -29, 230, 69,  -14, -32, 198, 112, -22,
       -29, 157, 157, -29, -22, 112, 198, -32, -14, 69,  230, -29, -6,  31,  249, -18},
      // Coefficients 6 taps
      {-2, 3,   254, 3,   -2,  0,  7,  -21, 243, 35,  -12, 4,  12, -36, 223, 72,  -23, 8,  14, -42, 193, 114, -34, 11,
       14, -40, 154, 154, -40, 14, 11, -34, 114, 193, -42, 14, 8,  -23, 72,  223, -36, 12, 4,  -12, 35,  243, -21, 7},
    },

    // Band_8_1036
    {
      // Scaler inc range
      {1035, 1036},
      // Coefficients 4 taps
      {2,   252, 2,   0,   -18, 248, 32,  -6,  -29, 230, 69,  -14, -32, 197, 113, -22,
       -29, 157, 157, -29, -22, 113, 197, -32, -14, 69,  230, -29, -6,  32,  248, -18},
      // Coefficients 6 taps
      {-2, 3,   254, 3,   -2,  0,  7,  -21, 243, 35,  -12, 4,  12, -36, 223, 73,  -24, 8,  14, -42, 193, 114, -34, 11,
       14, -40, 154, 154, -40, 14, 11, -34, 114, 193, -42, 14, 8,  -24, 73,  223, -36, 12, 4,  -12, 35,  243, -21, 7},
    },

    // Band_8_1037
    {
      // Scaler inc range
      {1036, 1037},
      // Coefficients 4 taps
      {2,   252, 2,   0,   -18, 248, 32,  -6,  -29, 229, 70,  -14, -32, 197, 113, -22,
       -29, 157, 157, -29, -22, 113, 197, -32, -14, 70,  229, -29, -6,  32,  248, -18},
      // Coefficients 6 taps
      {-2, 3,   254, 3,   -2,  0,  6,  -21, 245, 35,  -13, 4,  12, -36, 223, 73,  -24, 8,  14, -42, 193, 114, -34, 11,
       14, -40, 154, 154, -40, 14, 11, -34, 114, 193, -42, 14, 8,  -24, 73,  223, -36, 12, 4,  -13, 35,  245, -21, 6},
    },

    // Band_8_1038
    {
      // Scaler inc range
      {1037, 1038},
      // Coefficients 4 taps
      {3,   250, 3,   0,   -17, 248, 32,  -7,  -29, 229, 70,  -14, -32, 197, 113, -22,
       -29, 157, 157, -29, -22, 113, 197, -32, -14, 70,  229, -29, -7,  32,  248, -17},
      // Coefficients 6 taps
      {-2, 3,   254, 3,   -2,  0,  6,  -21, 245, 35,  -13, 4,  12, -36, 223, 73,  -24, 8,  14, -42, 193, 114, -34, 11,
       14, -40, 154, 154, -40, 14, 11, -34, 114, 193, -42, 14, 8,  -24, 73,  223, -36, 12, 4,  -13, 35,  245, -21, 6},
    },

    // Band_8_1039
    {
      // Scaler inc range
      {1038, 1039},
      // Coefficients 4 taps
      {3,   250, 3,   0,   -17, 248, 32,  -7,  -29, 229, 70,  -14, -32, 197, 113, -22,
       -29, 157, 157, -29, -22, 113, 197, -32, -14, 70,  229, -29, -7,  32,  248, -17},
      // Coefficients 6 taps
      {-2, 3,   254, 3,   -2,  0,  6,  -21, 245, 35,  -13, 4,  12, -35, 222, 73,  -24, 8,  14, -41, 192, 114, -34, 11,
       14, -40, 154, 154, -40, 14, 11, -34, 114, 192, -41, 14, 8,  -24, 73,  222, -35, 12, 4,  -13, 35,  245, -21, 6},
    },

    // Band_8_1040
    {
      // Scaler inc range
      {1039, 1040},
      // Coefficients 4 taps
      {3,   250, 3,   0,   -17, 248, 32,  -7,  -28, 228, 70,  -14, -32, 197, 113, -22,
       -29, 157, 157, -29, -22, 113, 197, -32, -14, 70,  228, -28, -7,  32,  248, -17},
      // Coefficients 6 taps
      {-2, 4,   252, 4,   -2,  0,  6,  -20, 243, 36,  -13, 4,  12, -35, 222, 73,  -24, 8,  14, -41, 192, 114, -34, 11,
       14, -40, 154, 154, -40, 14, 11, -34, 114, 192, -41, 14, 8,  -24, 73,  222, -35, 12, 4,  -13, 36,  243, -20, 6},
    },

    // Band_8_1041
    {
      // Scaler inc range
      {1040, 1041},
      // Coefficients 4 taps
      {3,   250, 3,   0,   -17, 248, 32,  -7,  -28, 228, 70,  -14, -32, 197, 113, -22,
       -29, 157, 157, -29, -22, 113, 197, -32, -14, 70,  228, -28, -7,  32,  248, -17},
      // Coefficients 6 taps
      {-3, 4,   254, 4,   -3,  0,  6,  -20, 243, 36,  -13, 4,  11, -35, 223, 73,  -24, 8,  14, -41, 192, 114, -34, 11,
       14, -40, 154, 154, -40, 14, 11, -34, 114, 192, -41, 14, 8,  -24, 73,  223, -35, 11, 4,  -13, 36,  243, -20, 6},
    },

    // Band_8_1042
    {
      // Scaler inc range
      {1041, 1042},
      // Coefficients 4 taps
      {3,   250, 3,   0,   -17, 247, 33,  -7,  -28, 228, 70,  -14, -32, 197, 113, -22,
       -29, 157, 157, -29, -22, 113, 197, -32, -14, 70,  228, -28, -7,  33,  247, -17},
      // Coefficients 6 taps
      {-3, 4,   254, 4,   -3,  0,  6,  -20, 242, 36,  -13, 5,  11, -35, 222, 74,  -24, 8,  14, -41, 192, 114, -34, 11,
       14, -40, 154, 154, -40, 14, 11, -34, 114, 192, -41, 14, 8,  -24, 74,  222, -35, 11, 5,  -13, 36,  242, -20, 6},
    },

    // Band_8_1043
    {
      // Scaler inc range
      {1042, 1043},
      // Coefficients 4 taps
      {4,   248, 4,   0,   -17, 247, 33,  -7,  -28, 228, 70,  -14, -32, 197, 113, -22,
       -29, 157, 157, -29, -22, 113, 197, -32, -14, 70,  228, -28, -7,  33,  247, -17},
      // Coefficients 6 taps
      {-3, 4,   254, 4,   -3,  0,  6,  -20, 242, 36,  -13, 5,  11, -35, 222, 74,  -24, 8,  14, -41, 191, 114, -34, 12,
       14, -40, 154, 154, -40, 14, 12, -34, 114, 191, -41, 14, 8,  -24, 74,  222, -35, 11, 5,  -13, 36,  242, -20, 6},
    },

    // Band_8_1044
    {
      // Scaler inc range
      {1043, 1044},
      // Coefficients 4 taps
      {4,   248, 4,   0,   -16, 246, 33,  -7,  -28, 228, 70,  -14, -31, 196, 113, -22,
       -29, 157, 157, -29, -22, 113, 196, -31, -14, 70,  228, -28, -7,  33,  246, -16},
      // Coefficients 6 taps
      {-3, 4,   254, 4,   -3,  0,  6,  -20, 243, 36,  -14, 5,  11, -35, 222, 74,  -24, 8,  14, -41, 191, 114, -34, 12,
       14, -40, 154, 154, -40, 14, 12, -34, 114, 191, -41, 14, 8,  -24, 74,  222, -35, 11, 5,  -14, 36,  243, -20, 6},
    },

    // Band_8_1045
    {
      // Scaler inc range
      {1044, 1045},
      // Coefficients 4 taps
      {4,   248, 4,   0,   -16, 246, 33,  -7,  -28, 227, 71,  -14, -31, 196, 113, -22,
       -29, 157, 157, -29, -22, 113, 196, -31, -14, 71,  227, -28, -7,  33,  246, -16},
      // Coefficients 6 taps
      {-3, 5,   252, 5,   -3,  0,  5,  -19, 242, 37,  -14, 5,  11, -34, 222, 74,  -25, 8,  14, -41, 191, 114, -34, 12,
       14, -40, 154, 154, -40, 14, 12, -34, 114, 191, -41, 14, 8,  -25, 74,  222, -34, 11, 5,  -14, 37,  242, -19, 5},
    },

    // Band_8_1046
    {
      // Scaler inc range
      {1045, 1046},
      // Coefficients 4 taps
      {4,   248, 4,   0,   -16, 246, 33,  -7,  -28, 228, 71,  -15, -31, 196, 113, -22,
       -29, 157, 157, -29, -22, 113, 196, -31, -15, 71,  228, -28, -7,  33,  246, -16},
      // Coefficients 6 taps
      {-3, 5,   252, 5,   -3,  0,  5,  -19, 242, 37,  -14, 5,  11, -34, 222, 74,  -25, 8,  14, -41, 190, 115, -34, 12,
       14, -40, 154, 154, -40, 14, 12, -34, 115, 190, -41, 14, 8,  -25, 74,  222, -34, 11, 5,  -14, 37,  242, -19, 5},
    },

    // Band_8_1048
    {
      // Scaler inc range
      {1046, 1048},
      // Coefficients 4 taps
      {4,   248, 4,   0,   -16, 245, 34,  -7,  -28, 228, 71,  -15, -31, 196, 113, -22,
       -29, 157, 157, -29, -22, 113, 196, -31, -15, 71,  228, -28, -7,  34,  245, -16},
      // Coefficients 6 taps
      {-3, 5,   252, 5,   -3,  0,  5,  -19, 242, 37,  -14, 5,  11, -34, 222, 74,  -25, 8,  14, -41, 190, 115, -34, 12,
       14, -40, 154, 154, -40, 14, 12, -34, 115, 190, -41, 14, 8,  -25, 74,  222, -34, 11, 5,  -14, 37,  242, -19, 5},
    },

    // Band_8_1049
    {
      // Scaler inc range
      {1048, 1049},
      // Coefficients 4 taps
      {5,   246, 5,   0,   -16, 245, 34,  -7,  -27, 227, 71,  -15, -31, 197, 113, -23,
       -29, 157, 157, -29, -23, 113, 197, -31, -15, 71,  227, -27, -7,  34,  245, -16},
      // Coefficients 6 taps
      {-4, 5,   254, 5,   -4,  0,  5,  -19, 242, 37,  -14, 5,  11, -34, 221, 75,  -25, 8,  13, -41, 191, 115, -34, 12,
       14, -40, 154, 154, -40, 14, 12, -34, 115, 191, -41, 13, 8,  -25, 75,  221, -34, 11, 5,  -14, 37,  242, -19, 5},
    },

    // Band_8_1050
    {
      // Scaler inc range
      {1049, 1050},
      // Coefficients 4 taps
      {5,   246, 5,   0,   -16, 245, 34,  -7,  -27, 227, 71,  -15, -31, 196, 114, -23,
       -29, 157, 157, -29, -23, 114, 196, -31, -15, 71,  227, -27, -7,  34,  245, -16},
      // Coefficients 6 taps
      {-4, 6,   252, 6,   -4,  0,  5,  -19, 241, 38,  -14, 5,  11, -34, 220, 75,  -25, 9,  13, -41, 191, 115, -34, 12,
       14, -40, 154, 154, -40, 14, 12, -34, 115, 191, -41, 13, 9,  -25, 75,  220, -34, 11, 5,  -14, 38,  241, -19, 5},
    },

    // Band_8_1051
    {
      // Scaler inc range
      {1050, 1051},
      // Coefficients 4 taps
      {5,   246, 5,   0,   -15, 244, 34,  -7,  -27, 227, 71,  -15, -31, 196, 114, -23,
       -29, 157, 157, -29, -23, 114, 196, -31, -15, 71,  227, -27, -7,  34,  244, -15},
      // Coefficients 6 taps
      {-4, 6,   252, 6,   -4,  0,  5,  -18, 240, 38,  -14, 5,  11, -34, 220, 75,  -25, 9,  13, -41, 191, 115, -34, 12,
       14, -40, 154, 154, -40, 14, 12, -34, 115, 191, -41, 13, 9,  -25, 75,  220, -34, 11, 5,  -14, 38,  240, -18, 5},
    },

    // Band_8_1052
    {
      // Scaler inc range
      {1051, 1052},
      // Coefficients 4 taps
      {5,   246, 5,   0,   -15, 244, 34,  -7,  -27, 227, 71,  -15, -31, 196, 114, -23,
       -29, 157, 157, -29, -23, 114, 196, -31, -15, 71,  227, -27, -7,  34,  244, -15},
      // Coefficients 6 taps
      {-4, 6,   252, 6,   -4,  0,  5,  -18, 240, 38,  -14, 5,  10, -34, 221, 75,  -25, 9,  13, -40, 190, 115, -34, 12,
       13, -40, 155, 155, -40, 13, 12, -34, 115, 190, -40, 13, 9,  -25, 75,  221, -34, 10, 5,  -14, 38,  240, -18, 5},
    },

    // Band_8_1053
    {
      // Scaler inc range
      {1052, 1053},
      // Coefficients 4 taps
      {5,   246, 5,   0,   -15, 245, 34,  -8,  -27, 226, 72,  -15, -31, 196, 114, -23,
       -29, 157, 157, -29, -23, 114, 196, -31, -15, 72,  226, -27, -8,  34,  245, -15},
      // Coefficients 6 taps
      {-4, 6,   252, 6,   -4,  0,  4,  -18, 242, 38,  -15, 5,  10, -33, 220, 75,  -25, 9,  13, -40, 190, 115, -34, 12,
       13, -40, 155, 155, -40, 13, 12, -34, 115, 190, -40, 13, 9,  -25, 75,  220, -33, 10, 5,  -15, 38,  242, -18, 4},
    },

    // Band_8_1054
    {
      // Scaler inc range
      {1053, 1054},
      // Coefficients 4 taps
      {5,   246, 5,   0,   -15, 244, 35,  -8,  -27, 226, 72,  -15, -31, 196, 114, -23,
       -29, 157, 157, -29, -23, 114, 196, -31, -15, 72,  226, -27, -8,  35,  244, -15},
      // Coefficients 6 taps
      {-4, 7,   250, 7,   -4,  0,  4,  -18, 242, 38,  -15, 5,  10, -33, 220, 75,  -25, 9,  13, -40, 190, 115, -34, 12,
       13, -40, 155, 155, -40, 13, 12, -34, 115, 190, -40, 13, 9,  -25, 75,  220, -33, 10, 5,  -15, 38,  242, -18, 4},
    },

    // Band_8_1055
    {
      // Scaler inc range
      {1054, 1055},
      // Coefficients 4 taps
      {6,   244, 6,   0,   -15, 244, 35,  -8,  -27, 226, 72,  -15, -31, 196, 114, -23,
       -29, 157, 157, -29, -23, 114, 196, -31, -15, 72,  226, -27, -8,  35,  244, -15},
      // Coefficients 6 taps
      {-5, 7,   252, 7,   -5,  0,  4,  -18, 241, 39,  -15, 5,  10, -33, 219, 76,  -25, 9,  13, -40, 191, 115, -35, 12,
       13, -40, 155, 155, -40, 13, 12, -35, 115, 191, -40, 13, 9,  -25, 76,  219, -33, 10, 5,  -15, 39,  241, -18, 4},
    },

    // Band_8_1056
    {
      // Scaler inc range
      {1055, 1056},
      // Coefficients 4 taps
      {6,   244, 6,   0,   -14, 243, 35,  -8,  -26, 225, 72,  -15, -31, 196, 114, -23,
       -29, 157, 157, -29, -23, 114, 196, -31, -15, 72,  225, -26, -8,  35,  243, -14},
      // Coefficients 6 taps
      {-5, 7,   252, 7,   -5,  0,  4,  -17, 240, 39,  -15, 5,  10, -33, 220, 76,  -26, 9,  13, -40, 191, 115, -35, 12,
       13, -40, 155, 155, -40, 13, 12, -35, 115, 191, -40, 13, 9,  -26, 76,  220, -33, 10, 5,  -15, 39,  240, -17, 4},
    },

    // Band_8_1057
    {
      // Scaler inc range
      {1056, 1057},
      // Coefficients 4 taps
      {6,   244, 6,   0,   -14, 243, 35,  -8,  -26, 225, 72,  -15, -31, 196, 114, -23,
       -29, 157, 157, -29, -23, 114, 196, -31, -15, 72,  225, -26, -8,  35,  243, -14},
      // Coefficients 6 taps
      {-5, 7,   252, 7,   -5,  0,  4,  -17, 240, 39,  -15, 5,  10, -33, 220, 76,  -26, 9,  13, -40, 191, 115, -35, 12,
       13, -40, 155, 155, -40, 13, 12, -35, 115, 191, -40, 13, 9,  -26, 76,  220, -33, 10, 5,  -15, 39,  240, -17, 4},
    },

    // Band_8_1058
    {
      // Scaler inc range
      {1057, 1058},
      // Coefficients 4 taps
      {6,   244, 6,   0,   -14, 243, 35,  -8,  -26, 225, 72,  -15, -31, 196, 114, -23,
       -29, 157, 157, -29, -23, 114, 196, -31, -15, 72,  225, -26, -8,  35,  243, -14},
      // Coefficients 6 taps
      {-5, 7,   252, 7,   -5,  0,  4,  -17, 240, 39,  -15, 5,  10, -33, 220, 76,  -26, 9,  13, -40, 191, 115, -35, 12,
       13, -40, 155, 155, -40, 13, 12, -35, 115, 191, -40, 13, 9,  -26, 76,  220, -33, 10, 5,  -15, 39,  240, -17, 4},
    },

    // Band_8_1059
    {
      // Scaler inc range
      {1058, 1059},
      // Coefficients 4 taps
      {6,   244, 6,   0,   -14, 242, 36,  -8,  -26, 225, 72,  -15, -30, 195, 114, -23,
       -29, 157, 157, -29, -23, 114, 195, -30, -15, 72,  225, -26, -8,  36,  242, -14},
      // Coefficients 6 taps
      {-5, 8,   250, 8,   -5,  0,  4,  -17, 239, 39,  -15, 6,  10, -32, 219, 76,  -26, 9,  13, -40, 190, 116, -35, 12,
       13, -40, 155, 155, -40, 13, 12, -35, 116, 190, -40, 13, 9,  -26, 76,  219, -32, 10, 6,  -15, 39,  239, -17, 4},
    },

    // Band_8_1060
    {
      // Scaler inc range
      {1059, 1060},
      // Coefficients 4 taps
      {7,   242, 7,   0,   -14, 242, 36,  -8,  -26, 224, 73,  -15, -30, 195, 114, -23,
       -29, 157, 157, -29, -23, 114, 195, -30, -15, 73,  224, -26, -8,  36,  242, -14},
      // Coefficients 6 taps
      {-5, 8,   250, 8,   -5,  0,  4,  -17, 239, 40,  -16, 6,  10, -32, 219, 76,  -26, 9,  13, -40, 190, 116, -35, 12,
       13, -40, 155, 155, -40, 13, 12, -35, 116, 190, -40, 13, 9,  -26, 76,  219, -32, 10, 6,  -16, 40,  239, -17, 4},
    },

    // Band_8_1061
    {
      // Scaler inc range
      {1060, 1061},
      // Coefficients 4 taps
      {7,   242, 7,   0,   -14, 242, 36,  -8,  -26, 224, 73,  -15, -30, 195, 114, -23,
       -29, 157, 157, -29, -23, 114, 195, -30, -15, 73,  224, -26, -8,  36,  242, -14},
      // Coefficients 6 taps
      {-5, 8,   250, 8,   -5,  0,  3,  -16, 239, 40,  -16, 6,  9, -32, 219, 77,  -26, 9, 13, -40, 190, 116, -35, 12,
       13, -40, 155, 155, -40, 13, 12, -35, 116, 190, -40, 13, 9, -26, 77,  219, -32, 9, 6,  -16, 40,  239, -16, 3},
    },

    // Band_8_1062
    {
      // Scaler inc range
      {1061, 1062},
      // Coefficients 4 taps
      {7,   242, 7,   0,   -13, 241, 36,  -8,  -26, 224, 73,  -15, -30, 195, 114, -23,
       -29, 157, 157, -29, -23, 114, 195, -30, -15, 73,  224, -26, -8,  36,  241, -13},
      // Coefficients 6 taps
      {-6, 8,   252, 8,   -6,  0,  3,  -16, 239, 40,  -16, 6,  9, -32, 219, 77,  -26, 9, 13, -39, 189, 116, -35, 12,
       13, -40, 155, 155, -40, 13, 12, -35, 116, 189, -39, 13, 9, -26, 77,  219, -32, 9, 6,  -16, 40,  239, -16, 3},
    },

    // Band_8_1063
    {
      // Scaler inc range
      {1062, 1063},
      // Coefficients 4 taps
      {7,   242, 7,   0,   -13, 241, 36,  -8,  -26, 224, 73,  -15, -30, 195, 114, -23,
       -28, 156, 156, -28, -23, 114, 195, -30, -15, 73,  224, -26, -8,  36,  241, -13},
      // Coefficients 6 taps
      {-6, 9,   250, 9,   -6,  0,  3,  -16, 239, 40,  -16, 6,  9, -32, 219, 77,  -26, 9, 13, -39, 189, 116, -35, 12,
       13, -40, 155, 155, -40, 13, 12, -35, 116, 189, -39, 13, 9, -26, 77,  219, -32, 9, 6,  -16, 40,  239, -16, 3},
    },

    // Band_8_1065
    {
      // Scaler inc range
      {1063, 1065},
      // Coefficients 4 taps
      {7,   242, 7,   0,   -13, 241, 36,  -8,  -25, 224, 73,  -16, -30, 195, 114, -23,
       -28, 156, 156, -28, -23, 114, 195, -30, -16, 73,  224, -25, -8,  36,  241, -13},
      // Coefficients 6 taps
      {-6, 9,   250, 9,   -6,  0,  3,  -16, 239, 40,  -16, 6,  9, -32, 219, 77,  -26, 9, 12, -39, 190, 116, -35, 12,
       13, -40, 155, 155, -40, 13, 12, -35, 116, 190, -39, 12, 9, -26, 77,  219, -32, 9, 6,  -16, 40,  239, -16, 3},
    },

    // Band_8_1066
    {
      // Scaler inc range
      {1065, 1066},
      // Coefficients 4 taps
      {7,   242, 7,   0,   -13, 240, 37,  -8,  -25, 224, 73,  -16, -30, 194, 115, -23,
       -28, 156, 156, -28, -23, 115, 194, -30, -16, 73,  224, -25, -8,  37,  240, -13},
      // Coefficients 6 taps
      {-6, 9,   250, 9,   -6,  0,  3,  -16, 238, 41,  -16, 6,  9, -32, 219, 77,  -26, 9, 12, -39, 190, 116, -35, 12,
       13, -40, 155, 155, -40, 13, 12, -35, 116, 190, -39, 12, 9, -26, 77,  219, -32, 9, 6,  -16, 41,  238, -16, 3},
    },

    // Band_8_1067
    {
      // Scaler inc range
      {1066, 1067},
      // Coefficients 4 taps
      {8,   240, 8,   0,   -13, 240, 37,  -8,  -25, 224, 73,  -16, -30, 194, 115, -23,
       -28, 156, 156, -28, -23, 115, 194, -30, -16, 73,  224, -25, -8,  37,  240, -13},
      // Coefficients 6 taps
      {-6, 9,   250, 9,   -6,  0,  3,  -15, 237, 41,  -16, 6,  9, -31, 219, 77,  -27, 9, 12, -39, 190, 116, -35, 12,
       13, -40, 155, 155, -40, 13, 12, -35, 116, 190, -39, 12, 9, -27, 77,  219, -31, 9, 6,  -16, 41,  237, -15, 3},
    },

    // Band_8_1068
    {
      // Scaler inc range
      {1067, 1068},
      // Coefficients 4 taps
      {8,   240, 8,   0,   -13, 240, 37,  -8,  -25, 224, 73,  -16, -30, 194, 115, -23,
       -28, 156, 156, -28, -23, 115, 194, -30, -16, 73,  224, -25, -8,  37,  240, -13},
      // Coefficients 6 taps
      {-6, 9,   250, 9,   -6,  0,  3,  -15, 238, 41,  -17, 6,  9, -31, 219, 77,  -27, 9, 12, -39, 190, 116, -35, 12,
       13, -40, 155, 155, -40, 13, 12, -35, 116, 190, -39, 12, 9, -27, 77,  219, -31, 9, 6,  -17, 41,  238, -15, 3},
    },

    // Band_8_1069
    {
      // Scaler inc range
      {1068, 1069},
      // Coefficients 4 taps
      {8,   240, 8,   0,   -12, 240, 37,  -9,  -25, 223, 74,  -16, -30, 194, 115, -23,
       -28, 156, 156, -28, -23, 115, 194, -30, -16, 74,  223, -25, -9,  37,  240, -12},
      // Coefficients 6 taps
      {-7, 10,  250, 10,  -7,  0,  2,  -15, 239, 41,  -17, 6,  9, -31, 218, 78,  -27, 9, 12, -39, 190, 116, -35, 12,
       13, -40, 155, 155, -40, 13, 12, -35, 116, 190, -39, 12, 9, -27, 78,  218, -31, 9, 6,  -17, 41,  239, -15, 2},
    },

    // Band_8_1070
    {
      // Scaler inc range
      {1069, 1070},
      // Coefficients 4 taps
      {8,   240, 8,   0,   -12, 240, 37,  -9,  -25, 223, 74,  -16, -30, 194, 115, -23,
       -28, 156, 156, -28, -23, 115, 194, -30, -16, 74,  223, -25, -9,  37,  240, -12},
      // Coefficients 6 taps
      {-7, 10,  250, 10,  -7,  0,  2,  -15, 239, 41,  -17, 6,  9, -31, 218, 78,  -27, 9, 12, -39, 190, 116, -35, 12,
       13, -40, 155, 155, -40, 13, 12, -35, 116, 190, -39, 12, 9, -27, 78,  218, -31, 9, 6,  -17, 41,  239, -15, 2},
    },

    // Band_8_1071
    {
      // Scaler inc range
      {1070, 1071},
      // Coefficients 4 taps
      {8,   240, 8,   0,   -12, 240, 37,  -9,  -25, 223, 74,  -16, -30, 194, 115, -23,
       -28, 156, 156, -28, -23, 115, 194, -30, -16, 74,  223, -25, -9,  37,  240, -12},
      // Coefficients 6 taps
      {-7, 10,  250, 10,  -7,  0,  2,  -15, 238, 42,  -17, 6,  8, -31, 219, 78,  -27, 9, 12, -39, 189, 117, -35, 12,
       13, -40, 155, 155, -40, 13, 12, -35, 117, 189, -39, 12, 9, -27, 78,  219, -31, 8, 6,  -17, 42,  238, -15, 2},
    },

    // Band_8_1072
    {
      // Scaler inc range
      {1071, 1072},
      // Coefficients 4 taps
      {9,   238, 9,   0,   -12, 239, 38,  -9,  -24, 222, 74,  -16, -29, 193, 115, -23,
       -28, 156, 156, -28, -23, 115, 193, -29, -16, 74,  222, -24, -9,  38,  239, -12},
      // Coefficients 6 taps
      {-7, 10,  250, 10,  -7,  0,  2,  -14, 237, 42,  -17, 6,  8, -31, 219, 78,  -27, 9, 12, -39, 189, 117, -35, 12,
       13, -40, 155, 155, -40, 13, 12, -35, 117, 189, -39, 12, 9, -27, 78,  219, -31, 8, 6,  -17, 42,  237, -14, 2},
    },

    // Band_8_1073
    {
      // Scaler inc range
      {1072, 1073},
      // Coefficients 4 taps
      {9,   238, 9,   0,   -12, 239, 38,  -9,  -24, 222, 74,  -16, -29, 193, 115, -23,
       -28, 156, 156, -28, -23, 115, 193, -29, -16, 74,  222, -24, -9,  38,  239, -12},
      // Coefficients 6 taps
      {-7, 11,  248, 11,  -7,  0,  2,  -14, 237, 42,  -17, 6,  8, -30, 218, 78,  -27, 9, 12, -38, 188, 117, -35, 12,
       13, -40, 155, 155, -40, 13, 12, -35, 117, 188, -38, 12, 9, -27, 78,  218, -30, 8, 6,  -17, 42,  237, -14, 2},
    },

    // Band_8_1074
    {
      // Scaler inc range
      {1073, 1074},
      // Coefficients 4 taps
      {9,   238, 9,   0,   -12, 239, 38,  -9,  -24, 222, 74,  -16, -29, 193, 115, -23,
       -28, 156, 156, -28, -23, 115, 193, -29, -16, 74,  222, -24, -9,  38,  239, -12},
      // Coefficients 6 taps
      {-7, 11,  248, 11,  -7,  0,  2,  -14, 237, 42,  -17, 6,  8, -30, 218, 78,  -27, 9, 12, -38, 188, 117, -35, 12,
       13, -40, 155, 155, -40, 13, 12, -35, 117, 188, -38, 12, 9, -27, 78,  218, -30, 8, 6,  -17, 42,  237, -14, 2},
    },

    // Band_8_1075
    {
      // Scaler inc range
      {1074, 1075},
      // Coefficients 4 taps
      {9,   238, 9,   0,   -11, 238, 38,  -9,  -24, 222, 74,  -16, -29, 193, 115, -23,
       -28, 156, 156, -28, -23, 115, 193, -29, -16, 74,  222, -24, -9,  38,  238, -11},
      // Coefficients 6 taps
      {-7, 11,  248, 11,  -7,  0,  2,  -14, 237, 42,  -17, 6,  8, -30, 217, 79,  -27, 9, 12, -38, 188, 117, -35, 12,
       13, -39, 154, 154, -39, 13, 12, -35, 117, 188, -38, 12, 9, -27, 79,  217, -30, 8, 6,  -17, 42,  237, -14, 2},
    },

    // Band_8_1077
    {
      // Scaler inc range
      {1075, 1077},
      // Coefficients 4 taps
      {9,   238, 9,   0,   -11, 238, 38,  -9,  -24, 222, 74,  -16, -29, 193, 115, -23,
       -28, 156, 156, -28, -23, 115, 193, -29, -16, 74,  222, -24, -9,  38,  238, -11},
      // Coefficients 6 taps
      {-8, 11,  250, 11,  -8,  0,  1,  -14, 238, 43,  -18, 6,  8,  -30, 216, 79,  -27, 10, 12, -38, 188, 117, -35, 12,
       13, -39, 154, 154, -39, 13, 12, -35, 117, 188, -38, 12, 10, -27, 79,  216, -30, 8,  6,  -18, 43,  238, -14, 1},
    },

    // Band_8_1078
    {
      // Scaler inc range
      {1077, 1078},
      // Coefficients 4 taps
      {9,   238, 9,   0,   -11, 238, 38,  -9,  -24, 221, 75,  -16, -29, 193, 115, -23,
       -28, 156, 156, -28, -23, 115, 193, -29, -16, 75,  221, -24, -9,  38,  238, -11},
      // Coefficients 6 taps
      {-8, 12,  248, 12,  -8,  0,  1,  -13, 237, 43,  -18, 6,  8,  -30, 216, 79,  -27, 10, 12, -38, 188, 117, -35, 12,
       13, -39, 154, 154, -39, 13, 12, -35, 117, 188, -38, 12, 10, -27, 79,  216, -30, 8,  6,  -18, 43,  237, -13, 1},
    },

    // Band_8_1079
    {
      // Scaler inc range
      {1078, 1079},
      // Coefficients 4 taps
      {10,  236, 10,  0,   -11, 237, 39,  -9,  -24, 221, 75,  -16, -29, 193, 115, -23,
       -28, 156, 156, -28, -23, 115, 193, -29, -16, 75,  221, -24, -9,  39,  237, -11},
      // Coefficients 6 taps
      {-8, 12,  248, 12,  -8,  0,  1,  -13, 236, 43,  -18, 7,  8,  -30, 217, 79,  -28, 10, 11, -38, 189, 117, -35, 12,
       13, -39, 154, 154, -39, 13, 12, -35, 117, 189, -38, 11, 10, -28, 79,  217, -30, 8,  7,  -18, 43,  236, -13, 1},
    },

    // Band_8_1080
    {
      // Scaler inc range
      {1079, 1080},
      // Coefficients 4 taps
      {10,  236, 10,  0,   -11, 237, 39,  -9,  -23, 220, 75,  -16, -29, 193, 115, -23,
       -28, 156, 156, -28, -23, 115, 193, -29, -16, 75,  220, -23, -9,  39,  237, -11},
      // Coefficients 6 taps
      {-8, 12,  248, 12,  -8,  0,  1,  -13, 236, 43,  -18, 7,  8,  -29, 216, 79,  -28, 10, 11, -38, 189, 117, -35, 12,
       13, -39, 154, 154, -39, 13, 12, -35, 117, 189, -38, 11, 10, -28, 79,  216, -29, 8,  7,  -18, 43,  236, -13, 1},
    },

    // Band_8_1081
    {
      // Scaler inc range
      {1080, 1081},
      // Coefficients 4 taps
      {10,  236, 10,  0,   -10, 236, 39,  -9,  -23, 220, 75,  -16, -29, 193, 115, -23,
       -28, 156, 156, -28, -23, 115, 193, -29, -16, 75,  220, -23, -9,  39,  236, -10},
      // Coefficients 6 taps
      {-8, 12,  248, 12,  -8,  0,  1,  -13, 236, 43,  -18, 7,  7,  -29, 217, 79,  -28, 10, 11, -38, 189, 117, -35, 12,
       13, -39, 154, 154, -39, 13, 12, -35, 117, 189, -38, 11, 10, -28, 79,  217, -29, 7,  7,  -18, 43,  236, -13, 1},
    },

    // Band_8_1082
    {
      // Scaler inc range
      {1081, 1082},
      // Coefficients 4 taps
      {10,  236, 10,  0,   -10, 236, 39,  -9,  -23, 220, 75,  -16, -29, 193, 115, -23,
       -28, 156, 156, -28, -23, 115, 193, -29, -16, 75,  220, -23, -9,  39,  236, -10},
      // Coefficients 6 taps
      {-8, 12,  248, 12,  -8,  0,  1,  -13, 235, 44,  -18, 7,  7,  -29, 217, 79,  -28, 10, 11, -38, 189, 117, -35, 12,
       13, -39, 154, 154, -39, 13, 12, -35, 117, 189, -38, 11, 10, -28, 79,  217, -29, 7,  7,  -18, 44,  235, -13, 1},
    },

    // Band_8_1083
    {
      // Scaler inc range
      {1082, 1083},
      // Coefficients 4 taps
      {10,  236, 10,  0,   -10, 236, 39,  -9,  -23, 220, 75,  -16, -29, 193, 115, -23,
       -28, 156, 156, -28, -23, 115, 193, -29, -16, 75,  220, -23, -9,  39,  236, -10},
      // Coefficients 6 taps
      {-8, 13,  246, 13,  -8,  0,  1,  -12, 234, 44,  -18, 7,  7,  -29, 216, 80,  -28, 10, 11, -37, 188, 117, -35, 12,
       13, -39, 154, 154, -39, 13, 12, -35, 117, 188, -37, 11, 10, -28, 80,  216, -29, 7,  7,  -18, 44,  234, -12, 1},
    },

    // Band_8_1084
    {
      // Scaler inc range
      {1083, 1084},
      // Coefficients 4 taps
      {10,  236, 10,  0,   -10, 236, 39,  -9,  -23, 220, 75,  -16, -28, 192, 115, -23,
       -28, 156, 156, -28, -23, 115, 192, -28, -16, 75,  220, -23, -9,  39,  236, -10},
      // Coefficients 6 taps
      {-9, 13,  248, 13,  -9,  0,  0,  -12, 235, 44,  -18, 7,  7,  -29, 216, 80,  -28, 10, 11, -37, 187, 118, -35, 12,
       13, -39, 154, 154, -39, 13, 12, -35, 118, 187, -37, 11, 10, -28, 80,  216, -29, 7,  7,  -18, 44,  235, -12, 0},
    },

    // Band_8_1085
    {
      // Scaler inc range
      {1084, 1085},
      // Coefficients 4 taps
      {11,  234, 11,  0,   -10, 235, 40,  -9,  -23, 220, 75,  -16, -28, 191, 116, -23,
       -28, 156, 156, -28, -23, 116, 191, -28, -16, 75,  220, -23, -9,  40,  235, -10},
      // Coefficients 6 taps
      {-9, 13,  248, 13,  -9,  0,  0,  -12, 236, 44,  -19, 7,  7,  -29, 216, 80,  -28, 10, 11, -37, 187, 118, -35, 12,
       12, -39, 155, 155, -39, 12, 12, -35, 118, 187, -37, 11, 10, -28, 80,  216, -29, 7,  7,  -19, 44,  236, -12, 0},
    },

    // Band_8_1087
    {
      // Scaler inc range
      {1085, 1087},
      // Coefficients 4 taps
      {11,  234, 11,  0,   -10, 236, 40,  -10, -23, 219, 76,  -16, -28, 191, 116, -23,
       -28, 156, 156, -28, -23, 116, 191, -28, -16, 76,  219, -23, -10, 40,  236, -10},
      // Coefficients 6 taps
      {-9, 13,  248, 13,  -9,  0,  0,  -12, 236, 44,  -19, 7,  7,  -28, 215, 80,  -28, 10, 11, -37, 187, 118, -35, 12,
       12, -39, 155, 155, -39, 12, 12, -35, 118, 187, -37, 11, 10, -28, 80,  215, -28, 7,  7,  -19, 44,  236, -12, 0},
    },

    // Band_8_1088
    {
      // Scaler inc range
      {1087, 1088},
      // Coefficients 4 taps
      {11,  234, 11,  0,   -9,  235, 40,  -10, -22, 218, 76,  -16, -28, 191, 116, -23,
       -28, 156, 156, -28, -23, 116, 191, -28, -16, 76,  218, -22, -10, 40,  235, -9},
      // Coefficients 6 taps
      {-9, 14,  246, 14,  -9,  0,  0,  -11, 234, 45,  -19, 7,  7,  -28, 215, 80,  -28, 10, 11, -37, 188, 118, -36, 12,
       12, -39, 155, 155, -39, 12, 12, -36, 118, 188, -37, 11, 10, -28, 80,  215, -28, 7,  7,  -19, 45,  234, -11, 0},
    },

    // Band_8_1089
    {
      // Scaler inc range
      {1088, 1089},
      // Coefficients 4 taps
      {11,  234, 11,  0,   -9,  235, 40,  -10, -22, 219, 76,  -17, -28, 191, 116, -23,
       -28, 156, 156, -28, -23, 116, 191, -28, -17, 76,  219, -22, -10, 40,  235, -9},
      // Coefficients 6 taps
      {-9, 14,  246, 14,  -9,  0,  0,  -11, 234, 45,  -19, 7,  7,  -28, 215, 80,  -28, 10, 11, -37, 188, 118, -36, 12,
       12, -39, 155, 155, -39, 12, 12, -36, 118, 188, -37, 11, 10, -28, 80,  215, -28, 7,  7,  -19, 45,  234, -11, 0},
    },

    // Band_8_1090
    {
      // Scaler inc range
      {1089, 1090},
      // Coefficients 4 taps
      {11,  234, 11,  0,   -9,  235, 40,  -10, -22, 219, 76,  -17, -28, 191, 116, -23,
       -28, 156, 156, -28, -23, 116, 191, -28, -17, 76,  219, -22, -10, 40,  235, -9},
      // Coefficients 6 taps
      {-9, 14,  246, 14,  -9,  0,  0,  -11, 234, 45,  -19, 7,  6,  -28, 215, 81,  -28, 10, 11, -37, 188, 118, -36, 12,
       12, -39, 155, 155, -39, 12, 12, -36, 118, 188, -37, 11, 10, -28, 81,  215, -28, 6,  7,  -19, 45,  234, -11, 0},
    },

    // Band_8_1091
    {
      // Scaler inc range
      {1090, 1091},
      // Coefficients 4 taps
      {12,  232, 12,  0,   -9,  235, 40,  -10, -22, 219, 76,  -17, -28, 191, 116, -23,
       -28, 156, 156, -28, -23, 116, 191, -28, -17, 76,  219, -22, -10, 40,  235, -9},
      // Coefficients 6 taps
      {-10, 14,  248, 14,  -10, 0,  0,  -11, 234, 45,  -19, 7,  6,  -28, 215, 81,  -28, 10, 11, -37, 188, 118, -36, 12,
       12,  -39, 155, 155, -39, 12, 12, -36, 118, 188, -37, 11, 10, -28, 81,  215, -28, 6,  7,  -19, 45,  234, -11, 0},
    },

    // Band_8_1092
    {
      // Scaler inc range
      {1091, 1092},
      // Coefficients 4 taps
      {12,  232, 12,  0,   -9,  234, 41,  -10, -22, 219, 76,  -17, -28, 191, 116, -23,
       -28, 156, 156, -28, -23, 116, 191, -28, -17, 76,  219, -22, -10, 41,  234, -9},
      // Coefficients 6 taps
      {-10, 14,  248, 14,  -10, 0,  -1, -11, 235, 45,  -19, 7,  6,  -28, 216, 81,  -29, 10, 10, -37, 189, 118, -36, 12,
       12,  -39, 155, 155, -39, 12, 12, -36, 118, 189, -37, 10, 10, -29, 81,  216, -28, 6,  7,  -19, 45,  235, -11, -1},
    },

    // Band_8_1093
    {
      // Scaler inc range
      {1092, 1093},
      // Coefficients 4 taps
      {12,  232, 12,  0,   -9,  234, 41,  -10, -22, 219, 76,  -17, -28, 191, 116, -23,
       -28, 156, 156, -28, -23, 116, 191, -28, -17, 76,  219, -22, -10, 41,  234, -9},
      // Coefficients 6 taps
      {-10, 15,  246, 15,  -10, 0,  -1, -10, 233, 46,  -19, 7,  6,  -27, 215, 81,  -29, 10, 10, -36, 188, 118, -36, 12,
       12,  -39, 155, 155, -39, 12, 12, -36, 118, 188, -36, 10, 10, -29, 81,  215, -27, 6,  7,  -19, 46,  233, -10, -1},
    },

    // Band_8_1095
    {
      // Scaler inc range
      {1093, 1095},
      // Coefficients 4 taps
      {12,  232, 12,  0,   -8,  233, 41,  -10, -22, 219, 76,  -17, -28, 191, 116, -23,
       -28, 156, 156, -28, -23, 116, 191, -28, -17, 76,  219, -22, -10, 41,  233, -8},
      // Coefficients 6 taps
      {-10, 15,  246, 15,  -10, 0,  -1, -10, 234, 46,  -20, 7,  6,  -27, 215, 81,  -29, 10, 10, -36, 188, 118, -36, 12,
       12,  -39, 155, 155, -39, 12, 12, -36, 118, 188, -36, 10, 10, -29, 81,  215, -27, 6,  7,  -20, 46,  234, -10, -1},
    },

    // Band_8_1096
    {
      // Scaler inc range
      {1095, 1096},
      // Coefficients 4 taps
      {12,  232, 12,  0,   -8,  233, 41,  -10, -21, 218, 76,  -17, -28, 191, 116, -23,
       -27, 155, 155, -27, -23, 116, 191, -28, -17, 76,  218, -21, -10, 41,  233, -8},
      // Coefficients 6 taps
      {-10, 15,  246, 15,  -10, 0,  -1, -10, 234, 46,  -20, 7,  6,  -27, 215, 81,  -29, 10, 10, -36, 188, 118, -36, 12,
       12,  -39, 155, 155, -39, 12, 12, -36, 118, 188, -36, 10, 10, -29, 81,  215, -27, 6,  7,  -20, 46,  234, -10, -1},
    },

    // Band_8_1097
    {
      // Scaler inc range
      {1096, 1097},
      // Coefficients 4 taps
      {12,  232, 12,  0,   -8,  233, 41,  -10, -21, 217, 77,  -17, -27, 190, 116, -23,
       -27, 155, 155, -27, -23, 116, 190, -27, -17, 77,  217, -21, -10, 41,  233, -8},
      // Coefficients 6 taps
      {-10, 15,  246, 15,  -10, 0,  -1, -10, 234, 46,  -20, 7,  6,  -27, 215, 81,  -29, 10, 10, -36, 188, 118, -36, 12,
       12,  -39, 155, 155, -39, 12, 12, -36, 118, 188, -36, 10, 10, -29, 81,  215, -27, 6,  7,  -20, 46,  234, -10, -1},
    },

    // Band_8_1098
    {
      // Scaler inc range
      {1097, 1098},
      // Coefficients 4 taps
      {13,  230, 13,  0,   -8,  233, 41,  -10, -21, 217, 77,  -17, -27, 190, 116, -23,
       -27, 155, 155, -27, -23, 116, 190, -27, -17, 77,  217, -21, -10, 41,  233, -8},
      // Coefficients 6 taps
      {-10, 16,  244, 16,  -10, 0,  -1, -10, 234, 46,  -20, 7,  6,  -27, 214, 82,  -29, 10, 10, -36, 188, 118, -36, 12,
       12,  -39, 155, 155, -39, 12, 12, -36, 118, 188, -36, 10, 10, -29, 82,  214, -27, 6,  7,  -20, 46,  234, -10, -1},
    },

    // Band_8_1099
    {
      // Scaler inc range
      {1098, 1099},
      // Coefficients 4 taps
      {13,  230, 13,  0,   -8,  232, 42,  -10, -21, 217, 77,  -17, -27, 190, 116, -23,
       -27, 155, 155, -27, -23, 116, 190, -27, -17, 77,  217, -21, -10, 42,  232, -8},
      // Coefficients 6 taps
      {-11, 16,  246, 16,  -11, 0,  -1, -9,  232, 47,  -20, 7,  6,  -26, 213, 82,  -29, 10, 10, -36, 187, 119, -36, 12,
       12,  -39, 155, 155, -39, 12, 12, -36, 119, 187, -36, 10, 10, -29, 82,  213, -26, 6,  7,  -20, 47,  232, -9,  -1},
    },

    // Band_8_1100
    {
      // Scaler inc range
      {1099, 1100},
      // Coefficients 4 taps
      {13,  230, 13,  0,   -8,  232, 42,  -10, -21, 217, 77,  -17, -27, 190, 116, -23,
       -27, 155, 155, -27, -23, 116, 190, -27, -17, 77,  217, -21, -10, 42,  232, -8},
      // Coefficients 6 taps
      {-11, 16,  246, 16,  -11, 0,  -2, -9,  233, 47,  -20, 7,  5,  -26, 214, 82,  -29, 10, 10, -36, 187, 119, -36, 12,
       12,  -38, 154, 154, -38, 12, 12, -36, 119, 187, -36, 10, 10, -29, 82,  214, -26, 5,  7,  -20, 47,  233, -9,  -2},
    },

    // Band_8_1101
    {
      // Scaler inc range
      {1100, 1101},
      // Coefficients 4 taps
      {13,  230, 13,  0,   -7,  231, 42,  -10, -21, 217, 77,  -17, -27, 190, 116, -23,
       -27, 155, 155, -27, -23, 116, 190, -27, -17, 77,  217, -21, -10, 42,  231, -7},
      // Coefficients 6 taps
      {-11, 16,  246, 16,  -11, 0,  -2, -9,  233, 47,  -20, 7,  5,  -26, 214, 82,  -29, 10, 10, -36, 187, 119, -36, 12,
       12,  -38, 154, 154, -38, 12, 12, -36, 119, 187, -36, 10, 10, -29, 82,  214, -26, 5,  7,  -20, 47,  233, -9,  -2},
    },

    // Band_8_1103
    {
      // Scaler inc range
      {1101, 1103},
      // Coefficients 4 taps
      {13,  230, 13,  0,   -7,  231, 42,  -10, -21, 217, 77,  -17, -27, 190, 116, -23,
       -27, 155, 155, -27, -23, 116, 190, -27, -17, 77,  217, -21, -10, 42,  231, -7},
      // Coefficients 6 taps
      {-11, 17,  244, 17,  -11, 0,  -2, -9,  233, 47,  -20, 7,  5,  -26, 214, 82,  -29, 10, 10, -36, 187, 119, -36, 12,
       12,  -38, 154, 154, -38, 12, 12, -36, 119, 187, -36, 10, 10, -29, 82,  214, -26, 5,  7,  -20, 47,  233, -9,  -2},
    },

    // Band_8_1104
    {
      // Scaler inc range
      {1103, 1104},
      // Coefficients 4 taps
      {13,  230, 13,  0,   -7,  231, 42,  -10, -20, 216, 77,  -17, -27, 190, 116, -23,
       -27, 155, 155, -27, -23, 116, 190, -27, -17, 77,  216, -20, -10, 42,  231, -7},
      // Coefficients 6 taps
      {-11, 17,  244, 17,  -11, 0,  -2, -9,  232, 47,  -20, 8,  5,  -26, 214, 82,  -29, 10, 10, -35, 186, 119, -36, 12,
       12,  -38, 154, 154, -38, 12, 12, -36, 119, 186, -35, 10, 10, -29, 82,  214, -26, 5,  8,  -20, 47,  232, -9,  -2},
    },

    // Band_8_1105
    {
      // Scaler inc range
      {1104, 1105},
      // Coefficients 4 taps
      {14,  228, 14,  0,   -7,  231, 42,  -10, -20, 216, 77,  -17, -27, 190, 116, -23,
       -27, 155, 155, -27, -23, 116, 190, -27, -17, 77,  216, -20, -10, 42,  231, -7},
      // Coefficients 6 taps
      {-11, 17,  244, 17,  -11, 0,  -2, -8,  231, 48,  -21, 8, 5,  -26, 214, 82,  -29, 10, 9, -35, 187, 119, -36, 12,
       12,  -38, 154, 154, -38, 12, 12, -36, 119, 187, -35, 9, 10, -29, 82,  214, -26, 5,  8, -21, 48,  231, -8,  -2},
    },

    // Band_8_1106
    {
      // Scaler inc range
      {1105, 1106},
      // Coefficients 4 taps
      {14,  228, 14,  0,   -7,  230, 43,  -10, -20, 215, 78,  -17, -27, 190, 116, -23,
       -27, 155, 155, -27, -23, 116, 190, -27, -17, 78,  215, -20, -10, 43,  230, -7},
      // Coefficients 6 taps
      {-11, 17,  244, 17,  -11, 0,  -2, -8,  231, 48,  -21, 8, 5,  -25, 212, 83,  -29, 10, 9, -35, 187, 119, -36, 12,
       12,  -38, 154, 154, -38, 12, 12, -36, 119, 187, -35, 9, 10, -29, 83,  212, -25, 5,  8, -21, 48,  231, -8,  -2},
    },

    // Band_8_1107
    {
      // Scaler inc range
      {1106, 1107},
      // Coefficients 4 taps
      {14,  228, 14,  0,   -6,  230, 43,  -11, -20, 215, 78,  -17, -27, 190, 116, -23,
       -27, 155, 155, -27, -23, 116, 190, -27, -17, 78,  215, -20, -11, 43,  230, -6},
      // Coefficients 6 taps
      {-12, 17,  246, 17,  -12, 0,  -2, -8,  231, 48,  -21, 8, 5,  -25, 212, 83,  -29, 10, 9, -35, 187, 119, -36, 12,
       12,  -38, 154, 154, -38, 12, 12, -36, 119, 187, -35, 9, 10, -29, 83,  212, -25, 5,  8, -21, 48,  231, -8,  -2},
    },

    // Band_8_1108
    {
      // Scaler inc range
      {1107, 1108},
      // Coefficients 4 taps
      {14,  228, 14,  0,   -6,  230, 43,  -11, -20, 215, 78,  -17, -26, 189, 116, -23,
       -27, 155, 155, -27, -23, 116, 189, -26, -17, 78,  215, -20, -11, 43,  230, -6},
      // Coefficients 6 taps
      {-12, 18,  244, 18,  -12, 0,  -2, -8,  231, 48,  -21, 8, 5,  -25, 213, 83,  -30, 10, 9, -35, 187, 119, -36, 12,
       11,  -38, 155, 155, -38, 11, 12, -36, 119, 187, -35, 9, 10, -30, 83,  213, -25, 5,  8, -21, 48,  231, -8,  -2},
    },

    // Band_8_1110
    {
      // Scaler inc range
      {1108, 1110},
      // Coefficients 4 taps
      {14,  228, 14,  0,   -6,  230, 43,  -11, -20, 215, 78,  -17, -26, 189, 116, -23,
       -27, 155, 155, -27, -23, 116, 189, -26, -17, 78,  215, -20, -11, 43,  230, -6},
      // Coefficients 6 taps
      {-12, 18,  244, 18,  -12, 0,  -3, -7,  231, 48,  -21, 8, 4,  -25, 214, 83,  -30, 10, 9, -35, 187, 119, -36, 12,
       11,  -38, 155, 155, -38, 11, 12, -36, 119, 187, -35, 9, 10, -30, 83,  214, -25, 4,  8, -21, 48,  231, -7,  -3},
    },

    // Band_8_1111
    {
      // Scaler inc range
      {1110, 1111},
      // Coefficients 4 taps
      {14,  228, 14,  0,   -6,  230, 43,  -11, -20, 215, 78,  -17, -26, 189, 116, -23,
       -27, 155, 155, -27, -23, 116, 189, -26, -17, 78,  215, -20, -11, 43,  230, -6},
      // Coefficients 6 taps
      {-12, 18,  244, 18,  -12, 0,  -3, -7,  230, 49,  -21, 8, 4,  -25, 214, 83,  -30, 10, 9, -35, 187, 119, -36, 12,
       11,  -38, 155, 155, -38, 11, 12, -36, 119, 187, -35, 9, 10, -30, 83,  214, -25, 4,  8, -21, 49,  230, -7,  -3},
    },

    // Band_8_1112
    {
      // Scaler inc range
      {1111, 1112},
      // Coefficients 4 taps
      {15,  226, 15,  0,   -6,  230, 43,  -11, -19, 214, 78,  -17, -26, 189, 116, -23,
       -27, 155, 155, -27, -23, 116, 189, -26, -17, 78,  214, -19, -11, 43,  230, -6},
      // Coefficients 6 taps
      {-12, 18,  244, 18,  -12, 0,  -3, -7,  230, 49,  -21, 8, 4,  -25, 214, 83,  -30, 10, 9, -35, 187, 119, -36, 12,
       11,  -38, 155, 155, -38, 11, 12, -36, 119, 187, -35, 9, 10, -30, 83,  214, -25, 4,  8, -21, 49,  230, -7,  -3},
    },

    // Band_8_1113
    {
      // Scaler inc range
      {1112, 1113},
      // Coefficients 4 taps
      {15,  226, 15,  0,   -6,  230, 43,  -11, -19, 214, 78,  -17, -26, 188, 117, -23,
       -27, 155, 155, -27, -23, 117, 188, -26, -17, 78,  214, -19, -11, 43,  230, -6},
      // Coefficients 6 taps
      {-12, 19,  242, 19,  -12, 0,  -3, -7,  230, 49,  -21, 8, 4,  -24, 212, 84,  -30, 10, 9, -34, 186, 119, -36, 12,
       11,  -38, 155, 155, -38, 11, 12, -36, 119, 186, -34, 9, 10, -30, 84,  212, -24, 4,  8, -21, 49,  230, -7,  -3},
    },

    // Band_8_1114
    {
      // Scaler inc range
      {1113, 1114},
      // Coefficients 4 taps
      {15,  226, 15,  0,   -5,  228, 44,  -11, -19, 214, 78,  -17, -26, 188, 117, -23,
       -27, 155, 155, -27, -23, 117, 188, -26, -17, 78,  214, -19, -11, 44,  228, -5},
      // Coefficients 6 taps
      {-12, 19,  242, 19,  -12, 0,  -3, -7,  230, 49,  -21, 8, 4,  -24, 212, 84,  -30, 10, 9, -34, 186, 120, -36, 11,
       11,  -38, 155, 155, -38, 11, 11, -36, 120, 186, -34, 9, 10, -30, 84,  212, -24, 4,  8, -21, 49,  230, -7,  -3},
    },

    // Band_8_1116
    {
      // Scaler inc range
      {1114, 1116},
      // Coefficients 4 taps
      {15,  226, 15,  0,   -5,  228, 44,  -11, -19, 214, 78,  -17, -26, 188, 117, -23,
       -27, 155, 155, -27, -23, 117, 188, -26, -17, 78,  214, -19, -11, 44,  228, -5},
      // Coefficients 6 taps
      {-13, 19,  244, 19,  -13, 0,  -3, -6,  230, 49,  -22, 8, 4,  -24, 212, 84,  -30, 10, 9, -34, 186, 120, -36, 11,
       11,  -38, 155, 155, -38, 11, 11, -36, 120, 186, -34, 9, 10, -30, 84,  212, -24, 4,  8, -22, 49,  230, -6,  -3},
    },

    // Band_8_1117
    {
      // Scaler inc range
      {1116, 1117},
      // Coefficients 4 taps
      {15,  226, 15,  0,   -5,  228, 44,  -11, -19, 214, 78,  -17, -26, 188, 117, -23,
       -27, 155, 155, -27, -23, 117, 188, -26, -17, 78,  214, -19, -11, 44,  228, -5},
      // Coefficients 6 taps
      {-13, 19,  244, 19,  -13, 0,  -3, -6,  229, 50,  -22, 8, 4,  -24, 212, 84,  -30, 10, 9, -34, 186, 120, -36, 11,
       11,  -38, 155, 155, -38, 11, 11, -36, 120, 186, -34, 9, 10, -30, 84,  212, -24, 4,  8, -22, 50,  229, -6,  -3},
    },

    // Band_8_1118
    {
      // Scaler inc range
      {1117, 1118},
      // Coefficients 4 taps
      {16,  224, 16,  0,   -5,  228, 44,  -11, -19, 213, 79,  -17, -26, 188, 117, -23,
       -27, 155, 155, -27, -23, 117, 188, -26, -17, 79,  213, -19, -11, 44,  228, -5},
      // Coefficients 6 taps
      {-13, 20,  242, 20,  -13, 0,  -4, -6,  230, 50,  -22, 8, 4,  -24, 212, 84,  -30, 10, 8, -34, 187, 120, -36, 11,
       11,  -38, 155, 155, -38, 11, 11, -36, 120, 187, -34, 8, 10, -30, 84,  212, -24, 4,  8, -22, 50,  230, -6,  -4},
    },

    // Band_8_1119
    {
      // Scaler inc range
      {1118, 1119},
      // Coefficients 4 taps
      {16,  224, 16,  0,   -5,  228, 44,  -11, -19, 213, 79,  -17, -25, 187, 117, -23,
       -27, 155, 155, -27, -23, 117, 187, -25, -17, 79,  213, -19, -11, 44,  228, -5},
      // Coefficients 6 taps
      {-13, 20,  242, 20,  -13, 0,  -4, -6,  230, 50,  -22, 8, 3,  -23, 212, 84,  -30, 10, 8, -34, 187, 120, -36, 11,
       11,  -38, 155, 155, -38, 11, 11, -36, 120, 187, -34, 8, 10, -30, 84,  212, -23, 3,  8, -22, 50,  230, -6,  -4},
    },

    // Band_8_1120
    {
      // Scaler inc range
      {1119, 1120},
      // Coefficients 4 taps
      {16,  224, 16,  0,   -5,  228, 44,  -11, -18, 212, 79,  -17, -25, 187, 117, -23,
       -26, 154, 154, -26, -23, 117, 187, -25, -17, 79,  212, -18, -11, 44,  228, -5},
      // Coefficients 6 taps
      {-13, 20,  242, 20,  -13, 0,  -4, -6,  230, 50,  -22, 8, 3,  -23, 212, 84,  -30, 10, 8, -34, 187, 120, -36, 11,
       11,  -37, 154, 154, -37, 11, 11, -36, 120, 187, -34, 8, 10, -30, 84,  212, -23, 3,  8, -22, 50,  230, -6,  -4},
    },

    // Band_8_1121
    {
      // Scaler inc range
      {1120, 1121},
      // Coefficients 4 taps
      {16,  224, 16,  0,   -4,  226, 45,  -11, -18, 212, 79,  -17, -25, 187, 117, -23,
       -26, 154, 154, -26, -23, 117, 187, -25, -17, 79,  212, -18, -11, 45,  226, -4},
      // Coefficients 6 taps
      {-13, 20,  242, 20,  -13, 0,  -4, -5,  229, 50,  -22, 8, 3,  -23, 211, 85,  -30, 10, 8, -34, 187, 120, -36, 11,
       11,  -37, 154, 154, -37, 11, 11, -36, 120, 187, -34, 8, 10, -30, 85,  211, -23, 3,  8, -22, 50,  229, -5,  -4},
    },

    // Band_8_1123
    {
      // Scaler inc range
      {1121, 1123},
      // Coefficients 4 taps
      {16,  224, 16,  0,   -4,  226, 45,  -11, -18, 213, 79,  -18, -25, 187, 117, -23,
       -26, 154, 154, -26, -23, 117, 187, -25, -18, 79,  213, -18, -11, 45,  226, -4},
      // Coefficients 6 taps
      {-13, 20,  242, 20,  -13, 0,  -4, -5,  228, 51,  -22, 8, 3,  -23, 211, 85,  -30, 10, 8, -33, 186, 120, -36, 11,
       11,  -37, 154, 154, -37, 11, 11, -36, 120, 186, -33, 8, 10, -30, 85,  211, -23, 3,  8, -22, 51,  228, -5,  -4},
    },

    // Band_8_1124
    {
      // Scaler inc range
      {1123, 1124},
      // Coefficients 4 taps
      {16,  224, 16,  0,   -4,  226, 45,  -11, -18, 213, 79,  -18, -25, 187, 117, -23,
       -26, 154, 154, -26, -23, 117, 187, -25, -18, 79,  213, -18, -11, 45,  226, -4},
      // Coefficients 6 taps
      {-13, 21,  240, 21,  -13, 0,  -4, -5,  228, 51,  -22, 8, 3,  -23, 211, 85,  -30, 10, 8, -33, 186, 120, -36, 11,
       11,  -37, 154, 154, -37, 11, 11, -36, 120, 186, -33, 8, 10, -30, 85,  211, -23, 3,  8, -22, 51,  228, -5,  -4},
    },

    // Band_8_1125
    {
      // Scaler inc range
      {1124, 1125},
      // Coefficients 4 taps
      {17,  222, 17,  0,   -4,  226, 45,  -11, -18, 213, 79,  -18, -25, 187, 117, -23,
       -26, 154, 154, -26, -23, 117, 187, -25, -18, 79,  213, -18, -11, 45,  226, -4},
      // Coefficients 6 taps
      {-14, 21,  242, 21,  -14, 0,  -4, -5,  228, 51,  -22, 8, 3,  -23, 211, 85,  -30, 10, 8, -33, 186, 120, -36, 11,
       11,  -37, 154, 154, -37, 11, 11, -36, 120, 186, -33, 8, 10, -30, 85,  211, -23, 3,  8, -22, 51,  228, -5,  -4},
    },

    // Band_8_1126
    {
      // Scaler inc range
      {1125, 1126},
      // Coefficients 4 taps
      {17,  222, 17,  0,   -4,  226, 45,  -11, -18, 213, 79,  -18, -25, 187, 117, -23,
       -26, 154, 154, -26, -23, 117, 187, -25, -18, 79,  213, -18, -11, 45,  226, -4},
      // Coefficients 6 taps
      {-14, 21,  242, 21,  -14, 0,  -5, -4,  228, 51,  -22, 8, 3,  -22, 210, 85,  -30, 10, 8, -33, 186, 120, -36, 11,
       11,  -37, 154, 154, -37, 11, 11, -36, 120, 186, -33, 8, 10, -30, 85,  210, -22, 3,  8, -22, 51,  228, -4,  -5},
    },

    // Band_8_1128
    {
      // Scaler inc range
      {1126, 1128},
      // Coefficients 4 taps
      {17,  222, 17,  0,   -3,  225, 45,  -11, -17, 212, 79,  -18, -25, 187, 117, -23,
       -26, 154, 154, -26, -23, 117, 187, -25, -18, 79,  212, -17, -11, 45,  225, -3},
      // Coefficients 6 taps
      {-14, 21,  242, 21,  -14, 0,  -5, -4,  229, 51,  -23, 8, 3,  -22, 210, 85,  -30, 10, 8, -33, 186, 120, -36, 11,
       10,  -37, 155, 155, -37, 10, 11, -36, 120, 186, -33, 8, 10, -30, 85,  210, -22, 3,  8, -23, 51,  229, -4,  -5},
    },

    // Band_8_1129
    {
      // Scaler inc range
      {1128, 1129},
      // Coefficients 4 taps
      {17,  222, 17,  0,   -3,  225, 45,  -11, -17, 211, 80,  -18, -25, 187, 117, -23,
       -26, 154, 154, -26, -23, 117, 187, -25, -18, 80,  211, -17, -11, 45,  225, -3},
      // Coefficients 6 taps
      {-14, 22,  240, 22,  -14, 0,  -5, -4,  228, 52,  -23, 8, 2,  -22, 212, 85,  -31, 10, 8, -33, 186, 120, -36, 11,
       10,  -37, 155, 155, -37, 10, 11, -36, 120, 186, -33, 8, 10, -31, 85,  212, -22, 2,  8, -23, 52,  228, -4,  -5},
    },

    // Band_8_1130
    {
      // Scaler inc range
      {1129, 1130},
      // Coefficients 4 taps
      {17,  222, 17,  0,   -3,  224, 46,  -11, -17, 211, 80,  -18, -25, 187, 117, -23,
       -26, 154, 154, -26, -23, 117, 187, -25, -18, 80,  211, -17, -11, 46,  224, -3},
      // Coefficients 6 taps
      {-14, 22,  240, 22,  -14, 0,  -5, -4,  228, 52,  -23, 8, 2,  -22, 211, 86,  -31, 10, 7, -33, 186, 121, -36, 11,
       10,  -37, 155, 155, -37, 10, 11, -36, 121, 186, -33, 7, 10, -31, 86,  211, -22, 2,  8, -23, 52,  228, -4,  -5},
    },

    // Band_8_1131
    {
      // Scaler inc range
      {1130, 1131},
      // Coefficients 4 taps
      {17,  222, 17,  0,   -3,  225, 46,  -12, -17, 211, 80,  -18, -24, 186, 117, -23,
       -26, 154, 154, -26, -23, 117, 186, -24, -18, 80,  211, -17, -12, 46,  225, -3},
      // Coefficients 6 taps
      {-14, 22,  240, 22,  -14, 0,  -5, -4,  228, 52,  -23, 8, 2,  -22, 211, 86,  -31, 10, 7, -32, 185, 121, -36, 11,
       10,  -37, 155, 155, -37, 10, 11, -36, 121, 185, -32, 7, 10, -31, 86,  211, -22, 2,  8, -23, 52,  228, -4,  -5},
    },

    // Band_8_1132
    {
      // Scaler inc range
      {1131, 1132},
      // Coefficients 4 taps
      {18,  220, 18,  0,   -3,  225, 46,  -12, -17, 211, 80,  -18, -24, 186, 117, -23,
       -26, 154, 154, -26, -23, 117, 186, -24, -18, 80,  211, -17, -12, 46,  225, -3},
      // Coefficients 6 taps
      {-14, 22,  240, 22,  -14, 0,  -5, -3,  227, 52,  -23, 8, 2,  -21, 210, 86,  -31, 10, 7, -32, 185, 121, -36, 11,
       10,  -37, 155, 155, -37, 10, 11, -36, 121, 185, -32, 7, 10, -31, 86,  210, -21, 2,  8, -23, 52,  227, -3,  -5},
    },

    // Band_8_1134
    {
      // Scaler inc range
      {1132, 1134},
      // Coefficients 4 taps
      {18,  220, 18,  0,   -3,  225, 46,  -12, -17, 211, 80,  -18, -24, 186, 117, -23,
       -26, 154, 154, -26, -23, 117, 186, -24, -18, 80,  211, -17, -12, 46,  225, -3},
      // Coefficients 6 taps
      {-15, 22,  242, 22,  -15, 0,  -5, -3,  227, 52,  -23, 8, 2,  -21, 210, 86,  -31, 10, 7, -32, 185, 121, -36, 11,
       10,  -37, 155, 155, -37, 10, 11, -36, 121, 185, -32, 7, 10, -31, 86,  210, -21, 2,  8, -23, 52,  227, -3,  -5},
    },

    // Band_8_1135
    {
      // Scaler inc range
      {1134, 1135},
      // Coefficients 4 taps
      {18,  220, 18,  0,   -2,  224, 46,  -12, -17, 211, 80,  -18, -24, 186, 117, -23,
       -26, 154, 154, -26, -23, 117, 186, -24, -18, 80,  211, -17, -12, 46,  224, -2},
      // Coefficients 6 taps
      {-15, 23,  240, 23,  -15, 0,  -6, -3,  228, 52,  -23, 8, 2,  -21, 210, 86,  -31, 10, 7, -32, 185, 121, -36, 11,
       10,  -37, 155, 155, -37, 10, 11, -36, 121, 185, -32, 7, 10, -31, 86,  210, -21, 2,  8, -23, 52,  228, -3,  -6},
    },

    // Band_8_1136
    {
      // Scaler inc range
      {1135, 1136},
      // Coefficients 4 taps
      {18,  220, 18,  0,   -2,  224, 46,  -12, -16, 210, 80,  -18, -24, 186, 117, -23,
       -26, 154, 154, -26, -23, 117, 186, -24, -18, 80,  210, -16, -12, 46,  224, -2},
      // Coefficients 6 taps
      {-15, 23,  240, 23,  -15, 0,  -6, -3,  227, 53,  -23, 8, 2,  -21, 210, 86,  -31, 10, 7, -32, 185, 121, -36, 11,
       10,  -37, 155, 155, -37, 10, 11, -36, 121, 185, -32, 7, 10, -31, 86,  210, -21, 2,  8, -23, 53,  227, -3,  -6},
    },

    // Band_8_1137
    {
      // Scaler inc range
      {1136, 1137},
      // Coefficients 4 taps
      {18,  220, 18,  0,   -2,  223, 47,  -12, -16, 210, 80,  -18, -24, 186, 117, -23,
       -26, 154, 154, -26, -23, 117, 186, -24, -18, 80,  210, -16, -12, 47,  223, -2},
      // Coefficients 6 taps
      {-15, 23,  240, 23,  -15, 0,  -6, -3,  227, 53,  -23, 8, 2,  -21, 210, 86,  -31, 10, 7, -32, 185, 121, -36, 11,
       10,  -36, 154, 154, -36, 10, 11, -36, 121, 185, -32, 7, 10, -31, 86,  210, -21, 2,  8, -23, 53,  227, -3,  -6},
    },

    // Band_8_1139
    {
      // Scaler inc range
      {1137, 1139},
      // Coefficients 4 taps
      {18,  220, 18,  0,   -2,  223, 47,  -12, -16, 210, 80,  -18, -24, 186, 117, -23,
       -26, 154, 154, -26, -23, 117, 186, -24, -18, 80,  210, -16, -12, 47,  223, -2},
      // Coefficients 6 taps
      {-15, 23,  240, 23,  -15, 0,  -6, -2,  226, 53,  -23, 8, 1,  -21, 211, 86,  -31, 10, 7, -32, 185, 121, -36, 11,
       10,  -36, 154, 154, -36, 10, 11, -36, 121, 185, -32, 7, 10, -31, 86,  211, -21, 1,  8, -23, 53,  226, -2,  -6},
    },

    // Band_8_1140
    {
      // Scaler inc range
      {1139, 1140},
      // Coefficients 4 taps
      {19,  218, 19,  0,   -2,  223, 47,  -12, -16, 210, 80,  -18, -24, 186, 117, -23,
       -25, 153, 153, -25, -23, 117, 186, -24, -18, 80,  210, -16, -12, 47,  223, -2},
      // Coefficients 6 taps
      {-15, 24,  238, 24,  -15, 0,  -6, -2,  227, 53,  -24, 8, 1,  -20, 209, 87,  -31, 10, 7, -32, 185, 121, -36, 11,
       10,  -36, 154, 154, -36, 10, 11, -36, 121, 185, -32, 7, 10, -31, 87,  209, -20, 1,  8, -24, 53,  227, -2,  -6},
    },

    // Band_8_1141
    {
      // Scaler inc range
      {1140, 1141},
      // Coefficients 4 taps
      {19,  218, 19,  0,   -2,  223, 47,  -12, -16, 209, 81,  -18, -24, 186, 117, -23,
       -25, 153, 153, -25, -23, 117, 186, -24, -18, 81,  209, -16, -12, 47,  223, -2},
      // Coefficients 6 taps
      {-15, 24,  238, 24,  -15, 0,  -6, -2,  227, 53,  -24, 8, 1,  -20, 209, 87,  -31, 10, 6, -31, 185, 121, -36, 11,
       10,  -36, 154, 154, -36, 10, 11, -36, 121, 185, -31, 6, 10, -31, 87,  209, -20, 1,  8, -24, 53,  227, -2,  -6},
    },

    // Band_8_1142
    {
      // Scaler inc range
      {1141, 1142},
      // Coefficients 4 taps
      {19,  218, 19,  0,   -1,  222, 47,  -12, -16, 209, 81,  -18, -23, 185, 117, -23,
       -25, 153, 153, -25, -23, 117, 185, -23, -18, 81,  209, -16, -12, 47,  222, -1},
      // Coefficients 6 taps
      {-15, 24,  238, 24,  -15, 0,  -6, -2,  226, 54,  -24, 8, 1,  -20, 209, 87,  -31, 10, 6, -31, 185, 121, -36, 11,
       10,  -36, 154, 154, -36, 10, 11, -36, 121, 185, -31, 6, 10, -31, 87,  209, -20, 1,  8, -24, 54,  226, -2,  -6},
    },

    // Band_8_1143
    {
      // Scaler inc range
      {1142, 1143},
      // Coefficients 4 taps
      {19,  218, 19,  0,   -1,  222, 47,  -12, -15, 208, 81,  -18, -23, 185, 117, -23,
       -25, 153, 153, -25, -23, 117, 185, -23, -18, 81,  208, -15, -12, 47,  222, -1},
      // Coefficients 6 taps
      {-16, 24,  240, 24,  -16, 0,  -6, -1,  224, 54,  -24, 9, 1,  -20, 209, 87,  -31, 10, 6, -31, 185, 121, -36, 11,
       10,  -36, 154, 154, -36, 10, 11, -36, 121, 185, -31, 6, 10, -31, 87,  209, -20, 1,  9, -24, 54,  224, -1,  -6},
    },

    // Band_8_1145
    {
      // Scaler inc range
      {1143, 1145},
      // Coefficients 4 taps
      {19,  218, 19,  0,   -1,  222, 47,  -12, -15, 208, 81,  -18, -23, 185, 117, -23,
       -25, 153, 153, -25, -23, 117, 185, -23, -18, 81,  208, -15, -12, 47,  222, -1},
      // Coefficients 6 taps
      {-16, 25,  238, 25,  -16, 0, -7, -1,  225, 54,  -24, 9, 1,  -20, 209, 87,  -31, 10, 6, -31, 185, 121, -36, 11,
       9,   -36, 155, 155, -36, 9, 11, -36, 121, 185, -31, 6, 10, -31, 87,  209, -20, 1,  9, -24, 54,  225, -1,  -7},
    },

    // Band_8_1146
    {
      // Scaler inc range
      {1145, 1146},
      // Coefficients 4 taps
      {19,  218, 19,  0,   -1,  221, 48,  -12, -15, 208, 81,  -18, -23, 185, 117, -23,
       -25, 153, 153, -25, -23, 117, 185, -23, -18, 81,  208, -15, -12, 48,  221, -1},
      // Coefficients 6 taps
      {-16, 25,  238, 25,  -16, 0, -7, -1,  225, 54,  -24, 9, 1,  -19, 208, 87,  -31, 10, 6, -31, 185, 121, -36, 11,
       9,   -36, 155, 155, -36, 9, 11, -36, 121, 185, -31, 6, 10, -31, 87,  208, -19, 1,  9, -24, 54,  225, -1,  -7},
    },

    // Band_8_1147
    {
      // Scaler inc range
      {1146, 1147},
      // Coefficients 4 taps
      {19,  218, 19,  0,   -1,  221, 48,  -12, -15, 208, 81,  -18, -23, 185, 117, -23,
       -25, 153, 153, -25, -23, 117, 185, -23, -18, 81,  208, -15, -12, 48,  221, -1},
      // Coefficients 6 taps
      {-16, 25,  238, 25,  -16, 0, -7, -1,  225, 54,  -24, 9, 0,  -19, 209, 87,  -31, 10, 6, -31, 184, 122, -36, 11,
       9,   -36, 155, 155, -36, 9, 11, -36, 122, 184, -31, 6, 10, -31, 87,  209, -19, 0,  9, -24, 54,  225, -1,  -7},
    },

    // Band_8_1148
    {
      // Scaler inc range
      {1147, 1148},
      // Coefficients 4 taps
      {20,  216, 20,  0,   0,   220, 48,  -12, -15, 208, 81,  -18, -23, 185, 117, -23,
       -25, 153, 153, -25, -23, 117, 185, -23, -18, 81,  208, -15, -12, 48,  220, 0},
      // Coefficients 6 taps
      {-16, 25,  238, 25,  -16, 0, -7, -1,  224, 55,  -24, 9, 0,  -19, 208, 88,  -31, 10, 6, -31, 184, 122, -36, 11,
       9,   -36, 155, 155, -36, 9, 11, -36, 122, 184, -31, 6, 10, -31, 88,  208, -19, 0,  9, -24, 55,  224, -1,  -7},
    },

    // Band_8_1150
    {
      // Scaler inc range
      {1148, 1150},
      // Coefficients 4 taps
      {20,  216, 20,  0,   0,   220, 48,  -12, -15, 208, 81,  -18, -23, 185, 117, -23,
       -25, 153, 153, -25, -23, 117, 185, -23, -18, 81,  208, -15, -12, 48,  220, 0},
      // Coefficients 6 taps
      {-16, 25,  238, 25,  -16, 0, -7, 0,   223, 55,  -24, 9, 0,  -19, 208, 88,  -31, 10, 6, -30, 184, 122, -36, 10,
       9,   -36, 155, 155, -36, 9, 10, -36, 122, 184, -30, 6, 10, -31, 88,  208, -19, 0,  9, -24, 55,  223, 0,   -7},
    },

    // Band_8_1151
    {
      // Scaler inc range
      {1150, 1151},
      // Coefficients 4 taps
      {20,  216, 20,  0,   0,   220, 48,  -12, -15, 208, 81,  -18, -23, 185, 117, -23,
       -25, 153, 153, -25, -23, 117, 185, -23, -18, 81,  208, -15, -12, 48,  220, 0},
      // Coefficients 6 taps
      {-16, 26,  236, 26,  -16, 0, -7, 0,   223, 55,  -24, 9, 0,  -19, 208, 88,  -31, 10, 6, -30, 184, 122, -36, 10,
       9,   -36, 155, 155, -36, 9, 10, -36, 122, 184, -30, 6, 10, -31, 88,  208, -19, 0,  9, -24, 55,  223, 0,   -7},
    },

    // Band_8_1152
    {
      // Scaler inc range
      {1151, 1152},
      // Coefficients 4 taps
      {20,  216, 20,  0,   0,   220, 48,  -12, -14, 207, 81,  -18, -22, 184, 117, -23,
       -25, 153, 153, -25, -23, 117, 184, -22, -18, 81,  207, -14, -12, 48,  220, 0},
      // Coefficients 6 taps
      {-17, 26,  238, 26,  -17, 0, -7, 0,   223, 55,  -24, 9, 0,  -18, 207, 88,  -31, 10, 6, -30, 183, 122, -35, 10,
       9,   -35, 154, 154, -35, 9, 10, -35, 122, 183, -30, 6, 10, -31, 88,  207, -18, 0,  9, -24, 55,  223, 0,   -7},
    },

    // Band_8_1154
    {
      // Scaler inc range
      {1152, 1154},
      // Coefficients 4 taps
      {20,  216, 20,  0,   0,   220, 48,  -12, -14, 207, 81,  -18, -22, 184, 117, -23,
       -25, 153, 153, -25, -23, 117, 184, -22, -18, 81,  207, -14, -12, 48,  220, 0},
      // Coefficients 6 taps
      {-17, 26,  238, 26,  -17, 0, -8, 0,   225, 55,  -25, 9, 0,  -18, 207, 88,  -31, 10, 5, -30, 184, 122, -35, 10,
       9,   -35, 154, 154, -35, 9, 10, -35, 122, 184, -30, 5, 10, -31, 88,  207, -18, 0,  9, -25, 55,  225, 0,   -8},
    },

    // Band_8_1155
    {
      // Scaler inc range
      {1154, 1155},
      // Coefficients 4 taps
      {20,  216, 20,  0,   0,   219, 49,  -12, -14, 206, 82,  -18, -22, 184, 117, -23,
       -25, 153, 153, -25, -23, 117, 184, -22, -18, 82,  206, -14, -12, 49,  219, 0},
      // Coefficients 6 taps
      {-17, 26,  238, 26,  -17, 0, -8, 0,   224, 56,  -25, 9, 0,  -18, 207, 88,  -31, 10, 5, -30, 184, 122, -35, 10,
       9,   -35, 154, 154, -35, 9, 10, -35, 122, 184, -30, 5, 10, -31, 88,  207, -18, 0,  9, -25, 56,  224, 0,   -8},
    },

    // Band_8_1156
    {
      // Scaler inc range
      {1155, 1156},
      // Coefficients 4 taps
      {21,  214, 21,  0,   1,   218, 49,  -12, -14, 206, 82,  -18, -22, 184, 117, -23,
       -25, 153, 153, -25, -23, 117, 184, -22, -18, 82,  206, -14, -12, 49,  218, 1},
      // Coefficients 6 taps
      {-17, 27,  236, 27,  -17, 0, -8, 1,   223, 56,  -25, 9, 0,  -18, 207, 88,  -31, 10, 5, -30, 184, 122, -35, 10,
       9,   -35, 154, 154, -35, 9, 10, -35, 122, 184, -30, 5, 10, -31, 88,  207, -18, 0,  9, -25, 56,  223, 1,   -8},
    },

    // Band_8_1157
    {
      // Scaler inc range
      {1156, 1157},
      // Coefficients 4 taps
      {21,  214, 21,  0,   1,   218, 49,  -12, -14, 206, 82,  -18, -22, 184, 117, -23,
       -25, 153, 153, -25, -23, 117, 184, -22, -18, 82,  206, -14, -12, 49,  218, 1},
      // Coefficients 6 taps
      {-17, 27,  236, 27,  -17, 0, -8, 1,   223, 56,  -25, 9, -1, -18, 208, 89,  -32, 10, 5, -30, 184, 122, -35, 10,
       9,   -35, 154, 154, -35, 9, 10, -35, 122, 184, -30, 5, 10, -32, 89,  208, -18, -1, 9, -25, 56,  223, 1,   -8},
    },

    // Band_8_1159
    {
      // Scaler inc range
      {1157, 1159},
      // Coefficients 4 taps
      {21,  214, 21,  0,   1,   218, 49,  -12, -14, 206, 82,  -18, -22, 184, 117, -23,
       -24, 152, 152, -24, -23, 117, 184, -22, -18, 82,  206, -14, -12, 49,  218, 1},
      // Coefficients 6 taps
      {-17, 27,  236, 27,  -17, 0, -8, 1,   223, 56,  -25, 9, -1, -17, 207, 89,  -32, 10, 5, -29, 183, 122, -35, 10,
       9,   -35, 154, 154, -35, 9, 10, -35, 122, 183, -29, 5, 10, -32, 89,  207, -17, -1, 9, -25, 56,  223, 1,   -8},
    },

    // Band_8_1160
    {
      // Scaler inc range
      {1159, 1160},
      // Coefficients 4 taps
      {21,  214, 21,  0,   1,   218, 49,  -12, -13, 205, 82,  -18, -22, 184, 117, -23,
       -24, 152, 152, -24, -23, 117, 184, -22, -18, 82,  205, -13, -12, 49,  218, 1},
      // Coefficients 6 taps
      {-17, 27,  236, 27,  -17, 0, -8, 1,   223, 56,  -25, 9, -1, -17, 207, 89,  -32, 10, 5, -29, 183, 122, -35, 10,
       8,   -35, 155, 155, -35, 8, 10, -35, 122, 183, -29, 5, 10, -32, 89,  207, -17, -1, 9, -25, 56,  223, 1,   -8},
    },

    // Band_8_1161
    {
      // Scaler inc range
      {1160, 1161},
      // Coefficients 4 taps
      {21,  214, 21,  0,   1,   218, 49,  -12, -13, 205, 82,  -18, -22, 184, 117, -23,
       -24, 152, 152, -24, -23, 117, 184, -22, -18, 82,  205, -13, -12, 49,  218, 1},
      // Coefficients 6 taps
      {-17, 27,  236, 27,  -17, 0, -8, 2,   222, 56,  -25, 9, -1, -17, 207, 89,  -32, 10, 5, -29, 183, 122, -35, 10,
       8,   -35, 155, 155, -35, 8, 10, -35, 122, 183, -29, 5, 10, -32, 89,  207, -17, -1, 9, -25, 56,  222, 2,   -8},
    },

    // Band_8_1163
    {
      // Scaler inc range
      {1161, 1163},
      // Coefficients 4 taps
      {21,  214, 21,  0,   1,   218, 49,  -12, -13, 205, 82,  -18, -22, 184, 117, -23,
       -24, 152, 152, -24, -23, 117, 184, -22, -18, 82,  205, -13, -12, 49,  218, 1},
      // Coefficients 6 taps
      {-17, 28,  234, 28,  -17, 0, -9, 2,   222, 57,  -25, 9, -1, -17, 207, 89,  -32, 10, 5, -29, 183, 122, -35, 10,
       8,   -35, 155, 155, -35, 8, 10, -35, 122, 183, -29, 5, 10, -32, 89,  207, -17, -1, 9, -25, 57,  222, 2,   -9},
    },

    // Band_8_1164
    {
      // Scaler inc range
      {1163, 1164},
      // Coefficients 4 taps
      {22,  212, 22,  0,   2,   217, 50,  -13, -13, 205, 82,  -18, -21, 182, 117, -22,
       -24, 152, 152, -24, -22, 117, 182, -21, -18, 82,  205, -13, -13, 50,  217, 2},
      // Coefficients 6 taps
      {-18, 28,  236, 28,  -18, 0, -9, 2,   222, 57,  -25, 9, -1, -17, 207, 89,  -32, 10, 5, -29, 183, 122, -35, 10,
       8,   -35, 155, 155, -35, 8, 10, -35, 122, 183, -29, 5, 10, -32, 89,  207, -17, -1, 9, -25, 57,  222, 2,   -9},
    },

    // Band_8_1165
    {
      // Scaler inc range
      {1164, 1165},
      // Coefficients 4 taps
      {22,  212, 22,  0,   2,   217, 50,  -13, -13, 205, 82,  -18, -21, 182, 117, -22,
       -24, 152, 152, -24, -22, 117, 182, -21, -18, 82,  205, -13, -13, 50,  217, 2},
      // Coefficients 6 taps
      {-18, 28,  236, 28,  -18, 0, -9, 2,   222, 57,  -25, 9, -1, -17, 207, 89,  -32, 10, 4, -29, 183, 123, -35, 10,
       8,   -35, 155, 155, -35, 8, 10, -35, 123, 183, -29, 4, 10, -32, 89,  207, -17, -1, 9, -25, 57,  222, 2,   -9},
    },

    // Band_8_1166
    {
      // Scaler inc range
      {1165, 1166},
      // Coefficients 4 taps
      {22,  212, 22,  0,   2,   217, 50,  -13, -13, 205, 82,  -18, -21, 182, 117, -22,
       -24, 152, 152, -24, -22, 117, 182, -21, -18, 82,  205, -13, -13, 50,  217, 2},
      // Coefficients 6 taps
      {-18, 28,  236, 28,  -18, 0, -9, 2,   222, 57,  -25, 9, -1, -16, 206, 89,  -32, 10, 4, -28, 182, 123, -35, 10,
       8,   -34, 154, 154, -34, 8, 10, -35, 123, 182, -28, 4, 10, -32, 89,  206, -16, -1, 9, -25, 57,  222, 2,   -9},
    },

    // Band_8_1168
    {
      // Scaler inc range
      {1166, 1168},
      // Coefficients 4 taps
      {22,  212, 22,  0,   2,   217, 50,  -13, -13, 205, 82,  -18, -21, 181, 118, -22,
       -24, 152, 152, -24, -22, 118, 181, -21, -18, 82,  205, -13, -13, 50,  217, 2},
      // Coefficients 6 taps
      {-18, 29,  234, 29,  -18, 0, -9, 3,   221, 57,  -25, 9, -2, -16, 206, 90,  -32, 10, 4, -28, 182, 123, -35, 10,
       8,   -34, 154, 154, -34, 8, 10, -35, 123, 182, -28, 4, 10, -32, 90,  206, -16, -2, 9, -25, 57,  221, 3,   -9},
    },

    // Band_8_1169
    {
      // Scaler inc range
      {1168, 1169},
      // Coefficients 4 taps
      {22,  212, 22,  0,   2,   217, 50,  -13, -12, 204, 82,  -18, -21, 181, 118, -22,
       -24, 152, 152, -24, -22, 118, 181, -21, -18, 82,  204, -12, -13, 50,  217, 2},
      // Coefficients 6 taps
      {-18, 29,  234, 29,  -18, 0, -9, 3,   220, 58,  -25, 9, -2, -16, 206, 90,  -32, 10, 4, -28, 182, 123, -35, 10,
       8,   -34, 154, 154, -34, 8, 10, -35, 123, 182, -28, 4, 10, -32, 90,  206, -16, -2, 9, -25, 58,  220, 3,   -9},
    },

    // Band_8_1170
    {
      // Scaler inc range
      {1169, 1170},
      // Coefficients 4 taps
      {22,  212, 22,  0,   2,   217, 50,  -13, -12, 203, 83,  -18, -21, 181, 118, -22,
       -24, 152, 152, -24, -22, 118, 181, -21, -18, 83,  203, -12, -13, 50,  217, 2},
      // Coefficients 6 taps
      {-18, 29,  234, 29,  -18, 0, -9, 3,   221, 58,  -26, 9, -2, -16, 206, 90,  -32, 10, 4, -28, 182, 123, -35, 10,
       8,   -34, 154, 154, -34, 8, 10, -35, 123, 182, -28, 4, 10, -32, 90,  206, -16, -2, 9, -26, 58,  221, 3,   -9},
    },

    // Band_8_1172
    {
      // Scaler inc range
      {1170, 1172},
      // Coefficients 4 taps
      {23,  210, 23,  0,   3,   216, 50,  -13, -12, 203, 83,  -18, -21, 181, 118, -22,
       -24, 152, 152, -24, -22, 118, 181, -21, -18, 83,  203, -12, -13, 50,  216, 3},
      // Coefficients 6 taps
      {-18, 29,  234, 29,  -18, 0, -9, 3,   221, 58,  -26, 9, -2, -16, 206, 90,  -32, 10, 4, -28, 182, 123, -35, 10,
       8,   -34, 154, 154, -34, 8, 10, -35, 123, 182, -28, 4, 10, -32, 90,  206, -16, -2, 9, -26, 58,  221, 3,   -9},
    },

    // Band_8_1173
    {
      // Scaler inc range
      {1172, 1173},
      // Coefficients 4 taps
      {23,  210, 23,  0,   3,   216, 50,  -13, -12, 203, 83,  -18, -21, 181, 118, -22,
       -24, 152, 152, -24, -22, 118, 181, -21, -18, 83,  203, -12, -13, 50,  216, 3},
      // Coefficients 6 taps
      {-18, 29,  234, 29,  -18, 0, -10, 4,   221, 58,  -26, 9, -2, -15, 205, 90,  -32, 10, 4, -28, 182, 123, -35, 10,
       8,   -34, 154, 154, -34, 8, 10,  -35, 123, 182, -28, 4, 10, -32, 90,  205, -15, -2, 9, -26, 58,  221, 4,   -10},
    },

    // Band_8_1174
    {
      // Scaler inc range
      {1173, 1174},
      // Coefficients 4 taps
      {23,  210, 23,  0,   3,   215, 51,  -13, -12, 203, 83,  -18, -20, 180, 118, -22,
       -24, 152, 152, -24, -22, 118, 180, -20, -18, 83,  203, -12, -13, 51,  215, 3},
      // Coefficients 6 taps
      {-19, 30,  234, 30,  -19, 0, -10, 4,   221, 58,  -26, 9, -2, -15, 205, 90,  -32, 10, 4, -28, 182, 123, -35, 10,
       8,   -34, 154, 154, -34, 8, 10,  -35, 123, 182, -28, 4, 10, -32, 90,  205, -15, -2, 9, -26, 58,  221, 4,   -10},
    },

    // Band_8_1176
    {
      // Scaler inc range
      {1174, 1176},
      // Coefficients 4 taps
      {23,  210, 23,  0,   3,   215, 51,  -13, -12, 203, 83,  -18, -20, 180, 118, -22,
       -23, 151, 151, -23, -22, 118, 180, -20, -18, 83,  203, -12, -13, 51,  215, 3},
      // Coefficients 6 taps
      {-19, 30,  234, 30,  -19, 0, -10, 4,   221, 58,  -26, 9, -2, -15, 205, 90,  -32, 10, 3, -27, 183, 123, -35, 9,
       7,   -34, 155, 155, -34, 7, 9,   -35, 123, 183, -27, 3, 10, -32, 90,  205, -15, -2, 9, -26, 58,  221, 4,   -10},
    },

    // Band_8_1177
    {
      // Scaler inc range
      {1176, 1177},
      // Coefficients 4 taps
      {23,  210, 23,  0,   3,   215, 51,  -13, -11, 202, 83,  -18, -20, 180, 118, -22,
       -23, 151, 151, -23, -22, 118, 180, -20, -18, 83,  202, -11, -13, 51,  215, 3},
      // Coefficients 6 taps
      {-19, 30,  234, 30,  -19, 0, -10, 4,   220, 59,  -26, 9, -2, -15, 205, 90,  -32, 10, 3, -27, 183, 123, -35, 9,
       7,   -34, 155, 155, -34, 7, 9,   -35, 123, 183, -27, 3, 10, -32, 90,  205, -15, -2, 9, -26, 59,  220, 4,   -10},
    },

    // Band_8_1178
    {
      // Scaler inc range
      {1177, 1178},
      // Coefficients 4 taps
      {23,  210, 23,  0,   4,   214, 51,  -13, -11, 202, 83,  -18, -20, 180, 118, -22,
       -23, 151, 151, -23, -22, 118, 180, -20, -18, 83,  202, -11, -13, 51,  214, 4},
      // Coefficients 6 taps
      {-19, 30,  234, 30,  -19, 0, -10, 4,   220, 59,  -26, 9, -3, -15, 205, 91,  -32, 10, 3, -27, 183, 123, -35, 9,
       7,   -34, 155, 155, -34, 7, 9,   -35, 123, 183, -27, 3, 10, -32, 91,  205, -15, -3, 9, -26, 59,  220, 4,   -10},
    },

    // Band_8_1180
    {
      // Scaler inc range
      {1178, 1180},
      // Coefficients 4 taps
      {23,  210, 23,  0,   4,   214, 51,  -13, -11, 202, 83,  -18, -20, 180, 118, -22,
       -23, 151, 151, -23, -22, 118, 180, -20, -18, 83,  202, -11, -13, 51,  214, 4},
      // Coefficients 6 taps
      {-19, 31,  232, 31,  -19, 0, -10, 5,   219, 59,  -26, 9, -3, -14, 204, 91,  -32, 10, 3, -27, 183, 123, -35, 9,
       7,   -33, 154, 154, -33, 7, 9,   -35, 123, 183, -27, 3, 10, -32, 91,  204, -14, -3, 9, -26, 59,  219, 5,   -10},
    },

    // Band_8_1181
    {
      // Scaler inc range
      {1180, 1181},
      // Coefficients 4 taps
      {24,  208, 24,  0,   4,   214, 51,  -13, -11, 202, 83,  -18, -20, 180, 118, -22,
       -23, 151, 151, -23, -22, 118, 180, -20, -18, 83,  202, -11, -13, 51,  214, 4},
      // Coefficients 6 taps
      {-19, 31,  232, 31,  -19, 0, -10, 5,   219, 59,  -26, 9, -3, -14, 204, 91,  -32, 10, 3, -27, 183, 123, -35, 9,
       7,   -33, 154, 154, -33, 7, 9,   -35, 123, 183, -27, 3, 10, -32, 91,  204, -14, -3, 9, -26, 59,  219, 5,   -10},
    },

    // Band_8_1182
    {
      // Scaler inc range
      {1181, 1182},
      // Coefficients 4 taps
      {24,  208, 24,  0,   4,   214, 51,  -13, -11, 202, 83,  -18, -20, 180, 118, -22,
       -23, 151, 151, -23, -22, 118, 180, -20, -18, 83,  202, -11, -13, 51,  214, 4},
      // Coefficients 6 taps
      {-19, 31,  232, 31,  -19, 0, -10, 5,   219, 59,  -26, 9, -3, -14, 204, 91,  -32, 10, 3, -27, 183, 123, -35, 9,
       7,   -33, 154, 154, -33, 7, 9,   -35, 123, 183, -27, 3, 10, -32, 91,  204, -14, -3, 9, -26, 59,  219, 5,   -10},
    },

    // Band_8_1183
    {
      // Scaler inc range
      {1182, 1183},
      // Coefficients 4 taps
      {24,  208, 24,  0,   4,   213, 52,  -13, -11, 202, 83,  -18, -20, 180, 118, -22,
       -23, 151, 151, -23, -22, 118, 180, -20, -18, 83,  202, -11, -13, 52,  213, 4},
      // Coefficients 6 taps
      {-19, 31,  232, 31,  -19, 0, -11, 5,   220, 59,  -26, 9, -3, -14, 204, 91,  -32, 10, 3, -26, 182, 123, -35, 9,
       7,   -33, 154, 154, -33, 7, 9,   -35, 123, 182, -26, 3, 10, -32, 91,  204, -14, -3, 9, -26, 59,  220, 5,   -11},
    },

    // Band_8_1185
    {
      // Scaler inc range
      {1183, 1185},
      // Coefficients 4 taps
      {24,  208, 24,  0,   4,   213, 52,  -13, -10, 201, 83,  -18, -19, 179, 118, -22,
       -23, 151, 151, -23, -22, 118, 179, -19, -18, 83,  201, -10, -13, 52,  213, 4},
      // Coefficients 6 taps
      {-19, 31,  232, 31,  -19, 0, -11, 5,   219, 60,  -26, 9, -3, -14, 204, 91,  -32, 10, 3, -26, 182, 123, -35, 9,
       7,   -33, 154, 154, -33, 7, 9,   -35, 123, 182, -26, 3, 10, -32, 91,  204, -14, -3, 9, -26, 60,  219, 5,   -11},
    },

    // Band_8_1186
    {
      // Scaler inc range
      {1185, 1186},
      // Coefficients 4 taps
      {24,  208, 24,  0,   5,   212, 52,  -13, -10, 200, 84,  -18, -19, 179, 118, -22,
       -23, 151, 151, -23, -22, 118, 179, -19, -18, 84,  200, -10, -13, 52,  212, 5},
      // Coefficients 6 taps
      {-20, 32,  232, 32,  -20, 0, -11, 6,   218, 60,  -26, 9, -3, -13, 203, 91,  -32, 10, 3, -26, 181, 124, -35, 9,
       7,   -33, 154, 154, -33, 7, 9,   -35, 124, 181, -26, 3, 10, -32, 91,  203, -13, -3, 9, -26, 60,  218, 6,   -11},
    },

    // Band_8_1188
    {
      // Scaler inc range
      {1186, 1188},
      // Coefficients 4 taps
      {24,  208, 24,  0,   5,   212, 52,  -13, -10, 200, 84,  -18, -19, 179, 118, -22,
       -23, 151, 151, -23, -22, 118, 179, -19, -18, 84,  200, -10, -13, 52,  212, 5},
      // Coefficients 6 taps
      {-20, 32,  232, 32,  -20, 0, -11, 6,   218, 60,  -26, 9, -4, -13, 203, 92,  -32, 10, 2, -26, 182, 124, -35, 9,
       7,   -33, 154, 154, -33, 7, 9,   -35, 124, 182, -26, 2, 10, -32, 92,  203, -13, -4, 9, -26, 60,  218, 6,   -11},
    },

    // Band_8_1189
    {
      // Scaler inc range
      {1188, 1189},
      // Coefficients 4 taps
      {25,  206, 25,  0,   5,   212, 52,  -13, -10, 200, 84,  -18, -19, 179, 118, -22,
       -23, 151, 151, -23, -22, 118, 179, -19, -18, 84,  200, -10, -13, 52,  212, 5},
      // Coefficients 6 taps
      {-20, 32,  232, 32,  -20, 0, -11, 6,   219, 60,  -27, 9, -4, -13, 204, 92,  -32, 9,  2, -26, 181, 124, -34, 9,
       7,   -33, 154, 154, -33, 7, 9,   -34, 124, 181, -26, 2, 9,  -32, 92,  204, -13, -4, 9, -27, 60,  219, 6,   -11},
    },

    // Band_8_1190
    {
      // Scaler inc range
      {1189, 1190},
      // Coefficients 4 taps
      {25,  206, 25,  0,   5,   212, 52,  -13, -10, 200, 84,  -18, -19, 179, 118, -22,
       -23, 151, 151, -23, -22, 118, 179, -19, -18, 84,  200, -10, -13, 52,  212, 5},
      // Coefficients 6 taps
      {-20, 32,  232, 32,  -20, 0, -11, 6,   219, 60,  -27, 9, -4, -13, 204, 92,  -32, 9,  2, -26, 181, 124, -34, 9,
       6,   -33, 155, 155, -33, 6, 9,   -34, 124, 181, -26, 2, 9,  -32, 92,  204, -13, -4, 9, -27, 60,  219, 6,   -11},
    },

    // Band_8_1192
    {
      // Scaler inc range
      {1190, 1192},
      // Coefficients 4 taps
      {25,  206, 25,  0,   5,   212, 52,  -13, -10, 200, 84,  -18, -19, 179, 118, -22,
       -22, 150, 150, -22, -22, 118, 179, -19, -18, 84,  200, -10, -13, 52,  212, 5},
      // Coefficients 6 taps
      {-20, 32,  232, 32,  -20, 0, -11, 7,   217, 61,  -27, 9, -4, -13, 204, 92,  -32, 9,  2, -25, 180, 124, -34, 9,
       6,   -33, 155, 155, -33, 6, 9,   -34, 124, 180, -25, 2, 9,  -32, 92,  204, -13, -4, 9, -27, 61,  217, 7,   -11},
    },

    // Band_8_1193
    {
      // Scaler inc range
      {1192, 1193},
      // Coefficients 4 taps
      {25,  206, 25,  0,   5,   211, 53,  -13, -10, 200, 84,  -18, -19, 179, 118, -22,
       -22, 150, 150, -22, -22, 118, 179, -19, -18, 84,  200, -10, -13, 53,  211, 5},
      // Coefficients 6 taps
      {-20, 33,  230, 33,  -20, 0, -11, 7,   217, 61,  -27, 9, -4, -12, 203, 92,  -32, 9,  2, -25, 180, 124, -34, 9,
       6,   -32, 154, 154, -32, 6, 9,   -34, 124, 180, -25, 2, 9,  -32, 92,  203, -12, -4, 9, -27, 61,  217, 7,   -11},
    },

    // Band_8_1194
    {
      // Scaler inc range
      {1193, 1194},
      // Coefficients 4 taps
      {25,  206, 25,  0,   6,   210, 53,  -13, -9,  199, 84,  -18, -19, 179, 118, -22,
       -22, 150, 150, -22, -22, 118, 179, -19, -18, 84,  199, -9,  -13, 53,  210, 6},
      // Coefficients 6 taps
      {-20, 33,  230, 33,  -20, 0, -12, 7,   218, 61,  -27, 9, -4, -12, 203, 92,  -32, 9,  2, -25, 180, 124, -34, 9,
       6,   -32, 154, 154, -32, 6, 9,   -34, 124, 180, -25, 2, 9,  -32, 92,  203, -12, -4, 9, -27, 61,  218, 7,   -12},
    },

    // Band_8_1196
    {
      // Scaler inc range
      {1194, 1196},
      // Coefficients 4 taps
      {25,  206, 25,  0,   6,   210, 53,  -13, -9,  199, 84,  -18, -18, 178, 118, -22,
       -22, 150, 150, -22, -22, 118, 178, -18, -18, 84,  199, -9,  -13, 53,  210, 6},
      // Coefficients 6 taps
      {-20, 33,  230, 33,  -20, 0, -12, 7,   218, 61,  -27, 9, -4, -12, 203, 92,  -32, 9,  2, -25, 180, 124, -34, 9,
       6,   -32, 154, 154, -32, 6, 9,   -34, 124, 180, -25, 2, 9,  -32, 92,  203, -12, -4, 9, -27, 61,  218, 7,   -12},
    },

    // Band_8_1197
    {
      // Scaler inc range
      {1196, 1197},
      // Coefficients 4 taps
      {25,  206, 25,  0,   6,   210, 53,  -13, -9,  199, 84,  -18, -18, 178, 118, -22,
       -22, 150, 150, -22, -22, 118, 178, -18, -18, 84,  199, -9,  -13, 53,  210, 6},
      // Coefficients 6 taps
      {-20, 33,  230, 33,  -20, 0, -12, 7,   218, 61,  -27, 9, -4, -12, 203, 92,  -32, 9,  2, -25, 181, 124, -34, 8,
       6,   -32, 154, 154, -32, 6, 8,   -34, 124, 181, -25, 2, 9,  -32, 92,  203, -12, -4, 9, -27, 61,  218, 7,   -12},
    },

    // Band_8_1198
    {
      // Scaler inc range
      {1197, 1198},
      // Coefficients 4 taps
      {26,  204, 26,  0,   6,   210, 53,  -13, -9,  199, 84,  -18, -18, 178, 118, -22,
       -22, 150, 150, -22, -22, 118, 178, -18, -18, 84,  199, -9,  -13, 53,  210, 6},
      // Coefficients 6 taps
      {-20, 34,  228, 34,  -20, 0, -12, 8,   217, 61,  -27, 9, -5, -12, 204, 92,  -32, 9,  2, -25, 181, 124, -34, 8,
       6,   -32, 154, 154, -32, 6, 8,   -34, 124, 181, -25, 2, 9,  -32, 92,  204, -12, -5, 9, -27, 61,  217, 8,   -12},
    },

    // Band_8_1200
    {
      // Scaler inc range
      {1198, 1200},
      // Coefficients 4 taps
      {26,  204, 26,  0,   6,   210, 53,  -13, -9,  199, 84,  -18, -18, 177, 118, -21,
       -22, 150, 150, -22, -21, 118, 177, -18, -18, 84,  199, -9,  -13, 53,  210, 6},
      // Coefficients 6 taps
      {-21, 34,  230, 34,  -21, 0, -12, 8,   216, 62,  -27, 9, -5, -11, 202, 93,  -32, 9,  1, -25, 182, 124, -34, 8,
       6,   -32, 154, 154, -32, 6, 8,   -34, 124, 182, -25, 1, 9,  -32, 93,  202, -11, -5, 9, -27, 62,  216, 8,   -12},
    },

    // Band_8_1201
    {
      // Scaler inc range
      {1200, 1201},
      // Coefficients 4 taps
      {26,  204, 26,  0,   6,   210, 53,  -13, -9,  199, 84,  -18, -18, 177, 118, -21,
       -22, 150, 150, -22, -21, 118, 177, -18, -18, 84,  199, -9,  -13, 53,  210, 6},
      // Coefficients 6 taps
      {-21, 34,  230, 34,  -21, 0, -12, 8,   216, 62,  -27, 9, -5, -11, 202, 93,  -32, 9,  1, -24, 181, 124, -34, 8,
       6,   -32, 154, 154, -32, 6, 8,   -34, 124, 181, -24, 1, 9,  -32, 93,  202, -11, -5, 9, -27, 62,  216, 8,   -12},
    },

    // Band_8_1202
    {
      // Scaler inc range
      {1201, 1202},
      // Coefficients 4 taps
      {26,  204, 26,  0,   7,   209, 53,  -13, -8,  198, 84,  -18, -18, 177, 118, -21,
       -22, 150, 150, -22, -21, 118, 177, -18, -18, 84,  198, -8,  -13, 53,  209, 7},
      // Coefficients 6 taps
      {-21, 34,  230, 34,  -21, 0, -12, 8,   216, 62,  -27, 9, -5, -11, 202, 93,  -32, 9,  1, -24, 181, 124, -34, 8,
       6,   -32, 154, 154, -32, 6, 8,   -34, 124, 181, -24, 1, 9,  -32, 93,  202, -11, -5, 9, -27, 62,  216, 8,   -12},
    },

    // Band_8_1204
    {
      // Scaler inc range
      {1202, 1204},
      // Coefficients 4 taps
      {26,  204, 26,  0,   7,   208, 54,  -13, -8,  198, 84,  -18, -18, 177, 118, -21,
       -22, 150, 150, -22, -21, 118, 177, -18, -18, 84,  198, -8,  -13, 54,  208, 7},
      // Coefficients 6 taps
      {-21, 34,  230, 34,  -21, 0, -12, 9,   215, 62,  -27, 9, -5, -11, 202, 93,  -32, 9,  1, -24, 181, 124, -34, 8,
       5,   -32, 155, 155, -32, 5, 8,   -34, 124, 181, -24, 1, 9,  -32, 93,  202, -11, -5, 9, -27, 62,  215, 9,   -12},
    },

    // Band_8_1205
    {
      // Scaler inc range
      {1204, 1205},
      // Coefficients 4 taps
      {26,  204, 26,  0,   7,   208, 54,  -13, -8,  197, 85,  -18, -17, 176, 118, -21,
       -22, 150, 150, -22, -21, 118, 176, -17, -18, 85,  197, -8,  -13, 54,  208, 7},
      // Coefficients 6 taps
      {-21, 35,  228, 35,  -21, 0, -13, 9,   216, 62,  -27, 9, -5, -11, 202, 93,  -32, 9,  1, -24, 181, 124, -34, 8,
       5,   -31, 154, 154, -31, 5, 8,   -34, 124, 181, -24, 1, 9,  -32, 93,  202, -11, -5, 9, -27, 62,  216, 9,   -13},
    },

    // Band_8_1207
    {
      // Scaler inc range
      {1205, 1207},
      // Coefficients 4 taps
      {27,  202, 27,  0,   7,   208, 54,  -13, -8,  197, 85,  -18, -17, 176, 118, -21,
       -21, 149, 149, -21, -21, 118, 176, -17, -18, 85,  197, -8,  -13, 54,  208, 7},
      // Coefficients 6 taps
      {-21, 35,  228, 35,  -21, 0, -13, 9,   216, 62,  -27, 9, -5, -10, 201, 93,  -32, 9,  1, -24, 181, 124, -34, 8,
       5,   -31, 154, 154, -31, 5, 8,   -34, 124, 181, -24, 1, 9,  -32, 93,  201, -10, -5, 9, -27, 62,  216, 9,   -13},
    },

    // Band_8_1208
    {
      // Scaler inc range
      {1207, 1208},
      // Coefficients 4 taps
      {27,  202, 27,  0,   7,   208, 54,  -13, -8,  197, 85,  -18, -17, 176, 118, -21,
       -21, 149, 149, -21, -21, 118, 176, -17, -18, 85,  197, -8,  -13, 54,  208, 7},
      // Coefficients 6 taps
      {-21, 35,  228, 35,  -21, 0, -13, 9,   215, 63,  -27, 9, -5, -10, 201, 93,  -32, 9,  1, -24, 181, 124, -34, 8,
       5,   -31, 154, 154, -31, 5, 8,   -34, 124, 181, -24, 1, 9,  -32, 93,  201, -10, -5, 9, -27, 63,  215, 9,   -13},
    },

    // Band_8_1209
    {
      // Scaler inc range
      {1208, 1209},
      // Coefficients 4 taps
      {27,  202, 27,  0,   7,   208, 54,  -13, -8,  197, 85,  -18, -17, 176, 118, -21,
       -21, 149, 149, -21, -21, 118, 176, -17, -18, 85,  197, -8,  -13, 54,  208, 7},
      // Coefficients 6 taps
      {-21, 35,  228, 35,  -21, 0, -13, 9,   215, 63,  -27, 9, -6, -10, 202, 93,  -32, 9,  1, -23, 180, 124, -34, 8,
       5,   -31, 154, 154, -31, 5, 8,   -34, 124, 180, -23, 1, 9,  -32, 93,  202, -10, -6, 9, -27, 63,  215, 9,   -13},
    },

    // Band_8_1211
    {
      // Scaler inc range
      {1209, 1211},
      // Coefficients 4 taps
      {27,  202, 27,  0,   8,   207, 54,  -13, -7,  196, 85,  -18, -17, 176, 118, -21,
       -21, 149, 149, -21, -21, 118, 176, -17, -18, 85,  196, -7,  -13, 54,  207, 8},
      // Coefficients 6 taps
      {-21, 35,  228, 35,  -21, 0, -13, 10,  214, 63,  -27, 9, -6, -10, 201, 94,  -32, 9,  0, -23, 180, 125, -34, 8,
       5,   -31, 154, 154, -31, 5, 8,   -34, 125, 180, -23, 0, 9,  -32, 94,  201, -10, -6, 9, -27, 63,  214, 10,  -13},
    },

    // Band_8_1212
    {
      // Scaler inc range
      {1211, 1212},
      // Coefficients 4 taps
      {27,  202, 27,  0,   8,   207, 54,  -13, -7,  196, 85,  -18, -17, 176, 118, -21,
       -21, 149, 149, -21, -21, 118, 176, -17, -18, 85,  196, -7,  -13, 54,  207, 8},
      // Coefficients 6 taps
      {-21, 36,  226, 36,  -21, 0, -13, 10,  214, 63,  -27, 9, -6, -10, 201, 94,  -32, 9,  0, -23, 180, 125, -34, 8,
       5,   -31, 154, 154, -31, 5, 8,   -34, 125, 180, -23, 0, 9,  -32, 94,  201, -10, -6, 9, -27, 63,  214, 10,  -13},
    },

    // Band_8_1214
    {
      // Scaler inc range
      {1212, 1214},
      // Coefficients 4 taps
      {27,  202, 27,  0,   8,   207, 54,  -13, -7,  196, 85,  -18, -17, 176, 118, -21,
       -21, 149, 149, -21, -21, 118, 176, -17, -18, 85,  196, -7,  -13, 54,  207, 8},
      // Coefficients 6 taps
      {-21, 36,  226, 36,  -21, 0, -13, 10,  214, 63,  -27, 9, -6, -9,  200, 94,  -32, 9,  0, -23, 179, 125, -33, 8,
       5,   -31, 154, 154, -31, 5, 8,   -33, 125, 179, -23, 0, 9,  -32, 94,  200, -9,  -6, 9, -27, 63,  214, 10,  -13},
    },

    // Band_8_1215
    {
      // Scaler inc range
      {1214, 1215},
      // Coefficients 4 taps
      {27,  202, 27,  0,   8,   207, 55,  -14, -7,  196, 85,  -18, -17, 176, 118, -21,
       -21, 149, 149, -21, -21, 118, 176, -17, -18, 85,  196, -7,  -14, 55,  207, 8},
      // Coefficients 6 taps
      {-22, 36,  228, 36,  -22, 0, -13, 10,  215, 63,  -28, 9, -6, -9,  200, 94,  -32, 9,  0, -23, 179, 125, -33, 8,
       5,   -31, 154, 154, -31, 5, 8,   -33, 125, 179, -23, 0, 9,  -32, 94,  200, -9,  -6, 9, -28, 63,  215, 10,  -13},
    },

    // Band_8_1216
    {
      // Scaler inc range
      {1215, 1216},
      // Coefficients 4 taps
      {28,  200, 28,  0,   8,   207, 55,  -14, -7,  196, 85,  -18, -16, 175, 118, -21,
       -21, 149, 149, -21, -21, 118, 175, -16, -18, 85,  196, -7,  -14, 55,  207, 8},
      // Coefficients 6 taps
      {-22, 36,  228, 36,  -22, 0, -13, 10,  214, 64,  -28, 9, -6, -9,  200, 94,  -32, 9,  0, -23, 180, 125, -33, 7,
       5,   -30, 153, 153, -30, 5, 7,   -33, 125, 180, -23, 0, 9,  -32, 94,  200, -9,  -6, 9, -28, 64,  214, 10,  -13},
    },

    // Band_8_1218
    {
      // Scaler inc range
      {1216, 1218},
      // Coefficients 4 taps
      {28,  200, 28,  0,   8,   207, 55,  -14, -7,  196, 85,  -18, -16, 176, 117, -21,
       -21, 149, 149, -21, -21, 117, 176, -16, -18, 85,  196, -7,  -14, 55,  207, 8},
      // Coefficients 6 taps
      {-22, 37,  226, 37,  -22, 0, -14, 11,  214, 64,  -28, 9, -6, -9,  200, 94,  -32, 9,  0, -22, 179, 125, -33, 7,
       4,   -30, 154, 154, -30, 4, 7,   -33, 125, 179, -22, 0, 9,  -32, 94,  200, -9,  -6, 9, -28, 64,  214, 11,  -14},
    },

    // Band_8_1219
    {
      // Scaler inc range
      {1218, 1219},
      // Coefficients 4 taps
      {28,  200, 28,  0,   9,   206, 55,  -14, -7,  196, 85,  -18, -16, 176, 117, -21,
       -21, 149, 149, -21, -21, 117, 176, -16, -18, 85,  196, -7,  -14, 55,  206, 9},
      // Coefficients 6 taps
      {-22, 37,  226, 37,  -22, 0, -14, 11,  214, 64,  -28, 9, -6, -9,  200, 94,  -32, 9,  0, -22, 179, 125, -33, 7,
       4,   -30, 154, 154, -30, 4, 7,   -33, 125, 179, -22, 0, 9,  -32, 94,  200, -9,  -6, 9, -28, 64,  214, 11,  -14},
    },

    // Band_8_1221
    {
      // Scaler inc range
      {1219, 1221},
      // Coefficients 4 taps
      {28,  200, 28,  0,   9,   206, 55,  -14, -6,  195, 85,  -18, -16, 176, 117, -21,
       -21, 149, 149, -21, -21, 117, 176, -16, -18, 85,  195, -6,  -14, 55,  206, 9},
      // Coefficients 6 taps
      {-22, 37,  226, 37,  -22, 0, -14, 11,  215, 64,  -28, 8, -7, -8,  200, 94,  -32, 9,  0, -22, 179, 125, -33, 7,
       4,   -30, 154, 154, -30, 4, 7,   -33, 125, 179, -22, 0, 9,  -32, 94,  200, -8,  -7, 8, -28, 64,  215, 11,  -14},
    },

    // Band_8_1222
    {
      // Scaler inc range
      {1221, 1222},
      // Coefficients 4 taps
      {28,  200, 28,  0,   9,   206, 55,  -14, -6,  195, 85,  -18, -16, 176, 117, -21,
       -20, 148, 148, -20, -21, 117, 176, -16, -18, 85,  195, -6,  -14, 55,  206, 9},
      // Coefficients 6 taps
      {-22, 37,  226, 37,  -22, 0, -14, 11,  215, 64,  -28, 8, -7,  -8, 200, 94, -32, 9, -1,  -22, 180, 125, -33, 7, 4,
       -30, 154, 154, -30, 4,   7, -33, 125, 180, -22, -1,  9, -32, 94, 200, -8, -7,  8, -28, 64,  215, 11,  -14},
    },

    // Band_8_1224
    {
      // Scaler inc range
      {1222, 1224},
      // Coefficients 4 taps
      {28,  200, 28,  0,   9,   206, 55,  -14, -6,  195, 85,  -18, -16, 176, 117, -21,
       -20, 148, 148, -20, -21, 117, 176, -16, -18, 85,  195, -6,  -14, 55,  206, 9},
      // Coefficients 6 taps
      {-22, 37,  226, 37,  -22, 0, -14, 12,  214, 64,  -28, 8, -7,  -8, 200, 95, -32, 8, -1,  -22, 180, 125, -33, 7, 4,
       -30, 154, 154, -30, 4,   7, -33, 125, 180, -22, -1,  8, -32, 95, 200, -8, -7,  8, -28, 64,  214, 12,  -14},
    },

    // Band_8_1225
    {
      // Scaler inc range
      {1224, 1225},
      // Coefficients 4 taps
      {28,  200, 28,  0,   9,   206, 55,  -14, -6,  195, 85,  -18, -16, 176, 117, -21,
       -20, 148, 148, -20, -21, 117, 176, -16, -18, 85,  195, -6,  -14, 55,  206, 9},
      // Coefficients 6 taps
      {-22, 38,  224, 38,  -22, 0, -14, 12,  213, 65,  -28, 8, -7,  -8, 200, 95, -32, 8, -1,  -21, 179, 125, -33, 7, 4,
       -30, 154, 154, -30, 4,   7, -33, 125, 179, -21, -1,  8, -32, 95, 200, -8, -7,  8, -28, 65,  213, 12,  -14},
    },

    // Band_8_1226
    {
      // Scaler inc range
      {1225, 1226},
      // Coefficients 4 taps
      {29,  198, 29,  0,   9,   205, 56,  -14, -6,  194, 86,  -18, -16, 176, 117, -21,
       -20, 148, 148, -20, -21, 117, 176, -16, -18, 86,  194, -6,  -14, 56,  205, 9},
      // Coefficients 6 taps
      {-22, 38,  224, 38,  -22, 0, -14, 12,  213, 65,  -28, 8, -7,  -7, 199, 95, -32, 8, -1,  -21, 179, 125, -33, 7, 4,
       -30, 154, 154, -30, 4,   7, -33, 125, 179, -21, -1,  8, -32, 95, 199, -7, -7,  8, -28, 65,  213, 12,  -14},
    },

    // Band_8_1228
    {
      // Scaler inc range
      {1226, 1228},
      // Coefficients 4 taps
      {29,  198, 29,  0,   10,  204, 56,  -14, -6,  194, 86,  -18, -15, 174, 117, -20,
       -20, 148, 148, -20, -20, 117, 174, -15, -18, 86,  194, -6,  -14, 56,  204, 10},
      // Coefficients 6 taps
      {-22, 38,  224, 38,  -22, 0, -14, 12,  213, 65,  -28, 8, -7,  -7, 199, 95, -32, 8, -1,  -21, 179, 125, -33, 7, 4,
       -29, 153, 153, -29, 4,   7, -33, 125, 179, -21, -1,  8, -32, 95, 199, -7, -7,  8, -28, 65,  213, 12,  -14},
    },

    // Band_8_1229
    {
      // Scaler inc range
      {1228, 1229},
      // Coefficients 4 taps
      {29,  198, 29,  0,   10,  204, 56,  -14, -5,  193, 86,  -18, -15, 174, 117, -20,
       -20, 148, 148, -20, -20, 117, 174, -15, -18, 86,  193, -5,  -14, 56,  204, 10},
      // Coefficients 6 taps
      {-22, 38,  224, 38,  -22, 0, -14, 12,  213, 65,  -28, 8, -7,  -7, 199, 95, -32, 8, -1,  -21, 179, 125, -33, 7, 4,
       -29, 153, 153, -29, 4,   7, -33, 125, 179, -21, -1,  8, -32, 95, 199, -7, -7,  8, -28, 65,  213, 12,  -14},
    },

    // Band_8_1231
    {
      // Scaler inc range
      {1229, 1231},
      // Coefficients 4 taps
      {29,  198, 29,  0,   10,  204, 56,  -14, -5,  193, 86,  -18, -15, 174, 117, -20,
       -20, 148, 148, -20, -20, 117, 174, -15, -18, 86,  193, -5,  -14, 56,  204, 10},
      // Coefficients 6 taps
      {-23, 38,  226, 38,  -23, 0, -15, 13,  213, 65,  -28, 8, -7,  -7, 199, 95, -32, 8, -1,  -21, 179, 125, -33, 7, 4,
       -29, 153, 153, -29, 4,   7, -33, 125, 179, -21, -1,  8, -32, 95, 199, -7, -7,  8, -28, 65,  213, 13,  -15},
    },

    // Band_8_1232
    {
      // Scaler inc range
      {1231, 1232},
      // Coefficients 4 taps
      {29,  198, 29,  0,   10,  204, 56,  -14, -5,  193, 86,  -18, -15, 174, 117, -20,
       -20, 148, 148, -20, -20, 117, 174, -15, -18, 86,  193, -5,  -14, 56,  204, 10},
      // Coefficients 6 taps
      {-23, 39,  224, 39,  -23, 0, -15, 13,  213, 65,  -28, 8, -8,  -7, 200, 95, -32, 8, -1,  -21, 179, 125, -33, 7, 3,
       -29, 154, 154, -29, 3,   7, -33, 125, 179, -21, -1,  8, -32, 95, 200, -7, -8,  8, -28, 65,  213, 13,  -15},
    },

    // Band_8_1234
    {
      // Scaler inc range
      {1232, 1234},
      // Coefficients 4 taps
      {29,  198, 29,  0,   10,  204, 56,  -14, -5,  193, 86,  -18, -15, 174, 117, -20,
       -20, 148, 148, -20, -20, 117, 174, -15, -18, 86,  193, -5,  -14, 56,  204, 10},
      // Coefficients 6 taps
      {-23, 39,  224, 39,  -23, 0, -15, 13,  212, 66,  -28, 8, -8,  -6, 199, 95, -32, 8, -1,  -20, 178, 125, -33, 7, 3,
       -29, 154, 154, -29, 3,   7, -33, 125, 178, -20, -1,  8, -32, 95, 199, -6, -8,  8, -28, 66,  212, 13,  -15},
    },

    // Band_8_1235
    {
      // Scaler inc range
      {1234, 1235},
      // Coefficients 4 taps
      {29,  198, 29,  0,   10,  204, 56,  -14, -5,  193, 86,  -18, -15, 174, 117, -20,
       -20, 148, 148, -20, -20, 117, 174, -15, -18, 86,  193, -5,  -14, 56,  204, 10},
      // Coefficients 6 taps
      {-23, 39,  224, 39,  -23, 0, -15, 13,  212, 66,  -28, 8, -8,  -6, 199, 95, -32, 8, -2,  -20, 179, 125, -32, 6, 3,
       -29, 154, 154, -29, 3,   6, -32, 125, 179, -20, -2,  8, -32, 95, 199, -6, -8,  8, -28, 66,  212, 13,  -15},
    },

    // Band_8_1237
    {
      // Scaler inc range
      {1235, 1237},
      // Coefficients 4 taps
      {30,  196, 30,  0,   11,  203, 56,  -14, -5,  193, 86,  -18, -15, 174, 117, -20,
       -19, 147, 147, -19, -20, 117, 174, -15, -18, 86,  193, -5,  -14, 56,  203, 11},
      // Coefficients 6 taps
      {-23, 39,  224, 39,  -23, 0, -15, 14,  211, 66,  -28, 8, -8,  -6, 198, 96, -32, 8, -2,  -20, 179, 125, -32, 6, 3,
       -29, 154, 154, -29, 3,   6, -32, 125, 179, -20, -2,  8, -32, 96, 198, -6, -8,  8, -28, 66,  211, 14,  -15},
    },

    // Band_8_1238
    {
      // Scaler inc range
      {1237, 1238},
      // Coefficients 4 taps
      {30,  196, 30,  0,   11,  203, 56,  -14, -4,  192, 86,  -18, -14, 173, 117, -20,
       -19, 147, 147, -19, -20, 117, 173, -14, -18, 86,  192, -4,  -14, 56,  203, 11},
      // Coefficients 6 taps
      {-23, 39,  224, 39,  -23, 0, -15, 14,  211, 66,  -28, 8, -8,  -6, 198, 96, -32, 8, -2,  -20, 179, 125, -32, 6, 3,
       -29, 154, 154, -29, 3,   6, -32, 125, 179, -20, -2,  8, -32, 96, 198, -6, -8,  8, -28, 66,  211, 14,  -15},
    },

    // Band_8_1239
    {
      // Scaler inc range
      {1238, 1239},
      // Coefficients 4 taps
      {30,  196, 30,  0,   11,  202, 57,  -14, -4,  192, 86,  -18, -14, 173, 117, -20,
       -19, 147, 147, -19, -20, 117, 173, -14, -18, 86,  192, -4,  -14, 57,  202, 11},
      // Coefficients 6 taps
      {-23, 40,  222, 40,  -23, 0, -15, 14,  211, 66,  -28, 8, -8,  -6, 198, 96, -32, 8, -2,  -20, 179, 125, -32, 6, 3,
       -28, 153, 153, -28, 3,   6, -32, 125, 179, -20, -2,  8, -32, 96, 198, -6, -8,  8, -28, 66,  211, 14,  -15},
    },

    // Band_8_1241
    {
      // Scaler inc range
      {1239, 1241},
      // Coefficients 4 taps
      {30,  196, 30,  0,   11,  202, 57,  -14, -4,  192, 86,  -18, -14, 173, 117, -20,
       -19, 147, 147, -19, -20, 117, 173, -14, -18, 86,  192, -4,  -14, 57,  202, 11},
      // Coefficients 6 taps
      {-23, 40,  222, 40,  -23, 0, -15, 14,  211, 66,  -28, 8, -8,  -5, 197, 96, -32, 8, -2,  -20, 178, 126, -32, 6, 3,
       -28, 153, 153, -28, 3,   6, -32, 126, 178, -20, -2,  8, -32, 96, 197, -5, -8,  8, -28, 66,  211, 14,  -15},
    },

    // Band_8_1242
    {
      // Scaler inc range
      {1241, 1242},
      // Coefficients 4 taps
      {30,  196, 30,  0,   11,  202, 57,  -14, -4,  192, 86,  -18, -14, 173, 117, -20,
       -19, 147, 147, -19, -20, 117, 173, -14, -18, 86,  192, -4,  -14, 57,  202, 11},
      // Coefficients 6 taps
      {-23, 40,  222, 40,  -23, 0, -15, 14,  210, 67,  -28, 8, -8,  -5, 197, 96, -32, 8, -2,  -19, 177, 126, -32, 6, 3,
       -28, 153, 153, -28, 3,   6, -32, 126, 177, -19, -2,  8, -32, 96, 197, -5, -8,  8, -28, 67,  210, 14,  -15},
    },

    // Band_8_1244
    {
      // Scaler inc range
      {1242, 1244},
      // Coefficients 4 taps
      {30,  196, 30,  0,   11,  202, 57,  -14, -4,  192, 86,  -18, -14, 173, 117, -20,
       -19, 147, 147, -19, -20, 117, 173, -14, -18, 86,  192, -4,  -14, 57,  202, 11},
      // Coefficients 6 taps
      {-23, 40,  222, 40,  -23, 0, -16, 15,  210, 67,  -28, 8, -9,  -5, 198, 96, -32, 8, -2,  -19, 177, 126, -32, 6, 3,
       -28, 153, 153, -28, 3,   6, -32, 126, 177, -19, -2,  8, -32, 96, 198, -5, -9,  8, -28, 67,  210, 15,  -16},
    },

    // Band_8_1245
    {
      // Scaler inc range
      {1244, 1245},
      // Coefficients 4 taps
      {30,  196, 30,  0,   12,  201, 57,  -14, -4,  192, 86,  -18, -14, 173, 117, -20,
       -19, 147, 147, -19, -20, 117, 173, -14, -18, 86,  192, -4,  -14, 57,  201, 12},
      // Coefficients 6 taps
      {-23, 40,  222, 40,  -23, 0, -16, 15,  210, 67,  -28, 8, -9,  -5, 198, 96, -32, 8, -2,  -19, 177, 126, -32, 6, 2,
       -28, 154, 154, -28, 2,   6, -32, 126, 177, -19, -2,  8, -32, 96, 198, -5, -9,  8, -28, 67,  210, 15,  -16},
    },

    // Band_8_1247
    {
      // Scaler inc range
      {1245, 1247},
      // Coefficients 4 taps
      {31,  194, 31,  0,   12,  201, 57,  -14, -3,  191, 86,  -18, -14, 173, 117, -20,
       -19, 147, 147, -19, -20, 117, 173, -14, -18, 86,  191, -3,  -14, 57,  201, 12},
      // Coefficients 6 taps
      {-23, 41,  220, 41,  -23, 0, -16, 15,  210, 67,  -28, 8, -9,  -5, 198, 96, -32, 8, -3,  -19, 178, 126, -32, 6, 2,
       -28, 154, 154, -28, 2,   6, -32, 126, 178, -19, -3,  8, -32, 96, 198, -5, -9,  8, -28, 67,  210, 15,  -16},
    },

    // Band_8_1248
    {
      // Scaler inc range
      {1247, 1248},
      // Coefficients 4 taps
      {31,  194, 31,  0,   12,  201, 57,  -14, -3,  191, 86,  -18, -13, 172, 117, -20,
       -19, 147, 147, -19, -20, 117, 172, -13, -18, 86,  191, -3,  -14, 57,  201, 12},
      // Coefficients 6 taps
      {-23, 41,  220, 41,  -23, 0, -16, 15,  210, 67,  -28, 8, -9,  -4, 197, 96, -32, 8, -3,  -19, 178, 126, -32, 6, 2,
       -28, 154, 154, -28, 2,   6, -32, 126, 178, -19, -3,  8, -32, 96, 197, -4, -9,  8, -28, 67,  210, 15,  -16},
    },

    // Band_8_1250
    {
      // Scaler inc range
      {1248, 1250},
      // Coefficients 4 taps
      {31,  194, 31,  0,   12,  201, 57,  -14, -3,  191, 86,  -18, -13, 172, 117, -20,
       -19, 147, 147, -19, -20, 117, 172, -13, -18, 86,  191, -3,  -14, 57,  201, 12},
      // Coefficients 6 taps
      {-23, 41,  220, 41,  -23, 0, -16, 15,  210, 67,  -28, 8, -9,  -4, 197, 97, -32, 7, -3,  -18, 177, 126, -32, 6, 2,
       -27, 153, 153, -27, 2,   6, -32, 126, 177, -18, -3,  7, -32, 97, 197, -4, -9,  8, -28, 67,  210, 15,  -16},
    },

    // Band_8_1251
    {
      // Scaler inc range
      {1250, 1251},
      // Coefficients 4 taps
      {31,  194, 31,  0,   12,  201, 57,  -14, -3,  190, 87,  -18, -13, 172, 117, -20,
       -18, 146, 146, -18, -20, 117, 172, -13, -18, 87,  190, -3,  -14, 57,  201, 12},
      // Coefficients 6 taps
      {-24, 41,  222, 41,  -24, 0, -16, 16,  209, 67,  -28, 8, -9,  -4, 197, 97, -32, 7, -3,  -18, 178, 126, -32, 5, 2,
       -27, 153, 153, -27, 2,   5, -32, 126, 178, -18, -3,  7, -32, 97, 197, -4, -9,  8, -28, 67,  209, 16,  -16},
    },

    // Band_8_1253
    {
      // Scaler inc range
      {1251, 1253},
      // Coefficients 4 taps
      {31,  194, 31,  0,   12,  200, 58,  -14, -3,  190, 87,  -18, -13, 171, 117, -19,
       -18, 146, 146, -18, -19, 117, 171, -13, -18, 87,  190, -3,  -14, 58,  200, 12},
      // Coefficients 6 taps
      {-24, 41,  222, 41,  -24, 0, -16, 16,  208, 68,  -28, 8, -9,  -4, 197, 97, -32, 7, -3,  -18, 177, 126, -31, 5, 2,
       -27, 153, 153, -27, 2,   5, -31, 126, 177, -18, -3,  7, -32, 97, 197, -4, -9,  8, -28, 68,  208, 16,  -16},
    },

    // Band_8_1254
    {
      // Scaler inc range
      {1253, 1254},
      // Coefficients 4 taps
      {31,  194, 31,  0,   13,  199, 58,  -14, -3,  189, 87,  -17, -13, 171, 117, -19,
       -18, 146, 146, -18, -19, 117, 171, -13, -17, 87,  189, -3,  -14, 58,  199, 13},
      // Coefficients 6 taps
      {-24, 42,  220, 42,  -24, 0, -16, 16,  208, 68,  -28, 8, -9,  -4, 197, 97, -32, 7, -3,  -18, 177, 126, -31, 5, 2,
       -27, 153, 153, -27, 2,   5, -31, 126, 177, -18, -3,  7, -32, 97, 197, -4, -9,  8, -28, 68,  208, 16,  -16},
    },

    // Band_8_1256
    {
      // Scaler inc range
      {1254, 1256},
      // Coefficients 4 taps
      {31,  194, 31,  0,   13,  199, 58,  -14, -3,  189, 87,  -17, -13, 171, 117, -19,
       -18, 146, 146, -18, -19, 117, 171, -13, -17, 87,  189, -3,  -14, 58,  199, 13},
      // Coefficients 6 taps
      {-24, 42,  220, 42,  -24, 0, -16, 16,  208, 68,  -28, 8, -9,  -3, 196, 97, -32, 7, -3,  -18, 177, 126, -31, 5, 2,
       -27, 153, 153, -27, 2,   5, -31, 126, 177, -18, -3,  7, -32, 97, 196, -3, -9,  8, -28, 68,  208, 16,  -16},
    },

    // Band_8_1257
    {
      // Scaler inc range
      {1256, 1257},
      // Coefficients 4 taps
      {32,  192, 32,  0,   13,  199, 58,  -14, -2,  188, 87,  -17, -13, 171, 117, -19,
       -18, 146, 146, -18, -19, 117, 171, -13, -17, 87,  188, -2,  -14, 58,  199, 13},
      // Coefficients 6 taps
      {-24, 42,  220, 42,  -24, 0, -16, 17,  207, 68,  -28, 8, -10, -3, 196, 97, -31, 7, -3,  -18, 177, 126, -31, 5, 2,
       -27, 153, 153, -27, 2,   5, -31, 126, 177, -18, -3,  7, -31, 97, 196, -3, -10, 8, -28, 68,  207, 17,  -16},
    },

    // Band_8_1259
    {
      // Scaler inc range
      {1257, 1259},
      // Coefficients 4 taps
      {32,  192, 32,  0,   13,  199, 58,  -14, -2,  188, 87,  -17, -13, 171, 117, -19,
       -18, 146, 146, -18, -19, 117, 171, -13, -17, 87,  188, -2,  -14, 58,  199, 13},
      // Coefficients 6 taps
      {-24, 42,  220, 42,  -24, 0, -17, 17,  208, 68,  -28, 8, -10, -3, 196, 97, -31, 7, -3,  -17, 176, 126, -31, 5, 2,
       -27, 153, 153, -27, 2,   5, -31, 126, 176, -17, -3,  7, -31, 97, 196, -3, -10, 8, -28, 68,  208, 17,  -17},
    },

    // Band_8_1260
    {
      // Scaler inc range
      {1259, 1260},
      // Coefficients 4 taps
      {32,  192, 32,  0,   13,  199, 58,  -14, -2,  188, 87,  -17, -12, 170, 117, -19,
       -18, 146, 146, -18, -19, 117, 170, -12, -17, 87,  188, -2,  -14, 58,  199, 13},
      // Coefficients 6 taps
      {-24, 42,  220, 42,  -24, 0, -17, 17,  208, 68,  -28, 8, -10, -3, 196, 97, -31, 7, -4,  -17, 177, 126, -31, 5, 1,
       -27, 154, 154, -27, 1,   5, -31, 126, 177, -17, -4,  7, -31, 97, 196, -3, -10, 8, -28, 68,  208, 17,  -17},
    },

    // Band_8_1262
    {
      // Scaler inc range
      {1260, 1262},
      // Coefficients 4 taps
      {32,  192, 32,  0,   13,  199, 58,  -14, -2,  188, 87,  -17, -12, 170, 117, -19,
       -18, 146, 146, -18, -19, 117, 170, -12, -17, 87,  188, -2,  -14, 58,  199, 13},
      // Coefficients 6 taps
      {-24, 43,  218, 43,  -24, 0, -17, 17,  207, 69,  -28, 8, -10, -2, 195, 97, -31, 7, -4,  -17, 177, 126, -31, 5, 1,
       -26, 153, 153, -26, 1,   5, -31, 126, 177, -17, -4,  7, -31, 97, 195, -2, -10, 8, -28, 69,  207, 17,  -17},
    },

    // Band_8_1263
    {
      // Scaler inc range
      {1262, 1263},
      // Coefficients 4 taps
      {32,  192, 32,  0,   14,  198, 58,  -14, -2,  188, 87,  -17, -12, 170, 117, -19,
       -18, 146, 146, -18, -19, 117, 170, -12, -17, 87,  188, -2,  -14, 58,  198, 14},
      // Coefficients 6 taps
      {-24, 43,  218, 43,  -24, 0, -17, 17,  208, 69,  -29, 8, -10, -2, 195, 97, -31, 7, -4,  -17, 177, 126, -31, 5, 1,
       -26, 153, 153, -26, 1,   5, -31, 126, 177, -17, -4,  7, -31, 97, 195, -2, -10, 8, -29, 69,  208, 17,  -17},
    },

    // Band_8_1265
    {
      // Scaler inc range
      {1263, 1265},
      // Coefficients 4 taps
      {32,  192, 32,  0,   14,  198, 58,  -14, -2,  188, 87,  -17, -12, 170, 117, -19,
       -18, 146, 146, -18, -19, 117, 170, -12, -17, 87,  188, -2,  -14, 58,  198, 14},
      // Coefficients 6 taps
      {-24, 43,  218, 43,  -24, 0, -17, 18,  207, 69,  -29, 8, -10, -2, 195, 97, -31, 7, -4,  -17, 177, 126, -31, 5, 1,
       -26, 153, 153, -26, 1,   5, -31, 126, 177, -17, -4,  7, -31, 97, 195, -2, -10, 8, -29, 69,  207, 18,  -17},
    },

    // Band_8_1266
    {
      // Scaler inc range
      {1265, 1266},
      // Coefficients 4 taps
      {32,  192, 32,  0,   14,  197, 59,  -14, -1,  187, 87,  -17, -12, 170, 117, -19,
       -17, 145, 145, -17, -19, 117, 170, -12, -17, 87,  187, -1,  -14, 59,  197, 14},
      // Coefficients 6 taps
      {-24, 43,  218, 43,  -24, 0, -17, 18,  207, 69,  -29, 8, -10, -2, 194, 98, -31, 7, -4,  -16, 176, 126, -31, 5, 1,
       -26, 153, 153, -26, 1,   5, -31, 126, 176, -16, -4,  7, -31, 98, 194, -2, -10, 8, -29, 69,  207, 18,  -17},
    },

    // Band_8_1268
    {
      // Scaler inc range
      {1266, 1268},
      // Coefficients 4 taps
      {33,  190, 33,  0,   14,  197, 59,  -14, -1,  187, 87,  -17, -12, 170, 117, -19,
       -17, 145, 145, -17, -19, 117, 170, -12, -17, 87,  187, -1,  -14, 59,  197, 14},
      // Coefficients 6 taps
      {-24, 43,  218, 43,  -24, 0, -17, 18,  207, 69,  -29, 8, -10, -2, 194, 98, -31, 7, -4,  -16, 176, 126, -31, 5, 1,
       -26, 153, 153, -26, 1,   5, -31, 126, 176, -16, -4,  7, -31, 98, 194, -2, -10, 8, -29, 69,  207, 18,  -17},
    },

    // Band_8_1269
    {
      // Scaler inc range
      {1268, 1269},
      // Coefficients 4 taps
      {33,  190, 33,  0,   14,  197, 59,  -14, -1,  187, 87,  -17, -12, 170, 117, -19,
       -17, 145, 145, -17, -19, 117, 170, -12, -17, 87,  187, -1,  -14, 59,  197, 14},
      // Coefficients 6 taps
      {-24, 44,  216, 44,  -24, 0, -17, 18,  208, 69,  -29, 7, -10, -1, 193, 98, -31, 7, -4,  -16, 176, 126, -30, 4, 1,
       -26, 153, 153, -26, 1,   4, -30, 126, 176, -16, -4,  7, -31, 98, 193, -1, -10, 7, -29, 69,  208, 18,  -17},
    },

    // Band_8_1271
    {
      // Scaler inc range
      {1269, 1271},
      // Coefficients 4 taps
      {33,  190, 33,  0,   14,  197, 59,  -14, -1,  187, 87,  -17, -11, 169, 117, -19,
       -17, 145, 145, -17, -19, 117, 169, -11, -17, 87,  187, -1,  -14, 59,  197, 14},
      // Coefficients 6 taps
      {-24, 44,  216, 44,  -24, 0, -17, 18,  208, 69,  -29, 7, -11, -1, 194, 98, -31, 7, -4,  -16, 176, 126, -30, 4, 1,
       -25, 152, 152, -25, 1,   4, -30, 126, 176, -16, -4,  7, -31, 98, 194, -1, -11, 7, -29, 69,  208, 18,  -17},
    },

    // Band_8_1273
    {
      // Scaler inc range
      {1271, 1273},
      // Coefficients 4 taps
      {33,  190, 33,  0,   15,  196, 59,  -14, -1,  187, 87,  -17, -11, 169, 117, -19,
       -17, 145, 145, -17, -19, 117, 169, -11, -17, 87,  187, -1,  -14, 59,  196, 15},
      // Coefficients 6 taps
      {-24, 44,  216, 44,  -24, 0, -17, 19,  206, 70,  -29, 7, -11, -1, 195, 98, -31, 6, -5,  -16, 177, 126, -30, 4, 1,
       -25, 152, 152, -25, 1,   4, -30, 126, 177, -16, -5,  6, -31, 98, 195, -1, -11, 7, -29, 70,  206, 19,  -17},
    },

    // Band_8_1274
    {
      // Scaler inc range
      {1273, 1274},
      // Coefficients 4 taps
      {33,  190, 33,  0,   15,  196, 59,  -14, -1,  187, 87,  -17, -11, 169, 117, -19,
       -17, 145, 145, -17, -19, 117, 169, -11, -17, 87,  187, -1,  -14, 59,  196, 15},
      // Coefficients 6 taps
      {-24, 44,  216, 44,  -24, 0, -17, 19,  206, 70,  -29, 7, -11, -1, 195, 98, -31, 6, -5,  -16, 177, 126, -30, 4, 0,
       -25, 153, 153, -25, 0,   4, -30, 126, 177, -16, -5,  6, -31, 98, 195, -1, -11, 7, -29, 70,  206, 19,  -17},
    },

    // Band_8_1276
    {
      // Scaler inc range
      {1274, 1276},
      // Coefficients 4 taps
      {33,  190, 33,  0,   15,  196, 59,  -14, 0,   186, 87,  -17, -11, 168, 117, -18,
       -17, 145, 145, -17, -18, 117, 168, -11, -17, 87,  186, 0,   -14, 59,  196, 15},
      // Coefficients 6 taps
      {-24, 44,  216, 44,  -24, 0, -18, 19,  207, 70,  -29, 7, -11, -1, 195, 98, -31, 6, -5,  -15, 176, 126, -30, 4, 0,
       -25, 153, 153, -25, 0,   4, -30, 126, 176, -15, -5,  6, -31, 98, 195, -1, -11, 7, -29, 70,  207, 19,  -18},
    },

    // Band_8_1277
    {
      // Scaler inc range
      {1276, 1277},
      // Coefficients 4 taps
      {33,  190, 33,  0,   15,  196, 59,  -14, 0,   186, 87,  -17, -11, 168, 117, -18,
       -17, 145, 145, -17, -18, 117, 168, -11, -17, 87,  186, 0,   -14, 59,  196, 15},
      // Coefficients 6 taps
      {-25, 44,  218, 44,  -25, 0, -18, 19,  207, 70,  -29, 7, -11, 0,  194, 98, -31, 6, -5,  -15, 176, 126, -30, 4, 0,
       -25, 153, 153, -25, 0,   4, -30, 126, 176, -15, -5,  6, -31, 98, 194, 0,  -11, 7, -29, 70,  207, 19,  -18},
    },

    // Band_8_1279
    {
      // Scaler inc range
      {1277, 1279},
      // Coefficients 4 taps
      {33,  190, 33,  0,   15,  196, 59,  -14, 0,   186, 87,  -17, -11, 168, 117, -18,
       -17, 145, 145, -17, -18, 117, 168, -11, -17, 87,  186, 0,   -14, 59,  196, 15},
      // Coefficients 6 taps
      {-25, 45,  216, 45,  -25, 0, -18, 19,  207, 70,  -29, 7, -11, 0,  194, 98, -31, 6, -5,  -15, 176, 126, -30, 4, 0,
       -25, 153, 153, -25, 0,   4, -30, 126, 176, -15, -5,  6, -31, 98, 194, 0,  -11, 7, -29, 70,  207, 19,  -18},
    },

    // Band_8_1280
    {
      // Scaler inc range
      {1279, 1280},
      // Coefficients 4 taps
      {34,  188, 34,  0,   15,  195, 60,  -14, 0,   186, 87,  -17, -11, 168, 117, -18,
       -16, 144, 144, -16, -18, 117, 168, -11, -17, 87,  186, 0,   -14, 60,  195, 15},
      // Coefficients 6 taps
      {-25, 45,  216, 45,  -25, 0, -18, 20,  206, 70,  -29, 7, -11, 0,  194, 98, -31, 6, -5,  -15, 176, 126, -30, 4, 0,
       -25, 153, 153, -25, 0,   4, -30, 126, 176, -15, -5,  6, -31, 98, 194, 0,  -11, 7, -29, 70,  206, 20,  -18},
    },

    // Band_8_1282
    {
      // Scaler inc range
      {1280, 1282},
      // Coefficients 4 taps
      {34,  188, 34,  0,   16,  194, 60,  -14, 0,   185, 88,  -17, -10, 167, 117, -18,
       -16, 144, 144, -16, -18, 117, 167, -10, -17, 88,  185, 0,   -14, 60,  194, 16},
      // Coefficients 6 taps
      {-25, 45,  216, 45,  -25, 0, -18, 20,  206, 70,  -29, 7, -11, 0,  193, 99, -31, 6, -5,  -15, 176, 126, -30, 4, 0,
       -24, 152, 152, -24, 0,   4, -30, 126, 176, -15, -5,  6, -31, 99, 193, 0,  -11, 7, -29, 70,  206, 20,  -18},
    },

    // Band_8_1283
    {
      // Scaler inc range
      {1282, 1283},
      // Coefficients 4 taps
      {34,  188, 34,  0,   16,  194, 60,  -14, 0,   185, 88,  -17, -10, 167, 117, -18,
       -16, 144, 144, -16, -18, 117, 167, -10, -17, 88,  185, 0,   -14, 60,  194, 16},
      // Coefficients 6 taps
      {-25, 45,  216, 45,  -25, 0, -18, 20,  205, 71,  -29, 7, -11, 0,  193, 99, -31, 6, -5,  -14, 175, 126, -30, 4, 0,
       -24, 152, 152, -24, 0,   4, -30, 126, 175, -14, -5,  6, -31, 99, 193, 0,  -11, 7, -29, 71,  205, 20,  -18},
    },

    // Band_8_1285
    {
      // Scaler inc range
      {1283, 1285},
      // Coefficients 4 taps
      {34,  188, 34,  0,   16,  194, 60,  -14, 1,   184, 88,  -17, -10, 167, 117, -18,
       -16, 144, 144, -16, -18, 117, 167, -10, -17, 88,  184, 1,   -14, 60,  194, 16},
      // Coefficients 6 taps
      {-25, 45,  216, 45,  -25, 0, -18, 20,  205, 71,  -29, 7, -12, 1,  193, 99, -31, 6, -5,  -14, 175, 126, -29, 3, 0,
       -24, 152, 152, -24, 0,   3, -29, 126, 175, -14, -5,  6, -31, 99, 193, 1,  -12, 7, -29, 71,  205, 20,  -18},
    },

    // Band_8_1287
    {
      // Scaler inc range
      {1285, 1287},
      // Coefficients 4 taps
      {34,  188, 34,  0,   16,  194, 60,  -14, 1,   184, 88,  -17, -10, 167, 117, -18,
       -16, 144, 144, -16, -18, 117, 167, -10, -17, 88,  184, 1,   -14, 60,  194, 16},
      // Coefficients 6 taps
      {-25, 46,  214, 46,  -25, 0, -18, 21,  204, 71,  -29, 7, -12, 1,  193, 99, -31, 6, -6,  -14, 175, 127, -29, 3, 0,
       -24, 152, 152, -24, 0,   3, -29, 127, 175, -14, -6,  6, -31, 99, 193, 1,  -12, 7, -29, 71,  204, 21,  -18},
    },

    // Band_8_1288
    {
      // Scaler inc range
      {1287, 1288},
      // Coefficients 4 taps
      {34,  188, 34,  0,   16,  194, 60,  -14, 1,   184, 88,  -17, -10, 167, 117, -18,
       -16, 144, 144, -16, -18, 117, 167, -10, -17, 88,  184, 1,   -14, 60,  194, 16},
      // Coefficients 6 taps
      {-25, 46,  214, 46,  -25, 0, -18, 21,  204, 71,  -29, 7, -12, 1,  193, 99, -31, 6, -6,  -14, 175, 127, -29, 3, -1,
       -24, 153, 153, -24, -1,  3, -29, 127, 175, -14, -6,  6, -31, 99, 193, 1,  -12, 7, -29, 71,  204, 21,  -18},
    },

    // Band_8_1290
    {
      // Scaler inc range
      {1288, 1290},
      // Coefficients 4 taps
      {34,  188, 34,  0,   16,  194, 60,  -14, 1,   184, 88,  -17, -10, 168, 116, -18,
       -16, 144, 144, -16, -18, 116, 168, -10, -17, 88,  184, 1,   -14, 60,  194, 16},
      // Coefficients 6 taps
      {-25, 46,  214, 46,  -25, 0, -18, 21,  204, 71,  -29, 7, -12, 1,  193, 99, -31, 6, -6,  -14, 175, 127, -29, 3, -1,
       -24, 153, 153, -24, -1,  3, -29, 127, 175, -14, -6,  6, -31, 99, 193, 1,  -12, 7, -29, 71,  204, 21,  -18},
    },

    // Band_8_1291
    {
      // Scaler inc range
      {1290, 1291},
      // Coefficients 4 taps
      {35,  186, 35,  0,   16,  194, 60,  -14, 1,   184, 88,  -17, -10, 168, 116, -18,
       -16, 144, 144, -16, -18, 116, 168, -10, -17, 88,  184, 1,   -14, 60,  194, 16},
      // Coefficients 6 taps
      {-25, 46,  214, 46,  -25, 0, -18, 21,  204, 71,  -29, 7, -12, 2,  191, 99, -30, 6, -6,  -13, 174, 127, -29, 3, -1,
       -24, 153, 153, -24, -1,  3, -29, 127, 174, -13, -6,  6, -30, 99, 191, 2,  -12, 7, -29, 71,  204, 21,  -18},
    },

    // Band_8_1293
    {
      // Scaler inc range
      {1291, 1293},
      // Coefficients 4 taps
      {35,  186, 35,  0,   17,  193, 60,  -14, 1,   184, 88,  -17, -9,  167, 116, -18,
       -16, 144, 144, -16, -18, 116, 167, -9,  -17, 88,  184, 1,   -14, 60,  193, 17},
      // Coefficients 6 taps
      {-25, 46,  214, 46,  -25, 0, -18, 21,  204, 71,  -29, 7, -12, 2,  191, 99, -30, 6, -6,  -13, 174, 127, -29, 3, -1,
       -23, 152, 152, -23, -1,  3, -29, 127, 174, -13, -6,  6, -30, 99, 191, 2,  -12, 7, -29, 71,  204, 21,  -18},
    },

    // Band_8_1295
    {
      // Scaler inc range
      {1293, 1295},
      // Coefficients 4 taps
      {35,  186, 35,  0,   17,  193, 60,  -14, 1,   184, 88,  -17, -9,  167, 116, -18,
       -15, 143, 143, -15, -18, 116, 167, -9,  -17, 88,  184, 1,   -14, 60,  193, 17},
      // Coefficients 6 taps
      {-25, 46,  214, 46,  -25, 0, -19, 22,  203, 72,  -29, 7, -12, 2,  192, 99, -30, 5, -6,  -13, 174, 127, -29, 3, -1,
       -23, 152, 152, -23, -1,  3, -29, 127, 174, -13, -6,  5, -30, 99, 192, 2,  -12, 7, -29, 72,  203, 22,  -19},
    },

    // Band_8_1296
    {
      // Scaler inc range
      {1295, 1296},
      // Coefficients 4 taps
      {35,  186, 35,  0,   17,  192, 61,  -14, 2,   183, 88,  -17, -9,  167, 116, -18,
       -15, 143, 143, -15, -18, 116, 167, -9,  -17, 88,  183, 2,   -14, 61,  192, 17},
      // Coefficients 6 taps
      {-25, 47,  212, 47,  -25, 0, -19, 22,  203, 72,  -29, 7, -12, 2,  192, 99, -30, 5, -6,  -13, 174, 127, -29, 3, -1,
       -23, 152, 152, -23, -1,  3, -29, 127, 174, -13, -6,  5, -30, 99, 192, 2,  -12, 7, -29, 72,  203, 22,  -19},
    },

    // Band_8_1298
    {
      // Scaler inc range
      {1296, 1298},
      // Coefficients 4 taps
      {35,  186, 35,  0,   17,  192, 61,  -14, 2,   183, 88,  -17, -9,  166, 116, -17,
       -15, 143, 143, -15, -17, 116, 166, -9,  -17, 88,  183, 2,   -14, 61,  192, 17},
      // Coefficients 6 taps
      {-25, 47,  212, 47,  -25, 0, -19, 22,  203, 72,  -29, 7, -12, 2,  192, 99, -30, 5, -6,  -13, 174, 127, -29, 3, -1,
       -23, 152, 152, -23, -1,  3, -29, 127, 174, -13, -6,  5, -30, 99, 192, 2,  -12, 7, -29, 72,  203, 22,  -19},
    },

    // Band_8_1299
    {
      // Scaler inc range
      {1298, 1299},
      // Coefficients 4 taps
      {35,  186, 35,  0,   17,  192, 61,  -14, 2,   183, 88,  -17, -9,  166, 116, -17,
       -15, 143, 143, -15, -17, 116, 166, -9,  -17, 88,  183, 2,   -14, 61,  192, 17},
      // Coefficients 6 taps
      {-25, 47,  212, 47,  -25, 0, -19, 22,  203, 72,  -29, 7, -13, 3,  192, 99, -30, 5, -7,  -12, 173, 127, -28, 3, -1,
       -23, 152, 152, -23, -1,  3, -28, 127, 173, -12, -7,  5, -30, 99, 192, 3,  -13, 7, -29, 72,  203, 22,  -19},
    },

    // Band_8_1301
    {
      // Scaler inc range
      {1299, 1301},
      // Coefficients 4 taps
      {35,  186, 35,  0,   17,  192, 61,  -14, 2,   183, 88,  -17, -9,  166, 116, -17,
       -15, 143, 143, -15, -17, 116, 166, -9,  -17, 88,  183, 2,   -14, 61,  192, 17},
      // Coefficients 6 taps
      {-25, 47,  212, 47,  -25, 0,   -19, 22,  203, 72,  -29, 7,   -13, 3,   191, 100,
       -30, 5,   -7,  -12, 173, 127, -28, 3,   -1,  -23, 152, 152, -23, -1,  3,   -28,
       127, 173, -12, -7,  5,   -30, 100, 191, 3,   -13, 7,   -29, 72,  203, 22,  -19},
    },

    // Band_8_1303
    {
      // Scaler inc range
      {1301, 1303},
      // Coefficients 4 taps
      {35,  186, 35,  0,   18,  191, 61,  -14, 2,   182, 88,  -16, -9,  166, 116, -17,
       -15, 143, 143, -15, -17, 116, 166, -9,  -16, 88,  182, 2,   -14, 61,  191, 18},
      // Coefficients 6 taps
      {-25, 47,  212, 47,  -25, 0,   -19, 23,  203, 72,  -29, 6,   -13, 3,   191, 100,
       -30, 5,   -7,  -12, 174, 127, -28, 2,   -2,  -22, 152, 152, -22, -2,  2,   -28,
       127, 174, -12, -7,  5,   -30, 100, 191, 3,   -13, 6,   -29, 72,  203, 23,  -19},
    },

    // Band_8_1304
    {
      // Scaler inc range
      {1303, 1304},
      // Coefficients 4 taps
      {36,  184, 36,  0,   18,  191, 61,  -14, 2,   182, 88,  -16, -8,  165, 116, -17,
       -15, 143, 143, -15, -17, 116, 165, -8,  -16, 88,  182, 2,   -14, 61,  191, 18},
      // Coefficients 6 taps
      {-25, 48,  210, 48,  -25, 0,   -19, 23,  203, 72,  -29, 6,   -13, 3,   191, 100,
       -30, 5,   -7,  -12, 174, 127, -28, 2,   -2,  -22, 152, 152, -22, -2,  2,   -28,
       127, 174, -12, -7,  5,   -30, 100, 191, 3,   -13, 6,   -29, 72,  203, 23,  -19},
    },

    // Band_8_1306
    {
      // Scaler inc range
      {1304, 1306},
      // Coefficients 4 taps
      {36,  184, 36,  0,   18,  191, 61,  -14, 3,   181, 88,  -16, -8,  165, 116, -17,
       -15, 143, 143, -15, -17, 116, 165, -8,  -16, 88,  181, 3,   -14, 61,  191, 18},
      // Coefficients 6 taps
      {-25, 48,  210, 48,  -25, 0,   -19, 23,  202, 73,  -29, 6,   -13, 3,   191, 100,
       -30, 5,   -7,  -12, 174, 127, -28, 2,   -2,  -22, 152, 152, -22, -2,  2,   -28,
       127, 174, -12, -7,  5,   -30, 100, 191, 3,   -13, 6,   -29, 73,  202, 23,  -19},
    },

    // Band_8_1307
    {
      // Scaler inc range
      {1306, 1307},
      // Coefficients 4 taps
      {36,  184, 36,  0,   18,  191, 61,  -14, 3,   181, 88,  -16, -8,  165, 116, -17,
       -15, 143, 143, -15, -17, 116, 165, -8,  -16, 88,  181, 3,   -14, 61,  191, 18},
      // Coefficients 6 taps
      {-25, 48,  210, 48,  -25, 0,   -19, 23,  202, 73,  -29, 6,   -13, 4,   190, 100,
       -30, 5,   -7,  -11, 173, 127, -28, 2,   -2,  -22, 152, 152, -22, -2,  2,   -28,
       127, 173, -11, -7,  5,   -30, 100, 190, 4,   -13, 6,   -29, 73,  202, 23,  -19},
    },

    // Band_8_1309
    {
      // Scaler inc range
      {1307, 1309},
      // Coefficients 4 taps
      {36,  184, 36,  0,   18,  191, 61,  -14, 3,   181, 88,  -16, -8,  165, 116, -17,
       -14, 142, 142, -14, -17, 116, 165, -8,  -16, 88,  181, 3,   -14, 61,  191, 18},
      // Coefficients 6 taps
      {-25, 48,  210, 48,  -25, 0,   -19, 23,  201, 73,  -28, 6,   -13, 4,   190, 100,
       -30, 5,   -7,  -11, 173, 127, -28, 2,   -2,  -22, 152, 152, -22, -2,  2,   -28,
       127, 173, -11, -7,  5,   -30, 100, 190, 4,   -13, 6,   -28, 73,  201, 23,  -19},
    },

    // Band_8_1311
    {
      // Scaler inc range
      {1309, 1311},
      // Coefficients 4 taps
      {36,  184, 36,  0,   18,  191, 61,  -14, 3,   181, 88,  -16, -8,  165, 116, -17,
       -14, 142, 142, -14, -17, 116, 165, -8,  -16, 88,  181, 3,   -14, 61,  191, 18},
      // Coefficients 6 taps
      {-25, 48,  210, 48,  -25, 0,   -19, 24,  200, 73,  -28, 6,   -13, 4,   190, 100,
       -30, 5,   -7,  -11, 173, 127, -28, 2,   -2,  -22, 152, 152, -22, -2,  2,   -28,
       127, 173, -11, -7,  5,   -30, 100, 190, 4,   -13, 6,   -28, 73,  200, 24,  -19},
    },

    // Band_8_1312
    {
      // Scaler inc range
      {1311, 1312},
      // Coefficients 4 taps
      {36,  184, 36,  0,   19,  189, 62,  -14, 3,   181, 88,  -16, -8,  165, 116, -17,
       -14, 142, 142, -14, -17, 116, 165, -8,  -16, 88,  181, 3,   -14, 62,  189, 19},
      // Coefficients 6 taps
      {-25, 48,  210, 48,  -25, 0,   -19, 24,  200, 73,  -28, 6,   -13, 4,   190, 100,
       -30, 5,   -7,  -11, 173, 127, -28, 2,   -2,  -21, 151, 151, -21, -2,  2,   -28,
       127, 173, -11, -7,  5,   -30, 100, 190, 4,   -13, 6,   -28, 73,  200, 24,  -19},
    },

    // Band_8_1314
    {
      // Scaler inc range
      {1312, 1314},
      // Coefficients 4 taps
      {36,  184, 36,  0,   19,  189, 62,  -14, 3,   181, 88,  -16, -8,  165, 116, -17,
       -14, 142, 142, -14, -17, 116, 165, -8,  -16, 88,  181, 3,   -14, 62,  189, 19},
      // Coefficients 6 taps
      {-25, 49,  208, 49,  -25, 0,   -19, 24,  200, 73,  -28, 6,   -13, 4,   190, 100,
       -30, 5,   -8,  -11, 173, 127, -27, 2,   -2,  -21, 151, 151, -21, -2,  2,   -27,
       127, 173, -11, -8,  5,   -30, 100, 190, 4,   -13, 6,   -28, 73,  200, 24,  -19},
    },

    // Band_8_1316
    {
      // Scaler inc range
      {1314, 1316},
      // Coefficients 4 taps
      {37,  182, 37,  0,   19,  189, 62,  -14, 4,   180, 88,  -16, -7,  164, 116, -17,
       -14, 142, 142, -14, -17, 116, 164, -7,  -16, 88,  180, 4,   -14, 62,  189, 19},
      // Coefficients 6 taps
      {-25, 49,  208, 49,  -25, 0,   -19, 24,  200, 73,  -28, 6,   -14, 5,   191, 100,
       -30, 4,   -8,  -11, 173, 127, -27, 2,   -2,  -21, 151, 151, -21, -2,  2,   -27,
       127, 173, -11, -8,  4,   -30, 100, 191, 5,   -14, 6,   -28, 73,  200, 24,  -19},
    },

    // Band_8_1317
    {
      // Scaler inc range
      {1316, 1317},
      // Coefficients 4 taps
      {37,  182, 37,  0,   19,  189, 62,  -14, 4,   180, 88,  -16, -7,  164, 116, -17,
       -14, 142, 142, -14, -17, 116, 164, -7,  -16, 88,  180, 4,   -14, 62,  189, 19},
      // Coefficients 6 taps
      {-25, 49,  208, 49,  -25, 0,   -20, 24,  201, 73,  -28, 6,   -14, 5,   190, 100,
       -29, 4,   -8,  -10, 172, 127, -27, 2,   -3,  -21, 152, 152, -21, -3,  2,   -27,
       127, 172, -10, -8,  4,   -29, 100, 190, 5,   -14, 6,   -28, 73,  201, 24,  -20},
    },

    // Band_8_1319
    {
      // Scaler inc range
      {1317, 1319},
      // Coefficients 4 taps
      {37,  182, 37,  0,   19,  189, 62,  -14, 4,   180, 88,  -16, -7,  163, 116, -16,
       -14, 142, 142, -14, -16, 116, 163, -7,  -16, 88,  180, 4,   -14, 62,  189, 19},
      // Coefficients 6 taps
      {-25, 49,  208, 49,  -25, 0,   -20, 25,  199, 74,  -28, 6,   -14, 5,   190, 100,
       -29, 4,   -8,  -10, 173, 127, -27, 1,   -3,  -21, 152, 152, -21, -3,  1,   -27,
       127, 173, -10, -8,  4,   -29, 100, 190, 5,   -14, 6,   -28, 74,  199, 25,  -20},
    },

    // Band_8_1321
    {
      // Scaler inc range
      {1319, 1321},
      // Coefficients 4 taps
      {37,  182, 37,  0,   19,  189, 62,  -14, 4,   180, 88,  -16, -7,  163, 116, -16,
       -14, 142, 142, -14, -16, 116, 163, -7,  -16, 88,  180, 4,   -14, 62,  189, 19},
      // Coefficients 6 taps
      {-25, 49,  208, 49,  -25, 0,   -20, 25,  199, 74,  -28, 6,   -14, 5,   190, 100,
       -29, 4,   -8,  -10, 173, 127, -27, 1,   -3,  -21, 152, 152, -21, -3,  1,   -27,
       127, 173, -10, -8,  4,   -29, 100, 190, 5,   -14, 6,   -28, 74,  199, 25,  -20},
    },

    // Band_8_1322
    {
      // Scaler inc range
      {1321, 1322},
      // Coefficients 4 taps
      {37,  182, 37,  0,   19,  189, 62,  -14, 4,   179, 89,  -16, -7,  163, 116, -16,
       -14, 142, 142, -14, -16, 116, 163, -7,  -16, 89,  179, 4,   -14, 62,  189, 19},
      // Coefficients 6 taps
      {-25, 49,  208, 49,  -25, 0,   -20, 25,  199, 74,  -28, 6,   -14, 6,   188, 101,
       -29, 4,   -8,  -10, 173, 127, -27, 1,   -3,  -20, 151, 151, -20, -3,  1,   -27,
       127, 173, -10, -8,  4,   -29, 101, 188, 6,   -14, 6,   -28, 74,  199, 25,  -20},
    },

    // Band_8_1324
    {
      // Scaler inc range
      {1322, 1324},
      // Coefficients 4 taps
      {37,  182, 37,  0,   20,  188, 62,  -14, 4,   179, 89,  -16, -7,  163, 116, -16,
       -13, 141, 141, -13, -16, 116, 163, -7,  -16, 89,  179, 4,   -14, 62,  188, 20},
      // Coefficients 6 taps
      {-26, 50,  208, 50,  -26, 0,   -20, 25,  199, 74,  -28, 6,   -14, 6,   188, 101,
       -29, 4,   -8,  -10, 173, 127, -27, 1,   -3,  -20, 151, 151, -20, -3,  1,   -27,
       127, 173, -10, -8,  4,   -29, 101, 188, 6,   -14, 6,   -28, 74,  199, 25,  -20},
    },

    // Band_8_1326
    {
      // Scaler inc range
      {1324, 1326},
      // Coefficients 4 taps
      {37,  182, 37,  0,   20,  188, 62,  -14, 4,   179, 89,  -16, -7,  163, 116, -16,
       -13, 141, 141, -13, -16, 116, 163, -7,  -16, 89,  179, 4,   -14, 62,  188, 20},
      // Coefficients 6 taps
      {-26, 50,  208, 50,  -26, 0, -20, 25,  199, 74, -28, 6, -14, 6,   188, 101, -29, 4, -8,  -9, 172, 127, -27, 1, -3,
       -20, 151, 151, -20, -3,  1, -27, 127, 172, -9, -8,  4, -29, 101, 188, 6,   -14, 6, -28, 74, 199, 25,  -20},
    },

    // Band_8_1327
    {
      // Scaler inc range
      {1326, 1327},
      // Coefficients 4 taps
      {37,  182, 37,  0,   20,  188, 62,  -14, 5,   178, 89,  -16, -6,  162, 116, -16,
       -13, 141, 141, -13, -16, 116, 162, -6,  -16, 89,  178, 5,   -14, 62,  188, 20},
      // Coefficients 6 taps
      {-26, 50,  208, 50,  -26, 0, -20, 26,  198, 74, -28, 6, -14, 6,   188, 101, -29, 4, -8,  -9, 171, 127, -26, 1, -3,
       -20, 151, 151, -20, -3,  1, -26, 127, 171, -9, -8,  4, -29, 101, 188, 6,   -14, 6, -28, 74, 198, 26,  -20},
    },

    // Band_8_1329
    {
      // Scaler inc range
      {1327, 1329},
      // Coefficients 4 taps
      {38,  180, 38,  0,   20,  187, 63,  -14, 5,   178, 89,  -16, -6,  162, 116, -16,
       -13, 141, 141, -13, -16, 116, 162, -6,  -16, 89,  178, 5,   -14, 63,  187, 20},
      // Coefficients 6 taps
      {-26, 50,  208, 50,  -26, 0, -20, 26,  198, 74, -28, 6, -14, 6,   188, 101, -29, 4, -8,  -9, 171, 127, -26, 1, -3,
       -20, 151, 151, -20, -3,  1, -26, 127, 171, -9, -8,  4, -29, 101, 188, 6,   -14, 6, -28, 74, 198, 26,  -20},
    },

    // Band_8_1331
    {
      // Scaler inc range
      {1329, 1331},
      // Coefficients 4 taps
      {38,  180, 38,  0,   20,  187, 63,  -14, 5,   178, 89,  -16, -6,  162, 116, -16,
       -13, 141, 141, -13, -16, 116, 162, -6,  -16, 89,  178, 5,   -14, 63,  187, 20},
      // Coefficients 6 taps
      {-26, 50,  208, 50,  -26, 0, -20, 26,  198, 74, -28, 6, -14, 7,   187, 101, -29, 4, -9,  -9, 172, 127, -26, 1, -3,
       -20, 151, 151, -20, -3,  1, -26, 127, 172, -9, -9,  4, -29, 101, 187, 7,   -14, 6, -28, 74, 198, 26,  -20},
    },

    // Band_8_1332
    {
      // Scaler inc range
      {1331, 1332},
      // Coefficients 4 taps
      {38,  180, 38,  0,   20,  187, 63,  -14, 5,   178, 89,  -16, -6,  162, 116, -16,
       -13, 141, 141, -13, -16, 116, 162, -6,  -16, 89,  178, 5,   -14, 63,  187, 20},
      // Coefficients 6 taps
      {-26, 50,  208, 50,  -26, 0, -20, 26,  198, 75, -28, 5, -14, 7,   187, 101, -29, 4, -9,  -9, 172, 127, -26, 1, -4,
       -19, 151, 151, -19, -4,  1, -26, 127, 172, -9, -9,  4, -29, 101, 187, 7,   -14, 5, -28, 75, 198, 26,  -20},
    },

    // Band_8_1334
    {
      // Scaler inc range
      {1332, 1334},
      // Coefficients 4 taps
      {38,  180, 38,  0,   20,  186, 63,  -13, 5,   178, 89,  -16, -6,  162, 116, -16,
       -13, 141, 141, -13, -16, 116, 162, -6,  -16, 89,  178, 5,   -13, 63,  186, 20},
      // Coefficients 6 taps
      {-26, 51,  206, 51,  -26, 0, -20, 27,  197, 75, -28, 5, -15, 7,   188, 101, -29, 4, -9,  -8, 171, 127, -26, 1, -4,
       -19, 151, 151, -19, -4,  1, -26, 127, 171, -8, -9,  4, -29, 101, 188, 7,   -15, 5, -28, 75, 197, 27,  -20},
    },

    // Band_8_1336
    {
      // Scaler inc range
      {1334, 1336},
      // Coefficients 4 taps
      {38,  180, 38,  0,   21,  185, 63,  -13, 5,   178, 89,  -16, -6,  163, 115, -16,
       -13, 141, 141, -13, -16, 115, 163, -6,  -16, 89,  178, 5,   -13, 63,  185, 21},
      // Coefficients 6 taps
      {-26, 51,  206, 51,  -26, 0, -20, 27,  197, 75, -28, 5, -15, 7,   188, 101, -29, 4, -9,  -8, 172, 127, -26, 0, -4,
       -19, 151, 151, -19, -4,  0, -26, 127, 172, -8, -9,  4, -29, 101, 188, 7,   -15, 5, -28, 75, 197, 27,  -20},
    },

    // Band_8_1337
    {
      // Scaler inc range
      {1336, 1337},
      // Coefficients 4 taps
      {38,  180, 38,  0,   21,  185, 63,  -13, 6,   177, 89,  -16, -6,  163, 115, -16,
       -12, 140, 140, -12, -16, 115, 163, -6,  -16, 89,  177, 6,   -13, 63,  185, 21},
      // Coefficients 6 taps
      {-26, 51,  206, 51,  -26, 0, -20, 27,  197, 75, -28, 5, -15, 7,   189, 101, -29, 3, -9,  -8, 172, 127, -26, 0, -4,
       -19, 151, 151, -19, -4,  0, -26, 127, 172, -8, -9,  3, -29, 101, 189, 7,   -15, 5, -28, 75, 197, 27,  -20},
    },

    // Band_8_1339
    {
      // Scaler inc range
      {1337, 1339},
      // Coefficients 4 taps
      {38,  180, 38,  0,   21,  185, 63,  -13, 6,   176, 89,  -15, -5,  161, 115, -15,
       -12, 140, 140, -12, -15, 115, 161, -5,  -15, 89,  176, 6,   -13, 63,  185, 21},
      // Coefficients 6 taps
      {-26, 51,  206, 51,  -26, 0, -20, 27,  197, 75, -28, 5, -15, 8,   187, 101, -28, 3, -9,  -8, 172, 127, -26, 0, -4,
       -19, 151, 151, -19, -4,  0, -26, 127, 172, -8, -9,  3, -28, 101, 187, 8,   -15, 5, -28, 75, 197, 27,  -20},
    },

    // Band_8_1341
    {
      // Scaler inc range
      {1339, 1341},
      // Coefficients 4 taps
      {38,  180, 38,  0,   21,  185, 63,  -13, 6,   176, 89,  -15, -5,  161, 115, -15,
       -12, 140, 140, -12, -15, 115, 161, -5,  -15, 89,  176, 6,   -13, 63,  185, 21},
      // Coefficients 6 taps
      {-26, 51,  206, 51,  -26, 0, -20, 27,  197, 75, -28, 5, -15, 8,   187, 101, -28, 3, -9,  -8, 171, 127, -25, 0, -4,
       -19, 151, 151, -19, -4,  0, -25, 127, 171, -8, -9,  3, -28, 101, 187, 8,   -15, 5, -28, 75, 197, 27,  -20},
    },

    // Band_8_1343
    {
      // Scaler inc range
      {1341, 1343},
      // Coefficients 4 taps
      {39,  178, 39,  0,   21,  185, 63,  -13, 6,   176, 89,  -15, -5,  161, 115, -15,
       -12, 140, 140, -12, -15, 115, 161, -5,  -15, 89,  176, 6,   -13, 63,  185, 21},
      // Coefficients 6 taps
      {-26, 51,  206, 51,  -26, 0, -20, 28,  196, 75, -28, 5, -15, 8,   187, 101, -28, 3, -9,  -7, 170, 127, -25, 0, -4,
       -18, 150, 150, -18, -4,  0, -25, 127, 170, -7, -9,  3, -28, 101, 187, 8,   -15, 5, -28, 75, 196, 28,  -20},
    },

    // Band_8_1344
    {
      // Scaler inc range
      {1343, 1344},
      // Coefficients 4 taps
      {39,  178, 39,  0,   21,  185, 63,  -13, 6,   176, 89,  -15, -5,  161, 115, -15,
       -12, 140, 140, -12, -15, 115, 161, -5,  -15, 89,  176, 6,   -13, 63,  185, 21},
      // Coefficients 6 taps
      {-26, 52,  204, 52,  -26, 0, -21, 28,  197, 75, -28, 5, -15, 8,   187, 101, -28, 3, -9,  -7, 170, 127, -25, 0, -4,
       -18, 150, 150, -18, -4,  0, -25, 127, 170, -7, -9,  3, -28, 101, 187, 8,   -15, 5, -28, 75, 197, 28,  -21},
    },

    // Band_8_1346
    {
      // Scaler inc range
      {1344, 1346},
      // Coefficients 4 taps
      {39,  178, 39,  0,   22,  184, 63,  -13, 6,   176, 89,  -15, -5,  161, 115, -15,
       -12, 140, 140, -12, -15, 115, 161, -5,  -15, 89,  176, 6,   -13, 63,  184, 22},
      // Coefficients 6 taps
      {-26, 52,  204, 52,  -26, 0, -21, 28,  196, 76, -28, 5, -15, 8,   186, 102, -28, 3, -10, -7, 171, 127, -25, 0, -4,
       -18, 150, 150, -18, -4,  0, -25, 127, 171, -7, -10, 3, -28, 102, 186, 8,   -15, 5, -28, 76, 196, 28,  -21},
    },

    // Band_8_1348
    {
      // Scaler inc range
      {1346, 1348},
      // Coefficients 4 taps
      {39,  178, 39,  0,   22,  184, 63,  -13, 6,   176, 89,  -15, -5,  161, 115, -15,
       -12, 140, 140, -12, -15, 115, 161, -5,  -15, 89,  176, 6,   -13, 63,  184, 22},
      // Coefficients 6 taps
      {-26, 52,  204, 52,  -26, 0, -21, 28,  196, 76, -28, 5, -15, 9,   185, 102, -28, 3, -10, -7, 171, 127, -25, 0, -4,
       -18, 150, 150, -18, -4,  0, -25, 127, 171, -7, -10, 3, -28, 102, 185, 9,   -15, 5, -28, 76, 196, 28,  -21},
    },

    // Band_8_1350
    {
      // Scaler inc range
      {1348, 1350},
      // Coefficients 4 taps
      {39,  178, 39,  0,   22,  183, 64,  -13, 7,   175, 89,  -15, -5,  161, 115, -15,
       -12, 140, 140, -12, -15, 115, 161, -5,  -15, 89,  175, 7,   -13, 64,  183, 22},
      // Coefficients 6 taps
      {-26, 52,  204, 52,  -26, 0, -21, 28,  196, 76, -28, 5, -15, 9,   185, 102, -28, 3, -10, -7, 171, 127, -25, 0, -5,
       -18, 151, 151, -18, -5,  0, -25, 127, 171, -7, -10, 3, -28, 102, 185, 9,   -15, 5, -28, 76, 196, 28,  -21},
    },

    // Band_8_1351
    {
      // Scaler inc range
      {1350, 1351},
      // Coefficients 4 taps
      {39,  178, 39,  0,   22,  183, 64,  -13, 7,   175, 89,  -15, -4,  160, 115, -15,
       -12, 140, 140, -12, -15, 115, 160, -4,  -15, 89,  175, 7,   -13, 64,  183, 22},
      // Coefficients 6 taps
      {-26, 52,  204, 52,  -26, 0, -21, 29,  195, 76, -28, 5, -15, 9,   185, 102, -28, 3, -10, -6, 170, 127, -25, 0, -5,
       -18, 151, 151, -18, -5,  0, -25, 127, 170, -6, -10, 3, -28, 102, 185, 9,   -15, 5, -28, 76, 195, 29,  -21},
    },

    // Band_8_1353
    {
      // Scaler inc range
      {1351, 1353},
      // Coefficients 4 taps
      {39,  178, 39,  0,   22,  183, 64,  -13, 7,   175, 89,  -15, -4,  160, 115, -15,
       -11, 139, 139, -11, -15, 115, 160, -4,  -15, 89,  175, 7,   -13, 64,  183, 22},
      // Coefficients 6 taps
      {-26, 52,  204, 52,  -26, 0,   -21, 29,  195, 76,  -28, 5,   -15, 9,   185, 102,
       -28, 3,   -10, -6,  171, 127, -25, -1,  -5,  -17, 150, 150, -17, -5,  -1,  -25,
       127, 171, -6,  -10, 3,   -28, 102, 185, 9,   -15, 5,   -28, 76,  195, 29,  -21},
    },

    // Band_8_1355
    {
      // Scaler inc range
      {1353, 1355},
      // Coefficients 4 taps
      {39,  178, 39,  0,   22,  183, 64,  -13, 7,   175, 89,  -15, -4,  160, 115, -15,
       -11, 139, 139, -11, -15, 115, 160, -4,  -15, 89,  175, 7,   -13, 64,  183, 22},
      // Coefficients 6 taps
      {-26, 53,  202, 53,  -26, 0,   -21, 29,  195, 76,  -28, 5,   -16, 10,  185, 102,
       -28, 3,   -10, -6,  170, 127, -24, -1,  -5,  -17, 150, 150, -17, -5,  -1,  -24,
       127, 170, -6,  -10, 3,   -28, 102, 185, 10,  -16, 5,   -28, 76,  195, 29,  -21},
    },

    // Band_8_1357
    {
      // Scaler inc range
      {1355, 1357},
      // Coefficients 4 taps
      {40,  176, 40,  0,   22,  183, 64,  -13, 7,   175, 89,  -15, -4,  160, 115, -15,
       -11, 139, 139, -11, -15, 115, 160, -4,  -15, 89,  175, 7,   -13, 64,  183, 22},
      // Coefficients 6 taps
      {-26, 53,  202, 53,  -26, 0,   -21, 29,  195, 76,  -28, 5,   -16, 10,  186, 102,
       -28, 2,   -10, -6,  170, 127, -24, -1,  -5,  -17, 150, 150, -17, -5,  -1,  -24,
       127, 170, -6,  -10, 2,   -28, 102, 186, 10,  -16, 5,   -28, 76,  195, 29,  -21},
    },

    // Band_8_1358
    {
      // Scaler inc range
      {1357, 1358},
      // Coefficients 4 taps
      {40,  176, 40,  0,   23,  182, 64,  -13, 7,   175, 89,  -15, -4,  160, 115, -15,
       -11, 139, 139, -11, -15, 115, 160, -4,  -15, 89,  175, 7,   -13, 64,  182, 23},
      // Coefficients 6 taps
      {-26, 53,  202, 53,  -26, 0,   -21, 29,  196, 76,  -28, 4,   -16, 10,  186, 102,
       -28, 2,   -10, -6,  170, 127, -24, -1,  -5,  -17, 150, 150, -17, -5,  -1,  -24,
       127, 170, -6,  -10, 2,   -28, 102, 186, 10,  -16, 4,   -28, 76,  196, 29,  -21},
    },

    // Band_8_1360
    {
      // Scaler inc range
      {1358, 1360},
      // Coefficients 4 taps
      {40,  176, 40,  0,   23,  182, 64,  -13, 8,   174, 89,  -15, -4,  159, 115, -14,
       -11, 139, 139, -11, -14, 115, 159, -4,  -15, 89,  174, 8,   -13, 64,  182, 23},
      // Coefficients 6 taps
      {-26, 53,  202, 53,  -26, 0,   -21, 29,  195, 77,  -28, 4,   -16, 10,  185, 102,
       -27, 2,   -10, -5,  169, 127, -24, -1,  -5,  -17, 150, 150, -17, -5,  -1,  -24,
       127, 169, -5,  -10, 2,   -27, 102, 185, 10,  -16, 4,   -28, 77,  195, 29,  -21},
    },

    // Band_8_1362
    {
      // Scaler inc range
      {1360, 1362},
      // Coefficients 4 taps
      {40,  176, 40,  0,   23,  182, 64,  -13, 8,   174, 89,  -15, -4,  159, 115, -14,
       -11, 139, 139, -11, -14, 115, 159, -4,  -15, 89,  174, 8,   -13, 64,  182, 23},
      // Coefficients 6 taps
      {-26, 53,  202, 53,  -26, 0,   -21, 30,  194, 77,  -28, 4,   -16, 10,  185, 102,
       -27, 2,   -10, -5,  169, 127, -24, -1,  -5,  -17, 150, 150, -17, -5,  -1,  -24,
       127, 169, -5,  -10, 2,   -27, 102, 185, 10,  -16, 4,   -28, 77,  194, 30,  -21},
    },

    // Band_8_1364
    {
      // Scaler inc range
      {1362, 1364},
      // Coefficients 4 taps
      {40,  176, 40,  0,   23,  182, 64,  -13, 8,   174, 89,  -15, -3,  158, 115, -14,
       -11, 139, 139, -11, -14, 115, 158, -3,  -15, 89,  174, 8,   -13, 64,  182, 23},
      // Coefficients 6 taps
      {-26, 53,  202, 53,  -26, 0,   -21, 30,  193, 77,  -27, 4,   -16, 11,  184, 102,
       -27, 2,   -11, -5,  170, 127, -24, -1,  -5,  -16, 149, 149, -16, -5,  -1,  -24,
       127, 170, -5,  -11, 2,   -27, 102, 184, 11,  -16, 4,   -27, 77,  193, 30,  -21},
    },

    // Band_8_1365
    {
      // Scaler inc range
      {1364, 1365},
      // Coefficients 4 taps
      {40,  176, 40,  0,   23,  182, 64,  -13, 8,   174, 89,  -15, -3,  158, 115, -14,
       -11, 139, 139, -11, -14, 115, 158, -3,  -15, 89,  174, 8,   -13, 64,  182, 23},
      // Coefficients 6 taps
      {-26, 53,  202, 53,  -26, 0,   -21, 30,  193, 77,  -27, 4,   -16, 11,  184, 102,
       -27, 2,   -11, -5,  170, 127, -24, -1,  -6,  -16, 150, 150, -16, -6,  -1,  -24,
       127, 170, -5,  -11, 2,   -27, 102, 184, 11,  -16, 4,   -27, 77,  193, 30,  -21},
    },

    // Band_8_1367
    {
      // Scaler inc range
      {1365, 1367},
      // Coefficients 4 taps
      {40,  176, 40,  0,   23,  182, 64,  -13, 8,   174, 89,  -15, -3,  158, 115, -14,
       -10, 138, 138, -10, -14, 115, 158, -3,  -15, 89,  174, 8,   -13, 64,  182, 23},
      // Coefficients 6 taps
      {-26, 54,  200, 54,  -26, 0,   -21, 30,  193, 77,  -27, 4,   -16, 11,  184, 102,
       -27, 2,   -11, -5,  169, 127, -23, -1,  -6,  -16, 150, 150, -16, -6,  -1,  -23,
       127, 169, -5,  -11, 2,   -27, 102, 184, 11,  -16, 4,   -27, 77,  193, 30,  -21},
    },

    // Band_8_1369
    {
      // Scaler inc range
      {1367, 1369},
      // Coefficients 4 taps
      {40,  176, 40,  0,   23,  181, 65,  -13, 8,   174, 89,  -15, -3,  158, 115, -14,
       -10, 138, 138, -10, -14, 115, 158, -3,  -15, 89,  174, 8,   -13, 65,  181, 23},
      // Coefficients 6 taps
      {-26, 54,  200, 54,  -26, 0,   -21, 30,  193, 77,  -27, 4,   -16, 11,  184, 102,
       -27, 2,   -11, -4,  168, 127, -23, -1,  -6,  -16, 150, 150, -16, -6,  -1,  -23,
       127, 168, -4,  -11, 2,   -27, 102, 184, 11,  -16, 4,   -27, 77,  193, 30,  -21},
    },

    // Band_8_1371
    {
      // Scaler inc range
      {1369, 1371},
      // Coefficients 4 taps
      {40,  176, 40,  0,   24,  180, 65,  -13, 8,   174, 89,  -15, -3,  158, 115, -14,
       -10, 138, 138, -10, -14, 115, 158, -3,  -15, 89,  174, 8,   -13, 65,  180, 24},
      // Coefficients 6 taps
      {-26, 54,  200, 54,  -26, 0,   -21, 31,  192, 77,  -27, 4,   -16, 11,  184, 102,
       -27, 2,   -11, -4,  169, 127, -23, -2,  -6,  -16, 150, 150, -16, -6,  -2,  -23,
       127, 169, -4,  -11, 2,   -27, 102, 184, 11,  -16, 4,   -27, 77,  192, 31,  -21},
    },

    // Band_8_1372
    {
      // Scaler inc range
      {1371, 1372},
      // Coefficients 4 taps
      {41,  174, 41,  0,   24,  180, 65,  -13, 9,   172, 89,  -14, -3,  158, 115, -14,
       -10, 138, 138, -10, -14, 115, 158, -3,  -14, 89,  172, 9,   -13, 65,  180, 24},
      // Coefficients 6 taps
      {-26, 54,  200, 54,  -26, 0,   -21, 31,  192, 77,  -27, 4,   -16, 12,  183, 102,
       -27, 2,   -11, -4,  169, 127, -23, -2,  -6,  -15, 149, 149, -15, -6,  -2,  -23,
       127, 169, -4,  -11, 2,   -27, 102, 183, 12,  -16, 4,   -27, 77,  192, 31,  -21},
    },

    // Band_8_1374
    {
      // Scaler inc range
      {1372, 1374},
      // Coefficients 4 taps
      {41,  174, 41,  0,   24,  180, 65,  -13, 9,   172, 89,  -14, -3,  159, 114, -14,
       -10, 138, 138, -10, -14, 114, 159, -3,  -14, 89,  172, 9,   -13, 65,  180, 24},
      // Coefficients 6 taps
      {-26, 54,  200, 54,  -26, 0,   -21, 31,  192, 77,  -27, 4,   -16, 12,  183, 102,
       -27, 2,   -11, -4,  169, 127, -23, -2,  -6,  -15, 149, 149, -15, -6,  -2,  -23,
       127, 169, -4,  -11, 2,   -27, 102, 183, 12,  -16, 4,   -27, 77,  192, 31,  -21},
    },

    // Band_8_1376
    {
      // Scaler inc range
      {1374, 1376},
      // Coefficients 4 taps
      {41,  174, 41,  0,   24,  180, 65,  -13, 9,   172, 89,  -14, -2,  158, 114, -14,
       -10, 138, 138, -10, -14, 114, 158, -2,  -14, 89,  172, 9,   -13, 65,  180, 24},
      // Coefficients 6 taps
      {-26, 54,  200, 54,  -26, 0,   -21, 31,  191, 78,  -27, 4,   -16, 12,  183, 102,
       -27, 2,   -11, -4,  169, 127, -23, -2,  -6,  -15, 149, 149, -15, -6,  -2,  -23,
       127, 169, -4,  -11, 2,   -27, 102, 183, 12,  -16, 4,   -27, 78,  191, 31,  -21},
    },

    // Band_8_1378
    {
      // Scaler inc range
      {1376, 1378},
      // Coefficients 4 taps
      {41,  174, 41,  0,   24,  180, 65,  -13, 9,   172, 89,  -14, -2,  158, 114, -14,
       -10, 138, 138, -10, -14, 114, 158, -2,  -14, 89,  172, 9,   -13, 65,  180, 24},
      // Coefficients 6 taps
      {-26, 54,  200, 54,  -26, 0,   -21, 31,  191, 78,  -27, 4,   -16, 12,  182, 103,
       -26, 1,   -11, -3,  168, 127, -23, -2,  -6,  -15, 149, 149, -15, -6,  -2,  -23,
       127, 168, -3,  -11, 1,   -26, 103, 182, 12,  -16, 4,   -27, 78,  191, 31,  -21},
    },

    // Band_8_1380
    {
      // Scaler inc range
      {1378, 1380},
      // Coefficients 4 taps
      {41,  174, 41,  0,   24,  180, 65,  -13, 9,   172, 89,  -14, -2,  157, 114, -13,
       -10, 138, 138, -10, -13, 114, 157, -2,  -14, 89,  172, 9,   -13, 65,  180, 24},
      // Coefficients 6 taps
      {-26, 55,  198, 55,  -26, 0,   -21, 32,  190, 78,  -27, 4,   -17, 12,  183, 103,
       -26, 1,   -11, -3,  167, 127, -22, -2,  -6,  -15, 149, 149, -15, -6,  -2,  -22,
       127, 167, -3,  -11, 1,   -26, 103, 183, 12,  -17, 4,   -27, 78,  190, 32,  -21},
    },

    // Band_8_1382
    {
      // Scaler inc range
      {1380, 1382},
      // Coefficients 4 taps
      {41,  174, 41,  0,   24,  180, 65,  -13, 9,   172, 89,  -14, -2,  157, 114, -13,
       -10, 138, 138, -10, -13, 114, 157, -2,  -14, 89,  172, 9,   -13, 65,  180, 24},
      // Coefficients 6 taps
      {-26, 55,  198, 55,  -26, 0,   -21, 32,  190, 78,  -27, 4,   -17, 13,  182, 103,
       -26, 1,   -11, -3,  167, 127, -22, -2,  -6,  -15, 149, 149, -15, -6,  -2,  -22,
       127, 167, -3,  -11, 1,   -26, 103, 182, 13,  -17, 4,   -27, 78,  190, 32,  -21},
    },

    // Band_8_1383
    {
      // Scaler inc range
      {1382, 1383},
      // Coefficients 4 taps
      {41, 174, 41,  0,  25,  179, 65,  -13, 9,   172, 89,  -14, -2,  157, 114, -13,
       -9, 137, 137, -9, -13, 114, 157, -2,  -14, 89,  172, 9,   -13, 65,  179, 25},
      // Coefficients 6 taps
      {-26, 55,  198, 55,  -26, 0,   -21, 32,  191, 78,  -27, 3,   -17, 13,  182, 103,
       -26, 1,   -12, -3,  168, 127, -22, -2,  -7,  -14, 149, 149, -14, -7,  -2,  -22,
       127, 168, -3,  -12, 1,   -26, 103, 182, 13,  -17, 3,   -27, 78,  191, 32,  -21},
    },

    // Band_8_1385
    {
      // Scaler inc range
      {1383, 1385},
      // Coefficients 4 taps
      {41, 174, 41,  0,  25,  179, 65,  -13, 10,  171, 89,  -14, -2,  157, 114, -13,
       -9, 137, 137, -9, -13, 114, 157, -2,  -14, 89,  171, 10,  -13, 65,  179, 25},
      // Coefficients 6 taps
      {-26, 55,  198, 55,  -26, 0,   -21, 32,  191, 78,  -27, 3,   -17, 13,  182, 103,
       -26, 1,   -12, -3,  168, 127, -22, -2,  -7,  -14, 149, 149, -14, -7,  -2,  -22,
       127, 168, -3,  -12, 1,   -26, 103, 182, 13,  -17, 3,   -27, 78,  191, 32,  -21},
    },

    // Band_8_1387
    {
      // Scaler inc range
      {1385, 1387},
      // Coefficients 4 taps
      {42, 172, 42,  0,  25,  179, 65,  -13, 10,  170, 90,  -14, -2,  157, 114, -13,
       -9, 137, 137, -9, -13, 114, 157, -2,  -14, 90,  170, 10,  -13, 65,  179, 25},
      // Coefficients 6 taps
      {-26, 55,  198, 55,  -26, 0,   -22, 32,  192, 78,  -27, 3,   -17, 13,  182, 103,
       -26, 1,   -12, -2,  167, 127, -22, -2,  -7,  -14, 149, 149, -14, -7,  -2,  -22,
       127, 167, -2,  -12, 1,   -26, 103, 182, 13,  -17, 3,   -27, 78,  192, 32,  -22},
    },

    // Band_8_1389
    {
      // Scaler inc range
      {1387, 1389},
      // Coefficients 4 taps
      {42, 172, 42,  0,  25,  179, 65,  -13, 10,  170, 90,  -14, -1,  156, 114, -13,
       -9, 137, 137, -9, -13, 114, 156, -1,  -14, 90,  170, 10,  -13, 65,  179, 25},
      // Coefficients 6 taps
      {-25, 55,  196, 55,  -25, 0,   -22, 33,  191, 78,  -27, 3,   -17, 14,  181, 103,
       -26, 1,   -12, -2,  168, 127, -22, -3,  -7,  -14, 149, 149, -14, -7,  -3,  -22,
       127, 168, -2,  -12, 1,   -26, 103, 181, 14,  -17, 3,   -27, 78,  191, 33,  -22},
    },

    // Band_8_1391
    {
      // Scaler inc range
      {1389, 1391},
      // Coefficients 4 taps
      {42, 172, 42,  0,  25,  179, 65,  -13, 10,  170, 90,  -14, -1,  156, 114, -13,
       -9, 137, 137, -9, -13, 114, 156, -1,  -14, 90,  170, 10,  -13, 65,  179, 25},
      // Coefficients 6 taps
      {-25, 55,  196, 55,  -25, 0,   -22, 33,  191, 78,  -27, 3,   -17, 14,  181, 103,
       -26, 1,   -12, -2,  168, 127, -22, -3,  -7,  -14, 149, 149, -14, -7,  -3,  -22,
       127, 168, -2,  -12, 1,   -26, 103, 181, 14,  -17, 3,   -27, 78,  191, 33,  -22},
    },

    // Band_8_1393
    {
      // Scaler inc range
      {1391, 1393},
      // Coefficients 4 taps
      {42, 172, 42,  0,  25,  178, 66,  -13, 10,  170, 90,  -14, -1,  156, 114, -13,
       -9, 137, 137, -9, -13, 114, 156, -1,  -14, 90,  170, 10,  -13, 66,  178, 25},
      // Coefficients 6 taps
      {-25, 56,  194, 56,  -25, 0,   -22, 33,  191, 78,  -27, 3,   -17, 14,  181, 103,
       -26, 1,   -12, -2,  167, 127, -21, -3,  -7,  -13, 148, 148, -13, -7,  -3,  -21,
       127, 167, -2,  -12, 1,   -26, 103, 181, 14,  -17, 3,   -27, 78,  191, 33,  -22},
    },

    // Band_8_1394
    {
      // Scaler inc range
      {1393, 1394},
      // Coefficients 4 taps
      {42, 172, 42,  0,  25,  178, 66,  -13, 10,  170, 90,  -14, -1,  156, 114, -13,
       -9, 137, 137, -9, -13, 114, 156, -1,  -14, 90,  170, 10,  -13, 66,  178, 25},
      // Coefficients 6 taps
      {-25, 56,  194, 56,  -25, 0,   -22, 33,  190, 79,  -27, 3,   -17, 14,  181, 103,
       -26, 1,   -12, -1,  166, 127, -21, -3,  -7,  -13, 148, 148, -13, -7,  -3,  -21,
       127, 166, -1,  -12, 1,   -26, 103, 181, 14,  -17, 3,   -27, 79,  190, 33,  -22},
    },

    // Band_8_1396
    {
      // Scaler inc range
      {1394, 1396},
      // Coefficients 4 taps
      {42, 172, 42,  0,  26,  177, 66,  -13, 11,  169, 90,  -14, -1,  156, 114, -13,
       -9, 137, 137, -9, -13, 114, 156, -1,  -14, 90,  169, 11,  -13, 66,  177, 26},
      // Coefficients 6 taps
      {-25, 56,  194, 56,  -25, 0,   -22, 33,  190, 79,  -27, 3,   -17, 14,  180, 103,
       -25, 1,   -12, -1,  166, 127, -21, -3,  -7,  -13, 148, 148, -13, -7,  -3,  -21,
       127, 166, -1,  -12, 1,   -25, 103, 180, 14,  -17, 3,   -27, 79,  190, 33,  -22},
    },

    // Band_8_1398
    {
      // Scaler inc range
      {1396, 1398},
      // Coefficients 4 taps
      {42, 172, 42,  0,  26,  177, 66,  -13, 11,  169, 90,  -14, -1,  156, 114, -13,
       -8, 136, 136, -8, -13, 114, 156, -1,  -14, 90,  169, 11,  -13, 66,  177, 26},
      // Coefficients 6 taps
      {-25, 56,  194, 56,  -25, 0,   -22, 34,  188, 79,  -26, 3,   -17, 15,  180, 103,
       -25, 0,   -12, -1,  166, 127, -21, -3,  -7,  -13, 148, 148, -13, -7,  -3,  -21,
       127, 166, -1,  -12, 0,   -25, 103, 180, 15,  -17, 3,   -26, 79,  188, 34,  -22},
    },

    // Band_8_1400
    {
      // Scaler inc range
      {1398, 1400},
      // Coefficients 4 taps
      {42, 172, 42,  0,  26,  177, 66,  -13, 11,  169, 90,  -14, -1,  155, 114, -12,
       -8, 136, 136, -8, -12, 114, 155, -1,  -14, 90,  169, 11,  -13, 66,  177, 26},
      // Coefficients 6 taps
      {-25, 56,  194, 56,  -25, 0,   -22, 34,  188, 79,  -26, 3,   -17, 15,  180, 103,
       -25, 0,   -12, -1,  166, 127, -21, -3,  -7,  -13, 148, 148, -13, -7,  -3,  -21,
       127, 166, -1,  -12, 0,   -25, 103, 180, 15,  -17, 3,   -26, 79,  188, 34,  -22},
    },

    // Band_8_1402
    {
      // Scaler inc range
      {1400, 1402},
      // Coefficients 4 taps
      {42, 172, 42,  0,  26,  177, 66,  -13, 11,  169, 90,  -14, 0,   154, 114, -12,
       -8, 136, 136, -8, -12, 114, 154, 0,   -14, 90,  169, 11,  -13, 66,  177, 26},
      // Coefficients 6 taps
      {-25, 56,  194, 56,  -25, 0,   -22, 34,  188, 79,  -26, 3,   -17, 15,  180, 103,
       -25, 0,   -12, -1,  166, 127, -21, -3,  -8,  -13, 149, 149, -13, -8,  -3,  -21,
       127, 166, -1,  -12, 0,   -25, 103, 180, 15,  -17, 3,   -26, 79,  188, 34,  -22},
    },

    // Band_8_1404
    {
      // Scaler inc range
      {1402, 1404},
      // Coefficients 4 taps
      {43, 170, 43,  0,  26,  177, 66,  -13, 11,  168, 90,  -13, 0,   154, 114, -12,
       -8, 136, 136, -8, -12, 114, 154, 0,   -13, 90,  168, 11,  -13, 66,  177, 26},
      // Coefficients 6 taps
      {-25, 56,  194, 56,  -25, 0,   -22, 34,  188, 79,  -26, 3,   -17, 15,  180, 103,
       -25, 0,   -12, 0,   164, 127, -20, -3,  -8,  -12, 148, 148, -12, -8,  -3,  -20,
       127, 164, 0,   -12, 0,   -25, 103, 180, 15,  -17, 3,   -26, 79,  188, 34,  -22},
    },

    // Band_8_1406
    {
      // Scaler inc range
      {1404, 1406},
      // Coefficients 4 taps
      {43, 170, 43,  0,  26,  176, 66,  -12, 11,  168, 90,  -13, 0,   154, 114, -12,
       -8, 136, 136, -8, -12, 114, 154, 0,   -13, 90,  168, 11,  -12, 66,  176, 26},
      // Coefficients 6 taps
      {-25, 57,  192, 57,  -25, 0,   -22, 34,  188, 79,  -26, 3,   -17, 15,  180, 103,
       -25, 0,   -13, 0,   165, 127, -20, -3,  -8,  -12, 148, 148, -12, -8,  -3,  -20,
       127, 165, 0,   -13, 0,   -25, 103, 180, 15,  -17, 3,   -26, 79,  188, 34,  -22},
    },

    // Band_8_1407
    {
      // Scaler inc range
      {1406, 1407},
      // Coefficients 4 taps
      {43, 170, 43,  0,  26,  176, 66,  -12, 11,  168, 90,  -13, 0,   154, 114, -12,
       -8, 136, 136, -8, -12, 114, 154, 0,   -13, 90,  168, 11,  -12, 66,  176, 26},
      // Coefficients 6 taps
      {-25, 57,  192, 57,  -25, 0,   -22, 34,  188, 79,  -26, 3,   -17, 16,  179, 103,
       -25, 0,   -13, 0,   165, 127, -20, -3,  -8,  -12, 148, 148, -12, -8,  -3,  -20,
       127, 165, 0,   -13, 0,   -25, 103, 179, 16,  -17, 3,   -26, 79,  188, 34,  -22},
    },

    // Band_8_1409
    {
      // Scaler inc range
      {1407, 1409},
      // Coefficients 4 taps
      {43, 170, 43,  0,  27,  175, 66,  -12, 12,  167, 90,  -13, 0,   154, 114, -12,
       -8, 136, 136, -8, -12, 114, 154, 0,   -13, 90,  167, 12,  -12, 66,  175, 27},
      // Coefficients 6 taps
      {-25, 57,  192, 57,  -25, 0,   -22, 35,  188, 79,  -26, 2,   -18, 16,  180, 103,
       -25, 0,   -13, 0,   167, 126, -20, -4,  -8,  -12, 148, 148, -12, -8,  -4,  -20,
       126, 167, 0,   -13, 0,   -25, 103, 180, 16,  -18, 2,   -26, 79,  188, 35,  -22},
    },

    // Band_8_1411
    {
      // Scaler inc range
      {1409, 1411},
      // Coefficients 4 taps
      {43, 170, 43,  0,  27,  175, 66,  -12, 12,  167, 90,  -13, 0,   155, 113, -12,
       -8, 136, 136, -8, -12, 113, 155, 0,   -13, 90,  167, 12,  -12, 66,  175, 27},
      // Coefficients 6 taps
      {-25, 57,  192, 57,  -25, 0,   -22, 35,  188, 79,  -26, 2,   -18, 16,  180, 103,
       -25, 0,   -13, 0,   167, 126, -20, -4,  -8,  -12, 148, 148, -12, -8,  -4,  -20,
       126, 167, 0,   -13, 0,   -25, 103, 180, 16,  -18, 2,   -26, 79,  188, 35,  -22},
    },

    // Band_8_1413
    {
      // Scaler inc range
      {1411, 1413},
      // Coefficients 4 taps
      {43, 170, 43,  0,  27,  175, 66,  -12, 12,  167, 90,  -13, 0,   155, 113, -12,
       -7, 135, 135, -7, -12, 113, 155, 0,   -13, 90,  167, 12,  -12, 66,  175, 27},
      // Coefficients 6 taps
      {-25, 57,  192, 57,  -25, 0,   -22, 35,  188, 79,  -26, 2,   -18, 16,  179, 103,
       -24, 0,   -13, 1,   166, 126, -20, -4,  -8,  -11, 147, 147, -11, -8,  -4,  -20,
       126, 166, 1,   -13, 0,   -24, 103, 179, 16,  -18, 2,   -26, 79,  188, 35,  -22},
    },

    // Band_8_1415
    {
      // Scaler inc range
      {1413, 1415},
      // Coefficients 4 taps
      {43, 170, 43,  0,  27,  175, 66,  -12, 12,  167, 90,  -13, 1,   154, 113, -12,
       -7, 135, 135, -7, -12, 113, 154, 1,   -13, 90,  167, 12,  -12, 66,  175, 27},
      // Coefficients 6 taps
      {-25, 57,  192, 57,  -25, 0,   -22, 35,  187, 80,  -26, 2,   -18, 16,  179, 103,
       -24, 0,   -13, 1,   166, 126, -20, -4,  -8,  -11, 147, 147, -11, -8,  -4,  -20,
       126, 166, 1,   -13, 0,   -24, 103, 179, 16,  -18, 2,   -26, 80,  187, 35,  -22},
    },

    // Band_8_1417
    {
      // Scaler inc range
      {1415, 1417},
      // Coefficients 4 taps
      {43, 170, 43,  0,  27,  174, 67,  -12, 12,  167, 90,  -13, 1,   154, 113, -12,
       -7, 135, 135, -7, -12, 113, 154, 1,   -13, 90,  167, 12,  -12, 67,  174, 27},
      // Coefficients 6 taps
      {-25, 57,  192, 57,  -25, 0,   -22, 35,  187, 80,  -26, 2,   -18, 17,  178, 103,
       -24, 0,   -13, 1,   165, 126, -19, -4,  -8,  -11, 147, 147, -11, -8,  -4,  -19,
       126, 165, 1,   -13, 0,   -24, 103, 178, 17,  -18, 2,   -26, 80,  187, 35,  -22},
    },

    // Band_8_1419
    {
      // Scaler inc range
      {1417, 1419},
      // Coefficients 4 taps
      {44, 168, 44,  0,  27,  174, 67,  -12, 12,  167, 90,  -13, 1,   153, 113, -11,
       -7, 135, 135, -7, -11, 113, 153, 1,   -13, 90,  167, 12,  -12, 67,  174, 27},
      // Coefficients 6 taps
      {-25, 57,  192, 57,  -25, 0,   -22, 36,  186, 80,  -26, 2,   -18, 17,  179, 103,
       -24, -1,  -13, 1,   165, 126, -19, -4,  -8,  -11, 147, 147, -11, -8,  -4,  -19,
       126, 165, 1,   -13, -1,  -24, 103, 179, 17,  -18, 2,   -26, 80,  186, 36,  -22},
    },

    // Band_8_1421
    {
      // Scaler inc range
      {1419, 1421},
      // Coefficients 4 taps
      {44, 168, 44,  0,  27,  174, 67,  -12, 12,  167, 90,  -13, 1,   153, 113, -11,
       -7, 135, 135, -7, -11, 113, 153, 1,   -13, 90,  167, 12,  -12, 67,  174, 27},
      // Coefficients 6 taps
      {-25, 58,  190, 58,  -25, 0,   -22, 36,  186, 80,  -26, 2,   -18, 17,  179, 103,
       -24, -1,  -13, 1,   165, 126, -19, -4,  -8,  -11, 147, 147, -11, -8,  -4,  -19,
       126, 165, 1,   -13, -1,  -24, 103, 179, 17,  -18, 2,   -26, 80,  186, 36,  -22},
    },

    // Band_8_1423
    {
      // Scaler inc range
      {1421, 1423},
      // Coefficients 4 taps
      {44, 168, 44,  0,  28,  173, 67,  -12, 13,  166, 90,  -13, 1,   153, 113, -11,
       -7, 135, 135, -7, -11, 113, 153, 1,   -13, 90,  166, 13,  -12, 67,  173, 28},
      // Coefficients 6 taps
      {-25, 58,  190, 58,  -25, 0,   -22, 36,  186, 80,  -26, 2,   -18, 17,  179, 103,
       -24, -1,  -13, 2,   164, 126, -19, -4,  -9,  -11, 148, 148, -11, -9,  -4,  -19,
       126, 164, 2,   -13, -1,  -24, 103, 179, 17,  -18, 2,   -26, 80,  186, 36,  -22},
    },

    // Band_8_1425
    {
      // Scaler inc range
      {1423, 1425},
      // Coefficients 4 taps
      {44, 168, 44,  0,  28,  173, 67,  -12, 13,  166, 90,  -13, 1,   153, 113, -11,
       -7, 135, 135, -7, -11, 113, 153, 1,   -13, 90,  166, 13,  -12, 67,  173, 28},
      // Coefficients 6 taps
      {-25, 58,  190, 58,  -25, 0,   -22, 36,  185, 80,  -25, 2,   -18, 17,  178, 104,
       -24, -1,  -13, 2,   164, 126, -19, -4,  -9,  -10, 147, 147, -10, -9,  -4,  -19,
       126, 164, 2,   -13, -1,  -24, 104, 178, 17,  -18, 2,   -25, 80,  185, 36,  -22},
    },

    // Band_8_1427
    {
      // Scaler inc range
      {1425, 1427},
      // Coefficients 4 taps
      {44, 168, 44,  0,  28,  173, 67,  -12, 13,  166, 90,  -13, 1,   153, 113, -11,
       -7, 135, 135, -7, -11, 113, 153, 1,   -13, 90,  166, 13,  -12, 67,  173, 28},
      // Coefficients 6 taps
      {-25, 58,  190, 58,  -25, 0,   -22, 36,  185, 80,  -25, 2,   -18, 18,  177, 104,
       -24, -1,  -13, 2,   164, 126, -19, -4,  -9,  -10, 147, 147, -10, -9,  -4,  -19,
       126, 164, 2,   -13, -1,  -24, 104, 177, 18,  -18, 2,   -25, 80,  185, 36,  -22},
    },

    // Band_8_1429
    {
      // Scaler inc range
      {1427, 1429},
      // Coefficients 4 taps
      {44, 168, 44,  0,  28,  173, 67,  -12, 13,  166, 90,  -13, 2,   152, 113, -11,
       -6, 134, 134, -6, -11, 113, 152, 2,   -13, 90,  166, 13,  -12, 67,  173, 28},
      // Coefficients 6 taps
      {-25, 58,  190, 58,  -25, 0,   -22, 36,  185, 80,  -25, 2,   -18, 18,  176, 104,
       -23, -1,  -13, 2,   164, 126, -18, -5,  -9,  -10, 147, 147, -10, -9,  -5,  -18,
       126, 164, 2,   -13, -1,  -23, 104, 176, 18,  -18, 2,   -25, 80,  185, 36,  -22},
    },

    // Band_8_1431
    {
      // Scaler inc range
      {1429, 1431},
      // Coefficients 4 taps
      {44, 168, 44,  0,  28,  173, 67,  -12, 13,  166, 90,  -13, 2,   152, 113, -11,
       -6, 134, 134, -6, -11, 113, 152, 2,   -13, 90,  166, 13,  -12, 67,  173, 28},
      // Coefficients 6 taps
      {-25, 58,  190, 58,  -25, 0,   -22, 37,  184, 80,  -25, 2,   -18, 18,  176, 104,
       -23, -1,  -14, 2,   165, 126, -18, -5,  -9,  -10, 147, 147, -10, -9,  -5,  -18,
       126, 165, 2,   -14, -1,  -23, 104, 176, 18,  -18, 2,   -25, 80,  184, 37,  -22},
    },

    // Band_8_1432
    {
      // Scaler inc range
      {1431, 1432},
      // Coefficients 4 taps
      {44, 168, 44,  0,  28,  173, 67,  -12, 13,  165, 90,  -12, 2,   152, 113, -11,
       -6, 134, 134, -6, -11, 113, 152, 2,   -12, 90,  165, 13,  -12, 67,  173, 28},
      // Coefficients 6 taps
      {-25, 58,  190, 58,  -25, 0,   -22, 37,  184, 80,  -25, 2,   -18, 18,  176, 104,
       -23, -1,  -14, 3,   164, 126, -18, -5,  -9,  -10, 147, 147, -10, -9,  -5,  -18,
       126, 164, 3,   -14, -1,  -23, 104, 176, 18,  -18, 2,   -25, 80,  184, 37,  -22},
    },

    // Band_8_1434
    {
      // Scaler inc range
      {1432, 1434},
      // Coefficients 4 taps
      {44, 168, 44,  0,  28,  173, 67,  -12, 13,  165, 90,  -12, 2,   152, 113, -11,
       -6, 134, 134, -6, -11, 113, 152, 2,   -12, 90,  165, 13,  -12, 67,  173, 28},
      // Coefficients 6 taps
      {-25, 58,  190, 58,  -25, 0,   -22, 37,  185, 80,  -25, 1,   -18, 18,  176, 104,
       -23, -1,  -14, 3,   164, 126, -18, -5,  -9,  -9,  146, 146, -9,  -9,  -5,  -18,
       126, 164, 3,   -14, -1,  -23, 104, 176, 18,  -18, 1,   -25, 80,  185, 37,  -22},
    },

    // Band_8_1436
    {
      // Scaler inc range
      {1434, 1436},
      // Coefficients 4 taps
      {45, 166, 45,  0,  29,  172, 67,  -12, 14,  164, 90,  -12, 2,   152, 113, -11,
       -6, 134, 134, -6, -11, 113, 152, 2,   -12, 90,  164, 14,  -12, 67,  172, 29},
      // Coefficients 6 taps
      {-25, 59,  188, 59,  -25, 0,   -22, 37,  184, 81,  -25, 1,   -18, 19,  175, 104,
       -23, -1,  -14, 3,   164, 126, -18, -5,  -9,  -9,  146, 146, -9,  -9,  -5,  -18,
       126, 164, 3,   -14, -1,  -23, 104, 175, 19,  -18, 1,   -25, 81,  184, 37,  -22},
    },

    // Band_8_1438
    {
      // Scaler inc range
      {1436, 1438},
      // Coefficients 4 taps
      {45, 166, 45,  0,  29,  172, 67,  -12, 14,  164, 90,  -12, 2,   151, 113, -10,
       -6, 134, 134, -6, -10, 113, 151, 2,   -12, 90,  164, 14,  -12, 67,  172, 29},
      // Coefficients 6 taps
      {-25, 59,  188, 59,  -25, 0,   -22, 37,  184, 81,  -25, 1,   -18, 19,  175, 104,
       -23, -1,  -14, 3,   164, 126, -18, -5,  -9,  -9,  146, 146, -9,  -9,  -5,  -18,
       126, 164, 3,   -14, -1,  -23, 104, 175, 19,  -18, 1,   -25, 81,  184, 37,  -22},
    },

    // Band_8_1440
    {
      // Scaler inc range
      {1438, 1440},
      // Coefficients 4 taps
      {45, 166, 45,  0,  29,  172, 67,  -12, 14,  164, 90,  -12, 2,   151, 113, -10,
       -6, 134, 134, -6, -10, 113, 151, 2,   -12, 90,  164, 14,  -12, 67,  172, 29},
      // Coefficients 6 taps
      {-25, 59,  188, 59,  -25, 0,   -22, 37,  184, 81,  -25, 1,   -18, 19,  175, 104,
       -23, -1,  -14, 3,   163, 126, -17, -5,  -9,  -9,  146, 146, -9,  -9,  -5,  -17,
       126, 163, 3,   -14, -1,  -23, 104, 175, 19,  -18, 1,   -25, 81,  184, 37,  -22},
    },

    // Band_8_1442
    {
      // Scaler inc range
      {1440, 1442},
      // Coefficients 4 taps
      {45, 166, 45,  0,  29,  172, 67,  -12, 14,  164, 90,  -12, 3,   150, 113, -10,
       -6, 134, 134, -6, -10, 113, 150, 3,   -12, 90,  164, 14,  -12, 67,  172, 29},
      // Coefficients 6 taps
      {-25, 59,  188, 59,  -25, 0,   -22, 38,  183, 81,  -25, 1,   -18, 19,  176, 104,
       -23, -2,  -14, 4,   162, 126, -17, -5,  -9,  -9,  146, 146, -9,  -9,  -5,  -17,
       126, 162, 4,   -14, -2,  -23, 104, 176, 19,  -18, 1,   -25, 81,  183, 38,  -22},
    },

    // Band_8_1444
    {
      // Scaler inc range
      {1442, 1444},
      // Coefficients 4 taps
      {45, 166, 45,  0,  29,  171, 68,  -12, 14,  164, 90,  -12, 3,   150, 113, -10,
       -5, 133, 133, -5, -10, 113, 150, 3,   -12, 90,  164, 14,  -12, 68,  171, 29},
      // Coefficients 6 taps
      {-24, 59,  186, 59,  -24, 0,   -22, 38,  183, 81,  -25, 1,   -18, 19,  175, 104,
       -22, -2,  -14, 4,   162, 126, -17, -5,  -9,  -8,  145, 145, -8,  -9,  -5,  -17,
       126, 162, 4,   -14, -2,  -22, 104, 175, 19,  -18, 1,   -25, 81,  183, 38,  -22},
    },

    // Band_8_1446
    {
      // Scaler inc range
      {1444, 1446},
      // Coefficients 4 taps
      {45, 166, 45,  0,  29,  171, 68,  -12, 14,  164, 90,  -12, 3,   151, 112, -10,
       -5, 133, 133, -5, -10, 112, 151, 3,   -12, 90,  164, 14,  -12, 68,  171, 29},
      // Coefficients 6 taps
      {-24, 59,  186, 59,  -24, 0,   -22, 38,  183, 81,  -25, 1,   -18, 20,  174, 104,
       -22, -2,  -14, 4,   162, 126, -17, -5,  -10, -8,  146, 146, -8,  -10, -5,  -17,
       126, 162, 4,   -14, -2,  -22, 104, 174, 20,  -18, 1,   -25, 81,  183, 38,  -22},
    },

    // Band_8_1448
    {
      // Scaler inc range
      {1446, 1448},
      // Coefficients 4 taps
      {45, 166, 45,  0,  29,  171, 68,  -12, 15,  163, 90,  -12, 3,   151, 112, -10,
       -5, 133, 133, -5, -10, 112, 151, 3,   -12, 90,  163, 15,  -12, 68,  171, 29},
      // Coefficients 6 taps
      {-24, 59,  186, 59,  -24, 0,   -22, 38,  182, 81,  -24, 1,   -18, 20,  174, 104,
       -22, -2,  -14, 4,   162, 126, -17, -5,  -10, -8,  146, 146, -8,  -10, -5,  -17,
       126, 162, 4,   -14, -2,  -22, 104, 174, 20,  -18, 1,   -24, 81,  182, 38,  -22},
    },

    // Band_8_1450
    {
      // Scaler inc range
      {1448, 1450},
      // Coefficients 4 taps
      {45, 166, 45,  0,  29,  171, 68,  -12, 15,  163, 90,  -12, 3,   151, 112, -10,
       -5, 133, 133, -5, -10, 112, 151, 3,   -12, 90,  163, 15,  -12, 68,  171, 29},
      // Coefficients 6 taps
      {-24, 59,  186, 59,  -24, 0,   -22, 38,  182, 81,  -24, 1,   -19, 20,  175, 104,
       -22, -2,  -14, 4,   163, 126, -17, -6,  -10, -8,  146, 146, -8,  -10, -6,  -17,
       126, 163, 4,   -14, -2,  -22, 104, 175, 20,  -19, 1,   -24, 81,  182, 38,  -22},
    },

    // Band_8_1452
    {
      // Scaler inc range
      {1450, 1452},
      // Coefficients 4 taps
      {45, 166, 45,  0,  30,  170, 68,  -12, 15,  163, 90,  -12, 3,   151, 112, -10,
       -5, 133, 133, -5, -10, 112, 151, 3,   -12, 90,  163, 15,  -12, 68,  170, 30},
      // Coefficients 6 taps
      {-24, 59,  186, 59,  -24, 0,   -22, 39,  181, 81,  -24, 1,   -19, 20,  175, 104,
       -22, -2,  -14, 5,   161, 126, -16, -6,  -10, -8,  146, 146, -8,  -10, -6,  -16,
       126, 161, 5,   -14, -2,  -22, 104, 175, 20,  -19, 1,   -24, 81,  181, 39,  -22},
    },

    // Band_8_1454
    {
      // Scaler inc range
      {1452, 1454},
      // Coefficients 4 taps
      {45, 166, 45,  0,  30,  170, 68,  -12, 15,  163, 90,  -12, 3,   151, 112, -10,
       -5, 133, 133, -5, -10, 112, 151, 3,   -12, 90,  163, 15,  -12, 68,  170, 30},
      // Coefficients 6 taps
      {-24, 60,  184, 60,  -24, 0,   -22, 39,  181, 81,  -24, 1,   -19, 20,  175, 104,
       -22, -2,  -14, 5,   161, 126, -16, -6,  -10, -7,  145, 145, -7,  -10, -6,  -16,
       126, 161, 5,   -14, -2,  -22, 104, 175, 20,  -19, 1,   -24, 81,  181, 39,  -22},
    },

    // Band_8_1456
    {
      // Scaler inc range
      {1454, 1456},
      // Coefficients 4 taps
      {46, 164, 46,  0,  30,  169, 68,  -11, 15,  163, 90,  -12, 3,   151, 112, -10,
       -5, 133, 133, -5, -10, 112, 151, 3,   -12, 90,  163, 15,  -11, 68,  169, 30},
      // Coefficients 6 taps
      {-24, 60,  184, 60,  -24, 0,   -22, 39,  181, 81,  -24, 1,   -19, 21,  174, 104,
       -22, -2,  -14, 5,   161, 126, -16, -6,  -10, -7,  145, 145, -7,  -10, -6,  -16,
       126, 161, 5,   -14, -2,  -22, 104, 174, 21,  -19, 1,   -24, 81,  181, 39,  -22},
    },

    // Band_8_1458
    {
      // Scaler inc range
      {1456, 1458},
      // Coefficients 4 taps
      {46, 164, 46,  0,  30, 169, 68,  -11, 15,  163, 90,  -12, 4,   149, 112, -9,
       -5, 133, 133, -5, -9, 112, 149, 4,   -12, 90,  163, 15,  -11, 68,  169, 30},
      // Coefficients 6 taps
      {-24, 60,  184, 60,  -24, 0,   -22, 39,  181, 81,  -24, 1,   -19, 21,  174, 104,
       -22, -2,  -14, 5,   161, 126, -16, -6,  -10, -7,  145, 145, -7,  -10, -6,  -16,
       126, 161, 5,   -14, -2,  -22, 104, 174, 21,  -19, 1,   -24, 81,  181, 39,  -22},
    },

    // Band_8_1460
    {
      // Scaler inc range
      {1458, 1460},
      // Coefficients 4 taps
      {46, 164, 46,  0,  30, 169, 68,  -11, 15,  162, 90,  -11, 4,   149, 112, -9,
       -4, 132, 132, -4, -9, 112, 149, 4,   -11, 90,  162, 15,  -11, 68,  169, 30},
      // Coefficients 6 taps
      {-24, 60,  184, 60,  -24, 0,   -22, 39,  182, 81,  -24, 0,   -19, 21,  173, 104,
       -21, -2,  -15, 5,   163, 125, -16, -6,  -10, -7,  145, 145, -7,  -10, -6,  -16,
       125, 163, 5,   -15, -2,  -21, 104, 173, 21,  -19, 0,   -24, 81,  182, 39,  -22},
    },

    // Band_8_1462
    {
      // Scaler inc range
      {1460, 1462},
      // Coefficients 4 taps
      {46, 164, 46,  0,  30, 169, 68,  -11, 16,  161, 90,  -11, 4,   149, 112, -9,
       -4, 132, 132, -4, -9, 112, 149, 4,   -11, 90,  161, 16,  -11, 68,  169, 30},
      // Coefficients 6 taps
      {-24, 60,  184, 60,  -24, 0,   -22, 39,  181, 82,  -24, 0,   -19, 21,  173, 104,
       -21, -2,  -15, 6,   162, 125, -16, -6,  -10, -7,  145, 145, -7,  -10, -6,  -16,
       125, 162, 6,   -15, -2,  -21, 104, 173, 21,  -19, 0,   -24, 82,  181, 39,  -22},
    },

    // Band_8_1464
    {
      // Scaler inc range
      {1462, 1464},
      // Coefficients 4 taps
      {46, 164, 46,  0,  30, 169, 68,  -11, 16,  161, 90,  -11, 4,   149, 112, -9,
       -4, 132, 132, -4, -9, 112, 149, 4,   -11, 90,  161, 16,  -11, 68,  169, 30},
      // Coefficients 6 taps
      {-24, 60,  184, 60,  -24, 0,   -22, 40,  180, 82,  -24, 0,   -19, 21,  174, 104,
       -21, -3,  -15, 6,   161, 125, -15, -6,  -10, -6,  144, 144, -6,  -10, -6,  -15,
       125, 161, 6,   -15, -3,  -21, 104, 174, 21,  -19, 0,   -24, 82,  180, 40,  -22},
    },

    // Band_8_1467
    {
      // Scaler inc range
      {1464, 1467},
      // Coefficients 4 taps
      {46, 164, 46,  0,  31, 168, 68,  -11, 16,  161, 90,  -11, 4,   149, 112, -9,
       -4, 132, 132, -4, -9, 112, 149, 4,   -11, 90,  161, 16,  -11, 68,  168, 31},
      // Coefficients 6 taps
      {-24, 60,  184, 60,  -24, 0,   -22, 40,  180, 82,  -24, 0,   -19, 22,  173, 104,
       -21, -3,  -15, 6,   161, 125, -15, -6,  -10, -6,  144, 144, -6,  -10, -6,  -15,
       125, 161, 6,   -15, -3,  -21, 104, 173, 22,  -19, 0,   -24, 82,  180, 40,  -22},
    },

    // Band_8_1469
    {
      // Scaler inc range
      {1467, 1469},
      // Coefficients 4 taps
      {46, 164, 46,  0,  31, 168, 68,  -11, 16,  161, 90,  -11, 4,   149, 112, -9,
       -4, 132, 132, -4, -9, 112, 149, 4,   -11, 90,  161, 16,  -11, 68,  168, 31},
      // Coefficients 6 taps
      {-24, 60,  184, 60,  -24, 0,   -22, 40,  180, 82,  -24, 0,   -19, 22,  173, 104,
       -21, -3,  -15, 6,   161, 125, -15, -6,  -10, -6,  144, 144, -6,  -10, -6,  -15,
       125, 161, 6,   -15, -3,  -21, 104, 173, 22,  -19, 0,   -24, 82,  180, 40,  -22},
    },

    // Band_8_1471
    {
      // Scaler inc range
      {1469, 1471},
      // Coefficients 4 taps
      {46, 164, 46,  0,  31, 168, 68,  -11, 16,  161, 90,  -11, 4,   149, 112, -9,
       -4, 132, 132, -4, -9, 112, 149, 4,   -11, 90,  161, 16,  -11, 68,  168, 31},
      // Coefficients 6 taps
      {-24, 60,  184, 60,  -24, 0,   -22, 40,  179, 82,  -23, 0,   -19, 22,  173, 104,
       -21, -3,  -15, 6,   161, 125, -15, -6,  -11, -6,  145, 145, -6,  -11, -6,  -15,
       125, 161, 6,   -15, -3,  -21, 104, 173, 22,  -19, 0,   -23, 82,  179, 40,  -22},
    },

    // Band_8_1473
    {
      // Scaler inc range
      {1471, 1473},
      // Coefficients 4 taps
      {46, 164, 46,  0,  31, 168, 68,  -11, 16,  161, 90,  -11, 5,   148, 112, -9,
       -4, 132, 132, -4, -9, 112, 148, 5,   -11, 90,  161, 16,  -11, 68,  168, 31},
      // Coefficients 6 taps
      {-24, 61,  182, 61,  -24, 0,   -22, 40,  179, 82,  -23, 0,   -19, 22,  173, 104,
       -21, -3,  -15, 7,   160, 125, -15, -6,  -11, -6,  145, 145, -6,  -11, -6,  -15,
       125, 160, 7,   -15, -3,  -21, 104, 173, 22,  -19, 0,   -23, 82,  179, 40,  -22},
    },

    // Band_8_1475
    {
      // Scaler inc range
      {1473, 1475},
      // Coefficients 4 taps
      {47, 162, 47,  0,  31, 168, 68,  -11, 16,  161, 90,  -11, 5,   148, 112, -9,
       -4, 132, 132, -4, -9, 112, 148, 5,   -11, 90,  161, 16,  -11, 68,  168, 31},
      // Coefficients 6 taps
      {-24, 61,  182, 61,  -24, 0,   -22, 40,  179, 82,  -23, 0,   -19, 22,  172, 104,
       -20, -3,  -15, 7,   161, 125, -15, -7,  -11, -6,  145, 145, -6,  -11, -7,  -15,
       125, 161, 7,   -15, -3,  -20, 104, 172, 22,  -19, 0,   -23, 82,  179, 40,  -22},
    },

    // Band_8_1477
    {
      // Scaler inc range
      {1475, 1477},
      // Coefficients 4 taps
      {47, 162, 47,  0,  31, 167, 69,  -11, 16,  161, 90,  -11, 5,   148, 112, -9,
       -3, 131, 131, -3, -9, 112, 148, 5,   -11, 90,  161, 16,  -11, 69,  167, 31},
      // Coefficients 6 taps
      {-24, 61,  182, 61,  -24, 0,   -22, 41,  178, 82,  -23, 0,   -19, 23,  171, 104,
       -20, -3,  -15, 7,   160, 125, -14, -7,  -11, -5,  144, 144, -5,  -11, -7,  -14,
       125, 160, 7,   -15, -3,  -20, 104, 171, 23,  -19, 0,   -23, 82,  178, 41,  -22},
    },

    // Band_8_1479
    {
      // Scaler inc range
      {1477, 1479},
      // Coefficients 4 taps
      {47, 162, 47,  0,  31, 167, 69,  -11, 17,  160, 90,  -11, 5,   148, 111, -8,
       -3, 131, 131, -3, -8, 111, 148, 5,   -11, 90,  160, 17,  -11, 69,  167, 31},
      // Coefficients 6 taps
      {-24, 61,  182, 61,  -24, 0,   -22, 41,  178, 82,  -23, 0,   -19, 23,  171, 104,
       -20, -3,  -15, 7,   160, 125, -14, -7,  -11, -5,  144, 144, -5,  -11, -7,  -14,
       125, 160, 7,   -15, -3,  -20, 104, 171, 23,  -19, 0,   -23, 82,  178, 41,  -22},
    },

    // Band_8_1481
    {
      // Scaler inc range
      {1479, 1481},
      // Coefficients 4 taps
      {47, 162, 47,  0,  31, 167, 69,  -11, 17,  160, 90,  -11, 5,   148, 111, -8,
       -3, 131, 131, -3, -8, 111, 148, 5,   -11, 90,  160, 17,  -11, 69,  167, 31},
      // Coefficients 6 taps
      {-23, 61,  180, 61,  -23, 0,   -22, 41,  178, 82,  -23, 0,   -19, 23,  171, 104,
       -20, -3,  -15, 7,   160, 125, -14, -7,  -11, -5,  144, 144, -5,  -11, -7,  -14,
       125, 160, 7,   -15, -3,  -20, 104, 171, 23,  -19, 0,   -23, 82,  178, 41,  -22},
    },

    // Band_8_1483
    {
      // Scaler inc range
      {1481, 1483},
      // Coefficients 4 taps
      {47, 162, 47,  0,  32, 166, 69,  -11, 17,  160, 90,  -11, 5,   148, 111, -8,
       -3, 131, 131, -3, -8, 111, 148, 5,   -11, 90,  160, 17,  -11, 69,  166, 32},
      // Coefficients 6 taps
      {-23, 61,  180, 61,  -23, 0,   -22, 41,  178, 82,  -23, 0,   -19, 23,  171, 104,
       -20, -3,  -15, 8,   159, 125, -14, -7,  -11, -5,  144, 144, -5,  -11, -7,  -14,
       125, 159, 8,   -15, -3,  -20, 104, 171, 23,  -19, 0,   -23, 82,  178, 41,  -22},
    },

    // Band_8_1485
    {
      // Scaler inc range
      {1483, 1485},
      // Coefficients 4 taps
      {47, 162, 47,  0,  32, 166, 69,  -11, 17,  160, 90,  -11, 5,   148, 111, -8,
       -3, 131, 131, -3, -8, 111, 148, 5,   -11, 90,  160, 17,  -11, 69,  166, 32},
      // Coefficients 6 taps
      {-23, 61,  180, 61,  -23, 0,   -22, 41,  178, 82,  -23, 0,   -19, 23,  171, 104,
       -20, -3,  -15, 8,   159, 125, -14, -7,  -11, -5,  144, 144, -5,  -11, -7,  -14,
       125, 159, 8,   -15, -3,  -20, 104, 171, 23,  -19, 0,   -23, 82,  178, 41,  -22},
    },

    // Band_8_1487
    {
      // Scaler inc range
      {1485, 1487},
      // Coefficients 4 taps
      {47, 162, 47,  0,  32, 166, 69,  -11, 17,  160, 90,  -11, 6,   147, 111, -8,
       -3, 131, 131, -3, -8, 111, 147, 6,   -11, 90,  160, 17,  -11, 69,  166, 32},
      // Coefficients 6 taps
      {-23, 61,  180, 61,  -23, 0,   -22, 41,  179, 82,  -23, -1,  -19, 23,  171, 104,
       -20, -3,  -15, 8,   159, 125, -14, -7,  -11, -4,  143, 143, -4,  -11, -7,  -14,
       125, 159, 8,   -15, -3,  -20, 104, 171, 23,  -19, -1,  -23, 82,  179, 41,  -22},
    },

    // Band_8_1489
    {
      // Scaler inc range
      {1487, 1489},
      // Coefficients 4 taps
      {47, 162, 47,  0,  32, 166, 69,  -11, 17,  159, 90,  -10, 6,   147, 111, -8,
       -3, 131, 131, -3, -8, 111, 147, 6,   -10, 90,  159, 17,  -11, 69,  166, 32},
      // Coefficients 6 taps
      {-23, 61,  180, 61,  -23, 0,   -22, 42,  178, 82,  -23, -1,  -19, 24,  170, 104,
       -19, -4,  -15, 8,   158, 125, -13, -7,  -11, -4,  143, 143, -4,  -11, -7,  -13,
       125, 158, 8,   -15, -4,  -19, 104, 170, 24,  -19, -1,  -23, 82,  178, 42,  -22},
    },

    // Band_8_1492
    {
      // Scaler inc range
      {1489, 1492},
      // Coefficients 4 taps
      {47, 162, 47,  0,  32, 166, 69,  -11, 17,  159, 90,  -10, 6,   147, 111, -8,
       -3, 131, 131, -3, -8, 111, 147, 6,   -10, 90,  159, 17,  -11, 69,  166, 32},
      // Coefficients 6 taps
      {-23, 61,  180, 61,  -23, 0,   -22, 42,  177, 82,  -22, -1,  -19, 24,  170, 104,
       -19, -4,  -15, 8,   158, 125, -13, -7,  -11, -4,  143, 143, -4,  -11, -7,  -13,
       125, 158, 8,   -15, -4,  -19, 104, 170, 24,  -19, -1,  -22, 82,  177, 42,  -22},
    },

    // Band_8_1494
    {
      // Scaler inc range
      {1492, 1494},
      // Coefficients 4 taps
      {47, 162, 47,  0,  32, 166, 69,  -11, 18,  158, 90,  -10, 6,   147, 111, -8,
       -2, 130, 130, -2, -8, 111, 147, 6,   -10, 90,  158, 18,  -11, 69,  166, 32},
      // Coefficients 6 taps
      {-23, 62,  178, 62,  -23, 0,   -22, 42,  176, 83,  -22, -1,  -19, 24,  170, 104,
       -19, -4,  -15, 9,   157, 125, -13, -7,  -11, -4,  143, 143, -4,  -11, -7,  -13,
       125, 157, 9,   -15, -4,  -19, 104, 170, 24,  -19, -1,  -22, 83,  176, 42,  -22},
    },

    // Band_8_1496
    {
      // Scaler inc range
      {1494, 1496},
      // Coefficients 4 taps
      {48, 160, 48,  0,  32, 166, 69,  -11, 18,  158, 90,  -10, 6,   147, 111, -8,
       -2, 130, 130, -2, -8, 111, 147, 6,   -10, 90,  158, 18,  -11, 69,  166, 32},
      // Coefficients 6 taps
      {-23, 62,  178, 62,  -23, 0,   -22, 42,  176, 83,  -22, -1,  -19, 24,  170, 104,
       -19, -4,  -15, 9,   157, 125, -13, -7,  -11, -4,  143, 143, -4,  -11, -7,  -13,
       125, 157, 9,   -15, -4,  -19, 104, 170, 24,  -19, -1,  -22, 83,  176, 42,  -22},
    },

    // Band_8_1498
    {
      // Scaler inc range
      {1496, 1498},
      // Coefficients 4 taps
      {48, 160, 48,  0,  32, 166, 69,  -11, 18,  158, 90,  -10, 6,   147, 111, -8,
       -2, 130, 130, -2, -8, 111, 147, 6,   -10, 90,  158, 18,  -11, 69,  166, 32},
      // Coefficients 6 taps
      {-23, 62,  178, 62,  -23, 0,   -22, 42,  176, 83,  -22, -1,  -19, 24,  170, 104,
       -19, -4,  -15, 9,   158, 124, -13, -7,  -11, -3,  142, 142, -3,  -11, -7,  -13,
       124, 158, 9,   -15, -4,  -19, 104, 170, 24,  -19, -1,  -22, 83,  176, 42,  -22},
    },

    // Band_8_1500
    {
      // Scaler inc range
      {1498, 1500},
      // Coefficients 4 taps
      {48, 160, 48,  0,  33, 164, 69,  -10, 18,  158, 90,  -10, 6,   146, 111, -7,
       -2, 130, 130, -2, -7, 111, 146, 6,   -10, 90,  158, 18,  -10, 69,  164, 33},
      // Coefficients 6 taps
      {-23, 62,  178, 62,  -23, 0,   -22, 42,  176, 83,  -22, -1,  -19, 25,  169, 104,
       -19, -4,  -16, 9,   160, 124, -13, -8,  -12, -3,  143, 143, -3,  -12, -8,  -13,
       124, 160, 9,   -16, -4,  -19, 104, 169, 25,  -19, -1,  -22, 83,  176, 42,  -22},
    },

    // Band_8_1502
    {
      // Scaler inc range
      {1500, 1502},
      // Coefficients 4 taps
      {48, 160, 48,  0,  33, 164, 69,  -10, 18,  158, 90,  -10, 7,   145, 111, -7,
       -2, 130, 130, -2, -7, 111, 145, 7,   -10, 90,  158, 18,  -10, 69,  164, 33},
      // Coefficients 6 taps
      {-23, 62,  178, 62,  -23, 0,   -22, 42,  176, 83,  -22, -1,  -19, 25,  169, 104,
       -19, -4,  -16, 9,   159, 124, -12, -8,  -12, -3,  143, 143, -3,  -12, -8,  -12,
       124, 159, 9,   -16, -4,  -19, 104, 169, 25,  -19, -1,  -22, 83,  176, 42,  -22},
    },

    // Band_8_1504
    {
      // Scaler inc range
      {1502, 1504},
      // Coefficients 4 taps
      {48, 160, 48,  0,  33, 164, 69,  -10, 18,  158, 90,  -10, 7,   145, 111, -7,
       -2, 130, 130, -2, -7, 111, 145, 7,   -10, 90,  158, 18,  -10, 69,  164, 33},
      // Coefficients 6 taps
      {-23, 62,  178, 62,  -23, 0,   -22, 43,  175, 83,  -22, -1,  -19, 25,  168, 104,
       -18, -4,  -16, 10,  158, 124, -12, -8,  -12, -3,  143, 143, -3,  -12, -8,  -12,
       124, 158, 10,  -16, -4,  -18, 104, 168, 25,  -19, -1,  -22, 83,  175, 43,  -22},
    },

    // Band_8_1507
    {
      // Scaler inc range
      {1504, 1507},
      // Coefficients 4 taps
      {48, 160, 48,  0,  33, 164, 69,  -10, 18,  158, 90,  -10, 7,   145, 111, -7,
       -2, 130, 130, -2, -7, 111, 145, 7,   -10, 90,  158, 18,  -10, 69,  164, 33},
      // Coefficients 6 taps
      {-23, 62,  178, 62,  -23, 0,   -22, 43,  175, 83,  -22, -1,  -19, 25,  168, 104,
       -18, -4,  -16, 10,  158, 124, -12, -8,  -12, -3,  143, 143, -3,  -12, -8,  -12,
       124, 158, 10,  -16, -4,  -18, 104, 168, 25,  -19, -1,  -22, 83,  175, 43,  -22},
    },

    // Band_8_1509
    {
      // Scaler inc range
      {1507, 1509},
      // Coefficients 4 taps
      {48, 160, 48,  0,  33, 164, 69,  -10, 19,  157, 90,  -10, 7,   145, 111, -7,
       -2, 130, 130, -2, -7, 111, 145, 7,   -10, 90,  157, 19,  -10, 69,  164, 33},
      // Coefficients 6 taps
      {-23, 62,  178, 62,  -23, 0,   -22, 43,  175, 83,  -22, -1,  -19, 25,  168, 104,
       -18, -4,  -16, 10,  158, 124, -12, -8,  -12, -2,  142, 142, -2,  -12, -8,  -12,
       124, 158, 10,  -16, -4,  -18, 104, 168, 25,  -19, -1,  -22, 83,  175, 43,  -22},
    },

    // Band_8_1511
    {
      // Scaler inc range
      {1509, 1511},
      // Coefficients 4 taps
      {48, 160, 48,  0,  33, 164, 69,  -10, 19,  157, 90,  -10, 7,   145, 111, -7,
       -1, 129, 129, -1, -7, 111, 145, 7,   -10, 90,  157, 19,  -10, 69,  164, 33},
      // Coefficients 6 taps
      {-22, 62,  176, 62,  -22, 0,   -22, 43,  174, 83,  -21, -1,  -19, 26,  167, 104,
       -18, -4,  -16, 10,  158, 124, -12, -8,  -12, -2,  142, 142, -2,  -12, -8,  -12,
       124, 158, 10,  -16, -4,  -18, 104, 167, 26,  -19, -1,  -21, 83,  174, 43,  -22},
    },

    // Band_8_1513
    {
      // Scaler inc range
      {1511, 1513},
      // Coefficients 4 taps
      {48, 160, 48,  0,  33, 163, 70,  -10, 19,  157, 90,  -10, 7,   146, 110, -7,
       -1, 129, 129, -1, -7, 110, 146, 7,   -10, 90,  157, 19,  -10, 70,  163, 33},
      // Coefficients 6 taps
      {-22, 62,  176, 62,  -22, 0,   -22, 43,  175, 83,  -21, -2,  -19, 26,  167, 104,
       -18, -4,  -16, 10,  157, 124, -11, -8,  -12, -2,  142, 142, -2,  -12, -8,  -11,
       124, 157, 10,  -16, -4,  -18, 104, 167, 26,  -19, -2,  -21, 83,  175, 43,  -22},
    },

    // Band_8_1515
    {
      // Scaler inc range
      {1513, 1515},
      // Coefficients 4 taps
      {49, 158, 49,  0,  33, 163, 70,  -10, 19, 156, 90,  -9, 7,   146, 110, -7,
       -1, 129, 129, -1, -7, 110, 146, 7,   -9, 90,  156, 19, -10, 70,  163, 33},
      // Coefficients 6 taps
      {-22, 63,  174, 63,  -22, 0,   -21, 43,  174, 83,  -21, -2,  -19, 26,  168, 104,
       -18, -5,  -16, 11,  156, 124, -11, -8,  -12, -2,  142, 142, -2,  -12, -8,  -11,
       124, 156, 11,  -16, -5,  -18, 104, 168, 26,  -19, -2,  -21, 83,  174, 43,  -21},
    },

    // Band_8_1517
    {
      // Scaler inc range
      {1515, 1517},
      // Coefficients 4 taps
      {49, 158, 49,  0,  34, 162, 70,  -10, 19, 156, 90,  -9, 8,   145, 110, -7,
       -1, 129, 129, -1, -7, 110, 145, 8,   -9, 90,  156, 19, -10, 70,  162, 34},
      // Coefficients 6 taps
      {-22, 63,  174, 63,  -22, 0,   -21, 44,  173, 83,  -21, -2,  -19, 26,  168, 104,
       -18, -5,  -16, 11,  156, 124, -11, -8,  -12, -2,  142, 142, -2,  -12, -8,  -11,
       124, 156, 11,  -16, -5,  -18, 104, 168, 26,  -19, -2,  -21, 83,  173, 44,  -21},
    },

    // Band_8_1520
    {
      // Scaler inc range
      {1517, 1520},
      // Coefficients 4 taps
      {49, 158, 49,  0,  34, 162, 70,  -10, 19, 156, 90,  -9, 8,   144, 110, -6,
       -1, 129, 129, -1, -6, 110, 144, 8,   -9, 90,  156, 19, -10, 70,  162, 34},
      // Coefficients 6 taps
      {-22, 63,  174, 63,  -22, 0,   -21, 44,  173, 83,  -21, -2,  -19, 26,  167, 104,
       -17, -5,  -16, 11,  156, 124, -11, -8,  -12, -1,  141, 141, -1,  -12, -8,  -11,
       124, 156, 11,  -16, -5,  -17, 104, 167, 26,  -19, -2,  -21, 83,  173, 44,  -21},
    },

    // Band_8_1522
    {
      // Scaler inc range
      {1520, 1522},
      // Coefficients 4 taps
      {49, 158, 49,  0,  34, 162, 70,  -10, 19, 156, 90,  -9, 8,   144, 110, -6,
       -1, 129, 129, -1, -6, 110, 144, 8,   -9, 90,  156, 19, -10, 70,  162, 34},
      // Coefficients 6 taps
      {-22, 63,  174, 63,  -22, 0,   -21, 44,  173, 83,  -21, -2,  -19, 26,  167, 104,
       -17, -5,  -16, 11,  156, 124, -11, -8,  -12, -1,  141, 141, -1,  -12, -8,  -11,
       124, 156, 11,  -16, -5,  -17, 104, 167, 26,  -19, -2,  -21, 83,  173, 44,  -21},
    },

    // Band_8_1524
    {
      // Scaler inc range
      {1522, 1524},
      // Coefficients 4 taps
      {49, 158, 49,  0,  34, 162, 70,  -10, 20, 155, 90,  -9, 8,   144, 110, -6,
       -1, 129, 129, -1, -6, 110, 144, 8,   -9, 90,  155, 20, -10, 70,  162, 34},
      // Coefficients 6 taps
      {-22, 63,  174, 63,  -22, 0,   -21, 44,  173, 83,  -21, -2,  -19, 27,  166, 104,
       -17, -5,  -16, 11,  156, 124, -11, -8,  -12, -1,  141, 141, -1,  -12, -8,  -11,
       124, 156, 11,  -16, -5,  -17, 104, 166, 27,  -19, -2,  -21, 83,  173, 44,  -21},
    },

    // Band_8_1526
    {
      // Scaler inc range
      {1524, 1526},
      // Coefficients 4 taps
      {49, 158, 49,  0,  34, 162, 70,  -10, 20, 155, 90,  -9, 8,   144, 110, -6,
       -1, 129, 129, -1, -6, 110, 144, 8,   -9, 90,  155, 20, -10, 70,  162, 34},
      // Coefficients 6 taps
      {-22, 63,  174, 63,  -22, 0,   -21, 44,  173, 83,  -21, -2,  -19, 27,  166, 104,
       -17, -5,  -16, 12,  154, 124, -10, -8,  -12, -1,  141, 141, -1,  -12, -8,  -10,
       124, 154, 12,  -16, -5,  -17, 104, 166, 27,  -19, -2,  -21, 83,  173, 44,  -21},
    },

    // Band_8_1529
    {
      // Scaler inc range
      {1526, 1529},
      // Coefficients 4 taps
      {49, 158, 49,  0, 34, 162, 70,  -10, 20, 155, 90,  -9, 8,   144, 110, -6,
       0,  128, 128, 0, -6, 110, 144, 8,   -9, 90,  155, 20, -10, 70,  162, 34},
      // Coefficients 6 taps
      {-22, 63,  174, 63,  -22, 0,   -21, 44,  173, 83,  -21, -2,  -19, 27,  166, 104,
       -17, -5,  -16, 12,  156, 123, -10, -9,  -12, -1,  141, 141, -1,  -12, -9,  -10,
       123, 156, 12,  -16, -5,  -17, 104, 166, 27,  -19, -2,  -21, 83,  173, 44,  -21},
    },

    // Band_8_1531
    {
      // Scaler inc range
      {1529, 1531},
      // Coefficients 4 taps
      {49, 158, 49,  0, 34, 162, 70,  -10, 20, 155, 90,  -9, 8,   144, 110, -6,
       0,  128, 128, 0, -6, 110, 144, 8,   -9, 90,  155, 20, -10, 70,  162, 34},
      // Coefficients 6 taps
      {-22, 63,  174, 63,  -22, 0,   -21, 44,  172, 83,  -20, -2,  -19, 27,  166, 104,
       -17, -5,  -16, 12,  156, 123, -10, -9,  -12, 0,   140, 140, 0,   -12, -9,  -10,
       123, 156, 12,  -16, -5,  -17, 104, 166, 27,  -19, -2,  -20, 83,  172, 44,  -21},
    },

    // Band_8_1533
    {
      // Scaler inc range
      {1531, 1533},
      // Coefficients 4 taps
      {49, 158, 49,  0, 34, 162, 70,  -10, 20, 155, 90,  -9, 8,   144, 110, -6,
       0,  128, 128, 0, -6, 110, 144, 8,   -9, 90,  155, 20, -10, 70,  162, 34},
      // Coefficients 6 taps
      {-22, 63,  174, 63,  -22, 0,   -21, 45,  170, 84,  -20, -2,  -19, 27,  165, 104,
       -16, -5,  -16, 12,  156, 123, -10, -9,  -12, 0,   140, 140, 0,   -12, -9,  -10,
       123, 156, 12,  -16, -5,  -16, 104, 165, 27,  -19, -2,  -20, 84,  170, 45,  -21},
    },

    // Band_8_1535
    {
      // Scaler inc range
      {1533, 1535},
      // Coefficients 4 taps
      {49, 158, 49,  0, 35, 161, 70,  -10, 20, 155, 90,  -9, 9,   143, 110, -6,
       0,  128, 128, 0, -6, 110, 143, 9,   -9, 90,  155, 20, -10, 70,  161, 35},
      // Coefficients 6 taps
      {-22, 63,  174, 63,  -22, 0,   -21, 45,  170, 84,  -20, -2,  -19, 28,  164, 104,
       -16, -5,  -16, 12,  156, 123, -10, -9,  -12, 0,   140, 140, 0,   -12, -9,  -10,
       123, 156, 12,  -16, -5,  -16, 104, 164, 28,  -19, -2,  -20, 84,  170, 45,  -21},
    },

    // Band_8_1538
    {
      // Scaler inc range
      {1535, 1538},
      // Coefficients 4 taps
      {50, 156, 50,  0, 35, 161, 70,  -10, 20, 155, 90,  -9, 9,   143, 110, -6,
       0,  128, 128, 0, -6, 110, 143, 9,   -9, 90,  155, 20, -10, 70,  161, 35},
      // Coefficients 6 taps
      {-21, 63,  172, 63,  -21, 0,   -21, 45,  170, 84,  -20, -2,  -19, 28,  164, 104,
       -16, -5,  -16, 13,  154, 123, -9,  -9,  -13, 0,   141, 141, 0,   -13, -9,  -9,
       123, 154, 13,  -16, -5,  -16, 104, 164, 28,  -19, -2,  -20, 84,  170, 45,  -21},
    },

    // Band_8_1540
    {
      // Scaler inc range
      {1538, 1540},
      // Coefficients 4 taps
      {50, 156, 50,  0, 35, 161, 70,  -10, 20, 155, 90,  -9, 9,   143, 110, -6,
       0,  128, 128, 0, -6, 110, 143, 9,   -9, 90,  155, 20, -10, 70,  161, 35},
      // Coefficients 6 taps
      {-21, 63,  172, 63,  -21, 0,   -21, 45,  170, 84,  -20, -2,  -19, 28,  164, 104,
       -16, -5,  -16, 13,  154, 123, -9,  -9,  -13, 0,   141, 141, 0,   -13, -9,  -9,
       123, 154, 13,  -16, -5,  -16, 104, 164, 28,  -19, -2,  -20, 84,  170, 45,  -21},
    },

    // Band_8_1542
    {
      // Scaler inc range
      {1540, 1542},
      // Coefficients 4 taps
      {50, 156, 50,  0, 35, 160, 70,  -9, 21, 154, 90,  -9, 9,  142, 110, -5,
       0,  128, 128, 0, -5, 110, 142, 9,  -9, 90,  154, 21, -9, 70,  160, 35},
      // Coefficients 6 taps
      {-21, 64,  170, 64,  -21, 0,   -21, 45,  170, 84,  -20, -2,  -19, 28,  164, 104,
       -16, -5,  -16, 13,  154, 123, -9,  -9,  -13, 1,   140, 140, 1,   -13, -9,  -9,
       123, 154, 13,  -16, -5,  -16, 104, 164, 28,  -19, -2,  -20, 84,  170, 45,  -21},
    },

    // Band_8_1544
    {
      // Scaler inc range
      {1542, 1544},
      // Coefficients 4 taps
      {50, 156, 50,  0, 35, 160, 70,  -9, 21, 153, 90,  -8, 9,  142, 110, -5,
       0,  128, 128, 0, -5, 110, 142, 9,  -8, 90,  153, 21, -9, 70,  160, 35},
      // Coefficients 6 taps
      {-21, 64,  170, 64,  -21, 0,   -21, 45,  171, 84,  -20, -3,  -19, 28,  165, 104,
       -16, -6,  -16, 13,  154, 123, -9,  -9,  -13, 1,   140, 140, 1,   -13, -9,  -9,
       123, 154, 13,  -16, -6,  -16, 104, 165, 28,  -19, -3,  -20, 84,  171, 45,  -21},
    },

    // Band_8_1547
    {
      // Scaler inc range
      {1544, 1547},
      // Coefficients 4 taps
      {50, 156, 50,  0, 35, 160, 70,  -9, 21, 153, 90,  -8, 9,  142, 110, -5,
       1,  127, 127, 1, -5, 110, 142, 9,  -8, 90,  153, 21, -9, 70,  160, 35},
      // Coefficients 6 taps
      {-21, 64,  170, 64,  -21, 0,   -21, 45,  171, 84,  -20, -3,  -19, 28,  165, 104,
       -16, -6,  -16, 13,  154, 123, -9,  -9,  -13, 1,   140, 140, 1,   -13, -9,  -9,
       123, 154, 13,  -16, -6,  -16, 104, 165, 28,  -19, -3,  -20, 84,  171, 45,  -21},
    },

    // Band_8_1549
    {
      // Scaler inc range
      {1547, 1549},
      // Coefficients 4 taps
      {50, 156, 50,  0, 35, 160, 70,  -9, 21, 153, 90,  -8, 9,  143, 109, -5,
       1,  127, 127, 1, -5, 109, 143, 9,  -8, 90,  153, 21, -9, 70,  160, 35},
      // Coefficients 6 taps
      {-21, 64,  170, 64,  -21, 0,   -21, 46,  169, 84,  -19, -3,  -19, 29,  163, 104,
       -15, -6,  -16, 14,  153, 123, -9,  -9,  -13, 1,   140, 140, 1,   -13, -9,  -9,
       123, 153, 14,  -16, -6,  -15, 104, 163, 29,  -19, -3,  -19, 84,  169, 46,  -21},
    },

    // Band_8_1551
    {
      // Scaler inc range
      {1549, 1551},
      // Coefficients 4 taps
      {50, 156, 50,  0, 35, 160, 70,  -9, 21, 153, 90,  -8, 10, 142, 109, -5,
       1,  127, 127, 1, -5, 109, 142, 10, -8, 90,  153, 21, -9, 70,  160, 35},
      // Coefficients 6 taps
      {-21, 64,  170, 64,  -21, 0,   -21, 46,  169, 84,  -19, -3,  -19, 29,  163, 104,
       -15, -6,  -16, 14,  152, 123, -8,  -9,  -13, 1,   140, 140, 1,   -13, -9,  -8,
       123, 152, 14,  -16, -6,  -15, 104, 163, 29,  -19, -3,  -19, 84,  169, 46,  -21},
    },

    // Band_8_1553
    {
      // Scaler inc range
      {1551, 1553},
      // Coefficients 4 taps
      {50, 156, 50,  0, 35, 160, 70,  -9, 21, 153, 90,  -8, 10, 142, 109, -5,
       1,  127, 127, 1, -5, 109, 142, 10, -8, 90,  153, 21, -9, 70,  160, 35},
      // Coefficients 6 taps
      {-21, 64,  170, 64,  -21, 0,   -21, 46,  169, 84,  -19, -3,  -19, 29,  163, 104,
       -15, -6,  -16, 14,  152, 123, -8,  -9,  -13, 1,   140, 140, 1,   -13, -9,  -8,
       123, 152, 14,  -16, -6,  -15, 104, 163, 29,  -19, -3,  -19, 84,  169, 46,  -21},
    },

    // Band_8_1556
    {
      // Scaler inc range
      {1553, 1556},
      // Coefficients 4 taps
      {50, 156, 50,  0, 36, 158, 71,  -9, 21, 153, 90,  -8, 10, 142, 109, -5,
       1,  127, 127, 1, -5, 109, 142, 10, -8, 90,  153, 21, -9, 71,  158, 36},
      // Coefficients 6 taps
      {-21, 64,  170, 64,  -21, 0,   -21, 46,  169, 84,  -19, -3,  -19, 29,  163, 104,
       -15, -6,  -16, 14,  152, 123, -8,  -9,  -13, 2,   139, 139, 2,   -13, -9,  -8,
       123, 152, 14,  -16, -6,  -15, 104, 163, 29,  -19, -3,  -19, 84,  169, 46,  -21},
    },

    // Band_8_1558
    {
      // Scaler inc range
      {1556, 1558},
      // Coefficients 4 taps
      {50, 156, 50,  0, 36, 158, 71,  -9, 22, 152, 90,  -8, 10, 142, 109, -5,
       1,  127, 127, 1, -5, 109, 142, 10, -8, 90,  152, 22, -9, 71,  158, 36},
      // Coefficients 6 taps
      {-21, 64,  170, 64,  -21, 0,   -21, 46,  169, 84,  -19, -3,  -19, 29,  163, 104,
       -15, -6,  -16, 14,  153, 122, -8,  -9,  -13, 2,   139, 139, 2,   -13, -9,  -8,
       122, 153, 14,  -16, -6,  -15, 104, 163, 29,  -19, -3,  -19, 84,  169, 46,  -21},
    },

    // Band_8_1560
    {
      // Scaler inc range
      {1558, 1560},
      // Coefficients 4 taps
      {50, 156, 50,  0, 36, 158, 71,  -9, 22, 152, 90,  -8, 10, 142, 109, -5,
       1,  127, 127, 1, -5, 109, 142, 10, -8, 90,  152, 22, -9, 71,  158, 36},
      // Coefficients 6 taps
      {-21, 64,  170, 64,  -21, 0,   -21, 46,  169, 84,  -19, -3,  -19, 29,  163, 104,
       -15, -6,  -16, 15,  152, 122, -8,  -9,  -13, 2,   139, 139, 2,   -13, -9,  -8,
       122, 152, 15,  -16, -6,  -15, 104, 163, 29,  -19, -3,  -19, 84,  169, 46,  -21},
    },

    // Band_8_1563
    {
      // Scaler inc range
      {1560, 1563},
      // Coefficients 4 taps
      {51, 154, 51,  0, 36, 158, 71,  -9, 22, 152, 90,  -8, 10, 141, 109, -4,
       1,  127, 127, 1, -4, 109, 141, 10, -8, 90,  152, 22, -9, 71,  158, 36},
      // Coefficients 6 taps
      {-20, 64,  168, 64,  -20, 0,   -21, 46,  169, 84,  -19, -3,  -19, 30,  161, 104,
       -14, -6,  -16, 15,  153, 122, -8,  -10, -13, 2,   139, 139, 2,   -13, -10, -8,
       122, 153, 15,  -16, -6,  -14, 104, 161, 30,  -19, -3,  -19, 84,  169, 46,  -21},
    },

    // Band_8_1565
    {
      // Scaler inc range
      {1563, 1565},
      // Coefficients 4 taps
      {51, 154, 51,  0, 36, 158, 71,  -9, 22, 152, 90,  -8, 10, 141, 109, -4,
       2,  126, 126, 2, -4, 109, 141, 10, -8, 90,  152, 22, -9, 71,  158, 36},
      // Coefficients 6 taps
      {-20, 64,  168, 64,  -20, 0,   -21, 47,  168, 84,  -19, -3,  -19, 30,  161, 104,
       -14, -6,  -16, 15,  152, 122, -7,  -10, -13, 2,   139, 139, 2,   -13, -10, -7,
       122, 152, 15,  -16, -6,  -14, 104, 161, 30,  -19, -3,  -19, 84,  168, 47,  -21},
    },

    // Band_8_1567
    {
      // Scaler inc range
      {1565, 1567},
      // Coefficients 4 taps
      {51, 154, 51,  0, 36, 158, 71,  -9, 22, 152, 90,  -8, 10, 141, 109, -4,
       2,  126, 126, 2, -4, 109, 141, 10, -8, 90,  152, 22, -9, 71,  158, 36},
      // Coefficients 6 taps
      {-20, 64,  168, 64,  -20, 0,   -20, 47,  166, 84,  -18, -3,  -19, 30,  161, 104,
       -14, -6,  -16, 15,  152, 122, -7,  -10, -13, 3,   138, 138, 3,   -13, -10, -7,
       122, 152, 15,  -16, -6,  -14, 104, 161, 30,  -19, -3,  -18, 84,  166, 47,  -20},
    },

    // Band_8_1570
    {
      // Scaler inc range
      {1567, 1570},
      // Coefficients 4 taps
      {51, 154, 51,  0, 36, 158, 71,  -9, 22, 152, 90,  -8, 11, 140, 109, -4,
       2,  126, 126, 2, -4, 109, 140, 11, -8, 90,  152, 22, -9, 71,  158, 36},
      // Coefficients 6 taps
      {-20, 64,  168, 64,  -20, 0,   -20, 47,  166, 84,  -18, -3,  -19, 30,  161, 104,
       -14, -6,  -16, 15,  152, 122, -7,  -10, -13, 3,   138, 138, 3,   -13, -10, -7,
       122, 152, 15,  -16, -6,  -14, 104, 161, 30,  -19, -3,  -18, 84,  166, 47,  -20},
    },

    // Band_8_1572
    {
      // Scaler inc range
      {1570, 1572},
      // Coefficients 4 taps
      {51, 154, 51,  0, 36, 158, 71,  -9, 22, 151, 90,  -7, 11, 140, 109, -4,
       2,  126, 126, 2, -4, 109, 140, 11, -7, 90,  151, 22, -9, 71,  158, 36},
      // Coefficients 6 taps
      {-20, 65,  166, 65,  -20, 0,   -20, 47,  166, 84,  -18, -3,  -19, 30,  161, 104,
       -14, -6,  -16, 16,  151, 122, -7,  -10, -13, 3,   138, 138, 3,   -13, -10, -7,
       122, 151, 16,  -16, -6,  -14, 104, 161, 30,  -19, -3,  -18, 84,  166, 47,  -20},
    },

    // Band_8_1574
    {
      // Scaler inc range
      {1572, 1574},
      // Coefficients 4 taps
      {51, 154, 51,  0, 37, 157, 71,  -9, 22, 151, 90,  -7, 11, 140, 109, -4,
       2,  126, 126, 2, -4, 109, 140, 11, -7, 90,  151, 22, -9, 71,  157, 37},
      // Coefficients 6 taps
      {-20, 65,  166, 65,  -20, 0,   -20, 47,  167, 84,  -18, -4,  -19, 31,  160, 104,
       -14, -6,  -17, 16,  152, 122, -7,  -10, -13, 3,   138, 138, 3,   -13, -10, -7,
       122, 152, 16,  -17, -6,  -14, 104, 160, 31,  -19, -4,  -18, 84,  167, 47,  -20},
    },

    // Band_8_1577
    {
      // Scaler inc range
      {1574, 1577},
      // Coefficients 4 taps
      {51, 154, 51,  0, 37, 157, 71,  -9, 23, 150, 90,  -7, 11, 140, 109, -4,
       2,  126, 126, 2, -4, 109, 140, 11, -7, 90,  150, 23, -9, 71,  157, 37},
      // Coefficients 6 taps
      {-20, 65,  166, 65,  -20, 0,   -20, 47,  167, 84,  -18, -4,  -19, 31,  161, 103,
       -13, -7,  -17, 16,  151, 122, -6,  -10, -13, 3,   138, 138, 3,   -13, -10, -6,
       122, 151, 16,  -17, -7,  -13, 103, 161, 31,  -19, -4,  -18, 84,  167, 47,  -20},
    },

    // Band_8_1579
    {
      // Scaler inc range
      {1577, 1579},
      // Coefficients 4 taps
      {51, 154, 51,  0, 37, 157, 71,  -9, 23, 150, 90,  -7, 11, 140, 109, -4,
       2,  126, 126, 2, -4, 109, 140, 11, -7, 90,  150, 23, -9, 71,  157, 37},
      // Coefficients 6 taps
      {-20, 65,  166, 65,  -20, 0,   -20, 47,  167, 84,  -18, -4,  -19, 31,  161, 103,
       -13, -7,  -17, 16,  151, 122, -6,  -10, -13, 4,   137, 137, 4,   -13, -10, -6,
       122, 151, 16,  -17, -7,  -13, 103, 161, 31,  -19, -4,  -18, 84,  167, 47,  -20},
    },

    // Band_8_1582
    {
      // Scaler inc range
      {1579, 1582},
      // Coefficients 4 taps
      {51, 154, 51,  0, 37, 156, 71,  -8, 23, 150, 90,  -7, 11, 140, 109, -4,
       2,  126, 126, 2, -4, 109, 140, 11, -7, 90,  150, 23, -8, 71,  156, 37},
      // Coefficients 6 taps
      {-20, 65,  166, 65,  -20, 0,   -20, 48,  166, 84,  -18, -4,  -19, 31,  161, 103,
       -13, -7,  -17, 16,  151, 122, -6,  -10, -13, 4,   137, 137, 4,   -13, -10, -6,
       122, 151, 16,  -17, -7,  -13, 103, 161, 31,  -19, -4,  -18, 84,  166, 48,  -20},
    },

    // Band_8_1584
    {
      // Scaler inc range
      {1582, 1584},
      // Coefficients 4 taps
      {51, 154, 51,  0, 37, 156, 71,  -8, 23, 150, 90,  -7, 11, 140, 108, -3,
       3,  125, 125, 3, -3, 108, 140, 11, -7, 90,  150, 23, -8, 71,  156, 37},
      // Coefficients 6 taps
      {-20, 65,  166, 65,  -20, 0,   -20, 48,  166, 84,  -18, -4,  -19, 31,  161, 103,
       -13, -7,  -17, 16,  152, 121, -6,  -10, -13, 4,   137, 137, 4,   -13, -10, -6,
       121, 152, 16,  -17, -7,  -13, 103, 161, 31,  -19, -4,  -18, 84,  166, 48,  -20},
    },

    // Band_8_1586
    {
      // Scaler inc range
      {1584, 1586},
      // Coefficients 4 taps
      {52, 152, 52,  0, 37, 156, 71,  -8, 23, 150, 90,  -7, 12, 139, 108, -3,
       3,  125, 125, 3, -3, 108, 139, 12, -7, 90,  150, 23, -8, 71,  156, 37},
      // Coefficients 6 taps
      {-19, 65,  164, 65,  -19, 0,   -20, 48,  165, 84,  -17, -4,  -19, 31,  161, 103,
       -13, -7,  -17, 17,  151, 121, -6,  -10, -13, 4,   137, 137, 4,   -13, -10, -6,
       121, 151, 17,  -17, -7,  -13, 103, 161, 31,  -19, -4,  -17, 84,  165, 48,  -20},
    },

    // Band_8_1589
    {
      // Scaler inc range
      {1586, 1589},
      // Coefficients 4 taps
      {52, 152, 52,  0, 37, 156, 71,  -8, 23, 150, 90,  -7, 12, 139, 108, -3,
       3,  125, 125, 3, -3, 108, 139, 12, -7, 90,  150, 23, -8, 71,  156, 37},
      // Coefficients 6 taps
      {-19, 65,  164, 65,  -19, 0,   -20, 48,  165, 84,  -17, -4,  -19, 32,  160, 103,
       -13, -7,  -17, 17,  150, 121, -5,  -10, -14, 4,   138, 138, 4,   -14, -10, -5,
       121, 150, 17,  -17, -7,  -13, 103, 160, 32,  -19, -4,  -17, 84,  165, 48,  -20},
    },

    // Band_8_1591
    {
      // Scaler inc range
      {1589, 1591},
      // Coefficients 4 taps
      {52, 152, 52,  0, 37, 156, 71,  -8, 23, 150, 90,  -7, 12, 139, 108, -3,
       3,  125, 125, 3, -3, 108, 139, 12, -7, 90,  150, 23, -8, 71,  156, 37},
      // Coefficients 6 taps
      {-19, 65,  164, 65,  -19, 0,   -20, 48,  165, 84,  -17, -4,  -19, 32,  160, 103,
       -13, -7,  -17, 17,  150, 121, -5,  -10, -14, 5,   137, 137, 5,   -14, -10, -5,
       121, 150, 17,  -17, -7,  -13, 103, 160, 32,  -19, -4,  -17, 84,  165, 48,  -20},
    },

    // Band_8_1594
    {
      // Scaler inc range
      {1591, 1594},
      // Coefficients 4 taps
      {52, 152, 52,  0, 37, 156, 71,  -8, 23, 150, 90,  -7, 12, 139, 108, -3,
       3,  125, 125, 3, -3, 108, 139, 12, -7, 90,  150, 23, -8, 71,  156, 37},
      // Coefficients 6 taps
      {-19, 65,  164, 65,  -19, 0,   -20, 48,  165, 84,  -17, -4,  -19, 32,  159, 103,
       -12, -7,  -17, 17,  150, 121, -5,  -10, -14, 5,   137, 137, 5,   -14, -10, -5,
       121, 150, 17,  -17, -7,  -12, 103, 159, 32,  -19, -4,  -17, 84,  165, 48,  -20},
    },

    // Band_8_1596
    {
      // Scaler inc range
      {1594, 1596},
      // Coefficients 4 taps
      {52, 152, 52,  0, 38, 155, 71,  -8, 24, 149, 90,  -7, 12, 139, 108, -3,
       3,  125, 125, 3, -3, 108, 139, 12, -7, 90,  149, 24, -8, 71,  155, 38},
      // Coefficients 6 taps
      {-19, 65,  164, 65,  -19, 0,   -20, 48,  165, 84,  -17, -4,  -19, 32,  159, 103,
       -12, -7,  -17, 17,  150, 121, -5,  -10, -14, 5,   137, 137, 5,   -14, -10, -5,
       121, 150, 17,  -17, -7,  -12, 103, 159, 32,  -19, -4,  -17, 84,  165, 48,  -20},
    },

    // Band_8_1598
    {
      // Scaler inc range
      {1596, 1598},
      // Coefficients 4 taps
      {52, 152, 52,  0, 38, 155, 71,  -8, 24, 148, 90,  -6, 12, 139, 108, -3,
       3,  125, 125, 3, -3, 108, 139, 12, -6, 90,  148, 24, -8, 71,  155, 38},
      // Coefficients 6 taps
      {-19, 65,  164, 65,  -19, 0,   -20, 48,  165, 84,  -17, -4,  -19, 32,  159, 103,
       -12, -7,  -17, 18,  149, 121, -5,  -10, -14, 5,   137, 137, 5,   -14, -10, -5,
       121, 149, 18,  -17, -7,  -12, 103, 159, 32,  -19, -4,  -17, 84,  165, 48,  -20},
    },

    // Band_8_1601
    {
      // Scaler inc range
      {1598, 1601},
      // Coefficients 4 taps
      {52, 152, 52,  0, 38, 155, 71,  -8, 24, 148, 90,  -6, 12, 139, 108, -3,
       3,  125, 125, 3, -3, 108, 139, 12, -6, 90,  148, 24, -8, 71,  155, 38},
      // Coefficients 6 taps
      {-19, 65,  164, 65,  -19, 0,   -20, 49,  164, 84,  -17, -4,  -19, 32,  159, 103,
       -12, -7,  -17, 18,  149, 121, -5,  -10, -14, 5,   137, 137, 5,   -14, -10, -5,
       121, 149, 18,  -17, -7,  -12, 103, 159, 32,  -19, -4,  -17, 84,  164, 49,  -20},
    },

    // Band_8_1603
    {
      // Scaler inc range
      {1601, 1603},
      // Coefficients 4 taps
      {52, 152, 52,  0, 38, 155, 71,  -8, 24, 148, 90,  -6, 12, 139, 108, -3,
       4,  124, 124, 4, -3, 108, 139, 12, -6, 90,  148, 24, -8, 71,  155, 38},
      // Coefficients 6 taps
      {-19, 65,  164, 65,  -19, 0,   -20, 49,  162, 85,  -16, -4,  -19, 33,  158, 103,
       -12, -7,  -17, 18,  148, 121, -4,  -10, -14, 6,   136, 136, 6,   -14, -10, -4,
       121, 148, 18,  -17, -7,  -12, 103, 158, 33,  -19, -4,  -16, 85,  162, 49,  -20},
    },

    // Band_8_1606
    {
      // Scaler inc range
      {1603, 1606},
      // Coefficients 4 taps
      {52, 152, 52,  0, 38, 155, 71,  -8, 24, 148, 90,  -6, 13, 137, 108, -2,
       4,  124, 124, 4, -2, 108, 137, 13, -6, 90,  148, 24, -8, 71,  155, 38},
      // Coefficients 6 taps
      {-19, 65,  164, 65,  -19, 0,   -20, 49,  162, 85,  -16, -4,  -19, 33,  158, 103,
       -12, -7,  -17, 18,  149, 121, -4,  -11, -14, 6,   136, 136, 6,   -14, -11, -4,
       121, 149, 18,  -17, -7,  -12, 103, 158, 33,  -19, -4,  -16, 85,  162, 49,  -20},
    },

    // Band_8_1608
    {
      // Scaler inc range
      {1606, 1608},
      // Coefficients 4 taps
      {52, 152, 52,  0, 38, 154, 72,  -8, 24, 148, 90,  -6, 13, 137, 108, -2,
       4,  124, 124, 4, -2, 108, 137, 13, -6, 90,  148, 24, -8, 72,  154, 38},
      // Coefficients 6 taps
      {-19, 65,  164, 65,  -19, 0,   -19, 49,  161, 85,  -16, -4,  -19, 33,  157, 103,
       -11, -7,  -17, 18,  150, 120, -4,  -11, -14, 6,   136, 136, 6,   -14, -11, -4,
       120, 150, 18,  -17, -7,  -11, 103, 157, 33,  -19, -4,  -16, 85,  161, 49,  -19},
    },

    // Band_8_1611
    {
      // Scaler inc range
      {1608, 1611},
      // Coefficients 4 taps
      {52, 152, 52,  0, 38, 154, 72,  -8, 24, 148, 90,  -6, 13, 137, 108, -2,
       4,  124, 124, 4, -2, 108, 137, 13, -6, 90,  148, 24, -8, 72,  154, 38},
      // Coefficients 6 taps
      {-18, 66,  160, 66,  -18, 0,   -19, 49,  162, 85,  -16, -5,  -19, 33,  157, 103,
       -11, -7,  -17, 19,  149, 120, -4,  -11, -14, 6,   136, 136, 6,   -14, -11, -4,
       120, 149, 19,  -17, -7,  -11, 103, 157, 33,  -19, -5,  -16, 85,  162, 49,  -19},
    },

    // Band_8_1613
    {
      // Scaler inc range
      {1611, 1613},
      // Coefficients 4 taps
      {53, 150, 53,  0, 38, 154, 72,  -8, 24, 148, 90,  -6, 13, 137, 108, -2,
       4,  124, 124, 4, -2, 108, 137, 13, -6, 90,  148, 24, -8, 72,  154, 38},
      // Coefficients 6 taps
      {-18, 66,  160, 66,  -18, 0,   -19, 49,  162, 85,  -16, -5,  -19, 33,  158, 103,
       -11, -8,  -17, 19,  149, 120, -4,  -11, -14, 6,   136, 136, 6,   -14, -11, -4,
       120, 149, 19,  -17, -8,  -11, 103, 158, 33,  -19, -5,  -16, 85,  162, 49,  -19},
    },

    // Band_8_1616
    {
      // Scaler inc range
      {1613, 1616},
      // Coefficients 4 taps
      {53, 150, 53,  0, 38, 154, 72,  -8, 25, 147, 90,  -6, 13, 137, 108, -2,
       4,  124, 124, 4, -2, 108, 137, 13, -6, 90,  147, 25, -8, 72,  154, 38},
      // Coefficients 6 taps
      {-18, 66,  160, 66,  -18, 0,   -19, 49,  162, 85,  -16, -5,  -19, 33,  158, 103,
       -11, -8,  -17, 19,  148, 120, -3,  -11, -14, 7,   135, 135, 7,   -14, -11, -3,
       120, 148, 19,  -17, -8,  -11, 103, 158, 33,  -19, -5,  -16, 85,  162, 49,  -19},
    },

    // Band_8_1618
    {
      // Scaler inc range
      {1616, 1618},
      // Coefficients 4 taps
      {53, 150, 53,  0, 39, 153, 72,  -8, 25, 147, 90,  -6, 13, 138, 107, -2,
       4,  124, 124, 4, -2, 107, 138, 13, -6, 90,  147, 25, -8, 72,  153, 39},
      // Coefficients 6 taps
      {-18, 66,  160, 66,  -18, 0,   -19, 49,  162, 85,  -16, -5,  -19, 33,  158, 103,
       -11, -8,  -17, 19,  148, 120, -3,  -11, -14, 7,   135, 135, 7,   -14, -11, -3,
       120, 148, 19,  -17, -8,  -11, 103, 158, 33,  -19, -5,  -16, 85,  162, 49,  -19},
    },

    // Band_8_1621
    {
      // Scaler inc range
      {1618, 1621},
      // Coefficients 4 taps
      {53, 150, 53,  0, 39, 152, 72,  -7, 25, 147, 90,  -6, 13, 138, 107, -2,
       4,  124, 124, 4, -2, 107, 138, 13, -6, 90,  147, 25, -7, 72,  152, 39},
      // Coefficients 6 taps
      {-18, 66,  160, 66,  -18, 0,   -19, 50,  160, 85,  -15, -5,  -18, 34,  156, 103,
       -11, -8,  -17, 19,  148, 120, -3,  -11, -14, 7,   135, 135, 7,   -14, -11, -3,
       120, 148, 19,  -17, -8,  -11, 103, 156, 34,  -18, -5,  -15, 85,  160, 50,  -19},
    },

    // Band_8_1623
    {
      // Scaler inc range
      {1621, 1623},
      // Coefficients 4 taps
      {53, 150, 53,  0, 39, 152, 72,  -7, 25, 147, 90,  -6, 13, 138, 107, -2,
       5,  123, 123, 5, -2, 107, 138, 13, -6, 90,  147, 25, -7, 72,  152, 39},
      // Coefficients 6 taps
      {-18, 66,  160, 66,  -18, 0,   -19, 50,  160, 85,  -15, -5,  -18, 34,  155, 103,
       -10, -8,  -17, 19,  148, 120, -3,  -11, -14, 7,   135, 135, 7,   -14, -11, -3,
       120, 148, 19,  -17, -8,  -10, 103, 155, 34,  -18, -5,  -15, 85,  160, 50,  -19},
    },

    // Band_8_1626
    {
      // Scaler inc range
      {1623, 1626},
      // Coefficients 4 taps
      {53, 150, 53,  0, 39, 152, 72,  -7, 25, 147, 90,  -6, 14, 137, 107, -2,
       5,  123, 123, 5, -2, 107, 137, 14, -6, 90,  147, 25, -7, 72,  152, 39},
      // Coefficients 6 taps
      {-18, 66,  160, 66,  -18, 0,   -19, 50,  160, 85,  -15, -5,  -18, 34,  155, 103,
       -10, -8,  -17, 20,  147, 120, -3,  -11, -14, 7,   135, 135, 7,   -14, -11, -3,
       120, 147, 20,  -17, -8,  -10, 103, 155, 34,  -18, -5,  -15, 85,  160, 50,  -19},
    },

    // Band_8_1628
    {
      // Scaler inc range
      {1626, 1628},
      // Coefficients 4 taps
      {53, 150, 53,  0, 39, 152, 72,  -7, 25, 146, 90,  -5, 14, 137, 107, -2,
       5,  123, 123, 5, -2, 107, 137, 14, -5, 90,  146, 25, -7, 72,  152, 39},
      // Coefficients 6 taps
      {-18, 66,  160, 66,  -18, 0,   -19, 50,  160, 85,  -15, -5,  -18, 34,  155, 103,
       -10, -8,  -17, 20,  147, 120, -3,  -11, -14, 7,   135, 135, 7,   -14, -11, -3,
       120, 147, 20,  -17, -8,  -10, 103, 155, 34,  -18, -5,  -15, 85,  160, 50,  -19},
    },

    // Band_8_1631
    {
      // Scaler inc range
      {1628, 1631},
      // Coefficients 4 taps
      {53, 150, 53,  0, 39, 152, 72,  -7, 25, 146, 90,  -5, 14, 136, 107, -1,
       5,  123, 123, 5, -1, 107, 136, 14, -5, 90,  146, 25, -7, 72,  152, 39},
      // Coefficients 6 taps
      {-18, 66,  160, 66,  -18, 0,   -19, 50,  160, 85,  -15, -5,  -18, 34,  155, 103,
       -10, -8,  -17, 20,  147, 119, -2,  -11, -14, 8,   134, 134, 8,   -14, -11, -2,
       119, 147, 20,  -17, -8,  -10, 103, 155, 34,  -18, -5,  -15, 85,  160, 50,  -19},
    },

    // Band_8_1633
    {
      // Scaler inc range
      {1631, 1633},
      // Coefficients 4 taps
      {53, 150, 53,  0, 39, 152, 72,  -7, 25, 146, 90,  -5, 14, 136, 107, -1,
       5,  123, 123, 5, -1, 107, 136, 14, -5, 90,  146, 25, -7, 72,  152, 39},
      // Coefficients 6 taps
      {-17, 66,  158, 66,  -17, 0,   -19, 50,  160, 85,  -15, -5,  -18, 34,  155, 103,
       -10, -8,  -17, 20,  147, 119, -2,  -11, -14, 8,   134, 134, 8,   -14, -11, -2,
       119, 147, 20,  -17, -8,  -10, 103, 155, 34,  -18, -5,  -15, 85,  160, 50,  -19},
    },

    // Band_8_1636
    {
      // Scaler inc range
      {1633, 1636},
      // Coefficients 4 taps
      {53, 150, 53,  0, 39, 152, 72,  -7, 26, 145, 90,  -5, 14, 136, 107, -1,
       5,  123, 123, 5, -1, 107, 136, 14, -5, 90,  145, 26, -7, 72,  152, 39},
      // Coefficients 6 taps
      {-17, 66,  158, 66,  -17, 0,   -19, 50,  160, 85,  -15, -5,  -18, 35,  154, 103,
       -10, -8,  -17, 20,  147, 119, -2,  -11, -14, 8,   134, 134, 8,   -14, -11, -2,
       119, 147, 20,  -17, -8,  -10, 103, 154, 35,  -18, -5,  -15, 85,  160, 50,  -19},
    },

    // Band_8_1638
    {
      // Scaler inc range
      {1636, 1638},
      // Coefficients 4 taps
      {53, 150, 53,  0, 39, 152, 72,  -7, 26, 145, 90,  -5, 14, 136, 107, -1,
       5,  123, 123, 5, -1, 107, 136, 14, -5, 90,  145, 26, -7, 72,  152, 39},
      // Coefficients 6 taps
      {-17, 66,  158, 66,  -17, 0,   -19, 50,  159, 85,  -14, -5,  -18, 35,  154, 102,
       -9,  -8,  -17, 20,  147, 119, -2,  -11, -14, 8,   134, 134, 8,   -14, -11, -2,
       119, 147, 20,  -17, -8,  -9,  102, 154, 35,  -18, -5,  -14, 85,  159, 50,  -19},
    },

    // Band_8_1641
    {
      // Scaler inc range
      {1638, 1641},
      // Coefficients 4 taps
      {54, 148, 54,  0, 40, 151, 72,  -7, 26, 145, 90,  -5, 14, 136, 107, -1,
       5,  123, 123, 5, -1, 107, 136, 14, -5, 90,  145, 26, -7, 72,  151, 40},
      // Coefficients 6 taps
      {-17, 66,  158, 66,  -17, 0,   -19, 50,  159, 85,  -14, -5,  -18, 35,  154, 102,
       -9,  -8,  -17, 21,  146, 119, -2,  -11, -14, 8,   134, 134, 8,   -14, -11, -2,
       119, 146, 21,  -17, -8,  -9,  102, 154, 35,  -18, -5,  -14, 85,  159, 50,  -19},
    },

    // Band_8_1644
    {
      // Scaler inc range
      {1641, 1644},
      // Coefficients 4 taps
      {54, 148, 54,  0, 40, 151, 72,  -7, 26, 145, 90,  -5, 15, 135, 107, -1,
       6,  122, 122, 6, -1, 107, 135, 15, -5, 90,  145, 26, -7, 72,  151, 40},
      // Coefficients 6 taps
      {-17, 66,  158, 66,  -17, 0,   -18, 51,  157, 85,  -14, -5,  -18, 35,  154, 102,
       -9,  -8,  -17, 21,  145, 119, -1,  -11, -14, 9,   133, 133, 9,   -14, -11, -1,
       119, 145, 21,  -17, -8,  -9,  102, 154, 35,  -18, -5,  -14, 85,  157, 51,  -18},
    },

    // Band_8_1646
    {
      // Scaler inc range
      {1644, 1646},
      // Coefficients 4 taps
      {54, 148, 54,  0, 40, 151, 72,  -7, 26, 145, 90,  -5, 15, 135, 107, -1,
       6,  122, 122, 6, -1, 107, 135, 15, -5, 90,  145, 26, -7, 72,  151, 40},
      // Coefficients 6 taps
      {-17, 66,  158, 66,  -17, 0,   -18, 51,  157, 85,  -14, -5,  -18, 35,  154, 102,
       -9,  -8,  -17, 21,  145, 119, -1,  -11, -14, 9,   133, 133, 9,   -14, -11, -1,
       119, 145, 21,  -17, -8,  -9,  102, 154, 35,  -18, -5,  -14, 85,  157, 51,  -18},
    },

    // Band_8_1649
    {
      // Scaler inc range
      {1646, 1649},
      // Coefficients 4 taps
      {54, 148, 54,  0, 40, 151, 72,  -7, 26, 145, 90,  -5, 15, 135, 107, -1,
       6,  122, 122, 6, -1, 107, 135, 15, -5, 90,  145, 26, -7, 72,  151, 40},
      // Coefficients 6 taps
      {-17, 66,  158, 66,  -17, 0,   -18, 51,  157, 85,  -14, -5,  -18, 35,  154, 102,
       -9,  -8,  -17, 21,  145, 119, -1,  -11, -14, 9,   133, 133, 9,   -14, -11, -1,
       119, 145, 21,  -17, -8,  -9,  102, 154, 35,  -18, -5,  -14, 85,  157, 51,  -18},
    },

    // Band_8_1651
    {
      // Scaler inc range
      {1649, 1651},
      // Coefficients 4 taps
      {54, 148, 54,  0, 40, 151, 72,  -7, 26, 145, 90,  -5, 15, 135, 107, -1,
       6,  122, 122, 6, -1, 107, 135, 15, -5, 90,  145, 26, -7, 72,  151, 40},
      // Coefficients 6 taps
      {-17, 66,  158, 66,  -17, 0,   -18, 51,  158, 85,  -14, -6,  -18, 35,  154, 102,
       -9,  -8,  -16, 21,  144, 119, -1,  -11, -14, 9,   133, 133, 9,   -14, -11, -1,
       119, 144, 21,  -16, -8,  -9,  102, 154, 35,  -18, -6,  -14, 85,  158, 51,  -18},
    },

    // Band_8_1654
    {
      // Scaler inc range
      {1651, 1654},
      // Coefficients 4 taps
      {54, 148, 54,  0, 40, 151, 72,  -7, 26, 145, 90,  -5, 15, 134, 107, 0,
       6,  122, 122, 6, 0,  107, 134, 15, -5, 90,  145, 26, -7, 72,  151, 40},
      // Coefficients 6 taps
      {-16, 66,  156, 66,  -16, 0,   -18, 51,  158, 85,  -14, -6,  -18, 36,  152, 102,
       -8,  -8,  -16, 22,  144, 118, -1,  -11, -14, 9,   133, 133, 9,   -14, -11, -1,
       118, 144, 22,  -16, -8,  -8,  102, 152, 36,  -18, -6,  -14, 85,  158, 51,  -18},
    },

    // Band_8_1657
    {
      // Scaler inc range
      {1654, 1657},
      // Coefficients 4 taps
      {54, 148, 54,  0, 40, 151, 72,  -7, 27, 143, 90,  -4, 15, 135, 106, 0,
       6,  122, 122, 6, 0,  106, 135, 15, -4, 90,  143, 27, -7, 72,  151, 40},
      // Coefficients 6 taps
      {-16, 66,  156, 66,  -16, 0,   -18, 51,  157, 85,  -13, -6,  -18, 36,  152, 102,
       -8,  -8,  -16, 22,  144, 118, -1,  -11, -14, 10,  132, 132, 10,  -14, -11, -1,
       118, 144, 22,  -16, -8,  -8,  102, 152, 36,  -18, -6,  -13, 85,  157, 51,  -18},
    },

    // Band_8_1659
    {
      // Scaler inc range
      {1657, 1659},
      // Coefficients 4 taps
      {54, 148, 54,  0, 40, 150, 72,  -6, 27, 143, 90,  -4, 15, 135, 106, 0,
       6,  122, 122, 6, 0,  106, 135, 15, -4, 90,  143, 27, -6, 72,  150, 40},
      // Coefficients 6 taps
      {-16, 66,  156, 66,  -16, 0,   -18, 51,  157, 85,  -13, -6,  -18, 36,  152, 102,
       -8,  -8,  -16, 22,  143, 118, 0,   -11, -14, 10,  132, 132, 10,  -14, -11, 0,
       118, 143, 22,  -16, -8,  -8,  102, 152, 36,  -18, -6,  -13, 85,  157, 51,  -18},
    },

    // Band_8_1662
    {
      // Scaler inc range
      {1659, 1662},
      // Coefficients 4 taps
      {54, 148, 54,  0, 40, 150, 72,  -6, 27, 143, 90,  -4, 15, 135, 106, 0,
       6,  122, 122, 6, 0,  106, 135, 15, -4, 90,  143, 27, -6, 72,  150, 40},
      // Coefficients 6 taps
      {-16, 67,  154, 67,  -16, 0,   -18, 51,  157, 85,  -13, -6,  -18, 36,  153, 102,
       -8,  -9,  -16, 22,  143, 118, 0,   -11, -14, 10,  132, 132, 10,  -14, -11, 0,
       118, 143, 22,  -16, -9,  -8,  102, 153, 36,  -18, -6,  -13, 85,  157, 51,  -18},
    },

    // Band_8_1664
    {
      // Scaler inc range
      {1662, 1664},
      // Coefficients 4 taps
      {54, 148, 54,  0, 40, 150, 72,  -6, 27, 143, 90,  -4, 16, 134, 106, 0,
       7,  121, 121, 7, 0,  106, 134, 16, -4, 90,  143, 27, -6, 72,  150, 40},
      // Coefficients 6 taps
      {-16, 67,  154, 67,  -16, 0,   -18, 52,  156, 85,  -13, -6,  -18, 36,  153, 102,
       -8,  -9,  -16, 22,  143, 118, 0,   -11, -14, 10,  132, 132, 10,  -14, -11, 0,
       118, 143, 22,  -16, -9,  -8,  102, 153, 36,  -18, -6,  -13, 85,  156, 52,  -18},
    },

    // Band_8_1667
    {
      // Scaler inc range
      {1664, 1667},
      // Coefficients 4 taps
      {54, 148, 54,  0, 41, 149, 72,  -6, 27, 143, 90,  -4, 16, 134, 106, 0,
       7,  121, 121, 7, 0,  106, 134, 16, -4, 90,  143, 27, -6, 72,  149, 41},
      // Coefficients 6 taps
      {-16, 67,  154, 67,  -16, 0,   -18, 52,  156, 85,  -13, -6,  -18, 36,  153, 102,
       -8,  -9,  -16, 22,  144, 118, 0,   -12, -14, 10,  132, 132, 10,  -14, -12, 0,
       118, 144, 22,  -16, -9,  -8,  102, 153, 36,  -18, -6,  -13, 85,  156, 52,  -18},
    },

    // Band_8_1670
    {
      // Scaler inc range
      {1667, 1670},
      // Coefficients 4 taps
      {54, 148, 54,  0, 41, 149, 72,  -6, 27, 144, 89,  -4, 16, 134, 106, 0,
       7,  121, 121, 7, 0,  106, 134, 16, -4, 89,  144, 27, -6, 72,  149, 41},
      // Coefficients 6 taps
      {-16, 67,  154, 67,  -16, 0,   -18, 52,  156, 85,  -13, -6,  -18, 37,  151, 102,
       -7,  -9,  -16, 23,  143, 118, 0,   -12, -14, 10,  132, 132, 10,  -14, -12, 0,
       118, 143, 23,  -16, -9,  -7,  102, 151, 37,  -18, -6,  -13, 85,  156, 52,  -18},
    },

    // Band_8_1672
    {
      // Scaler inc range
      {1670, 1672},
      // Coefficients 4 taps
      {55, 146, 55,  0, 41, 149, 72,  -6, 27, 144, 89,  -4, 16, 134, 106, 0,
       7,  121, 121, 7, 0,  106, 134, 16, -4, 89,  144, 27, -6, 72,  149, 41},
      // Coefficients 6 taps
      {-16, 67,  154, 67,  -16, 0,   -18, 52,  156, 85,  -13, -6,  -18, 37,  151, 102,
       -7,  -9,  -16, 23,  142, 118, 1,   -12, -14, 11,  131, 131, 11,  -14, -12, 1,
       118, 142, 23,  -16, -9,  -7,  102, 151, 37,  -18, -6,  -13, 85,  156, 52,  -18},
    },

    // Band_8_1675
    {
      // Scaler inc range
      {1672, 1675},
      // Coefficients 4 taps
      {55, 146, 55,  0, 41, 148, 73,  -6, 27, 144, 89,  -4, 16, 134, 106, 0,
       7,  121, 121, 7, 0,  106, 134, 16, -4, 89,  144, 27, -6, 73,  148, 41},
      // Coefficients 6 taps
      {-16, 67,  154, 67,  -16, 0,   -17, 52,  154, 85,  -12, -6,  -18, 37,  151, 102,
       -7,  -9,  -16, 23,  142, 118, 1,   -12, -14, 11,  131, 131, 11,  -14, -12, 1,
       118, 142, 23,  -16, -9,  -7,  102, 151, 37,  -18, -6,  -12, 85,  154, 52,  -17},
    },

    // Band_8_1678
    {
      // Scaler inc range
      {1675, 1678},
      // Coefficients 4 taps
      {55, 146, 55,  0, 41, 148, 73,  -6, 28, 143, 89,  -4, 16, 133, 106, 1,
       7,  121, 121, 7, 1,  106, 133, 16, -4, 89,  143, 28, -6, 73,  148, 41},
      // Coefficients 6 taps
      {-15, 67,  152, 67,  -15, 0,   -17, 52,  154, 85,  -12, -6,  -17, 37,  150, 102,
       -7,  -9,  -16, 23,  143, 117, 1,   -12, -14, 11,  131, 131, 11,  -14, -12, 1,
       117, 143, 23,  -16, -9,  -7,  102, 150, 37,  -17, -6,  -12, 85,  154, 52,  -17},
    },

    // Band_8_1680
    {
      // Scaler inc range
      {1678, 1680},
      // Coefficients 4 taps
      {55, 146, 55,  0, 41, 148, 73,  -6, 28, 143, 89,  -4, 16, 133, 106, 1,
       7,  121, 121, 7, 1,  106, 133, 16, -4, 89,  143, 28, -6, 73,  148, 41},
      // Coefficients 6 taps
      {-15, 67,  152, 67,  -15, 0,   -17, 52,  154, 85,  -12, -6,  -17, 37,  150, 102,
       -7,  -9,  -16, 23,  143, 117, 1,   -12, -14, 11,  131, 131, 11,  -14, -12, 1,
       117, 143, 23,  -16, -9,  -7,  102, 150, 37,  -17, -6,  -12, 85,  154, 52,  -17},
    },

    // Band_8_1683
    {
      // Scaler inc range
      {1680, 1683},
      // Coefficients 4 taps
      {55, 146, 55,  0, 41, 148, 73,  -6, 28, 143, 89,  -4, 16, 133, 106, 1,
       7,  121, 121, 7, 1,  106, 133, 16, -4, 89,  143, 28, -6, 73,  148, 41},
      // Coefficients 6 taps
      {-15, 67,  152, 67,  -15, 0,   -17, 52,  154, 85,  -12, -6,  -17, 37,  149, 102,
       -6,  -9,  -16, 23,  143, 117, 1,   -12, -14, 11,  131, 131, 11,  -14, -12, 1,
       117, 143, 23,  -16, -9,  -6,  102, 149, 37,  -17, -6,  -12, 85,  154, 52,  -17},
    },

    // Band_8_1686
    {
      // Scaler inc range
      {1683, 1686},
      // Coefficients 4 taps
      {55, 146, 55,  0, 41, 148, 73,  -6, 28, 142, 89,  -3, 17, 132, 106, 1,
       8,  120, 120, 8, 1,  106, 132, 17, -3, 89,  142, 28, -6, 73,  148, 41},
      // Coefficients 6 taps
      {-15, 67,  152, 67,  -15, 0,   -17, 52,  154, 85,  -12, -6,  -17, 37,  150, 101,
       -6,  -9,  -16, 24,  142, 117, 1,   -12, -14, 12,  130, 130, 12,  -14, -12, 1,
       117, 142, 24,  -16, -9,  -6,  101, 150, 37,  -17, -6,  -12, 85,  154, 52,  -17},
    },

    // Band_8_1689
    {
      // Scaler inc range
      {1686, 1689},
      // Coefficients 4 taps
      {55, 146, 55,  0, 41, 148, 73,  -6, 28, 142, 89,  -3, 17, 132, 106, 1,
       8,  120, 120, 8, 1,  106, 132, 17, -3, 89,  142, 28, -6, 73,  148, 41},
      // Coefficients 6 taps
      {-15, 67,  152, 67,  -15, 0,   -17, 52,  154, 85,  -12, -6,  -17, 38,  149, 101,
       -6,  -9,  -16, 24,  141, 117, 2,   -12, -14, 12,  130, 130, 12,  -14, -12, 2,
       117, 141, 24,  -16, -9,  -6,  101, 149, 38,  -17, -6,  -12, 85,  154, 52,  -17},
    },

    // Band_8_1691
    {
      // Scaler inc range
      {1689, 1691},
      // Coefficients 4 taps
      {55, 146, 55,  0, 41, 148, 73,  -6, 28, 142, 89,  -3, 17, 132, 106, 1,
       8,  120, 120, 8, 1,  106, 132, 17, -3, 89,  142, 28, -6, 73,  148, 41},
      // Coefficients 6 taps
      {-15, 67,  152, 67,  -15, 0,   -17, 53,  153, 85,  -12, -6,  -17, 38,  149, 101,
       -6,  -9,  -16, 24,  141, 117, 2,   -12, -14, 12,  130, 130, 12,  -14, -12, 2,
       117, 141, 24,  -16, -9,  -6,  101, 149, 38,  -17, -6,  -12, 85,  153, 53,  -17},
    },

    // Band_8_1694
    {
      // Scaler inc range
      {1691, 1694},
      // Coefficients 4 taps
      {55, 146, 55,  0, 42, 147, 73,  -6, 28, 142, 89,  -3, 17, 133, 105, 1,
       8,  120, 120, 8, 1,  105, 133, 17, -3, 89,  142, 28, -6, 73,  147, 42},
      // Coefficients 6 taps
      {-15, 67,  152, 67,  -15, 0,   -17, 53,  152, 85,  -11, -6,  -17, 38,  149, 101,
       -6,  -9,  -16, 24,  141, 117, 2,   -12, -14, 12,  130, 130, 12,  -14, -12, 2,
       117, 141, 24,  -16, -9,  -6,  101, 149, 38,  -17, -6,  -11, 85,  152, 53,  -17},
    },

    // Band_8_1697
    {
      // Scaler inc range
      {1694, 1697},
      // Coefficients 4 taps
      {55, 146, 55,  0, 42, 147, 73,  -6, 28, 142, 89,  -3, 17, 133, 105, 1,
       8,  120, 120, 8, 1,  105, 133, 17, -3, 89,  142, 28, -6, 73,  147, 42},
      // Coefficients 6 taps
      {-15, 67,  152, 67,  -15, 0,   -17, 53,  152, 85,  -11, -6,  -17, 38,  149, 101,
       -6,  -9,  -16, 24,  141, 117, 2,   -12, -14, 12,  130, 130, 12,  -14, -12, 2,
       117, 141, 24,  -16, -9,  -6,  101, 149, 38,  -17, -6,  -11, 85,  152, 53,  -17},
    },

    // Band_8_1699
    {
      // Scaler inc range
      {1697, 1699},
      // Coefficients 4 taps
      {55, 146, 55,  0, 42, 146, 73,  -5, 28, 142, 89,  -3, 17, 133, 105, 1,
       8,  120, 120, 8, 1,  105, 133, 17, -3, 89,  142, 28, -5, 73,  146, 42},
      // Coefficients 6 taps
      {-14, 67,  150, 67,  -14, 0,   -17, 53,  152, 85,  -11, -6,  -17, 38,  148, 101,
       -5,  -9,  -16, 24,  142, 116, 2,   -12, -14, 12,  130, 130, 12,  -14, -12, 2,
       116, 142, 24,  -16, -9,  -5,  101, 148, 38,  -17, -6,  -11, 85,  152, 53,  -17},
    },

    // Band_8_1702
    {
      // Scaler inc range
      {1699, 1702},
      // Coefficients 4 taps
      {55, 146, 55,  0, 42, 146, 73,  -5, 29, 141, 89,  -3, 17, 132, 105, 2,
       8,  120, 120, 8, 2,  105, 132, 17, -3, 89,  141, 29, -5, 73,  146, 42},
      // Coefficients 6 taps
      {-14, 67,  150, 67,  -14, 0,   -17, 53,  153, 85,  -11, -7,  -17, 38,  148, 101,
       -5,  -9,  -16, 25,  140, 116, 3,   -12, -14, 13,  129, 129, 13,  -14, -12, 3,
       116, 140, 25,  -16, -9,  -5,  101, 148, 38,  -17, -7,  -11, 85,  153, 53,  -17},
    },

    // Band_8_1705
    {
      // Scaler inc range
      {1702, 1705},
      // Coefficients 4 taps
      {56, 144, 56,  0, 42, 146, 73,  -5, 29, 141, 89,  -3, 17, 132, 105, 2,
       8,  120, 120, 8, 2,  105, 132, 17, -3, 89,  141, 29, -5, 73,  146, 42},
      // Coefficients 6 taps
      {-14, 67,  150, 67,  -14, 0,   -16, 53,  152, 85,  -11, -7,  -17, 38,  148, 101,
       -5,  -9,  -16, 25,  140, 116, 3,   -12, -14, 13,  129, 129, 13,  -14, -12, 3,
       116, 140, 25,  -16, -9,  -5,  101, 148, 38,  -17, -7,  -11, 85,  152, 53,  -16},
    },

    // Band_8_1708
    {
      // Scaler inc range
      {1705, 1708},
      // Coefficients 4 taps
      {56, 144, 56,  0, 42, 146, 73,  -5, 29, 141, 89,  -3, 18, 131, 105, 2,
       8,  120, 120, 8, 2,  105, 131, 18, -3, 89,  141, 29, -5, 73,  146, 42},
      // Coefficients 6 taps
      {-14, 67,  150, 67,  -14, 0,   -16, 53,  152, 85,  -11, -7,  -17, 38,  148, 101,
       -5,  -9,  -16, 25,  140, 116, 3,   -12, -14, 13,  129, 129, 13,  -14, -12, 3,
       116, 140, 25,  -16, -9,  -5,  101, 148, 38,  -17, -7,  -11, 85,  152, 53,  -16},
    },

    // Band_8_1711
    {
      // Scaler inc range
      {1708, 1711},
      // Coefficients 4 taps
      {56, 144, 56,  0, 42, 146, 73,  -5, 29, 141, 89,  -3, 18, 131, 105, 2,
       9,  119, 119, 9, 2,  105, 131, 18, -3, 89,  141, 29, -5, 73,  146, 42},
      // Coefficients 6 taps
      {-14, 67,  150, 67,  -14, 0,   -16, 53,  151, 85,  -10, -7,  -17, 39,  147, 101,
       -5,  -9,  -16, 25,  140, 116, 3,   -12, -14, 13,  129, 129, 13,  -14, -12, 3,
       116, 140, 25,  -16, -9,  -5,  101, 147, 39,  -17, -7,  -10, 85,  151, 53,  -16},
    },

    // Band_8_1713
    {
      // Scaler inc range
      {1711, 1713},
      // Coefficients 4 taps
      {56, 144, 56,  0, 42, 146, 73,  -5, 29, 141, 89,  -3, 18, 131, 105, 2,
       9,  119, 119, 9, 2,  105, 131, 18, -3, 89,  141, 29, -5, 73,  146, 42},
      // Coefficients 6 taps
      {-14, 67,  150, 67,  -14, 0,   -16, 53,  151, 85,  -10, -7,  -17, 39,  147, 101,
       -5,  -9,  -16, 25,  140, 116, 3,   -12, -14, 13,  129, 129, 13,  -14, -12, 3,
       116, 140, 25,  -16, -9,  -5,  101, 147, 39,  -17, -7,  -10, 85,  151, 53,  -16},
    },

    // Band_8_1716
    {
      // Scaler inc range
      {1713, 1716},
      // Coefficients 4 taps
      {56, 144, 56,  0, 42, 146, 73,  -5, 29, 140, 89,  -2, 18, 131, 105, 2,
       9,  119, 119, 9, 2,  105, 131, 18, -2, 89,  140, 29, -5, 73,  146, 42},
      // Coefficients 6 taps
      {-14, 67,  150, 67,  -14, 0,   -16, 53,  151, 85,  -10, -7,  -17, 39,  146, 101,
       -4,  -9,  -16, 25,  140, 116, 3,   -12, -14, 14,  128, 128, 14,  -14, -12, 3,
       116, 140, 25,  -16, -9,  -4,  101, 146, 39,  -17, -7,  -10, 85,  151, 53,  -16},
    },

    // Band_8_1719
    {
      // Scaler inc range
      {1716, 1719},
      // Coefficients 4 taps
      {56, 144, 56,  0, 42, 146, 73,  -5, 29, 140, 89,  -2, 18, 131, 105, 2,
       9,  119, 119, 9, 2,  105, 131, 18, -2, 89,  140, 29, -5, 73,  146, 42},
      // Coefficients 6 taps
      {-14, 67,  150, 67,  -14, 0,   -16, 54,  150, 85,  -10, -7,  -17, 39,  146, 101,
       -4,  -9,  -16, 26,  138, 116, 4,   -12, -14, 14,  128, 128, 14,  -14, -12, 4,
       116, 138, 26,  -16, -9,  -4,  101, 146, 39,  -17, -7,  -10, 85,  150, 54,  -16},
    },

    // Band_8_1722
    {
      // Scaler inc range
      {1719, 1722},
      // Coefficients 4 taps
      {56, 144, 56,  0, 43, 145, 73,  -5, 29, 140, 89,  -2, 18, 131, 105, 2,
       9,  119, 119, 9, 2,  105, 131, 18, -2, 89,  140, 29, -5, 73,  145, 43},
      // Coefficients 6 taps
      {-13, 67,  148, 67,  -13, 0,   -16, 54,  150, 85,  -10, -7,  -16, 39,  145, 101,
       -4,  -9,  -16, 26,  139, 115, 4,   -12, -14, 14,  128, 128, 14,  -14, -12, 4,
       115, 139, 26,  -16, -9,  -4,  101, 145, 39,  -16, -7,  -10, 85,  150, 54,  -16},
    },

    // Band_8_1725
    {
      // Scaler inc range
      {1722, 1725},
      // Coefficients 4 taps
      {56, 144, 56,  0, 43, 145, 73,  -5, 29, 140, 89,  -2, 18, 131, 105, 2,
       9,  119, 119, 9, 2,  105, 131, 18, -2, 89,  140, 29, -5, 73,  145, 43},
      // Coefficients 6 taps
      {-13, 67,  148, 67,  -13, 0,   -16, 54,  150, 85,  -10, -7,  -16, 39,  145, 101,
       -4,  -9,  -16, 26,  139, 115, 4,   -12, -14, 14,  128, 128, 14,  -14, -12, 4,
       115, 139, 26,  -16, -9,  -4,  101, 145, 39,  -16, -7,  -10, 85,  150, 54,  -16},
    },

    // Band_8_1727
    {
      // Scaler inc range
      {1725, 1727},
      // Coefficients 4 taps
      {56, 144, 56,  0, 43, 145, 73,  -5, 30, 139, 89,  -2, 18, 130, 105, 3,
       9,  119, 119, 9, 3,  105, 130, 18, -2, 89,  139, 30, -5, 73,  145, 43},
      // Coefficients 6 taps
      {-13, 67,  148, 67,  -13, 0,   -16, 54,  150, 85,  -10, -7,  -16, 39,  146, 100,
       -4,  -9,  -16, 26,  139, 115, 4,   -12, -14, 14,  128, 128, 14,  -14, -12, 4,
       115, 139, 26,  -16, -9,  -4,  100, 146, 39,  -16, -7,  -10, 85,  150, 54,  -16},
    },

    // Band_8_1730
    {
      // Scaler inc range
      {1727, 1730},
      // Coefficients 4 taps
      {56, 144, 56,  0, 43, 145, 73,  -5, 30, 139, 89,  -2, 18, 130, 105, 3,
       9,  119, 119, 9, 3,  105, 130, 18, -2, 89,  139, 30, -5, 73,  145, 43},
      // Coefficients 6 taps
      {-13, 67, 148, 67,  -13, 0,   -16, 54, 149, 85,  -9, -7,  -16, 40, 146, 100, -4, -10, -16, 26, 139, 115, 4,  -12,
       -14, 14, 128, 128, 14,  -14, -12, 4,  115, 139, 26, -16, -10, -4, 100, 146, 40, -16, -7,  -9, 85,  149, 54, -16},
    },

    // Band_8_1733
    {
      // Scaler inc range
      {1730, 1733},
      // Coefficients 4 taps
      {56, 144, 56,  0,  43, 145, 73,  -5, 30, 139, 89,  -2, 19, 130, 104, 3,
       10, 118, 118, 10, 3,  104, 130, 19, -2, 89,  139, 30, -5, 73,  145, 43},
      // Coefficients 6 taps
      {-13, 67, 148, 67,  -13, 0,   -15, 54, 148, 85,  -9, -7,  -16, 40, 145, 100, -3, -10, -16, 26, 138, 115, 5,  -12,
       -14, 15, 127, 127, 15,  -14, -12, 5,  115, 138, 26, -16, -10, -3, 100, 145, 40, -16, -7,  -9, 85,  148, 54, -15},
    },

    // Band_8_1736
    {
      // Scaler inc range
      {1733, 1736},
      // Coefficients 4 taps
      {56, 144, 56,  0,  43, 144, 73,  -4, 30, 139, 89,  -2, 19, 130, 104, 3,
       10, 118, 118, 10, 3,  104, 130, 19, -2, 89,  139, 30, -4, 73,  144, 43},
      // Coefficients 6 taps
      {-13, 67, 148, 67,  -13, 0,   -15, 54, 148, 85,  -9, -7,  -16, 40, 145, 100, -3, -10, -16, 27, 137, 115, 5,  -12,
       -14, 15, 127, 127, 15,  -14, -12, 5,  115, 137, 27, -16, -10, -3, 100, 145, 40, -16, -7,  -9, 85,  148, 54, -15},
    },

    // Band_8_1739
    {
      // Scaler inc range
      {1736, 1739},
      // Coefficients 4 taps
      {57, 142, 57,  0,  43, 144, 73,  -4, 30, 139, 89,  -2, 19, 130, 104, 3,
       10, 118, 118, 10, 3,  104, 130, 19, -2, 89,  139, 30, -4, 73,  144, 43},
      // Coefficients 6 taps
      {-13, 68, 146, 68,  -13, 0,   -15, 54, 148, 85,  -9, -7,  -16, 40, 145, 100, -3, -10, -16, 27, 137, 115, 5,  -12,
       -14, 15, 127, 127, 15,  -14, -12, 5,  115, 137, 27, -16, -10, -3, 100, 145, 40, -16, -7,  -9, 85,  148, 54, -15},
    },

    // Band_8_1742
    {
      // Scaler inc range
      {1739, 1742},
      // Coefficients 4 taps
      {57, 142, 57,  0,  43, 144, 73,  -4, 30, 139, 89,  -2, 19, 130, 104, 3,
       10, 118, 118, 10, 3,  104, 130, 19, -2, 89,  139, 30, -4, 73,  144, 43},
      // Coefficients 6 taps
      {-13, 68, 146, 68,  -13, 0,   -15, 54, 148, 85,  -9, -7,  -16, 40, 145, 100, -3, -10, -16, 27, 138, 114, 5,  -12,
       -14, 15, 127, 127, 15,  -14, -12, 5,  114, 138, 27, -16, -10, -3, 100, 145, 40, -16, -7,  -9, 85,  148, 54, -15},
    },

    // Band_8_1745
    {
      // Scaler inc range
      {1742, 1745},
      // Coefficients 4 taps
      {57, 142, 57,  0,  43, 144, 73,  -4, 30, 138, 89,  -1, 19, 130, 104, 3,
       10, 118, 118, 10, 3,  104, 130, 19, -1, 89,  138, 30, -4, 73,  144, 43},
      // Coefficients 6 taps
      {-12, 68, 144, 68,  -12, 0,   -15, 54, 148, 85,  -9, -7,  -16, 40, 145, 100, -3, -10, -15, 27, 137, 114, 5,  -12,
       -14, 15, 127, 127, 15,  -14, -12, 5,  114, 137, 27, -15, -10, -3, 100, 145, 40, -16, -7,  -9, 85,  148, 54, -15},
    },

    // Band_8_1748
    {
      // Scaler inc range
      {1745, 1748},
      // Coefficients 4 taps
      {57, 142, 57,  0,  43, 144, 73,  -4, 30, 138, 89,  -1, 19, 130, 104, 3,
       10, 118, 118, 10, 3,  104, 130, 19, -1, 89,  138, 30, -4, 73,  144, 43},
      // Coefficients 6 taps
      {-12, 68, 144, 68,  -12, 0,   -15, 54, 147, 85,  -8, -7,  -16, 40, 145, 100, -3, -10, -15, 27, 137, 114, 5,  -12,
       -14, 15, 127, 127, 15,  -14, -12, 5,  114, 137, 27, -15, -10, -3, 100, 145, 40, -16, -7,  -8, 85,  147, 54, -15},
    },

    // Band_8_1751
    {
      // Scaler inc range
      {1748, 1751},
      // Coefficients 4 taps
      {57, 142, 57,  0,  43, 144, 73,  -4, 30, 138, 89,  -1, 19, 130, 104, 3,
       10, 118, 118, 10, 3,  104, 130, 19, -1, 89,  138, 30, -4, 73,  144, 43},
      // Coefficients 6 taps
      {-12, 68, 144, 68,  -12, 0,   -15, 54, 147, 85,  -8, -7,  -16, 40, 144, 100, -2, -10, -15, 27, 136, 114, 6,  -12,
       -14, 16, 126, 126, 16,  -14, -12, 6,  114, 136, 27, -15, -10, -2, 100, 144, 40, -16, -7,  -8, 85,  147, 54, -15},
    },

    // Band_8_1753
    {
      // Scaler inc range
      {1751, 1753},
      // Coefficients 4 taps
      {57, 142, 57,  0,  44, 143, 73,  -4, 31, 137, 89,  -1, 19, 129, 104, 4,
       10, 118, 118, 10, 4,  104, 129, 19, -1, 89,  137, 31, -4, 73,  143, 44},
      // Coefficients 6 taps
      {-12, 68, 144, 68,  -12, 0,   -15, 55, 146, 85,  -8, -7,  -16, 41, 143, 100, -2, -10, -15, 27, 136, 114, 6,  -12,
       -14, 16, 126, 126, 16,  -14, -12, 6,  114, 136, 27, -15, -10, -2, 100, 143, 41, -16, -7,  -8, 85,  146, 55, -15},
    },

    // Band_8_1756
    {
      // Scaler inc range
      {1753, 1756},
      // Coefficients 4 taps
      {57, 142, 57,  0,  44, 143, 73,  -4, 31, 137, 89,  -1, 20, 128, 104, 4,
       11, 117, 117, 11, 4,  104, 128, 20, -1, 89,  137, 31, -4, 73,  143, 44},
      // Coefficients 6 taps
      {-12, 68, 144, 68,  -12, 0,   -15, 55, 146, 85,  -8, -7,  -16, 41, 143, 100, -2, -10, -15, 28, 135, 114, 6,  -12,
       -14, 16, 126, 126, 16,  -14, -12, 6,  114, 135, 28, -15, -10, -2, 100, 143, 41, -16, -7,  -8, 85,  146, 55, -15},
    },

    // Band_8_1759
    {
      // Scaler inc range
      {1756, 1759},
      // Coefficients 4 taps
      {57, 142, 57,  0,  44, 143, 73,  -4, 31, 137, 89,  -1, 20, 128, 104, 4,
       11, 117, 117, 11, 4,  104, 128, 20, -1, 89,  137, 31, -4, 73,  143, 44},
      // Coefficients 6 taps
      {-12, 68, 144, 68,  -12, 0,   -14, 55, 145, 85,  -8, -7,  -16, 41, 143, 100, -2, -10, -15, 28, 135, 114, 6,  -12,
       -14, 16, 126, 126, 16,  -14, -12, 6,  114, 135, 28, -15, -10, -2, 100, 143, 41, -16, -7,  -8, 85,  145, 55, -14},
    },

    // Band_8_1762
    {
      // Scaler inc range
      {1759, 1762},
      // Coefficients 4 taps
      {57, 142, 57,  0,  44, 143, 73,  -4, 31, 137, 89,  -1, 20, 128, 104, 4,
       11, 117, 117, 11, 4,  104, 128, 20, -1, 89,  137, 31, -4, 73,  143, 44},
      // Coefficients 6 taps
      {-12, 68, 144, 68,  -12, 0,   -14, 55, 146, 84,  -8, -7,  -15, 41, 142, 100, -2, -10, -15, 28, 135, 114, 6,  -12,
       -14, 16, 126, 126, 16,  -14, -12, 6,  114, 135, 28, -15, -10, -2, 100, 142, 41, -15, -7,  -8, 84,  146, 55, -14},
    },

    // Band_8_1765
    {
      // Scaler inc range
      {1762, 1765},
      // Coefficients 4 taps
      {57, 142, 57,  0,  44, 143, 73,  -4, 31, 137, 89,  -1, 20, 128, 104, 4,
       11, 117, 117, 11, 4,  104, 128, 20, -1, 89,  137, 31, -4, 73,  143, 44},
      // Coefficients 6 taps
      {-11, 68, 142, 68,  -11, 0,   -14, 55, 146, 84,  -8, -7,  -15, 41, 143, 99,  -2, -10, -15, 28, 135, 113, 7,  -12,
       -14, 16, 126, 126, 16,  -14, -12, 7,  113, 135, 28, -15, -10, -2, 99,  143, 41, -15, -7,  -8, 84,  146, 55, -14},
    },

    // Band_8_1768
    {
      // Scaler inc range
      {1765, 1768},
      // Coefficients 4 taps
      {57, 142, 57,  0,  44, 142, 74,  -4, 31, 137, 89,  -1, 20, 128, 104, 4,
       11, 117, 117, 11, 4,  104, 128, 20, -1, 89,  137, 31, -4, 74,  142, 44},
      // Coefficients 6 taps
      {-11, 68, 142, 68,  -11, 0,   -14, 55, 145, 84,  -7, -7,  -15, 41, 142, 99,  -1, -10, -15, 28, 135, 113, 7,  -12,
       -14, 17, 125, 125, 17,  -14, -12, 7,  113, 135, 28, -15, -10, -1, 99,  142, 41, -15, -7,  -7, 84,  145, 55, -14},
    },

    // Band_8_1771
    {
      // Scaler inc range
      {1768, 1771},
      // Coefficients 4 taps
      {57, 142, 57,  0,  44, 142, 74,  -4, 31, 137, 89,  -1, 20, 128, 104, 4,
       11, 117, 117, 11, 4,  104, 128, 20, -1, 89,  137, 31, -4, 74,  142, 44},
      // Coefficients 6 taps
      {-11, 68, 142, 68,  -11, 0,   -14, 55, 146, 84,  -7, -8,  -15, 41, 142, 99,  -1, -10, -15, 28, 135, 113, 7,  -12,
       -14, 17, 125, 125, 17,  -14, -12, 7,  113, 135, 28, -15, -10, -1, 99,  142, 41, -15, -8,  -7, 84,  146, 55, -14},
    },

    // Band_8_1774
    {
      // Scaler inc range
      {1771, 1774},
      // Coefficients 4 taps
      {57, 142, 57,  0,  44, 142, 74,  -4, 31, 137, 89,  -1, 20, 129, 103, 4,
       11, 117, 117, 11, 4,  103, 129, 20, -1, 89,  137, 31, -4, 74,  142, 44},
      // Coefficients 6 taps
      {-11, 68, 142, 68,  -11, 0,   -14, 55, 146, 84,  -7, -8,  -15, 41, 142, 99,  -1, -10, -15, 29, 134, 113, 7,  -12,
       -14, 17, 125, 125, 17,  -14, -12, 7,  113, 134, 29, -15, -10, -1, 99,  142, 41, -15, -8,  -7, 84,  146, 55, -14},
    },

    // Band_8_1777
    {
      // Scaler inc range
      {1774, 1777},
      // Coefficients 4 taps
      {58, 140, 58,  0,  44, 141, 74,  -3, 31, 136, 89,  0,  20, 129, 103, 4,
       11, 117, 117, 11, 4,  103, 129, 20, 0,  89,  136, 31, -3, 74,  141, 44},
      // Coefficients 6 taps
      {-11, 68, 142, 68,  -11, 0,   -14, 55, 146, 84,  -7, -8,  -15, 42, 141, 99,  -1, -10, -15, 29, 134, 113, 7,  -12,
       -14, 17, 125, 125, 17,  -14, -12, 7,  113, 134, 29, -15, -10, -1, 99,  141, 42, -15, -8,  -7, 84,  146, 55, -14},
    },

    // Band_8_1780
    {
      // Scaler inc range
      {1777, 1780},
      // Coefficients 4 taps
      {58, 140, 58,  0,  44, 141, 74,  -3, 32, 135, 89,  0,  20, 128, 103, 5,
       11, 117, 117, 11, 5,  103, 128, 20, 0,  89,  135, 32, -3, 74,  141, 44},
      // Coefficients 6 taps
      {-11, 68, 142, 68,  -11, 0,   -14, 55, 146, 84,  -7, -8,  -15, 42, 141, 99,  -1, -10, -15, 29, 134, 113, 7,  -12,
       -14, 17, 125, 125, 17,  -14, -12, 7,  113, 134, 29, -15, -10, -1, 99,  141, 42, -15, -8,  -7, 84,  146, 55, -14},
    },

    // Band_8_1783
    {
      // Scaler inc range
      {1780, 1783},
      // Coefficients 4 taps
      {58, 140, 58,  0,  44, 141, 74,  -3, 32, 135, 89,  0,  21, 127, 103, 5,
       12, 116, 116, 12, 5,  103, 127, 21, 0,  89,  135, 32, -3, 74,  141, 44},
      // Coefficients 6 taps
      {-11, 68, 142, 68,  -11, 0,   -14, 55, 146, 84,  -7, -8,  -15, 42, 140, 99,  0,  -10, -15, 29, 133, 113, 8,  -12,
       -14, 17, 125, 125, 17,  -14, -12, 8,  113, 133, 29, -15, -10, 0,  99,  140, 42, -15, -8,  -7, 84,  146, 55, -14},
    },

    // Band_8_1786
    {
      // Scaler inc range
      {1783, 1786},
      // Coefficients 4 taps
      {58, 140, 58,  0,  45, 140, 74,  -3, 32, 135, 89,  0,  21, 127, 103, 5,
       12, 116, 116, 12, 5,  103, 127, 21, 0,  89,  135, 32, -3, 74,  140, 45},
      // Coefficients 6 taps
      {-11, 68, 142, 68,  -11, 0,   -13, 55, 144, 84,  -6, -8,  -15, 42, 140, 99,  0,  -10, -15, 29, 134, 112, 8,  -12,
       -14, 18, 124, 124, 18,  -14, -12, 8,  112, 134, 29, -15, -10, 0,  99,  140, 42, -15, -8,  -6, 84,  144, 55, -13},
    },

    // Band_8_1789
    {
      // Scaler inc range
      {1786, 1789},
      // Coefficients 4 taps
      {58, 140, 58,  0,  45, 140, 74,  -3, 32, 135, 89,  0,  21, 127, 103, 5,
       12, 116, 116, 12, 5,  103, 127, 21, 0,  89,  135, 32, -3, 74,  140, 45},
      // Coefficients 6 taps
      {-10, 68, 140, 68,  -10, 0,   -13, 56, 143, 84,  -6, -8,  -15, 42, 140, 99,  0,  -10, -15, 29, 134, 112, 8,  -12,
       -14, 18, 124, 124, 18,  -14, -12, 8,  112, 134, 29, -15, -10, 0,  99,  140, 42, -15, -8,  -6, 84,  143, 56, -13},
    },

    // Band_8_1792
    {
      // Scaler inc range
      {1789, 1792},
      // Coefficients 4 taps
      {58, 140, 58,  0,  45, 140, 74,  -3, 32, 135, 89,  0,  21, 127, 103, 5,
       12, 116, 116, 12, 5,  103, 127, 21, 0,  89,  135, 32, -3, 74,  140, 45},
      // Coefficients 6 taps
      {-10, 68, 140, 68,  -10, 0,   -13, 56, 143, 84,  -6, -8,  -15, 42, 140, 99,  0,  -10, -15, 29, 134, 112, 8,  -12,
       -14, 18, 124, 124, 18,  -14, -12, 8,  112, 134, 29, -15, -10, 0,  99,  140, 42, -15, -8,  -6, 84,  143, 56, -13},
    },

    // Band_8_1796
    {
      // Scaler inc range
      {1792, 1796},
      // Coefficients 4 taps
      {58, 140, 58,  0,  45, 140, 74,  -3, 32, 135, 89,  0,  21, 127, 103, 5,
       12, 116, 116, 12, 5,  103, 127, 21, 0,  89,  135, 32, -3, 74,  140, 45},
      // Coefficients 6 taps
      {-10, 68, 140, 68,  -10, 0,   -13, 56, 143, 84,  -6, -8,  -15, 42, 140, 99,  0,  -10, -15, 30, 133, 112, 8,  -12,
       -14, 18, 124, 124, 18,  -14, -12, 8,  112, 133, 30, -15, -10, 0,  99,  140, 42, -15, -8,  -6, 84,  143, 56, -13},
    },

    // Band_8_1799
    {
      // Scaler inc range
      {1796, 1799},
      // Coefficients 4 taps
      {58, 140, 58,  0,  45, 140, 74,  -3, 32, 135, 89,  0,  21, 127, 103, 5,
       12, 116, 116, 12, 5,  103, 127, 21, 0,  89,  135, 32, -3, 74,  140, 45},
      // Coefficients 6 taps
      {-10, 68, 140, 68,  -10, 0,   -13, 56, 143, 84,  -6, -8,  -14, 42, 139, 99,  0,  -10, -15, 30, 133, 112, 8,  -12,
       -14, 18, 124, 124, 18,  -14, -12, 8,  112, 133, 30, -15, -10, 0,  99,  139, 42, -14, -8,  -6, 84,  143, 56, -13},
    },

    // Band_8_1802
    {
      // Scaler inc range
      {1799, 1802},
      // Coefficients 4 taps
      {58, 140, 58,  0,  45, 140, 74,  -3, 32, 135, 89,  0,  21, 127, 103, 5,
       12, 116, 116, 12, 5,  103, 127, 21, 0,  89,  135, 32, -3, 74,  140, 45},
      // Coefficients 6 taps
      {-10, 68, 140, 68,  -10, 0,   -13, 56, 143, 84,  -6, -8,  -14, 42, 139, 98,  1,  -10, -15, 30, 132, 112, 9,  -12,
       -14, 18, 124, 124, 18,  -14, -12, 9,  112, 132, 30, -15, -10, 1,  98,  139, 42, -14, -8,  -6, 84,  143, 56, -13},
    },

    // Band_8_1805
    {
      // Scaler inc range
      {1802, 1805},
      // Coefficients 4 taps
      {58, 140, 58,  0,  45, 140, 74,  -3, 32, 135, 89,  0,  21, 127, 103, 5,
       12, 116, 116, 12, 5,  103, 127, 21, 0,  89,  135, 32, -3, 74,  140, 45},
      // Coefficients 6 taps
      {-10, 68, 140, 68,  -10, 0,   -13, 56, 143, 84,  -6, -8,  -14, 43, 138, 98,  1,  -10, -14, 30, 131, 112, 9,  -12,
       -14, 19, 123, 123, 19,  -14, -12, 9,  112, 131, 30, -14, -10, 1,  98,  138, 43, -14, -8,  -6, 84,  143, 56, -13},
    },

    // Band_8_1808
    {
      // Scaler inc range
      {1805, 1808},
      // Coefficients 4 taps
      {58, 140, 58,  0,  45, 140, 74,  -3, 33, 134, 89,  0,  22, 126, 103, 5,
       12, 116, 116, 12, 5,  103, 126, 22, 0,  89,  134, 33, -3, 74,  140, 45},
      // Coefficients 6 taps
      {-10, 68, 140, 68,  -10, 0,   -13, 56, 142, 84,  -5, -8,  -14, 43, 138, 98,  1,  -10, -14, 30, 132, 111, 9,  -12,
       -14, 19, 123, 123, 19,  -14, -12, 9,  111, 132, 30, -14, -10, 1,  98,  138, 43, -14, -8,  -5, 84,  142, 56, -13},
    },

    // Band_8_1811
    {
      // Scaler inc range
      {1808, 1811},
      // Coefficients 4 taps
      {58, 140, 58,  0,  45, 140, 74,  -3, 33, 133, 89,  1,  22, 125, 103, 6,
       13, 115, 115, 13, 6,  103, 125, 22, 1,  89,  133, 33, -3, 74,  140, 45},
      // Coefficients 6 taps
      {-9,  68, 138, 68,  -9, 0,   -13, 56, 142, 84,  -5, -8,  -14, 43, 138, 98,  1,  -10, -14, 30, 132, 111, 9,  -12,
       -14, 19, 123, 123, 19, -14, -12, 9,  111, 132, 30, -14, -10, 1,  98,  138, 43, -14, -8,  -5, 84,  142, 56, -13},
    },

    // Band_8_1814
    {
      // Scaler inc range
      {1811, 1814},
      // Coefficients 4 taps
      {58, 140, 58,  0,  45, 140, 74,  -3, 33, 134, 88,  1,  22, 125, 103, 6,
       13, 115, 115, 13, 6,  103, 125, 22, 1,  88,  134, 33, -3, 74,  140, 45},
      // Coefficients 6 taps
      {-9,  68, 138, 68,  -9, 0,   -12, 56, 141, 84,  -5, -8,  -14, 43, 138, 98,  1,  -10, -14, 30, 132, 111, 9,  -12,
       -14, 19, 123, 123, 19, -14, -12, 9,  111, 132, 30, -14, -10, 1,  98,  138, 43, -14, -8,  -5, 84,  141, 56, -12},
    },

    // Band_8_1817
    {
      // Scaler inc range
      {1814, 1817},
      // Coefficients 4 taps
      {59, 138, 59,  0,  45, 139, 74,  -2, 33, 134, 88,  1,  22, 126, 102, 6,
       13, 115, 115, 13, 6,  102, 126, 22, 1,  88,  134, 33, -2, 74,  139, 45},
      // Coefficients 6 taps
      {-9,  68, 138, 68,  -9, 0,   -12, 56, 141, 84,  -5, -8,  -14, 43, 138, 98,  1,  -10, -14, 31, 131, 111, 9,  -12,
       -13, 19, 122, 122, 19, -13, -12, 9,  111, 131, 31, -14, -10, 1,  98,  138, 43, -14, -8,  -5, 84,  141, 56, -12},
    },

    // Band_8_1820
    {
      // Scaler inc range
      {1817, 1820},
      // Coefficients 4 taps
      {59, 138, 59,  0,  46, 138, 74,  -2, 33, 134, 88,  1,  22, 126, 102, 6,
       13, 115, 115, 13, 6,  102, 126, 22, 1,  88,  134, 33, -2, 74,  138, 46},
      // Coefficients 6 taps
      {-9,  68, 138, 68,  -9, 0,   -12, 56, 141, 84,  -5, -8,  -14, 43, 137, 98,  2,  -10, -14, 31, 130, 111, 10, -12,
       -13, 19, 122, 122, 19, -13, -12, 10, 111, 130, 31, -14, -10, 2,  98,  137, 43, -14, -8,  -5, 84,  141, 56, -12},
    },

    // Band_8_1824
    {
      // Scaler inc range
      {1820, 1824},
      // Coefficients 4 taps
      {59, 138, 59,  0,  46, 138, 74,  -2, 33, 134, 88,  1,  22, 126, 102, 6,
       13, 115, 115, 13, 6,  102, 126, 22, 1,  88,  134, 33, -2, 74,  138, 46},
      // Coefficients 6 taps
      {-9,  68, 138, 68,  -9, 0,   -12, 56, 141, 84,  -5, -8,  -14, 43, 137, 98,  2,  -10, -14, 31, 130, 111, 10, -12,
       -13, 20, 121, 121, 20, -13, -12, 10, 111, 130, 31, -14, -10, 2,  98,  137, 43, -14, -8,  -5, 84,  141, 56, -12},
    },

    // Band_8_1827
    {
      // Scaler inc range
      {1824, 1827},
      // Coefficients 4 taps
      {59, 138, 59,  0,  46, 138, 74,  -2, 33, 134, 88,  1,  22, 126, 102, 6,
       13, 115, 115, 13, 6,  102, 126, 22, 1,  88,  134, 33, -2, 74,  138, 46},
      // Coefficients 6 taps
      {-9,  68, 138, 68,  -9, 0,   -12, 56, 140, 84,  -4, -8,  -14, 43, 137, 98,  2,  -10, -14, 31, 130, 111, 10, -12,
       -13, 20, 121, 121, 20, -13, -12, 10, 111, 130, 31, -14, -10, 2,  98,  137, 43, -14, -8,  -4, 84,  140, 56, -12},
    },

    // Band_8_1830
    {
      // Scaler inc range
      {1827, 1830},
      // Coefficients 4 taps
      {59, 138, 59,  0,  46, 138, 74,  -2, 33, 134, 88,  1,  22, 126, 102, 6,
       13, 115, 115, 13, 6,  102, 126, 22, 1,  88,  134, 33, -2, 74,  138, 46},
      // Coefficients 6 taps
      {-9,  68, 138, 68,  -9, 0,   -12, 56, 140, 84,  -4, -8,  -14, 43, 137, 98,  2,  -10, -14, 31, 131, 110, 10, -12,
       -13, 20, 121, 121, 20, -13, -12, 10, 110, 131, 31, -14, -10, 2,  98,  137, 43, -14, -8,  -4, 84,  140, 56, -12},
    },

    // Band_8_1833
    {
      // Scaler inc range
      {1830, 1833},
      // Coefficients 4 taps
      {59, 138, 59,  0,  46, 138, 74,  -2, 33, 134, 88,  1,  22, 126, 102, 6,
       13, 115, 115, 13, 6,  102, 126, 22, 1,  88,  134, 33, -2, 74,  138, 46},
      // Coefficients 6 taps
      {-8,  68, 136, 68,  -8, 0,   -12, 57, 139, 84,  -4, -8,  -13, 44, 135, 98,  2,  -10, -14, 31, 131, 110, 10, -12,
       -13, 20, 121, 121, 20, -13, -12, 10, 110, 131, 31, -14, -10, 2,  98,  135, 44, -13, -8,  -4, 84,  139, 57, -12},
    },

    // Band_8_1836
    {
      // Scaler inc range
      {1833, 1836},
      // Coefficients 4 taps
      {59, 138, 59,  0,  46, 138, 74,  -2, 33, 134, 88,  1,  23, 125, 102, 6,
       14, 114, 114, 14, 6,  102, 125, 23, 1,  88,  134, 33, -2, 74,  138, 46},
      // Coefficients 6 taps
      {-8,  68, 136, 68,  -8, 0,   -12, 57, 139, 84,  -4, -8,  -13, 44, 136, 97,  2,  -10, -14, 31, 131, 110, 10, -12,
       -13, 20, 121, 121, 20, -13, -12, 10, 110, 131, 31, -14, -10, 2,  97,  136, 44, -13, -8,  -4, 84,  139, 57, -12},
    },

    // Band_8_1840
    {
      // Scaler inc range
      {1836, 1840},
      // Coefficients 4 taps
      {59, 138, 59,  0,  46, 138, 74,  -2, 34, 133, 88,  1,  23, 124, 102, 7,
       14, 114, 114, 14, 7,  102, 124, 23, 1,  88,  133, 34, -2, 74,  138, 46},
      // Coefficients 6 taps
      {-8,  68, 136, 68,  -8, 0,   -12, 57, 139, 84,  -4, -8,  -13, 44, 135, 97,  3,  -10, -14, 32, 129, 110, 11, -12,
       -13, 20, 121, 121, 20, -13, -12, 11, 110, 129, 32, -14, -10, 3,  97,  135, 44, -13, -8,  -4, 84,  139, 57, -12},
    },

    // Band_8_1843
    {
      // Scaler inc range
      {1840, 1843},
      // Coefficients 4 taps
      {59, 138, 59,  0,  46, 138, 74,  -2, 34, 132, 88,  2,  23, 124, 102, 7,
       14, 114, 114, 14, 7,  102, 124, 23, 2,  88,  132, 34, -2, 74,  138, 46},
      // Coefficients 6 taps
      {-8,  68, 136, 68,  -8, 0,   -11, 57, 138, 84,  -4, -8,  -13, 44, 135, 97,  3,  -10, -14, 32, 129, 110, 11, -12,
       -13, 21, 120, 120, 21, -13, -12, 11, 110, 129, 32, -14, -10, 3,  97,  135, 44, -13, -8,  -4, 84,  138, 57, -11},
    },

    // Band_8_1846
    {
      // Scaler inc range
      {1843, 1846},
      // Coefficients 4 taps
      {59, 138, 59,  0,  46, 138, 74,  -2, 34, 132, 88,  2,  23, 124, 102, 7,
       14, 114, 114, 14, 7,  102, 124, 23, 2,  88,  132, 34, -2, 74,  138, 46},
      // Coefficients 6 taps
      {-8,  68, 136, 68,  -8, 0,   -11, 57, 137, 84,  -3, -8,  -13, 44, 135, 97,  3,  -10, -14, 32, 129, 110, 11, -12,
       -13, 21, 120, 120, 21, -13, -12, 11, 110, 129, 32, -14, -10, 3,  97,  135, 44, -13, -8,  -3, 84,  137, 57, -11},
    },

    // Band_8_1849
    {
      // Scaler inc range
      {1846, 1849},
      // Coefficients 4 taps
      {59, 138, 59,  0,  46, 138, 74,  -2, 34, 132, 88,  2,  23, 124, 102, 7,
       14, 114, 114, 14, 7,  102, 124, 23, 2,  88,  132, 34, -2, 74,  138, 46},
      // Coefficients 6 taps
      {-8,  68, 136, 68,  -8, 0,   -11, 57, 137, 84,  -3, -8,  -13, 44, 135, 97,  3,  -10, -14, 32, 130, 109, 11, -12,
       -13, 21, 120, 120, 21, -13, -12, 11, 109, 130, 32, -14, -10, 3,  97,  135, 44, -13, -8,  -3, 84,  137, 57, -11},
    },

    // Band_8_1853
    {
      // Scaler inc range
      {1849, 1853},
      // Coefficients 4 taps
      {59, 138, 59,  0,  46, 138, 74,  -2, 34, 132, 88,  2,  23, 124, 102, 7,
       14, 114, 114, 14, 7,  102, 124, 23, 2,  88,  132, 34, -2, 74,  138, 46},
      // Coefficients 6 taps
      {-8,  68, 136, 68,  -8, 0,   -11, 57, 137, 84,  -3, -8,  -13, 44, 135, 97,  3,  -10, -13, 32, 129, 109, 11, -12,
       -13, 21, 120, 120, 21, -13, -12, 11, 109, 129, 32, -13, -10, 3,  97,  135, 44, -13, -8,  -3, 84,  137, 57, -11},
    },

    // Band_8_1856
    {
      // Scaler inc range
      {1853, 1856},
      // Coefficients 4 taps
      {59, 138, 59,  0,  46, 138, 74,  -2, 34, 132, 88,  2,  23, 124, 102, 7,
       14, 114, 114, 14, 7,  102, 124, 23, 2,  88,  132, 34, -2, 74,  138, 46},
      // Coefficients 6 taps
      {-7,  68, 134, 68,  -7, 0,   -11, 57, 137, 84,  -3, -8,  -13, 44, 135, 97,  3,  -10, -13, 32, 129, 109, 11, -12,
       -13, 21, 120, 120, 21, -13, -12, 11, 109, 129, 32, -13, -10, 3,  97,  135, 44, -13, -8,  -3, 84,  137, 57, -11},
    },

    // Band_8_1859
    {
      // Scaler inc range
      {1856, 1859},
      // Coefficients 4 taps
      {59, 138, 59,  0,  47, 136, 74,  -1, 34, 132, 88,  2,  23, 124, 102, 7,
       14, 114, 114, 14, 7,  102, 124, 23, 2,  88,  132, 34, -1, 74,  136, 47},
      // Coefficients 6 taps
      {-7,  68, 134, 68,  -7, 0,   -11, 57, 137, 84,  -3, -8,  -13, 44, 134, 97,  4,  -10, -13, 32, 128, 109, 12, -12,
       -13, 21, 120, 120, 21, -13, -12, 12, 109, 128, 32, -13, -10, 4,  97,  134, 44, -13, -8,  -3, 84,  137, 57, -11},
    },

    // Band_8_1862
    {
      // Scaler inc range
      {1859, 1862},
      // Coefficients 4 taps
      {60, 136, 60,  0,  47, 136, 74,  -1, 34, 132, 88,  2,  23, 125, 101, 7,
       14, 114, 114, 14, 7,  101, 125, 23, 2,  88,  132, 34, -1, 74,  136, 47},
      // Coefficients 6 taps
      {-7,  68, 134, 68,  -7, 0,   -11, 57, 137, 84,  -3, -8,  -13, 44, 134, 97,  4,  -10, -13, 32, 128, 109, 12, -12,
       -13, 21, 120, 120, 21, -13, -12, 12, 109, 128, 32, -13, -10, 4,  97,  134, 44, -13, -8,  -3, 84,  137, 57, -11},
    },

    // Band_8_1866
    {
      // Scaler inc range
      {1862, 1866},
      // Coefficients 4 taps
      {60, 136, 60,  0,  47, 136, 74,  -1, 34, 132, 88,  2,  24, 124, 101, 7,
       15, 113, 113, 15, 7,  101, 124, 24, 2,  88,  132, 34, -1, 74,  136, 47},
      // Coefficients 6 taps
      {-7,  68, 134, 68,  -7, 0,   -11, 57, 138, 83,  -3, -8,  -12, 45, 132, 97,  4,  -10, -13, 33, 127, 109, 12, -12,
       -13, 22, 119, 119, 22, -13, -12, 12, 109, 127, 33, -13, -10, 4,  97,  132, 45, -12, -8,  -3, 83,  138, 57, -11},
    },

    // Band_8_1869
    {
      // Scaler inc range
      {1866, 1869},
      // Coefficients 4 taps
      {60, 136, 60,  0,  47, 136, 74,  -1, 34, 132, 88,  2,  24, 123, 101, 8,
       15, 113, 113, 15, 8,  101, 123, 24, 2,  88,  132, 34, -1, 74,  136, 47},
      // Coefficients 6 taps
      {-7,  68, 134, 68,  -7, 0,   -10, 57, 136, 83,  -2, -8,  -12, 45, 132, 97,  4,  -10, -13, 33, 127, 109, 12, -12,
       -13, 22, 119, 119, 22, -13, -12, 12, 109, 127, 33, -13, -10, 4,  97,  132, 45, -12, -8,  -2, 83,  136, 57, -10},
    },

    // Band_8_1872
    {
      // Scaler inc range
      {1869, 1872},
      // Coefficients 4 taps
      {60, 136, 60,  0,  47, 136, 74,  -1, 35, 131, 88,  2,  24, 123, 101, 8,
       15, 113, 113, 15, 8,  101, 123, 24, 2,  88,  131, 35, -1, 74,  136, 47},
      // Coefficients 6 taps
      {-7,  68, 134, 68,  -7, 0,   -10, 57, 136, 83,  -2, -8,  -12, 45, 133, 96,  4,  -10, -13, 33, 128, 108, 12, -12,
       -13, 22, 119, 119, 22, -13, -12, 12, 108, 128, 33, -13, -10, 4,  96,  133, 45, -12, -8,  -2, 83,  136, 57, -10},
    },

    // Band_8_1876
    {
      // Scaler inc range
      {1872, 1876},
      // Coefficients 4 taps
      {60, 136, 60,  0,  47, 136, 74,  -1, 35, 131, 88,  2,  24, 123, 101, 8,
       15, 113, 113, 15, 8,  101, 123, 24, 2,  88,  131, 35, -1, 74,  136, 47},
      // Coefficients 6 taps
      {-7,  68, 134, 68,  -7, 0,   -10, 57, 136, 83,  -2, -8,  -12, 45, 133, 96,  4,  -10, -13, 33, 128, 108, 12, -12,
       -13, 22, 119, 119, 22, -13, -12, 12, 108, 128, 33, -13, -10, 4,  96,  133, 45, -12, -8,  -2, 83,  136, 57, -10},
    },

    // Band_8_1879
    {
      // Scaler inc range
      {1876, 1879},
      // Coefficients 4 taps
      {60, 136, 60,  0,  47, 136, 74,  -1, 35, 130, 88,  3,  24, 123, 101, 8,
       15, 113, 113, 15, 8,  101, 123, 24, 3,  88,  130, 35, -1, 74,  136, 47},
      // Coefficients 6 taps
      {-7,  68, 134, 68,  -7, 0,   -10, 57, 136, 83,  -2, -8,  -12, 45, 132, 96,  5,  -10, -13, 33, 127, 108, 13, -12,
       -13, 22, 119, 119, 22, -13, -12, 13, 108, 127, 33, -13, -10, 5,  96,  132, 45, -12, -8,  -2, 83,  136, 57, -10},
    },

    // Band_8_1883
    {
      // Scaler inc range
      {1879, 1883},
      // Coefficients 4 taps
      {60, 136, 60,  0,  47, 136, 74,  -1, 35, 130, 88,  3,  24, 123, 101, 8,
       15, 113, 113, 15, 8,  101, 123, 24, 3,  88,  130, 35, -1, 74,  136, 47},
      // Coefficients 6 taps
      {-6,  68, 132, 68,  -6, 0,   -10, 57, 136, 83,  -2, -8,  -12, 45, 132, 96,  5,  -10, -13, 33, 127, 108, 13, -12,
       -13, 22, 119, 119, 22, -13, -12, 13, 108, 127, 33, -13, -10, 5,  96,  132, 45, -12, -8,  -2, 83,  136, 57, -10},
    },

    // Band_8_1886
    {
      // Scaler inc range
      {1883, 1886},
      // Coefficients 4 taps
      {60, 136, 60,  0,  47, 136, 74,  -1, 35, 130, 88,  3,  24, 123, 101, 8,
       15, 113, 113, 15, 8,  101, 123, 24, 3,  88,  130, 35, -1, 74,  136, 47},
      // Coefficients 6 taps
      {-6,  68, 132, 68,  -6, 0,   -10, 57, 136, 83,  -2, -8,  -12, 45, 132, 96,  5,  -10, -13, 33, 127, 108, 13, -12,
       -13, 23, 118, 118, 23, -13, -12, 13, 108, 127, 33, -13, -10, 5,  96,  132, 45, -12, -8,  -2, 83,  136, 57, -10},
    },

    // Band_8_1889
    {
      // Scaler inc range
      {1886, 1889},
      // Coefficients 4 taps
      {60, 136, 60,  0,  47, 136, 74,  -1, 35, 130, 88,  3,  24, 123, 101, 8,
       15, 113, 113, 15, 8,  101, 123, 24, 3,  88,  130, 35, -1, 74,  136, 47},
      // Coefficients 6 taps
      {-6,  68, 132, 68,  -6, 0,   -10, 58, 134, 83,  -1, -8,  -12, 45, 132, 96,  5,  -10, -13, 33, 127, 108, 13, -12,
       -13, 23, 118, 118, 23, -13, -12, 13, 108, 127, 33, -13, -10, 5,  96,  132, 45, -12, -8,  -1, 83,  134, 58, -10},
    },

    // Band_8_1893
    {
      // Scaler inc range
      {1889, 1893},
      // Coefficients 4 taps
      {60, 136, 60,  0,  47, 136, 74,  -1, 35, 130, 88,  3,  24, 123, 101, 8,
       15, 113, 113, 15, 8,  101, 123, 24, 3,  88,  130, 35, -1, 74,  136, 47},
      // Coefficients 6 taps
      {-6,  68, 132, 68,  -6, 0,   -10, 58, 134, 83,  -1, -8,  -12, 45, 132, 96,  5,  -10, -13, 34, 126, 108, 13, -12,
       -12, 23, 117, 117, 23, -12, -12, 13, 108, 126, 34, -13, -10, 5,  96,  132, 45, -12, -8,  -1, 83,  134, 58, -10},
    },

    // Band_8_1896
    {
      // Scaler inc range
      {1893, 1896},
      // Coefficients 4 taps
      {60, 136, 60,  0,  47, 136, 74,  -1, 35, 130, 88,  3,  25, 122, 101, 8,
       16, 112, 112, 16, 8,  101, 122, 25, 3,  88,  130, 35, -1, 74,  136, 47},
      // Coefficients 6 taps
      {-6,  68, 132, 68,  -6, 0,   -9,  58, 133, 83,  -1, -8,  -12, 45, 132, 96,  5,  -10, -13, 34, 127, 107, 13, -12,
       -12, 23, 117, 117, 23, -12, -12, 13, 107, 127, 34, -13, -10, 5,  96,  132, 45, -12, -8,  -1, 83,  133, 58, -9},
    },

    // Band_8_1900
    {
      // Scaler inc range
      {1896, 1900},
      // Coefficients 4 taps
      {60, 136, 60,  0,  48, 134, 74,  0,  35, 130, 88,  3,  25, 121, 101, 9,
       16, 112, 112, 16, 9,  101, 121, 25, 3,  88,  130, 35, 0,  74,  134, 48},
      // Coefficients 6 taps
      {-6,  68, 132, 68,  -6, 0,   -9,  58, 133, 83,  -1, -8,  -11, 45, 130, 96,  6,  -10, -12, 34, 125, 107, 14, -12,
       -12, 23, 117, 117, 23, -12, -12, 14, 107, 125, 34, -12, -10, 6,  96,  130, 45, -11, -8,  -1, 83,  133, 58, -9},
    },

    // Band_8_1903
    {
      // Scaler inc range
      {1900, 1903},
      // Coefficients 4 taps
      {60, 136, 60,  0,  48, 134, 74,  0,  35, 130, 88,  3,  25, 121, 101, 9,
       16, 112, 112, 16, 9,  101, 121, 25, 3,  88,  130, 35, 0,  74,  134, 48},
      // Coefficients 6 taps
      {-6,  68, 132, 68,  -6, 0,   -9,  58, 133, 83,  -1, -8,  -11, 46, 129, 96,  6,  -10, -12, 34, 125, 107, 14, -12,
       -12, 23, 117, 117, 23, -12, -12, 14, 107, 125, 34, -12, -10, 6,  96,  129, 46, -11, -8,  -1, 83,  133, 58, -9},
    },

    // Band_8_1907
    {
      // Scaler inc range
      {1903, 1907},
      // Coefficients 4 taps
      {60, 136, 60,  0,  48, 134, 74,  0,  36, 129, 88,  3,  25, 121, 101, 9,
       16, 112, 112, 16, 9,  101, 121, 25, 3,  88,  129, 36, 0,  74,  134, 48},
      // Coefficients 6 taps
      {-5,  68, 130, 68,  -5, 0,   -9,  58, 133, 83,  -1, -8,  -11, 46, 130, 95,  6,  -10, -12, 34, 124, 107, 14, -11,
       -12, 23, 117, 117, 23, -12, -11, 14, 107, 124, 34, -12, -10, 6,  95,  130, 46, -11, -8,  -1, 83,  133, 58, -9},
    },

    // Band_8_1910
    {
      // Scaler inc range
      {1907, 1910},
      // Coefficients 4 taps
      {61, 134, 61,  0,  48, 134, 74,  0,  36, 129, 88,  3,  25, 122, 100, 9,
       16, 112, 112, 16, 9,  100, 122, 25, 3,  88,  129, 36, 0,  74,  134, 48},
      // Coefficients 6 taps
      {-5,  68, 130, 68,  -5, 0,   -9,  58, 132, 83,  0,  -8,  -11, 46, 130, 95,  6,  -10, -12, 34, 124, 107, 14, -11,
       -12, 24, 116, 116, 24, -12, -11, 14, 107, 124, 34, -12, -10, 6,  95,  130, 46, -11, -8,  0,  83,  132, 58, -9},
    },

    // Band_8_1913
    {
      // Scaler inc range
      {1910, 1913},
      // Coefficients 4 taps
      {61, 134, 61,  0,  48, 134, 74,  0,  36, 128, 88,  4,  25, 122, 100, 9,
       16, 112, 112, 16, 9,  100, 122, 25, 4,  88,  128, 36, 0,  74,  134, 48},
      // Coefficients 6 taps
      {-5,  68, 130, 68,  -5, 0,   -9,  58, 132, 83,  0,  -8,  -11, 46, 130, 95,  6,  -10, -12, 34, 124, 107, 14, -11,
       -12, 24, 116, 116, 24, -12, -11, 14, 107, 124, 34, -12, -10, 6,  95,  130, 46, -11, -8,  0,  83,  132, 58, -9},
    },

    // Band_8_1917
    {
      // Scaler inc range
      {1913, 1917},
      // Coefficients 4 taps
      {61, 134, 61,  0,  48, 134, 74,  0,  36, 128, 88,  4,  25, 122, 100, 9,
       16, 112, 112, 16, 9,  100, 122, 25, 4,  88,  128, 36, 0,  74,  134, 48},
      // Coefficients 6 taps
      {-5,  68, 130, 68,  -5, 0,   -9,  58, 132, 83,  0,  -8,  -11, 46, 130, 95,  6,  -10, -12, 34, 125, 106, 14, -11,
       -12, 24, 116, 116, 24, -12, -11, 14, 106, 125, 34, -12, -10, 6,  95,  130, 46, -11, -8,  0,  83,  132, 58, -9},
    },

    // Band_8_1920
    {
      // Scaler inc range
      {1917, 1920},
      // Coefficients 4 taps
      {61, 134, 61,  0,  48, 134, 74,  0,  36, 128, 88,  4,  25, 122, 100, 9,
       16, 112, 112, 16, 9,  100, 122, 25, 4,  88,  128, 36, 0,  74,  134, 48},
      // Coefficients 6 taps
      {-5,  68, 130, 68,  -5, 0,   -9,  58, 132, 83,  0,  -8,  -11, 46, 129, 95,  7,  -10, -12, 35, 123, 106, 15, -11,
       -12, 24, 116, 116, 24, -12, -11, 15, 106, 123, 35, -12, -10, 7,  95,  129, 46, -11, -8,  0,  83,  132, 58, -9},
    },

    // Band_8_1924
    {
      // Scaler inc range
      {1920, 1924},
      // Coefficients 4 taps
      {61, 134, 61,  0,  48, 133, 75,  0,  36, 128, 88,  4,  25, 122, 100, 9,
       16, 112, 112, 16, 9,  100, 122, 25, 4,  88,  128, 36, 0,  75,  133, 48},
      // Coefficients 6 taps
      {-5,  68, 130, 68,  -5, 0,   -8,  58, 131, 83,  0,  -8,  -11, 46, 129, 95,  7,  -10, -12, 35, 123, 106, 15, -11,
       -12, 24, 116, 116, 24, -12, -11, 15, 106, 123, 35, -12, -10, 7,  95,  129, 46, -11, -8,  0,  83,  131, 58, -8},
    },

    // Band_8_1928
    {
      // Scaler inc range
      {1924, 1928},
      // Coefficients 4 taps
      {61, 134, 61,  0,  48, 133, 75,  0,  36, 128, 88,  4,  26, 121, 100, 9,
       17, 111, 111, 17, 9,  100, 121, 26, 4,  88,  128, 36, 0,  75,  133, 48},
      // Coefficients 6 taps
      {-5,  68, 130, 68,  -5, 0,   -8,  58, 131, 83,  0,  -8,  -11, 46, 129, 95,  7,  -10, -12, 35, 123, 106, 15, -11,
       -12, 24, 116, 116, 24, -12, -11, 15, 106, 123, 35, -12, -10, 7,  95,  129, 46, -11, -8,  0,  83,  131, 58, -8},
    },

    // Band_8_1931
    {
      // Scaler inc range
      {1928, 1931},
      // Coefficients 4 taps
      {61, 134, 61,  0,  48, 133, 75,  0,  36, 128, 88,  4,  26, 121, 100, 9,
       17, 111, 111, 17, 9,  100, 121, 26, 4,  88,  128, 36, 0,  75,  133, 48},
      // Coefficients 6 taps
      {-4,  68, 128, 68,  -4, 0,   -8,  58, 131, 83,  0,  -8,  -11, 46, 129, 95,  7,  -10, -12, 35, 123, 106, 15, -11,
       -12, 24, 116, 116, 24, -12, -11, 15, 106, 123, 35, -12, -10, 7,  95,  129, 46, -11, -8,  0,  83,  131, 58, -8},
    },

    // Band_8_1935
    {
      // Scaler inc range
      {1931, 1935},
      // Coefficients 4 taps
      {61, 134, 61,  0,  48, 133, 75,  0,  36, 128, 88,  4,  26, 120, 100, 10,
       17, 111, 111, 17, 10, 100, 120, 26, 4,  88,  128, 36, 0,  75,  133, 48},
      // Coefficients 6 taps
      {-4,  68, 128, 68,  -4, 0,   -8,  58, 130, 83,  1,  -8,  -10, 46, 128, 95,  7,  -10, -12, 35, 123, 106, 15, -11,
       -12, 25, 115, 115, 25, -12, -11, 15, 106, 123, 35, -12, -10, 7,  95,  128, 46, -10, -8,  1,  83,  130, 58, -8},
    },

    // Band_8_1938
    {
      // Scaler inc range
      {1935, 1938},
      // Coefficients 4 taps
      {61, 134, 61,  0,  48, 133, 75,  0,  36, 128, 88,  4,  26, 120, 100, 10,
       17, 111, 111, 17, 10, 100, 120, 26, 4,  88,  128, 36, 0,  75,  133, 48},
      // Coefficients 6 taps
      {-4,  68, 128, 68,  -4, 0,   -8,  58, 130, 83,  1,  -8,  -10, 46, 128, 95,  7,  -10, -12, 35, 123, 106, 15, -11,
       -12, 25, 115, 115, 25, -12, -11, 15, 106, 123, 35, -12, -10, 7,  95,  128, 46, -10, -8,  1,  83,  130, 58, -8},
    },

    // Band_8_1942
    {
      // Scaler inc range
      {1938, 1942},
      // Coefficients 4 taps
      {61, 134, 61,  0,  48, 132, 75,  1,  36, 128, 88,  4,  26, 120, 100, 10,
       17, 111, 111, 17, 10, 100, 120, 26, 4,  88,  128, 36, 1,  75,  132, 48},
      // Coefficients 6 taps
      {-4,  68, 128, 68,  -4, 0,   -8,  58, 131, 82,  1,  -8,  -10, 47, 127, 94,  8,  -10, -11, 35, 122, 105, 16, -11,
       -12, 25, 115, 115, 25, -12, -11, 16, 105, 122, 35, -11, -10, 8,  94,  127, 47, -10, -8,  1,  82,  131, 58, -8},
    },

    // Band_8_1945
    {
      // Scaler inc range
      {1942, 1945},
      // Coefficients 4 taps
      {61, 134, 61,  0,  49, 131, 75,  1,  37, 127, 88,  4,  26, 120, 100, 10,
       17, 111, 111, 17, 10, 100, 120, 26, 4,  88,  127, 37, 1,  75,  131, 49},
      // Coefficients 6 taps
      {-4,  68, 128, 68,  -4, 0,   -8,  58, 131, 82,  1,  -8,  -10, 47, 127, 94,  8,  -10, -11, 35, 122, 105, 16, -11,
       -12, 25, 115, 115, 25, -12, -11, 16, 105, 122, 35, -11, -10, 8,  94,  127, 47, -10, -8,  1,  82,  131, 58, -8},
    },

    // Band_8_1949
    {
      // Scaler inc range
      {1945, 1949},
      // Coefficients 4 taps
      {61, 134, 61,  0,  49, 131, 75,  1,  37, 127, 87,  5,  26, 120, 100, 10,
       17, 111, 111, 17, 10, 100, 120, 26, 5,  87,  127, 37, 1,  75,  131, 49},
      // Coefficients 6 taps
      {-4,  68, 128, 68,  -4, 0,   -8,  58, 131, 82,  1,  -8,  -10, 47, 127, 94,  8,  -10, -11, 36, 121, 105, 16, -11,
       -12, 25, 115, 115, 25, -12, -11, 16, 105, 121, 36, -11, -10, 8,  94,  127, 47, -10, -8,  1,  82,  131, 58, -8},
    },

    // Band_8_1953
    {
      // Scaler inc range
      {1949, 1953},
      // Coefficients 4 taps
      {61, 134, 61,  0,  49, 131, 75,  1,  37, 127, 87,  5,  26, 120, 100, 10,
       17, 111, 111, 17, 10, 100, 120, 26, 5,  87,  127, 37, 1,  75,  131, 49},
      // Coefficients 6 taps
      {-4,  68, 128, 68,  -4, 0,   -7,  58, 130, 82,  1,  -8,  -10, 47, 127, 94,  8,  -10, -11, 36, 121, 105, 16, -11,
       -11, 25, 114, 114, 25, -11, -11, 16, 105, 121, 36, -11, -10, 8,  94,  127, 47, -10, -8,  1,  82,  130, 58, -7},
    },

    // Band_8_1956
    {
      // Scaler inc range
      {1953, 1956},
      // Coefficients 4 taps
      {61, 134, 61,  0,  49, 131, 75,  1,  37, 127, 87,  5,  26, 120, 100, 10,
       17, 111, 111, 17, 10, 100, 120, 26, 5,  87,  127, 37, 1,  75,  131, 49},
      // Coefficients 6 taps
      {-3,  68, 126, 68,  -3, 0,   -7,  58, 129, 82,  2,  -8,  -10, 47, 127, 94,  8,  -10, -11, 36, 121, 105, 16, -11,
       -11, 25, 114, 114, 25, -11, -11, 16, 105, 121, 36, -11, -10, 8,  94,  127, 47, -10, -8,  2,  82,  129, 58, -7},
    },

    // Band_8_1960
    {
      // Scaler inc range
      {1956, 1960},
      // Coefficients 4 taps
      {62, 132, 62,  0,  49, 131, 75,  1,  37, 127, 87,  5,  26, 120, 100, 10,
       18, 110, 110, 18, 10, 100, 120, 26, 5,  87,  127, 37, 1,  75,  131, 49},
      // Coefficients 6 taps
      {-3,  68, 126, 68,  -3, 0,   -7,  58, 129, 82,  2,  -8,  -10, 47, 127, 94,  8,  -10, -11, 36, 121, 105, 16, -11,
       -11, 26, 113, 113, 26, -11, -11, 16, 105, 121, 36, -11, -10, 8,  94,  127, 47, -10, -8,  2,  82,  129, 58, -7},
    },

    // Band_8_1964
    {
      // Scaler inc range
      {1960, 1964},
      // Coefficients 4 taps
      {62, 132, 62,  0,  49, 131, 75,  1,  37, 127, 87,  5,  27, 120, 99,  10,
       18, 110, 110, 18, 10, 99,  120, 27, 5,  87,  127, 37, 1,  75,  131, 49},
      // Coefficients 6 taps
      {-3,  68, 126, 68,  -3, 0,   -7,  58, 129, 82,  2,  -8,  -10, 47, 127, 94,  8,  -10, -11, 36, 122, 104, 16, -11,
       -11, 26, 113, 113, 26, -11, -11, 16, 104, 122, 36, -11, -10, 8,  94,  127, 47, -10, -8,  2,  82,  129, 58, -7},
    },

    // Band_8_1967
    {
      // Scaler inc range
      {1964, 1967},
      // Coefficients 4 taps
      {62, 132, 62,  0,  49, 131, 75,  1,  37, 127, 87,  5,  27, 119, 99,  11,
       18, 110, 110, 18, 11, 99,  119, 27, 5,  87,  127, 37, 1,  75,  131, 49},
      // Coefficients 6 taps
      {-3,  68, 126, 68,  -3, 0,   -7,  59, 128, 82,  2,  -8,  -9,  47, 125, 94,  9,  -10, -11, 36, 121, 104, 17, -11,
       -11, 26, 113, 113, 26, -11, -11, 17, 104, 121, 36, -11, -10, 9,  94,  125, 47, -9,  -8,  2,  82,  128, 59, -7},
    },

    // Band_8_1971
    {
      // Scaler inc range
      {1967, 1971},
      // Coefficients 4 taps
      {62, 132, 62,  0,  49, 131, 75,  1,  37, 127, 87,  5,  27, 119, 99,  11,
       18, 110, 110, 18, 11, 99,  119, 27, 5,  87,  127, 37, 1,  75,  131, 49},
      // Coefficients 6 taps
      {-3,  68, 126, 68,  -3, 0,   -7,  59, 128, 82,  2,  -8,  -9,  47, 125, 94,  9,  -10, -11, 36, 121, 104, 17, -11,
       -11, 26, 113, 113, 26, -11, -11, 17, 104, 121, 36, -11, -10, 9,  94,  125, 47, -9,  -8,  2,  82,  128, 59, -7},
    },

    // Band_8_1975
    {
      // Scaler inc range
      {1971, 1975},
      // Coefficients 4 taps
      {62, 132, 62,  0,  49, 131, 75,  1,  37, 127, 87,  5,  27, 119, 99,  11,
       18, 110, 110, 18, 11, 99,  119, 27, 5,  87,  127, 37, 1,  75,  131, 49},
      // Coefficients 6 taps
      {-3,  68, 126, 68,  -3, 0,   -7,  59, 128, 82,  2,  -8,  -9,  47, 126, 93,  9,  -10, -11, 36, 121, 104, 17, -11,
       -11, 26, 113, 113, 26, -11, -11, 17, 104, 121, 36, -11, -10, 9,  93,  126, 47, -9,  -8,  2,  82,  128, 59, -7},
    },

    // Band_8_1978
    {
      // Scaler inc range
      {1975, 1978},
      // Coefficients 4 taps
      {62, 132, 62,  0,  49, 131, 75,  1,  37, 127, 87,  5,  27, 119, 99,  11,
       18, 110, 110, 18, 11, 99,  119, 27, 5,  87,  127, 37, 1,  75,  131, 49},
      // Coefficients 6 taps
      {-3,  68, 126, 68,  -3, 0,   -6,  59, 126, 82,  3,  -8,  -9,  47, 126, 93,  9,  -10, -11, 36, 121, 104, 17, -11,
       -11, 26, 113, 113, 26, -11, -11, 17, 104, 121, 36, -11, -10, 9,  93,  126, 47, -9,  -8,  3,  82,  126, 59, -6},
    },

    // Band_8_1982
    {
      // Scaler inc range
      {1978, 1982},
      // Coefficients 4 taps
      {62, 132, 62,  0,  49, 131, 75,  1,  38, 126, 87,  5,  27, 119, 99,  11,
       18, 110, 110, 18, 11, 99,  119, 27, 5,  87,  126, 38, 1,  75,  131, 49},
      // Coefficients 6 taps
      {-2,  68, 124, 68,  -2, 0,   -6,  59, 126, 82,  3,  -8,  -9,  47, 126, 93,  9,  -10, -10, 37, 119, 104, 17, -11,
       -11, 26, 113, 113, 26, -11, -11, 17, 104, 119, 37, -10, -10, 9,  93,  126, 47, -9,  -8,  3,  82,  126, 59, -6},
    },

    // Band_8_1986
    {
      // Scaler inc range
      {1982, 1986},
      // Coefficients 4 taps
      {62, 132, 62,  0,  49, 131, 75,  1,  38, 126, 87,  5,  27, 119, 99,  11,
       18, 110, 110, 18, 11, 99,  119, 27, 5,  87,  126, 38, 1,  75,  131, 49},
      // Coefficients 6 taps
      {-2,  68, 124, 68,  -2, 0,   -6,  59, 126, 82,  3,  -8,  -9,  47, 126, 93,  9,  -10, -10, 37, 120, 103, 17, -11,
       -11, 26, 113, 113, 26, -11, -11, 17, 103, 120, 37, -10, -10, 9,  93,  126, 47, -9,  -8,  3,  82,  126, 59, -6},
    },

    // Band_8_1990
    {
      // Scaler inc range
      {1986, 1990},
      // Coefficients 4 taps
      {62, 132, 62,  0,  49, 130, 75,  2,  38, 125, 87,  6,  27, 119, 99,  11,
       18, 110, 110, 18, 11, 99,  119, 27, 6,  87,  125, 38, 2,  75,  130, 49},
      // Coefficients 6 taps
      {-2,  68, 124, 68,  -2, 0,   -6,  59, 126, 82,  3,  -8,  -9,  48, 124, 93,  10, -10, -10, 37, 120, 103, 17, -11,
       -11, 27, 112, 112, 27, -11, -11, 17, 103, 120, 37, -10, -10, 10, 93,  124, 48, -9,  -8,  3,  82,  126, 59, -6},
    },

    // Band_8_1993
    {
      // Scaler inc range
      {1990, 1993},
      // Coefficients 4 taps
      {62, 132, 62,  0,  50, 129, 75,  2,  38, 125, 87,  6,  27, 119, 99,  11,
       18, 110, 110, 18, 11, 99,  119, 27, 6,  87,  125, 38, 2,  75,  129, 50},
      // Coefficients 6 taps
      {-2,  68, 124, 68,  -2, 0,   -6,  59, 126, 82,  3,  -8,  -9,  48, 124, 93,  10, -10, -10, 37, 119, 103, 18, -11,
       -11, 27, 112, 112, 27, -11, -11, 18, 103, 119, 37, -10, -10, 10, 93,  124, 48, -9,  -8,  3,  82,  126, 59, -6},
    },

    // Band_8_1997
    {
      // Scaler inc range
      {1993, 1997},
      // Coefficients 4 taps
      {62, 132, 62,  0,  50, 129, 75,  2,  38, 125, 87,  6,  27, 119, 99,  11,
       19, 109, 109, 19, 11, 99,  119, 27, 6,  87,  125, 38, 2,  75,  129, 50},
      // Coefficients 6 taps
      {-2,  68, 124, 68,  -2, 0,   -6,  59, 126, 82,  3,  -8,  -8,  48, 123, 93,  10, -10, -10, 37, 118, 103, 18, -10,
       -11, 27, 112, 112, 27, -11, -10, 18, 103, 118, 37, -10, -10, 10, 93,  123, 48, -8,  -8,  3,  82,  126, 59, -6},
    },

    // Band_8_2001
    {
      // Scaler inc range
      {1997, 2001},
      // Coefficients 4 taps
      {62, 132, 62,  0,  50, 129, 75,  2,  38, 125, 87,  6,  28, 118, 99,  11,
       19, 109, 109, 19, 11, 99,  118, 28, 6,  87,  125, 38, 2,  75,  129, 50},
      // Coefficients 6 taps
      {-2,  68, 124, 68,  -2, 0,   -6,  59, 126, 82,  3,  -8,  -8,  48, 123, 93,  10, -10, -10, 37, 118, 103, 18, -10,
       -11, 27, 112, 112, 27, -11, -10, 18, 103, 118, 37, -10, -10, 10, 93,  123, 48, -8,  -8,  3,  82,  126, 59, -6},
    },

    // Band_8_2005
    {
      // Scaler inc range
      {2001, 2005},
      // Coefficients 4 taps
      {62, 132, 62,  0,  50, 129, 75,  2,  38, 125, 87,  6,  28, 117, 99,  12,
       19, 109, 109, 19, 12, 99,  117, 28, 6,  87,  125, 38, 2,  75,  129, 50},
      // Coefficients 6 taps
      {-2,  68, 124, 68,  -2, 0,   -6,  59, 125, 82,  4,  -8,  -8,  48, 123, 93,  10, -10, -10, 37, 118, 103, 18, -10,
       -11, 27, 112, 112, 27, -11, -10, 18, 103, 118, 37, -10, -10, 10, 93,  123, 48, -8,  -8,  4,  82,  125, 59, -6},
    },

    // Band_8_2009
    {
      // Scaler inc range
      {2005, 2009},
      // Coefficients 4 taps
      {62, 132, 62,  0,  50, 129, 75,  2,  38, 125, 87,  6,  28, 117, 99,  12,
       19, 109, 109, 19, 12, 99,  117, 28, 6,  87,  125, 38, 2,  75,  129, 50},
      // Coefficients 6 taps
      {-1,  68, 122, 68,  -1, 0,   -5,  59, 125, 81,  4,  -8,  -8,  48, 124, 92,  10, -10, -10, 37, 118, 103, 18, -10,
       -10, 27, 111, 111, 27, -10, -10, 18, 103, 118, 37, -10, -10, 10, 92,  124, 48, -8,  -8,  4,  81,  125, 59, -5},
    },

    // Band_8_2013
    {
      // Scaler inc range
      {2009, 2013},
      // Coefficients 4 taps
      {62, 132, 62,  0,  50, 129, 75,  2,  38, 125, 87,  6,  28, 117, 99,  12,
       19, 109, 109, 19, 12, 99,  117, 28, 6,  87,  125, 38, 2,  75,  129, 50},
      // Coefficients 6 taps
      {-1,  68, 122, 68,  -1, 0,   -5,  59, 125, 81,  4,  -8,  -8,  48, 123, 92,  11, -10, -10, 37, 119, 102, 18, -10,
       -10, 27, 111, 111, 27, -10, -10, 18, 102, 119, 37, -10, -10, 11, 92,  123, 48, -8,  -8,  4,  81,  125, 59, -5},
    },

    // Band_8_2016
    {
      // Scaler inc range
      {2013, 2016},
      // Coefficients 4 taps
      {62, 132, 62,  0,  50, 129, 75,  2,  38, 125, 87,  6,  28, 118, 98,  12,
       19, 109, 109, 19, 12, 98,  118, 28, 6,  87,  125, 38, 2,  75,  129, 50},
      // Coefficients 6 taps
      {-1,  68, 122, 68,  -1, 0,   -5,  59, 125, 81,  4,  -8,  -8,  48, 123, 92,  11, -10, -10, 37, 118, 102, 19, -10,
       -10, 28, 110, 110, 28, -10, -10, 19, 102, 118, 37, -10, -10, 11, 92,  123, 48, -8,  -8,  4,  81,  125, 59, -5},
    },

    // Band_8_2020
    {
      // Scaler inc range
      {2016, 2020},
      // Coefficients 4 taps
      {63, 130, 63,  0,  50, 129, 75,  2,  38, 125, 87,  6,  28, 118, 98,  12,
       19, 109, 109, 19, 12, 98,  118, 28, 6,  87,  125, 38, 2,  75,  129, 50},
      // Coefficients 6 taps
      {-1,  68, 122, 68,  -1, 0,   -5,  59, 125, 81,  4,  -8,  -8, 48, 122, 92,  11, -9, -10, 38, 117, 102, 19, -10,
       -10, 28, 110, 110, 28, -10, -10, 19, 102, 117, 38, -10, -9, 11, 92,  122, 48, -8, -8,  4,  81,  125, 59, -5},
    },

    // Band_8_2024
    {
      // Scaler inc range
      {2020, 2024},
      // Coefficients 4 taps
      {63, 130, 63,  0,  50, 129, 75,  2,  38, 125, 87,  6,  28, 118, 98,  12,
       19, 109, 109, 19, 12, 98,  118, 28, 6,  87,  125, 38, 2,  75,  129, 50},
      // Coefficients 6 taps
      {-1,  68, 122, 68,  -1, 0,   -5,  59, 125, 81,  4,  -8, -8, 48, 122, 92,  11, -9, -9, 38, 116, 102, 19, -10,
       -10, 28, 110, 110, 28, -10, -10, 19, 102, 116, 38, -9, -9, 11, 92,  122, 48, -8, -8, 4,  81,  125, 59, -5},
    },

    // Band_8_2028
    {
      // Scaler inc range
      {2024, 2028},
      // Coefficients 4 taps
      {63, 130, 63,  0,  50, 129, 75,  2,  39, 123, 87,  7,  28, 118, 98,  12,
       19, 109, 109, 19, 12, 98,  118, 28, 7,  87,  123, 39, 2,  75,  129, 50},
      // Coefficients 6 taps
      {-1,  68, 122, 68,  -1, 0,   -5,  59, 124, 81,  5,  -8, -8, 48, 122, 92,  11, -9, -9, 38, 116, 102, 19, -10,
       -10, 28, 110, 110, 28, -10, -10, 19, 102, 116, 38, -9, -9, 11, 92,  122, 48, -8, -8, 5,  81,  124, 59, -5},
    },

    // Band_8_2032
    {
      // Scaler inc range
      {2028, 2032},
      // Coefficients 4 taps
      {63, 130, 63,  0,  50, 129, 75,  2,  39, 123, 87,  7,  28, 118, 98,  12,
       20, 108, 108, 20, 12, 98,  118, 28, 7,  87,  123, 39, 2,  75,  129, 50},
      // Coefficients 6 taps
      {-1,  68, 122, 68,  -1, 0,   -5,  59, 124, 81,  5,  -8, -7, 48, 121, 92,  11, -9, -9, 38, 116, 102, 19, -10,
       -10, 28, 110, 110, 28, -10, -10, 19, 102, 116, 38, -9, -9, 11, 92,  121, 48, -7, -8, 5,  81,  124, 59, -5},
    },

    // Band_8_2036
    {
      // Scaler inc range
      {2032, 2036},
      // Coefficients 4 taps
      {63, 130, 63,  0,  50, 128, 75,  3,  39, 123, 87,  7,  28, 118, 98,  12,
       20, 108, 108, 20, 12, 98,  118, 28, 7,  87,  123, 39, 3,  75,  128, 50},
      // Coefficients 6 taps
      {0,   68, 120, 68,  0,  0,   -4,  59, 123, 81,  5,  -8, -7, 48, 121, 92,  11, -9, -9, 38, 117, 101, 19, -10,
       -10, 28, 110, 110, 28, -10, -10, 19, 101, 117, 38, -9, -9, 11, 92,  121, 48, -7, -8, 5,  81,  123, 59, -4},
    },

    // Band_8_2040
    {
      // Scaler inc range
      {2036, 2040},
      // Coefficients 4 taps
      {63, 130, 63,  0,  50, 128, 75,  3,  39, 123, 87,  7,  29, 117, 98,  12,
       20, 108, 108, 20, 12, 98,  117, 29, 7,  87,  123, 39, 3,  75,  128, 50},
      // Coefficients 6 taps
      {0,   68, 120, 68,  0,  0,   -4,  59, 123, 81,  5,  -8, -7, 48, 120, 92,  12, -9, -9, 38, 117, 101, 19, -10,
       -10, 28, 110, 110, 28, -10, -10, 19, 101, 117, 38, -9, -9, 12, 92,  120, 48, -7, -8, 5,  81,  123, 59, -4},
    },

    // Band_8_2044
    {
      // Scaler inc range
      {2040, 2044},
      // Coefficients 4 taps
      {63, 130, 63,  0,  50, 128, 75,  3,  39, 123, 87,  7,  29, 116, 98,  13,
       20, 108, 108, 20, 13, 98,  116, 29, 7,  87,  123, 39, 3,  75,  128, 50},
      // Coefficients 6 taps
      {0,   68, 120, 68,  0,  0,   -4,  59, 123, 81,  5,  -8, -7, 49, 120, 91,  12, -9, -9, 38, 116, 101, 20, -10,
       -10, 28, 110, 110, 28, -10, -10, 20, 101, 116, 38, -9, -9, 12, 91,  120, 49, -7, -8, 5,  81,  123, 59, -4},
    },

    // Band_8_2048
    {
      // Scaler inc range
      {2044, 2048},
      // Coefficients 4 taps
      {63, 130, 63,  0,  50, 128, 75,  3,  39, 123, 87,  7,  29, 116, 98,  13,
       20, 108, 108, 20, 13, 98,  116, 29, 7,  87,  123, 39, 3,  75,  128, 50},
      // Coefficients 6 taps
      {0,   68, 120, 68,  0,  0,   -4,  59, 123, 81,  5,  -8, -7, 49, 120, 91,  12, -9, -9, 38, 116, 101, 20, -10,
       -10, 29, 109, 109, 29, -10, -10, 20, 101, 116, 38, -9, -9, 12, 91,  120, 49, -7, -8, 5,  81,  123, 59, -4},
    },

    // Band_8_2052
    {
      // Scaler inc range
      {2048, 2052},
      // Coefficients 4 taps
      {63, 130, 63,  0,  51, 127, 75,  3,  39, 123, 87,  7,  29, 116, 98,  13,
       20, 108, 108, 20, 13, 98,  116, 29, 7,  87,  123, 39, 3,  75,  127, 51},
      // Coefficients 6 taps
      {0,   68, 120, 68,  0,  0,   -4,  59, 123, 81,  5,  -8, -7, 49, 120, 91,  12, -9, -9, 38, 116, 101, 20, -10,
       -10, 29, 109, 109, 29, -10, -10, 20, 101, 116, 38, -9, -9, 12, 91,  120, 49, -7, -8, 5,  81,  123, 59, -4},
    },

    // Band_8_2056
    {
      // Scaler inc range
      {2052, 2056},
      // Coefficients 4 taps
      {63, 130, 63,  0,  51, 127, 75,  3,  39, 123, 87,  7,  29, 116, 98,  13,
       20, 108, 108, 20, 13, 98,  116, 29, 7,  87,  123, 39, 3,  75,  127, 51},
      // Coefficients 6 taps
      {0,   68, 120, 68,  0,  0,   -4,  59, 122, 81,  6,  -8, -7, 49, 120, 91,  12, -9, -9, 38, 116, 101, 20, -10,
       -10, 29, 109, 109, 29, -10, -10, 20, 101, 116, 38, -9, -9, 12, 91,  120, 49, -7, -8, 6,  81,  122, 59, -4},
    },

    // Band_8_2060
    {
      // Scaler inc range
      {2056, 2060},
      // Coefficients 4 taps
      {63, 130, 63,  0,  51, 127, 75,  3,  39, 123, 87,  7,  29, 116, 98,  13,
       20, 108, 108, 20, 13, 98,  116, 29, 7,  87,  123, 39, 3,  75,  127, 51},
      // Coefficients 6 taps
      {0,  68, 120, 68,  0,  0,  -4,  59, 122, 81,  6,  -8, -7, 49, 120, 91,  12, -9, -9, 39, 116, 100, 20, -10,
       -9, 29, 108, 108, 29, -9, -10, 20, 100, 116, 39, -9, -9, 12, 91,  120, 49, -7, -8, 6,  81,  122, 59, -4},
    },

    // Band_8_2064
    {
      // Scaler inc range
      {2060, 2064},
      // Coefficients 4 taps
      {63, 130, 63,  0,  51, 127, 75,  3,  39, 123, 87,  7,  29, 116, 98,  13,
       20, 108, 108, 20, 13, 98,  116, 29, 7,  87,  123, 39, 3,  75,  127, 51},
      // Coefficients 6 taps
      {1,  68, 118, 68,  1,  0,  -4,  59, 122, 81,  6,  -8, -7, 49, 120, 91,  12, -9, -8, 39, 115, 100, 20, -10,
       -9, 29, 108, 108, 29, -9, -10, 20, 100, 115, 39, -8, -9, 12, 91,  120, 49, -7, -8, 6,  81,  122, 59, -4},
    },

    // Band_8_2068
    {
      // Scaler inc range
      {2064, 2068},
      // Coefficients 4 taps
      {63, 130, 63,  0,  51, 127, 75,  3,  39, 123, 87,  7,  29, 116, 98,  13,
       20, 108, 108, 20, 13, 98,  116, 29, 7,  87,  123, 39, 3,  75,  127, 51},
      // Coefficients 6 taps
      {1,  68, 118, 68,  1,  0,  -3,  59, 121, 81,  6,  -8, -6, 49, 118, 91,  13, -9, -8, 39, 115, 100, 20, -10,
       -9, 29, 108, 108, 29, -9, -10, 20, 100, 115, 39, -8, -9, 13, 91,  118, 49, -6, -8, 6,  81,  121, 59, -3},
    },

    // Band_8_2072
    {
      // Scaler inc range
      {2068, 2072},
      // Coefficients 4 taps
      {63, 130, 63,  0,  51, 127, 75,  3,  40, 121, 87,  8,  29, 116, 98,  13,
       21, 107, 107, 21, 13, 98,  116, 29, 8,  87,  121, 40, 3,  75,  127, 51},
      // Coefficients 6 taps
      {1,  68, 118, 68,  1,  0,  -3, 59, 122, 80,  6,  -8, -6, 49, 118, 91,  13, -9, -8, 39, 113, 100, 21, -9,
       -9, 29, 108, 108, 29, -9, -9, 21, 100, 113, 39, -8, -9, 13, 91,  118, 49, -6, -8, 6,  80,  122, 59, -3},
    },

    // Band_8_2076
    {
      // Scaler inc range
      {2072, 2076},
      // Coefficients 4 taps
      {63, 130, 63,  0,  51, 127, 75,  3,  40, 121, 87,  8,  29, 117, 97,  13,
       21, 107, 107, 21, 13, 97,  117, 29, 8,  87,  121, 40, 3,  75,  127, 51},
      // Coefficients 6 taps
      {1,  68, 118, 68,  1,  0,  -3, 59, 122, 80,  6,  -8, -6, 49, 118, 91,  13, -9, -8, 39, 113, 100, 21, -9,
       -9, 29, 108, 108, 29, -9, -9, 21, 100, 113, 39, -8, -9, 13, 91,  118, 49, -6, -8, 6,  80,  122, 59, -3},
    },

    // Band_8_2081
    {
      // Scaler inc range
      {2076, 2081},
      // Coefficients 4 taps
      {63, 130, 63,  0,  51, 127, 75,  3,  40, 121, 87,  8,  30, 115, 97,  14,
       21, 107, 107, 21, 14, 97,  115, 30, 8,  87,  121, 40, 3,  75,  127, 51},
      // Coefficients 6 taps
      {1,  68, 118, 68,  1,  0,  -3, 59, 121, 80,  7,  -8, -6, 49, 119, 90,  13, -9, -8, 39, 113, 100, 21, -9,
       -9, 29, 108, 108, 29, -9, -9, 21, 100, 113, 39, -8, -9, 13, 90,  119, 49, -6, -8, 7,  80,  121, 59, -3},
    },

    // Band_8_2085
    {
      // Scaler inc range
      {2081, 2085},
      // Coefficients 4 taps
      {64, 128, 64,  0,  51, 126, 75,  4,  40, 122, 86,  8,  30, 115, 97,  14,
       21, 107, 107, 21, 14, 97,  115, 30, 8,  86,  122, 40, 4,  75,  126, 51},
      // Coefficients 6 taps
      {1,  68, 118, 68,  1,  0,  -3, 59, 121, 80,  7,  -8, -6, 49, 119, 90,  13, -9, -8, 39, 113, 100, 21, -9,
       -9, 30, 107, 107, 30, -9, -9, 21, 100, 113, 39, -8, -9, 13, 90,  119, 49, -6, -8, 7,  80,  121, 59, -3},
    },

    // Band_8_2089
    {
      // Scaler inc range
      {2085, 2089},
      // Coefficients 4 taps
      {64, 128, 64,  0,  51, 126, 75,  4,  40, 122, 86,  8,  30, 115, 97,  14,
       21, 107, 107, 21, 14, 97,  115, 30, 8,  86,  122, 40, 4,  75,  126, 51},
      // Coefficients 6 taps
      {1,  68, 118, 68,  1,  0,  -3, 59, 121, 80,  7,  -8, -6, 49, 119, 90,  13, -9, -8, 39, 114, 99,  21, -9,
       -9, 30, 107, 107, 30, -9, -9, 21, 99,  114, 39, -8, -9, 13, 90,  119, 49, -6, -8, 7,  80,  121, 59, -3},
    },

    // Band_8_2093
    {
      // Scaler inc range
      {2089, 2093},
      // Coefficients 4 taps
      {64, 128, 64,  0,  51, 126, 75,  4,  40, 122, 86,  8,  30, 115, 97,  14,
       21, 107, 107, 21, 14, 97,  115, 30, 8,  86,  122, 40, 4,  75,  126, 51},
      // Coefficients 6 taps
      {2,  68, 116, 68,  2,  0,  -3, 59, 121, 80,  7,  -8, -6, 49, 118, 90,  14, -9, -8, 39, 114, 99,  21, -9,
       -9, 30, 107, 107, 30, -9, -9, 21, 99,  114, 39, -8, -9, 14, 90,  118, 49, -6, -8, 7,  80,  121, 59, -3},
    },

    // Band_8_2097
    {
      // Scaler inc range
      {2093, 2097},
      // Coefficients 4 taps
      {64, 128, 64,  0,  51, 126, 75,  4,  40, 122, 86,  8,  30, 115, 97,  14,
       21, 107, 107, 21, 14, 97,  115, 30, 8,  86,  122, 40, 4,  75,  126, 51},
      // Coefficients 6 taps
      {2,  68, 116, 68,  2,  0,  -2, 59, 120, 80,  7,  -8, -6, 49, 118, 90,  14, -9, -8, 39, 114, 99,  21, -9,
       -9, 30, 107, 107, 30, -9, -9, 21, 99,  114, 39, -8, -9, 14, 90,  118, 49, -6, -8, 7,  80,  120, 59, -2},
    },

    // Band_8_2101
    {
      // Scaler inc range
      {2097, 2101},
      // Coefficients 4 taps
      {64, 128, 64,  0,  51, 126, 75,  4,  40, 122, 86,  8,  30, 115, 97,  14,
       21, 107, 107, 21, 14, 97,  115, 30, 8,  86,  122, 40, 4,  75,  126, 51},
      // Coefficients 6 taps
      {2,  68, 116, 68,  2,  0,  -2, 59, 120, 80,  7,  -8, -5, 49, 117, 90,  14, -9, -8, 39, 114, 99,  21, -9,
       -9, 30, 107, 107, 30, -9, -9, 21, 99,  114, 39, -8, -9, 14, 90,  117, 49, -5, -8, 7,  80,  120, 59, -2},
    },

    // Band_8_2106
    {
      // Scaler inc range
      {2101, 2106},
      // Coefficients 4 taps
      {64, 128, 64,  0,  51, 126, 75,  4,  40, 122, 86,  8,  30, 115, 97,  14,
       21, 107, 107, 21, 14, 97,  115, 30, 8,  86,  122, 40, 4,  75,  126, 51},
      // Coefficients 6 taps
      {2,  68, 116, 68,  2,  0,  -2, 60, 119, 80,  7,  -8, -5, 49, 117, 90,  14, -9, -7, 40, 111, 99,  22, -9,
       -9, 30, 107, 107, 30, -9, -9, 22, 99,  111, 40, -7, -9, 14, 90,  117, 49, -5, -8, 7,  80,  119, 60, -2},
    },

    // Band_8_2110
    {
      // Scaler inc range
      {2106, 2110},
      // Coefficients 4 taps
      {64, 128, 64,  0,  51, 126, 75,  4,  40, 122, 86,  8,  30, 115, 97,  14,
       21, 107, 107, 21, 14, 97,  115, 30, 8,  86,  122, 40, 4,  75,  126, 51},
      // Coefficients 6 taps
      {2,  68, 116, 68,  2,  0,  -2, 60, 118, 80,  8,  -8, -5, 49, 117, 90,  14, -9, -7, 40, 111, 99,  22, -9,
       -9, 30, 107, 107, 30, -9, -9, 22, 99,  111, 40, -7, -9, 14, 90,  117, 49, -5, -8, 8,  80,  118, 60, -2},
    },

    // Band_8_2114
    {
      // Scaler inc range
      {2110, 2114},
      // Coefficients 4 taps
      {64, 128, 64,  0,  52, 125, 75,  4,  40, 121, 86,  9,  30, 115, 97,  14,
       22, 106, 106, 22, 14, 97,  115, 30, 9,  86,  121, 40, 4,  75,  125, 52},
      // Coefficients 6 taps
      {2,  68, 116, 68,  2,  0,  -2, 60, 118, 80,  8,  -8, -5, 50, 116, 90,  14, -9, -7, 40, 112, 98,  22, -9,
       -8, 30, 106, 106, 30, -8, -9, 22, 98,  112, 40, -7, -9, 14, 90,  116, 50, -5, -8, 8,  80,  118, 60, -2},
    },

    // Band_8_2118
    {
      // Scaler inc range
      {2114, 2118},
      // Coefficients 4 taps
      {64, 128, 64,  0,  52, 125, 75,  4,  40, 121, 86,  9,  30, 115, 97,  14,
       22, 106, 106, 22, 14, 97,  115, 30, 9,  86,  121, 40, 4,  75,  125, 52},
      // Coefficients 6 taps
      {2,  68, 116, 68,  2,  0,  -2, 60, 118, 80,  8,  -8, -5, 50, 117, 89,  14, -9, -7, 40, 112, 98,  22, -9,
       -8, 31, 105, 105, 31, -8, -9, 22, 98,  112, 40, -7, -9, 14, 89,  117, 50, -5, -8, 8,  80,  118, 60, -2},
    },

    // Band_8_2123
    {
      // Scaler inc range
      {2118, 2123},
      // Coefficients 4 taps
      {64, 128, 64,  0,  52, 125, 75,  4,  41, 120, 86,  9,  30, 114, 97,  15,
       22, 106, 106, 22, 15, 97,  114, 30, 9,  86,  120, 41, 4,  75,  125, 52},
      // Coefficients 6 taps
      {3,  68, 114, 68,  3,  0,  -2, 60, 118, 80,  8,  -8, -5, 50, 116, 89,  15, -9, -7, 40, 112, 98,  22, -9,
       -8, 31, 105, 105, 31, -8, -9, 22, 98,  112, 40, -7, -9, 15, 89,  116, 50, -5, -8, 8,  80,  118, 60, -2},
    },

    // Band_8_2127
    {
      // Scaler inc range
      {2123, 2127},
      // Coefficients 4 taps
      {64, 128, 64,  0,  52, 125, 75,  4,  41, 120, 86,  9,  31, 113, 97,  15,
       22, 106, 106, 22, 15, 97,  113, 31, 9,  86,  120, 41, 4,  75,  125, 52},
      // Coefficients 6 taps
      {3,  68, 114, 68,  3,  0,  -1, 60, 117, 80,  8,  -8, -5, 50, 115, 89,  15, -8, -7, 40, 112, 98,  22, -9,
       -8, 31, 105, 105, 31, -8, -9, 22, 98,  112, 40, -7, -8, 15, 89,  115, 50, -5, -8, 8,  80,  117, 60, -1},
    },

    // Band_8_2131
    {
      // Scaler inc range
      {2127, 2131},
      // Coefficients 4 taps
      {64, 128, 64,  0,  52, 125, 75,  4,  41, 120, 86,  9,  31, 113, 97,  15,
       22, 106, 106, 22, 15, 97,  113, 31, 9,  86,  120, 41, 4,  75,  125, 52},
      // Coefficients 6 taps
      {3,  68, 114, 68,  3,  0,  -1, 60, 117, 80,  8,  -8, -5, 50, 115, 89,  15, -8, -7, 40, 112, 98,  22, -9,
       -8, 31, 105, 105, 31, -8, -9, 22, 98,  112, 40, -7, -8, 15, 89,  115, 50, -5, -8, 8,  80,  117, 60, -1},
    },

    // Band_8_2136
    {
      // Scaler inc range
      {2131, 2136},
      // Coefficients 4 taps
      {64, 128, 64,  0,  52, 124, 75,  5,  41, 120, 86,  9,  31, 113, 97,  15,
       22, 106, 106, 22, 15, 97,  113, 31, 9,  86,  120, 41, 5,  75,  124, 52},
      // Coefficients 6 taps
      {3,  68, 114, 68,  3,  0,  -1, 60, 118, 79,  8,  -8, -4, 50, 114, 89,  15, -8, -7, 40, 111, 98,  23, -9,
       -8, 31, 105, 105, 31, -8, -9, 23, 98,  111, 40, -7, -8, 15, 89,  114, 50, -4, -8, 8,  79,  118, 60, -1},
    },

    // Band_8_2140
    {
      // Scaler inc range
      {2136, 2140},
      // Coefficients 4 taps
      {64, 128, 64,  0,  52, 124, 75,  5,  41, 120, 86,  9,  31, 114, 96,  15,
       22, 106, 106, 22, 15, 96,  114, 31, 9,  86,  120, 41, 5,  75,  124, 52},
      // Coefficients 6 taps
      {3,  68, 114, 68,  3,  0,  -1, 60, 117, 79,  9,  -8, -4, 50, 114, 89,  15, -8, -7, 40, 111, 97,  23, -8,
       -8, 31, 105, 105, 31, -8, -8, 23, 97,  111, 40, -7, -8, 15, 89,  114, 50, -4, -8, 9,  79,  117, 60, -1},
    },

    // Band_8_2144
    {
      // Scaler inc range
      {2140, 2144},
      // Coefficients 4 taps
      {64, 128, 64,  0,  52, 124, 75,  5,  41, 120, 86,  9,  31, 114, 96,  15,
       22, 106, 106, 22, 15, 96,  114, 31, 9,  86,  120, 41, 5,  75,  124, 52},
      // Coefficients 6 taps
      {3,  68, 114, 68,  3,  0,  -1, 60, 117, 79,  9,  -8, -4, 50, 114, 89,  15, -8, -6, 40, 110, 97,  23, -8,
       -8, 31, 105, 105, 31, -8, -8, 23, 97,  110, 40, -6, -8, 15, 89,  114, 50, -4, -8, 9,  79,  117, 60, -1},
    },

    // Band_8_2149
    {
      // Scaler inc range
      {2144, 2149},
      // Coefficients 4 taps
      {64, 128, 64,  0,  52, 124, 75,  5,  41, 120, 86,  9,  31, 114, 96,  15,
       22, 106, 106, 22, 15, 96,  114, 31, 9,  86,  120, 41, 5,  75,  124, 52},
      // Coefficients 6 taps
      {3,  68, 114, 68,  3,  0,  -1, 60, 117, 79,  9,  -8, -4, 50, 114, 89,  15, -8, -6, 40, 110, 97,  23, -8,
       -8, 31, 105, 105, 31, -8, -8, 23, 97,  110, 40, -6, -8, 15, 89,  114, 50, -4, -8, 9,  79,  117, 60, -1},
    },

    // Band_8_2153
    {
      // Scaler inc range
      {2149, 2153},
      // Coefficients 4 taps
      {65, 126, 65,  0,  52, 124, 75,  5,  41, 120, 86,  9,  31, 114, 96,  15,
       22, 106, 106, 22, 15, 96,  114, 31, 9,  86,  120, 41, 5,  75,  124, 52},
      // Coefficients 6 taps
      {4,  68, 112, 68,  4,  0,  -1, 60, 117, 79,  9,  -8, -4, 50, 113, 89,  16, -8, -6, 40, 110, 97,  23, -8,
       -8, 31, 105, 105, 31, -8, -8, 23, 97,  110, 40, -6, -8, 16, 89,  113, 50, -4, -8, 9,  79,  117, 60, -1},
    },

    // Band_8_2158
    {
      // Scaler inc range
      {2153, 2158},
      // Coefficients 4 taps
      {65, 126, 65,  0,  52, 124, 75,  5,  41, 120, 86,  9,  31, 114, 96,  15,
       23, 105, 105, 23, 15, 96,  114, 31, 9,  86,  120, 41, 5,  75,  124, 52},
      // Coefficients 6 taps
      {4,  68, 112, 68,  4,  0,  -1, 60, 117, 79,  9,  -8, -4, 50, 114, 88,  16, -8, -6, 41, 109, 97,  23, -8,
       -8, 32, 104, 104, 32, -8, -8, 23, 97,  109, 41, -6, -8, 16, 88,  114, 50, -4, -8, 9,  79,  117, 60, -1},
    },

    // Band_8_2162
    {
      // Scaler inc range
      {2158, 2162},
      // Coefficients 4 taps
      {65, 126, 65,  0,  52, 124, 75,  5,  41, 119, 86,  10, 31, 114, 96,  15,
       23, 105, 105, 23, 15, 96,  114, 31, 10, 86,  119, 41, 5,  75,  124, 52},
      // Coefficients 6 taps
      {4,  67, 114, 67,  4,  0,  0,  60, 116, 79,  9,  -8, -4, 50, 114, 88,  16, -8, -6, 41, 109, 97,  23, -8,
       -7, 32, 103, 103, 32, -7, -8, 23, 97,  109, 41, -6, -8, 16, 88,  114, 50, -4, -8, 9,  79,  116, 60, 0},
    },

    // Band_8_2166
    {
      // Scaler inc range
      {2162, 2166},
      // Coefficients 4 taps
      {65, 126, 65,  0,  52, 124, 75,  5,  41, 119, 86,  10, 31, 113, 96,  16,
       23, 105, 105, 23, 16, 96,  113, 31, 10, 86,  119, 41, 5,  75,  124, 52},
      // Coefficients 6 taps
      {4,  67, 114, 67,  4,  0,  0,  60, 116, 79,  9,  -8, -4, 50, 114, 88,  16, -8, -6, 41, 110, 96,  23, -8,
       -7, 32, 103, 103, 32, -7, -8, 23, 96,  110, 41, -6, -8, 16, 88,  114, 50, -4, -8, 9,  79,  116, 60, 0},
    },

    // Band_8_2171
    {
      // Scaler inc range
      {2166, 2171},
      // Coefficients 4 taps
      {65, 126, 65,  0,  52, 124, 75,  5,  41, 119, 86,  10, 32, 112, 96,  16,
       23, 105, 105, 23, 16, 96,  112, 32, 10, 86,  119, 41, 5,  75,  124, 52},
      // Coefficients 6 taps
      {4,  67, 114, 67,  4,  0,  0,  60, 115, 79,  10, -8, -3, 50, 113, 88,  16, -8, -6, 41, 109, 96,  24, -8,
       -7, 32, 103, 103, 32, -7, -8, 24, 96,  109, 41, -6, -8, 16, 88,  113, 50, -3, -8, 10, 79,  115, 60, 0},
    },

    // Band_8_2175
    {
      // Scaler inc range
      {2171, 2175},
      // Coefficients 4 taps
      {65, 126, 65,  0,  52, 124, 75,  5,  41, 119, 86,  10, 32, 112, 96,  16,
       23, 105, 105, 23, 16, 96,  112, 32, 10, 86,  119, 41, 5,  75,  124, 52},
      // Coefficients 6 taps
      {4,  67, 114, 67,  4,  0,  0,  60, 114, 79,  10, -7, -3, 50, 113, 88,  16, -8, -6, 41, 109, 96,  24, -8,
       -7, 32, 103, 103, 32, -7, -8, 24, 96,  109, 41, -6, -8, 16, 88,  113, 50, -3, -7, 10, 79,  114, 60, 0},
    },

    // Band_8_2180
    {
      // Scaler inc range
      {2175, 2180},
      // Coefficients 4 taps
      {65, 126, 65,  0,  52, 124, 75,  5,  42, 118, 86,  10, 32, 112, 96,  16,
       23, 105, 105, 23, 16, 96,  112, 32, 10, 86,  118, 42, 5,  75,  124, 52},
      // Coefficients 6 taps
      {4,  67, 114, 67,  4,  0,  0,  60, 114, 79,  10, -7, -3, 50, 113, 88,  16, -8, -6, 41, 109, 96,  24, -8,
       -7, 32, 103, 103, 32, -7, -8, 24, 96,  109, 41, -6, -8, 16, 88,  113, 50, -3, -7, 10, 79,  114, 60, 0},
    },

    // Band_8_2185
    {
      // Scaler inc range
      {2180, 2185},
      // Coefficients 4 taps
      {65, 126, 65,  0,  53, 123, 75,  5,  42, 118, 86,  10, 32, 112, 96,  16,
       23, 105, 105, 23, 16, 96,  112, 32, 10, 86,  118, 42, 5,  75,  123, 53},
      // Coefficients 6 taps
      {4,  67, 114, 67,  4,  0,  0,  60, 114, 79,  10, -7, -3, 50, 113, 88,  16, -8, -5, 41, 108, 96,  24, -8,
       -7, 32, 103, 103, 32, -7, -8, 24, 96,  108, 41, -5, -8, 16, 88,  113, 50, -3, -7, 10, 79,  114, 60, 0},
    },

    // Band_8_2189
    {
      // Scaler inc range
      {2185, 2189},
      // Coefficients 4 taps
      {65, 126, 65,  0,  53, 122, 75,  6,  42, 118, 86,  10, 32, 112, 96,  16,
       23, 105, 105, 23, 16, 96,  112, 32, 10, 86,  118, 42, 6,  75,  122, 53},
      // Coefficients 6 taps
      {5,  67, 112, 67,  5,  0,  0,  60, 114, 79,  10, -7, -3, 50, 112, 88,  17, -8, -5, 41, 108, 96,  24, -8,
       -7, 32, 103, 103, 32, -7, -8, 24, 96,  108, 41, -5, -8, 17, 88,  112, 50, -3, -7, 10, 79,  114, 60, 0},
    },

    // Band_8_2194
    {
      // Scaler inc range
      {2189, 2194},
      // Coefficients 4 taps
      {65, 126, 65,  0,  53, 122, 75,  6,  42, 118, 86,  10, 32, 112, 96,  16,
       23, 105, 105, 23, 16, 96,  112, 32, 10, 86,  118, 42, 6,  75,  122, 53},
      // Coefficients 6 taps
      {5,  67, 112, 67,  5,  0,  1,  60, 113, 79,  10, -7, -3, 50, 113, 87,  17, -8, -5, 41, 108, 96,  24, -8,
       -7, 32, 103, 103, 32, -7, -8, 24, 96,  108, 41, -5, -8, 17, 87,  113, 50, -3, -7, 10, 79,  113, 60, 1},
    },

    // Band_8_2198
    {
      // Scaler inc range
      {2194, 2198},
      // Coefficients 4 taps
      {65, 126, 65,  0,  53, 122, 75,  6,  42, 118, 86,  10, 32, 112, 96,  16,
       23, 105, 105, 23, 16, 96,  112, 32, 10, 86,  118, 42, 6,  75,  122, 53},
      // Coefficients 6 taps
      {5,  67, 112, 67,  5,  0,  1,  60, 114, 78,  10, -7, -3, 50, 113, 87,  17, -8, -5, 41, 109, 95,  24, -8,
       -7, 32, 103, 103, 32, -7, -8, 24, 95,  109, 41, -5, -8, 17, 87,  113, 50, -3, -7, 10, 78,  114, 60, 1},
    },

    // Band_8_2203
    {
      // Scaler inc range
      {2198, 2203},
      // Coefficients 4 taps
      {65, 126, 65,  0,  53, 122, 75,  6,  42, 118, 86,  10, 32, 112, 96,  16,
       24, 104, 104, 24, 16, 96,  112, 32, 10, 86,  118, 42, 6,  75,  122, 53},
      // Coefficients 6 taps
      {5,  67, 112, 67,  5,  0,  1,  60, 113, 78,  11, -7, -3, 50, 113, 87,  17, -8, -5, 41, 108, 95,  24, -7,
       -7, 33, 102, 102, 33, -7, -7, 24, 95,  108, 41, -5, -8, 17, 87,  113, 50, -3, -7, 11, 78,  113, 60, 1},
    },

    // Band_8_2208
    {
      // Scaler inc range
      {2203, 2208},
      // Coefficients 4 taps
      {65, 126, 65,  0,  53, 122, 75,  6,  42, 118, 86,  10, 32, 113, 95,  16,
       24, 104, 104, 24, 16, 95,  113, 32, 10, 86,  118, 42, 6,  75,  122, 53},
      // Coefficients 6 taps
      {5,  67, 112, 67,  5,  0,  1,  60, 113, 78,  11, -7, -2, 51, 111, 87,  17, -8, -5, 41, 107, 95,  25, -7,
       -7, 33, 102, 102, 33, -7, -7, 25, 95,  107, 41, -5, -8, 17, 87,  111, 51, -2, -7, 11, 78,  113, 60, 1},
    },

    // Band_8_2212
    {
      // Scaler inc range
      {2208, 2212},
      // Coefficients 4 taps
      {65, 126, 65,  0,  53, 122, 75,  6,  42, 117, 86,  11, 32, 113, 95,  16,
       24, 104, 104, 24, 16, 95,  113, 32, 11, 86,  117, 42, 6,  75,  122, 53},
      // Coefficients 6 taps
      {5,  67, 112, 67,  5,  0,  1,  60, 113, 78,  11, -7, -2, 51, 111, 87,  17, -8, -5, 41, 107, 95,  25, -7,
       -6, 33, 101, 101, 33, -6, -7, 25, 95,  107, 41, -5, -8, 17, 87,  111, 51, -2, -7, 11, 78,  113, 60, 1},
    },

    // Band_8_2217
    {
      // Scaler inc range
      {2212, 2217},
      // Coefficients 4 taps
      {65, 126, 65,  0,  53, 122, 75,  6,  42, 117, 86,  11, 32, 112, 95,  17,
       24, 104, 104, 24, 17, 95,  112, 32, 11, 86,  117, 42, 6,  75,  122, 53},
      // Coefficients 6 taps
      {5,  67, 112, 67,  5,  0,  1,  60, 113, 78,  11, -7, -2, 51, 110, 87,  17, -7, -5, 42, 106, 95,  25, -7,
       -6, 33, 101, 101, 33, -6, -7, 25, 95,  106, 42, -5, -7, 17, 87,  110, 51, -2, -7, 11, 78,  113, 60, 1},
    },

    // Band_8_2222
    {
      // Scaler inc range
      {2217, 2222},
      // Coefficients 4 taps
      {65, 126, 65,  0,  53, 122, 75,  6,  42, 117, 86,  11, 33, 111, 95,  17,
       24, 104, 104, 24, 17, 95,  111, 33, 11, 86,  117, 42, 6,  75,  122, 53},
      // Coefficients 6 taps
      {6,  67, 110, 67,  6,  0,  1,  60, 113, 78,  11, -7, -2, 51, 109, 87,  18, -7, -5, 42, 106, 95,  25, -7,
       -6, 33, 101, 101, 33, -6, -7, 25, 95,  106, 42, -5, -7, 18, 87,  109, 51, -2, -7, 11, 78,  113, 60, 1},
    },

    // Band_8_2226
    {
      // Scaler inc range
      {2222, 2226},
      // Coefficients 4 taps
      {65, 126, 65,  0,  53, 122, 75,  6,  42, 117, 86,  11, 33, 111, 95,  17,
       24, 104, 104, 24, 17, 95,  111, 33, 11, 86,  117, 42, 6,  75,  122, 53},
      // Coefficients 6 taps
      {6,  67, 110, 67,  6,  0,  1,  60, 113, 78,  11, -7, -2, 51, 109, 87,  18, -7, -4, 42, 106, 94,  25, -7,
       -6, 33, 101, 101, 33, -6, -7, 25, 94,  106, 42, -4, -7, 18, 87,  109, 51, -2, -7, 11, 78,  113, 60, 1},
    },

    // Band_8_2231
    {
      // Scaler inc range
      {2226, 2231},
      // Coefficients 4 taps
      {65, 126, 65,  0,  53, 122, 75,  6,  42, 118, 85,  11, 33, 111, 95,  17,
       24, 104, 104, 24, 17, 95,  111, 33, 11, 85,  118, 42, 6,  75,  122, 53},
      // Coefficients 6 taps
      {6,  67, 110, 67,  6,  0,  2,  60, 112, 78,  11, -7, -2, 51, 109, 87,  18, -7, -4, 42, 106, 94,  25, -7,
       -6, 33, 101, 101, 33, -6, -7, 25, 94,  106, 42, -4, -7, 18, 87,  109, 51, -2, -7, 11, 78,  112, 60, 2},
    },

    // Band_8_2236
    {
      // Scaler inc range
      {2231, 2236},
      // Coefficients 4 taps
      {66, 124, 66,  0,  53, 122, 75,  6,  42, 118, 85,  11, 33, 111, 95,  17,
       24, 104, 104, 24, 17, 95,  111, 33, 11, 85,  118, 42, 6,  75,  122, 53},
      // Coefficients 6 taps
      {6,  67, 110, 67,  6,  0,  2,  60, 111, 78,  12, -7, -2, 51, 110, 86,  18, -7, -4, 42, 106, 94,  25, -7,
       -6, 33, 101, 101, 33, -6, -7, 25, 94,  106, 42, -4, -7, 18, 86,  110, 51, -2, -7, 12, 78,  111, 60, 2},
    },

    // Band_8_2241
    {
      // Scaler inc range
      {2236, 2241},
      // Coefficients 4 taps
      {66, 124, 66,  0,  53, 122, 75,  6,  43, 117, 85,  11, 33, 111, 95,  17,
       24, 104, 104, 24, 17, 95,  111, 33, 11, 85,  117, 43, 6,  75,  122, 53},
      // Coefficients 6 taps
      {6,  67, 110, 67,  6,  0,  2,  60, 111, 78,  12, -7, -2, 51, 110, 86,  18, -7, -4, 42, 106, 94,  25, -7,
       -6, 33, 101, 101, 33, -6, -7, 25, 94,  106, 42, -4, -7, 18, 86,  110, 51, -2, -7, 12, 78,  111, 60, 2},
    },

    // Band_8_2245
    {
      // Scaler inc range
      {2241, 2245},
      // Coefficients 4 taps
      {66, 124, 66,  0,  53, 121, 75,  7,  43, 117, 85,  11, 33, 111, 95,  17,
       24, 104, 104, 24, 17, 95,  111, 33, 11, 85,  117, 43, 7,  75,  121, 53},
      // Coefficients 6 taps
      {6,  67, 110, 67,  6,  0,  2,  60, 111, 78,  12, -7, -1, 51, 109, 86,  18, -7, -4, 42, 106, 94,  25, -7,
       -6, 33, 101, 101, 33, -6, -7, 25, 94,  106, 42, -4, -7, 18, 86,  109, 51, -1, -7, 12, 78,  111, 60, 2},
    },

    // Band_8_2250
    {
      // Scaler inc range
      {2245, 2250},
      // Coefficients 4 taps
      {66, 124, 66,  0,  53, 121, 75,  7,  43, 117, 85,  11, 33, 111, 95,  17,
       25, 103, 103, 25, 17, 95,  111, 33, 11, 85,  117, 43, 7,  75,  121, 53},
      // Coefficients 6 taps
      {6,  67, 110, 67,  6,  0,  2,  60, 111, 78,  12, -7, -1, 51, 109, 86,  18, -7, -4, 42, 105, 94,  26, -7,
       -6, 34, 100, 100, 34, -6, -7, 26, 94,  105, 42, -4, -7, 18, 86,  109, 51, -1, -7, 12, 78,  111, 60, 2},
    },

    // Band_8_2255
    {
      // Scaler inc range
      {2250, 2255},
      // Coefficients 4 taps
      {66, 124, 66,  0,  53, 121, 75,  7,  43, 117, 85,  11, 33, 111, 95,  17,
       25, 103, 103, 25, 17, 95,  111, 33, 11, 85,  117, 43, 7,  75,  121, 53},
      // Coefficients 6 taps
      {7,  67, 108, 67,  7,  0,  2,  60, 111, 78,  12, -7, -1, 51, 109, 86,  18, -7, -4, 42, 105, 94,  26, -7,
       -6, 34, 100, 100, 34, -6, -7, 26, 94,  105, 42, -4, -7, 18, 86,  109, 51, -1, -7, 12, 78,  111, 60, 2},
    },

    // Band_8_2260
    {
      // Scaler inc range
      {2255, 2260},
      // Coefficients 4 taps
      {66, 124, 66,  0,  53, 121, 75,  7,  43, 117, 85,  11, 33, 111, 95,  17,
       25, 103, 103, 25, 17, 95,  111, 33, 11, 85,  117, 43, 7,  75,  121, 53},
      // Coefficients 6 taps
      {7,  67, 108, 67,  7,  0,  2,  60, 111, 78,  12, -7, -1, 51, 108, 86,  19, -7, -4, 42, 106, 93,  26, -7,
       -6, 34, 100, 100, 34, -6, -7, 26, 93,  106, 42, -4, -7, 19, 86,  108, 51, -1, -7, 12, 78,  111, 60, 2},
    },

    // Band_8_2265
    {
      // Scaler inc range
      {2260, 2265},
      // Coefficients 4 taps
      {66, 124, 66,  0,  54, 120, 75,  7,  43, 116, 85,  12, 33, 110, 95,  18,
       25, 103, 103, 25, 18, 95,  110, 33, 12, 85,  116, 43, 7,  75,  120, 54},
      // Coefficients 6 taps
      {7,  67, 108, 67, 7,  0,  3,  60, 111, 77,  12, -7, -1, 51, 108, 86,  19, -7, -4, 42, 105, 93,  26, -6,
       -5, 34, 99,  99, 34, -5, -6, 26, 93,  105, 42, -4, -7, 19, 86,  108, 51, -1, -7, 12, 77,  111, 60, 3},
    },

    // Band_8_2270
    {
      // Scaler inc range
      {2265, 2270},
      // Coefficients 4 taps
      {66, 124, 66,  0,  54, 120, 75,  7,  43, 116, 85,  12, 33, 110, 95,  18,
       25, 103, 103, 25, 18, 95,  110, 33, 12, 85,  116, 43, 7,  75,  120, 54},
      // Coefficients 6 taps
      {7,  67, 108, 67, 7,  0,  3,  60, 110, 77,  13, -7, -1, 51, 108, 86,  19, -7, -3, 42, 104, 93,  26, -6,
       -5, 34, 99,  99, 34, -5, -6, 26, 93,  104, 42, -3, -7, 19, 86,  108, 51, -1, -7, 13, 77,  110, 60, 3},
    },

    // Band_8_2275
    {
      // Scaler inc range
      {2270, 2275},
      // Coefficients 4 taps
      {66, 124, 66,  0,  54, 120, 75,  7,  43, 116, 85,  12, 33, 110, 95,  18,
       25, 103, 103, 25, 18, 95,  110, 33, 12, 85,  116, 43, 7,  75,  120, 54},
      // Coefficients 6 taps
      {7,  67, 108, 67, 7,  0,  3,  60, 110, 77,  13, -7, -1, 51, 109, 85,  19, -7, -3, 42, 104, 93,  26, -6,
       -5, 34, 99,  99, 34, -5, -6, 26, 93,  104, 42, -3, -7, 19, 85,  109, 51, -1, -7, 13, 77,  110, 60, 3},
    },

    // Band_8_2280
    {
      // Scaler inc range
      {2275, 2280},
      // Coefficients 4 taps
      {66, 124, 66,  0,  54, 120, 75,  7,  43, 116, 85,  12, 34, 109, 95,  18,
       25, 103, 103, 25, 18, 95,  109, 34, 12, 85,  116, 43, 7,  75,  120, 54},
      // Coefficients 6 taps
      {7,  67, 108, 67, 7,  0,  3,  60, 110, 77,  13, -7, -1, 51, 109, 85,  19, -7, -3, 42, 104, 93,  26, -6,
       -5, 34, 99,  99, 34, -5, -6, 26, 93,  104, 42, -3, -7, 19, 85,  109, 51, -1, -7, 13, 77,  110, 60, 3},
    },

    // Band_8_2284
    {
      // Scaler inc range
      {2280, 2284},
      // Coefficients 4 taps
      {66, 124, 66,  0,  54, 120, 75,  7,  43, 116, 85,  12, 34, 110, 94,  18,
       25, 103, 103, 25, 18, 94,  110, 34, 12, 85,  116, 43, 7,  75,  120, 54},
      // Coefficients 6 taps
      {7,  67, 108, 67, 7,  0,  3,  60, 110, 77,  13, -7, 0,  51, 108, 85,  19, -7, -3, 43, 103, 93,  26, -6,
       -5, 34, 99,  99, 34, -5, -6, 26, 93,  103, 43, -3, -7, 19, 85,  108, 51, 0,  -7, 13, 77,  110, 60, 3},
    },

    // Band_8_2289
    {
      // Scaler inc range
      {2284, 2289},
      // Coefficients 4 taps
      {66, 124, 66,  0,  54, 120, 75,  7,  43, 116, 85,  12, 34, 110, 94,  18,
       25, 103, 103, 25, 18, 94,  110, 34, 12, 85,  116, 43, 7,  75,  120, 54},
      // Coefficients 6 taps
      {8,  67, 106, 67, 8,  0,  3,  60, 110, 77,  13, -7, 0,  51, 108, 85,  19, -7, -3, 43, 104, 92,  26, -6,
       -5, 34, 99,  99, 34, -5, -6, 26, 92,  104, 43, -3, -7, 19, 85,  108, 51, 0,  -7, 13, 77,  110, 60, 3},
    },

    // Band_8_2294
    {
      // Scaler inc range
      {2289, 2294},
      // Coefficients 4 taps
      {66, 124, 66,  0,  54, 120, 75,  7,  43, 116, 85,  12, 34, 110, 94,  18,
       25, 103, 103, 25, 18, 94,  110, 34, 12, 85,  116, 43, 7,  75,  120, 54},
      // Coefficients 6 taps
      {8,  67, 106, 67, 8,  0,  3,  60, 109, 77,  13, -6, 0,  51, 106, 85,  20, -6, -3, 43, 103, 92,  27, -6,
       -5, 34, 99,  99, 34, -5, -6, 27, 92,  103, 43, -3, -6, 20, 85,  106, 51, 0,  -6, 13, 77,  109, 60, 3},
    },

    // Band_8_2300
    {
      // Scaler inc range
      {2294, 2300},
      // Coefficients 4 taps
      {66, 124, 66,  0,  54, 120, 75,  7,  43, 116, 85,  12, 34, 110, 94,  18,
       25, 103, 103, 25, 18, 94,  110, 34, 12, 85,  116, 43, 7,  75,  120, 54},
      // Coefficients 6 taps
      {8,  67, 106, 67, 8,  0,  3,  60, 109, 77,  13, -6, 0,  51, 106, 85,  20, -6, -3, 43, 103, 92,  27, -6,
       -5, 34, 99,  99, 34, -5, -6, 27, 92,  103, 43, -3, -6, 20, 85,  106, 51, 0,  -6, 13, 77,  109, 60, 3},
    },

    // Band_8_2305
    {
      // Scaler inc range
      {2300, 2305},
      // Coefficients 4 taps
      {66, 124, 66,  0,  54, 119, 75,  8,  44, 115, 85,  12, 34, 110, 94,  18,
       26, 102, 102, 26, 18, 94,  110, 34, 12, 85,  115, 44, 8,  75,  119, 54},
      // Coefficients 6 taps
      {8,  67, 106, 67, 8,  0,  4,  60, 108, 77,  13, -6, 0,  51, 106, 85,  20, -6, -3, 43, 103, 92,  27, -6,
       -5, 35, 98,  98, 35, -5, -6, 27, 92,  103, 43, -3, -6, 20, 85,  106, 51, 0,  -6, 13, 77,  108, 60, 4},
    },

    // Band_8_2310
    {
      // Scaler inc range
      {2305, 2310},
      // Coefficients 4 taps
      {66, 124, 66,  0,  54, 119, 75,  8,  44, 115, 85,  12, 34, 110, 94,  18,
       26, 102, 102, 26, 18, 94,  110, 34, 12, 85,  115, 44, 8,  75,  119, 54},
      // Coefficients 6 taps
      {8,  67, 106, 67, 8,  0,  4,  60, 107, 77,  14, -6, 0,  51, 106, 85,  20, -6, -3, 43, 103, 92,  27, -6,
       -5, 35, 98,  98, 35, -5, -6, 27, 92,  103, 43, -3, -6, 20, 85,  106, 51, 0,  -6, 14, 77,  107, 60, 4},
    },

    // Band_8_2315
    {
      // Scaler inc range
      {2310, 2315},
      // Coefficients 4 taps
      {66, 124, 66,  0,  54, 119, 75,  8,  44, 115, 85,  12, 34, 110, 94,  18,
       26, 102, 102, 26, 18, 94,  110, 34, 12, 85,  115, 44, 8,  75,  119, 54},
      // Coefficients 6 taps
      {8,  67, 106, 67, 8,  0,  4,  60, 107, 77,  14, -6, 0,  51, 106, 85,  20, -6, -2, 43, 102, 92,  27, -6,
       -4, 35, 97,  97, 35, -4, -6, 27, 92,  102, 43, -2, -6, 20, 85,  106, 51, 0,  -6, 14, 77,  107, 60, 4},
    },

    // Band_8_2320
    {
      // Scaler inc range
      {2315, 2320},
      // Coefficients 4 taps
      {66, 124, 66,  0,  54, 119, 75,  8,  44, 114, 85,  13, 34, 109, 94,  19,
       26, 102, 102, 26, 19, 94,  109, 34, 13, 85,  114, 44, 8,  75,  119, 54},
      // Coefficients 6 taps
      {8,  67, 106, 67, 8,  0,  4,  60, 107, 77,  14, -6, 0,  51, 107, 84,  20, -6, -2, 43, 103, 91,  27, -6,
       -4, 35, 97,  97, 35, -4, -6, 27, 91,  103, 43, -2, -6, 20, 84,  107, 51, 0,  -6, 14, 77,  107, 60, 4},
    },

    // Band_8_2325
    {
      // Scaler inc range
      {2320, 2325},
      // Coefficients 4 taps
      {66, 124, 66,  0,  54, 119, 75,  8,  44, 114, 85,  13, 34, 109, 94,  19,
       26, 102, 102, 26, 19, 94,  109, 34, 13, 85,  114, 44, 8,  75,  119, 54},
      // Coefficients 6 taps
      {8,  67, 106, 67, 8,  0,  4,  60, 107, 77,  14, -6, 1,  51, 106, 84,  20, -6, -2, 43, 102, 91,  27, -5,
       -4, 35, 97,  97, 35, -4, -5, 27, 91,  102, 43, -2, -6, 20, 84,  106, 51, 1,  -6, 14, 77,  107, 60, 4},
    },

    // Band_8_2330
    {
      // Scaler inc range
      {2325, 2330},
      // Coefficients 4 taps
      {67, 122, 67,  0,  54, 119, 75,  8,  44, 114, 85,  13, 34, 109, 94,  19,
       26, 102, 102, 26, 19, 94,  109, 34, 13, 85,  114, 44, 8,  75,  119, 54},
      // Coefficients 6 taps
      {9,  67, 104, 67, 9,  0,  4,  60, 108, 76,  14, -6, 1,  51, 106, 84,  20, -6, -2, 43, 102, 91,  27, -5,
       -4, 35, 97,  97, 35, -4, -5, 27, 91,  102, 43, -2, -6, 20, 84,  106, 51, 1,  -6, 14, 76,  108, 60, 4},
    },

    // Band_8_2335
    {
      // Scaler inc range
      {2330, 2335},
      // Coefficients 4 taps
      {67, 122, 67,  0,  54, 119, 75,  8,  44, 114, 85,  13, 34, 109, 94,  19,
       26, 102, 102, 26, 19, 94,  109, 34, 13, 85,  114, 44, 8,  75,  119, 54},
      // Coefficients 6 taps
      {9,  67, 104, 67, 9,  0,  4,  60, 108, 76,  14, -6, 1,  51, 105, 84,  21, -6, -2, 43, 102, 91,  27, -5,
       -4, 35, 97,  97, 35, -4, -5, 27, 91,  102, 43, -2, -6, 21, 84,  105, 51, 1,  -6, 14, 76,  108, 60, 4},
    },

    // Band_8_2341
    {
      // Scaler inc range
      {2335, 2341},
      // Coefficients 4 taps
      {67, 122, 67,  0,  54, 119, 75,  8,  44, 114, 85,  13, 35, 108, 94,  19,
       26, 102, 102, 26, 19, 94,  108, 35, 13, 85,  114, 44, 8,  75,  119, 54},
      // Coefficients 6 taps
      {9,  67, 104, 67, 9,  0,  5,  60, 107, 76,  14, -6, 1,  51, 105, 84,  21, -6, -2, 43, 101, 91,  28, -5,
       -4, 35, 97,  97, 35, -4, -5, 28, 91,  101, 43, -2, -6, 21, 84,  105, 51, 1,  -6, 14, 76,  107, 60, 5},
    },

    // Band_8_2346
    {
      // Scaler inc range
      {2341, 2346},
      // Coefficients 4 taps
      {67, 122, 67,  0,  54, 119, 75,  8,  44, 114, 85,  13, 35, 108, 94,  19,
       26, 102, 102, 26, 19, 94,  108, 35, 13, 85,  114, 44, 8,  75,  119, 54},
      // Coefficients 6 taps
      {9,  67, 104, 67, 9,  0,  5,  60, 106, 76,  15, -6, 1,  52, 104, 84,  21, -6, -2, 43, 101, 91,  28, -5,
       -4, 35, 97,  97, 35, -4, -5, 28, 91,  101, 43, -2, -6, 21, 84,  104, 52, 1,  -6, 15, 76,  106, 60, 5},
    },

    // Band_8_2351
    {
      // Scaler inc range
      {2346, 2351},
      // Coefficients 4 taps
      {67, 122, 67,  0,  54, 119, 75,  8,  44, 114, 85,  13, 35, 108, 94,  19,
       26, 102, 102, 26, 19, 94,  108, 35, 13, 85,  114, 44, 8,  75,  119, 54},
      // Coefficients 6 taps
      {9,  67, 104, 67, 9,  0,  5,  60, 106, 76,  15, -6, 1,  52, 104, 84,  21, -6, -2, 43, 101, 91,  28, -5,
       -4, 35, 97,  97, 35, -4, -5, 28, 91,  101, 43, -2, -6, 21, 84,  104, 52, 1,  -6, 15, 76,  106, 60, 5},
    },

    // Band_8_2356
    {
      // Scaler inc range
      {2351, 2356},
      // Coefficients 4 taps
      {67, 122, 67,  0,  55, 118, 75,  8,  44, 114, 85,  13, 35, 108, 94,  19,
       26, 102, 102, 26, 19, 94,  108, 35, 13, 85,  114, 44, 8,  75,  118, 55},
      // Coefficients 6 taps
      {9,  67, 104, 67, 9,  0,  5,  60, 106, 76,  15, -6, 1,  52, 104, 84,  21, -6, -2, 43, 102, 90,  28, -5,
       -4, 35, 97,  97, 35, -4, -5, 28, 90,  102, 43, -2, -6, 21, 84,  104, 52, 1,  -6, 15, 76,  106, 60, 5},
    },

    // Band_8_2362
    {
      // Scaler inc range
      {2356, 2362},
      // Coefficients 4 taps
      {67, 122, 67,  0,  55, 118, 75,  8,  44, 114, 85,  13, 35, 108, 94,  19,
       27, 101, 101, 27, 19, 94,  108, 35, 13, 85,  114, 44, 8,  75,  118, 55},
      // Coefficients 6 taps
      {9,  67, 104, 67, 9,  0,  5,  60, 106, 76,  15, -6, 1,  52, 105, 83,  21, -6, -1, 43, 101, 90,  28, -5,
       -4, 35, 97,  97, 35, -4, -5, 28, 90,  101, 43, -1, -6, 21, 83,  105, 52, 1,  -6, 15, 76,  106, 60, 5},
    },

    // Band_8_2367
    {
      // Scaler inc range
      {2362, 2367},
      // Coefficients 4 taps
      {67, 122, 67,  0,  55, 117, 75,  9,  44, 114, 85,  13, 35, 109, 93,  19,
       27, 101, 101, 27, 19, 93,  109, 35, 13, 85,  114, 44, 9,  75,  117, 55},
      // Coefficients 6 taps
      {10, 66, 104, 66, 10, 0,  5,  60, 106, 76,  15, -6, 2,  52, 104, 83,  21, -6, -1, 43, 101, 90,  28, -5,
       -3, 36, 95,  95, 36, -3, -5, 28, 90,  101, 43, -1, -6, 21, 83,  104, 52, 2,  -6, 15, 76,  106, 60, 5},
    },

    // Band_8_2372
    {
      // Scaler inc range
      {2367, 2372},
      // Coefficients 4 taps
      {67, 122, 67,  0,  55, 117, 75,  9,  44, 114, 85,  13, 35, 109, 93,  19,
       27, 101, 101, 27, 19, 93,  109, 35, 13, 85,  114, 44, 9,  75,  117, 55},
      // Coefficients 6 taps
      {10, 66, 104, 66, 10, 0,  5,  60, 106, 76,  15, -6, 2,  52, 103, 83,  21, -5, -1, 44, 100, 90,  28, -5,
       -3, 36, 95,  95, 36, -3, -5, 28, 90,  100, 44, -1, -5, 21, 83,  103, 52, 2,  -6, 15, 76,  106, 60, 5},
    },

    // Band_8_2378
    {
      // Scaler inc range
      {2372, 2378},
      // Coefficients 4 taps
      {67, 122, 67,  0,  55, 117, 75,  9,  44, 113, 85,  14, 35, 108, 93,  20,
       27, 101, 101, 27, 20, 93,  108, 35, 14, 85,  113, 44, 9,  75,  117, 55},
      // Coefficients 6 taps
      {10, 66, 104, 66, 10, 0,  5,  60, 106, 76,  15, -6, 2,  52, 103, 83,  21, -5, -1, 44, 100, 90,  28, -5,
       -3, 36, 95,  95, 36, -3, -5, 28, 90,  100, 44, -1, -5, 21, 83,  103, 52, 2,  -6, 15, 76,  106, 60, 5},
    },

    // Band_8_2383
    {
      // Scaler inc range
      {2378, 2383},
      // Coefficients 4 taps
      {67, 122, 67,  0,  55, 117, 75,  9,  45, 112, 85,  14, 35, 108, 93,  20,
       27, 101, 101, 27, 20, 93,  108, 35, 14, 85,  112, 45, 9,  75,  117, 55},
      // Coefficients 6 taps
      {10, 66, 104, 66, 10, 0,  6,  60, 105, 76,  15, -6, 2,  52, 102, 83,  22, -5, -1, 44, 100, 90,  28, -5,
       -3, 36, 95,  95, 36, -3, -5, 28, 90,  100, 44, -1, -5, 22, 83,  102, 52, 2,  -6, 15, 76,  105, 60, 6},
    },

    // Band_8_2389
    {
      // Scaler inc range
      {2383, 2389},
      // Coefficients 4 taps
      {67, 122, 67,  0,  55, 117, 75,  9,  45, 112, 85,  14, 35, 108, 93,  20,
       27, 101, 101, 27, 20, 93,  108, 35, 14, 85,  112, 45, 9,  75,  117, 55},
      // Coefficients 6 taps
      {10, 66, 104, 66, 10, 0,  6,  60, 104, 76, 16, -6, 2,  52, 102, 83,  22, -5, -1, 44, 99, 89,  29, -4,
       -3, 36, 95,  95, 36, -3, -4, 29, 89,  99, 44, -1, -5, 22, 83,  102, 52, 2,  -6, 16, 76, 104, 60, 6},
    },

    // Band_8_2394
    {
      // Scaler inc range
      {2389, 2394},
      // Coefficients 4 taps
      {67, 122, 67,  0,  55, 117, 75,  9,  45, 112, 85,  14, 35, 108, 93,  20,
       27, 101, 101, 27, 20, 93,  108, 35, 14, 85,  112, 45, 9,  75,  117, 55},
      // Coefficients 6 taps
      {10, 66, 104, 66, 10, 0,  6,  60, 104, 76, 16, -6, 2,  52, 102, 83,  22, -5, -1, 44, 99, 89,  29, -4,
       -3, 36, 95,  95, 36, -3, -4, 29, 89,  99, 44, -1, -5, 22, 83,  102, 52, 2,  -6, 16, 76, 104, 60, 6},
    },

    // Band_8_2399
    {
      // Scaler inc range
      {2394, 2399},
      // Coefficients 4 taps
      {67, 122, 67,  0,  55, 117, 75,  9,  45, 113, 84,  14, 35, 108, 93,  20,
       27, 101, 101, 27, 20, 93,  108, 35, 14, 84,  113, 45, 9,  75,  117, 55},
      // Coefficients 6 taps
      {10, 66, 104, 66, 10, 0,  6,  60, 104, 75, 16, -5, 2,  52, 102, 83,  22, -5, -1, 44, 99, 89,  29, -4,
       -3, 36, 95,  95, 36, -3, -4, 29, 89,  99, 44, -1, -5, 22, 83,  102, 52, 2,  -5, 16, 75, 104, 60, 6},
    },

    // Band_8_2405
    {
      // Scaler inc range
      {2399, 2405},
      // Coefficients 4 taps
      {67, 122, 67,  0,  55, 117, 75,  9,  45, 113, 84,  14, 36, 107, 93,  20,
       27, 101, 101, 27, 20, 93,  107, 36, 14, 84,  113, 45, 9,  75,  117, 55},
      // Coefficients 6 taps
      {10, 66, 104, 66, 10, 0,  6,  60, 104, 75, 16, -5, 2,  52, 102, 83,  22, -5, -1, 44, 99, 89,  29, -4,
       -3, 36, 95,  95, 36, -3, -4, 29, 89,  99, 44, -1, -5, 22, 83,  102, 52, 2,  -5, 16, 75, 104, 60, 6},
    },

    // Band_8_2411
    {
      // Scaler inc range
      {2405, 2411},
      // Coefficients 4 taps
      {67, 122, 67,  0,  55, 117, 75,  9,  45, 113, 84,  14, 36, 107, 93,  20,
       27, 101, 101, 27, 20, 93,  107, 36, 14, 84,  113, 45, 9,  75,  117, 55},
      // Coefficients 6 taps
      {11, 66, 102, 66, 11, 0,  6,  60, 104, 75, 16, -5, 3,  52, 102, 82,  22, -5, 0,  44, 98, 89,  29, -4,
       -3, 36, 95,  95, 36, -3, -4, 29, 89,  98, 44, 0,  -5, 22, 82,  102, 52, 3,  -5, 16, 75, 104, 60, 6},
    },

    // Band_8_2416
    {
      // Scaler inc range
      {2411, 2416},
      // Coefficients 4 taps
      {67, 122, 67,  0,  55, 117, 75,  9,  45, 113, 84,  14, 36, 107, 93,  20,
       27, 101, 101, 27, 20, 93,  107, 36, 14, 84,  113, 45, 9,  75,  117, 55},
      // Coefficients 6 taps
      {11, 66, 102, 66, 11, 0,  6,  60, 104, 75, 16, -5, 3,  52, 102, 82,  22, -5, 0,  44, 98, 89,  29, -4,
       -3, 36, 95,  95, 36, -3, -4, 29, 89,  98, 44, 0,  -5, 22, 82,  102, 52, 3,  -5, 16, 75, 104, 60, 6},
    },

    // Band_8_2422
    {
      // Scaler inc range
      {2416, 2422},
      // Coefficients 4 taps
      {67, 122, 67,  0,  55, 117, 75,  9,  45, 113, 84,  14, 36, 107, 93,  20,
       27, 101, 101, 27, 20, 93,  107, 36, 14, 84,  113, 45, 9,  75,  117, 55},
      // Coefficients 6 taps
      {11, 66, 102, 66, 11, 0,  7,  60, 103, 75, 16, -5, 3,  52, 102, 82,  22, -5, 0,  44, 98, 89,  29, -4,
       -2, 36, 94,  94, 36, -2, -4, 29, 89,  98, 44, 0,  -5, 22, 82,  102, 52, 3,  -5, 16, 75, 103, 60, 7},
    },

    // Band_8_2427
    {
      // Scaler inc range
      {2422, 2427},
      // Coefficients 4 taps
      {67, 122, 67,  0,  55, 117, 75,  9,  45, 113, 84,  14, 36, 107, 93,  20,
       28, 100, 100, 28, 20, 93,  107, 36, 14, 84,  113, 45, 9,  75,  117, 55},
      // Coefficients 6 taps
      {11, 66, 102, 66, 11, 0,  7,  60, 103, 75, 16, -5, 3,  52, 101, 82,  23, -5, 0,  44, 99, 88,  29, -4,
       -2, 36, 94,  94, 36, -2, -4, 29, 88,  99, 44, 0,  -5, 23, 82,  101, 52, 3,  -5, 16, 75, 103, 60, 7},
    },

    // Band_8_2433
    {
      // Scaler inc range
      {2427, 2433},
      // Coefficients 4 taps
      {68, 120, 68,  0,  55, 116, 75,  10, 45, 113, 84,  14, 36, 107, 93,  20,
       28, 100, 100, 28, 20, 93,  107, 36, 14, 84,  113, 45, 10, 75,  116, 55},
      // Coefficients 6 taps
      {11, 66, 102, 66, 11, 0,  7,  60, 102, 75, 17, -5, 3,  52, 101, 82,  23, -5, 0,  44, 99, 88,  29, -4,
       -2, 37, 93,  93, 37, -2, -4, 29, 88,  99, 44, 0,  -5, 23, 82,  101, 52, 3,  -5, 17, 75, 102, 60, 7},
    },

    // Band_8_2439
    {
      // Scaler inc range
      {2433, 2439},
      // Coefficients 4 taps
      {68, 120, 68,  0,  55, 116, 75,  10, 45, 112, 84,  15, 36, 106, 93,  21,
       28, 100, 100, 28, 21, 93,  106, 36, 15, 84,  112, 45, 10, 75,  116, 55},
      // Coefficients 6 taps
      {11, 66, 102, 66, 11, 0,  7,  60, 102, 75, 17, -5, 3,  52, 101, 82,  23, -5, 0,  44, 99, 88,  29, -4,
       -2, 37, 93,  93, 37, -2, -4, 29, 88,  99, 44, 0,  -5, 23, 82,  101, 52, 3,  -5, 17, 75, 102, 60, 7},
    },

    // Band_8_2444
    {
      // Scaler inc range
      {2439, 2444},
      // Coefficients 4 taps
      {68, 120, 68,  0,  55, 116, 75,  10, 45, 112, 84,  15, 36, 106, 93,  21,
       28, 100, 100, 28, 21, 93,  106, 36, 15, 84,  112, 45, 10, 75,  116, 55},
      // Coefficients 6 taps
      {11, 66, 102, 66, 11, 0,  7,  60, 102, 75, 17, -5, 3,  52, 101, 82,  23, -5, 0,  44, 98, 88,  30, -4,
       -2, 37, 93,  93, 37, -2, -4, 30, 88,  98, 44, 0,  -5, 23, 82,  101, 52, 3,  -5, 17, 75, 102, 60, 7},
    },

    // Band_8_2450
    {
      // Scaler inc range
      {2444, 2450},
      // Coefficients 4 taps
      {68, 120, 68,  0,  55, 116, 75,  10, 45, 112, 84,  15, 36, 106, 93,  21,
       28, 100, 100, 28, 21, 93,  106, 36, 15, 84,  112, 45, 10, 75,  116, 55},
      // Coefficients 6 taps
      {11, 66, 102, 66, 11, 0,  7,  60, 102, 75, 17, -5, 3,  52, 100, 82,  23, -4, 0,  44, 97, 88,  30, -3,
       -2, 37, 93,  93, 37, -2, -3, 30, 88,  97, 44, 0,  -4, 23, 82,  100, 52, 3,  -5, 17, 75, 102, 60, 7},
    },

    // Band_8_2456
    {
      // Scaler inc range
      {2450, 2456},
      // Coefficients 4 taps
      {68, 120, 68,  0,  55, 116, 75,  10, 45, 112, 84,  15, 36, 107, 92,  21,
       28, 100, 100, 28, 21, 92,  107, 36, 15, 84,  112, 45, 10, 75,  116, 55},
      // Coefficients 6 taps
      {12, 66, 100, 66, 12, 0,  7,  60, 102, 75, 17, -5, 4,  52, 99, 82, 23, -4, 0,  44, 97, 88,  30, -3,
       -2, 37, 93,  93, 37, -2, -3, 30, 88,  97, 44, 0,  -4, 23, 82, 99, 52, 4,  -5, 17, 75, 102, 60, 7},
    },

    // Band_8_2461
    {
      // Scaler inc range
      {2456, 2461},
      // Coefficients 4 taps
      {68, 120, 68,  0,  55, 116, 75,  10, 45, 112, 84,  15, 36, 107, 92,  21,
       28, 100, 100, 28, 21, 92,  107, 36, 15, 84,  112, 45, 10, 75,  116, 55},
      // Coefficients 6 taps
      {12, 66, 100, 66, 12, 0,  7,  60, 102, 75, 17, -5, 4,  52, 100, 81,  23, -4, 1,  44, 96, 88,  30, -3,
       -2, 37, 93,  93, 37, -2, -3, 30, 88,  96, 44, 1,  -4, 23, 81,  100, 52, 4,  -5, 17, 75, 102, 60, 7},
    },

    // Band_8_2467
    {
      // Scaler inc range
      {2461, 2467},
      // Coefficients 4 taps
      {68, 120, 68,  0,  56, 115, 75,  10, 46, 111, 84,  15, 36, 107, 92,  21,
       28, 100, 100, 28, 21, 92,  107, 36, 15, 84,  111, 46, 10, 75,  115, 56},
      // Coefficients 6 taps
      {12, 66, 100, 66, 12, 0,  8,  60, 101, 75, 17, -5, 4,  52, 100, 81,  23, -4, 1,  44, 97, 87,  30, -3,
       -2, 37, 93,  93, 37, -2, -3, 30, 87,  97, 44, 1,  -4, 23, 81,  100, 52, 4,  -5, 17, 75, 101, 60, 8},
    },

    // Band_8_2473
    {
      // Scaler inc range
      {2467, 2473},
      // Coefficients 4 taps
      {68, 120, 68,  0,  56, 115, 75,  10, 46, 111, 84,  15, 36, 107, 92,  21,
       28, 100, 100, 28, 21, 92,  107, 36, 15, 84,  111, 46, 10, 75,  115, 56},
      // Coefficients 6 taps
      {12, 66, 100, 66, 12, 0,  8,  60, 102, 74, 17, -5, 4,  52, 100, 81,  23, -4, 1,  44, 97, 87,  30, -3,
       -1, 37, 92,  92, 37, -1, -3, 30, 87,  97, 44, 1,  -4, 23, 81,  100, 52, 4,  -5, 17, 74, 102, 60, 8},
    },

    // Band_8_2479
    {
      // Scaler inc range
      {2473, 2479},
      // Coefficients 4 taps
      {68, 120, 68,  0,  56, 115, 75,  10, 46, 111, 84,  15, 37, 106, 92,  21,
       28, 100, 100, 28, 21, 92,  106, 37, 15, 84,  111, 46, 10, 75,  115, 56},
      // Coefficients 6 taps
      {12, 66, 100, 66, 12, 0,  8,  60, 101, 74, 18, -5, 4,  52, 99, 81, 24, -4, 1,  44, 97, 87,  30, -3,
       -1, 37, 92,  92, 37, -1, -3, 30, 87,  97, 44, 1,  -4, 24, 81, 99, 52, 4,  -5, 18, 74, 101, 60, 8},
    },

    // Band_8_2485
    {
      // Scaler inc range
      {2479, 2485},
      // Coefficients 4 taps
      {68, 120, 68,  0,  56, 115, 75,  10, 46, 111, 84,  15, 37, 106, 92,  21,
       28, 100, 100, 28, 21, 92,  106, 37, 15, 84,  111, 46, 10, 75,  115, 56},
      // Coefficients 6 taps
      {12, 66, 100, 66, 12, 0,  8,  60, 101, 74, 18, -5, 4,  52, 99, 81, 24, -4, 1,  45, 96, 87,  30, -3,
       -1, 37, 92,  92, 37, -1, -3, 30, 87,  96, 45, 1,  -4, 24, 81, 99, 52, 4,  -5, 18, 74, 101, 60, 8},
    },

    // Band_8_2491
    {
      // Scaler inc range
      {2485, 2491},
      // Coefficients 4 taps
      {68, 120, 68, 0,  56, 115, 75,  10, 46, 111, 84,  15, 37, 106, 92,  21,
       29, 99,  99, 29, 21, 92,  106, 37, 15, 84,  111, 46, 10, 75,  115, 56},
      // Coefficients 6 taps
      {12, 66, 100, 66, 12, 0,  8,  60, 101, 74, 18, -5, 4,  52, 99, 81, 24, -4, 1,  45, 96, 87,  30, -3,
       -1, 37, 92,  92, 37, -1, -3, 30, 87,  96, 45, 1,  -4, 24, 81, 99, 52, 4,  -5, 18, 74, 101, 60, 8},
    },

    // Band_8_2497
    {
      // Scaler inc range
      {2491, 2497},
      // Coefficients 4 taps
      {68, 120, 68, 0,  56, 115, 75,  10, 46, 111, 84,  15, 37, 106, 92,  21,
       29, 99,  99, 29, 21, 92,  106, 37, 15, 84,  111, 46, 10, 75,  115, 56},
      // Coefficients 6 taps
      {12, 66, 100, 66, 12, 0,  8,  60, 100, 74, 18, -4, 4,  52, 99, 81, 24, -4, 1,  45, 96, 87,  30, -3,
       -1, 37, 92,  92, 37, -1, -3, 30, 87,  96, 45, 1,  -4, 24, 81, 99, 52, 4,  -4, 18, 74, 100, 60, 8},
    },

    // Band_8_2503
    {
      // Scaler inc range
      {2497, 2503},
      // Coefficients 4 taps
      {68, 120, 68, 0,  56, 114, 75,  11, 46, 111, 84,  15, 37, 105, 92,  22,
       29, 99,  99, 29, 22, 92,  105, 37, 15, 84,  111, 46, 11, 75,  114, 56},
      // Coefficients 6 taps
      {13, 66, 98, 66, 13, 0,  8,  60, 100, 74, 18, -4, 4,  52, 99, 81, 24, -4, 1,  45, 96, 87,  30, -3,
       -1, 37, 92, 92, 37, -1, -3, 30, 87,  96, 45, 1,  -4, 24, 81, 99, 52, 4,  -4, 18, 74, 100, 60, 8},
    },

    // Band_8_2509
    {
      // Scaler inc range
      {2503, 2509},
      // Coefficients 4 taps
      {68, 120, 68, 0,  56, 114, 75,  11, 46, 110, 84,  16, 37, 105, 92,  22,
       29, 99,  99, 29, 22, 92,  105, 37, 16, 84,  110, 46, 11, 75,  114, 56},
      // Coefficients 6 taps
      {13, 66, 98, 66, 13, 0,  8,  60, 100, 74, 18, -4, 5,  52, 99, 80, 24, -4, 2,  45, 95, 86,  31, -3,
       -1, 37, 92, 92, 37, -1, -3, 31, 86,  95, 45, 2,  -4, 24, 80, 99, 52, 5,  -4, 18, 74, 100, 60, 8},
    },

    // Band_8_2515
    {
      // Scaler inc range
      {2509, 2515},
      // Coefficients 4 taps
      {68, 120, 68, 0,  56, 114, 75,  11, 46, 110, 84,  16, 37, 105, 92,  22,
       29, 99,  99, 29, 22, 92,  105, 37, 16, 84,  110, 46, 11, 75,  114, 56},
      // Coefficients 6 taps
      {13, 66, 98, 66, 13, 0,  9,  60, 99, 74, 18, -4, 5,  52, 99, 80, 24, -4, 2,  45, 94, 86, 31, -2,
       -1, 38, 91, 91, 38, -1, -2, 31, 86, 94, 45, 2,  -4, 24, 80, 99, 52, 5,  -4, 18, 74, 99, 60, 9},
    },

    // Band_8_2521
    {
      // Scaler inc range
      {2515, 2521},
      // Coefficients 4 taps
      {68, 120, 68, 0,  56, 114, 75,  11, 46, 110, 84,  16, 37, 105, 92,  22,
       29, 99,  99, 29, 22, 92,  105, 37, 16, 84,  110, 46, 11, 75,  114, 56},
      // Coefficients 6 taps
      {13, 66, 98, 66, 13, 0,  9,  60, 99, 74, 18, -4, 5,  52, 99, 80, 24, -4, 2,  45, 94, 86, 31, -2,
       -1, 38, 91, 91, 38, -1, -2, 31, 86, 94, 45, 2,  -4, 24, 80, 99, 52, 5,  -4, 18, 74, 99, 60, 9},
    },

    // Band_8_2527
    {
      // Scaler inc range
      {2521, 2527},
      // Coefficients 4 taps
      {68, 120, 68, 0,  56, 114, 75,  11, 46, 110, 84,  16, 37, 105, 92,  22,
       29, 99,  99, 29, 22, 92,  105, 37, 16, 84,  110, 46, 11, 75,  114, 56},
      // Coefficients 6 taps
      {13, 66, 98, 66, 13, 0,  9,  60, 98, 74, 19, -4, 5,  52, 98, 80, 24, -3, 2,  45, 94, 86, 31, -2,
       -1, 38, 91, 91, 38, -1, -2, 31, 86, 94, 45, 2,  -3, 24, 80, 98, 52, 5,  -4, 19, 74, 98, 60, 9},
    },

    // Band_8_2533
    {
      // Scaler inc range
      {2527, 2533},
      // Coefficients 4 taps
      {68, 120, 68, 0,  56, 114, 75,  11, 46, 110, 84,  16, 37, 105, 92,  22,
       29, 99,  99, 29, 22, 92,  105, 37, 16, 84,  110, 46, 11, 75,  114, 56},
      // Coefficients 6 taps
      {13, 66, 98, 66, 13, 0, 9,  60, 98, 74, 19, -4, 5,  52, 97, 80, 25, -3, 2,  45, 94, 86, 31, -2,
       0,  38, 90, 90, 38, 0, -2, 31, 86, 94, 45, 2,  -3, 25, 80, 97, 52, 5,  -4, 19, 74, 98, 60, 9},
    },

    // Band_8_2539
    {
      // Scaler inc range
      {2533, 2539},
      // Coefficients 4 taps
      {68, 120, 68, 0,  56, 114, 75,  11, 46, 110, 84,  16, 37, 105, 92,  22,
       29, 99,  99, 29, 22, 92,  105, 37, 16, 84,  110, 46, 11, 75,  114, 56},
      // Coefficients 6 taps
      {13, 66, 98, 66, 13, 0, 9,  60, 98, 74, 19, -4, 5,  52, 97, 80, 25, -3, 2,  45, 94, 86, 31, -2,
       0,  38, 90, 90, 38, 0, -2, 31, 86, 94, 45, 2,  -3, 25, 80, 97, 52, 5,  -4, 19, 74, 98, 60, 9},
    },

    // Band_8_2545
    {
      // Scaler inc range
      {2539, 2545},
      // Coefficients 4 taps
      {68, 120, 68, 0,  56, 114, 75,  11, 46, 110, 84,  16, 37, 105, 92,  22,
       29, 99,  99, 29, 22, 92,  105, 37, 16, 84,  110, 46, 11, 75,  114, 56},
      // Coefficients 6 taps
      {13, 66, 98, 66, 13, 0, 9,  60, 99, 73, 19, -4, 5,  52, 97, 80, 25, -3, 2,  45, 95, 85, 31, -2,
       0,  38, 90, 90, 38, 0, -2, 31, 85, 95, 45, 2,  -3, 25, 80, 97, 52, 5,  -4, 19, 73, 99, 60, 9},
    },

    // Band_8_2551
    {
      // Scaler inc range
      {2545, 2551},
      // Coefficients 4 taps
      {68, 120, 68, 0,  56, 114, 75,  11, 46, 110, 84,  16, 37, 105, 92,  22,
       29, 99,  99, 29, 22, 92,  105, 37, 16, 84,  110, 46, 11, 75,  114, 56},
      // Coefficients 6 taps
      {14, 66, 96, 66, 14, 0, 9,  60, 99, 73, 19, -4, 5,  52, 97, 80, 25, -3, 2,  45, 95, 85, 31, -2,
       0,  38, 90, 90, 38, 0, -2, 31, 85, 95, 45, 2,  -3, 25, 80, 97, 52, 5,  -4, 19, 73, 99, 60, 9},
    },

    // Band_8_2558
    {
      // Scaler inc range
      {2551, 2558},
      // Coefficients 4 taps
      {69, 118, 69, 0,  56, 114, 75,  11, 46, 110, 84,  16, 38, 105, 91,  22,
       29, 99,  99, 29, 22, 91,  105, 38, 16, 84,  110, 46, 11, 75,  114, 56},
      // Coefficients 6 taps
      {14, 66, 96, 66, 14, 0, 9,  59, 100, 73, 19, -4, 6,  52, 96, 80, 25, -3, 2,  45, 95, 85,  31, -2,
       0,  38, 90, 90, 38, 0, -2, 31, 85,  95, 45, 2,  -3, 25, 80, 96, 52, 6,  -4, 19, 73, 100, 59, 9},
    },

    // Band_8_2564
    {
      // Scaler inc range
      {2558, 2564},
      // Coefficients 4 taps
      {69, 118, 69, 0,  56, 114, 75,  11, 47, 109, 84,  16, 38, 105, 91,  22,
       30, 98,  98, 30, 22, 91,  105, 38, 16, 84,  109, 47, 11, 75,  114, 56},
      // Coefficients 6 taps
      {14, 65, 98, 65, 14, 0, 10, 59, 99, 73, 19, -4, 6,  52, 97, 79, 25, -3, 3,  45, 94, 85, 31, -2,
       0,  38, 90, 90, 38, 0, -2, 31, 85, 94, 45, 3,  -3, 25, 79, 97, 52, 6,  -4, 19, 73, 99, 59, 10},
    },

    // Band_8_2570
    {
      // Scaler inc range
      {2564, 2570},
      // Coefficients 4 taps
      {69, 118, 69, 0,  56, 114, 75,  11, 47, 109, 84,  16, 38, 104, 91,  23,
       30, 98,  98, 30, 23, 91,  104, 38, 16, 84,  109, 47, 11, 75,  114, 56},
      // Coefficients 6 taps
      {14, 65, 98, 65, 14, 0, 10, 59, 99, 73, 19, -4, 6,  52, 97, 79, 25, -3, 3,  45, 94, 85, 31, -2,
       0,  38, 90, 90, 38, 0, -2, 31, 85, 94, 45, 3,  -3, 25, 79, 97, 52, 6,  -4, 19, 73, 99, 59, 10},
    },

    // Band_8_2576
    {
      // Scaler inc range
      {2570, 2576},
      // Coefficients 4 taps
      {69, 118, 69, 0,  56, 114, 75,  11, 47, 108, 84,  17, 38, 104, 91,  23,
       30, 98,  98, 30, 23, 91,  104, 38, 17, 84,  108, 47, 11, 75,  114, 56},
      // Coefficients 6 taps
      {14, 65, 98, 65, 14, 0, 10, 59, 99, 73, 19, -4, 6,  52, 97, 79, 25, -3, 3,  45, 93, 85, 32, -2,
       0,  38, 90, 90, 38, 0, -2, 32, 85, 93, 45, 3,  -3, 25, 79, 97, 52, 6,  -4, 19, 73, 99, 59, 10},
    },

    // Band_8_2583
    {
      // Scaler inc range
      {2576, 2583},
      // Coefficients 4 taps
      {69, 118, 69, 0,  56, 113, 75,  12, 47, 108, 84,  17, 38, 104, 91,  23,
       30, 98,  98, 30, 23, 91,  104, 38, 17, 84,  108, 47, 12, 75,  113, 56},
      // Coefficients 6 taps
      {14, 65, 98, 65, 14, 0, 10, 59, 98, 73, 20, -4, 6,  52, 97, 79, 25, -3, 3,  45, 92, 85, 32, -1,
       0,  38, 90, 90, 38, 0, -1, 32, 85, 92, 45, 3,  -3, 25, 79, 97, 52, 6,  -4, 20, 73, 98, 59, 10},
    },

    // Band_8_2589
    {
      // Scaler inc range
      {2583, 2589},
      // Coefficients 4 taps
      {69, 118, 69, 0,  56, 113, 75,  12, 47, 109, 83,  17, 38, 104, 91,  23,
       30, 98,  98, 30, 23, 91,  104, 38, 17, 83,  109, 47, 12, 75,  113, 56},
      // Coefficients 6 taps
      {14, 65, 98, 65, 14, 0, 10, 59, 97, 73, 20, -3, 6,  52, 97, 79, 25, -3, 3,  45, 93, 84, 32, -1,
       1,  38, 89, 89, 38, 1, -1, 32, 84, 93, 45, 3,  -3, 25, 79, 97, 52, 6,  -3, 20, 73, 97, 59, 10},
    },

    // Band_8_2595
    {
      // Scaler inc range
      {2589, 2595},
      // Coefficients 4 taps
      {69, 118, 69, 0,  56, 113, 75,  12, 47, 109, 83,  17, 38, 104, 91,  23,
       30, 98,  98, 30, 23, 91,  104, 38, 17, 83,  109, 47, 12, 75,  113, 56},
      // Coefficients 6 taps
      {14, 65, 98, 65, 14, 0, 10, 59, 97, 73, 20, -3, 6,  52, 96, 79, 26, -3, 3,  45, 93, 84, 32, -1,
       1,  38, 89, 89, 38, 1, -1, 32, 84, 93, 45, 3,  -3, 26, 79, 96, 52, 6,  -3, 20, 73, 97, 59, 10},
    },

    // Band_8_2602
    {
      // Scaler inc range
      {2595, 2602},
      // Coefficients 4 taps
      {69, 118, 69, 0,  57, 112, 75,  12, 47, 109, 83,  17, 38, 104, 91,  23,
       30, 98,  98, 30, 23, 91,  104, 38, 17, 83,  109, 47, 12, 75,  112, 57},
      // Coefficients 6 taps
      {15, 65, 96, 65, 15, 0, 10, 59, 97, 73, 20, -3, 6,  52, 96, 79, 26, -3, 3,  45, 93, 84, 32, -1,
       1,  38, 89, 89, 38, 1, -1, 32, 84, 93, 45, 3,  -3, 26, 79, 96, 52, 6,  -3, 20, 73, 97, 59, 10},
    },

    // Band_8_2608
    {
      // Scaler inc range
      {2602, 2608},
      // Coefficients 4 taps
      {69, 118, 69, 0,  57, 112, 75,  12, 47, 109, 83,  17, 38, 104, 91,  23,
       30, 98,  98, 30, 23, 91,  104, 38, 17, 83,  109, 47, 12, 75,  112, 57},
      // Coefficients 6 taps
      {15, 65, 96, 65, 15, 0, 10, 59, 97, 73, 20, -3, 7,  52, 94, 79, 26, -2, 3,  45, 93, 84, 32, -1,
       1,  39, 88, 88, 39, 1, -1, 32, 84, 93, 45, 3,  -2, 26, 79, 94, 52, 7,  -3, 20, 73, 97, 59, 10},
    },

    // Band_8_2615
    {
      // Scaler inc range
      {2608, 2615},
      // Coefficients 4 taps
      {69, 118, 69, 0,  57, 112, 75,  12, 47, 109, 83,  17, 38, 104, 91,  23,
       30, 98,  98, 30, 23, 91,  104, 38, 17, 83,  109, 47, 12, 75,  112, 57},
      // Coefficients 6 taps
      {15, 65, 96, 65, 15, 0, 11, 59, 96, 73, 20, -3, 7,  52, 94, 79, 26, -2, 3,  45, 93, 84, 32, -1,
       1,  39, 88, 88, 39, 1, -1, 32, 84, 93, 45, 3,  -2, 26, 79, 94, 52, 7,  -3, 20, 73, 96, 59, 11},
    },

    // Band_8_2621
    {
      // Scaler inc range
      {2615, 2621},
      // Coefficients 4 taps
      {69, 118, 69, 0,  57, 112, 75,  12, 47, 109, 83,  17, 38, 104, 91,  23,
       30, 98,  98, 30, 23, 91,  104, 38, 17, 83,  109, 47, 12, 75,  112, 57},
      // Coefficients 6 taps
      {15, 65, 96, 65, 15, 0, 11, 59, 96, 73, 20, -3, 7,  52, 95, 78, 26, -2, 4,  45, 92, 84, 32, -1,
       1,  39, 88, 88, 39, 1, -1, 32, 84, 92, 45, 4,  -2, 26, 78, 95, 52, 7,  -3, 20, 73, 96, 59, 11},
    },

    // Band_8_2628
    {
      // Scaler inc range
      {2621, 2628},
      // Coefficients 4 taps
      {69, 118, 69, 0,  57, 112, 75,  12, 47, 109, 83,  17, 38, 104, 91,  23,
       30, 98,  98, 30, 23, 91,  104, 38, 17, 83,  109, 47, 12, 75,  112, 57},
      // Coefficients 6 taps
      {15, 65, 96, 65, 15, 0, 11, 59, 97, 72, 20, -3, 7,  52, 95, 78, 26, -2, 4,  45, 92, 84, 32, -1,
       1,  39, 88, 88, 39, 1, -1, 32, 84, 92, 45, 4,  -2, 26, 78, 95, 52, 7,  -3, 20, 72, 97, 59, 11},
    },

    // Band_8_2635
    {
      // Scaler inc range
      {2628, 2635},
      // Coefficients 4 taps
      {69, 118, 69, 0,  57, 112, 75,  12, 47, 109, 83,  17, 38, 104, 91,  23,
       30, 98,  98, 30, 23, 91,  104, 38, 17, 83,  109, 47, 12, 75,  112, 57},
      // Coefficients 6 taps
      {15, 65, 96, 65, 15, 0, 11, 59, 97, 72, 20, -3, 7,  52, 95, 78, 26, -2, 4,  46, 92, 83, 32, -1,
       1,  39, 88, 88, 39, 1, -1, 32, 83, 92, 46, 4,  -2, 26, 78, 95, 52, 7,  -3, 20, 72, 97, 59, 11},
    },

    // Band_8_2641
    {
      // Scaler inc range
      {2635, 2641},
      // Coefficients 4 taps
      {69, 118, 69, 0,  57, 112, 75,  12, 47, 109, 83,  17, 38, 104, 91,  23,
       30, 98,  98, 30, 23, 91,  104, 38, 17, 83,  109, 47, 12, 75,  112, 57},
      // Coefficients 6 taps
      {15, 65, 96, 65, 15, 0, 11, 59, 96, 72, 21, -3, 7,  52, 95, 78, 26, -2, 4,  46, 92, 83, 32, -1,
       1,  39, 88, 88, 39, 1, -1, 32, 83, 92, 46, 4,  -2, 26, 78, 95, 52, 7,  -3, 21, 72, 96, 59, 11},
    },

    // Band_8_2648
    {
      // Scaler inc range
      {2641, 2648},
      // Coefficients 4 taps
      {69, 118, 69, 0,  57, 112, 75,  12, 47, 109, 83,  17, 39, 102, 91,  24,
       31, 97,  97, 31, 24, 91,  102, 39, 17, 83,  109, 47, 12, 75,  112, 57},
      // Coefficients 6 taps
      {15, 65, 96, 65, 15, 0, 11, 59, 96, 72, 21, -3, 7,  52, 95, 78, 26, -2, 4,  46, 92, 83, 32, -1,
       1,  39, 88, 88, 39, 1, -1, 32, 83, 92, 46, 4,  -2, 26, 78, 95, 52, 7,  -3, 21, 72, 96, 59, 11},
    },

    // Band_8_2655
    {
      // Scaler inc range
      {2648, 2655},
      // Coefficients 4 taps
      {69, 118, 69, 0,  57, 112, 75,  12, 47, 108, 83,  18, 39, 102, 91,  24,
       31, 97,  97, 31, 24, 91,  102, 39, 18, 83,  108, 47, 12, 75,  112, 57},
      // Coefficients 6 taps
      {16, 65, 94, 65, 16, 0, 11, 59, 96, 72, 21, -3, 7,  52, 95, 78, 26, -2, 4,  46, 90, 83, 33, 0,
       2,  39, 87, 87, 39, 2, 0,  33, 83, 90, 46, 4,  -2, 26, 78, 95, 52, 7,  -3, 21, 72, 96, 59, 11},
    },

    // Band_8_2661
    {
      // Scaler inc range
      {2655, 2661},
      // Coefficients 4 taps
      {69, 118, 69, 0,  57, 112, 75,  12, 47, 108, 83,  18, 39, 102, 91,  24,
       31, 97,  97, 31, 24, 91,  102, 39, 18, 83,  108, 47, 12, 75,  112, 57},
      // Coefficients 6 taps
      {16, 65, 94, 65, 16, 0, 11, 59, 96, 72, 21, -3, 8,  52, 93, 78, 27, -2, 4,  46, 90, 83, 33, 0,
       2,  39, 87, 87, 39, 2, 0,  33, 83, 90, 46, 4,  -2, 27, 78, 93, 52, 8,  -3, 21, 72, 96, 59, 11},
    },

    // Band_8_2668
    {
      // Scaler inc range
      {2661, 2668},
      // Coefficients 4 taps
      {69, 118, 69, 0,  57, 111, 75,  13, 47, 108, 83,  18, 39, 102, 91,  24,
       31, 97,  97, 31, 24, 91,  102, 39, 18, 83,  108, 47, 13, 75,  111, 57},
      // Coefficients 6 taps
      {16, 65, 94, 65, 16, 0, 11, 59, 96, 72, 21, -3, 8,  52, 93, 78, 27, -2, 4,  46, 90, 83, 33, 0,
       2,  39, 87, 87, 39, 2, 0,  33, 83, 90, 46, 4,  -2, 27, 78, 93, 52, 8,  -3, 21, 72, 96, 59, 11},
    },

    // Band_8_2675
    {
      // Scaler inc range
      {2668, 2675},
      // Coefficients 4 taps
      {69, 118, 69, 0,  57, 111, 75,  13, 48, 107, 83,  18, 39, 103, 90,  24,
       31, 97,  97, 31, 24, 90,  103, 39, 18, 83,  107, 48, 13, 75,  111, 57},
      // Coefficients 6 taps
      {16, 65, 94, 65, 16, 0, 12, 59, 95, 72, 21, -3, 8,  52, 93, 78, 27, -2, 4,  46, 90, 83, 33, 0,
       2,  39, 87, 87, 39, 2, 0,  33, 83, 90, 46, 4,  -2, 27, 78, 93, 52, 8,  -3, 21, 72, 95, 59, 12},
    },

    // Band_8_2682
    {
      // Scaler inc range
      {2675, 2682},
      // Coefficients 4 taps
      {69, 118, 69, 0,  57, 111, 75,  13, 48, 107, 83,  18, 39, 103, 90,  24,
       31, 97,  97, 31, 24, 90,  103, 39, 18, 83,  107, 48, 13, 75,  111, 57},
      // Coefficients 6 taps
      {16, 65, 94, 65, 16, 0, 12, 59, 95, 72, 21, -3, 8,  52, 94, 77, 27, -2, 5,  46, 89, 83, 33, 0,
       2,  39, 87, 87, 39, 2, 0,  33, 83, 89, 46, 5,  -2, 27, 77, 94, 52, 8,  -3, 21, 72, 95, 59, 12},
    },

    // Band_8_2689
    {
      // Scaler inc range
      {2682, 2689},
      // Coefficients 4 taps
      {69, 118, 69, 0,  57, 111, 75,  13, 48, 107, 83,  18, 39, 103, 90,  24,
       31, 97,  97, 31, 24, 90,  103, 39, 18, 83,  107, 48, 13, 75,  111, 57},
      // Coefficients 6 taps
      {16, 65, 94, 65, 16, 0, 12, 59, 94, 72, 21, -2, 8,  52, 93, 77, 27, -1, 5,  46, 90, 82, 33, 0,
       2,  39, 87, 87, 39, 2, 0,  33, 82, 90, 46, 5,  -1, 27, 77, 93, 52, 8,  -2, 21, 72, 94, 59, 12},
    },

    // Band_8_2696
    {
      // Scaler inc range
      {2689, 2696},
      // Coefficients 4 taps
      {69, 118, 69, 0,  57, 111, 75,  13, 48, 107, 83,  18, 39, 103, 90,  24,
       31, 97,  97, 31, 24, 90,  103, 39, 18, 83,  107, 48, 13, 75,  111, 57},
      // Coefficients 6 taps
      {16, 65, 94, 65, 16, 0, 12, 59, 94, 72, 21, -2, 8,  52, 93, 77, 27, -1, 5,  46, 90, 82, 33, 0,
       2,  39, 87, 87, 39, 2, 0,  33, 82, 90, 46, 5,  -1, 27, 77, 93, 52, 8,  -2, 21, 72, 94, 59, 12},
    },

    // Band_8_2703
    {
      // Scaler inc range
      {2696, 2703},
      // Coefficients 4 taps
      {69, 118, 69, 0,  57, 111, 75,  13, 48, 107, 83,  18, 39, 103, 90,  24,
       31, 97,  97, 31, 24, 90,  103, 39, 18, 83,  107, 48, 13, 75,  111, 57},
      // Coefficients 6 taps
      {16, 65, 94, 65, 16, 0, 12, 59, 93, 72, 22, -2, 8,  52, 93, 77, 27, -1, 5,  46, 90, 82, 33, 0,
       2,  39, 87, 87, 39, 2, 0,  33, 82, 90, 46, 5,  -1, 27, 77, 93, 52, 8,  -2, 22, 72, 93, 59, 12},
    },

    // Band_8_2709
    {
      // Scaler inc range
      {2703, 2709},
      // Coefficients 4 taps
      {70, 116, 70, 0,  57, 111, 75,  13, 48, 107, 83,  18, 39, 103, 90,  24,
       31, 97,  97, 31, 24, 90,  103, 39, 18, 83,  107, 48, 13, 75,  111, 57},
      // Coefficients 6 taps
      {17, 65, 92, 65, 17, 0, 12, 59, 94, 71, 22, -2, 8,  52, 93, 77, 27, -1, 5,  46, 90, 82, 33, 0,
       2,  39, 87, 87, 39, 2, 0,  33, 82, 90, 46, 5,  -1, 27, 77, 93, 52, 8,  -2, 22, 71, 94, 59, 12},
    },

    // Band_8_2717
    {
      // Scaler inc range
      {2709, 2717},
      // Coefficients 4 taps
      {70, 116, 70, 0,  57, 111, 75,  13, 48, 107, 83,  18, 39, 103, 90,  24,
       31, 97,  97, 31, 24, 90,  103, 39, 18, 83,  107, 48, 13, 75,  111, 57},
      // Coefficients 6 taps
      {17, 65, 92, 65, 17, 0, 12, 59, 94, 71, 22, -2, 8,  53, 92, 77, 27, -1, 5,  46, 90, 82, 33, 0,
       3,  39, 86, 86, 39, 3, 0,  33, 82, 90, 46, 5,  -1, 27, 77, 92, 53, 8,  -2, 22, 71, 94, 59, 12},
    },

    // Band_8_2724
    {
      // Scaler inc range
      {2717, 2724},
      // Coefficients 4 taps
      {70, 116, 70, 0,  57, 111, 75,  13, 48, 107, 83,  18, 39, 103, 90,  24,
       31, 97,  97, 31, 24, 90,  103, 39, 18, 83,  107, 48, 13, 75,  111, 57},
      // Coefficients 6 taps
      {17, 65, 92, 65, 17, 0, 12, 59, 94, 71, 22, -2, 9,  53, 91, 77, 27, -1, 5,  46, 89, 82, 33, 1,
       3,  40, 85, 85, 40, 3, 1,  33, 82, 89, 46, 5,  -1, 27, 77, 91, 53, 9,  -2, 22, 71, 94, 59, 12},
    },

    // Band_8_2731
    {
      // Scaler inc range
      {2724, 2731},
      // Coefficients 4 taps
      {70, 116, 70, 0,  57, 111, 75,  13, 48, 107, 83,  18, 39, 102, 90,  25,
       32, 96,  96, 32, 25, 90,  102, 39, 18, 83,  107, 48, 13, 75,  111, 57},
      // Coefficients 6 taps
      {17, 65, 92, 65, 17, 0, 13, 59, 93, 71, 22, -2, 9,  53, 91, 77, 27, -1, 5,  46, 89, 82, 33, 1,
       3,  40, 85, 85, 40, 3, 1,  33, 82, 89, 46, 5,  -1, 27, 77, 91, 53, 9,  -2, 22, 71, 93, 59, 13},
    },

    // Band_8_2738
    {
      // Scaler inc range
      {2731, 2738},
      // Coefficients 4 taps
      {70, 116, 70, 0,  57, 111, 75,  13, 48, 106, 83,  19, 39, 102, 90,  25,
       32, 96,  96, 32, 25, 90,  102, 39, 19, 83,  106, 48, 13, 75,  111, 57},
      // Coefficients 6 taps
      {17, 65, 92, 65, 17, 0, 13, 59, 93, 71, 22, -2, 9,  53, 90, 77, 28, -1, 6,  46, 89, 81, 33, 1,
       3,  40, 85, 85, 40, 3, 1,  33, 81, 89, 46, 6,  -1, 28, 77, 90, 53, 9,  -2, 22, 71, 93, 59, 13},
    },

    // Band_8_2745
    {
      // Scaler inc range
      {2738, 2745},
      // Coefficients 4 taps
      {70, 116, 70, 0,  57, 111, 75,  13, 48, 106, 83,  19, 40, 101, 90,  25,
       32, 96,  96, 32, 25, 90,  101, 40, 19, 83,  106, 48, 13, 75,  111, 57},
      // Coefficients 6 taps
      {17, 65, 92, 65, 17, 0, 13, 59, 93, 71, 22, -2, 9,  53, 91, 76, 28, -1, 6,  46, 88, 81, 34, 1,
       3,  40, 85, 85, 40, 3, 1,  34, 81, 88, 46, 6,  -1, 28, 76, 91, 53, 9,  -2, 22, 71, 93, 59, 13},
    },

    // Band_8_2752
    {
      // Scaler inc range
      {2745, 2752},
      // Coefficients 4 taps
      {70, 116, 70, 0,  57, 110, 75,  14, 48, 106, 83,  19, 40, 101, 90,  25,
       32, 96,  96, 32, 25, 90,  101, 40, 19, 83,  106, 48, 14, 75,  110, 57},
      // Coefficients 6 taps
      {17, 65, 92, 65, 17, 0, 13, 59, 93, 71, 22, -2, 9,  53, 91, 76, 28, -1, 6,  46, 88, 81, 34, 1,
       3,  40, 85, 85, 40, 3, 1,  34, 81, 88, 46, 6,  -1, 28, 76, 91, 53, 9,  -2, 22, 71, 93, 59, 13},
    },

    // Band_8_2759
    {
      // Scaler inc range
      {2752, 2759},
      // Coefficients 4 taps
      {70, 116, 70, 0,  57, 110, 75,  14, 48, 106, 83,  19, 40, 101, 90,  25,
       32, 96,  96, 32, 25, 90,  101, 40, 19, 83,  106, 48, 14, 75,  110, 57},
      // Coefficients 6 taps
      {17, 65, 92, 65, 17, 0, 13, 59, 93, 71, 22, -2, 9,  53, 91, 76, 28, -1, 6,  46, 88, 81, 34, 1,
       3,  40, 85, 85, 40, 3, 1,  34, 81, 88, 46, 6,  -1, 28, 76, 91, 53, 9,  -2, 22, 71, 93, 59, 13},
    },

    // Band_8_2767
    {
      // Scaler inc range
      {2759, 2767},
      // Coefficients 4 taps
      {70, 116, 70, 0,  58, 109, 75,  14, 48, 106, 83,  19, 40, 101, 90,  25,
       32, 96,  96, 32, 25, 90,  101, 40, 19, 83,  106, 48, 14, 75,  109, 58},
      // Coefficients 6 taps
      {17, 65, 92, 65, 17, 0, 13, 59, 92, 71, 23, -2, 9,  53, 91, 76, 28, -1, 6,  46, 88, 81, 34, 1,
       3,  40, 85, 85, 40, 3, 1,  34, 81, 88, 46, 6,  -1, 28, 76, 91, 53, 9,  -2, 23, 71, 92, 59, 13},
    },

    // Band_8_2774
    {
      // Scaler inc range
      {2767, 2774},
      // Coefficients 4 taps
      {70, 116, 70, 0,  58, 109, 75,  14, 48, 106, 83,  19, 40, 101, 90,  25,
       32, 96,  96, 32, 25, 90,  101, 40, 19, 83,  106, 48, 14, 75,  109, 58},
      // Coefficients 6 taps
      {18, 65, 90, 65, 18, 0, 13, 59, 92, 71, 23, -2, 9, 53, 90, 76, 28, 0, 6,  46, 88, 81, 34, 1,
       3,  40, 85, 85, 40, 3, 1,  34, 81, 88, 46, 6,  0, 28, 76, 90, 53, 9, -2, 23, 71, 92, 59, 13},
    },

    // Band_8_2781
    {
      // Scaler inc range
      {2774, 2781},
      // Coefficients 4 taps
      {70, 116, 70, 0,  58, 109, 75,  14, 48, 106, 83,  19, 40, 101, 90,  25,
       32, 96,  96, 32, 25, 90,  101, 40, 19, 83,  106, 48, 14, 75,  109, 58},
      // Coefficients 6 taps
      {18, 64, 92, 64, 18, 0, 13, 59, 92, 71, 23, -2, 9, 53, 90, 76, 28, 0, 6,  46, 88, 81, 34, 1,
       3,  40, 85, 85, 40, 3, 1,  34, 81, 88, 46, 6,  0, 28, 76, 90, 53, 9, -2, 23, 71, 92, 59, 13},
    },

    // Band_8_2789
    {
      // Scaler inc range
      {2781, 2789},
      // Coefficients 4 taps
      {70, 116, 70, 0,  58, 109, 75,  14, 48, 106, 83,  19, 40, 101, 90,  25,
       32, 96,  96, 32, 25, 90,  101, 40, 19, 83,  106, 48, 14, 75,  109, 58},
      // Coefficients 6 taps
      {18, 64, 92, 64, 18, 0, 13, 59, 91, 71, 23, -1, 10, 53, 89, 76, 28, 0,  6,  46, 89, 80, 34, 1,
       4,  40, 84, 84, 40, 4, 1,  34, 80, 89, 46, 6,  0,  28, 76, 89, 53, 10, -1, 23, 71, 91, 59, 13},
    },

    // Band_8_2796
    {
      // Scaler inc range
      {2789, 2796},
      // Coefficients 4 taps
      {70, 116, 70, 0,  58, 109, 75,  14, 48, 106, 83,  19, 40, 101, 90,  25,
       32, 96,  96, 32, 25, 90,  101, 40, 19, 83,  106, 48, 14, 75,  109, 58},
      // Coefficients 6 taps
      {18, 64, 92, 64, 18, 0, 14, 59, 90, 71, 23, -1, 10, 53, 89, 76, 28, 0,  6,  46, 89, 80, 34, 1,
       4,  40, 84, 84, 40, 4, 1,  34, 80, 89, 46, 6,  0,  28, 76, 89, 53, 10, -1, 23, 71, 90, 59, 14},
    },

    // Band_8_2804
    {
      // Scaler inc range
      {2796, 2804},
      // Coefficients 4 taps
      {70, 116, 70, 0,  58, 109, 75,  14, 49, 105, 83,  19, 40, 102, 89,  25,
       32, 96,  96, 32, 25, 89,  102, 40, 19, 83,  105, 49, 14, 75,  109, 58},
      // Coefficients 6 taps
      {18, 64, 92, 64, 18, 0, 14, 59, 91, 70, 23, -1, 10, 53, 89, 76, 28, 0,  7,  46, 87, 80, 34, 2,
       4,  40, 84, 84, 40, 4, 2,  34, 80, 87, 46, 7,  0,  28, 76, 89, 53, 10, -1, 23, 70, 91, 59, 14},
    },

    // Band_8_2811
    {
      // Scaler inc range
      {2804, 2811},
      // Coefficients 4 taps
      {70, 116, 70, 0,  58, 109, 75,  14, 49, 105, 83,  19, 40, 102, 89,  25,
       32, 96,  96, 32, 25, 89,  102, 40, 19, 83,  105, 49, 14, 75,  109, 58},
      // Coefficients 6 taps
      {18, 64, 92, 64, 18, 0, 14, 59, 91, 70, 23, -1, 10, 53, 89, 76, 28, 0,  7,  46, 87, 80, 34, 2,
       4,  40, 84, 84, 40, 4, 2,  34, 80, 87, 46, 7,  0,  28, 76, 89, 53, 10, -1, 23, 70, 91, 59, 14},
    },

    // Band_8_2819
    {
      // Scaler inc range
      {2811, 2819},
      // Coefficients 4 taps
      {70, 116, 70, 0,  58, 109, 75,  14, 49, 105, 83,  19, 40, 102, 89,  25,
       32, 96,  96, 32, 25, 89,  102, 40, 19, 83,  105, 49, 14, 75,  109, 58},
      // Coefficients 6 taps
      {18, 64, 92, 64, 18, 0, 14, 59, 91, 70, 23, -1, 10, 53, 89, 75, 29, 0,  7,  46, 87, 80, 34, 2,
       4,  40, 84, 84, 40, 4, 2,  34, 80, 87, 46, 7,  0,  29, 75, 89, 53, 10, -1, 23, 70, 91, 59, 14},
    },

    // Band_8_2826
    {
      // Scaler inc range
      {2819, 2826},
      // Coefficients 4 taps
      {70, 116, 70, 0,  58, 109, 75,  14, 49, 106, 82,  19, 40, 101, 89,  26,
       32, 96,  96, 32, 26, 89,  101, 40, 19, 82,  106, 49, 14, 75,  109, 58},
      // Coefficients 6 taps
      {18, 64, 92, 64, 18, 0, 14, 59, 91, 70, 23, -1, 10, 53, 89, 75, 29, 0,  7,  46, 87, 80, 34, 2,
       4,  40, 84, 84, 40, 4, 2,  34, 80, 87, 46, 7,  0,  29, 75, 89, 53, 10, -1, 23, 70, 91, 59, 14},
    },

    // Band_8_2834
    {
      // Scaler inc range
      {2826, 2834},
      // Coefficients 4 taps
      {70, 116, 70, 0,  58, 109, 75,  14, 49, 105, 82,  20, 40, 101, 89,  26,
       33, 95,  95, 33, 26, 89,  101, 40, 20, 82,  105, 49, 14, 75,  109, 58},
      // Coefficients 6 taps
      {18, 64, 92, 64, 18, 0, 14, 59, 91, 70, 23, -1, 10, 53, 89, 75, 29, 0,  7,  46, 87, 80, 34, 2,
       4,  40, 84, 84, 40, 4, 2,  34, 80, 87, 46, 7,  0,  29, 75, 89, 53, 10, -1, 23, 70, 91, 59, 14},
    },

    // Band_8_2842
    {
      // Scaler inc range
      {2834, 2842},
      // Coefficients 4 taps
      {70, 116, 70, 0,  58, 109, 75,  14, 49, 105, 82,  20, 40, 101, 89,  26,
       33, 95,  95, 33, 26, 89,  101, 40, 20, 82,  105, 49, 14, 75,  109, 58},
      // Coefficients 6 taps
      {19, 64, 90, 64, 19, 0, 14, 59, 90, 70, 24, -1, 10, 53, 89, 75, 29, 0,  7,  46, 87, 80, 34, 2,
       4,  40, 84, 84, 40, 4, 2,  34, 80, 87, 46, 7,  0,  29, 75, 89, 53, 10, -1, 24, 70, 90, 59, 14},
    },

    // Band_8_2849
    {
      // Scaler inc range
      {2842, 2849},
      // Coefficients 4 taps
      {70, 116, 70, 0,  58, 108, 75,  15, 49, 105, 82,  20, 40, 101, 89,  26,
       33, 95,  95, 33, 26, 89,  101, 40, 20, 82,  105, 49, 15, 75,  108, 58},
      // Coefficients 6 taps
      {19, 64, 90, 64, 19, 0, 14, 59, 90, 70, 24, -1, 10, 53, 89, 75, 29, 0,  7,  46, 88, 79, 34, 2,
       4,  40, 84, 84, 40, 4, 2,  34, 79, 88, 46, 7,  0,  29, 75, 89, 53, 10, -1, 24, 70, 90, 59, 14},
    },

    // Band_8_2857
    {
      // Scaler inc range
      {2849, 2857},
      // Coefficients 4 taps
      {70, 116, 70, 0,  58, 108, 75,  15, 49, 105, 82,  20, 40, 101, 89,  26,
       33, 95,  95, 33, 26, 89,  101, 40, 20, 82,  105, 49, 15, 75,  108, 58},
      // Coefficients 6 taps
      {19, 64, 90, 64, 19, 0, 14, 59, 90, 70, 24, -1, 11, 53, 88, 75, 29, 0,  7,  46, 87, 79, 35, 2,
       4,  40, 84, 84, 40, 4, 2,  35, 79, 87, 46, 7,  0,  29, 75, 88, 53, 11, -1, 24, 70, 90, 59, 14},
    },

    // Band_8_2865
    {
      // Scaler inc range
      {2857, 2865},
      // Coefficients 4 taps
      {70, 116, 70, 0,  58, 108, 75,  15, 49, 105, 82,  20, 41, 100, 89,  26,
       33, 95,  95, 33, 26, 89,  100, 41, 20, 82,  105, 49, 15, 75,  108, 58},
      // Coefficients 6 taps
      {19, 64, 90, 64, 19, 0, 15, 59, 89, 70, 24, -1, 11, 53, 87, 75, 29, 1,  7,  46, 87, 79, 35, 2,
       5,  40, 83, 83, 40, 5, 2,  35, 79, 87, 46, 7,  1,  29, 75, 87, 53, 11, -1, 24, 70, 89, 59, 15},
    },

    // Band_8_2873
    {
      // Scaler inc range
      {2865, 2873},
      // Coefficients 4 taps
      {70, 116, 70, 0,  58, 108, 75,  15, 49, 105, 82,  20, 41, 100, 89,  26,
       33, 95,  95, 33, 26, 89,  100, 41, 20, 82,  105, 49, 15, 75,  108, 58},
      // Coefficients 6 taps
      {19, 64, 90, 64, 19, 0, 15, 59, 89, 70, 24, -1, 11, 53, 87, 75, 29, 1,  7,  46, 87, 79, 35, 2,
       5,  41, 82, 82, 41, 5, 2,  35, 79, 87, 46, 7,  1,  29, 75, 87, 53, 11, -1, 24, 70, 89, 59, 15},
    },

    // Band_8_2881
    {
      // Scaler inc range
      {2873, 2881},
      // Coefficients 4 taps
      {70, 116, 70, 0,  58, 108, 75,  15, 49, 105, 82,  20, 41, 100, 89,  26,
       33, 95,  95, 33, 26, 89,  100, 41, 20, 82,  105, 49, 15, 75,  108, 58},
      // Coefficients 6 taps
      {19, 64, 90, 64, 19, 0, 15, 59, 89, 70, 24, -1, 11, 53, 87, 75, 29, 1,  8,  47, 84, 79, 35, 3,
       5,  41, 82, 82, 41, 5, 3,  35, 79, 84, 47, 8,  1,  29, 75, 87, 53, 11, -1, 24, 70, 89, 59, 15},
    },

    // Band_8_2889
    {
      // Scaler inc range
      {2881, 2889},
      // Coefficients 4 taps
      {71, 114, 71, 0,  58, 108, 75,  15, 49, 105, 82,  20, 41, 100, 89,  26,
       33, 95,  95, 33, 26, 89,  100, 41, 20, 82,  105, 49, 15, 75,  108, 58},
      // Coefficients 6 taps
      {19, 64, 90, 64, 19, 0, 15, 59, 88, 70, 24, 0, 11, 53, 88, 74, 29, 1,  8, 47, 84, 79, 35, 3,
       5,  41, 82, 82, 41, 5, 3,  35, 79, 84, 47, 8, 1,  29, 74, 88, 53, 11, 0, 24, 70, 88, 59, 15},
    },

    // Band_8_2897
    {
      // Scaler inc range
      {2889, 2897},
      // Coefficients 4 taps
      {71, 114, 71, 0,  58, 108, 75,  15, 49, 105, 82,  20, 41, 100, 89,  26,
       33, 95,  95, 33, 26, 89,  100, 41, 20, 82,  105, 49, 15, 75,  108, 58},
      // Coefficients 6 taps
      {19, 64, 90, 64, 19, 0, 15, 59, 88, 70, 24, 0, 11, 53, 88, 74, 29, 1,  8, 47, 84, 79, 35, 3,
       5,  41, 82, 82, 41, 5, 3,  35, 79, 84, 47, 8, 1,  29, 74, 88, 53, 11, 0, 24, 70, 88, 59, 15},
    },

    // Band_8_2905
    {
      // Scaler inc range
      {2897, 2905},
      // Coefficients 4 taps
      {71, 114, 71, 0,  58, 108, 75,  15, 49, 105, 82,  20, 41, 100, 89,  26,
       33, 95,  95, 33, 26, 89,  100, 41, 20, 82,  105, 49, 15, 75,  108, 58},
      // Coefficients 6 taps
      {19, 64, 90, 64, 19, 0, 15, 58, 90, 69, 24, 0, 11, 53, 88, 74, 29, 1,  8, 47, 84, 79, 35, 3,
       5,  41, 82, 82, 41, 5, 3,  35, 79, 84, 47, 8, 1,  29, 74, 88, 53, 11, 0, 24, 69, 90, 58, 15},
    },

    // Band_8_2913
    {
      // Scaler inc range
      {2905, 2913},
      // Coefficients 4 taps
      {71, 114, 71, 0,  58, 108, 75,  15, 49, 105, 82,  20, 41, 100, 89,  26,
       33, 95,  95, 33, 26, 89,  100, 41, 20, 82,  105, 49, 15, 75,  108, 58},
      // Coefficients 6 taps
      {20, 64, 88, 64, 20, 0, 15, 58, 90, 69, 24, 0, 11, 53, 87, 74, 30, 1,  8, 47, 85, 78, 35, 3,
       5,  41, 82, 82, 41, 5, 3,  35, 78, 85, 47, 8, 1,  30, 74, 87, 53, 11, 0, 24, 69, 90, 58, 15},
    },

    // Band_8_2921
    {
      // Scaler inc range
      {2913, 2921},
      // Coefficients 4 taps
      {71, 114, 71, 0,  58, 108, 75,  15, 49, 105, 82,  20, 41, 100, 89,  26,
       33, 95,  95, 33, 26, 89,  100, 41, 20, 82,  105, 49, 15, 75,  108, 58},
      // Coefficients 6 taps
      {20, 64, 88, 64, 20, 0, 15, 58, 89, 69, 25, 0, 11, 53, 87, 74, 30, 1,  8, 47, 85, 78, 35, 3,
       5,  41, 82, 82, 41, 5, 3,  35, 78, 85, 47, 8, 1,  30, 74, 87, 53, 11, 0, 25, 69, 89, 58, 15},
    },

    // Band_8_2929
    {
      // Scaler inc range
      {2921, 2929},
      // Coefficients 4 taps
      {71, 114, 71, 0,  58, 108, 75, 15, 49, 105, 82,  20, 41, 99, 89,  27,
       33, 95,  95, 33, 27, 89,  99, 41, 20, 82,  105, 49, 15, 75, 108, 58},
      // Coefficients 6 taps
      {20, 64, 88, 64, 20, 0, 15, 58, 89, 69, 25, 0, 12, 53, 86, 74, 30, 1,  8, 47, 85, 78, 35, 3,
       5,  41, 82, 82, 41, 5, 3,  35, 78, 85, 47, 8, 1,  30, 74, 86, 53, 12, 0, 25, 69, 89, 58, 15},
    },

    // Band_8_2937
    {
      // Scaler inc range
      {2929, 2937},
      // Coefficients 4 taps
      {71, 114, 71, 0,  58, 108, 75, 15, 49, 104, 82,  21, 41, 99, 89,  27,
       34, 94,  94, 34, 27, 89,  99, 41, 21, 82,  104, 49, 15, 75, 108, 58},
      // Coefficients 6 taps
      {20, 64, 88, 64, 20, 0, 16, 58, 88, 69, 25, 0, 12, 53, 86, 74, 30, 1,  8, 47, 85, 78, 35, 3,
       6,  41, 81, 81, 41, 6, 3,  35, 78, 85, 47, 8, 1,  30, 74, 86, 53, 12, 0, 25, 69, 88, 58, 16},
    },

    // Band_8_2945
    {
      // Scaler inc range
      {2937, 2945},
      // Coefficients 4 taps
      {71, 114, 71, 0,  58, 108, 75, 15, 49, 104, 82,  21, 41, 99, 89,  27,
       34, 94,  94, 34, 27, 89,  99, 41, 21, 82,  104, 49, 15, 75, 108, 58},
      // Coefficients 6 taps
      {20, 64, 88, 64, 20, 0, 16, 58, 88, 69, 25, 0, 12, 53, 86, 74, 30, 1,  8, 47, 85, 78, 35, 3,
       6,  41, 81, 81, 41, 6, 3,  35, 78, 85, 47, 8, 1,  30, 74, 86, 53, 12, 0, 25, 69, 88, 58, 16},
    },

    // Band_8_2954
    {
      // Scaler inc range
      {2945, 2954},
      // Coefficients 4 taps
      {71, 114, 71, 0,  58, 108, 75,  15, 50, 103, 82,  21, 41, 100, 88,  27,
       34, 94,  94, 34, 27, 88,  100, 41, 21, 82,  103, 50, 15, 75,  108, 58},
      // Coefficients 6 taps
      {20, 64, 88, 64, 20, 0, 16, 58, 88, 69, 25, 0, 12, 53, 85, 74, 30, 2,  9, 47, 84, 78, 35, 3,
       6,  41, 81, 81, 41, 6, 3,  35, 78, 84, 47, 9, 2,  30, 74, 85, 53, 12, 0, 25, 69, 88, 58, 16},
    },

    // Band_8_2962
    {
      // Scaler inc range
      {2954, 2962},
      // Coefficients 4 taps
      {71, 114, 71, 0,  58, 107, 75,  16, 50, 103, 82,  21, 41, 100, 88,  27,
       34, 94,  94, 34, 27, 88,  100, 41, 21, 82,  103, 50, 16, 75,  107, 58},
      // Coefficients 6 taps
      {20, 64, 88, 64, 20, 0, 16, 58, 88, 69, 25, 0, 12, 53, 85, 74, 30, 2,  9, 47, 84, 78, 35, 3,
       6,  41, 81, 81, 41, 6, 3,  35, 78, 84, 47, 9, 2,  30, 74, 85, 53, 12, 0, 25, 69, 88, 58, 16},
    },

    // Band_8_2970
    {
      // Scaler inc range
      {2962, 2970},
      // Coefficients 4 taps
      {71, 114, 71, 0,  58, 107, 75,  16, 50, 103, 82,  21, 41, 100, 88,  27,
       34, 94,  94, 34, 27, 88,  100, 41, 21, 82,  103, 50, 16, 75,  107, 58},
      // Coefficients 6 taps
      {20, 64, 88, 64, 20, 0, 16, 58, 88, 69, 25, 0, 12, 53, 86, 73, 30, 2,  9, 47, 83, 78, 35, 4,
       6,  41, 81, 81, 41, 6, 4,  35, 78, 83, 47, 9, 2,  30, 73, 86, 53, 12, 0, 25, 69, 88, 58, 16},
    },

    // Band_8_2979
    {
      // Scaler inc range
      {2970, 2979},
      // Coefficients 4 taps
      {71, 114, 71, 0,  59, 106, 75,  16, 50, 103, 82,  21, 41, 100, 88,  27,
       34, 94,  94, 34, 27, 88,  100, 41, 21, 82,  103, 50, 16, 75,  106, 59},
      // Coefficients 6 taps
      {20, 64, 88, 64, 20, 0, 16, 58, 88, 69, 25, 0, 12, 53, 86, 73, 30, 2,  9, 47, 83, 77, 36, 4,
       6,  41, 81, 81, 41, 6, 4,  36, 77, 83, 47, 9, 2,  30, 73, 86, 53, 12, 0, 25, 69, 88, 58, 16},
    },

    // Band_8_2987
    {
      // Scaler inc range
      {2979, 2987},
      // Coefficients 4 taps
      {71, 114, 71, 0,  59, 106, 75,  16, 50, 103, 82,  21, 41, 100, 88,  27,
       34, 94,  94, 34, 27, 88,  100, 41, 21, 82,  103, 50, 16, 75,  106, 59},
      // Coefficients 6 taps
      {21, 64, 86, 64, 21, 0, 16, 58, 88, 69, 25, 0, 12, 53, 86, 73, 30, 2,  9, 47, 83, 77, 36, 4,
       6,  41, 81, 81, 41, 6, 4,  36, 77, 83, 47, 9, 2,  30, 73, 86, 53, 12, 0, 25, 69, 88, 58, 16},
    },

    // Band_8_2996
    {
      // Scaler inc range
      {2987, 2996},
      // Coefficients 4 taps
      {71, 114, 71, 0,  59, 106, 75, 16, 50, 103, 82,  21, 42, 99, 88,  27,
       34, 94,  94, 34, 27, 88,  99, 42, 21, 82,  103, 50, 16, 75, 106, 59},
      // Coefficients 6 taps
      {21, 64, 86, 64, 21, 0, 16, 58, 87, 69, 25, 1, 12, 53, 86, 73, 30, 2,  9, 47, 83, 77, 36, 4,
       6,  41, 81, 81, 41, 6, 4,  36, 77, 83, 47, 9, 2,  30, 73, 86, 53, 12, 1, 25, 69, 87, 58, 16},
    },

    // Band_8_3005
    {
      // Scaler inc range
      {2996, 3005},
      // Coefficients 4 taps
      {71, 114, 71, 0,  59, 106, 75, 16, 50, 103, 82,  21, 42, 99, 88,  27,
       34, 94,  94, 34, 27, 88,  99, 42, 21, 82,  103, 50, 16, 75, 106, 59},
      // Coefficients 6 taps
      {21, 64, 86, 64, 21, 0, 16, 58, 87, 69, 25, 1, 13, 53, 85, 73, 30, 2,  9, 47, 83, 77, 36, 4,
       6,  41, 81, 81, 41, 6, 4,  36, 77, 83, 47, 9, 2,  30, 73, 85, 53, 13, 1, 25, 69, 87, 58, 16},
    },

    // Band_8_3013
    {
      // Scaler inc range
      {3005, 3013},
      // Coefficients 4 taps
      {71, 114, 71, 0,  59, 106, 75, 16, 50, 103, 82,  21, 42, 99, 88,  27,
       34, 94,  94, 34, 27, 88,  99, 42, 21, 82,  103, 50, 16, 75, 106, 59},
      // Coefficients 6 taps
      {21, 64, 86, 64, 21, 0, 17, 58, 86, 68, 26, 1, 13, 53, 85, 73, 30, 2,  9, 47, 83, 77, 36, 4,
       6,  41, 81, 81, 41, 6, 4,  36, 77, 83, 47, 9, 2,  30, 73, 85, 53, 13, 1, 26, 68, 86, 58, 17},
    },

    // Band_8_3022
    {
      // Scaler inc range
      {3013, 3022},
      // Coefficients 4 taps
      {71, 114, 71, 0,  59, 106, 75, 16, 50, 103, 82,  21, 42, 99, 88,  27,
       34, 94,  94, 34, 27, 88,  99, 42, 21, 82,  103, 50, 16, 75, 106, 59},
      // Coefficients 6 taps
      {21, 64, 86, 64, 21, 0, 17, 58, 86, 68, 26, 1, 13, 53, 84, 73, 31, 2,  9, 47, 83, 77, 36, 4,
       7,  41, 80, 80, 41, 7, 4,  36, 77, 83, 47, 9, 2,  31, 73, 84, 53, 13, 1, 26, 68, 86, 58, 17},
    },

    // Band_8_3031
    {
      // Scaler inc range
      {3022, 3031},
      // Coefficients 4 taps
      {71, 114, 71, 0,  59, 106, 75, 16, 50, 103, 82,  21, 42, 99, 88,  27,
       34, 94,  94, 34, 27, 88,  99, 42, 21, 82,  103, 50, 16, 75, 106, 59},
      // Coefficients 6 taps
      {21, 63, 88, 63, 21, 0, 17, 58, 86, 68, 26, 1,  13, 53, 84, 73, 31, 2,  10, 47, 82, 77, 36, 4,
       7,  41, 80, 80, 41, 7, 4,  36, 77, 82, 47, 10, 2,  31, 73, 84, 53, 13, 1,  26, 68, 86, 58, 17},
    },

    // Band_8_3039
    {
      // Scaler inc range
      {3031, 3039},
      // Coefficients 4 taps
      {71, 114, 71, 0,  59, 106, 75, 16, 50, 103, 82,  21, 42, 98, 88,  28,
       34, 94,  94, 34, 28, 88,  98, 42, 21, 82,  103, 50, 16, 75, 106, 59},
      // Coefficients 6 taps
      {21, 63, 88, 63, 21, 0, 17, 58, 86, 68, 26, 1,  13, 53, 84, 73, 31, 2,  10, 47, 82, 77, 36, 4,
       7,  41, 80, 80, 41, 7, 4,  36, 77, 82, 47, 10, 2,  31, 73, 84, 53, 13, 1,  26, 68, 86, 58, 17},
    },

    // Band_8_3048
    {
      // Scaler inc range
      {3039, 3048},
      // Coefficients 4 taps
      {71, 114, 71, 0,  59, 106, 75, 16, 50, 102, 82,  22, 42, 98, 88,  28,
       34, 94,  94, 34, 28, 88,  98, 42, 22, 82,  102, 50, 16, 75, 106, 59},
      // Coefficients 6 taps
      {21, 63, 88, 63, 21, 0, 17, 58, 86, 68, 26, 1,  13, 53, 84, 73, 31, 2,  10, 47, 83, 76, 36, 4,
       7,  41, 80, 80, 41, 7, 4,  36, 76, 83, 47, 10, 2,  31, 73, 84, 53, 13, 1,  26, 68, 86, 58, 17},
    },

    // Band_8_3057
    {
      // Scaler inc range
      {3048, 3057},
      // Coefficients 4 taps
      {71, 114, 71, 0,  59, 106, 75, 16, 50, 102, 82,  22, 42, 98, 88,  28,
       34, 94,  94, 34, 28, 88,  98, 42, 22, 82,  102, 50, 16, 75, 106, 59},
      // Coefficients 6 taps
      {21, 63, 88, 63, 21, 0, 17, 58, 86, 68, 26, 1,  13, 53, 84, 72, 31, 3,  10, 47, 82, 76, 36, 5,
       7,  41, 80, 80, 41, 7, 5,  36, 76, 82, 47, 10, 3,  31, 72, 84, 53, 13, 1,  26, 68, 86, 58, 17},
    },

    // Band_8_3066
    {
      // Scaler inc range
      {3057, 3066},
      // Coefficients 4 taps
      {71, 114, 71, 0,  59, 106, 75, 16, 50, 102, 82,  22, 42, 98, 88,  28,
       35, 93,  93, 35, 28, 88,  98, 42, 22, 82,  102, 50, 16, 75, 106, 59},
      // Coefficients 6 taps
      {21, 63, 88, 63, 21, 0, 17, 58, 86, 68, 26, 1,  13, 53, 84, 72, 31, 3,  10, 47, 82, 76, 36, 5,
       7,  41, 80, 80, 41, 7, 5,  36, 76, 82, 47, 10, 3,  31, 72, 84, 53, 13, 1,  26, 68, 86, 58, 17},
    },

    // Band_8_3075
    {
      // Scaler inc range
      {3066, 3075},
      // Coefficients 4 taps
      {71, 114, 71, 0,  59, 105, 75, 17, 50, 102, 82,  22, 42, 98, 88,  28,
       35, 93,  93, 35, 28, 88,  98, 42, 22, 82,  102, 50, 17, 75, 105, 59},
      // Coefficients 6 taps
      {22, 63, 86, 63, 22, 0, 17, 58, 86, 68, 26, 1,  13, 52, 85, 72, 31, 3,  10, 47, 82, 76, 36, 5,
       7,  42, 79, 79, 42, 7, 5,  36, 76, 82, 47, 10, 3,  31, 72, 85, 52, 13, 1,  26, 68, 86, 58, 17},
    },

    // Band_8_3084
    {
      // Scaler inc range
      {3075, 3084},
      // Coefficients 4 taps
      {71, 114, 71, 0,  59, 105, 75, 17, 50, 102, 82,  22, 42, 98, 88,  28,
       35, 93,  93, 35, 28, 88,  98, 42, 22, 82,  102, 50, 17, 75, 105, 59},
      // Coefficients 6 taps
      {22, 63, 86, 63, 22, 0, 17, 58, 86, 68, 26, 1,  14, 52, 84, 72, 31, 3,  10, 47, 82, 76, 36, 5,
       7,  42, 79, 79, 42, 7, 5,  36, 76, 82, 47, 10, 3,  31, 72, 84, 52, 14, 1,  26, 68, 86, 58, 17},
    },

    // Band_8_3093
    {
      // Scaler inc range
      {3084, 3093},
      // Coefficients 4 taps
      {71, 114, 71, 0,  59, 105, 75, 17, 50, 102, 82,  22, 42, 98, 88,  28,
       35, 93,  93, 35, 28, 88,  98, 42, 22, 82,  102, 50, 17, 75, 105, 59},
      // Coefficients 6 taps
      {22, 63, 86, 63, 22, 0, 17, 58, 86, 68, 26, 1,  14, 52, 84, 72, 31, 3,  10, 47, 82, 76, 36, 5,
       7,  42, 79, 79, 42, 7, 5,  36, 76, 82, 47, 10, 3,  31, 72, 84, 52, 14, 1,  26, 68, 86, 58, 17},
    },

    // Band_8_3102
    {
      // Scaler inc range
      {3093, 3102},
      // Coefficients 4 taps
      {71, 114, 71, 0,  59, 105, 75, 17, 50, 102, 82,  22, 42, 98, 88,  28,
       35, 93,  93, 35, 28, 88,  98, 42, 22, 82,  102, 50, 17, 75, 105, 59},
      // Coefficients 6 taps
      {22, 63, 86, 63, 22, 0, 18, 58, 85, 68, 26, 1,  14, 52, 84, 72, 31, 3,  10, 47, 82, 76, 36, 5,
       7,  42, 79, 79, 42, 7, 5,  36, 76, 82, 47, 10, 3,  31, 72, 84, 52, 14, 1,  26, 68, 85, 58, 18},
    },

    // Band_8_3112
    {
      // Scaler inc range
      {3102, 3112},
      // Coefficients 4 taps
      {72, 112, 72, 0,  59, 105, 75, 17, 50, 103, 81,  22, 42, 98, 88,  28,
       35, 93,  93, 35, 28, 88,  98, 42, 22, 81,  103, 50, 17, 75, 105, 59},
      // Coefficients 6 taps
      {22, 63, 86, 63, 22, 0, 18, 58, 84, 68, 26, 2,  14, 52, 84, 72, 31, 3,  10, 47, 82, 76, 36, 5,
       8,  42, 78, 78, 42, 8, 5,  36, 76, 82, 47, 10, 3,  31, 72, 84, 52, 14, 2,  26, 68, 84, 58, 18},
    },

    // Band_8_3121
    {
      // Scaler inc range
      {3112, 3121},
      // Coefficients 4 taps
      {72, 112, 72, 0,  59, 105, 75, 17, 50, 103, 81,  22, 42, 98, 88,  28,
       35, 93,  93, 35, 28, 88,  98, 42, 22, 81,  103, 50, 17, 75, 105, 59},
      // Coefficients 6 taps
      {22, 63, 86, 63, 22, 0, 18, 58, 83, 68, 27, 2,  14, 52, 84, 72, 31, 3,  11, 47, 82, 75, 36, 5,
       8,  42, 78, 78, 42, 8, 5,  36, 75, 82, 47, 11, 3,  31, 72, 84, 52, 14, 2,  27, 68, 83, 58, 18},
    },

    // Band_8_3130
    {
      // Scaler inc range
      {3121, 3130},
      // Coefficients 4 taps
      {72, 112, 72, 0,  59, 105, 75, 17, 50, 103, 81,  22, 42, 98, 88,  28,
       35, 93,  93, 35, 28, 88,  98, 42, 22, 81,  103, 50, 17, 75, 105, 59},
      // Coefficients 6 taps
      {22, 63, 86, 63, 22, 0, 18, 58, 83, 68, 27, 2,  14, 52, 84, 72, 31, 3,  11, 47, 81, 75, 37, 5,
       8,  42, 78, 78, 42, 8, 5,  37, 75, 81, 47, 11, 3,  31, 72, 84, 52, 14, 2,  27, 68, 83, 58, 18},
    },

    // Band_8_3139
    {
      // Scaler inc range
      {3130, 3139},
      // Coefficients 4 taps
      {72, 112, 72, 0,  59, 105, 75, 17, 51, 102, 81,  22, 42, 99, 87,  28,
       35, 93,  93, 35, 28, 87,  99, 42, 22, 81,  102, 51, 17, 75, 105, 59},
      // Coefficients 6 taps
      {22, 63, 86, 63, 22, 0, 18, 58, 84, 67, 27, 2,  14, 52, 83, 72, 32, 3,  11, 47, 81, 75, 37, 5,
       8,  42, 78, 78, 42, 8, 5,  37, 75, 81, 47, 11, 3,  32, 72, 83, 52, 14, 2,  27, 67, 84, 58, 18},
    },

    // Band_8_3149
    {
      // Scaler inc range
      {3139, 3149},
      // Coefficients 4 taps
      {72, 112, 72, 0,  59, 105, 75, 17, 51, 102, 81,  22, 43, 98, 87,  28,
       35, 93,  93, 35, 28, 87,  98, 43, 22, 81,  102, 51, 17, 75, 105, 59},
      // Coefficients 6 taps
      {22, 63, 86, 63, 22, 0, 18, 58, 84, 67, 27, 2,  14, 52, 84, 71, 32, 3,  11, 47, 81, 75, 37, 5,
       8,  42, 78, 78, 42, 8, 5,  37, 75, 81, 47, 11, 3,  32, 71, 84, 52, 14, 2,  27, 67, 84, 58, 18},
    },

    // Band_8_3158
    {
      // Scaler inc range
      {3149, 3158},
      // Coefficients 4 taps
      {72, 112, 72, 0,  59, 105, 75, 17, 51, 102, 81,  22, 43, 98, 87,  28,
       35, 93,  93, 35, 28, 87,  98, 43, 22, 81,  102, 51, 17, 75, 105, 59},
      // Coefficients 6 taps
      {23, 63, 84, 63, 23, 0, 18, 58, 84, 67, 27, 2,  14, 52, 83, 71, 32, 4,  11, 47, 80, 75, 37, 6,
       8,  42, 78, 78, 42, 8, 6,  37, 75, 80, 47, 11, 4,  32, 71, 83, 52, 14, 2,  27, 67, 84, 58, 18},
    },

    // Band_8_3168
    {
      // Scaler inc range
      {3158, 3168},
      // Coefficients 4 taps
      {72, 112, 72, 0,  59, 105, 75, 17, 51, 101, 81,  23, 43, 97, 87,  29,
       35, 93,  93, 35, 29, 87,  97, 43, 23, 81,  101, 51, 17, 75, 105, 59},
      // Coefficients 6 taps
      {23, 63, 84, 63, 23, 0, 18, 58, 84, 67, 27, 2,  14, 52, 83, 71, 32, 4,  11, 47, 80, 75, 37, 6,
       8,  42, 78, 78, 42, 8, 6,  37, 75, 80, 47, 11, 4,  32, 71, 83, 52, 14, 2,  27, 67, 84, 58, 18},
    },

    // Band_8_3178
    {
      // Scaler inc range
      {3168, 3178},
      // Coefficients 4 taps
      {72, 112, 72, 0,  59, 105, 75, 17, 51, 101, 81,  23, 43, 97, 87,  29,
       35, 93,  93, 35, 29, 87,  97, 43, 23, 81,  101, 51, 17, 75, 105, 59},
      // Coefficients 6 taps
      {23, 63, 84, 63, 23, 0, 18, 58, 84, 67, 27, 2,  15, 52, 82, 71, 32, 4,  11, 47, 80, 75, 37, 6,
       8,  42, 78, 78, 42, 8, 6,  37, 75, 80, 47, 11, 4,  32, 71, 82, 52, 15, 2,  27, 67, 84, 58, 18},
    },

    // Band_8_3187
    {
      // Scaler inc range
      {3178, 3187},
      // Coefficients 4 taps
      {72, 112, 72, 0,  59, 105, 75, 17, 51, 101, 81,  23, 43, 97, 87,  29,
       35, 93,  93, 35, 29, 87,  97, 43, 23, 81,  101, 51, 17, 75, 105, 59},
      // Coefficients 6 taps
      {23, 63, 84, 63, 23, 0, 18, 58, 84, 67, 27, 2,  15, 52, 82, 71, 32, 4,  11, 47, 80, 75, 37, 6,
       8,  42, 78, 78, 42, 8, 6,  37, 75, 80, 47, 11, 4,  32, 71, 82, 52, 15, 2,  27, 67, 84, 58, 18},
    },

    // Band_8_3197
    {
      // Scaler inc range
      {3187, 3197},
      // Coefficients 4 taps
      {72, 112, 72, 0,  59, 105, 75, 17, 51, 101, 81,  23, 43, 97, 87,  29,
       35, 93,  93, 35, 29, 87,  97, 43, 23, 81,  101, 51, 17, 75, 105, 59},
      // Coefficients 6 taps
      {23, 63, 84, 63, 23, 0, 19, 58, 83, 67, 27, 2,  15, 52, 82, 71, 32, 4,  11, 47, 80, 75, 37, 6,
       8,  42, 78, 78, 42, 8, 6,  37, 75, 80, 47, 11, 4,  32, 71, 82, 52, 15, 2,  27, 67, 83, 58, 19},
    },

    // Band_8_3207
    {
      // Scaler inc range
      {3197, 3207},
      // Coefficients 4 taps
      {72, 112, 72, 0,  59, 104, 75, 18, 51, 101, 81,  23, 43, 97, 87,  29,
       36, 92,  92, 36, 29, 87,  97, 43, 23, 81,  101, 51, 18, 75, 104, 59},
      // Coefficients 6 taps
      {23, 63, 84, 63, 23, 0, 19, 58, 83, 67, 27, 2,  15, 52, 82, 71, 32, 4,  11, 47, 81, 74, 37, 6,
       9,  42, 77, 77, 42, 9, 6,  37, 74, 81, 47, 11, 4,  32, 71, 82, 52, 15, 2,  27, 67, 83, 58, 19},
    },

    // Band_8_3216
    {
      // Scaler inc range
      {3207, 3216},
      // Coefficients 4 taps
      {72, 112, 72, 0,  59, 104, 75, 18, 51, 101, 81,  23, 43, 97, 87,  29,
       36, 92,  92, 36, 29, 87,  97, 43, 23, 81,  101, 51, 18, 75, 104, 59},
      // Coefficients 6 taps
      {23, 63, 84, 63, 23, 0, 19, 58, 83, 67, 27, 2,  15, 52, 82, 71, 32, 4,  12, 47, 80, 74, 37, 6,
       9,  42, 77, 77, 42, 9, 6,  37, 74, 80, 47, 12, 4,  32, 71, 82, 52, 15, 2,  27, 67, 83, 58, 19},
    },

    // Band_8_3226
    {
      // Scaler inc range
      {3216, 3226},
      // Coefficients 4 taps
      {72, 112, 72, 0,  59, 104, 75, 18, 51, 101, 81,  23, 43, 97, 87,  29,
       36, 92,  92, 36, 29, 87,  97, 43, 23, 81,  101, 51, 18, 75, 104, 59},
      // Coefficients 6 taps
      {23, 63, 84, 63, 23, 0, 19, 58, 83, 67, 27, 2,  15, 52, 82, 71, 32, 4,  12, 47, 80, 74, 37, 6,
       9,  42, 77, 77, 42, 9, 6,  37, 74, 80, 47, 12, 4,  32, 71, 82, 52, 15, 2,  27, 67, 83, 58, 19},
    },

    // Band_8_3236
    {
      // Scaler inc range
      {3226, 3236},
      // Coefficients 4 taps
      {72, 112, 72, 0,  59, 104, 75, 18, 51, 101, 81,  23, 43, 97, 87,  29,
       36, 92,  92, 36, 29, 87,  97, 43, 23, 81,  101, 51, 18, 75, 104, 59},
      // Coefficients 6 taps
      {23, 63, 84, 63, 23, 0, 19, 58, 81, 67, 28, 3,  15, 52, 82, 71, 32, 4,  12, 47, 80, 74, 37, 6,
       9,  42, 77, 77, 42, 9, 6,  37, 74, 80, 47, 12, 4,  32, 71, 82, 52, 15, 3,  28, 67, 81, 58, 19},
    },

    // Band_8_3246
    {
      // Scaler inc range
      {3236, 3246},
      // Coefficients 4 taps
      {72, 112, 72, 0,  59, 104, 75, 18, 51, 101, 81,  23, 43, 97, 87,  29,
       36, 92,  92, 36, 29, 87,  97, 43, 23, 81,  101, 51, 18, 75, 104, 59},
      // Coefficients 6 taps
      {23, 63, 84, 63, 23, 0, 19, 57, 82, 67, 28, 3,  15, 52, 83, 70, 32, 4,  12, 47, 80, 74, 37, 6,
       9,  42, 77, 77, 42, 9, 6,  37, 74, 80, 47, 12, 4,  32, 70, 83, 52, 15, 3,  28, 67, 82, 57, 19},
    },

    // Band_8_3256
    {
      // Scaler inc range
      {3246, 3256},
      // Coefficients 4 taps
      {72, 112, 72, 0,  59, 104, 75, 18, 51, 101, 81,  23, 43, 97, 87,  29,
       36, 92,  92, 36, 29, 87,  97, 43, 23, 81,  101, 51, 18, 75, 104, 59},
      // Coefficients 6 taps
      {24, 63, 82, 63, 24, 0, 19, 57, 82, 67, 28, 3,  15, 52, 83, 70, 32, 4,  12, 47, 79, 74, 37, 7,
       9,  42, 77, 77, 42, 9, 7,  37, 74, 79, 47, 12, 4,  32, 70, 83, 52, 15, 3,  28, 67, 82, 57, 19},
    },

    // Band_8_3267
    {
      // Scaler inc range
      {3256, 3267},
      // Coefficients 4 taps
      {72, 112, 72, 0,  59, 104, 75, 18, 51, 101, 81,  23, 43, 97, 87,  29,
       36, 92,  92, 36, 29, 87,  97, 43, 23, 81,  101, 51, 18, 75, 104, 59},
      // Coefficients 6 taps
      {24, 63, 82, 63, 24, 0, 19, 57, 83, 66, 28, 3,  15, 52, 83, 70, 32, 4,  12, 47, 79, 74, 37, 7,
       9,  42, 77, 77, 42, 9, 7,  37, 74, 79, 47, 12, 4,  32, 70, 83, 52, 15, 3,  28, 66, 83, 57, 19},
    },

    // Band_8_3277
    {
      // Scaler inc range
      {3267, 3277},
      // Coefficients 4 taps
      {72, 112, 72, 0,  60, 103, 75, 18, 51, 101, 81,  23, 43, 97, 87,  29,
       36, 92,  92, 36, 29, 87,  97, 43, 23, 81,  101, 51, 18, 75, 103, 60},
      // Coefficients 6 taps
      {24, 63, 82, 63, 24, 0, 19, 57, 83, 66, 28, 3,  16, 52, 81, 70, 32, 5,  12, 47, 79, 74, 37, 7,
       9,  42, 77, 77, 42, 9, 7,  37, 74, 79, 47, 12, 5,  32, 70, 81, 52, 16, 3,  28, 66, 83, 57, 19},
    },

    // Band_8_3287
    {
      // Scaler inc range
      {3277, 3287},
      // Coefficients 4 taps
      {72, 112, 72, 0,  60, 103, 75, 18, 51, 101, 81,  23, 43, 97, 87,  29,
       36, 92,  92, 36, 29, 87,  97, 43, 23, 81,  101, 51, 18, 75, 103, 60},
      // Coefficients 6 taps
      {24, 63, 82, 63, 24, 0, 19, 57, 83, 66, 28, 3,  16, 52, 80, 70, 33, 5,  12, 47, 80, 73, 37, 7,
       9,  42, 77, 77, 42, 9, 7,  37, 73, 80, 47, 12, 5,  33, 70, 80, 52, 16, 3,  28, 66, 83, 57, 19},
    },

    // Band_8_3297
    {
      // Scaler inc range
      {3287, 3297},
      // Coefficients 4 taps
      {72, 112, 72, 0,  60, 103, 75, 18, 51, 101, 81,  23, 43, 97, 87,  29,
       36, 92,  92, 36, 29, 87,  97, 43, 23, 81,  101, 51, 18, 75, 103, 60},
      // Coefficients 6 taps
      {24, 63, 82, 63, 24, 0, 20, 57, 82, 66, 28, 3,  16, 52, 80, 70, 33, 5,  12, 47, 80, 73, 37, 7,
       9,  42, 77, 77, 42, 9, 7,  37, 73, 80, 47, 12, 5,  33, 70, 80, 52, 16, 3,  28, 66, 82, 57, 20},
    },

    // Band_8_3308
    {
      // Scaler inc range
      {3297, 3308},
      // Coefficients 4 taps
      {72, 112, 72, 0,  60, 103, 75, 18, 51, 100, 81,  24, 43, 97, 87,  29,
       36, 92,  92, 36, 29, 87,  97, 43, 24, 81,  100, 51, 18, 75, 103, 60},
      // Coefficients 6 taps
      {24, 63, 82, 63, 24, 0,  20, 57, 82, 66, 28, 3,  16, 52, 80, 70, 33, 5,  12, 47, 80, 73, 37, 7,
       10, 42, 76, 76, 42, 10, 7,  37, 73, 80, 47, 12, 5,  33, 70, 80, 52, 16, 3,  28, 66, 82, 57, 20},
    },

    // Band_8_3318
    {
      // Scaler inc range
      {3308, 3318},
      // Coefficients 4 taps
      {72, 112, 72, 0,  60, 103, 75, 18, 51, 100, 81,  24, 43, 96, 87,  30,
       36, 92,  92, 36, 30, 87,  96, 43, 24, 81,  100, 51, 18, 75, 103, 60},
      // Coefficients 6 taps
      {24, 63, 82, 63, 24, 0,  20, 57, 82, 66, 28, 3,  16, 52, 80, 70, 33, 5,  13, 47, 79, 73, 37, 7,
       10, 42, 76, 76, 42, 10, 7,  37, 73, 79, 47, 13, 5,  33, 70, 80, 52, 16, 3,  28, 66, 82, 57, 20},
    },

    // Band_8_3329
    {
      // Scaler inc range
      {3318, 3329},
      // Coefficients 4 taps
      {72, 112, 72, 0,  60, 103, 75, 18, 51, 100, 81,  24, 44, 95, 87,  30,
       36, 92,  92, 36, 30, 87,  95, 44, 24, 81,  100, 51, 18, 75, 103, 60},
      // Coefficients 6 taps
      {24, 63, 82, 63, 24, 0,  20, 57, 82, 66, 28, 3,  16, 52, 80, 70, 33, 5,  13, 47, 78, 73, 38, 7,
       10, 42, 76, 76, 42, 10, 7,  38, 73, 78, 47, 13, 5,  33, 70, 80, 52, 16, 3,  28, 66, 82, 57, 20},
    },

    // Band_8_3339
    {
      // Scaler inc range
      {3329, 3339},
      // Coefficients 4 taps
      {72, 112, 72, 0,  60, 103, 75, 18, 51, 100, 81,  24, 44, 95, 87,  30,
       36, 92,  92, 36, 30, 87,  95, 44, 24, 81,  100, 51, 18, 75, 103, 60},
      // Coefficients 6 taps
      {24, 62, 84, 62, 24, 0,  20, 57, 82, 66, 28, 3,  16, 52, 80, 70, 33, 5,  13, 47, 78, 73, 38, 7,
       10, 42, 76, 76, 42, 10, 7,  38, 73, 78, 47, 13, 5,  33, 70, 80, 52, 16, 3,  28, 66, 82, 57, 20},
    },

    // Band_8_3350
    {
      // Scaler inc range
      {3339, 3350},
      // Coefficients 4 taps
      {72, 112, 72, 0,  60, 103, 75, 18, 51, 100, 81,  24, 44, 96, 86,  30,
       36, 92,  92, 36, 30, 86,  96, 44, 24, 81,  100, 51, 18, 75, 103, 60},
      // Coefficients 6 taps
      {24, 62, 84, 62, 24, 0,  20, 57, 82, 66, 28, 3,  16, 52, 80, 70, 33, 5,  13, 47, 78, 73, 38, 7,
       10, 42, 76, 76, 42, 10, 7,  38, 73, 78, 47, 13, 5,  33, 70, 80, 52, 16, 3,  28, 66, 82, 57, 20},
    },

    // Band_8_3361
    {
      // Scaler inc range
      {3350, 3361},
      // Coefficients 4 taps
      {72, 112, 72, 0,  60, 102, 75, 19, 51, 100, 81,  24, 44, 96, 86,  30,
       36, 92,  92, 36, 30, 86,  96, 44, 24, 81,  100, 51, 19, 75, 102, 60},
      // Coefficients 6 taps
      {24, 62, 84, 62, 24, 0,  20, 57, 81, 66, 29, 3,  16, 52, 81, 69, 33, 5,  13, 47, 78, 73, 38, 7,
       10, 42, 76, 76, 42, 10, 7,  38, 73, 78, 47, 13, 5,  33, 69, 81, 52, 16, 3,  29, 66, 81, 57, 20},
    },

    // Band_8_3372
    {
      // Scaler inc range
      {3361, 3372},
      // Coefficients 4 taps
      {72, 112, 72, 0,  60, 102, 75, 19, 52, 99, 81, 24, 44, 96, 86,  30,
       37, 91,  91, 37, 30, 86,  96, 44, 24, 81, 99, 52, 19, 75, 102, 60},
      // Coefficients 6 taps
      {25, 62, 82, 62, 25, 0,  20, 57, 80, 66, 29, 4,  16, 52, 81, 69, 33, 5,  13, 47, 77, 73, 38, 8,
       10, 43, 75, 75, 43, 10, 8,  38, 73, 77, 47, 13, 5,  33, 69, 81, 52, 16, 4,  29, 66, 80, 57, 20},
    },

    // Band_8_3383
    {
      // Scaler inc range
      {3372, 3383},
      // Coefficients 4 taps
      {72, 112, 72, 0,  60, 102, 75, 19, 52, 99, 81, 24, 44, 96, 86,  30,
       37, 91,  91, 37, 30, 86,  96, 44, 24, 81, 99, 52, 19, 75, 102, 60},
      // Coefficients 6 taps
      {25, 62, 82, 62, 25, 0,  20, 57, 80, 66, 29, 4,  17, 52, 80, 69, 33, 5,  13, 47, 77, 73, 38, 8,
       10, 43, 75, 75, 43, 10, 8,  38, 73, 77, 47, 13, 5,  33, 69, 80, 52, 17, 4,  29, 66, 80, 57, 20},
    },

    // Band_8_3393
    {
      // Scaler inc range
      {3383, 3393},
      // Coefficients 4 taps
      {72, 112, 72, 0,  60, 102, 75, 19, 52, 99, 81, 24, 44, 96, 86,  30,
       37, 91,  91, 37, 30, 86,  96, 44, 24, 81, 99, 52, 19, 75, 102, 60},
      // Coefficients 6 taps
      {25, 62, 82, 62, 25, 0,  20, 57, 80, 66, 29, 4,  17, 52, 79, 69, 33, 6,  13, 47, 78, 72, 38, 8,
       10, 43, 75, 75, 43, 10, 8,  38, 72, 78, 47, 13, 6,  33, 69, 79, 52, 17, 4,  29, 66, 80, 57, 20},
    },

    // Band_8_3404
    {
      // Scaler inc range
      {3393, 3404},
      // Coefficients 4 taps
      {73, 110, 73, 0,  60, 102, 75, 19, 52, 99, 81, 24, 44, 96, 86,  30,
       37, 91,  91, 37, 30, 86,  96, 44, 24, 81, 99, 52, 19, 75, 102, 60},
      // Coefficients 6 taps
      {25, 62, 82, 62, 25, 0,  20, 57, 80, 66, 29, 4,  17, 52, 79, 69, 33, 6,  13, 47, 78, 72, 38, 8,
       10, 43, 75, 75, 43, 10, 8,  38, 72, 78, 47, 13, 6,  33, 69, 79, 52, 17, 4,  29, 66, 80, 57, 20},
    },

    // Band_8_3416
    {
      // Scaler inc range
      {3404, 3416},
      // Coefficients 4 taps
      {73, 110, 73, 0,  60, 102, 75, 19, 52, 99, 81, 24, 44, 96, 86,  30,
       37, 91,  91, 37, 30, 86,  96, 44, 24, 81, 99, 52, 19, 75, 102, 60},
      // Coefficients 6 taps
      {25, 62, 82, 62, 25, 0,  21, 57, 79, 66, 29, 4,  17, 52, 79, 69, 33, 6,  13, 47, 78, 72, 38, 8,
       10, 43, 75, 75, 43, 10, 8,  38, 72, 78, 47, 13, 6,  33, 69, 79, 52, 17, 4,  29, 66, 79, 57, 21},
    },

    // Band_8_3427
    {
      // Scaler inc range
      {3416, 3427},
      // Coefficients 4 taps
      {73, 110, 73, 0,  60, 102, 75, 19, 52, 99, 81, 24, 44, 96, 86,  30,
       37, 91,  91, 37, 30, 86,  96, 44, 24, 81, 99, 52, 19, 75, 102, 60},
      // Coefficients 6 taps
      {25, 62, 82, 62, 25, 0,  21, 57, 80, 65, 29, 4,  17, 52, 79, 69, 33, 6,  14, 47, 77, 72, 38, 8,
       11, 43, 74, 74, 43, 11, 8,  38, 72, 77, 47, 14, 6,  33, 69, 79, 52, 17, 4,  29, 65, 80, 57, 21},
    },

    // Band_8_3438
    {
      // Scaler inc range
      {3427, 3438},
      // Coefficients 4 taps
      {73, 110, 73, 0,  60, 102, 75, 19, 52, 99, 81, 24, 44, 96, 86,  30,
       37, 91,  91, 37, 30, 86,  96, 44, 24, 81, 99, 52, 19, 75, 102, 60},
      // Coefficients 6 taps
      {25, 62, 82, 62, 25, 0,  21, 57, 80, 65, 29, 4,  17, 52, 79, 69, 33, 6,  14, 47, 77, 72, 38, 8,
       11, 43, 74, 74, 43, 11, 8,  38, 72, 77, 47, 14, 6,  33, 69, 79, 52, 17, 4,  29, 65, 80, 57, 21},
    },

    // Band_8_3449
    {
      // Scaler inc range
      {3438, 3449},
      // Coefficients 4 taps
      {73, 110, 73, 0,  60, 102, 75, 19, 52, 99, 81, 24, 44, 96, 86,  30,
       37, 91,  91, 37, 30, 86,  96, 44, 24, 81, 99, 52, 19, 75, 102, 60},
      // Coefficients 6 taps
      {25, 62, 82, 62, 25, 0,  21, 57, 80, 65, 29, 4,  17, 52, 79, 69, 33, 6,  14, 47, 77, 72, 38, 8,
       11, 43, 74, 74, 43, 11, 8,  38, 72, 77, 47, 14, 6,  33, 69, 79, 52, 17, 4,  29, 65, 80, 57, 21},
    },

    // Band_8_3461
    {
      // Scaler inc range
      {3449, 3461},
      // Coefficients 4 taps
      {73, 110, 73, 0,  60, 102, 75, 19, 52, 99, 81, 24, 44, 96, 86,  30,
       37, 91,  91, 37, 30, 86,  96, 44, 24, 81, 99, 52, 19, 75, 102, 60},
      // Coefficients 6 taps
      {25, 62, 82, 62, 25, 0,  21, 57, 80, 65, 29, 4,  17, 52, 78, 69, 34, 6,  14, 47, 77, 72, 38, 8,
       11, 43, 74, 74, 43, 11, 8,  38, 72, 77, 47, 14, 6,  34, 69, 78, 52, 17, 4,  29, 65, 80, 57, 21},
    },

    // Band_8_3472
    {
      // Scaler inc range
      {3461, 3472},
      // Coefficients 4 taps
      {73, 110, 73, 0,  60, 102, 75, 19, 52, 98, 81, 25, 44, 96, 86,  30,
       37, 91,  91, 37, 30, 86,  96, 44, 25, 81, 98, 52, 19, 75, 102, 60},
      // Coefficients 6 taps
      {25, 62, 82, 62, 25, 0,  21, 57, 80, 65, 29, 4,  17, 52, 78, 69, 34, 6,  14, 47, 77, 72, 38, 8,
       11, 43, 74, 74, 43, 11, 8,  38, 72, 77, 47, 14, 6,  34, 69, 78, 52, 17, 4,  29, 65, 80, 57, 21},
    },

    // Band_8_3484
    {
      // Scaler inc range
      {3472, 3484},
      // Coefficients 4 taps
      {73, 110, 73, 0,  60, 102, 75, 19, 52, 98, 81, 25, 44, 95, 86,  31,
       37, 91,  91, 37, 31, 86,  95, 44, 25, 81, 98, 52, 19, 75, 102, 60},
      // Coefficients 6 taps
      {25, 62, 82, 62, 25, 0,  21, 57, 80, 65, 29, 4,  17, 52, 79, 68, 34, 6,  14, 47, 77, 72, 38, 8,
       11, 43, 74, 74, 43, 11, 8,  38, 72, 77, 47, 14, 6,  34, 68, 79, 52, 17, 4,  29, 65, 80, 57, 21},
    },

    // Band_8_3495
    {
      // Scaler inc range
      {3484, 3495},
      // Coefficients 4 taps
      {73, 110, 73, 0,  60, 102, 75, 19, 52, 98, 81, 25, 44, 95, 86,  31,
       37, 91,  91, 37, 31, 86,  95, 44, 25, 81, 98, 52, 19, 75, 102, 60},
      // Coefficients 6 taps
      {26, 62, 80, 62, 26, 0,  21, 57, 80, 65, 29, 4,  17, 52, 79, 68, 34, 6,  14, 47, 77, 71, 38, 9,
       11, 43, 74, 74, 43, 11, 9,  38, 71, 77, 47, 14, 6,  34, 68, 79, 52, 17, 4,  29, 65, 80, 57, 21},
    },

    // Band_8_3507
    {
      // Scaler inc range
      {3495, 3507},
      // Coefficients 4 taps
      {73, 110, 73, 0,  60, 102, 75, 19, 52, 99, 80, 25, 44, 95, 86,  31,
       37, 91,  91, 37, 31, 86,  95, 44, 25, 80, 99, 52, 19, 75, 102, 60},
      // Coefficients 6 taps
      {26, 62, 80, 62, 26, 0,  21, 57, 79, 65, 29, 5,  18, 52, 78, 68, 34, 6,  14, 47, 77, 71, 38, 9,
       11, 43, 74, 74, 43, 11, 9,  38, 71, 77, 47, 14, 6,  34, 68, 78, 52, 18, 5,  29, 65, 79, 57, 21},
    },

    // Band_8_3519
    {
      // Scaler inc range
      {3507, 3519},
      // Coefficients 4 taps
      {73, 110, 73, 0,  60, 102, 75, 19, 52, 99, 80, 25, 44, 95, 86,  31,
       37, 91,  91, 37, 31, 86,  95, 44, 25, 80, 99, 52, 19, 75, 102, 60},
      // Coefficients 6 taps
      {26, 62, 80, 62, 26, 0,  21, 57, 78, 65, 30, 5,  18, 52, 78, 68, 34, 6,  14, 48, 76, 71, 38, 9,
       11, 43, 74, 74, 43, 11, 9,  38, 71, 76, 48, 14, 6,  34, 68, 78, 52, 18, 5,  30, 65, 78, 57, 21},
    },

    // Band_8_3531
    {
      // Scaler inc range
      {3519, 3531},
      // Coefficients 4 taps
      {73, 110, 73, 0,  60, 101, 75, 20, 52, 99, 80, 25, 44, 95, 86,  31,
       37, 91,  91, 37, 31, 86,  95, 44, 25, 80, 99, 52, 20, 75, 101, 60},
      // Coefficients 6 taps
      {26, 62, 80, 62, 26, 0,  21, 57, 78, 65, 30, 5,  18, 52, 77, 68, 34, 7,  14, 48, 76, 71, 38, 9,
       11, 43, 74, 74, 43, 11, 9,  38, 71, 76, 48, 14, 7,  34, 68, 77, 52, 18, 5,  30, 65, 78, 57, 21},
    },

    // Band_8_3542
    {
      // Scaler inc range
      {3531, 3542},
      // Coefficients 4 taps
      {73, 110, 73, 0,  60, 101, 75, 20, 52, 99, 80, 25, 45, 94, 86,  31,
       37, 91,  91, 37, 31, 86,  94, 45, 25, 80, 99, 52, 20, 75, 101, 60},
      // Coefficients 6 taps
      {26, 62, 80, 62, 26, 0,  22, 57, 77, 65, 30, 5,  18, 52, 77, 68, 34, 7,  14, 48, 76, 71, 38, 9,
       11, 43, 74, 74, 43, 11, 9,  38, 71, 76, 48, 14, 7,  34, 68, 77, 52, 18, 5,  30, 65, 77, 57, 22},
    },

    // Band_8_3554
    {
      // Scaler inc range
      {3542, 3554},
      // Coefficients 4 taps
      {73, 110, 73, 0,  60, 101, 75, 20, 52, 99, 80, 25, 45, 94, 86,  31,
       37, 91,  91, 37, 31, 86,  94, 45, 25, 80, 99, 52, 20, 75, 101, 60},
      // Coefficients 6 taps
      {26, 62, 80, 62, 26, 0,  22, 57, 77, 65, 30, 5,  18, 52, 77, 68, 34, 7,  15, 48, 75, 71, 38, 9,
       12, 43, 73, 73, 43, 12, 9,  38, 71, 75, 48, 15, 7,  34, 68, 77, 52, 18, 5,  30, 65, 77, 57, 22},
    },

    // Band_8_3567
    {
      // Scaler inc range
      {3554, 3567},
      // Coefficients 4 taps
      {73, 110, 73, 0,  60, 101, 75, 20, 52, 99, 80, 25, 45, 94, 86,  31,
       38, 90,  90, 38, 31, 86,  94, 45, 25, 80, 99, 52, 20, 75, 101, 60},
      // Coefficients 6 taps
      {26, 62, 80, 62, 26, 0,  22, 57, 77, 65, 30, 5,  18, 52, 77, 68, 34, 7,  15, 48, 75, 71, 38, 9,
       12, 43, 73, 73, 43, 12, 9,  38, 71, 75, 48, 15, 7,  34, 68, 77, 52, 18, 5,  30, 65, 77, 57, 22},
    },

    // Band_8_3579
    {
      // Scaler inc range
      {3567, 3579},
      // Coefficients 4 taps
      {73, 110, 73, 0,  60, 101, 75, 20, 52, 99, 80, 25, 45, 94, 86,  31,
       38, 90,  90, 38, 31, 86,  94, 45, 25, 80, 99, 52, 20, 75, 101, 60},
      // Coefficients 6 taps
      {26, 62, 80, 62, 26, 0,  22, 57, 77, 65, 30, 5,  18, 52, 77, 68, 34, 7,  15, 48, 75, 71, 38, 9,
       12, 43, 73, 73, 43, 12, 9,  38, 71, 75, 48, 15, 7,  34, 68, 77, 52, 18, 5,  30, 65, 77, 57, 22},
    },

    // Band_8_3591
    {
      // Scaler inc range
      {3579, 3591},
      // Coefficients 4 taps
      {73, 110, 73, 0,  60, 101, 75, 20, 52, 99, 80, 25, 45, 94, 86,  31,
       38, 90,  90, 38, 31, 86,  94, 45, 25, 80, 99, 52, 20, 75, 101, 60},
      // Coefficients 6 taps
      {26, 62, 80, 62, 26, 0,  22, 57, 78, 64, 30, 5,  18, 52, 77, 68, 34, 7,  15, 48, 74, 71, 39, 9,
       12, 43, 73, 73, 43, 12, 9,  39, 71, 74, 48, 15, 7,  34, 68, 77, 52, 18, 5,  30, 64, 78, 57, 22},
    },

    // Band_8_3603
    {
      // Scaler inc range
      {3591, 3603},
      // Coefficients 4 taps
      {73, 110, 73, 0,  60, 101, 75, 20, 52, 99, 80, 25, 45, 94, 86,  31,
       38, 90,  90, 38, 31, 86,  94, 45, 25, 80, 99, 52, 20, 75, 101, 60},
      // Coefficients 6 taps
      {26, 62, 80, 62, 26, 0,  22, 57, 78, 64, 30, 5,  18, 52, 77, 68, 34, 7,  15, 48, 74, 71, 39, 9,
       12, 43, 73, 73, 43, 12, 9,  39, 71, 74, 48, 15, 7,  34, 68, 77, 52, 18, 5,  30, 64, 78, 57, 22},
    },

    // Band_8_3616
    {
      // Scaler inc range
      {3603, 3616},
      // Coefficients 4 taps
      {73, 110, 73, 0,  60, 101, 75, 20, 52, 99, 80, 25, 45, 95, 85,  31,
       38, 90,  90, 38, 31, 85,  95, 45, 25, 80, 99, 52, 20, 75, 101, 60},
      // Coefficients 6 taps
      {26, 62, 80, 62, 26, 0,  22, 57, 78, 64, 30, 5,  18, 52, 77, 68, 34, 7,  15, 48, 75, 70, 39, 9,
       12, 43, 73, 73, 43, 12, 9,  39, 70, 75, 48, 15, 7,  34, 68, 77, 52, 18, 5,  30, 64, 78, 57, 22},
    },

    // Band_8_3628
    {
      // Scaler inc range
      {3616, 3628},
      // Coefficients 4 taps
      {73, 110, 73, 0,  60, 101, 75, 20, 52, 99, 80, 25, 45, 95, 85,  31,
       38, 90,  90, 38, 31, 85,  95, 45, 25, 80, 99, 52, 20, 75, 101, 60},
      // Coefficients 6 taps
      {27, 62, 78, 62, 27, 0,  22, 56, 79, 64, 30, 5,  18, 52, 78, 67, 34, 7,  15, 48, 74, 70, 39, 10,
       12, 43, 73, 73, 43, 12, 10, 39, 70, 74, 48, 15, 7,  34, 67, 78, 52, 18, 5,  30, 64, 79, 56, 22},
    },

    // Band_8_3641
    {
      // Scaler inc range
      {3628, 3641},
      // Coefficients 4 taps
      {73, 110, 73, 0,  60, 101, 75, 20, 52, 99, 80, 25, 45, 95, 85,  31,
       38, 90,  90, 38, 31, 85,  95, 45, 25, 80, 99, 52, 20, 75, 101, 60},
      // Coefficients 6 taps
      {27, 62, 78, 62, 27, 0,  22, 56, 79, 64, 30, 5,  19, 52, 77, 67, 34, 7,  15, 48, 74, 70, 39, 10,
       12, 43, 73, 73, 43, 12, 10, 39, 70, 74, 48, 15, 7,  34, 67, 77, 52, 19, 5,  30, 64, 79, 56, 22},
    },

    // Band_8_3654
    {
      // Scaler inc range
      {3641, 3654},
      // Coefficients 4 taps
      {73, 110, 73, 0,  60, 101, 75, 20, 52, 98, 80, 26, 45, 95, 85,  31,
       38, 90,  90, 38, 31, 85,  95, 45, 26, 80, 98, 52, 20, 75, 101, 60},
      // Coefficients 6 taps
      {27, 62, 78, 62, 27, 0,  22, 56, 79, 64, 30, 5,  19, 52, 77, 67, 34, 7,  15, 48, 74, 70, 39, 10,
       12, 43, 73, 73, 43, 12, 10, 39, 70, 74, 48, 15, 7,  34, 67, 77, 52, 19, 5,  30, 64, 79, 56, 22},
    },

    // Band_8_3666
    {
      // Scaler inc range
      {3654, 3666},
      // Coefficients 4 taps
      {73, 110, 73, 0,  60, 101, 75, 20, 53, 97, 80, 26, 45, 95, 85,  31,
       38, 90,  90, 38, 31, 85,  95, 45, 26, 80, 97, 53, 20, 75, 101, 60},
      // Coefficients 6 taps
      {27, 62, 78, 62, 27, 0,  22, 56, 78, 64, 30, 6,  19, 52, 77, 67, 34, 7,  15, 48, 74, 70, 39, 10,
       12, 43, 73, 73, 43, 12, 10, 39, 70, 74, 48, 15, 7,  34, 67, 77, 52, 19, 6,  30, 64, 78, 56, 22},
    },

    // Band_8_3679
    {
      // Scaler inc range
      {3666, 3679},
      // Coefficients 4 taps
      {73, 110, 73, 0,  60, 101, 75, 20, 53, 97, 80, 26, 45, 94, 85,  32,
       38, 90,  90, 38, 32, 85,  94, 45, 26, 80, 97, 53, 20, 75, 101, 60},
      // Coefficients 6 taps
      {27, 62, 78, 62, 27, 0,  22, 56, 78, 64, 30, 6,  19, 52, 75, 67, 35, 8,  15, 48, 74, 70, 39, 10,
       12, 43, 73, 73, 43, 12, 10, 39, 70, 74, 48, 15, 8,  35, 67, 75, 52, 19, 6,  30, 64, 78, 56, 22},
    },

    // Band_8_3692
    {
      // Scaler inc range
      {3679, 3692},
      // Coefficients 4 taps
      {73, 110, 73, 0,  60, 101, 75, 20, 53, 97, 80, 26, 45, 94, 85,  32,
       38, 90,  90, 38, 32, 85,  94, 45, 26, 80, 97, 53, 20, 75, 101, 60},
      // Coefficients 6 taps
      {27, 62, 78, 62, 27, 0,  23, 56, 77, 64, 30, 6,  19, 52, 75, 67, 35, 8,  16, 48, 73, 70, 39, 10,
       13, 43, 72, 72, 43, 13, 10, 39, 70, 73, 48, 16, 8,  35, 67, 75, 52, 19, 6,  30, 64, 77, 56, 23},
    },

    // Band_8_3705
    {
      // Scaler inc range
      {3692, 3705},
      // Coefficients 4 taps
      {73, 110, 73, 0,  61, 100, 75, 20, 53, 97, 80, 26, 45, 94, 85,  32,
       38, 90,  90, 38, 32, 85,  94, 45, 26, 80, 97, 53, 20, 75, 100, 61},
      // Coefficients 6 taps
      {27, 62, 78, 62, 27, 0,  23, 56, 76, 64, 31, 6,  19, 52, 75, 67, 35, 8,  16, 48, 73, 70, 39, 10,
       13, 43, 72, 72, 43, 13, 10, 39, 70, 73, 48, 16, 8,  35, 67, 75, 52, 19, 6,  31, 64, 76, 56, 23},
    },

    // Band_8_3718
    {
      // Scaler inc range
      {3705, 3718},
      // Coefficients 4 taps
      {73, 110, 73, 0,  61, 100, 75, 20, 53, 97, 80, 26, 45, 94, 85,  32,
       38, 90,  90, 38, 32, 85,  94, 45, 26, 80, 97, 53, 20, 75, 100, 61},
      // Coefficients 6 taps
      {27, 62, 78, 62, 27, 0,  23, 56, 76, 64, 31, 6,  19, 52, 75, 67, 35, 8,  16, 48, 73, 70, 39, 10,
       13, 43, 72, 72, 43, 13, 10, 39, 70, 73, 48, 16, 8,  35, 67, 75, 52, 19, 6,  31, 64, 76, 56, 23},
    },

    // Band_8_3732
    {
      // Scaler inc range
      {3718, 3732},
      // Coefficients 4 taps
      {73, 110, 73, 0,  61, 99, 75, 21, 53, 97, 80, 26, 45, 94, 85, 32,
       38, 90,  90, 38, 32, 85, 94, 45, 26, 80, 97, 53, 21, 75, 99, 61},
      // Coefficients 6 taps
      {27, 61, 80, 61, 27, 0,  23, 56, 76, 64, 31, 6,  19, 52, 75, 67, 35, 8,  16, 48, 73, 70, 39, 10,
       13, 43, 72, 72, 43, 13, 10, 39, 70, 73, 48, 16, 8,  35, 67, 75, 52, 19, 6,  31, 64, 76, 56, 23},
    },

    // Band_8_3745
    {
      // Scaler inc range
      {3732, 3745},
      // Coefficients 4 taps
      {73, 110, 73, 0,  61, 99, 75, 21, 53, 97, 80, 26, 45, 94, 85, 32,
       38, 90,  90, 38, 32, 85, 94, 45, 26, 80, 97, 53, 21, 75, 99, 61},
      // Coefficients 6 taps
      {27, 61, 80, 61, 27, 0,  23, 56, 76, 64, 31, 6,  19, 52, 75, 67, 35, 8,  16, 48, 74, 69, 39, 10,
       13, 43, 72, 72, 43, 13, 10, 39, 69, 74, 48, 16, 8,  35, 67, 75, 52, 19, 6,  31, 64, 76, 56, 23},
    },

    // Band_8_3758
    {
      // Scaler inc range
      {3745, 3758},
      // Coefficients 4 taps
      {73, 110, 73, 0,  61, 99, 75, 21, 53, 97, 80, 26, 45, 94, 85, 32,
       38, 90,  90, 38, 32, 85, 94, 45, 26, 80, 97, 53, 21, 75, 99, 61},
      // Coefficients 6 taps
      {27, 61, 80, 61, 27, 0,  23, 56, 76, 64, 31, 6,  19, 52, 75, 67, 35, 8,  16, 48, 74, 69, 39, 10,
       13, 43, 72, 72, 43, 13, 10, 39, 69, 74, 48, 16, 8,  35, 67, 75, 52, 19, 6,  31, 64, 76, 56, 23},
    },

    // Band_8_3772
    {
      // Scaler inc range
      {3758, 3772},
      // Coefficients 4 taps
      {73, 110, 73, 0,  61, 99, 75, 21, 53, 97, 80, 26, 45, 94, 85, 32,
       38, 90,  90, 38, 32, 85, 94, 45, 26, 80, 97, 53, 21, 75, 99, 61},
      // Coefficients 6 taps
      {27, 61, 80, 61, 27, 0,  23, 56, 76, 64, 31, 6,  19, 52, 75, 67, 35, 8,  16, 48, 74, 69, 39, 10,
       13, 43, 72, 72, 43, 13, 10, 39, 69, 74, 48, 16, 8,  35, 67, 75, 52, 19, 6,  31, 64, 76, 56, 23},
    },

    // Band_8_3785
    {
      // Scaler inc range
      {3772, 3785},
      // Coefficients 4 taps
      {73, 110, 73, 0,  61, 99, 75, 21, 53, 97, 80, 26, 45, 94, 85, 32,
       38, 90,  90, 38, 32, 85, 94, 45, 26, 80, 97, 53, 21, 75, 99, 61},
      // Coefficients 6 taps
      {28, 61, 78, 61, 28, 0,  23, 56, 76, 64, 31, 6,  20, 52, 75, 66, 35, 8,  16, 48, 73, 69, 39, 11,
       13, 43, 72, 72, 43, 13, 11, 39, 69, 73, 48, 16, 8,  35, 66, 75, 52, 20, 6,  31, 64, 76, 56, 23},
    },

    // Band_8_3799
    {
      // Scaler inc range
      {3785, 3799},
      // Coefficients 4 taps
      {73, 110, 73, 0,  61, 99, 75, 21, 53, 97, 80, 26, 45, 94, 85, 32,
       39, 89,  89, 39, 32, 85, 94, 45, 26, 80, 97, 53, 21, 75, 99, 61},
      // Coefficients 6 taps
      {28, 61, 78, 61, 28, 0,  23, 56, 77, 63, 31, 6,  20, 52, 75, 66, 35, 8,  16, 48, 73, 69, 39, 11,
       13, 43, 72, 72, 43, 13, 11, 39, 69, 73, 48, 16, 8,  35, 66, 75, 52, 20, 6,  31, 63, 77, 56, 23},
    },

    // Band_8_3813
    {
      // Scaler inc range
      {3799, 3813},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 99, 75, 21, 53, 97, 80, 26, 46, 93, 85, 32,
       39, 89,  89, 39, 32, 85, 93, 46, 26, 80, 97, 53, 21, 75, 99, 61},
      // Coefficients 6 taps
      {28, 61, 78, 61, 28, 0,  23, 56, 77, 63, 31, 6,  20, 52, 75, 66, 35, 8,  16, 48, 73, 69, 39, 11,
       13, 43, 72, 72, 43, 13, 11, 39, 69, 73, 48, 16, 8,  35, 66, 75, 52, 20, 6,  31, 63, 77, 56, 23},
    },

    // Band_8_3827
    {
      // Scaler inc range
      {3813, 3827},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 99, 75, 21, 53, 97, 80, 26, 46, 93, 85, 32,
       39, 89,  89, 39, 32, 85, 93, 46, 26, 80, 97, 53, 21, 75, 99, 61},
      // Coefficients 6 taps
      {28, 61, 78, 61, 28, 0,  23, 56, 77, 63, 31, 6,  20, 52, 75, 66, 35, 8,  16, 48, 73, 69, 39, 11,
       13, 43, 72, 72, 43, 13, 11, 39, 69, 73, 48, 16, 8,  35, 66, 75, 52, 20, 6,  31, 63, 77, 56, 23},
    },

    // Band_8_3841
    {
      // Scaler inc range
      {3827, 3841},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 99, 75, 21, 53, 97, 80, 26, 46, 93, 85, 32,
       39, 89,  89, 39, 32, 85, 93, 46, 26, 80, 97, 53, 21, 75, 99, 61},
      // Coefficients 6 taps
      {28, 61, 78, 61, 28, 0,  23, 56, 76, 63, 31, 7,  20, 52, 74, 66, 35, 9,  17, 48, 72, 69, 39, 11,
       14, 43, 71, 71, 43, 14, 11, 39, 69, 72, 48, 17, 9,  35, 66, 74, 52, 20, 7,  31, 63, 76, 56, 23},
    },

    // Band_8_3855
    {
      // Scaler inc range
      {3841, 3855},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 100, 74, 21, 53, 97, 80, 26, 46, 93, 85,  32,
       39, 89,  89, 39, 32, 85,  93, 46, 26, 80, 97, 53, 21, 74, 100, 61},
      // Coefficients 6 taps
      {28, 61, 78, 61, 28, 0,  24, 56, 75, 63, 31, 7,  20, 52, 74, 66, 35, 9,  17, 48, 72, 69, 39, 11,
       14, 43, 71, 71, 43, 14, 11, 39, 69, 72, 48, 17, 9,  35, 66, 74, 52, 20, 7,  31, 63, 75, 56, 24},
    },

    // Band_8_3869
    {
      // Scaler inc range
      {3855, 3869},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 100, 74, 21, 53, 96, 80, 27, 46, 93, 85,  32,
       39, 89,  89, 39, 32, 85,  93, 46, 27, 80, 96, 53, 21, 74, 100, 61},
      // Coefficients 6 taps
      {28, 61, 78, 61, 28, 0,  24, 56, 75, 63, 31, 7,  20, 52, 74, 66, 35, 9,  17, 48, 72, 69, 39, 11,
       14, 43, 71, 71, 43, 14, 11, 39, 69, 72, 48, 17, 9,  35, 66, 74, 52, 20, 7,  31, 63, 75, 56, 24},
    },

    // Band_8_3884
    {
      // Scaler inc range
      {3869, 3884},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 100, 74, 21, 53, 96, 80, 27, 46, 93, 85,  32,
       39, 89,  89, 39, 32, 85,  93, 46, 27, 80, 96, 53, 21, 74, 100, 61},
      // Coefficients 6 taps
      {28, 61, 78, 61, 28, 0,  24, 56, 75, 63, 31, 7,  20, 52, 74, 66, 35, 9,  17, 48, 73, 68, 39, 11,
       14, 43, 71, 71, 43, 14, 11, 39, 68, 73, 48, 17, 9,  35, 66, 74, 52, 20, 7,  31, 63, 75, 56, 24},
    },

    // Band_8_3898
    {
      // Scaler inc range
      {3884, 3898},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 100, 74, 21, 53, 96, 80, 27, 46, 93, 85,  32,
       39, 89,  89, 39, 32, 85,  93, 46, 27, 80, 96, 53, 21, 74, 100, 61},
      // Coefficients 6 taps
      {28, 61, 78, 61, 28, 0,  24, 56, 75, 63, 31, 7,  20, 52, 74, 66, 35, 9,  17, 48, 73, 68, 39, 11,
       14, 44, 70, 70, 44, 14, 11, 39, 68, 73, 48, 17, 9,  35, 66, 74, 52, 20, 7,  31, 63, 75, 56, 24},
    },

    // Band_8_3913
    {
      // Scaler inc range
      {3898, 3913},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 100, 74, 21, 53, 96, 80, 27, 46, 92, 85,  33,
       39, 89,  89, 39, 33, 85,  92, 46, 27, 80, 96, 53, 21, 74, 100, 61},
      // Coefficients 6 taps
      {28, 61, 78, 61, 28, 0,  24, 56, 75, 63, 31, 7,  20, 52, 74, 66, 35, 9,  17, 48, 73, 68, 39, 11,
       14, 44, 70, 70, 44, 14, 11, 39, 68, 73, 48, 17, 9,  35, 66, 74, 52, 20, 7,  31, 63, 75, 56, 24},
    },

    // Band_8_3927
    {
      // Scaler inc range
      {3913, 3927},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 100, 74, 21, 53, 96, 80, 27, 46, 92, 85,  33,
       39, 89,  89, 39, 33, 85,  92, 46, 27, 80, 96, 53, 21, 74, 100, 61},
      // Coefficients 6 taps
      {28, 61, 78, 61, 28, 0,  24, 56, 74, 63, 32, 7,  20, 52, 74, 66, 35, 9,  17, 48, 73, 68, 39, 11,
       14, 44, 70, 70, 44, 14, 11, 39, 68, 73, 48, 17, 9,  35, 66, 74, 52, 20, 7,  32, 63, 74, 56, 24},
    },

    // Band_8_3942
    {
      // Scaler inc range
      {3927, 3942},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 100, 74, 21, 53, 96, 80, 27, 46, 92, 85,  33,
       39, 89,  89, 39, 33, 85,  92, 46, 27, 80, 96, 53, 21, 74, 100, 61},
      // Coefficients 6 taps
      {28, 61, 78, 61, 28, 0,  24, 56, 74, 63, 32, 7,  20, 52, 74, 66, 35, 9,  17, 48, 73, 68, 39, 11,
       14, 44, 70, 70, 44, 14, 11, 39, 68, 73, 48, 17, 9,  35, 66, 74, 52, 20, 7,  32, 63, 74, 56, 24},
    },

    // Band_8_3957
    {
      // Scaler inc range
      {3942, 3957},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 99, 74, 22, 53, 96, 80, 27, 46, 93, 84, 33,
       39, 89,  89, 39, 33, 84, 93, 46, 27, 80, 96, 53, 22, 74, 99, 61},
      // Coefficients 6 taps
      {29, 61, 76, 61, 29, 0,  24, 56, 74, 63, 32, 7,  20, 52, 73, 66, 36, 9,  17, 48, 71, 68, 40, 12,
       14, 44, 70, 70, 44, 14, 12, 40, 68, 71, 48, 17, 9,  36, 66, 73, 52, 20, 7,  32, 63, 74, 56, 24},
    },

    // Band_8_3972
    {
      // Scaler inc range
      {3957, 3972},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 99, 74, 22, 53, 96, 80, 27, 46, 93, 84, 33,
       39, 89,  89, 39, 33, 84, 93, 46, 27, 80, 96, 53, 22, 74, 99, 61},
      // Coefficients 6 taps
      {29, 61, 76, 61, 29, 0,  24, 56, 74, 63, 32, 7,  21, 52, 73, 65, 36, 9,  17, 48, 71, 68, 40, 12,
       14, 44, 70, 70, 44, 14, 12, 40, 68, 71, 48, 17, 9,  36, 65, 73, 52, 21, 7,  32, 63, 74, 56, 24},
    },

    // Band_8_3987
    {
      // Scaler inc range
      {3972, 3987},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 99, 74, 22, 53, 96, 80, 27, 46, 93, 84, 33,
       39, 89,  89, 39, 33, 84, 93, 46, 27, 80, 96, 53, 22, 74, 99, 61},
      // Coefficients 6 taps
      {29, 61, 76, 61, 29, 0,  24, 56, 74, 63, 32, 7,  21, 52, 73, 65, 36, 9,  17, 48, 71, 68, 40, 12,
       14, 44, 70, 70, 44, 14, 12, 40, 68, 71, 48, 17, 9,  36, 65, 73, 52, 21, 7,  32, 63, 74, 56, 24},
    },

    // Band_8_4002
    {
      // Scaler inc range
      {3987, 4002},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 99, 74, 22, 53, 96, 80, 27, 46, 93, 84, 33,
       39, 89,  89, 39, 33, 84, 93, 46, 27, 80, 96, 53, 22, 74, 99, 61},
      // Coefficients 6 taps
      {29, 61, 76, 61, 29, 0,  24, 56, 74, 63, 32, 7,  21, 52, 73, 65, 36, 9,  17, 48, 71, 68, 40, 12,
       14, 44, 70, 70, 44, 14, 12, 40, 68, 71, 48, 17, 9,  36, 65, 73, 52, 21, 7,  32, 63, 74, 56, 24},
    },

    // Band_8_4018
    {
      // Scaler inc range
      {4002, 4018},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 99, 74, 22, 53, 96, 80, 27, 46, 93, 84, 33,
       39, 89,  89, 39, 33, 84, 93, 46, 27, 80, 96, 53, 22, 74, 99, 61},
      // Coefficients 6 taps
      {29, 61, 76, 61, 29, 0,  24, 56, 74, 63, 32, 7,  21, 52, 72, 65, 36, 10, 18, 48, 70, 68, 40, 12,
       15, 44, 69, 69, 44, 15, 12, 40, 68, 70, 48, 18, 10, 36, 65, 72, 52, 21, 7,  32, 63, 74, 56, 24},
    },

    // Band_8_4033
    {
      // Scaler inc range
      {4018, 4033},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 99, 74, 22, 53, 96, 80, 27, 46, 93, 84, 33,
       39, 89,  89, 39, 33, 84, 93, 46, 27, 80, 96, 53, 22, 74, 99, 61},
      // Coefficients 6 taps
      {29, 61, 76, 61, 29, 0,  24, 56, 74, 62, 32, 8,  21, 52, 72, 65, 36, 10, 18, 48, 70, 68, 40, 12,
       15, 44, 69, 69, 44, 15, 12, 40, 68, 70, 48, 18, 10, 36, 65, 72, 52, 21, 8,  32, 62, 74, 56, 24},
    },

    // Band_8_4049
    {
      // Scaler inc range
      {4033, 4049},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 99, 74, 22, 53, 96, 80, 27, 46, 93, 84, 33,
       39, 89,  89, 39, 33, 84, 93, 46, 27, 80, 96, 53, 22, 74, 99, 61},
      // Coefficients 6 taps
      {29, 61, 76, 61, 29, 0,  25, 56, 73, 62, 32, 8,  21, 52, 72, 65, 36, 10, 18, 48, 71, 67, 40, 12,
       15, 44, 69, 69, 44, 15, 12, 40, 67, 71, 48, 18, 10, 36, 65, 72, 52, 21, 8,  32, 62, 73, 56, 25},
    },

    // Band_8_4064
    {
      // Scaler inc range
      {4049, 4064},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 99, 74, 22, 53, 97, 79, 27, 46, 93, 84, 33,
       39, 89,  89, 39, 33, 84, 93, 46, 27, 79, 97, 53, 22, 74, 99, 61},
      // Coefficients 6 taps
      {29, 61, 76, 61, 29, 0,  25, 56, 73, 62, 32, 8,  21, 52, 72, 65, 36, 10, 18, 48, 71, 67, 40, 12,
       15, 44, 69, 69, 44, 15, 12, 40, 67, 71, 48, 18, 10, 36, 65, 72, 52, 21, 8,  32, 62, 73, 56, 25},
    },

    // Band_8_4080
    {
      // Scaler inc range
      {4064, 4080},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 99, 74, 22, 54, 96, 79, 27, 46, 93, 84, 33,
       40, 88,  88, 40, 33, 84, 93, 46, 27, 79, 96, 54, 22, 74, 99, 61},
      // Coefficients 6 taps
      {29, 61, 76, 61, 29, 0,  25, 56, 73, 62, 32, 8,  21, 52, 72, 65, 36, 10, 18, 48, 71, 67, 40, 12,
       15, 44, 69, 69, 44, 15, 12, 40, 67, 71, 48, 18, 10, 36, 65, 72, 52, 21, 8,  32, 62, 73, 56, 25},
    },

    // Band_8_4096
    {
      // Scaler inc range
      {4080, 4096},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 99, 74, 22, 54, 96, 79, 27, 46, 93, 84, 33,
       40, 88,  88, 40, 33, 84, 93, 46, 27, 79, 96, 54, 22, 74, 99, 61},
      // Coefficients 6 taps
      {29, 61, 76, 61, 29, 0,  25, 56, 73, 62, 32, 8,  21, 52, 72, 65, 36, 10, 18, 48, 71, 67, 40, 12,
       15, 44, 69, 69, 44, 15, 12, 40, 67, 71, 48, 18, 10, 36, 65, 72, 52, 21, 8,  32, 62, 73, 56, 25},
    },

    // Band_8_4112
    {
      // Scaler inc range
      {4096, 4112},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 99, 74, 22, 54, 96, 79, 27, 46, 93, 84, 33,
       40, 88,  88, 40, 33, 84, 93, 46, 27, 79, 96, 54, 22, 74, 99, 61},
      // Coefficients 6 taps
      {29, 61, 76, 61, 29, 0,  25, 56, 73, 62, 32, 8,  21, 52, 72, 65, 36, 10, 18, 48, 71, 67, 40, 12,
       15, 44, 69, 69, 44, 15, 12, 40, 67, 71, 48, 18, 10, 36, 65, 72, 52, 21, 8,  32, 62, 73, 56, 25},
    },

    // Band_8_4128
    {
      // Scaler inc range
      {4112, 4128},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 99, 74, 22, 54, 95, 79, 28, 46, 93, 84, 33,
       40, 88,  88, 40, 33, 84, 93, 46, 28, 79, 95, 54, 22, 74, 99, 61},
      // Coefficients 6 taps
      {29, 61, 76, 61, 29, 0,  25, 55, 74, 62, 32, 8,  21, 52, 72, 65, 36, 10, 18, 48, 71, 67, 40, 12,
       15, 44, 69, 69, 44, 15, 12, 40, 67, 71, 48, 18, 10, 36, 65, 72, 52, 21, 8,  32, 62, 74, 55, 25},
    },

    // Band_8_4145
    {
      // Scaler inc range
      {4128, 4145},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 99, 74, 22, 54, 95, 79, 28, 46, 93, 84, 33,
       40, 88,  88, 40, 33, 84, 93, 46, 28, 79, 95, 54, 22, 74, 99, 61},
      // Coefficients 6 taps
      {29, 61, 76, 61, 29, 0,  25, 55, 74, 62, 32, 8,  21, 52, 72, 65, 36, 10, 18, 48, 70, 67, 40, 13,
       15, 44, 69, 69, 44, 15, 13, 40, 67, 70, 48, 18, 10, 36, 65, 72, 52, 21, 8,  32, 62, 74, 55, 25},
    },

    // Band_8_4161
    {
      // Scaler inc range
      {4145, 4161},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 99, 74, 22, 54, 95, 79, 28, 47, 92, 84, 33,
       40, 88,  88, 40, 33, 84, 92, 47, 28, 79, 95, 54, 22, 74, 99, 61},
      // Coefficients 6 taps
      {30, 61, 74, 61, 30, 0,  25, 55, 74, 62, 32, 8,  22, 52, 71, 65, 36, 10, 18, 48, 70, 67, 40, 13,
       15, 44, 69, 69, 44, 15, 13, 40, 67, 70, 48, 18, 10, 36, 65, 71, 52, 22, 8,  32, 62, 74, 55, 25},
    },

    // Band_8_4178
    {
      // Scaler inc range
      {4161, 4178},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 99, 74, 22, 54, 95, 79, 28, 47, 91, 84, 34,
       40, 88,  88, 40, 34, 84, 91, 47, 28, 79, 95, 54, 22, 74, 99, 61},
      // Coefficients 6 taps
      {30, 61, 74, 61, 30, 0,  25, 55, 74, 62, 32, 8,  22, 52, 72, 64, 36, 10, 18, 48, 70, 67, 40, 13,
       15, 44, 69, 69, 44, 15, 13, 40, 67, 70, 48, 18, 10, 36, 64, 72, 52, 22, 8,  32, 62, 74, 55, 25},
    },

    // Band_8_4194
    {
      // Scaler inc range
      {4178, 4194},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 99, 74, 22, 54, 95, 79, 28, 47, 91, 84, 34,
       40, 88,  88, 40, 34, 84, 91, 47, 28, 79, 95, 54, 22, 74, 99, 61},
      // Coefficients 6 taps
      {30, 61, 74, 61, 30, 0,  25, 55, 73, 62, 33, 8,  22, 52, 72, 64, 36, 10, 18, 48, 70, 67, 40, 13,
       15, 44, 69, 69, 44, 15, 13, 40, 67, 70, 48, 18, 10, 36, 64, 72, 52, 22, 8,  33, 62, 73, 55, 25},
    },

    // Band_8_4211
    {
      // Scaler inc range
      {4194, 4211},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 99, 74, 22, 54, 95, 79, 28, 47, 91, 84, 34,
       40, 88,  88, 40, 34, 84, 91, 47, 28, 79, 95, 54, 22, 74, 99, 61},
      // Coefficients 6 taps
      {30, 61, 74, 61, 30, 0,  25, 55, 73, 62, 33, 8,  22, 52, 72, 64, 36, 10, 19, 48, 69, 67, 40, 13,
       16, 44, 68, 68, 44, 16, 13, 40, 67, 69, 48, 19, 10, 36, 64, 72, 52, 22, 8,  33, 62, 73, 55, 25},
    },

    // Band_8_4228
    {
      // Scaler inc range
      {4211, 4228},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 98, 74, 23, 54, 95, 79, 28, 47, 91, 84, 34,
       40, 88,  88, 40, 34, 84, 91, 47, 28, 79, 95, 54, 23, 74, 98, 61},
      // Coefficients 6 taps
      {30, 61, 74, 61, 30, 0,  25, 55, 73, 62, 33, 8,  22, 52, 71, 64, 36, 11, 19, 48, 69, 67, 40, 13,
       16, 44, 68, 68, 44, 16, 13, 40, 67, 69, 48, 19, 11, 36, 64, 71, 52, 22, 8,  33, 62, 73, 55, 25},
    },

    // Band_8_4245
    {
      // Scaler inc range
      {4228, 4245},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 98, 74, 23, 54, 95, 79, 28, 47, 91, 84, 34,
       40, 88,  88, 40, 34, 84, 91, 47, 28, 79, 95, 54, 23, 74, 98, 61},
      // Coefficients 6 taps
      {30, 61, 74, 61, 30, 0,  25, 55, 73, 62, 33, 8,  22, 52, 71, 64, 36, 11, 19, 48, 70, 66, 40, 13,
       16, 44, 68, 68, 44, 16, 13, 40, 66, 70, 48, 19, 11, 36, 64, 71, 52, 22, 8,  33, 62, 73, 55, 25},
    },

    // Band_8_4263
    {
      // Scaler inc range
      {4245, 4263},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 98, 74, 23, 54, 95, 79, 28, 47, 91, 84, 34,
       40, 88,  88, 40, 34, 84, 91, 47, 28, 79, 95, 54, 23, 74, 98, 61},
      // Coefficients 6 taps
      {30, 61, 74, 61, 30, 0,  25, 55, 72, 62, 33, 9,  22, 51, 72, 64, 36, 11, 19, 48, 70, 66, 40, 13,
       16, 44, 68, 68, 44, 16, 13, 40, 66, 70, 48, 19, 11, 36, 64, 72, 51, 22, 9,  33, 62, 72, 55, 25},
    },

    // Band_8_4280
    {
      // Scaler inc range
      {4263, 4280},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 98, 74, 23, 54, 95, 79, 28, 47, 91, 84, 34,
       40, 88,  88, 40, 34, 84, 91, 47, 28, 79, 95, 54, 23, 74, 98, 61},
      // Coefficients 6 taps
      {30, 60, 76, 60, 30, 0,  26, 55, 71, 62, 33, 9,  22, 51, 72, 64, 36, 11, 19, 48, 70, 66, 40, 13,
       16, 44, 68, 68, 44, 16, 13, 40, 66, 70, 48, 19, 11, 36, 64, 72, 51, 22, 9,  33, 62, 71, 55, 26},
    },

    // Band_8_4297
    {
      // Scaler inc range
      {4280, 4297},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 98, 74, 23, 54, 95, 79, 28, 47, 91, 84, 34,
       40, 88,  88, 40, 34, 84, 91, 47, 28, 79, 95, 54, 23, 74, 98, 61},
      // Coefficients 6 taps
      {30, 60, 76, 60, 30, 0,  26, 55, 71, 62, 33, 9,  22, 51, 72, 64, 36, 11, 19, 48, 70, 66, 40, 13,
       16, 44, 68, 68, 44, 16, 13, 40, 66, 70, 48, 19, 11, 36, 64, 72, 51, 22, 9,  33, 62, 71, 55, 26},
    },

    // Band_8_4315
    {
      // Scaler inc range
      {4297, 4315},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 98, 74, 23, 54, 95, 79, 28, 47, 91, 84, 34,
       40, 88,  88, 40, 34, 84, 91, 47, 28, 79, 95, 54, 23, 74, 98, 61},
      // Coefficients 6 taps
      {30, 60, 76, 60, 30, 0,  26, 55, 72, 61, 33, 9,  22, 51, 71, 64, 37, 11, 19, 48, 70, 66, 40, 13,
       16, 44, 68, 68, 44, 16, 13, 40, 66, 70, 48, 19, 11, 37, 64, 71, 51, 22, 9,  33, 61, 72, 55, 26},
    },

    // Band_8_4333
    {
      // Scaler inc range
      {4315, 4333},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 98, 74, 23, 54, 95, 79, 28, 47, 91, 84, 34,
       40, 88,  88, 40, 34, 84, 91, 47, 28, 79, 95, 54, 23, 74, 98, 61},
      // Coefficients 6 taps
      {30, 60, 76, 60, 30, 0,  26, 55, 72, 61, 33, 9,  22, 51, 71, 64, 37, 11, 19, 48, 70, 66, 40, 13,
       16, 44, 68, 68, 44, 16, 13, 40, 66, 70, 48, 19, 11, 37, 64, 71, 51, 22, 9,  33, 61, 72, 55, 26},
    },

    // Band_8_4351
    {
      // Scaler inc range
      {4333, 4351},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 98, 74, 23, 54, 95, 79, 28, 47, 91, 84, 34,
       40, 88,  88, 40, 34, 84, 91, 47, 28, 79, 95, 54, 23, 74, 98, 61},
      // Coefficients 6 taps
      {30, 60, 76, 60, 30, 0,  26, 55, 72, 61, 33, 9,  22, 51, 71, 64, 37, 11, 19, 48, 70, 66, 40, 13,
       16, 44, 68, 68, 44, 16, 13, 40, 66, 70, 48, 19, 11, 37, 64, 71, 51, 22, 9,  33, 61, 72, 55, 26},
    },

    // Band_8_4369
    {
      // Scaler inc range
      {4351, 4369},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 98, 74, 23, 54, 95, 79, 28, 47, 91, 84, 34,
       40, 88,  88, 40, 34, 84, 91, 47, 28, 79, 95, 54, 23, 74, 98, 61},
      // Coefficients 6 taps
      {30, 60, 76, 60, 30, 0,  26, 55, 72, 61, 33, 9,  22, 51, 71, 64, 37, 11, 19, 48, 69, 66, 40, 14,
       16, 44, 68, 68, 44, 16, 14, 40, 66, 69, 48, 19, 11, 37, 64, 71, 51, 22, 9,  33, 61, 72, 55, 26},
    },

    // Band_8_4387
    {
      // Scaler inc range
      {4369, 4387},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 98, 74, 23, 54, 95, 79, 28, 47, 91, 84, 34,
       40, 88,  88, 40, 34, 84, 91, 47, 28, 79, 95, 54, 23, 74, 98, 61},
      // Coefficients 6 taps
      {30, 60, 76, 60, 30, 0,  26, 55, 72, 61, 33, 9,  22, 51, 71, 64, 37, 11, 19, 48, 69, 66, 40, 14,
       16, 44, 68, 68, 44, 16, 14, 40, 66, 69, 48, 19, 11, 37, 64, 71, 51, 22, 9,  33, 61, 72, 55, 26},
    },

    // Band_8_4406
    {
      // Scaler inc range
      {4387, 4406},
      // Coefficients 4 taps
      {74, 108, 74, 0,  61, 98, 74, 23, 54, 95, 79, 28, 47, 91, 84, 34,
       40, 88,  88, 40, 34, 84, 91, 47, 28, 79, 95, 54, 23, 74, 98, 61},
      // Coefficients 6 taps
      {31, 60, 74, 60, 31, 0,  26, 55, 72, 61, 33, 9,  23, 51, 70, 64, 37, 11, 19, 48, 69, 66, 40, 14,
       16, 44, 68, 68, 44, 16, 14, 40, 66, 69, 48, 19, 11, 37, 64, 70, 51, 23, 9,  33, 61, 72, 55, 26},
    },

    // Band_8_4424
    {
      // Scaler inc range
      {4406, 4424},
      // Coefficients 4 taps
      {75, 106, 75, 0,  61, 98, 74, 23, 54, 95, 79, 28, 47, 92, 83, 34,
       40, 88,  88, 40, 34, 83, 92, 47, 28, 79, 95, 54, 23, 74, 98, 61},
      // Coefficients 6 taps
      {31, 60, 74, 60, 31, 0,  26, 55, 72, 61, 33, 9,  23, 51, 70, 64, 37, 11, 19, 48, 69, 66, 40, 14,
       16, 44, 68, 68, 44, 16, 14, 40, 66, 69, 48, 19, 11, 37, 64, 70, 51, 23, 9,  33, 61, 72, 55, 26},
    },

    // Band_8_4443
    {
      // Scaler inc range
      {4424, 4443},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 97, 74, 23, 54, 95, 79, 28, 47, 92, 83, 34,
       41, 87,  87, 41, 34, 83, 92, 47, 28, 79, 95, 54, 23, 74, 97, 62},
      // Coefficients 6 taps
      {31, 60, 74, 60, 31, 0,  26, 55, 72, 61, 33, 9,  23, 51, 71, 63, 37, 11, 19, 48, 69, 66, 40, 14,
       17, 44, 67, 67, 44, 17, 14, 40, 66, 69, 48, 19, 11, 37, 63, 71, 51, 23, 9,  33, 61, 72, 55, 26},
    },

    // Band_8_4462
    {
      // Scaler inc range
      {4443, 4462},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 97, 74, 23, 54, 94, 79, 29, 47, 92, 83, 34,
       41, 87,  87, 41, 34, 83, 92, 47, 29, 79, 94, 54, 23, 74, 97, 62},
      // Coefficients 6 taps
      {31, 60, 74, 60, 31, 0,  26, 55, 72, 61, 33, 9,  23, 51, 70, 63, 37, 12, 20, 48, 69, 65, 40, 14,
       17, 44, 67, 67, 44, 17, 14, 40, 65, 69, 48, 20, 12, 37, 63, 70, 51, 23, 9,  33, 61, 72, 55, 26},
    },

    // Band_8_4481
    {
      // Scaler inc range
      {4462, 4481},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 97, 74, 23, 54, 94, 79, 29, 47, 92, 83, 34,
       41, 87,  87, 41, 34, 83, 92, 47, 29, 79, 94, 54, 23, 74, 97, 62},
      // Coefficients 6 taps
      {31, 60, 74, 60, 31, 0,  26, 55, 72, 61, 33, 9,  23, 51, 70, 63, 37, 12, 20, 48, 69, 65, 40, 14,
       17, 44, 67, 67, 44, 17, 14, 40, 65, 69, 48, 20, 12, 37, 63, 70, 51, 23, 9,  33, 61, 72, 55, 26},
    },

    // Band_8_4500
    {
      // Scaler inc range
      {4481, 4500},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 97, 74, 23, 54, 94, 79, 29, 47, 92, 83, 34,
       41, 87,  87, 41, 34, 83, 92, 47, 29, 79, 94, 54, 23, 74, 97, 62},
      // Coefficients 6 taps
      {31, 60, 74, 60, 31, 0,  26, 55, 72, 61, 33, 9,  23, 51, 70, 63, 37, 12, 20, 48, 69, 65, 40, 14,
       17, 44, 67, 67, 44, 17, 14, 40, 65, 69, 48, 20, 12, 37, 63, 70, 51, 23, 9,  33, 61, 72, 55, 26},
    },

    // Band_8_4520
    {
      // Scaler inc range
      {4500, 4520},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 97, 74, 23, 54, 94, 79, 29, 47, 92, 83, 34,
       41, 87,  87, 41, 34, 83, 92, 47, 29, 79, 94, 54, 23, 74, 97, 62},
      // Coefficients 6 taps
      {31, 60, 74, 60, 31, 0,  26, 55, 71, 61, 33, 10, 23, 51, 70, 63, 37, 12, 20, 48, 69, 65, 40, 14,
       17, 44, 67, 67, 44, 17, 14, 40, 65, 69, 48, 20, 12, 37, 63, 70, 51, 23, 10, 33, 61, 71, 55, 26},
    },

    // Band_8_4539
    {
      // Scaler inc range
      {4520, 4539},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 97, 74, 23, 54, 94, 79, 29, 47, 91, 83, 35,
       41, 87,  87, 41, 35, 83, 91, 47, 29, 79, 94, 54, 23, 74, 97, 62},
      // Coefficients 6 taps
      {31, 60, 74, 60, 31, 0,  26, 55, 70, 61, 34, 10, 23, 51, 70, 63, 37, 12, 20, 48, 68, 65, 41, 14,
       17, 44, 67, 67, 44, 17, 14, 41, 65, 68, 48, 20, 12, 37, 63, 70, 51, 23, 10, 34, 61, 70, 55, 26},
    },

    // Band_8_4559
    {
      // Scaler inc range
      {4539, 4559},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 96, 74, 24, 54, 94, 79, 29, 47, 91, 83, 35,
       41, 87,  87, 41, 35, 83, 91, 47, 29, 79, 94, 54, 24, 74, 96, 62},
      // Coefficients 6 taps
      {31, 60, 74, 60, 31, 0,  27, 55, 69, 61, 34, 10, 23, 51, 70, 63, 37, 12, 20, 48, 68, 65, 41, 14,
       17, 44, 67, 67, 44, 17, 14, 41, 65, 68, 48, 20, 12, 37, 63, 70, 51, 23, 10, 34, 61, 69, 55, 27},
    },

    // Band_8_4579
    {
      // Scaler inc range
      {4559, 4579},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 96, 74, 24, 54, 94, 79, 29, 47, 91, 83, 35,
       41, 87,  87, 41, 35, 83, 91, 47, 29, 79, 94, 54, 24, 74, 96, 62},
      // Coefficients 6 taps
      {31, 60, 74, 60, 31, 0,  27, 55, 69, 61, 34, 10, 23, 51, 70, 63, 37, 12, 20, 48, 68, 65, 41, 14,
       17, 44, 67, 67, 44, 17, 14, 41, 65, 68, 48, 20, 12, 37, 63, 70, 51, 23, 10, 34, 61, 69, 55, 27},
    },

    // Band_8_4599
    {
      // Scaler inc range
      {4579, 4599},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 96, 74, 24, 54, 94, 79, 29, 47, 91, 83, 35,
       41, 87,  87, 41, 35, 83, 91, 47, 29, 79, 94, 54, 24, 74, 96, 62},
      // Coefficients 6 taps
      {31, 60, 74, 60, 31, 0,  27, 55, 69, 61, 34, 10, 23, 51, 70, 63, 37, 12, 20, 48, 68, 65, 41, 14,
       17, 44, 67, 67, 44, 17, 14, 41, 65, 68, 48, 20, 12, 37, 63, 70, 51, 23, 10, 34, 61, 69, 55, 27},
    },

    // Band_8_4619
    {
      // Scaler inc range
      {4599, 4619},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 96, 74, 24, 54, 94, 79, 29, 47, 91, 83, 35,
       41, 87,  87, 41, 35, 83, 91, 47, 29, 79, 94, 54, 24, 74, 96, 62},
      // Coefficients 6 taps
      {31, 60, 74, 60, 31, 0,  27, 55, 69, 61, 34, 10, 23, 51, 70, 63, 37, 12, 20, 48, 67, 65, 41, 15,
       17, 44, 67, 67, 44, 17, 15, 41, 65, 67, 48, 20, 12, 37, 63, 70, 51, 23, 10, 34, 61, 69, 55, 27},
    },

    // Band_8_4640
    {
      // Scaler inc range
      {4619, 4640},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 96, 74, 24, 54, 94, 79, 29, 48, 90, 83, 35,
       41, 87,  87, 41, 35, 83, 90, 48, 29, 79, 94, 54, 24, 74, 96, 62},
      // Coefficients 6 taps
      {31, 60, 74, 60, 31, 0,  27, 55, 69, 61, 34, 10, 23, 51, 70, 63, 37, 12, 20, 48, 67, 65, 41, 15,
       17, 44, 67, 67, 44, 17, 15, 41, 65, 67, 48, 20, 12, 37, 63, 70, 51, 23, 10, 34, 61, 69, 55, 27},
    },

    // Band_8_4660
    {
      // Scaler inc range
      {4640, 4660},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 96, 74, 24, 54, 94, 79, 29, 48, 90, 83, 35,
       41, 87,  87, 41, 35, 83, 90, 48, 29, 79, 94, 54, 24, 74, 96, 62},
      // Coefficients 6 taps
      {31, 60, 74, 60, 31, 0,  27, 55, 70, 60, 34, 10, 23, 51, 70, 63, 37, 12, 20, 48, 67, 65, 41, 15,
       17, 44, 67, 67, 44, 17, 15, 41, 65, 67, 48, 20, 12, 37, 63, 70, 51, 23, 10, 34, 60, 70, 55, 27},
    },

    // Band_8_4681
    {
      // Scaler inc range
      {4660, 4681},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 96, 74, 24, 54, 94, 79, 29, 48, 90, 83, 35,
       41, 87,  87, 41, 35, 83, 90, 48, 29, 79, 94, 54, 24, 74, 96, 62},
      // Coefficients 6 taps
      {32, 60, 72, 60, 32, 0,  27, 55, 70, 60, 34, 10, 24, 51, 69, 63, 37, 12, 20, 48, 67, 65, 41, 15,
       17, 44, 67, 67, 44, 17, 15, 41, 65, 67, 48, 20, 12, 37, 63, 69, 51, 24, 10, 34, 60, 70, 55, 27},
    },

    // Band_8_4702
    {
      // Scaler inc range
      {4681, 4702},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 96, 74, 24, 55, 93, 79, 29, 48, 90, 83, 35,
       41, 87,  87, 41, 35, 83, 90, 48, 29, 79, 93, 55, 24, 74, 96, 62},
      // Coefficients 6 taps
      {32, 60, 72, 60, 32, 0,  27, 55, 70, 60, 34, 10, 24, 51, 69, 63, 37, 12, 20, 48, 67, 65, 41, 15,
       18, 44, 66, 66, 44, 18, 15, 41, 65, 67, 48, 20, 12, 37, 63, 69, 51, 24, 10, 34, 60, 70, 55, 27},
    },

    // Band_8_4723
    {
      // Scaler inc range
      {4702, 4723},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 96, 74, 24, 55, 93, 79, 29, 48, 90, 83, 35,
       41, 87,  87, 41, 35, 83, 90, 48, 29, 79, 93, 55, 24, 74, 96, 62},
      // Coefficients 6 taps
      {32, 60, 72, 60, 32, 0,  27, 55, 70, 60, 34, 10, 24, 51, 69, 63, 37, 12, 21, 48, 67, 64, 41, 15,
       18, 44, 66, 66, 44, 18, 15, 41, 64, 67, 48, 21, 12, 37, 63, 69, 51, 24, 10, 34, 60, 70, 55, 27},
    },

    // Band_8_4745
    {
      // Scaler inc range
      {4723, 4745},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 96, 74, 24, 55, 93, 79, 29, 48, 90, 83, 35,
       41, 87,  87, 41, 35, 83, 90, 48, 29, 79, 93, 55, 24, 74, 96, 62},
      // Coefficients 6 taps
      {32, 60, 72, 60, 32, 0,  27, 55, 70, 60, 34, 10, 24, 51, 69, 62, 37, 13, 21, 48, 67, 64, 41, 15,
       18, 44, 66, 66, 44, 18, 15, 41, 64, 67, 48, 21, 13, 37, 62, 69, 51, 24, 10, 34, 60, 70, 55, 27},
    },

    // Band_8_4766
    {
      // Scaler inc range
      {4745, 4766},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 96, 74, 24, 55, 93, 79, 29, 48, 90, 83, 35,
       41, 87,  87, 41, 35, 83, 90, 48, 29, 79, 93, 55, 24, 74, 96, 62},
      // Coefficients 6 taps
      {32, 60, 72, 60, 32, 0,  27, 55, 70, 60, 34, 10, 24, 51, 69, 62, 37, 13, 21, 48, 67, 64, 41, 15,
       18, 44, 66, 66, 44, 18, 15, 41, 64, 67, 48, 21, 13, 37, 62, 69, 51, 24, 10, 34, 60, 70, 55, 27},
    },

    // Band_8_4788
    {
      // Scaler inc range
      {4766, 4788},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 96, 74, 24, 55, 93, 79, 29, 48, 90, 83, 35,
       41, 87,  87, 41, 35, 83, 90, 48, 29, 79, 93, 55, 24, 74, 96, 62},
      // Coefficients 6 taps
      {32, 60, 72, 60, 32, 0,  27, 55, 70, 60, 34, 10, 24, 51, 69, 62, 37, 13, 21, 48, 67, 64, 41, 15,
       18, 44, 66, 66, 44, 18, 15, 41, 64, 67, 48, 21, 13, 37, 62, 69, 51, 24, 10, 34, 60, 70, 55, 27},
    },

    // Band_8_4810
    {
      // Scaler inc range
      {4788, 4810},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 96, 74, 24, 55, 93, 79, 29, 48, 90, 83, 35,
       41, 87,  87, 41, 35, 83, 90, 48, 29, 79, 93, 55, 24, 74, 96, 62},
      // Coefficients 6 taps
      {32, 60, 72, 60, 32, 0,  27, 54, 70, 60, 34, 11, 24, 51, 69, 62, 37, 13, 21, 48, 67, 64, 41, 15,
       18, 44, 66, 66, 44, 18, 15, 41, 64, 67, 48, 21, 13, 37, 62, 69, 51, 24, 11, 34, 60, 70, 54, 27},
    },

    // Band_8_4832
    {
      // Scaler inc range
      {4810, 4832},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 96, 74, 24, 55, 93, 79, 29, 48, 90, 83, 35,
       41, 87,  87, 41, 35, 83, 90, 48, 29, 79, 93, 55, 24, 74, 96, 62},
      // Coefficients 6 taps
      {32, 60, 72, 60, 32, 0,  27, 54, 70, 60, 34, 11, 24, 51, 69, 62, 37, 13, 21, 48, 67, 64, 41, 15,
       18, 44, 66, 66, 44, 18, 15, 41, 64, 67, 48, 21, 13, 37, 62, 69, 51, 24, 11, 34, 60, 70, 54, 27},
    },

    // Band_8_4855
    {
      // Scaler inc range
      {4832, 4855},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 96, 74, 24, 55, 92, 79, 30, 48, 90, 83, 35,
       41, 87,  87, 41, 35, 83, 90, 48, 30, 79, 92, 55, 24, 74, 96, 62},
      // Coefficients 6 taps
      {32, 60, 72, 60, 32, 0,  27, 54, 70, 60, 34, 11, 24, 51, 68, 62, 38, 13, 21, 48, 67, 64, 41, 15,
       18, 44, 66, 66, 44, 18, 15, 41, 64, 67, 48, 21, 13, 38, 62, 68, 51, 24, 11, 34, 60, 70, 54, 27},
    },

    // Band_8_4877
    {
      // Scaler inc range
      {4855, 4877},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 96, 74, 24, 55, 92, 79, 30, 48, 90, 83, 35,
       41, 87,  87, 41, 35, 83, 90, 48, 30, 79, 92, 55, 24, 74, 96, 62},
      // Coefficients 6 taps
      {32, 60, 72, 60, 32, 0,  27, 54, 70, 60, 34, 11, 24, 51, 68, 62, 38, 13, 21, 48, 67, 64, 41, 15,
       18, 44, 66, 66, 44, 18, 15, 41, 64, 67, 48, 21, 13, 38, 62, 68, 51, 24, 11, 34, 60, 70, 54, 27},
    },

    // Band_8_4900
    {
      // Scaler inc range
      {4877, 4900},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 96, 74, 24, 55, 92, 79, 30, 48, 90, 83, 35,
       41, 87,  87, 41, 35, 83, 90, 48, 30, 79, 92, 55, 24, 74, 96, 62},
      // Coefficients 6 taps
      {32, 60, 72, 60, 32, 0,  28, 54, 69, 60, 34, 11, 24, 51, 68, 62, 38, 13, 21, 48, 67, 64, 41, 15,
       18, 44, 66, 66, 44, 18, 15, 41, 64, 67, 48, 21, 13, 38, 62, 68, 51, 24, 11, 34, 60, 69, 54, 28},
    },

    // Band_8_4923
    {
      // Scaler inc range
      {4900, 4923},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 96, 74, 24, 55, 92, 79, 30, 48, 90, 83, 35,
       41, 87,  87, 41, 35, 83, 90, 48, 30, 79, 92, 55, 24, 74, 96, 62},
      // Coefficients 6 taps
      {32, 60, 72, 60, 32, 0,  28, 54, 69, 60, 34, 11, 24, 51, 68, 62, 38, 13, 21, 48, 66, 64, 41, 16,
       18, 44, 66, 66, 44, 18, 16, 41, 64, 66, 48, 21, 13, 38, 62, 68, 51, 24, 11, 34, 60, 69, 54, 28},
    },

    // Band_8_4946
    {
      // Scaler inc range
      {4923, 4946},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 96, 74, 24, 55, 92, 79, 30, 48, 90, 83, 35,
       42, 86,  86, 42, 35, 83, 90, 48, 30, 79, 92, 55, 24, 74, 96, 62},
      // Coefficients 6 taps
      {32, 60, 72, 60, 32, 0,  28, 54, 69, 60, 34, 11, 24, 51, 68, 62, 38, 13, 21, 48, 66, 64, 41, 16,
       18, 44, 66, 66, 44, 18, 16, 41, 64, 66, 48, 21, 13, 38, 62, 68, 51, 24, 11, 34, 60, 69, 54, 28},
    },

    // Band_8_4970
    {
      // Scaler inc range
      {4946, 4970},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 95, 74, 25, 55, 93, 78, 30, 48, 90, 83, 35,
       42, 86,  86, 42, 35, 83, 90, 48, 30, 78, 93, 55, 25, 74, 95, 62},
      // Coefficients 6 taps
      {32, 60, 72, 60, 32, 0,  28, 54, 69, 60, 34, 11, 24, 51, 68, 62, 38, 13, 21, 48, 66, 64, 41, 16,
       18, 44, 66, 66, 44, 18, 16, 41, 64, 66, 48, 21, 13, 38, 62, 68, 51, 24, 11, 34, 60, 69, 54, 28},
    },

    // Band_8_4993
    {
      // Scaler inc range
      {4970, 4993},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 95, 74, 25, 55, 93, 78, 30, 48, 89, 83, 36,
       42, 86,  86, 42, 36, 83, 89, 48, 30, 78, 93, 55, 25, 74, 95, 62},
      // Coefficients 6 taps
      {32, 60, 72, 60, 32, 0,  28, 54, 69, 60, 34, 11, 24, 51, 68, 62, 38, 13, 21, 48, 66, 64, 41, 16,
       18, 44, 66, 66, 44, 18, 16, 41, 64, 66, 48, 21, 13, 38, 62, 68, 51, 24, 11, 34, 60, 69, 54, 28},
    },

    // Band_8_5017
    {
      // Scaler inc range
      {4993, 5017},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 95, 74, 25, 55, 93, 78, 30, 48, 89, 83, 36,
       42, 86,  86, 42, 36, 83, 89, 48, 30, 78, 93, 55, 25, 74, 95, 62},
      // Coefficients 6 taps
      {32, 60, 72, 60, 32, 0,  28, 54, 69, 60, 34, 11, 25, 51, 67, 62, 38, 13, 21, 48, 66, 64, 41, 16,
       18, 44, 66, 66, 44, 18, 16, 41, 64, 66, 48, 21, 13, 38, 62, 67, 51, 25, 11, 34, 60, 69, 54, 28},
    },

    // Band_8_5041
    {
      // Scaler inc range
      {5017, 5041},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 95, 74, 25, 55, 93, 78, 30, 48, 89, 83, 36,
       42, 86,  86, 42, 36, 83, 89, 48, 30, 78, 93, 55, 25, 74, 95, 62},
      // Coefficients 6 taps
      {33, 60, 70, 60, 33, 0,  28, 54, 68, 60, 35, 11, 25, 51, 67, 62, 38, 13, 21, 48, 67, 63, 41, 16,
       19, 44, 65, 65, 44, 19, 16, 41, 63, 67, 48, 21, 13, 38, 62, 67, 51, 25, 11, 35, 60, 68, 54, 28},
    },

    // Band_8_5066
    {
      // Scaler inc range
      {5041, 5066},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 95, 74, 25, 55, 93, 78, 30, 48, 89, 83, 36,
       42, 86,  86, 42, 36, 83, 89, 48, 30, 78, 93, 55, 25, 74, 95, 62},
      // Coefficients 6 taps
      {33, 60, 70, 60, 33, 0,  28, 54, 68, 60, 35, 11, 25, 51, 67, 62, 38, 13, 22, 48, 66, 63, 41, 16,
       19, 44, 65, 65, 44, 19, 16, 41, 63, 66, 48, 22, 13, 38, 62, 67, 51, 25, 11, 35, 60, 68, 54, 28},
    },

    // Band_8_5090
    {
      // Scaler inc range
      {5066, 5090},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 95, 74, 25, 55, 93, 78, 30, 48, 90, 82, 36,
       42, 86,  86, 42, 36, 82, 90, 48, 30, 78, 93, 55, 25, 74, 95, 62},
      // Coefficients 6 taps
      {33, 60, 70, 60, 33, 0,  28, 54, 68, 60, 35, 11, 25, 51, 66, 62, 38, 14, 22, 48, 66, 63, 41, 16,
       19, 44, 65, 65, 44, 19, 16, 41, 63, 66, 48, 22, 14, 38, 62, 66, 51, 25, 11, 35, 60, 68, 54, 28},
    },

    // Band_8_5115
    {
      // Scaler inc range
      {5090, 5115},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 95, 74, 25, 55, 93, 78, 30, 48, 90, 82, 36,
       42, 86,  86, 42, 36, 82, 90, 48, 30, 78, 93, 55, 25, 74, 95, 62},
      // Coefficients 6 taps
      {33, 59, 72, 59, 33, 0,  28, 54, 69, 59, 35, 11, 25, 51, 67, 61, 38, 14, 22, 48, 66, 63, 41, 16,
       19, 44, 65, 65, 44, 19, 16, 41, 63, 66, 48, 22, 14, 38, 61, 67, 51, 25, 11, 35, 59, 69, 54, 28},
    },

    // Band_8_5140
    {
      // Scaler inc range
      {5115, 5140},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 95, 74, 25, 55, 93, 78, 30, 48, 90, 82, 36,
       42, 86,  86, 42, 36, 82, 90, 48, 30, 78, 93, 55, 25, 74, 95, 62},
      // Coefficients 6 taps
      {33, 59, 72, 59, 33, 0,  28, 54, 69, 59, 35, 11, 25, 51, 67, 61, 38, 14, 22, 48, 66, 63, 41, 16,
       19, 44, 65, 65, 44, 19, 16, 41, 63, 66, 48, 22, 14, 38, 61, 67, 51, 25, 11, 35, 59, 69, 54, 28},
    },

    // Band_8_5165
    {
      // Scaler inc range
      {5140, 5165},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 95, 74, 25, 55, 93, 78, 30, 48, 90, 82, 36,
       42, 86,  86, 42, 36, 82, 90, 48, 30, 78, 93, 55, 25, 74, 95, 62},
      // Coefficients 6 taps
      {33, 59, 72, 59, 33, 0,  28, 54, 68, 59, 35, 12, 25, 51, 67, 61, 38, 14, 22, 48, 66, 63, 41, 16,
       19, 44, 65, 65, 44, 19, 16, 41, 63, 66, 48, 22, 14, 38, 61, 67, 51, 25, 12, 35, 59, 68, 54, 28},
    },

    // Band_8_5191
    {
      // Scaler inc range
      {5165, 5191},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 95, 74, 25, 55, 93, 78, 30, 48, 90, 82, 36,
       42, 86,  86, 42, 36, 82, 90, 48, 30, 78, 93, 55, 25, 74, 95, 62},
      // Coefficients 6 taps
      {33, 59, 72, 59, 33, 0,  28, 54, 68, 59, 35, 12, 25, 51, 67, 61, 38, 14, 22, 48, 66, 63, 41, 16,
       19, 44, 65, 65, 44, 19, 16, 41, 63, 66, 48, 22, 14, 38, 61, 67, 51, 25, 12, 35, 59, 68, 54, 28},
    },

    // Band_8_5217
    {
      // Scaler inc range
      {5191, 5217},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 95, 74, 25, 55, 93, 78, 30, 48, 90, 82, 36,
       42, 86,  86, 42, 36, 82, 90, 48, 30, 78, 93, 55, 25, 74, 95, 62},
      // Coefficients 6 taps
      {33, 59, 72, 59, 33, 0,  28, 54, 68, 59, 35, 12, 25, 51, 67, 61, 38, 14, 22, 48, 66, 63, 41, 16,
       19, 44, 65, 65, 44, 19, 16, 41, 63, 66, 48, 22, 14, 38, 61, 67, 51, 25, 12, 35, 59, 68, 54, 28},
    },

    // Band_8_5243
    {
      // Scaler inc range
      {5217, 5243},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 95, 74, 25, 55, 93, 78, 30, 48, 90, 82, 36,
       42, 86,  86, 42, 36, 82, 90, 48, 30, 78, 93, 55, 25, 74, 95, 62},
      // Coefficients 6 taps
      {33, 59, 72, 59, 33, 0,  28, 54, 68, 59, 35, 12, 25, 51, 67, 61, 38, 14, 22, 48, 66, 63, 41, 16,
       19, 44, 65, 65, 44, 19, 16, 41, 63, 66, 48, 22, 14, 38, 61, 67, 51, 25, 12, 35, 59, 68, 54, 28},
    },

    // Band_8_5269
    {
      // Scaler inc range
      {5243, 5269},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 95, 74, 25, 55, 93, 78, 30, 48, 90, 82, 36,
       42, 86,  86, 42, 36, 82, 90, 48, 30, 78, 93, 55, 25, 74, 95, 62},
      // Coefficients 6 taps
      {33, 59, 72, 59, 33, 0,  28, 54, 68, 59, 35, 12, 25, 51, 67, 61, 38, 14, 22, 48, 66, 63, 41, 16,
       19, 44, 65, 65, 44, 19, 16, 41, 63, 66, 48, 22, 14, 38, 61, 67, 51, 25, 12, 35, 59, 68, 54, 28},
    },

    // Band_8_5296
    {
      // Scaler inc range
      {5269, 5296},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 95, 74, 25, 55, 93, 78, 30, 48, 90, 82, 36,
       42, 86,  86, 42, 36, 82, 90, 48, 30, 78, 93, 55, 25, 74, 95, 62},
      // Coefficients 6 taps
      {33, 59, 72, 59, 33, 0,  28, 54, 68, 59, 35, 12, 25, 51, 67, 61, 38, 14, 22, 48, 65, 63, 41, 17,
       19, 44, 65, 65, 44, 19, 17, 41, 63, 65, 48, 22, 14, 38, 61, 67, 51, 25, 12, 35, 59, 68, 54, 28},
    },

    // Band_8_5323
    {
      // Scaler inc range
      {5296, 5323},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 95, 74, 25, 55, 93, 78, 30, 48, 90, 82, 36,
       42, 86,  86, 42, 36, 82, 90, 48, 30, 78, 93, 55, 25, 74, 95, 62},
      // Coefficients 6 taps
      {33, 59, 72, 59, 33, 0,  28, 54, 68, 59, 35, 12, 25, 51, 67, 61, 38, 14, 22, 48, 65, 63, 41, 17,
       19, 44, 65, 65, 44, 19, 17, 41, 63, 65, 48, 22, 14, 38, 61, 67, 51, 25, 12, 35, 59, 68, 54, 28},
    },

    // Band_8_5350
    {
      // Scaler inc range
      {5323, 5350},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 95, 74, 25, 55, 93, 78, 30, 49, 89, 82, 36,
       42, 86,  86, 42, 36, 82, 89, 49, 30, 78, 93, 55, 25, 74, 95, 62},
      // Coefficients 6 taps
      {33, 59, 72, 59, 33, 0,  29, 54, 67, 59, 35, 12, 25, 51, 67, 61, 38, 14, 22, 48, 65, 63, 41, 17,
       19, 44, 65, 65, 44, 19, 17, 41, 63, 65, 48, 22, 14, 38, 61, 67, 51, 25, 12, 35, 59, 67, 54, 29},
    },

    // Band_8_5377
    {
      // Scaler inc range
      {5350, 5377},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 95, 74, 25, 55, 92, 78, 31, 49, 89, 82, 36,
       42, 86,  86, 42, 36, 82, 89, 49, 31, 78, 92, 55, 25, 74, 95, 62},
      // Coefficients 6 taps
      {33, 59, 72, 59, 33, 0,  29, 54, 67, 59, 35, 12, 25, 51, 67, 61, 38, 14, 22, 48, 65, 63, 41, 17,
       19, 44, 65, 65, 44, 19, 17, 41, 63, 65, 48, 22, 14, 38, 61, 67, 51, 25, 12, 35, 59, 67, 54, 29},
    },

    // Band_8_5405
    {
      // Scaler inc range
      {5377, 5405},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 95, 74, 25, 55, 92, 78, 31, 49, 89, 82, 36,
       42, 86,  86, 42, 36, 82, 89, 49, 31, 78, 92, 55, 25, 74, 95, 62},
      // Coefficients 6 taps
      {33, 59, 72, 59, 33, 0,  29, 54, 67, 59, 35, 12, 25, 51, 67, 61, 38, 14, 22, 48, 65, 63, 41, 17,
       19, 44, 65, 65, 44, 19, 17, 41, 63, 65, 48, 22, 14, 38, 61, 67, 51, 25, 12, 35, 59, 67, 54, 29},
    },

    // Band_8_5433
    {
      // Scaler inc range
      {5405, 5433},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 95, 74, 25, 55, 92, 78, 31, 49, 89, 82, 36,
       42, 86,  86, 42, 36, 82, 89, 49, 31, 78, 92, 55, 25, 74, 95, 62},
      // Coefficients 6 taps
      {33, 59, 72, 59, 33, 0,  29, 54, 67, 59, 35, 12, 25, 51, 67, 61, 38, 14, 22, 48, 66, 62, 41, 17,
       20, 44, 64, 64, 44, 20, 17, 41, 62, 66, 48, 22, 14, 38, 61, 67, 51, 25, 12, 35, 59, 67, 54, 29},
    },

    // Band_8_5461
    {
      // Scaler inc range
      {5433, 5461},
      // Coefficients 4 taps
      {75, 106, 75, 0,  62, 94, 74, 26, 55, 92, 78, 31, 49, 89, 82, 36,
       42, 86,  86, 42, 36, 82, 89, 49, 31, 78, 92, 55, 26, 74, 94, 62},
      // Coefficients 6 taps
      {33, 59, 72, 59, 33, 0,  29, 54, 67, 59, 35, 12, 26, 51, 66, 61, 38, 14, 22, 48, 66, 62, 41, 17,
       20, 44, 64, 64, 44, 20, 17, 41, 62, 66, 48, 22, 14, 38, 61, 66, 51, 26, 12, 35, 59, 67, 54, 29},
    },

    // Band_8_5490
    {
      // Scaler inc range
      {5461, 5490},
      // Coefficients 4 taps
      {76, 104, 76, 0,  62, 94, 74, 26, 55, 92, 78, 31, 49, 89, 82, 36,
       42, 86,  86, 42, 36, 82, 89, 49, 31, 78, 92, 55, 26, 74, 94, 62},
      // Coefficients 6 taps
      {34, 59, 70, 59, 34, 0,  29, 54, 67, 59, 35, 12, 26, 51, 65, 61, 38, 15, 23, 48, 65, 62, 41, 17,
       20, 44, 64, 64, 44, 20, 17, 41, 62, 65, 48, 23, 15, 38, 61, 65, 51, 26, 12, 35, 59, 67, 54, 29},
    },

    // Band_8_5519
    {
      // Scaler inc range
      {5490, 5519},
      // Coefficients 4 taps
      {76, 104, 76, 0,  62, 94, 74, 26, 55, 92, 78, 31, 49, 89, 82, 36,
       42, 86,  86, 42, 36, 82, 89, 49, 31, 78, 92, 55, 26, 74, 94, 62},
      // Coefficients 6 taps
      {34, 59, 70, 59, 34, 0,  29, 54, 67, 59, 35, 12, 26, 51, 65, 61, 38, 15, 23, 48, 65, 62, 41, 17,
       20, 45, 63, 63, 45, 20, 17, 41, 62, 65, 48, 23, 15, 38, 61, 65, 51, 26, 12, 35, 59, 67, 54, 29},
    },

    // Band_8_5548
    {
      // Scaler inc range
      {5519, 5548},
      // Coefficients 4 taps
      {76, 104, 76, 0,  62, 94, 74, 26, 55, 92, 78, 31, 49, 89, 82, 36,
       42, 86,  86, 42, 36, 82, 89, 49, 31, 78, 92, 55, 26, 74, 94, 62},
      // Coefficients 6 taps
      {34, 59, 70, 59, 34, 0,  29, 54, 67, 59, 35, 12, 26, 51, 65, 61, 38, 15, 23, 48, 65, 62, 41, 17,
       20, 45, 63, 63, 45, 20, 17, 41, 62, 65, 48, 23, 15, 38, 61, 65, 51, 26, 12, 35, 59, 67, 54, 29},
    },

    // Band_8_5578
    {
      // Scaler inc range
      {5548, 5578},
      // Coefficients 4 taps
      {76, 104, 76, 0,  62, 94, 74, 26, 55, 92, 78, 31, 49, 89, 82, 36,
       42, 86,  86, 42, 36, 82, 89, 49, 31, 78, 92, 55, 26, 74, 94, 62},
      // Coefficients 6 taps
      {34, 59, 70, 59, 34, 0,  29, 54, 66, 59, 35, 13, 26, 51, 65, 61, 38, 15, 23, 48, 65, 62, 41, 17,
       20, 45, 63, 63, 45, 20, 17, 41, 62, 65, 48, 23, 15, 38, 61, 65, 51, 26, 13, 35, 59, 66, 54, 29},
    },

    // Band_8_5607
    {
      // Scaler inc range
      {5578, 5607},
      // Coefficients 4 taps
      {76, 104, 76, 0,  62, 94, 74, 26, 55, 92, 78, 31, 49, 88, 82, 37,
       42, 86,  86, 42, 37, 82, 88, 49, 31, 78, 92, 55, 26, 74, 94, 62},
      // Coefficients 6 taps
      {34, 59, 70, 59, 34, 0,  29, 54, 66, 59, 35, 13, 26, 51, 66, 60, 38, 15, 23, 48, 65, 62, 41, 17,
       20, 45, 63, 63, 45, 20, 17, 41, 62, 65, 48, 23, 15, 38, 60, 66, 51, 26, 13, 35, 59, 66, 54, 29},
    },

    // Band_8_5638
    {
      // Scaler inc range
      {5607, 5638},
      // Coefficients 4 taps
      {76, 104, 76, 0,  62, 94, 74, 26, 55, 92, 78, 31, 49, 88, 82, 37,
       43, 85,  85, 43, 37, 82, 88, 49, 31, 78, 92, 55, 26, 74, 94, 62},
      // Coefficients 6 taps
      {34, 59, 70, 59, 34, 0,  29, 54, 66, 59, 35, 13, 26, 51, 66, 60, 38, 15, 23, 48, 65, 62, 41, 17,
       20, 45, 63, 63, 45, 20, 17, 41, 62, 65, 48, 23, 15, 38, 60, 66, 51, 26, 13, 35, 59, 66, 54, 29},
    },

    // Band_8_5668
    {
      // Scaler inc range
      {5638, 5668},
      // Coefficients 4 taps
      {76, 104, 76, 0,  62, 94, 74, 26, 55, 92, 78, 31, 49, 88, 82, 37,
       43, 85,  85, 43, 37, 82, 88, 49, 31, 78, 92, 55, 26, 74, 94, 62},
      // Coefficients 6 taps
      {34, 59, 70, 59, 34, 0,  29, 54, 66, 59, 35, 13, 26, 51, 66, 60, 38, 15, 23, 48, 65, 62, 41, 17,
       20, 45, 63, 63, 45, 20, 17, 41, 62, 65, 48, 23, 15, 38, 60, 66, 51, 26, 13, 35, 59, 66, 54, 29},
    },

    // Band_8_5699
    {
      // Scaler inc range
      {5668, 5699},
      // Coefficients 4 taps
      {76, 104, 76, 0,  62, 94, 74, 26, 55, 92, 78, 31, 49, 88, 82, 37,
       43, 85,  85, 43, 37, 82, 88, 49, 31, 78, 92, 55, 26, 74, 94, 62},
      // Coefficients 6 taps
      {34, 59, 70, 59, 34, 0,  29, 54, 66, 59, 35, 13, 26, 51, 66, 60, 38, 15, 23, 48, 64, 62, 42, 17,
       20, 45, 63, 63, 45, 20, 17, 42, 62, 64, 48, 23, 15, 38, 60, 66, 51, 26, 13, 35, 59, 66, 54, 29},
    },

    // Band_8_5730
    {
      // Scaler inc range
      {5699, 5730},
      // Coefficients 4 taps
      {76, 104, 76, 0,  62, 94, 74, 26, 55, 92, 78, 31, 49, 88, 82, 37,
       43, 85,  85, 43, 37, 82, 88, 49, 31, 78, 92, 55, 26, 74, 94, 62},
      // Coefficients 6 taps
      {34, 59, 70, 59, 34, 0,  29, 54, 66, 58, 36, 13, 26, 51, 65, 60, 39, 15, 23, 48, 63, 62, 42, 18,
       20, 45, 63, 63, 45, 20, 18, 42, 62, 63, 48, 23, 15, 39, 60, 65, 51, 26, 13, 36, 58, 66, 54, 29},
    },

    // Band_8_5761
    {
      // Scaler inc range
      {5730, 5761},
      // Coefficients 4 taps
      {76, 104, 76, 0,  62, 94, 74, 26, 56, 91, 78, 31, 49, 88, 82, 37,
       43, 85,  85, 43, 37, 82, 88, 49, 31, 78, 91, 56, 26, 74, 94, 62},
      // Coefficients 6 taps
      {34, 59, 70, 59, 34, 0,  29, 54, 66, 58, 36, 13, 26, 51, 65, 60, 39, 15, 23, 48, 63, 62, 42, 18,
       20, 45, 63, 63, 45, 20, 18, 42, 62, 63, 48, 23, 15, 39, 60, 65, 51, 26, 13, 36, 58, 66, 54, 29},
    },

    // Band_8_5793
    {
      // Scaler inc range
      {5761, 5793},
      // Coefficients 4 taps
      {76, 104, 76, 0,  62, 94, 74, 26, 56, 91, 78, 31, 49, 88, 82, 37,
       43, 85,  85, 43, 37, 82, 88, 49, 31, 78, 91, 56, 26, 74, 94, 62},
      // Coefficients 6 taps
      {34, 59, 70, 59, 34, 0,  29, 54, 66, 58, 36, 13, 26, 51, 65, 60, 39, 15, 23, 48, 63, 62, 42, 18,
       20, 45, 63, 63, 45, 20, 18, 42, 62, 63, 48, 23, 15, 39, 60, 65, 51, 26, 13, 36, 58, 66, 54, 29},
    },

    // Band_8_5825
    {
      // Scaler inc range
      {5793, 5825},
      // Coefficients 4 taps
      {76, 104, 76, 0,  62, 94, 74, 26, 56, 91, 78, 31, 49, 88, 82, 37,
       43, 85,  85, 43, 37, 82, 88, 49, 31, 78, 91, 56, 26, 74, 94, 62},
      // Coefficients 6 taps
      {34, 59, 70, 59, 34, 0,  29, 54, 66, 58, 36, 13, 26, 51, 65, 60, 39, 15, 23, 48, 63, 62, 42, 18,
       20, 45, 63, 63, 45, 20, 18, 42, 62, 63, 48, 23, 15, 39, 60, 65, 51, 26, 13, 36, 58, 66, 54, 29},
    },

    // Band_8_5858
    {
      // Scaler inc range
      {5825, 5858},
      // Coefficients 4 taps
      {76, 104, 76, 0,  62, 94, 74, 26, 56, 91, 78, 31, 49, 88, 82, 37,
       43, 85,  85, 43, 37, 82, 88, 49, 31, 78, 91, 56, 26, 74, 94, 62},
      // Coefficients 6 taps
      {34, 59, 70, 59, 34, 0,  29, 53, 67, 58, 36, 13, 26, 51, 65, 60, 39, 15, 23, 48, 63, 62, 42, 18,
       20, 45, 63, 63, 45, 20, 18, 42, 62, 63, 48, 23, 15, 39, 60, 65, 51, 26, 13, 36, 58, 67, 53, 29},
    },

    // Band_8_5891
    {
      // Scaler inc range
      {5858, 5891},
      // Coefficients 4 taps
      {76, 104, 76, 0,  62, 94, 74, 26, 56, 91, 78, 31, 49, 88, 82, 37,
       43, 85,  85, 43, 37, 82, 88, 49, 31, 78, 91, 56, 26, 74, 94, 62},
      // Coefficients 6 taps
      {34, 59, 70, 59, 34, 0,  29, 53, 67, 58, 36, 13, 26, 51, 65, 60, 39, 15, 23, 48, 63, 62, 42, 18,
       20, 45, 63, 63, 45, 20, 18, 42, 62, 63, 48, 23, 15, 39, 60, 65, 51, 26, 13, 36, 58, 67, 53, 29},
    },

    // Band_8_5924
    {
      // Scaler inc range
      {5891, 5924},
      // Coefficients 4 taps
      {76, 104, 76, 0,  62, 94, 74, 26, 56, 91, 78, 31, 49, 88, 82, 37,
       43, 85,  85, 43, 37, 82, 88, 49, 31, 78, 91, 56, 26, 74, 94, 62},
      // Coefficients 6 taps
      {34, 59, 70, 59, 34, 0,  29, 53, 67, 58, 36, 13, 26, 51, 65, 60, 39, 15, 23, 48, 64, 61, 42, 18,
       21, 45, 62, 62, 45, 21, 18, 42, 61, 64, 48, 23, 15, 39, 60, 65, 51, 26, 13, 36, 58, 67, 53, 29},
    },

    // Band_8_5958
    {
      // Scaler inc range
      {5924, 5958},
      // Coefficients 4 taps
      {76, 104, 76, 0,  62, 94, 74, 26, 56, 91, 78, 31, 49, 88, 82, 37,
       43, 85,  85, 43, 37, 82, 88, 49, 31, 78, 91, 56, 26, 74, 94, 62},
      // Coefficients 6 taps
      {34, 59, 70, 59, 34, 0,  30, 53, 66, 58, 36, 13, 26, 51, 65, 60, 39, 15, 23, 48, 64, 61, 42, 18,
       21, 45, 62, 62, 45, 21, 18, 42, 61, 64, 48, 23, 15, 39, 60, 65, 51, 26, 13, 36, 58, 66, 53, 30},
    },

    // Band_8_5992
    {
      // Scaler inc range
      {5958, 5992},
      // Coefficients 4 taps
      {76, 104, 76, 0,  62, 94, 74, 26, 56, 91, 78, 31, 49, 88, 82, 37,
       43, 85,  85, 43, 37, 82, 88, 49, 31, 78, 91, 56, 26, 74, 94, 62},
      // Coefficients 6 taps
      {34, 59, 70, 59, 34, 0,  30, 53, 66, 58, 36, 13, 26, 51, 64, 60, 39, 16, 23, 48, 64, 61, 42, 18,
       21, 45, 62, 62, 45, 21, 18, 42, 61, 64, 48, 23, 16, 39, 60, 64, 51, 26, 13, 36, 58, 66, 53, 30},
    },

    // Band_8_6026
    {
      // Scaler inc range
      {5992, 6026},
      // Coefficients 4 taps
      {76, 104, 76, 0,  62, 94, 74, 26, 56, 91, 78, 31, 49, 88, 82, 37,
       43, 85,  85, 43, 37, 82, 88, 49, 31, 78, 91, 56, 26, 74, 94, 62},
      // Coefficients 6 taps
      {34, 59, 70, 59, 34, 0,  30, 53, 66, 58, 36, 13, 26, 50, 65, 60, 39, 16, 24, 48, 63, 61, 42, 18,
       21, 45, 62, 62, 45, 21, 18, 42, 61, 63, 48, 24, 16, 39, 60, 65, 50, 26, 13, 36, 58, 66, 53, 30},
    },

    // Band_8_6061
    {
      // Scaler inc range
      {6026, 6061},
      // Coefficients 4 taps
      {76, 104, 76, 0,  62, 94, 74, 26, 56, 91, 78, 31, 49, 88, 82, 37,
       43, 85,  85, 43, 37, 82, 88, 49, 31, 78, 91, 56, 26, 74, 94, 62},
      // Coefficients 6 taps
      {35, 59, 68, 59, 35, 0,  30, 53, 66, 58, 36, 13, 27, 50, 64, 60, 39, 16, 24, 48, 63, 61, 42, 18,
       21, 45, 62, 62, 45, 21, 18, 42, 61, 63, 48, 24, 16, 39, 60, 64, 50, 27, 13, 36, 58, 66, 53, 30},
    },

    // Band_8_6096
    {
      // Scaler inc range
      {6061, 6096},
      // Coefficients 4 taps
      {76, 104, 76, 0,  62, 94, 74, 26, 56, 90, 78, 32, 49, 88, 82, 37,
       43, 85,  85, 43, 37, 82, 88, 49, 32, 78, 90, 56, 26, 74, 94, 62},
      // Coefficients 6 taps
      {35, 59, 68, 59, 35, 0,  30, 53, 66, 58, 36, 13, 27, 50, 64, 60, 39, 16, 24, 48, 63, 61, 42, 18,
       21, 45, 62, 62, 45, 21, 18, 42, 61, 63, 48, 24, 16, 39, 60, 64, 50, 27, 13, 36, 58, 66, 53, 30},
    },

    // Band_8_6132
    {
      // Scaler inc range
      {6096, 6132},
      // Coefficients 4 taps
      {76, 104, 76, 0,  62, 93, 74, 27, 56, 90, 78, 32, 49, 88, 82, 37,
       43, 85,  85, 43, 37, 82, 88, 49, 32, 78, 90, 56, 27, 74, 93, 62},
      // Coefficients 6 taps
      {35, 59, 68, 59, 35, 0,  30, 53, 65, 58, 36, 14, 27, 50, 64, 60, 39, 16, 24, 48, 63, 61, 42, 18,
       21, 45, 62, 62, 45, 21, 18, 42, 61, 63, 48, 24, 16, 39, 60, 64, 50, 27, 14, 36, 58, 65, 53, 30},
    },

    // Band_8_6168
    {
      // Scaler inc range
      {6132, 6168},
      // Coefficients 4 taps
      {76, 104, 76, 0,  62, 93, 74, 27, 56, 90, 78, 32, 49, 89, 81, 37,
       43, 85,  85, 43, 37, 81, 89, 49, 32, 78, 90, 56, 27, 74, 93, 62},
      // Coefficients 6 taps
      {35, 59, 68, 59, 35, 0,  30, 53, 65, 58, 36, 14, 27, 50, 64, 60, 39, 16, 24, 48, 63, 61, 42, 18,
       21, 45, 62, 62, 45, 21, 18, 42, 61, 63, 48, 24, 16, 39, 60, 64, 50, 27, 14, 36, 58, 65, 53, 30},
    },

    // Band_8_6205
    {
      // Scaler inc range
      {6168, 6205},
      // Coefficients 4 taps
      {76, 104, 76, 0,  62, 93, 74, 27, 56, 90, 78, 32, 49, 89, 81, 37,
       43, 85,  85, 43, 37, 81, 89, 49, 32, 78, 90, 56, 27, 74, 93, 62},
      // Coefficients 6 taps
      {35, 59, 68, 59, 35, 0,  30, 53, 65, 58, 36, 14, 27, 50, 64, 60, 39, 16, 24, 48, 63, 61, 42, 18,
       21, 45, 62, 62, 45, 21, 18, 42, 61, 63, 48, 24, 16, 39, 60, 64, 50, 27, 14, 36, 58, 65, 53, 30},
    },

    // Band_8_6242
    {
      // Scaler inc range
      {6205, 6242},
      // Coefficients 4 taps
      {76, 104, 76, 0,  62, 93, 74, 27, 56, 90, 78, 32, 49, 89, 81, 37,
       43, 85,  85, 43, 37, 81, 89, 49, 32, 78, 90, 56, 27, 74, 93, 62},
      // Coefficients 6 taps
      {35, 59, 68, 59, 35, 0,  30, 53, 65, 58, 36, 14, 27, 50, 65, 59, 39, 16, 24, 48, 63, 61, 42, 18,
       21, 45, 62, 62, 45, 21, 18, 42, 61, 63, 48, 24, 16, 39, 59, 65, 50, 27, 14, 36, 58, 65, 53, 30},
    },

    // Band_8_6279
    {
      // Scaler inc range
      {6242, 6279},
      // Coefficients 4 taps
      {76, 104, 76, 0,  62, 93, 74, 27, 56, 90, 78, 32, 49, 89, 81, 37,
       43, 85,  85, 43, 37, 81, 89, 49, 32, 78, 90, 56, 27, 74, 93, 62},
      // Coefficients 6 taps
      {35, 59, 68, 59, 35, 0,  30, 53, 65, 58, 36, 14, 27, 50, 65, 59, 39, 16, 24, 48, 63, 61, 42, 18,
       21, 45, 62, 62, 45, 21, 18, 42, 61, 63, 48, 24, 16, 39, 59, 65, 50, 27, 14, 36, 58, 65, 53, 30},
    },

    // Band_8_6317
    {
      // Scaler inc range
      {6279, 6317},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 74, 27, 56, 90, 78, 32, 49, 89, 81, 37,
       43, 85,  85, 43, 37, 81, 89, 49, 32, 78, 90, 56, 27, 74, 92, 63},
      // Coefficients 6 taps
      {35, 59, 68, 59, 35, 0,  30, 53, 65, 58, 36, 14, 27, 50, 65, 59, 39, 16, 24, 48, 63, 61, 42, 18,
       21, 45, 62, 62, 45, 21, 18, 42, 61, 63, 48, 24, 16, 39, 59, 65, 50, 27, 14, 36, 58, 65, 53, 30},
    },

    // Band_8_6355
    {
      // Scaler inc range
      {6317, 6355},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 74, 27, 56, 90, 78, 32, 49, 89, 81, 37,
       43, 85,  85, 43, 37, 81, 89, 49, 32, 78, 90, 56, 27, 74, 92, 63},
      // Coefficients 6 taps
      {35, 59, 68, 59, 35, 0,  30, 53, 65, 58, 36, 14, 27, 50, 65, 59, 39, 16, 24, 48, 62, 61, 42, 19,
       21, 45, 62, 62, 45, 21, 19, 42, 61, 62, 48, 24, 16, 39, 59, 65, 50, 27, 14, 36, 58, 65, 53, 30},
    },

    // Band_8_6394
    {
      // Scaler inc range
      {6355, 6394},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 74, 27, 56, 90, 78, 32, 49, 89, 81, 37,
       43, 85,  85, 43, 37, 81, 89, 49, 32, 78, 90, 56, 27, 74, 92, 63},
      // Coefficients 6 taps
      {35, 59, 68, 59, 35, 0,  30, 53, 65, 58, 36, 14, 27, 50, 65, 59, 39, 16, 24, 48, 62, 61, 42, 19,
       21, 45, 62, 62, 45, 21, 19, 42, 61, 62, 48, 24, 16, 39, 59, 65, 50, 27, 14, 36, 58, 65, 53, 30},
    },

    // Band_8_6433
    {
      // Scaler inc range
      {6394, 6433},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 74, 27, 56, 90, 78, 32, 49, 89, 81, 37,
       43, 85,  85, 43, 37, 81, 89, 49, 32, 78, 90, 56, 27, 74, 92, 63},
      // Coefficients 6 taps
      {35, 59, 68, 59, 35, 0,  30, 53, 65, 58, 36, 14, 27, 50, 65, 59, 39, 16, 24, 48, 62, 61, 42, 19,
       21, 45, 62, 62, 45, 21, 19, 42, 61, 62, 48, 24, 16, 39, 59, 65, 50, 27, 14, 36, 58, 65, 53, 30},
    },

    // Band_8_6473
    {
      // Scaler inc range
      {6433, 6473},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 74, 27, 56, 90, 78, 32, 49, 89, 81, 37,
       43, 85,  85, 43, 37, 81, 89, 49, 32, 78, 90, 56, 27, 74, 92, 63},
      // Coefficients 6 taps
      {35, 59, 68, 59, 35, 0,  30, 53, 65, 58, 36, 14, 27, 50, 65, 59, 39, 16, 24, 48, 62, 61, 42, 19,
       21, 45, 62, 62, 45, 21, 19, 42, 61, 62, 48, 24, 16, 39, 59, 65, 50, 27, 14, 36, 58, 65, 53, 30},
    },

    // Band_8_6513
    {
      // Scaler inc range
      {6473, 6513},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 74, 27, 56, 90, 78, 32, 50, 87, 81, 38,
       43, 85,  85, 43, 38, 81, 87, 50, 32, 78, 90, 56, 27, 74, 92, 63},
      // Coefficients 6 taps
      {35, 59, 68, 59, 35, 0,  30, 53, 65, 58, 36, 14, 27, 50, 65, 59, 39, 16, 24, 48, 62, 61, 42, 19,
       21, 45, 62, 62, 45, 21, 19, 42, 61, 62, 48, 24, 16, 39, 59, 65, 50, 27, 14, 36, 58, 65, 53, 30},
    },

    // Band_8_6554
    {
      // Scaler inc range
      {6513, 6554},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 74, 27, 56, 90, 78, 32, 50, 87, 81, 38,
       43, 85,  85, 43, 38, 81, 87, 50, 32, 78, 90, 56, 27, 74, 92, 63},
      // Coefficients 6 taps
      {35, 59, 68, 59, 35, 0,  30, 53, 65, 58, 36, 14, 27, 50, 65, 59, 39, 16, 24, 48, 62, 61, 42, 19,
       21, 45, 62, 62, 45, 21, 19, 42, 61, 62, 48, 24, 16, 39, 59, 65, 50, 27, 14, 36, 58, 65, 53, 30},
    },

    // Band_8_6595
    {
      // Scaler inc range
      {6554, 6595},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 74, 27, 56, 90, 78, 32, 50, 87, 81, 38,
       43, 85,  85, 43, 38, 81, 87, 50, 32, 78, 90, 56, 27, 74, 92, 63},
      // Coefficients 6 taps
      {35, 59, 68, 59, 35, 0,  30, 53, 66, 57, 36, 14, 27, 50, 65, 59, 39, 16, 24, 48, 62, 61, 42, 19,
       22, 45, 61, 61, 45, 22, 19, 42, 61, 62, 48, 24, 16, 39, 59, 65, 50, 27, 14, 36, 57, 66, 53, 30},
    },

    // Band_8_6637
    {
      // Scaler inc range
      {6595, 6637},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 74, 27, 56, 90, 78, 32, 50, 87, 81, 38,
       43, 85,  85, 43, 38, 81, 87, 50, 32, 78, 90, 56, 27, 74, 92, 63},
      // Coefficients 6 taps
      {35, 59, 68, 59, 35, 0,  30, 53, 66, 57, 36, 14, 27, 50, 65, 59, 39, 16, 24, 48, 63, 60, 42, 19,
       22, 45, 61, 61, 45, 22, 19, 42, 60, 63, 48, 24, 16, 39, 59, 65, 50, 27, 14, 36, 57, 66, 53, 30},
    },

    // Band_8_6679
    {
      // Scaler inc range
      {6637, 6679},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 74, 27, 56, 90, 78, 32, 50, 87, 81, 38,
       43, 85,  85, 43, 38, 81, 87, 50, 32, 78, 90, 56, 27, 74, 92, 63},
      // Coefficients 6 taps
      {35, 59, 68, 59, 35, 0,  30, 53, 66, 57, 36, 14, 27, 50, 64, 59, 39, 17, 24, 48, 63, 60, 42, 19,
       22, 45, 61, 61, 45, 22, 19, 42, 60, 63, 48, 24, 17, 39, 59, 64, 50, 27, 14, 36, 57, 66, 53, 30},
    },

    // Band_8_6722
    {
      // Scaler inc range
      {6679, 6722},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 74, 27, 56, 90, 78, 32, 50, 87, 81, 38,
       44, 84,  84, 44, 38, 81, 87, 50, 32, 78, 90, 56, 27, 74, 92, 63},
      // Coefficients 6 taps
      {35, 58, 70, 58, 35, 0,  30, 53, 66, 57, 36, 14, 27, 50, 64, 59, 39, 17, 24, 47, 64, 60, 42, 19,
       22, 45, 61, 61, 45, 22, 19, 42, 60, 64, 47, 24, 17, 39, 59, 64, 50, 27, 14, 36, 57, 66, 53, 30},
    },

    // Band_8_6765
    {
      // Scaler inc range
      {6722, 6765},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 74, 27, 56, 90, 78, 32, 50, 87, 81, 38,
       44, 84,  84, 44, 38, 81, 87, 50, 32, 78, 90, 56, 27, 74, 92, 63},
      // Coefficients 6 taps
      {35, 58, 70, 58, 35, 0,  30, 53, 66, 57, 36, 14, 27, 50, 64, 59, 39, 17, 24, 47, 64, 60, 42, 19,
       22, 45, 61, 61, 45, 22, 19, 42, 60, 64, 47, 24, 17, 39, 59, 64, 50, 27, 14, 36, 57, 66, 53, 30},
    },

    // Band_8_6809
    {
      // Scaler inc range
      {6765, 6809},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 74, 27, 56, 91, 77, 32, 50, 87, 81, 38,
       44, 84,  84, 44, 38, 81, 87, 50, 32, 77, 91, 56, 27, 74, 92, 63},
      // Coefficients 6 taps
      {35, 58, 70, 58, 35, 0,  31, 53, 64, 57, 36, 15, 27, 50, 64, 59, 39, 17, 25, 47, 63, 60, 42, 19,
       22, 45, 61, 61, 45, 22, 19, 42, 60, 63, 47, 25, 17, 39, 59, 64, 50, 27, 15, 36, 57, 64, 53, 31},
    },

    // Band_8_6853
    {
      // Scaler inc range
      {6809, 6853},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 74, 27, 56, 91, 77, 32, 50, 87, 81, 38,
       44, 84,  84, 44, 38, 81, 87, 50, 32, 77, 91, 56, 27, 74, 92, 63},
      // Coefficients 6 taps
      {35, 58, 70, 58, 35, 0,  31, 53, 64, 57, 36, 15, 28, 50, 63, 59, 39, 17, 25, 47, 63, 60, 42, 19,
       22, 45, 61, 61, 45, 22, 19, 42, 60, 63, 47, 25, 17, 39, 59, 63, 50, 28, 15, 36, 57, 64, 53, 31},
    },

    // Band_8_6899
    {
      // Scaler inc range
      {6853, 6899},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 74, 27, 56, 91, 77, 32, 50, 87, 81, 38,
       44, 84,  84, 44, 38, 81, 87, 50, 32, 77, 91, 56, 27, 74, 92, 63},
      // Coefficients 6 taps
      {36, 58, 68, 58, 36, 0,  31, 53, 64, 57, 36, 15, 28, 50, 63, 59, 39, 17, 25, 47, 63, 60, 42, 19,
       22, 45, 61, 61, 45, 22, 19, 42, 60, 63, 47, 25, 17, 39, 59, 63, 50, 28, 15, 36, 57, 64, 53, 31},
    },

    // Band_8_6944
    {
      // Scaler inc range
      {6899, 6944},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 74, 27, 56, 91, 77, 32, 50, 87, 81, 38,
       44, 84,  84, 44, 38, 81, 87, 50, 32, 77, 91, 56, 27, 74, 92, 63},
      // Coefficients 6 taps
      {36, 58, 68, 58, 36, 0,  31, 53, 63, 57, 37, 15, 28, 50, 63, 59, 39, 17, 25, 47, 63, 60, 42, 19,
       22, 45, 61, 61, 45, 22, 19, 42, 60, 63, 47, 25, 17, 39, 59, 63, 50, 28, 15, 37, 57, 63, 53, 31},
    },

    // Band_8_6991
    {
      // Scaler inc range
      {6944, 6991},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 91, 74, 28, 56, 91, 77, 32, 50, 87, 81, 38,
       44, 84,  84, 44, 38, 81, 87, 50, 32, 77, 91, 56, 28, 74, 91, 63},
      // Coefficients 6 taps
      {36, 58, 68, 58, 36, 0,  31, 53, 63, 57, 37, 15, 28, 50, 63, 59, 39, 17, 25, 47, 63, 60, 42, 19,
       22, 45, 61, 61, 45, 22, 19, 42, 60, 63, 47, 25, 17, 39, 59, 63, 50, 28, 15, 37, 57, 63, 53, 31},
    },

    // Band_8_7037
    {
      // Scaler inc range
      {6991, 7037},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 91, 74, 28, 56, 91, 77, 32, 50, 87, 81, 38,
       44, 84,  84, 44, 38, 81, 87, 50, 32, 77, 91, 56, 28, 74, 91, 63},
      // Coefficients 6 taps
      {36, 58, 68, 58, 36, 0,  31, 53, 63, 57, 37, 15, 28, 50, 63, 59, 39, 17, 25, 47, 63, 60, 42, 19,
       22, 45, 61, 61, 45, 22, 19, 42, 60, 63, 47, 25, 17, 39, 59, 63, 50, 28, 15, 37, 57, 63, 53, 31},
    },

    // Band_8_7085
    {
      // Scaler inc range
      {7037, 7085},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 73, 28, 56, 91, 77, 32, 50, 87, 81, 38,
       44, 84,  84, 44, 38, 81, 87, 50, 32, 77, 91, 56, 28, 73, 92, 63},
      // Coefficients 6 taps
      {36, 58, 68, 58, 36, 0,  31, 53, 63, 57, 37, 15, 28, 50, 63, 59, 39, 17, 25, 47, 63, 60, 42, 19,
       22, 45, 61, 61, 45, 22, 19, 42, 60, 63, 47, 25, 17, 39, 59, 63, 50, 28, 15, 37, 57, 63, 53, 31},
    },

    // Band_8_7133
    {
      // Scaler inc range
      {7085, 7133},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 73, 28, 56, 90, 77, 33, 50, 87, 81, 38,
       44, 84,  84, 44, 38, 81, 87, 50, 33, 77, 90, 56, 28, 73, 92, 63},
      // Coefficients 6 taps
      {36, 58, 68, 58, 36, 0,  31, 53, 63, 57, 37, 15, 28, 50, 63, 59, 39, 17, 25, 47, 62, 60, 42, 20,
       22, 45, 61, 61, 45, 22, 20, 42, 60, 62, 47, 25, 17, 39, 59, 63, 50, 28, 15, 37, 57, 63, 53, 31},
    },

    // Band_8_7182
    {
      // Scaler inc range
      {7133, 7182},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 73, 28, 56, 90, 77, 33, 50, 87, 81, 38,
       44, 84,  84, 44, 38, 81, 87, 50, 33, 77, 90, 56, 28, 73, 92, 63},
      // Coefficients 6 taps
      {36, 58, 68, 58, 36, 0,  31, 53, 63, 57, 37, 15, 28, 50, 63, 59, 39, 17, 25, 47, 62, 60, 42, 20,
       22, 45, 61, 61, 45, 22, 20, 42, 60, 62, 47, 25, 17, 39, 59, 63, 50, 28, 15, 37, 57, 63, 53, 31},
    },

    // Band_8_7232
    {
      // Scaler inc range
      {7182, 7232},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 73, 28, 56, 90, 77, 33, 50, 87, 81, 38,
       44, 84,  84, 44, 38, 81, 87, 50, 33, 77, 90, 56, 28, 73, 92, 63},
      // Coefficients 6 taps
      {36, 58, 68, 58, 36, 0,  31, 53, 63, 57, 37, 15, 28, 50, 64, 58, 39, 17, 25, 47, 62, 60, 42, 20,
       22, 45, 61, 61, 45, 22, 20, 42, 60, 62, 47, 25, 17, 39, 58, 64, 50, 28, 15, 37, 57, 63, 53, 31},
    },

    // Band_8_7282
    {
      // Scaler inc range
      {7232, 7282},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 73, 28, 56, 90, 77, 33, 50, 87, 81, 38,
       44, 84,  84, 44, 38, 81, 87, 50, 33, 77, 90, 56, 28, 73, 92, 63},
      // Coefficients 6 taps
      {36, 58, 68, 58, 36, 0,  31, 53, 63, 57, 37, 15, 28, 50, 64, 58, 39, 17, 25, 47, 62, 60, 42, 20,
       22, 45, 61, 61, 45, 22, 20, 42, 60, 62, 47, 25, 17, 39, 58, 64, 50, 28, 15, 37, 57, 63, 53, 31},
    },

    // Band_8_7333
    {
      // Scaler inc range
      {7282, 7333},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 73, 28, 56, 90, 77, 33, 50, 87, 81, 38,
       44, 84,  84, 44, 38, 81, 87, 50, 33, 77, 90, 56, 28, 73, 92, 63},
      // Coefficients 6 taps
      {36, 58, 68, 58, 36, 0,  31, 53, 63, 57, 37, 15, 28, 50, 64, 58, 39, 17, 25, 47, 62, 60, 42, 20,
       22, 45, 61, 61, 45, 22, 20, 42, 60, 62, 47, 25, 17, 39, 58, 64, 50, 28, 15, 37, 57, 63, 53, 31},
    },

    // Band_8_7384
    {
      // Scaler inc range
      {7333, 7384},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 73, 28, 56, 90, 77, 33, 50, 87, 81, 38,
       44, 84,  84, 44, 38, 81, 87, 50, 33, 77, 90, 56, 28, 73, 92, 63},
      // Coefficients 6 taps
      {36, 58, 68, 58, 36, 0,  31, 53, 63, 57, 37, 15, 28, 50, 64, 58, 39, 17, 25, 47, 62, 60, 42, 20,
       22, 45, 61, 61, 45, 22, 20, 42, 60, 62, 47, 25, 17, 39, 58, 64, 50, 28, 15, 37, 57, 63, 53, 31},
    },

    // Band_8_7437
    {
      // Scaler inc range
      {7384, 7437},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 73, 28, 56, 90, 77, 33, 50, 87, 81, 38,
       44, 84,  84, 44, 38, 81, 87, 50, 33, 77, 90, 56, 28, 73, 92, 63},
      // Coefficients 6 taps
      {36, 58, 68, 58, 36, 0,  31, 53, 63, 57, 37, 15, 28, 50, 64, 58, 39, 17, 25, 47, 62, 60, 42, 20,
       22, 45, 61, 61, 45, 22, 20, 42, 60, 62, 47, 25, 17, 39, 58, 64, 50, 28, 15, 37, 57, 63, 53, 31},
    },

    // Band_8_7490
    {
      // Scaler inc range
      {7437, 7490},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 73, 28, 56, 90, 77, 33, 50, 87, 81, 38,
       44, 84,  84, 44, 38, 81, 87, 50, 33, 77, 90, 56, 28, 73, 92, 63},
      // Coefficients 6 taps
      {36, 58, 68, 58, 36, 0,  31, 53, 63, 57, 37, 15, 28, 50, 64, 58, 39, 17, 25, 47, 62, 60, 42, 20,
       22, 45, 61, 61, 45, 22, 20, 42, 60, 62, 47, 25, 17, 39, 58, 64, 50, 28, 15, 37, 57, 63, 53, 31},
    },

    // Band_8_7544
    {
      // Scaler inc range
      {7490, 7544},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 73, 28, 56, 90, 77, 33, 50, 87, 81, 38,
       44, 84,  84, 44, 38, 81, 87, 50, 33, 77, 90, 56, 28, 73, 92, 63},
      // Coefficients 6 taps
      {36, 58, 68, 58, 36, 0,  31, 53, 63, 57, 37, 15, 28, 50, 64, 58, 39, 17, 25, 47, 62, 60, 42, 20,
       23, 45, 60, 60, 45, 23, 20, 42, 60, 62, 47, 25, 17, 39, 58, 64, 50, 28, 15, 37, 57, 63, 53, 31},
    },

    // Band_8_7598
    {
      // Scaler inc range
      {7544, 7598},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 73, 28, 56, 90, 77, 33, 50, 87, 81, 38,
       44, 84,  84, 44, 38, 81, 87, 50, 33, 77, 90, 56, 28, 73, 92, 63},
      // Coefficients 6 taps
      {36, 58, 68, 58, 36, 0,  31, 53, 63, 57, 37, 15, 28, 50, 62, 58, 40, 18, 25, 47, 63, 59, 42, 20,
       23, 45, 60, 60, 45, 23, 20, 42, 59, 63, 47, 25, 18, 40, 58, 62, 50, 28, 15, 37, 57, 63, 53, 31},
    },

    // Band_8_7654
    {
      // Scaler inc range
      {7598, 7654},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 73, 28, 56, 90, 77, 33, 50, 87, 81, 38,
       44, 84,  84, 44, 38, 81, 87, 50, 33, 77, 90, 56, 28, 73, 92, 63},
      // Coefficients 6 taps
      {36, 58, 68, 58, 36, 0,  31, 53, 63, 57, 37, 15, 28, 50, 62, 58, 40, 18, 25, 47, 63, 59, 42, 20,
       23, 45, 60, 60, 45, 23, 20, 42, 59, 63, 47, 25, 18, 40, 58, 62, 50, 28, 15, 37, 57, 63, 53, 31},
    },

    // Band_8_7710
    {
      // Scaler inc range
      {7654, 7710},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 73, 28, 56, 90, 77, 33, 50, 87, 81, 38,
       44, 84,  84, 44, 38, 81, 87, 50, 33, 77, 90, 56, 28, 73, 92, 63},
      // Coefficients 6 taps
      {36, 58, 68, 58, 36, 0,  31, 53, 62, 57, 37, 16, 28, 50, 62, 58, 40, 18, 25, 47, 63, 59, 42, 20,
       23, 45, 60, 60, 45, 23, 20, 42, 59, 63, 47, 25, 18, 40, 58, 62, 50, 28, 16, 37, 57, 62, 53, 31},
    },

    // Band_8_7767
    {
      // Scaler inc range
      {7710, 7767},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 73, 28, 56, 90, 77, 33, 50, 87, 81, 38,
       44, 84,  84, 44, 38, 81, 87, 50, 33, 77, 90, 56, 28, 73, 92, 63},
      // Coefficients 6 taps
      {36, 58, 68, 58, 36, 0,  31, 53, 62, 57, 37, 16, 28, 50, 62, 58, 40, 18, 25, 47, 63, 59, 42, 20,
       23, 45, 60, 60, 45, 23, 20, 42, 59, 63, 47, 25, 18, 40, 58, 62, 50, 28, 16, 37, 57, 62, 53, 31},
    },

    // Band_8_7825
    {
      // Scaler inc range
      {7767, 7825},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 73, 28, 56, 90, 77, 33, 50, 87, 81, 38,
       44, 84,  84, 44, 38, 81, 87, 50, 33, 77, 90, 56, 28, 73, 92, 63},
      // Coefficients 6 taps
      {36, 58, 68, 58, 36, 0,  31, 53, 62, 57, 37, 16, 28, 50, 62, 58, 40, 18, 25, 47, 63, 59, 42, 20,
       23, 45, 60, 60, 45, 23, 20, 42, 59, 63, 47, 25, 18, 40, 58, 62, 50, 28, 16, 37, 57, 62, 53, 31},
    },

    // Band_8_7884
    {
      // Scaler inc range
      {7825, 7884},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 73, 28, 56, 90, 77, 33, 50, 87, 81, 38,
       44, 84,  84, 44, 38, 81, 87, 50, 33, 77, 90, 56, 28, 73, 92, 63},
      // Coefficients 6 taps
      {36, 58, 68, 58, 36, 0,  31, 52, 63, 57, 37, 16, 28, 50, 62, 58, 40, 18, 25, 47, 63, 59, 42, 20,
       23, 45, 60, 60, 45, 23, 20, 42, 59, 63, 47, 25, 18, 40, 58, 62, 50, 28, 16, 37, 57, 63, 52, 31},
    },

    // Band_8_7944
    {
      // Scaler inc range
      {7884, 7944},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 73, 28, 56, 90, 77, 33, 50, 87, 81, 38,
       44, 84,  84, 44, 38, 81, 87, 50, 33, 77, 90, 56, 28, 73, 92, 63},
      // Coefficients 6 taps
      {36, 58, 68, 58, 36, 0,  31, 52, 64, 56, 37, 16, 28, 50, 62, 58, 40, 18, 26, 47, 62, 59, 42, 20,
       23, 45, 60, 60, 45, 23, 20, 42, 59, 62, 47, 26, 18, 40, 58, 62, 50, 28, 16, 37, 56, 64, 52, 31},
    },

    // Band_8_8004
    {
      // Scaler inc range
      {7944, 8004},
      // Coefficients 4 taps
      {76, 104, 76, 0,  63, 92, 73, 28, 56, 90, 77, 33, 50, 86, 81, 39,
       44, 84,  84, 44, 39, 81, 86, 50, 33, 77, 90, 56, 28, 73, 92, 63},
      // Coefficients 6 taps
      {36, 58, 68, 58, 36, 0,  31, 52, 64, 56, 37, 16, 28, 50, 62, 58, 40, 18, 26, 47, 62, 59, 42, 20,
       23, 45, 60, 60, 45, 23, 20, 42, 59, 62, 47, 26, 18, 40, 58, 62, 50, 28, 16, 37, 56, 64, 52, 31},
    },

    // Band_8_8066
    {
      // Scaler inc range
      {8004, 8066},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 91, 73, 29, 56, 90, 77, 33, 50, 86, 81, 39,
       44, 84,  84, 44, 39, 81, 86, 50, 33, 77, 90, 56, 29, 73, 91, 63},
      // Coefficients 6 taps
      {36, 58, 68, 58, 36, 0,  31, 52, 64, 56, 37, 16, 28, 50, 62, 58, 40, 18, 26, 47, 62, 59, 42, 20,
       23, 45, 60, 60, 45, 23, 20, 42, 59, 62, 47, 26, 18, 40, 58, 62, 50, 28, 16, 37, 56, 64, 52, 31},
    },

    // Band_8_8128
    {
      // Scaler inc range
      {8066, 8128},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 91, 73, 29, 56, 90, 77, 33, 50, 86, 81, 39,
       44, 84,  84, 44, 39, 81, 86, 50, 33, 77, 90, 56, 29, 73, 91, 63},
      // Coefficients 6 taps
      {36, 58, 68, 58, 36, 0,  31, 52, 64, 56, 37, 16, 28, 50, 62, 58, 40, 18, 26, 47, 62, 59, 42, 20,
       23, 45, 60, 60, 45, 23, 20, 42, 59, 62, 47, 26, 18, 40, 58, 62, 50, 28, 16, 37, 56, 64, 52, 31},
    },

    // Band_8_8192
    {
      // Scaler inc range
      {8128, 8192},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 91, 73, 29, 56, 90, 77, 33, 50, 86, 81, 39,
       44, 84,  84, 44, 39, 81, 86, 50, 33, 77, 90, 56, 29, 73, 91, 63},
      // Coefficients 6 taps
      {36, 58, 68, 58, 36, 0,  31, 52, 64, 56, 37, 16, 29, 50, 61, 58, 40, 18, 26, 47, 62, 59, 42, 20,
       23, 45, 60, 60, 45, 23, 20, 42, 59, 62, 47, 26, 18, 40, 58, 61, 50, 29, 16, 37, 56, 64, 52, 31},
    },

    // Band_8_8257
    {
      // Scaler inc range
      {8192, 8257},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 91, 73, 29, 56, 90, 77, 33, 50, 86, 81, 39,
       44, 84,  84, 44, 39, 81, 86, 50, 33, 77, 90, 56, 29, 73, 91, 63},
      // Coefficients 6 taps
      {37, 58, 66, 58, 37, 0,  31, 52, 64, 56, 37, 16, 29, 50, 61, 58, 40, 18, 26, 47, 62, 59, 42, 20,
       23, 45, 60, 60, 45, 23, 20, 42, 59, 62, 47, 26, 18, 40, 58, 61, 50, 29, 16, 37, 56, 64, 52, 31},
    },

    // Band_8_8322
    {
      // Scaler inc range
      {8257, 8322},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 91, 73, 29, 57, 89, 77, 33, 50, 87, 80, 39,
       44, 84,  84, 44, 39, 80, 87, 50, 33, 77, 89, 57, 29, 73, 91, 63},
      // Coefficients 6 taps
      {37, 58, 66, 58, 37, 0,  32, 52, 63, 56, 37, 16, 29, 50, 61, 58, 40, 18, 26, 47, 61, 59, 42, 21,
       23, 45, 60, 60, 45, 23, 21, 42, 59, 61, 47, 26, 18, 40, 58, 61, 50, 29, 16, 37, 56, 63, 52, 32},
    },

    // Band_8_8389
    {
      // Scaler inc range
      {8322, 8389},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 91, 73, 29, 57, 89, 77, 33, 50, 87, 80, 39,
       44, 84,  84, 44, 39, 80, 87, 50, 33, 77, 89, 57, 29, 73, 91, 63},
      // Coefficients 6 taps
      {37, 58, 66, 58, 37, 0,  32, 52, 63, 56, 37, 16, 29, 50, 61, 58, 40, 18, 26, 47, 61, 59, 42, 21,
       23, 45, 60, 60, 45, 23, 21, 42, 59, 61, 47, 26, 18, 40, 58, 61, 50, 29, 16, 37, 56, 63, 52, 32},
    },

    // Band_8_8456
    {
      // Scaler inc range
      {8389, 8456},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 91, 73, 29, 57, 89, 77, 33, 50, 87, 80, 39,
       44, 84,  84, 44, 39, 80, 87, 50, 33, 77, 89, 57, 29, 73, 91, 63},
      // Coefficients 6 taps
      {37, 58, 66, 58, 37, 0,  32, 52, 63, 56, 37, 16, 29, 50, 61, 58, 40, 18, 26, 47, 61, 59, 42, 21,
       23, 45, 60, 60, 45, 23, 21, 42, 59, 61, 47, 26, 18, 40, 58, 61, 50, 29, 16, 37, 56, 63, 52, 32},
    },

    // Band_8_8525
    {
      // Scaler inc range
      {8456, 8525},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 91, 73, 29, 57, 89, 77, 33, 50, 87, 80, 39,
       44, 84,  84, 44, 39, 80, 87, 50, 33, 77, 89, 57, 29, 73, 91, 63},
      // Coefficients 6 taps
      {37, 58, 66, 58, 37, 0,  32, 52, 63, 56, 37, 16, 29, 50, 61, 58, 40, 18, 26, 47, 61, 59, 42, 21,
       23, 45, 60, 60, 45, 23, 21, 42, 59, 61, 47, 26, 18, 40, 58, 61, 50, 29, 16, 37, 56, 63, 52, 32},
    },

    // Band_8_8595
    {
      // Scaler inc range
      {8525, 8595},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 91, 73, 29, 57, 89, 77, 33, 50, 87, 80, 39,
       44, 84,  84, 44, 39, 80, 87, 50, 33, 77, 89, 57, 29, 73, 91, 63},
      // Coefficients 6 taps
      {37, 58, 66, 58, 37, 0,  32, 52, 63, 56, 37, 16, 29, 50, 61, 58, 40, 18, 26, 47, 61, 59, 42, 21,
       23, 45, 60, 60, 45, 23, 21, 42, 59, 61, 47, 26, 18, 40, 58, 61, 50, 29, 16, 37, 56, 63, 52, 32},
    },

    // Band_8_8666
    {
      // Scaler inc range
      {8595, 8666},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 91, 73, 29, 57, 88, 77, 34, 50, 87, 80, 39,
       44, 84,  84, 44, 39, 80, 87, 50, 34, 77, 88, 57, 29, 73, 91, 63},
      // Coefficients 6 taps
      {37, 58, 66, 58, 37, 0,  32, 52, 63, 56, 37, 16, 29, 50, 61, 58, 40, 18, 26, 47, 61, 59, 42, 21,
       23, 45, 60, 60, 45, 23, 21, 42, 59, 61, 47, 26, 18, 40, 58, 61, 50, 29, 16, 37, 56, 63, 52, 32},
    },

    // Band_8_8738
    {
      // Scaler inc range
      {8666, 8738},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 91, 73, 29, 57, 88, 77, 34, 50, 87, 80, 39,
       45, 83,  83, 45, 39, 80, 87, 50, 34, 77, 88, 57, 29, 73, 91, 63},
      // Coefficients 6 taps
      {37, 58, 66, 58, 37, 0,  32, 52, 63, 56, 37, 16, 29, 50, 61, 58, 40, 18, 26, 47, 61, 59, 42, 21,
       23, 45, 60, 60, 45, 23, 21, 42, 59, 61, 47, 26, 18, 40, 58, 61, 50, 29, 16, 37, 56, 63, 52, 32},
    },

    // Band_8_8812
    {
      // Scaler inc range
      {8738, 8812},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 91, 73, 29, 57, 88, 77, 34, 50, 87, 80, 39,
       45, 83,  83, 45, 39, 80, 87, 50, 34, 77, 88, 57, 29, 73, 91, 63},
      // Coefficients 6 taps
      {37, 58, 66, 58, 37, 0,  32, 52, 63, 56, 37, 16, 29, 50, 62, 57, 40, 18, 26, 47, 61, 59, 42, 21,
       23, 45, 60, 60, 45, 23, 21, 42, 59, 61, 47, 26, 18, 40, 57, 62, 50, 29, 16, 37, 56, 63, 52, 32},
    },

    // Band_8_8886
    {
      // Scaler inc range
      {8812, 8886},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 91, 73, 29, 57, 88, 77, 34, 50, 87, 80, 39,
       45, 83,  83, 45, 39, 80, 87, 50, 34, 77, 88, 57, 29, 73, 91, 63},
      // Coefficients 6 taps
      {37, 58, 66, 58, 37, 0,  32, 52, 63, 56, 37, 16, 29, 50, 62, 57, 40, 18, 26, 47, 61, 59, 42, 21,
       23, 45, 60, 60, 45, 23, 21, 42, 59, 61, 47, 26, 18, 40, 57, 62, 50, 29, 16, 37, 56, 63, 52, 32},
    },

    // Band_8_8962
    {
      // Scaler inc range
      {8886, 8962},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 91, 73, 29, 57, 88, 77, 34, 51, 86, 80, 39,
       45, 83,  83, 45, 39, 80, 86, 51, 34, 77, 88, 57, 29, 73, 91, 63},
      // Coefficients 6 taps
      {37, 58, 66, 58, 37, 0,  32, 52, 62, 56, 37, 17, 29, 50, 61, 57, 40, 19, 26, 47, 61, 59, 42, 21,
       23, 45, 60, 60, 45, 23, 21, 42, 59, 61, 47, 26, 19, 40, 57, 61, 50, 29, 17, 37, 56, 62, 52, 32},
    },

    // Band_8_9039
    {
      // Scaler inc range
      {8962, 9039},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 91, 73, 29, 57, 88, 77, 34, 51, 86, 80, 39,
       45, 83,  83, 45, 39, 80, 86, 51, 34, 77, 88, 57, 29, 73, 91, 63},
      // Coefficients 6 taps
      {37, 58, 66, 58, 37, 0,  32, 52, 62, 56, 37, 17, 29, 50, 61, 57, 40, 19, 26, 47, 61, 59, 42, 21,
       23, 45, 60, 60, 45, 23, 21, 42, 59, 61, 47, 26, 19, 40, 57, 61, 50, 29, 17, 37, 56, 62, 52, 32},
    },

    // Band_8_9118
    {
      // Scaler inc range
      {9039, 9118},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 91, 73, 29, 57, 88, 77, 34, 51, 86, 80, 39,
       45, 83,  83, 45, 39, 80, 86, 51, 34, 77, 88, 57, 29, 73, 91, 63},
      // Coefficients 6 taps
      {37, 58, 66, 58, 37, 0,  32, 52, 62, 56, 37, 17, 29, 50, 61, 57, 40, 19, 26, 47, 61, 59, 42, 21,
       24, 45, 59, 59, 45, 24, 21, 42, 59, 61, 47, 26, 19, 40, 57, 61, 50, 29, 17, 37, 56, 62, 52, 32},
    },

    // Band_8_9198
    {
      // Scaler inc range
      {9118, 9198},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 91, 73, 29, 57, 88, 77, 34, 51, 86, 80, 39,
       45, 83,  83, 45, 39, 80, 86, 51, 34, 77, 88, 57, 29, 73, 91, 63},
      // Coefficients 6 taps
      {37, 58, 66, 58, 37, 0,  32, 52, 62, 56, 37, 17, 29, 50, 61, 57, 40, 19, 26, 47, 62, 58, 42, 21,
       24, 45, 59, 59, 45, 24, 21, 42, 58, 62, 47, 26, 19, 40, 57, 61, 50, 29, 17, 37, 56, 62, 52, 32},
    },

    // Band_8_9279
    {
      // Scaler inc range
      {9198, 9279},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 91, 73, 29, 57, 88, 77, 34, 51, 86, 80, 39,
       45, 83,  83, 45, 39, 80, 86, 51, 34, 77, 88, 57, 29, 73, 91, 63},
      // Coefficients 6 taps
      {37, 58, 66, 58, 37, 0,  32, 52, 62, 56, 37, 17, 29, 50, 61, 57, 40, 19, 26, 47, 62, 58, 42, 21,
       24, 45, 59, 59, 45, 24, 21, 42, 58, 62, 47, 26, 19, 40, 57, 61, 50, 29, 17, 37, 56, 62, 52, 32},
    },

    // Band_8_9362
    {
      // Scaler inc range
      {9279, 9362},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 90, 73, 30, 57, 88, 77, 34, 51, 86, 80, 39,
       45, 83,  83, 45, 39, 80, 86, 51, 34, 77, 88, 57, 30, 73, 90, 63},
      // Coefficients 6 taps
      {37, 58, 66, 58, 37, 0,  32, 52, 62, 56, 37, 17, 29, 50, 61, 57, 40, 19, 26, 47, 62, 58, 42, 21,
       24, 45, 59, 59, 45, 24, 21, 42, 58, 62, 47, 26, 19, 40, 57, 61, 50, 29, 17, 37, 56, 62, 52, 32},
    },

    // Band_8_9447
    {
      // Scaler inc range
      {9362, 9447},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 90, 73, 30, 57, 88, 77, 34, 51, 86, 80, 39,
       45, 83,  83, 45, 39, 80, 86, 51, 34, 77, 88, 57, 30, 73, 90, 63},
      // Coefficients 6 taps
      {37, 58, 66, 58, 37, 0,  32, 52, 62, 56, 37, 17, 29, 50, 61, 57, 40, 19, 26, 47, 62, 58, 42, 21,
       24, 45, 59, 59, 45, 24, 21, 42, 58, 62, 47, 26, 19, 40, 57, 61, 50, 29, 17, 37, 56, 62, 52, 32},
    },

    // Band_8_9533
    {
      // Scaler inc range
      {9447, 9533},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 90, 73, 30, 57, 88, 77, 34, 51, 86, 80, 39,
       45, 83,  83, 45, 39, 80, 86, 51, 34, 77, 88, 57, 30, 73, 90, 63},
      // Coefficients 6 taps
      {37, 58, 66, 58, 37, 0,  32, 52, 62, 56, 37, 17, 29, 50, 61, 57, 40, 19, 26, 47, 62, 58, 42, 21,
       24, 45, 59, 59, 45, 24, 21, 42, 58, 62, 47, 26, 19, 40, 57, 61, 50, 29, 17, 37, 56, 62, 52, 32},
    },

    // Band_8_9620
    {
      // Scaler inc range
      {9533, 9620},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 90, 73, 30, 57, 88, 77, 34, 51, 86, 80, 39,
       45, 83,  83, 45, 39, 80, 86, 51, 34, 77, 88, 57, 30, 73, 90, 63},
      // Coefficients 6 taps
      {37, 58, 66, 58, 37, 0,  32, 52, 62, 56, 37, 17, 29, 50, 61, 57, 40, 19, 26, 47, 62, 58, 42, 21,
       24, 45, 59, 59, 45, 24, 21, 42, 58, 62, 47, 26, 19, 40, 57, 61, 50, 29, 17, 37, 56, 62, 52, 32},
    },

    // Band_8_9709
    {
      // Scaler inc range
      {9620, 9709},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 90, 73, 30, 57, 88, 77, 34, 51, 86, 80, 39,
       45, 83,  83, 45, 39, 80, 86, 51, 34, 77, 88, 57, 30, 73, 90, 63},
      // Coefficients 6 taps
      {37, 58, 66, 58, 37, 0,  32, 52, 62, 56, 37, 17, 29, 50, 61, 57, 40, 19, 26, 47, 62, 58, 42, 21,
       24, 45, 59, 59, 45, 24, 21, 42, 58, 62, 47, 26, 19, 40, 57, 61, 50, 29, 17, 37, 56, 62, 52, 32},
    },

    // Band_8_9800
    {
      // Scaler inc range
      {9709, 9800},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 90, 73, 30, 57, 88, 77, 34, 51, 86, 80, 39,
       45, 83,  83, 45, 39, 80, 86, 51, 34, 77, 88, 57, 30, 73, 90, 63},
      // Coefficients 6 taps
      {37, 58, 66, 58, 37, 0,  32, 52, 62, 56, 37, 17, 29, 50, 61, 57, 40, 19, 26, 47, 62, 58, 42, 21,
       24, 45, 59, 59, 45, 24, 21, 42, 58, 62, 47, 26, 19, 40, 57, 61, 50, 29, 17, 37, 56, 62, 52, 32},
    },

    // Band_8_9892
    {
      // Scaler inc range
      {9800, 9892},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 90, 73, 30, 57, 88, 77, 34, 51, 86, 80, 39,
       45, 83,  83, 45, 39, 80, 86, 51, 34, 77, 88, 57, 30, 73, 90, 63},
      // Coefficients 6 taps
      {37, 58, 66, 58, 37, 0,  32, 52, 61, 56, 38, 17, 29, 50, 61, 57, 40, 19, 27, 47, 61, 58, 42, 21,
       24, 45, 59, 59, 45, 24, 21, 42, 58, 61, 47, 27, 19, 40, 57, 61, 50, 29, 17, 38, 56, 61, 52, 32},
    },

    // Band_8_9986
    {
      // Scaler inc range
      {9892, 9986},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 90, 73, 30, 57, 88, 77, 34, 51, 86, 80, 39,
       45, 83,  83, 45, 39, 80, 86, 51, 34, 77, 88, 57, 30, 73, 90, 63},
      // Coefficients 6 taps
      {37, 58, 66, 58, 37, 0,  32, 52, 61, 56, 38, 17, 29, 50, 61, 57, 40, 19, 27, 47, 61, 58, 42, 21,
       24, 45, 59, 59, 45, 24, 21, 42, 58, 61, 47, 27, 19, 40, 57, 61, 50, 29, 17, 38, 56, 61, 52, 32},
    },

    // Band_8_10082
    {
      // Scaler inc range
      {9986, 10082},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 90, 73, 30, 57, 88, 77, 34, 51, 86, 80, 39,
       45, 83,  83, 45, 39, 80, 86, 51, 34, 77, 88, 57, 30, 73, 90, 63},
      // Coefficients 6 taps
      {37, 58, 66, 58, 37, 0,  32, 52, 61, 56, 38, 17, 29, 50, 61, 57, 40, 19, 27, 47, 61, 58, 42, 21,
       24, 45, 59, 59, 45, 24, 21, 42, 58, 61, 47, 27, 19, 40, 57, 61, 50, 29, 17, 38, 56, 61, 52, 32},
    },

    // Band_8_10180
    {
      // Scaler inc range
      {10082, 10180},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 90, 73, 30, 57, 88, 77, 34, 51, 86, 80, 39,
       45, 83,  83, 45, 39, 80, 86, 51, 34, 77, 88, 57, 30, 73, 90, 63},
      // Coefficients 6 taps
      {37, 58, 66, 58, 37, 0,  32, 52, 61, 56, 38, 17, 29, 50, 61, 57, 40, 19, 27, 47, 61, 58, 42, 21,
       24, 45, 59, 59, 45, 24, 21, 42, 58, 61, 47, 27, 19, 40, 57, 61, 50, 29, 17, 38, 56, 61, 52, 32},
    },

    // Band_8_10280
    {
      // Scaler inc range
      {10180, 10280},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 90, 73, 30, 57, 88, 77, 34, 51, 86, 80, 39,
       45, 83,  83, 45, 39, 80, 86, 51, 34, 77, 88, 57, 30, 73, 90, 63},
      // Coefficients 6 taps
      {37, 58, 66, 58, 37, 0,  32, 52, 61, 56, 38, 17, 29, 50, 61, 57, 40, 19, 27, 47, 61, 58, 42, 21,
       24, 45, 59, 59, 45, 24, 21, 42, 58, 61, 47, 27, 19, 40, 57, 61, 50, 29, 17, 38, 56, 61, 52, 32},
    },

    // Band_8_10382
    {
      // Scaler inc range
      {10280, 10382},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 90, 73, 30, 57, 88, 77, 34, 51, 86, 80, 39,
       45, 83,  83, 45, 39, 80, 86, 51, 34, 77, 88, 57, 30, 73, 90, 63},
      // Coefficients 6 taps
      {37, 58, 66, 58, 37, 0,  32, 52, 61, 56, 38, 17, 29, 50, 61, 57, 40, 19, 27, 47, 60, 58, 42, 22,
       24, 45, 59, 59, 45, 24, 22, 42, 58, 60, 47, 27, 19, 40, 57, 61, 50, 29, 17, 38, 56, 61, 52, 32},
    },

    // Band_8_10486
    {
      // Scaler inc range
      {10382, 10486},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 90, 73, 30, 57, 88, 77, 34, 51, 86, 80, 39,
       45, 83,  83, 45, 39, 80, 86, 51, 34, 77, 88, 57, 30, 73, 90, 63},
      // Coefficients 6 taps
      {37, 58, 66, 58, 37, 0,  32, 52, 61, 56, 38, 17, 29, 50, 61, 57, 40, 19, 27, 47, 60, 58, 42, 22,
       24, 45, 59, 59, 45, 24, 22, 42, 58, 60, 47, 27, 19, 40, 57, 61, 50, 29, 17, 38, 56, 61, 52, 32},
    },

    // Band_8_10592
    {
      // Scaler inc range
      {10486, 10592},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 90, 73, 30, 57, 88, 77, 34, 51, 85, 80, 40,
       45, 83,  83, 45, 40, 80, 85, 51, 34, 77, 88, 57, 30, 73, 90, 63},
      // Coefficients 6 taps
      {37, 58, 66, 58, 37, 0,  32, 52, 61, 55, 38, 18, 29, 50, 61, 57, 40, 19, 27, 47, 59, 58, 43, 22,
       24, 45, 59, 59, 45, 24, 22, 43, 58, 59, 47, 27, 19, 40, 57, 61, 50, 29, 18, 38, 55, 61, 52, 32},
    },

    // Band_8_10700
    {
      // Scaler inc range
      {10592, 10700},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 90, 73, 30, 57, 87, 77, 35, 51, 85, 80, 40,
       45, 83,  83, 45, 40, 80, 85, 51, 35, 77, 87, 57, 30, 73, 90, 63},
      // Coefficients 6 taps
      {38, 58, 64, 58, 38, 0,  32, 52, 61, 55, 38, 18, 30, 50, 60, 57, 40, 19, 27, 47, 59, 58, 43, 22,
       24, 45, 59, 59, 45, 24, 22, 43, 58, 59, 47, 27, 19, 40, 57, 60, 50, 30, 18, 38, 55, 61, 52, 32},
    },

    // Band_8_10810
    {
      // Scaler inc range
      {10700, 10810},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 89, 73, 31, 57, 87, 77, 35, 51, 85, 80, 40,
       45, 83,  83, 45, 40, 80, 85, 51, 35, 77, 87, 57, 31, 73, 89, 63},
      // Coefficients 6 taps
      {38, 58, 64, 58, 38, 0,  32, 52, 61, 55, 38, 18, 30, 50, 60, 57, 40, 19, 27, 47, 59, 58, 43, 22,
       24, 45, 59, 59, 45, 24, 22, 43, 58, 59, 47, 27, 19, 40, 57, 60, 50, 30, 18, 38, 55, 61, 52, 32},
    },

    // Band_8_10923
    {
      // Scaler inc range
      {10810, 10923},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 89, 73, 31, 57, 87, 77, 35, 51, 85, 80, 40,
       45, 83,  83, 45, 40, 80, 85, 51, 35, 77, 87, 57, 31, 73, 89, 63},
      // Coefficients 6 taps
      {38, 58, 64, 58, 38, 0,  32, 52, 61, 55, 38, 18, 30, 50, 60, 57, 40, 19, 27, 47, 59, 58, 43, 22,
       24, 45, 59, 59, 45, 24, 22, 43, 58, 59, 47, 27, 19, 40, 57, 60, 50, 30, 18, 38, 55, 61, 52, 32},
    },

    // Band_8_11038
    {
      // Scaler inc range
      {10923, 11038},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 89, 73, 31, 57, 87, 77, 35, 51, 85, 80, 40,
       45, 83,  83, 45, 40, 80, 85, 51, 35, 77, 87, 57, 31, 73, 89, 63},
      // Coefficients 6 taps
      {38, 58, 64, 58, 38, 0,  32, 52, 61, 55, 38, 18, 30, 50, 59, 57, 40, 20, 27, 47, 59, 58, 43, 22,
       24, 45, 59, 59, 45, 24, 22, 43, 58, 59, 47, 27, 20, 40, 57, 59, 50, 30, 18, 38, 55, 61, 52, 32},
    },

    // Band_8_11155
    {
      // Scaler inc range
      {11038, 11155},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 89, 73, 31, 57, 87, 77, 35, 51, 85, 80, 40,
       45, 83,  83, 45, 40, 80, 85, 51, 35, 77, 87, 57, 31, 73, 89, 63},
      // Coefficients 6 taps
      {38, 58, 64, 58, 38, 0,  32, 52, 61, 55, 38, 18, 30, 50, 59, 57, 40, 20, 27, 47, 59, 58, 43, 22,
       24, 45, 59, 59, 45, 24, 22, 43, 58, 59, 47, 27, 20, 40, 57, 59, 50, 30, 18, 38, 55, 61, 52, 32},
    },

    // Band_8_11275
    {
      // Scaler inc range
      {11155, 11275},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 89, 73, 31, 57, 88, 76, 35, 51, 85, 80, 40,
       45, 83,  83, 45, 40, 80, 85, 51, 35, 76, 88, 57, 31, 73, 89, 63},
      // Coefficients 6 taps
      {38, 58, 64, 58, 38, 0,  32, 52, 61, 55, 38, 18, 30, 50, 59, 57, 40, 20, 27, 47, 59, 58, 43, 22,
       24, 45, 59, 59, 45, 24, 22, 43, 58, 59, 47, 27, 20, 40, 57, 59, 50, 30, 18, 38, 55, 61, 52, 32},
    },

    // Band_8_11398
    {
      // Scaler inc range
      {11275, 11398},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 89, 73, 31, 57, 88, 76, 35, 51, 85, 80, 40,
       45, 83,  83, 45, 40, 80, 85, 51, 35, 76, 88, 57, 31, 73, 89, 63},
      // Coefficients 6 taps
      {38, 58, 64, 58, 38, 0,  32, 52, 61, 55, 38, 18, 30, 50, 59, 57, 40, 20, 27, 47, 59, 58, 43, 22,
       24, 45, 59, 59, 45, 24, 22, 43, 58, 59, 47, 27, 20, 40, 57, 59, 50, 30, 18, 38, 55, 61, 52, 32},
    },

    // Band_8_11523
    {
      // Scaler inc range
      {11398, 11523},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 89, 73, 31, 57, 88, 76, 35, 51, 85, 80, 40,
       45, 83,  83, 45, 40, 80, 85, 51, 35, 76, 88, 57, 31, 73, 89, 63},
      // Coefficients 6 taps
      {38, 58, 64, 58, 38, 0,  32, 52, 61, 55, 38, 18, 30, 50, 59, 57, 40, 20, 27, 47, 59, 58, 43, 22,
       24, 45, 59, 59, 45, 24, 22, 43, 58, 59, 47, 27, 20, 40, 57, 59, 50, 30, 18, 38, 55, 61, 52, 32},
    },

    // Band_8_11651
    {
      // Scaler inc range
      {11523, 11651},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 89, 73, 31, 57, 88, 76, 35, 51, 85, 80, 40,
       45, 83,  83, 45, 40, 80, 85, 51, 35, 76, 88, 57, 31, 73, 89, 63},
      // Coefficients 6 taps
      {38, 58, 64, 58, 38, 0,  32, 52, 61, 55, 38, 18, 30, 50, 59, 57, 40, 20, 27, 47, 59, 58, 43, 22,
       24, 45, 59, 59, 45, 24, 22, 43, 58, 59, 47, 27, 20, 40, 57, 59, 50, 30, 18, 38, 55, 61, 52, 32},
    },

    // Band_8_11782
    {
      // Scaler inc range
      {11651, 11782},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 89, 73, 31, 57, 88, 76, 35, 51, 85, 80, 40,
       45, 83,  83, 45, 40, 80, 85, 51, 35, 76, 88, 57, 31, 73, 89, 63},
      // Coefficients 6 taps
      {38, 58, 64, 58, 38, 0,  33, 52, 60, 55, 38, 18, 30, 49, 61, 56, 40, 20, 27, 47, 59, 58, 43, 22,
       24, 45, 59, 59, 45, 24, 22, 43, 58, 59, 47, 27, 20, 40, 56, 61, 49, 30, 18, 38, 55, 60, 52, 33},
    },

    // Band_8_11916
    {
      // Scaler inc range
      {11782, 11916},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 89, 73, 31, 57, 88, 76, 35, 51, 85, 80, 40,
       45, 83,  83, 45, 40, 80, 85, 51, 35, 76, 88, 57, 31, 73, 89, 63},
      // Coefficients 6 taps
      {38, 57, 66, 57, 38, 0,  33, 52, 60, 55, 38, 18, 30, 49, 61, 56, 40, 20, 27, 47, 59, 58, 43, 22,
       24, 45, 59, 59, 45, 24, 22, 43, 58, 59, 47, 27, 20, 40, 56, 61, 49, 30, 18, 38, 55, 60, 52, 33},
    },

    // Band_8_12053
    {
      // Scaler inc range
      {11916, 12053},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 89, 73, 31, 57, 88, 76, 35, 51, 85, 80, 40,
       45, 83,  83, 45, 40, 80, 85, 51, 35, 76, 88, 57, 31, 73, 89, 63},
      // Coefficients 6 taps
      {38, 57, 66, 57, 38, 0,  33, 52, 60, 55, 38, 18, 30, 49, 61, 56, 40, 20, 27, 47, 59, 58, 43, 22,
       25, 45, 58, 58, 45, 25, 22, 43, 58, 59, 47, 27, 20, 40, 56, 61, 49, 30, 18, 38, 55, 60, 52, 33},
    },

    // Band_8_12193
    {
      // Scaler inc range
      {12053, 12193},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 89, 73, 31, 57, 88, 76, 35, 51, 85, 80, 40,
       45, 83,  83, 45, 40, 80, 85, 51, 35, 76, 88, 57, 31, 73, 89, 63},
      // Coefficients 6 taps
      {38, 57, 66, 57, 38, 0,  33, 52, 60, 55, 38, 18, 30, 49, 61, 56, 40, 20, 27, 47, 59, 58, 43, 22,
       25, 45, 58, 58, 45, 25, 22, 43, 58, 59, 47, 27, 20, 40, 56, 61, 49, 30, 18, 38, 55, 60, 52, 33},
    },

    // Band_8_12336
    {
      // Scaler inc range
      {12193, 12336},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 88, 73, 32, 57, 88, 76, 35, 51, 85, 80, 40,
       45, 83,  83, 45, 40, 80, 85, 51, 35, 76, 88, 57, 32, 73, 88, 63},
      // Coefficients 6 taps
      {38, 57, 66, 57, 38, 0,  33, 52, 60, 55, 38, 18, 30, 49, 61, 56, 40, 20, 27, 47, 60, 57, 43, 22,
       25, 45, 58, 58, 45, 25, 22, 43, 57, 60, 47, 27, 20, 40, 56, 61, 49, 30, 18, 38, 55, 60, 52, 33},
    },

    // Band_8_12483
    {
      // Scaler inc range
      {12336, 12483},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 88, 73, 32, 57, 88, 76, 35, 51, 85, 80, 40,
       45, 83,  83, 45, 40, 80, 85, 51, 35, 76, 88, 57, 32, 73, 88, 63},
      // Coefficients 6 taps
      {38, 57, 66, 57, 38, 0,  33, 52, 60, 55, 38, 18, 30, 49, 61, 56, 40, 20, 27, 47, 60, 57, 43, 22,
       25, 45, 58, 58, 45, 25, 22, 43, 57, 60, 47, 27, 20, 40, 56, 61, 49, 30, 18, 38, 55, 60, 52, 33},
    },

    // Band_8_12633
    {
      // Scaler inc range
      {12483, 12633},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 89, 72, 32, 57, 88, 76, 35, 51, 85, 80, 40,
       45, 83,  83, 45, 40, 80, 85, 51, 35, 76, 88, 57, 32, 72, 89, 63},
      // Coefficients 6 taps
      {38, 57, 66, 57, 38, 0,  33, 52, 60, 55, 38, 18, 30, 49, 61, 56, 40, 20, 27, 47, 60, 57, 43, 22,
       25, 45, 58, 58, 45, 25, 22, 43, 57, 60, 47, 27, 20, 40, 56, 61, 49, 30, 18, 38, 55, 60, 52, 33},
    },

    // Band_8_12788
    {
      // Scaler inc range
      {12633, 12788},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 89, 72, 32, 57, 88, 76, 35, 51, 85, 80, 40,
       45, 83,  83, 45, 40, 80, 85, 51, 35, 76, 88, 57, 32, 72, 89, 63},
      // Coefficients 6 taps
      {38, 57, 66, 57, 38, 0,  33, 52, 59, 55, 38, 19, 30, 49, 61, 56, 40, 20, 27, 47, 60, 57, 43, 22,
       25, 45, 58, 58, 45, 25, 22, 43, 57, 60, 47, 27, 20, 40, 56, 61, 49, 30, 19, 38, 55, 59, 52, 33},
    },

    // Band_8_12945
    {
      // Scaler inc range
      {12788, 12945},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 89, 72, 32, 57, 87, 76, 36, 51, 85, 80, 40,
       46, 82,  82, 46, 40, 80, 85, 51, 36, 76, 87, 57, 32, 72, 89, 63},
      // Coefficients 6 taps
      {38, 57, 66, 57, 38, 0,  33, 52, 59, 55, 38, 19, 30, 49, 61, 56, 40, 20, 27, 47, 60, 57, 43, 22,
       25, 45, 58, 58, 45, 25, 22, 43, 57, 60, 47, 27, 20, 40, 56, 61, 49, 30, 19, 38, 55, 59, 52, 33},
    },

    // Band_8_13107
    {
      // Scaler inc range
      {12945, 13107},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 89, 72, 32, 57, 87, 76, 36, 51, 86, 79, 40,
       46, 82,  82, 46, 40, 79, 86, 51, 36, 76, 87, 57, 32, 72, 89, 63},
      // Coefficients 6 taps
      {38, 57, 66, 57, 38, 0,  33, 51, 60, 55, 38, 19, 30, 49, 61, 56, 40, 20, 27, 47, 60, 57, 43, 22,
       25, 45, 58, 58, 45, 25, 22, 43, 57, 60, 47, 27, 20, 40, 56, 61, 49, 30, 19, 38, 55, 60, 51, 33},
    },

    // Band_8_13273
    {
      // Scaler inc range
      {13107, 13273},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 89, 72, 32, 57, 87, 76, 36, 51, 86, 79, 40,
       46, 82,  82, 46, 40, 79, 86, 51, 36, 76, 87, 57, 32, 72, 89, 63},
      // Coefficients 6 taps
      {38, 57, 66, 57, 38, 0,  33, 51, 60, 55, 38, 19, 30, 49, 61, 56, 40, 20, 27, 47, 60, 57, 43, 22,
       25, 45, 58, 58, 45, 25, 22, 43, 57, 60, 47, 27, 20, 40, 56, 61, 49, 30, 19, 38, 55, 60, 51, 33},
    },

    // Band_8_13443
    {
      // Scaler inc range
      {13273, 13443},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 89, 72, 32, 57, 87, 76, 36, 51, 86, 79, 40,
       46, 82,  82, 46, 40, 79, 86, 51, 36, 76, 87, 57, 32, 72, 89, 63},
      // Coefficients 6 taps
      {38, 57, 66, 57, 38, 0,  33, 51, 60, 55, 38, 19, 30, 49, 61, 56, 40, 20, 27, 47, 60, 57, 43, 22,
       25, 45, 58, 58, 45, 25, 22, 43, 57, 60, 47, 27, 20, 40, 56, 61, 49, 30, 19, 38, 55, 60, 51, 33},
    },

    // Band_8_13618
    {
      // Scaler inc range
      {13443, 13618},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 89, 72, 32, 57, 87, 76, 36, 51, 86, 79, 40,
       46, 82,  82, 46, 40, 79, 86, 51, 36, 76, 87, 57, 32, 72, 89, 63},
      // Coefficients 6 taps
      {38, 57, 66, 57, 38, 0,  33, 51, 60, 55, 38, 19, 30, 49, 61, 56, 40, 20, 27, 47, 60, 57, 43, 22,
       25, 45, 58, 58, 45, 25, 22, 43, 57, 60, 47, 27, 20, 40, 56, 61, 49, 30, 19, 38, 55, 60, 51, 33},
    },

    // Band_8_13797
    {
      // Scaler inc range
      {13618, 13797},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 89, 72, 32, 57, 87, 76, 36, 51, 86, 79, 40,
       46, 82,  82, 46, 40, 79, 86, 51, 36, 76, 87, 57, 32, 72, 89, 63},
      // Coefficients 6 taps
      {38, 57, 66, 57, 38, 0,  33, 51, 60, 55, 38, 19, 30, 49, 61, 56, 40, 20, 27, 47, 59, 57, 43, 23,
       25, 45, 58, 58, 45, 25, 23, 43, 57, 59, 47, 27, 20, 40, 56, 61, 49, 30, 19, 38, 55, 60, 51, 33},
    },

    // Band_8_13981
    {
      // Scaler inc range
      {13797, 13981},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 88, 72, 33, 57, 87, 76, 36, 51, 85, 79, 41,
       46, 82,  82, 46, 41, 79, 85, 51, 36, 76, 87, 57, 33, 72, 88, 63},
      // Coefficients 6 taps
      {38, 57, 66, 57, 38, 0,  33, 51, 60, 55, 38, 19, 30, 49, 60, 56, 40, 21, 27, 47, 59, 57, 43, 23,
       25, 45, 58, 58, 45, 25, 23, 43, 57, 59, 47, 27, 21, 40, 56, 60, 49, 30, 19, 38, 55, 60, 51, 33},
    },

    // Band_8_14170
    {
      // Scaler inc range
      {13981, 14170},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 88, 72, 33, 57, 87, 76, 36, 51, 85, 79, 41,
       46, 82,  82, 46, 41, 79, 85, 51, 36, 76, 87, 57, 33, 72, 88, 63},
      // Coefficients 6 taps
      {38, 57, 66, 57, 38, 0,  33, 51, 60, 55, 38, 19, 30, 49, 60, 56, 40, 21, 27, 47, 59, 57, 43, 23,
       25, 45, 58, 58, 45, 25, 23, 43, 57, 59, 47, 27, 21, 40, 56, 60, 49, 30, 19, 38, 55, 60, 51, 33},
    },

    // Band_8_14364
    {
      // Scaler inc range
      {14170, 14364},
      // Coefficients 4 taps
      {77, 102, 77, 0,  63, 88, 72, 33, 57, 87, 76, 36, 51, 85, 79, 41,
       46, 82,  82, 46, 41, 79, 85, 51, 36, 76, 87, 57, 33, 72, 88, 63},
      // Coefficients 6 taps
      {38, 57, 66, 57, 38, 0,  33, 51, 60, 55, 38, 19, 30, 49, 60, 56, 40, 21, 28, 47, 58, 57, 43, 23,
       25, 45, 58, 58, 45, 25, 23, 43, 57, 58, 47, 28, 21, 40, 56, 60, 49, 30, 19, 38, 55, 60, 51, 33},
    },

    // Band_8_14564
    {
      // Scaler inc range
      {14364, 14564},
      // Coefficients 4 taps
      {77, 102, 77, 0,  62, 89, 72, 33, 57, 87, 76, 36, 51, 85, 79, 41,
       46, 82,  82, 46, 41, 79, 85, 51, 36, 76, 87, 57, 33, 72, 89, 62},
      // Coefficients 6 taps
      {38, 57, 66, 57, 38, 0,  33, 51, 60, 55, 38, 19, 30, 49, 60, 56, 40, 21, 28, 47, 58, 57, 43, 23,
       25, 45, 58, 58, 45, 25, 23, 43, 57, 58, 47, 28, 21, 40, 56, 60, 49, 30, 19, 38, 55, 60, 51, 33},
    },

    // Band_8_14769
    {
      // Scaler inc range
      {14564, 14769},
      // Coefficients 4 taps
      {77, 102, 77, 0,  62, 89, 72, 33, 57, 87, 76, 36, 51, 85, 79, 41,
       46, 82,  82, 46, 41, 79, 85, 51, 36, 76, 87, 57, 33, 72, 89, 62},
      // Coefficients 6 taps
      {38, 57, 66, 57, 38, 0,  33, 51, 60, 55, 38, 19, 30, 49, 60, 56, 40, 21, 28, 47, 58, 57, 43, 23,
       25, 45, 58, 58, 45, 25, 23, 43, 57, 58, 47, 28, 21, 40, 56, 60, 49, 30, 19, 38, 55, 60, 51, 33},
    },

    // Band_8_14980
    {
      // Scaler inc range
      {14769, 14980},
      // Coefficients 4 taps
      {77, 102, 77, 0,  62, 89, 72, 33, 57, 87, 76, 36, 51, 85, 79, 41,
       46, 82,  82, 46, 41, 79, 85, 51, 36, 76, 87, 57, 33, 72, 89, 62},
      // Coefficients 6 taps
      {38, 57, 66, 57, 38, 0,  33, 51, 60, 55, 38, 19, 30, 49, 60, 56, 40, 21, 28, 47, 58, 57, 43, 23,
       25, 45, 58, 58, 45, 25, 23, 43, 57, 58, 47, 28, 21, 40, 56, 60, 49, 30, 19, 38, 55, 60, 51, 33},
    },

    // Band_8_15197
    {
      // Scaler inc range
      {14980, 15197},
      // Coefficients 4 taps
      {77, 102, 77, 0,  62, 89, 72, 33, 57, 86, 76, 37, 51, 85, 79, 41,
       46, 82,  82, 46, 41, 79, 85, 51, 37, 76, 86, 57, 33, 72, 89, 62},
      // Coefficients 6 taps
      {38, 57, 66, 57, 38, 0,  33, 51, 60, 55, 38, 19, 30, 49, 60, 56, 40, 21, 28, 47, 58, 57, 43, 23,
       25, 45, 58, 58, 45, 25, 23, 43, 57, 58, 47, 28, 21, 40, 56, 60, 49, 30, 19, 38, 55, 60, 51, 33},
    },

    // Band_8_15420
    {
      // Scaler inc range
      {15197, 15420},
      // Coefficients 4 taps
      {77, 102, 77, 0,  62, 88, 72, 34, 57, 86, 76, 37, 51, 85, 79, 41,
       46, 82,  82, 46, 41, 79, 85, 51, 37, 76, 86, 57, 34, 72, 88, 62},
      // Coefficients 6 taps
      {38, 57, 66, 57, 38, 0,  33, 51, 59, 55, 38, 20, 30, 49, 60, 56, 40, 21, 28, 47, 58, 57, 43, 23,
       25, 45, 58, 58, 45, 25, 23, 43, 57, 58, 47, 28, 21, 40, 56, 60, 49, 30, 20, 38, 55, 59, 51, 33},
    },

    // Band_8_15650
    {
      // Scaler inc range
      {15420, 15650},
      // Coefficients 4 taps
      {77, 102, 77, 0,  62, 88, 72, 34, 57, 86, 76, 37, 51, 85, 79, 41,
       46, 82,  82, 46, 41, 79, 85, 51, 37, 76, 86, 57, 34, 72, 88, 62},
      // Coefficients 6 taps
      {38, 57, 66, 57, 38, 0,  33, 51, 59, 55, 38, 20, 30, 49, 60, 56, 40, 21, 28, 47, 58, 57, 43, 23,
       25, 45, 58, 58, 45, 25, 23, 43, 57, 58, 47, 28, 21, 40, 56, 60, 49, 30, 20, 38, 55, 59, 51, 33},
    },

    // Band_8_15888
    {
      // Scaler inc range
      {15650, 15888},
      // Coefficients 4 taps
      {77, 102, 77, 0,  62, 88, 72, 34, 57, 86, 76, 37, 51, 85, 79, 41,
       46, 82,  82, 46, 41, 79, 85, 51, 37, 76, 86, 57, 34, 72, 88, 62},
      // Coefficients 6 taps
      {38, 57, 66, 57, 38, 0,  33, 51, 59, 55, 38, 20, 30, 49, 60, 56, 40, 21, 28, 47, 58, 57, 43, 23,
       25, 45, 58, 58, 45, 25, 23, 43, 57, 58, 47, 28, 21, 40, 56, 60, 49, 30, 20, 38, 55, 59, 51, 33},
    },

    // Band_8_16132
    {
      // Scaler inc range
      {15888, 16132},
      // Coefficients 4 taps
      {77, 102, 77, 0,  62, 88, 72, 34, 57, 86, 76, 37, 51, 85, 79, 41,
       46, 82,  82, 46, 41, 79, 85, 51, 37, 76, 86, 57, 34, 72, 88, 62},
      // Coefficients 6 taps
      {38, 57, 66, 57, 38, 0,  33, 51, 60, 54, 38, 20, 30, 49, 60, 56, 40, 21, 28, 47, 58, 57, 43, 23,
       25, 45, 58, 58, 45, 25, 23, 43, 57, 58, 47, 28, 21, 40, 56, 60, 49, 30, 20, 38, 54, 60, 51, 33},
    },

    // Band_8_16384
    {
      // Scaler inc range
      {16132, 16384},
      // Coefficients 4 taps
      {77, 102, 77, 0,  62, 88, 72, 34, 57, 86, 76, 37, 51, 85, 79, 41,
       46, 82,  82, 46, 41, 79, 85, 51, 37, 76, 86, 57, 34, 72, 88, 62},
      // Coefficients 6 taps
      {38, 57, 66, 57, 38, 0,  33, 51, 60, 54, 38, 20, 30, 49, 60, 56, 40, 21, 28, 47, 58, 57, 43, 23,
       25, 45, 58, 58, 45, 25, 23, 43, 57, 58, 47, 28, 21, 40, 56, 60, 49, 30, 20, 38, 54, 60, 51, 33},
    },

    // Band_8_16644
    {
      // Scaler inc range
      {16384, 16644},
      // Coefficients 4 taps
      {77, 102, 77, 0,  62, 88, 72, 34, 57, 86, 76, 37, 51, 85, 79, 41,
       46, 82,  82, 46, 41, 79, 85, 51, 37, 76, 86, 57, 34, 72, 88, 62},
      // Coefficients 6 taps
      {38, 57, 66, 57, 38, 0,  33, 51, 60, 54, 38, 20, 30, 49, 60, 56, 40, 21, 28, 47, 58, 57, 43, 23,
       25, 45, 58, 58, 45, 25, 23, 43, 57, 58, 47, 28, 21, 40, 56, 60, 49, 30, 20, 38, 54, 60, 51, 33},
    },

    // Band_8_16913
    {
      // Scaler inc range
      {16644, 16913},
      // Coefficients 4 taps
      {77, 102, 77, 0,  62, 88, 72, 34, 57, 87, 75, 37, 52, 83, 79, 42,
       46, 82,  82, 46, 42, 79, 83, 52, 37, 75, 87, 57, 34, 72, 88, 62},
      // Coefficients 6 taps
      {38, 57, 66, 57, 38, 0,  33, 51, 60, 54, 38, 20, 30, 49, 60, 56, 40, 21, 28, 47, 58, 57, 43, 23,
       25, 45, 58, 58, 45, 25, 23, 43, 57, 58, 47, 28, 21, 40, 56, 60, 49, 30, 20, 38, 54, 60, 51, 33},
    },

    // Band_8_17190
    {
      // Scaler inc range
      {16913, 17190},
      // Coefficients 4 taps
      {77, 102, 77, 0,  62, 87, 72, 35, 57, 86, 75, 38, 52, 83, 79, 42,
       46, 82,  82, 46, 42, 79, 83, 52, 38, 75, 86, 57, 35, 72, 87, 62},
      // Coefficients 6 taps
      {38, 57, 66, 57, 38, 0,  33, 51, 60, 54, 38, 20, 30, 49, 60, 56, 40, 21, 28, 47, 58, 57, 43, 23,
       25, 45, 58, 58, 45, 25, 23, 43, 57, 58, 47, 28, 21, 40, 56, 60, 49, 30, 20, 38, 54, 60, 51, 33},
    },

    // Band_8_17476
    {
      // Scaler inc range
      {17190, 17476},
      // Coefficients 4 taps
      {77, 102, 77, 0,  62, 87, 72, 35, 57, 86, 75, 38, 52, 83, 79, 42,
       46, 82,  82, 46, 42, 79, 83, 52, 38, 75, 86, 57, 35, 72, 87, 62},
      // Coefficients 6 taps
      {38, 57, 66, 57, 38, 0,  33, 51, 60, 54, 38, 20, 30, 49, 59, 56, 40, 22, 28, 47, 58, 57, 43, 23,
       26, 45, 57, 57, 45, 26, 23, 43, 57, 58, 47, 28, 22, 40, 56, 59, 49, 30, 20, 38, 54, 60, 51, 33},
    },

    // Band_8_17772
    {
      // Scaler inc range
      {17476, 17772},
      // Coefficients 4 taps
      {77, 102, 77, 0,  62, 87, 72, 35, 57, 86, 75, 38, 52, 83, 79, 42,
       46, 82,  82, 46, 42, 79, 83, 52, 38, 75, 86, 57, 35, 72, 87, 62},
      // Coefficients 6 taps
      {39, 57, 64, 57, 39, 0,  33, 51, 60, 54, 38, 20, 30, 49, 59, 56, 40, 22, 28, 47, 58, 57, 43, 23,
       26, 45, 57, 57, 45, 26, 23, 43, 57, 58, 47, 28, 22, 40, 56, 59, 49, 30, 20, 38, 54, 60, 51, 33},
    },

    // Band_8_18079
    {
      // Scaler inc range
      {17772, 18079},
      // Coefficients 4 taps
      {77, 102, 77, 0,  62, 88, 71, 35, 57, 86, 75, 38, 52, 83, 79, 42,
       47, 81,  81, 47, 42, 79, 83, 52, 38, 75, 86, 57, 35, 71, 88, 62},
      // Coefficients 6 taps
      {39, 57, 64, 57, 39, 0,  33, 51, 60, 54, 38, 20, 30, 49, 59, 56, 40, 22, 28, 47, 58, 57, 43, 23,
       26, 45, 57, 57, 45, 26, 23, 43, 57, 58, 47, 28, 22, 40, 56, 59, 49, 30, 20, 38, 54, 60, 51, 33},
    },

    // Band_8_18396
    {
      // Scaler inc range
      {18079, 18396},
      // Coefficients 4 taps
      {77, 102, 77, 0,  62, 88, 71, 35, 57, 86, 75, 38, 52, 84, 78, 42,
       47, 81,  81, 47, 42, 78, 84, 52, 38, 75, 86, 57, 35, 71, 88, 62},
      // Coefficients 6 taps
      {39, 57, 64, 57, 39, 0,  33, 51, 59, 54, 38, 21, 30, 49, 60, 55, 40, 22, 28, 47, 57, 57, 43, 24,
       26, 45, 57, 57, 45, 26, 24, 43, 57, 57, 47, 28, 22, 40, 55, 60, 49, 30, 21, 38, 54, 59, 51, 33},
    },

    // Band_8_18725
    {
      // Scaler inc range
      {18396, 18725},
      // Coefficients 4 taps
      {77, 102, 77, 0,  62, 87, 71, 36, 57, 86, 75, 38, 52, 84, 78, 42,
       47, 81,  81, 47, 42, 78, 84, 52, 38, 75, 86, 57, 36, 71, 87, 62},
      // Coefficients 6 taps
      {39, 57, 64, 57, 39, 0,  33, 51, 59, 54, 38, 21, 31, 49, 59, 55, 40, 22, 28, 47, 57, 57, 43, 24,
       26, 45, 57, 57, 45, 26, 24, 43, 57, 57, 47, 28, 22, 40, 55, 59, 49, 31, 21, 38, 54, 59, 51, 33},
    },

    // Band_8_19065
    {
      // Scaler inc range
      {18725, 19065},
      // Coefficients 4 taps
      {77, 102, 77, 0,  62, 87, 71, 36, 57, 85, 75, 39, 52, 84, 78, 42,
       47, 81,  81, 47, 42, 78, 84, 52, 39, 75, 85, 57, 36, 71, 87, 62},
      // Coefficients 6 taps
      {39, 57, 64, 57, 39, 0,  33, 51, 59, 54, 38, 21, 31, 49, 59, 55, 40, 22, 28, 47, 58, 56, 43, 24,
       26, 45, 57, 57, 45, 26, 24, 43, 56, 58, 47, 28, 22, 40, 55, 59, 49, 31, 21, 38, 54, 59, 51, 33},
    },

    // Band_8_19418
    {
      // Scaler inc range
      {19065, 19418},
      // Coefficients 4 taps
      {77, 102, 77, 0,  62, 87, 71, 36, 57, 85, 75, 39, 52, 84, 78, 42,
       47, 81,  81, 47, 42, 78, 84, 52, 39, 75, 85, 57, 36, 71, 87, 62},
      // Coefficients 6 taps
      {39, 57, 64, 57, 39, 0,  33, 51, 59, 54, 38, 21, 31, 49, 59, 55, 40, 22, 28, 47, 58, 56, 43, 24,
       26, 45, 57, 57, 45, 26, 24, 43, 56, 58, 47, 28, 22, 40, 55, 59, 49, 31, 21, 38, 54, 59, 51, 33},
    },

    // Band_8_19784
    {
      // Scaler inc range
      {19418, 19784},
      // Coefficients 4 taps
      {78, 100, 78, 0,  62, 87, 71, 36, 57, 85, 75, 39, 52, 83, 78, 43,
       47, 81,  81, 47, 43, 78, 83, 52, 39, 75, 85, 57, 36, 71, 87, 62},
      // Coefficients 6 taps
      {39, 57, 64, 57, 39, 0,  33, 51, 59, 54, 38, 21, 31, 49, 59, 55, 40, 22, 28, 47, 58, 56, 43, 24,
       26, 45, 57, 57, 45, 26, 24, 43, 56, 58, 47, 28, 22, 40, 55, 59, 49, 31, 21, 38, 54, 59, 51, 33},
    },

    // Band_8_20165
    {
      // Scaler inc range
      {19784, 20165},
      // Coefficients 4 taps
      {78, 100, 78, 0,  62, 86, 71, 37, 57, 85, 75, 39, 52, 83, 78, 43,
       47, 81,  81, 47, 43, 78, 83, 52, 39, 75, 85, 57, 37, 71, 86, 62},
      // Coefficients 6 taps
      {39, 57, 64, 57, 39, 0,  33, 51, 59, 54, 38, 21, 31, 49, 59, 55, 40, 22, 28, 47, 58, 56, 43, 24,
       26, 45, 57, 57, 45, 26, 24, 43, 56, 58, 47, 28, 22, 40, 55, 59, 49, 31, 21, 38, 54, 59, 51, 33},
    },

    // Band_8_20560
    {
      // Scaler inc range
      {20165, 20560},
      // Coefficients 4 taps
      {78, 100, 78, 0,  62, 86, 71, 37, 57, 85, 75, 39, 52, 83, 78, 43,
       47, 81,  81, 47, 43, 78, 83, 52, 39, 75, 85, 57, 37, 71, 86, 62},
      // Coefficients 6 taps
      {39, 57, 64, 57, 39, 0,  33, 51, 59, 54, 38, 21, 31, 49, 59, 55, 40, 22, 28, 47, 58, 56, 43, 24,
       26, 45, 57, 57, 45, 26, 24, 43, 56, 58, 47, 28, 22, 40, 55, 59, 49, 31, 21, 38, 54, 59, 51, 33},
    },

    // Band_8_20972
    {
      // Scaler inc range
      {20560, 20972},
      // Coefficients 4 taps
      {78, 100, 78, 0,  62, 86, 71, 37, 57, 84, 75, 40, 52, 83, 78, 43,
       47, 81,  81, 47, 43, 78, 83, 52, 40, 75, 84, 57, 37, 71, 86, 62},
      // Coefficients 6 taps
      {39, 57, 64, 57, 39, 0,  33, 51, 59, 54, 38, 21, 31, 49, 58, 55, 40, 23, 28, 47, 58, 56, 43, 24,
       26, 45, 57, 57, 45, 26, 24, 43, 56, 58, 47, 28, 23, 40, 55, 58, 49, 31, 21, 38, 54, 59, 51, 33},
    },

    // Band_8_21400
    {
      // Scaler inc range
      {20972, 21400},
      // Coefficients 4 taps
      {78, 100, 78, 0,  62, 86, 71, 37, 57, 84, 75, 40, 52, 83, 78, 43,
       47, 81,  81, 47, 43, 78, 83, 52, 40, 75, 84, 57, 37, 71, 86, 62},
      // Coefficients 6 taps
      {39, 57, 64, 57, 39, 0,  33, 51, 58, 54, 38, 22, 31, 49, 58, 55, 40, 23, 28, 47, 58, 56, 43, 24,
       26, 45, 57, 57, 45, 26, 24, 43, 56, 58, 47, 28, 23, 40, 55, 58, 49, 31, 22, 38, 54, 58, 51, 33},
    },

    // Band_8_21845
    {
      // Scaler inc range
      {21400, 21845},
      // Coefficients 4 taps
      {78, 100, 78, 0,  62, 85, 71, 38, 57, 85, 74, 40, 52, 83, 78, 43,
       47, 81,  81, 47, 43, 78, 83, 52, 40, 74, 85, 57, 38, 71, 85, 62},
      // Coefficients 6 taps
      {39, 57, 64, 57, 39, 0,  33, 51, 58, 54, 38, 22, 31, 49, 58, 55, 40, 23, 28, 47, 58, 56, 43, 24,
       26, 45, 57, 57, 45, 26, 24, 43, 56, 58, 47, 28, 23, 40, 55, 58, 49, 31, 22, 38, 54, 58, 51, 33},
    },

    // Band_8_22310
    {
      // Scaler inc range
      {21845, 22310},
      // Coefficients 4 taps
      {78, 100, 78, 0,  62, 85, 71, 38, 57, 85, 74, 40, 52, 82, 78, 44,
       48, 80,  80, 48, 44, 78, 82, 52, 40, 74, 85, 57, 38, 71, 85, 62},
      // Coefficients 6 taps
      {39, 57, 64, 57, 39, 0,  33, 51, 58, 54, 38, 22, 31, 49, 58, 55, 40, 23, 28, 47, 58, 56, 43, 24,
       26, 45, 57, 57, 45, 26, 24, 43, 56, 58, 47, 28, 23, 40, 55, 58, 49, 31, 22, 38, 54, 58, 51, 33},
    },

    // Band_8_22795
    {
      // Scaler inc range
      {22310, 22795},
      // Coefficients 4 taps
      {78, 100, 78, 0,  62, 86, 70, 38, 57, 84, 74, 41, 52, 83, 77, 44,
       48, 80,  80, 48, 44, 77, 83, 52, 41, 74, 84, 57, 38, 70, 86, 62},
      // Coefficients 6 taps
      {39, 57, 64, 57, 39, 0,  33, 51, 58, 54, 38, 22, 31, 49, 58, 55, 40, 23, 29, 47, 57, 56, 42, 25,
       26, 45, 57, 57, 45, 26, 25, 42, 56, 57, 47, 29, 23, 40, 55, 58, 49, 31, 22, 38, 54, 58, 51, 33},
    },

    // Band_8_23302
    {
      // Scaler inc range
      {22795, 23302},
      // Coefficients 4 taps
      {78, 100, 78, 0,  62, 85, 70, 39, 57, 84, 74, 41, 52, 83, 77, 44,
       48, 80,  80, 48, 44, 77, 83, 52, 41, 74, 84, 57, 39, 70, 85, 62},
      // Coefficients 6 taps
      {39, 57, 64, 57, 39, 0,  33, 50, 59, 54, 38, 22, 31, 49, 58, 55, 40, 23, 29, 47, 57, 56, 42, 25,
       26, 45, 57, 57, 45, 26, 25, 42, 56, 57, 47, 29, 23, 40, 55, 58, 49, 31, 22, 38, 54, 59, 50, 33},
    },

    // Band_8_23831
    {
      // Scaler inc range
      {23302, 23831},
      // Coefficients 4 taps
      {78, 100, 78, 0,  62, 85, 70, 39, 57, 84, 74, 41, 52, 83, 77, 44,
       48, 80,  80, 48, 44, 77, 83, 52, 41, 74, 84, 57, 39, 70, 85, 62},
      // Coefficients 6 taps
      {39, 57, 64, 57, 39, 0,  33, 50, 59, 54, 38, 22, 31, 49, 58, 55, 40, 23, 29, 47, 57, 56, 42, 25,
       27, 45, 56, 56, 45, 27, 25, 42, 56, 57, 47, 29, 23, 40, 55, 58, 49, 31, 22, 38, 54, 59, 50, 33},
    },

    // Band_8_24385
    {
      // Scaler inc range
      {23831, 24385},
      // Coefficients 4 taps
      {78, 100, 78, 0,  62, 85, 70, 39, 57, 84, 74, 41, 52, 82, 77, 45,
       48, 80,  80, 48, 45, 77, 82, 52, 41, 74, 84, 57, 39, 70, 85, 62},
      // Coefficients 6 taps
      {39, 57, 64, 57, 39, 0,  33, 50, 58, 54, 38, 23, 31, 49, 58, 55, 40, 23, 29, 47, 57, 56, 42, 25,
       27, 45, 56, 56, 45, 27, 25, 42, 56, 57, 47, 29, 23, 40, 55, 58, 49, 31, 23, 38, 54, 58, 50, 33},
    },

    // Band_8_24966
    {
      // Scaler inc range
      {24385, 24966},
      // Coefficients 4 taps
      {78, 100, 78, 0,  62, 84, 70, 40, 57, 83, 74, 42, 53, 81, 77, 45,
       48, 80,  80, 48, 45, 77, 81, 53, 42, 74, 83, 57, 40, 70, 84, 62},
      // Coefficients 6 taps
      {39, 57, 64, 57, 39, 0,  33, 50, 59, 53, 38, 23, 31, 49, 57, 55, 40, 24, 29, 47, 57, 56, 42, 25,
       27, 45, 56, 56, 45, 27, 25, 42, 56, 57, 47, 29, 24, 40, 55, 57, 49, 31, 23, 38, 53, 59, 50, 33},
    },

    // Band_8_25575
    {
      // Scaler inc range
      {24966, 25575},
      // Coefficients 4 taps
      {78, 100, 78, 0,  62, 84, 70, 40, 57, 84, 73, 42, 53, 81, 77, 45,
       49, 79,  79, 49, 45, 77, 81, 53, 42, 73, 84, 57, 40, 70, 84, 62},
      // Coefficients 6 taps
      {39, 57, 64, 57, 39, 0,  33, 50, 59, 53, 38, 23, 31, 48, 58, 55, 40, 24, 29, 47, 57, 56, 42, 25,
       27, 45, 56, 56, 45, 27, 25, 42, 56, 57, 47, 29, 24, 40, 55, 58, 48, 31, 23, 38, 53, 59, 50, 33},
    },

    // Band_8_26214
    {
      // Scaler inc range
      {25575, 26214},
      // Coefficients 4 taps
      {78, 100, 78, 0,  62, 83, 70, 41, 57, 83, 73, 43, 53, 82, 76, 45,
       49, 79,  79, 49, 45, 76, 82, 53, 43, 73, 83, 57, 41, 70, 83, 62},
      // Coefficients 6 taps
      {39, 57, 64, 57, 39, 0,  33, 50, 59, 53, 38, 23, 31, 48, 58, 55, 40, 24, 29, 46, 58, 56, 42, 25,
       27, 44, 57, 57, 44, 27, 25, 42, 56, 58, 46, 29, 24, 40, 55, 58, 48, 31, 23, 38, 53, 59, 50, 33},
    },

    // Band_8_26887
    {
      // Scaler inc range
      {26214, 26887},
      // Coefficients 4 taps
      {78, 100, 78, 0,  62, 83, 70, 41, 57, 83, 73, 43, 53, 81, 76, 46,
       49, 79,  79, 49, 46, 76, 81, 53, 43, 73, 83, 57, 41, 70, 83, 62},
      // Coefficients 6 taps
      {39, 57, 64, 57, 39, 0,  33, 50, 59, 53, 38, 23, 31, 48, 59, 54, 40, 24, 29, 46, 58, 56, 42, 25,
       27, 44, 57, 57, 44, 27, 25, 42, 56, 58, 46, 29, 24, 40, 54, 59, 48, 31, 23, 38, 53, 59, 50, 33},
    },

    // Band_8_27594
    {
      // Scaler inc range
      {26887, 27594},
      // Coefficients 4 taps
      {78, 100, 78, 0,  61, 85, 69, 41, 57, 83, 73, 43, 53, 81, 76, 46,
       49, 79,  79, 49, 46, 76, 81, 53, 43, 73, 83, 57, 41, 69, 85, 61},
      // Coefficients 6 taps
      {39, 57, 64, 57, 39, 0,  33, 50, 58, 53, 38, 24, 31, 48, 59, 54, 40, 24, 29, 46, 58, 55, 42, 26,
       27, 44, 57, 57, 44, 27, 26, 42, 55, 58, 46, 29, 24, 40, 54, 59, 48, 31, 24, 38, 53, 58, 50, 33},
    },

    // Band_8_28340
    {
      // Scaler inc range
      {27594, 28340},
      // Coefficients 4 taps
      {78, 100, 78, 0,  61, 84, 69, 42, 57, 82, 73, 44, 53, 81, 76, 46,
       50, 78,  78, 50, 46, 76, 81, 53, 44, 73, 82, 57, 42, 69, 84, 61},
      // Coefficients 6 taps
      {39, 57, 64, 57, 39, 0,  33, 50, 58, 53, 38, 24, 31, 48, 58, 54, 40, 25, 29, 46, 58, 55, 42, 26,
       27, 44, 57, 57, 44, 27, 26, 42, 55, 58, 46, 29, 25, 40, 54, 58, 48, 31, 24, 38, 53, 58, 50, 33},
    },

    // Band_8_29127
    {
      // Scaler inc range
      {28340, 29127},
      // Coefficients 4 taps
      {78, 100, 78, 0,  61, 84, 69, 42, 57, 82, 73, 44, 53, 80, 76, 47,
       50, 78,  78, 50, 47, 76, 80, 53, 44, 73, 82, 57, 42, 69, 84, 61},
      // Coefficients 6 taps
      {39, 57, 64, 57, 39, 0,  34, 50, 57, 53, 38, 24, 31, 48, 58, 54, 40, 25, 29, 46, 58, 55, 42, 26,
       28, 44, 56, 56, 44, 28, 26, 42, 55, 58, 46, 29, 25, 40, 54, 58, 48, 31, 24, 38, 53, 57, 50, 34},
    },

    // Band_8_29959
    {
      // Scaler inc range
      {29127, 29959},
      // Coefficients 4 taps
      {78, 100, 78, 0,  61, 83, 69, 43, 57, 82, 72, 45, 54, 80, 75, 47,
       50, 78,  78, 50, 47, 75, 80, 54, 45, 72, 82, 57, 43, 69, 83, 61},
      // Coefficients 6 taps
      {39, 57, 64, 57, 39, 0,  34, 50, 57, 53, 38, 24, 31, 48, 58, 54, 40, 25, 30, 46, 57, 55, 42, 26,
       28, 44, 56, 56, 44, 28, 26, 42, 55, 57, 46, 30, 25, 40, 54, 58, 48, 31, 24, 38, 53, 57, 50, 34},
    },

    // Band_8_30840
    {
      // Scaler inc range
      {29959, 30840},
      // Coefficients 4 taps
      {78, 100, 78, 0,  61, 82, 69, 44, 58, 81, 72, 45, 54, 80, 75, 47,
       50, 78,  78, 50, 47, 75, 80, 54, 45, 72, 81, 58, 44, 69, 82, 61},
      // Coefficients 6 taps
      {40, 57, 62, 57, 40, 0,  34, 50, 56, 53, 38, 25, 32, 48, 57, 54, 40, 25, 30, 46, 57, 55, 42, 26,
       28, 44, 56, 56, 44, 28, 26, 42, 55, 57, 46, 30, 25, 40, 54, 57, 48, 32, 25, 38, 53, 56, 50, 34},
    },

    // Band_8_31775
    {
      // Scaler inc range
      {30840, 31775},
      // Coefficients 4 taps
      {79, 98, 79, 0,  61, 82, 69, 44, 58, 80, 72, 46, 54, 79, 75, 48,
       51, 77, 77, 51, 48, 75, 79, 54, 46, 72, 80, 58, 44, 69, 82, 61},
      // Coefficients 6 taps
      {40, 57, 62, 57, 40, 0,  34, 50, 56, 53, 38, 25, 32, 48, 56, 54, 40, 26, 30, 46, 56, 55, 42, 27,
       28, 44, 56, 56, 44, 28, 27, 42, 55, 56, 46, 30, 26, 40, 54, 56, 48, 32, 25, 38, 53, 56, 50, 34},
    },

    // Band_8_32768
    {
      // Scaler inc range
      {31775, 32768},
      // Coefficients 4 taps
      {79, 98, 79, 0,  61, 81, 69, 45, 58, 80, 72, 46, 54, 80, 74, 48,
       51, 77, 77, 51, 48, 74, 80, 54, 46, 72, 80, 58, 45, 69, 81, 61},
      // Coefficients 6 taps
      {40, 57, 62, 57, 40, 0,  34, 50, 57, 52, 38, 25, 32, 48, 56, 54, 40, 26, 30, 46, 56, 55, 42, 27,
       28, 44, 56, 56, 44, 28, 27, 42, 55, 56, 46, 30, 26, 40, 54, 56, 48, 32, 25, 38, 52, 57, 50, 34},
    },

    // Band_8_33825
    {
      // Scaler inc range
      {32768, 33825},
      // Coefficients 4 taps
      {79, 98, 79, 0,  61, 82, 68, 45, 58, 80, 71, 47, 55, 78, 74, 49,
       51, 77, 77, 51, 49, 74, 78, 55, 47, 71, 80, 58, 45, 68, 82, 61},
      // Coefficients 6 taps
      {40, 57, 62, 57, 40, 0,  34, 49, 58, 52, 38, 25, 32, 48, 56, 54, 40, 26, 30, 46, 56, 55, 42, 27,
       29, 44, 55, 55, 44, 29, 27, 42, 55, 56, 46, 30, 26, 40, 54, 56, 48, 32, 25, 38, 52, 58, 49, 34},
    },

    // Band_8_34953
    {
      // Scaler inc range
      {33825, 34953},
      // Coefficients 4 taps
      {79, 98, 79, 0,  61, 81, 68, 46, 58, 80, 71, 47, 55, 78, 74, 49,
       52, 76, 76, 52, 49, 74, 78, 55, 47, 71, 80, 58, 46, 68, 81, 61},
      // Coefficients 6 taps
      {40, 56, 64, 56, 40, 0,  34, 49, 57, 52, 38, 26, 32, 48, 57, 53, 40, 26, 30, 46, 57, 54, 42, 27,
       29, 44, 55, 55, 44, 29, 27, 42, 54, 57, 46, 30, 26, 40, 53, 57, 48, 32, 26, 38, 52, 57, 49, 34},
    },

    // Band_8_36158
    {
      // Scaler inc range
      {34953, 36158},
      // Coefficients 4 taps
      {79, 98, 79, 0,  62, 79, 68, 47, 58, 79, 71, 48, 55, 78, 73, 50,
       52, 76, 76, 52, 50, 73, 78, 55, 48, 71, 79, 58, 47, 68, 79, 62},
      // Coefficients 6 taps
      {40, 56, 64, 56, 40, 0,  34, 49, 57, 52, 38, 26, 32, 47, 57, 53, 40, 27, 31, 46, 55, 54, 42, 28,
       29, 44, 55, 55, 44, 29, 28, 42, 54, 55, 46, 31, 27, 40, 53, 57, 47, 32, 26, 38, 52, 57, 49, 34},
    },

    // Band_8_37449
    {
      // Scaler inc range
      {36158, 37449},
      // Coefficients 4 taps
      {79, 98, 79, 0,  62, 78, 68, 48, 58, 79, 70, 49, 55, 77, 73, 51,
       53, 75, 75, 53, 51, 73, 77, 55, 49, 70, 79, 58, 48, 68, 78, 62},
      // Coefficients 6 taps
      {40, 56, 64, 56, 40, 0,  34, 49, 56, 52, 38, 27, 32, 47, 57, 53, 40, 27, 31, 46, 55, 54, 42, 28,
       29, 44, 55, 55, 44, 29, 28, 42, 54, 55, 46, 31, 27, 40, 53, 57, 47, 32, 27, 38, 52, 56, 49, 34},
    },

    // Band_8_38836
    {
      // Scaler inc range
      {37449, 38836},
      // Coefficients 4 taps
      {80, 96, 80, 0,  62, 78, 68, 48, 59, 78, 70, 49, 56, 76, 73, 51,
       53, 75, 75, 53, 51, 73, 76, 56, 49, 70, 78, 59, 48, 68, 78, 62},
      // Coefficients 6 taps
      {40, 56, 64, 56, 40, 0,  34, 49, 56, 52, 38, 27, 33, 47, 55, 53, 40, 28, 31, 46, 55, 54, 42, 28,
       30, 44, 54, 54, 44, 30, 28, 42, 54, 55, 46, 31, 28, 40, 53, 55, 47, 33, 27, 38, 52, 56, 49, 34},
    },

    // Band_8_40330
    {
      // Scaler inc range
      {38836, 40330},
      // Coefficients 4 taps
      {80, 96, 80, 0,  62, 78, 67, 49, 59, 77, 70, 50, 56, 76, 72, 52,
       54, 74, 74, 54, 52, 72, 76, 56, 50, 70, 77, 59, 49, 67, 78, 62},
      // Coefficients 6 taps
      {41, 56, 62, 56, 41, 0,  35, 49, 56, 51, 38, 27, 33, 47, 55, 53, 40, 28, 31, 45, 55, 54, 42, 29,
       30, 44, 54, 54, 44, 30, 29, 42, 54, 55, 45, 31, 28, 40, 53, 55, 47, 33, 27, 38, 51, 56, 49, 35},
    },

    // Band_8_41943
    {
      // Scaler inc range
      {40330, 41943},
      // Coefficients 4 taps
      {80, 96, 80, 0,  62, 77, 67, 50, 59, 77, 69, 51, 57, 74, 72, 53,
       54, 74, 74, 54, 53, 72, 74, 57, 51, 69, 77, 59, 50, 67, 77, 62},
      // Coefficients 6 taps
      {41, 56, 62, 56, 41, 0,  35, 48, 56, 51, 38, 28, 33, 47, 56, 52, 40, 28, 32, 45, 55, 53, 42, 29,
       30, 44, 54, 54, 44, 30, 29, 42, 53, 55, 45, 32, 28, 40, 52, 56, 47, 33, 28, 38, 51, 56, 48, 35},
    },

    // Band_8_43691
    {
      // Scaler inc range
      {41943, 43691},
      // Coefficients 4 taps
      {80, 96, 80, 0,  62, 76, 67, 51, 59, 76, 69, 52, 57, 75, 71, 53,
       55, 73, 73, 55, 53, 71, 75, 57, 52, 69, 76, 59, 51, 67, 76, 62},
      // Coefficients 6 taps
      {41, 56, 62, 56, 41, 0,  35, 48, 56, 51, 38, 28, 33, 47, 55, 52, 40, 29, 32, 45, 54, 53, 42, 30,
       31, 43, 54, 54, 43, 31, 30, 42, 53, 54, 45, 32, 29, 40, 52, 55, 47, 33, 28, 38, 51, 56, 48, 35},
    },

    // Band_8_45590
    {
      // Scaler inc range
      {43691, 45590},
      // Coefficients 4 taps
      {81, 94, 81, 0,  62, 75, 67, 52, 60, 74, 69, 53, 58, 73, 71, 54,
       56, 72, 72, 56, 54, 71, 73, 58, 53, 69, 74, 60, 52, 67, 75, 62},
      // Coefficients 6 taps
      {41, 56, 62, 56, 41, 0,  35, 48, 55, 51, 38, 29, 34, 47, 54, 52, 40, 29, 32, 45, 54, 53, 42, 30,
       31, 43, 54, 54, 43, 31, 30, 42, 53, 54, 45, 32, 29, 40, 52, 54, 47, 34, 29, 38, 51, 55, 48, 35},
    },

    // Band_8_47663
    {
      // Scaler inc range
      {45590, 47663},
      // Coefficients 4 taps
      {81, 94, 81, 0,  62, 75, 66, 53, 60, 74, 68, 54, 58, 73, 70, 55,
       56, 72, 72, 56, 55, 70, 73, 58, 54, 68, 74, 60, 53, 66, 75, 62},
      // Coefficients 6 taps
      {42, 56, 60, 56, 42, 0,  35, 48, 55, 50, 38, 30, 34, 46, 55, 51, 40, 30, 33, 45, 53, 52, 42, 31,
       32, 43, 53, 53, 43, 32, 31, 42, 52, 53, 45, 33, 30, 40, 51, 55, 46, 34, 30, 38, 50, 55, 48, 35},
    },

    // Band_8_49932
    {
      // Scaler inc range
      {47663, 49932},
      // Coefficients 4 taps
      {81, 94, 81, 0,  62, 74, 66, 54, 60, 73, 68, 55, 59, 71, 70, 56,
       57, 71, 71, 57, 56, 70, 71, 59, 55, 68, 73, 60, 54, 66, 74, 62},
      // Coefficients 6 taps
      {42, 55, 62, 55, 42, 0,  36, 48, 53, 50, 39, 30, 34, 46, 54, 51, 40, 31, 33, 45, 53, 52, 42, 31,
       32, 43, 53, 53, 43, 32, 31, 42, 52, 53, 45, 33, 31, 40, 51, 54, 46, 34, 30, 39, 50, 53, 48, 36},
    },

    // Band_8_52429
    {
      // Scaler inc range
      {49932, 52429},
      // Coefficients 4 taps
      {82, 92, 82, 0,  62, 73, 66, 55, 61, 72, 67, 56, 59, 71, 69, 57,
       58, 70, 70, 58, 57, 69, 71, 59, 56, 67, 72, 61, 55, 66, 73, 62},
      // Coefficients 6 taps
      {42, 55, 62, 55, 42, 0,  36, 47, 53, 50, 39, 31, 35, 46, 53, 51, 40, 31, 34, 45, 52, 51, 42, 32,
       33, 43, 52, 52, 43, 33, 32, 42, 51, 52, 45, 34, 31, 40, 51, 53, 46, 35, 31, 39, 50, 53, 47, 36},
    },

    // Band_8_55188
    {
      // Scaler inc range
      {52429, 55188},
      // Coefficients 4 taps
      {82, 92, 82, 0,  63, 71, 66, 56, 61, 71, 67, 57, 60, 70, 68, 58,
       59, 69, 69, 59, 58, 68, 70, 60, 57, 67, 71, 61, 56, 66, 71, 63},
      // Coefficients 6 taps
      {43, 55, 60, 55, 43, 0,  36, 47, 53, 49, 39, 32, 35, 46, 53, 50, 40, 32, 34, 44, 52, 51, 42, 33,
       33, 43, 52, 52, 43, 33, 33, 42, 51, 52, 44, 34, 32, 40, 50, 53, 46, 35, 32, 39, 49, 53, 47, 36},
    },

    // Band_8_58254
    {
      // Scaler inc range
      {55188, 58254},
      // Coefficients 4 taps
      {83, 90, 83, 0,  63, 71, 65, 57, 62, 69, 67, 58, 60, 70, 68, 58,
       59, 69, 69, 59, 58, 68, 70, 60, 58, 67, 69, 62, 57, 65, 71, 63},
      // Coefficients 6 taps
      {43, 55, 60, 55, 43, 0,  37, 47, 52, 49, 39, 32, 36, 45, 52, 50, 40, 33, 35, 44, 52, 50, 42, 33,
       34, 43, 51, 51, 43, 34, 33, 42, 50, 52, 44, 35, 33, 40, 50, 52, 45, 36, 32, 39, 49, 52, 47, 37},
    },

    // Band_8_61681
    {
      // Scaler inc range
      {58254, 61681},
      // Coefficients 4 taps
      {83, 90, 83, 0,  63, 70, 65, 58, 62, 69, 66, 59, 61, 69, 67, 59,
       60, 68, 68, 60, 59, 67, 69, 61, 59, 66, 69, 62, 58, 65, 70, 63},
      // Coefficients 6 taps
      {44, 54, 60, 54, 44, 0,  37, 46, 53, 48, 39, 33, 36, 45, 51, 49, 41, 34, 35, 44, 51, 50, 42, 34,
       35, 43, 50, 50, 43, 35, 34, 42, 50, 51, 44, 35, 34, 41, 49, 51, 45, 36, 33, 39, 48, 53, 46, 37},
    },

    // Band_8_65536
    {
      // Scaler inc range
      {61681, 65536},
      // Coefficients 4 taps
      {83, 90, 83, 0,  63, 69, 65, 59, 62, 68, 66, 60, 62, 68, 66, 60,
       61, 67, 67, 61, 60, 66, 68, 62, 60, 66, 68, 62, 59, 65, 69, 63},
      // Coefficients 6 taps
      {45, 54, 58, 54, 45, 0,  38, 46, 50, 48, 40, 34, 37, 45, 50, 49, 41, 34, 36, 44, 50, 49, 42, 35,
       35, 43, 50, 50, 43, 35, 35, 42, 49, 50, 44, 36, 34, 41, 49, 50, 45, 37, 34, 40, 48, 50, 46, 38},
    },

    // Band_8_69905
    {
      // Scaler inc range
      {65536, 69905},
      // Coefficients 4 taps
      {84, 88, 84, 0,  63, 67, 65, 61, 63, 67, 65, 61, 62, 67, 66, 61,
       62, 66, 66, 62, 61, 66, 67, 62, 61, 65, 67, 63, 61, 65, 67, 63},
      // Coefficients 6 taps
      {45, 54, 58, 54, 45, 0,  38, 46, 50, 47, 40, 35, 38, 45, 49, 48, 41, 35, 37, 44, 49, 48, 42, 36,
       36, 43, 49, 49, 43, 36, 36, 42, 48, 49, 44, 37, 35, 41, 48, 49, 45, 38, 35, 40, 47, 50, 46, 38},
    },

    // Band_8_74898
    {
      // Scaler inc range
      {69905, 74898},
      // Coefficients 4 taps
      {84, 88, 84, 0,  64, 66, 65, 61, 63, 66, 65, 62, 63, 66, 65, 62,
       62, 66, 66, 62, 62, 65, 66, 63, 62, 65, 66, 63, 61, 65, 66, 64},
      // Coefficients 6 taps
      {46, 53, 58, 53, 46, 0,  39, 45, 49, 47, 40, 36, 38, 44, 50, 47, 41, 36, 38, 44, 47, 48, 42, 37,
       37, 43, 48, 48, 43, 37, 37, 42, 48, 47, 44, 38, 36, 41, 47, 50, 44, 38, 36, 40, 47, 49, 45, 39},
    },

    // Band_8_80660
    {
      // Scaler inc range
      {74898, 80660},
      // Coefficients 4 taps
      {85, 86, 85, 0,  64, 66, 64, 62, 63, 66, 65, 62, 63, 65, 65, 63,
       63, 65, 65, 63, 63, 65, 65, 63, 62, 65, 66, 63, 62, 64, 66, 64},
      // Coefficients 6 taps
      {47, 53, 56, 53, 47, 0,  39, 45, 48, 46, 41, 37, 39, 44, 49, 46, 41, 37, 38, 43, 48, 47, 42, 38,
       38, 43, 47, 47, 43, 38, 38, 42, 47, 48, 43, 38, 37, 41, 46, 49, 44, 39, 37, 41, 46, 48, 45, 39},
    },

    // Band_8_87381
    {
      // Scaler inc range
      {80660, 87381},
      // Coefficients 4 taps
      {85, 86, 85, 0,  64, 65, 64, 63, 64, 65, 64, 63, 63, 65, 65, 63,
       63, 65, 65, 63, 63, 65, 65, 63, 63, 64, 65, 64, 63, 64, 65, 64},
      // Coefficients 6 taps
      {48, 53, 54, 53, 48, 0,  40, 44, 48, 45, 41, 38, 40, 44, 46, 46, 42, 38, 39, 43, 47, 46, 42, 39,
       39, 43, 46, 46, 43, 39, 39, 42, 46, 47, 43, 39, 38, 42, 46, 46, 44, 40, 38, 41, 45, 48, 44, 40},
    },

    // Band_8_95325
    {
      // Scaler inc range
      {87381, 95325},
      // Coefficients 4 taps
      {85, 86, 85, 0,  64, 65, 64, 63, 64, 65, 64, 63, 64, 64, 64, 64,
       64, 64, 64, 64, 64, 64, 64, 64, 63, 64, 65, 64, 63, 64, 65, 64},
      // Coefficients 6 taps
      {49, 52, 54, 52, 49, 0,  41, 44, 46, 45, 41, 39, 40, 44, 45, 45, 42, 40, 40, 43, 46, 45, 42, 40,
       40, 43, 45, 45, 43, 40, 40, 42, 45, 46, 43, 40, 40, 42, 45, 45, 44, 40, 39, 41, 45, 46, 44, 41},
    },

    // Band_8_104858
    {
      // Scaler inc range
      {95325, 104858},
      // Coefficients 4 taps
      {85, 86, 85, 0,  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
       64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64},
      // Coefficients 6 taps
      {49, 52, 54, 52, 49, 0,  41, 44, 45, 44, 42, 40, 41, 43, 45, 44, 42, 41, 41, 43, 45, 44, 42, 41,
       41, 43, 44, 44, 43, 41, 41, 42, 44, 45, 43, 41, 41, 42, 44, 45, 43, 41, 40, 42, 44, 45, 44, 41},
    },

    // Band_8_116508
    {
      // Scaler inc range
      {104858, 116508},
      // Coefficients 4 taps
      {85, 86, 85, 0,  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
       64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64},
      // Coefficients 6 taps
      {50, 52, 52, 52, 50, 0,  42, 43, 45, 43, 42, 41, 42, 43, 44, 44, 42, 41, 42, 43, 43, 44, 43, 41,
       42, 43, 43, 43, 43, 42, 41, 43, 44, 43, 43, 42, 41, 42, 44, 44, 43, 42, 41, 42, 43, 45, 43, 42},
    },

    // Band_8_131072
    {
      // Scaler inc range
      {116508, 131072},
      // Coefficients 4 taps
      {85, 86, 85, 0,  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
       64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64},
      // Coefficients 6 taps
      {51, 51, 52, 51, 51, 0,  42, 43, 44, 43, 42, 42, 42, 43, 43, 43, 43, 42, 42, 43, 43, 43, 43, 42,
       42, 43, 43, 43, 43, 42, 42, 43, 43, 43, 43, 42, 42, 43, 43, 43, 43, 42, 42, 42, 43, 44, 43, 42},
    },

    // Band_8_149797
    {
      // Scaler inc range
      {131072, 149797},
      // Coefficients 4 taps
      {85, 86, 85, 0,  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
       64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64},
      // Coefficients 6 taps
      {51, 51, 52, 51, 51, 0,  43, 43, 42, 43, 43, 42, 43, 43, 42, 43, 43, 42, 42, 43, 43, 43, 43, 42,
       42, 43, 43, 43, 43, 42, 42, 43, 43, 43, 43, 42, 42, 43, 43, 42, 43, 43, 42, 43, 43, 42, 43, 43},
    },

    // Band_8_174763
    {
      // Scaler inc range
      {149797, 174763},
      // Coefficients 4 taps
      {85, 86, 85, 0,  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
       64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64},
      // Coefficients 6 taps
      {51, 51, 52, 51, 51, 0,  43, 43, 41, 43, 43, 43, 43, 43, 41, 43, 43, 43, 43, 43, 41, 43, 43, 43,
       43, 43, 42, 42, 43, 43, 43, 43, 43, 41, 43, 43, 43, 43, 43, 41, 43, 43, 43, 43, 43, 41, 43, 43},
    },

    // Band_8_209715
    {
      // Scaler inc range
      {174763, 209715},
      // Coefficients 4 taps
      {85, 86, 85, 0,  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
       64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64},
      // Coefficients 6 taps
      {51, 51, 52, 51, 51, 0,  43, 43, 41, 43, 43, 43, 43, 43, 41, 43, 43, 43, 43, 43, 41, 43, 43, 43,
       43, 43, 42, 42, 43, 43, 43, 43, 43, 41, 43, 43, 43, 43, 43, 41, 43, 43, 43, 43, 43, 41, 43, 43},
    },

    // Band_8_262144
    {
      // Scaler inc range
      {209715, 262144},
      // Coefficients 4 taps
      {85, 86, 85, 0,  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
       64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64},
      // Coefficients 6 taps
      {51, 51, 52, 51, 51, 0,  43, 43, 41, 43, 43, 43, 43, 43, 41, 43, 43, 43, 43, 43, 41, 43, 43, 43,
       43, 43, 42, 42, 43, 43, 43, 43, 43, 41, 43, 43, 43, 43, 43, 41, 43, 43, 43, 43, 43, 41, 43, 43},
    },

    // Band_8_349525
    {
      // Scaler inc range
      {262144, 349525},
      // Coefficients 4 taps
      {85, 86, 85, 0,  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
       64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64},
      // Coefficients 6 taps
      {51, 51, 52, 51, 51, 0,  43, 43, 41, 43, 43, 43, 43, 43, 41, 43, 43, 43, 43, 43, 41, 43, 43, 43,
       43, 43, 42, 42, 43, 43, 43, 43, 43, 41, 43, 43, 43, 43, 43, 41, 43, 43, 43, 43, 43, 41, 43, 43},
    },

    // Band_8_524288
    {
      // Scaler inc range
      {349525, 524288},
      // Coefficients 4 taps
      {85, 86, 85, 0,  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
       64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64},
      // Coefficients 6 taps
      {51, 51, 52, 51, 51, 0,  43, 43, 41, 43, 43, 43, 43, 43, 41, 43, 43, 43, 43, 43, 41, 43, 43, 43,
       43, 43, 42, 42, 43, 43, 43, 43, 43, 41, 43, 43, 43, 43, 43, 41, 43, 43, 43, 43, 43, 41, 43, 43},
    },

    // Band_8_1048576
    {
      // Scaler inc range
      {524288, 1048576},
      // Coefficients 4 taps
      {85, 86, 85, 0,  64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
       64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64},
      // Coefficients 6 taps
      {51, 51, 52, 51, 51, 0,  43, 43, 41, 43, 43, 43, 43, 43, 41, 43, 43, 43, 43, 43, 41, 43, 43, 43,
       43, 43, 42, 42, 43, 43, 43, 43, 43, 41, 43, 43, 43, 43, 43, 41, 43, 43, 43, 43, 43, 41, 43, 43},
    },
  },
};

#endif  // YUV_SCALER_PARA_SET
