class CommandParserContent {
  static contentId() {
    return 'command-parser';
  }

  static parsePluginCommand(parser, command, args) {
    let methodName = 'parse_' + command.toLowerCase();
    while (methodName.indexOf('-') >= 0) {
      methodName = methodName.replace('-', '_');
    }  

    if (this[methodName] !== undefined) {
      try {
        this[methodName](parser);
      }
      catch(exception) {
        console.error(exception);
      }

      return true;
    }

    return false;
  }

  static parse_play(parser) {
    if (parser.upperName == 'CHILD') {
      if (parser.hasMoreArgs()) {
        var eventName = parser.nextArg();
        Managers.CommonEvent.playChildEvent(eventName, parser.interpreter);
      }

      return;
    }

    if (parser.upperName == 'SOUND') {
      if (parser.hasMoreArgs()) {
        var soundName = parser.nextArg(true);
        var pitch;
        
        if (parser.hasMoreArgs()) {
          pitch = Number(parser.nextArg(true));
          if (isNaN(pitch)) {
            pitch = undefined;
          }
        }

        if (Managers.Sound[`play${ soundName }`]) {
          Managers.Sound[`play${ soundName }`]();
          return;
        }

        Engine.Audio.playSe(soundName, pitch);
      }

      return;
    }

    if (parser.upperName == 'EVENT') {
      if (parser.hasMoreArgs()) {
        const thisEventName = parser.nextArg(true);
        const async = parser.hasMoreArgs() && parser.nextUpperArg(true) === 'ASYNC';

        if (async) {
          Managers.CommonEvent.playEventAsync(thisEventName);
        } else {
          Managers.CommonEvent.playEvent(thisEventName);
        }
      }
      return;
    }

    Managers.CommonEvent.playEvent(parser.name);
  }

  static parse_teleport(parser) {
    if (parser.nextUpperArg() == 'TO') {
      parser.skipArg();
    }

    if (parser.hasMoreArgs()) {
      var mapLabel = '';
      var mapId = 0;
      
      do {
        mapLabel = (mapLabel + ' ' + parser.nextArg(true)).trim();
        mapId = Managers.Map.getMapIdByLabel(mapLabel);
      } while (mapId === 0 && parser.hasMoreArgs());

      if (mapId > 0) {
        var autoSave = true;
        var allowFestival = false;

        if (parser.nextUpperArg() == 'FROM') {
          parser.skipArg();

          if (parser.nextUpperArg() == 'FESTIVAL') {
            allowFestival = true;
            parser.skipArg();
          }
        }

        if (parser.nextUpperArg() == 'WITHOUT') {
          parser.skipArg();
          if (parser.nextUpperArg() == 'SAVING') {
            autoSave = false;
            parser.skipArg();
          }
        }

        var params = Managers.EventScript.parseList(parser.remainingLine());

        if (params.length >= 2) {
          var x = parseInt(params[0], 10);
          var y = parseInt(params[1], 10);
          var d = 2;

          if (params.length > 2) {
            d = DirectionHelper.parseDirection(params[2], 2);
          }

          if (mapId == $gameMap._mapId) {
            $gamePlayer.requestMapReload();
          }
          $gamePlayer.reserveTransfer(mapId, x, y, d, 0, autoSave, allowFestival);
        }
      }
    }
  }

  static parse_include(parser) {
    if (parser.upperName == 'CHAT') {
      Managers.CommonEvent.includeChild($gameTemp.chatName || '', parser.interpreter);
      return;
    }

    Managers.CommonEvent.includeChild(parser.nameMaybeVariable(), parser.interpreter);
  }

  static parse_say(parser) {
    Managers.CommonEvent.playEvent(parser.name, parser.interpreter.character());
  }

  static parse_add(parser) {
    if (parser.upperName == 'CHOICE') {
      parser.interpreter.addChoice(parser.line.trim().substring(6).trim());
      return;
    }

    if (parser.upperName == 'HOURS') {
      const hours = parser.nextArgAsFloat();
      if (!isNaN(hours)) {
        Managers.Time.addHours(hours);
        return;
      }
    }
  }

  static parse_set(parser) {
    if (['HOUR', 'MINUTE', 'SECOND', 'DAY', 'MONTH', 'YEAR'].includes(parser.upperName)) {
      const value = parser.nextArgAsFloat();
      if (!isNaN(value)) {
        if (parser.upperName === 'SECOND') {
          Managers.Time.seconds = value;
        } else {
          Managers.Time[parser.name.toLowerCase()] = value;
        }

        Managers.Time.updateTime(true, false);
      }
      return;
    }

    if (parser.upperName == 'TIME') {
      const time = Managers.Time.convertStringToConfg(parser.nextArg());
      Managers.Time.setTime(time.seconds, time.minute, time.hour);
      return;
    }
  }

  static parse_add_choice(parser) {
    parser.interpreter.addChoice(parser.line);
  }

  static parse_clear(parser) {
    if (parser.upperName == 'CHOICES') {
      parser.interpreter.clearReservedChoices();
    }
  }

  static parse_clear_choices(parser) {
    parser.interpreter.clearReservedChoices();
  }

  static parse_show(parser) {
    if (parser.upperName == 'ANIMATION') {
      let villagerName = parser.nextArg(true);
      if (villagerName.toUpperCase() == 'VARIABLE') {
        const variableName = parser.nextArg(true);
        villagerName = Variables[variableName];
      }

      const animationName = parser.nextUpperArg(true);
      switch (animationName) {
        case 'TELEPORT':
          Managers.Villagers.showAnimation(villagerName, Constants.TELEPORT_ANIMATION);
          return;
        case 'THUNDER':
          Managers.Villagers.showAnimation(villagerName, Constants.THUNDER_ANIMATION);
          return;
        case 'SPA':
          if (villagerName.toLowerCase() === 'player') {
            $gamePlayer.animateSpaJump();
          }
          return;
      }
    }
  }

  static parse_start(parser) {
    if (parser.upperName == 'CUTSCENE') {
      var killVillagers = false;
      var hideCreatures = false;

      for (var i = 0; i < parser.args.length; i++) {
        if (parser.args[i].toUpperCase() == 'NO' && parser.args.length > i + 1) {
          if (parser.args[i + 1].toUpperCase() == 'VILLAGERS') {
            killVillagers = true;
          }

          if (parser.args[i + 1].toUpperCase() == 'CREATURES') {
            hideCreatures = true;
          }
        }
      }

      $gameSystem.startCutscene(killVillagers, hideCreatures);
    }
  }

  static parse_end(parser) {
    if (parser.upperName == 'CUTSCENE') {
      $gameSystem.endCutscene();
    }
  }

  static parse_retain(parser) {
    if (parser.upperName == 'CUTSCENE') {
      $gameSystem.retainCutscene();
      if (parser.hasMoreArgs() && parser.nextUpperArg() == 'UNTIL') {
        parser.skipArg();

        if (parser.hasMoreArgs() && parser.nextUpperArg() == 'TELEPORT') {
          parser.skipArg();
          $gameSystem.letPlayerLeaveCutscene();
        }
      }
    }
  }

  static parse_abort(parser) {
    if (parser.upperName == 'CUTSCENE') {
      $gameSystem.abortCutscene();
    }
  }

  static parse_refresh(parser) {
    if (parser.upperName == 'HUD') {
      Managers.Items.requestHudRefresh();
      return;
    }

    if (parser.upperName == 'TIME') {
      Managers.Villagers.killAllVillagerEvents();
      Managers.Time.updateTime();
      return;
    }

    if (parser.upperName == 'MAP') {
      $gameMap.refreshWholeMap();
      return;
    }

    if (parser.upperName == 'BGM') {
      $gameMap.autoplay();
      return;
    }
  }

  static parse_switch(parser) {
    if (parser.hasMoreArgs()) {
      var value = parser.nextArg(true);

      Switches[parser.name] = Utils.evaluatesToTrue(value);
    } else {
      Switches[parser.name] = true;
    }
  }

  static parse_through(parser) {
    let value = true;
    if (parser.hasMoreArgs()) {
      value = Utils.evaluatesToTrue(parser.nextArg(true));
    }

    Managers.Villagers.setThrough(parser.name, value);
  }

  static parse_talk(parser) {
    Managers.Villagers.talkToCharacter(parser.name);
  }

  static parse_meet(parser) {
    Managers.Relationship.markCharacterAsKnown(parser.name);
  }

  static parse_variable(parser) {
    if (parser.hasMoreArgs()) {
      let value = parser.nextArg(true);

      try {
        const floatValue = parseFloat(value, 10);
        if (!isNaN(floatValue)) {
          value = floatValue;
        }
      }
      catch(e) {
        // if it didn't parse as float, then save whatever it was
      }

      Variables[parser.name] = value;
    }
  }

  static parse_accept(parser) {
    Managers.Tasks.acceptTask(parser.name);
  }

  static parse_refuse(parser) {
    if (parser.upperName == 'ITEM') {
      parser.parse_rejectparser();
      return;
    }

    Managers.Tasks.refuseTask(parser.name);
  }

  static parse_learn(parser) {
    Managers.Cooking.registerLearntRecipe(parser.name);
  }

  static parse_take(parser) {
    if (parser.upperName == 'ITEM' || parser.upperName == 'GIFT') {
      Managers.Items.takeGift();
    }
  }

  static parse_reject(parser) {
    if (parser.upperName == 'ITEM' || parser.upperName == 'GIFT') {
      Managers.Items.rejectGift();
    }
  }

  static parse_replace(parser) {
    if (parser.upperName == 'TASK') {
      if (parser.hasMoreArgs(2)) {
        var oldTaskId = parser.nextArg(true);
        var newTaskId = parser.nextArg(true);

        Managers.Tasks.replaceTask(oldTaskId, newTaskId);
      }
    }
  }

  static parse_deliver(parser) {
    const itemName = parser.name;
    let amount = 1;

    if (parser.hasMoreArgs(2)) {
      if (parser.nextUpperArg() == 'X') {
        parser.skipArg();
        amount = parser.nextArgAsFloat();
      }
    }

    Managers.Items.deliverItemId(itemName, amount);
  }

  static parse_lose(parser) {
    if (parser.upperName == 'GOLD') {
      if (parser.nextUpperArg() === 'VARIABLE') {
        parser.skipArg(1);
        const variableName = parser.nextArg(true);
        const value = Variables[variableName];
        Managers.Items.loseGold(value);
        return;
      }

      const gold = parser.nextArgAsFloat(true);
      Managers.Items.loseGold(gold);
      return;
    }

    let itemName = parser.name;
    let selected = false;
    if (itemName.toUpperCase() === 'SELECTED') {
      selected = true;
      parser.skipArg(1);
    }

    if (parser.upperName == 'ITEM') {
      itemName = parser.nextArg(true);
    }

    if (itemName.toUpperCase() == 'VARIABLE') {
      const variableName = parser.nextArg(true);
      itemName = Variables[variableName];
    }

    if (parser.hasMoreArgs(2)) {
      if (parser.nextUpperArg() == 'X') {
        parser.skipArg();
        const amount = parser.nextArgAsFloat();

        if (selected) {
          Managers.Items.loseUnitOfSelectedItem(amount);
        } else {
          Managers.Items.loseItemId(itemName, amount);
        }
      }
    } else {
      if (selected) {
        Managers.Items.loseUnitOfSelectedItem();
      } else {
        Managers.Items.loseItemId(itemName);
      }
    }
  }

  static parse_increase(parser) {
    return this.parse_gain(parser);
  }

  static parse_pick(parser) {
    let itemName = parser.name;
    if (parser.upperName == 'ITEM') {
      itemName = parser.nextArg(true);
    }

    if (itemName.toUpperCase() == 'VARIABLE') {
      const variableName = parser.nextArg(true);
      itemName = Variables[variableName];
    }

    if (parser.hasMoreArgs(2)) {
      if (parser.nextUpperArg() == 'X') {
        parser.skipArg();
        const amount = parser.nextArgAsFloatMaybeVariable();

        Managers.Items.pickItemId(itemName, amount);
      }
    } else {
      Managers.Items.pickItemId(itemName);
    }
  }

  static parse_gain(parser) {
    if (parser.upperName == 'GOLD') {
      if (parser.nextUpperArg() == 'X') {
        parser.skipArg();
      }

      const gold = parser.nextArgAsFloat(true);
      Managers.Items.gainGold(gold);
      return;
    }

    if (parser.upperName == 'ACHIEVEMENT') {
      const achievementName = parser.nextArg(true);
      if (Managers.Achievements.isAchievementRequirementsMet(achievementName)) {
        Managers.Steam.activateAchievement(achievementName);
      }
      return;
    }

    if (parser.upperName == 'FRIENDSHIP') {
      let villagerName = parser.nextArg(true);
      if (villagerName.toUpperCase() == 'VARIABLE') {
        const variableName = parser.nextArg(true);
        villagerName = Variables[variableName];
      }
      let amount = 10;

      if (parser.hasMoreArgs()) {
        if (parser.nextUpperArg() == 'X') {
          parser.skipArg();
        }

        amount = parser.nextArgAsFloat();
      }

      Managers.Relationship.increaseFriendship(villagerName, amount);
      return;
    }

    let itemName = parser.name;
    if (parser.upperName == 'ITEM') {
      itemName = parser.nextArg(true);
    }

    if (itemName.toUpperCase() == 'VARIABLE') {
      const variableName = parser.nextArg(true);
      itemName = Variables[variableName];
    }

    if (parser.hasMoreArgs(2)) {
      if (parser.nextUpperArg() == 'X') {
        parser.skipArg();
        const amount = parser.nextArgAsFloat();

        Managers.Items.forceGainItemId(itemName, amount);
      }
    } else {
      Managers.Items.forceGainItemId(itemName);
    }
  }

  static parse_complete(parser) {
    var taskName = parser.name;
    if (parser.hasMoreArgs() && parser.nextUpperArg() == 'HIDDEN') {
      Managers.Tasks.completeTask(taskName);
    } else {
      Managers.Tasks.completeTaskAnimation(taskName);
    }
  }

  static parse_remove(parser) {
    if (parser.upperName == 'EVENT') {
      if (!parser.hasMoreArgs()) return;

      let mapEventOnly = false;
      let hardPositionOnly = false;
      let objectsOnly = false;
      let priorityOnly = false;

      if (parser.nextUpperArg() == 'HARD') {
        hardPositionOnly = true;
        parser.skipArg(1);
      }

      if (parser.nextUpperArg() == 'OBJECT') {
        objectsOnly = true;
        parser.skipArg(1);
      }

      if (parser.nextUpperArg() == 'PRIORITY') {
        parser.skipArg(1);
        priorityOnly = parser.nextArgAsFloat();
        parser.skipArg(1);
      }

      if (parser.nextUpperArg() == 'ONLY') {
        mapEventOnly = true;
        parser.skipArg(1);
      }

      const params = Managers.EventScript.parseList(parser.remainingLine());
      if (params.length >= 2) {
        const x = parseInt(params[0].trim(), 10);
        const y = parseInt(params[1].trim(), 10);

        let events;
        if (hardPositionOnly) {
          events = $gameMap.eventsHardXy(x, y);
        } else if (mapEventOnly) {
          events = $gameMap.eventsXy(x, y);
        } else if (objectsOnly) {
          events = $gameMap.farmObjectsXy(x, y);
        } else {
          events = $gameMap.anythingXy(x, y);
        }

        if (events && events.length) {
          for (let event of events) {
            if (priorityOnly !== false) {
              if (event._priorityType !== priorityOnly) {
                continue;
              }
            }

            event && event.erase();
          }
        }
      }
      return;
    }

    var villagerName = parser.name;

    if (parser.upperName == 'VILLAGER') {
      if (!parser.hasMoreArgs()) return;

      villagerName = parser.nextArg(true);
      if (villagerName.toUpperCase() == 'VARIABLE') {
        const variableName = parser.nextArg(true);
        villagerName = Variables[variableName];
      }
    }

    Managers.Villagers.removeVillagerEvent(villagerName);
  }

  static parse_create(parser) {
    if (parser.upperName == 'VILLAGER') {
      if (!parser.hasMoreArgs()) return;

      let villagerName = parser.nextArg(true);
      if (villagerName.toUpperCase() == 'VARIABLE') {
        const variableName = parser.nextArg(true);
        villagerName = Variables[variableName];
      }

      const params = Managers.EventScript.parseList(parser.remainingLine());

      if (params.length >= 2) {
        const x = parseInt(params[0].trim(), 10);
        const y = parseInt(params[1].trim(), 10);
        var d = 2;

        if (params.length > 2) {
          d = DirectionHelper.parseDirection(params[2].trim(), 2);
        }

        $gameMap.createVillagerByNameAt(villagerName, x, y, d);
      }
      return;
    }

    if (parser.upperName == 'ITEM') {
      if (!parser.hasMoreArgs()) return;

      let priority = 1;
      if (parser.nextUpperArg() === 'FLYING') {
        priority = 2;
        parser.skipArg(1);
      }

      const itemName = parser.nextArg(true);
      const params = Managers.EventScript.parseList(parser.remainingLine());

      if (params.length >= 2) {
        const x = parseInt(params[0].trim(), 10);
        const y = parseInt(params[1].trim(), 10);

        const itemData = Managers.Items.getItemData(itemName);
        if (itemData) {
          $gameMap.createStaticIconEventAt(itemData.iconIndex, x, y, priority, true);
        }
      }

      return;
    }
  }

  static parse_speed(parser) {
    const villagerName = parser.name;
    const speedStr = parser.nextArg(false);

    if (speedStr.toUpperCase() == 'RESTORE') {
      Managers.Villagers.restoreSpeed(villagerName);
      return;
    }

    const speed = parser.nextArgAsFloat();

    Managers.Villagers.changeSpeed(villagerName, speed);
  }

  static parse_offset(parser) {
    let villagerName = parser.name;

    if (!parser.hasMoreArgs()) return;
    if (parser.upperName == 'VILLAGER') {
      villagerName = parser.nextArg(true);
    }

    if (villagerName.toUpperCase() == 'VARIABLE') {
      const variableName = parser.nextArg(true);
      villagerName = Variables[variableName];
    }

    let propName;

    if (parser.nextUpperArg(false) == 'Y') {
      propName = 'offsetY';
    } else {
      propName = 'offsetX';
    }
    parser.skipArg(1);

    const value = parseFloat(Managers.EventScript.parseList(parser.remainingLine()), 10);
    const event = Managers.Villagers.getVillagerData(villagerName);

    event[propName] = value;
    return;
  }

  static parse_stop(parser) {
    let villagerName = parser.name;

    if (parser.upperName == 'VILLAGER') {
      villagerName = parser.nextArg(true);
      if (villagerName.toUpperCase() == 'VARIABLE') {
        const variableName = parser.nextArg(true);
        villagerName = Variables[variableName];
      }
    }

    Managers.Villagers.stop(villagerName);
  }

  static parse_move(parser) {
    let villagerName = parser.name;

    if (parser.upperName == 'VILLAGER') {
      villagerName = parser.nextArg(true);
      if (villagerName.toUpperCase() == 'VARIABLE') {
        const variableName = parser.nextArg(true);
        villagerName = Variables[variableName];
      }
    }

    if (!parser.hasMoreArgs()) return;

    let setPosition = false;
    if (parser.nextUpperArg(false) == 'NOW') {
      setPosition = true;
      parser.skipArg(1);
    }

    const params = Managers.EventScript.parseList(parser.remainingLine());

    if (params.length >= 2) {
      const x = parseInt(params[0].trim(), 10);
      const y = parseInt(params[1].trim(), 10);

      if (setPosition) {
        Managers.Villagers.setPosition(villagerName, x, y);
        Managers.Villagers.stop(villagerName);
        return;
      }

      let d = 0;
      if (params.length > 2) {
        d = DirectionHelper.parseDirection(params[2].trim(), 0);
      }

      Managers.Villagers.move(villagerName, x, y, d);
    }
  }

  static parse_turn(parser) {
    let villagerName = parser.name;
    if (!parser.hasMoreArgs()) return;

    if (parser.upperName == 'VILLAGER') {
      villagerName = parser.nextArg(true);
      if (villagerName.toUpperCase() == 'VARIABLE') {
        const variableName = parser.nextArg(true);
        villagerName = Variables[variableName];
      }
      if (!parser.hasMoreArgs()) return;
    }

    const dName = parser.nextArg(true);
    if (Managers.Villagers.getVillagerData(dName)) {
      Managers.Villagers.turn(villagerName, dName);
      return;
    }

    const d = DirectionHelper.parseDirection(dName, 2);
    Managers.Villagers.turn(villagerName, d);
  }

  static parse_assign(parser) {
    if (parser.upperName == 'EVENT') {
      if (!parser.hasMoreArgs(3)) return;

      var eventName = parser.nextArg(true);

      if (parser.nextUpperArg() == 'TO') {
        parser.skipArg();
      }

      if (parser.nextUpperArg() == 'VILLAGER') {
        parser.skipArg();
        if (!parser.hasMoreArgs()) return;

        let villagerName = parser.nextArg(true);
        if (villagerName.toUpperCase() == 'VARIABLE') {
          const variableName = parser.nextArg(true);
          villagerName = Variables[variableName];
        }

        Managers.Villagers.assignEvent(villagerName, eventName);
        return;
      }

      if (parser.nextUpperArg() == 'ITEM') {
        parser.skipArg();
        if (!parser.hasMoreArgs()) return;

        var itemName = parser.nextArg(true);

        if (parser.hasMoreArgs()) {
          if (parser.nextUpperArg() == 'ON') {
            parser.skipArg();
          }
          if (parser.nextUpperArg() == 'ACTION') {
            parser.skipArg();
          }

          if (parser.hasMoreArgs()) {
            var actionType = parser.nextArg(true);

            Managers.Items.assignEvent(itemName, eventName, actionType);
            return;
          }
        }

        Managers.Items.assignEvent(itemName, eventName);
        return;
      }
    }
  }

  static parse_scroll(parser) {
    const direction = parser.upperName;

    if (!parser.hasMoreArgs()) return;

    let distance;
    try {
      distance = parseInt(parser.remainingLine(), 10);
    } catch(e) {
      distance = 1;
    }

    switch(direction) {
      case 'DOWN':
        $gameMap.scrollDown(distance);
        break;
      case 'LEFT':
        $gameMap.scrollLeft(distance);
        break;
      case 'RIGHT':
        $gameMap.scrollRight(distance);
        break;
      case 'UP':
        $gameMap.scrollUp(distance);
        break;
    }
  }

  static parse_restore(parser) {
    if (parser.upperName == 'CAMERA') {
      $gameMap.restoreScroll(parser.nextUpperArg() == 'NOW');
      return;
    }

    if (parser.upperName == 'HEALTH') {
      if (parser.nextUpperArg() === 'SPA') {
        Managers.Health.applySpaEffect();
        return;
      }

      if (parser.nextUpperArg() === 'FULL') {
        parser.skipArg(1);

        if (parser.nextUpperArg() === 'STAMINA') {
          parser.skipArg(1);
          Managers.Health.fullyRestoreStamina();
        }

        if (parser.nextUpperArg() === 'FATIGUE') {
          parser.skipArg(1);
          Managers.Health.fullyRestoreFatigue();
        }

        return;
      }

    }
  }

  static parse_snap(parser) {
    Managers.Scenes.snapForBackground();
  }

  static parse_villager_item_balloon(parser) {
    const villagerName = parser.name;
    if (!parser.hasMoreArgs(1)) {
      return;
    }

    const itemName = parser.nextArg();
    Managers.Villagers.showItemBalloon(villagerName, itemName);
  }

  static parse_villager_item_display(parser) {
    const villagerName = parser.name;
    let itemName;

    if (parser.hasMoreArgs(1)) {
      itemName = parser.nextArg();
    }

    if (itemName && (itemName.toLowerCase() === 'none') || (itemName.toLowerCase() === 'undefined'))  {
      itemName = undefined;
    }

    Managers.Villagers.pickItemId(villagerName, itemName);
  }

  static parse_displayitemid(parser) {
    const itemName = parser.nameMaybeVariable();
    Managers.Items.displayItemId = itemName.toUpperCase() === 'UNDEFINED' ? undefined : itemName;
  }

  static parse_erase(parser) {
    parser.interpreter.command214();
  }

  static parse_open(parser) {
    if (parser.upperName === 'STORAGE') {
      Managers.Items.openStorage();
      return;
    }
  }

  static parse_force(parser) {
    if (parser.upperName === 'CHAT') {
      let villagerName = parser.nextArg(true);
      if (villagerName.toUpperCase() == 'VARIABLE') {
        const variableName = parser.nextArg(true);
        villagerName = Variables[variableName];
      }

      Managers.Villagers.forceChatWithVillager(villagerName);
      return;
    }
  }

  static parse_trashcan(parser) {
    const trashContainer = parser.name || 'default';

    Managers.Items.doTrashCanEvent(trashContainer);
  }

  static parse_transfer(parser) {
    if (parser.upperName === 'STORE') {
      $gamePlayer.backupTransferPosition();
      return;
    }

    if (parser.upperName === 'NOW') {
      $gamePlayer.transferToStoredPosition();
      return;
    }
  }

  static parse_change(parser) {
    let villagerName = parser.name;

    if (parser.upperName == 'VILLAGER') {
      villagerName = parser.nextArg(true);
      if (villagerName.toUpperCase() == 'VARIABLE') {
        const variableName = parser.nextArg(true);
        villagerName = Variables[variableName];
      }
    }

    if (!parser.hasMoreArgs()) return;

    if (parser.nextUpperArg() === 'SPRITE') {
      parser.skipArg();
      const command = parser.nextUpperArg(true);
      if (command === 'NAME') {
        const newFile = parser.nextArg();
        Managers.Villagers.changeSpriteImage(villagerName, newFile);
      } else if (command === 'INDEX') {
        const newIndex = parser.nextArgAsFloat();
        Managers.Villagers.changeSpriteIndex(villagerName, newIndex);
      }
    }
  }

  static parse_pause(parser) {
    const paused = Utils.evaluatesToTrue(parser.command);
    Managers.Time.paused = paused;
  }

  static parse_preload(parser) {
    if (parser.upperName === 'CHARACTER') {
      const name = parser.nextArg();
      Managers.Images.loadCharacter(name);
    }
  }

  static parse_release(parser) {
    if (parser.upperName === 'CHARACTER') {
      const name = parser.nextArg();
      Managers.Images.releaseCharacter(name);
    }
  }

  static parse_tool(parser) {
    if (parser.upperName === 'UNEQUIP') {
      Managers.Tools.setTool();
      return;
    }

    if (parser.upperName === 'USE') {
      Managers.Tools.useTool();
      return;
    }
  }

  static parse_use(parser) {
    if (parser.upperName === 'TOOL') {
      Managers.Tools.useTool();
      return;
    }
  }

  static parse_history(parser) {
    if (parser.upperName === 'REGISTER') {
      const registrationType = parser.nextUpperArg(true);

      if (registrationType === 'EVENT') {
        const eventName = parser.nextArgMaybeVariable(true);
        if (eventName) {
          Managers.History.registerEvent(eventName);
          return;
        }
      }
    }
  }

  static parse_debugger(parser) {
    /* eslint-disable no-debugger */
    debugger;
    /* eslint-enable no-debugger */
  }
}

Managers.Content.registerContentClass(CommandParserContent);
