require('game/managers/Content');
require('game/objects/Object');
require('engine/sprites/Sprite_Object');
require('game/models/BaseModel');
require('game/tools/Tool');

Models.CheeseMaker = class CheeseMakerModel extends Models.BaseModel {
  static get modelName() {
    return 'cheese-maker';
  }

  static defaultData(gameObject) {
    return {
      spriteName: 'machines',
      spriteIndex: 0,
      direction: 2,
      pattern: 0,
      priority: 1,
      hitboxHeight: 0.5,
      hitboxY: 0.5,
      hitboxWidth: 0.5,
      hitboxX: 0.25
    };
  }
};

Models.MayoMaker = class MayoMakerModel extends Models.BaseModel {
  static get modelName() {
    return 'mayo-maker';
  }

  static defaultData(gameObject) {
    return {
      spriteName: 'machines',
      spriteIndex: 1,
      direction: 2,
      pattern: 0,
      priority: 1,
      hitboxHeight: 0.5,
      hitboxY: 0.5,
      hitboxWidth: 0.5,
      hitboxX: 0.25
    };
  }
};

Models.HayMaker = class HayMakerModel extends Models.BaseModel {
  static get modelName() {
    return 'hay-maker';
  }

  static defaultData(gameObject) {
    return {
      spriteName: '-hay-maker',
      spriteIndex: 0,
      direction: 2,
      pattern: 0,
      priority: 1,
      hitboxHeight: 1,
      hitboxY: 0,
      hitboxWidth: 1.5,
      hitboxX: 0,
      offsetX: 8
    };
  }
};

Models.FeedMaker = class FeedMakerModel extends Models.BaseModel {
  static get modelName() {
    return 'feed-maker';
  }

  static defaultData(gameObject) {
    return {
      spriteName: '$$$feed-maker',
      spriteIndex: 0,
      direction: 2,
      pattern: 0,
      priority: 1,
      hitboxHeight: 1,
      hitboxY: 0,
      hitboxWidth: 1.5,
      hitboxX: 0
    };
  }
};

Models.SeedMaker = class SeedMakerModel extends Models.BaseModel {
  static get modelName() {
    return 'seed-maker';
  }

  static defaultData(gameObject) {
    return {
      spriteName: '-seed-maker',
      spriteIndex: 0,
      direction: 2,
      pattern: 0,
      priority: 1,
      hitboxHeight: 1,
      hitboxY: 0,
      hitboxWidth: 1,
      hitboxX: 0
    };
  }
};

Models.YarnMaker = class YarnMakerModel extends Models.BaseModel {
  static get modelName() {
    return 'yarn-maker';
  }

  static defaultData(gameObject) {
    return {
      spriteName: '$$$yarn-maker',
      spriteIndex: 0,
      direction: 2,
      pattern: 0,
      priority: 1,
      hitboxHeight: 1,
      hitboxY: 0.25,
      hitboxWidth: 2,
      hitboxX: -0.5
    };
  }
};

class Sprite_Machine extends Sprite_Object {
  updateShader() {
    // No shaders
  }
}

class MachineObject extends Objects.Object {
  useMachine() {
    if (Managers.Items.waiting) {
      return;
    }

    const farmObjectData = this._farmObjectData;
    if (farmObjectData) {
      farmObjectData.lastTouch = false;
    }

    if (!this.isHoldingValidItem()) {
      return;
    }

    const itemId = this.resultingItemId();
    if (!itemId) {
      return;
    }

    if (!Managers.Items.canGainItemId(itemId)) {
      this.playNoRoomEvent();
      return;
    }

    this.playProductionEvent();
  }

  resultingItemId() {
    return '';
  }

  isHoldingValidItem() {
    return false;
  }

  playNoRoomEvent() {

  }

  playProductionEvent() {

  }

  start(activatedByMouse) {
    if (this.usageTimeout) {
      clearTimeout(this.usageTimeout);
      this.usageTimeout = false;
    }

    if (activatedByMouse && !TouchInput.isTriggered()) {
      return false;
    }

    if (this.isHoldingValidItem()) {
      this.useMachine();
      return true;
    }

    const started = super.start(activatedByMouse);
    if (started) {
      return started;
    }

    this.usageTimeout = setTimeout(() => {
      this.usageTimeout = false;
      this.useMachine();
    }, 500);
    return true;
  }
}

class CheeseMaker extends MachineObject {
  getSpriteClass() {
    return Sprite_Machine;
  }

  resultingItemId() {
    if (Managers.Items.isHoldingItem('milk')) {
      return 'cheese';
    }
    if (Managers.Items.isHoldingItem('goat-milk')) {
      return 'goat-cheese';
    }
    if (Managers.Items.isHoldingItem('mountain-goat-milk')) {
      return 'mountain-goat-cheese';
    }

    return '';
  }

  isHoldingValidItem() {
    return Managers.Items.isHoldingItem('milk') || Managers.Items.isHoldingItem('goat-milk') || Managers.Items.isHoldingItem('mountain-goat-milk');
  }

  playNoRoomEvent() {
    Managers.CommonEvent.playEvent('check_cheese_maker_no_room');
  }

  playProductionEvent() {
    Managers.CommonEvent.playEvent('use_cheese_maker');
  }
}

class MayoMaker extends MachineObject {
  getSpriteClass() {
    return Sprite_Machine;
  }

  resultingItemId() {
    return 'mayo';
  }

  isHoldingValidItem() {
    return Managers.Items.isHoldingItem('egg') || Managers.Items.isHoldingItem('duck-egg') || Managers.Items.isHoldingItem('fertilized-egg');
  }

  playNoRoomEvent() {
    Managers.CommonEvent.playEvent('check_mayo_maker_no_room');
  }

  playProductionEvent() {
    Managers.CommonEvent.playEvent('use_mayo_maker');
  }
}

class HayMaker extends MachineObject {
  getSpriteClass() {
    return Sprite_Machine;
  }

  resultingItemId() {
    return 'hay';
  }

  isHoldingValidItem() {
    return Managers.Items.isHoldingItem('weeds') || Managers.Items.isHoldingItem('yellow-weeds') || Managers.Items.isHoldingItem('red-weeds') || Managers.Items.isHoldingItem('wheat');
  }

  playNoRoomEvent() {
    Managers.CommonEvent.playEvent('check_hay_maker_no_room');
  }

  playProductionEvent() {
    Managers.CommonEvent.playEvent('use_hay_maker');
  }
}

class FeedMaker extends MachineObject {
  getSpriteClass() {
    return Sprite_Machine;
  }

  resultingItemId() {
    return 'chicken-feed';
  }

  isHoldingValidItem() {
    return Managers.Items.isHoldingItem('corn') || Managers.Items.isHoldingItem('bait');
  }

  playNoRoomEvent() {
    Managers.CommonEvent.playEvent('check_feed_maker_no_room');
  }

  playProductionEvent() {
    Managers.CommonEvent.playEvent('use_feed_maker');
  }
}

class SeedMaker extends MachineObject {
  getSpriteClass() {
    return Sprite_Machine;
  }

  resultingItemId() {
    if (!Managers.Items.isHoldingItem()) return '';

    const itemId = Managers.Items.selectedItem.id;
    return `${ itemId }-seeds`;
  }

  isHoldingValidItem() {
    if (!Managers.Items.isHoldingItem()) return false;

    const itemId = Managers.Items.selectedItem.id;
    const seedId = `${ itemId }-seeds`;

    const seedItem = Managers.Items.getItemData(seedId);
    if (!seedItem) return false;
    if (seedItem.disabled) return false;

    return true;
  }

  playNoRoomEvent() {
    Managers.CommonEvent.playEvent('check_seed_maker_no_room');
  }

  playProductionEvent() {
    if (Managers.Items.isHoldingItem()) {
      Variables.seedId = `${ Managers.Items.selectedItem.id }-seeds`;
    } else {
      Variables.seedId = 'jasmine-seeds';
    }

    const seedData = Managers.Items.getItemData(Variables.seedId);
    Variables.seedAmount = (seedData && seedData.daysToRegrow) ? 1 : (Math.randomInt(3) + 1);

    Managers.CommonEvent.playEvent('use_seed_maker');
  }
}

class YarnMaker extends MachineObject {
  getSpriteClass() {
    return Sprite_Machine;
  }

  resultingItemId() {
    if (!Managers.Items.isHoldingItem()) return '';

    const itemId = Managers.Items.selectedItem.id;
    return itemId.replace('wool', 'yarn');
  }

  isHoldingValidItem() {
    if (!Managers.Items.isHoldingItem()) return false;

    const itemId = Managers.Items.selectedItem.id;
    if (!itemId.includes('wool')) return false;

    return true;
  }

  playNoRoomEvent() {
    Managers.CommonEvent.playEvent('check_yarn_maker_no_room');
  }

  playProductionEvent() {
    if (this.isHoldingValidItem()) {
      Variables.yarnId = Managers.Items.selectedItem.id.replace('wool', 'yarn');
    } else {
      Variables.yarnId = 'yarn';
    }

    Managers.CommonEvent.playEvent('use_yarn_maker');
  }
}

const machines = [
  'cheese-maker',
  'mayo-maker',
  'hay-maker',
  'feed-maker',
  'seed-maker',
  'yarn-maker'
];

class MachinesContent {
  static contentId() {
    return 'machines';
  }

  static isFarmObjectLarge(farmObjectData) {
    if (farmObjectData.modelName == 'hay-maker') return true;
    if (farmObjectData.modelName == 'yarn-maker') return true;
    if (farmObjectData.modelName == 'feed-maker') return true;
  }

  static checkSpecialItemDrop(targetX, targetY) {
    if (!Managers.Items.selectedItem) return;

    const itemId = Managers.Items.selectedItem.id;

    if (machines.indexOf(itemId) >= 0) {
      return Managers.Items.useSelectedItem(targetX, targetY);
    }
  }

  static getObjectClassType(object) {
    if (object.modelName == 'cheese-maker') {
      return CheeseMaker;
    }

    if (object.modelName == 'mayo-maker') {
      return MayoMaker;
    }

    if (object.modelName == 'hay-maker') {
      return HayMaker;
    }

    if (object.modelName == 'feed-maker') {
      return FeedMaker;
    }

    if (object.modelName == 'seed-maker') {
      return SeedMaker;
    }

    if (object.modelName == 'yarn-maker') {
      return YarnMaker;
    }
  }

  static hasAnythingToRun(event, farmObjectData, activatedByMouse) {
    if (machines.indexOf(farmObjectData.modelName) >= 0) {
      return true;
    }
  }

  static touchObject(farmObjectData) {
    if (machines.indexOf(farmObjectData.modelName) >= 0) {
      Managers.FarmObjects.touchFence(farmObjectData, farmObjectData.modelName);
      return true;
    }
  }

  static pickBoughtItem(item) {
    switch(item.id) {
      case 'cheese-maker':
        Managers.Villagers.playDialogue('Viktor', 'trigger_cheese_maker_craft');
        return true;
      case 'mayo-maker':
        Managers.Villagers.playDialogue('Viktor', 'trigger_mayo_maker_craft');
        return true;
      case 'hay-maker':
        Managers.Villagers.playDialogue('Viktor', 'trigger_hay_maker_craft');
        return true;
      case 'feed-maker':
        Managers.Villagers.playDialogue('Viktor', 'trigger_feed_maker_craft');
        return true;
      case 'seed-maker':
        Managers.Villagers.playDialogue('Viktor', 'trigger_seed_maker_craft');
        return true;
      case 'yarn-maker':
        Managers.Villagers.playDialogue('Viktor', 'trigger_yarn_maker_craft');
        return true;
    }
  }

  static makeBlacksmithProductList(container) {
    container.gainItemId('cheese-maker');
    container.gainItemId('mayo-maker');
    container.gainItemId('hay-maker');
    container.gainItemId('feed-maker');
    container.gainItemId('seed-maker');
    container.gainItemId('yarn-maker');
  }

  // static parsePluginCommand(parser, command, args) {
  //   if (command.toUpperCase() !== 'MACHINE') {
  //     return;
  //   }
  // }
}

class MachineTool extends Tools.BaseTool {
  static doToolEffect() {
    const tile = TileHelper.getSmartFrontTile(1);
    const farmObjects = Managers.FarmObjects.getFarmObjectsXy($gameMap._mapId, tile.x, tile.y);

    for (let i = 0; i < farmObjects.length; i++) {
      const farmObject = farmObjects[i];
      if (!farmObject) {
        continue;
      }

      if (farmObject.isCropOrTree) return;
      if (farmObject.hasCollision()) {
        return;
      }
    }

    if (Managers.FarmObjects.isTileProtected($gameMap._mapId, tile.x, tile.y)) return;
    //Allow machine objects on any tile even if they aren't soil
    if (!TileHelper.checkIfTileCanGetObject(tile.x, tile.y, true, this.toolName, false, false)) {
      return;
    }

    if ($gamePlayer) {
      $gamePlayer.delayItemSprite();
      $gamePlayer.doToolDelayAnimation(this.toolName);
    }

    for (let i = 0; i < farmObjects.length; i++) {
      const replacedFarmObject = farmObjects[i];
      if (!replacedFarmObject) continue;
      if (replacedFarmObject.tilled) {
        replacedFarmObject.tilled = false;
        replacedFarmObject.updateEvents(true, false);
      }
    }

    const machine = Managers.FarmObjects.createFarmObjectAt($gameMap._mapId, tile.x, tile.y, this.toolName);
    Managers.Sound.playDropItem();
    machine.updateEvents(true, false);
    Managers.Items.loseItemId(this.toolName);
    $gamePlayer.leaveTile(tile.x, tile.y);
  }

  static use(level, targetTile = null) {
    this.doToolEffect(level);
  }

  static getAffectedTiles(level, includeInvalid) {
    return [];
  }
}

Tools.CheeseMaker = class CheeseMakerTool extends MachineTool {
  static get toolName() {
    return 'cheese-maker';
  }
};

Tools.MayoMaker = class MayoMakerTool extends MachineTool {
  static get toolName() {
    return 'mayo-maker';
  }
};

Tools.HayMaker = class HayMakerTool extends MachineTool {
  static get toolName() {
    return 'hay-maker';
  }
};

Tools.FeedMaker = class FeedMakerTool extends MachineTool {
  static get toolName() {
    return 'feed-maker';
  }
};

Tools.SeedMaker = class SeedMakerTool extends MachineTool {
  static get toolName() {
    return 'seed-maker';
  }
};

Tools.YarnMaker = class YarnMakerTool extends MachineTool {
  static get toolName() {
    return 'yarn-maker';
  }
};

Managers.Tools.registerTool(Tools.CheeseMaker);
Managers.Tools.registerTool(Tools.MayoMaker);
Managers.Tools.registerTool(Tools.HayMaker);
Managers.Tools.registerTool(Tools.FeedMaker);
Managers.Tools.registerTool(Tools.SeedMaker);
Managers.Tools.registerTool(Tools.YarnMaker);
Managers.Content.registerContentClass(MachinesContent);