require('game/managers/Content');
require('game/managers/Festivals');
require('game/core/Constants');

class FlowerFestivalContent {
  static contentId() {
    return 'festival-flower';
  }

  // FestivalManager.startDay
  static startDay() {
    this.clearVariables();
  }

  static clearVariables() {
    Variables.flowerFestivalFirstPlace = '';
    Variables.flowerFestivalSecondPlace = '';
    Variables.flowerFestivalThirdPlace = '';
    Variables.flowerFestivalFourthPlace = '';
    Variables.flowerFestivalFifthPlace = '';
    Variables.flowerFestivalSixthPlace = '';
    Variables.flowerFestivalSeventhPlace = '';

    Variables.flowerFestivalViktorScore = 0;
    Variables.flowerFestivalCindyScore = 0;
    Variables.flowerFestivalBrittanyScore = 0;
    Variables.flowerFestivalNathaliaScore = 0;
    Variables.flowerFestivalDevinScore = 0;
    Variables.flowerFestivalBillyScore = 0;
    Variables.flowerFestivalScore = 0;

    Variables.flowerFestivalWinner = '';
    Switches.playerWonFlowerFestival = false;
    Switches.playerGotSecondPlaceOnFlowerFestival = false;
    Switches.playerGotThirdPlaceOnFlowerFestival = false;
  }

  // ContentManager.getLetterList
  static getLetterList() {
    if (Managers.Time.yearDay !== Festivals.FLOWER + 1) return [];

    if (Switches.playerWonFlowerFestival) {
      return ['flower_festival_first_prize_letter'];
    }

    if (Switches.playerGotSecondPlaceOnFlowerFestival) {
      return ['flower_festival_second_prize_letter'];
    }

    if (Switches.playerGotThirdPlaceOnFlowerFestival) {
      return ['flower_festival_third_prize_letter'];
    }

    return [];
  }

  // ContentManager.setupMapEvents
  static setupMapEvents(firstOfTheDay) {
    if (!Switches.isFlowerFestival) return;

    if ($gameMap._mapId == Maps.SPRING_FLOWERS) {
      this.setupCompetitionMapEvents();
      return;
    }

    if ($gameMap._mapId !== Maps.FESTIVAL_RECEPTION) return;

    // if the festival is over, don't generate the custom events

    if (Managers.Time.hour >= 18 && !Switches.isInsideFestival) return;

    if (Managers.Time.hour < 10) {
      Managers.Festivals.leaveUnpreparedFestivalArea();
      return;
    }

    if (!Switches.isInsideFestival) {
      Managers.Festivals.enterFestival();
    }

    this.createFestivalEvents();
  }

  static setupCompetitionMapEvents() {
    Managers.Villagers.killAllVillagerEvents();

    $gameMap.createVillagerByNameAt('Amanda', 17, 30, Direction.DOWN, true);

    $gameMap.createVillagerByNameAt('Brittany', 14, 33, Direction.UP, true, Objects.FlowerFestivalVillager);
    $gameMap.createVillagerByNameAt('Billy', 15, 33, Direction.UP, true, Objects.FlowerFestivalVillager);
    $gameMap.createVillagerByNameAt('Devin', 16, 33, Direction.UP, true, Objects.FlowerFestivalVillager);
    $gameMap.createVillagerByNameAt('Nathalia', 18, 33, Direction.UP, true, Objects.FlowerFestivalVillager);
    $gameMap.createVillagerByNameAt('Cindy', 19, 33, Direction.UP, true, Objects.FlowerFestivalVillager);
    $gameMap.createVillagerByNameAt('Viktor', 20, 33, Direction.UP, true, Objects.FlowerFestivalVillager);

    Managers.Villagers.assignEvent('Amanda', 'amanda_flower_festival_competition');
    Managers.Villagers.assignEvent('Brittany', 'brittany_flower_festival_competition');
    Managers.Villagers.assignEvent('Billy', 'billy_flower_festival_competition');
    Managers.Villagers.assignEvent('Devin', 'devin_flower_festival_competition');
    Managers.Villagers.assignEvent('Nathalia', 'nathalia_flower_festival_competition');
    Managers.Villagers.assignEvent('Cindy', 'cindy_flower_festival_competition');
    Managers.Villagers.assignEvent('Viktor', 'viktor_flower_festival_competition');

  }

  static makeBouquetEventsAtRandom() {
    for (let event of $gameMap.events()) {
      if (!event) continue;

      if (event.iconIndex() > 0) {
        event.erase();
        continue;
      }
    }

    const positions = [];
    for (let x = 1; x < $gameMap.width() - 1; x++) {
      for (let y = 1; y < $gameMap.height() - 1; y++) {
        const regionId = $gameMap.regionId(x, y);
        if (regionId === Region.PLACEHOLDER) {
          positions.push({x, y});
        }
      }
    }

    const getRandomPosition = () => {
      if (!positions.length) {
        return {x: 38, y: 24};
      }

      const idx = Math.randomInt(positions.length);
      if (!positions[idx]) {
        return getRandomPosition();
      }

      const pos = positions[idx];
      positions.splice(idx, 1);
      return pos;
    };

    const makeNewBouquet = (itemId) => {
      const pos = getRandomPosition();

      const iconIndex = Managers.Items.getItemIcon(itemId);
      const eventName = `flower_festival_${ itemId.replace('-', '_') }`;
      $gameMap.createActionIconEventAt(iconIndex, pos.x, pos.y, eventName);
    };

    if (Switches.colorblindMode) {
      switch (Variables.flowerFestivalStep) {
        case 0: makeNewBouquet('red-bouquet'); break;
        case 1: makeNewBouquet('yellow-bouquet'); break;
        case 2: makeNewBouquet('white-bouquet'); break;
        case 3: makeNewBouquet('purple-bouquet'); break;
        case 4: makeNewBouquet('pink-bouquet'); break;
        case 5: makeNewBouquet('blue-bouquet'); break;
      }
    } else {
      makeNewBouquet('red-bouquet');
      makeNewBouquet('yellow-bouquet');
      makeNewBouquet('white-bouquet');
      makeNewBouquet('purple-bouquet');
      makeNewBouquet('pink-bouquet');
      makeNewBouquet('blue-bouquet');
    }
  }

  // FlowerFestival.createVillagerEvents
  static createVillagerEvents() {
    Managers.Villagers.killAllVillagerEvents();

    $gameMap.createVillagerByNameAt('Amanda', 18, 12, Direction.DOWN);
    $gameMap.createVillagerByNameAt('Julia', 20, 12, Direction.DOWN);

    $gameMap.createVillagerByNameAt('Brittany', 36, 15, Direction.LEFT);

    $gameMap.createVillagerByNameAt('Nathalia', 17, 17, Direction.UP);
    $gameMap.createVillagerByNameAt('Devin', 18, 17, Direction.UP);

    if (Managers.Relationship.isCharacterKnown('Annie')) {
      $gameMap.createVillagerByNameAt('Annie', 3, 20, Direction.RIGHT);
    }

    $gameMap.createVillagerByNameAt('Benjamin', 28, 12, Direction.DOWN);
    $gameMap.createVillagerByNameAt('Raphael', 29, 12, Direction.DOWN);

    $gameMap.createVillagerByNameAt('Viktor', 19, 21, Direction.UP);

    $gameMap.createVillagerByNameAt('Chloe', 35, 21, Direction.LEFT);
    $gameMap.createVillagerByNameAt('Serge', 35, 23, Direction.LEFT);

    $gameMap.createVillagerByNameAt('Gabriel', 13, 28, Direction.LEFT);
    $gameMap.createVillagerByNameAt('Lucas', 21, 33, Direction.UP);

    $gameMap.createVillagerByNameAt('Billy', 9, 13, Direction.DOWN);

    $gameMap.createVillagerByNameAt('Cindy', 24, 29, Direction.RIGHT);
    $gameMap.createVillagerByNameAt('Bonnie', 24, 28, Direction.RIGHT);

    if (!Switches.isFestivalEnding) {
      $gameMap.createVillagerByNameAt('Ilda', 12, 28, Direction.RIGHT);
      $gameMap.createVillagerByNameAt('Stella', 31, 10, Direction.DOWN);
      $gameMap.createVillagerByNameAt('Phi', 26, 28, Direction.LEFT);
      $gameMap.createVillagerByNameAt('Richard', 20, 21, Direction.UP);
    }

    const suffix = Switches.isFestivalEnding ? '_ending' : '';

    Managers.Villagers.assignEvent('Brittany', 'brittany_flower_festival' + suffix);
    Managers.Villagers.assignEvent('Benjamin', 'benjamin_flower_festival' + suffix);
    Managers.Villagers.assignEvent('Raphael', 'raphael_flower_festival' + suffix);
    Managers.Villagers.assignEvent('Julia', 'julia_flower_festival' + suffix);
    Managers.Villagers.assignEvent('Viktor', 'viktor_flower_festival' + suffix);
    Managers.Villagers.assignEvent('Richard', 'richard_flower_festival' + suffix);
    Managers.Villagers.assignEvent('Serge', 'serge_flower_festival' + suffix);
    Managers.Villagers.assignEvent('Chloe', 'chloe_flower_festival' + suffix);
    Managers.Villagers.assignEvent('Gabriel', 'gabriel_flower_festival' + suffix);
    Managers.Villagers.assignEvent('Lucas', 'lucas_flower_festival' + suffix);
    Managers.Villagers.assignEvent('Stella', 'stella_flower_festival' + suffix);
    Managers.Villagers.assignEvent('Nathalia', 'nathalia_flower_festival' + suffix);
    Managers.Villagers.assignEvent('Devin', 'devin_flower_festival' + suffix);
    Managers.Villagers.assignEvent('Bonnie', 'bonnie_flower_festival' + suffix);
    Managers.Villagers.assignEvent('Cindy', 'cindy_flower_festival' + suffix);
    Managers.Villagers.assignEvent('Billy', 'billy_flower_festival' + suffix);
    Managers.Villagers.assignEvent('Amanda', 'amanda_flower_festival' + suffix);
    Managers.Villagers.assignEvent('Ilda', 'ilda_flower_festival' + suffix);

    if (Managers.Relationship.isCharacterKnown('Annie')) {
      Managers.Villagers.assignEvent('Annie', 'annie_flower_festival' + suffix);
    }

    if (!Switches.isFestivalEnding) {
      Managers.Villagers.assignEvent('Phi', 'phi_flower_festival' + suffix);
    }
  }

  // FlowerFestival.createFlowerEvents
  static createFlowerEvents() {
    $gameMap.createStaticEventAt('festivals/$festival-flowers', 0, 12, 13, 2, 0, 1, true);
    $gameMap.createStaticEventAt('festivals/$festival-flowers', 0, 26, 13, 2, 1, 1, true);
    $gameMap.createStaticEventAt('festivals/$festival-flowers', 0, 27, 26, 2, 2, 1, true);
    $gameMap.createStaticEventAt('festivals/$festival-flowers', 0, 23, 26, 2, 0, 1, true);
    $gameMap.createStaticEventAt('festivals/$festival-flowers', 0, 15, 26, 2, 1, 1, true);
    $gameMap.createStaticEventAt('festivals/$festival-flowers', 0, 11, 26, 2, 2, 1, true);
    
    $gameMap.createStaticEventAt('festivals/$festival_tables', 0, 7, 20, 2, 0, 1, true);
    $gameMap.createStaticEventAt('festivals/$festival_tables', 0, 7, 25, 2, 1, 1, true);
    $gameMap.createStaticEventAt('festivals/$festival_tables', 0, 31, 20, 2, 2, 1, true);
    $gameMap.createStaticEventAt('festivals/$festival_tables', 0, 31, 25, 2, 0, 1, true);
  }

  // FlowerFestival.createFestivalEvents
  static createFestivalEvents() {
    this.createVillagerEvents();
    this.createFlowerEvents();

    $gameSystem.resetMap(Maps.SPRING_FLOWERS, true);
  }

  // ContentManager.checkCutscenes
  static checkCutscenes(mapId) {
    if (Switches.amandaToldAboutFlowerFestival) return false;

    if (mapId !== Maps.FARM) return false;
    if (!$gameMap.isHome()) return false;
    if (Managers.Time.yearDay !== Festivals.FLOWER) return false;
    if (Managers.Time.hour < 6) return false;
    if (Managers.Time.hour >= 18) return false;

    $gameTemp.reserveCutscene('amanda_tell_about_flower_festival');
    return true;
  }

  static isFestivalPrepared() {
    if (Managers.Time.yearDay == Festivals.FLOWER) {
      return Managers.Time.hour >= 10;
    }
  }

  // ContentManager.playCustomTvChannel
  static playCustomTvChannel(playFn) {
    if (Managers.Time.yearDay == Festivals.FLOWER - 1) {
      playFn('flower');
      return true;
    }
  }

  // ContentManager.getTvChannelFrameLength
  static getTvChannelFrameLength(channelName) {
    if (channelName == 'flower') {
      return 28;
    }
  }

  // ContentManager.getPurchaseDiscount
  static getPurchaseDiscount(itemId) {
    if (!Switches.isFlowerFestival) return;
    if (!Switches.isInsideFestival) return;

    const data = Managers.Items.getItemData(itemId);
    if (!data) return;
    if (data.type == 'seed-box') {
      if (data.id == 'jasmine-seed-box' || data.id == 'pink-daisy-seed-box') {
        return 15;
      }

      if (data.id == 'daisy-seed-box') {
        return 10;
      }

      return 5;
    }
  }

  static increaseFriendship() {
    // Increase friendship with every one who is in the festival
    Managers.Relationship.increaseFriendshipForFestival('Brittany');
    Managers.Relationship.increaseFriendshipForFestival('Benjamin');
    Managers.Relationship.increaseFriendshipForFestival('Raphael');
    Managers.Relationship.increaseFriendshipForFestival('Julia');
    Managers.Relationship.increaseFriendshipForFestival('Viktor');
    Managers.Relationship.increaseFriendshipForFestival('Richard');
    Managers.Relationship.increaseFriendshipForFestival('Serge');
    Managers.Relationship.increaseFriendshipForFestival('Chloe');
    Managers.Relationship.increaseFriendshipForFestival('Gabriel');
    Managers.Relationship.increaseFriendshipForFestival('Lucas');
    Managers.Relationship.increaseFriendshipForFestival('Stella');
    Managers.Relationship.increaseFriendshipForFestival('Nathalia');
    Managers.Relationship.increaseFriendshipForFestival('Devin');
    Managers.Relationship.increaseFriendshipForFestival('Bonnie');
    Managers.Relationship.increaseFriendshipForFestival('Cindy');
    Managers.Relationship.increaseFriendshipForFestival('Billy');
    Managers.Relationship.increaseFriendshipForFestival('Amanda');
    Managers.Relationship.increaseFriendshipForFestival('Ilda');
    Managers.Relationship.increaseFriendshipForFestival('Annie');
    Managers.Relationship.increaseFriendshipForFestival('Phi');
  }

  static startFlowerFestival() {
    this.increaseFriendship();

    Variables.flowerFestivalScore = 0;
    Variables.flowerFestivalStep = 0;

    this.startNextStep();

    const names = ['Brittany', 'Billy', 'Devin', 'Cindy', 'Nathalia', 'Viktor'];
    for (let name of names) {
      const event = Managers.Villagers.getVillagerData(name);
      if (event) {
        event.allowMovementOnFrozenTime = true;
        event._moveType = 1;
        event._moveSpeed = 3;
        event._moveFrequency = 5;
      }
    }
  }

  // ContentManager.parsePluginCommand
  static parsePluginCommand(parser, command, args) {
    if (command.toUpperCase() !== 'FLOWERFESTIVAL') {
      return;
    }

    switch(parser.name.toUpperCase()) {
      case 'START':
        this.startFlowerFestival();
        return true;
      case 'OPENSHOP':
        Managers.Scenes.push(GameScenes.FlowerShop);
        return true;
      case 'OPENSNACKSHOP':
        Managers.Scenes.push(GameScenes.FlowerFestivalSnackShop);
        return true;
      case 'CHECKBOUQUET':
        this.checkBouquet();
        return true;
      case 'NEXTSTEP':
        this.startNextStep();
        return true;
      case 'ANNOUNCEWINNERS':
        this.announceWinners();
        return true;
      case 'REPLACEBOUQUET':
        this.replaceBouquet(args);
        return true;
      case 'BACKTOSQUARE':
        this.goBackToSquare();
        return true;
      case 'PAUSE':
        this.pauseScore();
        return true;
      case 'RESUME':
        this.continueScore();
        return true;
    }
  }

  static _getEventByIconIndex(iconIndex) {
    for (let event of $gameMap.events()) {
      if (!event) continue;

      if (event.iconIndex() == iconIndex) {
        return event;
      }
    }
  }

  static pauseScore() {
    this._pausedScore = Graphics.frameCount - this._currentStepFrameCount;
  }

  static continueScore() {
    this._currentStepFrameCount = Graphics.frameCount - this._pausedScore;
  }

  static replaceBouquet(args) {
    const itemId = args.pop();

    const itemIcon = Managers.Items.getItemIcon(itemId);
    if (itemIcon) {
      const event = this._getEventByIconIndex(itemIcon);
      if (event) {
        if (Managers.Items.displayItemId && Managers.Items.displayItemId != 'none') {
          const oldItemIcon = Managers.Items.getItemIcon(Managers.Items.displayItemId);
          const oldItemEvent = `flower_festival_${ Managers.Items.displayItemId.replace('-', '_') }`;
          $gameMap.createActionIconEventAt(oldItemIcon, event.x, event.y, oldItemEvent);
        }

        event.erase();
      }
    }
    
    Managers.Sound.playPickItem();
    Managers.Items.displayItemId = itemId;
  }

  static announceWinners() {
    this.setupCompetitionMapEvents();

    const extraScore = Math.min(Managers.Time.year, 5) * 4;

    Variables.flowerFestivalViktorScore = Math.floor(100 + Math.randomInt(10) + extraScore);
    Variables.flowerFestivalCindyScore = Math.floor(95 + Math.randomInt(13) + extraScore);
    Variables.flowerFestivalBrittanyScore = Math.floor(95 + Math.randomInt(7) + extraScore);
    Variables.flowerFestivalNathaliaScore = Math.floor(90 + Math.randomInt(13) + extraScore);
    Variables.flowerFestivalDevinScore = Math.floor(75 + Math.randomInt(25) + extraScore);
    Variables.flowerFestivalBillyScore = Math.floor(60 + Math.randomInt(35) + extraScore);

    const participants = [{
      name : 'player',
      score : Variables.flowerFestivalScore,
      displayName: Managers.Player.getName()
    }, {
      name : 'Viktor',
      score : Variables.flowerFestivalViktorScore,
      displayName: t('Viktor')
    }, {
      name : 'Brittany',
      score : Variables.flowerFestivalBrittanyScore,
      displayName: t('Brittany')
    }, {
      name : 'Nathalia',
      score : Variables.flowerFestivalNathaliaScore,
      displayName: t('Nathalia')
    }, {
      name : 'Devin',
      score : Variables.flowerFestivalDevinScore,
      displayName: t('Devin')
    }, {
      name : 'Cindy',
      score : Variables.flowerFestivalCindyScore,
      displayName: t('Cindy')
    }, {
      name : 'Billy',
      score : Variables.flowerFestivalBillyScore,
      displayName: t('Billy')
    }];

    participants.sort(function(item1, item2){ return item2.score - item1.score; });

    Variables.flowerFestivalFirstPlace = participants[0].displayName;
    Variables.flowerFestivalSecondPlace = participants[1].displayName;
    Variables.flowerFestivalThirdPlace = participants[2].displayName;
    Variables.flowerFestivalFourthPlace = participants[3].displayName;
    Variables.flowerFestivalFifthPlace = participants[4].displayName;
    Variables.flowerFestivalSixthPlace = participants[5].displayName;
    Variables.flowerFestivalSeventhPlace = participants[6].displayName;

    Variables.flowerFestivalViktorScore = Utils.padNumber(Variables.flowerFestivalViktorScore);
    Variables.flowerFestivalCindyScore = Utils.padNumber(Variables.flowerFestivalCindyScore);
    Variables.flowerFestivalBrittanyScore = Utils.padNumber(Variables.flowerFestivalBrittanyScore);
    Variables.flowerFestivalNathaliaScore = Utils.padNumber(Variables.flowerFestivalNathaliaScore);
    Variables.flowerFestivalDevinScore = Utils.padNumber(Variables.flowerFestivalDevinScore);
    Variables.flowerFestivalBillyScore = Utils.padNumber(Variables.flowerFestivalBillyScore);
    Variables.flowerFestivalScore = Utils.padNumber(Variables.flowerFestivalScore);

    Variables.flowerFestivalWinner = participants[0].name;
    Switches.playerWonFlowerFestival = participants[0].name == 'player';
    Switches.playerGotSecondPlaceOnFlowerFestival = participants[1].name == 'player';
    Switches.playerGotThirdPlaceOnFlowerFestival = participants[2].name == 'player';

    if (Switches.playerWonFlowerFestival) {
      Variables.flowerFestivalWins += 1;
    }

    Managers.CommonEvent.playEvent('flower_festival_announce_winnters');
  }

  static startNextStep() {
    this.makeBouquetEventsAtRandom();

    Variables.flowerFestivalStep++;
    if (Variables.flowerFestivalStep <= 6) {
      const eventName = `flower_festival_step_${ Variables.flowerFestivalStep }`;
      this._currentStepFrameCount = Graphics.frameCount;
      Managers.CommonEvent.playEvent(eventName);
    }
  }

  static checkBouquet() {
    if (!Managers.Items.displayItemId) {
      Managers.CommonEvent.playEvent('flower_festival_amanda_no_bouquet');
      return;
    }

    const iconIndex = Managers.Items.getItemIcon(Managers.Items.displayItemId);
    if (!iconIndex) {
      Managers.CommonEvent.playEvent('flower_festival_amanda_no_bouquet');
      return;
    }

    const amandaEvent = Managers.Villagers.getVillagerData('Amanda');
    if (!amandaEvent) {
      return;
    }

    if (amandaEvent._itemIconBalloon == iconIndex) {
      const diff = Graphics.frameCount - this._currentStepFrameCount;
      const maxTime = 2900 + (125 * Math.min(Managers.Time.year, 5));

      const score = Math.max(0, Math.floor(maxTime - diff) / 100);
      Variables.flowerFestivalScore = Variables.flowerFestivalScore + score;

      Managers.CommonEvent.playEvent('flower_festival_amanda_right_bouquet');
    } else {
      Managers.CommonEvent.playEvent('flower_festival_amanda_wrong_bouquet');
    }
  }

  static goBackToSquare() {
    Managers.Time.hour = 18;
    Managers.Time.updateTime();
    $gameMap.autoplay();

    Switches.isFestivalEnding = true;
    Switches.isInsideFestivalCompetition = false;
    Managers.Items.displayItemId = undefined;

    $gamePlayer.reserveTransfer(Maps.FESTIVAL_RECEPTION, 19, 33, Direction.DOWN, 0, false, true);
  }
}

Managers.Content.registerContentClass(FlowerFestivalContent);
Managers.Festivals.registerFestival('flower', Festivals.FLOWER, FlowerFestivalContent, 'isFlowerFestival', 233, "Flower Festival");