require('game/managers/Content');

class AutoTasksContent {
  static contentId() {
    return 'auto-tasks';
  }

  static processNewDay() {
    Managers.Tasks.clearCustomTasks();

    // Only generate requests after 10 in game days
    if (Managers.Time.totalDays < 10) return;

    for (let i = 0; i < this.autoTasks.length; i++) {
      const data = this.autoTasks[i];
      if (this.checkIfTaskShouldActivate(data.chance)) {
        data.generatorFn.call(this);
      }
    }

    Managers.Tasks.updateVariables();
  }

  static checkIfTaskShouldActivate(averageDays) {
    const chance = 100 / averageDays;
    const randomValue = Math.randomInt(100);
    return randomValue < chance;
  }

  static getCustomDatabaseSummaryLines() {
    return [{
      weight: 4,
      title: t("Requests Fulfilled"),
      dataFn: function() {
        let totalRequests = 0;
        let completedRequests = 0;

        for (let i = 0; i < AutoTasksContent.autoTasks.length; i++) {
          const data = AutoTasksContent.autoTasks[i];
          if (!data) continue;

          totalRequests++;
          if (Managers.Tasks.getTaskStatus(data.id) == TaskStatus.COMPLETED) {
            completedRequests++;
          }
        }

        return [completedRequests, totalRequests];
      }
    }];
  }

  static getRandomItemId(possibleItems) {
    return possibleItems[Math.randomInt(possibleItems.length)];
  }

  static getRandomAvailableItemId(filterFn) {
    const possibleItems = [];
    const len = $dataItems.length;

    for (let i = 0; i < len; i++) {
      if (!$dataItems[i]) continue;

      const itemData = $dataItems[i];
      if (!itemData.useOnRequests) continue;
      if (!itemData.taskPrice) continue;

      if (filterFn && !filterFn(itemData)) continue;

      // Only ask for items that the player has shipped at least once
      if (Managers.History.getShippedItems(itemData.id) > 0) {
        possibleItems.push(itemData.id);
      }
    }

    return this.getRandomItemId(possibleItems);
  }

  static getItemRequest(itemId) {
    const request = [{
      type : 'item',
      amount : 1,
      id : itemId
    }];

    return request;
  }

  static makeReward(villagerName, friendshipAmount, goldAmount) {
    const reward = [{
      type: 'friendship',
      amount: friendshipAmount,
      villager: villagerName
    },{
      type : 'gold',
      amount : goldAmount
    }];

    return reward;
  }

  static registerTask(taskId, chance, taskGeneratorFn) {
    AutoTasksContent.autoTasks.push({
      id: taskId,
      chance: chance,
      generatorFn: taskGeneratorFn
    });
  }
}

AutoTasksContent.autoTasks = [];

AutoTasksContent.registerTask('viktor-terrible-gift-taste', 30, function() {
  if (!Managers.Relationship.isCharacterKnown('Viktor')) return;

  var taskId = 'viktor-terrible-gift-taste';
  var title = t("Picking a Gift");

  var itemId = this.getRandomAvailableItemId();
  if (!itemId) return;

  var request = this.getItemRequest(itemId);
  var description = "Viktor wants one <color:important> <var:viktorTerribleGiftItemName> </color> to give to Richard.";

  var rewardPrice = Managers.Items.getTaskPrice(itemId);

  var reward = this.makeReward('Viktor', 50, rewardPrice);

  Variables.viktorTerribleGiftItemId = itemId;
  Variables.viktorTerribleGiftItemName = Managers.Text.item(itemId);
  Variables.viktorTerribleGiftReward = rewardPrice;

  var taskData = {
    id: 'viktor-terrible-gift-taste',
    villagerPicture: 'menu/villagers/viktor',
    villagerName: 'Viktor',
    title : title,
    requester: 'Viktor',
    description : description,
    request : request,
    reward : reward,
    requestEvent: 'viktor-terrible-gift-taste-request',
    town: 'Orange',
    autoComplete : {
      villager: 'Viktor',
      event: 'viktor-terrible-gift-taste-ready'
    }
  };

  Managers.Tasks.addCustomTaskObject(taskId, taskData);
});

AutoTasksContent.registerTask('phi-weird-tree', 30, function() {
  if (!Managers.Relationship.isCharacterKnown('Phi')) return;

  var taskId = 'phi-weird-tree';
  var title = t('Phi Wants a Tree');

  var itemId = this.getRandomAvailableItemId();
  if (!itemId) return;
  var request = this.getItemRequest(itemId);
  var description = "Phi wants to plant a <color:important><var:phiWeirdTreeItemName></color> tree.";

  var rewardPrice = Managers.Items.getTaskPrice(itemId);

  var reward = this.makeReward('Phi', 50, rewardPrice);

  Variables.phiWeirdTreeItemId = itemId;
  Variables.phiWeirdTreeItemName = Managers.Text.item(itemId);
  Variables.phiWeirdTreeReward = rewardPrice;

  var taskData = {
    id: 'phi-weird-tree',
    title : title,
    requester : 'Phi',
    villagerPicture: 'menu/villagers/phi',
    villagerName: 'Phi',
    description: description,
    request : request,
    reward : reward,
    requestEvent : 'phi-weird-tree-request',
    town: 'Orange',
    autoComplete: {
      villager: 'Phi',
      event: 'phi-weird-tree-ready'
    }
  };

  Managers.Tasks.addCustomTaskObject(taskId, taskData);
});

AutoTasksContent.registerTask('annie-experiment', 45, function() {
  if (!Managers.Relationship.isCharacterKnown('Annie')) return;

  var taskId = 'annie-experiment';
  var title = t("Annie's Experiment");

  var request = [{
    type : 'item',
    amount : 1,
    id : 'chocolate'
  },{
    type : 'item',
    amount : 1,
    id : 'onion'
  },{
    type : 'item',
    amount : 1,
    id : 'sugarcane'
  }];
  var description = "Annie needs <color:important>1 Chocolate, 1 Onion and 1 Sugarcane</color> for some kind of experiment.";

  var reward = this.makeReward('Annie', 100, 400);

  var taskData = {
    id: 'annie-experiment',
    villagerPicture: 'menu/villagers/annie',
    villagerName: 'Annie',
    title : title,
    requester : 'Annie',
    description: description,
    request : request,
    reward : reward,
    requestEvent : 'annie-experiment-request',
    town: 'Orange',
    autoComplete: {
      villager: 'Annie',
      event: 'annie-experiment-ready'
    }
  };

  Managers.Tasks.addCustomTaskObject(taskId, taskData);
});

AutoTasksContent.registerTask('bonnie-treatment-help', 30, function() {
  if (!Managers.Relationship.isCharacterKnown('Bonnie')) return;

  var taskId = 'bonnie-treatment-help';
  var title = t("A Patient Needs Help");

  var itemId = this.getRandomAvailableItemId();
  if (!itemId) return;
  var request = this.getItemRequest(itemId);
  var description = "Bonnie needs one <color:important><var:bonnieTreatmentItemName></color> to complete a patient's treatment.";

  var rewardPrice = Managers.Items.getTaskPrice(itemId);

  var reward = this.makeReward('Bonnie', 50, rewardPrice);

  Variables.bonnieTreatmentItemId = itemId;
  Variables.bonnieTreatmentItemName = Managers.Text.item(itemId);
  Variables.bonnieTreatmentReward = rewardPrice;

  var taskData = {
    id: 'bonnie-treatment-help',
    villagerPicture: 'menu/villagers/bonnie',
    villagerName: 'Bonnie',
    title : title,
    requester : 'Bonnie',
    description: description,
    request : request,
    reward : reward,
    requestEvent : 'bonnie-treatment-help-request',
    town: 'Orange',
    autoComplete: {
      villager: 'Bonnie',
      event: 'bonnie-treatment-help-ready'
    }
  };

  Managers.Tasks.addCustomTaskObject(taskId, taskData);
});

AutoTasksContent.registerTask('benjamin-fertilizer-ingredients', 30, function() {
  if (!Managers.Relationship.isCharacterKnown('Benjamin')) return;

  var taskId = 'benjamin-fertilizer-ingredients';
  var title = t("Fertilizer Ingredients");

  var itemId = this.getRandomAvailableItemId();
  if (!itemId) return;
  var request = this.getItemRequest(itemId);
  var description = "Benjamin is trying to make a new kind of fertilizer. He asked for one <color:important><var:benjaminFertilizerItemName></color> to finish his formula.";

  var rewardPrice = Managers.Items.getTaskPrice(itemId);

  var reward = this.makeReward('Benjamin', 50, rewardPrice);

  Variables.benjaminFertilizerItemId = itemId;
  Variables.benjaminFertilizerItemName = Managers.Text.item(itemId);
  Variables.benjaminFertilizerReward = rewardPrice;

  var taskData = {
    id: 'benjamin-fertilizer-ingredients',
    villagerPicture: 'menu/villagers/benjamin',
    villagerName: 'Benjamin',
    title : title,
    requester : 'Benjamin',
    description: description,
    request : request,
    reward : reward,
    requestEvent : 'benjamin-fertilizer-ingredients-request',
    town: 'Orange',
    autoComplete: {
      villager: 'Benjamin',
      event: 'benjamin-fertilizer-ingredients-ready'
    }
  };

  Managers.Tasks.addCustomTaskObject(taskId, taskData);
});

AutoTasksContent.registerTask('chloe-saci-defense', 30, function() {
  if (!Managers.Relationship.isCharacterKnown('Chloe')) return;

  var taskId = 'chloe-saci-defense';
  var title = t("Chloe's Defence");

  var itemId = this.getRandomAvailableItemId();
  if (!itemId) return;
  var request = this.getItemRequest(itemId);
  var description = "Chloe wants one <color:important><var:chloeSaciDefenseItemName></color> for self-defense in case the Saci shows up.";

  var rewardPrice = Managers.Items.getTaskPrice(itemId);

  var reward = this.makeReward('Chloe', 50, rewardPrice);

  Variables.chloeSaciDefenseItemId = itemId;
  Variables.chloeSaciDefenseItemName = Managers.Text.item(itemId);
  Variables.chloeSaciDefenseReward = rewardPrice;

  var taskData = {
    id: 'chloe-saci-defense',
    villagerPicture: 'menu/villagers/chloe',
    villagerName: 'Chloe',
    title : title,
    requester : 'Chloe',
    description: description,
    request : request,
    reward : reward,
    requestEvent : 'chloe-saci-defense-request',
    town: 'Orange',
    autoComplete: {
      villager: 'Chloe',
      event: 'chloe-saci-defense-ready'
    }
  };

  Managers.Tasks.addCustomTaskObject(taskId, taskData);
});

AutoTasksContent.registerTask('devin-broken-furniture', 30, function() {
  if (!Managers.Relationship.isCharacterKnown('Devin')) return;

  var taskId = 'devin-broken-furniture';
  var title = t("Broken Furniture");

  var itemId = this.getRandomAvailableItemId();
  if (!itemId) return;
  var request = this.getItemRequest(itemId);
  var description = "Devin needs one <color:important><var:devinBrokenFurnitureItemName></color> to fix some piece of furniture in his room.";

  var rewardPrice = Managers.Items.getTaskPrice(itemId);
  var reward = this.makeReward('Devin', 50, rewardPrice);

  Variables.devinBrokenFurnitureItemId = itemId;
  Variables.devinBrokenFurnitureItemName = Managers.Text.item(itemId);
  Variables.devinBrokenFurnitureReward = rewardPrice;

  var taskData = {
    id: 'devin-broken-furniture',
    villagerPicture: 'menu/villagers/devin',
    villagerName: 'Devin',
    title : title,
    requester : 'Devin',
    description: description,
    request : request,
    reward : reward,
    requestEvent : 'devin-broken-furniture-request',
    town: 'Orange',
    autoComplete: {
      villager: 'Devin',
      event: 'devin-broken-furniture-ready'
    }
  };

  Managers.Tasks.addCustomTaskObject(taskId, taskData);
});

AutoTasksContent.registerTask('karl-ultimate-cake-recipe', 30, function() {
  if (!Managers.Relationship.isCharacterKnown('Karl')) return;

  var taskId = 'karl-ultimate-cake-recipe';
  var title = t("Karl's Ultimate Cake");

  var itemId = this.getRandomAvailableItemId(function(itemData){
    if (itemData.id == 'cake') return false;
    return true;
  });

  if (!itemId) return;

  var request = this.getItemRequest(itemId);
  var description = "Karl needs one <color:important><var:karlCakeIngredientName></color> to make the ultimate cake recipe.";
  var rewardPrice = Managers.Items.getTaskPrice(itemId);
  var reward = this.makeReward('Karl', 50, rewardPrice);

  Variables.karlCakeIngredientItemId = itemId;
  Variables.karlCakeIngredientName = Managers.Text.item(itemId);
  Variables.karlCakeIngredientReward = rewardPrice;

  var taskData = {
    id: 'karl-ultimate-cake-recipe',
    villagerPicture: 'menu/villagers/karl',
    villagerName: 'Karl',
    title: title,
    requester: 'Karl',
    description: description,
    request: request,
    reward: reward,
    requestEvent: 'karl-ultimate-cake-recipe-request',
    town: 'Orange',
    autoComplete: {
      villager: 'Karl',
      event: 'karl-ultimate-cake-recipe-ready'
    }
  };

  Managers.Tasks.addCustomTaskObject(taskId, taskData);
});

AutoTasksContent.registerTask('richard-request', 30, function() {
  if (!Managers.Relationship.isCharacterKnown('Richard')) return;

  const taskId = 'richard-request';
  const title = t("Richard's Request");

  const itemId = this.getRandomAvailableItemId();
  if (!itemId) return;

  const request = this.getItemRequest(itemId);
  const description = "Richard wants a new <color:important><var:richardRequestName></color> because his last one was stolen.";
  const rewardPrice = Managers.Items.getTaskPrice(itemId);
  const reward = this.makeReward('Richard', 50, rewardPrice);

  Variables.richardRequestItemId = itemId;
  Variables.richardRequestName = Managers.Text.item(itemId);
  Variables.richardRequestReward = rewardPrice;

  const taskData = {
    id: taskId,
    villagerPicture: 'menu/villagers/richard',
    villagerName: 'Richard',
    title,
    requester: 'Richard',
    description,
    request,
    reward,
    requestEvent: 'richard-request',
    town: 'Orange',
    autoComplete: {
      villager: 'Richard',
      event: 'richard-request-ready'
    }
  };

  Managers.Tasks.addCustomTaskObject(taskId, taskData);
});

AutoTasksContent.registerTask('gabriel-request', 50, function() {
  if (!Managers.Relationship.isCharacterKnown('Gabriel')) return;
  if (Managers.Time.weekDay < Days.THURSDAY || Managers.Time.weekDay > Days.FRIDAY) return false;

  const taskId = 'gabriel-request';
  const title = t("Gabriel's Request");

  const itemId = this.getRandomAvailableItemId();
  if (!itemId) return;

  const request = this.getItemRequest(itemId);
  const description = "Gabriel needs one <color:important><var:gabrielRequestName></color> for tomorrow's show. He will pay me with exposure.";
  
  const reward = [
    { type: 'friendship', amount: 100, villager: 'Gabriel' },
    { type: 'friendship', amount: 100, villager: 'Amanda' },
    { type: 'friendship', amount: 100, villager: 'Annie' },
    { type: 'friendship', amount: 100, villager: 'Benjamin' },
    { type: 'friendship', amount: 100, villager: 'Billy' },
    { type: 'friendship', amount: 100, villager: 'Bonnie' },
    { type: 'friendship', amount: 100, villager: 'Brittany' },
    { type: 'friendship', amount: 100, villager: 'Chloe' },
    { type: 'friendship', amount: 100, villager: 'Cindy' },
    { type: 'friendship', amount: 100, villager: 'Devin' },
    { type: 'friendship', amount: 100, villager: 'Julia' },
    { type: 'friendship', amount: 100, villager: 'Karl' },
    { type: 'friendship', amount: 100, villager: 'Lucas' },
    { type: 'friendship', amount: 100, villager: 'Nathalia' },
    { type: 'friendship', amount: 100, villager: 'Phi' },
    { type: 'friendship', amount: 100, villager: 'Raphael' },
    { type: 'friendship', amount: 100, villager: 'Richard' },
    { type: 'friendship', amount: 100, villager: 'Serge' },
    { type: 'friendship', amount: 100, villager: 'Stella' },
    { type: 'friendship', amount: 100, villager: 'Viktor' }
  ];

  Variables.gabrielRequestItemId = itemId;
  Variables.gabrielRequestName = Managers.Text.item(itemId);

  const taskData = {
    id: taskId,
    villagerPicture: 'menu/villagers/gabriel',
    villagerName: 'Gabriel',
    title,
    requester: 'Gabriel',
    description,
    request,
    reward,
    requestEvent: 'gabriel-request',
    town: 'Orange',
    autoComplete: {
      villager: 'Gabriel',
      event: 'gabriel-request-ready'
    }
  };

  Managers.Tasks.addCustomTaskObject(taskId, taskData);
});

AutoTasksContent.registerTask('julia-request', 30, function() {
  if (!Managers.Relationship.isCharacterKnown('Julia')) return;

  const taskId = 'julia-request';
  const title = t("Julia's Request");

  const itemId = this.getRandomAvailableItemId();
  if (!itemId) return;

  const request = this.getItemRequest(itemId);
  const description = "Julia urgently needs a <color:important><var:juliaRequestName></color> for the future of Orange Town.";
  const rewardPrice = Managers.Items.getTaskPrice(itemId);
  const reward = this.makeReward('Julia', 50, rewardPrice);

  Variables.juliaRequestItemId = itemId;
  Variables.juliaRequestName = Managers.Text.item(itemId);
  Variables.juliaRequestReward = rewardPrice;

  const taskData = {
    id: taskId,
    villagerPicture: 'menu/villagers/julia',
    villagerName: 'Julia',
    title,
    requester: 'Julia',
    description,
    request,
    reward,
    requestEvent: 'julia-request',
    town: 'Orange',
    autoComplete: {
      villager: 'Julia',
      event: 'julia-request-ready'
    }
  };

  Managers.Tasks.addCustomTaskObject(taskId, taskData);
});

AutoTasksContent.registerTask('billy-request', 30, function() {
  if (!Managers.Relationship.isCharacterKnown('Billy')) return;

  const taskId = 'billy-request';
  const title = t("Billy's Request");

  const itemId = this.getRandomAvailableItemId();
  if (!itemId) return;

  const request = this.getItemRequest(itemId);
  const description = "Billy needs one <color:important><var:billyRequestName></color> to save his skin.";
  const rewardPrice = Managers.Items.getTaskPrice(itemId);
  const reward = this.makeReward('Billy', 50, rewardPrice);

  Variables.billyRequestItemId = itemId;
  Variables.billyRequestName = Managers.Text.item(itemId);
  Variables.billyRequestReward = rewardPrice;

  const taskData = {
    id: taskId,
    villagerPicture: 'menu/villagers/billy',
    villagerName: 'Billy',
    title,
    requester: 'Billy',
    description,
    request,
    reward,
    requestEvent: 'billy-request',
    town: 'Orange',
    autoComplete: {
      villager: 'Billy',
      event: 'billy-request-ready'
    }
  };

  Managers.Tasks.addCustomTaskObject(taskId, taskData);
});

AutoTasksContent.registerTask('brittany-request', 30, function() {
  if (!Managers.Relationship.isCharacterKnown('Brittany')) return;

  const taskId = 'brittany-request';
  const title = t("Brittany's Request");

  const itemId = this.getRandomAvailableItemId();
  if (!itemId) return;

  const request = this.getItemRequest(itemId);
  const description = "Brittany is looking for one <color:important><var:brittanyRequestName></color> for her farm.";
  const rewardPrice = Managers.Items.getTaskPrice(itemId);
  const reward = this.makeReward('Brittany', 50, rewardPrice);

  Variables.brittanyRequestItemId = itemId;
  Variables.brittanyRequestName = Managers.Text.item(itemId);
  Variables.brittanyRequestReward = rewardPrice;

  const taskData = {
    id: taskId,
    villagerPicture: 'menu/villagers/brittany',
    villagerName: 'Brittany',
    title,
    requester: 'Brittany',
    description,
    request,
    reward,
    requestEvent: 'brittany-request',
    town: 'Orange',
    autoComplete: {
      villager: 'Brittany',
      event: 'brittany-request-ready'
    }
  };

  Managers.Tasks.addCustomTaskObject(taskId, taskData);
});

AutoTasksContent.registerTask('cindy-request', 30, function() {
  if (!Managers.Relationship.isCharacterKnown('Cindy')) return;
  if (!Switches.cindyMovedOut) return;

  const taskId = 'cindy-request';
  const title = t("Cindy's Request");

  const itemId = this.getRandomAvailableItemId();
  if (!itemId) return;

  const request = this.getItemRequest(itemId);
  const description = "Cindy needs one <color:important><var:cindyRequestName></color> for her studies.";
  const rewardPrice = Managers.Items.getTaskPrice(itemId);
  const reward = this.makeReward('Cindy', 50, rewardPrice);

  Variables.cindyRequestItemId = itemId;
  Variables.cindyRequestName = Managers.Text.item(itemId);
  Variables.cindyRequestReward = rewardPrice;

  const taskData = {
    id: taskId,
    villagerPicture: 'menu/villagers/cindy',
    villagerName: 'Cindy',
    title,
    requester: 'Cindy',
    description,
    request,
    reward,
    requestEvent: 'cindy-request',
    town: 'Orange',
    autoComplete: {
      villager: 'Cindy',
      event: 'cindy-request-ready'
    }
  };

  Managers.Tasks.addCustomTaskObject(taskId, taskData);
});

AutoTasksContent.registerTask('serge-request', 30, function() {
  if (!Managers.Relationship.isCharacterKnown('Serge')) return;

  const taskId = 'serge-request';
  const title = t("Serge's Request");

  const itemId = this.getRandomAvailableItemId();
  if (!itemId) return;

  const request = this.getItemRequest(itemId);
  const description = "Serge needs to replace one <color:important><var:sergeRequestName></color> that is missing from his storage.";
  const rewardPrice = Managers.Items.getTaskPrice(itemId);
  const reward = this.makeReward('Serge', 50, rewardPrice);

  Variables.sergeRequestItemId = itemId;
  Variables.sergeRequestName = Managers.Text.item(itemId);
  Variables.sergeRequestReward = rewardPrice;

  const taskData = {
    id: taskId,
    villagerPicture: 'menu/villagers/serge',
    villagerName: 'Serge',
    title,
    requester: 'Serge',
    description,
    request,
    reward,
    requestEvent: 'serge-request',
    town: 'Orange',
    autoComplete: {
      villager: 'Serge',
      event: 'serge-request-ready'
    }
  };

  Managers.Tasks.addCustomTaskObject(taskId, taskData);
});

AutoTasksContent.registerTask('nathalia-request', 30, function() {
  if (!Managers.Relationship.isCharacterKnown('Nathalia')) return;

  const taskId = 'nathalia-request';
  const title = t("Nathalia's Request");

  const itemId = this.getRandomAvailableItemId();
  if (!itemId) return;

  const request = this.getItemRequest(itemId);
  const description = "Nathalia wants one <color:important><var:nathaliaRequestName></color> to check if it can fly.";
  const rewardPrice = Managers.Items.getTaskPrice(itemId);
  const reward = this.makeReward('Nathalia', 50, rewardPrice);

  Variables.nathaliaRequestItemId = itemId;
  Variables.nathaliaRequestName = Managers.Text.item(itemId);
  Variables.nathaliaRequestReward = rewardPrice;

  const taskData = {
    id: taskId,
    villagerPicture: 'menu/villagers/nathalia',
    villagerName: 'Nathalia',
    title,
    requester: 'Nathalia',
    description,
    request,
    reward,
    requestEvent: 'nathalia-request',
    town: 'Orange',
    autoComplete: {
      villager: 'Nathalia',
      event: 'nathalia-request-ready'
    }
  };

  Managers.Tasks.addCustomTaskObject(taskId, taskData);
});

AutoTasksContent.registerTask('raphael-request', 30, function() {
  if (!Managers.Relationship.isCharacterKnown('Raphael')) return;

  const taskId = 'raphael-request';
  const title = t("Raphael's Request");

  const itemId = this.getRandomAvailableItemId();
  if (!itemId) return;

  const request = this.getItemRequest(itemId);
  const description = "Raphael is asking for one <color:important><var:raphaelRequestName></color> for his work.";
  const rewardPrice = Managers.Items.getTaskPrice(itemId);
  const reward = this.makeReward('Raphael', 50, rewardPrice);

  Variables.raphaelRequestItemId = itemId;
  Variables.raphaelRequestName = Managers.Text.item(itemId);
  Variables.raphaelRequestReward = rewardPrice;

  const taskData = {
    id: taskId,
    villagerPicture: 'menu/villagers/raphael',
    villagerName: 'Raphael',
    title,
    requester: 'Raphael',
    description,
    request,
    reward,
    requestEvent: 'raphael-request',
    town: 'Orange',
    autoComplete: {
      villager: 'Raphael',
      event: 'raphael-request-ready'
    }
  };

  Managers.Tasks.addCustomTaskObject(taskId, taskData);
});

AutoTasksContent.registerTask('lucas-request', 30, function() {
  if (!Managers.Relationship.isCharacterKnown('Lucas')) return;

  const taskId = 'lucas-request';
  const title = t("Lucas's Request");

  const itemId = this.getRandomAvailableItemId();
  if (!itemId) return;

  const request = this.getItemRequest(itemId);
  const description = "Lucas is looking for one <color:important><var:lucasRequestName></color> for... reasons?";
  const rewardPrice = Managers.Items.getTaskPrice(itemId);
  const reward = this.makeReward('Lucas', 50, rewardPrice);

  Variables.lucasRequestItemId = itemId;
  Variables.lucasRequestName = Managers.Text.item(itemId);
  Variables.lucasRequestReward = rewardPrice;

  const taskData = {
    id: taskId,
    villagerPicture: 'menu/villagers/lucas',
    villagerName: 'Lucas',
    title,
    requester: 'Lucas',
    description,
    request,
    reward,
    requestEvent: 'lucas-request',
    town: 'Orange',
    autoComplete: {
      villager: 'Lucas',
      event: 'lucas-request-ready'
    }
  };

  Managers.Tasks.addCustomTaskObject(taskId, taskData);
});

AutoTasksContent.registerTask('amanda-request', 30, function() {
  if (!Managers.Relationship.isCharacterKnown('Amanda')) return;

  const taskId = 'amanda-request';
  const title = t("Amanda's Request");

  const itemId = this.getRandomAvailableItemId();
  if (!itemId) return;

  const request = this.getItemRequest(itemId);
  const description = "Amanda says her flowers are asking for one <color:important><var:amandaRequestName></color>.";
  const rewardPrice = Managers.Items.getTaskPrice(itemId);
  const reward = this.makeReward('Amanda', 50, rewardPrice);

  Variables.amandaRequestItemId = itemId;
  Variables.amandaRequestName = Managers.Text.item(itemId);
  Variables.amandaRequestReward = rewardPrice;

  const taskData = {
    id: taskId,
    villagerPicture: 'menu/villagers/amanda',
    villagerName: 'Amanda',
    title,
    requester: 'Amanda',
    description,
    request,
    reward,
    requestEvent: 'amanda-request',
    town: 'Orange',
    autoComplete: {
      villager: 'Amanda',
      event: 'amanda-request-ready'
    }
  };

  Managers.Tasks.addCustomTaskObject(taskId, taskData);
});


Managers.Content.registerContentClass(AutoTasksContent);