//=============================================================================
// main.js
//=============================================================================


const _log = console.log;
let logFileName;

try {
  if (typeof window.require == 'function') {
    const fs = require('fs');
    const date = new Date();
    // Remove the offset so that the .toISOString method will generate a string with the local date instead of UTC
    date.setMinutes(date.getMinutes() - date.getTimezoneOffset());

    logFileName = 'logs/' + (date.toISOString().substr(0, 10));
    let fileHandler;
    fs.open(logFileName, 'a', 0o666, (err, handler) => {
      if (err) {
        _log('Failed to create file handler for logging.');
      }
      fileHandler = handler;
    });

    window.log = (...args) => {
      const prefix = new Date().toLocaleTimeString();

      for (let line of args) {
        if (line === undefined || line === null) continue;

        if (typeof line != 'string') {
          if (!line.toString) continue;

          const obj = line;
          line = line.toString();
          if (line == '[object Object]') {
            try {
              line = JSON.stringify(obj);
            }
            catch(e) {
              continue;
            }
          }
        }

        fs.appendFile(fileHandler || logFileName, `\r\n${ prefix } | ${ line }`, () => {});
      }

      return args;
    };

    window.error = (...args) => {
      if (args.length > 0 && typeof args[0] == 'object') {
        const e = args[0];

        const logs = [];

        if (e.message) {
          logs.push('Message: ' + e.message);
        }
        if (e.filename) {
          logs.push('File: ' + e.filename);
        }
        if (e.lineno) {
          logs.push('Line: ' + e.lineno);
        }
        if (e.stack) {
          logs.push('Stack: ' + e.stack);
        }

        window.log(...logs);
      }
      return args;
    };

    window.warn = (...args) => {
      return window.log('WARNING', ...args);
    };
  }
} catch(e) {
  console.log('Failed to overwrite log method.');
  window.log = (...args) => args;
  window.error = (...args) => args;
  window.warn = (...args) => args;
}

window.onload = function() {
  console.log(...log('Starting boot process.'));

  // require('nw.gui').Window.get().showDevTools();

  OrangeSeason.registerRequiredFile('main.js');
  OrangeSeason.registerRequiredFile('OrangeSeason.js');

  try {
    OrangeSeason.include(Wrapper.joinFileNames('engine', 'libs', 'pixi.js'));
    OrangeSeason.includeFolder(Wrapper.joinFileNames('engine', 'libs'));

    OrangeSeason.includeFolder('engine');

    OrangeSeason.includeFolder('.', ['tests']);
    OrangeSeason.includeFolder(Wrapper.joinFileNames('..', 'packages'));

    OrangeSeason.validateWatchList();
  } catch(e) {
    console.error(...window.error(e));
    alert('Failed to load game files. Please contact the developer. Check log file for details.');
    return;
  }

  if (window.$dataSystem) {
    window.isDemo = $dataSystem.isDemoMode;
  }

  OrangeSeason.startGame();
};

window.openFeedbackForm = function(takeScreenshot = true) {
  const screenshotList = window.screenshotList || [];
  const screenshot = takeScreenshot ? Managers.Scenes.snap() : false;

  if (window.feedbackWindow) {
    window.feedbackWindow.show(true);
    window.feedbackWindow.focus();

    if (window.$gamePlayer) {
      window.feedbackWindow.pos = { x : $gamePlayer._realX, y : $gamePlayer._realY };
    }

    if (screenshot) {
      window.feedbackWindow.screenshot = screenshot._canvas.toDataURL();
      screenshotList.push(screenshot);
    }

    if (logFileName) {
      if (window.fs.existsSync(logFileName)) {
        const logText = window.fs.readFileSync(logFileName);
        window.feedbackWindow.logText = logText;
      }
    }

    if (window.$gameTemp && window.$gameTemp.getRealLoadedGamePath()) {
      const saveContent = window.fs.readFileSync(window.$gameTemp.getRealLoadedGamePath());
      window.feedbackWindow.saveContent = saveContent;
    }

    return;
  }

  if (!window.feedbackCount) {
    const globalInfo = Engine.Data.loadGlobalInfo();
    window.feedbackCount = globalInfo.feedbackCount || 0;
  }

  window.feedbackCount++;
  Engine.Data.updateGlobalInfo();

  const orangePath = Managers.Storage.localFileDirectoryPath(true);
  const feedbackPath = Wrapper.joinFileNames(orangePath, 'feedbacks');

  const url = window.location.pathname.replace('index', 'feedback');

  window.gui.Window.open(`${ url }?id=${ window.feedbackCount }`, {
    focus: true,
  }, (win) => {
    win.on('loaded', () => {
      window.feedbackWindow = win;
      window.screenshotList = screenshotList;
      win.feedbackPath = feedbackPath;

      win.resizeTo(1000, 800);

      if (screenshot) {
        if (!screenshotList.includes(screenshot)) {
          screenshotList.push(screenshot);
        }
      }

      if (screenshotList.length > 0) {
        if (screenshotList.length > 1) {
          let i = 1;
          let handler = setInterval(() => {
            if (i >= screenshotList.length) {
              clearInterval(handler);
              return;
            }

            win.screenshot = screenshotList[i]._canvas.toDataURL();
            i++;
          }, 500);
        }

        if (window.$gamePlayer) {
          window.feedbackWindow.pos = { x : $gamePlayer._realX, y : $gamePlayer._realY };
        }
        win.screenshot = screenshotList[0]._canvas.toDataURL();
      }

      if (logFileName) {
        if (window.fs.existsSync(logFileName)) {
          const logText = window.fs.readFileSync(logFileName);
          win.logText = logText;
        }
      }

      if (window.$gameTemp && window.$gameTemp.getRealLoadedGamePath()) {
        const saveContent = window.fs.readFileSync(window.$gameTemp.getRealLoadedGamePath());
        win.saveContent = saveContent;
      }
    });

    win.on('close', () => {
      if (window.feedbackWindow == win) {
        delete window.feedbackWindow;
        window.screenshotList = [];
      }

      win.close(true);
    });
  });
};

console.log(...log('main file loaded'));