const Window_Base = require('engine/windows/Window_Base');

const STAGE_BACKGROUND = 0;
const STAGE_LOGO = 1;
const STAGE_START = 2;
const STAGE_OPTIONS = 3;
const STAGE_CHANGELOG = 4;

const BUTTON_STATE_HIDDEN = 0;
const BUTTON_STATE_FADING = 1;
const BUTTON_STATE_VISIBLE = 2;

class Window_TitleOptions extends Window_Base {
  initialize(x, y, width, height) {
    super.initialize(x, y, width, height);

    this._newGameBtnOffBitmap = Managers.Images.loadPicture('title/newgame_off');
    this._newGameBtnOnBitmap = Managers.Images.loadPicture('title/newgame_on');
    this._saveBtnOffBitmap = Managers.Images.loadPicture('title/save_off');
    this._saveBtnOnBitmap = Managers.Images.loadPicture('title/save_on');
    this._saveBtnOnDisabledBitmap = Managers.Images.loadPicture('title/save_on_disabled');
    this._optionBtnOffBitmap = Managers.Images.loadPicture('title/option_off');
    this._optionBtnOnBitmap = Managers.Images.loadPicture('title/option_on');
    this._updatesBtnOffBitmap = Managers.Images.loadPicture('title/updates_off');
    this._updatesBtnOnBitmap = Managers.Images.loadPicture('title/updates_on');
    this._quitBtnOffBitmap = Managers.Images.loadPicture('title/quit_off');
    this._quitBtnOnBitmap = Managers.Images.loadPicture('title/quit_on');

    this._textBox1Bitmap = Managers.Images.loadPicture('title/button_text_1');
    this._textBox2Bitmap = Managers.Images.loadPicture('title/button_text_2');
    this._textBox3Bitmap = Managers.Images.loadPicture('title/button_text_3');
    this._earlyAccessBoxBitmap = Managers.Images.loadPicture('title/box_earlyaccess');

    this._changeLogbgBitmap = Managers.Images.loadBackground('changelog_layout');

    this._handlers = {};

    this._zoomLevel = Graphics.windowZoomLevel;
    this._selectedIndex = Engine.Data.anySaveExists() ? 1 : 0;
    this._selectionStage = 0;

    this._animationStage = STAGE_BACKGROUND;
    this._backgroundStage = 0;
    this._backgroundTimeout = Utils.getFrameCount(60);
    this._transferDelay = -1;
    this._optionFadingStage = 0;

    this._startAnimationStage = 0;

    this._logoY = 500;
    this._logoDelay = 0;

    this._animatingPressedOption = false;
  }

  setHandler(symbol, method) {
    this._handlers[symbol] = method;
  }

  isHandled(symbol) {
    return !!this._handlers[symbol];
  }

  animateHandler(symbol) {
    if (symbol == 'load') {
      if (!Engine.Data.anySaveExists()) {
        Managers.Sound.playBuzzer();
        return;
      }
    }

    this._animatingPressedOption = true;
    this.refresh();
    setTimeout(() => {
      this._animatingPressedOption = false;
      this.refresh();
      setTimeout(() => {
        this.callHandler(symbol);
      }, 30);
    }, 90);
  }

  callHandler(symbol) {
    if (this.isHandled(symbol)) {
      this._handlers[symbol]();
    }
  }

  enableWindowFrame() {
    return false;
  }

  standardFontSize() {
    return Utils.convertFontSize(26, Graphics.windowZoomLevel);
  }

  resetTextColor() {
    this.changeTextColor('#3d2b25');
  }

  resetFontSettings() {
    super.resetFontSettings();
    this.contents.outlineColor = '#f4dea8';
    this.contents.outlineWidth = 0;
  }

  lineHeight() {
    return this.standardFontSize();
  }

  refresh() {
    if (this.contents) {
      this.contents.clear();
      this.drawContent();
    }
  }

  drawBitmap(bitmap, originalX, originalY, zoom) {
    this.requireBitmap(bitmap);
    if (!bitmap.isReady()) return;

    const width = bitmap._canvas.width;
    const height = bitmap._canvas.height;
    const drawWidth = width * this._zoomLevel * (zoom || 1);
    const drawHeight = height * this._zoomLevel * (zoom || 1);

    let x = originalX * this._zoomLevel;
    let y = originalY * this._zoomLevel;

    if (zoom && zoom !== 1) {
      const widthDiff = (width - (width * zoom)) * this._zoomLevel;
      const heightDiff = (height - (height * zoom)) * this._zoomLevel;

      x += widthDiff / 2;
      y += heightDiff / 2;
    }

    this.contents.bltImage(bitmap, 0, 0, width, height, x, y, drawWidth, drawHeight);
  }

  drawOptions() {
    if (this._animationStage != STAGE_OPTIONS) return;

    let textBoxX = -1000;
    let text = '';
    let textOpacity = 1;
    let selectedX = 2000;
    let selectedY = 259;

    if (this._button1State !== undefined) {
      if (this._button1State >= BUTTON_STATE_VISIBLE) {
        if (this._selectedIndex === 0) {
          this.drawBitmap(this._newGameBtnOnBitmap, 129, 259);
          textBoxX = 122;
          text = t("New Game");

          selectedX = 129;
        } else {
          this.drawBitmap(this._newGameBtnOffBitmap, 132, 262);
        }
      } else if (this._button1State === BUTTON_STATE_FADING) {
        const fadeStage = this._optionFadingStage;
        const zoom = (0.1 * fadeStage).clamp(0, 1);
        this.drawBitmap(this._newGameBtnOffBitmap, 132, 262, zoom);
      }
    }

    if (this._button2State !== undefined) {
      const loadEnabled = Engine.Data.anySaveExists();

      if (this._button2State >= BUTTON_STATE_VISIBLE) {
        if (this._selectedIndex === 1) {
          this.drawBitmap(loadEnabled ? this._saveBtnOnBitmap : this._saveBtnOnDisabledBitmap, 206, 259);
          selectedX = 206;
          textBoxX = 199;
          text = t("Load Game");
          textOpacity = loadEnabled ? 1 : 0.6;
        } else {
          this.drawBitmap(this._saveBtnOffBitmap, 209, 262);
        }

      } else if (this._button2State === BUTTON_STATE_FADING) {
        const fadeStage = this._optionFadingStage - 8;
        const zoom = (0.1 * fadeStage).clamp(0, 1);
        this.drawBitmap(this._saveBtnOffBitmap, 209, 262, zoom);
      }

    }

    if (this._button3State !== undefined) {
      if (this._button3State >= BUTTON_STATE_VISIBLE) {
        if (this._selectedIndex === 2) {
          this.drawBitmap(this._optionBtnOnBitmap, 283, 259);
          selectedX = 283;
          textBoxX = 276;
          text = t("Options");
        } else {
          this.drawBitmap(this._optionBtnOffBitmap, 286, 262);
        }
      } else if (this._button3State === BUTTON_STATE_FADING) {
        const fadeStage = this._optionFadingStage - 16;
        const zoom = (0.1 * fadeStage).clamp(0, 1);
        this.drawBitmap(this._optionBtnOffBitmap, 286, 262, zoom);
      }
    }

    if (this._button4State !== undefined) {
      if (this._button4State >= BUTTON_STATE_VISIBLE) {
        if (this._selectedIndex === 3) {
          this.drawBitmap(this._updatesBtnOnBitmap, 361, 259);
          selectedX = 361;
          textBoxX = 354;
          text = t("Patch Notes");
        } else {
          this.drawBitmap(this._updatesBtnOffBitmap, 364, 262);
        }
      } else if (this._button4State === BUTTON_STATE_FADING) {
        const fadeStage = this._optionFadingStage - 24;
        const zoom = (0.1 * fadeStage).clamp(0, 1);
        this.drawBitmap(this._updatesBtnOffBitmap, 364, 262, zoom);
      }
    }

    if (this._button5State !== undefined) {
      if (this._button5State >= BUTTON_STATE_VISIBLE) {
        if (this._selectedIndex === 4) {
          this.drawBitmap(this._quitBtnOnBitmap, 439, 259);
          selectedX = 439;
          textBoxX = 432;
          text = t("Exit");
        } else {
          this.drawBitmap(this._quitBtnOffBitmap, 442, 262);
        }
      } else if (this._button5State === BUTTON_STATE_FADING) {
        const fadeStage = this._optionFadingStage - 32;
        const zoom = (0.1 * fadeStage).clamp(0, 1);
        this.drawBitmap(this._quitBtnOffBitmap, 442, 262, zoom);
      }
    }

    if (this._animatingPressedOption) {
      this.contents._context.globalAlpha = 0.8;
      this.contents.fillRect((selectedX + 5) * this._zoomLevel, (selectedY + 5) * this._zoomLevel, 62 * this._zoomLevel, 62 * this._zoomLevel, '#FFFFFF');
      this.contents._context.globalAlpha = 1;
    }

    if (textBoxX > 0) {
      const stage = Math.floor(this._selectionStage / 20).clamp(0, 3);
      switch(stage) {
        case 0:
          break;
        case 1:
          this.drawBitmap(this._textBox1Bitmap, textBoxX, 323);
          break;
        case 2:
          this.drawBitmap(this._textBox2Bitmap, textBoxX, 321);
          break;
        case 3:
          {
            this.drawBitmap(this._textBox3Bitmap, textBoxX, 319);

            const textX = (textBoxX + 11) * this._zoomLevel;
            const textY = 319 * this._zoomLevel;
            const textWidth = 64 * this._zoomLevel;
            this.changeTextColor('#d7a678');
            this.contents.outlineWidth = 0;
            this.contents.outlineColor = '#d7a678';

            this.contents._context.globalAlpha = textOpacity;
            this.drawText(text.toUpperCase(), textX, textY, textWidth, 'center');
            this.contents._context.globalAlpha = 1;
          }

          break;
      }
    }
  }

  drawStart() {
    if (this._animationStage != STAGE_START) return;

    const opacity = (this._startAnimationStage < 2) ? 0.6 : 1;

    const textX = 0 * this._zoomLevel;
    const textY = 310 * this._zoomLevel;
    const textWidth = 640 * this._zoomLevel;
    const text = t('Start');

    this.contents._context.globalAlpha = opacity;
    this.contents.fontSize = Utils.convertFontSize(120, Graphics.windowZoomLevel);
    this.changeTextColor('#f4dea8');
    this.contents.outlineWidth = 10;
    this.contents.outlineColor = '#d7a678';
    this.drawText(text.toUpperCase(), textX, textY, textWidth, 'center');

    this.contents._context.globalAlpha = 1;
    this.resetFontSettings();
  }

  drawEarlyAccessBox() {
    if (this._animationStage < STAGE_START) return;

    this.drawBitmap(this._earlyAccessBoxBitmap, 496, 10);

    const textX = 506 * this._zoomLevel;
    const textY = 12 * this._zoomLevel;
    const textWidth = 114 * this._zoomLevel;
    const text = t("Early Access");

    this.changeTextColor('#fb3153');
    this.contents.outlineWidth = 0;
    this.drawText(text.toUpperCase(), textX, textY, textWidth, 'center');
  }

  drawChangeLog() {
    if (this._animationStage != STAGE_CHANGELOG) return;

    const width = 351 * this._zoomLevel;
    const height = 336 * this._zoomLevel;

    const y = (Graphics.height - height) / this._zoomLevel;
    const x = (Graphics.width - width) / this._zoomLevel / 2;

    this.drawBitmap(this._changeLogbgBitmap, x, y, 1);
  }

  drawContent() {
    this.drawLogo();
    this.drawStart();
    this.drawOptions();
    this.drawEarlyAccessBox();
    this.drawChangeLog();
  }

  standardPadding() {
    return 0;
  }

  clearBackground() {
    const canvasHeight = this.contents.height;
    const canvasWidth = this.contents.width;

    this.contents.clear();
    this.contents.fillRect(0, 0, canvasWidth, canvasHeight, '#FFAB57');
  }

  updateBackground() {
    const edgeBitmap = Managers.Images.loadSystem('l2r-transition-edge');
    if (!$gameTemp._loadedTitleOnce) {
      if (this._backgroundTimeout > 0) {
        this._backgroundTimeout--;
        this.clearBackground();
        return;
      }
    }

    if (!edgeBitmap.isReady()) {
      this.clearBackground();
      return;
    }

    const speed = 10 * Graphics.windowZoomLevel;
    const edgeWidth = edgeBitmap.width;
    const edgeHeight = edgeBitmap.height;
    const canvasHeight = this.contents.height;
    const canvasWidth = this.contents.width;

    this.contents.clear();
    const width = Math.floor(canvasWidth - this._backgroundStage);
    this.contents.fillRect(0, 0, width, canvasHeight, '#FFAB57');

    if (this._backgroundStage <= canvasWidth + speed) {
      this.contents._context.globalCompositeOperation = 'source-over';
      this.contents._context.imageSmoothingEnabled = false;

      let y = 0;
      let x = Math.floor(canvasWidth - this._backgroundStage);

      while (y < canvasHeight) {
        this.contents._context.drawImage(edgeBitmap._image, 0, 0, edgeWidth, edgeHeight, x, y, edgeWidth, edgeHeight);
        y += edgeHeight;
      }

      this.contents._setDirty();
      this._backgroundStage += speed;

      if (TouchInput.isTriggered() || Engine.Input.isTriggered('ok') || $gameTemp._loadedTitleOnce) {
        this._backgroundStage = canvasWidth + speed + 1;
      }

    } else {
      this._transferDelay = Utils.getFrameCount(30);
    }
  }

  drawLogo() {
    const logoBitmap = Managers.Images.loadPicture('title/logo');
    if (!logoBitmap.isReady()) {
      return;
    }

    let y = this._logoY;
    if (y < 56) {
      y = 56 + (56 - y);
    }
    const logoX = 214;
    this.drawBitmap(logoBitmap, logoX, y);
  }

  updateLogo() {
    if (this._logoDelay > 0) {
      this._logoDelay--;
      return;
    }

    const logoMinY = 46;

    if (this._logoY > logoMinY) {
      this._logoY -= 6;

      if (TouchInput.isTriggered() || Engine.Input.isTriggered('ok')) {
        this._logoY = logoMinY;
      }

      if (this._logoY <= logoMinY) {
        this._logoDelay = Utils.getFrameCount(30);
      }

      this.refresh();
    } else {
      this._transferDelay = 1;
    }
  }

  updateStart() {
    this._startAnimationStage++;

    if (this._startAnimationStage > 30) {
      this._startAnimationStage = 0;
    }

    if (TouchInput.isTriggered() || Engine.Input.isAnyTriggered()) {
      this._transferDelay = 1;
    }

    this.refresh();
  }

  updateOptions() {
    if (this._optionFadingStage <= 44) {
      this._optionFadingStage += 1;

      if (TouchInput.isTriggered() || Engine.Input.isTriggered('ok')) {
        this._optionFadingStage = 45;
      }

      this._button1State = this._optionFadingStage > 12 ? BUTTON_STATE_VISIBLE : (this._optionFadingStage > 2 ? BUTTON_STATE_FADING : BUTTON_STATE_HIDDEN);
      this._button2State = this._optionFadingStage > 20 ? BUTTON_STATE_VISIBLE : (this._optionFadingStage > 10 ? BUTTON_STATE_FADING : BUTTON_STATE_HIDDEN);
      this._button3State = this._optionFadingStage > 28 ? BUTTON_STATE_VISIBLE : (this._optionFadingStage > 18 ? BUTTON_STATE_FADING : BUTTON_STATE_HIDDEN);
      this._button4State = this._optionFadingStage > 36 ? BUTTON_STATE_VISIBLE : (this._optionFadingStage > 26 ? BUTTON_STATE_FADING : BUTTON_STATE_HIDDEN);
      this._button5State = this._optionFadingStage > 44 ? BUTTON_STATE_VISIBLE : (this._optionFadingStage > 34 ? BUTTON_STATE_FADING : BUTTON_STATE_HIDDEN);

      this.refresh();
      return;
    }

    $gameTemp._loadedTitleOnce = true;

    if (Managers.Scenes.isSceneChanging()) {
      return;
    }

    let needsRefresh = false;
    if (this._selectionStage < 80) {
      const oldStage = Math.floor(this._selectionStage / 20);
      this._selectionStage += Utils.getFrameCount(4);
      if (Math.floor(this._selectionStage / 20) > oldStage) {
        needsRefresh = true;
      }
    }

    if (!this._animatingPressedOption) {
      if (Engine.Input.isRepeated('left')) {
        this._selectedIndex--;
        if (this._selectedIndex < 0) {
          this._selectedIndex = 4;
        }
        needsRefresh = true;
      } else if (Engine.Input.isRepeated('right')) {
        this._selectedIndex++;
        if (this._selectedIndex > 4) {
          this._selectedIndex = 0;
        }
        needsRefresh = true;
      } else if (Engine.Input.isTriggered('ok')) {
        const symbol = ['newGame', 'load', 'options', 'updates', 'exit'][this._selectedIndex];
        this.animateHandler(symbol);
      } else {
        const mouseX = Math.floor(TouchInput.mouseX / this._zoomLevel);
        const mouseY = Math.floor(TouchInput.mouseY / this._zoomLevel);

        if (mouseX != this._oldMouseX || mouseY != this._oldMouseY || TouchInput.isTriggered()) {
          if (mouseY >= 262 && mouseY <= 328 && mouseX > 132 && mouseX <= 508) {
            const index = Math.floor((mouseX - 127) / 78);
            if (index != this._selectedIndex) {
              this._selectedIndex = index;
              this._selectionStage = 0;
              needsRefresh = true;
            }

            if (TouchInput.isTriggered() && index >= 0 && index <= 4) {
              const symbol = ['newGame', 'load', 'options', 'updates', 'exit'][index];
              this._selectedIndex = index;
              this.animateHandler(symbol);
            }
          }

          this._oldMouseX = mouseX;
          this._oldMouseY = mouseY;
        }
      }
    }
    
    if (needsRefresh) {
      this.refresh();
    }
  }

  updateChangelog() {


  }

  update() {
    super.update();

    if (this._transferDelay >= 0) {
      this._transferDelay--;
      if (this._transferDelay < 0) {
        this._animationStage++;
        this.refresh();
      }
      return;
    } 

    switch (this._animationStage) {
      case STAGE_BACKGROUND:
        this.updateBackground();
        break;
      case STAGE_LOGO:
        this.updateLogo();
        break;
      case STAGE_START:
        this.updateStart();
        break;
      case STAGE_OPTIONS:
        this.updateOptions();
        break;
      case STAGE_CHANGELOG:
        this.updateChangelog();
        break;
    }
  }

  openChangeLog() {
    this._animationStage = STAGE_CHANGELOG;
    this._transferDelay = -1;
    this.refresh();
  }

  closeChangeLog() {
    this._animationStage = STAGE_OPTIONS;
    this._selectionStage = 0;
    this._transferDelay = -1;
  }

  canCheckInput() {
    return this._animationStage >= STAGE_OPTIONS;
  }
}

module.exports = Window_TitleOptions;