const Window_Selectable = require('engine/windows/Window_Selectable');

//-----------------------------------------------------------------------------
// ShopNumberSubWindow
//
// The window for inputting quantity of items to buy or sell on the shop
// screen.

class ShopNumberSubWindow extends Window_Selectable {
  initialize(x, y) {
    const width = this.windowWidth();
    const height = this.windowHeight();

    super.initialize(x, y, width, height);
    this._zoomLevel = Graphics.windowZoomLevel;

    this._item = null;
    this._max = 1;
    this._price = 0;
    this._number = 1;
    this._ingredients = [];
    this._currencyUnit = Managers.Text.currencyUnit;
    this._selectedDigit = 0;

    this.createButtons();
  }

  windowWidth() {
    return 154 * Graphics.windowZoomLevel;
  }

  windowHeight() {
    return 70 * Graphics.windowZoomLevel;
  }

  enableWindowFrame() {
    return false;
  }

  number() {
    return this._number;
  }

  topMargin() {
    return 0;
  }

  leftMargin() {
    return 0;
  }

  totalNumber() {
    return Math.round(this._number / this._unitMultiplier);
  }

  item() {
    return this._item;
  }

  setUnitMultiplierForItem(item) {
    if (item.type == 'seed-box') {
      this._unitMultiplier = 10;
    } else {
      this._unitMultiplier = 1;
    }
  }

  setup(item, max, price, ingredients) {
    this._item = item;
    this._max = Math.min(Math.floor(max), 999);
    this._price = price;
    this._ingredients = ingredients || [];

    this.setUnitMultiplierForItem(item);

    if (this._unitMultiplier > 10) {
      this._selectedDigit = 2;
    } else if (this._unitMultiplier > 1) {
      this._selectedDigit = 1;
    } else {
      this._selectedDigit = 0;
    }

    this._max = Math.min(this._max * this._unitMultiplier, Math.floor(999 / this._unitMultiplier) * this._unitMultiplier);
    this._number = this._unitMultiplier;

    this.placeButtons();
    this.updateButtonsVisibility();
    this.changeNumber(0);
    this.refresh();
  }

  unitMultiplier() {
    return this._unitMultiplier || 1;
  }

  setCurrencyUnit(currencyUnit) {
    this._currencyUnit = currencyUnit;
    this.refresh();
  }

  createArrowButtons() {
    const bitmap = Managers.Images.loadSystem('ShopButtonSet');
    let newBitmap;
    if (Graphics.windowZoomLevel == 2) {
      newBitmap = bitmap;
    } else {
      newBitmap = new Bitmap(96 * Graphics.windowZoomLevel / 2, 32 * Graphics.windowZoomLevel / 2);

      bitmap.addLoadListener(() => {
        newBitmap.bltBitmap(bitmap, 0, 0, bitmap.width, bitmap.height, 0, 0, newBitmap.width, newBitmap.height);
      });
    }

    const buttonWidth = 16 * Graphics.windowZoomLevel / 2;
    const buttonHeight = 16 * Graphics.windowZoomLevel / 2;

    for (let i = 0; i < 6; i++) {
      let buttonIndex = i % 2;

      const button = new Sprite_Button();
      const x = buttonWidth * buttonIndex;
      const w = buttonWidth;
      button.bitmap = newBitmap;
      button.setColdFrame(x, 0, w, buttonHeight);
      button.setHotFrame(x, buttonHeight, w, buttonHeight);
      button.visible = false;
      this._buttons.push(button);
      this.addChild(button);
    }

    this._buttons[2].setClickHandler(this.onButtonDown.bind(this));
    this._buttons[3].setClickHandler(this.onButtonUp.bind(this));
    this._buttons[4].setClickHandler(this.onButtonDown2.bind(this));
    this._buttons[5].setClickHandler(this.onButtonUp2.bind(this));
    this._buttons[6].setClickHandler(this.onButtonDown3.bind(this));
    this._buttons[7].setClickHandler(this.onButtonUp3.bind(this));
  }

  createButtons() {
    const bitmap = Managers.Images.loadSystem('ShopButtonSet');
    const newBitmap = new Bitmap(96 * Graphics.windowZoomLevel, 32 * Graphics.windowZoomLevel);

    bitmap.addLoadListener(() => {
      newBitmap.bltBitmap(bitmap, 0, 0, bitmap.width, bitmap.height, 0, 0, newBitmap.width, newBitmap.height);
    });

    const buttonWidth = 16 * Graphics.windowZoomLevel;
    const buttonHeight = 16 * Graphics.windowZoomLevel;
    this._buttons = [];

    for (let i = 0; i < 2; i++) {
      const button = new Sprite_Button();
      const x = buttonWidth * (i + 2);
      const w = buttonWidth;
      button.bitmap = newBitmap;
      button.setColdFrame(x, 0, w, buttonHeight);
      button.setHotFrame(x, buttonHeight, w, buttonHeight);
      button.visible = false;
      this._buttons.push(button);
      this.addChild(button);
    }

    this._buttons[0].setClickHandler(this.onButtonOk.bind(this));
    this._buttons[1].setClickHandler(this.onButtonCancel.bind(this));

    this.createArrowButtons();
  }

  placeButtons() {
    const z = Graphics.windowZoomLevel;
    const top = this.topMargin();
    const left = this.leftMargin();

    this._buttons[0].x = left + 100 * z;
    this._buttons[0].y = top  + 51 * z;
    this._buttons[1].x = left + 116 * z;
    this._buttons[1].y = top  + 51 * z;

    this._buttons[2].x = left + 63 * z;
    this._buttons[2].y = top  + 35 * z;
    this._buttons[3].x = left + 63 * z;
    this._buttons[3].y = top  + 15 * z;

    this._buttons[4].x = left + 53 * z;
    this._buttons[4].y = top  + 35 * z;
    this._buttons[5].x = left + 53 * z;
    this._buttons[5].y = top  + 15 * z;

    this._buttons[6].x = left + 43 * z;
    this._buttons[6].y = top  + 35 * z;
    this._buttons[7].x = left + 43 * z;
    this._buttons[7].y = top  + 15 * z;
  }

  updateButtonsVisibility() {
    // this.hideButtons();
    this.showButtons();
  }

  showButtons() {
    for (let i = 0; i < this._buttons.length; i++) {
      this._buttons[i].visible = true;
    }

    this._buttons[0].visible = true;
    this._buttons[1].visible = true;

    const unitMultiplier = this.unitMultiplier();

    this._buttons[2].visible = unitMultiplier <= 1;
    this._buttons[3].visible = unitMultiplier <= 1;

    this._buttons[4].visible = unitMultiplier <= 10;
    this._buttons[5].visible = unitMultiplier <= 10;

    this._buttons[6].visible = unitMultiplier <= 100;
    this._buttons[7].visible = unitMultiplier <= 100;
  }

  hideButtons() {
    for (let i = 0; i < this._buttons.length; i++) {
      this._buttons[i].visible = false;
    }
  }

  requestUpdate() {
    this._needsRefresh = true;
  }

  drawContentBackground() {

  }

  refresh() {
    this.contents.clear();
    this.drawContentBackground();

    this.drawSelectedItemIcon();
    this.drawNumber();
    this.drawTotalPrice();

    // if (this._ingredientsWindow) {
    //   this._ingredientsWindow.refresh();
    // }
  }

  drawSelectedItemName() {
    const top = this.topMargin();
    const left = this.leftMargin();

    const x = left + 5 * this._zoomLevel;
    const y = top + 4 * this._zoomLevel;
    const width = 75 * this._zoomLevel;

    this.drawItemName(this._item, x, y - this.contents.fontSize / 2, width, false, false, 'center');
  }

  getIconX() {
    const left = this.leftMargin();
    const x = left + 14 * this._zoomLevel;

    return x;
  }

  getIconY() {
    const top = this.topMargin();
    const y = top + 17 * this._zoomLevel;

    return y;
  }

  drawSelectedItemIcon() {
    const x = this.getIconX();
    const y = this.getIconY();

    this.drawIcon(this._item.iconIndex, x, y);
  }

  drawNumber() {
    const top = this.topMargin();
    const left = this.leftMargin();

    const digits = this._number.toString().split('');
    while (digits.length < 3) {
      digits.splice(0, 0, '0');
    }

    const y = top + 25 * Graphics.windowZoomLevel;
    const selectedNumber = 2 - (this._selectedDigit % 3);

    let digitIndex = 3;
    this.contents.fontSize = Utils.convertFontSize(30, Graphics.windowZoomLevel);

    for (let i = digits.length -1; i >= 0; i--) {
      const digit = digits[i];
      digitIndex--;

      const x = left + (43 + digitIndex * 10) * Graphics.windowZoomLevel;
      const w = 9 * Graphics.windowZoomLevel;
      this.drawText(digit, x, y - this.contents.fontSize / 2, w, 'center');

      if (digitIndex == selectedNumber) {
        this.contents.fillRect(x, top + 34 * Graphics.windowZoomLevel, 9 * Graphics.windowZoomLevel, 1 * Graphics.windowZoomLevel, this.blackColor());
      }

      if (digitIndex === 0) {
        break;
      }
    }

    this.contents.fontSize = this.standardFontSize();
  }

  drawTotalPrice() {
    const top = this.topMargin();
    const left = this.leftMargin();

    const number = this._number || 1;
    const total = this._price * number / this.unitMultiplier();

    const digits = total.toString().split('');
    const y = top + 34 * Graphics.windowZoomLevel;

    let digitIndex = 6;
    this.contents.fontSize = Utils.convertFontSize(28, Graphics.windowZoomLevel);

    if (digits.length < 5) {
      while (digits.length < 5) {
        digits.splice(0, 0, '');
      }
      if (digits.length < 6) {
        digits.splice(0, 0, '$');
      }
    }

    for (let i = digits.length -1; i >= 0; i--) {
      const digit = digits[i];
      digitIndex--;

      const x = left + (92 + digitIndex * 8) * Graphics.windowZoomLevel;
      const w = 7 * Graphics.windowZoomLevel;
      this.drawText(digit, x, y - this.contents.fontSize / 2, w, 'center');

      if (digitIndex === 0) {
        break;
      }
    }

    this.contents.fontSize = this.standardFontSize();
  }

  maxDigits() {
    return 3;
  }

  update() {
    if (this._needsRefresh) {
      this._needsRefresh = false;
      this.refresh();
    }

    super.update();
    this.processNumberChange();
  }

  isOkTriggered() {
    return Engine.Input.isTriggered('ok');
  }

  playOkSound() {
  }

  selectNextDigit() {
    const minUnit = this.unitMultiplier();
    if (minUnit > 10) {
      this._selectedDigit = 2;
    } else if (minUnit > 1) {
      this._selectedDigit = 1 + this._selectedDigit % 2;
    } else {
      this._selectedDigit = (this._selectedDigit + 2) % 3;
    }

    this.refresh();
  }

  selectPrevDigit() {
    const minUnit = this.unitMultiplier();
    if (minUnit > 10) {
      this._selectedDigit = 2;
    } else if (minUnit > 1) {
      this._selectedDigit = 1 + this._selectedDigit % 2;
    } else {
      this._selectedDigit = (this._selectedDigit + 1) % 3;
    }

    this.refresh();
  }

  processNumberChange() {
    if (!this.isOpenAndActive()) {
      return;
    }

    if (Engine.Input.isRepeated('right')) {
      this.selectNextDigit();
      return;
    }

    if (Engine.Input.isRepeated('left')) {
      this.selectPrevDigit();
      return;
    }

    if (Engine.Input.isRepeated('up')) {
      switch(this._selectedDigit) {
        case 0:
          this.changeNumber(1);
          break;
        case 1:
          this.changeNumber(10);
          break;
        case 2:
          this.changeNumber(100);
          break;
      }
    }

    if (Engine.Input.isRepeated('down')) {
      switch(this._selectedDigit) {
        case 0:
          this.changeNumber(-1);
          break;
        case 1:
          this.changeNumber(-10);
          break;
        case 2:
          this.changeNumber(-100);
          break;
      }
    }
  }

  changeNumber(amount) {
    const lastNumber = this._number;
    const minUnit = this.unitMultiplier();

    this._number = (this._number + amount).clamp(minUnit, this._max);

    if (this._number !== lastNumber) {
      Managers.Sound.playCursor();
      this.refresh();
    }
  }

  updateCursor() {
    if (!this.visible) return;

  }

  onButtonUp() {
    this.changeNumber(1);
  }

  onButtonUp2() {
    this.changeNumber(10);
  }

  onButtonUp3() {
    this.changeNumber(100);
  }

  onButtonDown() {
    this.changeNumber(-1);
  }

  onButtonDown2() {
    this.changeNumber(-10);
  }

  onButtonDown3() {
    this.changeNumber(-100);
  }

  onButtonOk() {
    this.processOk();
  }

  onButtonCancel() {
    super.processCancel();
  }

  standardFontSize() {
    return Utils.convertFontSize(22, Graphics.windowZoomLevel);
  }

  scrollDown() {
    this.changeNumber(-1);
  }

  scrollUp() {
    this.changeNumber(1);
  }

  setIngredientsWindow(ingredientsWindow) {
    this._ingredientsWindow = ingredientsWindow;
  }

  standardOutlineWidth() {
    return 0;
  }

  normalColor() {
    return this.blackColor();
  }

  standardPadding() {
    return 0;
  }

  textPadding() {
    return 15 * Graphics.windowZoomLevel;
  }
}

module.exports = ShopNumberSubWindow;
