const Window_ModernItemContainer = require('../containers/ModernItemContainer');

//-----------------------------------------------------------------------------
// Window_ShopItemList
//
// The window for selecting an item on the shop screen.

class Window_ShopItemList extends Window_ModernItemContainer {
  initialize(itemContainer) {
    this._zoomLevel = Graphics.windowZoomLevel;

    const width = 271 * this._zoomLevel;
    const height = 185 * this._zoomLevel;
    const x = (Graphics.width - width) / 2;
    const y = (Graphics.height - height) / 2;

    super.initialize(x, y, width, height, itemContainer, itemContainer.maxItemTypes());
  }

  enableCursorSprite() {
    return false;
  }

  isCurrentItemEnabled() {
    return this.isEnabled(this.item());
  }

  includes(item) {
    return true;
  }

  isEnabled(item) {
    // Removed this check to allow buying items even with the full backpack (item is sent to the mailbox)

    // if (!Managers.Items.isHoldingItem(item.id)) {
    //   if (!Managers.Items.canGainItemId(item.id)) return false;
    // }

    if (!Managers.Items.canBuyItem(item.id)) return false;

    return true;
  }

  setInfoWindow(infoWindow) {
    this._infoWindow = infoWindow;
  }

  updateInfoWindow() {
    if (!this._infoWindow) return;

    this._infoWindow.setItem(this.item());
  }

  select(index) {
    super.select(index);
    this.updateInfoWindow();
  }

  maxCols() {
    return 7;
  }

  itemWidth() {
    return 24 * this._zoomLevel;
  }

  itemHeight() {
    return 24 * this._zoomLevel;
  }

  itemRect(index) {
    const rect = this.originalItemRect(index);
    const cols = this.maxCols();
    const topRow = this.topRow();
    const topIndex = cols * topRow;
    const relativeIndex = index - topIndex;

    if (relativeIndex >= cols) {
      rect.y += Math.floor(relativeIndex / cols) * 7 * this._zoomLevel;
    }

    if ((relativeIndex % cols) > 0) {
      rect.x += (relativeIndex % cols) * 7 * this._zoomLevel;
    }

    return rect;
  }

  drawItem(index) {
    const item = this.getItem(index);
    if (!item) {
      return;
    }

    const rect = this.itemRect(index);
    const areaColor = this.darkBaseColor();

    if (this._itemBuying) {
      if (this._itemBuying.id == item.id) {
        this.changePaintOpacity(true);
      } else {
        this.contents.paintOpacity = 30;
      }
    } else if (index == this._index) {
      this.changePaintOpacity(true);
    } else {
      this.contents.paintOpacity = 60;
    }

    this.contents.fillRect(rect.x, rect.y, rect.width, rect.height, areaColor);
    this.drawIcon(item.iconIndex, rect.x, rect.y);

    const price = Managers.Items.getDiscountedBuyPrice(item);
    const priceStr = Managers.Text.formatMoney(price);
    this.changeTextColor(this.blackColor());
    this.contents.outlineWidth = 4;
    this.contents.outlineColor = this.baseColor();

    const priceY = this.calcTextY(rect.y + rect.height);
    this.drawText(priceStr, rect.x, priceY, rect.width, 'center');

    this.changePaintOpacity(true);
  }

  drawExtraContent() {
    const money = Managers.Text.formatMoney(Managers.Items.gold);
    this.contents.fontSize = Utils.convertFontSize(26, this._zoomLevel);
    this.contents.outlineWidth = 0;
    this.changeTextColor(this.blackColor);

    this.drawText(money, 20 * this._zoomLevel, 18 * this._zoomLevel - (this.contents.fontSize / 2), 50 * this._zoomLevel, 'right');
  }

  standardFontSize() {
    return Utils.convertFontSize(22, this._zoomLevel);
  }

  drawSelectedItem() {
  }

  standardPadding() {
    return 0;
  }

  spacing() {
    return 0;
  }

  standardOutlineWidth() {
    return 0;
  }

  normalColor() {
    return this.blackColor();
  }

  backgroundBitmap() {
    if (this._itemBuying) {
      return Managers.Images.loadBackground('layout_shop_open');
    }

    return Managers.Images.loadBackground('layout_shop_closed');
  }

  drawWindowBackground() {
    const bitmap = this.backgroundBitmap();

    if (!bitmap.isReady()) {
      this.requireBitmap(bitmap);
      return;
    }

    this.drawBitmap(bitmap, 0, 0, this._zoomLevel);
  }

  topMargin() {
    return 47 * this._zoomLevel;
  }

  bottomMargin() {
    return 83 * this._zoomLevel;
  }

  horizontalMargin() {
    return 30 * this._zoomLevel;
  }

  drawArrows() {
    if (this._itemBuying) {
      return;
    }

    const zoom = Graphics.windowZoomLevel;
    const x = (this.width / 2) - 5.5 * zoom;

    if (this.maxTopRow() > 0) {
      this.drawPicture('menu/arrow_up', x, 37 * zoom, zoom);
      this.drawPicture('menu/arrow_down', x, 103 * zoom, zoom);
    }
  }

  updateArrowTouch(mouseX, mouseY) {
    if (this._itemBuying) {
      return;
    }

    const x = (this.width / Graphics.windowZoomLevel / 2) - 4.5;

    if (mouseX < x) return;
    if (mouseX > x + 8) return;

    const topRow = this.topRow();

    if (mouseY > 35 && mouseY < 45) {
      this.cursorPageup();

      if (topRow !== this.topRow()) {
        Managers.Sound.playCursor();
      }
      return;
    }

    if (mouseY > 101 && mouseY < 111) {
      this.cursorPagedown();
      if (topRow !== this.topRow()) {
        Managers.Sound.playCursor();
      }
      return;
    }
  }

  prepareToBuyItem(item) {
    this._itemBuying = item;
    this.refresh();
  }

  endPurchase() {
    this._itemBuying = false;
    this.refresh();
  }
}

module.exports = Window_ShopItemList;
