const Window_Selectable = require('engine/windows/Window_Selectable');
const AnimalWindowHelper = require('../contexts/AnimalWindowHelper');

//-----------------------------------------------------------------------------
// Window_AnimalsForSale
//
// A selectable window with a list of animals the player owns

class Window_SellAnimals extends Window_Selectable {
  initialize() {
    this._zoomLevel = Graphics.windowZoomLevel;

    const height = 274 * this._zoomLevel;
    const width = 274 * this._zoomLevel;
    const x = (Graphics.width - width) / 2;
    const y = (Graphics.height - height) / 2;

    this.makeItemList();

    super.initialize(x, y, width, height);
    this.refresh();
    this.select(0);
    this.activate();
  }

  enableWindowFrame() {
    return false;
  }

  enableCursorSprite() {
    return false;
  }

  topMargin() {
    return 29 * this._zoomLevel;
  }

  bottomMargin() {
    return 0;
  }

  horizontalMargin() {
    return 27 * this._zoomLevel;
  }

  maxItems() {
    return this._animalList.length;
  }

  maxCols() {
    return 1;
  }

  standardFontSize() {
    return Utils.convertFontSize(20, this._zoomLevel);
  }

  animal() {
    var idx = this.index();
    if (idx >= 0 && idx < this._animalList.length) {
      return this._animalList[idx];
    }

    return undefined;
  }

  refresh() {
    this.makeItemList();
    this.createContents();

    this.redraw();
  }

  redraw() {
    this.contents.clear();

    this._needsRedraw = false;

    this.drawBackground();
    this.drawAllItems();
    this.drawExtraContent();
  }

  update() {
    super.update();

    if (this._needsRedraw === true) {
      this.redraw();
    } else if (this._needsRedraw) {
      this.requireBitmap(this._needsRedraw);
    }
  }

  select(index) {
    super.select(index);
    this._needsRedraw = true;
  }

  drawBackground() {
    const bitmap = Managers.Images.loadBackground('animal_shop_layout');
    if (!bitmap.isReady()) {
      this._needsRedraw = bitmap;
      return;
    }

    this.drawBitmap(bitmap, 0, 0, this._zoomLevel);
  }

  drawExtraContent() {
    const money = Managers.Text.formatMoney(Managers.Items.gold);
    this.contents.fontSize = Utils.convertFontSize(26, this._zoomLevel);
    this.contents.outlineWidth = 0;
    this.changeTextColor(this.blackColor);

    this.drawText(money, 20 * this._zoomLevel, 12 * this._zoomLevel - (this.contents.fontSize / 2), 50 * this._zoomLevel, 'right');
  }

  itemHeight() {
    return 58 * this._zoomLevel;
  }

  makeItemList() {
    this._animalList = Managers.Creatures.getAnimalList(data => {
      if (!data._creatureName) return false;

      const animalType = Managers.Creatures.getCreatureType(data.type, false);
      if (!animalType) return false;
      if (!animalType._price) return false;

      Managers.Images.loadPicture('menu/animals/' + animalType.creatureName);
      return true;
    });
  }

  drawItem(index) {
    const rect = this.itemRect(index);

    this.contents.fontSize = Utils.convertFontSize(26, this._zoomLevel);

    const bitmap = Managers.Images.loadPicture('shop/animal_shop_sell_slot');
    if (bitmap.isReady()) {
      this.drawZoomedBitmap(bitmap, rect.x, rect.y, this._zoomLevel);
    }

    this.resetTextColor();

    if (index < 0) return;

    const animalData = this._animalList[index];
    if (!animalData) {
      return;
    }

    const { x, y } = rect;
    const bgX = x + 6 * this._zoomLevel;
    const bgY = y + 4 * this._zoomLevel;

    const faceBitmap = Managers.Images.loadPicture('menu/animals/' + animalData.type);
    AnimalWindowHelper.drawAnimalFace(this, animalData, bgX, bgY, this._zoomLevel, faceBitmap);

    const heartsNum = Managers.Relationship.friendshipToHeartNumber(animalData.friendship || 0);
    AnimalWindowHelper.drawAnimalBadgeHearts(this, heartsNum, bgX, bgY, this._zoomLevel);

    const animalName = animalData.creatureName;
    const price = Managers.Creatures.getAnimalSellPrice(animalData);
    const priceStr = Managers.Text.formatMoney(price);

    this.contents.outlineWidth = 0;
    this.changeTextColor(this.blackColor);

    let textY = y + (42 * this._zoomLevel) - (this.contents.fontSize / 2);

    this.drawText(animalName, x + 8 * this._zoomLevel, textY, 43 * this._zoomLevel, 'center');
    this.resetTextColor();

    this.changeTextColor(this.textColor(3));
    textY = y - (this.contents.fontSize / 2);
    this.drawText(priceStr, x + 174 * this._zoomLevel, y - (this.contents.fontSize / 2), 40 * this._zoomLevel, 'center');

    const buttonX = x + 174 * this._zoomLevel;
    const buttonY = y + 38 * this._zoomLevel;
    const buttonW = 40 * this._zoomLevel;
    const buttonH = 12 * this._zoomLevel;
    const buttonColor = this.blueColor();
    const buttonText = t("Sell");

    this.contents.fillRect(buttonX, buttonY, buttonW, buttonH, buttonColor);

    this.changeTextColor(this.baseColor());
    this.drawText(buttonText.toUpperCase(), buttonX, buttonY + 2 * this._zoomLevel - (this.contents.fontSize / 2), buttonW, 'center');

    if (index == this._index) {
      const cursorBitmap = Managers.Images.loadPicture('menu/cursor');
      if (cursorBitmap.isReady()) {
        this.drawZoomedBitmap(cursorBitmap, x + 223 * this._zoomLevel, y + 27 * this._zoomLevel, this._zoomLevel);
      } else {
        this._needsRedraw = cursorBitmap;
      }
    }

    this.drawAnimalInfo(rect, animalData);
  }

  drawAnimalInfo(rect, animalData) {
    let infos = [];
    infos.push(`* ${ t("Type") }: ${ Managers.Text.creature(animalData.type) }`);
    infos.push(`* ${ t("Age") }: ${ Managers.Time.getFormattedAge(animalData.age || 0)}`);
    infos.push(`* ${ t("Location") }: ${ Managers.Map.getMapDisplayName(animalData.mapId) }`);

    let evaluation = null;

    const typeData = Managers.Creatures.getCreatureType(animalData.type);

    switch(typeData.creatureKindName) {
      case 'chicken':
        evaluation = this.getChickenEvaluation(animalData);
        break;
    }

    if (evaluation !== null) {
      infos.push(`* ${ t("Evaluation") }: ${ evaluation } / 100`);
    }

    const text = infos.join('<BR>');
    this.resetFontSettings();
    this.drawTextEx(`<WordWrap>${t(text)}`, rect.x + 55 * this._zoomLevel, rect.y, 120 * this._zoomLevel);
  }

  extraLineHeight() {
    return 0;
  }

  normalColor() {
    return this.blackColor();
  }

  getChickenEvaluation(animalData) {
    const speedScore = Math.round(animalData._chickenSpeed * 20);
    const luckScore = Math.round(animalData._chickenLuck * 20);
    const eggScore = Math.round(animalData._chickenLuck * 20);

    return Math.round((speedScore + luckScore + eggScore) / 3);
  }

  standardOutlineWidth() {
    return 0;
  }

  isCurrentItemEnabled() {
    if (this._index < 0) return false;

    return true;
  }

  standardPadding() {
    return 0;
  }

  spacing() {
    return 0;
  }

  requireBitmap(bitmap) {
    if (!bitmap) return;

    if (!bitmap.isReady()) {
      this._needsRedraw = bitmap;
    }
  }
}

module.exports = Window_SellAnimals;