const Window_Selectable = require('engine/windows/Window_Selectable');
const AnimalWindowHelper = require('../contexts/AnimalWindowHelper');
//-----------------------------------------------------------------------------
// Window_AnimalsForSale
//
// A selectable window with a list of animals the player owns

class Window_AnimalsForSale extends Window_Selectable {
  initialize(typeList) {
    this._typeList = typeList;
    this._zoomLevel = Graphics.windowZoomLevel;

    const height = 274 * this._zoomLevel;
    const width = 274 * this._zoomLevel;
    const x = (Graphics.width - width) / 2;
    const y = (Graphics.height - height) / 2;

    this.makeItemList();

    super.initialize(x, y, width, height);
    this.refresh();
    this.select(0);
    this.activate();
  }

  enableWindowFrame() {
    return false;
  }

  enableCursorSprite() {
    return false;
  }

  topMargin() {
    return 29 * this._zoomLevel;
  }

  bottomMargin() {
    return 0;
  }

  horizontalMargin() {
    return 27 * this._zoomLevel;
  }

  maxItems() {
    return this._animalList.length;
  }

  maxCols() {
    return 1;
  }

  standardFontSize() {
    return Utils.convertFontSize(20, this._zoomLevel);
  }

  animal() {
    var idx = this.index();
    if (idx >= 0 && idx < this._animalList.length) {
      return this._animalList[idx];
    }

    return undefined;
  }

  refresh() {
    this.makeItemList();
    this.createContents();

    this.redraw();
  }

  redraw() {
    this.contents.clear();

    this._needsRedraw = false;

    this.drawBackground();
    this.drawAllItems();
    this.drawExtraContent();
  }

  update() {
    super.update();

    if (this._needsRedraw === true) {
      this.redraw();
    } else if (this._needsRedraw) {
      this.requireBitmap(this._needsRedraw);
    }
  }

  select(index) {
    super.select(index);
    this._needsRedraw = true;
  }

  drawBackground() {
    const bitmap = Managers.Images.loadBackground('animal_shop_layout');
    if (!bitmap.isReady()) {
      this._needsRedraw = bitmap;
      return;
    }

    this.drawBitmap(bitmap, 0, 0, this._zoomLevel);
  }

  drawExtraContent() {
    const money = Managers.Text.formatMoney(Managers.Items.gold);
    this.contents.fontSize = Utils.convertFontSize(26, this._zoomLevel);
    this.contents.outlineWidth = 0;
    this.changeTextColor(this.blackColor);

    this.drawText(money, 20 * this._zoomLevel, 12 * this._zoomLevel - (this.contents.fontSize / 2), 50 * this._zoomLevel, 'right');
  }

  itemHeight() {
    return 58 * this._zoomLevel;
  }

  makeItemList() {
    this._animalList = [];

    for (let i = 0; i < this._typeList.length; i++) {
      const creatureType = Managers.Creatures.getCreatureType(this._typeList[i]);

      if (!creatureType) continue;

      this._animalList.push(creatureType);
      Managers.Images.loadPicture('menu/animals/' + creatureType.creatureName);
    }
  }

  drawItem(index) {
    const rect = this.itemRect(index);

    this.contents.fontSize = Utils.convertFontSize(26, this._zoomLevel);

    const bitmap = Managers.Images.loadPicture('shop/animal_shop_buy_slot');
    if (bitmap.isReady()) {
      this.drawZoomedBitmap(bitmap, rect.x, rect.y, this._zoomLevel);
    }

    this.resetTextColor();

    if (index < 0) return;

    const typeData = this._animalList[index];
    if (!typeData) {
      return;
    }

    const { x, y } = rect;

    const faceBitmap = Managers.Images.loadPicture('menu/animals/' + typeData.creatureName);
    if (faceBitmap.isReady()) {
      AnimalWindowHelper.drawAnimalTypeFace(this, typeData.creatureName, x + 6 * this._zoomLevel, y + 4 * this._zoomLevel, this._zoomLevel, faceBitmap);
    } else {
      this._needsRedraw = faceBitmap;
    }

    const typeName = Managers.Text.creature(typeData.creatureName);
    const price = typeData._price;
    const priceStr = Managers.Text.formatMoney(price);

    const enoughMoney = Managers.Items.gold >= price;
    const enoughRoom = Managers.Creatures.hasRoomForAnimal(typeData.creatureName);

    this.contents.outlineWidth = 0;
    this.changeTextColor(this.blackColor);

    let textY = y + (42 * this._zoomLevel) - (this.contents.fontSize / 2);

    this.drawText(typeName, x + 8 * this._zoomLevel, textY, 43 * this._zoomLevel, 'center');
    this.resetTextColor();

    this.changeTextColor(this.textColor(3));
    textY = y - (this.contents.fontSize / 2);
    this.drawText(priceStr, x + 174 * this._zoomLevel, y - (this.contents.fontSize / 2), 40 * this._zoomLevel, 'center');

    const buttonX = x + 174 * this._zoomLevel;
    const buttonY = y + 38 * this._zoomLevel;
    const buttonW = 40 * this._zoomLevel;
    const buttonH = 12 * this._zoomLevel;
    let buttonColor = this.blueColor();
    let buttonText = t("Buy");
    if (!enoughMoney) {
      buttonText = t("Can't Afford");
      buttonColor = this.redColor();
    } else if (!enoughRoom) {
      buttonText = t("No Room");
      buttonColor = this.redColor();
    }

    this.contents.fillRect(buttonX, buttonY, buttonW, buttonH, buttonColor);

    this.changeTextColor(this.baseColor());
    this.drawText(buttonText.toUpperCase(), buttonX, buttonY + 2 * this._zoomLevel - (this.contents.fontSize / 2), buttonW, 'center');

    if (index == this._index) {
      const cursorBitmap = Managers.Images.loadPicture('menu/cursor');
      if (cursorBitmap.isReady()) {
        this.drawZoomedBitmap(cursorBitmap, x + 223 * this._zoomLevel, y + 27 * this._zoomLevel, this._zoomLevel);
      } else {
        this._needsRedraw = cursorBitmap;
      }
    }

    this.drawAnimalInfo(rect, typeData);
  }

  drawAnimalInfo(rect, typeData) {
    let infos;

    switch(typeData.creatureKindName) {
      case 'chicken':
        if (typeData.creatureName.endsWith('chick')) {
          infos = this.getChickInfo();
        } else {
          infos = this.getChickenInfo();
        }
        break;
      case 'goat':
        infos = this.getGoatInfo();
        break;
      case 'cow':
        infos = this.getCowInfo();
        break;
      case 'sheep':
        infos = this.getSheepInfo();
        break;
    }

    if (!infos) {
      return;
    }

    for (let i = 0; i < infos.length; i++) {
      infos[i] = `* ${ t(infos[i]) }`;
    }

    const text = infos.join('<BR>');
    this.resetFontSettings();
    this.drawTextEx(`<WordWrap>${t(text)}`, rect.x + 55 * this._zoomLevel, rect.y, 120 * this._zoomLevel);
  }

  extraLineHeight() {
    return 0;
  }

  normalColor() {
    return this.blackColor();
  }

  getChickInfo() {
    return [
      "Grows into a chicken in 4 days.",
      "Eats one bird feed per day.",
    ];
  }

  getChickenInfo() {
    return [
      "Lays up to two eggs per day.",
      "Eats one bird feed per day.",
    ];
  }

  getCowInfo() {
    return [
      "Gives milk once per day.",
      "Eats one hay per day.",
    ];
  }

  getSheepInfo() {
    return [
      "Produces wool once per week.",
      "Eats one hay per day.",
    ];
  }

  getGoatInfo() {
    return [
      "Gives milk once per day.",
      "Eats one hay per day.",
    ];
  }

  standardOutlineWidth() {
    return 0;
  }

  isCurrentItemEnabled() {
    if (this._index < 0) return false;

    const typeData = this._animalList[this._index];

    const enoughMoney = Managers.Items.gold >= typeData._price;
    const enoughRoom = Managers.Creatures.hasRoomForAnimal(typeData.creatureName);

    return enoughMoney && enoughRoom;
  }

  standardPadding() {
    return 0;
  }

  spacing() {
    return 0;
  }
}

module.exports = Window_AnimalsForSale;