const Window_Selectable = require('engine/windows/Window_Selectable');

class BaseStampListWindow extends Window_Selectable {
  initialize() {
    this.setup();
  }

  getList() {
    throw new Error('getList method needs to be overwritten');
  }

  makeItemList() {
    throw new Error('makeItemList method needs to be overwritten');
  }

  drawItemText(index, rect, arrowMargin, data, textWidth) {
    throw new Error('drawItemText method needs to be overwritten');
  }

  setup() {
    this.makeItemList();
    this._bgBitmap = Managers.Images.loadPicture('menu/submenu_background');
    
    this._closeBitmap = Managers.Images.loadPicture('menu/exit_menu');
    this._arrowRightBitmap = Managers.Images.loadPicture('menu/arrow_right');
    this._arrowDownBitmap = Managers.Images.loadPicture('menu/arrow_down');
    this._arrowUpBitmap = Managers.Images.loadPicture('menu/arrow_up');

    const width = 266 * Graphics.windowZoomLevel;
    const height = 295 * Graphics.windowZoomLevel;

    super.initialize(0, 0, width, height);
    this.refresh();
    this.select(0);
  }

  enableWindowFrame() {
    return false;
  }

  drawItem(index) {
    if (index < 0) return;
    this.resetTextColor();
    const rect = this.itemRect(index);
    const zoom = Graphics.windowZoomLevel;

    let arrowMargin = 0;

    if (this._arrowRightBitmap.isReady() && index == this._index) {
      arrowMargin = (this._arrowRightBitmap.width + 2) * zoom;
      this.drawBitmap(this._arrowRightBitmap, rect.x / zoom, rect.y / zoom + 3, zoom);

      this.changeTextColor('#fb3153');
    }

    const list = this.getList();
    const data = list[index];
    const textWidth = rect.width - arrowMargin;

    if (data) {
      this.drawItemText(index, rect, arrowMargin, data, textWidth);
    }

    if (index == this._index) {
      this.contents.fillRect(rect.x + arrowMargin, rect.y + rect.height - 1 * zoom, rect.width - arrowMargin, 1 * zoom, '#fb3153');
    }
  }

  drawAllItems() {
    // If the data is already outdated, don't bother drawing it
    if (this._needsRefresh) return;

    const list = this.getList();

    const zoom = Graphics.windowZoomLevel;

    this.drawBitmap(this._bgBitmap, 0, 0, Graphics.windowZoomLevel);
    if (this._closeBitmap.isReady()) {
      this.drawBitmap(this._closeBitmap, (this.width / zoom) - this._closeBitmap.width, 0, zoom);
    }

    if (this._arrowUpBitmap.isReady()) {
      this.drawBitmap(this._arrowUpBitmap, 130.5, 2, zoom);
    } else {
      this.requireBitmap(this._arrowUpBitmap);
    }

    super.drawAllItems();

    const maxIdx = list.length -1;
    if (maxIdx > 0) {
      if (this._arrowDownBitmap.isReady()) {
        this.drawBitmap(this._arrowDownBitmap, 130.5, 281, zoom);
      } else {
        this.requireBitmap(this._arrowDownBitmap);
      }
    }
  }

  horizontalMargin() {
    return 10 * Graphics.windowZoomLevel;
  }

  maxCols() {
    return 1;
  }

  horizontalPadding() {
    return 0 * Graphics.windowZoomLevel;
  }

  verticalPadding() {
    return 0 * Graphics.windowZoomLevel;
  }

  verticalMargin() {
    return 8 * Graphics.windowZoomLevel;
  }

  maxItems() {
    return this.getList().length;
  }

  standardFontSize() {
    return Utils.convertFontSize(26, Graphics.windowZoomLevel);
  }

  resetTextColor() {
    this.changeTextColor('#3d2b25');
  }

  resetFontSettings() {
    super.resetFontSettings();
    this.contents.outlineWidth = 0;
  }

  lineHeight() {
    return this.standardFontSize();
  }

  requireBitmap(bitmap) {
    if (bitmap && !bitmap.isReady()) {
      this._needsRefresh = true;
    }
  }

  update() {
    super.update();
    if (this._needsRefresh) {
      this.refresh();
    }
  }

  refresh() {
    this._needsRefresh = false;
    this.requireBitmap(this._bgBitmap);

    this._refreshExpected = this._needsRefresh;
    super.refresh();
  }

  select(index) {
    if (index != this._index) {
      super.select(index);
      super.refresh();
    }
  }

  itemWidth() {
    const leftMargin = 10 * Graphics.windowZoomLevel;
    const rightMargin = 6 * Graphics.windowZoomLevel;

    const totalWidth = this.width - (this.paddingH * 2) - (leftMargin + rightMargin) + this.spacing();

    return Math.floor(totalWidth / this.maxCols() - this.spacing());
  }

  isMouseOverCloseButton() {
    const bitmap = this._closeBitmap;
    if (!bitmap.isReady()) {
      return false;
    }

    const zoom = Graphics.windowZoomLevel;
    const mouseX = (TouchInput.mouseX - this.x) / zoom;
    const mouseY = (TouchInput.mouseY - this.y) / zoom;

    if (mouseY < 0) return false;

    const right = this.width / zoom;
    const x = right - bitmap.width;

    if (mouseX < x) return false;
    if (mouseY > bitmap.height) return false;
    if (mouseX > right) return false;

    return true;
  }

  selectPrior() {
    const list = this.getList();
    const maxIdx = list.length -1;
    const index = this._index -1;

    if (index < 0) {
      this.select(maxIdx);
    } else {
      this.select(index);
    }
  }

  selectNext() {
    const list = this.getList();
    const maxIdx = list.length -1;

    const index = this._index + 1;

    if (index > maxIdx) {
      this.select(0);
    } else {
      this.select(index);
    }
  }

  updateTouch() {
    const zoom = Graphics.windowZoomLevel;
    const mouseX = (TouchInput.mouseX - this.x) / zoom;
    const mouseY = (TouchInput.mouseY - this.y) / zoom;

    if (mouseY < 0) return;
    if (mouseX < 0) return;

    const right = this.width / zoom;
    if (mouseX > right) return;

    const bottom = this.height / zoom;
    if (mouseY > bottom) return;


    if (mouseX >= 130 && mouseX <= 140) {
      if (mouseY <= 12) {
        this.selectPrior();
        return true;
      }

      if (mouseY >= 280) {
        this.selectNext();
        return true;
      }
    }
  }
}

module.exports = BaseStampListWindow;