const Window_BaseDiaryContent = require('./BaseContent');

class Window_TasksContent extends Window_BaseDiaryContent {
  initialize(x, y, width, height) {
    super.initialize(x, y, width, height);

    this.loadTasks();
    this.loadPictures();

    this._index = 0;
    this._canScrollUp = false;
    this._canScrollDown = false;
  }

  loadPictures() {
    this._taskPictureBmp = Managers.Images.loadPicture('menu/task_picture');
    this._taskSingleIconBmp = Managers.Images.loadPicture('menu/task_single_icon');
    this._taskDoubleIconBmp = Managers.Images.loadPicture('menu/task_double_icon');
    this._taskTripleIconBmp = Managers.Images.loadPicture('menu/task_triple_icon');
    this._taskDescriptionBmp = Managers.Images.loadPicture('menu/task_description');
    this._arrowDownBitmap = Managers.Images.loadPicture('menu/arrow_down');
    this._arrowUpBitmap = Managers.Images.loadPicture('menu/arrow_up');
  }

  loadTasks() {
    const allTasks = Managers.Tasks.getActiveTasks();
    this._tasks = [];

    for (let taskId of allTasks) {
      const data = Managers.Tasks.getTaskData(taskId);
      if (!data) {
        continue;
      }

      const { request, villagerPicture, villagerName, title } = data;
      const cardData = {
        slots: [],
        data,
        taskId,
        title: t(title)
      };

      cardData.slots.push({
        type: 'villager',
        pictureName: villagerPicture || '',
        pictureLabel: villagerName ? t(villagerName) : ''
      });

      let done = false;

      if (data.itemDescription) {

        if (Array.isArray(data.itemDescription)) {
          for (let line of data.itemDescription) {
            cardData.slots.push({
              type: 'description',
              itemDescription: t(line)
            });
          }
        } else {
          cardData.slots.push({
            type: 'description',
            itemDescription: t(data.itemDescription)
          });
        }

        cardData.taskType = 'description';
        done = true;
      }

      if (!done && data.pictureName) {
        const sprite = Managers.Images.loadPicture(data.pictureName);
        if (sprite) {
          cardData.slots.push({
            type: 'picture',
            pictureName: data.pictureName,
            pictureLabel: data.pictureLabel ? t(data.pictureLabel) : '',
            pictureItem: data.pictureItem
          });

          cardData.taskType = 'picture';
          done = true;
        }
      }

      if (!done && data.iconIndex) {
        cardData.slots.push({
          type: 'icon',
          iconIndex: data.iconIndex,
          amount: -1,
          iconLabel: data.iconLabel ? t(data.iconLabel) : '',
          itemId: false
        });
        cardData.taskType = 'single_icon';
      }

      if (!done && request) {
        for (let requestItem of request) {
          let iconIndex = -1;
          let iconLabel;
          const { type, amount } = requestItem;
          const itemId = requestItem.id;

          if (type == 'gold') {
            iconIndex = 192;
            iconLabel = t('Money');
          } else if (type == 'item') {
            const itemData = Managers.Items.getItemData(itemId);
            if (!itemData) {
              continue;
            }

            iconIndex = itemData.iconIndex;
            iconLabel = Managers.Text.item(itemId);
          } else {
            continue;
          }

          cardData.slots.push({
            type,
            iconIndex,
            amount,
            itemId,
            iconLabel
          });

          if (cardData.taskType == 'single_icon') {
            cardData.taskType = 'double_icon';
          } else if (cardData.taskType == 'double_icon') {
            cardData.taskType = 'triple_icon';
            break;
          } else {
            cardData.taskType = 'single_icon';
          }
        }
      }

      this._tasks.push(cardData);
    }
  }

  drawTaskItems(items, slotX, slotY, taskType) {
    if (taskType == 'triple_icon') return;

    const zoom = this._zoomLevel;
    const textX = (slotX + 64) * zoom;
    const textY = (slotY + 14) * zoom;
    const textW = (taskType == 'description' ? 142 : 90) * zoom;
    const color = '#3d2b25';

    this.changeTextColor(color);
    this.contents.outlineColor = color;
    this.contents.outlineWidth = 0;

    this.contents.fontSize = Utils.convertFontSize(22, this._zoomLevel);

    for (let i = 0; i < items.length; i++) {
      const y = textY + 13 * i * zoom;

      this.drawText('* ' + items[i], textX, y, textW, 'left');
    }
  }

  drawTaskTitle(data, slotX, slotY) {
    const zoom = this._zoomLevel;

    let textX = (slotX + 64) * zoom;
    let textY = slotY * zoom;
    let textW = 142 * zoom;
    
    const color = '#328ffa';

    this.changeTextColor(color);
    this.contents.outlineColor = color;
    this.contents.outlineWidth = 0;

    this.contents.fontSize = Utils.convertFontSize(24, this._zoomLevel);
    this.drawText(data.title, textX, textY, textW, 'left');
  }

  drawPictureTask(pictureName, slotX, slotY, pictureLabel, positionType) {
    const zoom = this._zoomLevel;
    const sprite = Managers.Images.loadPicture(pictureName);
  
    if (!sprite.isReady()) {
      this.requireBitmap(sprite);
      return;
    }

    let spriteX;
    let textX;

    switch(positionType) {
      case 'villager':
        spriteX = (slotX + 15) * zoom;
        textX = (slotX + 17) * zoom;
        break;
      case 'picture':
        spriteX = (slotX + 164) * zoom;
        textX = (slotX + 160) * zoom;
        break;
      default:
        return;
    }
    
    const spriteY = (slotY - 1) * zoom;
    const textY = (slotY + 39) * zoom;
    const textW = 42 * zoom;

    this.drawPicture(pictureName, spriteX, spriteY, zoom);

    if (!pictureLabel) {
      return;
    }

    const color = '#328ffa';

    this.changeTextColor(color);
    this.contents.outlineColor = color;
    this.contents.outlineWidth = 0;

    this.contents.fontSize = Utils.convertFontSize(24, this._zoomLevel);
    this.drawText(pictureLabel, textX, textY, textW, 'center');
  }

  drawIconTask(data, slotX, slotY, iconIndex, positionType) {
    const zoom = this._zoomLevel;

    let iconX;
    let iconY;

    switch(positionType) {
      case 'double_icon':
        if (iconIndex === 0) {
          iconX = (slotX + 155) * zoom;
          iconY = (slotY + 11) * zoom;
        } else {
          iconX = (slotX + 183) * zoom;
          iconY = (slotY + 20) * zoom;
        }
        break;
      case 'triple_icon':
        if (iconIndex % 2 === 0) {
          iconY = (slotY + 20) * zoom;          
        } else {
          iconY = (slotY + 11) * zoom;
        }

        switch(iconIndex) {
          case 0:
            iconX = (slotX + 127) * zoom;
            break;
          case 1:
            iconX = (slotX + 155) * zoom;
            break;
          case 2:
            iconX = (slotX + 183) * zoom;
            break;
        }
        break;
      default:
        iconX = (slotX + 168) * zoom;
        iconY = (slotY + 13) * zoom;
        break;
    }

    this.drawIcon(data.iconIndex, iconX, iconY, zoom);

    let text;
    let color = '#328ffa';
    
    if (data.type === 'icon') {
      if (data.iconLabel && data.iconLabel.length <= 12) {
        text = data.iconLabel;
      } else {
        return;
      }
    }

    const textX = iconX;
    const textY = iconY + (20 * zoom);
    const textW = 24 * zoom;

    if (data.type !== 'icon') {
      const maxAmount = data.amount;
      const currentAmount = data.type == 'gold' ? Managers.Items.gold : Managers.Items.numItemsById(data.itemId);
      const amount = currentAmount >= maxAmount ? maxAmount : currentAmount;

      if (amount == maxAmount) {
        color = '#fb3153';
      }

      if (data.type == 'gold') {
        text = Managers.Text.formatMoney(maxAmount);
      } else {
        text = `${ amount }/${ maxAmount }`;
      }
    }

    this.changeTextColor(color);
    this.contents.outlineColor = color;
    this.contents.outlineWidth = 0;

    this.contents.fontSize = 24;
    this.drawText(text, textX, textY, textW, 'center');
  }

  drawTasks() {
    let drawnCardIndex = 0;
    let skippedIndexes = 0;
    this._canScrollUp = false;
    this._canScrollDown = false;

    for (let idx = 0; idx < this._tasks.length; idx++) {
      const data = this._tasks[idx];
      if (!data.slots.length) {
        continue;
      }

      if (!data.taskType) {
        continue;
      }

      let bitmap;
      switch(data.taskType) {
        case 'description':
          bitmap = this._taskDescriptionBmp;
          break;
        case 'single_icon':
          bitmap = this._taskSingleIconBmp;
          break;
        case 'double_icon':
          bitmap = this._taskDoubleIconBmp;
          break;
        case 'triple_icon':
          bitmap = this._taskTripleIconBmp;
          break;
        case 'picture':
          bitmap = this._taskPictureBmp;
          break;
        default:
          bitmap = this._taskPictureBmp;
          break;
      }

      if (!bitmap) {
        continue;
      }

      if (this._index && skippedIndexes < this._index) {
        skippedIndexes++;
        this._canScrollUp = true;
        continue;
      }

      if (drawnCardIndex >= 3) {
        this._canScrollDown = true;
        break;
      }

      const bitmapY = 58 + (68 * drawnCardIndex);
      const bitmapX = 0;

      this.drawBitmap(bitmap, bitmapX, bitmapY);
      this.drawCard(data, bitmapX, bitmapY);

      drawnCardIndex++;
    }

    if (this._canScrollUp) {
      this.drawBitmap(this._arrowUpBitmap, 104, 48);
    }

    if (this._canScrollDown) {
      this.drawBitmap(this._arrowDownBitmap, 104, 250);
    }
  }

  drawCard(data, x, y) {
    let iconIndex = 0;

    const items = [];

    for (let slotIdx = 0; slotIdx < data.slots.length; slotIdx++) {
      const slotData = data.slots[slotIdx];

      if (['picture', 'villager'].indexOf(slotData.type) >= 0) {
        this.drawPictureTask(slotData.pictureName, x, y, slotData.pictureLabel, slotData.type);

        if (slotData.pictureItem) {
          items.push(slotData.pictureItem);
        }
      } else if (['icon', 'item', 'gold'].indexOf(slotData.type) >= 0) {
        this.drawIconTask(slotData, x, y, iconIndex, data.taskType);
        iconIndex++;

        if (slotData.iconLabel) {
          items.push(slotData.iconLabel);
        }
      } else if (slotData.type == 'description') {
        items.push(slotData.itemDescription);
      }
    }

    if (data.title) {
      this.drawTaskTitle(data, x, y);
    }

    if (items.length > 0) {
      this.drawTaskItems(items, x, y, data.taskType);
    }
  }

  drawContent() {
    this.drawTitle();    
    this.drawTasks();
  }

  drawTitle() {
    const zoom = this._zoomLevel;

    const x = 47 * zoom;
    const y = 20 * zoom;
    const w = 122 * zoom;
    const h = Utils.convertFontSize(56, zoom);

    this.contents.fontSize = h;
    this.contents.outlineColor = this.baseColor();
    this.contents.outlineWidth = 0;

    this.changeTextColor(this.baseColor());
    this.drawText(t("Tasks").toUpperCase(), x, y, w, 'center');
  }

  scrollUp() {
    this._index -= 3;
    if (this._index < 0) {
      this._index = 0;
    }
  }

  scrollDown() {
    this._index += 3;
  }

  update() {
    this.refresh();

    const zoom = Graphics.windowZoomLevel;
    const mouseX = (TouchInput.mouseX - this.x) / zoom;
    const mouseY = (TouchInput.mouseY - this.y) / zoom;

    if (this._canScrollDown) {
      if (Engine.Input.isTriggered('down')) {
        this.scrollDown();
        return;
      }

      if (TouchInput.isTriggered()) {
        if (mouseX >= 104 && mouseX <= 112) {
          if (mouseY >= 250 && mouseY <= 260) {
            this.scrollDown();
            return;
          }
        }
      }
    }

    if (this._canScrollUp) {
      if (Engine.Input.isTriggered('up')) {
        this.scrollUp();
        return;
      }

      if (TouchInput.isTriggered()) {
        if (mouseX >= 104 && mouseX <= 112) {
          if (mouseY >= 48 && mouseY <= 58) {
            this.scrollUp();
            return;
          }
        }
      }
    }
  }
}

module.exports = Window_TasksContent;