const Window_Command = require('engine/windows/Window_Command');

class Window_SettingsList extends Window_Command {
  initialize(x, y, width, height) {
    this._width = width;
    this._height = height;
    this._optionIndicatorBitmap = Managers.Images.loadPicture('menu/new_option');
    this._option1Bitmap = Managers.Images.loadPicture('menu/option_1');
    this._option1SelectedBitmap = Managers.Images.loadPicture('menu/option_1_selected');
    this._option2Bitmap = Managers.Images.loadPicture('menu/option_2');
    this._arrowsBitmap = Managers.Images.loadPicture('menu/arrows');
    this._arrowIndex = 0;
    this._zoomLevel = Graphics.windowZoomLevel;
    this._mappingControls = false;

    super.initialize(x, y);

    this.clearCommandList();
    this.makeCommandList();
  }

  enableWindowFrame() {
    return false;
  }

  standardPadding() {
    return 0;
  }

  windowWidth() {
    return this._width;
  }

  windowHeight() {
    return this._height;
  }

  makeGraphicsCommandList() {
    this.addCommand(t('Resolution'), 'resolution', true, 'list');
    this.addCommand(t('Window Size'), 'windowSize', true, 'list');
    this.addCommand(t('Fullscreen'), 'fullscreen', true, 'boolean');

    this.addCommand(t('Stretch To Fit'), 'stretchToFit', true, 'boolean');
    this.addCommand(t('Renderer'), 'renderer', true, 'list');
    this.addCommand(t('Game Speed'), 'gameFps', true, 'list');
    this.addCommand(t('FPS Calibration'), 'fpsCalibration', true, 'number');
    this.addCommand(t('Enable Animations'), 'enableAnimations', true, 'boolean');
    // this.addCommand(t('Max Frame Skip'), 'maxAutoFrameSkip', true, 'number');
    this.addCommand(t('Use Screen Refresh Rate'), 'useScreenRefreshRate', true, 'boolean');

    this.addCommand(t('Back'), 'back', true, 'action');
  }

  makeAudioCommandList() {
    this.addCommand(t('Master Volume'), 'masterVolume', true, 'volume');
    this.addCommand(t('BGM Volume'), 'bgmVolume', true, 'volume');
    this.addCommand(t('SE Volume'), 'seVolume', true, 'volume');

    this.addCommand(t('Back'), 'back', true, 'action');
  }

  makeControlsCommandList() {
    this.addCommand(t('Enable Mouse'), 'enableMouse', true, 'boolean');
    this.addCommand(t('Keyboard Mapping'), 'keyboardMapping', true, 'empty');
    this.addCommand(t("Gamepad Profiles"), 'gamepadProfiles', true, 'empty');

    this.addCommand(t('Back'), 'back', true, 'action');
  }

  makeModsCommandList() {
    Managers.Content.makeModList(this);

    this.addCommand(t('Back'), 'back', true, 'action');
  }

  makeGeneralCommandList() {
    this.addCommand(t('Item Bar'), 'drawItemBar', true, 'boolean');
    this.addCommand(t('Tool Bar'), 'drawToolBar', true, 'boolean');
    this.addCommand(t('Quick Text'), 'quickText', true, 'boolean');
    this.addCommand(t('Small HUD'), 'smallHud', true, 'boolean');

    this.addCommand(t('Back'), 'back', true, 'action');
  }

  makeControllerButtonList() {
    this.addControllerCommand('ok');
    this.addControllerCommand('alternate');
    this.addControllerCommand('use');
    this.addControllerCommand('extra');
    this.addControllerCommand('left_backbutton');
    this.addControllerCommand('right_backbutton');
    this.addControllerCommand('left_trigger');
    this.addControllerCommand('right_trigger');
    this.addControllerCommand('map');
    this.addControllerCommand('start');
    this.addControllerCommand('change_backpack');
    this.addControllerCommand('inventory');
    this.addControllerCommand('quicksave');

    // this.addCommand(t('Reset Defaults'), `gamepad_reset`, true, 'action');
    this.addCommand(t('Back'), 'back', true, 'action');
  }

  addControllerCommand(buttonAction) {
    this.addCommand(this.getButtonActionLabel(buttonAction), `gamepad_${ buttonAction}`, true, 'gamepad');
  }

  getButtonActionLabel(buttonAction) {
    switch(buttonAction) {
      case 'up':
        return t("Up");
      case 'down':
        return t("Down");
      case 'left':
        return t("Left");
      case 'right':
        return t("Right");
      case 'ok':
        return t("Confirm, Action");
      case 'alternate':
        return t("Cancel, Run");
      case 'use':
        return t("Use Item or Tool");
      case 'map':
        return t("Map");
      case 'extra':
        return t("Extra");
      case 'start':
        return t("Cancel, Open Menu");
      case 'left_backbutton':
        return t("Previous Tool");
      case 'right_backbutton':
        return t("Next Tool");
      case 'left_trigger':
        return t("Previous Item");
      case 'right_trigger':
        return t("Next Item");
      case 'quicksave':
        return t("Quick Save");
      case 'change_backpack':
        return t("Change Backpack");
      case 'inventory':
        return t("Inventory");
    }

    return false;
  }


  makeGamepadList() {
    const allTypes = Engine.Input.getAllGamepadTypes();
    const listedTypes = [];

    if (allTypes.includes('LJOY') && allTypes.includes('RJOY')) {
      allTypes.push('JOYCON-PAIR');
    }

    for (const type of allTypes) {
      if (listedTypes.includes(type)) {
        continue;
      }

      let label = type;

      switch(type.toLowerCase()) {
        case 'ps2':
          label = t('PS2 Controller');
          break;
        case 'ps3':
          label = t('PS3 Controller');
          break;
        case 'ps4':
          label = t('PS4 Controller');
          break;
        case 'sony':
          label = t('Sony Controller');
          break;
        case 'x360':
          label = t('Xbox 360 Controller');
          break;
        case 'xone':
          label = t('Xbox One Controller');
          break;
        case 'xbox':
          label = t('Xbox Controller');
          break;
        case 'ms':
          label = t('Microsoft Controller');
          break;
        case 'ljoy':
          label = t('Nintendo Switch Left JoyCon');
          break;
        case 'rjoy':
          label = t('Nintendo Switch Right JoyCon');
          break;
        case 'joycon-pair':
          label = t('Pair of Nintendo Switch JoyCons');
          break;
        case 'pro':
          label = t('Nintendo Switch Pro Controller');
          break;
        case 'xinput':
          label = t('XInput Compatible Controller');
          break;
        default:
          label = t('Unknown Controller');
          break;
      }

      listedTypes.push(type);
      this.addCommand(label, type, true, 'controller');
    }


    this.addCommand(t('Back'), 'back', true, 'action');
  }

  makeMappingCommandList() {
    this.addKeyCommand('up');
    this.addKeyCommand('down');
    this.addKeyCommand('left');
    this.addKeyCommand('right');
    this.addKeyCommand('ok');
    this.addKeyCommand('escape');
    this.addKeyCommand('use');
    this.addKeyCommand('map');
    this.addKeyCommand('extra');
    this.addKeyCommand('pageup');
    this.addKeyCommand('pagedown');
    this.addKeyCommand('change_backpack');
    this.addKeyCommand('inventory');
    this.addKeyCommand('quicksave');
    
    this.addCommand(t('Reset Defaults'), `key_reset`, true, 'action');
    this.addCommand(t('Back'), 'back', true, 'action');
  }

  addKeyCommand(key) {
    this.addCommand(this.getKeyLabel(key), `key_${ key }`, true, 'key');
  }

  getKeyLabel(keySymbol) {
    switch(keySymbol) {
      case 'up':
        return t("Up");
      case 'down':
        return t("Down");
      case 'left':
        return t("Left");
      case 'right':
        return t("Right");
      case 'ok':
        return t("Confirm, Action");
      case 'escape':
        return t("Cancel, Open Menu");
      case 'use':
        return t("Use Item or Tool");
      case 'map':
        return t("Map");
      case 'extra':
        return t("Extra");
      case 'pageup':
        return t("Change Tool");
      case 'pagedown':
        return t("Change Item");
      case 'quicksave':
        return t("Quick Save");
      case 'change_backpack':
        return t("Change Backpack");
      case 'inventory':
        return t("Inventory");
    }

    return false;
  }

  makeKeyCommandList() {
    const key = this._selectedKey;

    this.addCommand(this.getKeyLabel(key), `do_nothing`, true, 'label');
    this.addCommand(t("First Key"), `key_${ key }_key1`, true, 'key');
    this.addCommand(t("Second Key"), `key_${ key }_key2`, true, 'key');

    this.addCommand(t('Back'), 'back', true, 'action');
  }

  makeCommandList() {
    if (this._waitingForKey) {
      return;
    }

    const category = this.category();
    switch (category) {
      case 'gamepad':
        this._index = -1;
        break;
      case 'keyboard':
        if (this._selectedKey) {
          this.makeKeyCommandList();
        } else {
          this.makeMappingCommandList();
        }

        break;
      case 'gamepads':
        if (this._selectedController) {
          this.makeControllerButtonList();
        } else {
          this.makeGamepadList();
        }

        break;
      case 'graphics':
        this.makeGraphicsCommandList();
        break;
      case 'audio':
        this.makeAudioCommandList();
        break;
      case 'controls':
        this.makeControlsCommandList();
        break;
      case 'mods':
        if (this._subCategory) {
          if (Managers.Content.makeSettingsList(this)) {
            this.addCommand(t('Back'), 'back', true, 'action');
            return;
          }
        }

        this.makeModsCommandList();
        break;
      case 'general':
        this.makeGeneralCommandList();
        break;
      default:
        this._index = -1;
        break;
    }
  }

  desiredWidth() {
    if (this._mappingControls == 'gamepads') {
      return 186;
    } else if (this._selectedController) {
      return 160;
    } else {
      return 140;
    }
  }

  drawItem(index) {
    const rect = this.itemRectForText(index);
    const itemType = this.getExt(index);

    if (itemType == 'action') {
      this.changeTextColor('#328ffa');
    } else if (index == this.index()) {
      this.resetTextColor();
    } else {
      this.changeTextColor('#3d2b25');
    }

    const bitmap = this._optionIndicatorBitmap;
    if (bitmap.isReady()) {
      this.contents.bltImage(bitmap, 0, 0, bitmap.width, bitmap.height, 0, rect.y, bitmap.width * this._zoomLevel, bitmap.height * this._zoomLevel);
    } else {
      this.requireBitmap(bitmap);
    }


    let baseWidth = this.desiredWidth();

    const commandWidth =  (baseWidth - 17) * this._zoomLevel;
    this.drawText(this.commandName(index), rect.x + 4 * this._zoomLevel , rect.y - 2 * this._zoomLevel, commandWidth, 'left');

    this.resetTextColor();
    this.drawValue(index, rect);
  }

  resetTextColor() {
    this.changeTextColor('#f4dea8');
  }

  drawArrow(x, y, direction) {
    const bitmap = this._arrowsBitmap;

    if (!bitmap.isReady()) {
      this.requireBitmap(bitmap);
      return;
    }

    const directionIndex = [Direction.UP, Direction.RIGHT, Direction.LEFT, Direction.DOWN].indexOf(direction);
    if (directionIndex < 0) {
      return;
    }

    const sx = 10 * this._arrowIndex;
    const sy = 10 * directionIndex;

    this.contents.blt(this._arrowsBitmap, sx, sy, 10, 10, x, y, 10 * this._zoomLevel, 10 * this._zoomLevel);
  }

  _drawBooleanValue(value, symbol, rect, index) {
    const bitmap1 = value ? this._option1Bitmap : this._option1SelectedBitmap;
    const bitmap2 = value ? this._option1SelectedBitmap : this._option1Bitmap;

    let x = rect.x + 140 * this._zoomLevel;

    if (bitmap1.isReady()) {
      this.contents.bltImage(bitmap1, 0, 0, bitmap1.width, bitmap1.height, x, rect.y, bitmap1.width * this._zoomLevel, bitmap1.height * this._zoomLevel);

      this.drawText(t('No'), x, rect.y - 2 * this._zoomLevel, bitmap1.width * this._zoomLevel, 'center');
      x += (bitmap1.width + 2) * this._zoomLevel;
    } else {
      this.requireBitmap(bitmap1);
    }

    if (bitmap2.isReady()) {
      this.contents.bltImage(bitmap2, 0, 0, bitmap2.width, bitmap2.height, x, rect.y, bitmap2.width * this._zoomLevel, bitmap2.height * this._zoomLevel);
      this.drawText(t('Yes'), x, rect.y - 2 * this._zoomLevel, bitmap2.width * this._zoomLevel, 'center');

      x += (bitmap2.width + 2) * this._zoomLevel;
    } else {
      this.requireBitmap(bitmap2);
    }

    this.drawSelectionArrows(index, rect, !value, value);
  }

  drawBooleanValue(symbol, rect, index) {
    const value = Managers.Config[symbol];

    this._drawBooleanValue(value, symbol, rect, index);
  }

  getRendererText() {
    const value = Managers.Config.renderer;

    if (value == 'webgl') return 'WebGL';
    if (value == 'canvas') return 'Canvas';
    if (value == 'auto') return 'Auto';
    return value;
  }

  getTextValue(symbol) {
    switch(symbol) {
      case 'resolution':
        return Managers.Config.resolutionLabel;
      case 'windowSize':
        if (Managers.Config.windowSize == 'disabled') {
          return t('Disabled');
        }
        return `${Managers.Config.windowWidth} x ${Managers.Config.windowHeight}`;
      case 'gameFps':
        return `${Managers.Config.gameFps} FPS`;
      case 'fpsCalibration':
        return Managers.Config.fpsCalibration < 0 ? `${Managers.Config.fpsCalibration}` : `+ ${Managers.Config.fpsCalibration}`;
      case 'maxAutoFrameSkip':
        return Managers.Config.maxAutoFrameSkip;
      case 'renderer':
        return this.getRendererText();
      default:
        return '';
    }
  }

  drawSelectionArrows(index, rect, suppressLeft, suppressRight) {
    if (index == this._index) {
      if (!suppressLeft) {
        const x1 = rect.x + 130 * this._zoomLevel;
        this.drawArrow(x1, rect.y + 2 * this._zoomLevel, Direction.LEFT);
      }

      if (!suppressRight) {
        const x2 = rect.x + 186 * this._zoomLevel;
        this.drawArrow(x2, rect.y + 2 * this._zoomLevel, Direction.RIGHT);
      }

      if (!suppressLeft || !suppressRight) {
        this._arrowTimeoutHandler = setTimeout(() => {
          this._arrowTimeoutHandler = null;
          if (this._index == index) {
            this._arrowIndex = (this._arrowIndex + 1) % 4;
            this.refresh();
          }
        }, 100);
      }
    }
  }

  drawListValue(symbol, rect, index) {
    this.drawTextValue(symbol, rect);
    const minValue = this.getMinValue(symbol);
    const maxValue = this.getMaxValue(symbol);

    this.drawSelectionArrows(index, rect, false, false, minValue, maxValue);
  }

  _drawNumberValue(value, symbol, rect, index, minValue, maxValue) {
    const bitmap = this._option2Bitmap;
    const x = rect.x + 140 * this._zoomLevel;

    if (minValue === undefined) {
      minValue = this.getMinValue(symbol);
    }
    if (maxValue === undefined) {
      maxValue = this.getMaxValue(symbol);
    }

    if (bitmap.isReady()) {
      this.contents.bltImage(bitmap, 0, 0, bitmap.width, bitmap.height, x, rect.y, bitmap.width * this._zoomLevel, bitmap.height * this._zoomLevel);
      this.drawText(value, x, rect.y - 2 * this._zoomLevel, bitmap.width * this._zoomLevel, 'center');
    } else {
      this.requireBitmap(bitmap);
    }

    if (minValue !== undefined && maxValue !== undefined) {
      this.drawSelectionArrows(index, rect, value <= minValue, value >= maxValue);
    }
  }

  drawNumberValue(symbol, rect, index) {
    const value = Managers.Config[symbol];
    this._drawNumberValue(value, symbol, rect, index);
  }

  drawVolumeValue(symbol, rect, index) {
    const bitmap = this._option2Bitmap;
    const x = rect.x + 140 * this._zoomLevel;
    const value = Managers.Config[symbol];

    if (bitmap.isReady()) {
      this.contents.bltImage(bitmap, 0, 0, bitmap.width, bitmap.height, x, rect.y, bitmap.width * this._zoomLevel, bitmap.height * this._zoomLevel);

      const progressWidth = Math.floor(((bitmap.width * this._zoomLevel) / 100) * value);
      if (progressWidth > 0) {
        this.contents.fillRect(x, rect.y, progressWidth, bitmap.height * this._zoomLevel, '#fb3153');
      }

      this.drawText(value, x, rect.y - 2 * this._zoomLevel, bitmap.width * this._zoomLevel, 'center');
    } else {
      this.requireBitmap(bitmap);
    }

    this.drawSelectionArrows(index, rect, value <= 0, value >= 100);
  }

  _drawTextValue(value, symbol, rect, index) {
    const bitmap = this._option2Bitmap;
    const x = rect.x + 140 * this._zoomLevel;

    if (bitmap.isReady()) {
      this.contents.bltImage(bitmap, 0, 0, bitmap.width, bitmap.height, x, rect.y, bitmap.width * this._zoomLevel, bitmap.height * this._zoomLevel);
      this.drawText(value, x, rect.y - 2 * this._zoomLevel, bitmap.width * this._zoomLevel, 'center');
    } else {
      this.requireBitmap(bitmap);
    }
  }

  drawTextValue(symbol, rect, index) {
    const value = this.getTextValue(symbol);
    this._drawTextValue(value, symbol, rect, index);
  }

  getKeyDescriptions(keyNameOrSymbol, useArray) {
    return Managers.Config.getKeyDescriptions(keyNameOrSymbol, useArray);
  }

  drawKeyValue(symbol, rect, index) {
    const bitmap = this._option2Bitmap;
    const x = rect.x + 140 * this._zoomLevel;

    if (bitmap.isReady()) {
      this.contents.bltImage(bitmap, 0, 0, bitmap.width, bitmap.height, x, rect.y, bitmap.width * this._zoomLevel, bitmap.height * this._zoomLevel);

      const descriptions = this.getKeyDescriptions(symbol.replace('_key1', '').replace('_key2', ''));
      let text = descriptions.toString();

      if (symbol.startsWith('key_')) {
        if (symbol.endsWith('_key1')) {
          text = descriptions[0] || '';
        } else if (symbol.endsWith('_key2')) {
          text = descriptions[1] || '';
        }
      }

      this.drawText(text, x, rect.y - 2 * this._zoomLevel, bitmap.width * this._zoomLevel, 'center');
    } else {
      this.requireBitmap(bitmap);
    }
  }

  drawGamepadValue(symbol, rect, index) {
    const x = rect.x + 160 * this._zoomLevel;

    const iconIndex = Managers.Config.getButtonIconIndex(symbol, this._selectedController);
    this.drawIcon(iconIndex, x + 2, rect.y - 5 * this._zoomLevel);
  }

  standardOutlineWidth() {
    return 0;
  }

  _updateCursor() {
    if (this.active) {
      super._updateCursor();
    }
  }

  quickSelect(index) {
    super.select(index);
  }

  select(index) {
    const refresh = this._index != index;
    super.select(index);

    if (refresh) {
      if (this._categoryWindow) {
        this._categoryWindow.deactivate();
        this.activate();
      }
      this.refresh();
    }
  }

  drawValue(index, rect) {
    const data = index >= 0 ? this._list[index] : null;
    if (!data) {
      return;
    }

    const category = this.category();
    const symbol = data.symbol;
    const ext = data.ext;

    if (category == 'mods') {
      if (Managers.Content.drawModSettingValue(this, symbol, rect, index)) {
        return;
      }
    }

    switch (ext) {
      case 'boolean':
        this.drawBooleanValue(symbol, rect, index);
        break;
      case 'text':
        this.drawTextValue(symbol, rect, index);
        break;
      case 'key':
        this.drawKeyValue(symbol, rect, index);
        break;
      case 'gamepad':
        this.drawGamepadValue(symbol, rect, index);
        break;
      case 'list':
        this.drawListValue(symbol, rect, index);
        break;
      case 'number':
        this.drawNumberValue(symbol, rect, index);
        break;
      case 'volume':
        this.drawVolumeValue(symbol, rect, index);
        break;
    }
  }

  configureCursorRect(rect) {
    const width = this.desiredWidth();
    this.setCursorRect(rect.x, rect.y, width * this._zoomLevel, rect.height);
  }

  lineHeight() {
    return this._zoomLevel * 14;
  }

  spacing() {
    return 0;
  }

  setCategoryWindow(categoryWindow) {
    this._categoryWindow = categoryWindow;
    this.refresh();
  }

  category() {
    if (this._mappingControls) {
      return this._mappingControls;
    }

    if (!this._categoryWindow) {
      return 'general';
    }

    return this._categoryWindow.currentSymbol() || 'general';
  }

  // itemHeight() {
  //   return this._zoomLevel * 18;
  // }

  verticalItemSpacing() {
    return this._zoomLevel * 4;
  }

  refresh() {
    if (this._arrowTimeoutHandler) {
      clearTimeout(this._arrowTimeoutHandler);
      this._arrowTimeoutHandler = null;
    }

    if (!this.parent) {
      return;
    }

    super.refresh();

    if (this._index >= this._list.length) {
      this.select(0);
    }
  }

  checkItemTouch(index, x, y, rect) {
    const originalX = x / this._zoomLevel;

    this.select(index);
    if (originalX < 130) {
      return;
    }

    const width = this.desiredWidth();
    if (originalX < (width - 10)) {
      return;
    }

    if (originalX < 140) {
      this.changeToPreviousValue();
      return;
    }

    if (originalX > 186) {
      this.changeToNextValue();
      return;
    }

    this.onValueClick(originalX);
  }

  changeValue(symbol, newValue) {
    Managers.Config[symbol] = newValue;
    switch (symbol) {
      case 'fullscreen':
        if (newValue) {
          Graphics._requestFullScreen();
        } else {
          Graphics._cancelFullScreen();
        }
        break;
      case 'gameFps':
        Managers.Scenes.updateGameSpeed(60 / newValue);
        break;
      case 'resolution':
        Managers.Config.forceResolution = false;
        if (!Utils.windowSizeFitsScreenAndResolution(Managers.Config.windowSize, Managers.Config.resolution)) {
          if (Utils.windowSizeFitsScreenAndResolution(Managers.Config.resolution)) {
            Managers.Config.windowSize = Managers.Config.resolution;
          } else {
            Managers.Config.windowSize = Utils.getNextWindowSize('none');
          }
        } else if (Managers.Config.windowSize == 'disabled') {
          Managers.Config.windowSize = Utils.getNextWindowSize('none');
        }

        break;
    }
  }

  onValueClick(originalX) {
    const data = this.currentData();
    if (!data) {
      return;
    }

    const symbol = data.symbol;
    const ext = data.ext;

    if (ext == 'boolean') {
      if (originalX < 163) {
        this.changeValue(symbol, false);
      } else {
        this.changeValue(symbol, true);
      }
    }
  }

  getValueList(symbol) {
    switch(symbol) {
      case 'resolution':
        return Utils.getResolutionList();
      case 'windowSize':
        return Utils.getWindowSizeList();
      case 'renderer':
        return ['auto', 'canvas'];
      case 'gameFps':
        return [30, 40, 48, 60];
    }

    return [];
  }

  getMaxValue(symbol) {
    switch(symbol) {
      case 'fpsCalibration':
        return 60;
      case 'maxAutoFrameSkip':
        return 15;
    }
  }

  getMinValue(symbol) {
    switch(symbol) {
      case 'fpsCalibration':
        return -30;
      case 'maxAutoFrameSkip':
        return 0;
    }
  }

  changeToPreviousValue() {
    const data = this.currentData();
    if (!data) {
      return;
    }

    const category = this.category();
    const symbol = data.symbol;
    const ext = data.ext;

    if (category == 'mods') {
      if (Managers.Content.changeToPreviousValue(this, symbol, ext)) {
        return;
      }
    }

    const value = Managers.Config[symbol];

    switch (ext) {
      case 'boolean':
        if (value) {
          this.changeValue(symbol, !value);
        }
        break;
      case 'list':
        {
          const list = this.getValueList(symbol);
          if (list) {
            const currIndex = list.indexOf(value);
            const prevIndex = currIndex > 0 ? currIndex - 1 : list.length -1;
            const newValue = list[prevIndex];
            this.changeValue(symbol, newValue);
          }
        }

        break;
      case 'number':
        {
          const minValue = this.getMinValue(symbol);
          if (value > minValue) {
            this.changeValue(symbol, value - 1);
          } else {
            this.changeValue(symbol, minValue);
          }
        }

        break;
      case 'volume':
        if (value >= 5) {
          this.changeValue(symbol, value - 5);
        } else {
          this.changeValue(symbol, 0);
        }
        break;
    }
  }

  changeToNextValue() {
    const data = this.currentData();
    if (!data) {
      return;
    }

    const category = this.category();
    const symbol = data.symbol;
    const ext = data.ext;

    if (category == 'mods') {
      if (Managers.Content.changeToNextValue(this, symbol, ext)) {
        return;
      }
    }

    const value = Managers.Config[symbol];

    switch (ext) {
      case 'boolean':
        if (!value) {
          this.changeValue(symbol, !value);
        }
        break;
      case 'list':
        {
          const list = this.getValueList(symbol);
          if (list) {
            const currIndex = list.indexOf(value);
            const prevIndex = currIndex < list.length -1 ? currIndex + 1 : 0;
            const newValue = list[prevIndex];
            this.changeValue(symbol, newValue);
          }
        }

        break;
      case 'number':
        {
          const maxValue = this.getMaxValue(symbol);
          if (value < maxValue) {
            this.changeValue(symbol, value + 1);
          } else {
            this.changeValue(symbol, maxValue);
          }
        }

        break;
      case 'volume':
        if (value <= 95) {
          this.changeValue(symbol, value + 5);
        } else {
          this.changeValue(symbol, 100);
        }
        break;
    }

  }

  checkPageTouch(x, y) {
    for (let i in this._list) {
      const rect = this.itemRect(i);
      if (rect.y > y || rect.x > x) break;
      if (rect.y + rect.height < y) continue;
      if (rect.x + rect.width < x) continue;

      return this.checkItemTouch(i, x, y, rect);
    }
  }

  processCursorMove() {
    if (!this._checkInput) return;

    if (this.isCursorMovable() && this.active) {
      const lastIndex = this.index();
      if (Engine.Input.isRepeated('down')) {
        this.cursorDown(Engine.Input.isTriggered('down'));
      }
      if (Engine.Input.isRepeated('up')) {
        this.cursorUp(Engine.Input.isTriggered('up'));
      }
      if (Engine.Input.isRepeated('right')) {
        this.changeToNextValue();
      }
      if (Engine.Input.isRepeated('left')) {
        this.changeToPreviousValue();
      }
      if (!this.isHandled('pagedown')) {
        if (Engine.Input.isTriggered('pagedown') || Engine.Input.isTriggered('right_trigger')) {
          this.cursorPagedown();
        }
      }
      if (!this.isHandled('pageup')) {
        if (Engine.Input.isTriggered('pageup') || Engine.Input.isTriggered('left_trigger')) {
          this.cursorPageup();
        }
      }
      if (this.index() !== lastIndex) {
        Managers.Sound.playCursor();
      }
    }
  }

  callCancelHandler() {
    if (this._waitingForKey) {
      return;
    }

    if (this._mappingControls) {
      if (this._selectedController) {
        delete this._selectedController;
        super.select(-1);
        this.refresh();
        this.activate();
        return;
      }

      if (this._selectedKey) {
        delete this._selectedKey;
        this.refresh();
        return;
      }

      this._mappingControls = false;
      this.refresh();
      this.activate();
      return;
    }

    if (this._subCategory) {
      delete this._subCategory;

      // If the category changed, then we were not in the subcategory anymore
      if (this.category() == 'mods') {
        this.activate();
        this.refresh();
        return;
      }
    }

    super.callCancelHandler();
  }

  callOkHandler() {
    if (this._waitingForKey) {
      return;
    }

    if (this._index < 0 || this._index >= this._list.length) {
      this.select(0);
      this.activate();
      return;
    }

    const symbol = this.currentSymbol();
    const ext = this.currentExt();

    const isMod = this.category() == 'mods';

    if (ext == 'subcategory') {
      if (isMod) {
        this._subCategory = symbol;
        super.select(0);
        this.activate();
        this.refresh();
        return;
      }
    }

    if (symbol == 'back') {
      if (this._selectedController) {
        delete this._selectedController;
        super.select(-1);
        this.refresh();
        this.activate();
        return;
      }

      if (this._selectedKey) {
        delete this._selectedKey;
        this.refresh();
        return;
      }

      if (this._subCategory) {
        delete this._subCategory;

        // If the category changed, then we were not in the subcategory anymore
        if (this.category() == 'mods') {
          this.activate();
          this.refresh();
          return;
        }
      }

      if (this._mappingControls) {
        this._mappingControls = false;
        this.refresh();
        this.activate();
      } else {
        this.deactivate();
        this._categoryWindow.activate();
      }
      return;
    }

    if (symbol == 'keyboardMapping') {
      this._mappingControls = 'keyboard';
      super.select(0);
      this.refresh();
      this.activate();
      return;
    }

    if (symbol == 'gamepadProfiles') {
      this._mappingControls = 'gamepads';
      super.select(0);
      this.refresh();
      this.activate();
      return;
    }

    if (symbol == 'key_reset') {
      Managers.Config.reloadKeys();
      this.refresh();
      return;
    }

    if (ext == 'controller') {
      this._selectedController = symbol;
      super.select(-1);
      this.refresh();
      this.activate();
      return;
    }

    if (ext == 'key') {
      if (symbol.endsWith('_key1') || symbol.endsWith('_key2')) {
        this._waitingForKey = true;
        this._waitingTime = Utils.getFrameCount(60 * 8);
        this._selectedKey = symbol.replace('key_', '');
        super.select(-1);
      } else {
        this._selectedKey = symbol.replace('key_', '');
        super.select(0);
      }

      this.refresh();
      this.activate();
      return;
    }

    if (isMod && Managers.Content.confirmModSetting(this, symbol)) {
      return;
    }

    super.callOkHandler();
  }

  drawAllItems() {
    if (this._waitingForKey && this._selectedKey) {
      const textWidth = 180 * this._zoomLevel;
      const line1 = Managers.Text.replaceBasicTextVariables('<color:black>') + t("Press a single key to assign to the selected action...");
      const keySymbol = this._selectedKey.replace('_key1', '').replace('_key2', '');
      const index = this._selectedKey.endsWith('_key2') ? 1 : 0;
      const keyDescriptions = this.getKeyDescriptions(keySymbol);
      const action = this.getKeyLabel(keySymbol);
      const currentKey = keyDescriptions[index] || '';

      const rect = this.itemRect(0);
      this.drawTextEx(`<WordWrap>${ line1 }`, rect.x , rect.y, textWidth);

      this.changeTextColor(this.blackColor());
      this.drawText(t("Action") + ':', rect.x, rect.y + rect.height * 4, textWidth, 'left');
      this.drawText(action, rect.x, rect.y + rect.height * 4, textWidth, 'right');
      
      this.drawText(t("Current Key") + ':', rect.x, rect.y + rect.height * 5, textWidth, 'left');
      this.drawText(currentKey, rect.x, rect.y + rect.height * 5, textWidth, 'right');

      let waitingTimeStr = '';
      for (let i = 0; i < Math.ceil(this._waitingTime / 60); i++) {
        waitingTimeStr += '.';
      }
      this.drawText(waitingTimeStr, rect.x, rect.y + rect.height * 7, textWidth, 'center');

      return;
    }

    super.drawAllItems();
  }

  isCursorVisible() {
    if (!super.isCursorVisible()) {
      return false;
    }

    if (this._index < 0 || this._index >= this._list.length) {
      return false;
    }

    return true;
  }

  updateKeyMapping() {
    if (!this._waitingForKey) {
      return;
    }

    const keySymbol = this._selectedKey.replace('_key1', '').replace('_key2', '');
    const index = this._selectedKey.endsWith('_key2') ? 1 : 0;

    if (Engine.Input._lastKeys && Engine.Input._lastKeys.length) {
      for (let i = 0; i < Engine.Input._lastKeys.length; i++) {
        const key = Engine.Input._lastKeys[i];
        if (Engine.Input.protectedKeyCodes[key] !== undefined && Engine.Input.protectedKeyCodes[key] !== keySymbol) {
          continue;
        }

        Managers.Config.changeKey(keySymbol, index, key);
        delete this._waitingForKey;
        delete this._waitingTime;
        this._selectedKey = keySymbol;
        this.refresh();
        return;
      }

    }

    if (this._waitingTime > 0) {
      this._waitingTime--;

      if (Math.ceil(this._waitingTime / 60) != this._oldWaitingTime) {
        this._oldWaitingTime = Math.ceil(this._waitingTime / 60);
        this.refresh();
      }
    } else {
      delete this._waitingForKey;
      delete this._waitingTime;
      this._selectedKey = keySymbol;
      this.refresh();
    }
  }

  update() {
    this.updateKeyMapping();
    super.update();
  }

}

module.exports = Window_SettingsList;
