const Window_Base = require('engine/windows/Window_Base');

let tileSize = 4;

class Window_MapDisplay extends Window_Base {
  initialize(x, y, width, height) {
    let finalY = y;
    let finalHeight = height;
    tileSize = 4;
    const zoom = Graphics.windowZoomLevel;

    const mountainMaps = Managers.Map.getMountainMapIds();
    const mountainTopMaps = Managers.Map.getMountainTopMapIds();
    if (mountainMaps.includes($gameMap._mapId) || mountainTopMaps.includes($gameMap._mapId)) {
      tileSize = 4.25;
    }

    super.initialize(x, finalY, width, finalHeight);

    this._zoomLevel = zoom;
    this._delay = 0;
  }

  enableWindowFrame() {
    return false;
  }

  standardFontSize() {
    return 26;
  }

  resetTextColor() {
    this.changeTextColor('#3d2b25');
  }

  resetFontSettings() {
    super.resetFontSettings();
    this.contents.outlineColor = '#f4dea8';
    this.contents.outlineWidth = 0;
  }

  lineHeight() {
    return this.standardFontSize();
  }

  refresh() {
    if (this.contents) {
      this.contents.clear();
      this.drawContent();
    }
  }

  drawBitmap(bitmap, originalX, originalY, zoomX, zoomY) {
    this.requireBitmap(bitmap);
    if (!bitmap.isReady()) return;

    const width = bitmap._canvas.width;
    const height = bitmap._canvas.height;
    const drawWidth = width * (zoomX || 1) * this._zoomLevel;
    const drawHeight = height * (zoomY || 1) * this._zoomLevel;

    const x = originalX * this._zoomLevel;
    const y = originalY * this._zoomLevel;

    this.contents.bltImage(bitmap, 0, 0, width, height, x, y, drawWidth, drawHeight);
  }

  getMapName() {
    const townMaps = Managers.Map.getTownMapIds();
    const forestMaps = Managers.Map.getForestMapIds();
    const mountainMaps = Managers.Map.getMountainMapIds();
    const mountainTopMaps = Managers.Map.getMountainTopMapIds();
    const farmMaps = Managers.Map.getFarmMapIds();

    const mapId = $gameMap._mapId;

    if (townMaps.indexOf(mapId) >= 0) {
      return 'town';
    }

    if (forestMaps.indexOf(mapId) >= 0) {
      return 'forest';
    }

    if (mountainMaps.indexOf(mapId) >= 0) {
      return 'mountain';
    }

    if (mountainTopMaps.indexOf(mapId) >= 0) {
      return 'mountainTop';
    }

    if (farmMaps.indexOf(mapId) >= 0) {
      return 'town';
    }

    return 'unknown';
  }

  drawPlayerPin(x, y) {
    const playerSprite = Managers.Player.generatePlayerSpriteName();

    const playerPin = Managers.Images.loadPicture(`maps/player/pin-${ playerSprite }`);
    if (!playerPin.isReady()) {
      this.requireBitmap(playerPin);
      return;
    }

    this.drawBitmap(playerPin, x - playerPin.width, y - playerPin.height, 2, 2);
  }

  _drawCharacterPin(villagerName, x, y) {
    const pin = Managers.Images.loadPicture(`maps/pin-${ villagerName }`);
    if (!pin.isReady()) {
      this._requiresRefresh = true;
      // this.requireBitmap(pin);
      return;
    }

    this.drawBitmap(pin, x - (pin.width * 0.75), y - (pin.height * 0.75), 1.5, 1.5);
  }

  getPositionForMap(mapId, characterName, mapBeingShown) {
    let pos;

    if (mapBeingShown === Maps.ORANGE_TOWN) {
      switch(mapId) {
        case Maps.GENERAL_STORE:
          pos = { x : 5 * tileSize, y: 34 * tileSize};
          break;
        case Maps.BLACKSMITH:
        case Maps.BLACKSMITH_ROOMS:
          pos = { x : 3 * tileSize, y: 20 * tileSize};
          break;
        case Maps.BENJAMINS_HOUSE:
          pos = { x : 46 * tileSize, y: 16 * tileSize};
          break;
        case Maps.CLINIC:
          pos = { x : 28 * tileSize, y: 3 * tileSize};
          break;
        case Maps.LIBRARY:
          pos = { x : 20 * tileSize, y: 34 * tileSize};
          break;
        case Maps.MIAS_HOUSE:
        case Maps.ANIMAL_CLINIC:
          pos = { x : 20 * tileSize, y: 4 * tileSize};
          break;
        case Maps.LUCAS_HOUSE:
          pos = { x : 65 * tileSize, y: 32 * tileSize};
          break;
        case Maps.BRITTANYS_FARM:
          pos = { x : 12 * tileSize, y: 68 * tileSize};
          break;
        case Maps.CARPENTER_HOUSE:
          pos = { x : 69 * tileSize, y: 55 * tileSize};
          break;
        case Maps.RESTAURANT:
        case Maps.INN:
          pos = { x : 41 * tileSize, y: 34 * tileSize};
          break;
        case Maps.CITY_HALL:
          pos = { x : 60 * tileSize, y: 23 * tileSize};
          break;
        case Maps.RORYS_FARM:
          pos = { x : 28 * tileSize, y: 25 * tileSize};
          break;
        case Maps.AMANDAS_HOUSE:
          pos = { x : 56 * tileSize, y: 32 * tileSize};
          break;
        case Maps.ILDAS_HOUSE:
          pos = { x : 63 * tileSize, y: 47 * tileSize};
          break;
        case Maps.GABRIELS_HOUSE:
          pos = { x : 59 * tileSize, y: 68 * tileSize};
          break;
        case Maps.KARLS_HOUSE:
          pos = { x : 44 * tileSize, y: 57 * tileSize};
          break;
        case Maps.DUKES_HOUSE:
          pos = { x : 31 * tileSize, y: 72 * tileSize};
          break;
        case Maps.PETTING_ZOO:
        case Maps.BEACH:
          pos = { x : 63 * tileSize, y: 81 * tileSize};
          break;
        case Maps.FARM:
        case Maps.FARM_CENTER:
        case Maps.HOME:
        case Maps.HOME_2:
        case Maps.HOME_3:
        case Maps.HOME_3B:
        case Maps.COOP:
        case Maps.BARN:
          pos = { x : 3 * tileSize, y: 81 * tileSize};
          break;
      }
    }

    if (pos) {
      const count = (this._mapCounts[mapId] || 0);
      this._mapCounts[mapId] = count+1;

      if (characterName == 'Player') {
        pos.y += 3;
      }

      pos.x += (count % 3) * 3 * tileSize;
      if (count >= 3) {
        pos.x += 1 * tileSize;
        pos.y += 3 * tileSize;
      }

      return pos;
    }
    return { x: 0, y : 0};
  }

  drawVillagerPin(villagerName, mainMapId, startX, startY) {
    if (!Managers.Relationship.isCharacterKnown(villagerName)) return;

    const position = Managers.Villagers.getVillagerPosition(villagerName);
    if (!position) return;

    const { mapId, x, y} = position;

    if (mapId == mainMapId) {
      const villagerX = x * tileSize;
      const villagerY = y * tileSize;

      this._drawCharacterPin(villagerName.toLowerCase(), startX + villagerX, startY + villagerY);
    } else {
      const { x, y } = this.getPositionForMap(mapId, villagerName, mainMapId);
      if (x && y) {
        this._drawCharacterPin(villagerName.toLowerCase(), startX + x, startY + y);
      }
    }
  }

  drawTownMapContent(x, y) {
    const startX = x + 158;
    const startY = y + 64;

    if ($gameMap._mapId == Maps.ORANGE_TOWN) {
      const playerX = $gamePlayer.x * tileSize;
      const playerY = $gamePlayer.y * tileSize;

      this.drawPlayerPin(startX + playerX, startY + playerY);
    } else {
      const { x, y } = this.getPositionForMap($gameMap._mapId, 'Player');
      if (x && y) {
        this.drawPlayerPin(startX + x, startY + y);
      }
    }

    this.drawVillagerPin('Amanda', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Annie', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Benjamin', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Billy', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Bonnie', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Brittany', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Chloe', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Cindy', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Curupira', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Devin', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Duke', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Gabriel', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Gary', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Ilda', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Julia', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Kaori', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Karl', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Lucas', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Martin', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Mia', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Nathalia', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Phi', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Prince', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Raphael', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Richard', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Rory', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Sarah', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Serge', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Shinnosuke', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Stella', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Thalia', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Viktor', Maps.ORANGE_TOWN, startX, startY);
    this.drawVillagerPin('Yumiko', Maps.ORANGE_TOWN, startX, startY);
  }

  drawForestMapContent(x, y) {
    const startX = x + 155;
    const startY = y + 75;

    if ($gameMap._mapId == Maps.FOREST) {
      const playerX = $gamePlayer.x * tileSize;
      const playerY = $gamePlayer.y * tileSize;

      this.drawPlayerPin(startX + playerX, startY + playerY);
    } else {
      const { x, y } = this.getPositionForMap($gameMap._mapId, 'Player');
      if (x && y) {
        this.drawPlayerPin(startX + x, startY + y);
      }
    }

    this.drawVillagerPin('Amanda', Maps.FOREST, startX, startY);
    this.drawVillagerPin('Gary', Maps.FOREST, startX, startY);
    this.drawVillagerPin('Karl', Maps.FOREST, startX, startY);
    this.drawVillagerPin('Sarah', Maps.FOREST, startX, startY);
  }

  drawMountainMapContent(x, y) {
    const startX = x + 174;
    const startY = y + 61;

    if ($gameMap._mapId == Maps.MOUNTAIN) {
      const playerX = $gamePlayer.x * tileSize;
      const playerY = $gamePlayer.y * tileSize;

      this.drawPlayerPin(startX + playerX, startY + playerY);
    } else {
      const { x, y } = this.getPositionForMap($gameMap._mapId, 'Player');
      if (x && y) {
        this.drawPlayerPin(startX + x, startY + y);
      }
    }

    this.drawVillagerPin('Gabriel', Maps.MOUNTAIN, startX, startY);
    this.drawVillagerPin('Sarah', Maps.MOUNTAIN, startX, startY);
    this.drawVillagerPin('Martin', Maps.MOUNTAIN, startX, startY);
  }

  drawMountainTopMapContent(x, y) {
    const startX = x + 174;
    const startY = y + 182;

    if ($gameMap._mapId == Maps.MOUNTAIN_TOP) {
      const playerX = $gamePlayer.x * tileSize;
      const playerY = $gamePlayer.y * tileSize;

      this.drawPlayerPin(startX + playerX, startY + playerY);
    } else {
      const { x, y } = this.getPositionForMap($gameMap._mapId, 'Player');
      if (x && y) {
        this.drawPlayerPin(startX + x, startY + y);
      }
    }

    this.drawVillagerPin('Sarah', Maps.MOUNTAIN_TOP, startX, startY);
  }

  drawMapContent(mapName, x, y) {
    this._mapCounts = {};

    switch(mapName) {
      case 'town':
        this.drawTownMapContent(x, y);
        break;
      case 'forest':
        this.drawForestMapContent(x, y);
        break;
      case 'mountain':
        this.drawMountainMapContent(x, y);
        break;
      case 'mountainTop':
        this.drawMountainTopMapContent(x, y);
        break;
    }
  }

  drawContent() {
    const mapName = this.getMapName();
    const bitmap = Managers.Images.loadPicture(`maps/${ mapName }`);

    if (!bitmap.isReady()) {
      this.requireBitmap(bitmap);
      return;
    }

    const width = bitmap.width * this._zoomLevel;
    const height = bitmap.height * this._zoomLevel;
    const x = (Graphics.width - width) / 2 / this._zoomLevel;
    const y = (Graphics.height - height) / 2 / this._zoomLevel;

    this.drawBitmap(bitmap, x, y);
    this.drawMapContent(mapName, x, y);
    this.drawCloseButton(y > 0 ? 0 : (-y / this._zoomLevel));
  }

  drawCloseButton(y) {
    const bitmap = Managers.Images.loadPicture('menu/exit_menu2');
    if (!bitmap.isReady()) {
      this.requireBitmap(bitmap);
      return;
    }

    const x = (this.width / this._zoomLevel) - bitmap.width;
    this._closeButtonY = y;

    this.drawBitmap(bitmap, x, y);
  }

  isMouseOverExitButton() {
    const bitmap = Managers.Images.loadPicture('menu/exit_menu2');
    if (!bitmap.isReady()) {
      return false;
    }

    const mouseX = (TouchInput.mouseX - this.x) / this._zoomLevel;
    const mouseY = (TouchInput.mouseY - this.y) / this._zoomLevel;
    
    const x = (this.width / this._zoomLevel) - bitmap.width;

    if (mouseX < x) return false;
    if (mouseY > (this._closeButtonY + bitmap.height)) return false;

    return true;
  }

  standardPadding() {
    return 0;
  }

  update() {
    this._requiresRefresh = false;
    super.update();

    if (this._delay > 0) {
      this._delay--;
      return;
    }

    const isMouseOnExit = this.isMouseOverExitButton();
    if (isMouseOnExit != this._oldMouseOverExitButton) {
      this.refresh();
      this._delay = 30;
      this._oldMouseOverExitButton = isMouseOnExit;
    }

    this._delay = 12;

    if (this._requiresRefresh) {
      setTimeout(() => {
        this.refresh();
      }, 100);
    }
  }
}

module.exports = Window_MapDisplay;