const Window_Selectable = require('engine/windows/Window_Selectable');
const FriendWindowHelper = require('../contexts/FriendWindowHelper');

class Window_FriendList extends Window_Selectable {
  initialize(x, y, w, h) {
    this.makeItemList();
    
    this._layoutBitmap = Managers.Images.loadPicture('menu/friend_layout_dark');
    this._layoutSelectedBitmap = Managers.Images.loadPicture('menu/friend_layout_selected_dark');
    this._redHeartBitmap = Managers.Images.loadPicture('menu/red_heart_dark');
    this._halfRedHeartBitmap = Managers.Images.loadPicture('menu/half_red_heart_dark');
    
    this._arrowDownBitmap = Managers.Images.loadPicture('menu/arrow_down');
    this._arrowUpBitmap = Managers.Images.loadPicture('menu/arrow_up');

    super.initialize(x, y, w, h);
    this.refresh();
  }

  makeItemList() {
    this._friendList = Managers.Relationship.getMenuVillagers();

    FriendWindowHelper.requestPortraitList(this._friendList);
  }

  isCursorVisible() {
    return false;
  }

  enableWindowFrame() {
    return false;
  }

  selectedFriend() {
    return this._friendList[this._index];
  }

  drawItem(index) {
    if (index < 0) return;

    const rect = this.itemRect(index);
    const zoom = Graphics.windowZoomLevel;

    FriendWindowHelper.drawFriendBadge(this, rect, this._friendList[index], zoom, true, index === this._index);
  }

  drawAllItems() {
    // If the data is already outdated, don't bother drawing it
    if (this._needsRefresh) return;
    const zoom = Graphics.windowZoomLevel;
    const maxIdx = this._friendList.length -1;
    const rect = this.itemRect(maxIdx);

    if (maxIdx > 0 && rect.y + rect.height > this.contentsHeight()) {
      if (this._arrowUpBitmap.isReady()) {
        this.drawBitmap(this._arrowUpBitmap, 103, 0, zoom);
      } else {
        this.requireBitmap(this._arrowUpBitmap);
      }
    }

    super.drawAllItems();

    if (maxIdx > 0 && rect.y + rect.height > this.contentsHeight()) {
      if (this._arrowDownBitmap.isReady()) {
        this.drawBitmap(this._arrowDownBitmap, 103, 205, zoom);
      } else {
        this.requireBitmap(this._arrowDownBitmap);
      }
    }
  }

  horizontalMargin() {
    return 10 * Graphics.windowZoomLevel;
  }

  maxCols() {
    return 3;
  }

  horizontalPadding() {
    return 0 * Graphics.windowZoomLevel;
  }

  verticalPadding() {
    return 0 * Graphics.windowZoomLevel;
  }

  verticalMargin() {
    return 8 * Graphics.windowZoomLevel;
  }

  bottomMargin() {
    return 0;
  }

  maxItems() {
    return this._friendList.length;
  }

  itemHeight() {
    return 62 * Graphics.windowZoomLevel;
  }

  standardFontSize() {
    return Utils.convertFontSize(26, Graphics.windowZoomLevel);
  }

  verticalItemSpacing() {
    return 4 * Graphics.windowZoomLevel;
  }

  resetTextColor() {
    this.changeTextColor(this.blackColor());
  }

  resetFontSettings() {
    super.resetFontSettings();
    this.contents.outlineWidth = 0;
  }

  lineHeight() {
    return this.standardFontSize();
  }

  requireBitmap(bitmap) {
    if (bitmap && !bitmap.isReady()) {
      this._needsRefresh = true;
    }
  }

  update() {
    super.update();
    if (this._needsRefresh) {
      this.refresh();
    }
  }

  refresh() {
    this._needsRefresh = false;

    this.requireBitmap(this._layoutBitmap);
    this.requireBitmap(this._layoutSelectedBitmap);
    this.requireBitmap(this._redHeartBitmap);
    this.requireBitmap(this._halfRedHeartBitmap);
    this._refreshExpected = this._needsRefresh;
    super.refresh();
  }

  select(index) {
    if (index != this._index) {
      super.select(index);
      super.refresh();
    }

    if (this._detailsWindow) {
      this._detailsWindow.setFriendName(this._friendList[index] || '');
    }
  }

  setDetailsWindow(detailsWindow) {
    this._detailsWindow = detailsWindow;
  }

  itemWidth() {
    const leftMargin = 10 * Graphics.windowZoomLevel;
    const rightMargin = 6 * Graphics.windowZoomLevel;

    const totalWidth = this.width - (this.paddingH * 2) - (leftMargin + rightMargin) + this.spacing();

    return Math.floor(totalWidth / this.maxCols() - this.spacing());
  }

  selectPrior() {
    const list = this._friendList;
    const maxIdx = list.length -1;

    if (this._index <= 0) {
      this.select(maxIdx);
      return;
    }

    const index = this._index - this.maxCols();
    if (index < 0) {
      this.select(0);
    } else {
      this.select(index);
    }
  }

  selectNext() {
    const list = this._friendList;
    const maxIdx = list.length -1;

    if (this._index === maxIdx) {
      this.select(0);
      return;
    }

    const index = this._index + this.maxCols();

    if (index > maxIdx) {
      this.select(maxIdx);
    } else {
      this.select(index);
    }
  }

  updateTouch() {
    const zoom = Graphics.windowZoomLevel;
    const mouseX = (TouchInput.mouseX - this.x) / zoom;
    const mouseY = (TouchInput.mouseY - this.y) / zoom;

    if (mouseY < 0) return;
    if (mouseX < 0) return;

    const right = this.width / zoom;
    if (mouseX > right) return;

    const bottom = this.height / zoom;
    if (mouseY > bottom) return;


    if (mouseX >= 100 && mouseX <= 115) {
      if (mouseY <= 12) {
        this.selectPrior();
        return true;
      }

      if (mouseY >= 203) {
        this.selectNext();
        return true;
      }
    }
  }

  cursorPagedown() {
  }

  cursorPageup() {
  }
}

module.exports = Window_FriendList;