const Window_BaseDiaryContent = require('./BaseContent');
const AnimalWindowHelper = require('../contexts/AnimalWindowHelper');

class Window_DataDetailedContent extends Window_BaseDiaryContent {
  initialize(x, y, width, height) {
    super.initialize(x, y, width, height);

    this._profileBackgroundBitmap = Managers.Images.loadPicture('menu/profile_background_1');
    this._emptySlotBitmap = Managers.Images.loadPicture('menu/empty_slot');
    this._staminaSlotBitmap = Managers.Images.loadPicture('menu/stamina_slot');
    this._staminaIconBitmap = Managers.Images.loadPicture('menu/stamina_icon');
    this._fatigueSlotBitmap = Managers.Images.loadPicture('menu/fatigue_slot');
    this._fatigueIconBitmap = Managers.Images.loadPicture('menu/fatigue_icon');

    this._friendBackgroundBitmap = Managers.Images.loadPicture('menu/friend_layout');
    this._redHeartBitmap = Managers.Images.loadPicture('menu/red_heart');
    this._halfRedHeartBitmap = Managers.Images.loadPicture('menu/half_red_heart');

  }

  standardFontSize() {
    return Utils.convertFontSize(24, this._zoomLevel);
  }

  getFullData(saveFilePath) {
    if (!this._saveFilePath) return Engine.Data.makeSaveContents();

    try {
      const cachedData = Engine.Data.getCachedSaveData(this._saveFilePath);
      if (cachedData) {
        return cachedData;
      }

      const json = Managers.Storage.loadSaveFile(this._saveFilePath);
      return JsonEx.parse(json);
    }
    catch(e) {
      console.error(e);
    }

    return false;
  }

  drawHealthData(health) {
    const maxStamina = Math.min(20, Math.round(health.maxStamina / 10));
    const staminaLevel = Math.floor(health.stamina * 100 / health.maxStamina);

    const stamina = staminaLevel / 10 * maxStamina;

    const maxFatigue = Math.min(20, Math.round(health.maxFatigue / 10));
    const fatigueLevel = Math.floor(health.fatigue * 100 / health.maxFatigue);
    const fatigue = (100 - fatigueLevel) / 10 * maxFatigue;

    this.drawBitmap(this._staminaIconBitmap, 81, 27);

    let x = 99;
    for (let staminaSlotIndex = 0; staminaSlotIndex < maxStamina; staminaSlotIndex++) {
      let bitmap = stamina > staminaSlotIndex ? this._staminaSlotBitmap : this._emptySlotBitmap;
      this.drawBitmap(bitmap, x, 33);
      x += 5;
    }

    this.drawBitmap(this._fatigueIconBitmap, 81, 46);
    x = 99;
    for (let fatigueSlotIndex = 0; fatigueSlotIndex < maxFatigue; fatigueSlotIndex++) {
      let bitmap = fatigue > fatigueSlotIndex ? this._fatigueSlotBitmap : this._emptySlotBitmap;
      this.drawBitmap(bitmap, x, 55);
      x += 5;
    }
  }

  drawPlayerSprite(playerData) {
    let spriteName = '';
    const { sprite, hairType, skinType } = playerData;

    if (!sprite || !hairType || !skinType) return;

    spriteName += sprite;
    spriteName += hairType;
    spriteName += skinType;

    // Main Sprite:
    const bitmap = Managers.Images.loadCharacterIcon(spriteName, 0, this._zoomLevel);
    if (bitmap.isReady()) {
      this.drawCharacterIcon(spriteName, 0, 34 * this._zoomLevel, 14 * this._zoomLevel, this._zoomLevel);
    } else {
      setTimeout(() => {
        this.refresh();
      }, 80);
    }
  }

  drawHearts(heartsNum, bgX, bgY) {
    if (heartsNum >= 0.5) {
      let heartBitmap = heartsNum >= 1 ? this._redHeartBitmap : this._halfRedHeartBitmap;

      this.requireBitmap(this._redHeartBitmap);
      this.requireBitmap(this._halfRedHeartBitmap);

      if (this._redHeartBitmap.isReady() && this._halfRedHeartBitmap.isReady()) {
        this.drawBitmap(heartBitmap, bgX + 35, bgY + 29);

        if (heartsNum >= 1.5) {
          heartBitmap = heartsNum >= 2 ? this._redHeartBitmap : this._halfRedHeartBitmap;

          this.drawBitmap(heartBitmap, bgX + 40, bgY + 22);

          if (heartsNum >= 2.5) {
            heartBitmap = heartsNum >= 3 ? this._redHeartBitmap : this._halfRedHeartBitmap;

            this.drawBitmap(heartBitmap, bgX + 42, bgY + 14);

            if (heartsNum >= 3.5) {
              heartBitmap = heartsNum >= 4 ? this._redHeartBitmap : this._halfRedHeartBitmap;

              this.drawBitmap(heartBitmap, bgX + 40, bgY + 6);

              if (heartsNum >= 4.5) {
                heartBitmap = heartsNum >= 5 ? this._redHeartBitmap : this._halfRedHeartBitmap;

                this.drawBitmap(heartBitmap, bgX + 35, bgY);
              }
            }
          }
        }
      }
    }
  }

  drawFriend(bgX, bgY, friendName, friendship) {
    if (!friendName) return;

    const textY = bgY + 34;
    this.drawText(friendName, bgX * this._zoomLevel, textY * this._zoomLevel, 47 * this._zoomLevel, 'center');

    const faceBitmap = Managers.Images.loadPortrait('mini/' + friendName.toLowerCase());
    this.requireBitmap(faceBitmap);

    if (faceBitmap.isReady()) {
      this.drawBitmap(faceBitmap, bgX + 2, bgY - 10);
    }

    const heartsNum = Managers.Relationship.friendshipToHeartNumber(friendship);
    this.drawHearts(heartsNum, bgX, bgY);
  }

  drawAnimal(bgX, bgY, animalData) {
    const textY = bgY + 34;
    this.drawText(animalData._creatureName, bgX * this._zoomLevel, textY * this._zoomLevel, 47 * this._zoomLevel, 'center');

    const faceBitmap = Managers.Images.loadPicture('menu/animals/' + animalData.type);
    if (faceBitmap.isReady()) {
      AnimalWindowHelper.drawAnimalFace(this, animalData, bgX * this._zoomLevel, bgY * this._zoomLevel, this._zoomLevel, faceBitmap);
    } else {
      this.requireBitmap(faceBitmap);
    }

    const heartsNum = Managers.Relationship.friendshipToHeartNumber(animalData.friendship || 0);
    this.drawHearts(heartsNum, bgX, bgY);
  }

  drawBestFriends(fullData) {
    const { relationship } = fullData;
    if (!relationship) return;

    const { knownVillagers, friendship } = relationship;
    if (!knownVillagers) return;
    if (!friendship) return;

    const bitmap = this._friendBackgroundBitmap;
    if (!bitmap.isReady()) {
      this.requireBitmap(bitmap);
      return;
    }

    const visibleVillagers = knownVillagers.filter((npc) => {
      const data = Managers.Villagers.getActorData(npc);
      if (!data) return false;
      return Boolean(data.showOnFriendList);
    });

    visibleVillagers.sort((a, b) => {
      if (friendship[a] === undefined) {
        friendship[a] = 0;
      }
      if (friendship[b] === undefined) {
        friendship[b] = 0;
      }

      return friendship[b] - friendship[a];
    });

    let x = 20;
    const y = 101;
    for (let i = 0; i < 3; i++) {
      this.drawBitmap(bitmap, x, y);

      if (i < visibleVillagers.length) {
        this.drawFriend(x, y, visibleVillagers[i], friendship[visibleVillagers[i]]);
      }

      x += 62;
    }
  }

  drawBestAnimals(creaturesData) {
    if (!creaturesData) return;

    const { creatures } = creaturesData;
    if (!creatures) return;

    const bitmap = this._friendBackgroundBitmap;
    if (!bitmap.isReady()) {
      this.requireBitmap(bitmap);
      return;
    }

    const animals = [];
    const filterFn = (data) => !!data._creatureName;

    for (const mapId in creatures) {
      if (isNaN(mapId)) continue;

      for (let i = 0; i < creatures[mapId].length; i++) {
        if (!creatures[mapId][i]) continue;
        if (!filterFn(creatures[mapId][i])) continue;

        animals.push(creatures[mapId][i]);
      }
    }

    animals.sort(Managers.Creatures.animalSorterFn);

    let x = 20;
    const y = 165;
    for (let i = 0; i < 3; i++) {
      this.drawBitmap(bitmap, x, y);

      if (i < animals.length) {
        this.drawAnimal(x, y, animals[i]);
      }

      x += 62;
    }
  }

  drawContent() {
    this.drawBitmap(this._profileBackgroundBitmap, 22, 11);

    const fullData = this.getFullData();
    if (!fullData) return;

    const { playerName, gold, health, playerData, currentDateTime } = fullData;

    if (playerName) {
      this.drawLabel(82, 10, playerName);
    } else if (this._saveFilePath) {
      this.drawLabel(82, 10, Utils.getFileNameFromPath(this._saveFilePath).replace('.ffsave', ''));
    }

    if (gold) {
      this.drawLabel(33, 67, Managers.Text.formatMoney(gold));
    }

    if (health) {
      this.drawHealthData(health);
    }

    if (playerData) {
      this.drawPlayerSprite(playerData);
    }

    // Labels:
    this.drawLabel(5, 88, t("Friends"));
    this.drawLabel(5, 152, t("Animals"));
    // this.drawLabel(86, 259, t("Tomorrow"), 46 * this._zoomLevel, 'center');

    if (currentDateTime) {
      const monthString = t(Managers.Time.getMonthName(currentDateTime.month));
      this.changeTextColor('#f4dea8');
      this.contents.outlineWidth = 0;
      this.drawLabel(17, 225, monthString, 50 * this._zoomLevel, 'center');

      const yearString = Managers.Time.getDateDescription(currentDateTime, "Year {year}");
      this.changeTextColor('#f4dea8');
      this.contents.outlineWidth = 0;
      this.drawLabel(17, 235, yearString, 50 * this._zoomLevel, 'center');

      const weekDay = Managers.Time.calculateWeekDay(currentDateTime);
      const dayNameString = t(Managers.Time.getWeekDayLabel(weekDay));
      this.changeTextColor('#f4dea8');
      this.contents.outlineWidth = 0;
      this.drawLabel(155, 225, dayNameString, 50 * this._zoomLevel, 'center');

      const dayString = Managers.Time.getDateDescription(currentDateTime, "Day {day}");
      this.changeTextColor('#f4dea8');
      this.contents.outlineWidth = 0;
      this.drawLabel(155, 235, dayString, 50 * this._zoomLevel, 'center');
    }

    this.drawBestFriends(fullData);
    this.drawBestAnimals(fullData.creatures);
  }

  setSaveFilePath(saveFilePath) {
    if (this._saveFilePath !== saveFilePath) {
      this._saveFilePath = saveFilePath;
      this.refresh();
    }
  }
}

module.exports = Window_DataDetailedContent;