const Window_BaseDiaryContent = require('./BaseContent');
const Window_DataCommand = require('./DataCommand');

class Window_DataContent extends Window_BaseDiaryContent {
  initialize(x, y, width, height) {
    super.initialize(x, y, width, height);

    this._loadSuccess = false;

    this.loadPictures();

    this.setInitialMode();
    this._index = 0;
    this._topIndex = 0;

    this.updateSaves();
    this._currentSaves = [];
    this._onEmptyListAction = 'saveMode';
  }

  loadModePictures() {
    this._leftTitle = Managers.Images.loadPicture('menu/left_title');
    this._rightTitle = Managers.Images.loadPicture('menu/right_title');
  }

  loadSlotPicutres() {
    this._selectedSlotBitmap = Managers.Images.loadPicture('menu/slot_1');
    this._regularSlotBitmap = Managers.Images.loadPicture('menu/slot_2');
    this._newSlotBitmap = Managers.Images.loadPicture('menu/slot_3');
  }

  loadPictures() {
    this.loadModePictures();
    this.loadSlotPicutres();

    this._choiceTypeBitmap = Managers.Images.loadPicture('menu/choices/choice_type_1');
    this._okBitmap = Managers.Images.loadPicture('menu/choices/ok');
    this._okBitmapSelected = Managers.Images.loadPicture('menu/choices/ok_selected');
    this._deleteBitmap = Managers.Images.loadPicture('menu/choices/nop');
    this._deleteBitmapSelected = Managers.Images.loadPicture('menu/choices/nop_selected');
    this._arrowDownBitmap = Managers.Images.loadPicture('menu/arrow_down');
    this._arrowUpBitmap = Managers.Images.loadPicture('menu/arrow_up');

    this._stampFontBitmap = Managers.Images.loadPicture('menu/stamp_font');
  }

  setInitialMode() {
    this._mode = 'save';
  }

  updateSaves() {
    this._allSaves = Engine.Data.getAllExistingSaves(['crash.ffsave']);

    this._allSaves.sort((a, b) => {
      const dataA = Engine.Data.getCachedSaveData(a);
      if (!dataA) return 0;
      const dataB = Engine.Data.getCachedSaveData(b);
      if (!dataB) return 0;

      const timeA = dataA.info ? dataA.info.timestamp : 0;
      const timeB = dataB.info ? dataB.info.timestamp : 0;

      if (timeA == timeB) {
        if (dataA.info && dataB.info) {
          return dataB.info.gold - dataA.info.gold;
        }
      }

      return timeB - timeA;
    });

    this._savesToOverwrite = this._allSaves.filter((savePath) => {
      const fileName = Utils.getFileNameFromPath(savePath);
      if (fileName == 'auto.ffsave' || fileName == 'quick.ffsave') {
        return false;
      }

      return true;
    });
  }

  drawModes() {
    let titleBitmap;
    let titleX = 0;
    const titleY = 17;
    let saveColor = this.darkBaseColor();
    let loadColor = this.darkBaseColor();

    if (this._mode == 'save') {
      titleBitmap = this._leftTitle;
      saveColor = this.baseColor();
    } else {
      titleBitmap = this._rightTitle;
      loadColor = this.baseColor();
      titleX = 128;
    }

    this.drawBitmap(titleBitmap, titleX, titleY);

    let x = 17;
    let y = 18;
    const w = 100 * this._zoomLevel;
    let h = Utils.convertFontSize(56, this._zoomLevel);

    this.changeTextColor(saveColor);
    this.contents.fontSize = h;
    this.contents.outlineColor = saveColor;
    this.contents.outlineWidth = 0;
    this.drawLabel(x, y, t("Save"), w, 'right');

    x = 130;
    this.changeTextColor(loadColor);
    this.contents.fontSize = h;
    this.contents.outlineColor = loadColor;
    this.contents.outlineWidth = 0;
    this.drawLabel(x, y, t("Load"), w, 'left');
  }

  drawContent() {
    this.requireBitmap(this._stampFontBitmap);
    this.drawModes();

    const saves = this._mode == 'save' ? this._savesToOverwrite : this._allSaves;
    this._currentSaves = saves;

    if (saves.length > 0) {
      if (this._arrowUpBitmap.isReady()) {
        this.drawBitmap(this._arrowUpBitmap, 120, 45);
      } else {
        this.requireBitmap(this._arrowUpBitmap);
      }
    }

    let y = 57;
    let drewLast = false;

    for (let i = this._topIndex; i < saves.length; i++) {
      if (i == this._index) {
        if (y >= 202) {
          break;
        }

        this.drawSelectedSlot(i, y);

        y += 48;
      } else {
        if (y >= 225) {
          break;
        }

        this.drawRegularSlot(i, y);
        y += 25;
      }

      drewLast = i == saves.length -1;
    }

    if (this._mode == 'save' && y < 225 && (drewLast || saves.length === 0)) {
      this.drawNewSlot(saves.length, y, this._index == this._currentSaves.length);
      y += 25;
    }

    if (saves.length > 0) {
      if (this._arrowDownBitmap.isReady()) {
        this.drawBitmap(this._arrowDownBitmap, 120, 236);
      } else {
        this.requireBitmap(this._arrowDownBitmap);
      }
    }

    this.drawBitmap(this._choiceTypeBitmap, 189, 241);

    this._oldMouseOverOkButton = this.isMouseOverOkButton();
    this._oldMouseOverDeleteButton = this.isMouseOverDeleteButton();

    if (this._oldMouseOverDeleteButton) {
      this.drawBitmap(this._deleteBitmapSelected, 196, 240);
    } else {
      this.drawBitmap(this._deleteBitmap, 196, 240);
    }

    if (this._oldMouseOverOkButton) {
      this.drawBitmap(this._okBitmapSelected, 212, 240);
    } else {
      this.drawBitmap(this._okBitmap, 212, 240);
    }
  }

  getSaveDataHeader(saveData) {
    if (saveData.info) {
      return saveData.info;
    }

    const newHeader = {};

    newHeader.playerName = saveData.playerName || t('Unknown');
    newHeader.fileName = saveData.fileName;
    newHeader.playtime = t('Unknown');
    newHeader.timestamp = false;
    newHeader.currentDateTime = false;

    return newHeader;
  }

  drawSelectedSlot(index, y) {
    this.requireBitmap(this._selectedSlotBitmap);
    if (!this._selectedSlotBitmap.isReady()) return;

    this.drawBitmap(this._selectedSlotBitmap, 17, y);
    this.drawSlotNumber(index, 21, y + 9);

    const savePath = this._currentSaves[index];
    if (!savePath) {
      return;
    }

    let slotBitmap;
    let slotLabel;

    const saveFileName = Utils.getFileNameFromPath(savePath);

    if (saveFileName == 'auto.ffsave') {
      slotLabel = t("Auto Save");
      slotBitmap = Managers.Images.loadPicture('menu/auto_slot');
    } else if (saveFileName == 'quick.ffsave') {
      slotLabel = t("Quick Save");
      slotBitmap = Managers.Images.loadPicture('menu/quick_slot');
    }

    if (slotBitmap && slotLabel) {
      if (slotBitmap.isReady()) {
        this.drawBitmap(slotBitmap, 169, y + 3);
        this.contents.fontSize = Utils.convertFontSize(20, this._zoomLevel);
        this.drawLabel(172, y - 3, slotLabel, 35 * this._zoomLevel, 'left');
      } else {
        this.requireBitmap(slotBitmap);
      }
    }
    const saveData = Engine.Data.getCachedSaveData(savePath);
    if (!saveData) return;

    const saveHeader = this.getSaveDataHeader(saveData);
    const { playerName, fileName, playtime, timestamp, currentDateTime } = saveHeader;

    this.changeTextColor(this.baseColor());
    this.contents.outlineColor = this.baseColor();
    this.contents.outlineWidth = 0;
    this.contents.fontSize = Utils.convertFontSize(42, this._zoomLevel);

    this.drawLabel(65, y, playerName || fileName || t('Player'));

    if (playtime) {
      this.changeTextColor(this.baseColor());
      this.contents.outlineColor = this.darkBaseColor();
      this.contents.outlineWidth = 1;
      this.contents.fontSize = Utils.convertFontSize(26, this._zoomLevel);

      this.drawLabel(65, y + 12, t('Play Time') + ': ' + playtime);
    }

    if (timestamp) {
      const date = new Date(timestamp);
      const dateStr = date.toLocaleString();

      this.changeTextColor(this.baseColor());
      this.contents.outlineColor = this.darkBaseColor();
      this.contents.outlineWidth = 1;
      this.contents.fontSize = Utils.convertFontSize(24, this._zoomLevel);

      this.drawLabel(65, y + 27, dateStr, 162 * this._zoomLevel, 'right');
    }

    if (saveFileName) {
      this.changeTextColor(this.baseColor());
      this.contents.outlineColor = this.darkBaseColor();
      this.contents.outlineWidth = 1;
      this.contents.fontSize = Utils.convertFontSize(20, this._zoomLevel);
      this.drawLabel(65, y + 20, saveFileName, 166 * this._zoomLevel, 'left');
    }

    if (currentDateTime) {
      const { month } = currentDateTime;
      let bitmap;
      switch(month) {
        case 1:
          bitmap = Managers.Images.loadPicture('menu/slot_spring');
          break;
        case 2:
          bitmap = Managers.Images.loadPicture('menu/slot_summer');
          break;
        case 3:
          bitmap = Managers.Images.loadPicture('menu/slot_fall');
          break;
        case 4:
          bitmap = Managers.Images.loadPicture('menu/slot_winter');
          break;
      }

      if (bitmap) {
        if (bitmap.isReady()) {
          this.drawBitmap(bitmap, 209, y - 8);
        } else {
          this.requireBitmap(bitmap);
        }
      }
    }
  }

  selectedSaveFilePath() {
    if (this._index >= 0 && this._index < this._currentSaves.length) {
      return this._currentSaves[this._index];
    }

    return false;
  }

  drawRegularSlot(index, y) {
    this.requireBitmap(this._regularSlotBitmap);
    if (!this._regularSlotBitmap.isReady()) return;

    this.drawBitmap(this._regularSlotBitmap, 17, y);
    this.drawSlotNumber(index, 21, y);

    const savePath = this._currentSaves[index];
    if (!savePath) {
      return;
    }

    const saveData = Engine.Data.getCachedSaveData(savePath);
    if (!saveData) return;

    const saveHeader = this.getSaveDataHeader(saveData);
    const { playerName, fileName, playtime } = saveHeader;

    this.changeTextColor(this.baseColor());
    this.contents.outlineColor = this.baseColor();
    this.contents.outlineWidth = 0;
    this.contents.fontSize = Utils.convertFontSize(42, this._zoomLevel);

    this.drawLabel(65, y - 3, playerName || fileName || t('Player'));

    if (playtime) {
      this.changeTextColor(this.baseColor());
      this.contents.outlineColor = this.darkBaseColor();
      this.contents.outlineWidth = 1;
      this.contents.fontSize = Utils.convertFontSize(26, this._zoomLevel);

      this.drawLabel(65, y + 9, t('Play Time') + ': ' + playtime);
    }
  }

  drawNewSlot(index, y, selected) {
    if (selected) {
      this.drawBitmap(this._selectedSlotBitmap, 17, y);
      this.drawPercentValue(20 * this._zoomLevel, (y + 9) * this._zoomLevel, '+', 40 * this._zoomLevel, 'center', false);
    } else {
      this.drawBitmap(this._newSlotBitmap, 17, y);
    }
  }

  drawSlotNumber(index, x, y) {
    if (index < 99) {
      const value = index + 1;
      const text = value < 10 ? '0' + value : value.toString();

      this.drawPercentValue(x * this._zoomLevel, y * this._zoomLevel, text, 40 * this._zoomLevel, 'center', false);
    }
  }

  getIndexUnderCursor() {
    const x = (TouchInput.mouseX - this.x) / Graphics.windowZoomLevel;
    const y = (TouchInput.mouseY - this.y) / Graphics.windowZoomLevel;

    if (x < 17 || x > this.width) {
      return -1;
    }

    if (y < 57 || y > 250) {
      return -1;
    }

    let index = this._topIndex;
    let countedY = 57;
    while (index <= this._topIndex + 5) {
      if (index == this._index) {
        if (countedY + 48 > y) {
          return index;
        } else {
          countedY += 48;
          index++;
        }
      } else {
        if (countedY + 25 > y) {
          return index;
        } else {
          countedY += 25;
          index++;
        }
      }
    }

    return index;
  }

  resetCursor() {
    this._topIndex = 0;
    this._index = 0;
  }

  isMouseOverOkButton() {
    return this.isMouseOverPosition(212, 240, 14, 12);
  }

  isMouseOverDeleteButton() {
    return this.isMouseOverPosition(196, 240, 14, 12);
  }

  isMouseOverPosition(x, y, w, h) {
    const mouseX = (TouchInput.mouseX - this.x) / Graphics.windowZoomLevel;
    const mouseY = (TouchInput.mouseY - this.y) / Graphics.windowZoomLevel;

    if (mouseX < x) return false;
    if (mouseY < y) return false;
    if (mouseX > x + w) return false;
    if (mouseY > y + h) return false;

    return true;
  }

  needsRefresh() {
    // if (this._oldMouseX != TouchInput.mouseX) {
    //   return true;
    // }

    // if (this._oldMouseY != TouchInput.mouseY) {
    //   return true;
    // }

    return false;
  }

  refreshIfNeeded() {
    if (this.needsRefresh()) {
      this.refresh();
    }
  }

  maybeChangeMode(newMode) {
    if (newMode == 'load') {
      if (this._allSaves.length == 0) {
        this.refresh();
        return;
      }
    }

    if (newMode != this._mode) {
      this._mode = newMode;
      this.refresh();
    }
  }

  deleteSaveFile(index) {
    if (index < 0 || index >= this._currentSaves.length) return;
    
    const saveFilePath = this._currentSaves[index];
    Engine.Data.deleteSave(saveFilePath);

    this.resetCursor();
    this.updateSaves();
    this.refresh();

    this.afterDeleteFile();
  }

  afterDeleteFile() {
    if (this._allSaves.length === 0) {
      switch (this._onEmptyListAction) {
        case 'saveMode':
          this._mode = 'save';
          this.updateSaves();
          this.refresh();
          break;
        case 'popScene':
          Managers.Scenes.pop();
          break;
      }
    }
  }

  confirmSaveFile(index) {
    if (index < 0 || index > this._currentSaves.length) return;
    let saveFilePath;

    if (index == this._currentSaves.length) {
      if (this._mode == 'load') return;

      saveFilePath = Managers.Storage.getPathForNewSave();
    } else {
      saveFilePath = this._currentSaves[index];
    }

    if (this._mode == 'load') {
      try {
        if (Engine.Data.loadGame(saveFilePath)) {
          this.onLoadSuccess();
        } else {
          this.onLoadFailure();
        }
      } catch(e) {
        this.onLoadFailure(e);
      }
    } else if (this._mode == 'save') {
      $gameSystem.onBeforeSave();

      if (Engine.Data.saveGame(saveFilePath)) {
        this.onSaveSuccess();
      } else {
        this.onSaveFailure();
      }
    }
  }

  reloadMap() {
    if (!$gamePlayer.isTransferring()) {
      $gamePlayer.reserveTransfer($gameMap.mapId(), $gamePlayer.x, $gamePlayer.y, $gamePlayer._direction, undefined, false, true);
      $gamePlayer.requestMapReload();
    }
  }

  onSaveSuccess() {
    Managers.Sound.playSave();
    Managers.Scenes.pop();
  }

  onSaveFailure() {
    Managers.Sound.playBuzzer();
  }

  onLoadSuccess() {
    Managers.Sound.playLoad();
    Managers.Scenes._scene.fadeOutAll();
    this.reloadMap();
    Managers.Scenes.goToScene(GameScenes.PreloadSave);
    this._loadSuccess = true;
  }

  onLoadFailure(e) {
    Managers.Sound.playBuzzer();

    // If a load operation fails midway, it can completely mess with the existing data, so we're forced to send the player back to the title to load a valid save again
    Managers.Scenes.goToScene(GameScenes.Title);
  }

  updateTouchDataMode() {
    if (this.isMouseOverPosition(0, 18, 257, 14)) {
      if (this.isMouseOverPosition(0, 18, 126, 14)) {
        this.maybeChangeMode('save');
      } else if (this.isMouseOverPosition(127, 18, 131, 14)) {
        this.maybeChangeMode('load');
      }

      return true;
    }

    return false;
  }

  selectPrior() {
    const index = this._index - 1;
    if (index < 0) {
      this._index = this.maxSlots() - 1;
    } else {
      this._index = index;
    }

    this.updateScroll();
    this.refresh();
  }

  selectNext() {
    const index = this._index + 1;
    if (index >= this.maxSlots()) {
      this._index = 0;
    } else {
      this._index = index;
    }

    this.updateScroll();
    this.refresh();
  }

  updateTouch() {
    if (TouchInput.isTriggered()) {
      if (this.updateTouchDataMode()) {
        return;
      }

      if (this.isMouseOverPosition(196, 240, 14, 12)) {
        this.showCommandWindow('delete');
        return;
      } else if (this.isMouseOverPosition(212, 240, 14, 12)) {
        this.showCommandWindow();
        return;
      } else if (this.isMouseOverPosition(118, 42, 10, 10)) {
        this.selectPrior();
        return;
      } else if (this.isMouseOverPosition(118, 233, 10, 10)) {
        this.selectNext();
        return;
      }

      const index = this.getIndexUnderCursor();
      if (index >= 0 && index != this._index && index <= this._topIndex + 5 && index < this.maxSlots()) {
        this._index = index;
        this.updateScroll();
        this.refresh();
      }
    }
  }

  updateScroll() {
    if (this._topIndex > this._index) {
      this._topIndex = Math.min(this._index, this.maxSlots() -1);
    } else if (this._topIndex + 5 < this._index) {
      this._topIndex = Math.max(0, this._index - 5);
    }
  }

  maxSlots() {
    if (this._mode == 'save') {
      return this._currentSaves.length + 1;
    }

    return this._currentSaves.length;
  }

  updateInput() {
    if (Engine.Input.isTriggered('down') || Engine.Input.isRepeated('down')) {
      if (this._index < this.maxSlots() -1) {
        this._index++;
      } else {
        this._index = 0;
      }

      this.updateScroll();
      this.refresh();
    } else if (Engine.Input.isTriggered('up') || Engine.Input.isRepeated('up')) {
      if (this._index > 0) {
        this._index--;
      } else {
        this._index = this.maxSlots() -1;
      }

      this.updateScroll();
      this.refresh();
    } else if (Engine.Input.isTriggered('left') || Engine.Input.isRepeated('right')) {
      this.switchDataMode();
    } else if (Engine.Input.isTriggered('ok')) {
      this.showCommandWindow();
    } else if (Engine.Input.isTriggered('extra')) {
      this.showCommandWindow('delete');
    }
  }

  switchDataMode() {
    const newMode = this._mode == 'save' ? 'load' : 'save';
    this.maybeChangeMode(newMode);
  }

  showCommandWindow(mode) {
    if (this._index < 0) return;
    if (this.maxSlots() === 0) return;

    if (!this._commandWindow) {
      this._commandWindow = new Window_DataCommand();

      this._commandWindow.setHandler('save', this.commandSave.bind(this));
      this._commandWindow.setHandler('load', this.commandLoad.bind(this));
      this._commandWindow.setHandler('delete', this.commandDelete.bind(this));
      this._commandWindow.setHandler('cancel', this.commandCancel.bind(this));
    } else {
      this._commandWindow.show();
    }

    this.parent.addChild(this._commandWindow);
    this._commandWindow.setMode(mode || this._mode);
    this._commandWindow.activate();
    this._commandWindow.select(0);
  }

  commandSave() {
    this.confirmSaveFile(this._index);
    this._commandWindow.hide();
    this._commandWindow.deactivate();
  }

  commandLoad() {
    this.confirmSaveFile(this._index);
    this._commandWindow.hide();
    this._commandWindow.deactivate();
  }

  commandDelete() {
    this.deleteSaveFile(this._index);
    this._commandWindow.hide();
    this._commandWindow.deactivate();
  }

  commandCancel() {
    this._commandWindow.hide();
    this._commandWindow.deactivate();
  }

  isLocked() {
    return this._commandWindow && this._commandWindow.visible;
  }

  update() {
    if (this._commandWindow && this._commandWindow.visible) {
      return;
    }

    if (!this.visible) {
      return;
    }

    if (this._oldMouseOverOkButton !== this.isMouseOverOkButton() || this._oldMouseOverDeleteButton !== this.isMouseOverDeleteButton()) {
      this.refresh();
    }

    this.updateTouch();
    this.updateInput();
  }

  setEmptyListAction(action) {
    this._onEmptyListAction = action;
  }
}

module.exports = Window_DataContent;