const Window_Selectable = require('engine/windows/Window_Selectable');

//-----------------------------------------------------------------------------
// Window_RecipeList
//
// A selectable window with a list of learnt recipes

class RecipeList extends Window_Selectable {
  initialize(x, y, width, height) {
    this._zoomLevel = Graphics.windowZoomLevel;
    this.makeItemList();
    super.initialize(x, y, width, height);

    this._itemBackground = Managers.Images.loadPicture('crafts/recipe_slot');
    this._arrowDownBitmap = Managers.Images.loadPicture('menu/arrow_down');
    this._arrowUpBitmap = Managers.Images.loadPicture('menu/arrow_up');

    this.refresh();
    this.select(0);
    this.activate();
  }

  numVisibleRows() {
    return 4;
  }

  // enableCursorSprite() {
  //   return false;
  // }

  enableWindowFrame() {
    return false;
  }

  maxCols() {
    return 1;
  }

  maxItems() {
    return this._recipeList.length;
  }

  verticalItemSpacing() {
    return 4 * Graphics.windowZoomLevel;
  }

  allBitmapsReady() {
    if (!this._itemBackground.isReady()) {
      this.requireBitmap(this._itemBackground);
      return false;
    }

    if (!this._arrowUpBitmap.isReady()) {
      this.requireBitmap(this._arrowUpBitmap);
      return false;
    }

    if (!this._arrowDownBitmap.isReady()) {
      this.requireBitmap(this._arrowDownBitmap);
      return false;
    }

    return true;
  }

  drawAllItems() {
    if (!this.allBitmapsReady()) {
      return;
    }

    const zoom = this._zoomLevel;
    const iconX = 137.5;
    if (this._arrowUpBitmap.isReady()) {
      this.drawBitmap(this._arrowUpBitmap, iconX, 0, zoom);
    }

    super.drawAllItems();

    if (this._arrowDownBitmap.isReady()) {
      this.drawBitmap(this._arrowDownBitmap, iconX, 215, zoom);
    }
  }

  drawItem(index) {
    var rect = this.itemRectForText(index);

    if (this._itemBackground.isReady()) {
      this.drawZoomedBitmap(this._itemBackground, rect.x + 21 * this._zoomLevel, rect.y, this._zoomLevel);
    }

    this.drawText(index + 1, rect.x, rect.y + 12 * this._zoomLevel, 12 * this._zoomLevel, 'right');

    const data = Managers.Cooking.getRecipeData(this._recipeList[index]);
    if (!data) {
      return;
    }

    if (data.utensil) {
      const utensilData = Managers.Items.getItemData(data.utensil);

      if (utensilData) {
        const iconX = rect.x + 14 * this._zoomLevel;
        const iconY = rect.y + 10 * this._zoomLevel;
        this.drawIcon(utensilData.iconIndex, iconX, iconY);
      }
    }

    const itemData = Managers.Items.getItemData(data.itemId);
    if (itemData) {
      const iconX = rect.x + 226 * this._zoomLevel;
      const iconY = rect.y + 11 * this._zoomLevel;
      this.drawIcon(itemData.iconIndex, iconX, iconY);
    }

    const ingredients = data.ingredients;
    const y = rect.y + 11 * this._zoomLevel;
    let x = rect.x + 45 * this._zoomLevel;

    for (let i = 0; i < ingredients.length; i++) {
      if (!ingredients[i]) continue;

      const ingredientData = Managers.Items.getItemData(ingredients[i]);
      if (ingredientData) {
        this.drawIcon(ingredientData.iconIndex, x, y);
      }

      x += 28 * this._zoomLevel;
    }
  }

  selectedRecipe() {
    if (this._index >= 0 && this._index < this._recipeList.length) {
      return Managers.Cooking.getRecipeData(this._recipeList[this._index]);
    }

    return false;
  }

  standardFontSize() {
    return Utils.convertFontSize(28, Graphics.windowZoomLevel);
  }

  itemTextAlign() {
    return 'left';
  }

  makeItemList() {
    this._recipeList = Managers.Cooking.getSortedLearntRecipes() || [];
  }

  refresh() {
    this.makeItemList();
    this.createContents();
    super.refresh();
  }

  itemHeight() {
    return 48 * this._zoomLevel;
  }

  standardOutlineWidth() {
    return 0;
  }

  normalColor() {
    return this.blackColor();
  }

  verticalMargin() {
    return 10 * this._zoomLevel;
  }

  updateHelp() {
    if (!this._infoWindow) {
      return;
    }

    if (this._index < 0) {
      this._infoWindow.clear();
      return;
    }

    const recipeId = this._recipeList[this._index];
    const data = Managers.Cooking.getRecipeData(recipeId);

    if (!data) {
      this._infoWindow.clear();
      return;
    }

    const itemId = data.itemId;
    const itemData = Managers.Items.getItemData(itemId);

    this._infoWindow.setItem(itemData);
  }


  maxPageRows() {
    const pageHeight = this.height - this.paddingV * 2 - this.verticalMargin();
    return Math.floor(pageHeight / this.itemFullHeight());
  }

  update() {
    super.update();

    if (!TouchInput.isRepeated()) {
      return;
    }

    const x = TouchInput.mouseX - this.x;
    const y = TouchInput.mouseY - this.y;

    if (x < 0) return;
    if (x > this.width) return;
    if (y < 0) return;
    if (y > this.height) return;

    const localY = y / this._zoomLevel;

    if (localY < 20) {
      this.cursorUp(TouchInput.isTriggered());
      return;
    }

    if (localY > 215) {
      this.cursorDown(TouchInput.isTriggered());
      return;
    }
  }

  allowScrollOnHover() {
    return false;
  }
}

module.exports = RecipeList;