const Window_Base = require('engine/windows/Window_Base');

//-----------------------------------------------------------------------------
// Window_Cooking
//
// A window to select utensil and ingredients for cooking

class Cooking extends Window_Base {
  initialize(x, y, width, height) {
    this._zoomLevel = Graphics.windowZoomLevel;
    super.initialize(x, y, width, height);

    this._ingredients = [];
    this._index = -1;

    this.refresh();
    this.activate();
  }

  enableWindowFrame() {
    return false;
  }

  standardFontSize() {
    return Utils.convertFontSize(28, Graphics.windowZoomLevel);
  }

  setRecipe(recipe) {
    this._recipe = recipe;
    this._ingredients = [].concat(recipe.ingredients);
    this._utensil = recipe.utensil;
    this._index = 12;

    this.refresh();
  }

  refresh() {
    this.createContents();
    super.refresh();

    const z = Graphics.windowZoomLevel;
    const selectedSlotPicture = 'crafts/cooking_selected_slot';
    const selectedSlotPicture2 = 'crafts/cooking_selected_slot2';

    if (Switches.hasKitchenKnife) {
      const knifeData = Managers.Items.getItemData('kitchen-knife');
      const iconIndex = knifeData ? knifeData.iconIndex : 417;

      if (this._index === 0) {
        this.changePaintOpacity(true);
        this.drawPicture(selectedSlotPicture, 25 * z, 3 * z, z, true);
      }
      this.changePaintOpacity(this._utensil == 'kitchen-knife');
      this.drawIcon(iconIndex, 26 * z, 4 * z, z);
    }

    if (Switches.hasKitchenPot) {
      const potData = Managers.Items.getItemData('kitchen-pot');
      const iconIndex = potData ? potData.iconIndex : 420;

      if (this._index === 1) {
        this.changePaintOpacity(true);
        this.drawPicture(selectedSlotPicture, 53 * z, 3 * z, z, true);
      }
      this.changePaintOpacity(this._utensil == 'kitchen-pot');
      this.drawIcon(iconIndex, 54 * z, 4 * z, z);
    }

    if (Switches.hasKitchenOven) {
      const ovenData = Managers.Items.getItemData('kitchen-oven');
      const iconIndex = ovenData ? ovenData.iconIndex : 435;

      if (this._index === 2) {
        this.changePaintOpacity(true);
        this.drawPicture(selectedSlotPicture, 81 * z, 3 * z, z, true);
      }
      this.changePaintOpacity(this._utensil == 'kitchen-oven');
      this.drawIcon(iconIndex, 82 * z, 4 * z, z);
    }

    if (Switches.hasKitchenPan) {
      const panData = Managers.Items.getItemData('kitchen-pan');
      const iconIndex = panData ? panData.iconIndex : 423;

      if (this._index === 3) {
        this.changePaintOpacity(true);
        this.drawPicture(selectedSlotPicture, 25 * z, 30 * z, z, true);
      }
      this.changePaintOpacity(this._utensil == 'kitchen-pan');
      this.drawIcon(iconIndex, 26 * z, 31 * z, z);
    }

    if (Switches.hasKitchenBlender) {
      const blenderData = Managers.Items.getItemData('kitchen-blender');
      const iconIndex = blenderData ? blenderData.iconIndex : 383;

      if (this._index === 4) {
        this.changePaintOpacity(true);
        this.drawPicture(selectedSlotPicture, 53 * z, 30 * z, z, true);
      }
      this.changePaintOpacity(this._utensil == 'kitchen-blender');
      this.drawIcon(iconIndex, 54 * z, 31 * z, z);
    }

    if (Switches.hasKitchenMixer) {
      const mixerData = Managers.Items.getItemData('kitchen-mixer');
      const iconIndex = mixerData ? mixerData.iconIndex : 398;

      if (this._index === 5) {
        this.changePaintOpacity(true);
        this.drawPicture(selectedSlotPicture, 81 * z, 30 * z, z, true);
      }
      this.changePaintOpacity(this._utensil == 'kitchen-mixer');
      this.drawIcon(iconIndex, 82 * z, 31 * z, z);
    }

    let idx = 0;
    for (let ingredient of this._ingredients) {
      if (idx >= 6) {
        break;
      }

      const ingredientData = Managers.Items.getItemData(ingredient);
      if (!ingredientData) {
        continue;
      }

      const x = 126 + 28 * idx;

      if (this._index === 6 + idx) {
        this.drawPicture(selectedSlotPicture, (x - 1) * z, 16 * z, z, true);
      }

      const totalNum = Managers.Items.numItemsById(ingredientData) + Managers.Items.storageContainer.numItemsById(ingredientData);

      this.changePaintOpacity(totalNum > 0);
      this.drawIcon(ingredientData.iconIndex, x * z, 17 * z, z);
      idx++;
    }

    if (this._index == 12) {
      this.drawPicture(selectedSlotPicture2, 310 * z, 5 * z, z, true);
      this.contents.fillRect(306 * z, 37 * z, 36 * z, 11 * z, this.blueColor());
    } else if (this._index >= 6 + idx) {
      const x = 126 + 28 * (this._index - 6);
      this.drawPicture(selectedSlotPicture, (x - 1) * z, 16 * z, z, true);
    }

    const recipe = Managers.Cooking.findRecipeForItems(this._utensil, this._ingredients, true);
    let recipeIcon = Constants.UNKNOWN_RECIPE_ICON;
    if (recipe) {
      const itemData = Managers.Items.getItemData(recipe.itemId);
      if (itemData) {
        recipeIcon = itemData.iconIndex;
      }
    }
    
    this.changePaintOpacity(true);
    this.drawIcon(recipeIcon, 312 * z, 12 * z, z);
    this.changeTextColor(this.blackColor());
    this.drawText(t("Cook").toUpperCase(), 306 * z, 32 * z, 36 * z, 'center');
  }

  standardOutlineWidth() {
    return 0;
  }

  normalColor() {
    return this.blackColor();
  }

  standardPadding() {
    return 0;
  }

  drawPicture(filename, x, y, zoomLevel) {
    super.drawPicture(filename, x, y, zoomLevel, true);
  }

  confirmUtensil() {
    if (this._index < 0 || this._index >= 6) {
      return;
    }

    switch(this._index) {
      case 0:
        this._utensil = this._utensil == 'kitchen-knife' ? false : 'kitchen-knife';
        break;
      case 1:
        this._utensil = this._utensil == 'kitchen-pot' ? false : 'kitchen-pot';
        break;
      case 2:
        this._utensil = this._utensil == 'kitchen-oven' ? false : 'kitchen-oven';
        break;
      case 3:
        this._utensil = this._utensil == 'kitchen-pan' ? false : 'kitchen-pan';
        break;
      case 4:
        this._utensil = this._utensil == 'kitchen-blender' ? false : 'kitchen-blender';
        break;
      case 5:
        this._utensil = this._utensil == 'kitchen-mixer' ? false : 'kitchen-mixer';
        break;
    }

    if (this._utensil) {
      Managers.Sound.playPickItem();
    } else {
      Managers.Sound.playDropItem();
    }
    this.refresh();
  }

  removeSelectedIngredient() {
    if (this._index < 6 || this._index >= 12) {
      return;
    }

    const idx = this._index - 6;
    if (idx >= this._ingredients.length) {
      return;
    }

    this._ingredients.splice(idx, 1);
    Managers.Sound.playDropItem();
    this.refresh();
  }

  addIngredient(itemId) {
    if (this._ingredients.includes(itemId)) {
      const idx = this._ingredients.indexOf(itemId);
      this._ingredients.splice(idx, 1);
      Managers.Sound.playDropItem();
      this.refresh();
      return true;
    }

    if (this._ingredients.length >= 6) {
      Managers.Sound.playBuzzer();
      return false;
    }

    this._ingredients.push(itemId);
    this.refresh();
    Managers.Sound.playPickItem();
  }

  isPositionInsideFrame(x, y) {
    return x >= 0 && y >= 0 && x < this.width && y < this.height;
  }

  processTouch() {
    if (!Managers.Config.enableMouse) return;

    const lastIndex = this._index;
    const x = this.canvasToLocalX(TouchInput.mouseX);
    const y = this.canvasToLocalY(TouchInput.mouseY);

    if (x == this._oldX && y == this._oldY) {
      return;
    }

    if (!this.isPositionInsideFrame(x, y)) {
      return;
    }

    this._oldX = x;
    this._oldY = y;

    const hitIndex = this.hitTest(x, y);
    if (hitIndex >= 0 && hitIndex !== lastIndex) {
      this._index = hitIndex;
      Managers.Sound.playCursor();
      this.refresh();
    }
  }

  hitTest(x, y) {
    let cx = Math.round(x / Graphics.windowZoomLevel);
    let cy = Math.round(y / Graphics.windowZoomLevel);

    if (cx < 24 || cx > 344) {
      return -1;
    }

    if (cx < 106) {
      if (cx < 52) {
        return cy < 30 ? 0 : 3;
      }

      if (cx < 80 ){
        return cy < 30 ? 1 : 4;
      }

      return cy < 30 ? 2 : 5;
    }

    if (cx > 292) {
      if (cx >= 304 && cx <= 344 && cy >= 8 && cy <= 50) {
        return 12;
      }

      return -1;
    }

    if (cy < 16 || cy > 42 || cx <= 124) {
      return -1;
    }

    return Math.floor(6 + (cx - 124) / 28);
  }

  update() {
    super.update();
    this.processTouch();
  }

  isTouchedInsideItem() {
    const x = this.canvasToLocalX(TouchInput.x);
    const y = this.canvasToLocalY(TouchInput.y);

    if (x >= 0 && y >= 0 && x < this.width && y < this.height) {
      return this.hitTest(x, y) >= 0;
    }

    return false;
  }
}

module.exports = Cooking;