const Window_Base = require('engine/windows/Window_Base');

class Window_BaseItemInfo extends Window_Base {
  initialize(x, y, width, height) {
    super.initialize(x, y, width, height);
    this._zoomLevel = Graphics.windowZoomLevel;
  }

  setItem(item) {
    if (item && this._item && item.id == this._item.id) {
      return;
    }

    if (!item && !this._item) {
      return;
    }

    this._item = item;
    this.refresh();
  }

  setDiscountInformationEnabled(enable) {
    this._discountInformationEnabled = Boolean(enable);
  }

  enableWindowFrame() {
    return false;
  }

  drawBackground() {
  }

  labelWidth() {
    return 125 * Graphics.windowZoomLevel;
  }

  labelX() {
    return 11 * Graphics.windowZoomLevel;
  }

  valueWidth() {
    return 38 * Graphics.windowZoomLevel;
  }

  valueX() {
    return 138 * Graphics.windowZoomLevel;
  }

  drawItemName(y) {
    const labelX = this.labelX();
    const labelWidth = this.labelWidth();

    const itemName = Managers.Text.item(this._item.id).toUpperCase();
    this.changeTextColor(this.redColor());
    this.drawText(itemName, labelX, y, labelWidth, 'center');
    this.resetTextColor();

    return this.largeLineHeight();
  }

  refresh() {
    this.contents.clear();
    this.drawBackground();

    if (!this._item) return;

    this.resetFontSettings();

    const lineHeight = this.lineHeight();
    let y = this.topMargin();
    const labelX = this.labelX();
    const labelWidth = this.labelWidth();
    const valueWidth = this.valueWidth();
    const valueX = this.valueX();

    y += (this.drawItemName(y) || 0);

    const data = Managers.Items.getItemData(this._item.id);
    const quantity = Managers.Items.numItems(this._item);

    const buyPrice = Managers.Items.getBuyPrice(this._item.id);
    let sellPrice = Managers.Items.getSellPrice(this._item.id);

    //#ToDo: Use the content manager for this
    if (!data['placeholder-for']) {
      this.drawText(t('Quantity:'), labelX, y, labelWidth, 'left');
      this.drawText(quantity, valueX, y, valueWidth, 'right');
      y += lineHeight;
    }

    if (buyPrice > 0) {
      let buyText = t("Buy Price:");
      if (data && data.ingredients && data.ingredients.length > 0) {
        buyText = t("Craft Price:");
      } else if (data && data.type == 'seed-box') {
        buyText = t("Buy Price (box with 10):");
      }

      this.drawText(buyText, labelX, y, labelWidth, 'left');

      this.drawText(Managers.Text.formatMoney(buyPrice), valueX, y, valueWidth, 'right');
      y += lineHeight;
    }

    if (sellPrice > 0) {
      this.drawText(t('Sell Price:'), labelX, y, labelWidth, 'left');
      this.drawText(Managers.Text.formatMoney(sellPrice), valueX, y, valueWidth, 'right');
      y += lineHeight;
    }

    if (this._discountInformationEnabled) {
      const discount = Managers.Items.getPurchaseDiscount(this._item.id);
      if (discount && discount > 0 && discount < 100) {
        this.changeTextColor(this.blueColor());

        this.drawText(t('Discount:'), labelX, y, labelWidth, 'left');
        this.drawText(`${discount}%`, valueX, y, valueWidth, 'right');
        this.resetTextColor();

        y += lineHeight;
      }
    }

    const cropData = Managers.Items.getSeedCropData(this._item.id);
    if (cropData) {
      sellPrice = Managers.Items.getSellPrice(cropData.id);

      if (sellPrice > 0) {
        this.drawText(t('Crop Sell Price:'), labelX, y, labelWidth, 'left');
        this.drawText(Managers.Text.formatMoney(sellPrice), valueX, y, valueWidth, 'right');
        y += lineHeight;
      }

      if (data.seasons) {
        this.drawText(t('Grows on:'), labelX, y, labelWidth, 'left');

        let shiftPosition = false;
        for (let i = 0; i < data.seasons.length; i++) {
          if (shiftPosition) {
            y += lineHeight;
          } else {
            shiftPosition = true;
          }

          this.drawText(t(data.seasons[i]), valueX, y, valueWidth, 'right');
        }

        y += lineHeight;
      }
    }

    if (data && data.daysToComplete) {
      this.drawText(t('Craft Time:'), labelX, y, labelWidth, 'left');
      if (data.daysToComplete == 1) {
        this.drawText(t('1 day'), valueX, y, valueWidth, 'right');
      } else {
        this.drawText(`${data.daysToComplete} ${t('days')}`, valueX, y, valueWidth, 'right');
      }
      y += lineHeight;
    }

    if (data && data.daysToGrow) {
      this.drawText(t('Time to grow:'), labelX, y, labelWidth, 'left');
      if (data.daysToGrow == 1) {
        this.drawText(t('1 day'), valueX, y, valueWidth, 'right');
      } else {
        this.drawText(`${data.daysToGrow} ${t('days')}`, valueX, y, valueWidth, 'right');
      }
      y += lineHeight;

      if (data.daysToRegrow) {
        this.drawText(t('Grows again in:'), labelX, y, labelWidth, 'left');
        if (data.daysToRegrow == 1) {
          this.drawText(t('1 day'), valueX, y, valueWidth, 'right');
        } else {
          this.drawText(`${data.daysToRegrow} ${t('days')}`, valueX, y, valueWidth, 'right');
        }
        y += lineHeight;
      }
    }

    if (data && data.ingredients && data.ingredients.length > 0) {
      this.changeTextColor(this.blueColor());
      this.outlineWidth = 0;

      this.drawText(t('Required Items:'), labelX, y, labelWidth, 'left');
      y += lineHeight;
      this.resetTextColor();

      const ingredientsIncluded = [];
      for (let idx = 0; idx < data.ingredients.length; idx++) {
        const ingredientId = data.ingredients[idx];
        if (ingredientsIncluded.includes(ingredientId)) continue;
        ingredientsIncluded.push(ingredientId);

        const ingredientData = Managers.Items.getItemData(ingredientId);

        let ingredientCount = 1;
        for (let newIdx = idx +1; newIdx < data.ingredients.length; newIdx++) {
          if (data.ingredients[newIdx] == ingredientId) {
            ingredientCount++;
          }
        }

        let ownedCount = Managers.Items.numItemsById(ingredientId);

        this.changePaintOpacity(ownedCount >= ingredientCount);

        this.drawTextEx(`\\hi[${ingredientData.iconIndex}]`, labelX, y);
        this.drawText(Managers.Text.item(ingredientId), labelX + 12 * Graphics.windowZoomLevel, y, labelWidth - 12 * Graphics.windowZoomLevel, 'left');
        this.changePaintOpacity(true);

        if (ingredientCount > 0) {
          ownedCount = Math.min(ownedCount, ingredientCount);
          this.drawText(`${ownedCount} / ${ingredientCount}`, valueX, y, valueWidth, 'right');
        }

        y += this.largeLineHeight();
      }

      if (data.expRequired) {
        this.resetTextColor();
        this.drawText(t("Experience:"), labelX, y, labelWidth, 'left');

        let expNeeded = 0;
        let exp = 0;

        for (const expType in data.expRequired) {
          const attrName = `${ expType }Exp`;
          const needed = data.expRequired[expType] || 1000;
          const current = Managers.Player[attrName] || 0;
          expNeeded += needed;
          exp += Math.min(current, needed);
        }

        if (expNeeded > 0) {
          if (exp >= expNeeded) {
            this.changeTextColor(this.blueColor());
            this.drawText('100%', valueX, y, valueWidth, 'right');
          } else {
            this.changeTextColor(this.redColor());
            const rate = Math.fix(Math.floor(exp * 10000 / expNeeded) / 100);
            this.drawText(`${rate} %`, valueX, y, valueWidth, 'right');
          }
          this.resetTextColor();
        }

        y += lineHeight;
      }
    }

    //#ToDo: Use the content manager for this
    if (data && data['placeholder-for']) {
      y += this.shortLineHeight();

      this.drawText(t('Generated Items:'), labelX, y, labelWidth, 'left');
      this.resetTextColor();

      const placeholderId = data['placeholder-for'];
      const placeholderQuantity = data['placeholder-quantity'] || 1;
      const placeholderData = Managers.Items.getItemData(placeholderId);

      y += lineHeight;

      this.changePaintOpacity(true);
      this.drawTextEx(`\\hi[${placeholderData.iconIndex}]`, labelX, y);
      this.drawText(Managers.Text.item(placeholderId), labelX + 12 * Graphics.windowZoomLevel, y, labelWidth - 12 * Graphics.windowZoomLevel, 'left');

      if (placeholderQuantity > 0) {
        this.drawText(` x ${placeholderQuantity}`, valueX, y, valueWidth, 'right');
      }

      y += lineHeight;
    }

    this.resetTextColor();
    y += this.drawAdditionalInformation(y, data);
    this.resetTextColor();
    y += this.drawItemEffects(y, data, labelX, labelWidth, lineHeight, valueX, valueWidth);
    this.resetTextColor();
  }

  drawItemEffects(y, data, labelX, labelWidth, lineHeight, valueX, valueWidth) {
    let value;
    let extraY = 0;

    if (data.restores) {
      if (data.restores.fatigue) {
        value = data.restores.fatigue.toString();
        value = Managers.Text.replaceTextVariables(value);

        this.drawText(`${t('Fatigue')}:`, labelX, y + extraY, labelWidth, 'left');
        this.changeTextColor(this.blueColor());

        if (value > 0) {
          this.drawText(`-${ value }`, valueX, y + extraY, valueWidth, 'right');
        } else {
          this.drawText(`+${ Math.abs(value) }`, valueX, y + extraY, valueWidth, 'right');
        }

        this.resetTextColor();
        extraY += lineHeight;
      }

      if (data.restores.stamina) {
        let valueColor = this.blueColor();

        if (data.restores.stamina > 0) {
          value = `+${data.restores.stamina}`;
        } else {
          valueColor = this.redColor();
          value = `${data.restores.stamina}`;
        }

        value = Managers.Text.replaceTextVariables(value);
        this.drawText(`${t('Stamina')}:`, labelX, y + extraY, labelWidth, 'left');
        this.changeTextColor(valueColor);
        this.drawText(value, valueX, y + extraY, valueWidth, 'right');
        this.resetTextColor();
        extraY += lineHeight;
      }

      if (data.restores.maxStamina) {
        let valueColor = this.blueColor();
        if (data.restores.maxStamina > 0) {
          value = `+${data.restores.maxStamina}`;
        } else {
          valueColor = this.redColor();
          value = `${data.restores.maxStamina}`;
        }

        value = Managers.Text.replaceTextVariables(value);
        this.drawText(`${t('Max Stamina')}:`, labelX, y + extraY, labelWidth, 'left');
        this.changeTextColor(valueColor);
        this.drawText(value, valueX, y + extraY, valueWidth, 'right');
        this.resetTextColor();
        extraY += lineHeight;
      }

      const buffsTable = [];
      if (data.restores.buffs) {
        for (let buff of data.restores.buffs) {
          if (!buff) {
            continue;
          }

          let effectStr = false;
          let labelColor = 'blue';
          let valueColor = this.blueColor();
          const timeStr = Managers.Time.convertSecondsToTimeString(buff.time);

          if (buff.rate === 0) {
            effectStr = t("Frozen");
          } else if (buff.rate <= 1) {
            effectStr = Math.floor(100 * buff.rate) + '%';
          } else {
            labelColor = 'red';
            valueColor = this.redColor();
            effectStr = Math.floor(100 * buff.rate) + '%';
          }

          effectStr = Managers.Text.replaceBasicTextVariables(`<color:${ labelColor }>${ effectStr }</color>`);

          switch (buff.type) {
            case 'stamina':
              buffsTable.push({
                label: t("Stamina Rate") + ': ' + effectStr,
                valueColor,
                duration: timeStr
              });

              break;
            case 'fatigue':
              buffsTable.push({
                label: t("Fatigue Rate") + ': ' + effectStr,
                valueColor,
                duration: timeStr
              });
              break;
          }
        }
      }

      if (buffsTable && buffsTable.length) {
        this.resetTextColor();
        this.drawText(t("Buff"), labelX, y + extraY, labelWidth, 'left');
        this.drawText(t("Duration"), valueX, y + extraY, valueWidth, 'right');
        extraY += lineHeight;

        this.contents.fillRect(labelX, y + extraY, labelWidth, 1 * this._zoomLevel, '#fb3153');
        this.contents.fillRect(valueX, y + extraY, valueWidth, 1 * this._zoomLevel, '#fb3153');

        for (let buff of buffsTable) {
          this.resetTextColor();
          this.drawTextEx(buff.label, labelX, y + extraY, labelWidth);
          this.changeTextColor(buff.valueColor);
          this.drawText(buff.duration, valueX, y + extraY, valueWidth, 'right');
          extraY += lineHeight;
        }
      }
    }

    return extraY;
  }

  drawAdditionalInformation(y, data) {
    if (!data) return;

    const x = this.labelX();
    const w = (this.valueX() - x) + this.valueWidth();
    let extraY = 0;

    const lineHeight = this.lineHeight();
    let bitmapNeedsRefresh;

    if (data.possibleGift) {
      let defaultGiftText;
      let showWhoLoves = true;
      let showWhoLikes = true;
      let showWhoDislikes = true;
      let showWhoHates = true;

      switch(data.defaultGift) {
        case 'love':
          defaultGiftText = t("This is usually a <color:blue>great</color> gift.");
          showWhoLoves = false;
          showWhoLikes = false;
          break;
        case 'like':
          defaultGiftText = t("This is usually a <color:blue>good</color> gift.");
          showWhoLikes = false;
          break;
        case 'dislike':
          defaultGiftText = t("This is usually a <color:red>bad</color> gift.");
          showWhoDislikes = false;
          break;
        case 'hate':
          defaultGiftText = t("This is usually a <color:red>terrible</color> gift.");
          showWhoHates = false;
          showWhoDislikes = false;
          break;
        case 'normal':
          defaultGiftText = t("This is usually an uninspiring gift.");
          break;
      }

      if (defaultGiftText) {
        this.drawTextEx(defaultGiftText, x, y + extraY);
        extraY += lineHeight;
      }

      const drawPinList = (label, list) => {
        const knownList = [];

        for (let name of list) {
          if (peopleGifted.includes(name)) {
            knownList.push(name);
          }
        }

        if (knownList.length == 0) {
          return false;
        }

        this.drawText(label, x, y + extraY, w, 'left');
        let pinIndex = 0;
        for (let personName of knownList) {
          const pinBitmap = Managers.Images.loadPicture(`maps/pin-${ personName.toLowerCase() }`);
          if (pinBitmap.isReady()) {
            if (pinIndex >= 8) {
              pinIndex = 0;
              extraY += lineHeight;
            }

            const pinX = x + w - (pinBitmap.width * Graphics.windowZoomLevel  * (pinIndex + 1));
            this.drawZoomedBitmap(pinBitmap, pinX, y + extraY, Graphics.windowZoomLevel);
            pinIndex++;
          } else {
            bitmapNeedsRefresh = pinBitmap;
          }
        }

        extraY += lineHeight;
        return true;
      };

      const peopleGifted = Managers.Relationship.whoReceivedGift(data.id);
      if (showWhoLoves) {
        const lovers = Managers.Villagers.whoLovesItem(data.id);
        if (drawPinList(t("Loved by:"), lovers)) {
          showWhoLikes = false;
        }
      }

      if (showWhoLikes) {
        const likers = Managers.Villagers.whoLikesItem(data.id);
        drawPinList(t("Liked by:"), likers);
      }

      if (showWhoHates) {
        const haters = Managers.Villagers.whoHatesItem(data.id);
        if (drawPinList(t("Hated by:"), haters)) {
          showWhoDislikes = false;
        }
      }

      if (showWhoDislikes) {
        const dislikers = Managers.Villagers.whoDislikesItem(data.id);
        drawPinList(t("Disliked by:"), dislikers);
      }
    }

    if (bitmapNeedsRefresh) {
      this.requireBitmap(bitmapNeedsRefresh);
    }

    return extraY;
  }

  textPadding() {
    return 6 * Graphics.windowZoomLevel;
  }

  standardPadding() {
    return 0;
  }

  lineHeight() {
    return 10.5 * Graphics.windowZoomLevel;
  }

  shortLineHeight() {
    return 7.5 * Graphics.windowZoomLevel;
  }

  largeLineHeight() {
    return 13.5 * Graphics.windowZoomLevel;
  }

  standardFontSize() {
    return Utils.convertFontSize(24, Graphics.windowZoomLevel);
  }

  normalColor() {
    return this.blackColor();
  }

  resetFontSettings() {
    super.resetFontSettings();
    // this.contents.outlineColor = '#f4dea8';
    this.contents.outlineWidth = 0;
  }

  horizontalMargin() {
    return 5 * Graphics.windowZoomLevel;
  }

  topMargin() {
    return 5 * Graphics.windowZoomLevel;
  }
}

module.exports = Window_BaseItemInfo;