const Window_Selectable = require('engine/windows/Window_Selectable');

//-----------------------------------------------------------------------------
// Window_ShopNumber
//
// The window for inputting quantity of items to buy or sell on the shop
// screen.

class Window_ShopNumber extends Window_Selectable {
  initialize() {
    const width = this.windowWidth();
    const height = this.windowHeight();

    const x = (Graphics.width - width) / 2;
    const y = (Graphics.height - height) / 2;

    super.initialize(x, y, width, height);
    this._zoomLevel = Graphics.windowZoomLevel;

    this._item = null;
    this._max = 1;
    this._price = 0;
    this._number = 1;
    this._ingredients = [];
    this._currencyUnit = Managers.Text.currencyUnit;

    this._imageBackground = Managers.Images.loadBackground(this.backgroundImageName());
    this.createButtons();
  }

  windowWidth() {
    return 166 * Graphics.windowZoomLevel;
  }

  windowHeight() {
    return 80 * Graphics.windowZoomLevel;
  }

  backgroundImageName() {
    return 'popup';
  }

  enableWindowFrame() {
    return false;
  }

  number() {
    return this._number;
  }

  item() {
    return this._item;
  }

  setup(item, max, price, ingredients) {
    this._item = item;
    this._max = Math.floor(max);
    this._price = price;
    this._number = 1;
    this._ingredients = ingredients || [];

    this.placeButtons();
    this.updateButtonsVisibility();
    this.changeNumber(0);
    this.refresh();
  }

  setCurrencyUnit(currencyUnit) {
    this._currencyUnit = currencyUnit;
    this.refresh();
  }

  createButtons() {
    const bitmap = Managers.Images.loadSystem('ButtonSet');
    const buttonWidth = 48;
    const buttonHeight = 48;
    this._buttons = [];

    for (let i = 0; i < 6; i++) {
      const button = new Sprite_Button();
      const x = buttonWidth * i;
      const w = buttonWidth;
      button.bitmap = bitmap;
      button.setColdFrame(x, 0, w, buttonHeight);
      button.setHotFrame(x, buttonHeight, w, buttonHeight);
      button.visible = false;
      this._buttons.push(button);
      this.addChild(button);
    }

    this._buttons[0].setClickHandler(this.onButtonDown2.bind(this));
    this._buttons[1].setClickHandler(this.onButtonDown.bind(this));
    this._buttons[2].setClickHandler(this.onButtonUp.bind(this));
    this._buttons[3].setClickHandler(this.onButtonUp2.bind(this));
    this._buttons[4].setClickHandler(this.onButtonOk.bind(this));
    this._buttons[5].setClickHandler(this.onButtonCancel.bind(this));
  }

  placeButtons() {
    const numButtons = this._buttons.length;
    let totalWidth = 0;

    for (let i = 0; i < numButtons; i++) {
      totalWidth += (24 * this._zoomLevel);
    }

    let x = (this.contentsWidth() - totalWidth) / 2;
    const y = 15 * this._zoomLevel + this.lineHeight() * 2;

    for (let j = 0; j < numButtons; j++) {
      const button = this._buttons[j];
      button.x = x;
      button.y = y;
      x += (24 * this._zoomLevel);
    }
  }

  updateButtonsVisibility() {
    this.showButtons();
  }

  showButtons() {
    for (let i = 0; i < this._buttons.length; i++) {
      this._buttons[i].visible = true;
    }
  }

  hideButtons() {
    for (let i = 0; i < this._buttons.length; i++) {
      this._buttons[i].visible = false;
    }
  }

  requestUpdate() {
    this._needsRefresh = true;
  }

  refresh() {
    this.contents.clear();

    if (this._imageBackground.isReady()) {
      const w = this._imageBackground.width;
      const h = this._imageBackground.height;

      const dW = w * this._zoomLevel;
      const dH = h * this._zoomLevel;

      this.contents.bltImage(this._imageBackground, 0, 0, w, h, 0, 0, dW, dH);
    } else {
      this.requireBitmap(this._imageBackground);
    }

    let y = 15 * this._zoomLevel;

    this.drawItemName(this._item, this.textPadding(), y, undefined, false, false);
    this.drawNumber(y);

    y += this.lineHeight() * 0.8;

    this.drawTotalPrice(y);

    if (this._ingredientsWindow) {
      this._ingredientsWindow.refresh();
    }
  }

  drawNumber(y) {
    const x = this.textPadding();
    const width = this.contentsWidth() - this.textPadding() * 2;
    this.resetTextColor();
    this.drawText(`\u00d7 ${ this._number }`, x, y, width, 'right');
  }

  drawTotalPrice(y) {
    const number = this._number || 1;

    const total = this._price * number;
    const width = this.contentsWidth() - this.textPadding() * 2;
    
    const money = Managers.Text.formatMoney(total);
    this.drawText(money, this.textPadding(), y, width, 'right');
  }

  maxDigits() {
    return 3;
  }

  update() {
    if (this._needsRefresh) {
      this._needsRefresh = false;
      this.refresh();
    }

    super.update();
    this.processNumberChange();
  }

  isOkTriggered() {
    return Engine.Input.isTriggered('ok');
  }

  playOkSound() {
  }

  processNumberChange() {
    if (this.isOpenAndActive()) {
      if (Engine.Input.isRepeated('right')) {
        this.changeNumber(1);
      }
      if (Engine.Input.isRepeated('left')) {
        this.changeNumber(-1);
      }
      if (Engine.Input.isRepeated('up')) {
        this.changeNumber(10);
      }
      if (Engine.Input.isRepeated('down')) {
        this.changeNumber(-10);
      }
    }
  }

  changeNumber(amount) {
    const lastNumber = this._number;
    this._number = (this._number + amount).clamp(1, this._max);
    if (this._number !== lastNumber) {
      Managers.Sound.playCursor();
      this.refresh();
    }
  }

  updateCursor() {
    if (!this.visible) return;

    // this.setCursorRect(this.cursorX(), this.itemY(), this.cursorWidth(), this.lineHeight());
  }

  onButtonUp() {
    this.changeNumber(1);
  }

  onButtonUp2() {
    this.changeNumber(10);
  }

  onButtonDown() {
    this.changeNumber(-1);
  }

  onButtonDown2() {
    this.changeNumber(-10);
  }

  onButtonOk() {
    this.processOk();
  }

  onButtonCancel() {
    super.processCancel();
  }

  standardFontSize() {
    return Utils.convertFontSize(22, Graphics.windowZoomLevel);
  }

  scrollDown() {
    this.changeNumber(-1);
  }

  scrollUp() {
    this.changeNumber(1);
  }

  setIngredientsWindow(ingredientsWindow) {
    this._ingredientsWindow = ingredientsWindow;
  }

  standardOutlineWidth() {
    return 0;
  }

  normalColor() {
    return this.blackColor();
  }

  standardPadding() {
    return 0;
  }

  textPadding() {
    return 15 * Graphics.windowZoomLevel;
  }
}

module.exports = Window_ShopNumber;
