const Window_Selectable = require('engine/windows/Window_Selectable');

class Window_ModernItemContainer extends Window_Selectable {
  initialize(x, y, width, height, container, maxItems) {
    this._maxItems = maxItems;
    this._container = container;
    this._filteredList = [];
    super.initialize(x, y, width, height);
    this._index = -1;
    this._allowTouchWhenInactive = false;
  }

  enableWindowFrame() {
    return false;
  }

  drawEmptyItem(x, y, selectedItem) {
    this.drawItemBgPicture(x, y, selectedItem);
  }

  getItemAmount(item) {
    return item.amount;
  }

  resetTextColor() {
    this.changeTextColor('#328ffa');
  }

  verticalMargin() {
    return 10 * Graphics.windowZoomLevel;
  }

  drawItemAmount(amount, x, y) {
    if (amount <= 1) {
      return;
    }

    this.contents.outlineWidth = 3;
    this.contents.fontSize = Utils.convertFontSize(26, Graphics.windowZoomLevel);

    this.drawText(amount, x, y + 10, 24 * Graphics.windowZoomLevel, 'right');
  }

  extraSpacing() {
    return 0;
  }

  standardPadding() {
    return 0;
  }

  drawNormalItem(item, x, y, selectedItem) {
    const menuZoom = Graphics.windowZoomLevel;

    const w = 24 * menuZoom;
    const h = 24 * menuZoom;

    if (!!item && !!item.id) {
      x += this.extraSpacing();
      y += this.extraSpacing();
      
      const iconX = x;
      const iconY = y;

      this.drawItemBgPicture(x, y, selectedItem);
      this.drawItemBadge(item.iconIndex, iconX, iconY, w, h);

      const amount = this.getItemAmount(item);
      this.drawItemAmount(amount, x, y);
    } else {
      this.drawEmptyItem(x, y, selectedItem);
    }
  }

  drawItemBgPicture(x, y, selectedItem) {
    const menuZoom = Graphics.windowZoomLevel;
    this.contents.fillRect(x, y, 24 * menuZoom, 24 * menuZoom, '#d7a678');

    if (selectedItem) {
      this.contents.fillRect(x + 2 * menuZoom, y + 2 * menuZoom, 20 * menuZoom, 20 * menuZoom, '#f4dea8');
    }
  }

  getIndexUnderCursor(x, y) {
    if (x < this.x) return false;
    if (y < this.y) return false;

    if (x > this.x + this._width) return false;
    if (y > this.y + this._height) return false;

    const zoom = Graphics.windowZoomLevel;
    const mouseY = (y - this.y) / zoom;
    if (mouseY < 15) return false;
    if (mouseY > 210) return false;

    const marginX = this.x + this.spacing() + this.extraSpacing() + this.padding;
    const marginY = this.y + this.extraSpacing() + this.padding;

    for (let i = 0 ; i < this.maxItems(); i++) {
      const rect = this.itemRect(i);

      if (x < rect.x + marginX) continue;
      if (x > rect.x + marginX + rect.width) continue;
      if (y < rect.y + marginY) continue;
      if (y > rect.y + marginY + rect.height) continue;

      return i;
    }

    return false;
  }

  processMapTouch(x, y) {
    const index = this.getIndexUnderCursor(x, y);
    if (index === false) return;

    this.onClickItem(index);
  }

  onClickItem(index) {
    // Nothing to do by default
  }

  playOkSound() {
  }

  maxItems() {
    return this._maxItems;
  }

  item() {
    return this.getItem(this.index());
  }

  getItem(index) {
    return this._filteredList && index >= 0 ? this._filteredList[index] : null;
  }

  filterList() {
    this._filteredList = [];

    if (this._container) {
      if (this._category && this._category !== 'all') {
        for (const item of this._container._items) {
          if (!item) continue;
          if (item.category != this._category) continue;

          this._filteredList.push(item);
        }
      } else {
        this._filteredList = this._container._items;
      }
    }
  }

  refresh() {
    this.createContents();
    this.filterList();

    this.drawWindowBackground();
    this.drawItemBackgrounds();
    this.drawAllItems();

    this.drawSelectedItem();

    this.drawArrows();
    this.drawExtraContent();

    this._needsRefresh = false;
  }

  drawWindowBackground() {
  }

  drawExtraContent() {
  }

  drawArrows() {
    const zoom = Graphics.windowZoomLevel;
    const x = (this.width / 2) - 5.5 * zoom;

    if (this._scrollY > 0) {
      const y = 3 * zoom;
      this.drawPicture('menu/arrow_up', x, y, zoom);
    }

    if (this.topRow() < this.maxTopRow()) {
      const y = 216 * zoom;
      this.drawPicture('menu/arrow_down', x, y, zoom);
    }
  }

  update() {
    Window_Selectable.prototype.update.call(this);
    if (this._needsRefresh) {
      this.refresh();
    }

    this.updateTouch();
  }

  updateTouch() {
    if (!TouchInput.isTriggered() && !TouchInput.isRepeated()) return;

    const zoom = Graphics.windowZoomLevel;
    const mouseX = (TouchInput.mouseX - this.x) / zoom;
    const mouseY = (TouchInput.mouseY - this.y) / zoom;

    this.updateArrowTouch(mouseX, mouseY);
  }

  updateArrowTouch(mouseX, mouseY) {
    const x = (this.width / Graphics.windowZoomLevel / 2) - 4.5;

    if (mouseX < x) return;
    if (mouseX > x + 8) return;

    const topRow = this.topRow();

    if (mouseY > 0 && mouseY < 10) {
      this.cursorPageup();

      if (topRow !== this.topRow()) {
        Managers.Sound.playCursor();
      }
      return;
    }

    if (mouseY > 216 && mouseY < 224) {
      this.cursorPagedown();
      if (topRow !== this.topRow()) {
        Managers.Sound.playCursor();
      }
      return;
    }
  }

  drawItemBackgrounds() {
    const maxItems = this._maxItems;
    const maxCols = this.maxCols();
    const index = this.index();
    const maxHeight = this.contentsHeight();

    for (let i = 0; i < maxItems; i++) {
      if (i == index) continue;
      if (this.getItem(i)) continue;

      const row = Math.floor(i / maxCols);
      if (row < this.topRow()) continue;

      const rect = this.itemRect(i);
      if (rect.y + rect.height >= maxHeight) continue;

      this.drawEmptyItem(rect.x, rect.y, false);
    }
  }

  drawSelectedItem() {
    const index = this.index();
    if (index < 0) return;

    const rect = this.itemRect(index);

    // Avoid drawing outside the regular area
    if (rect.y < 0) return;
    if (rect.y > 210 * Graphics.windowZoomLevel) return;

    const item = this.getItem(index);
    this.drawNormalItem(item, rect.x, rect.y, true);
  }

  drawItem(index) {
    if (index == this.index()) return;

    const row = Math.floor(index / this.maxCols());
    if (row < this.topRow()) return;

    const item = this.getItem(index);
    if (item) {
      const rect = this.itemRect(index);
      rect.width -= this.textPadding();

      const maxHeight = this.contentsHeight();
      if (rect.y + rect.height >= maxHeight) return;

      this.drawNormalItem(item, rect.x, rect.y, false);
    }
  }

  itemWidth() {
    return this.lineHeight();
  }

  itemHeight() {
    return this.lineHeight();
  }

  lineHeight() {
    return 28 * Graphics.windowZoomLevel;
  }

  maxCols() {
    return 5;
  }

  spacing() {
    return 0;
  }

  updateCursor() {
    this._needsRefresh = true;
    this._refreshExpected = true;
  }

  updateHelp() {
    this.setHelpWindowItem(this.item());


  }

  callUpdateHelp() {
    if (this._helpWindow || this._infoWindow) {
      this.updateHelp();
    }

    this.callHandler('updateHelp');
  }

  getIndexLabel(index) {
    const item = this.getItem(index);
    if (!item) return '';

    if (typeof(item) == "string") {
      return Managers.Text.item(item);
    }

    return Managers.Text.item(item.id);
  }

  setCategory(category) {
    this._category = category;
    this.refresh();
  }

  selectByTouch(index) {
    if (index >= 0) {
      this.callHandler('beforeSelectByTouch');
    }
    super.selectByTouch(index);
  }

  originalItemRect(index) {
    return super.itemRect(index);
  }

  itemRect(index) {
    const rect = this.originalItemRect(index);

    rect.y += 8 * Graphics.windowZoomLevel;
    return rect;
  }
}

module.exports = Window_ModernItemContainer;