const Window_Hud = require('./Window_Hud');

class Window_ToolHud extends Window_Hud {
  initialize() {
    super.initialize(Graphics.width - this.windowWidth(), Graphics.height - this.windowHeight(), this.windowWidth(), this.windowHeight());
    this._tools = [];
    this._items = [];
    this._includeEmptyTool = false;
    this._mouseText = '';
    this._mouseTextDrawn = false;
    this.preloadImages();
    this.refresh();
  }

  windowWidth() {
    return Graphics.width;
  }

  windowHeight() {
    return Graphics.height;
  }

  standardPadding() {
    return 0;
  }

  preloadImages() {
    Managers.Images.loadPicture('toolHud/number_font2');
  }

  drawPicture(filename, x, y, zoomLevel) {
    if (x === undefined) x = 0;
    if (y === undefined) y = 0;

    const me = this;
    const folderName = 'toolHud';

    const bitmap = Managers.Images.loadPicture(`${folderName}/${filename}`);
    if (!bitmap.isReady()) {
      bitmap.addLoadListener(() => {
        me.requestUpdate();
      });
    }

    const width = bitmap._canvas.width;
    const height = bitmap._canvas.height;
    let drawWidth = bitmap._canvas.width;
    let drawHeight = bitmap._canvas.height;

    drawWidth *= zoomLevel || 1;
    drawHeight *= zoomLevel || 1;

    this.contents.bltBitmap(bitmap, 0, 0, width, height, x, y, drawWidth, drawHeight);
  }

  refresh() {
    if (!this.shouldBeVisible()) return;

    if (this.contents) {
      if (Managers.Config.drawToolBar) {
        this._tools = Managers.Tools.tools(true);
      } else {
        this._tools = [];
      }

      if (Managers.Config.drawItemBar) {
        const minItem = 10 * Managers.Items.itemBarIndex;
        this._items = Managers.Items.items().slice(minItem, minItem + 10);
      } else {
        this._items = [];
      }
      while (this._items.length < 10) {
        this._items.push(null);
      }

      this.contents.clear();
      this.drawHud();
    }
  }

  itemsToDraw() {
    return this._items;
  }

  toolsToDraw() {
    return this._tools;
  }

  getIconSize() {
    return 27 * Graphics.hudZoomLevel;
  }

  toolbarTop() {
    return Graphics.height - 35 * Graphics.hudZoomLevel;
  }

  toolbarLeft() {
    const width = 313 * Graphics.hudZoomLevel;
    return Graphics.width - width;
  }

  itembarTop() {
    return Graphics.height - 35 * Graphics.hudZoomLevel;
  }

  itembarLeft() {
    return 0;
  }

  itembarWidth() {
    const x = this.firstItemLeft();
    const iconSize = this.getIconSize();

    return x + iconSize * 10 + 8 * Graphics.hudZoomLevel;
  }

  firstItemLeft() {
    let x = this.itembarLeft();
    x += (32 * Graphics.hudZoomLevel);

    return x;
  }

  shouldBeVisible() {
    if (Switches.forceToolHud) return true;

    if (!Switches.toolHudEnabled) return false;
    if ($gameSystem.isPlayerSleeping) return false;
    if ($gameSystem.isPlayingCutscene()) return false;
    if (Switches.isInsideFestival) return false;

    if (Managers.Scenes._scene instanceof GameScenes.GameMap) {
      if (Managers.Scenes._scene._playingDiaryAnimation) return false;
      if (Managers.Scenes._scene.menuCalling) return false;
    }

    return true;
  }

  drawTargetName() {
    $gameTemp.clearNamedObject();

    if (!Managers.Tools.isToolTypeEquipped('magnifier')) {
      return;
    }

    if ($gamePlayer.isRiding()) {
      return;
    }

    if (!Switches.finishedTutorial && Switches.startedTutorial) {
      return;
    }

    if ($gameSystem.isPlayingCutscene()) {
      return;
    }

    if ($gamePlayer.isMoving()) {
      return;
    }

    const checkPosition = (x, y) => {
      const creatures = $gameMap.creaturesXy(x, y);
      if (creatures && creatures.length) {
        for (let creature of creatures) {
          if (!creature._creatureData) {
            continue;
          }

          if (!creature._creatureData.creatureName) {
            continue;
          }

          this._drawCreatureName(creature);
          return true;
        }
      }

      const farmObjects = $gameMap.farmObjectsXyEx(x, y);
      if (farmObjects && farmObjects.length) {
        for (let object of farmObjects) {
          if (!object) continue;
          if (!object._farmObjectData) continue;
          
          if (object._farmObjectData.modelName) {
            if (object._farmObjectData.modelName.includes('-crop')) {
              if (this._drawTargetCrop(object)) {
                return true;
              }

              continue;
            }

            if (this._drawTargetObject(object)) {
              return true;
            }
          }

          if (object._farmObjectData.itemName && this._drawTargetItem(object)) {
            return true;
          }

          if (object._farmObjectData._rockType) {
            this._drawTargetRock(object, object._farmObjectData._rockType);
            return true;
          }
        }
      }
    };

    if (Managers.Config.enableMouse && TouchInput.wasMouseUsed()) {
      checkPosition($gameMap.canvasToMapX(TouchInput.mouseX), $gameMap.canvasToMapY(TouchInput.mouseY));
      return;
    }

    const tile = TileHelper.getSmartFrontTile(1);
    if (tile && checkPosition(tile.x, tile.y)) {
      return;
    }
  }

  _drawTargetCrop(event) {
    const data = event._farmObjectData;
    const itemId = data.modelName.replace('-crop', '');
    const itemData = Managers.Items.getItemData(itemId);

    if (!itemData) {
      return false;
    }

    this._drawItemName(event, itemId, itemData);
    return true;
  }

  _drawTargetObject(event) {
    const data = event._farmObjectData;
    const modelName = data.modelName;

    const drawName = (name) => {
      $gameTemp.setNamedObject(event);
      this._drawTargetName(name);
    };

    switch(modelName) {
      case 'weed':
      case 'small-weed':
        drawName(Managers.Text.item('weeds'));
        return true;
      case 'sprinkler':
      case 'super-sprinkler':
      case 'gate':
      case 'fence':
      // case 'barn':
      // case 'coop':
        drawName(Managers.Text.item(modelName));
        return true;
      case 'stone':
        drawName(t("Stone"));
        return true;
      case 'treeLog':
        drawName(t("Tree Log (Level 1)"));
        return true;
      case 'stump':
        drawName(t("Tree Stump (Level 2)"));
        return true;
      case 'termite-nest':
        drawName(t("Termite Nest"));
        return true;
      case 'fallen-tree':
        drawName(t("Fallen Tree (Level 4)"));
        return true;
      case 'rock-1':
        drawName(t("Rock (Level 1)"));
        return true;
      case 'rock-2':
        drawName(t("Rock (Level 2)"));
        return true;
      case 'rock-3':
        drawName(t("Rock (Level 3)"));
        return true;
      case 'rock-4':
        drawName(t("Rock (Level 4)"));
        return true;
      case 'boulder':
        drawName(t("Boulder (Level 4)"));
        return true;
    }

    if (modelName.startsWith('regular-tree')) {
      if ($gameMap.isFarm()) {
        drawName(t("Tree (Level 3)"));
        return true;
      }
    }

    return false;
  }

  _drawTargetRock(event, rockType) {
    $gameTemp.setNamedObject(event);
    this._drawTargetName(t(`rock-name-${ rockType }`));
  }

  _drawTargetItem(event) {
    const data = event._farmObjectData;
    const itemId = data.itemName;
    const itemData = Managers.Items.getItemData(itemId);

    if (!itemData) {
      return false;
    }

    this._drawItemName(event, itemId, itemData);
    return true;
  }

  _drawItemName(event, itemId, itemData) {
    $gameTemp.setNamedObject(event);
    this._drawTargetName(Managers.Text.item(itemId));
  }

  _drawCreatureName(event) {
    $gameTemp.setNamedObject(event);
    this._drawTargetName(event._creatureData.creatureName);
  }

  _drawTargetName(name) {
    const bitmap = Managers.Images.loadPicture('info_box');
    if (!bitmap.isReady()) {
      bitmap.addLoadListener(() => {
        this.requestUpdate();
      });
      return;
    }

    let y = this.toolbarTop() - bitmap.height * Graphics.hudZoomLevel;
    const x = Graphics.width - bitmap.width * Graphics.hudZoomLevel;
    const w = bitmap.width * Graphics.hudZoomLevel;
    const h = bitmap.height * Graphics.hudZoomLevel;

    this.contents.bltBitmap(bitmap, 0, 0, bitmap.width, bitmap.height, x, y, w, h);

    y -= 2 * Graphics.hudZoomLevel;
    this.drawText(name, x, y, w, 'center');
  }

  drawToolBar() {
    if (!Managers.Config.drawToolBar) return;

    let x = this.toolbarLeft();
    let y = this.toolbarTop();

    const tools = this.toolsToDraw();

    const zoomLevel = Graphics.hudZoomLevel;
    const iconSize = 27 * zoomLevel;

    x += 8 * zoomLevel;

    let slotX = x;
    let slotY = y;

    this.drawPicture('tools_slot', slotX, slotY, zoomLevel);
    x += (2 * zoomLevel);
    y += (7 * zoomLevel);

    this.resetTextColor();

    slotX = x;
    slotY = y;
    const minIndex = this._includeEmptyTool ? -1 : 0;

    for (let i = minIndex; i < tools.length; i++) {
      let hotkey = '';

      if (i < 0) {
        if (!Managers.Tools.toolId || Managers.Tools.toolId == 'none') {
          this.drawPicture('selected_slot', slotX + 2 * zoomLevel, slotY + 2 * zoomLevel, zoomLevel);
        }
      } else if (tools[i]) {
        if (tools[i].id == Managers.Tools.toolId) {
          this.drawPicture('selected_slot', slotX + 2 * zoomLevel, slotY + 2 * zoomLevel, zoomLevel);
        }

        this.drawItemBadge(tools[i].iconIndex, slotX + 2 * zoomLevel, slotY + 2 * zoomLevel, Core.BasicWindow._iconWidth * zoomLevel, Core.BasicWindow._iconHeight * zoomLevel);
        hotkey = tools[i].hotkey;
      }

      if (hotkey !== undefined) {
        const maxWidth = iconSize;
        let numberY = slotY + iconSize - 9 * zoomLevel;
        let numberX = slotX;

        this.drawValueUsingImage(numberX, numberY, hotkey, maxWidth, 'number_font2', 5, 7, true);
      }

      slotX += iconSize;
    }

    this.drawBigToolBadge();
  }

  maxItemTypes() {
    const maxItems = Managers.Items.maxItemTypes();
    return maxItems;
  }

  drawItemBar() {
    if (!Managers.Config.drawItemBar) return;

    let i;
    const iconSize = this.getIconSize();
    let x = this.itembarLeft();
    let y = this.itembarTop();

    const items = this.itemsToDraw();
    const zoomLevel = Graphics.hudZoomLevel;

    this.drawPicture('item_slot', x, y, zoomLevel);
    this.drawBigBadge();

    x = this.firstItemLeft();
    y += (7 * zoomLevel);

    let iconX = 0;
    let iconY = 0;

    for (i = 0; i < items.length; i++) {
      let amount = 1;

      if (i == Managers.Items._itemIndex && !Managers.Items.displayItemId) {
        this.drawPicture('selected_slot', x + 2 * zoomLevel, y + 2 * zoomLevel, zoomLevel);

        iconY = y - Core.BasicWindow._iconHeight * zoomLevel / 2;
        iconX = x + iconSize - Core.BasicWindow._iconWidth * zoomLevel / 2;
      }

      if (!items[i]) {
        x += iconSize;
        continue;
      }

      if (Managers.Items.itemTypeIsStackable(items[i].type)) {
        amount = Managers.Items.numItems(items[i]);
      }

      this.drawItemBadge(items[i].iconIndex, x + 2 * zoomLevel, y + 2 * zoomLevel, Core.BasicWindow._iconWidth * zoomLevel, Core.BasicWindow._iconHeight * zoomLevel);

      if (amount > 1) {
        const maxWidth = iconSize;
        let numberY = y + iconSize - 7 * zoomLevel;
        let numberX = x;

        this.drawValueUsingImage(numberX, numberY, amount > 999 ? '999+' : amount, maxWidth, 'number_font2', 5, 7, true);
      }

      x += iconSize;
    }

    const maxTabs = Managers.Items.maxItemBarIndex() + 1;
    const currTab = (Managers.Items.itemBarIndex || 0) + 1;

    if (maxTabs >= currTab && maxTabs <= 3) {
      const pictureName = `item_tab_${currTab}_of_${maxTabs}`;
      this.drawPicture(pictureName, x + 2 * zoomLevel, y, zoomLevel);
    }

    if (iconX && iconY) {
      this.drawIcon(570, iconX - 22, iconY + 20, zoomLevel);
    }
  }

  drawImagePiece(fileName, x, y, sx, sy, sw, sh, zoomLevel) {
    super.drawImagePiece(fileName, x, y, sx, sy, sw, sh, zoomLevel, false);
  }

  drawBigBadge() {
    let y;
    let x;
    const iconSize = this.getIconSize();
    let amount = 0;
    const zoomLevel = Graphics.hudZoomLevel;

    x = 0 - zoomLevel;
    y = this.itembarTop() + (6 * zoomLevel);

    const itemId = Managers.Items.getSelectedItemId();

    if (!!itemId && itemId !== 'none') {
      const itemData = Managers.Items.getItemData(itemId);

      if (itemData) {
        const iconX = x;
        const iconY = y;

        const w = Core.BasicWindow._iconWidth;
        const h = Core.BasicWindow._iconHeight;
        if (itemData.characterName && itemData.characterIndex !== undefined) {
          // this.drawCharacterIconCentered(itemData.characterName, itemData.characterIndex, iconX + 5 + 1.5 * zoomLevel, iconY + 1.5 * zoomLevel, w, h, zoomLevel);
        } else {
          this.drawItemBadge(itemData.iconIndex, iconX + 4 * zoomLevel, iconY + 3 * zoomLevel, w * zoomLevel, h * zoomLevel);
        }
      }

      amount = Managers.Items.selectedItem.amount || 1;

      if (amount > 1) {
        const maxWidth = this.getIconSize();
        const numberY = y + iconSize - 9 * zoomLevel;
        const numberX = x + 4 * zoomLevel;

        this.drawValueUsingImage(numberX, numberY, amount > 999 ? '999' : amount, maxWidth, 'number_font2', 5, 7, true);
      }
    }
  }

  drawBigToolBadge() {
    const zoomLevel = Graphics.hudZoomLevel;
    const iconSize = this.getIconSize();
    const y = this.toolbarTop() + (6 * zoomLevel);
    const x = Graphics.width - (31 * zoomLevel);
    let amount = 0;

    const toolId = Managers.Tools.toolId;

    if (toolId) {
      const toolData = Managers.Items.getItemData(toolId);

      if (toolData) {
        const iconX = x;
        const iconY = y;

        if (toolData.characterName && toolData.characterIndex !== undefined) {
          this.drawCharacterIconCentered(toolData.characterName, toolData.characterIndex, iconX, iconY, iconSize, iconSize, 1);
        } else {
          const w = Core.BasicWindow._iconWidth * zoomLevel;
          const h = Core.BasicWindow._iconHeight * zoomLevel;
          this.drawItemBadge(toolData.iconIndex, iconX + 4 * zoomLevel, iconY + 3 * zoomLevel, w, h);
        }
      }

      if (toolId.includes('watering-can')) {
        amount = Managers.Tools.water;

        const maxWidth = this.getIconSize();
        const numberY = y + iconSize - 9 * zoomLevel;
        const numberX = x + 4 * zoomLevel;

        this.drawValueUsingImage(numberX, numberY, amount, maxWidth, 'number_font2', 5, 7, true);
      }
    }
  }

  drawHud() {
    if (!this.shouldBeVisible()) return;

    this.resetTextColor();
    this.drawItemBar();
    this.drawToolBar();
    this.drawTargetName();

    if (Managers.Config.enableMouse && this._mouseText) {
      let x = TouchInput.mouseX;
      let y = TouchInput.mouseY;
      let align = 'left';

      if (y < 200) {
        y += 15;
      } else {
        y -= 30;
      }

      if (x > Graphics.width - 200) {
        x -= 400;
        align = 'right';
      }

      this.contents.outlineWidth = 4;
      this.contents.outlineColor = 'rgba(0, 0, 0, 1)';
      this.changeTextColor('#FFFFFF');
      this.drawText(this._mouseText, x, y, 400, align);
      this._mouseTextDrawn = true;
    } else {
      this._mouseTextDrawn = false;
    }
  }

  update() {
    if (this._isVisible) {
      if (Graphics.skipFrame) return;
      // if (Graphics.fps < 30 && Graphics.skipFrame) return;
    }

    super.update();
    let shouldRefresh = !!this._isDirty;

    if (!this.visible) {
      if (this._isVisible) {
        this.contents.clear();
      }

      this._isVisible = false;
      return;
    }

    if (!this._isVisible) {
      shouldRefresh = true;
    } else if (this._toolId != Managers.Tools.toolId) {
      shouldRefresh = true;
    } else if (this.updateMouseText()) {
      shouldRefresh = true;
    } else if (this._oldMapId != $gameMap._mapId) {
      shouldRefresh = true;
    } else if (this._mouseTextDrawn && !this._mouseText) {
      shouldRefresh = true;
    } else if (this._itemBarIndex != Managers.Items._itemBarIndex) {
      shouldRefresh = true;
    } else if (this._itemIndex != Managers.Items._itemIndex) {
      shouldRefresh = true;
    }

    // if ($gamePlayer.isMoving()) {
    //   shouldRefresh = true;
    // }

    this._isVisible = true;
    this._isDirty = false;
    this._toolId = Managers.Tools.toolId;
    this._oldMapId = $gameMap._mapId;
    this._itemBarIndex = Managers.Items._itemBarIndex;
    this._itemIndex = Managers.Items._itemIndex;

    if (this._smallHud !== Managers.Config.smallHud) {
      this._smallHud = Managers.Config.smallHud;
      shouldRefresh = true;
    }

    if (shouldRefresh) {
      this.refresh();
    }
  }

  requestUpdate() {
    this._isDirty = true;
    this._mouseText = '';
  }

  updateMouseText() {
    if (!Managers.Config.enableMouse) return false;
    if (this._lastMouseX == TouchInput.mouseX && this._lastMouseY == TouchInput.mouseY) return false;

    this._mouseText = '';

    this._lastMouseX = TouchInput.mouseX;
    this._lastMouseY = TouchInput.mouseY;

    if (!this.visible) return false;

    let x;
    let y;
    const iconSize = this.getIconSize();
    const zoomLevel = Graphics.hudZoomLevel;

    //Check if the player hovered on the item big icon
    x = 0;
    y = Graphics.height - iconSize;

    if (TouchInput.mouseY >= y && TouchInput.mouseY <= (y + iconSize) && TouchInput.mouseX > x && TouchInput.mouseX < x + iconSize) {
      return this.showBigIconName();
    }

    //Check if the player hovered on the tool big icon
    x = Graphics.width - iconSize;
    y = Graphics.height - iconSize;

    if (TouchInput.mouseY >= y && TouchInput.mouseY <= (y + iconSize) && TouchInput.mouseX > x && TouchInput.mouseX < x + iconSize) {
      return this.showToolBigIconName();
    }

    y = this.toolbarTop() + 7;

    //If an icon from the toolbar was touched
    if (TouchInput.mouseY >= y && TouchInput.mouseY < y + iconSize) {
      x = this.toolbarLeft();
      x += 8 * zoomLevel;
      if (this._includeEmptyTool) {
        x += iconSize;
      }

      if (TouchInput.mouseX >= x) {
        const tools = this.toolsToDraw();
        const toolIndex = Math.floor((TouchInput.mouseX - x) / iconSize);
        if (toolIndex < tools.length) {
          const tool = tools[toolIndex];

          if (tool) {
            return this.showTool(tool);
          }

          return false;
        } else if (toolIndex == tools.length) {
          return false;
        }
      }
    }

    y = this.itembarTop() + 7;

    //If an icon from the itembar was touched
    if (TouchInput.mouseY >= y && TouchInput.mouseY < y + iconSize) {
      x = this.firstItemLeft();
      //Just a small correction
      x += 1 * zoomLevel;

      if (TouchInput.mouseX >= x) {
        const items = this.itemsToDraw();
        const itemIndex = Math.floor((TouchInput.mouseX - x) / iconSize);
        if (itemIndex < items.length) {
          const item = items[itemIndex];

          if (item) {
            return this.showItem(item);
          }
        }
      }
    }

    return false;
  }

  showBigIconName() {
    if (Managers.Items.isHoldingItem()) {
      const animalData = Managers.Items.getItemInfo('animalData');

      if (animalData) {
        return this.showText(animalData.creatureName);
      }
    }

    const itemId = Managers.Items.getSelectedItemId();
    if (!itemId || itemId == 'none') return false;

    return this.showItemId(itemId);
  }

  showToolBigIconName() {
    const itemId = Managers.Tools.toolId;
    if (!itemId) return false;

    return this.showItemId(itemId);
  }

  showTool(tool) {
    return this.showText(Managers.Text.item(tool.id));
  }

  showItem(item) {
    return this.showItemId(item.id);
  }

  showItemId(itemId) {
    return this.showText(Managers.Text.item(itemId));
  }

  showText(text) {
    if (!text) return false;

    this._mouseText = text;
    return true;
  }

  standardFontFace() {
    if ($gameSystem.isChinese()) {
      return 'SimHei, Heiti TC, sans-serif';
    } else if ($gameSystem.isKorean()) {
      return 'Dotum, AppleGothic, sans-serif';
    } else {
      return 'GameFont';
    }
  }

  standardOutlineWidth() {
    return 0;
  }

  normalColor() {
    return this.blackColor();
  }

  drawValueUsingImage(x, y, value, maxWidth, imageName, charW, charH, rightAlignText) {
    super.drawValueUsingImage(x, y, value, maxWidth, imageName, charW, charH, rightAlignText, 'toolHud', Graphics.hudZoomLevel, true);
  }
}

module.exports = Window_ToolHud;
