const Window_Hud = require('./Window_Hud');

class Window_TimeHud extends Window_Hud {
  initialize() {
    super.initialize(0, 0, this.windowWidth(), this.windowHeight());
    this.refresh();
    this.opacity = 0;
    this.preloadImages();
  }

  windowWidth() {
    return Graphics.width;
  }

  windowHeight() {
    return 90 * Graphics.timeHudZoomLevel;
  }

  standardPadding() {
    return 0;
  }

  preloadImages() {
    Managers.Images.loadPicture('timeHud/empty_slot');
    Managers.Images.loadPicture('timeHud/fatigue_bar');
    Managers.Images.loadPicture('timeHud/fatigue_icon');
    Managers.Images.loadPicture('timeHud/fatigue_slot');
    Managers.Images.loadPicture('timeHud/icon_background');
    Managers.Images.loadPicture('timeHud/money_bar');
    Managers.Images.loadPicture('timeHud/money_font');
    Managers.Images.loadPicture('timeHud/stamina_icon');
    Managers.Images.loadPicture('timeHud/stamina_bar');
    Managers.Images.loadPicture('timeHud/stamina_slot');
    Managers.Images.loadPicture('timeHud/mail');
  }

  drawPictureRotated(filename, x, y, angle, widthMultiplier) {
    if (x === undefined) x = 0;
    if (y === undefined) y = 0;
    widthMultiplier = widthMultiplier || 1;

    const bitmap = Managers.Images.loadPicture(`timeHud/${filename}`);
    const me = this;

    let drawBitmap;

    drawBitmap = () => {
      const width = bitmap._image ? bitmap._image.width : bitmap._canvas.width;
      const height = bitmap._image ? bitmap._image.height : bitmap._canvas.height;

      const zoomLevel = Graphics.timeHudZoomLevel;
      let drawWidth = width * zoomLevel;
      const drawHeight = height * zoomLevel;

      x *= zoomLevel;
      y *= zoomLevel;

      drawWidth *= widthMultiplier;

      const rotationBitmap = new Bitmap(drawWidth, drawHeight);
      const ctx = rotationBitmap._canvas.getContext('2d');

      ctx.save();
      ctx.translate(drawWidth / 2, drawHeight / 2);
      ctx.rotate(angle * Math.PI / 180);

      rotationBitmap.bltImage(bitmap, 0, 0, width, height, -(drawWidth / 2), -(drawHeight / 2), drawWidth, drawHeight);
      ctx.restore();
      me.contents.blt(rotationBitmap, 0, 0, drawWidth, drawHeight, x, y, drawWidth, drawHeight);
    };

    drawBitmap.bind(this);
    bitmap.addLoadListener(drawBitmap);
  }

  drawPicture(filename, x, y, widthMultiplier) {
    if (x === undefined) x = 0;
    if (y === undefined) y = 0;
    widthMultiplier = widthMultiplier || 1;

    const bitmap = Managers.Images.loadPicture(`timeHud/${filename}`);
    const me = this;

    let drawBitmap;

    drawBitmap = () => {
      const width = bitmap._image ? bitmap._image.width : bitmap._canvas.width;
      const height = bitmap._image ? bitmap._image.height : bitmap._canvas.height;

      const zoomLevel = Graphics.timeHudZoomLevel;
      let drawWidth = width * zoomLevel;
      const drawHeight = height * zoomLevel;

      x *= zoomLevel;
      y *= zoomLevel;

      drawWidth *= widthMultiplier;
      me.contents.bltImage(bitmap, 0, 0, width, height, x, y, drawWidth, drawHeight);
    };

    if (bitmap.isReady()) {
      drawBitmap();
    } else {
      bitmap.addLoadListener(() => {
        this._dirty = true;
      });
    }
  }

  refresh() {
    if (this.contents) {
      this.contents.clear();
      this.drawHud();
    }
  }

  drawDayNumber(x, y, zoomLevel = null) {
    super.drawDayNumber(x, y, Graphics.timeHudZoomLevel);
  }

  drawMoneyValue(x, y, maxWidth) {
    let goldValueToDraw = Math.floor(Managers.Items.gold);
    let playCoinSound = false;
    if (goldValueToDraw != this._drawnGoldValue) {
      if (!isNaN(Number(this._drawnGoldValue))) {
        let diff = this._drawnGoldValue - Math.floor(Managers.Items.gold);

        if (Math.abs(diff) > 10) {
          if (Math.abs(diff) > 1000) {
            diff = Math.floor(diff / 10);
          } else if (Math.abs(diff) > 100) {
            diff = Math.floor(diff / 5);
          } else {
            diff = Math.floor(diff / 3);
          }

          goldValueToDraw = this._drawnGoldValue - diff;
        }

        playCoinSound = true;
      }
    }

    const goldStr = Math.floor(goldValueToDraw).toString();
    this._drawnGoldValue = goldValueToDraw;

    if (goldStr.length > 7) {
      this.contents.textColor = '#85412A';
      this.contents.outlineWidth = 0;
      this.drawHudText(t('$ A LOT'), x, y - 6, maxWidth - 2, 'right');
      return;
    }

    if (playCoinSound) {
      Managers.Sound.playCoin();
    }

    this.drawValueUsingImage(x, y, goldStr, maxWidth, 'money_font', 8, 9, true);
    if (goldStr.length < 6) {
      this.drawValueUsingImage(x, y, '$', maxWidth, 'money_font', 8, 9, false);
    }
  }

  drawTimeValue(x, y) {
    const timeStr = Managers.Time.getTimeStr();
    this.drawValueUsingImage(x, y, timeStr, 50, 'hour_font', 9, 10, false);
  }

  drawHud() {
    if (!Switches.forceTimeHud) {
      if (!Switches.timeHudEnabled) return;
      if ($gameSystem.isPlayerSleeping) return;
      if ($gameSystem.isPlayingCutscene()) return;
    }

    this.resetTextColor();

    let x = 10;
    const y = 0;

    if (!this.isPlayerUnderHealth()) {
      if (Switches.newMail) {
        this.drawPicture('mail', x + 44, y + 8);
      }

      this.drawPicture('money_bar', x + 0, y + 43);
      this.drawPicture('icon_background', x + 2, y + 10);

      const staminaTicks = this.staminaTicks();
      const fatigueTicks = this.fatigueTicks();

      this.drawPicture('fatigue_bar', x + 24, y + 31);
      if (fatigueTicks > 10) {
        const left = x + 24 + (fatigueTicks - 10) * 5;
        this.drawPicture('fatigue_bar', left, y + 31);
      }

      // Draw the stamina bar a second time over the first one, but further to the right, to increase the total width of the bar
      if (staminaTicks > 10) {
        const left = x + 26;
        const top = y + 16;

        this.drawPicture('fatigue_bar', left, top + 3);
        const newLeft = left + (staminaTicks - 10) * 5;

        this.drawPicture('stamina_bar', newLeft, top);
      } else {
        this.drawPicture('stamina_bar', x + 26, y + 16);
      }

      this.drawPicture('fatigue_icon', x + 12, y + 28);
      this.drawPicture('stamina_icon', x + 16, y + 16);

      const currentStaminaTicks = Math.round(Managers.Health.staminaLevel / 100 * staminaTicks);
      const currentFatigueTicks = Math.round((100 - Managers.Health.fatigueLevel) / 100 * fatigueTicks);

      const maxTicks = Math.max(staminaTicks, fatigueTicks);
      for (let i = 0; i < maxTicks; i++) {
        let slotX = 33 + (i * 5);
        let slotY = 21;

        let pictureName = 'stamina_slot';

        if (i < staminaTicks) {
          if (currentStaminaTicks <= i) {
            pictureName = 'empty_slot';
          }
          this.drawPicture(pictureName, x + slotX, y + slotY);
        }

        slotX -= 3;
        slotY += 12;

        if (i < fatigueTicks) {
          pictureName = 'fatigue_slot';
          if (currentFatigueTicks <= i) {
            pictureName = 'empty_slot';
          }
          this.drawPicture(pictureName, x + slotX, y + slotY);
        }
      }

      this.drawMoneyValue(x + 5, y + 58, 53);
    }

    if (!this.isPlayerUnderCalendar()) {
      const weatherPicture = this.getWeatherPicture();
      const drawWidth = Graphics.width / Graphics.timeHudZoomLevel;

      x = x + (drawWidth - 640);

      this.drawPicture(weatherPicture, x + 542, y);

      this.drawDayNumber(x + 561, y + 10);
      this.drawPicture('day_bar', x + 554, y + 50);
      this.drawClock(x + 564, y + 65);

      this.drawDayName(x + 556, y + 59);
      this.drawTimeValue(x + 581, y + 68);
    }
  }

  getWeatherPicture() {
    switch (Managers.Weather.currentWeather) {
      case WeatherType.RAIN:
        return 'rain';
      case WeatherType.STORM:
        return 'storm';
      case WeatherType.SNOW:
        return 'snow';
      case WeatherType.SNOWSTORM:
        return 'snowstorm';
      case WeatherType.HEAT:
        return 'heat';
      default:
        return 'sun';
    }
  }

  drawClock(x, y) {
    this.drawPicture('clock', x, y);

    const realMinute = (Managers.Time.minute * 10) + Managers.Time.seconds;
    let angle = realMinute * 6;

    this.drawPictureRotated('minute_finger', x, y, angle);

    angle = Math.floor((Managers.Time.hour * 30) + realMinute / 2);
    this.drawPictureRotated('hour_finger', x, y, angle);
  }

  drawDayName(x, y) {
    const textWidth = 66;
    let dayName = t(Managers.Time.getDayName());
    dayName = dayName.toUpperCase();

    this.contents.fontSize = Utils.convertFontSize(26, Graphics.timeHudZoomLevel);
    y = this.calcTextY(y * Graphics.timeHudZoomLevel) / Graphics.timeHudZoomLevel;

    this.contents.textColor = '#FB3153';
    this.contents.outlineWidth = 0;
    this.contents.outlineColor = 'rgba(0, 0, 0, 0)';
    this.drawHudText(dayName, x, y, textWidth, 'center');
  }

  lineHeight() {
    return 16 + (8 * Graphics.timeHudZoomLevel);
  }

  drawHudText(text, x, y, width, align) {
    x *= Graphics.timeHudZoomLevel;
    y *= Graphics.timeHudZoomLevel;
    width *= Graphics.timeHudZoomLevel;

    this.drawText(text, x, y, width, align);
  }

  update() {
    if (this._isVisible) {
      if (Graphics.skipFrame) return;
    }

    super.update();
    let shouldRefresh = !!this._dirty;

    if (!this.visible) {
      if (this._isVisible) {
        this.contents.clear();
      }

      this._isVisible = false;
      return;
    }

    if (!this._isVisible) {
      shouldRefresh = true;
    }

    if (this._weekDay != Managers.Time.weekDay || this._day != Managers.Time.day || this._hour != Managers.Time.hour || this._minute != Managers.Time.minute || this._second != Managers.Time.seconds) {
      shouldRefresh = true;
    }

    if (this._drawnGoldValue != Managers.Items.gold) {
      shouldRefresh = true;
    }

    if (!shouldRefresh) {
      if (this.isPlayerUnderCalendar() || this.isPlayerUnderHealth()) {
        shouldRefresh = true;
      }
    }

    this._isVisible = true;
    this._weekDay = Managers.Time.weekDay;
    this._day = Managers.Time.day;
    this._hour = Managers.Time.hour;
    this._minute = Managers.Time.minute;
    this._second = Managers.Time.seconds;
    this._dirty = false;

    if (this._smallHud !== Managers.Config.smallHud) {
      this.move(this.x, this.y, this.windowWidth(), this.windowHeight());
      this._smallHud = Managers.Config.smallHud;
      this.createContents();
      shouldRefresh = true;
    }

    if (shouldRefresh) {
      this.refresh();
    }
  }

  calendarWidth() {
    return 70 * Graphics.timeHudZoomLevel;
  }

  calendarHeight() {
    return 75 * Graphics.timeHudZoomLevel;
  }

  staminaTicks() {
    return Math.floor(Managers.Health.maxStamina / 10).clamp(10, 20);
  }

  fatigueTicks() {
    return Math.floor(Managers.Health.maxFatigue / 10).clamp(10, 20);
  }

  staminaWidth() {
    const staminaTicks = this.staminaTicks();

    return 6.5 * staminaTicks * Graphics.timeHudZoomLevel;
  }

  staminaHeight() {
    return 75 * Graphics.timeHudZoomLevel;
  }

  isPlayerUnderCalendar() {
    const playerY = $gameMap.mapToCanvasY($gamePlayer.y);
    const playerX = $gameMap.mapToCanvasX($gamePlayer.x);

    if (playerY <= this.calendarHeight() && playerX >= (Graphics.width - this.calendarWidth())) {
      return true;
    } else {
      return false;
    }
  }

  isPlayerUnderHealth() {
    const playerY = $gameMap.mapToCanvasY($gamePlayer.y);
    const playerX = $gameMap.mapToCanvasX($gamePlayer.x);

    if (playerY <= this.staminaHeight() && playerX <= this.staminaWidth()) {
      return true;
    } else {
      return false;
    }
  }

  requestRefresh() {
    this._dirty = true;
  }

  drawValueUsingImage(x, y, value, maxWidth, imageName, charW, charH, rightAlignText, hudName, zoomLevel) {
    super.drawValueUsingImage(x, y, value, maxWidth, imageName, charW, charH, rightAlignText, 'timeHud', Graphics.timeHudZoomLevel);
  }
}

module.exports = Window_TimeHud;
