const Window_Selectable = require('engine/windows/Window_Selectable');
//-----------------------------------------------------------------------------
// Window_NameInput
//
// The window for selecting text characters on the name input screen.

/* globals JAPAN1, JAPAN2, JAPAN3, RUSSIA, LATIN1, LATIN2 */

class Window_NameInput extends Window_Selectable {
  initialize(editWindow) {
    const x = editWindow.x;
    const y = editWindow.y + editWindow.height + 8;
    const width = editWindow.width;
    const height = this.windowHeight();
    super.initialize(x, y, width, height);
    this._editWindow = editWindow;
    this._page = 0;
    this._index = 0;
    this.refresh();
    this.updateCursor();
    this.activate();
  }

  windowHeight() {
    return this.fittingHeight(9);
  }

  table() {
    if ($gameSystem.isJapanese()) {
      return [JAPAN1, JAPAN2, JAPAN3];
    } else if ($gameSystem.isRussian()) {
      return [RUSSIA];
    } else {
      return [LATIN1, LATIN2];
    }
  }

  maxCols() {
    return 10;
  }

  maxItems() {
    return 90;
  }

  character() {
    return this._index < 88 ? this.table()[this._page][this._index] : '';
  }

  isPageChange() {
    return this._index === 88;
  }

  isOk() {
    return this._index === 89;
  }

  itemRect(index) {
    return {
      x: index % 10 * 42 + Math.floor(index % 10 / 5) * 24,
      y: Math.floor(index / 10) * this.lineHeight(),
      width: 42,
      height: this.lineHeight()
    };
  }

  refresh() {
    const table = this.table();
    this.contents.clear();
    this.resetTextColor();
    for (let i = 0; i < 90; i++) {
      const rect = this.itemRect(i);
      rect.x += 3;
      rect.width -= 6;
      this.drawText(table[this._page][i], rect.x, rect.y, rect.width, 'center');
    }
  }

  updateCursor() {
    const rect = this.itemRect(this._index);
    this.setCursorRect(rect.x, rect.y, rect.width, rect.height);
  }

  isCursorMovable() {
    return this.active;
  }

  cursorDown(wrap) {
    if (this._index < 80 || wrap) {
      this._index = (this._index + 10) % 90;
    }
  }

  cursorUp(wrap) {
    if (this._index >= 10 || wrap) {
      this._index = (this._index + 80) % 90;
    }
  }

  cursorRight(wrap) {
    if (this._index % 10 < 9) {
      this._index++;
    } else if (wrap) {
      this._index -= 9;
    }
  }

  cursorLeft(wrap) {
    if (this._index % 10 > 0) {
      this._index--;
    } else if (wrap) {
      this._index += 9;
    }
  }

  cursorPagedown() {
    this._page = (this._page + 1) % this.table().length;
    this.refresh();
  }

  cursorPageup() {
    this._page = (this._page + this.table().length - 1) % this.table().length;
    this.refresh();
  }

  processCursorMove() {
    const lastPage = this._page;
    super.processCursorMove();
    this.updateCursor();
    if (this._page !== lastPage) {
      Managers.Sound.playCursor();
    }
  }

  processHandling() {
    if ($gameTemp.shouldSkipSelectableWindowHandling()) return;

    if (this.isOpen() && this.active) {
      if (Engine.Input.isTriggered('start')) {
        this.onNameOk();
        return;
      }

      if (Engine.Input.isRepeated('cancel') || Engine.Input.isRepeated('alternate') || Engine.Input.isRepeated('backspace')) {
        if (!Engine.Input.isPressed('start')) {
          this.processBack();
        }
      }
      if (Engine.Input.isRepeated('ok')) {
        this.processOk();
      }
    }
  }

  isCancelEnabled() {
    return true;
  }

  processCancel() {
    this.processBack();
  }

  processJump() {
    if (this._index !== 89) {
      this._index = 89;
      Managers.Sound.playCursor();
    }
  }

  processBack() {
    if (this._editWindow.back()) {
      Managers.Sound.playCancel();
    }
  }

  processOk() {
    if (this.character()) {
      this.onNameAdd();
    } else if (this.isPageChange()) {
      Managers.Sound.playOk();
      this.cursorPagedown();
    } else if (this.isOk()) {
      this.onNameOk();
    }
  }

  onNameAdd() {
    if (this._editWindow.add(this.character())) {
      Managers.Sound.playOk();
    } else {
      Managers.Sound.playBuzzer();
    }
  }

  onNameOk() {
    if (this._editWindow.name() === '') {
      if (this._editWindow.restoreDefault()) {
        Managers.Sound.playOk();
      } else {
        Managers.Sound.playBuzzer();
      }
    } else {
      Managers.Sound.playOk();
      this.callOkHandler();
    }
  }
}

module.exports = Window_NameInput;
