const Window_ItemBase = require('./Window_ItemBase');
const Window_Selectable = require('engine/windows/Window_Selectable');

//-----------------------------------------------------------------------------
// Window_ItemContainer
//
// Displays the items from one item container

class Window_ItemContainer extends Window_ItemBase {
  initialize(x, y, width, height, container, maxItems) {
    this._maxItems = maxItems;
    this._container = container;
    this._filteredList = [];
    super.initialize(x, y, width, height);
    this._index = -1;
  }

  maxItems() {
    return this._maxItems;
  }

  item() {
    return this.getItem(this.index());
  }

  getItem(index) {
    return this._filteredList && index >= 0 ? this._filteredList[index] : null;
  }

  filterList() {
    this._filteredList = [];

    if (this._container) {
      if (this._category) {
        for (const item of this._container._items) {
          if (!item) continue;
          if (item.category != this._category) continue;

          this._filteredList.push(item);
        }
      } else {
        this._filteredList = this._container._items;
      }
    }
  }

  refresh() {
    this.createContents();
    this.filterList();

    this.drawItemBackgrounds();
    this.drawAllItems();

    this.drawSelectedItem();

    this._needsRefresh = false;
  }

  update() {
    Window_Selectable.prototype.update.call(this);
    if (this._needsRefresh) {
      this.refresh();
    }
  }

  drawItemBackgrounds() {
    const maxItems = this._maxItems;
    const maxCols = this.maxCols();
    const index = this.index();
    const maxHeight = this.contentsHeight();

    for (let i = 0; i < maxItems; i++) {
      if (i == index) continue;
      if (this.getItem(i)) continue;

      const row = Math.floor(i / maxCols);
      if (row < this.topRow()) continue;

      const rect = this.itemRect(i);
      if (rect.y + rect.height >= maxHeight) continue;

      this.drawEmptyItem(rect.x, rect.y, 1, i);
    }
  }

  drawSelectedItem() {
    const index = this.index();
    if (index < 0) return;

    const rect = this.itemRect(index);
    const item = this.getItem(index);
    this.drawNormalItem(item, rect.x, rect.y, true, index);
  }

  drawItem(index) {
    if (index == this.index()) return;

    const row = Math.floor(index / this.maxCols());
    if (row < this.topRow()) return;

    const item = this.getItem(index);
    if (item) {
      const rect = this.itemRect(index);
      rect.width -= this.textPadding();

      const maxHeight = this.contentsHeight();
      if (rect.y + rect.height >= maxHeight) return;

      this.drawNormalItem(item, rect.x, rect.y, false, index);
    }
  }

  itemWidth() {
    return this.lineHeight();
  }

  itemHeight() {
    return this.lineHeight();
  }

  lineHeight() {
    return 52;
  }

  maxCols() {
    return Math.floor((this.width - (this.padding + this.extraSpacing()) * 2) / this.itemWidth());
  }

  spacing() {
    return 0;
  }

  updateCursor() {
    this._needsRefresh = true;
  }

  updateHelp() {
    this.setHelpWindowItem(this.item());
  }

  callUpdateHelp() {
    if (this._helpWindow || this._infoWindow) {
      this.updateHelp();
    }
  }

  extraSpacing() {
    return 24;
  }

  getIndexLabel(index) {
    const item = this.getItem(index);
    if (!item) return '';

    if (typeof(item) == "string") {
      return Managers.Text.item(item);
    }

    return Managers.Text.item(item.id);
  }

  setCategory(category) {
    this._category = category;
    this.refresh();
  }

  drawItemAmount(amount, x, y, zoomedIn) {
    if (amount <= 1) {
      return;
    }

    if (amount > 999) {
      amount = '999+';
    }

    const zoom = Graphics.windowZoomLevel;

    this.contents.outlineColor = '#f4dea8';
    this.contents.fontSize = Utils.convertFontSize(28, zoom);

    if (zoom >= 2) {
      y += 12 + (zoom - 2) * 16;
    }

    this.drawText(amount, x, y, 24 * zoom, 'left');
  }
}

module.exports = Window_ItemContainer;
