const Window_Base = require('engine/windows/Window_Base');
//-----------------------------------------------------------------------------
// Window_CreateCharacter
//
// The window for configuring the player's character

class Window_CreateCharacter extends Window_Base {
  initialize() {
    const width = 312 * Graphics.windowZoomLevel;
    const height = 309 * Graphics.windowZoomLevel;

    this._genders = Managers.Player.getGenderList();
    this._sprites = Managers.Player.getSpriteList();
    this._hairColors = Managers.Player.getHairColorList();
    this._skins = Managers.Player.getSkinList();
    this._handlers = {};
    this._selectionAreas = [];

    this._bgImage = Managers.Images.loadBackground('layout_charactercreation');
    this._backArrowImage = Managers.Images.loadPicture('menu/arrow_left');
    this._arrowImage = Managers.Images.loadPicture('menu/arrow_right');
    this._frameImage = Managers.Images.loadPicture('portrait_frame');

    this._selectionIndex = 0;

    super.initialize(0, 0, width, height);
    this.updatePlacement();

    this.makeSelectionAreas();
  }

  enableWindowFrame() {
    return false;
  }

  updatePlacement() {
    this.x = (Graphics.width - this.width) / 2;
    this.y = (Graphics.height - this.height) / 2;
  }

  index() {
    return this._selectionIndex;
  }

  setHandler(symbol, callback) {
    this._handlers[symbol] = callback;
  }

  callHandler(symbol) {
    if (this._handlers[symbol]) {
      this._handlers[symbol]();
    }
  }

  allSymbols() {
    const symbols = [
      'name',
      'gender',
      'skinColor',
      'hairColor',
      'sprite',
      'farmName',
      'petName',
      'skipTutorialNo',
      'skipTutorialYes',
      'cancel',
      'startGame'
    ];

    return symbols;    
  }

  commandSymbol(index) {
    const symbols = this.allSymbols();

    if (index < 0 || index >= symbols.length) {
      return false;
    }

    return symbols[index];
  }

  selectedSymbol() {
    return this.commandSymbol(this._selectionIndex);
  }

  processOk() {
    const index = this.index();
    const symbol = this.commandSymbol(index);

    switch(symbol) {
      case 'ok' :
        this.callHandler('returnToGame');
        return;
      case 'startGame' :
        this.callHandler('startGame');
        return;
      case 'quickStart' :
        this.cursorRight(false);
        return;
      case 'cancel' :
        this.callHandler('cancel');
        return;
      case 'name' :
        this.callHandler('inputName');
        return;
      case 'petName' :
        this.callHandler('inputPetName');
        return;
      case 'farmName' :
        this.callHandler('inputFarmName');
        return;
      default:
        this.cursorRight(false);
        return;
    }
  }

  nextGender() {
    let idx = this._genders.indexOf(Managers.Player._gender);
    idx++;

    if (idx >= this._genders.length) idx = 0;

    return this._genders[idx];
  }

  prevGender() {
    let idx = this._genders.indexOf(Managers.Player._gender);
    idx--;

    if (idx < 0) idx = this._genders.length - 1;

    return this._genders[idx];
  }

  nextSprite() {
    let idx = this._sprites.indexOf(Managers.Player._sprite);
    idx++;

    if (idx >= this._sprites.length) idx = 0;

    return this._sprites[idx];
  }

  prevSprite() {
    let idx = this._sprites.indexOf(Managers.Player._sprite);
    idx--;

    if (idx < 0) idx = this._sprites.length - 1;

    return this._sprites[idx];
  }

  nextHairColor() {
    let idx = this._hairColors.indexOf(Managers.Player._hairType);
    idx++;

    if (idx >= this._hairColors.length) idx = 0;

    return this._hairColors[idx];
  }

  prevHairColor() {
    let idx = this._hairColors.indexOf(Managers.Player._hairType);
    idx--;

    if (idx < 0) idx = this._hairColors.length - 1;

    return this._hairColors[idx];
  }

  nextSkin() {
    let idx = this._skins.indexOf(Managers.Player._skinType);
    idx++;

    if (idx >= this._skins.length) idx = 0;

    return this._skins[idx];
  }

  prevSkin() {
    let idx = this._skins.indexOf(Managers.Player._skinType);
    idx--;

    if (idx < 0) idx = this._skins.length - 1;

    return this._skins[idx];
  }

  switchToNextGender() {
    Managers.Player._gender = this.nextGender();
    if (Managers.Player._gender == 'male' && Managers.Player._sprite == 'heroine') {
      Managers.Player._sprite = 'hero';
    }
    if (Managers.Player._gender == 'female' && Managers.Player._sprite == 'hero') {
      Managers.Player._sprite = 'heroine';
    }
  }

  cursorRight() {
    const index = this.index();
    const symbol = this.commandSymbol(index);

    switch(symbol) {
      case 'gender' :
        this.switchToNextGender();
        break;
      case 'sprite' :
        Managers.Player._sprite = this.nextSprite();
        break;
      case 'hairColor' :
        Managers.Player._hairType = this.nextHairColor();
        break;
      case 'skinColor' :
        Managers.Player._skinType = this.nextSkin();
        break;
      case 'quickStart' :
      case 'skipTutorialYes':
      case 'skipTutorialNo':
        Managers.Player._quickStart = !Managers.Player._quickStart;
        this.selectSymbol(Managers.Player._quickStart ? 'skipTutorialYes' : 'skipTutorialNo');
        break;
      case 'cancel':
        this.selectSymbol('startGame');
        break;
      case 'startGame':
        this.selectSymbol('cancel');
        break;
    }

    this.refresh();
  }

  cursorLeft() {
    const index = this.index();
    const symbol = this.commandSymbol(index);

    switch(symbol) {
      case 'gender' :
        Managers.Player._gender = this.prevGender();
        break;
      case 'sprite' :
        Managers.Player._sprite = this.prevSprite();
        break;
      case 'hairColor' :
        Managers.Player._hairType = this.prevHairColor();
        break;
      case 'skinColor' :
        Managers.Player._skinType = this.prevSkin();
        break;
      case 'quickStart' :
      case 'skipTutorialYes':
      case 'skipTutorialNo':
        Managers.Player._quickStart = !Managers.Player._quickStart;
        this.selectSymbol(Managers.Player._quickStart ? 'skipTutorialYes' : 'skipTutorialNo');
        break;
      case 'cancel':
        this.selectSymbol('startGame');
        break;
      case 'startGame':
        this.selectSymbol('cancel');
        break;
    }

    this.refresh();
  }

  standardPadding() {
    return 0;
  }

  spacing() {
    return 0;
  }

  makeSelectionAreas() {
    this._selectionAreas = [];
    this.addArea(151, 30, 141, 16, 'name');

    this.addArea(151, 48, 141, 16, 'gender');
    this.addArea(151, 66, 141, 16, 'birthday');
    this.addArea(34, 114, 97, 12, 'skinColor');
    this.addArea(34, 138, 136, 19, 'hairColor');
    this.addArea(151, 185, 141, 16, 'farmName');
    this.addArea(151, 203, 141, 16, 'petName');
    this.addArea(222, 100, 58, 67, 'sprite');
    
    this.addArea(222, 221, 35, 16, 'skipTutorialNo');
    this.addArea(257, 221, 35, 16, 'skipTutorialYes');
    
    this.addArea(175, 277, 58, 16, 'cancel');
    this.addArea(235, 277, 58, 16, 'startGame');
    
    this.addArea(220, 125, 15, 20, 'previousSprite');
    this.addArea(268, 125, 15, 20, 'nextSprite');
  }

  drawSelectedArea() {
    const symbol = this.selectedSymbol();

    for (let area of this._selectionAreas) {
      if (area.name == symbol) {
        this.drawSelection(area.x, area.y, area.w, area.h);
      }
    }
  }

  drawStartButtons() {
    const x = 20 * Graphics.windowZoomLevel;
    const y = 275 * Graphics.windowZoomLevel;
    const w = 150 * Graphics.windowZoomLevel;

    this.changeTextColor('#3D2B25');
    this.drawText(t("Start your journey") + '?', x, y, w, 'left');
    this.resetTextColor();

    const cancelX = 176 * Graphics.windowZoomLevel;
    const beginX = 236 * Graphics.windowZoomLevel;
    const buttonY = 278 * Graphics.windowZoomLevel;
    const buttonW = 56 * Graphics.windowZoomLevel;
    const buttonH = 14 * Graphics.windowZoomLevel;

    this.contents.fillRect(cancelX, buttonY, buttonW, buttonH, '#D7A678');
    this.contents.fillRect(beginX, buttonY, buttonW, buttonH, '#328FFA');

    this.drawText(t("Cancel"), cancelX, y, buttonW, 'center');
    this.drawText(t("Start"), beginX, y, buttonW, 'center');
  }

  refresh() {
    $gamePlayer._characterName = Managers.Player.generatePlayerSpriteName();

    this.makeSelectionAreas();

    this._refreshAllParts();
    this.createContents();

    this.drawBitmap(this._bgImage, 0, 0, Graphics.windowZoomLevel);
    this.drawSelectedArea();

    this.resetFontSettings();

    this.drawPlayerName();
    this.drawPlayerGender();
    // this.drawBirthday();
    this.drawFarmName();
    this.drawPetName();

    this.drawSkinColor();
    this.drawHairColor();

    this.drawCharacterPreview();

    this.drawSkipTutorial();
    this.drawStartButtons();
  }

  addArea(x, y, w, h, name) {
    this._selectionAreas.push({
      x,
      y,
      w,
      h,
      name
    });
  }

  drawPlayerName() {
    const y = 28 * Graphics.windowZoomLevel;

    const titleX = 84 * Graphics.windowZoomLevel;
    const titleW = 65 * Graphics.windowZoomLevel;

    const valueX = 153 * Graphics.windowZoomLevel;
    const valueW = 137 * Graphics.windowZoomLevel;

    // this.changeTextColor('#328FFA');
    this.drawText(t('Name') + ':', titleX, y, titleW, 'left');
    this.resetTextColor();

    const name = Managers.Player.getName();
    this.drawText(name, valueX, y, valueW, 'left');
  }

  drawSelection(x, y, w, h) {
    const z = Graphics.windowZoomLevel;
    const bgX = x * z;
    const bgY = y * z;
    const bgW = w * z;
    const bgH = h * z;

    this.contents.fillRect(bgX, bgY, bgW, z, '#3D2B25');
    this.contents.fillRect(bgX, bgY, z, bgH, '#3D2B25');
    this.contents.fillRect(bgX + bgW - z, bgY, z, bgH, '#3D2B25');
    this.contents.fillRect(bgX, bgY + bgH - z, bgW, z, '#3D2B25');

    this.contents.fillRect(bgX + z, bgY + z, z, z, '#3D2B25');
    this.contents.fillRect(bgX + z, bgY + bgH - z * 2, z, z, '#3D2B25');
    this.contents.fillRect(bgX + bgW - z * 2, bgY + z, z, z, '#3D2B25');
    this.contents.fillRect(bgX + bgW - z * 2, bgY + bgH - z * 2, z, z, '#3D2B25');
  }

  drawPlayerGender() {
    const titleX = 84 * Graphics.windowZoomLevel;
    const titleW = 65 * Graphics.windowZoomLevel;

    const valueX = 153 * Graphics.windowZoomLevel;
    const valueW = 137 * Graphics.windowZoomLevel;
    const y = 46 * Graphics.windowZoomLevel;

    // this.changeTextColor('#328FFA');
    this.drawText(t('Gender') + ':', titleX, y, titleW, 'left');
    this.resetTextColor();

    const gender = Managers.Player.genderName();
    this.drawText(gender, valueX, y, valueW, 'left');

    if (this._arrowImage.isReady()) {
      this.drawBitmap(this._arrowImage, 284, 51, Graphics.windowZoomLevel);
    }
  }

  drawHairColor() {
    const x = 36 * Graphics.windowZoomLevel;
    const y = 140 * Graphics.windowZoomLevel;
    const w = 14 * Graphics.windowZoomLevel;
    const h = 8 * Graphics.windowZoomLevel;

    const label = t("Hair Color") + ':';
    this.contents.fontSize = Utils.convertFontSize(26, Graphics.windowZoomLevel);
    this.drawText(label, x, y - 15 * Graphics.windowZoomLevel, 100 * Graphics.windowZoomLevel, 'left');

    const hairList = Managers.Player.getHairColorList();
    const hairColor = Managers.Player.getHairColor();
    const hairIndex = hairList.indexOf(hairColor);

    const drawX = x + (w - Graphics.windowZoomLevel) * (hairIndex % Math.ceil(hairList.length / 2));
    const drawY = y + (hairIndex >= (hairList.length / 2) ? h - Graphics.windowZoomLevel : 0);

    this.contents.fillRect(drawX, drawY, w, Graphics.windowZoomLevel, '#3D2B25');
    this.contents.fillRect(drawX, drawY + h - Graphics.windowZoomLevel, w, Graphics.windowZoomLevel, '#3D2B25');

    this.contents.fillRect(drawX, drawY, Graphics.windowZoomLevel, h, '#3D2B25');
    this.contents.fillRect(drawX + w - Graphics.windowZoomLevel, drawY, Graphics.windowZoomLevel, h, '#3D2B25');
  }

  drawSkinColor() {
    const x = 36 * Graphics.windowZoomLevel;
    const y = 116 * Graphics.windowZoomLevel;
    const w = 14 * Graphics.windowZoomLevel;
    const h = 8 * Graphics.windowZoomLevel;

    const label = t("Skin Color") + ':';
    this.contents.fontSize = Utils.convertFontSize(26, Graphics.windowZoomLevel);
    this.drawText(label, x, y - 15 * Graphics.windowZoomLevel, 100 * Graphics.windowZoomLevel, 'left');

    const skinList = Managers.Player.getSkinList();
    const skin = Managers.Player.getSkinValue();
    const skinIndex = skinList.indexOf(skin);

    const drawX = x + (w - Graphics.windowZoomLevel) * skinIndex;

    this.contents.fillRect(drawX, y, w, Graphics.windowZoomLevel, '#3D2B25');
    this.contents.fillRect(drawX, y + h - Graphics.windowZoomLevel, w, Graphics.windowZoomLevel, '#3D2B25');

    this.contents.fillRect(drawX, y, Graphics.windowZoomLevel, h, '#3D2B25');
    this.contents.fillRect(drawX + w - Graphics.windowZoomLevel, y, Graphics.windowZoomLevel, h, '#3D2B25');
  }

  drawFarmName() {
    const titleX = 84 * Graphics.windowZoomLevel;
    const titleW = 65 * Graphics.windowZoomLevel;

    const valueX = 153 * Graphics.windowZoomLevel;
    const valueW = 137 * Graphics.windowZoomLevel;

    const y = 183 * Graphics.windowZoomLevel;

    // this.changeTextColor('#328FFA');
    this.drawText(t("Farm Name") + ':', titleX, y, titleW, 'left');
    this.resetTextColor();

    const name = Variables.farmName;
    this.drawText(name, valueX, y, valueW, 'left');
  }

  drawBirthday() {
    const titleX = 84 * Graphics.windowZoomLevel;
    const titleW = 65 * Graphics.windowZoomLevel;

    const valueX = 153 * Graphics.windowZoomLevel;
    const valueW = 137 * Graphics.windowZoomLevel;
    const y = 64 * Graphics.windowZoomLevel;

    // this.changeTextColor('#328FFA');
    this.drawText(t("Birthday") + ':', titleX, y, titleW, 'left');
    this.resetTextColor();

    const name = Managers.Player.getBirthdayLabel();
    this.drawText(name, valueX, y, valueW, 'left');
  }

  drawSkipTutorial() {
    const x = 84 * Graphics.windowZoomLevel;
    const y = 219 * Graphics.windowZoomLevel;
    const w = 134 * Graphics.windowZoomLevel;

    this.changeTextColor('#3D2B25');
    this.drawText(t("Quick Start") + '?', x, y, w, 'left');
    this.resetTextColor();

    const noX = 223 * Graphics.windowZoomLevel;
    const yesX = 258 * Graphics.windowZoomLevel;
    const buttonY = 222 * Graphics.windowZoomLevel;
    const buttonW = 33 * Graphics.windowZoomLevel;
    const buttonH = 14 * Graphics.windowZoomLevel;

    const noColor = Managers.Player._quickStart ? '#D7A678' : '#FB3153';
    const yesColor = Managers.Player._quickStart ? '#FB3153' : '#D7A678';

    this.contents.fillRect(noX, buttonY, buttonW, buttonH, noColor);
    this.contents.fillRect(yesX, buttonY, buttonW, buttonH, yesColor);

    this.drawText(t("No"), noX, y, buttonW, 'center');
    this.drawText(t("Yes"), yesX, y, buttonW, 'center');
  }

  drawPetName() {
    const titleX = 84 * Graphics.windowZoomLevel;
    const titleW = 65 * Graphics.windowZoomLevel;

    const valueX = 153 * Graphics.windowZoomLevel;
    const valueW = 137 * Graphics.windowZoomLevel;
    const y = 201 * Graphics.windowZoomLevel;

    // this.changeTextColor('#328FFA');
    this.drawText(t("Pet Name") + ':', titleX, y, titleW, 'left');
    this.resetTextColor();

    const name = Variables.dogName;
    this.drawText(name, valueX, y, valueW, 'left');
  }

  drawCharacterPreview() {
    const actorData = $gamePlayer;

    const x = 225 * Graphics.windowZoomLevel;
    const y = 105 * Graphics.windowZoomLevel;
    const bitmap = Managers.Images.loadCharacter(actorData._characterName);

    if (bitmap.isReady()) {
      this._lastPlayerBitmap = bitmap;
      this.drawCharacterSprites(bitmap, actorData._characterName, actorData._characterIndex, x, y);
    } else {
      this.requireBitmap(bitmap);
      if (this._lastPlayerBitmap && this._lastPlayerBitmap.isReady()) {
        this.drawCharacterSprites(this._lastPlayerBitmap, actorData._characterName, actorData._characterIndex, x, y);
      }
    }

    if (this._backArrowImage.isReady()) {
      this.drawBitmap(this._backArrowImage, 225, 128, Graphics.windowZoomLevel);
    }
    if (this._arrowImage.isReady()) {
      this.drawBitmap(this._arrowImage, 270, 128, Graphics.windowZoomLevel);
    }

    const faceX = 25;
    const faceY = 27;
    const faceBitmap = Managers.Images.loadPortrait(`small/${actorData._characterName}`);

    if (faceBitmap.isReady()) {
      this._lastPlayerFaceBitmap = faceBitmap;
      this.drawBitmap(faceBitmap, faceX, faceY, Graphics.windowZoomLevel);
    } else {
      this.requireBitmap(faceBitmap);
      if (this._lastPlayerFaceBitmap && this._lastPlayerFaceBitmap.isReady()) {
        this.drawBitmap(this._lastPlayerFaceBitmap, faceX, faceY, Graphics.windowZoomLevel);
      }
    }

    if (this._frameImage.isReady()) {
      this.drawBitmap(this._frameImage, 21, 23, Graphics.windowZoomLevel);
    }

  }

  drawCharacterSprites(bitmap, characterName, characterIndex, x, y) {
    const big = Managers.Images.isBigCharacter(characterName);
    const pw = bitmap.width / (big ? 3 : 12);
    const ph = bitmap.height / (big ? 4 : 8);
    const n = characterIndex;
    const sx = (n % 4 * 3 + 1) * pw;
    const dw = pw * Graphics.windowZoomLevel;
    const dh = ph * Graphics.windowZoomLevel;
    const dx = x + (52 - pw) / 2 * Graphics.windowZoomLevel;
    const dy = y;

    const sy = (Math.floor(n / 4) * 4) * ph;

    this.contents.bltBitmap(bitmap, sx, sy, pw, ph, dx, dy, dw, dh);
  }

  standardOutlineWidth() {
    return 0;
  }

  normalColor() {
    return this.baseColor();
  }

  processCursorMove() {
    const lastIndex = this.index();
    if (Engine.Input.isRepeated('down')) {
      this.cursorDown();
    }

    if (Engine.Input.isRepeated('up')) {
      this.cursorUp();
    }

    if (Engine.Input.isRepeated('right')) {
      this.cursorRight();
    }

    if (Engine.Input.isRepeated('left')) {
      this.cursorLeft();
    }

    if (this.index() !== lastIndex) {
      Managers.Sound.playCursor();
      this.refresh();
    }
  }

  select(index) {
    const maxItems = this.allSymbols().length;

    if (index < 0) {
      this._selectionIndex = maxItems -1;
    } else if (index < maxItems) {
      this._selectionIndex = index;
    } else {
      this._selectionIndex = 0;
    }
  }

  selectSymbol(symbol) {
    const symbols = this.allSymbols();
    const idx = symbols.indexOf(symbol);

    if (idx >= 0) {
      this.select(idx);
    }
  }

  cursorDown() {
    const oldSymbol = this.selectedSymbol();
    if (oldSymbol === 'startGame' || oldSymbol === 'cancel') {
      this.selectSymbol('name');
      return;
    }

    this.select(this.index() + 1);

    const symbol = this.selectedSymbol();

    if (symbol == 'skipTutorialNo' && Managers.Player._quickStart) {
      this._selectionIndex++;
    } else if (symbol == 'skipTutorialYes' && !Managers.Player._quickStart) {
      this._selectionIndex += 2;
    }

    if (symbol == 'cancel') {
      this._selectionIndex++;
    }
  }

  cursorUp() {
    const oldSymbol = this.selectedSymbol();
    if (oldSymbol === 'startGame' || oldSymbol === 'cancel') {
      this.selectSymbol('skipTutorialYes');
    } else {
      this.select(this.index() - 1);
    }

    const symbol = this.selectedSymbol();
    if (symbol == 'skipTutorialYes' && !Managers.Player._quickStart) {
      this._selectionIndex--;
    } else if (symbol == 'skipTutorialNo' && Managers.Player._quickStart) {
      this._selectionIndex--;
    } 
  }

  update() {
    super.update();

    if (!this.active) {
      return;
    }

    this.processCursorMove();

    if (!$gameTemp.shouldSkipSelectableWindowHandling()) {
      if (Engine.Input.isTriggered('ok')) {
        this.processOk();
      }

      this.processTouch();
    }
  }

  processAreaTouch(areaName, x, y) {
    const symbols = this.allSymbols();
    const idx = symbols.indexOf(areaName);

    if (idx >= 0) {
      this.select(idx);
    }

    if (TouchInput.isTriggered()) {
      switch(areaName) {
        case 'skinColor':
          {
            const skinCol = Math.floor((x - 3) / 13);
            if (skinCol >= 0 && skinCol < this._skins.length) {
              const newSkin = this._skins[skinCol];
              Managers.Player._skinType = newSkin;
            }
          }

          break;
        case 'hairColor':
          {
            const hairCol = Math.floor((x -3) / 13);
            const hairRow = Math.floor((y -3) / 7);
            const hairIdx = hairRow * 10 + hairCol;
            if (hairIdx >= 0 && hairIdx < this._hairColors.length) {
              const newHair = this._hairColors[hairIdx];
              Managers.Player._hairType = newHair;
            }
          }

          break;
        case 'gender':
          this.switchToNextGender();
          break;
        case 'sprite':
          break;
        case 'previousSprite':
          Managers.Player._sprite = this.prevSprite();
          break;
        case 'nextSprite':
          Managers.Player._sprite = this.nextSprite();
          break;
        case 'skipTutorialYes':
          Managers.Player._quickStart = true;
          break;
        case 'skipTutorialNo':
          Managers.Player._quickStart = false;
          break;
        default:
          if (idx >= 0) {
            this.processOk();
          }
          break;
      }
    }

    this.refresh();
  }

  processTouch() {
    if (this._lastMouseX === TouchInput.mouseX && this._lastMouseY === TouchInput.mouseY) {
      if (!TouchInput.isTriggered()) {
        return;
      }
    }

    this._lastMouseX = TouchInput.mouseX;
    this._lastMouseY = TouchInput.mouseY;

    const x = (TouchInput.mouseX - this.x) / Graphics.windowZoomLevel;
    const y = (TouchInput.mouseY - this.y) / Graphics.windowZoomLevel;

    for (let area of this._selectionAreas) {
      if (x < area.x) continue;
      if (x >= area.x + area.w - 4) continue;
      if (y < area.y) continue;
      if (y >= area.y + area.h - 4) continue;

      this.processAreaTouch(area.name, x - area.x, y - area.y);
    }
  }
}

module.exports = Window_CreateCharacter;